//////////////////////////////////////////////////////////////////////////////
// Quake 2 Model implementation
//////////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <math.h>
#include <string.h>
#include "mdlib.h"





// load a Q2 model file
// TODO: put all mallocs() in one place to speed up
q2mdl_t Q2LoadFile(char *filename)
{
    PACKFILE *mdl_file;
    long i,j;
    long frame, trig;
    long bseek = 0;                  // number of bytes read
    V3D_f v1, v2, v3;
    q2mdl_t model;
    BITMAP *tmp;
    PALETTE pal;
    unsigned char buffer[MAX_VERTS*4+128];
    q2daliasframe_t *out;
    unsigned char skinname[256];


    mdl_file = pack_fopen(filename, F_READ);
    if(mdl_file == NULL)	// return an error and exit
    {
        printf("Q2LoadFile(): ");
        perror(filename);
        exit(1);
    }

    // read the header
    bseek += pack_fread(&model.header, sizeof(q2dmdl_t), mdl_file);

    // compute scaling factors for texture because Allegro needs
    // read the skin names
    if( model.header.num_skins )
    {
        pack_fseek(mdl_file, model.header.ofs_skins-bseek);
        bseek = model.header.ofs_skins;
        for(i=0; i<model.header.num_skins; i++)
            bseek += pack_fread(&model.skins[i], MAX_SKINNAME, mdl_file);
    }

    // read texture coordinates (dstvert_t)
    model.texture_st = (q2dstvert_t *) malloc(model.header.num_st*sizeof(q2dstvert_t));
    if(model.texture_st == NULL)
    {
        printf("Q2LoadFile(): Not enough memory for vertices\n");
        exit(1);
    }
    pack_fseek(mdl_file, model.header.ofs_st-bseek);
    bseek = model.header.ofs_st;
    bseek += pack_fread(model.texture_st, model.header.num_st*sizeof(q2dstvert_t), mdl_file);


    // read triangle array indexes
    model.tri_index = (q2dtriangle_t *)malloc( model.header.num_tris*sizeof(q2dtriangle_t) );
    if(model.tri_index == NULL)
    {
        printf("Q2LoadFile(): Not enough memory for triangle index\n");
        exit(1);
    }
    pack_fseek(mdl_file, model.header.ofs_tris-bseek);
    bseek = model.header.ofs_tris;
    for(i=0; i<model.header.num_tris; i++)
        bseek += pack_fread(&model.tri_index[i], sizeof(q2dtriangle_t), mdl_file);


    model.framelist = (q2framelist_t *)malloc( model.header.num_frames*sizeof(q2framelist_t) );
    if(model.framelist == NULL)
    {
        printf("Q2LoadFile(): Not enough memory for frame list\n");
        exit(1);
    }

    model.points = (V3D_f *)malloc( model.header.num_xyz*sizeof(V3D_f) );
    if(model.points == NULL)
    {
        printf("Q2LoadFile(): Not enough memory for 3D points\n");
        exit(1);
    }
    model.rpos = (V3D_f *)malloc( model.header.num_xyz*sizeof(V3D_f) );
    if(model.points == NULL)
    {
        printf("Q2LoadFile(): Not enough memory for 3D points projection\n");
        exit(1);
    }

    // read triangle vertices
    pack_fseek(mdl_file, model.header.ofs_frames-bseek);
    bseek = model.header.ofs_frames;

    for(i=0; i<model.header.num_frames; i++)
    {
        model.framelist[i].tris = (V3D_f *)malloc( model.header.num_xyz*sizeof(V3D_f) );
        if(model.framelist[i].tris == NULL)
        {
            printf("Q2LoadFile(): Not enough memory for triangles at frame %i\n", i);
            exit(1);
        }

        out = (q2daliasframe_t *) buffer;

        bseek += pack_fread(out, model.header.framesize, mdl_file);
        for(j=0; j<model.header.num_xyz; j++)
        {
            model.framelist[i].tris[j].x = out->verts[j].v[0] * out->scale[0] + out->translate[0];
            model.framelist[i].tris[j].y = out->verts[j].v[1] * out->scale[1] + out->translate[1];
            model.framelist[i].tris[j].z = out->verts[j].v[2] * out->scale[2] + out->translate[2];
            model.framelist[i].tris[j].u = model.texture_st[j].s;
            model.framelist[i].tris[j].v = model.texture_st[j].t;
            model.framelist[i].tris[j].c = 10;
        }
    }

    // read gl commands
    model.commands = (long *)malloc( model.header.num_glcmds*sizeof(long) );
    if(model.commands == NULL)
    {
        printf("Q2LoadFile(): Not enough memory for GL commands\n");
        exit(1);
    }
    pack_fseek(mdl_file, model.header.ofs_glcmds-bseek);
    bseek = model.header.ofs_glcmds;
    bseek += pack_fread(model.commands, model.header.num_glcmds*sizeof(long), mdl_file);


    // close file
    pack_fclose(mdl_file);


    // allocate memory for distance to midpoints list and for the order array
    model.dist2 = (float *)malloc( model.header.num_tris*sizeof(float) );
    model.trigorder = (long *)malloc( model.header.num_tris*sizeof(long) );
    model.drawtrig = (char *)malloc( model.header.num_tris*sizeof(char) );


    // textures that are power of 2  =>  512x256
    model.xscale = (float)XTEX_CONV/(float)model.header.skinwidth;
    model.yscale = (float)YTEX_CONV/(float)model.header.skinheight;

    for(j=0; j<model.header.num_xyz; j++)
    {
       model.rpos[j].u = model.texture_st[j].s * model.xscale;
       model.rpos[j].v = model.texture_st[j].t * model.yscale;
       model.rpos[j].c = j;
    }


    model.tex = create_bitmap(XTEX_CONV, YTEX_CONV);

/*
    strcpy( skinname, filename );
    skinname[ strlen(skinname)-3 ] = 0;
    strcat(skinname, "pcx");
    tmp = load_bitmap( skinname, pal );
*/
    tmp = load_bitmap( "skin.pcx", pal );
    if(tmp == NULL)
    {
//        printf("Q2LoadFile(): Could not find skin %s\n", skinname);
        printf("Q2LoadFile(): Could not find skin.pcx\n");
        exit(1);
    }

    // ATENCAO a esta cena... nao tou a usar o tex e tou a usar directamente o model.tex

    // adjust to 512x256
    stretch_blit(tmp, model.tex, 0, 0, model.header.skinwidth, model.header.skinheight, 0, 0, XTEX_CONV, YTEX_CONV);


    destroy_bitmap( tmp );

    return( model );
}

// free memory ocuppied by a model
// Fix this if you alter anything in the Q1LoadModel()
void Q2Unload(mdl_t *model)
{
   long i;

   free( model->q2mdl->texture_st );
   free( model->q2mdl->tri_index );
   free( model->q2mdl->points );
   free( model->q2mdl->rpos );

   for(i=0; i<model->q2mdl->header.num_frames; i++) free( model->q2mdl->framelist[i].tris );

   free( model->q2mdl->framelist );
   free( model->q2mdl->commands );
   free( model->q2mdl->dist2 );
   free( model->q2mdl->drawtrig );
   free( model->q2mdl->trigorder );


}

// show info on the model
void Q2Debug(mdl_t *model)
{
    long i,j;
    long frame;

    printf("Q2Debug()\n");

    printf("\nHeader:\n");
    printf("ident = %X\n", model->q2mdl->header.ident);
    printf("version = %i\n", model->q2mdl->header.version);
    printf("skinwidth = %i\n", model->q2mdl->header.skinwidth);
    printf("skinheight = %i\n", model->q2mdl->header.skinheight);
    printf("framesize = %i\n", model->q2mdl->header.framesize);

    printf("num_skins = %i\n", model->q2mdl->header.num_skins);
    printf("num_xyz = %i\n", model->q2mdl->header.num_xyz);
    printf("num_st = %i\n", model->q2mdl->header.num_st);
    printf("num_tris = %i\n", model->q2mdl->header.num_tris);
    printf("num_glcmds = %i\n", model->q2mdl->header.num_glcmds);
    printf("num_frames = %i\n", model->q2mdl->header.num_frames);

    printf("ofs_skins = 0x%X\n", model->q2mdl->header.ofs_skins);
    printf("ofs_st = 0x%X\n", model->q2mdl->header.ofs_st);
    printf("ofs_tris = 0x%X\n", model->q2mdl->header.ofs_tris);
    printf("ofs_frames = 0x%X\n", model->q2mdl->header.ofs_frames);
    printf("ofs_glcmds = 0x%X\n", model->q2mdl->header.ofs_glcmds);
    printf("ofs_end = 0x%X\n", model->q2mdl->header.ofs_end);

    printf("\nSkin Names:\n");
    for(i=0; i<model->q2mdl->header.num_skins; i++)
        printf("%s\n", model->q2mdl->skins[i]);

    printf("\nTexture Coord:\n");
    for(i=0; i<model->q2mdl->header.num_st; i++)
        printf("s = %i \t t = %i\n", model->q2mdl->texture_st[i].s, model->q2mdl->texture_st[i].s);

    printf("\nTriangle Index:\n");
    for(i=0; i<model->q2mdl->header.num_tris; i++)
        printf("(x,y,z) = %i %i %i \t (t1,t2,t3) = %i %i %i\n",
            model->q2mdl->tri_index[i].index_xyz[0], model->q2mdl->tri_index[i].index_xyz[1],
            model->q2mdl->tri_index[i].index_xyz[2], model->q2mdl->tri_index[i].index_st[0],
            model->q2mdl->tri_index[i].index_st[1],  model->q2mdl->tri_index[i].index_st[2]);
/*
    printf("\nFrames: %i\n");
    for(i=0; i<model->q2mdl->header.num_frames; i++)
    {
        printf("Frame %i\n", i);
        for(j=0; j<model->q2mdl->header.num_xyz; j++)
            printf("X Y Z = %lf %lf %lf\n", model->q2mdl->framelist[i].tris[j].x,
                model->q2mdl->framelist[i].tris[j].y, model->q2mdl->framelist[i].tris[j].z );
    }
*/

    printf("\nGL Commands:\n");
    for(i=0; i<model->q2mdl->header.num_glcmds; i++)
        printf("cmd %i = %0X\n", i, model->q2mdl->commands[i]);


}


// draw only the points
void Q2DrawPoints(BITMAP *dbuf, mdl_t *model)
{
   long i;

   // draw dots
   for(i=0; i<model->q2mdl->header.num_xyz; i++)
		putpixel(dbuf, (int)model->q2mdl->rpos[i].x, (int)model->q2mdl->rpos[i].y, 10);
}

// draw wireframe triangles
void Q2DrawWireframe(BITMAP *dbuf, mdl_t *model)
{
    V3D_f v1, v2, v3;
    long i, trig;

    // draw every triangle
    for(i=model->q2mdl->header.num_tris-1; i>=0; i--)
    {
        trig = model->q2mdl->trigorder[i];
        v1 = model->q2mdl->rpos[model->q2mdl->tri_index[i].index_xyz[0]];
        v2 = model->q2mdl->rpos[model->q2mdl->tri_index[i].index_xyz[2]];
        v3 = model->q2mdl->rpos[model->q2mdl->tri_index[i].index_xyz[1]];

//        if(model->q2mdl->drawtrig[i] && v1.z > Z_CLIP && v2.z > Z_CLIP && v3.z > Z_CLIP)
        {
            line(dbuf, v1.x, v1.y, v2.x, v2.y, 10);
            line(dbuf, v1.x, v1.y, v3.x, v3.y, 10);
            line(dbuf, v3.x, v3.y, v2.x, v2.y, 10);
        }
    }
}


// Draw textured triangles
void Q2DrawTexture(BITMAP *dbuf, mdl_t *model)
{
    long i, j;
    long trig;
    V3D_f v1, v2, v3;        // tmp variables

    // draw every triangle
    for(i=model->q2mdl->header.num_tris-1; i>=0; i--)
    {
        trig = model->q2mdl->trigorder[i];
        v1 = model->q2mdl->rpos[model->q2mdl->tri_index[trig].index_xyz[0]];
        v2 = model->q2mdl->rpos[model->q2mdl->tri_index[trig].index_xyz[2]];
        v3 = model->q2mdl->rpos[model->q2mdl->tri_index[trig].index_xyz[1]];

        v1.u = model->q2mdl->texture_st[ model->q2mdl->tri_index[trig].index_st[0] ].s * model->q2mdl->xscale;
        v1.v = model->q2mdl->texture_st[ model->q2mdl->tri_index[trig].index_st[0] ].t * model->q2mdl->yscale;
        v2.u = model->q2mdl->texture_st[ model->q2mdl->tri_index[trig].index_st[2] ].s * model->q2mdl->xscale;
        v2.v = model->q2mdl->texture_st[ model->q2mdl->tri_index[trig].index_st[2] ].t * model->q2mdl->yscale;
        v3.u = model->q2mdl->texture_st[ model->q2mdl->tri_index[trig].index_st[1] ].s * model->q2mdl->xscale;
        v3.v = model->q2mdl->texture_st[ model->q2mdl->tri_index[trig].index_st[1] ].t * model->q2mdl->yscale;


//      if(model->q2mdl->drawtrig[i] && v1.z > Z_CLIP && v2.z > Z_CLIP && v3.z > Z_CLIP)
           triangle3d_f(dbuf, POLYTYPE_ATEX, model->q2mdl->tex, &v1, &v2, &v3);

   }

}

// rotate and move the points and sort the traingles
void Q2UpdateView(camera_t cam, mdl_t *model)
{
    float xfront, yfront, zfront;
    float xup, yup, zup;
    float dot;
    long i, j;
//    long frameidx = 0;       // current frame
    MATRIX_f roller, camview;
    V3D_f v1, v2, v3, v4, vc, v4r, vn, vnr;   // tmp variables

    // calculate the in-front vector
    xfront = sin(cam.head) * cos(cam.pitch);
    yfront = sin(cam.pitch);
    zfront = cos(cam.head) * cos(cam.pitch);

    // rotate the up vector around the in-front vector by the roll angle
//    get_vector_rotation_matrix_f(&roller, xfront, yfront, zfront, cam.roll*128.0/M_PI);
//    apply_matrix_f(&roller, 0, -1, 0, &xup, &yup, &zup);
    get_transformation_matrix_f(&camview, 1., cam.head, cam.roll, cam.pitch, cam.x, cam.y, cam.z);
/*
    // build the camera matrix
    get_camera_matrix_f(&camview,
                        cam.x, cam.y, cam.z,     // camera position
                        xfront, yfront, zfront,  // in-front vector
                        xup, yup, zup,           // up vector
                        48,                      // field of view
                        1.);                     // aspect ratio
*/
    // rotate the points and
    for(i=0; i<model->q2mdl->header.num_xyz; i++)
        apply_matrix_f(&camview, model->q2mdl->framelist[ model->frameidx ].tris[i].x, model->q2mdl->framelist[ model->frameidx ].tris[i].y, model->q2mdl->framelist[ model->frameidx ].tris[i].z,
                       &model->q2mdl->rpos[i].x, &model->q2mdl->rpos[i].y, &model->q2mdl->rpos[i].z);

    // this might do the work for now!
    for(i=0; i<model->q2mdl->header.num_tris; i++)
    {
        v1 = model->q2mdl->rpos[ model->q2mdl->tri_index[i].index_xyz[0] ];
        v2 = model->q2mdl->rpos[ model->q2mdl->tri_index[i].index_xyz[2] ];
        v3 = model->q2mdl->rpos[ model->q2mdl->tri_index[i].index_xyz[1] ];

        // average of the triangle
        v4.x = (v3.x + v2.x + v1.x)/3.;
        v4.y = (v3.y + v2.y + v1.y)/3.;
        v4.z = (v3.z + v2.z + v1.z)/3.;

        // calculate distance from view to average triangle point
//         model->q2mdl->dist2[i] = (v4.x-xpos)*(v4.x-xpos) + (v4.y-ypos)*(v4.y-ypos) + (v4.z-zpos)*(v4.z-zpos);
        model->q2mdl->dist2[i] = (v4.x)*(v4.x) + (v4.y)*(v4.y) + (v4.z)*(v4.z);
        model->q2mdl->trigorder[i] = i;

        cross_product_f(v2.x-v1.x, v2.y-v1.y, v2.z-v1.z, v3.x-v1.x, v3.y-v1.y, v3.z-v1.z, &vc.x, &vc.y, &vc.z);
        dot = dot_product_f(vc.x, vc.y, vc.z, v4.x, v4.y, v4.z);

//        if(dot > 0) model->q2mdl->drawtrig[i] = 1;
//        else model->q2mdl->drawtrig[i] = 0;
        model->q2mdl->drawtrig[i] = 1;
    }

   // calculate perspective
   for(i=0; i<model->q2mdl->header.num_xyz; i++)
      persp_project_f(model->q2mdl->rpos[i].x, model->q2mdl->rpos[i].y, model->q2mdl->rpos[i].z, &model->q2mdl->rpos[i].x, &model->q2mdl->rpos[i].y);


   // sort the triangles
   Mysort(model->q2mdl->dist2, model->q2mdl->trigorder, 0, model->q2mdl->header.num_tris-1);

}

