indexing
    description    : "Allegro con Eiffel: joystick objects"
    status         : "Initial development"
    author         : "Peter Monks (pmonks@iname.com)"
    allegro_author : "Shawn Hargreaves (shawn@talula.demon.co.uk)"
    names          : joystick
    date_started   : "30th January, 1997"
    version        : "0.1 beta"
    platforms      : "MS-DOS"
    dependencies   : "Allegro v2.2, DJGPP v2.01"


class JOYSTICK


inherit
    ACE_INFORMATION_SINGLETON  -- implementation inheritance
    end  -- inherit ACE_INFORMATION_SINGLETON


creation { ANY }
    make


------------------------------------------------------ Creation features
feature { ANY }

    make is
    -- Create a new joystick
    require
        ace_initialised        : info.ace_initialised
        joystick_not_installed : not info.joystick_installed
    do
        c_inline_c("joy_type=JOY_TYPE_STANDARD;")

        centre_calibrated       := FALSE
        top_left_calibrated     := FALSE
        bottom_right_calibrated := FALSE
        throttle_min_calibrated := FALSE
        throttle_max_calibrated := FALSE
    ensure
        not_calibrated     : not centre_calibrated   and
                             not top_left_calibrated and
                             not bottom_right_calibrated
        throttle_not_calibrated : not throttle_min_calibrated and
                             not throttle_max_calibrated
    end  -- feature make


------------------------------------------------------ Joystick types
feature { ANY }

    standard : INTEGER is
    -- An ID for the standard joystick
    once
        c_inline_c("R=JOY_TYPE_STANDARD;")
    end  -- feature standard


    flightstick_pro : INTEGER is
    -- An ID for the flightstick pro joystick
    once
        c_inline_c("R=JOY_TYPE_FSPRO;")
    end  -- feature flightstick_pro


    four_button : INTEGER is
    -- An ID for the four button joystick
    once
        c_inline_c("R=JOY_TYPE_4BUTTON;")
    end  -- feature four_button


------------------------------------------------------ Joystick features
feature { ANY }

    type : INTEGER is
    -- What type is the joystick?
    require
        ace_initialised : info.ace_initialised
    do
        c_inline_c("R=joy_type;")
    ensure
        Result_is_valid : Result = standard or
                          Result = flightstick_pro or
                          Result = four_button
    end  -- feature type


    set_type(joystick_type : INTEGER) is
    -- Set the type of the joystick
    require
        ace_initialised        : info.ace_initialised
        joystick_not_installed : not info.joystick_installed
        joystick_type_is_valid : joystick_type = standard or
                                 joystick_type = flightstick_pro or
                                 joystick_type = four_button
    do
        c_inline_c("joy_type=a1;")
    ensure
        joystick_type_is_valid : type = joystick_type
    end  -- feature set_type


    installed : BOOLEAN is
    -- Is a joystick installed?
    require
        ace_initialised        : info.ace_initialised
        joystick_not_installed : not info.joystick_installed
    once
        c_inline_c("R=(initialise_joystick()==0?-1:0);")
        info.set_joystick_flag(Result)
    end  -- feature installed


    calibrated : BOOLEAN is
    -- Has the joystick been calibrated?
    require
        ace_initialised     : info.ace_initialised
        joystick_installed  : installed
    do
        Result := centre_calibrated   and
                  top_left_calibrated and
                  bottom_right_calibrated
    end  -- feature calibrated


    throttle_calibrated : BOOLEAN is
    -- Has the Flightstick Pro throttle been calibrated?
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
        type_is_valid      : type = flightstick_pro
    do
        Result := throttle_min_calibrated and throttle_max_calibrated
    end  -- feature calibrated


    calibrate_centre is
    -- Calibrate the centre of the joystick
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
    do
        c_inline_c("initialise_joystick();")
        centre_calibrated := TRUE
    ensure
        centre_calibrated : centre_calibrated
    end  -- feature calibrate_centre


    calibrate_top_left is
    -- Calibrate the top left of the joystick
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
    do
        c_inline_c("calibrate_joystick_tl();")
        top_left_calibrated := TRUE
    ensure
        top_left_calibrated : top_left_calibrated
    end  -- feature calibrate_top_left


    calibrate_bottom_right is
    -- Calibrate the bottom right of the joystick
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
    do
        c_inline_c("calibrate_joystick_br();")
        bottom_right_calibrated := TRUE
    ensure
        bottom_right_calibrated : bottom_right_calibrated
    end  -- feature calibrate_bottom_right


    calibrate_throttle_min is
    -- Calibrate the minimum position of the Flightstick Pro throttle
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
        type_is_valid      : type = flightstick_pro
    do
        c_inline_c("calibrate_joystick_throttle_min();")
        throttle_min_calibrated := TRUE
    ensure
        throttle_min_calibrated : throttle_min_calibrated
    end  -- feature calibrate_throttle_min


    calibrate_throttle_max is
    -- Calibrate the maximum position of the Flightstick Pro throttle
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
        type_is_valid      : type = flightstick_pro
    do
        c_inline_c("calibrate_joystick_throttle_max();")
        throttle_max_calibrated := TRUE
    ensure
        throttle_max_calibrated : throttle_max_calibrated
    end  -- feature calibrate_throttle_max


    save_calibration(filename : STRING) : BOOLEAN is
    -- Save the current calibration settings to the specified file
    require
        ace_initialised     : info.ace_initialised
        joystick_installed  : installed
        joystick_calibrated : calibrated and
                              (type = standard or throttle_calibrated)
        filename_is_valid   : filename /= Void
    local
        p : POINTER
    do
        p := filename.to_external
        c_inline_c("R=(save_joystick_data(_p)==0?-1:0);")
    end  -- feature save_calibration


    load_calibration(filename : STRING) : BOOLEAN is
    -- Load the current calibration settings from the specified file.
    -- Note: this initialises the joystick, even if it hasn't already
    --       been initialised.
    require
        ace_initialised   : info.ace_initialised
        filename_is_valid : filename /= Void
    local
        p : POINTER
    do
        if installed then
            p := filename.to_external
            c_inline_c("R=(load_joystick_data(_p)==0?-1:0);")

            if Result then
                centre_calibrated       := TRUE
                top_left_calibrated     := TRUE
                bottom_right_calibrated := TRUE

                if type = flightstick_pro then
                    throttle_min_calibrated := TRUE
                    throttle_max_calibrated := TRUE
                end  -- if
            end  -- if
        end  -- if
    end  -- feature load_calibration


    poll is
    -- Poll the joystick to update position and button states
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
    do
        c_inline_c("poll_joystick();")
    end  -- feature poll


    left : BOOLEAN is
    -- Is the joystick pointing left?
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
    do
        c_inline_c("R=joy_left;")
    end  -- feature left


    up : BOOLEAN is
    -- Is the joystick pointing up?
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
    do
        c_inline_c("R=joy_up;")
    end  -- feature up


    right : BOOLEAN is
    -- Is the joystick pointing right?
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
    do
        c_inline_c("R=joy_right;")
    end  -- feature right


    down : BOOLEAN is
    -- Is the joystick pointing down?
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
    do
        c_inline_c("R=joy_down;")
    end  -- feature down


    hat_left : BOOLEAN is
    -- Is the hat on the FlightStick Pro pointing left?
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
        type_is_valid      : type = flightstick_pro
    do
        c_inline_c("R=(joy_hat==JOY_HAT_LEFT?-1:0);")
    end  -- feature hat_left


    hat_up : BOOLEAN is
    -- Is the hat on the FlightStick Pro pointing up?
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
        type_is_valid      : type = flightstick_pro
    do
        c_inline_c("R=(joy_hat==JOY_HAT_UP?-1:0);")
    end  -- feature hat_up


    hat_right : BOOLEAN is
    -- Is the hat on the FlightStick Pro pointing right?
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
        type_is_valid      : type = flightstick_pro
    do
        c_inline_c("R=(joy_hat==JOY_HAT_RIGHT?-1:0);")
    end  -- feature hat_right


    hat_down : BOOLEAN is
    -- Is the hat on the FlightStick Pro pointing down?
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
        type_is_valid      : type = flightstick_pro
    do
        c_inline_c("R=(joy_hat==JOY_HAT_DOWN?-1:0);")
    end  -- feature hat_down


    hat_centre : BOOLEAN is
    -- Is the hat on the FlightStick Pro centred?
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
        type_is_valid      : type = flightstick_pro
    do
        c_inline_c("R=(joy_hat==JOY_HAT_CENTRE?-1:0);")
    end  -- feature hat_centre


    button1_pressed : BOOLEAN is
    -- Is joystick button 1 pressed?
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
    do
        c_inline_c("R=joy_b1;")
    end  -- feature button1_pressed


    button2_pressed : BOOLEAN is
    -- Is joystick button 2 pressed?
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
    do
        c_inline_c("R=joy_b2;")
    end  -- feature button2_pressed


    button3_pressed : BOOLEAN is
    -- Is joystick button 3 pressed?
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
        type_is_valid      : type = flightstick_pro or
                             type = four_button
    do
        c_inline_c("R=joy_b3;")
    end  -- feature button3_pressed


    button4_pressed : BOOLEAN is
    -- Is joystick button 4 pressed?
    require
        ace_initialised    : info.ace_initialised
        joystick_installed : installed
        type_is_valid      : type = flightstick_pro or
                             type = four_button
    do
        c_inline_c("R=joy_b4;")
    end  -- feature button4_pressed


    x_position : INTEGER is
    -- What is the joystick's x position?
    require
        ace_initialised     : info.ace_initialised
        joystick_installed  : installed
        joystick_calibrated : calibrated
    do
        c_inline_c("R=joy_x;")
    ensure
        Result_is_valid : Result >= -128 and Result <= 128
    end  -- feature x_position


    y_position : INTEGER is
    -- What is the joystick's y position?
    require
        ace_initialised     : info.ace_initialised
        joystick_installed  : installed
        joystick_calibrated : calibrated
    do
        c_inline_c("R=joy_y;")
    ensure
        Result_is_valid : Result >= -128 and Result <= 128
    end  -- feature y_position


    throttle_position : INTEGER is
    -- What is the joystick's y position?
    require
        ace_initialised     : info.ace_initialised
        joystick_installed  : installed
        type_is_valid       : type = flightstick_pro
        joystick_throttle_calibrated : throttle_calibrated
    do
        c_inline_c("R=joy_throttle;")
    ensure
        Result_is_valid : Result >= 0 and Result <= 255
    end  -- feature y


------------------------------------------------------ Internal features
feature { NONE }

    -- Has the joystick centre been calibrated?
    centre_calibrated : BOOLEAN

    -- Has the joystick top right been calibrated?
    top_left_calibrated : BOOLEAN


    -- Has the joystick bottom left been calibrated?
    bottom_right_calibrated : BOOLEAN


    -- Has the Flightstick Pro throttle minimum been calibrated?
    throttle_min_calibrated : BOOLEAN


    -- Has the Flightstick Pro throttle maximum been calibrated?
    throttle_max_calibrated : BOOLEAN


------------------------------------------------------ Class invariant
invariant

    type_is_valid : type = standard or
                    type = flightstick_pro or
                    type = four_button


end  -- class JOYSTICK
