/* Module:          SQLExecute.c
 *
 * Description:     Executes a prepared statement, using the current values 
 *					of the parameter marker variables if any parameter markers 
 *					exist in the statement.
 *
 * Classes:         
 *
 * API functions:   SQLExecute
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

SQLRETURN SQL_API SQLExecute(SQLHSTMT	hDrvStmt)
{
	static char *func="SQLExecute";
	StatementClass *stmt = (StatementClass *) hDrvStmt;
	ConnectionClass *conn;
	int i, retval;


	mylog("%s: entering...\n", func);

	if ( ! stmt)
	{
		SC_log_error(func, "", NULL);
		mylog("%s: NULL statement so return SQL_INVALID_HANDLE\n", func);
		return SQL_INVALID_HANDLE;
	}

	/*  If the statement is premature, it means we already executed
		it from an SQLPrepare/SQLDescribeCol type of scenario.  So
		just return success.
	*/
	if ( stmt->prepare && stmt->status == STMT_PREMATURE)
	{
		stmt->status = STMT_FINISHED;       
		if (stmt->errormsg == NULL)
		{
			mylog("%s: premature statement but return SQL_SUCCESS\n", func);
			return SQL_SUCCESS;
		}
		else
		{
			SC_log_error(func, "", stmt);
			mylog("%s: premature statement so return SQL_ERROR\n", func);
			return SQL_ERROR;
		}
	}  

	mylog("%s: clear errors...\n", func);

	SC_clear_error(stmt);

	conn = SC_get_conn(stmt);
	if (conn->status == CONN_EXECUTING)
	{
		stmt->errormsg = "Connection is already in use.";
		stmt->errornumber = STMT_SEQUENCE_ERROR;
		SC_log_error(func, "", stmt);
		mylog("%s: problem with connection\n", func);
		return SQL_ERROR;
	}

	if ( ! stmt->statement)
	{
		stmt->errornumber = STMT_NO_STMTSTRING;
		stmt->errormsg = "This handle does not have a SQL statement stored in it";
		SC_log_error(func, "", stmt);
		mylog("%s: problem with handle\n", func);
		return SQL_ERROR;
	}

	/*	If SQLExecute is being called again, recycle the statement.
		Note this should have been done by the application in a call
		to SQLFreeStmt(SQL_CLOSE) or SQLCancel.
	*/
	if (stmt->status == STMT_FINISHED)
	{
		mylog("%s: recycling statement (should have been done by app)...\n", func);
		SC_recycle_statement(stmt);
	}

	/*	Check if the statement is in the correct state */
	if ((stmt->prepare && stmt->status != STMT_READY) || 
		(stmt->status != STMT_ALLOCATED && stmt->status != STMT_READY))
	{
		stmt->errornumber = STMT_STATUS_ERROR;
		stmt->errormsg = "The handle does not point to a statement that is ready to be executed";
		SC_log_error(func, "", stmt);
		mylog("%s: problem with statement\n", func);
		return SQL_ERROR;
	}


	/*	The bound parameters could have possibly changed since the last execute
		of this statement?  Therefore check for params and re-copy.
	*/
	stmt->data_at_exec = -1;
	for (i = 0; i < stmt->parameters_allocated; i++)
	{
		/*	Check for data at execution parameters */
		if ( stmt->parameters[i].data_at_exec == TRUE)
		{
			if (stmt->data_at_exec < 0)
			{
				stmt->data_at_exec = 1;
			}
			else
			{
				stmt->data_at_exec++;
			}
		}
	}
	/*	If there are some data at execution parameters, return need data */
	/*	SQLParamData and SQLPutData will be used to send params and execute the statement. */
	if (stmt->data_at_exec > 0)
	{
		return SQL_NEED_DATA;
	}


	mylog("%s: copying statement params: trans_status=%d, len=%d, stmt='%s'\n", func, conn->transact_status, strlen(stmt->statement), stmt->statement);

	/*	Create the statement with parameters substituted. */
	retval = copy_statement_with_parameters(stmt);
	if( retval != SQL_SUCCESS)
	{
		/* error msg passed from above */
		return retval;
	}

	mylog("   stmt_with_params = '%s'\n", stmt->stmt_with_params);
	
	return SC_execute(stmt);

}


