/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#ifndef DBTABLEMANAGER_H
#define DBTABLEMANAGER_H

#include <string>
#include "../../exceptions/sqlexception.h"
#include "dbtableset.h"
#include "../dbconnection.h"
#include "dbtableschema.h"
#include "../dbbasemanager.h"

/**
 * This class is used to manage operations concerning database tables. The class
 * provides many generic functions that can be used to retrieve/create/modify
 * database tables.
 */
class DBTableManager : public DBBaseManager
{
					
public:

	/**
 	 * Constructor
   */		
  DBTableManager();

	/**
 	 * Constructor
 	 * It is assumed that the database connection object will remain alive during
 	 * the life of this object. Be very careful to ensure that no methods are called
 	 * on this object if the connection object no longer exists. If the connection
 	 * object has already been destroyed then unpredictable results will be returned. 	
   */		
  DBTableManager(DBConnection *poDBConn);
	
	/**
 	 * Destructor
   */		
	~DBTableManager();

	/**
	 * Used to create a new table.
	 * @param			roTableSchema		an object representing the new table to be created
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void createTableSchema(const DBTableSchema & roTableSchema)
				throw (SQLException, DBConnectionException);

	/**
	 * Used to drop a table/view.
	 * @param			rstrTableName		the name of the table/view to drop	
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void dropTableSchema(const string & rstrTableName)
				throw (SQLException, DBConnectionException);

	/**
	 * Used to modify the table comment.
	 * @param			rstrTableName		the name of the table
	 * @param			rstrComment			the comment to set for the table
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void modifyTableComment(const string & rstrTableName, const string & rstrComment)
				throw (SQLException, DBConnectionException);
								
	/**
	 * Used to add a table column.
	 * Currently constraints are not supported. Column defaults are supported.
	 * @param			rstrTableName		the table the column belong to
	 * @param			roTableColNew		the new column to add to the table
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void addTableColumn(const string & rstrTableName, const DBTableColumn & roTableColNew)
				throw (SQLException, DBConnectionException);

	/**
	 * Used to alter the table name.
	 * @param			rstrNewTableName		the new table name
	 * @param			rstrOldTableName		the old table name
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void alterTableName(const string & rstrNewTableName, const string & rstrOldTableName)
				throw (SQLException, DBConnectionException);
												
	/**
	 * Used to alter a table column.
	 * Currently the features supported by alter:
	 * - Change column name
	 * - Change the default of the column
	 * - Change the column comment
	 * @param			rstrTableName		the table the columns belong to
	 * @param			roTableColNew		the new version of the column
	 * @param			roTableColOld		the old version of the column
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void alterTableColumn(const string & rstrTableName, const DBTableColumn & roTableColNew,
				const DBTableColumn & roTableColOld)
				throw (SQLException, DBConnectionException);

	/**
	 * Used to modify the column comment.
	 * @param			rstrTableName		the name of the table
	 * @param			rstrColumnName	the name of the column	
	 * @param			rstrComment			the comment to set for the table
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void modifyColumnComment(const string & rstrTableName, const string & rstrColumnName,
				const string & rstrComment)
				throw (SQLException, DBConnectionException);
												
	/**
	 * Used to retrieve the a table schema.
	 * @param			rstrTableName
	 * @param			roTableSchema
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void retrieveTableSchema(const string & rstrTableName, DBTableSchema & roTableSchema)
				throw (SQLException, DBConnectionException);
				
	/**
	 * Used to retrieve the list of tables for this connection.
	 * @param	a DBTableSet object that contains the table details
	 * @param	the string parameter that is passed to the like clause
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void retrieveListOfTables(DBTableSet & roDBTableSet, const string & rstrLikeClause = "") throw (SQLException, DBConnectionException);
		
};

#endif

