-- ------------------------------------------------
-- Model   :   Top - Level System Component 
--
-- Author  :   Michael Mayer,
--             Department of Electrical Engineering
--             University of Missouri - Rolla
--
-- Date    :   October 10, 1997
--
-- Limitations :
-- ------------------------------------------------
 
LIBRARY ieee;
USE ieee.std_logic_1164.ALL;
LIBRARY mc8051;
USE mc8051.ALL;

ENTITY system IS
--    GENERIC (
--    );
--
--    PORT (
--    );
END ENTITY system;

ARCHITECTURE struct OF system IS

    COMPONENT mc8051 IS
--       GENERIC (
--          program_filename : string
--       );
       PORT (
          P0      : INOUT std_logic_vector(7 DOWNTO 0);  -- used for data i/o 
          P1      : INOUT std_logic_vector(7 DOWNTO 0);  -- low-order address byte
          P2      : INOUT std_logic_vector(7 DOWNTO 0);  -- high-order address byte
          P3      : INOUT std_logic_vector(7 DOWNTO 0);  -- high-order address byte
--          rxd     : INOUT std_logic;  --port 3.0, serial port receiver data
--          txd     : INOUT std_logic;  --port 3.1, serial port transmitter
--          int0_n  : INOUT std_logic;  --port 3.2, interrupt 0 input
--          int1_n  : INOUT std_logic;  --port 3.3, interrupt 1 input
--          t0      : INOUT std_logic;  --port 3.4, input to counter 0
--          t1      : INOUT std_logic;  --port 3.5, input to counter 1
--          wr_n    : INOUT std_logic;  --port 3.6, write control, latches port 0 to external
--          rd_n    : INOUT std_logic;  --port 3.7, read control, enables external to port 0
    
          rst     : IN    std_logic;  -- low to high causes reset, 
          xtal1   : IN    std_logic;  -- clock input 1.2 to 12 MHz 
          xtal2   : OUT   std_logic;  -- output from oscillator (for crystal) -- IGNORED!
          ale     : OUT   std_logic;  -- provides Address Latch Enable output,
                  -- also receives the program pulse input during programming
          psen_n  : OUT   std_logic;  -- program store enable
          ea_n    : IN    std_logic   -- when low, access external prog ram. 
       );
    END COMPONENT mc8051;
    
    COMPONENT nvsram IS
        GENERIC (
            download_filename  : STRING;  -- filename for RAM initial contents
            -- File must be Hex format
            -- Configuring RAM size
            size:      INTEGER :=  65536;     -- number of memory words
            addr_width: INTEGER :=  16;    -- number of address bits
            width:     INTEGER :=  8;     -- number of bits per memory word
     
            t_rc      : TIME := 70 ns;    -- read cycle time
            t_oe      : TIME := 35 ns;    -- oe to output valid
            t_ce      : TIME := 70 ns;    -- ce to output valid
            t_wc      : TIME := 70 ns;    -- write cycle time
            t_ds      : TIME := 30 ns;    -- data setup time
            t_dh1     : TIME := 7 ns      -- data hold time
        );
     
        PORT (
            addr      : IN    std_logic_vector(addr_width-1 DOWNTO 0); -- address
            data      : INOUT std_logic_vector(width-1 DOWNTO 0);  -- data in/out
            ce_N      : IN    std_logic;                     -- chip enable
            we_N      : IN    std_logic;                     -- write enable
            oe_N      : IN    std_logic                      -- output enable
        );
    END COMPONENT nvsram;
 

    COMPONENT latch373 IS
        PORT (
            D    : IN  std_logic_vector(7 DOWNTO 0);  -- used for data input
            Q    : OUT std_logic_vector(7 DOWNTO 0);  -- used for data input
            LE   : IN  std_logic;   -- latch enable
            OE_n : IN  std_logic   -- output enable (active low)
         );
    END COMPONENT latch373;

    FOR ALL : mc8051 USE ENTITY mc8051.mc8051;
    --FOR ALL : mc8051 USE ENTITY work.mc8051;

    SIGNAL wr_n, rd_n, mem_out     : std_logic;  -- write, read, output enable 
    SIGNAL p0               : std_logic_vector(7 DOWNTO 0);
    SIGNAL clock            : std_logic;
    SIGNAL psen_n, ale      : std_logic;
    SIGNAL pmem_en_N        : std_logic;
    SIGNAL addr             : std_logic_vector(15 DOWNTO 0);
    SIGNAL latch_oe_N       : std_logic;
    SIGNAL rxd, txd, int0_n, int1_n, t0, t1 : std_logic;
    SIGNAL reset            : std_logic;

BEGIN
   reset <= '1', '0' AFTER 10 us;

   clockgen : PROCESS IS 
   BEGIN
       -- 8 MHz clock, or 125 ns period
       clock <= '0';
       WAIT FOR 62 ns;
       clock <= '1';
       WAIT FOR 63 ns;
   END PROCESS clockgen;

   processor : mc8051
--       GENERIC MAP (
--           program_filename => "code.hex"
--       )
       PORT MAP (
           P0     => p0,
           P2     => addr(15 DOWNTO 8),
           rst    => reset, 
           xtal1  => clock,
           ea_n   => '0',
           P3(0)  => rxd,
           P3(1)  => txd,
           P3(2)  => int0_n,
           P3(3)  => int1_n,
           P3(4)  => t0,
           P3(5)  => t1,
           P3(6)  => wr_n,
           P3(7)  => rd_n,
           ale    => ale,
           psen_n => psen_n
    );

  pmem_en_N <= '0';

  mem_out <= '0' WHEN (psen_N = '0') OR (rd_N = '0') ELSE '1';

  external_pmem : nvsram
     GENERIC MAP (
        download_filename => "sram.hex"
     )
     PORT MAP (
        addr    => addr,
        data    => p0,
        ce_N  => pmem_en_N,
        oe_N  => mem_out,
        we_N  => '1'
      );

  low_addr_latch : latch373
      PORT MAP(
        D    => p0,
        Q    => addr(7 DOWNTO 0),
        LE   => ale,
        OE_n => '0' 
     );

END ARCHITECTURE struct;

