-- Sayuri GNU Public Licence 32 bit CPU design/implementation
-- Sayuri 32 bit RISC
-- Copyright (C) 2000 Toyoaki Sagawa.
-- This program/design/implementation is a free software/hardware design 
-- (and all intellectual property therein); you can redistribute it and/or
-- modify it under the terms of the GNU General Public License
-- as published by the Free Software Foundation; either version 2
-- of the License, or (at your option) any later version.
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
-- GNU General Public License for more details.
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
-- USA.
-- Toyoaki Sagawa,
-- PXW07530@nifty.ne.jp
-- http://www.morphyplanning.co.jp/FreeCPU/freecpu-e.html
-- Postal 113-0034 Yusima, 4-11-16-808 Bunkyo, Tokyo, Japan
-- 
-- Sayuri Data BUS interface module
-- 2000/08/14 Rev.1.0 Toyozou
-- 
-- This is memory access module of Sayuri.
-- keep in mind that, Sayuri doesn't have "address" space.
-- "address to Cash/SDRAM" is just a value in DP register.

library IEEE;
use IEEE.std_LOGIC_1164.all;
use IEEE.std_LOGIC_unsigned.all;
use IEEE.std_LOGIC_arith.all;

entity DBUS is
    port (
	DIN: in STD_LOGIC_VECTOR (31 downto 0);
	CLK: in STD_LOGIC;
	WE: in STD_LOGIC_VECTOR (1 downto 0);
	OE: in STD_LOGIC_VECTOR (1 downto 0);
 	MEMDIN : in STD_LOGIC_VECTOR(31 downto 0);
		-- input data from memory
 	RDY : out STD_LOGIC; 
		-- when 1, Core must wait for Memory I/O
	DOUT: out STD_LOGIC_VECTOR (31 downto 0);
		-- output data to CPU_CORE
 	MEMA: out STD_LOGIC_VECTOR(31 downto 0);
		-- memory address bus
 	MEMDOUT : out STD_LOGIC_VECTOR(31 downto 0);
		-- output data to memory
 	MEMWE : out STD_LOGIC;
		-- memory write signal
 	MEMOE : out STD_LOGIC
		-- memory read signal
    );
end DBUS;

architecture DBUS_arch of DBUS is
 	type STATE is (IDLE, DDREAD, READY); -- needs 3 state for memory read
 	signal CURRENT_STATE : STATE;
	signal DPREG: std_LOGIC_vector(31 downto 0); -- memory address register
	signal DDREG: std_LOGIC_vector(31 downto 0); -- memory data register
	signal inWE: STD_LOGIC_VECTOR (1 downto 0); -- keep action from CPU
	signal inOE: STD_LOGIC_VECTOR (1 downto 0); -- keep action from CPU
begin
  process(OE,DDREG,DPREG,CURRENT_STATE) begin
	if (OE = "01") then -- output from address register
		RDY <= '0';
		DOUT <= DPREG;
	elsif (OE = "10" and CURRENT_STATE = READY) then -- output from data register
		RDY <= '0';
		DOUT <= DDREG;
	else
		RDY <= '1';
		DOUT <= (others =>'0');
	end if;		
  end process;
  process(CLK) begin
	if(CLK'event and CLK='1') then
		inWE <= WE;
		inOE <= OE;
		case WE is
		when "01" =>  -- write to address register
			DPREG <= DIN;
			DDREG <= DDREG;
			CURRENT_STATE <= IDLE;
		when "10" =>  -- write to data register
			DPREG <= DPREG;
			DDREG <= DIN;
			CURRENT_STATE <= IDLE;
		when others =>
			DDREG <= DDREG;
			DPREG <= DPREG;
			CURRENT_STATE <= CURRENT_STATE;
		end case;
		if(OE = "10" and CURRENT_STATE = IDLE)then -- start read access to memory
			DDREG <= DDREG;
			CURRENT_STATE <= DDREAD;
		end if;
		if(CURRENT_STATE = DDREAD) then
			DDREG <= MEMDIN;
			CURRENT_STATE <= READY;
		end if;
		if(CURRENT_STATE = READY) then
			DDREG <= DDREG;
			CURRENT_STATE <= IDLE;
		end if;
	else
		inWE <= inWE;
		inOE <= inOE;
 	end if;
  end process;
  process(inWE, DDREG, DPREG, CURRENT_STATE) begin
	if(inWE = "10")then  -- memory write action
		MEMA <= DPREG;
		MEMDOUT <= DDREG;
		MEMWE <= '0';
		MEMOE <= '1';
	else
		MEMA <= (others =>'0');
		MEMDOUT <= (others => '0');
		MEMWE <= '1';
		MEMOE <= '1';
	end if;

	if (CURRENT_STATE = DDREAD) then -- memory read action
		MEMA <= DPREG;
		MEMDOUT <=(others => '0');
		MEMWE <= '1';
		MEMOE <= '0';
	end if;
	if (CURRENT_STATE = READY) then
		MEMA <= (others =>'0');
		MEMDOUT <= (others => '0');
		MEMWE <= '1';
		MEMOE <= '1';
	end if;
  end process;

end DBUS_arch;


