-- Sayuri GNU Public Licence 32 bit CPU design/implementation
-- Sayuri 32 bit RISC
-- Copyright (C) 2000 Toyoaki Sagawa.
-- This program/design/implementation is a free software/hardware design 
-- (and all intellectual property therein); you can redistribute it and/or
-- modify it under the terms of the GNU General Public License
-- as published by the Free Software Foundation; either version 2
-- of the License, or (at your option) any later version.
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
-- GNU General Public License for more details.
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
-- USA.
-- Toyoaki Sagawa,
-- PXW07530@nifty.ne.jp
-- http://www.morphyplanning.co.jp/FreeCPU/freecpu-e.html
-- Postal 113-0034 Yusima, 4-11-16-808 Bunkyo, Tokyo, Japan

-- 
-- Sayuri CPU module
-- 2000/08/16 Rev.1.0 Toyozou
-- 8/21 Change 3 state to OR gated
-- 9/7  change ECR out of core
--
library IEEE;
use IEEE.std_logic_1164.all;
-- use IEEE.std_logic_unsigned.all;
-- use IEEE.std_logic_arith.all;

entity CPU is  -- harvard architecture Three Instruction Set Computer core
    port (
	DMEMAO: out STD_LOGIC_VECTOR (31 downto 0);
	DMEMDIN: in STD_LOGIC_VECTOR (31 downto 0);
	DMEMDOUT: out STD_LOGIC_VECTOR (31 downto 0);
	IMEMAO: out STD_LOGIC_VECTOR (31 downto 0);
	IMEMDIN: in STD_LOGIC_VECTOR (31 downto 0);
	CLK, RESET: in STD_LOGIC;
	MEMWE : out STD_LOGIC;
	MEMOE : out STD_LOGIC;
	INTIN : in  STD_LOGIC;  -- interrupt, 1 enable
	MODEOUT : out STD_LOGIC; -- 1:kernel mode, 0:user mode
	WAITIN : in STD_LOGIC  -- 1:wait for access.
    );
end CPU;

architecture RTL of CPU is
component CORE -- instruction decode=execute
    port(
	STATIN: in STD_LOGIC;  -- input data from STATBIT
	INSTRUCTION: in STD_LOGIC_VECTOR(31 downto 0);  -- Instruction Mem.
	CORESTOP: in STD_LOGIC;
	DOUT: out STD_LOGIC_VECTOR(31 downto 0); -- output for Imm16
	WE: out STD_LOGIC_VECTOR(255 downto 0);
	OE: out STD_LOGIC_VECTOR(255 downto 0)
    );
end component;

component INTSTATE -- interrupt state machine
    port (
	INTERRUPT: in STD_LOGIC;		-- if 1, interrupt occured
	CLK: in STD_LOGIC;
	RESET: in STD_LOGIC;
	CORESTOP: out STD_LOGIC;		-- if 1, core & IP counter must stop
	MODEON: out STD_LOGIC;			-- if 1, mode register will change
	EPRWE: out STD_LOGIC;			-- 1st stage, WE for EPR register
	IPOE: out STD_LOGIC;			-- 1st stage, OE for IP
	IPWE: out STD_LOGIC;			-- 2nd stage, WE for IP
	EJROE: out STD_LOGIC			-- 2nd stage, OE for EJR
	);
end component;

component ModeReg -- kernel mode/interrupt mask register
    port (
	DIN: in STD_LOGIC_VECTOR(31 downto 0) ;
	DOUT: out STD_LOGIC_VECTOR(31 downto 0) ;
	MODEOUT: out STD_LOGIC;  -- kernel mode bit output for registers
	MASKOUT: out STD_LOGIC;  -- ALLMASK bit output for ECR register
	CLK: in STD_LOGIC;
	RESET: in STD_LOGIC;
	OE: in STD_LOGIC;
	WE: in STD_LOGIC;
	MODEON: in STD_LOGIC; -- interrupt routine in, from int state machine
	IPRETWE: in STD_LOGIC -- interrupt routine return, WE of IP+1 
    );
end component;

component StatReg -- move compare status here.
    port (
	DIN: in STD_LOGIC ;
	STATOUT: out STD_LOGIC;
	CLK: in STD_LOGIC;
	WE: in STD_LOGIC
    );
end component;

component REG -- generic register
    port(   CLK : in std_logic;
	WE : in std_logic_vector(31 downto 0);
	OE : in std_logic_vector(31 downto 0);
	DIN : in std_logic_vector(31 downto 0);
	DOUT : out std_logic_vector(31 downto 0)
    );
end component;

component ALU -- generic ALU unit
    port (
	DIN: in STD_LOGIC_VECTOR (31 downto 0);
	DOUT: out STD_LOGIC_VECTOR (31 downto 0);
	CLK: in STD_LOGIC;
	WE: in STD_LOGIC_VECTOR (7 downto 0);
	OE: in STD_LOGIC_VECTOR (1 downto 0)
    );
end component;

component CMP -- comparater unit
    port (
	DIN: in STD_LOGIC_VECTOR (31 downto 0);
	DOUT: out STD_LOGIC_VECTOR (31 downto 0);
	CLK: in STD_LOGIC;
	WE: in STD_LOGIC_VECTOR (7 downto 0);
	OE: in STD_LOGIC_VECTOR (1 downto 0)
    );
end component;

component INC_DEC -- increment/decrement unit
    port (
	DIN: in STD_LOGIC_VECTOR (31 downto 0);
	DOUT: out STD_LOGIC_VECTOR (31 downto 0);
	CLK: in STD_LOGIC;
	WE: in STD_LOGIC_VECTOR (2 downto 0);
	OE: in STD_LOGIC
    );
end component;

component LOG -- logical calucurator unit
    port (
	DIN: in STD_LOGIC_VECTOR (31 downto 0);
	DOUT: out STD_LOGIC_VECTOR (31 downto 0);
	CLK: in STD_LOGIC;
	WE: in STD_LOGIC_VECTOR (7 downto 0);
	OE: in STD_LOGIC_VECTOR (1 downto 0)
    );
end component;

component DBUS -- data bus interface
    port (
	DIN: in STD_LOGIC_VECTOR (31 downto 0);
	CLK: in STD_LOGIC;
	WE: in STD_LOGIC_VECTOR (1 downto 0);
	OE: in STD_LOGIC_VECTOR (1 downto 0);
	DOUT: out STD_LOGIC_VECTOR (31 downto 0);
	RDY: out STD_LOGIC; -- when 1,must wait

	MEMDIN : in STD_LOGIC_VECTOR(31 downto 0);
	MEMA: out STD_LOGIC_VECTOR(31 downto 0);
	MEMDOUT : out STD_LOGIC_VECTOR(31 downto 0);
	MEMWE : out STD_LOGIC;
	MEMOE : out STD_LOGIC
    );
end component;

component IP -- instruction supply unit
    port (
	DIN: in STD_LOGIC_VECTOR (31 downto 0);
	DOUT: out STD_LOGIC_VECTOR (31 downto 0);
	CLK: in STD_LOGIC;
	RESET: in STD_LOGIC; -- if 0, set IPREG 0
	COUNTEN: in STD_LOGIC; -- if 1, IP Count UP
	WE: in STD_LOGIC;
	OE: in STD_LOGIC;
	MEMA: out STD_LOGIC_VECTOR (31 downto 0)
    );
end component;

component EPREJR -- Exeption address control unit
    port (
	DIN: in STD_LOGIC_VECTOR (31 downto 0);  	-- produced address and jump address
	DOUT: out STD_LOGIC_VECTOR (31 downto 0);
	CLK: in STD_LOGIC;
	MODE: in STD_LOGIC; 						-- kernel mode/user mode, 1/0
	WE: in STD_LOGIC_VECTOR(1 downto 0);
	OE: in STD_LOGIC_VECTOR(1 downto 0)
    );
end component;


-- internal signals
signal WE : STD_LOGIC_VECTOR(255 downto 0);
 -- from core to any unit
signal OE : STD_LOGIC_VECTOR(255 downto 0);
 -- from core to any unit
signal D_BUS : STD_LOGIC_VECTOR(31 downto 0);
 -- data bus
signal DA : STD_LOGIC_VECTOR(31 downto 0);
signal DB : STD_LOGIC_VECTOR(31 downto 0);
signal DC : STD_LOGIC_VECTOR(31 downto 0);
signal DD : STD_LOGIC_VECTOR(31 downto 0);
signal DE : STD_LOGIC_VECTOR(31 downto 0);
signal DF : STD_LOGIC_VECTOR(31 downto 0);
signal DG : STD_LOGIC_VECTOR(31 downto 0);
signal DH : STD_LOGIC_VECTOR(31 downto 0);
signal DI : STD_LOGIC_VECTOR(31 downto 0);
signal DJ : STD_LOGIC_VECTOR(31 downto 0);

signal STATIN : STD_LOGIC;
 -- from statreg to core

signal DDRDY: STD_LOGIC;
signal CORESTOP : STD_LOGIC;
signal IPCOUNTEN : STD_LOGIC;
 -- from intstate to IP.
 -- IPCOUNTEN <= not(CORESTOP or WAITIN or DDRDY)

signal CPUMODE : STD_LOGIC;
 -- from modereg to EPREJR, cpu external

signal MODEON : STD_LOGIC;

signal EPRWE : STD_LOGIC;
signal EPRWEIN : STD_LOGIC;
 -- EPRWEIN <= EPRWE or WE(8).

signal IPOE : STD_LOGIC;
signal IPOEIN : STD_LOGIC;
 -- IPOEIN <= IPOE or OE(2).

signal IPWE : STD_LOGIC;
signal IPWEIN : STD_LOGIC;
 -- IPWEIN <= IPWE or WE(2) or WE(3)

signal EJROE : STD_LOGIC;
signal EJROEIN : STD_LOGIC;
 -- EJROEIN <= EJROE or OE(9)

signal INTMASK : STD_LOGIC;
signal INTERRUPT : STD_LOGIC;
 -- INTERRUPT <= (not INTMASK) and INTIN

begin

-- port maps
    U0 : CORE port map(
	-- input
	STATIN=>STATIN,INSTRUCTION=>IMEMDIN,CORESTOP=>CORESTOP,
	-- output
	DOUT=>DA , WE=>WE,OE=>OE );

    U1 : INTSTATE port map (
	-- input
	CLK =>CLK , RESET =>RESET , INTERRUPT =>INTERRUPT,
	-- output
	CORESTOP =>CORESTOP, MODEON=>MODEON, EPRWE =>EPRWE,IPOE =>IPOE,
        IPWE =>IPWE,  EJROE=>EJROE);

    U2 : ModeReg port map (
	-- input
	CLK =>CLK, RESET => RESET,
	DIN =>D_BUS , MODEON=>MODEON, IPRETWE => WE(3),
	WE => WE(0), OE => OE(0) ,
	-- output
	MODEOUT =>CPUMODE, MASKOUT =>INTMASK , DOUT => DB);

    U3 : StatReg port map(
	-- input
	CLK=>CLK ,DIN=>D_BUS(0),WE=>WE(1),
	-- output
	STATOUT=>STATIN);

    U4 : IP port map (
	-- input
	CLK=>CLK ,RESET=>RESET ,DIN=>D_BUS,WE=>IPWEIN ,OE=>IPOEIN,COUNTEN=>IPCOUNTEN,
	-- output
	DOUT=>DC,MEMA=>IMEMAO);

    U5 : DBUS port map (
	-- input
	CLK=>CLK ,DIN=>D_BUS ,WE=>WE(5 downto 4),OE=>OE(5 downto 4),
	-- output
 	DOUT=>DD ,RDY=>DDRDY, MEMDIN => DMEMDIN,MEMA=>DMEMAO ,MEMDOUT => DMEMDOUT,MEMWE => MEMWE,MEMOE => MEMOE);

    U6 : EPREJR port map (
	-- input 
	CLK=>CLK, DIN=>D_BUS ,WE(1) => WE(9), WE(0)=>EPRWEIN, OE(1)=>EJROEIN, OE(0)=>OE(8),MODE=>CPUMODE,
	-- output
	DOUT =>DE);

    U7 : ALU  port map (
	-- input
	CLK=>CLK ,DIN=>D_BUS ,WE=>WE(17 downto 10),OE=>OE(11 downto 10),
	-- output
	DOUT=>DF);

    U8 : CMP  port map (
	-- input
	CLK=>CLK ,DIN=>D_BUS ,WE=>WE(26 downto 19),OE=>OE(20 downto 19),
	-- output
	DOUT=>DG );

    U9 : LOG  port map (
	-- input 
	CLK=>CLK ,DIN=>D_BUS , WE=>WE(38 downto 31),OE=>OE(32 downto 31),
	-- output
	DOUT=>DH);

    U10 : INC_DEC port map(
	-- input 
	CLK=>CLK,DIN =>D_BUS ,WE=>WE(51 downto 49) , OE => OE(49),
	-- output
	 DOUT =>DI );
    U11 : REG  port map (
	-- input
	CLK=>CLK,DIN =>D_BUS ,WE =>WE(95 downto 64) ,OE =>OE(95 downto 64),
	-- output
	DOUT =>DJ );

-- equations

 MODEOUT <= CPUMODE; -- kernel mode output to CPU external
 D_BUS <= DA or DB or DC or DD or DE or DF or DG or DH or DI or DJ;
 EPRWEIN <= EPRWE or WE(8);
 IPOEIN <= IPOE or OE(2);
 IPWEIN <= IPWE or WE(2) or WE(3);
 EJROEIN <= EJROE or OE(9);
 IPCOUNTEN <= not(CORESTOP or WAITIN or (not DDRDY)) ;
 INTERRUPT <= (not INTMASK) and INTIN;

end RTL;



