-- Sayuri GNU Public Licence 32 bit CPU design/implementation
-- Sayuri 32 bit RISC
-- Copyright (C) 2000 Toyoaki Sagawa.
-- This program/design/implementation is a free software/hardware design 
-- (and all intellectual property therein); you can redistribute it and/or
-- modify it under the terms of the GNU General Public License
-- as published by the Free Software Foundation; either version 2
-- of the License, or (at your option) any later version.
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
-- GNU General Public License for more details.
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
-- USA.
-- Toyoaki Sagawa,
-- PXW07530@nifty.ne.jp
-- http://www.morphyplanning.co.jp/FreeCPU/freecpu-e.html
-- Postal 113-0034 Yusima, 4-11-16-808 Bunkyo, Tokyo, Japan

-- 
-- Sayuri ALU module
-- 2000/08/14 Rev.1.0 Toyozou
-- 
-- ALU has 2 register, "BASE" and "OUTPUT".
-- how to do "ADD" "SUB" is, ex:ADD R1+R2->R3
-- (1) MOV R1 to BASE register.
-- (2) MOV R2 to ADDu address.
-- (3) MOV OUTPUT to R3.
-- So Sayuri has only 3 opecode, but can do any instruction.
-- 

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;

entity ALU is
    port (
        DIN: in STD_LOGIC_VECTOR (31 downto 0);
        DOUT: out STD_LOGIC_VECTOR (31 downto 0);
        CLK: in STD_LOGIC;
        WE: in STD_LOGIC_VECTOR (7 downto 0);
        OE: in STD_LOGIC_VECTOR(1 downto 0)
    );
end ALU;

architecture ALU_arch of ALU is
	signal OUTREG: std_logic_vector(31 downto 0);
	signal REGA : std_logic_vector(31 downto 0);
begin

  process(OE, OUTREG, REGA) begin
	case OE is
	when "01" =>
		DOUT <= REGA;
	when "10" =>
		DOUT <= OUTREG;
	when others =>
		DOUT <= (others=>'0');
	end case;
  end process;

  process (CLK) begin
   if (CLK'event and CLK='1') then
	case WE is
		when "00000000" =>  REGA <= REGA; OUTREG <= OUTREG; -- No operation
		when "00000001" =>  REGA <= DIN; OUTREG <= OUTREG;
		when "00000010" =>  REGA <= REGA; OUTREG <= DIN;
	-- ADD unsigned
	when "00000100" =>  OUTREG <= REGA + DIN;
	-- ADD signed
	when "00001000" => 
		OUTREG <= CONV_STD_LOGIC_VECTOR((CONV_INTEGER(REGA) + CONV_INTEGER(DIN)), 32) ;
	-- SUB unsigned
	when "00010000" =>  OUTREG <= REGA - DIN;
	-- SUB signed
	when "00100000" => 
		OUTREG <= CONV_STD_LOGIC_VECTOR((CONV_INTEGER(REGA) - CONV_INTEGER(DIN)), 32) ;
	-- Shift Left
	when "01000000" =>  OUTREG <= SHL(REGA, DIN) ;
	-- Shift Right
	when "10000000" =>  OUTREG <= SHR(REGA, DIN) ;
	when others => OUTREG <= OUTREG;
	end case;	
    end if;
  end process;
  
end ALU_arch;


