/////////////////////////////////////////////////////////////////////
////                                                             ////
////  EQU                                                        ////
////  Floating Point Operator Equalization Unit                  ////
////                                                             ////
////  Author: Rudolf Usselmann                                   ////
////          russelmann@hotmail.com                             ////
////                                                             ////
/////////////////////////////////////////////////////////////////////
////                                                             ////
//// Copyright (C) 2000 Rudolf Usselmann                         ////
////                    russelmann@hotmail.com                   ////
////                                                             ////
//// This source file may be used and distributed without        ////
//// restriction provided that this copyright statement is not   ////
//// removed from the file and that any derivative work contains ////
//// the original copyright notice and the associated disclaimer.////
////                                                             ////
//// THIS SOURCE FILE IS PROVIDED "AS IS" AND WITHOUT ANY        ////
//// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT           ////
//// LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND   ////
//// FITNESS FOR A PARTICULAR PURPOSE.                           ////
////                                                             ////
/////////////////////////////////////////////////////////////////////


`timescale 1ns / 10ps


module equ(clk, add, opa, opb, fracta_out, fractb_out, exp_dn_out, sign, op_out);
input		clk;
input		add;
input	[31:0]	opa, opb;
output	[23:0]	fracta_out, fractb_out;
output	[7:0]	exp_dn_out;
output		sign;
output		op_out;

////////////////////////////////////////////////////////////////////////
//
// Local Wires and registers
//

wire		signa, signb;		// alias to opX sign
wire	[7:0]	expa, expb;		// alias to opX exponenet
wire	[22:0]	fracta, fractb;		// alias to opX fraction
wire		expa_lt_expb;		// expa is larger than expb indicator
wire		fractb_lt_fracta;	// fractb is larger than fracta indicator
reg	[7:0]	exp_dn_out;		// denormalized exponent output
wire	[7:0]	exp_small, exp_large;
wire	[7:0]	exp_diff;		// Numeric difference of the two exponents
wire	[23:0]	adj_op, adj_op_out;	// Fraction adjustment: input, output
wire	[23:0]	fracta_n, fractb_n;	// Fraction selection after normalising
wire	[23:0]	fracta_s, fractb_s;	// Fraction Sorting out
reg	[23:0]	fracta_out, fractb_out;	// Fraction Output
reg		sign, sign_d;		// Sign Output
reg		add_d;			// opration (add/sub)
reg		op_out;			// opration (add/sub) register

////////////////////////////////////////////////////////////////////////
//
// Aliases
//

assign  signa = opa[31];
assign  signb = opb[31];
assign   expa = opa[30:23];
assign   expb = opb[30:23];
assign fracta = opa[22:0];
assign fractb = opb[22:0];

////////////////////////////////////////////////////////////////////////
//
// Equalize exponents (and fractions)
//

assign expa_lt_expb = expa > expb;		// expa is larger than expb

// ---------------------------------------------------------------------
// Calculate the difference between the smaller and larger exponen

assign exp_small  = expa_lt_expb ? expb : expa;
assign exp_large  = expa_lt_expb ? expa : expb;
assign exp_diff   = exp_large - exp_small;

always @(posedge clk)
	exp_dn_out <= #1 exp_large;

// ---------------------------------------------------------------------
// Adjust the smaller fraction

assign adj_op[22:0] = expa_lt_expb ? fractb : fracta;
assign adj_op[23]   = 1'b1;			// recover hidden bit
assign adj_op_out   = adj_op >> exp_diff[7:0];	// FIX ME ??? (don't need all 7 bits ...)

// ---------------------------------------------------------------------
// Select operands for add/sub (recover hidden bit)

assign fracta_n = expa_lt_expb ? {1'b1, fracta} : adj_op_out;
assign fractb_n = expa_lt_expb ? adj_op_out : {1'b1, fractb};

// ---------------------------------------------------------------------
// Sort operands (for sub only)

assign fractb_lt_fracta = fractb_n > fracta_n;	// fractb is larger than fracta
assign fracta_s = fractb_lt_fracta ? fractb_n : fracta_n;
assign fractb_s = fractb_lt_fracta ? fracta_n : fractb_n;

always @(posedge clk)
	fracta_out <= #1 fracta_s;

always @(posedge clk)
	fractb_out <= #1 fractb_s;

// ---------------------------------------------------------------------
// Determine sign for the output

// sign: 0=Posetive Number; 1=Negative Number
always @(signa or signb or add or fractb_lt_fracta)
   case({signa, signb, add})		// synopsys full_case parallel_case

   	// Add
	3'b0_0_1: sign_d = 0;
	3'b0_1_1: sign_d = fractb_lt_fracta;
	3'b1_0_1: sign_d = !fractb_lt_fracta;
	3'b1_1_1: sign_d = 1;

	// Sub
	3'b0_0_0: sign_d = fractb_lt_fracta;
	3'b0_1_0: sign_d = 0;
	3'b1_0_0: sign_d = 1;
	3'b1_1_0: sign_d = !fractb_lt_fracta;
   endcase

always @(posedge clk)
	sign <= #1 sign_d;
	
////////////////////////////////////////////////////////////////////////
//
// Decode Add/Sub operation
//

// add: 1=Add; 0=Subtract
always @(signa or signb or add)
   case({signa, signb, add})		// synopsys full_case parallel_case
   
   	// Add
	3'b0_0_1: add_d = 1;
	3'b0_1_1: add_d = 0;
	3'b1_0_1: add_d = 0;
	3'b1_1_1: add_d = 1;
	
	// Sub
	3'b0_0_0: add_d = 0;
	3'b0_1_0: add_d = 1;
	3'b1_0_0: add_d = 1;
	3'b1_1_0: add_d = 0;
   endcase

always @(posedge clk)
	op_out <= #1 add_d;

endmodule
