/*****************************************************************************/
/*                                                                           */
/*                                 ENVIRON.CC                                */
/*                                                                           */
/* (C) 1995     Ullrich von Bassewitz                                        */
/*              Zwehrenbuehlstrasse 33                                       */
/*              D-72070 Tuebingen                                            */
/* EMail:       uz@ibb.schwaben.com                                          */
/*                                                                           */
/*****************************************************************************/



// $Id$
//
// $Log$
//
//



#include <stdlib.h>

#include "str.h"
#include "strparse.h"



/*****************************************************************************/
/*                                   Code                                    */
/*****************************************************************************/



String GetEnvVar (const char* Var)
// Retrieve an environment var. The result is empty if the string does not
// exist.
{
    return String (getenv (Var));
}



String GetEnvVar (const String& Var)
// Retrieve an environment var. The result is empty if the string does not
// exist.
{
    return String (getenv (Var.GetStr ()));
}



static String GetStr (String& ValStr, char C)
// Helper function for GetEnvVal. Extract from ValStr the string delimited
// by C. Delete this string (including C) from ValStr and return it.
// Call FAIL if ValStr is invalid
{
    // Get the position of the delimiter and make shure, it's there
    int DelimPos = ValStr.Pos (C);
    CHECK (DelimPos > 0);

    // Get the partial string
    String Val = ValStr.Cut (0, DelimPos);

    // Delete the partial string
    ValStr.Del (0, DelimPos + 1);

    // Return the extracted string
    return Val;
}



static i32 GetNumber (String& ValStr)
// Helper function for GetEnvVal. Extract the numeric value from ValStr and
// delete this number including the following '^'. Call FAIL if ValStr is
// invalid
{
    // Get the number
    String Val = GetStr (ValStr, '^');

    // Set up the string parser
    StringParser SP (Val);

    // Get the value
    i32 Value;
    CHECK (SP.GetI32 (Value) == 0);

    // return the value
    return Value;
}



i32 GetEnvVal (const char* Var, String ValStr)
// Read the environment variable Var and compare the value with values stored
// in ValStr. ValStr looks like this: "1^ON|0^OFF|1^YES|0^NO|", meaning:
// Return 1 if the value is "ON", return 0 if the value is "OFF" etc.
// Case is ignored when comparing the value. The first value stored in ValStr
// is the default, it is returned if Var does not exist or is invalid.
// The function calls FAIL if the given ValStr is invalid.
// Don't forget the trailing '|' !
{
    // Make ValStr uppercase
    ValStr.ToUpper ();

    // We have no default until now
    int NoDefault = 1;
    i32 Default = -1;

    // Get the environment value. If there is no value, we are already done
    String Val = GetEnvVar (Var);
    if (Val.IsEmpty ()) {
        return GetNumber (ValStr);
    }

    // Make the value uppercase
    Val.ToUpper ();

    // Repeat until ValStr is gone..
    do {

        // Get the next numeric value
        i32 NumVal = GetNumber (ValStr);

        // If we haven't had a default, this will change
        if (NoDefault) {
            Default = NumVal;
            NoDefault = 0;
        }

        // Compare the next value
        if (GetStr (ValStr, '|') == Val) {
            // Found the value
            return NumVal;
        }

    } while (!ValStr.IsEmpty ());

    // Not found
    return Default;
}



i32 GetEnvVal (const String& Var, const String& ValStr)
// Read the environment variable Var and compare the value with values stored
// in ValStr. ValStr looks like this: "1^ON|0^OFF|1^YES|0^NO|", meaning:
// Return 1 if the value is "ON", return 0 if the value is "OFF" etc.
// Case is ignored when comparing the value. The first value stored in ValStr
// is the default, it is returned if Var does not exist or is invalid.
// The function calls FAIL if the given ValStr is invalid. Don't forget the
// trailing '|' !
{
    // BC complains over a (nonexistant) ambiguity here, so do explicit
    // casting here...
    return GetEnvVal (Var.GetStr (), String (ValStr));
}



i32 GetEnvNum (const char* Var, i32 Default)
// Read the environment variable Var and treat the value as a number string.
// Return the converted number or Default if the variable does not exist or
// contains an invalid value.
{
    // Get the value
    String Val = GetEnvVar (Var);

    // Try to convert it to a number
    StringParser SP (Val);
    i32 Value;
    if (SP.GetI32 (Value) == 0) {
        // Done
        return Value;
    } else {
        // Error
        return Default;
    }
}



i32 GetEnvNum (const String& Var, i32 Default)
// Read the environment variable Var and treat the value as a number string.
// Return the converted number or Default if the variable does not exist or
// contains an invalid value.
{
    return GetEnvNum (Var.GetStr (), Default);
}



int GetEnvBool (const char* Var, int Default)
// Read the environment variable and treat it as a boolean value. Accepted
// values are "on" "off" "yes" "no" "1" "0". Default is returned if the
// variable does not exist or the value is invalid.
{
    if (Default) {
        return GetEnvVal (Var, String ("1^YES|0^NO|1^ON|0^OFF|1^1|0^0|"));
    } else {
        return GetEnvVal (Var, String ("0^NO|1^YES|0^OFF|1^ON|0^0|1^1|"));
    }
}



int GetEnvBool (const String& Var, int Default)
// Read the environment variable and treat it as a boolean value. Accepted
// values are "on" "off" "yes" "no" "1" "0". Default is returned if the
// variable does not exist or the value is invalid. Case of the value is
// ignored by this function.
{
    return GetEnvBool (Var.GetStr (), Default);
}



