/*****************************************************************************/
/*									     */
/*				   ICCOM.CC				     */
/*									     */
/* (C) 1995-96	Ullrich von Bassewitz					     */
/*		Zwehrenbuehlstrasse 33					     */
/*		D-72070 Tuebingen					     */
/* EMail:	uz@ibb.schwaben.com					     */
/*									     */
/*****************************************************************************/



// $Id$
//
// $Log$
//
//



#include "cpucvt.h"
#include "check.h"
#include "delay.h"
#include "sercom.h"

#include "icmsg.h"
#include "icconfig.h"
#include "icdlog.h"
#include "icdiag.h"
#include "iccom.h"



/*****************************************************************************/
/*				     Data				     */
/*****************************************************************************/



// Port address and irq used
unsigned PortBase = 0;
unsigned PortIRQ  = 0;


// Current ISTEC charges and a flag that is set to 1 on an update
IstecCharges Charges;
int ChargeUpdate	= 0;



// Com port instance
static ComPort* Port = NULL;



// Variables for the read routine
static enum {
    stIdle,			// No current message
    stInBlock,			// Currently reading a block
    stGotBlock,			// Got a complete block
    stInLastBlock,		// Currently reading the last block of a msg
    stFillBytes			// Reading the fill bytes of the last block
} ICReadStat = stIdle;

// Buffers and counters
static unsigned ICBlockCount = 0;
static unsigned ICReadCount = 0;
static unsigned ICFillCount = 0;
static unsigned ICMsgCount = 0;
static unsigned char ICReadBuf [512];

// Type of an istec message
struct IstecMsg {
    unsigned Size;
    unsigned char* Data;

    ~IstecMsg ();
    // Free the data area

    String AsciiData ();
    // Return the message converted to am string

    int IsDiagMsg ();
    // Return true if the message is a diagnostic message

    int IsChargeInfo ();
    // Return true if the message is a charge info message

};

// The last non diag message received
static IstecMsg* LastIstecMsg = NULL;

// Flag for the send routine to disable receive polling (this may result in
// receiving diag messages, calling the diag message handler, sending other
// requests while the first is not done)
static int SendInProgress = 0;



/*****************************************************************************/
/*				 Utility stuff				     */
/*****************************************************************************/



String AsciiData (unsigned char* Data, unsigned Size)
// Create an ascii string with the hex representation of the message data
{
    // Check the programmer :-)
    PRECONDITION (Data != NULL && Size > 0);

    String S (Size * 3);

    // Create the string
    for (unsigned I = 0; I < Size; I++) {
	S += FormatStr ("%02X ", Data [I]);
    }

    // Return the result
    return S;
}



static int IsDiagMsg (unsigned char* Data, unsigned Size)
// Return true if the given message is a diagnostic message
{
    PRECONDITION (Data != NULL);
    return Size == 6 && Data [0] == 0xdd;
}



static int IsChargeInfo (unsigned char* Data, unsigned Size)
// Return true if the message contains the ISTEC charges
{
    return Size == 129 && Data [0] == 0x15;
}



static unsigned SendMsgSize (const unsigned char* Data, unsigned BufSize)
// The function returns the real size of the given message to send. Since
// the messages given to this function are created by the programmer (that
// is me - hi!), a non determinable message size is a programming error.
// Note: BufSize is the size of the message buffer, the real message that
// is sent maybe smaller than this value.
{
    unsigned Version;

    // Check the given parameter
    PRECONDITION (Data != NULL && BufSize > 0);

    switch (Data [0]) {

	case 0x02:
	case 0x06:
	case 0x08:
	case 0x0a:
	case 0x0c:
	    return 1;

	case 0x05:
	    return 129;

	case 0x07:
	    return (FirmwareVersion < 1.93) ? 18 : 22;

	case 0x09:
	    // This is ugly, but I cannot help: Use the firmware version
	    // from the message instead of the saved one, since this we may
	    // not know about the version when receiving this message.
	    // To be shure not to access non-existing message members,
	    // check the length first.
	    CHECK (BufSize >= 94);
	    Version = unsigned (Data [4]) * 100 + unsigned (Data [5]);
	    return (Version < 193) ? 94 : 107;

	case 0xdd:
	    return 6;

	default:
	    FAIL ("SendMsgSize: Trying to create invalid message!");
	    return 0;
    }
}



static unsigned RecMsgSize (unsigned char* Data, unsigned Size)
// Calculate the size of a message. First byte is opcode.
{
    // Check the given parameter
    PRECONDITION (Data != NULL && Size > 0);

    unsigned S;
    switch (Data [0]) {

	case 0x11:
	case 0x12:
	    S = 1;
	    break;

	case 0x13:
	    S = 2;
	    WriteDebugLog ("Got error reply: " + AsciiData (Data, Size));
	    break;

	case 0x15:
	    S = 129;
	    break;

	case 0x16:
	    S = (FirmwareVersion < 1.93) ? 18 : 22;
	    break;

	case 0x17:
	    // This is ugly, but I cannot help: Use the firmware version
	    // from the message instead of the saved one, since this we may
	    // not know about the version when receiving this message.
	    // To be shure not to access non-existing message members,
	    // check the length first.
	    if (Size != 94 && Size != 107) {
		WriteDebugLog ("Error: Invalid message length: " +
			       AsciiData (Data, Size));
		S = Size;   // Prevent endless loop
	    } else {
		// Get the version
		unsigned Version = unsigned (Data [4]) * 100 + unsigned (Data [5]);
		S = Version < 193 ? 94 : 107;
	    }
	    break;

	case 0x18:
	    S = 2;
	    break;

	case 0xdd:
	    S = 6;
	    break;

	default:
	    WriteDebugLog ("Warning: Cannot determine message length: " +
			   AsciiData (Data, Size));
	    S = Size; // Prevent endless loop
	    break;
    }

    // The proposed length must not be greater than the real length
    if (S > Size) {
	// OOPS - this should not happen...
	WriteDebugLog ("Error: Got short message: " + AsciiData (Data, Size));
	S = Size;
    }

    // Return the length
    return S;
}



IstecMsg::~IstecMsg ()
{
    delete [] Data;
}



String IstecMsg::AsciiData ()
{
    return ::AsciiData (Data, Size);
}



int IstecMsg::IsDiagMsg ()
// Return true if the message is a diagnostic message
{
    // Unnecessary override for gcc
    return ::IsDiagMsg (Data, Size);
}



int IstecMsg::IsChargeInfo ()
// Return true if the message is a charge info message
{
    // Unnecessary override for gcc
    return ::IsChargeInfo (Data, Size);
}



/*****************************************************************************/
/*			     Com port related code			     */
/*****************************************************************************/



void CloseComPort ()
// Close the com port
{
    if (Port) {
	// Switch off DTR
	Port->DTROff ();

	// Delete the port object
	delete Port;
	Port = NULL;
    }
}



int OpenComPort (const String& PortName)
// Try to open the com port. If the port is already open, it is closed and
// reopened. The function returns 0 on success and an error code on failure.
{
    // Close the port in case it is open
    CloseComPort ();

    // Ok, now reopen it. Use 9600 8N1 without handshaking
    Port = new ComPort (PortName, 9600, 8, 'N', 1, 'D', 'D', PortBase, PortIRQ);
    int Result = Port->Open ();
    if (Result != 0) {
	// Error, maybe the device does not exist or is already in use
	CloseComPort ();
	return Result;
    }

    // Set the timeout value for sending and receiving
    Port->SetRXTimeout (3.0);
    Port->SetTXTimeout (3.0);

    // Make the RTS line active. This is needed for the new PCB of the istec
    // (beginning from version #3).
    Port->RTSOn ();

    // Success
    return 0;
}



int ComPortAvail ()
// Return 1 if the com port is open, 0 if not
{
    return Port != NULL;
}



/*****************************************************************************/
/*			 Low level ISTEC specific code			     */
/*****************************************************************************/



static void UpdateCharges (const unsigned char* NewCharges)
// Copy the charges
{
    // Unpack the new charges into the static area
    Charges.Unpack (NewCharges);

    // Handle the update
    ChargeUpdate = 1;
    HandleCallQueue ();
}



static int IstecWrite (unsigned char* Msg, unsigned BufSize)
// Recode the binary message into the istec format and send it via the port.
{
    // Check the parameters
    PRECONDITION (Msg != NULL && BufSize > 0);

    // Determine the real message size from the message
    unsigned Size = SendMsgSize (Msg, BufSize);

    // If the port is unavailable, we have nothing to do
    if (Port == NULL) {
	return iePortNotOpen;
    }

    // Block the poll routine
    SendInProgress = 1;

    // Log the outgoing message
    WriteDebugLog ("Outgoing: " + AsciiData (Msg, Size));

    // Send the message
    char Buf [4];		// ## should be unsigned
    while (Size) {

	// Set up the length of the paket
	unsigned BytesToSend = Size > 3 ? 3 : Size;
	Size -= BytesToSend;

	// Time to wait after chunk (the times used here are the times the
	// ISTEC software uses: 40ms after a chunk and an additional 60ms
	// [making a total of 100ms] after the last chunk)
	unsigned DelayTime = 40;

	// Set up the header byte
	Buf [0] = BytesToSend;
	if (Size == 0) {
	    // Last chunk, mark it
	    Buf [0] |= 0x80;

	    // Need a longer wait after the last chunk to swallow the command
	    DelayTime = 100;
	}

	// Copy parameter bytes
	unsigned I = 0;
	while (I < BytesToSend) {
	    I++;
	    Buf [I] = *Msg++;
	}

	// Fill rest with 0x00
	while (I < sizeof (Buf) - 1) {
	    I++;
	    Buf [I] = '\0';
	}

	// Send the paket
	for (I = 0; I < sizeof (Buf); I++) {
	    if (Port->TimedSend (Buf [I]) == -1) {
		// Timeout
		WriteDebugLog ("Error: Timeout on outgoing message!");
		SendInProgress = 0;
		return ieTimeout;
	    }
	}

	if (DelayTime) {
	    Delay (DelayTime);
	}

    }

    // Success
    SendInProgress = 0;
    return ieDone;
}



static IstecMsg* IstecReadChar (unsigned char C)
// Is called when a character is available on the serial line. According
// to the status in ICReadStat, the character is handled. If the received
// message is complete, a copy of the message is returned in a buffer allocated
// with new.
{
    // Check the status
    switch (ICReadStat) {

	case stIdle:
	case stGotBlock:
	    // We got a complete block and are waiting for the header of a
	    // new one. C contains the byte count of the new block.
	    if (C & 0x80) {
		// This one is the last block
		ICReadStat = stInLastBlock;
		ICBlockCount = C & 0x7F;
		ICFillCount = 3 - ICBlockCount;
	    } else {
		ICReadStat = stInBlock;
		ICBlockCount = C;
	    }
	    // If we get a block with an invalid block size, we are in
	    // trouble (this may happen if we are connected to the wrong
	    // port). Ignore the block and go back into idle state hoping
	    // that a timeout will clear things.
	    if (ICBlockCount == 0 || ICBlockCount > 3) {
		ICReadStat = stIdle;
	    }
	    break;

	case stInBlock:
	    // We are currently reading a block. ICBlockCount contains the
	    // count of outstanding characters for this block. Place the
	    // received character into the receive buffer.
	    if (ICReadCount < sizeof (ICReadBuf)) {
		ICReadBuf [ICReadCount++] = C;
	    }
	    ICBlockCount--;
	    if (ICBlockCount == 0) {
		// Got that block
		ICReadStat = stGotBlock;
	    }
	    break;

	case stInLastBlock:
	    // We are currently reading the last block. ICBlockCount contains
	    // the count of outstanding characters for this block. Place the
	    // received character into the receive buffer.
	    if (ICReadCount < sizeof (ICReadBuf)) {
		ICReadBuf [ICReadCount++] = C;
	    }
	    ICBlockCount--;
	    if (ICBlockCount == 0) {
		// Got that block. Receive fill bytes or end
		if (ICFillCount > 0) {
		    ICReadStat = stFillBytes;
		} else {
		    // Got a complete message
		    ICReadStat = stIdle;
		    ICMsgCount++;
		}
	    }
	    break;

	case stFillBytes:
	    // We are reading the fill bytes of the last block. Ignore the
	    // bytes and wait for the end of the message
	    ICFillCount--;
	    if (ICFillCount == 0) {
		// Got the fill bytes
		ICReadStat = stIdle;
		ICMsgCount++;
	    }
	    break;

	default:
	    FAIL ("IstecReadChar: Invalid machine state");
	    break;

    }

    // Check if we did receive a complete message
    while (ICMsgCount > 0) {

	// Check if there is one message in the buffer or more than one (this
	// seems to be an ISTEC error)
	IstecMsg* IM = new IstecMsg;
	IM->Size = RecMsgSize (ICReadBuf, ICReadCount); // Returns ICReadCount max
	IM->Data = new unsigned char [IM->Size];
	memcpy (IM->Data, ICReadBuf, IM->Size);
	if (IM->Size < ICReadCount) {
	    // There are bytes left
	    WriteDebugLog ("Error: Got multiple messages in one chunk: " +
			   AsciiData (ICReadBuf, ICReadCount));
	    memmove (ICReadBuf, &ICReadBuf [ICReadCount], ICReadCount - IM->Size);
	    ICReadCount -= IM->Size;
	} else {
	    // No more messages waiting
	    ICMsgCount = 0;
	    ICReadCount = 0;
	}

	// Log the received message
	WriteDebugLog ("Incoming: " + IM->AsciiData ());

	// Look for special messages and handle them directly (don't return
	// those messages to the caller)
	if (IM->IsDiagMsg ()) {

	    // The message is a diagnostic message
	    HandleDiagMsg (IM->Data);
	    delete IM;
	    IM = NULL;

	} else if (IM->IsChargeInfo ()) {

	    // The message is a charge info message
	    UpdateCharges (IM->Data + 1);
	    delete IM;
	    IM = NULL;

	} else {

	    // Cannot handle the message - return it
	    return IM;

	}

    }

    // No complete message left, return a NULL pointer
    return NULL;

}



static int IstecRead (unsigned char* Msg, unsigned Size, unsigned Size2 = 0)
// Read and return a complete message from the istec. The variable Size2 is
// a little hack: Firmware 1.93 changed some message sizes, so there are
// messages, where the exact length is unknown and can be one of two different
// values. If Size2 is not zero, it is a second possibility for the message
// size.
{
    IstecMsg* IM;

    // Check if there is an already received message
    if (LastIstecMsg != NULL) {

	// There is an already received message - grab it
	IM = LastIstecMsg;
	LastIstecMsg = NULL;

    } else {

	// No message, try to receive one
	do {
	    // Get a char with timeout
	    int C = Port->TimedReceive ();
	    if (C == -1) {
		// Timeout
		return ieTimeout;
	    }

	    // Handle the char, receive a complete message if available
	    IM = IstecReadChar (C);

	} while (IM == NULL);

    }

    // Ok, got a message. Check for the buffer size before copying
    unsigned MaxSize = Size > Size2? Size : Size2;
    unsigned BytesToCopy = MaxSize < IM->Size ? MaxSize : IM->Size;
    memcpy (Msg, IM->Data, BytesToCopy);

    // Set up the return code
    int RetCode = ieDone;
    if (IM->Size != Size && (Size2 == 0 || IM->Size != Size2)) {
	// OOPS - message size mismatch. Log this, then set the return code
	WriteDebugLog (FormatStr ("Error: Receive message size mismatch, "
				  "expected %d/%d bytes, got %d bytes: ",
				  Size, Size2, IM->Size) +
		       AsciiData (IM->Data, IM->Size));
	// Set the return code (this is somewhat wrong if Size2 is used, but
	// does not matter since the code is not evaluated besides mapping
	// it to a message
	if (Size > IM->Size) {
	    RetCode = ieRecBufUnderflow;
	} else if (Size < IM->Size) {
	    RetCode = ieRecBufOverflow;
	}
    }

    // Delete the message and return the result
    delete IM;
    return RetCode;
}



void IstecPoll ()
// Poll the istec for incoming diag messages. If we get a real message, store
// it in LastIstecMsg (there should be only one outstanding real message at a
// time).
{
    // If we don't have a valid port, ignore the call
    if (Port == NULL) {
	return;
    }

    // If we are currently sending something, ignore the call
    if (SendInProgress) {
	return;
    }

    // Handle all characters in the receive queue
    while (Port->RXCount () > 0) {

	// Get a char and handle it
	int C = Port->TimedReceive ();
	CHECK (C != -1);
	IstecMsg* IM = IstecReadChar (C);

	// If we have a message, save it into the message buffer
	if (IM) {

	    // This is not a diagnose message. As we have only room to
	    // buffer one message, delete the buffer contents before
	    // storing (can happen only if the ISTEC does not work
	    // correctly, ESTIC should not crash because of that).
	    if (LastIstecMsg) {
		// OOPS - there is a message already waiting
		WriteDebugLog ("Error: Overwriting waiting message: " +
			       AsciiData (LastIstecMsg->Data, LastIstecMsg->Size));
		delete LastIstecMsg;
	    }
	    LastIstecMsg = IM;

	}
    }
}



static int IstecReadAck (unsigned char Ack)
// Wait for an ack from the istec
{
    // Wait for the acknowledgement
    unsigned char Reply [1];
    int Result = IstecRead (Reply, sizeof (Reply));
    if (Result != ieDone) {
	return Result;
    }

    // Check the return value
    if (Reply [0] != Ack) {
	// OOPS - got wrong answer
	WriteDebugLog (FormatStr ("Error: Got wrong ACK: Expected 0x%02X, got 0x%02X",
				  Ack, Reply [0]));
	return ieInvalidReply;
    }

    // Ok, done
    return ieDone;
}



/*****************************************************************************/
/*			High level ISTEC specific code			     */
/*****************************************************************************/



void IstecErrorSync ()
// Try to resync the istec after an error
{
    // First, wait some time. IstecPoll will be called in this time
    Delay (250);

    // To be shure, call IstecPoll again
    IstecPoll ();

    // If we have a waiting message now, throw it away
    if (LastIstecMsg) {
	// Log the facts
	WriteDebugLog ("ErrorSync: Killing waiting message: " +
		       LastIstecMsg->AsciiData ());

	// Delete the message
	delete LastIstecMsg;
	LastIstecMsg = NULL;

    }
}



int IstecReady ()
// Check if the istec answers the "Ready" message.
{
    // ISTEC Command
    static unsigned char Msg [1] = { 0x02 };

    // Send the command to the istec
    int Result = IstecWrite (Msg, sizeof (Msg));
    if (Result != ieDone) {
	return Result;
    }

    // Ok, now wait for the reply
    unsigned char Reply [1];
    Result = IstecRead (Reply, sizeof (Reply));
    if (Result != ieDone) {
	return Result;

    }

    // Check the return message opcode
    if (Reply [0] != 0x12) {
	WriteDebugLog ("Error: Got invalid reply on request 0x02: " +
		       AsciiData (Reply, sizeof (Reply)));
	return ieInvalidReply;
    }

    // Ok, we got the answer from the istec!
    return ieDone;
}



int IstecGetBaseConfig (IstecBaseConfig& Config)
// Request the basic configuration from the istec.
{
    // ISTEC Command
    static unsigned char Msg [1] = { 0x0A };

    // Send the command to the istec
    int Result = IstecWrite (Msg, sizeof (Msg));
    if (Result != ieDone) {
	return Result;
    }

    // Ok, now wait for the reply
    unsigned char Reply [BaseConfigSize + 1];
    Result = IstecRead (Reply, 94, 107);
    if (Result != ieDone) {
	return Result;
    }

    // Check the return message opcode
    if (Reply [0] != 0x17) {
	WriteDebugLog ("Error: Got invalid reply on request 0x0A: " +
		       AsciiData (Reply, sizeof (Reply)));
	return ieInvalidReply;
    }

    // Ok, we got the answer from the istec. Copy the data into the config
    // struct, update the save firmware version from the istec data and
    // return a success code to the caller
    Config.Unpack (&Reply [1]);
    FirmwareVersion = double (Config.VersionHigh) +
		      double (Config.VersionLow) / 100.0;
    return ieDone;
}



int IstecGetDevConfig (DevConfig* Config, unsigned DevCount)
// Request the device configurations from the istec.
{
    // ISTEC Command
    static unsigned char Msg [1] = { 0x08 };

    // Send the command to the istec
    int Result = IstecWrite (Msg, sizeof (Msg));
    if (Result != ieDone) {
	return Result;
    }

    // Ok, now we get many replys
    for (unsigned I = 0; I < DevCount; I++) {

	unsigned char Reply [DevConfigSize + 1];
	int Result = IstecRead (Reply, 18, 22);
	if (Result != ieDone) {
	    return Result;
	}

	// Check the return message code
	if (Reply [0] != 0x16) {
	    WriteDebugLog ("Error: Got invalid reply on request 0x08: " +
			   AsciiData (Reply, sizeof (Reply)));
	    return ieInvalidReply;
	}

	// Ok, we got the answer from the istec. Copy the data into the config
	// struct
	Config [I].Unpack (&Reply [1]);
    }

    // Got it
    return ieDone;
}



int IstecRequestCharges ()
// Request the device charges from the istec. This function is different from
// the "Get" functions as it does not wait for a reply. The charge messages
// from the ISTEC are handled by the IstecPoll function in the background.
// If new charges are available, they are passed to the function NewChargeInfo
// of the application object.
{
    // ISTEC Command
    static unsigned char Msg [1] = { 0x06 };

    // Send the command to the istec
    int Result = IstecWrite (Msg, sizeof (Msg));
    if (Result != ieDone) {
	return Result;
    }

    // Success
    return ieDone;
}



int IstecGetConfig (IstecBaseConfig& BaseConfig, DevConfig* DevConfig)
// Get the complete configuration from the istec
{
    int Result;

    // Read the base configuration
    if ((Result = IstecGetBaseConfig (BaseConfig)) != ieDone) {
	return Result;
    }

    // Determine how many device infos come from the istec. Note: this depends
    // on the firmware version!
    unsigned DevCount = FirmwareVersion < 1.93? IstecDevCount : BaseConfig.AB_InterfaceCount;

    // Read the device configurations
    if ((Result = IstecGetDevConfig (DevConfig, DevCount)) != ieDone) {
	return Result;
    }

    // Ok, all done
    return ieDone;
}



int IstecPutCharges (const IstecCharges& NewCharges)
// Write the given charges to the istec
{
    // First byte is opcode, charges are parameters
    unsigned char Buf [ChargeSize + 1];
    Buf [0] = 0x05;
    NewCharges.Pack (&Buf [1]);

    // Write it out
    int Result = IstecWrite (Buf, sizeof (Buf));

    // If the write has been successful, copy the charges
    if (Result == ieDone) {
	Charges = NewCharges;
	ChargeUpdate = 1;
    }

    // Return the result
    return Result;
}



int IstecPutDevConfig (const DevConfig* Config, unsigned Count)
// Write a set of device configuration data to the istec.
{
    while (Count--) {

	// Set up the command
	unsigned char Buf [DevConfigSize + 1];
	Buf [0] = 0x07;
	Config->Pack (&Buf [1]);

	// Write it out
	int Result;
	if ((Result = IstecWrite (Buf, sizeof (Buf))) != ieDone) {
	    return Result;
	}

	// Wait for the reply
	unsigned char Reply [2];
	if ((Result = IstecRead (Reply, sizeof (Reply))) != ieDone) {
	    return ieDone;
	}

	// Check the reply
	if (Reply [0] != 0x18) {
	    return ieInvalidReply;
	}
	if (Reply [1] != Config->DevNum) {
	    // Reply has the wrong device number
	    return ieWrongDevice;
	}

	// Next device
	Config++;
    }

    // Ok, all done
    return ieDone;
}



int IstecPutBaseConfig (const IstecBaseConfig& Config)
// Write a base configuration to the istec
{
    unsigned char Buf [BaseConfigSize + 1];

    // First byte is opcode, base config is parameter
    Buf [0] = 0x09;
    Config.Pack (&Buf [1]);

    // Write it out
    return IstecWrite (Buf, sizeof (Buf));
}



int IstecMakePermanent ()
// Send the command to the istec to store the current configuration into the
// EEPROM.
{
    // Send the command
    static unsigned char Msg [1] = { 0x0C };
    int Result;
    if ((Result = IstecWrite (Msg, sizeof (Msg))) != ieDone) {
	return Result;
    }

    // Wait for the acknowledgement
    return IstecReadAck (0x11);
}



int IstecPutConfig (IstecBaseConfig& BaseConfig,
		    DevConfig* DevConfig,
		    unsigned DevCount)
// Write the complete configuration to the istec. To make things short, the
// number of devices is given as an parameter
{
    int Result;

    // Write the base configuration
    if ((Result = IstecPutBaseConfig (BaseConfig)) != ieDone) {
	return Result;
    }

    // Write the device configurations
    if ((Result = IstecPutDevConfig (DevConfig, DevCount)) != ieDone) {
	return Result;
    }

    // Ok, all done
    return ieDone;
}



int IstecDiagOn ()
// Switch the istec into diag mode
{
    static unsigned char Msg [6] = { 0xdd, 0x00, 0x69, 0x5a, 0x96, 0xa5 };
    return IstecWrite (Msg, sizeof (Msg));
}



int IstecDiagOff ()
// Disable diag mode
{
    static unsigned char Msg [6] = { 0xdd, 0x01, 0x00, 0x00, 0x00, 0x00 };
    return IstecWrite (Msg, sizeof (Msg));
}



