/*****************************************************************************/
/*									     */
/*				   FVIEWER.CC				     */
/*									     */
/* (C) 1993-95	Ullrich von Bassewitz					     */
/*		Zwehrenbuehlstrasse 33					     */
/*		D-72070 Tuebingen					     */
/* EMail:	uz@ibb.schwaben.com					     */
/*									     */
/*****************************************************************************/



// $Id$
//
// $Log$
//
//



#include "fviewer.h"
#include "progutil.h"



/*****************************************************************************/
/*			       class FileViewer				     */
/*****************************************************************************/



FileViewer::FileViewer (const String& Name, const Rect& Bounds,
			u16 aState, u16 aPalette,
			u16 Number) :
    ItemWindow (Bounds, aState, aPalette, Number),
    F (new TextFileStream (Name)),
    FirstLine (0),
    HOffset (0),
    HDelta (10),
    HLimit (1000)
{
    SetHeader (String (" ") + Name + ' ');
    Init ();
}



FileViewer::FileViewer (TextFileStream* S, const Rect& Bounds,
			u16 aState, u16 aPalette, u16 Number):
    ItemWindow (Bounds, aState, aPalette, Number),
    F (S),
    FirstLine (0),
    HOffset (0),
    HDelta (10),
    HLimit (1000)
{
    Init ();
}



void FileViewer::Init ()
// Caled from the constructors
{
    // Set the zoom size
    ZoomSize = OBounds;
    if (OBounds == Background->GetDesktop ()) {
	// We already have max size, choose a random smaller size
	ZoomSize.Grow (-3, -3);
    }

    if (F->GetStatus () == stOk) {
	DrawInterior ();
    }
}



FileViewer::~FileViewer ()
{
    // Delete the text file stream
    delete F;
}



void FileViewer::PgUp ()
{
    if (FirstLine != 0) {

	if (FirstLine > IYSize () - 1) {
	    FirstLine -= IYSize () - 1;
	} else {
	    FirstLine = 0;
	}

	DrawInterior ();

    }
}



void FileViewer::PgDn ()
{
    if (!AtEnd ()) {

	// Scroll window contents up
	FirstLine += IYSize () - 1;
	if (FirstLine > F->LineCount () - IYSize ()) {
	    FirstLine = F->LineCount () - IYSize ();
	}

	DrawInterior ();

    }
}



void FileViewer::Down ()
{
    if (!AtEnd ()) {

	// Don't do screen output
	Lock ();

	// Scroll window contents up
	FirstLine++;
	ScrollUp ();

	// Seek to the current starting line in the stream
	F->LineSeek (FirstLine + MaxY ());

	// Display line at the bottom of the window
	DrawLine (MaxY ());

	// Update the screen
	Unlock ();

    }
}



void FileViewer::Up ()
{
    if (!AtStart ()) {

	// Don't do screen output
	Lock ();

	// Scroll window contents down
	FirstLine--;
	ScrollDown ();

	// Seek to the current starting line in the stream
	F->LineSeek (FirstLine);

	// Display line at the top of the window
	DrawLine (0);

	// Update the screen
	Unlock ();

    }
}



void FileViewer::Left ()
{
    if (HOffset > 0) {
	if (HOffset > HDelta) {
	    HOffset -= HDelta;
	} else {
	    HOffset = 0;
	}
	DrawInterior ();
    }
}



void FileViewer::Right ()
{
    if (HOffset < HLimit) {
	if ((HOffset += HDelta) > HLimit) {
	    HOffset = HLimit;
	}
	DrawInterior ();
    }
}



void FileViewer::ToTop ()
{
    if (FirstLine != 0) {
	FirstLine = 0;
	DrawInterior ();
    }
}



void FileViewer::Home ()
{
    if (HOffset != 0 || FirstLine != 0) {
	HOffset = 0;
	FirstLine = 0;
	DrawInterior ();
    }
}



void FileViewer::ToBot ()
{
    if (!AtEnd ()) {
	FirstLine = F->LineCount () - IYSize ();
	DrawInterior ();
    }
}



void FileViewer::End ()
{
    if (HOffset != 0 || (!AtEnd ())) {
	FirstLine = F->LineCount () - IYSize ();
	HOffset = 0;
	DrawInterior ();
    }
}



void FileViewer::CenterLine (unsigned Line)
// Center the given line in the viewer (if possible)
{
    // Set the first line according to the given line
    if (Line < IYSize () / 2) {
	FirstLine = 0;
    } else {
	FirstLine = Line - IYSize () / 2;
    }
    DrawInterior ();
}



int FileViewer::LineIsVisible (unsigned Line)
// Returns 1 if the given line is visible in the window, 0 otherwise
{
    return (Line >= FirstLine && Line <= FirstLine + IYSize ());
}



void FileViewer::ShowLine (unsigned Line)
// Show the line in the window if it is not already visible
{
    if (!LineIsVisible (Line)) {
	CenterLine (Line);
    }
}



void FileViewer::HandleKey (Key& K)
{
    switch (K) {
	case vkUp:	    Up ();	K = kbNoKey;	break;
	case vkDown:	    Down ();	K = kbNoKey;	break;
	case vkLeft:	    Left ();	K = kbNoKey;	break;
	case vkRight:	    Right ();	K = kbNoKey;	break;
	case vkPgUp:	    PgUp ();	K = kbNoKey;	break;
	case vkPgDn:	    PgDn ();	K = kbNoKey;	break;
	case vkEnd:	    End ();	K = kbNoKey;	break;
	case vkCtrlPgDn:    ToBot ();	K = kbNoKey;	break;
	case vkHome:	    Home ();	K = kbNoKey;	break;
	case vkCtrlPgUp:    ToTop ();	K = kbNoKey;	break;
	default:	    ItemWindow::HandleKey (K);	break;
    }
}



void FileViewer::DrawLine (int Y, int Single)
// Draw one line at position Y. The line is read from the current position in
// the file
{
    unsigned XSize = IBounds.XSize ();

    // Read the line from the file
    String Line = F->GetLine ().Cut (HOffset, XSize);

    // Convert the line to the internally used character set
    Line.InputCvt ();

    // Pad the line to the window width if needed
    if (Single) {
	// Only one line, no ClrScr(), pad to line length
	Line.Pad (String::Right, XSize);
    }

    // print the line
    Write (0, Y, Line);
}



void FileViewer::DrawInterior ()
// Redraw the window interior
{
    // Seek to the current starting line in the stream
    F->LineSeek (FirstLine);

    // Lock screen output
    Lock ();

    // Clear the screen
    Clear ();

    // Now display all lines in the window
    u32 Cur = FirstLine;
    u32 LinesToDisplay = F->LineCount () - Cur;
    if (LinesToDisplay > IYSize ()) {
	LinesToDisplay = IYSize ();
    }
    for (int Y = 0; Y < LinesToDisplay; Y++) {
	DrawLine (Y, 0);
    }
    Unlock ();
}



u32 FileViewer::GetStatusFlags ()
// Returns the flags that are used to build the status line in Browse
{
    u32 Flags = ItemWindow::GetStatusFlags () | siCursPgKeys_Move;
    if (HasHelp ()) {
	Flags |= siHelp;
    }
    return Flags;
}



void FileViewer::Zoom ()
// Zoom the window
{
    // Get the desktop bounds
    Rect Desktop = Background->GetDesktop ();

    // Check if we must zoom in or out
    if (OBounds != Desktop) {
	// Remember the old size, then zoom out
	ZoomSize = OBounds;
	Resize (Desktop);
    } else {
	// Zoom in
	Resize (ZoomSize);
    }
}



