/*****************************************************************************/
/*									     */
/*				   ICDIAG.CC				     */
/*									     */
/* (C) 1995-96	Ullrich von Bassewitz					     */
/*		Zwehrenbuehlstrasse 33					     */
/*		D-72070 Tuebingen					     */
/* EMail:	uz@ibb.schwaben.com					     */
/*									     */
/*****************************************************************************/



// $Id$
//
// $Log$
//
//



#include <stdio.h>

#include "delay.h"
#include "strcvt.h"
#include "coll.h"
#include "datetime.h"
#include "menue.h"
#include "textitem.h"
#include "listbox.h"
#include "menuitem.h"
#include "progutil.h"
#include "settings.h"

#include "icobjid.h"
#include "icmsg.h"
#include "iclog.h"
#include "icdlog.h"
#include "iccwin.h"
#include "icalias.h"
#include "icwinmgr.h"
#include "istec.h"
#include "iccom.h"



/*****************************************************************************/
/*			       Message constants			     */
/*****************************************************************************/



const u16 msMsgWinHeader	= MSGBASE_ICDIAG +  0;
const u16 msOn			= MSGBASE_ICDIAG +  1;
const u16 msOff			= MSGBASE_ICDIAG +  2;
const u16 msLEDState		= MSGBASE_ICDIAG +  3;
const u16 msChargeState		= MSGBASE_ICDIAG +  4;
const u16 msTFEState		= MSGBASE_ICDIAG +  5;
const u16 msDoorState		= MSGBASE_ICDIAG +  6;
const u16 msSwitchState		= MSGBASE_ICDIAG +  7;
const u16 msPulseDial		= MSGBASE_ICDIAG +  8;
const u16 msToneDial		= MSGBASE_ICDIAG +  9;
const u16 msCallWinHeader	= MSGBASE_ICDIAG + 10;
const u16 msCallStart		= MSGBASE_ICDIAG + 11;
const u16 msCallEnd		= MSGBASE_ICDIAG + 12;
const u16 msCallTimeFmt		= MSGBASE_ICDIAG + 13;
const u16 msMatrixWindowTitle	= MSGBASE_ICDIAG + 14;
const u16 msMatrixWindowHeader	= MSGBASE_ICDIAG + 15;



/*****************************************************************************/
/*				     Data				     */
/*****************************************************************************/



// Wait time after a call before requesting the charges
int DebugWaitAfterCall	= 500;

// If true, log calls with a chargecount of zero
int LogZeroCostCalls	= 1;

// If true, reread the alias file before trying to resolve call data
int AutoReadAliases	= 0;

// The name of the matrix window position in the settings file
static const String MatrixWindowBounds = "MatrixWindow.Bounds";

// Width of the matrix window
static const unsigned MatrixWindowWidth = 64;

// Max length of the phone number
static const unsigned CallPhoneMaxLen = 16;

// The diag data for tone dial is somewhat weird: When using pulse dial, the
// diag message from the istec contains the device number. When using tone
// dial, the number of the internal connection is transmitted instead. To
// show the device number in both cases, we hold the device that uses an
// internal connection in the array below
static unsigned IntCon [3];



/*****************************************************************************/
/*		 Constants for the State field in DevStateInfo		     */
/*****************************************************************************/



const unsigned stSchleife	= 0x0001;
const unsigned stAmt1		= 0x0002;
const unsigned stAmt2		= 0x0004;
const unsigned stAmt		= stAmt1 | stAmt2;
const unsigned stInt1		= 0x0008;
const unsigned stInt2		= 0x0010;
const unsigned stInt3		= 0x0020;
const unsigned stInt		= stInt1 | stInt2 | stInt3;
const unsigned stTon		= 0x0040;
const unsigned stWTon		= 0x0080;
const unsigned stTFE		= 0x0100;
const unsigned stRuf		= 0x0200;



/*****************************************************************************/
/*			Explicit template instantiation			     */
/*****************************************************************************/



#ifdef EXPLICIT_TEMPLATES
template class Collection<class DevStateInfo>;
template class SortedCollection<class DevStateInfo, unsigned char>;
template class ListBox<class DevStateInfo>;
#endif



/*****************************************************************************/
/*			     Diag- and CallWindow			     */
/*****************************************************************************/



static void WriteDiagMsg (const String& /* DiagMsg */)
// This is empty now, but it may be reenabled
{
}



/*****************************************************************************/
/*			      class DevStateInfo			     */
/*****************************************************************************/



class DevStateInfo {

    // Columns for the matrix window
    //		 1	   2	     3	       4	 5	   6	     7
    // 01234567890123456789012345678901234567890123456789012345678901234567890
    //
    //	Nr. Amt1 Amt2 Int1 Int2 Int3 Ton WTon TFE Ruf Nummer
    enum {
	colNr	    =  1,
	colSchleife =  3,
	colAmt1     =  6,
	colAmt2     = 11,
	colInt1     = 16,
	colInt2     = 21,
	colInt3     = 26,
	colTon	    = 31,
	colWTon     = 35,
	colTFE	    = 40,
	colRuf	    = 44,
	colPhone    = 47
    };


    void ClearCallPhone ();
    // Clear the call phone number, reset the stuff in the matrix line


public:
    const unsigned char DevNum;		// Device number
    u32			State;		// Device state
    String		CallPhone;	// Phone number dialed
    Time		CallStart;	// Start of call
    TimeDiff		CallDuration;	// Duration of call
    u16			StartCharges;	// Charges at start of call
    u16			CallCharges;	// Charges for call
    unsigned		HasExt;		// True if device had an external call
    String		MatrixLine;	// The line that's displayed


    DevStateInfo (unsigned char Device);
    // Create a DevStateInfo object

    DevStateInfo (const DevStateInfo& X);
    // Copy constructor

    String LogMsg ();
    // Create a message for the logfile from the data

    int GetState (unsigned StateMask);
    void SetState (unsigned NewState);
    void ClrState (unsigned NewState);
    // Get/set/clear bits in State

    void SetSchleife (int State);
    void SetAmt (int State, unsigned Amt);
    void SetInt (int State, unsigned Int);
    void SetTon (int State);
    void SetWTon (int State);
    void SetTFE (int State);
    void SetRuf (int State);
    // Set specific matrix states

    void AddDigit (char Digit);
    // Add a digit to the phone number if the device is in a state where a digit
    // is accepted (dialed)
};



/*****************************************************************************/
/*				class CallQueue				     */
/*****************************************************************************/



class CallQueue: public Collection<DevStateInfo> {

public:
    CallQueue ();
    // Create a CallQueue

    DevStateInfo* At (int Index);
    // Return a pointer to the item at position Index.
    // OVERRIDE FOR DEBUGGING

};



CallQueue::CallQueue ():
    Collection <class DevStateInfo> (10, 10, 1)
{
}



DevStateInfo* CallQueue::At (int Index)
// Return a pointer to the item at position Index.
// OVERRIDE FOR DEBUGGING
{
    // Check range
    if (Index < 0 || Index >= Count) {
	FAIL ("CallQueue::At: Index out of bounds");
	return NULL;
    }

    return Collection<DevStateInfo>::At (Index);
}



static CallQueue Queue;



/*****************************************************************************/
/*			      class DevStateInfo			     */
/*****************************************************************************/



DevStateInfo::DevStateInfo (unsigned char Device):
    DevNum (Device),
    State (0),
    HasExt (0),
    MatrixLine (MatrixWindowWidth)
{
    // Set the width of the matrix line and insert the number
    MatrixLine.Set (0, MatrixWindowWidth, ' ');
    MatrixLine.Replace (colNr, U32Str (Device + 21));

    // Create the rest of the line
    MatrixLine.Replace (colAmt1, '-');
    MatrixLine.Replace (colAmt2, '-');
    MatrixLine.Replace (colInt1, '-');
    MatrixLine.Replace (colInt2, '-');
    MatrixLine.Replace (colInt3, '-');
    MatrixLine.Replace (colTon, '-');
    MatrixLine.Replace (colWTon, '-');
    MatrixLine.Replace (colTFE, '-');
    MatrixLine.Replace (colRuf, '-');
}



DevStateInfo::DevStateInfo (const DevStateInfo& X):
    DevNum (X.DevNum),
    State (X.State),
    CallPhone (X.CallPhone),
    CallStart (X.CallStart),
    CallDuration (X.CallDuration),
    StartCharges (X.StartCharges),
    CallCharges (X.CallCharges),
    HasExt (X.HasExt),
    MatrixLine (X.MatrixLine)
{
}



void DevStateInfo::ClearCallPhone ()
// Clear the call phone number, reset the stuff in the matrix line
{
    // Clear the number
    CallPhone.Clear ();

    // Reset the matrix line
    MatrixLine.Set (colPhone, CallPhoneMaxLen, ' ');
}



String DevStateInfo::LogMsg ()
// Create a message for the logfile from the data
{
    // If AutoReadAliases is set, reread the alias file before trying to
    // resolve call data
    if (AutoReadAliases) {
	ReadAliasFile ();
    }

    // Format of the line:
    //		 1	   2	     3	       4	 5	   6	     7	       8
    // 012345678901234567890123456789012345678901234567890123456789012345678901234567890
    // NR  ALIAS_______  DATE____  START___  DURATION  CALLPHONE_______  EINH  DM____

    String StartDate = CallStart.DateTimeStr ("%d.%m.%y");
    String StartTime = CallStart.DateTimeStr ("%H:%M:%S");
    unsigned DurSec = CallDuration.GetSec ();
    unsigned DurMin = (DurSec % 3600) / 60;
    unsigned DurHour = DurSec / 3600;
    DurSec %= 60;
    String DevAlias = GetAlias (DevNum+21);
    DevAlias.Trunc (12);

    // Use the alias of the phone number if one is defined, otherwise use
    // the number
    String Phone = GetAlias (CallPhone);
    if (Phone.IsEmpty ()) {
	Phone = CallPhone;
    }
    Phone.Trunc (CallPhoneMaxLen);

    return FormatStr ("%d  %-12s  %s  %s  %02d:%02d:%02d  %-16s  %4d  %6.2f",
		      DevNum+21,
		      DevAlias.GetStr (),
		      StartDate.GetStr (),
		      StartTime.GetStr (),
		      DurHour, DurMin, DurSec,
		      Phone.GetStr (),
		      CallCharges,
		      CallCharges * PricePerUnit);
}



inline int DevStateInfo::GetState (unsigned StateMask)
{
    return (State & StateMask) != 0;
}



inline void DevStateInfo::SetState (unsigned NewState)
{
    State |= NewState;
}



inline void DevStateInfo::ClrState (unsigned NewState)
{
    State &= ~NewState;
}



void DevStateInfo::SetSchleife (int State)
{
    // If we already have the correct settings, ignore the call
    if (State == GetState (stSchleife)) {
        // Ignore it
        return;
    }
    
    // Now check which change
    if (State) {

	// Mark the device in the matrix
	MatrixLine.Replace (colSchleife, '*');

	// Set the state bit
	SetState (stSchleife);

	// Clear the phone number
	ClearCallPhone ();

    } else {

	// Unmark the device
	MatrixLine.Replace (colSchleife, ' ');

	// Reset the state bit
	ClrState (stSchleife);

	// Clear the phone number if we have no external line. Otherwise
	// simulate the "Amt off" message to circumvent a bug in the istec
	// firmware
	if (GetState (stAmt) == 0) {

            // Clear the phonre number
	    ClearCallPhone ();

	} else {
	
	    // Simulate "Amt off"
	    SetAmt (0, GetState (stAmt1)? 1 : 2);

	}
    }
}



void DevStateInfo::SetAmt (int State, unsigned Amt)
{
    // If we already have the correct settings, ignore the call
    if (State == GetState (stAmt)) {
        // Ignore it
        return;
    }
    
    // Map the line number to a column number in the matrix
    unsigned Column = 0;		// Make gcc happy
    unsigned Mask = 0;			// Make gcc happy
    switch (Amt) {
	case 1:   Column = colAmt1; Mask = stAmt1;	break;
	case 2:   Column = colAmt2; Mask = stAmt2;	break;
	default:  FAIL ("SetAmt: Invalid value for Amt");
    }

    // Check which state change
    if (State) {

	// Update the matrix line
	MatrixLine.Replace (Column, 'x');

	// Remember the line used
	SetState (Mask);

	// This should be an external call, so handle the phone number.
	// This is somewhat difficult to handle since there are different
	// firmware versions, some that get a line if you pick up the phone,
	// others that need a predialed zero. Worse, sometimes the diag message
	// indicating that we have an external line comes late, after dialing
	// two or even three digits...
	// So assume the following: If we get the external line and the phone
	// number is empty, leave it alone. Otherwise delete the first digit.
	if (CallPhone.Len () > 0) {
	    // Delete the string in the matrix line
	    MatrixLine.Set (colPhone, CallPhoneMaxLen, ' ');

	    // Delete the first digit
	    CallPhone.Del (0, 1);

	    // Put the new number into the matrix line
	    if (CallPhone.Len () > 0) {
		MatrixLine.Replace (colPhone, CallPhone);
	    }
	}

    } else {

	// Update the matrix line
	MatrixLine.Replace (Column, '-');

	// Clear the used line
	ClrState (stAmt);

	// If we had an external call, print the info
	if (HasExt) {

	    // Get current time
	    Time Current = Now ();

	    // Calculate the durcation of the call
	    CallDuration = Current - CallStart;

	    // Reset flag for external call
	    HasExt = 0;

	    // Create an entry for the queue
	    DevStateInfo* DI = new DevStateInfo (*this);

	    // Wait some time
	    Delay (DebugWaitAfterCall);

	    // Insert the entry into into the call queue
	    Queue.Insert (DI);

	    // Request a charge update from the istec
	    IstecRequestCharges ();

	}

        // If the device is inactive now (Schleife == 0), clear the called
        // phone number. SetSchleife will act in a similiar way, clearing the
        // call phone if there is no external line, so the number is cleared
        // if both, Amt and Schleife are inactive.
        if (GetState (stSchleife) == 0) {
            // Both, Amt and Schleife are inactive, clear the displayed
            // phone number
            ClearCallPhone ();
        }
    }
}



void DevStateInfo::SetInt (int State, unsigned Int)
{
    // Map the line number to a column number in the matrix
    unsigned Column = 0;		// Make gcc happy
    unsigned Mask = 0;			// Make gcc happy
    switch (Int) {
	case 1:   Column = colInt1; Mask = stInt1;	break;
	case 2:   Column = colInt2; Mask = stInt2;	break;
	case 3:   Column = colInt3; Mask = stInt3;	break;
	default:  FAIL ("SetInt: Invalid column number");
    }

    // Ignore the call if we already have the correct state
    if (State == GetState (Mask)) {
        return;
    }
    
    // Check which state change we have
    if (State) {

	// Show the change
	MatrixLine.Replace (Column, 'x');

	// Set the state bit
	SetState (Mask);

	// Remember which device uses this internal line
	IntCon [Int-1] = DevNum;

    } else {

	// Show the change
	MatrixLine.Replace (Column, '-');

	// Reset the state bit
	ClrState (Mask);

	// If the internal connection goes off, but we have an external one,
	// we have an external call.
	if (GetState (stAmt)) {

	    // Remember the starting time
	    CallStart = Now ();

	    // Remember the charge info on start
	    StartCharges = Charges [DevNum];

	    // Remember that we are connected externally
	    HasExt = 1;

	}
    }
}



void DevStateInfo::SetTon (int State)
{
    if (State) {
	MatrixLine.Replace (colTon, 'x');
	SetState (stTon);
    } else {
	MatrixLine.Replace (colTon, '-');
	ClrState (stTon);
    }
}



void DevStateInfo::SetWTon (int State)
{
    if (State) {
	MatrixLine.Replace (colWTon, 'x');
	SetState (stWTon);
    } else {
	MatrixLine.Replace (colWTon, '-');
	ClrState (stWTon);
    }
}



void DevStateInfo::SetTFE (int State)
{
    if (State) {
	MatrixLine.Replace (colTFE, 'x');
	SetState (stTFE);
    } else {
	MatrixLine.Replace (colTFE, '-');
	ClrState (stTFE);
    }
}



void DevStateInfo::SetRuf (int State)
{
    if (State) {
	MatrixLine.Replace (colRuf, 'x');
	SetState (stRuf);
    } else {
	MatrixLine.Replace (colRuf, '-');
	ClrState (stRuf);
    }
}



void DevStateInfo::AddDigit (char Digit)
// Add a digit to the phone number if the device is in a state where a digit
// is accepted (dialed)
{
    if (Digit != 'R' && (GetState (stAmt) == 0 || HasExt == 0)) {
	MatrixLine.Replace (colPhone + CallPhone.Len (), Digit);
	CallPhone += Digit;
    }
}



/*****************************************************************************/
/*			      class DevStateColl			     */
/*****************************************************************************/



class DevStateColl: public SortedCollection<DevStateInfo, unsigned char> {

protected:
    virtual int Compare (const unsigned char* Key1, const unsigned char* Key2);
    virtual const unsigned char* KeyOf (const DevStateInfo* Item);

public:
    DevStateColl ();
    // Create a DevStateColl

    void DeleteDev (unsigned char Num);
    // Delete the device with the given numer.

    DevStateInfo* NewDev (unsigned char Num);
    // Create and insert a device with the given numer.

    DevStateInfo* GetDevStateInfo (unsigned char Num);
    // Return a pointer to the entry. Calls FAIL if the entry does not exist

    void SetState (unsigned char Dev, unsigned NewState);
    void ClrState (unsigned char Dev, unsigned NewState);
    // Set/reset the device state bits

    DevStateInfo* At (int Index);
    // Return a pointer to the item at position Index.
    // OVERRIDE FOR DEBUGGING

    void SetSchleife (unsigned Dev, int State);
    void SetAmt (unsigned Dev, int State, unsigned Amt);
    void SetInt (unsigned Dev, int State, unsigned Int);
    void SetTon (unsigned Dev, int State);
    void SetWTon (unsigned Dev, int State);
    void SetTFE (unsigned Dev, int State);
    void SetRuf (unsigned Dev, int State);
    // Set specific matrix states

    void AddDigit (unsigned Dev, char Digit);
    // Add a digit to the phone number if the device is in a state where a digit
    // is accepted (dialed)
};



int DevStateColl::Compare (const unsigned char* Key1, const unsigned char* Key2)
{
    if (*Key1 < *Key2) {
	return -1;
    } else if (*Key1 > *Key2) {
	return 1;
    } else {
	return 0;
    }
}



const unsigned char* DevStateColl::KeyOf (const DevStateInfo* Item)
{
    return &Item->DevNum;
}



DevStateColl::DevStateColl ():
    SortedCollection <DevStateInfo, unsigned char> (10, 10, 1)
{
    // Insert 8 devices (more devices are added dynamically if they are
    // detected the first time)
    for (unsigned char Dev = 0; Dev < 8; Dev++) {
	NewDev (Dev);
    }
}



void DevStateColl::DeleteDev (unsigned char Num)
// Delete the device with the given numer.
{
    // Search for the device
    int Index;
    CHECK (Search (&Num, Index) != 0);

    // Delete the entry
    AtDelete (Index);
}



DevStateInfo* DevStateColl::NewDev (unsigned char Num)
// Create and insert a device with the given numer.
{
    // Create a new entry
    DevStateInfo* DI = new DevStateInfo (Num);

    // Insert the new entry
    Insert (DI);

    // And return it
    return DI;
}



DevStateInfo* DevStateColl::GetDevStateInfo (unsigned char Num)
// Return a pointer to the entry. Creates an entry if none exists.
{
    // Search for the entry
    int Index;
    if (Search (&Num, Index) == 0) {
	// No entry til now, create one
	return NewDev (Num);
    } else {
	// Found, return it
	return At (Index);
    }
}



void DevStateColl::SetState (unsigned char Dev, unsigned NewState)
// Set the device state bits
{
    GetDevStateInfo (Dev)->SetState (NewState);
}



void DevStateColl::ClrState (unsigned char Dev, unsigned NewState)
// Reset the device state bits
{
    GetDevStateInfo (Dev)->ClrState (NewState);
}



DevStateInfo* DevStateColl::At (int Index)
// Return a pointer to the item at position Index.
// OVERRIDE FOR DEBUGGING
{
    // Check range
    if (Index < 0 || Index >= Count) {
	FAIL ("DevStateColl::At: Index out of bounds");
	return NULL;
    }

    return SortedCollection<DevStateInfo, unsigned char>::At (Index);
}



void DevStateColl::SetSchleife (unsigned Dev, int State)
{
    GetDevStateInfo (Dev)->SetSchleife (State);
}



void DevStateColl::SetAmt (unsigned Dev, int State, unsigned Amt)
{
    GetDevStateInfo (Dev)->SetAmt (State, Amt);
}



void DevStateColl::SetInt (unsigned Dev, int State, unsigned Int)
{
    GetDevStateInfo (Dev)->SetInt (State, Int);
}



void DevStateColl::SetTon (unsigned Dev, int State)
{
    GetDevStateInfo (Dev)->SetTon (State);
}



void DevStateColl::SetWTon (unsigned Dev, int State)
{
    GetDevStateInfo (Dev)->SetWTon (State);
}



void DevStateColl::SetTFE (unsigned Dev, int State)
{
    GetDevStateInfo (Dev)->SetTFE (State);
}



void DevStateColl::SetRuf (unsigned Dev, int State)
{
    GetDevStateInfo (Dev)->SetRuf (State);
}



void DevStateColl::AddDigit (unsigned Dev, char Digit)
// Add a digit to the phone number if the device is in a state where a digit
// is accepted (dialed)
{
    GetDevStateInfo (Dev)->AddDigit (Digit);
}



static DevStateColl DevState;



/*****************************************************************************/
/*			      class MatrixListBox			     */
/*****************************************************************************/



class MatrixListBox: public ListBox<DevStateInfo> {

protected:
    virtual void Print (int Index, int X, int Y, u16 Attr);
    // Display one of the listbox entries

public:
    MatrixListBox (i16 aID, const Point& aSize);
    // Create a matrix listbox

    MatrixListBox (StreamableInit);
    // Create an empty matrix listbox

    virtual ~MatrixListBox ();
    // Destroy a MatrixListBox

    virtual u16 StreamableID () const;
    // Return the streamable ID

    static Streamable* Build ();
    // Build an empty object
};



// Register the class
LINK (MatrixListBox, ID_MatrixListBox);



MatrixListBox::MatrixListBox (i16 aID, const Point& aSize):
    ListBox <DevStateInfo> ("", aID, aSize, atEditNormal, atEditBar, atEditNormal, NULL)
{
    // Set the collection
    SetColl (&DevState);
}



inline MatrixListBox::MatrixListBox (StreamableInit):
    ListBox<DevStateInfo> (Empty)
// Create an empty matrix listbox
{
    // Set the collection
    SetColl (&DevState);
}



MatrixListBox::~MatrixListBox ()
// Destroy a MatrixListBox
{
    // Beware: Before deleting, reset the collection pointer of the listbox
    SetColl (NULL);
}



void MatrixListBox::Print (int Index, int X, int Y, u16 Attr)
{
    // Get the line
    String S = Coll->At (Index)->MatrixLine;

    // Pad the line to length
    S.Pad (String::Right, Size.X);

    // Write out the string
    Owner->Write (X, Y, S, Attr);
}



u16 MatrixListBox::StreamableID () const
// Return the streamable ID
{
    return ID_MatrixListBox;
}



Streamable* MatrixListBox::Build ()
// Build an empty object
{
    return new MatrixListBox (Empty);
}



/*****************************************************************************/
/*			      class MatrixWindow			     */
/*****************************************************************************/



class MatrixWindow: public ItemWindow {

    MatrixListBox*		MatrixBox;	// Direct pointer to listbox
    u16				DevCount;	// Device count
    Rect			ZoomSize;	// Size for zooming

    static unsigned		WindowCount;	// Count of windows

public:
    static MatrixWindow*	Win;		// Pointer to the only window


protected:
    virtual void HandleKey (Key& K);
    // Key dispatcher used in Browse

public:
    MatrixWindow (const Point& Pos, unsigned aDevCount);
    // Create a matrix window

    MatrixWindow (StreamableInit);
    // Create an empty object

    virtual ~MatrixWindow ();
    // Destroy a matrix window

    virtual void Store (Stream&) const;
    // Store the object into a stream

    virtual void Load (Stream&);
    // Load the object from a stream

    virtual u16 StreamableID () const;
    // Return the streamable ID

    static Streamable* Build ();
    // Build an empty object

    virtual unsigned MinXSize () const;
    // Return the minimum X size of the window. Override this to limit resizing.

    virtual unsigned MinYSize () const;
    // Return the minumim Y size of the window. Override this to limit resizing.

    virtual void Resize (const Rect& NewBounds);
    // Resize the window to the new bounds (this can also be used to move the
    // window but Move is faster if the window should not be resized).

    virtual void Zoom ();
    // Zoom the window

    void DrawMatrix ();
    // Redraw the listbox after changes

};



// Register the class
LINK (MatrixWindow, ID_MatrixWindow);



void MatrixWindow::HandleKey (Key& K)
// Key dispatcher used in Browse
{
    // First call the derived function.
    ItemWindow::HandleKey (K);

    // Maybe the listbox has some work
    MatrixBox->HandleKey (K);
}



MatrixWindow::MatrixWindow (const Point& Pos, unsigned aDevCount):
    ItemWindow (Rect (Pos.X, Pos.Y, Pos.X+MatrixWindowWidth+2, Pos.Y+11),
		wfFramed | wfCanMove | wfCanResize | wfSaveVisible),
    MatrixBox (NULL),
    DevCount (aDevCount),
    ZoomSize (OBounds)
{
    // Count of windows must be zero, otherwise this is an error
    CHECK (WindowCount == 0);

    // Lock window output
    Lock ();

    // If there is a stored window size in the settings file, resize the
    // window to the stored rectangle.
    Rect StoredBounds = StgGetRect (MatrixWindowBounds, OBounds);
    if (StoredBounds != OBounds) {
	Resize (StoredBounds);
    }

    // Set the window title
    SetHeader (LoadAppMsg (msMatrixWindowTitle));

    // Create and insert the header line
    TextItem* HdrItem = new TextItem (LoadAppMsg (msMatrixWindowHeader),
				      100, atTextNormal, NULL);
    AddItem (HdrItem);
    HdrItem->SetWidth (IXSize ());
    HdrItem->SetPos (0, 0);

    // Create a listbox inside the window
    Point Size (IXSize (), IYSize () - 1);
    MatrixBox = new MatrixListBox (1, Size);
    AddItem (MatrixBox);
    MatrixBox->SetPos (0, 1);
    MatrixBox->Draw ();

    // Redraw the window contents
    DrawInterior ();

    // Unlock the window, allowing output
    Unlock ();

    // Ok, we have the window now, make it available globally
    WindowCount++;
    Win = this;

    // Disable the menue command to create more matrix windows
    DisableCommand (miMatrixWin);
}



inline MatrixWindow::MatrixWindow (StreamableInit):
    ItemWindow (Empty)
{
    // Be shure to check the window count
    CHECK (WindowCount++ == 0);

    // Disable the menue command to create more matrix windows
    DisableCommand (miMatrixWin);
}



MatrixWindow::~MatrixWindow ()
{
    // Store the current window position and size into the settings file
    StgPutRect (OBounds, MatrixWindowBounds);

    // Decrease the window count and invalidate the global pointer
    WindowCount--;
    Win = NULL;

    // Enable the menue command to create more matrix windows
    EnableCommand (miMatrixWin);
}



void MatrixWindow::Store (Stream& S) const
// Store the object into a stream
{
    // Before storing, be shure to reset the pointer for the device state
    // collection
    MatrixBox->SetColl (NULL);

    // Now use the inherited store
    ItemWindow::Store (S);

    // Reset the collection pointer
    MatrixBox->SetColl (&DevState);

    // Store additional data
    S << DevCount << ZoomSize;

}



void MatrixWindow::Load (Stream& S)
// Load the object from a stream
{
    // Load the derived data
    ItemWindow::Load (S);

    // Load the device count
    S >> DevCount >> ZoomSize;

    // Get a pointer to the listbox
    MatrixBox = (MatrixListBox*) ForcedItemWithID (1);

    // Assign a pointer to the global collection
    MatrixBox->SetColl (&DevState);

    // The inherited Load function did not draw the contents of the listbox
    // since the listbox had no valid data collection at that time. Do the
    // redraw now.
    DrawMatrix ();

    // Ok, the window is valid now. Do a check of the global pointer, then
    // replace it
    CHECK (Win == NULL);
    Win = this;
}



u16 MatrixWindow::StreamableID () const
// Return the streamable ID
{
    return ID_MatrixWindow;
}



Streamable* MatrixWindow::Build ()
// Build an empty object
{
    return new MatrixWindow (Empty);
}



unsigned MatrixWindow::MinXSize () const
// Return the minimum X size of the window. Override this to limit resizing.
{
    return 7;		// Device number + online flag
}



unsigned MatrixWindow::MinYSize () const
// Return the minumim Y size of the window. Override this to limit resizing.
{
    return 4;		// Header and one device
}



void MatrixWindow::Resize (const Rect& NewBounds)
// Resize the window to the new bounds (this can also be used to move the
// window but Move is faster if the window should not be resized).
{
    // If we have already a matrix listbox, resize it to fit into the new
    // window
    if (MatrixBox) {
	MatrixBox->SetWidth (NewBounds.XSize () - 2);
	MatrixBox->SetHeight (NewBounds.YSize () - 3);
    }

    // Now do the actual resize
    ItemWindow::Resize (NewBounds);
}



void MatrixWindow::Zoom ()
// Zoom the window
{
    // Get the desktop bounds
    Rect Desktop = Background->GetDesktop ();

    // Check if we must zoom in or out
    if (OBounds != Desktop) {
	// Remember the old size, then zoom out
	ZoomSize = OBounds;
	Resize (Desktop);
    } else {
	// Zoom in
	Resize (ZoomSize);
    }
}



void MatrixWindow::DrawMatrix ()
// Redraw the listbox after changes
{
    MatrixBox->Draw ();
}



/*****************************************************************************/
/*		    Static variables of class MatrixWindow		     */
/*****************************************************************************/



unsigned MatrixWindow::WindowCount = 0;
MatrixWindow* MatrixWindow::Win = NULL;



/*****************************************************************************/
/*			    Matrix update functions			     */
/*****************************************************************************/



static void RedrawMatrix ()
{
    // Update the matrix column
    if (MatrixWindow::Win) {
	MatrixWindow::Win->DrawMatrix ();
    }
}



static void WriteDialStatus (unsigned MsgNum, unsigned Dev, unsigned Digit)
{
    // Remap the digit code to a real digit
    switch (Digit) {
	case  1:	Digit = '1';	break;
	case  2:	Digit = '2';	break;
	case  3:	Digit = '3';	break;
	case  4:	Digit = '4';	break;
	case  5:	Digit = '5';	break;
	case  6:	Digit = '6';	break;
	case  7:	Digit = '7';	break;
	case  8:	Digit = '8';	break;
	case  9:	Digit = '9';	break;
	case 10:	Digit = '0';	break;
	case 16:	Digit = 'R';	break;
	default:	Digit = '?';	break;
    }

    // Assemble the message and print it
    WriteDiagMsg (FormatStr (LoadAppMsg (MsgNum).GetStr (), Dev+21, Digit));

    // Remember the digit, but only if we don't have already a connection
    DevState.AddDigit (Dev, (char) Digit);

    // Update the matrix window
    RedrawMatrix ();
}



static void WriteDiagStatus (unsigned MsgNum, int State)
{
    String OnOff = LoadAppMsg (State? msOn : msOff);
    WriteDiagMsg (FormatStr (LoadAppMsg (MsgNum).GetStr (), OnOff.GetStr ()));
}



static void SetMatrix (unsigned Col, unsigned Dev, int State)
{
    // Check which matrix column to set
    switch (Col) {

	case 0:
	    DevState.SetAmt (Dev, State, 1);
	    break;

	case 1:
	    DevState.SetAmt (Dev, State, 2);
	    break;

	case 2:
	    DevState.SetInt (Dev, State, 1);
	    break;

	case 3:
	    DevState.SetInt (Dev, State, 2);
	    break;

	case 4:
	    DevState.SetInt (Dev, State, 3);
	    break;

	case 5:
	    DevState.SetTon (Dev, State);
	    break;

	case 6:
	    DevState.SetWTon (Dev, State);
	    break;

	case 7:
	    DevState.SetTFE (Dev, State);
	    break;

	default:
	    WriteDiagMsg (FormatStr ("Unknown matrix col: %d", Col));
	    break;

    }

    // Update the matrix column
    RedrawMatrix ();
}



/*****************************************************************************/
/*				     Code				     */
/*****************************************************************************/



void InitMatrixWin (unsigned DevCount)
// Show the istec matrix window
{
    WinMgr->AddWindow (new MatrixWindow (Point (10, 10), DevCount));
}



void HandleDiagMsg (const unsigned char* Msg)
// Handle a diagnostic message from the istec
{
    // Check the event type
    String S;
    switch (Msg [1]) {

	case 0x02:
	    // connection matrix
	    SetMatrix (Msg [2], Msg [3], Msg [4]);
	    break;

	case 0x03:
	    // call
	    DevState.SetRuf (Msg [2], Msg [4]);
	    RedrawMatrix ();
	    break;

	case 0x04:
	    // Schleifenzustand
	    DevState.SetSchleife (Msg [2], Msg [4]);
	    RedrawMatrix ();
	    break;

	case 0x05:
	    // Pulse dial
	    WriteDialStatus (msPulseDial, Msg [2], Msg [4]);
	    break;

	case 0x06:
	    // Tone dial
	    WriteDialStatus (msToneDial, IntCon [Msg [2]], Msg [4]);
	    break;

	case 0x07:
	    // LED state
	    WriteDiagStatus (msLEDState, Msg [4]);
	    break;

	case 0x08:
	    // Charges
	    WriteDiagStatus (msChargeState, Msg [4]);
	    break;

	case 0x09:
	    // TFE amplifier
	    WriteDiagStatus (msTFEState, Msg [4]);
	    break;

	case 0x0A:
	    // Door opener
	    WriteDiagStatus (msDoorState, Msg [4]);
	    break;

	case 0x0D:
	    // Switch
	    WriteDiagStatus (msSwitchState, Msg [4]);
	    break;

	default:
	    // Unknown debug message!
	    S = FormatStr ("%02x %02x %02x %02x %02x",
			   Msg [1], Msg [2], Msg [3], Msg [4], Msg [5]);
	    WriteDiagMsg ("OOPS: " + S);
	    WriteDebugLog ("Warning: Got unknown diagnostic message: " + S);
	    break;

    }
}



void HandleCallQueue ()
// Handle messages in the call queue
{
    if (Queue.GetCount () > 0) {

	// We have a queue and there's something in it. Get first entry.
	DevStateInfo* DS = Queue.At (0);

	// Update the charges
	DS->CallCharges = Charges [DS->DevNum] - DS->StartCharges;

	// Get the log message
	String Msg = DS->LogMsg ();

	// Log the message into the logfiles and the window
	if (LogZeroCostCalls || DS->CallCharges > 0) {
	    LogCall (Msg, DS->CallStart, DS->DevNum);
	    WriteCallWin (Msg);
	}

	// Delete the entry
	Queue.AtDelete (0);

    }
}

