/*****************************************************************************/
/*                                                                           */
/*                                MEMCHECK.CC                                */
/*                                                                           */
/* (C) 1995     Ullrich von Bassewitz                                        */
/*              Zwehrenbuehlstrasse 33                                       */
/*              D-72070 Tuebingen                                            */
/* EMail:       uz@ibb.schwaben.com                                          */
/*                                                                           */
/*****************************************************************************/



// $Id$
//
// $Log$
//
//



// Poor man's memory checker. Overloads the global operators new and delete
// and does some additional checks if the variable MemCheck is set to true:
//
//      * Check if an allocated block is already allocated (heap corrupt)
//      * Check if a block that should be freed is allocated
//      * Check if there have been writes outside the blocks bounds (by
//        adding a signature to the end)
//      * Check if new does not provide a NULL pointer.



#include <stdio.h>

#include "coll.h"
#include "environ.h"
#include "memcheck.h"



/*****************************************************************************/
/*                                   Data                                    */
/*****************************************************************************/



// Signature of a memory block
static u32 MemSig = 0x12785634;

// Switch memory checking on or off
int MemCheck = 0;

// Switch memory filling on or off
int MemFill = 0;

// Statistics
u32 NewCount = 0;
u32 DelCount = 0;
u32 DelNULLCount = 0;
u32 NewCheckCount = 0;
u32 DelCheckCount = 0;



// This is the fill value for memory blocks if MemFill is true. On intel
// architectures, this is the code for "INT 3", an instruction that is
// often used by debuggers as a breakpoint.
const unsigned char FillVal = 0xCC;



/*****************************************************************************/
/*                             struct BlockInfo                              */
/*****************************************************************************/



struct BlockInfo {

    unsigned char*  Ptr;
    u32             Size;

};



/*****************************************************************************/
/*                      Explicit template instantiation                      */
/*****************************************************************************/



#ifdef EXPLICIT_TEMPLATES
template class Collection<BlockInfo>;
template class SortedCollection<BlockInfo, unsigned char*>;
#endif



/*****************************************************************************/
/*                            class BlockInfoColl                            */
/*****************************************************************************/



class BlockInfoColl: public SortedCollection<BlockInfo, unsigned char*> {

protected:
    virtual int Compare (unsigned char* const* Key1, unsigned char* const* Key2);
    virtual unsigned char* const * KeyOf (const BlockInfo* Item);

    // Allocate and deallocate item space
    virtual void** AllocItemSpace (int ItemCount);
    virtual void FreeItemSpace (void** Space);

public:
    BlockInfoColl (int aLimit, int aDelta);
    // Create a collection. Because there is only one (static) instance by
    // definition, the constructor also sets the default value for MemCheck
    // by evaluating an environment variable.

    virtual ~BlockInfoColl ();
    // Delete the collection. May check if the count of blocks still allocated
    // is zero.

};



BlockInfoColl::BlockInfoColl (int aLimit, int aDelta):
    SortedCollection<BlockInfo, unsigned char*> (aLimit, aDelta, 0)
// Create a collection. Because there is only one (static) instance by
// definition, the constructor also sets the default value for MemCheck
// by evaluating an environment variable.
{
    // Get the defaults for the memory checker
    MemCheck = GetEnvBool ("SPUNK_MEMCHECK", MemCheck);
    MemFill = GetEnvBool ("SPUNK_MEMFILL", MemFill);
}



BlockInfoColl::~BlockInfoColl ()
// Delete the collection. May check if the count of blocks still allocated
// is zero.
{
    // Switch off memory checking (not good if the collection is deleted :-)
    MemCheck = 0;

    // If the environment variable SPUNK_MEMLOGBLOCKS is set to something, use
    // this "something" as a filename to log a list of still allocated blocks
    String Name = GetEnvVar ("SPUNK_MEMLOGBLOCKS");
    if (!Name.IsEmpty ()) {
        MemLogBlocksInUse (Name);
    }
}



int BlockInfoColl::Compare (unsigned char* const* Key1, unsigned char* const* Key2)
{
    if (*Key1 < *Key2) {
        return -1;
    } else if (*Key1 > *Key2) {
        return 1;
    } else {
        return 0;
    }
}



unsigned char* const* BlockInfoColl::KeyOf (const BlockInfo* Item)
{
    return &Item->Ptr;
}



void** BlockInfoColl::AllocItemSpace (int ItemCount)
// Allocate and deallocate item space
{
    return (void**) malloc (ItemCount * sizeof (void*));
}



void BlockInfoColl::FreeItemSpace (void** Space)
// Allocate and deallocate item space
{
    free (Space);
}



/*****************************************************************************/
/*                                   Data                                    */
/*****************************************************************************/



// A static collection of memory blocks
static BlockInfoColl BIC (2000, 500);



/*****************************************************************************/
/*                                   Code                                    */
/*****************************************************************************/



void* operator new (size_t Size)
{
    // Count the calls to new
    NewCount++;

    unsigned char* Ptr;
    if (MemCheck) {

        // Count the checked calls
        NewCheckCount++;

        // Get a memory block
        Ptr = (unsigned char*) malloc (Size + sizeof (MemSig));

        // Make a signature at the end of the block
        memcpy (Ptr + Size, &MemSig, sizeof (MemSig));

        // Create a new BlockInfo struct. Do _not_ use new here to avoid
        // recursive calls to insert
        BlockInfo* BI = (BlockInfo*) malloc (sizeof (BlockInfo));

        // Insert the values
        BI->Ptr  = Ptr;
        BI->Size = Size;

        // Now search for the entry
        int Index;
        if (BIC.Search (&Ptr, Index) != 0) {
            // An item with this key exists. This means that the heap is
            // corrupted
            FAIL ("MemCheck: Duplicate block!");
        } else {
            // The returned pointer is not in the collection of already
            // allocated blocks, but it may point inside of an already
            // allocated block. Check this.
            // Note: Index is the index of the item _before the given
            // pointer, so simply check the range of the entry with index
            // Index.
            if (Index > 0) {
                // There is a block that's memory address is less than the
                // one returned by malloc
                const BlockInfo* BB = BIC.At (Index - 1);
                if (Ptr < BB->Ptr + BB->Size) {
                    // Pointer points inside the block below - heap corrupted
                    FAIL ("MemCheck: Heap corrupt!");
                }
            }

            // Heap ok, insert the new block
            BIC.AtInsert (Index, BI);
        }

    } else {

        // No memory checking. Allocate a memory block, but beware: New is
        // defined so that "new char [0]" points to a distinct object every
        // time it is called, so one cannot return NULL for a size of 0!
        Ptr = (unsigned char*) malloc (Size ? Size : 1);

    }

    // Check if we got memory, fail otherwise
    if (Ptr == NULL) {
        FAIL ("MemCheck: Out of memory");
    }

    // Fill the memory block if requested
    if (MemFill) {
        memset (Ptr, FillVal, Size);
    }

    // Return a pointer to the memory block
    return Ptr;
}



void operator delete (void* P)
{
    // Count the calls to delete
    DelCount++;

    // Deleting NULL pointers is always ok, nothing has to be done
    if (P == 0) {
        DelNULLCount++;
        return;
    }

    if (MemCheck) {

        // Count the calls
        DelCheckCount++;

        // Cast the pointer
        unsigned char* Ptr = (unsigned char*) P;

        // Search for the block
        int Index;
        if (BIC.Search (&Ptr, Index) != 0) {

            // The block exists. Check the signature, then delete it
            BlockInfo* BI = BIC.At (Index);
            if (memcmp (Ptr + BI->Size, &MemSig, sizeof (MemSig)) != 0) {
                // Signature overwritten
                FAIL ("MemCheck: Block signature overwritten");
            }

            // Fill the memory block if requested
            if (MemFill) {
                memset (Ptr, FillVal, BI->Size);
            }

            // Delete the entry
            BIC.AtDelete (Index);

            // Delete the info block
            free (BI);

            // Delete the memory block
            free (P);

        } else {
            // Trying to free a block that is not allocated
            FAIL ("MemCheck: Trying to free a block that is not allocated");
        }
    } else {

        // Free the block without checks
        free (P);

    }
}



u32 MemBlocksInUse ()
{
    return BIC.GetCount ();
}



void MemLogBlocksInUse (const String& Name)
{
    FILE* F = fopen (Name.GetStr (), "w+t");
    if (F == NULL) {
        // This is a debug function, so ignore the error
        return;
    }

    // Get the block count and log some statistics
    u32 Count = BIC.GetCount ();
    fprintf (F, "Blocks currently in use:               %8lu\n\n"
                "Calls to operator new:                 %8lu\n"
                "Calls to operator delete:              %8lu\n"
                "Checked calls to new:                  %8lu\n"
                "Checked calls to delete:               %8lu\n"
                "Calls to delete with a NULL arg:       %8lu\n\n",
                (unsigned long) Count,
                (unsigned long) NewCount,
                (unsigned long) DelCount,
                (unsigned long) NewCheckCount,
                (unsigned long) DelCheckCount,
                (unsigned long) DelNULLCount);

    // Log the blocks
    for (unsigned I = 0; I < Count; I++) {

        // Get a pointer to the block info
        const BlockInfo* BI = BIC.At (I);

        // Print a line describing the block (convert pointers to hex values)
        fprintf (F, "Block %5u: Loc = 0x%08lX, Size = %5lu\n",
                 I, (unsigned long) BI->Ptr, (unsigned long) BI->Size);

    }

    // Close the file
    fclose (F);
}

