/*****************************************************************************/
/*									     */
/*				     ISTEC.CC				     */
/*									     */
/* (C) 1995-96	Ullrich von Bassewitz					     */
/*		Zwehrenbuehlstrasse 33					     */
/*		D-72070 Tuebingen					     */
/* EMail:	uz@ibb.schwaben.com					     */
/*									     */
/*****************************************************************************/



// $Id$
//
// $Log$
//
//



#include <errno.h>
#include <stdio.h>
#include <stdlib.h>

#include "delay.h"
#include "screen.h"
#include "environ.h"
#include "syserror.h"
#include "filepath.h"
#include "filesel.h"
#include "fviewer.h"
#include "program.h"
#include "progutil.h"
#include "menuitem.h"
#include "menue.h"
#include "strcvt.h"
#include "stdmenue.h"
#include "stdmsg.h"
#include "memstrm.h"
#include "datetime.h"
#include "inifile.h"
#include "settings.h"

#include "icmsg.h"
#include "icerror.h"
#include "istec.h"
#include "icconfig.h"
#include "icdlog.h"
#include "iccom.h"
#include "iclog.h"
#include "icalias.h"
#include "icdevs.h"
#include "icident.h"
#include "icbaseed.h"
#include "icdiag.h"
#include "iccwin.h"
#include "icwinmgr.h"
#ifdef LINUX
#include "icimon.h"
#endif



/*****************************************************************************/
/*			      Message constants				     */
/*****************************************************************************/



const u16 msIstec			= MSGBASE_ISTEC +  0;
const u16 msAbout			= MSGBASE_ISTEC +  1;
const u16 msFile			= MSGBASE_ISTEC + 10;
const u16 msLoadFile			= MSGBASE_ISTEC + 11;
const u16 msSaveFile			= MSGBASE_ISTEC + 12;
const u16 msViewLog			= MSGBASE_ISTEC + 13;
const u16 msLoadIstec			= MSGBASE_ISTEC + 14;
const u16 msSaveIstec			= MSGBASE_ISTEC + 15;
const u16 msMakePerm			= MSGBASE_ISTEC + 16;
const u16 msReadAliases			= MSGBASE_ISTEC + 17;
const u16 msQuit			= MSGBASE_ISTEC + 18;

const u16 msConfig			= MSGBASE_ISTEC + 20;
const u16 msVersion			= MSGBASE_ISTEC + 21;
const u16 msSysParams			= MSGBASE_ISTEC + 22;
const u16 msDevParams			= MSGBASE_ISTEC + 23;
const u16 msReset			= MSGBASE_ISTEC + 24;

const u16 msCharges			= MSGBASE_ISTEC + 30;
const u16 msLoadCharges			= MSGBASE_ISTEC + 31;
const u16 msShowCharges			= MSGBASE_ISTEC + 32;
const u16 msPrintSettings		= MSGBASE_ISTEC + 33;
const u16 msPrintCharges		= MSGBASE_ISTEC + 34;
const u16 msResetCharges		= MSGBASE_ISTEC + 35;

const u16 msWindow			= MSGBASE_ISTEC + 40;
const u16 msOpen			= MSGBASE_ISTEC + 41;
const u16 msMatrixWin			= MSGBASE_ISTEC + 42;
const u16 msCallWin			= MSGBASE_ISTEC + 44;
const u16 msIMonWin			= MSGBASE_ISTEC + 45;
const u16 msTile			= MSGBASE_ISTEC + 46;
const u16 msCascade			= MSGBASE_ISTEC + 47;
const u16 msCloseAll			= MSGBASE_ISTEC + 48;
const u16 msRedraw			= MSGBASE_ISTEC + 49;
const u16 msResize			= MSGBASE_ISTEC + 50;
const u16 msZoom			= MSGBASE_ISTEC + 51;
const u16 msClose			= MSGBASE_ISTEC + 52;
const u16 msWindowList			= MSGBASE_ISTEC + 53;

const u16 msAboutInfo			= MSGBASE_ISTEC + 60;
const u16 msConnMehrgeraete		= MSGBASE_ISTEC + 61;
const u16 msConnAnlagen			= MSGBASE_ISTEC + 62;
const u16 msConnUnknown			= MSGBASE_ISTEC + 63;
const u16 msIstecConfig			= MSGBASE_ISTEC + 64;
const u16 msLoadFileHeader		= MSGBASE_ISTEC + 65;
const u16 msSaveFileHeader		= MSGBASE_ISTEC + 66;
const u16 msComPortNotOpen		= MSGBASE_ISTEC + 67;
const u16 msIstecTimeout		= MSGBASE_ISTEC + 68;
const u16 msWriteChanges		= MSGBASE_ISTEC + 69;
const u16 msAskMakePerm			= MSGBASE_ISTEC + 70;
const u16 msPrintHeader			= MSGBASE_ISTEC + 71;
const u16 msPrintHeader2		= MSGBASE_ISTEC + 72;
const u16 msChargeTableHeader		= MSGBASE_ISTEC + 73;
const u16 msChargeLine			= MSGBASE_ISTEC + 74;
const u16 msErrorComPortOpen		= MSGBASE_ISTEC + 75;
const u16 msRecBufOverflow		= MSGBASE_ISTEC + 76;
const u16 msRecBufUnderflow		= MSGBASE_ISTEC + 77;
const u16 msInvalidReply		= MSGBASE_ISTEC + 78;
const u16 msWrongDevice			= MSGBASE_ISTEC + 79;
const u16 msPrintFileSelHeader		= MSGBASE_ISTEC + 80;
const u16 msViewLogSelHeader		= MSGBASE_ISTEC + 81;
const u16 msSettingsFileError		= MSGBASE_ISTEC + 82;



/*****************************************************************************/
/*			       class IstecApp				     */
/*****************************************************************************/



IstecApp::IstecApp (int argc, char* argv []):
    Program (argc, argv, CreateMenueBar, CreateStatusLine, "estic"),
    RegularTask (RegularTask::rtFast),
    StatusFlags (0),
    ComPortName ("COM2"),
    SettingsFile ("estic.rc"),
    IstecPresent (1),
    Changes (0),
    WroteConfig (0),
    Currency (NLSData.CurrStr),
    VideoMode (vmAsk),
    ShowDateTime (1),
    LastUpdate (Now ()),
    DiagModeUpdateCounter (450),		// 7.5 min
    ChargesUpdateCounter (900)			// 15 min
{
    // Read the default values of some variables from the ini file
    ReadIniFile ();

    // Parse the command line
    int I = 1;
    while (I < ArgCount) {

	char* Item = ArgVec [I];
	if (*Item == '-') {

	    Item++;
	    switch (*Item) {

		case 'a':
		    PortBase = atoi (++Item);
		    break;

		case 'i':
		    PortIRQ = atoi (++Item);
		    break;

		case 'n':
		    IstecPresent = 0;
		    break;

		case 'p':
		    ComPortName = ++Item;
		    break;

	    }

	}

	// Next argument
	I++;
    }

    // Now switch the video mode
    ChangeVideoMode (VideoMode);

    // Try to open the debug logfile
    InitDebugLog (DebugLog);

    // Open the program option file
    if (!SettingsFile.IsEmpty ()) {
	// Ignore errors for now
	if (StgOpen (SettingsFile) != 0) {
	    ErrorMsg (LoadAppMsg (msSettingsFileError));
	}
    }

    // Try to initialize the com port if this was not prohibited
    if (IstecPresent) {

	// Assume no istec
	IstecPresent = 0;

	// Try to open the com port
	if (OpenComPort (ComPortName) != 0) {
	    // Port could not be opened
	    IstecError (msErrorComPortOpen);
	} else {
	    // Com port could be opened, check for the istec
	    if (EvalCmd (IstecReady ())) {
		// Istec is there
		IstecPresent = 1;
	    }

	}
    }

    // Read the complete configuration from the istec if possible, else reset
    // it to a known state
    InitIstecConfig ();

    // Load the window manager from the settings file
    WinMgr = (WindowManager*) StgGet ("WinMgr");
    if (WinMgr == NULL) {
	// Does not exist, create default
	WinMgr = new WindowManager;
    }

    // If the istec is not present, disable some of the menue choices, if
    // the istec is present, switch to debug mode if requested
    if (IstecPresent) {

	// Enable the istec diagnose messages
	EvalCmd (IstecDiagOn ());

    } else {
	// Disable some commands
	DisableCommand (miLoadIstec);
	DisableCommand (miSaveIstec);
	DisableCommand (miMatrixWin);
	DisableCommand (miCallWin);
	DisableCommand (miCharges);
    }

    // Display the istec configuration
    ShowIstecConfig ();

}



IstecApp::~IstecApp ()
// Destruct an application object
{
    // Close the com port
    CloseComPort ();

    // Delete the window manager
    delete WinMgr;

    // Close the settings file
    StgClose ();
}



void IstecApp::Work (const Time& Current)
// Idle function. Is used to check for debug messages in the receive queue.
// This function contains some hacks. No spunk program should rely on calls
// to App::Idle in a regular or even time based manner. But there is no
// other way to implement the needed background functions without using
// too much CPU resources. So I will do something dirty and use my knowledge
// about the internals of the KbdGet() and Delay() functions here...
// However, there are other solutions, but none of them is portable between
// the supported operating systems.
{
    // Get the system time and check if the time has changed
    TimeDiff Period = Current - LastUpdate;
    if (Period.GetSec () > 0) {

	// Check if the minute has changed
	int MinuteChange = Current.GetMin () != LastUpdate.GetMin ();

	// Remember the last update time
	LastUpdate = Current;

	// Check if we have to talk with the istec
	if (IstecPresent) {

	    // Update the time counters
	    DiagModeUpdateCounter -= Period.GetSec ();
	    ChargesUpdateCounter  -= Period.GetSec ();

	    // Check if there is work to do
	    if (DiagModeUpdateCounter <= 0) {
		DiagModeUpdateCounter += 15 * 60;
		EvalCmd (IstecDiagOn ());
	    }
	    if (ChargesUpdateCounter <= 0) {
		ChargesUpdateCounter += 15 * 60;
		EvalCmd (IstecRequestCharges ());
	    }

	    // If the minute has changed, call the cron handler
	    if (MinuteChange) {
		CronHandler ();
	    }
	}

	// If date/time display is selected, update that
	if (ShowDateTime) {
	    String S = Current.DateTimeStr (ShowDateTime > 1);
	    MainMenue->Write (MainMenue->MaxX () - S.Len (), 0, S);
	}

    }

    // Check for debug messages
    IstecPoll ();
}



void IstecApp::CronHandler ()
// Is called from idle every minute, checks periodic events
{
}



int IstecApp::GetNewCharges ()
// Request and wait for an update of the charge info. Return a result code.
{
    // Reset the flag
    ChargeUpdate = 0;

    // Send the command
    int Result = IstecRequestCharges ();

    if (Result == ieDone) {

	// Wait for the new charges
	unsigned I = 0;
	do {
	    Delay (200);
	} while (++I < 10 && ChargeUpdate == 0);

	// Check for a timeout
	if (ChargeUpdate == 0) {
	    // Timeout, pop up a message
	    Result = ieTimeout;
	} else {
	    Result = ieDone;
	}
    }

    return Result;
}



void IstecApp::ReadIniFile ()
// Read default settings from an ini file
{
    // Build the name of the ini file
    String IniName = GetProgName () + ".ini";

    // Search for the ini file in the following directories:
    //	-> the current dir
    //	-> the home directory if $HOME is defined
    //	-> the support path
    IniFile* F = new IniFile (IniName);
    if (F->GetStatus () != stOk) {
	// Not found, try the home dir
	delete F;
	String HomeDir = GetEnvVar ("HOME");
	AddPathSep (HomeDir);
	F = new IniFile (HomeDir + IniName);
	if (F->GetStatus () != stOk) {
	    // Ok, last resort: try the support path
	    delete F;
	    F = new IniFile (GetSupportPath () + IniName);
	    if (F->GetStatus () != stOk) {
		// No ini file
		delete F;
		return;
	    }
	}
    }

    // F points now to a valid ini file. Read the variables
    static const char* EsticSection = "ESTIC";
    static const char* PortSection = "Port";
    static const char* PrintSection = "Printing";
    static const char* WindowSection = "Windows";
    static const char* LogSection = "Call-Logs";
    static const char* AliasSection = "Alias";
    static const char* DebugSection = "Debug";
    static const char* FirmwareSection = "Firmware";

    SettingsFile = F->ReadString (EsticSection, "SettingsFile", SettingsFile);
    ComPortName = F->ReadString (PortSection, "PortName", ComPortName);
    PortBase = F->ReadInt (PortSection, "PortBase", PortBase);
    PortIRQ = F->ReadInt (PortSection, "PortIRQ", PortIRQ);
    Headline = F->ReadString (PrintSection, "Headline", Headline);
    Currency = F->ReadString (PrintSection, "Currency", Currency);
    PricePerUnit = F->ReadFloat (PrintSection, "PricePerUnit", PricePerUnit);
    VideoMode = F->ReadInt (WindowSection, "VideoMode", VideoMode);
    ShowDateTime = F->ReadInt (WindowSection, "ShowDateTime", ShowDateTime);
    Logfile1 = F->ReadString (LogSection, "Logfile1", Logfile1);
    Logfile2 = F->ReadString (LogSection, "Logfile2", Logfile2);
    Logfile3 = F->ReadString (LogSection, "Logfile3", Logfile3);
    LogZeroCostCalls = F->ReadInt (LogSection, "LogZeroCostCalls", LogZeroCostCalls);
    DebugWaitAfterCall = F->ReadInt (DebugSection, "WaitAfterCall", DebugWaitAfterCall);
    DebugLog = F->ReadString (DebugSection, "DebugLog", DebugLog);
    FirmwareVersion = F->ReadFloat (FirmwareSection, "FirmwareVersion", FirmwareVersion);
    AliasFile = F->ReadString (AliasSection, "AliasFile", AliasFile);
    AutoReadAliases = F->ReadInt (AliasSection, "AutoReadAliases", AutoReadAliases);

    // If an aliasfile is defined, read it. Otherwise try to read the aliases
    // from the ini file.
    if (AliasFile.IsEmpty ()) {
	// No aliasfile defined, read the device aliases from the ini file
	for (unsigned Dev = 21; Dev < 99; Dev++) {
	    String Alias = F->ReadString (AliasSection, U32Str (Dev), "");
	    if (!Alias.IsEmpty ()) {
		NewAlias (Dev, Alias);
	    }
	}
    } else {
	// Read the aliasfile
	ReadAliasFile ();

	// Enable the "Reread aliases" menu entry
	MainMenue->ActivateItem (miReadAliases);
    }

    // Now close the ini file
    delete F;
}



int IstecApp::LoadConfig ()
// Calls IstecGetConfig an returns the same codes but displays a message as
// this can last some time
{
    // Pop up a window
    Window* Win = PleaseWaitWindow ();

    // Load the stuff from the istec
    int Result = IstecGetConfig (BaseConfig, DevConfig);

    // If we could get the configuration, read also the charges
    if (Result == ieDone) {
	Result = GetNewCharges ();
    }

    // Delete the window
    delete Win;

    // Return the result
    return Result;
}



int IstecApp::StoreConfig ()
// Calls IstecPutConfig an returns the same codes but displays a message as
// this can last some time
{
    // Pop up a window
    Window* Win = PleaseWaitWindow ();

    // Load the stuff from the istec
    int Result = IstecPutConfig (BaseConfig, DevConfig, BaseConfig.AB_InterfaceCount);

    // Delete the window
    delete Win;

    if (Result == 1) {

	// Remember that we wrote a configuration
	WroteConfig = 1;
	Changes = 0;

	// Enable the "make permanent" selection
	EnableCommand (miMakePerm);

    }

    // Return the result
    return Result;
}



void IstecApp::InitIstecConfig ()
// Try to connect to the istec and download the istec configuration.
// If this is impossible, initialize the configuration data to known
// values.
{
    // If we could connect to the istec, try to download the configuration.
    if (IstecPresent == 0 || EvalCmd (LoadConfig ()) != ieDone) {

	// No istec or read error, use defaults
	IstecPresent = 0;
	LoadConfigDefault (BaseConfig, DevConfig, 1008);
	Charges.Clear ();

    }
}



unsigned IstecApp::GetIstecType ()
// Return the type of the istec, determined by the parameters of the base
// configuration.
{
    return IdentifyIstec (BaseConfig.AB_InterfaceCount,
			  BaseConfig.ExtS0,
			  BaseConfig.IntS0);
}



String IstecApp::GetIstecName ()
// Return the name of the istec, determined by the parameters of the base
// configuration.
{
    return ::GetIstecName (GetIstecType ());
}



const char* IstecApp::GetProtocolName (unsigned Prot)
// Return the protocol name used by the istec
{
    switch (Prot) {

	case pr1TR6:
	    return "1TR6";

	case prDSS1:
	    return "DSS1";

	default:
	    return "";
    }
}



const String& IstecApp::GetConnectionName (unsigned Conn)
// Return the connection type of the istec
{
    unsigned MsgNum;
    switch (Conn) {

	case coMehrgeraete:
	    MsgNum = msConnMehrgeraete;
	    break;

	case coAnlagen:
	    MsgNum = msConnAnlagen;
	    break;

	default:
	    MsgNum = msConnUnknown;
	    break;

    }

    return LoadAppMsg (MsgNum);
}



void IstecApp::ShowIstecConfig ()
// Show the istec configuration
{
    // Get some strings now to work around a gcc bug
    String IstecName = GetIstecName ();
    String ConnectionName = GetConnectionName (BaseConfig.Connection);

    // Set up the message to display
    String Msg =  FormatStr (
		    LoadAppMsg (msIstecConfig).GetStr (),
		    IstecName.GetStr (),
		    BaseConfig.VersionHigh, BaseConfig.VersionLow,
		    BaseConfig.ExtS0,
		    BaseConfig.IntS0,
		    BaseConfig.AB_InterfaceCount,
		    GetProtocolName (BaseConfig.Protocol),
		    ConnectionName.GetStr ()
		  );

    InformationMsg (Msg);

}



void IstecApp::DisableCommand (i16 ID)
// Disable the command bound to the menue item with the given ID
{
    // Gray the item
    MainMenue->GrayItem (ID);

    // Get the accel key of the item
    Key AccelKey = MainMenue->GetAccelKey (ID);

    // If this key is registered, unregister it
    if (AccelKey != kbNoKey && KeyIsRegistered (AccelKey)) {
	UnregisterKey (AccelKey);
    }

}



void IstecApp::EnableCommand (i16 ID)
// Enable the command bound to the menue item with the given ID
{
    // Enable the item
    MainMenue->ActivateItem (ID);

    // Get the accel key of the item
    Key AccelKey = MainMenue->GetAccelKey (ID);

    // If this key is not registered, do it
    if (AccelKey != kbNoKey && KeyIsRegistered (AccelKey) == 0) {
	RegisterKey (AccelKey);
    }
}



static MenueBarItem* GetMenueBarItem (u16 MsgID, i16 ItemID,
				      WindowItem* MenueList,
				      WindowItem* NextItem)
{
    return new MenueBarItem (App->LoadAppMsg (MsgID), ItemID, MenueList, NextItem);
}



static SubMenueItem* GetSubMenueItem (u16 MsgID, i16 ItemID,
				      Key AccelKey,
				      WindowItem* MenueList,
				      WindowItem* NextItem)
{
    return (SubMenueItem*) SetAccelKey (new SubMenueItem (App->LoadAppMsg (MsgID),
							  ItemID,
							   MenueList,
							   NextItem),
					AccelKey);
}



static MenueItem* GetMenueItem (u16 MsgID, i16 ItemID, Key AccelKey, WindowItem* NextItem)
{
    return (MenueItem*) SetAccelKey (new MenueItem (App->LoadAppMsg (MsgID),
						    ItemID,
						    NextItem),
				     AccelKey);
}



static MenueLine* GetMenueLine (WindowItem* NextItem)
{
    return new MenueLine (miNone, NextItem);
}



TopMenueBar* IstecApp::CreateMenueBar ()
{
    TopMenueBar* M = new TopMenueBar (
      GetMenueBarItem	(msIstec,	    miIstec,
	GetMenueItem	(msAbout,	    miAbout,		kbNoKey,
	NULL
      ),
      GetMenueBarItem	(msFile,	    miFile,
	GetMenueItem	(msLoadFile,	    miLoadFile,		kbNoKey,
	GetMenueItem	(msSaveFile,	    miSaveFile,		kbNoKey,
	GetMenueLine	(
	GetMenueItem	(msViewLog,	    miViewLog,		kbNoKey,
	GetMenueLine	(
	GetMenueItem	(msLoadIstec,	    miLoadIstec,	kbNoKey,
	GetMenueItem	(msSaveIstec,	    miSaveIstec,	kbNoKey,
	GetMenueItem	(msMakePerm,	    miMakePerm,		kbNoKey,
	GetMenueLine	(
	GetMenueItem	(msReadAliases,     miReadAliases,	kbNoKey,
	GetMenueLine	(
	GetMenueItem	(msQuit,	    miQuit,		vkQuit,
	NULL
      )))))))))))),
      GetMenueBarItem	(msConfig,	    miConfig,
	GetMenueItem	(msVersion,	    miVersion,		kbNoKey,
	GetMenueLine	(
	GetMenueItem	(msSysParams,	    miSysParams,	kbNoKey,
	GetMenueItem	(msDevParams,	    miDevParams,	kbNoKey,
	GetMenueLine	(
	GetMenueItem	(msReset,	    miReset,		kbNoKey,
	NULL
      )))))),
      GetMenueBarItem	(msCharges,	    miCharges,
	GetMenueItem	(msLoadCharges,     miLoadCharges,	kbNoKey,
	GetMenueItem	(msShowCharges,     miShowCharges,	kbNoKey,
	GetMenueLine	(
	GetMenueItem	(msPrintSettings,   miPrintSettings,	kbNoKey,
	GetMenueItem	(msPrintCharges,    miPrintCharges,	kbNoKey,
	GetMenueLine	(
	GetMenueItem	(msResetCharges,    miResetCharges,	kbNoKey,
	NULL
      ))))))),
      GetMenueBarItem	(msWindow,	    miWindow,
	GetSubMenueItem (msOpen,	    miOpen,		vkOpen,
	  GetMenueItem	(msMatrixWin,	    miMatrixWin,	kbNoKey,
	  GetMenueItem	(msCallWin,	    miCallWin,		kbNoKey,
	  GetMenueItem	(msIMonWin,	    miIMonWin,		kbNoKey,
	  NULL
	))),
	GetMenueItem	(msTile,	    miTile,		kbNoKey,
	GetMenueItem	(msCascade,	    miCascade,		kbNoKey,
	GetMenueItem	(msCloseAll,	    miCloseAll,		kbNoKey,
	GetMenueItem	(msRedraw,	    miRedraw,		kbNoKey,
	GetMenueLine	(
	GetMenueItem	(msZoom,	    miZoom,		vkZoom,
	GetMenueItem	(msResize,	    miResize,		vkResize,
	GetMenueItem	(msClose,	    miClose,		vkClose,
	GetMenueLine	(
	GetMenueItem	(msWindowList,	    miWindowList,	kbMeta0,
	NULL
      ))))))))))),
      NULL
    ))))));

    // Register the accel keys of the submenues
    App->RegisterKey (M->GetAccelKey (miIstec));
    App->RegisterKey (M->GetAccelKey (miFile));
    App->RegisterKey (M->GetAccelKey (miConfig));
    App->RegisterKey (M->GetAccelKey (miCharges));
    App->RegisterKey (M->GetAccelKey (miWindow));
    App->RegisterKey (M->GetAccelKey (miOpen));
    App->RegisterKey (M->GetAccelKey (miClose));

    // Register the accel keys
    App->RegisterKey (M->GetAccelKey (miQuit));
    App->RegisterKey (M->GetAccelKey (miWindowList));

    // Gray unused items
    M->GrayItem (miMakePerm);
    M->GrayItem (miReadAliases);
    M->GrayItem (miTile);
    M->GrayItem (miCascade);
    M->GrayItem (miCloseAll);
    M->GrayItem (miZoom);
    M->GrayItem (miResize);
    M->GrayItem (miClose);

#ifndef LINUX
    M->GrayItem (miIMonWin);
#endif

    // Return the result
    return M;
}



BottomStatusLine* IstecApp::CreateStatusLine ()
{
    const u32 Flags = siAltX_Exit;
    ((IstecApp*) App)->StatusFlags = Flags;
    return new BottomStatusLine (Flags);
}



void IstecApp::AskWriteChanges ()
// If there are differnces between the configuration stored in the istec and
// the configuration in memory, ask to store the config data into the istec.
{
    if (IstecPresent) {

	if (Changes) {

	    // We have changes
	    if (AskYesNo (LoadAppMsg (msWriteChanges)) == 2) {

		// We should write the changes...
		EvalCmd (StoreConfig ());

	    }
	}

    } else {

	// No istec present, assume changes are written
	Changes = 0;

    }
}



void IstecApp::AskMakePermanent ()
// If we have written a configuration to the istec, ask to make the changes
// permanent
{
    if (IstecPresent && WroteConfig) {

	if (AskYesNo (LoadAppMsg (msAskMakePerm)) == 2) {

	    // Changes should be made permanent
	    int Result;
	    EvalCmd (Result = IstecMakePermanent ());

	    // Reset flag and selection if operation was successful
	    if (Result == 1) {
		WroteConfig = 0;
		DisableCommand (miMakePerm);
	    }

	}
    }
}



int IstecApp::EvalCmd (int RetCode)
// Evaluate the return code of an istec command function. If the return
// code denotes an error, an error message is poped up. The function
// return 1 if the command has been error free, 0 on errors.
{
    switch (RetCode) {

	case ieRecBufOverflow:
	    // Receive buffer overflow
	    IstecError (msRecBufOverflow);
	    IstecErrorSync ();
	    return 0;

	case ieRecBufUnderflow:
	    // Receive buffer underflow
	    IstecError (msRecBufUnderflow);
	    IstecErrorSync ();
	    return 0;

	case ieInvalidReply:
	    // Invalid replay
	    IstecError (msInvalidReply);
	    IstecErrorSync ();
	    return 0;

	case ieWrongDevice:
	    // Wrong device number in reply
	    IstecError (msWrongDevice);
	    IstecErrorSync ();
	    return 0;

	case iePortNotOpen:
	    // COM port not open
	    IstecError (msComPortNotOpen);
	    return 0;

	case ieTimeout:
	    // Timeout
	    IstecError (msIstecTimeout);
	    return 0;

	case ieDone:
	    // Success
	    return 1;

	default:
	    FAIL ("IstecApp::EvalCmd: Unexpected return code");

    }

    // Never reached
    return 0;
}



void IstecApp::SetSaveDir (const String& Path)
// Strips the name part from the given filename and stores the remaining part
// (the directory) including the trailing path separator into SaveDir.
{
    String Name;
    FSplit (Path, SaveDir, Name);
}



void IstecApp::LoadFile ()
// Load the configuration from a file
{
    // Choose the file name
    FileSelector FS (LoadAppMsg (msLoadFileHeader), ".ic");
    String Sel = FS.GetChoice (SaveDir + "*.ic");
    if (!Sel.IsEmpty ()) {

	// Open the file
	FILE* F = fopen (Sel.GetStr (), "rb");
	if (F == NULL) {
	    ErrorMsg (GetSysErrorMsg (errno));
	    return;
	}

	// Read the configuration.
	unsigned char Buf [BaseConfigSize + IstecDevCount * DevConfigSize];
	fread (Buf, sizeof (Buf), 1, F);

	// Close the file
	fclose (F);

	// Unpack the configuration
	unsigned char* B = Buf;
	BaseConfig.Unpack (B);
	B += BaseConfigSize;
	for (unsigned I = 0; I < IstecDevCount; I++) {
	    DevConfig [I].Unpack (B);
	    B += DevConfigSize;
	}

	// Remember the directory used
	SetSaveDir (Sel);

	// Show the configuration just read
	ShowIstecConfig ();

	// Data has been changed
	Changes = 1;

	// Ask to write the changes to the istec
	AskWriteChanges ();
    }
}



void IstecApp::SaveFile ()
// Save the current configuration to a file
{
    unsigned char Buf [BaseConfigSize + IstecDevCount * DevConfigSize];
    unsigned char* B = Buf;

    // Pack the current configuration into the buffer
    BaseConfig.Pack (B);
    B += BaseConfigSize;
    for (unsigned I = 0; I < IstecDevCount; I++) {
	DevConfig [I].Pack (B);
	B += DevConfigSize;
    }

    // Choose the file name
    FileSelector FS (LoadAppMsg (msSaveFileHeader), ".ic", fsFileMayNotExist);
    String Sel = FS.GetChoice (SaveDir + "*.ic");
    if (!Sel.IsEmpty ()) {

	// Open the file
	FILE* F = fopen (Sel.GetStr (), "wb");
	if (F == NULL) {
	    ErrorMsg (GetSysErrorMsg (errno));
	    return;
	}

	// Write the configuration
	fwrite (Buf, sizeof (Buf), 1, F);

	// Close the file
	fclose (F);

	// Remember the directory used
	SetSaveDir (Sel);

    }
}



void IstecApp::ViewLog ()
// View a logfile
{
    // Ask for the file name
    FileSelector FS (LoadAppMsg (msViewLogSelHeader), ".log");
    String LogFile = FS.GetChoice ();
    if (LogFile.IsEmpty ()) {
	// User abort
	return;
    }

    // Open the file viewer
    FileViewer* Viewer = new FileViewer (LogFile,
					 Background->GetDesktop (),
					 wfFramed | wfCanMove | wfCanResize,
					 paBlue);

    // Check for errors, browse the file
    if (Viewer->GetStatus () != stOk) {
	// OOPS, error opening the file or something like that
	ErrorMsg (GetSysErrorMsg (Viewer->GetErrorInfo ()));
	delete Viewer;
    } else {
	// Ok, insert the file into the window manager
	WinMgr->AddWindow (Viewer);
	WinMgr->Browse (Viewer);
    }
}



void IstecApp::LoadIstec ()
// Load the configuration from the istec
{
    if (EvalCmd (IstecReady ())) {
	// Well, we _can_ talk to the istec, try to read the config
	if (EvalCmd (LoadConfig ())) {
	    // Error free, display the configuration
	    ShowIstecConfig ();
	}
    }
}



void IstecApp::SaveIstec ()
// Save the current config to the istec
{
    EvalCmd (StoreConfig ());
}



void IstecApp::SysParams ()
// Edit system parameters
{
    // Edit the parameters
    int NewChanges = 0;
    EditBaseConfig (BaseConfig, DevConfig, IstecPresent, NewChanges);

    // Remember if we had changes
    Changes = Changes || NewChanges;

    // Ask for writing the changes back if we have new changes
    if (NewChanges) {
	AskWriteChanges ();
    }
}



void IstecApp::DevParams ()
// Set the device parameters
{
    // Edit the parameters
    int NewChanges = 0;
    DeviceList (DevConfig, Charges, BaseConfig.AB_InterfaceCount, NewChanges);

    // Remember if we had changes
    Changes |= Changes || NewChanges;

    // Ask for writing the changes back if we have new changes
    if (NewChanges) {
	AskWriteChanges ();
    }
}



void IstecApp::Reset ()
// Reset the istec
{
    if (AskAreYouShure () == 2) {

	// Reset the device params
	LoadConfigDefault (BaseConfig, DevConfig, GetIstecType ());

	// We have changes now
	Changes = 1;

	// Ask to write those changes
	AskWriteChanges ();

    }
}



void IstecApp::LoadCharges ()
// Reload the charges from the istec
{
    // Pop up a window
    Window* Win = PleaseWaitWindow ();

    // Load the stuff from the istec
    int Result = GetNewCharges ();

    // Delete the window
    delete Win;

    // Evaluate the return code and display the charges if ok
    if (EvalCmd (Result)) {
	ShowCharges ();
    }
}



void IstecApp::ShowCharges ()
// Display an overview
{
    ShowDevCharges (DevConfig, Charges, BaseConfig.AB_InterfaceCount);
}



void IstecApp::PrintSettings ()
// Edit settings for printing charges
{
    // Name of the settings resource
    static const String StgPosName = "PrintSettings.PrintSettingsMenue.Position";

    // Load the menue
    Menue* M = (Menue*) LoadResource ("@ISTEC.PrintSettingsMenue");

    // If there is a stored window position, move the window to that position
    Point Pos = StgGetPoint (StgPosName, M->OuterBounds ().A);
    M->MoveAbs (Pos);

    // Remember the old values
    String OldHeadline	   = Headline;
    double OldPricePerUnit = PricePerUnit;

    // Transfer the current values to the menue
    M->SetStringValue (1, Headline);
    M->SetFloatValue (2, PricePerUnit);

    // Create a new status line
    PushStatusLine (siAbort | siSelectKeys | siAccept);

    // Activate the menue
    M->Activate ();

    // Accept user input
    int Done = 0;
    while (!Done) {

	// Get a selection
	int Sel = M->GetChoice ();

	// Evaluate the selection
	switch (Sel) {

	    case 1:
		Headline = M->GetStringValue (1);
		break;

	    case 2:
		PricePerUnit = M->GetFloatValue (2);
		break;

	    case 0:
		if (M->GetAbortKey () == vkAbort) {
		    // Abort - ask if we have changes
		    if (PricePerUnit != OldPricePerUnit || Headline != OldHeadline) {
			// We have changes
			if (AskDiscardChanges () == 2) {
			    // Discard changes
			    PricePerUnit = OldPricePerUnit;
			    Headline = OldHeadline;
			    Done = 1;
			}
		    } else {
			// No changes
			Done = 1;
		    }
		} else if (M->GetAbortKey () == vkAccept) {
		    // Accept the changes
		    Done = 1;
		}
		break;

	}

    }

    // Restore the old status line
    PopStatusLine ();

    // Save the current window position
    StgPutPoint (M->OuterBounds ().A, StgPosName);

    // Delete the menue
    delete M;
}



void IstecApp::PrintCharges ()
// Print the charges
{
    // Choose the file name
    FileSelector FS (LoadAppMsg (msPrintFileSelHeader), ".geb", fsFileMayNotExist);
    String Sel = FS.GetChoice (SaveDir + "*.geb");
    if (Sel.IsEmpty ()) {
	// Empty selection means abort
	return;
    }

    // Open the file
    FILE* F = fopen (Sel.GetStr (), "wt");
    if (F == NULL) {
	ErrorMsg (GetSysErrorMsg (errno));
	return;
    }

    // Some space
    fprintf (F, "\n\n");

    // Headline
    String Line = Headline;
    fprintf (F, "%s\n", Line.OutputCvt ().GetStr ());

    // Internal headline
    String TimeStr = Time ().DateTimeStr ();
    String IstecName = GetIstecName ();
    Line = FormatStr (LoadAppMsg (msPrintHeader).GetStr (),
		      IstecName.GetStr (),
		      TimeStr.GetStr ());
    fprintf (F, "%s\n", Line.OutputCvt ().GetStr ());

    // Divider
    Line = LoadAppMsg (msPrintHeader2).GetStr ();
    fprintf (F, "%s\n", Line.OutputCvt ().GetStr ());

    // Empty line followed by the table header
    Line = LoadAppMsg (msChargeTableHeader).GetStr ();
    fprintf (F, "\n%s\n\n", Line.OutputCvt ().GetStr ());

    // Ok, loop through the devices...
    for (unsigned Dev = 0; Dev < BaseConfig.AB_InterfaceCount; Dev++) {
	Line = LoadAppMsg (msChargeLine);
	Line = FormatStr (Line.GetStr (), Dev + 21, Charges [Dev]);
	String Price = FloatStr (Charges [Dev] * PricePerUnit, 4, 2);
	Price.Pad (String::Left, 7);
	Line += Price;
	fprintf (F, "%s\n", Line.OutputCvt ().GetStr ());
    }

    // Close the file
    fclose (F);

    // Remember the directory used
    SetSaveDir (Sel);

}



void IstecApp::ResetCharges ()
// Reset all charges
{
    if (AskAreYouShure () == 2) {

	// Create empty charges
	IstecCharges Charges;

	if (IstecPresent) {

	    // Pop up a window cause this could last some time
	    Window* WaitWin = PleaseWaitWindow ();

	    // Send the istec command
	    int RetCode = IstecPutCharges (Charges);

	    // Delete the window
	    delete WaitWin;

	    // Check the return code
	    EvalCmd (RetCode);

	}
    }
}



void IstecApp::CloseAll ()
// Close all windows
{
    WinMgr->CloseAll ();
}



void IstecApp::Resize (ItemWindow* Win)
// Resize a window
{
    if (Win) {
	Win->MoveResize ();
    }
}



void IstecApp::Zoom (ItemWindow* Win)
// Zoom a window
{
    if (Win && Win->CanResize ()) {
	Win->Zoom ();
    }
}



void IstecApp::Close (ItemWindow* Win)
// Close a window
{
    if (Win) {
	WinMgr->DeleteWindow (Win);
    }
}



int IstecApp::Run ()
{
    // Activate the main menue
    MainMenue->Activate ();

    // Main loop
    while (!Quitting ()) {

	Key K;

	// Switch according to the users choice
	switch (MainMenue->GetChoice ()) {

	    case miAbout:
		InformationMsg (LoadAppMsg (msAboutInfo));
		break;

	    case miLoadFile:
		LoadFile ();
		break;

	    case miSaveFile:
		SaveFile ();
		break;

	    case miViewLog:
		ViewLog ();
		break;

	    case miLoadIstec:
		LoadIstec ();
		break;

	    case miSaveIstec:
		SaveIstec ();
		break;

	    case miMakePerm:
		AskMakePermanent ();
		break;

	    case miReadAliases:
		ReadAliasFile ();
		break;

	    case miQuit:
		// Close the windows
		if (WinMgr->CanClose ()) {

		    // Save the window manager, then close all windows
		    StgPut (WinMgr, "WinMgr");
		    WinMgr->CloseAll ();

		    // Switch the istec out of diag mode (only if istec present)
		    if (IstecPresent) {
			EvalCmd (IstecDiagOff ());
		    }

		    // End the program. First ask to write any changes back
		    AskWriteChanges ();

		    // Now, if we have changes, ask to make them permanent
		    AskMakePermanent ();

		    // The end...
		    Quit = 1;
		}
		break;

	    case miVersion:
		ShowIstecConfig ();
		break;

	    case miSysParams:
		SysParams ();
		break;

	    case miDevParams:
		DevParams ();
		break;

	    case miReset:
		Reset ();
		break;

	    case miLoadCharges:
		LoadCharges ();
		break;

	    case miShowCharges:
		ShowCharges ();
		break;

	    case miPrintSettings:
		PrintSettings ();
		break;

	    case miPrintCharges:
		PrintCharges ();
		break;

	    case miResetCharges:
		ResetCharges ();
		break;

	    case miMatrixWin:
		InitMatrixWin (BaseConfig.AB_InterfaceCount);
		break;

	    case miCallWin:
		InitCallWin ();
		break;

#ifdef LINUX
	    case miIMonWin:
		WinMgr->AddWindow (new IMonWindow (Point (10, 10)));
		break;
#endif

	    case miTile:
		WinMgr->Tile ();
		break;

	    case miCascade:
		WinMgr->Cascade ();
		break;

	    case miCloseAll:
		CloseAll ();
		break;

	    case miRedraw:
		RedrawScreen ();
		break;

	    case miResize:
		Resize (WinMgr->GetTopWindow ());
		break;

	    case miZoom:
		Zoom (WinMgr->GetTopWindow ());
		break;

	    case miClose:
		Close (WinMgr->GetTopWindow ());
		break;

	    case miWindowList:
		WinMgr->Browse (WinMgr->ChooseWindow ());
		break;

	    case 0:
		K = MainMenue->GetAbortKey ();
		if (K != vkAbort) {
		    // Window hotkey
		    WinMgr->Browse (WinMgr->FindWindowWithKey (K));
		}
		break;

	}

    }

    // Close the debug logfile
    DoneDebugLog ();

    // Return the program exit code
    return 0;
}




int main (int argc, char* argv [])
{
    // Set the default language and country
    DefaultLanguage = laGerman;
    DefaultCountry = 49;

    // Declare an application object
    IstecApp MyApp (argc, argv);

    // Use it...
    return MyApp.Run ();

}




