// MPWriter.java
// $Id: MPWriter.java,v 1.1 1996/04/10 13:52:56 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.mux ;

import java.io.* ;
import java.net.* ;

class Chunker implements MUX {
    // For ggod band-width sharing, BUFSIZE should be kept smaller than
    // MUX_LENGTH (18 bits)
    private static final int BUFSIZE = 8192 ;
    private static final byte pad[] = {(byte) 0, (byte) 0, (byte) 0, (byte) 0};
    private static final boolean debug = false ;

    byte         buffer[] = new byte[BUFSIZE] ;
    int          buflen   = 0 ;
    OutputStream output   = null ;

    byte sheader[] = null ;
    byte bheader[] = null ;

    private final void trace (String msg) {
	if ( debug )
	    System.out.println (msg) ;
    }


    private final void dumpHeader(byte h[], int off) {
	if ( debug )
	    System.out.println ("h0="+h[off]
				+ " ,h1="+h[off+1]
				+ " ,h2="+h[off+2]
				+ " ,h3="+h[off+3]) ;
    }


    /**
     * Can we get this capacity from our buffer.
     * Make best effort (ie flush) to try getting the requested capacity. If
     * success, than return <strong>true</strong> otherwise, return 
     * <strong>false</strong>.
     * @param capacity Requested capacity.
     * @return A boolean <strong>true</strong if requested capacity is 
     *    available.
     * @exception IOException If flushing the buffer trigered some IO errors.
     */

    protected boolean ensureCapacity (int capacity) 
	throws IOException
    {
	if ( buflen + capacity < buffer.length ) {
	    return true ;
	} else if (buffer.length < capacity) {
	    flush() ;
	    return true ;
	} else {
	    return false ;
	}
    }
     
    /**
     * Encode a small message.
     * @param flags The header flags.
     * @param session The session.
     * @param len The message length.
     * @param into Target buffer.
     * @param dst Target buffer position.
     * @return Next offset in target buffer.
     */

    private final int encodeMessage (byte flags
				      , int session
				      , int length
				      , byte into[]
				      , int  dst) {
	into[dst++] = (byte) ((flags) | ((byte) (session & 0x3))) ;
	into[dst++] = (byte) ((session & 0xfc) | (length & 0x3)) ;
	into[dst++] = (byte) ((length & 0x3fc) >> 2) ;
	into[dst++] = (byte) ((length & 0x3fc00) >> 10) ;
dumpHeader (into, dst-4) ;
	return dst ;
    }

    /**
     * Encode a big message.
     * @param flags The header flags.
     * @param session The session identifier.
     * @param protocol The protocol identifier.
     * @param len The message length.
     */

    private final int encodeLongMessage (byte flags
					 , int session
					 , int protocol
					 , int length
					 , byte into[]
					 , int dst) {
	into[dst++] = (byte) ((flags) | ((byte) (session & 0x3))) ;
	into[dst++] = (byte) ((session & 0xfc) | (protocol & 0x3)) ;
	into[dst++] = (byte) ((protocol & 0x3fc) >> 2) ;
	into[dst++] = (byte) ((protocol & 0x3fc00) >> 10) ;
	into[dst++] = (byte) (length  & 0xff) ;
	into[dst++] = (byte) ((length & 0xff00) >> 8) ;
	into[dst++] = (byte) ((length & 0xff0000) >> 16) ;
	into[dst++] = (byte) ((length & 0xff000000) >> 24) ;
	return dst ;
    }

    /**
     * Write one chunk of data to the output.
     * The data is bufferized, to keep the number of write system calls 
     * low, except if the provided chunk length is really big, in which
     * case, the message is sent straight through.
     * @param session The session identifier this message belongs to.
     * @param buf The message to write.
     * @param off Offset in the buffer.
     * @param len The length of the buffer.
     * @exception IOException if something went wrong.
     */

    void writeChunk(int session, byte buf[], int off, int len) 
	throws IOException
    {
	writeMessage((byte) 0, session, -1, buf, off, len) ;
    }

    /**
     * Write one message to the output stream.
     */

    synchronized void writeMessage (byte flags
				    , int id
				    , int protid
				    , byte b[], int off, int len)
	throws IOException
    {
	int padlen = ((len & 0x3) != 0 ) ? (4 - (len & 0x3)) : 0 ;
	int netlen = len + padlen ;

	if ((len <=0x3ffff) && ((flags&MUX_LONG_LENGTH) != MUX_LONG_LENGTH) ) {
	    if ( ensureCapacity(netlen) ) {
		buflen = encodeMessage(flags, id, len, buffer, buflen);
		if ( len > 0 )
		    System.arraycopy (b, off, buffer, buflen, len) ;
		buflen += netlen ;
	    } else {
		flush() ;
		encodeMessage(flags, id, len, sheader, 0) ;
		output.write(sheader, 0, 4) ;
		if ( len > 0 )
		    output.write(b, off, len) ;
		output.write(pad, 0, padlen) ;
	    }
	} else {
	    if ( ensureCapacity(netlen) ) {
		buflen = encodeLongMessage((byte) flags
					   , id
					   , protid
					   , len
					   , buffer, buflen);
		if ( len > 0 )
		    System.arraycopy(b, off, buffer, buflen, len) ;
		buflen += netlen ;
	    } else {
		flush() ;
		encodeLongMessage(flags, id, protid, len, bheader, 0) ;
		output.write(bheader, 0, 8) ;
		if ( len > 0 )
		    output.write(b, off, len) ;
		output.write(pad, 0, padlen) ;
	    }
	}
    }

    synchronized void flush() 
	throws IOException
    {
trace ("Chunker: flushing "+buflen+" bytes.");	
	output.write(buffer, 0, buflen) ;
	buflen = 0 ;
    }

    Chunker(OutputStream output) {
	this.output  = output ;
	this.buffer  = new byte[BUFSIZE] ;
	this.buflen  = 0 ;
	this.sheader = new byte[4] ;
	this.bheader = new byte[4] ;
    }
}

/**
 * The multiplexed socket writer object.
 * Writing to the socket has to be done using an extra object, if we want
 * to take advantage of the simultaneous IO facility that threads offer.
 */

class MPWriter implements MUX {
    private static final int CHUNKSIZE = 128 ;

    private static final boolean debug = false ;

    OutputStream output  = null ;
    MPStream     stream  = null ;
    Chunker      chunker = null ;

    private final void trace (String msg) {
	if ( debug )
	    System.out.println (msg) ;
    }

    /**
     * Write one message of output.
     */

    protected synchronized void writeMessage (int session
					      , byte flags
					      , int protocol
					      , byte b[], int o, int l) 
	throws IOException
    {
	chunker.writeMessage (flags, session, protocol, b, o, l) ;
    }

    /**
     * Short cut to write data on a given session.
     */

    protected  void writeData (int session, byte b[], int o, int l) 
	throws IOException
    {
trace ("MPWriter: writing: " + l + " bytes.") ;
	int done = 0 ;
	// Chunk the data:
	for (done= 0 ; done <= l ; done += CHUNKSIZE) {
	    chunker.writeChunk (session, b, o, l) ;
	    o += CHUNKSIZE ;
	    l -= CHUNKSIZE ;
	}
	// Don't forget last chunk:
	if ( done < l ) {
	    int miss = l - done ;
	    chunker.writeChunk(session, b, o, miss) ;
	}
    }

    /**
     * Flush any pending output.
     */

    protected void flush() 
	throws IOException
    {
trace ("MPWriter: flush.") ;
	chunker.flush() ;
    }

    /**
     * Create a multiplexed writer on the given output stream.
     */

    MPWriter (MPStream stream, OutputStream output) 
	throws IOException
    {
	this.stream  = stream ;
	this.output  = output ;
	this.chunker = new Chunker(output) ;
    }

}
