// MIMEType.java
// $Id: MIMEType.java,v 1.1 1996/04/10 13:52:20 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.mime ;

import java.util.Vector ;

import java.io.DataOutputStream ;
import java.io.DataInputStream ;

/**
 * This class is used to represent parsed MIME types. 
 * It creates this representation from a string based representation of
 * the MIME type, as defined in the RFC 1345.
 */

public class MIMEType /* implements Savable */ {
    /** 
     * List of well known MIME types:
     */
    public static MIMEType TEXT_HTML_V3                      = null ;
    public static MIMEType TEXT_HTML_V2                      = null ;
    public static MIMEType TEXT_HTML                         = null ;
    public static MIMEType APPLICATION_POSTSCRIPT            = null ;
    public static MIMEType TEXT_PLAIN                        = null ;
    public static MIMEType APPLICATION_X_WWW_FORM_URLENCODED = null ;
    public static MIMEType MULTIPART_FORM_DATA               = null ;
    public static MIMEType APPLICATION_X_JAVA_AGENT          = null ;

    static {
	try {
	    TEXT_HTML_V3           
		= new MIMEType ("text/html;version=3.0") ;
	    TEXT_HTML_V2           
		= new MIMEType ("text/html") ;
	    TEXT_HTML = TEXT_HTML_V2 ;
	    APPLICATION_POSTSCRIPT 
		= new MIMEType ("application/postscript") ;
	    TEXT_PLAIN             
		= new MIMEType ("text/plain") ;
	    APPLICATION_X_WWW_FORM_URLENCODED
		= new MIMEType ("application/x-www-form-urlencoded") ;
	    MULTIPART_FORM_DATA
		= new MIMEType ("multipart/form-data") ;
	    APPLICATION_X_JAVA_AGENT
		= new MIMEType ("application/x-java-agent") ;
	} catch (MIMETypeFormatException e) {
	    System.out.println ("httpd.MIMEType: invalid static init.") ;
	    System.exit (1) ;
	}
    }

    public final int MATCH_TYPE          = 1 ;
    public final int MATCH_EXACT_TYPE    = 2 ;
    public final int MATCH_SUBTYPE       = 3 ;
    public final int MATCH_EXACT_SUBTYPE = 4 ;
    public final int MATCH_VERSION       = 5 ;
    public final int MATCH_EXACT_VERSION = 6 ;
    public final int MATCH_LEVEL         = 7 ;
    public final int MATCH_EXACT_LEVEL   = 8 ;

    protected String type    = null ;
    protected String subtype = null ;
    protected double version = Double.MIN_VALUE ;
    protected int    level   = -1 ;
    protected MIMETypeParameter parameters[] = null ;
    protected String external = null ;

    /**
     * How good the given MIMEType matches the receiver of the method ?
     *  This method returns a matching level among:
     * <dl><dt>TYPE<dd>Types match, through wildcards,</dd>
     * <dt>EXACT_TYPE<dd>Types match exactly,</dd>
     * <dt>SUBTYPE<dd>Types match, subtypes match through wildcards,</dd>
     * <dt>EXACT_SUBTYPE<dd>Types and subtypes match exactly,</dd>
     * <dt>VERSION<dd>Types and subtypes match, version ok</dd>
     * <dt>EXACT_VERSION<dd>Types and subtypes match, versions equal</dd>
     * <dt>LEVEL<dd>Types and subtypes match versions and level ok, </dd>
     * <dt>EXACT_LEVEL<dd>Types and subtypes match, versions and levels equals
     *     </dd>
     * </dl>
     * @param other The other MIMEType to match against ourself.
     */

    public int match (MIMEType other) {
	// match types:
	if ( type.equals("*") || other.type.equals("*") ) {
	    return MATCH_TYPE ;
	} else if ( ! type.equals (other.type) ) {
	    return 0 ;
	}
	// match subtypes:
	if ( subtype.equals("*") || other.subtype.equals("*") ) {
	    return MATCH_SUBTYPE ;
	} else if ( ! subtype.equals (other.subtype) ) {
	    return 0 ;
	}
	// match versions:
	if ((version==Double.MIN_VALUE) && (other.version==Double.MIN_VALUE)) {
	    return MATCH_EXACT_SUBTYPE ;
	} else if ( version < other.version ) {
	    return MATCH_EXACT_SUBTYPE ;
	} 
	    
	// match levels :
	if ( (level == -1) && (other.level ==-1) ) {
	    return ((version == other.version) 
		    ? MATCH_EXACT_VERSION 
		    : MATCH_VERSION) ;
	} else if ( level > other.level ) {
	    return MATCH_LEVEL ;
	} else if ( level == other.level ) {
	    return MATCH_EXACT_LEVEL ;
	}
	return MATCH_LEVEL ;
    }

    /**
     * A printable representation of this MIMEType. 
     * The printed representation is guaranteed to be parseable by the
     * String constructor.
     */

    public String toString () {
	if ( external == null ) {
	    StringBuffer sb = new StringBuffer (type) ;
	    sb.append((char) '/') ;
	    sb.append (subtype) ;
	    if ( ! (type.equals ("text") 
		    && subtype.equals ("html")
		    && (version == 2.0)
		    && (level == 1)) ) {
		if ( version != Double.MIN_VALUE )
		    sb.append (";version="+version) ;
		if ( level != -1 )
		    sb.append (";level="+level) ;
	    }
	    if ( parameters != null ) {
		for (int i = 0 ; i < parameters.length ; i++ )
		    sb.append (";"+parameters[i].getName()
			       +"="+parameters[i].getValue()) ;
	    }
	    external = sb.toString() ;
	}
	return external ;
    }


    /**
     * Does this MIME type has some value for the given parameter ?
     * @param name The parameter to check.
     * @return <strong>True</strong> if this parameter has a value, false
     *    otherwise.
     */

    public boolean hasParameter (String name) {
	if ( parameters != null ) {
	    for (int i = 0 ; i < parameters.length ; i++) {
		if ( parameters[i].getName().equals (name) ) 
		    return true ;
	    }
	}
	return false ;
    }

    /**
     * Get a mime type parameter value.
     * @param name The parameter whose value is to be returned.
     * @return The parameter value, or <b>null</b> if not found.
     */
    
    public String getParameterValue (String name) {
	if ( parameters != null ) {
	    for (int i = 0 ; i < parameters.length ; i++) {
		if ( parameters[i].getName().equals (name) ) 
		    return parameters[i].getValue () ;
	    }
	}
	return null ;
    }


    /**
     * Construct  MIMEType object for the given string.
     * The string should be the representation of the type. This methods
     * tries to be compliant with HTTP1.1, p 15, although it is not
     * (because of quoted-text not being accepted).
     * FIXME
     * @parameter spec A string representing a MIMEType
     * @return A MIMEType object
     * @exception MIMETypeFormatException, if the string couldn't be parsed.
     */

    public MIMEType (String spec)
	throws MIMETypeFormatException
    {
	int strl  = spec.length() ;
	int start = 0, look = -1 ;
	// skip leading/trailing blanks:
	while ((start < strl) && (spec.charAt (start)) <= ' ')
	    start++ ;
	while ((strl > start) && (spec.charAt (strl-1) <= ' '))
	    strl-- ;
	// get the type:
	StringBuffer sb = new StringBuffer () ;
	while ((start < strl) && ((look = spec.charAt(start)) != '/')) {
	    sb.append ((char) look) ;
	    start++ ;
	}
	if ( look != '/' ) 
	    throw new MIMETypeFormatException (spec) ;
	this.type = sb.toString() ;
	// get the subtype:
	start++ ;
	sb.setLength(0) ;
	while ((start < strl) 
	       && ((look = spec.charAt(start)) > ' ') && (look != ';')) {
	    sb.append ((char) look) ;
	    start++ ;
	}
	this.subtype = sb.toString() ;
	// get parameters, if any:
	while ((start < strl) && ((look = spec.charAt(start)) <= ' '))
	    start++ ;
	if ( start < strl ) {
	    if (spec.charAt(start) != ';') 
		throw new MIMETypeFormatException (spec) ;
	    start++ ;
	    Vector params = new Vector() ;
	    while ( start < strl ) {
		while ((start < strl) && (spec.charAt(start) <= ' ')) start++ ;
		// get parameter name:
		sb.setLength (0) ;
		while ((start < strl) 
		       && ((look=spec.charAt(start)) > ' ') && (look != '=')) {
		    sb.append (Character.toLowerCase((char) look)) ;
		    start++ ;
		}
		String name = sb.toString() ;
		// get the value:
		while ((start < strl) && (spec.charAt(start) <= ' ')) start++ ;
		if (spec.charAt(start) != '=') 
		    throw new MIMETypeFormatException (spec) ;
		start++ ;
		while ((start < strl) && (spec.charAt(start) <= ' ')) start++ ;
		sb.setLength(0) ;
		while ((start < strl) 
		       && ((look=spec.charAt(start)) > ' ') && (look != ';')) {
		    sb.append ((char) look) ;
		    start++ ;
		}
		while ((start < strl) && (spec.charAt(start) != ';')) start++ ;
		start++ ; 
		String value = sb.toString() ;
		// parse this parameter value if needed:
		if ( name.equals ("level") ) {
		    try {
			this.level = Integer.parseInt (value) ;
		    } catch (NumberFormatException e) {
			throw new MIMETypeFormatException (spec) ;
		    }
		} else if ( name.equals ("version") ) {
		    try {
			this.version = (Double.valueOf (value)).doubleValue() ;
		    } catch (NumberFormatException e) {
			throw new MIMETypeFormatException (spec) ;
		    }
		} else {
		    params.addElement (new MIMETypeParameter (name, value)) ;
		}
	    }
	    this.parameters = new MIMETypeParameter[params.size()] ;
	    params.copyInto (this.parameters) ;
	}
	// Fix HTML types, for backward compatiblity. I can't find back the
	// official ref for this hack :-(
	if ( this.type.equals ("text") 
	     && this.subtype.equals ("html")
	     && (this.version <= Double.MIN_VALUE) ) {
	    this.version = 2.0 ;
	    this.level   = 1 ;
	}
    }
    
    MIMEType (String type, String subtype
	      , int level
	      , MIMETypeParameter parameters[]) {
	this.type       = type ;
	this.subtype    = subtype ;
	this.level      = level ;
	this.parameters = parameters ;
    }

    public static void main (String args[]) {
	if ( args.length == 1) {
	    MIMEType type = null ;
	    try {
		type = new MIMEType (args[0]) ;
	    } catch (MIMETypeFormatException e) {
	    }
	    if ( type != null )
		System.out.println (type) ;
	    else
		System.out.println ("Invalid mime type specification.") ;
	} else {
	    System.out.println ("Usage: java MIMEType <type-to-parse>") ;
	}
    }

}
