// MIMEHeaders.java
// $Id: MIMEHeaders.java,v 1.1 1996/04/10 13:52:13 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.mime ;

import java.io.InputStream ;
import java.io.BufferedInputStream ;
import java.io.IOException ;

import java.util.Vector ;
import java.util.Date ;
import java.util.Hashtable ;
import java.util.StringTokenizer ;
import java.util.Enumeration ;

/**
 * This class is used as a container for MIME header fields.
 * It works in conjunction with the MIMEParser object. For efficiency reasons,
 * it keeps all headers value unparsed, until they are explicitly requested.
 * When such a thing happen, the header value gets parse to its native type
 * and svaed as such. All other calls for this header value will result in
 * the already parsed value.
 * @see w3c.mime.MIMEParser
 */

public class MIMEHeaders {
    // List of monthes, to parse Date:
    private static final Hashtable monthes = new Hashtable() ;
    static {
	monthes.put ("Jan", new Integer (0)) ;
	monthes.put ("Feb", new Integer (1)) ;
	monthes.put ("Mar", new Integer (2)) ;
	monthes.put ("Apr", new Integer (3)) ;
	monthes.put ("May", new Integer (4)) ;
	monthes.put ("Jun", new Integer (5)) ;
	monthes.put ("Jul", new Integer (6)) ;
	monthes.put ("Aug", new Integer (7)) ;
	monthes.put ("Sep", new Integer (8)) ;
	monthes.put ("Oct", new Integer (9)) ;
	monthes.put ("Nov", new Integer (10)) ;
	monthes.put ("Dec", new Integer (11)) ;
    }

    protected Hashtable  rheaders = null ;	// raw header values
    protected Hashtable  headers  = null ;	// parsed header values
    protected int        blength  = -1 ;	// body length

    protected void invalidFieldAccess (String name, String type) 
	throws MIMEException 
    {
	String msg = "Invalid request access: "+name+" is not "+type ;
	throw new MIMEException (msg) ;
    }

    /**
     * Dump all headers. Available for debug only.
     */

    public void dump () {
	Enumeration e = headers.keys() ;
	while ( e.hasMoreElements() ) {
	    String name = (String) e.nextElement() ;
	    System.out.println ("[parsed]"+name+"="+headers.get(name)) ;
	}
	e = rheaders.keys() ;
	while ( e.hasMoreElements() ) {
	    String name = (String) e.nextElement() ;
	    if ( headers.get (name) == null ) 
		System.out.println ("[raw   ]"+name+"="+rheaders.get(name));
	}
    }

    /**
     * Check if the given field has been defined for this request.
     * @parameter name A field name.
     * @return A boolean <b>true</b> if the field exists.
     */

    public boolean hasField (String name) {
	return ((rheaders.get ((Object) name) != null)
		|| (headers.get ((Object) name) != null)) ;
    }

    /** 
     * Define a new field, with the given value. If the field already exists
     * its old value is lost.
     * @parameter name The field name.
     * @parameter value The field value.
     */

    public void defineField (String name, Object value) {
	headers.put ((Object) name, value) ;
    }

    /** 
     * Get the raw value of a field. Before being parsed into their proper
     * types, field values are stored as a String.
     * @parameter name The field name.
     * @return The String handling the value or <strong>null</strong> if this
     *    header isn't defined.
     */
    
    public String getField (String name) {
	return (String) rheaders.get (name) ;
    }

    /** 
     * Get the parsed value of a field. If the field has not yet been parsed
     * triggers an exception.
     * @parameter name The field name.
     * @return The field value, as an Object, or <strong>null</strong> if the
     *    header isn't defined.
     */
    
    public Object getFieldObject (String name) {
	return headers.get (name) ;
    }

    /*
     * Parse the given String as a IETF Date. This version of parseDate is
     * more robust than the one found in Java Date class.
     * @param string The string to parse.
     * @exception MIMEException if date couldn't be parsed.
     */

    protected Long parseDate (String string) 
	throws MIMEException
    {
	StringTokenizer st     = new StringTokenizer (string, ", -:") ;
	int year, month, day, h, m, s ;
	
	if ( st.hasMoreTokens () ) {
	    try {
		String daystr = st.nextToken () ;
		if ( daystr.length() == 3 ) {	// rfc 822, or ANSI C
		    String next = st.nextToken() ;
		    switch (next.charAt(0)) {
		      case '0': case '1': case '2': case '3': case '4':
		      case '5': case '6': case '7': case '8': case '9': // 822
			day   = Integer.parseInt (next) ;
			month = ((Integer) monthes.get(next)).intValue() ;
			year  = Integer.parseInt (st.nextToken()) ;
			h 	  = Integer.parseInt (st.nextToken()) ;
			m 	  = Integer.parseInt (st.nextToken()) ;
			s 	  = Integer.parseInt (st.nextToken()) ;
			break ;
		      default:					        // ANSI
			month = ((Integer) monthes.get(next)).intValue() ;
			day   = Integer.parseInt (st.nextToken()) ;
			h     = Integer.parseInt (st.nextToken()) ;
			m     = Integer.parseInt (st.nextToken()) ;
			s     = Integer.parseInt (st.nextToken()) ;
			year  = Integer.parseInt (st.nextToken()) ;
			break ;
		    }
		} else {					         // 850
		    day   = Integer.parseInt (st.nextToken()) ;
		    month = ((Integer) monthes.get(st.nextToken())).intValue();
		    year  = Integer.parseInt (st.nextToken()) ;
		    h     = Integer.parseInt (st.nextToken()) ;
		    m     = Integer.parseInt (st.nextToken()) ;
		    s     = Integer.parseInt (st.nextToken()) ;
		}
		return new Long(Date.UTC (year, month, day, h, m, s)) ;
	    } catch (Exception e) {	// Null Pointer, NumberFormat
		throw new MIMEException ("Invalid date.") ;
	    }
	}
	throw new MIMEException ("Invalid date.") ;
    }

    /*
     * Parse the given String as an Integer object.
     * @param string The String to be parsed as an integer.
     * @exception MIMEException If the requested header isn't define, or if
     *    the integer couldn't be parsed.
     * @see java.lang.Integer
     */
    
    protected Integer parseInteger (String string)
	throws MIMEException 
    {
	int i ;
	try {
	    i = Integer.parseInt (string) ;
	} catch (NumberFormatException e) {
	    throw new MIMEException ("invalid Integer format") ;
	} 
	return new Integer (i) ;
    }

    /*
     * Parse the given String as a list of strings.
     * @param string The String to be parsed as a list.
     * @param sep The list separator.
     * @exception MIMEException If the requested header wasn't defined, or if
     *    the list couldn't be parsed.
     */ 
     
    protected String[] parseList (String string, char sep) 
	throws MIMEException
    {
	Vector items  = new Vector () ;
	int strl = string.length () ;
	int spos = 0 ;
	int epos = -1 ;
	int look ;
	// skip leading spaces:
	while (string.charAt (spos) <= ' ') {
	    if ( ++spos >= strl )
		return null ;
	}
    listloop:
	while ( (epos = string.indexOf (sep, spos)) != -1 ) {
	    int save = epos ;
	    // skip trailing spaces:
            for (--epos ; string.charAt (epos) <= ' ' ; ) {
		if ( --epos == spos ) 
		    continue listloop ;
	    }
	    // get item:
	    items.addElement (string.substring (spos, epos+1)) ;
	    // skip leading spaces of next item:
	    for (epos=save+1 ; string.charAt (epos) <= ' ' ; ) {
		if ( ++epos >= strl )
		    break listloop ;
	    }
	    spos = epos ;
	}
	// skip last item trailing spaces:
	for (epos=strl-1 ; string.charAt (epos) <= ' ' ; ) { 
	    if ( --epos == spos ) 
		break ;
	}
	if ( epos != spos )
	    items.addElement (string.substring (spos, epos+1)) ;
	String list[] = new String[items.size()] ;
	items.copyInto (list) ;
	return list ;
    }

    /**
     * Get the value of a Date field.
     * @parameter name field name.
     * @return A long, giving the UTC parsed date
     * @exception MIMEException If the requested header isn't define, or if
     *     the date couldn't be parsed.
     * @see java.util.Date
     */

    public long getFieldDate (String name) 
	throws MIMEException
    {
	Object value = headers.get ((Object) name) ;
	if ( value != null ) {
	    if ( value instanceof Long ) {
		return ((Long) value).longValue() ;
	    } else {
		invalidFieldAccess (name, "Date") ;
		return (long) -1 ;				// not reached
	    }
	} else {
	    String body = getField (name) ;
	    if ( body == null )
		invalidFieldAccess(name, "Date") ;
	    Long   l = null ;
	    try {
		l = parseDate (body) ;
	    } catch (MIMEException e) {
		invalidFieldAccess (name, "Date") ;
		return (long) -1 ;				// not reached
	    } 
	    headers.put ((Object) name, (Object) l) ;
	    return l.longValue() ;
	}
    }

    /**
     * Get the value of a list field. 
     * List values are represented as an array of strings. 
     * @parameter name field name.
     * @return An array of String, one per item.
     * @exception MIMEException If the requested header isn't define, or if the
     *     list couldn't be parsed.
     */

    public String[] getFieldList (String name) 
	throws MIMEException
    {
	Object value = headers.get (name) ;
	if ( value != null ) {
	    if ( value instanceof String[] ) {
		return (String[]) value ;
	    } else {
		invalidFieldAccess (name, "String[]") ;
		return null ;					// not reached
	    }
	} else {
	    String list[] ;
	    String body = getField (name) ;
	    try {
		list = parseList (body, ',') ;
	    } catch (MIMEException e) {
		invalidFieldAccess (name, "String[]") ;
		return null ;					// not reached
	    }
	    headers.put ((Object) name, (Object) list) ;
	    return list ;
	}
    }

    /**
     * Get the value of an integer field. 
     * @parameter name field name.
     * @return An array of Accept objects.
     * @exception MIMEException If the requested header isn't define, or if
     *    the integer couldn't be parsed.
     */

    public int getFieldInt (String name) 
	throws MIMEException
    {
	Object value = headers.get ((Object) name) ;
	if ( value != null ) {
	    if ( value instanceof Integer ) {
		return ((Integer) value).intValue() ;
	    } else {
		invalidFieldAccess (name, "Integer") ;
		return -1 ;					// not reached
	    }
	} else {
	    Integer i ;
	    String body = getField (name) ;
	    try {
		i = parseInteger (body) ;
	    } catch (MIMEException e) {
		invalidFieldAccess (name, "Integer") ;
		return -1 ;					// not reached
	    } 
	    headers.put ((Object) name, (Object) i) ;
	    return i.intValue() ;
	}
    }

    /**
     * Get the value of an string field. 
     * @parameter name field name.
     * @return An array of Accept objects.
     * @exception MIMEException If the requested header isn't define.
     * @see java.lang.String
     */

    public String getFieldString (String name) 
	throws MIMEException
    {
	Object value = headers.get (name) ;
	if ( value != null ) {
	    if ( value instanceof String ) {
		return (String) value ;
	    } else {
		invalidFieldAccess (name, "String") ;
		return null ;					// not reached
	    }
	} else {
	    String string = getField(name);
	    if ( string == null ) {
		invalidFieldAccess(name, "doesnt' exist");
		return null ;
	    }
	    headers.put ((Object) name, (Object) string) ;
	    return string ;
	}
    }

    /**
     * Get the value of the accept header. The accept header is parsed as an
     * array of Accept objects (see negociator.Accept).
     * @param name The name of the field to get.
     * @return An Accept object.
     * @exception MIMEException If the field wasn't defined, or if the 
     *    header couldn't be parsed.
     * @see w3c.mime.Accept
     */

    public Accept[] getFieldAccept (String name) 
	throws MIMEException
    {
	Object value = headers.get (name) ;
	if ( value != null ) {
	    if ( value instanceof Accept[] ) {
		return (Accept[]) value ;
	    } else {
		invalidFieldAccess (name, "Accept[]") ;
		return null ;					// not reached
	    }
	} else {
	    String list[]   = getFieldList (name) ;
	    Accept accept[] = new Accept[list.length] ;
	    for (int i = 0 ; i < list.length ; i++) {
		try {
		    accept[i] = new Accept (list[i]) ;
		} catch (MIMETypeFormatException e) {
		    invalidFieldAccess (name, "Accept[]") ;
		    return null ;				// not reached
		}
	    }
	    headers.put ((Object) name, (Object) accept) ;
	    return accept ;
	}
    }

    /**
     * Get the value of a MIMEType header. 
     * The given header is parsed and an appropriate MIMEType is returned.
     * @param name The name of the requested header.
     * @return A MIMEType object.
     * @exception MIMEEXception If the requested header wasn't defined, of if 
     *    the field couldn't be parsed.
     * @see w3c.mime.MIMEType
     */

    public MIMEType getFieldMIMEType (String name) 
	throws MIMEException
    {
	Object value = headers.get (name) ;
	if ( value != null ) {
	    if ( value instanceof MIMEType ) {
		return (MIMEType) value ;
	    } else {
		invalidFieldAccess (name, "MIMEType") ;
		return null ;					// not reached
	    }
	} else {
	    String   body = getField (name) ;
	    MIMEType type = null ;
	    try {
		type = new MIMEType (body) ;
	    } catch (MIMETypeFormatException e) {
		invalidFieldAccess (name, "MIMEType") ;
		return null ;					// not reached 
	    }
	    headers.put (name, type) ;
	    return type ;
	}
    }

    /**
     * Get the value of some field as an array of languages.
     * @param name The field's name.
     * @return An array of language objects.
     * @exception MIMEException If the requested field wasn't defined, or if
     *    field couldn't be parsed properly.
     * @see w3c.mime.Language
     */

    public Language[] getFieldLanguage (String name)
	throws MIMEException 
    {
	Object value = headers.get (name) ;
	if ( value != null ) {
	    if ( value instanceof Language[] ) {
		return (Language[]) value ;
	    } else {
		invalidFieldAccess (name, "Language[]") ;
		return null ;					// not reached
	    }
	} else {
	    String   list[]  = getFieldList (name) ;
	    Language langs[] = new Language[list.length] ;
	    try {
		for (int i = 0 ; i < list.length ; i++) {
		    langs[i] = new Language (list[i]) ;
		}
	    } catch (LanguageFormatException e) {
		invalidFieldAccess (name, "Language[]") ;
		return null ;					// not reached
	    }
	    headers.put (name, langs) ;
	    return langs ;
	}
    }

    /**
     * Get the value of a field as a list.
     * The list of keyword and optional value should be separated by a
     * <strong>;</strong>.
     * @param name The field's name.
     * @return A Hashtable containing the associations.
     * @exception MIMEParserException if the field couldn't be parsed.
     * @see java.util.Hashtable
     */

    public Hashtable getFieldHashtable (String name) 
	throws MIMEException
    {
	Object value = headers.get (name) ;
	if ( value != null ) {
	    if ( value instanceof Hashtable ) {
		return (Hashtable) value ;
	    } else {
		invalidFieldAccess (name, "Hashtable") ;
		return null ;					// not reached
	    }
	} else {
	    String list[]  = parseList (getField(name), ';') ;
	    Hashtable atts = new Hashtable (5) ;
	iloop:
	    for (int i = 0 ; i < list.length ; i++) {
		String item = list[i] ;
		int    ilen = item.length() ;
		int    ieq  = item.indexOf('=') ;
		if ( ieq < 0 ) {
		    // Key with no value
		    atts.put (item, "<none>") ;
		} else {
		    int ekey = ieq - 1 ; // end of key
		    int sval = ieq + 1 ; // start of value
		    while ((ekey > 0) && (item.charAt(ekey) <= ' '))
			ekey-- ;
		    while ((sval < ilen) && (item.charAt(sval)<= ' '))
			sval++ ;
		    String key = item.substring (0, ekey+1) ;
		    String val = null ;
		    if (item.charAt(sval) == '\"') {
			int eval = ++sval ;
			while ((eval < ilen) && (item.charAt(eval) != '\"'))
			    eval++ ;
			val = item.substring (sval, eval) ;
		    } else {
			val = item.substring (sval) ;
		    }
		    atts.put (key, val) ;
		}
	    }
	    headers.put (name, atts) ;
	    return atts ;
	}
    }

    /**
     * Create a new MIMEHeader instance.
     * The instance gets its parsed fields and its raw (unparsed) fields from
     * the two hashtables.
     * @param headers Parsed header values, indexed by header names.
     * @param rheaders Unparsed header values, indexed by header names.
     */

    public MIMEHeaders (Hashtable headers, Hashtable rheaders) {
	this.headers  = headers ;
	this.rheaders = rheaders ;
    }
}
