// HTTPResource.java
// $Id: HTTPResource.java,v 1.1 1996/04/10 13:49:59 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.resources ;

import java.io.*;

import w3c.jigsaw.http.* ;
import w3c.mime.* ;

/**
 * The basic HTTP resource.
 * Defines a bunch of attributes for all HTTP resources, and all the HTTP
 * method that provides access to them.
 */

public class HTTPResource extends Resource {
    /**
     * Attribute index - The index for our parent attribute.
     */
    protected static int ATTR_PARENT = -1 ;
    /**
     * Attribute index - The index for our URL attribute.
     */
    protected static int ATTR_URL = -1;
    /**
     * Attribute index - The index for our server attribute.
     */
    protected static int ATTR_SERVER = -1 ;
    /**
     * Attributes index - The index for the quality attribute.
     */
    protected static int ATTR_QUALITY = -1 ;
    /**
     * Attribute index - The index for the title attribute.
     */
    protected static int ATTR_TITLE = -1 ;
    /**
     * Attribute index - The index for the content languages attribute.
     */
    protected static int ATTR_CONTENT_LANGUAGE = -1 ;
    /**
     * Attribute index - The index for the content encodings attribute.
     */
    protected static int ATTR_CONTENT_ENCODING = -1 ;
    /**
     * Attribute index - The index for the content type attribute.
     */
    protected static int ATTR_CONTENT_TYPE = -1 ;
    /**
     * Attribute index - The index for the content length attribute.
     */
    protected static int ATTR_CONTENT_LENGTH = -1 ;
    /**
     * Attribute index - The index for the last-modified attribute.
     */
    protected static int ATTR_LAST_MODIFIED = -1 ;
    /**
     * Attribute index - The index for the expires attribute.
     */
    protected static int ATTR_EXPIRES = -1 ;
    /**
     * Attribute index - The icon (if any) associated to the resource.
     */
    protected static int ATTR_ICON = -1 ;
    /**
     * Attribute index - Max age: the maximum drift allowed from reality.
     */
    protected static int ATTR_MAXAGE = -1 ;

    static {
	Attribute a   = null ;
	Class     cls = null ;
	// Get a pointer to our class:
	try {
	    cls = Class.forName("w3c.jigsaw.resources.HTTPResource") ;
	} catch (Exception ex) {
	    ex.printStackTrace() ;
	    System.exit(1) ;
	}
	// Our parent resource (the one that created us)
	a = new ObjectAttribute("parent"
				, "w3c.jigsaw.resources.Resource"
				, null
				, Attribute.COMPUTED|Attribute.DONTSAVE);
	ATTR_PARENT = AttributeRegistery.registerAttribute(cls, a) ;
	// Our URL
	a = new StringAttribute("url"
				, null
				, Attribute.COMPUTED|Attribute.DONTSAVE);
	ATTR_URL = AttributeRegistery.registerAttribute(cls, a) ;
	// The server we are attached to
	a = new ObjectAttribute("server"
				, "w3c.jigsaw.http.httpd"
				, null
				, Attribute.COMPUTED|Attribute.DONTSAVE);
	ATTR_SERVER = AttributeRegistery.registerAttribute(cls, a) ;
	// The quality attribute:
	a = new DoubleAttribute("quality"
				, new Double(1.0) 
				, Attribute.EDITABLE);
	ATTR_QUALITY = AttributeRegistery.registerAttribute(cls, a) ;
	// The title attribute:
	a = new StringAttribute("title"
				, null
				, Attribute.EDITABLE) ;
	ATTR_TITLE = AttributeRegistery.registerAttribute(cls, a) ;
	// The content language attribute:
	a = new LanguageAttribute("content-language"
				  , null
				  , Attribute.EDITABLE) ;
	ATTR_CONTENT_LANGUAGE = AttributeRegistery.registerAttribute(cls,a);
	// The content encoding attribute:
	a = new EncodingAttribute("content-encoding"
				  , null
				  , Attribute.EDITABLE) ;
	ATTR_CONTENT_ENCODING = AttributeRegistery.registerAttribute(cls,a);
	// The content type attribute:
	a = new MIMETypeAttribute("content-type"
				  , MIMEType.TEXT_PLAIN
				  , Attribute.EDITABLE) ;
	ATTR_CONTENT_TYPE = AttributeRegistery.registerAttribute(cls,a);
	// The content length attribute:
	a = new IntegerAttribute("content-length"
				 , null
				 , Attribute.COMPUTED);
	ATTR_CONTENT_LENGTH = AttributeRegistery.registerAttribute(cls,a);
	// The last modified attribute:
	a = new DateAttribute("last-modified"
			      , null
			      , Attribute.COMPUTED|Attribute.EDITABLE) ;
	ATTR_LAST_MODIFIED = AttributeRegistery.registerAttribute(cls,a);
	// The expires attribute:
	a = new DateAttribute("expires"
			      , null
			      , Attribute.EDITABLE) ;
	ATTR_EXPIRES = AttributeRegistery.registerAttribute(cls,a);
	// The icon attribute:
	a = new StringAttribute("icon"
				, null
				, Attribute.EDITABLE) ;
	ATTR_ICON = AttributeRegistery.registerAttribute(cls, a) ;
	// The max age attribute (in ms)
	a = new LongAttribute("maxage"
			      , null
			      , Attribute.EDITABLE) ;
	ATTR_MAXAGE = AttributeRegistery.registerAttribute(cls, a) ;
    }

    /**
     * Set an HTTPResource attribute.
     * Mark modified should also update our last-modification time.
     * @param idx The index of the value to be set.
     * @param value Its new value.
     */

    public void markModified() {
	super.markModified() ;
	values[ATTR_LAST_MODIFIED] = new Long(System.currentTimeMillis());
    }

    /**
     * Get this resource parent resource.
     * The parent of a resource can be either <strong>null</strong> if it is
     * the server root resource, or any ContainerResource.
     * @return An instance of ContainerResource, or <strong>null</strong>
     */

    public ContainerResource getParent() {
	return (ContainerResource) getValue(ATTR_PARENT, null) ;
    }

    /**
     * Get the URL this resource is attached to.
     * @return An URL object specifying the location in the information 
     *    space of this resource.
     */

    public String getURL() {
	return getString(ATTR_URL, null) ;
    }

    /**
     * Get the server this resource is served by.
     * @return The first instance of Jigsaw this resource was attached to.
     */

    public httpd getServer() {
	return (httpd) getValue(ATTR_SERVER, null) ;
    }

    /**
     * Get this resource quality.
     * @return The resource quality, or some negative value if not defined.
     */

    public double getQuality() {
	return getDouble(ATTR_QUALITY, -1.0) ;
    }

    /**
     * Get this resource title.
     * @return This resource's title, or <strong>null</strong> if not 
     *    defined.
     */

    public String getTitle() {
	return getString(ATTR_TITLE, null) ;
    }

    /**
     * Get this resource content language.
     * Language are stored as a comma separated String of tokens.
     * @return A comma separated string of language tokens, or
     *    <strong>null</strong> if undefined.
     */

    public String getContentLanguage() {
	return (String) getValue(ATTR_CONTENT_LANGUAGE, null) ;
    } 

    /**
     * Get this resource content encoding.
     * The content encoding of a resource is stored as a comma separated
     * list of tokens (as decribed in the Content_encoding header of the
     * HTTP specification, and in the order they should appear in the header).
     * @return A string of comma separated encoding tokens, or
     *    <strong>null</strong> if not defined.
     */

    public String getContentEncoding() {
	String def = (String) attributes[ATTR_CONTENT_ENCODING].getDefault();
	return (String) getString (ATTR_CONTENT_ENCODING, def) ;
    }

    /**
     * Get this resource content type.
     * @return An instance of MIMEType, or <strong>null</strong> if not
     *    defined.
     */

    public MIMEType getContentType() {
	MIMEType def = (MIMEType) attributes[ATTR_CONTENT_TYPE].getDefault() ;
	return (MIMEType) getValue (ATTR_CONTENT_TYPE, def) ;
    }

    /**
     * Get this resource content length.
     * @return The resource content length, or <strong>-1</strong> if not
     *    defined.
     */

    public int getContentLength() {
	return getInt(ATTR_CONTENT_LENGTH, -1) ;
    }

    /**
     * Get this resource last modification time.
     * @return A long giving the date of the last modification time, or
     *    <strong>-1</strong> if undefined.
     */

    public long getLastModified() {
	return getLong(ATTR_LAST_MODIFIED, (long) -1) ;
    }

    /**
     * Get this resource expiration time.
     * @return A long giving the expires date, or <strong>-1</strong> if
     *    undefined.
     */

    public long getExpires() {
	return getLong(ATTR_EXPIRES, (long) -1) ;
    }

    /**
     * Get this resource's icon.
     */

    public String getIcon() {
	return getString(ATTR_ICON, null) ;
    }

    /**
     * Get this resource's max age.
     * The max age of a resource indicates how much drift is allowed between
     * the physicall version of the resource, and any in-memory cached version
     * of it.
     * <p>The max age attribute is a long number giving the number of 
     * milliseconds of allowed drift.
     */

    public long getMaxAge() {
	return getLong(ATTR_MAXAGE, (long) -1) ;
    }

    /**
     * The default GET method replies with a not implemented.
     * @param request The request to handle.
     * @exception HTTPException Always thrown, to return a NOT_IMPLEMENTED
     *    error.
     */

    public Reply get (Request request) 
	throws HTTPException
    {
	Reply error = request.makeReply(HTTP.NOT_IMPLEMENTED) ;
	error.setContent("Method GET not implemented.") ;
	throw new HTTPException (error) ;
    }

    /**
     * The default HEAD method replies does a GET and removes entity.
     * @param request The request to handle.
     * @exception HTTPException Always thrown, to return a NOT_IMPLEMENTED
     *    error.
     */

    public Reply head(Request request)
	throws HTTPException
    {
	Reply reply = get(request) ;
	reply.setStream((InputStream) null);
	return reply;
    }

    /**
     * The default POST method replies with a not implemented.
     * @param request The request to handle.
     * @exception HTTPException Always thrown, to return a NOT_IMPLEMENTED
     *    error.
     */

    public Reply post(Request request)
	throws HTTPException
    {
	Reply error = request.makeReply(HTTP.NOT_IMPLEMENTED) ;
	error.setContent("Method POST not implemented.") ;
	throw new HTTPException (error) ;
    }

    /**
     * The default PUT method replies with a not implemented.
     * @param request The request to handle.
     * @exception HTTPException Always thrown, to return a NOT_IMPLEMENTED
     *    error.
     */

    public Reply put(Request request)
	throws HTTPException
    {
	Reply error = request.makeReply(HTTP.NOT_IMPLEMENTED) ;
	error.setContent("Method PUT not implemented.") ;
	throw new HTTPException (error) ;
    }

    /**
     * The default DELETE method replies with a not implemented.
     * @param request The request to handle.
     * @exception HTTPException Always thrown, to return a NOT_IMPLEMENTED
     *    error.
     */

    public Reply delete(Request request)
	throws HTTPException
    {
	Reply error = request.makeReply(HTTP.NOT_IMPLEMENTED) ;
	error.setContent("Method DELETE not implemented.") ;
	throw new HTTPException (error) ;
    }

    /**
     * The default LINK method replies with a not implemented.
     * @param request The request to handle.
     * @exception HTTPException Always thrown, to return a NOT_IMPLEMENTED
     *    error.
     */

    public Reply link(Request request) 
	throws HTTPException
    {
	Reply error = request.makeReply(HTTP.NOT_IMPLEMENTED) ;
	error.setContent("Method LINK not implemented.") ;
	throw new HTTPException (error) ;
    }

    /**
     * The default UNLINK method replies with a not implemented.
     * @param request The request to handle.
     * @exception HTTPException Always thrown, to return a NOT_IMPLEMENTED
     *    error.
     */

    public Reply unlink(Request request)
	throws HTTPException
    {
	Reply error = request.makeReply(HTTP.NOT_IMPLEMENTED) ;
	error.setContent("Method UNLINK not implemented.") ;
	throw new HTTPException (error) ;
    }

    /**
     * The handler for unknown method replies with a not implemented.
     * @param request The request to handle.
     * @exception HTTPException Always thrown, to return a NOT_IMPLEMENTED
     *    error.
     */

    public Reply extended(Request request)
	throws HTTPException
    {
	Reply error = request.makeReply(HTTP.NOT_IMPLEMENTED) ;
	error.setContent("Method UNLINK not implemented.") ;
	throw new HTTPException (error) ;
    }

    /**
     * Perform the given request on this HTTP resource.
     * @param request The request to perform.
     * @exception HTTPException If the resource couldn't process this request.
     */

    public Reply perform(Request request) 
	throws HTTPException
    {
	String method = request.getMethod (null) ;
	if ( method == null ) {
	    Reply error = request.makeReply(HTTP.BAD_REQUEST) ;
	    error.setContent ("No method provided.") ;
	    throw new HTTPException (error) ;
	} else if ( method.equals ("GET") ) {
	    return get (request) ;
	} else if ( method.equals ("HEAD") ) {
	    return head (request) ;
	} else if ( method.equals ("POST") ) {
	    return post (request) ;
	} else if ( method.equals ("PUT") ) {
	    return put (request) ;
	} else if ( method.equals ("DELETE") ) {
	    return delete (request) ;
	} else if ( method.equals ("LINK") ) {
	    return link (request) ;
	} else if ( method.equals ("UNLINK") ) {
	    return unlink (request) ;
	} else {
	    return extended (request) ;
	}
    }

}
