// FileResource.java
// $Id: FileResource.java,v 1.5 1996/05/28 14:36:09 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.resources ;

import java.io.* ;

import w3c.jigsaw.http.* ;
import w3c.mime.* ;

public class FileResource extends FilteredResource {
    /**
     * Attributes index - The filename attribute.
     */
    protected static int ATTR_FILENAME = -1 ;
    /**
     * Attribute index - Do we allow PUT method on this file.
     */
    protected static int ATTR_PUTABLE = -1 ;
    /**
     * Attribute index - The date at which we last checked the file content.
     */
    protected static int ATTR_FILESTAMP = -1 ;

    static {
	Attribute a   = null ;
	Class     cls = null ;
	try {
	    cls = Class.forName("w3c.jigsaw.resources.FileResource") ;
	} catch (Exception ex) {
	    ex.printStackTrace();
	    System.exit(0);
	}
	// The filename attribute.
	a = new FilenameAttribute("filename"
				  , null
				  , Attribute.EDITABLE) ;
	ATTR_FILENAME = AttributeRegistery.registerAttribute(cls, a) ;
	// The putable flag:
	a = new BooleanAttribute("putable"
				 , Boolean.FALSE
				 , Attribute.EDITABLE) ;
	ATTR_PUTABLE = AttributeRegistery.registerAttribute(cls, a) ;
	// The file stamp attribute
	a = new DateAttribute("filestamp"
			      , new Long(-1) 
			      , Attribute.COMPUTED) ;
	ATTR_FILESTAMP = AttributeRegistery.registerAttribute(cls, a) ;
    }
    
    /**
     * The file we refer to.
     * This is a cached version of some attributes, so we need to override
     * the setValue method in order to be able to catch any changes to it.
     */
    protected File file = null ;

    /**
     * Get this resource filename attribute.
     */

    public String getFilename() {
	return (String) getValue(ATTR_FILENAME, null);
    }

    /**
     * Get the PUT'able flag (are we allow to PUT to the resource ?)
     */

    public boolean getPutableFlag() {
	return getBoolean(ATTR_PUTABLE, false) ;
    }

    /**
     * Get the date at which we last examined the file.
     */

    public long getFileStamp() {
	return getLong(ATTR_FILESTAMP, (long) -1) ;
    }

    /**
     * Get the name of the backup file for this resource.
     * @return A File object suitable to receive the backup version of this
     *    file.
     */

    public File getBackupFile() {
	File   file = getFile() ;
	String name = file.getName() ;
	return new File(file.getParent(), name+"~") ;
    }

    /**
     * Save the given stream as the underlying file content.
     * This method preserve the old file version in a <code>~</code> file.
     * @param in The input stream to use as the resource entity.
     * @exception IOException If dumping the content failed.
     */

    protected synchronized void newContent(InputStream in) 
	throws IOException
    {
	File   file   = getFile() ;
	String name   = file.getName() ;
	File   temp   = new File(file.getParent(), "#"+name+"#") ;
	String iomsg  = null ;

	// We are not catching IO exceptions here, except to remove temp:
	try {
	    FileOutputStream fout  = new FileOutputStream(temp) ;
	    byte             buf[] = new byte[4096] ;
	    for (int got = 0 ; (got = in.read(buf)) > 0 ; )
		fout.write(buf, 0, got) ;
	    fout.close() ;
	} catch (IOException ex) {
	    iomsg = ex.getMessage() ;
	} finally {
	    if ( iomsg != null ) {
		temp.delete() ;
		throw new IOException(iomsg) ;
	    } else {
		file.renameTo(getBackupFile()) ;
		temp.renameTo(file) ;
		// update our attributes for this new content:
		updateAttributes() ;
	    }
	}
    }

    /**
     * Set some of this resource attribute.
     * We just catch here any write access to the filename's, to update 
     * our cache file object.
     */

    public synchronized void setValue(int idx, Object value) {
	super.setValue(idx, value) ;
	if ((idx == ATTR_FILENAME) || (idx == ATTR_IDENTIFIER))
	      file = null ;
    }
	    
    /**
     * Get this file resource file name.
     */

    public synchronized File getFile() {
	// Have we already computed this ?
	if ( file == null ) {
	    // Get the file name:
	    String name = getFilename() ;
	    if ( name == null )
		name = getIdentifier() ;
	    // Get the file directory:
	    ContainerResource p = getParent() ;
	    while ((p != null) && ! (p instanceof DirectoryResource) )
		p = p.getParent() ;
	    if ( p == null )
		throw new RuntimeException ("no directory for "+getURL()) ;
	    file = new File(((DirectoryResource) p).getDirectory(), name);
	}
	return file ;
    }
	
    /**
     * The GET method on files.
     * Check for the last modified time against the IMS if any. If OK, emit
     * a not modified reply, otherwise, emit the whole file.
     * @param request The request to handle.
     * @exception HTTPException If some error occured.
     */

    public Reply get(Request request)
	throws HTTPException
    {
	File file = getFile() ;
	// Has this resource changed since last queried ? 
	long ims = request.getIfModifiedSince(-1) ;
	long lmt = file.lastModified() ;
	long cmt = getLastModified() ;
// This section of code, can't be turned on right now
// It allows to answer to IMS requests without touching the disk, however
// as of today (and due to people using PUT), the server has to go down to
// the file system to answer the question. This is unfortunate.
//	if ( lmt < 0 ) {
//	    file = getFile() ;
//	    lmt  = file.lastModified() ;
//	}
	if ((cmt < 0) || (cmt < lmt))
	    updateFileAttributes() ;
	if ((ims > 0) && (lmt > 0) && (lmt <= ims)) 
	    return request.makeReply(HTTP.NOT_MODIFIED) ;
	// Does this file really exists, if so send it back
	if ( file.exists() ) {
	    Reply reply  = request.makeReply(HTTP.OK) ;
	    int   length = getContentLength();
	    if ( length < 0 )
		length = (int) file.length() ;
	    reply.setContentLength(length) ;
	    reply.setContentType(getContentType()) ;
	    if ( definesAttribute(ATTR_CONTENT_ENCODING) ) 
  		reply.setContentEncoding(getContentEncoding()) ;
	    if ( definesAttribute(ATTR_CONTENT_LANGUAGE) )
		reply.setContentLanguage(getContentLanguage()) ;
	    reply.setLastModified(lmt) ;
	    try { 
		reply.setStream(new FileInputStream(file)) ;
	    } catch (IOException ex) {
		// I hate to have to loose time in tries
	    }
	    return reply ;
	} else {
	    Reply error = request.makeReply(HTTP.NOT_FOUND) ;
	    error.setContent ("<h1>Document not found</h1>"
			      + "<p>The document "
			      + request.getURI("<em>invalid URI</em>")
			      + " is indexed but not available."
			      + "<p>The server is misconfigured.") ;
	    throw new HTTPException (error) ;
	}
	// not reached
    }

    /**
     * Put a new entity in this resource.
     * @param request The request to handle.
     */

    public synchronized Reply put(Request request)
	throws HTTPException
    {
	// Is this resource writable ?
	if ( ! getPutableFlag() )
	    super.put(request) ;
	// Check the request:
	if ( ! request.hasField("content-length") ) {
	    Reply error = request.makeReply(HTTP.BAD_REQUEST) ;
	    error.setContent ("<p>Request has no content length header.") ;
	    throw new HTTPException (error) ;
	}
	// Check that if some type is provided it doesn't conflict:
	if ( request.hasField ("content-type") ) {
	    MIMEType rtype = request.getContentType(null) ;
	    MIMEType type  = getContentType() ;
	    if ( type == null ) {
		setValue ("content-type", rtype) ;
	    } else if ( rtype.match (type) < 0 ) {
		Reply error = request.makeReply(HTTP.BAD_REQUEST) ;
		error.setContent ("<p>Invalid content type: "+type.toString());
		throw new HTTPException (error) ;
	    }
	}
	// Get the request input stream:
	int         length = request.getContentLength(-1) ;
	InputStream in     = null ;
	in = new MIMEContentLengthInputStream(request.getInputStream(),length);
	// Write the body back to the file:
	try {
	    newContent(in) ;
	} catch (IOException ex) {
	    Reply error = request.makeReply(HTTP.BAD_REQUEST) ;
	    error.setContent ("IO error while saving new content: "
			      + ex.getMessage()) ;
	    throw new HTTPException (error) ;
	}
	// Refresh the client's display
	Reply reply = request.makeReply(HTTP.OK) ;
	reply.setContent ("<p>Entity body saved succesfully !") ;
	return reply ;
    }

    /**
     * Update the file related attributes.
     * The file we serve has changed since the last time we checked it, if
     * any of the attribute values depend on the file content, this is the
     * appropriate place to recompute them.
     */

    public void updateFileAttributes() {
	File file = getFile() ;
	setValue(ATTR_LAST_MODIFIED, new Long (file.lastModified())) ;
	setValue(ATTR_CONTENT_LENGTH, new Integer((int)file.length()));
	return ;
    }

    /**
     * Update our computed attributes.
     */

    public void updateAttributes() {
	long fstamp = getFile().lastModified() ;
	long stamp  = getLong(ATTR_FILESTAMP, -1) ;

	if ( stamp < fstamp ) {
	    updateFileAttributes() ;
	    setValue(ATTR_FILESTAMP, new Long(fstamp));
	}
    }

}
