// PICSFilter.java
// $Id: PICSFilter.java,v 1.1 1996/04/10 13:47:08 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.pics ;

import java.io.* ;
import java.util.*;
import java.net.*;

import w3c.jigsaw.http.* ;
import w3c.jigsaw.resources.*;
import w3c.jigsaw.html.HtmlGenerator ;

/** 
 * This package implements a PICS filter. The PICS filters allows server
 * administrator to rate the documents they deliver. The references for this
 * protocol is <a href="http://www.w3.org/hypertext/WWW/PICS/">here</a>.</p>
 * <p>The PICS filter defines the following attributes:</p>
 * <table border>
 * <caption>The list of parameters</caption>
 * <tr> 
 * <th align=left>Parameter name</th> 
 * <th align=left>Semantics</th>
 * <th align=left>Default value</th> 
 * <th align=left>Type</th>
 * </tr>
 * <tr> 
 * <th align=left>bureau</th> 
 * <th align=left>The label bureau to query for this entity labels.</th>
 * <th align=left><em>none</em></th> 
 * <th align=left>java.lang.String</th>
 * </tr>
 * </table>
 */

public class PICSFilter extends ResourceFilter {
    /**
     * Attribute index - The identifier of our bureau.
     */
    protected static int ATTR_BUREAU_IDENTIFIER = -1 ;

    static {
	Attribute a   = null ;
	Class     cls = null ;
	try {
	    cls = Class.forName("w3c.jigsaw.pics.PICSFilter");
	} catch (Exception ex) {
	    ex.printStackTrace() ;
	    System.exit(1) ;
	}
	// The bureau identifier attribute
	a = new FileAttribute("bureau"
			      , null
			      , Attribute.EDITABLE|Attribute.MANDATORY) ;
	ATTR_BUREAU_IDENTIFIER = AttributeRegistery.registerAttribute(cls, a);
    }

    /**
     * Our loaded lable bureau.
     */
    protected LabelBureauInterface bureau      = null ;

    /**
     * The PICS protocol version that this filter handles.
     */
    public  static final String PICS_PROTOCOL_ID = "PICS-1.0" ;

    /**
     * Get our label bureau identifier.
     */

    public File getBureauIdentifier() {
	return (File) getValue(ATTR_BUREAU_IDENTIFIER, null) ;
    }

    /**
     * Make sure our label bureau is loaded.
     */

    protected final void acquireBureau() {
	if ( bureau != null )
	    return ;
	bureau = LabelBureauFactory.getLabelBureau(getBureauIdentifier());
    }

    /**
     * Check the query to examine if it requires some PICS handling. 
     * If this is the case, it returns a <em>Bag</em> object
     * corresponding to the part of the <em>Accept-Protocol</em> header that
     * relates with PICS.
     * @param request The request to be checked.
     * @return A Bag object if PICS handling required, <string>null</strong>
     *         otherwise.
     */

    protected Bag isPICSQuery (Request request) 
	throws HTTPException
    {
	// If the request doesn't ask for labels, return right now.
	if ( ! request.hasField ("accept-protocol") )
	    return null ;
	Bag protocols = request.getFieldBag ("accept-protocol") ;
	if ( ! protocols.hasBag (PICS_PROTOCOL_ID) )
	    return null ;
	// Now, the request has some PICS stuff in it, let look inside this:
	Bag pics = null ;
	try {
	    pics = protocols.getBag (PICS_PROTOCOL_ID) ;
	} catch (ClassCastException e) {
	    return null ;
	}
	return pics ;
    }



    public Reply outgoingFilter (Request request, Reply reply) 
	throws HTTPException
    {
	Bag pics = isPICSQuery (request) ;
	if ( pics == null )
	    return reply ;
	// Get the requested services:
	Bag params    = pics.getBag ("params") ;
	Bag services  = params.getBag ("services") ;
	URL url       = null ;
	Client client = request.getClient() ;

	try {
	    String requri = request.getURI() ;
	    if ( requri.indexOf(':') < 0 ) {
		url = new URL ("http"
			       , client.getServer().getHost()
			       , client.getServer().getPort()
			       , requri) ;
	    } else {
		url = new URL(requri) ;
	    }
	} catch (MalformedURLException e) {
	    String m = "unable to URL \"" + request.getURI() + "\"" ;
	    throw new HTTPRuntimeException (this.getClass().getName()
					    , "outgoingFilter"
					    , m) ;
	}
	int format    = LabelBureauInterface.FMT_MINIMAL ;
	// Get any format parameter:
	if ( params.hasItem ("minimal") ) {
	    format = LabelBureauInterface.FMT_MINIMAL ;
	} else if ( params.hasItem ("short") ) {
	    format = LabelBureauInterface.FMT_SHORT ;
	} else if ( params.hasItem ("full") ) {
	    format = LabelBureauInterface.FMT_FULL ;
	} else if ( params.hasItem ("signed") ) {
	    format = LabelBureauInterface.FMT_SIGNED ;
	} else {
	    Reply error = request.makeReply(HTTP.BAD_REQUEST) ;
	    error.setContent ("Invalid label format: "+format) ;
	    throw new HTTPException (error) ;
	}
	// Get labels for each service, building out the ret hashtable
	StringBuffer sb = new StringBuffer(128) ;
	Enumeration  e  = services.keys() ;
	sb.append ("("+PICS_PROTOCOL_ID) ;
    sloop:
	while ( e.hasMoreElements() ) {
	    String                n = (String) e.nextElement() ;
	    LabelServiceInterface s = bureau.getLabelService (n) ;
	    if ( s == null ) {
		sb.append (" error "
			   + "(service-unavailable \"unknown service\")") ;
		continue sloop ;
	    } 
	    s.dump(sb, format) ;
	    LabelInterface l = s.getSpecificLabel (url) ;
	    if ( (l == null) && ((l = s.getGenericLabel (url)) == null) ) {
		sb.append (" error (not-labeled \"" + url +"\")") ;
	    }  else {
		sb.append (" labels ") ;
		l.dump (sb, format) ;
	    }
	}
	sb.append (")") ;
	// Add additional reply headers:
	reply.addHeader ("Protocol", ("{"
				      + PICS_PROTOCOL_ID 
				      + " {headers PICS-label}}")) ;
	reply.addHeader ("PICS-label", sb.toString()) ;
	return reply ;
    }

    public void initialize (Object values[]) {
	super.initialize(values);
	acquireBureau() ;
    }
}
