// IndexerResource.java
// $Id: IndexerResource.java,v 1.1 1996/04/10 13:45:39 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.indexer ;

import java.util.*;
import java.io.* ;

import w3c.jigsaw.resources.* ;
import w3c.jigsaw.formedit.* ;

/**
 * Indexer resource have some shared properties.
 * This class implements an Indexer Resource, wich hav the following 
 * properties:
 * <ul>
 * <li>They optionnaly handle a class (to be instanciated)
 * <li>They provide a default set of attributes for their non-existing target.
 * </ul>
 */

abstract
public class IndexerResource extends Resource implements ResourceShadower {
    /**
     * Attribute index - The class for this directory.
     */
    protected static int ATTR_CLASS = -1 ;

    static {
	Attribute  a  = null ;
	Class     cls = null ;

	try {
	    cls   = Class.forName("w3c.jigsaw.indexer.IndexerResource");
	} catch (Exception ex) {
	    ex.printStackTrace() ;
	    System.exit(1) ;
	}
	// The class attribute
	a = new ClassAttribute("class"
			       , null
			       , Attribute.EDITABLE) ;
	ATTR_CLASS = AttributeRegistery.registerAttribute(cls, a) ;
	// Register this resource editor:
	FormEditorFactory.registerEditor("w3c.jigsaw.indexer.IndexerResource"
					 , "w3c.jigsaw.formedit.IndexerResourceEditor");
    }

    /**
     * The set of attributes for our target class.
     */
    protected Attribute target_attributes[] = null ;
    /**
     * The default attribute values we hold for to be created reosurces.
     */
    protected Object    target_values[] = null ;
    

    /**
     * Get the target class for this directory.
     */

    public Class getTargetClass() {
	return (Class) getValue(ATTR_CLASS, null) ;
    }

    /**
     * Get the default target class.
     */

    abstract public Class getDefaultTargetClass() ;

    /**
     * ResourceShadower implementation - Get our target resource.
     * We don't really have a target resource here.
     */

    public Resource getTargetResource() {
	return null ;
    }

    /**
     * Get our target class attributes.
     */

    public Attribute[] getTargetAttributes() {
	return target_attributes;
    }

    /**
     * ResourceShadower implementation - Get target default attribute value.
     * @param idx The name of the target attribute we want a value for.
     * @param def The default value (if we don't define it ourselve).
     * @return Our proposed target attribute value, or the default 
     *     provided value if we don't define it.
     */

    public Object getTargetValue(int idx, Object def) {
	if (idx < 0)
	    throw new IllegalAttributeAccess(this, idx) ;
	if (idx >= target_values.length)
	    return def ;
	return (target_values[idx] == null) ? def : target_values[idx] ;
    }

    /**
     * ResourceShadower implementation - Get a shadowed attribute value
     * @param name The name of the attribute.
     * @param def The default value to return if undef.
     * @return The attribute value or the provided default value.
     */

    public Object getTargetValue(String name, Object def) {
	for (int i = 0 ; i < target_attributes.length ; i++) {
	    if ( name.equals(target_attributes[i].getName()) ) {
		return target_values[i] ;
	    }
	}
	return null ;
    }

    /**
     * ResourceShadower implementation - Set target default attribute value.
     * @param name The name of the attribute to set.
     * @param value Its Object value.
     */

    public void setTargetValue(String name, Object def) {
	for (int i = 0 ; i < target_attributes.length ; i++) {
	    if ( name.equals(target_attributes[i].getName()) ) {
		setTargetValue(i, def) ;
		markModified() ;
		return ;
	    }
	}
	throw new IllegalAttributeAccess(this, name) ;
    }

    /**
     * ResourceShadower implementation - Set target default attribute value.
     * @param idx The index of the attribute to set.
     * @param value Its new Object value.
     */

    public void setTargetValue(int idx, Object def) {
	if ((idx < 0) || (idx >= target_values.length))
	    throw new IllegalAttributeAccess(this, idx) ;
	target_values[idx] = def ;
	markModified() ;
    }

    /**
     * ResourceShadower implementation - Do we shadow this attribute.
     * @param idx The index of the target attribute.
     * @return A boolean <strong>true</strong if attribute is defined.
     */

    public boolean definesTargetAttribute(int idx) {
	if (idx < 0)
	    throw new IllegalAttributeAccess(this, idx) ;
	if (idx >= target_attributes.length)
	    return false ;
	return (target_values[idx] != null) ;
    }

    /**
     * ResourceShadower implementation - Do we shadow this attribute.
     * @param name The name of the target attribute.
     * @return A boolean <strong>true</strong if attribute is defined.
     */

    public boolean definesTargetAttribute(String name) {
	for (int i = 0 ; i < target_attributes.length ; i++) {
	    if ( name.equals(target_attributes[i].getName()) ) 
		return (target_values[i] != null) ;
	}
	return false ;
    }

    /**
     * Pickling an Extension is a little tricky.
     * @param out The output stream to picle to.
     */

    public void pickle(DataOutputStream out) 
	throws IOException
    {
	// Pickle myself:
	super.pickle(out) ;
	// Than pickle the attributes I have for the target:
	for (int i = 0 ; i < target_attributes.length ; i++) {
	    Attribute attr = target_attributes[i] ;
	    if ((target_values[i] != null) 
		&& ! attr.checkFlag(Attribute.DONTSAVE)) {
		out.writeBoolean(true) ;
		attr.pickle(out, target_values[i]);
	    } else {
		out.writeBoolean(false) ;
	    }
	}
    }

    /**
     * Unpickle myself.
     * @param in The input stream to unpickle from.
     * @param defs The proposed set of default attributes.
     */
    
    public AttributeHolder unpickleInstance(DataInputStream in, Hashtable defs)
	throws IOException
    {
	// Unpickle myself:
	super.unpickleInstance(in, defs) ;
	// Unpickle my additional informations.
	Class cls = getTargetClass() ;
	if ( cls == null )
	    cls = getDefaultTargetClass() ;
	this.target_attributes = AttributeRegistery.getClassAttributes(cls);
	this.target_values     = new Object[target_attributes.length] ;
	for (int i = 0 ; i < target_attributes.length ; i++) {
	    if ( in.readBoolean() ) {
		target_values[i] = target_attributes[i].unpickle(in) ;
	    } else {
		target_values[i] = null ;
	    }
	}
	return this ;
    }

    /**
     * Initialize a resource indexer instance.
     * @param values The proposed default attribute values.
     */

    public void initialize(Object values[]) {
	super.initialize(values) ;
	// Have we set our target stuff:
	Class cls = getTargetClass() ;
	if ( cls == null )
	    cls = getDefaultTargetClass() ;
	if ( cls != null ) {
	    if ( target_attributes == null )
		target_attributes = AttributeRegistery.getClassAttributes(cls);
	    if ( target_values == null )
		target_values = new Object[target_attributes.length] ;
	}
    }



}
