// ExtensionsEditor.java
// $Id: ExtensionsEditor.java,v 1.2 1996/04/12 21:00:57 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.indexer ;

import java.util.* ;

import w3c.tools.sorter.*;
import w3c.mime.*;
import w3c.jigsaw.http.* ;
import w3c.jigsaw.resources.* ;
import w3c.jigsaw.forms.* ;
import w3c.jigsaw.formedit.*;
import w3c.jigsaw.html.*;

class ExtensionEditorHandler extends FormCardHandler {
    Extension target = null ;

    public void notifyChange(FormFieldInterface field, Object value) 
	throws FormFieldException
    {
	target.setTargetValue(field.getName(), value) ;
    }

    public String notifyEndProcessing(FormCardResource card, Request request)
	throws FormProcessingException
    {
	return "..";
    }

    ExtensionEditorHandler(Extension target) {
	this.target = target ;
    }

}

class ExtensionEditor extends FormResource {
    Extension              target  = null ;
    ExtensionEditorHandler handler = null ;

    public void dumpHeader(HtmlGenerator into) {
	Class cls = target.getTargetClass() ;
	if ( cls == null ) 
	    cls = target.getDefaultTargetClass() ;
	into.append("<h2>"+cls.getName()+"</h2>");
    }

    ExtensionEditor(String url, Extension target) {
	super(url) ;
	this.target  = target ;
	this.handler = new ExtensionEditorHandler(target) ;
	// Define the card:
	FormCardResource c = defineCard(handler
					, "Attributes"
					, "Attributes for extension "
					+ target.getIdentifier()) ;
	Attribute attrs[] = target.getTargetAttributes() ;
	for (int i = 0 ; i < attrs.length ; i++) {
	    Attribute              a = attrs[i] ;
	    FormFieldInterface field = null ;

	    if ( ! a.checkFlag(Attribute.EDITABLE) )
		continue ;
	    field = FormFieldRegistery.getTargetField(target, a) ;
	    if ( field != null )
		c.addField(field) ;
	}
    }
}

class ExtensionsAddHandler extends FormCardHandler {
    ResourceIndexer indexer = null ;
    String          url     = null ;

    public void notifyButtonClick(String label) {
	indexer.save() ;
    }

    /**
     * We just handle the notification of end of processing.
     */

    public String notifyEndProcessing(FormCardResource card, Request request)
	throws FormProcessingException
    {
	// Get and check the input values:
	String name = (String) card.lookupField("name").getValue() ;
	Class  cls  = (Class) card.lookupField("class").getValue();
	if (name == null) {
	    String msg = "Fill in the name of the extension. If the class is "
		+ " left blank, than this extension will be associated to "
		+ " a set of HTTPResource attributes, but file having only "
		+ " this extension won't be exported.";
	    throw new FormProcessingException (msg) ;
	}
	// Create and register the new extension:
	Extension e = Extension.makeExtension(name, cls) ;
	indexer.registerExtension(e) ;
	// Relocate to the extension editor:
	return url+"/"+name;
    }

    ExtensionsAddHandler(String url, ResourceIndexer indexer) {
	this.url     = url ;
	this.indexer = indexer ;
    }

}

class ExtensionsListHandler extends FormCardHandler {
    ResourceIndexer indexer = null ;
    Vector          marked  = null ;
    String          url     = null ;

    /**
     * Update the form card.
     * We are really not clever here, this will be improved soon.
     */

    public FormCardResource updateFormCard(FormCardResource card) {
	// Reset the card of all its known fields:
	card.reset() ;
	// Rebuild the fields that are valid now:
	Enumeration enum = indexer.enumerateExtensionNames() ;
	Vector      exts = Sorter.sortStringEnumeration(enum) ;
	for (int i = 0 ; i < exts.size() ; i ++) {
	    String    name  = (String) exts.elementAt(i) ;
	    FormField field = new CheckBoxField(name
						, name
						, url + "/" + name
						, false);
	    card.addField(field);
	}
	return card ;
    }

    /**
     * On notifyChange, mark extension to be removed.
     */
    
    public void notifyChange(FormFieldInterface field, Object value)
	throws FormFieldException
    {
	if ( marked == null )
	    marked = new Vector() ;
	marked.addElement(field.getName()) ;
    }

    /**
     * On notifyEndProcessing, remove marked extensions.
     */

    public String notifyEndProcessing(FormCardResource card, Request request)
	throws FormProcessingException
    {
	// No extensions marked ?
	if ( marked == null )
	    return null;
	// Remove all the marked ones
	int vsize = marked.size() ;
	for (int i = 0 ; i < vsize ; i++) {
	    String id = (String) marked.elementAt(i) ;
	    indexer.unregisterExtension(id) ;
	    card.removeField(id) ;
	}
	marked.setSize(0) ;
	return null ;
    }

    ExtensionsListHandler(String url, ResourceIndexer indexer) {
	this.url     = url ;
	this.indexer = indexer ;
    }
}

public class ExtensionsEditor extends FormResource implements ResourceLocker {
    Hashtable       loaded  = null ;
    ResourceIndexer indexer = null ;

    /**
     * Define the form cards.
     * The extensions editor provides one form card for editing the list
     * of known extensions, and another one for adding new extensions.
     */

    protected void defineFormCards() {
	FormFieldInterface field = null ;
	String             url   = getURL() ;

	// The card to remove extensions:
	FormCardHandler list_handler = new ExtensionsListHandler(url,indexer);
	FormCardResource c = defineCard(list_handler
					, "Extensions"
					, "Defined extensions") ;
	list_handler.updateFormCard(c) ;
	// The card to define new extensions:
	FormCardHandler add_handler = new ExtensionsAddHandler(url,indexer);
	c = defineCard(add_handler, "AddExtension", "Add a new extension.");
	// The input field for the name of the extension:
	field = new TextField("name", "name", null, null) ;
	c.addField(field) ;
	// The input field for the class name:
	field = new ClassnameField("class", "class", null, null) ;
	c.addField(field) ;
	// A Save button:
	c.addButton("Save");
    }
	
    /**
     * Resource locker implementation - Our lock is being breaked.
     * @param resource The resource going down.
     */

    public synchronized void notifyResourceLockBreak(Resource resource) {
System.out.println("*** lock breaks on "+resource.getIdentifier()) ;
	loaded.remove(resource.getIdentifier()) ;
    }

    /** 
     * Lookup in the extensions editor for a defined extension.
     * @param state The current lookup state object.
     * @return A Resource to edit the extension or <strong>null</strong>.
     */
					    
    public synchronized HTTPResource lookup (LookupState state) 
	throws HTTPException
    {
	// Look for an already existing editor, fallback to super otherwise
	String          name   = state.peekNextComponent() ;
	ExtensionEditor editor = (ExtensionEditor) loaded.get(name) ;
	if ( editor != null ) {
	    state.getNextComponent() ;
	    return editor ;
	}
	// Install a new one
	Extension extension = indexer.loadExtension(name) ;
	if ( extension == null ) 
	    return super.lookup(state) ;
	state.getNextComponent() ;
	editor = new ExtensionEditor(getURL()+"/"+name, extension) ;
	extension.lock(this) ;
	loaded.put(name, editor) ;
	return editor ;
    }

    /**
     * Initialize the extension editor.
     */

    public void initialize(Object values[]) {
	super.initialize(values) ;
	this.indexer = getServer().getIndexer() ;
	this.loaded  = new Hashtable() ;
	defineFormCards() ;
    }

}
