// DirectoriesEditor.java
// $Id: DirectoriesEditor.java,v 1.2 1996/04/12 21:00:55 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.indexer ;

import java.util.* ;

import w3c.tools.sorter.*;
import w3c.mime.*;
import w3c.jigsaw.http.* ;
import w3c.jigsaw.resources.* ;
import w3c.jigsaw.forms.* ;
import w3c.jigsaw.formedit.*;

class DirectoriesAddHandler extends FormCardHandler {
    ResourceIndexer indexer = null ;
    String          url     = null ;

    public void notifyButtonClick(String label) {
	indexer.save() ;
    }

    /**
     * We just handle the notification of end of processing.
     */

    public String notifyEndProcessing(FormCardResource card, Request request)
	throws FormProcessingException
    {
	// Get and check the input values:
	String name = (String) card.lookupField("name").getValue() ;
	Class  cls  = (Class) card.lookupField("class").getValue();
	if (name == null) {
	    String msg = "Fill in the name of the directory. If the class is "
		+ " left blank, than this directory will be associated to "
		+ " the DirectoryResource class." ;
	    throw new FormProcessingException (msg) ;
	}
	// Create and register the new directory template:
	Directory d = Directory.makeDirectory(name, cls) ;
	indexer.registerDirectory(d) ;
	// Relocate to the directory template editor:
	return url+"/"+name;
    }

    DirectoriesAddHandler(String url, ResourceIndexer indexer) {
	this.url     = url ;
	this.indexer = indexer ;
    }

}

class DirectoriesListHandler extends FormCardHandler {
    ResourceIndexer indexer = null ;
    Vector          marked  = null ;
    String          url     = null ;

    /**
     * Update the form card.
     * We are really not clever here, this will be improved soon.
     */

    public FormCardResource updateFormCard(FormCardResource card) {
	// Reset the card of all its known fields:
	card.reset() ;
	// Rebuild the fields that are valid now:
	Enumeration enum = indexer.enumerateDirectoryNames() ;
	Vector      exts = Sorter.sortStringEnumeration(enum) ;
	for (int i = 0 ; i < exts.size() ; i ++) {
	    String    name  = (String) exts.elementAt(i) ;
	    FormField field = new CheckBoxField(name
						, name
						, url + "/" + name
						, false);
	    card.addField(field);
	}
	return card ;
    }

    /**
     * On notifyChange, mark directory to be removed.
     */
    
    public void notifyChange(FormFieldInterface field, Object value)
	throws FormFieldException
    {
	if ( marked == null )
	    marked = new Vector() ;
	marked.addElement(field.getName()) ;
    }

    /**
     * On notifyEndProcessing, remove marked directory templates.
     */

    public String notifyEndProcessing(FormCardResource card, Request request)
	throws FormProcessingException
    {
	// No extensions marked ?
	if ( marked == null )
	    return null;
	// Remove all the marked ones
	int vsize = marked.size() ;
	for (int i = 0 ; i < vsize ; i++) {
	    String id = (String) marked.elementAt(i) ;
	    indexer.unregisterDirectory(id) ;
	    card.removeField(id) ;
	}
	marked.setSize(0) ;
	return null ;
    }

    DirectoriesListHandler(String url, ResourceIndexer indexer) {
	this.url     = url ;
	this.indexer = indexer ;
    }
}

public class DirectoriesEditor extends FormResource implements ResourceLocker {
    Hashtable       loaded  = null ;
    ResourceIndexer indexer = null ;

    /**
     * Define the form cards.
     * The directories editor provides one form card for editing the list
     * of known directories, and another one for adding new directories.
     */

    protected void defineFormCards() {
	FormFieldInterface field = null ;
	String             url   = getURL() ;

	// The card to remove directories:
	FormCardHandler list_handler = new DirectoriesListHandler(url,indexer);
	FormCardResource c = defineCard(list_handler
					, "Directories"
					, "Defined directories") ;
	list_handler.updateFormCard(c) ;
	// The card to define new directories:
	FormCardHandler add_handler = new DirectoriesAddHandler(url,indexer);
	c = defineCard(add_handler, "AddDirectory", "Add a new directory.");
	// The input field for the name of the directory:
	field = new TextField("name", "name", null, null) ;
	c.addField(field) ;
	// The input field for the class name:
	field = new ClassnameField("class", "class", null, null) ;
	c.addField(field) ;
	// A Save button:
	c.addButton("Save");
    }

    /**
     * ResourceLocker implementation - The resource is going down.
     * @param resource The resource being brought down.
     */

    public synchronized void notifyResourceLockBreak(Resource resource) {
	loaded.remove(resource.getIdentifier()) ;
    }

    /** 
     * Lookup in the directories editor for a defined directory.
     * @param state The current lookup state object.
     * @return A Resource to edit the directory or <strong>null</strong>.
     */
					    
    public synchronized HTTPResource lookup (LookupState state) 
	throws HTTPException
    {
	// Look for an already existing editor, fallback to super otherwise
	String       name   = state.peekNextComponent() ;
	HTTPResource editor = (HTTPResource) loaded.get(name);
	if ( editor != null ) {
	    state.getNextComponent() ;
	    return editor ;
	}
	// Install a new one
	Directory directory = indexer.loadDirectory(name) ;
	if ( directory == null ) 
	    return super.lookup(state) ;
	state.getNextComponent() ;
	editor = FormEditorFactory.getEditor(directory) ;
	if ( editor != null ) {
	    Hashtable defs = new Hashtable(3) ;
	    defs.put("url", getURL()+"/"+name) ;
	    defs.put("target", directory) ;
	    editor.initialize(defs) ;
	    directory.lock(this) ;
	    loaded.put(name, editor) ;
	}
	return editor ;
    }

    /**
     * Initialize the directory templates editor.
     */

    public void initialize(Object values[]) {
	super.initialize(values) ;
	this.indexer = getServer().getIndexer() ;
	this.loaded  = new Hashtable() ;
	defineFormCards() ;
    }

}
