// httpdProperties.java
// $Id: httpdProperties.java,v 1.2 1996/05/02 15:47:00 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.http ;

import java.util.* ;
import java.io.File ;

/**
 * Subclass of Properties, to deal with http specific usage of them.
 * This class refines the basic Properties class, in order to tune them for
 * http specific needs.
 * The major enhancement provided by this class over the basic Properties class
 * is to allow for property monitoring. The Jigsaw server doesn't define only
 * one set of properties. Instead - beacuse it is extensible - it define
 * multiple sets of properties. For example, the <code>httpd</code> class
 * defines a set of global properties, the <code>CommonLogger</code> class
 * also defines its own set of properties, etc.
 * <p>Each property can be monitored, to allow for dynamic reconfiguration of
 * the server.
 * @see w3c.jigsaw.core.PropertyMonitoring
 */

public class httpdProperties extends Properties {
    PropertyMonitoring observers[]     = null ;
    int                observers_count = 0 ;

    /**
     * Subscribe for property monitoring.
     * @param observer The object that handles the PropertyMonitoring 
     *    interface.
     */
    
    public void registerObserver (PropertyMonitoring observer) {
	// Try looking for an empty slot:
	for (int i = 0 ; i < observers.length ; i++) {
	    if ( observers[i] == null ) {
		observers[i] = observer ;
		return ;
	    }
	}
	// Add the observer to the registered oned, resizing array if needed
	if ( observers_count + 1 >= observers.length ) {
	    PropertyMonitoring m[]=new PropertyMonitoring[observers.length*2];
	    System.arraycopy (observers, 0, m, 0, observers.length) ;
	    observers = m ;
	}
	observers[observers_count++] = observer ;
    }

    /**
     * Unsubscribe this object from the observers list.
     * @param observer The observer to unsubscribe.
     * @return A boolean <strong>true</strong> if object was succesfully 
     *     unsubscribed, <strong>false</strong> otherwise.
     */

    public boolean unregisterObserver (PropertyMonitoring observer) {
	for (int i = 0 ; i < observers.length ; i++) {
	    if ( observers[i] == observer ) {
		observers[i] = null ;
		return true ;
	    }
	}
	return false ;
    }
		
    /**
     * Update a property value.
     * Assign a value to a property. If the property value has really changed
     * notify our observers of the change.
     * @param name The name of the property to assign.
     * @param value The new value for this property, or <strong>null</strong>
     *    if the property setting is to be cancelled.
     * @return A boolean <strong>true</strong> if change was accepted by 
     *    our observers, <strong>false</strong> otherwise.
     */

    public synchronized boolean putValue (String name, String value) {
	// If null value, remove the prop definition:
	if ( value == null ) {
	    super.remove(name) ;
	    return true ;
	}
	// Otherwise, proceed:
	String old = (String) get (value) ;
	if ( (old == null) || (!  old.equals (value)) ) {
	    super.put (name, value) ;
	    for (int i = 0 ; i < observers.length ; i++) {
		if ( observers[i] == null )
		    continue ;
		if ( ! observers[i].propertyChanged (name) ) {
		    if ( old != null )
			super.put (name, old) ;
		    return false ;
		}
	    }
	}
	return true ;
    }
    
    /**
     * Get this property value, as a boolean.
     * @param name The name of the property to be fetched.
     * @param def The default value, if the property isn't defined.
     * @return A Boolean instance.
     */

    public boolean getBoolean(String name, boolean def) {
	String v = getProperty(name) ;
	if ( v != null )
	    return "true".equalsIgnoreCase(v) ? true : false ;
	return def ;
    }

    /**
     * Get this property value, as a String.
     * @param name The name of the property to be fetched.
     * @param def The default value, if the property isn't defined.
     * @return An instance of String.
     */

    public String getString(String name, String def) {
	String v = getProperty (name) ;
	if ( v != null )
	    return v ;
	return def ;
    }

    /**
     * Get this property value, as an integer.
     * @param name The name of the property to be fetched.
     * @param def The default value, if the property isn't defined.
     * @return An instance of Integer.
     */

    public int getInteger(String name, int def) {
	String v = getProperty (name) ;
	if ( v != null ) {
	    try {
		if (v.startsWith("0x")) {
		    return Integer.valueOf(v.substring(2), 16).intValue();
		}
		if (v.startsWith("#")) {
		    return Integer.valueOf(v.substring(1), 16).intValue();
		}
		if (v.startsWith("0")) {
		    return Integer.valueOf(v.substring(1), 8).intValue();
		}
		return Integer.valueOf(v).intValue();
	    } catch (NumberFormatException e) {
	    }
	}
	return def ;
    }

    /**
     * Get this property value, as a File.
     * @param name The name of the property to be fetched.
     * @param def The default value, if the property isn't defined.
     * @return An instance of File.
     */

    public File getFile(String name, File def) {
	String v = getProperty (name) ;
	if ( v != null )
	    return new File (v) ;
	return def ;
    }

    /**
     * Build an httpdProperties instance from a Properties instance.
     * @param props The Properties instance.
     */

    public httpdProperties (Properties props) {
	super (props) ;
	this.observers       = new PropertyMonitoring[5] ;
	this.observers_count = 0 ;
    }
   
}

 
