// Reply.java
// $Id: Reply.java,v 1.5 1996/05/28 14:03:07 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.http ;

import java.util.Date ;
import java.util.Hashtable ;
import java.util.Enumeration ;
import java.util.Vector ;

import java.io.* ;

import w3c.mime.MIMEType ;

public class Reply {
    static final byte eol[] = { (byte) 13, (byte) 10 } ;

    static final MIMEType DEFAULT_TYPE = MIMEType.TEXT_HTML ;
    
    static Hashtable status_messages = new Hashtable () ;

    static final int CONTENT_ENCODING_BIT          = (1<<0) ;
    static final int CONTENT_LENGTH_BIT            = (1<<1) ;
    static final int CONTENT_TYPE_BIT              = (1<<2) ;
    static final int LAST_MODIFIED_BIT             = (1<<4) ;
    static final int EXPIRES_BIT                   = (1<<5) ;
    static final int LOCATION_BIT                  = (1<<6) ;
    static final int URI_HEADER_BIT                = (1<<7) ;
    static final int CONNECTION_BIT                = (1<<8) ;
    static final int WWW_AUTHENTICATE_BIT          = (1<<9) ;
    static final int STATUS_BIT                    = (1<<10) ;
    static final int KEEP_ALIVE_BIT                = (1<<11) ;
    static final int PROXY_CONNECTION_BIT          = (1<<12) ;

    int        set                   	 = 0 ;	// Values explicitly set
    Integer    status                	 = null ;
    int        content_length        	 = 0 ;
    String     content_type          	 = null ;
    String     content_encoding      	 = null ;
    String     content_transfer_encoding = null ;
    String     uri                       = null ;
    String     location                  = null ;
    String     scheme                    = null ;
    String     auth_info                 = null ;
    String     connection                = null ;
    String     proxy_connection          = null ;
    String     keep_alive                = null ;
    long       last_modified             = -1 ;
    Date       expires                   = null ;

    Vector     eh_names  = null ;	// extension header names
    Vector     eh_values = null ;	// extension header values

    InputStream is        = null ;
    Client      client    = null ;

    boolean     keep_connection = true ;
    boolean     proxy_keep_connection = true ;
    boolean     is_proxy = false ;

    String version = "HTTP/1.0" ;

    static {
	// Initialize the status message hashtable
	Hashtable sm = status_messages ;
	sm.put (HTTP.OK, " OK") ;
	sm.put (HTTP.CREATED, " Created") ;
	sm.put (HTTP.ACCEPTED, " Accepted") ;
	sm.put (HTTP.PROVISIONAL_INFORMATION, " Provisional Information") ;
	sm.put (HTTP.NO_CONTENT, "No content") ;
	sm.put (HTTP.MULTIPLE_CHOICE, " Multiple choices") ;
	sm.put (HTTP.MOVED_PERMANENTLY, " Moved permanently") ;
	sm.put (HTTP.MOVED_TEMPORARILY, " Moved temporarily") ;
	sm.put (HTTP.SEE_OTHER, " See Other") ;
	sm.put (HTTP.NOT_MODIFIED, " Not Modified") ;
	sm.put (HTTP.BAD_REQUEST, " Bad Request") ;
	sm.put (HTTP.UNAUTHORISED, " Unauthorised") ;
	sm.put (HTTP.PAYMENT_REQUIRED, " Payment required") ;
	sm.put (HTTP.FORBIDDEN, " Forbidden") ;
	sm.put (HTTP.NOT_FOUND, " Not Found") ;
	sm.put (HTTP.NOT_ALLOWED, " Not Allowed") ;
	sm.put (HTTP.NONE_ACCEPTABLE, " No variants acceptable") ;
	sm.put (HTTP.PROXY_AUTH_REQUIRED, " Proxy authentication required") ;
	sm.put (HTTP.REQUEST_TIMEOUT, " Request timed out") ;
	sm.put (HTTP.CONFLICT, " Conflict") ;
	sm.put (HTTP.GONE, " Gone") ;
	sm.put (HTTP.INTERNAL_SERVER_ERROR, " Internal Server Error") ;
	sm.put (HTTP.NOT_IMPLEMENTED, " Not Implemented") ;
	sm.put (HTTP.BAD_GATEWAY, " Bad Gateway") ;
	sm.put (HTTP.SERVICE_UNAVAILABLE, " Service Unavailable") ;
	sm.put (HTTP.GATEWAY_TIMEOUT, " Gateway Timeout") ;
	sm.put (HTTP.NOHEADER, "Internal status code (no headers)") ;
    }

    /**
     * Is this reply a proxy reply.
     */

    public boolean isProxy() {
	return is_proxy ;
    }

    /**
     * Mark this reply as being a proxy reply.
     */

    public void setProxy () {
	is_proxy = true ;
    }

    /**
     * Add an extension header in the reply. 
     * This mechanism allows for protocol extensions.
     * @param name The extended header name.
     * @param value The extended header value.
     */

    public void addHeader (String name, String value) {
	if ( eh_names == null )
	    eh_names = new Vector (2) ;
	if ( eh_values == null )
	    eh_values = new Vector (2) ;
	eh_names.addElement (name) ;
	eh_values.addElement (value) ;
    }

    /**
     * Check for an extension header.
     * @param name The header name to test.
     * @return <string>true</string> if the named header already has some value
     * <strong>false</strong> otherwise.
     */

    public boolean hasHeader (String name) {
	if ( eh_names == null )
	    return false ;
	int size = eh_names.size() ;
	for (int i = 0 ; i < size ; i++) 
	    if ( name.equals(eh_names.elementAt(i)) )
		return true ;
	return false ;
    }
    
    /**
     * Set the status of this connection. 
     * Turining this toggle off will make 
     * the server break the connection at the end of request processing.
     * <p>This defaults to <b>true</b>
     * @param onoff Turn the toggle on or off.
     */

    public void setKeepConnection (boolean onoff) {
	keep_connection = onoff ;
    }

    public boolean keepConnection () {
	return keep_connection ;
    }

    /**
     * Set the status of this proxy'ed connection.
     * @param onoff Turn the toggle on or off.
     */

    public void setProxyKeepConnection (boolean onoff) {
	proxy_keep_connection = onoff ;
    }

    public boolean keepProxyConnection () {
	return proxy_keep_connection ;
    }

    /**
     * Sets the reply HTTP status.
     * If a default body for the provided status is given at configuration
     * time, this method will set the Reply stream accordingly. This means
     * that you should, as a simple rule of thumb always set the Reply status
     * <em>before</em> the Reply stream
     * @param status An Integer object indicating the status, see the publicly
     *               defined status code of this class.
     */

    public void setStatus (Integer status) {
	set |= STATUS_BIT ;
	this.status = status ;
    }

    /**
     * Sets the reply HTTP status.
     * <p>This method takes a true integer instead of an Integer object. Use
     * it only when required.
     */

    public void setStatus (int status) {
	switch (status) {
	  case 200: setStatus (HTTP.OK) ; break ;
	  case 201: setStatus (HTTP.CREATED) ; break ;
	  case 202: setStatus (HTTP.ACCEPTED) ; break ;
	  case 203: setStatus (HTTP.PROVISIONAL_INFORMATION) ; break ;
	  case 204: setStatus (HTTP.NO_CONTENT) ; break ;
	  case 300: setStatus (HTTP.MULTIPLE_CHOICE) ; break ;
	  case 301: setStatus (HTTP.MOVED_PERMANENTLY) ; break ;
	  case 302: setStatus (HTTP.MOVED_TEMPORARILY) ; break ;
	  case 303: setStatus (HTTP.SEE_OTHER) ; break ;
	  case 304: setStatus (HTTP.NOT_MODIFIED) ; break ;
	  case 400: setStatus (HTTP.BAD_REQUEST) ; break ;
	  case 401: setStatus (HTTP.UNAUTHORISED) ; break ;
	  case 402: setStatus (HTTP.PAYMENT_REQUIRED) ; break ;
	  case 403: setStatus (HTTP.FORBIDDEN) ; break ;
	  case 404: setStatus (HTTP.NOT_FOUND) ; break ;
	  case 405: setStatus (HTTP.NOT_ALLOWED) ; break ;
	  case 406: setStatus (HTTP.NONE_ACCEPTABLE) ; break ;
	  case 407: setStatus (HTTP.PROXY_AUTH_REQUIRED) ; break ;
	  case 408: setStatus (HTTP.REQUEST_TIMEOUT) ; break ;
	  case 409: setStatus (HTTP.CONFLICT) ; break ;
	  case 410: setStatus (HTTP.GONE) ; break ;
	  case 500: setStatus (HTTP.INTERNAL_SERVER_ERROR) ; break ;
	  case 501: setStatus (HTTP.NOT_IMPLEMENTED) ; break ;
	  case 502: setStatus (HTTP.BAD_GATEWAY) ; break ;
	  case 503: setStatus (HTTP.SERVICE_UNAVAILABLE) ; break ;
	  case 504: setStatus (HTTP.GATEWAY_TIMEOUT) ; break ;
	  default:
	      // FIXME Should trigger a Runtime exception
	      System.out.println ("Invalid status setting.") ;
	      System.exit (0) ;
	}
    }

    /**
     * Get this reply current status.
     */

    public Integer getStatus () {
	if ( (set & STATUS_BIT) == 0 ) 
	    throw new HTTPRuntimeException(this
				           , "getStatus"
				           , "no status defined.");
	return status ;
    }

    /**
     * Set this reply content encoding.
     * @param content_encoding The String describing the content encoding.
     */

    public void setContentEncoding (String content_encoding) {
	set |= CONTENT_ENCODING_BIT ;
	this.content_encoding = content_encoding ;
    }

    /**
     * Does the reply caries content length information ?
     * @return <strong>true</strong> if content length is available.
     */

    public boolean hasContentLength () {
	return ( (set & CONTENT_LENGTH_BIT) != 0 ) ;
    }

    /**
     * Set this reply content length.
     * @param len The content length.
     */

    public void setContentLength (int len) {
	set |= CONTENT_LENGTH_BIT ;
	this.content_length = len ;
    }

    /**
     * Unset the reply content length.
     * Usefull if you change on the fly the reply content.
     */

    public void unsetContentLength () {
	set &= (~CONTENT_LENGTH_BIT) ;
    }

    /**
     * Set this reply's content language.
     * @param lang The language tag(s)
     */

    public void setContentLanguage(String lang) {
	addHeader("Content-Language", lang) ;
    }

    /**
     * Get this reply content length.
     */

    public int getContentLength () {
	if ( (set & CONTENT_LENGTH_BIT) == 0 ) 
	    throw new HTTPRuntimeException (this
					    , "getContentLength"
					    , "no content-length set.") ;
	return content_length ;
    }			     

    /**
     * Set this reply content type.
     * @param type The reply content MIME type
     * @see w3c.mime.MIMEType
     */

    public void setContentType (MIMEType type) {
	set |= CONTENT_TYPE_BIT ;
	this.content_type = type.toString() ;
    }

    /**
     * Set the reply content type.
     * @param type The content type, given as a String.
     */

    public void setContentType (String type) {
	set |= CONTENT_TYPE_BIT ;
	this.content_type = type ;
    }

    /**
     * Get this reply content-type.
     * @return A string describing the content type.
     */

    public String getContentType () {
	if ( (set & CONTENT_TYPE_BIT) == 0 ) 
	    throw new HTTPRuntimeException(this
					   , "getContentType"
					   , "no type set.") ;
	return content_type ;
    }
    
    /**
     * Set the reply last-modified header.
     * @param ld The date of the content last modification.
     */

    public void setLastModified (long ld) {
	set |= LAST_MODIFIED_BIT ;
	this.last_modified = ld ;
    }

    /**
     * Unset any previously set last-modification date.
     */

    public void unsetLastModified () {
	set &= (~LAST_MODIFIED_BIT) ;
    }

    /**
     * Set the reply body expiration date.
     * @param d The date of expiration.
     */

    public void setExpires (Date d) {
	set |= EXPIRES_BIT ;
	this.expires = d ;
    }

    /**
     * Set the reply localtion header field.
     * @ param location The value of the location header field.
     */

    public void setLocation (String location) {
	set |= LOCATION_BIT ;
	this.location = location ;
    }

    /**
     * Set this reply URI header field.
     * @param uri The value of the URI header field.
     */
     
    public void setURIHeader (String uri) {
	set |= URI_HEADER_BIT ;
	if ( this.uri == null ) 
            this.uri = "<"+uri+">" ;
	else
            this.uri += ", <"+uri+">" ;
    }

    /**
     * Set this reply URI field, with varying dimensions.
     * @param uri The reply URI.
     * @param vary_dim The varying dimensions.
     */

    public void setURIHeader (String uri, String vary_dim) {
	set |= URI_HEADER_BIT ;
	if ( this.uri == null ) 
            this.uri = "<"+uri+">;vary=<"+vary_dim+">" ;
	else
            this.uri += ", <"+uri+">;vary=<"+vary_dim+">" ;
    }

    /**
     * Set this reply authenticate header value.
     * @param scheme The authentication scheme used to protecte entity.
     * @param info Any additional informations, as described in HTTP 
     *    specification.
     */

    public void setWWWAuthenticate (String scheme, String info) {
	set |= WWW_AUTHENTICATE_BIT ;
	this.scheme    = scheme ;
	this.auth_info = info ;
    }

    /**
     * Set this reply connection header.
     * @param value The value of the connection header.
     */

    public void setConnection (String value) {
	set |= CONNECTION_BIT ;
	this.connection = value ;
    }

    /**
     * Set this proxy'ed connection header.
     * @param value The value of the proxy-connection header.
     */

    public void setProxyConnection (String value) {
	set |= PROXY_CONNECTION_BIT ;
	this.proxy_connection = value ;
    }

    /**
     * Set this reply keep alive header field value.
     * @param value The value of the keep alive header field.
     */

    public void setKeepAlive (String value) {
	set |= KEEP_ALIVE_BIT ;
	this.keep_alive = value ;
    }

    /**
     * Does this request has a connection header value defined ?
     * @return <strong>true</strong> if a value is defined.
     */
     
    public boolean hasConnection () {
	return (set & CONNECTION_BIT) != 0 ;
    }

    /**
     * Sets the stream containing the replied entity.
     * <p>This method sets up the stream containing the entity body into the 
     * reply. The provided stream should be ready to read, and will be closed 
     * once emitted.
     * @param is The stream containing the replied entity body.
     */

    public void setStream (InputStream is) {
 	if ( this.is != null ) {
	    try { this.is.close() ; } catch (IOException e) {} 
	}
	this.is = is ;
    }

    /**
     * Sets the reply stream to the given HtmlGenerator stream.
     * @param g The HtmlGenerator whose output is to be used as the reply body.
     */

    public void setStream (w3c.jigsaw.html.HtmlGenerator g) {
	g.close() ;
	setContentLength (g.length()) ;
	setContentType (g.getMIMEType()) ;
	setStream (g.getInputStream()) ;
    }

    /**
     * Open this reply body stream.
     * This is used to send the reply body back to the client.
     * @return An InputStream containing the reply body, which is dumped
     *    back to the client.
     */

    public InputStream openStream () {
	return is ;
    }

    /**
     * Try to get the rpely file descriptyor, if available.
     * @return An FileDescriptor, of <strong>null</strong> if not available.
     */

    public FileDescriptor getInputFileDescriptor ()
	throws IOException
    {
	if (is instanceof FileInputStream )
	    return ((FileInputStream) is).getFD() ;
	return null ;
    }

    /**
     * Should this reply be chunked ?
     * @return If so, the reply should prepare itself to send back the
     *     appropriate transfer encoding header, and return 
     *     <strong>true</strong>, otherwise it should just return
     *     <strong>false</strong>.
     */

    protected Boolean chunkable = null ;

    public boolean canChunkTransfer() {
	// Have we already compute this ?
	if ( chunkable == null ) {
	    // Compute wether we can chunk the reply:
	    if ( hasContentLength() ) {
		chunkable = Boolean.FALSE ;
	    } else if ( version.equals("HTTP/1.1") ) {
		addHeader("Transfer-Encoding", "chunked") ;
		chunkable = Boolean.TRUE ;
	    } else {
		chunkable = Boolean.FALSE ;
	    }
	}
	return (chunkable == Boolean.TRUE) ;
    }

    /**
     * Set this reply content.
     * This method allows to set the reply content to a simple String instance.
     * @param msg The reply content.
     */

    public void setContent (String msg) {
	if ( (set & CONTENT_TYPE_BIT) == 0 )
	    setContentType (DEFAULT_TYPE) ;
	setContentLength (msg.length()) ;
	is = new StringBufferInputStream (msg) ;
    }

    // Emit header:
    // See, I am not using StringBuffer here, this should save some contention
    // for the monitor cache, although, I am not sure this is really the hot 
    // spot right now. 
    // Anyway, I am also saving a number of StringBuffer allocation, with
    // regard to the naive way of coding this.

    byte buffer[] = new byte[256] ;
    int  bufptr   = 0 ;

    private final void flush() 
	throws IOException
    {
	if ( client.output != null ) {
	    client.output.write(buffer, 0, bufptr) ;
	    bufptr = 0 ;
	} else {
	    throw new IOException("Client output stream closed.");
	}
    }
    
    private final void flush(byte buf[]) 
	throws IOException
    {
	flush() ;
	client.output.write(buf, 0, buf.length) ;
    }

    private final void flush(String s) 
	throws IOException
    {
	flush() ;
	client.output.writeBytes(s) ;
    }

    private final void append(String s) 
	throws IOException
    {
	int len = s.length() ;
	if (bufptr + len >= buffer.length) 
	    flush() ;
	if (len > buffer.length) {
	    flush(s) ;
	} else {
	    s.getBytes(0, len, buffer, bufptr) ;
	    bufptr += len ;
	}
    }
    
    private final void append(byte ch) 
	throws IOException
    {
	if ( bufptr+1 >= buffer.length )
	    flush() ;
	buffer[bufptr++] = ch ;
    }

    private final void append(byte buf[]) 
	throws IOException
    {
	if ( bufptr + buf.length >= buffer.length )
	    flush() ;
	if (buf.length >= buffer.length) {
	    flush(buf) ;
	} else {
	    System.arraycopy(buf, 0, buffer, bufptr, buf.length);
	    bufptr += buf.length ;
	}
    }

    private final void append(String s1, String s2) 
	throws IOException
    {
	append(s1) ;
	append((byte) ' ') ;
	append(s2) ;
	append(eol) ;
    }

    private final void append(String s1, String s2, String s3) 
	throws IOException
    {
	append(s1);
	append((byte) ' ');
	append(s2) ;
	append((byte) ' ');
	append(s3);
	append(eol) ;
    }
	
    private final void append(String s1, byte sep, String s2) 
	throws IOException
    {
	append(s1);
	append(sep);
	append(s2);
	append(eol);
    }

    /**
     * Emit this reply headers.
     * @exception IOException If IO error occurs while sending the reply
     *    headers.
     */

    public void emit () 
	throws IOException
    {
	if ( version.equals("HTTP/0.9") )
	    return ;
	if ( (set & STATUS_BIT ) == 0 ) {
	    client.trace ("Invalid reply header !") ;
	    return ;
	}
	if ( status == HTTP.NOHEADER )
	    return ;
	// Status line (HTTP/1.0, p 26)
	append("HTTP/1.0 "
	       , status.toString()
	       , (String) status_messages.get (status));
	// General-Header (HTTP/1.0, p 14)
	// Response-Header (HTTP/1.0, p 33)
	if ( (set & WWW_AUTHENTICATE_BIT) != 0 ) 
	    append("WWW-Authenticate:", scheme, auth_info) ;
	// Entity-Header (HTTP/1.0, p 35)
	if ( (set & CONTENT_ENCODING_BIT) != 0 ) 
	    append("Content-Encoding:", content_encoding) ;
	if ( (set & CONTENT_LENGTH_BIT) != 0 ) 
	    append("Content-Length:", Integer.toString(content_length)) ;
	if ( (set & CONTENT_TYPE_BIT) != 0 ) 
	    append("Content-Type:", content_type) ;
	if ( (set & EXPIRES_BIT) != 0 ) 
	    append("Expires:", expires.toGMTString()) ;
	if ( (set & LAST_MODIFIED_BIT) != 0 ) 
	    append("Last-Modified:", new Date(last_modified).toGMTString()) ;
	if ( (set & LOCATION_BIT) != 0 )
            append("Location:", location) ;
	if ( (set & URI_HEADER_BIT) != 0 )
            append("URI:", uri) ;
	if ( (set & CONNECTION_BIT) != 0 )
	    append("Connection:", connection) ;
	if ( (set & PROXY_CONNECTION_BIT) != 0 )
	    append("Proxy-Connection:", proxy_connection) ;
	if ( (set & KEEP_ALIVE_BIT) != 0 )
	    append("Keep-Alive:", keep_alive) ;
	append("Server:", client.getServer().getSoftware()) ;
	append("Date:", (new Date()).toGMTString()) ;
	// Extension headers (if any):
	if ( eh_names != null ) {
	    for (int i = 0 ; i < eh_names.size() ; i++) {
		append((String) eh_names.elementAt(i)
		       , (byte) ':'
		       , (String) eh_values.elementAt(i)) ;
	    }
	}
	append(eol);
	flush() ;
	// Entity-Body goes here.
    }

    /**
     * Create a new Reply instance for the given client.
     * @param client The client to who this reply  is directed.
     */

    public Reply (Client client) {
	this.client = client ;
    }

    /**
     * Create a new reply for the given client.
     * @param client The client ot who the reply is directed.
     * @reply status The reply status code.
     */

    public Reply (Client client, String version, Integer status) {
	this (client) ;
	this.version = version ;
	this.setStatus (status) ;
    }
}
