// CommonLogger.java
// $Id: CommonLogger.java,v 1.1 1996/04/10 13:44:35 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.http ;

import java.io.* ;
import java.util.Date ;

import w3c.mime.MIMEException ;

/**
 * The CommonLogger class implements the abstract Logger class.
 * The resulting log will conform to the 
 * <a href="http://www.w3.org/hypertext/WWW/Daemon/User/Config/Logging.html#common-logfile-format">common log format</a>).
 * @see w3c.jigsaw.core.Logger
 */

public class CommonLogger extends Logger implements PropertyMonitoring {
    /**
     * Name of the property indicating the log file.
     * This property indicates the name of the log file to use.
     * <p>This property defaults to the <code>log</code> file in the server
     * log directory.
     */
    public static final String LOGNAME_P = "w3c.jigsaw.logger.logname" ;
    /**
     * Name of the property indicating the error log file.
     * This property indicates the name of the error log file to use.
     * <p>This property defaults to the <code>errlog</code> file in the
     * server log directory.
     */
    public static final String ERRLOGNAME_P = "w3c.jigsaw.logger.errlogname" ;
    /**
     * Name of the property indicating the server trace file.
     * This property indicates the name of the trace file to use.
     * <p>This property defaults to the <code>trace</code> file in the 
     * server log directory.
     */
    public static final String TRACELOGNAME_P="w3c.jigsaw.logger.tracelogname";

    private   byte             msgbuf[] = null ;
    protected RandomAccessFile log      = null ;
    protected RandomAccessFile errlog   = null ;
    protected RandomAccessFile trace    = null ;
    protected httpd            server   = null ;
    protected httpdProperties  props    = null ;
    protected String           logdir   = "logs" ;

    /**
     * Property monitoring for the logger.
     * The logger allows you to dynamically (typically through the property
     * setter) change the names of the file to which it logs error, access
     * and traces.
     * @param name The name of the property that has changed.
     * @return A boolean, <strong>true</strong> if the change was made, 
     *    <strong>false</strong> otherwise.
     */

    public boolean propertyChanged (String name) {
	if ( name.equals (LOGNAME_P) ) {
	    try {
		openLogFile () ;
	    } catch (Exception e) {
		e.printStackTrace() ;
		return false ;
	    }
	    return true ;
	} else if ( name.equals (ERRLOGNAME_P) ) {
	    try {
		openErrorLogFile() ;
	    } catch (Exception e) {
		e.printStackTrace() ;
		return false ;
	    } 
	    return true ;
	} else if ( name.equals (TRACELOGNAME_P) ) {
	    try {
		openTraceFile() ;
	    } catch (Exception e) {
		e.printStackTrace() ;
		return false ;
	    }
	    return true ;
	} else {
	    return true ;
	}
    }

    /**
     * Output the given message to the given RandomAccessFile.
     * This method makes its best effort to avoid one byte writes (which you
     * get when writing the string as a whole). It first copies the string 
     * bytes into a private byte array, and than, write them all at once.
     * @param f The RandomAccessFile to write to, which should be one of
     *    log, errlog or trace.
     * @param msg The message to be written.
     * @exception IOException If writing to the output failed.
     */

    protected synchronized void output (RandomAccessFile f, String msg)
	throws IOException
    {
	int len = msg.length() ;
	if ( len > msgbuf.length ) 
	    msgbuf = new byte[len] ;
	msg.getBytes (0, len, msgbuf, 0) ;
	f.write (msgbuf, 0, len) ;
    }

    protected void logmsg (String msg) {
	if ( log != null ) {
	    try {
		output (log, msg) ;
	    } catch (IOException e) {
		throw new HTTPRuntimeException (this,"logmsg",e.getMessage()) ;
	    }
	}
    }
    
    protected synchronized void errlogmsg (String msg) {
	if ( errlog != null ) {
	    try {
		output (errlog, msg) ;
	    } catch (IOException e) {
		throw new HTTPRuntimeException (this
						, "errlogmsg"
						, e.getMessage()) ;
	    }
	}
    }
    
    protected synchronized void tracemsg (String msg) {
	if ( trace != null ) {
	    try {
		output (trace, msg) ;
	    } catch (IOException e) {
		throw new HTTPRuntimeException (this
						, "tracemsg"
						, e.getMessage()) ;
	    }
	}
    }

    public void log (Request request, Reply reply, int nbytes, long duration) {
	Client client = request.getClient() ;
	String entry  = null ;
	Date   now    = new Date() ;

	entry = client.getInetAddress()		// client inet address
	    + " " + "-"				// user name
	    + " " + request.getUser("-")		// auth user name
	    + " [" + (now.getDate() 		// current date
		      + "/" + now.getMonth()
		      + "/" + (now.getYear() + 1900)
		      + ":" + now.getHours()
		      + ":" + now.getMinutes()
		      + ":" + now.getSeconds()
		      + " +" + (now.getTimezoneOffset() / 60)
		      + "]")
	    + " \"" + request.getMethod("-")	// request line
	    + " " + request.getURI("-")
	    + " " + request.getHTTPVersion("-")
	    + "\" " + reply.getStatus()		// reply status
	    + " " + nbytes				// # of emited bytes
	    + "\n" ;
	logmsg (entry) ;
    }
    
    public void errlog (Client client, String msg) {
	errlogmsg (client.toString() + ": " + msg + "\n") ;
    }

    public void errlog (String msg) {
	errlogmsg ("<httpd>: " + msg + "\n") ;
    }

    public void trace (Client client, String msg) {
	tracemsg (client.toString() + ": " + msg + "\n") ;
    }

    public void trace (String msg) {
	tracemsg ("<httpd>: " + msg + "\n") ;
    }

    /**
     * Get the name for the file indicated by the provided property.
     * This method first looks for a property value. If none is found, it
     * than constructs a default filename from the server root, by 
     * using the provided default name.
     * <p>This method shall either succeed in getting a filename, or throw
     * a runtime exception.
     * @param propname The name of the property.
     * @param def The default file name to use.
     * @exception HTTPRuntimeException If no file name could be deduced from
     *     the provided set of properties.
     */

    protected String getFilename (String propname, String def) {
	String filename = props.getString (propname, null) ;
	if ( filename == null ) {
	    String root_dir = props.getString (httpd.ROOT_P, null) ;
	    if ( root_dir == null ) {
		String msg = "unable to build a default value for the \""
		    + propname + "\" value." ;
		throw new HTTPRuntimeException (this.getClass().getName()
						, "getFilename"
						, msg) ;
	    }
	    File flogdir = new File(root_dir, logdir) ;
	    return (new File(flogdir, def)).getAbsolutePath() ;
	} else {
	    return filename ;
	}
    }

    /**
     * Open this logger log file.
     */

    protected void openLogFile () {
	String logname = getFilename(LOGNAME_P, "log") ;
	try {
	    RandomAccessFile old = log ;
	    log = new RandomAccessFile (logname, "rw") ;
	    log.seek (log.length()) ;
	    if ( old != null )
		old.close () ;
	} catch (IOException e) {
	    throw new HTTPRuntimeException (this.getClass().getName()
					    , "openLogFile"
					    , "unable to open "+logname);
	}
    }

    /**
     * Open this logger error log file.
     */

    protected void openErrorLogFile () {
	String errlogname = getFilename (ERRLOGNAME_P, "errlog") ;
	try {
	    RandomAccessFile old = errlog ;
	    errlog = new RandomAccessFile (errlogname, "rw") ;
	    errlog.seek (errlog.length()) ;
	    if ( old != null )
		old.close() ;
	} catch (IOException e) {
	    throw new HTTPRuntimeException (this.getClass().getName()
					    , "openErrorLogFile"
					    , "unable to open "+errlogname);
	}
    }

    /**
     * Open this logger trace file.
     */

    protected void openTraceFile () {
	String tracename = getFilename (TRACELOGNAME_P, "traces");
	try {
	    RandomAccessFile old = trace ;
	    trace = new RandomAccessFile (tracename, "rw") ;
	    trace.seek (trace.length()) ;
	    if ( old != null )
		old.close() ;
	} catch (IOException e) {
	    throw new HTTPRuntimeException (this.getClass().getName()
					    , "openTraceFile"
					    , "unable to open "+tracename);
	}
    }

    /**
     * Shutdown this logger.
     */

    public synchronized void shutdown () {
	server.getProperties().unregisterObserver (this) ;
	try {
	    log.close() ; 
	    log = null ;
	    errlog.close() ;
	    errlog = null ;
	    trace.close() ;
	    trace = null ;
	} catch (IOException e) {
	    e.printStackTrace() ;
	}
    }
		
    /**
     * Initialize this logger for the given server.
     * This method gets the server properties describe above to
     * initialize its various log files.
     * @param server The server to which thiss logger should initialize.
     */

    public void initialize (httpd server) {
	this.server = server ;
	this.props  = server.getProperties() ;
	// Register for property changes:
	props.registerObserver (this) ;
	// Open the various logs:
	openLogFile () ;
	openErrorLogFile() ;
	openTraceFile() ;
	return ;
    }
	
    /**
     * Construct a new Logger instance.
     */
     
    CommonLogger () {
	this.msgbuf = new byte[128] ;
    }
}

