// ClientPool.java
// $Id: ClientPool.java,v 1.8 1996/05/29 22:10:45 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.http ;

import java.io.* ;
import java.net.* ;

class ClientState {
    // LRU for clients:
    ClientState next = null ;
    ClientState prev = null ;
    // Client being wrapped, and potential lock (when client is buzy)
    Client  client = null ;
    boolean locked = false ;
    boolean idle   = false ;

    /**
     * THis client enters its idle state.
     */

    final synchronized void markIdle() {
	idle = true ;
    }

    /**
     * Is this client currently idle ?
     * @return A boolean <strong>true</strong> if the client is idle.
     */

    final synchronized boolean isIdle() {
	return idle ;
    }

    /**
     * LRU management - Remove this client from the lru list.
     */

    final synchronized void LRUremove(ClientState head) {
	if ((prev != null) || (next != null)) {
	    synchronized (head) {
		prev.next = next ;
		next.prev = prev ;
		next = null ;
		prev = null ;
	    }
	}
    }

    /**
     * LRU management - Bring this client to the fromt of the lru list.
     * If the client is not registered in the list yet, add to the list first.
     */

    final synchronized void LRUfront(ClientState head) {
	idle = false ;
	if ((prev == null) && (next == null)) {
	    LRUinsert(head) ;
	} else {
	    synchronized(head) {
		prev.next = next ;
		next.prev = prev ;
		head.next.prev = this ;
		next           = head.next ;
		prev           = head ;
		head.next      = this ;
	    }
	}
    }

    /**
     * LRU management - Insert before given tail.
     */

    final synchronized void LRUinsert(ClientState head) {
	synchronized (head) {
	    head.next.prev = this ;
	    next           = head.next ;
	    prev           = head ;
	    head.next      = this ;
	}
    }

    ClientState(Client client, boolean locked) {
	this.client = client ;
	this.locked = locked ;
    }
}

/**
 * The client pool is a kind of client factory.
 * Each time the server gets a new connection, it calls the client pool
 * to bound a client object (newly created or spared) to handle it.
 */

public class ClientPool implements PropertyMonitoring {
    /**
     * The name of the property giving the maximum number of allowed clients.
     * This property gives the maximum number of simultaneous connections to
     * the server. When this number is exceeded, any incomming connection
     * will be rejected.
     */
    public final static String MAX_CLIENT_P = "w3c.jigsaw.client.max" ;
    /**
     * The name of the property giving the number of clients to keep.
     * The ClientPool object keep alive this number of clients, ready
     * to handle connections. 
     */
    public final static String MIN_CLIENT_P = "w3c.jigsaw.client.min" ;
     
    int             max       = 100 ;	// max number of allowed clients.
    int             min       = 32 ;	// number of clients to keep.
    int             count     = 0 ;	// number of created clients.
    httpd           server    = null ;
    ClientState     clients[] = null ;
    httpdProperties props     = null ;
    int             buzy      = 0 ;	// number of buzy clients.

    ClientState head = null ;
    ClientState tail = null ;

    /**
     * Resize our client array to the given number.
     * @param newmax The new maximum size.
     */

    protected synchronized void resizeClientArray (int newmax) {
	ClientState newclients[] = new ClientState[newmax] ;
	System.arraycopy (clients, 0, newclients, 0, clients.length) ;
	clients = newclients ;
	max     = newmax ;
    }
	
    /**
     * Create a new client for this pool.
     * @param locked Mark this client as buzy.
     */

    protected synchronized ClientState addClient (boolean locked) {
	if ( count >= max )
	    return null ;
	// Find the free slot, and create a new client. Insert in LRU.
	for (int i = 0 ; i < max ; i++) {
	    if ( clients[i] == null ) {
		Client      client = new Client(server, this, i) ;
		ClientState cs     = new ClientState(client, locked) ;
		count++ ;
		clients[i] = cs ;
		return cs ;
	    }
	}
	return null ;
    }

    protected void setMaxConnection () {
	int newmax = props.getInteger (MAX_CLIENT_P, max) ;
	if ( newmax > max ) {
	    resizeClientArray (newmax) ;
	} else if ( newmax < max ) {
	    max = newmax ;
	}
    }

    /**
     * Kill some client.
     * Our buzy watermark exceeds the config given min spare client. Pick
     * clients in LRU and kill them until the buzy count reaches the
     * water mark again.
     */

    protected synchronized void killSomeClients() {
	ClientState ptr  = tail.prev ;
	int         bcnt = buzy ;

	while ( ptr != null ) {
	    // Is this client kill'able ?
	    if ( ptr.isIdle() ) {
		ptr.client.kill(true) ;
		if ( --bcnt <= min )
		    break ;
	    }
	    ptr = ptr.next ;
	}
    }
     
    /**
     * Notify that this client has finished with its connection.
     * If the pool wants the client to be freed (because it has too many of 
     * them), it makes the client kill itself (which will trigger a call to
     * the clientFinished method, were enventual cleanup is performed).
     * @param client The client that is done with its connection.
     */

    protected synchronized void clientConnectionFinished (Client client) {
	// Cleanup our client state:
	int id = client.getIdentifier() ;
	clients[id].LRUremove(head) ;
	buzy-- ;
	// If we are in client excess, kill the client.
	if ( buzy > min ) {
	    // Kill this client, locking it until it dies.
	    clients[id].locked = true ;
	    client.kill (true) ;
	} else {
	    clients[id].locked = false ;
	}
    }

    /**
     * Notify that this client has been killed.
     * @param client The client that has terminate.
     */

    protected synchronized void clientFinished (Client client) {
	int id = client.getIdentifier() ;
	// If this client is still in LRU (which happens during shutdown)
	clients[id].LRUremove(head) ;
	// Cleanup the associated client state object:
	count-- ;
	if (clients[id].client == client)
	    clients[id] = null ;
    }

    /**
     * The client notifies the pool that is has been activated.
     * The client state object is updated to unmark the client as idle.
     * <p>This method needs not be synchronized, as it affect only the client
     * state, <em>not</em> the client list.
     * @param client The activated client.
     */

    protected void notifyUse(Client client) {
	clients[client.getIdentifier()].LRUfront(head) ;
    }

    /**
     * The client notifies the pool that it enters idle state.
     * <p>This method needs not be synchronized, as it affect only the client
     * state, <em>not</em> the client list.
     * @param client The client that is going to be idle.
     */

    protected void notifyIdle(Client client) {
	clients[client.getIdentifier()].markIdle() ;
    }

    /**
     * Some property have changed, update our setting.
     * @param name The name of the property that has changed.
     * @return A boolean, <strong>true</strong> if we updated ourself 
     *    successfully.
     */

    public boolean propertyChanged (String name) {
	if ( name.equals (MAX_CLIENT_P) ) {
	    setMaxConnection() ;
	} else if ( name.equals (MIN_CLIENT_P) ) {
	    min = props.getInteger (MIN_CLIENT_P, min) ;
	} 
	return true ;
    }

    /**
     * Handle the given connection.
     * Find a free client, bind it to the given socket, and run it. If we
     * have reached our maximum allowed number of clients, kill some
     * connections.
     * <p>A client enters the LRU list (and become a candidate for kill) only
     * after it has handle one request (or if some timeout expires). This is
     * performed by the first call to <code>notifyUse</code> which will
     * silently insert the client into the LRU if it was not there already.
     * <p>This client pool does a lot of nice thinigs, but could probably be
     * implemented in a much better way (while keeping the features it has).
     * Contention arond the pool is probably concern number 1 of performances.
     * @param socket The connection to handle.
     */

    public synchronized void handleConnection (Socket socket) {
	boolean freeslot = false ;
	// Pick a free client:
	// Walking through the whole client array is  a waste of
	// time, this will (?) be changed to a more efficient list 
	// representation.
	// Although, as there is some randomness as to when a client connection
	// finishes, we are not in the worst case (were clients would always
	// be found at the end of the array).
	for (int i = 0 ; i < max ; i++) {
	    if ( clients[i] == null ) {
		freeslot = true ;
		continue ;
	    } else if ( clients[i].locked ) {
		continue ;
	    } else {
		clients[i].client.bind (socket) ;
		clients[i].locked = true ;
		if (++buzy > min )
		    killSomeClients() ;
		return ;
	    }
	}
	// No free clients, try adding a new one (in locked state)
	ClientState cs = addClient(true) ;
	if ( cs != null ) {
	    if (++buzy > min)
		killSomeClients() ;
	    cs.client.bind (socket) ;
	    return ;
	}
	// No free client available, don't accept incoming connections
	try {
	    socket.close() ;
	} catch (IOException ex) {
	}
    }

    protected synchronized void killClients(boolean force) {
	for (int i = 0 ; i < clients.length ; i++) {
	    if ( clients[i] == null )
		continue ;
	    clients[i].client.kill (force) ;
	}
    }

    protected synchronized void cleanLRU() {
	head = null ;
	tail = null ;
    }

    /**
     * Shutdown the client pool. 
     * If force is <strong>true</strong>, kill all running clients right
     * now, otherwise, wait for them to terminate gracefully, and return
     * when done.
     * @param force Should we interrupt running clients.
     */

    public void shutdown (boolean force) {
	// First stage: kill all clients (synchronized)
	killClients(force) ;
	// Second stage (unsynchronized), join all client threads
	for (int i = 0 ; i < clients.length ; i++) {
	    if ( clients[i] == null )
		continue ;
	    // Wait for the client thread to terminate
	    Thread thread = clients[i].client.getThread() ;
	    if ( (thread != null) && (thread != Thread.currentThread()) ) {
		clients[i].client.trace ("exiting.") ;
		boolean cont = true ;
		while ( cont ) {
		    try {
			thread.join() ;
			cont = false ;
		    } catch (InterruptedException ex) {
		    }
		}
	    }
	    clients[i] = null ;
	}
	// Third stage (synchronized), clean up the lru list.
	cleanLRU() ;
    }

    ClientPool (httpd server) {
	this.server = server ;
	this.props  = server.getProperties() ;
	this.props.registerObserver (this) ;
	this.max    = props.getInteger (MAX_CLIENT_P, max) ;
	this.min    = props.getInteger (MIN_CLIENT_P, min) ;
	// Create the fake head an tail of LRU list:
	head = new ClientState(null, false) ;
	tail = new ClientState(null, false) ;
	head.next = tail ;
	head.prev = null ;
	tail.next = null ;
	tail.prev = head ;
	// Create all our clients:
	clients = new ClientState[max] ;
	for (int i = 0 ; i < min ; i++) {
	    if ( addClient(false) == null )
		throw new RuntimeException (this.getClass().getName()
					    + "[construstructor]"
					    + ": unable to create clients.");
	}
    }

}
