// PostableResource.java
// $Id: PostableResource.java,v 1.1 1996/04/10 13:42:05 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.forms ;

import java.io.* ;
import java.util.*;

import w3c.mime.* ;
import w3c.jigsaw.http.* ;
import w3c.jigsaw.resources.*;
import w3c.jigsaw.html.HtmlGenerator ;

/**
 * This class is the root class for  handling forms inside the server.
 * It will decode the form content, and trigger a call to the 
 * <b>handle</b> method of the instance.
 * <p>It inherits from the <b>FileResource</b> class to be able to send the
 * form text, and extends it with an implementation of the <b>post</b> method.
 * @see w3c.jigsaw.resources.FileResource
 */

public class PostableResource extends FileResource {
    private static MIMEType type = MIMEType.APPLICATION_X_WWW_FORM_URLENCODED ;
    /**
     * Attribute index - Should we overide form values when multiple ?
     */
    protected static int ATTR_OVERIDE = -1 ;
    /**
     * Attribute index - Should we silently convert GET to POST methods ?
     */
    protected static int ATTR_CONVERT_GET = -1 ;

    static {
	Attribute a   = null ;
	Class     cls = null ;
	try {
	    cls = Class.forName("w3c.jigsaw.forms.PostableResource") ;
	} catch (Exception ex) {
	    ex.printStackTrace() ;
	    System.exit(1) ;
	}
	// The overide attribute:
	a = new BooleanAttribute("overide"
				 , Boolean.FALSE
				 , Attribute.EDITABLE);
	ATTR_OVERIDE = AttributeRegistery.registerAttribute(cls, a) ;
	// The convert get attribute:
	a = new BooleanAttribute("convert-get"
				 , Boolean.TRUE
				 , Attribute.EDITABLE) ;
	ATTR_CONVERT_GET = AttributeRegistery.registerAttribute(cls, a) ;
    }

    /**
     * Get the 'convert GET to POST' flag.
     */

    public boolean getConvertGetFlag() {
	return getBoolean(ATTR_CONVERT_GET, false) ;
    }

    /**
     * Get the 'overide multiple form field value' flag.
     */

    public boolean getOverideFlag() {
	return getBoolean(ATTR_OVERIDE, true) ;
    }

    /**
     * Get this resource body.
     * If we are allowed to convert GET requests to POST, than we first
     * check to see if there is some search string in the request, and continue
     * with normal POST request processing.
     * <p>If there is no search string, or if we are not allowed to convert
     * GETs to POSTs, than we just invoke our <code>super</code> method,
     * which will perform the appropriate job.
     * @param request The request to handle.
     * @exception HTTPException If request couldn't be processed.
     */
     
    public Reply get (Request request) 
	throws HTTPException
    {
	// Check if we should handle it (is it a POST disguised in GET ?)
	if ((! getConvertGetFlag()) || ( ! request.hasField("query")))
	    return super.get (request) ;
	// Get the request entity, and decode it:
	String      query = request.getQueryString(null) ;
	InputStream in    = new StringBufferInputStream(query) ;
	URLDecoder  d     = new URLDecoder (in, getOverideFlag()) ;
	try {
	    d.parse () ;
	} catch (URLDecoderException e) {
	    Reply error = request.makeReply(HTTP.BAD_REQUEST) ;
	    error.setContent("Invalid request:unable to decode form data.");
	    throw new HTTPException (error) ;
	} catch (IOException e) {
	    Reply error = request.makeReply(HTTP.BAD_REQUEST) ;
	    error.setContent("Invalid request: unable to read form data.");
	    throw new HTTPException (error) ;
	}
	return handle (request, d) ;
    }

    /**
     * The POST method handling.
     * This method decode the posted data, and triggers a call to the
     * <strong>handle</strong> method.
     * @param request The request to handle.
     * @exception HTTPException If request processing failed.
     */

    public Reply post (Request request) 
	throws HTTPException
    {
	// Check that we are dealing with an application/x-www-form-urlencoded:
	if ( (! request.hasField ("content-type"))
	     || (type.match(request.getContentType()) < 0) ) {
	    Reply error = request.makeReply(HTTP.BAD_REQUEST) ;
	    throw new HTTPException (error) ;
	}
	// Get and decode the request entity:
	InputStream rin = request.getInputStream() ;
	int         len = request.getContentLength();
	InputStream in  = new MIMEContentLengthInputStream (rin, len) ;
	URLDecoder dec  = new URLDecoder (in, getOverideFlag()) ;
	try {
	    dec.parse () ;
	} catch (URLDecoderException e) {
	    Reply error = request.makeReply(HTTP.BAD_REQUEST) ;
	    error.setContent("Invalid request: unable to decode form data.") ;
	    throw new HTTPException (error) ;
	} catch (IOException e) {
	    Reply error = request.makeReply(HTTP.BAD_REQUEST) ;
	    error.setContent("Invalid request: unable to read form data.") ;
	    throw new HTTPException (error) ;
	}
	// Handle the stuff:
	return handle (request, dec) ;
    }

    /**
     * Handle the form submission, after posted data parsing.
     * <p>This method ought to be abstract, but for reasonable reason, it
     * will just dump (parsed) the form content back to the client, so that it
     * can be used for debugging.
     * @param request The request proper.
     * @param data The parsed data content.
     * @exception HTTPException If form data processing failed.
     * @see w3c.jigsaw.forms.URLDecoder
     */

    public Reply handle (Request request, URLDecoder data)
	throws HTTPException 
    {
	// Now we just dump back the variables we got:
	Enumeration   e = data.keys() ;
	HtmlGenerator g = new HtmlGenerator ("Form decoded values") ;
	g.append ("<p>List of variables and values:</p><ul>") ;
	while ( e.hasMoreElements () ) {
	    String name = (String) e.nextElement() ;
	    g.append ("<li><em>"
		      + name+"</em> = <b>"
		      + data.getValue(name)
		      + "</b></li>");
	}
	g.append ("</ul>") ;
	Reply reply = request.makeReply(HTTP.OK) ;
	reply.setStream (g) ;
	return reply ;
    }

}
