#include <kernel.h>
#include "iodlg.h"
#include "mpeg.h"
//
l_ulong AppVersion	= ULONG_ID(0,0,0,1);
char AppName[]		= "Mpeg Dialog Player";
l_uid	nUID		= "dMpegPlay";
l_uid NeededLibs[]	= { "allegui", "" };
//
PFileTypes Filter = 0;
//
/***************************************************************************/
static inline void mpeg_bits_fill_reverse_bits( mpeg_bits_t* stream, int bits )
{
   while( stream->bit_number > 7 )
   {
      stream->bfr >>= 8;
      stream->bfr_size -= 8;
      stream->bit_number -= 8;
   }
   while( stream->bfr_size - stream->bit_number < bits )
   {
      if ( stream->input_ptr )
         stream->bfr |= ( unsigned int )( *--stream->input_ptr ) << stream->bfr_size;
      else
         stream->bfr |= ( unsigned int )mpeg_demux_read_prev_char( stream->demuxer ) << stream->bfr_size;
      stream->bfr_size += 8;
   }
}
/***************************************************************************/
static inline void mpeg_bits_fill_bits( mpeg_bits_t* stream, int bits )
{
   while( stream->bit_number < bits )
   {
      stream->bfr <<= 8;
      if ( stream->input_ptr )
      {
         stream->bfr |= *stream->input_ptr++;
      }
      else
      {
         stream->bfr |= mpeg_demux_read_char( stream->demuxer );
      }
      stream->bit_number += 8;
      stream->bfr_size += 8;
      if ( stream->bfr_size > 32 )
         stream->bfr_size = 32;
   }
}
/***************************************************************************/
static inline unsigned int mpeg_bits_getbyte_noptr( mpeg_bits_t* stream )
{
   if ( stream->bit_number < 8 )
   {
      stream->bfr <<= 8;
      if ( stream->input_ptr )
         stream->bfr |= *stream->input_ptr++;
      else
         stream->bfr |= mpeg_demux_read_char( stream->demuxer );
      stream->bfr_size += 8;
      if ( stream->bfr_size > 32 ) stream->bfr_size = 32;
         return ( stream->bfr >> stream->bit_number ) & 0xff;
   }
   //
   return ( stream->bfr >> ( stream->bit_number -= 8 ) ) & 0xff;
}
/***************************************************************************/
static inline unsigned int mpeg_bits_getbit_noptr( mpeg_bits_t* stream )
{
   if ( !stream->bit_number )
   {
      stream->bfr <<= 8;
      stream->bfr |= mpeg_demux_read_char( stream->demuxer );
      stream->bfr_size += 8;
      if ( stream->bfr_size > 32 ) stream->bfr_size = 32;
         stream->bit_number = 7;
      return ( stream->bfr >> 7 ) & 0x1;
   }
   //
   return ( stream->bfr >> ( --stream->bit_number ) ) & ( 0x1 );
}
/***************************************************************************/
static inline unsigned int mpeg_bits_getbits( mpeg_bits_t* stream, int bits )
{
   if ( bits <= 0 )
      return 0;
   mpeg_bits_fill_bits( stream, bits );
   //
   return ( stream->bfr >> ( stream->bit_number -= bits ) ) & ( 0xffffffff >> ( 32 - bits ) );
}
/***************************************************************************/
static inline unsigned int mpeg_bits_showbits24_noptr( mpeg_bits_t* stream )
{
   while( stream->bit_number < 24 )
   {
      stream->bfr <<= 8;
      stream->bfr |= mpeg_demux_read_char( stream->demuxer );
      stream->bit_number += 8;
      stream->bfr_size += 8;
      if ( stream->bfr_size > 32 )
         stream->bfr_size = 32;
   }
   //
   return ( stream->bfr >> ( stream->bit_number - 24 ) ) & 0xffffff;
}
/***************************************************************************/
static inline unsigned int mpeg_bits_showbits32_noptr( mpeg_bits_t* stream )
{
   while( stream->bit_number < 32 )
   {
      stream->bfr <<= 8;
      stream->bfr |= mpeg_demux_read_char( stream->demuxer );
      stream->bit_number += 8;
      stream->bfr_size += 8;
      if ( stream->bfr_size > 32 )
         stream->bfr_size = 32;
   }
   //
   return stream->bfr;
}
/***************************************************************************/
static inline unsigned int mpeg_bits_showbits( mpeg_bits_t* stream, int bits )
{
   mpeg_bits_fill_bits( stream, bits );
   //
   return ( stream->bfr >> ( stream->bit_number - bits ) ) & ( 0xffffffff >> ( 32 - bits ) );
}
/***************************************************************************/
static inline unsigned int mpeg_bits_getbits_reverse( mpeg_bits_t* stream, int bits )
{
   unsigned int result;
   //
   mpeg_bits_fill_reverse_bits( stream, bits );
   result = ( stream->bfr >> stream->bit_number ) & ( 0xffffffff >> ( 32 - bits ) );
   stream->bit_number += bits;
   //
   return result;
}
/***************************************************************************/
static inline unsigned int mpeg_bits_showbits_reverse( mpeg_bits_t* stream, int bits )
{
   unsigned int result;
   //
   mpeg_bits_fill_reverse_bits( stream, bits );
   result = ( stream->bfr >> stream->bit_number ) & ( 0xffffffff >> ( 32 - bits ) );
   //
   return result;
}
/***************************************************************************/
static inline unsigned int mpeg_slice_getbit( mpeg_slice_buffer_t *buffer )
{
   if ( buffer->bits_size )
      return ( buffer->bits >> ( --buffer->bits_size ) ) & 0x1;
   else if ( buffer->current_position < buffer->buffer_size )
   {
      buffer->bits = buffer->data[ buffer->current_position++ ];
      buffer->bits_size = 7;
      return ( buffer->bits >> 7 ) & 0x1;
   }
   //
   return 0;
}
/***************************************************************************/
static inline unsigned int mpeg_slice_getbits2( mpeg_slice_buffer_t *buffer )
{
   if ( buffer->bits_size >= 2 )
      return ( buffer->bits >> ( buffer->bits_size -= 2 ) ) & 0x3;
   else if ( buffer->current_position < buffer->buffer_size )
   {
      buffer->bits <<= 8;
      buffer->bits |= buffer->data[ buffer->current_position++ ];
      buffer->bits_size += 6;
      return ( buffer->bits >> buffer->bits_size ) & 0x3;
   }
   //
   return 0;
}
/***************************************************************************/
static inline unsigned int mpeg_slice_getbyte( mpeg_slice_buffer_t *buffer )
{
   if ( buffer->bits_size >= 8 )
      return ( buffer->bits >> ( buffer->bits_size -= 8 ) ) & 0xff;
   else if ( buffer->current_position < buffer->buffer_size )
   {
      buffer->bits <<= 8;
      buffer->bits |= buffer->data[ buffer->current_position++ ];
      return ( buffer->bits >> buffer->bits_size ) & 0xff;
   }
   //
   return 0;
}
/***************************************************************************/
static inline unsigned int mpeg_slice_getbits( mpeg_slice_buffer_t *slice_buffer, int bits )
{
   if ( bits == 1 )
      return mpeg_slice_getbit( slice_buffer );
   //
   mpeg_slice_fillbits( slice_buffer, bits );
   //
   return ( slice_buffer->bits >> ( slice_buffer->bits_size -= bits ) ) & ( 0xffffffff >> ( 32 - bits ) );
}
/***************************************************************************/
static inline unsigned int mpeg_slice_showbits16( mpeg_slice_buffer_t *buffer )
{
   if ( buffer->bits_size >= 16 )
      return ( buffer->bits >> ( buffer->bits_size - 16 ) ) & 0xffff;
   else if ( buffer->current_position < buffer->buffer_size )
   {
      buffer->bits <<= 16;
      buffer->bits_size += 16;
      buffer->bits |= ( unsigned int )buffer->data[ buffer->current_position++ ] << 8;
      buffer->bits |= buffer->data[ buffer->current_position++ ];
      return ( buffer->bits >> ( buffer->bits_size - 16 ) ) & 0xffff;
   }
   //
   return 0;
}
/***************************************************************************/
static inline unsigned int mpeg_slice_showbits9( mpeg_slice_buffer_t *buffer )
{
   if ( buffer->bits_size >= 9 )
      return ( buffer->bits >> ( buffer->bits_size - 9 ) ) & 0x1ff;
   else if ( buffer->current_position < buffer->buffer_size )
   {
      buffer->bits <<= 16;
      buffer->bits_size += 16;
      buffer->bits |= ( unsigned int )buffer->data[ buffer->current_position++ ] << 8;
      buffer->bits |= buffer->data[ buffer->current_position++ ];
      return ( buffer->bits >> ( buffer->bits_size - 9 ) ) & 0x1ff;
   }
   //
   return 0;
}
/***************************************************************************/
static inline unsigned int mpeg_slice_showbits5( mpeg_slice_buffer_t *buffer )
{
   if ( buffer->bits_size >= 5 )
      return ( buffer->bits >> ( buffer->bits_size - 5 ) ) & 0x1f;
   else if ( buffer->current_position < buffer->buffer_size )
   {
      buffer->bits <<= 8;
      buffer->bits_size += 8;
      buffer->bits |= buffer->data[ buffer->current_position++ ];
      return ( buffer->bits >> ( buffer->bits_size - 5 ) ) & 0x1f;
   }
   //
   return 0;
}
/***************************************************************************/
static inline unsigned int mpeg_slice_showbits( mpeg_slice_buffer_t *slice_buffer, int bits )
{
   mpeg_slice_fillbits( slice_buffer, bits );
   //
   return ( slice_buffer->bits >> ( slice_buffer->bits_size - bits ) ) & ( 0xffffffff >> ( 32 - bits ) );
}
/***************************************************************************/
static inline unsigned int mpeg_io_read_int32( mpeg_fs_t *fs )
{
   int a, b, c, d;
   unsigned int result;
   //
   a = ( unsigned char )fgetc( fs->fd );
   b = ( unsigned char )fgetc( fs->fd );
   c = ( unsigned char )fgetc( fs->fd );
   d = ( unsigned char )fgetc( fs->fd );
   result = ( ( int )a << 24 ) | ( ( int )b << 16 ) | ( ( int )c << 8 ) | ( ( int )d );
   fs->current_byte += 4;
   //
   return result;
}
/***************************************************************************/
static inline unsigned int mpeg_io_read_char( mpeg_fs_t *fs )
{
   fs->current_byte++;
   //
   return fgetc( fs->fd );
}
/***************************************************************************/
static inline uint32_t bitstream_get( uint32_t num_bits )
{
   uint32_t result;
   //
   if ( num_bits < a52_bits_left ) 
   {
      result = ( a52_current_word << ( 32 - a52_bits_left ) ) >> ( 32 - num_bits );
      a52_bits_left -= num_bits;
      return result;
   }
   //
   return a52_bitstream_get_bh( num_bits );
}
/***************************************************************************/
static inline int32_t bitstream_get_2( uint32_t num_bits )
{
   int32_t result;
   //
   if ( num_bits < a52_bits_left ) 
   {
      result = ( ( ( int32_t )a52_current_word ) << ( 32 - a52_bits_left ) ) >> ( 32 - num_bits );
      a52_bits_left -= num_bits;
      return result;
   }
   //
   return a52_bitstream_get_bh_2( num_bits );
}
/***************************************************************************/
static inline mpeg_bits_t* mpeg_bits_new_stream( MPEG *file_mpeg, mpeg_demuxer_t *demuxer )
{
   mpeg_bits_t *stream = ( mpeg_bits_t * ) calloc( 1, sizeof( mpeg_bits_t ) );
   stream->bfr = 0;
   stream->bfr_size = 0;
   stream->bit_number = 0;
   stream->file_mpeg = file_mpeg;
   stream->demuxer = demuxer;
   stream->input_ptr = 0;
   //
   return stream;
}
/***************************************************************************/
static inline int mpeg_bits_delete_stream( mpeg_bits_t* stream )
{
   free( stream );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_bits_read_buffer( mpeg_bits_t* stream, unsigned char *buffer, int bytes )
{
   int result;
   //
   while( stream->bit_number > 0 )
   {
      stream->bit_number -= 8;
      mpeg_demux_read_prev_char( stream->demuxer );
   }
   stream->bit_number = 0;
   stream->bfr_size = 0;
   stream->bfr = 0;
   result = mpeg_demux_read_data( stream->demuxer, buffer, bytes );
   //
   return result;
}
/***************************************************************************/
static inline int mpeg_bits_use_ptr( mpeg_bits_t* stream, unsigned char *buffer )
{
   stream->bfr_size = stream->bit_number = 0;
   stream->bfr = 0;
   stream->input_ptr = buffer;
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_bits_use_demuxer( mpeg_bits_t* stream )
{
   if ( stream->input_ptr )
   {
      stream->bfr_size = stream->bit_number = 0;
      stream->input_ptr = 0;
      stream->bfr = 0;
   }
   //
   return 0;
}
/***************************************************************************/
static inline void mpeg_bits_start_reverse( mpeg_bits_t* stream )
{
   int i;
   //
   for( i = 0; i < stream->bfr_size; i += 8 )
   {
      if ( stream->input_ptr )
         stream->input_ptr--;
      else
         mpeg_demux_read_prev_char( stream->demuxer );
   }
}
/***************************************************************************/
static inline void mpeg_bits_start_forward( mpeg_bits_t* stream )
{
   int i;
   //
   for( i = 0; i < stream->bfr_size; i += 8 )
   {
      if ( stream->input_ptr )
         stream->input_ptr++;
      else
         mpeg_demux_read_char( stream->demuxer );
   }
}
/***************************************************************************/
static inline int mpeg_bits_refill( mpeg_bits_t* stream )
{
   stream->bit_number = 32;
   stream->bfr_size = 32;
   //
   if ( stream->input_ptr )
   {
      stream->bfr = ( unsigned int )( *stream->input_ptr++ ) << 24;
      stream->bfr |= ( unsigned int )( *stream->input_ptr++ ) << 16;
      stream->bfr |= ( unsigned int )( *stream->input_ptr++ ) << 8;
      stream->bfr |= *stream->input_ptr++;
   }
   else
   {
      stream->bfr = mpeg_demux_read_char( stream->demuxer ) << 24;
      stream->bfr |= mpeg_demux_read_char( stream->demuxer ) << 16;
      stream->bfr |= mpeg_demux_read_char( stream->demuxer ) << 8;
      stream->bfr |= mpeg_demux_read_char( stream->demuxer );
   }
   //
   return mpeg_demux_eof( stream->demuxer );
}
/***************************************************************************/
static inline int mpeg_bits_byte_align( mpeg_bits_t *stream )
{
   stream->bit_number = ( stream->bit_number + 7 ) & 0xf8;
   //
   return 0;
}
/***************************************************************************/
static inline long mpeg_demuxer_total_bytes( mpeg_demuxer_t *demuxer )
{
   mpeg_title_t *title = demuxer->titles[ demuxer->current_title ];
   //
   return title->total_bytes;
}
/***************************************************************************/
static inline int mpeg_bits_seek_end( mpeg_bits_t* stream )
{
   stream->bfr_size = stream->bit_number = 0;
   //
   return mpeg_demux_seek_byte( stream->demuxer, mpeg_demuxer_total_bytes( stream->demuxer ) );
}
/***************************************************************************/
static inline int mpeg_bits_seek_start( mpeg_bits_t* stream )
{
   stream->bfr_size = stream->bit_number = 0;
   //
   return mpeg_demux_seek_byte( stream->demuxer, 0 );
}
/***************************************************************************/
static inline int mpeg_bits_seek_time( mpeg_bits_t* stream, double time_position )
{
   stream->bfr_size = stream->bit_number = 0;
   //
   return mpeg_demux_seek_time( stream->demuxer, time_position );
}
/***************************************************************************/
static inline int mpeg_bits_seek_byte( mpeg_bits_t* stream, long position )
{
   stream->bfr_size = stream->bit_number = 0;
   //
   return mpeg_demux_seek_byte( stream->demuxer, position );
}
/***************************************************************************/
static inline int mpeg_bits_seek_percentage( mpeg_bits_t* stream, double percentage )
{
   stream->bfr_size = stream->bit_number = 0;
   //
   return mpeg_demux_seek_percentage( stream->demuxer, percentage );
}
/***************************************************************************/
static inline int mpeg_bits_getbitoffset( mpeg_bits_t *stream )
{
   return stream->bit_number & 7;
}
/***************************************************************************/
static inline mpeg_css_t* mpeg_new_css( void )
{
   return 0;
}
/***************************************************************************/
static inline int mpeg_delete_css( mpeg_css_t *css )
{
   return 0;
}
/***************************************************************************/
static inline int mpeg_get_keys( mpeg_css_t *css, char *path )
{
   return 1;
}
/***************************************************************************/
static inline int mpeg_decrypt_packet( mpeg_css_t *css, unsigned char *sector )
{
   return 1;
}
/***************************************************************************/
static inline MPEG* mpeg_new( char *path )
{
   MPEG *file_mpeg = ( MPEG * ) calloc( 1, sizeof( MPEG ) );
   //
   file_mpeg->cpus = 1;
   file_mpeg->fs = mpeg_new_fs( path );
   file_mpeg->demuxer = mpeg_new_demuxer( file_mpeg, 0, 0, -1 );
   //
   return file_mpeg;
}
/***************************************************************************/
static inline int mpeg_delete( MPEG *file_mpeg )
{
   int i;
   //
   for( i = 0; i < file_mpeg->total_vstreams; i++ )
      mpeg_delete_vtrack( file_mpeg, file_mpeg->vtrack[ i ] );
   for( i = 0; i < file_mpeg->total_astreams; i++ )
      mpeg_delete_atrack( file_mpeg, file_mpeg->atrack[ i ] );
   mpeg_delete_fs( file_mpeg->fs );
   mpeg_delete_demuxer( file_mpeg->demuxer );
   free( file_mpeg );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_check_sig( char *path )
{
   mpeg_fs_t *fs;
   unsigned int bits;
   char *ext;
   int result = 0;
   //
   fs = mpeg_new_fs( path );
   if ( mpeg_io_open_file( fs ) )
   {
      return 0;
   }
   bits = mpeg_io_read_int32( fs );
   if ( bits == MPEG3_TOC_PREFIX || bits == MPEG3_TOC_PREFIXLOWER )
   {
      result = 1;
   }
   else if ( ( ( ( bits >> 24 ) & 0xff ) == MPEG3_SYNC_BYTE ) || ( bits == MPEG3_PACK_START_CODE ) || ( ( bits & 0xfff00000 ) == 0xfff00000 ) || ( ( bits & 0xffff0000 ) == 0xffe30000 ) || ( bits == MPEG3_SEQUENCE_START_CODE ) || ( bits == MPEG3_PICTURE_START_CODE ) || ( ( ( bits & 0xffff0000 ) >> 16 ) == MPEG3_AC3_START_CODE ) || ( ( bits >> 8 ) == MPEG3_ID3_PREFIX ) || ( bits == MPEG3_RIFF_CODE ) )
   {
      result = 1;
      ext = strrchr( path, '.' );
      if ( ext )
      {
         if ( strncasecmp( ext, ".mp2", 4 ) && strncasecmp( ext, ".mp3", 4 ) && strncasecmp( ext, ".m1v", 4 ) && strncasecmp( ext, ".m2v", 4 ) && strncasecmp( ext, ".m2s", 4 ) && strncasecmp( ext, ".mpg", 4 ) && strncasecmp( ext, ".vob", 4 ) && strncasecmp( ext, ".mpeg", 4 ) && strncasecmp( ext, ".ac3", 4 ) )
            result = 0;
      }
   }
   mpeg_io_close_file( fs );
   mpeg_delete_fs( fs );
   //
   return result;
}
/***************************************************************************/
static inline MPEG* mpeg_open_copy( char *path, MPEG *old_file )
{
   MPEG *file_mpeg = 0;
   unsigned int bits;
   int i;
   //
   file_mpeg = mpeg_new( path );
   if ( mpeg_io_open_file( file_mpeg->fs ) )
   {
      mpeg_delete( file_mpeg );
      return 0;
   }
   bits = mpeg_io_read_int32( file_mpeg->fs );
   if ( bits == MPEG3_TOC_PREFIX || bits == MPEG3_TOC_PREFIXLOWER ) 
   {
      if ( mpeg_read_toc( file_mpeg ) )
      {
         mpeg_delete( file_mpeg );
         return 0;
      }
      mpeg_io_close_file( file_mpeg->fs );
   }
   else if ( ( ( bits >> 24 ) & 0xff ) == MPEG3_SYNC_BYTE )
   {
      file_mpeg->packet_size = MPEG3_TS_PACKET_SIZE;
      file_mpeg->is_transport_stream = 1;
   }
   else if ( bits == MPEG3_PACK_START_CODE )
   {
      file_mpeg->packet_size = MPEG3_DVD_PACKET_SIZE;
      file_mpeg->is_program_stream = 1;
   }
   else if ( ( bits & 0xfff00000 ) == 0xfff00000 || ( bits & 0xffff0000 ) == 0xffe30000 || ( ( bits >> 8 ) == MPEG3_ID3_PREFIX ) || ( bits == MPEG3_RIFF_CODE ) )
   {
      file_mpeg->packet_size = MPEG3_DVD_PACKET_SIZE;
      file_mpeg->has_audio = 1;
      file_mpeg->is_audio_stream = 1;
   }
   else if ( bits == MPEG3_SEQUENCE_START_CODE || bits == MPEG3_PICTURE_START_CODE )
   {
      file_mpeg->packet_size = MPEG3_DVD_PACKET_SIZE;
      file_mpeg->is_video_stream = 1;
   }
   else if ( ( ( bits & 0xffff0000 ) >> 16 ) == MPEG3_AC3_START_CODE )
   {
      file_mpeg->packet_size = MPEG3_DVD_PACKET_SIZE;
      file_mpeg->has_audio = 1;
      file_mpeg->is_audio_stream = 1;
   }
   else
   {
      mpeg_delete( file_mpeg );
	  DebugMessage( "MpegPlayer : mpeg_open: not an MPEG 2 stream\n" );
      return 0;
   }
   if ( old_file && mpeg_get_demuxer( old_file ) )
   {
      mpeg_demux_copy_titles( file_mpeg->demuxer, mpeg_get_demuxer( old_file ) );
   }
   else if ( !file_mpeg->demuxer->total_titles )
   {
      mpeg_demux_create_title( file_mpeg->demuxer, 0, 0 );
   }
   if ( file_mpeg->is_transport_stream || file_mpeg->is_program_stream )
   {
      for( i = 0; i < MPEG3_MAX_STREAMS; i++ )
      {
         if ( file_mpeg->demuxer->vstream_table[ i ] )
         {
            file_mpeg->vtrack[ file_mpeg->total_vstreams ] = mpeg_new_vtrack( file_mpeg, i, file_mpeg->demuxer );
            if ( file_mpeg->vtrack[ file_mpeg->total_vstreams ] )
               file_mpeg->total_vstreams++;
         }
      }
      for( i = 0; i < MPEG3_MAX_STREAMS; i++ )
      {
         if ( file_mpeg->demuxer->astream_table[ i ] )
         {
            file_mpeg->atrack[ file_mpeg->total_astreams ] = mpeg_new_atrack( file_mpeg, i, file_mpeg->demuxer->astream_table[ i ], file_mpeg->demuxer );
            if ( file_mpeg->atrack[ file_mpeg->total_astreams ] )
               file_mpeg->total_astreams++;
         }
      }
   }
   else if ( file_mpeg->is_video_stream )
   {
      file_mpeg->vtrack[ 0 ] = mpeg_new_vtrack( file_mpeg, -1, file_mpeg->demuxer );
      if ( file_mpeg->vtrack[ 0 ] )
         file_mpeg->total_vstreams++;
   }
   else if ( file_mpeg->is_audio_stream )
   {
      file_mpeg->atrack[ 0 ] = mpeg_new_atrack( file_mpeg, -1, AUDIO_UNKNOWN, file_mpeg->demuxer );
      if ( file_mpeg->atrack[ 0 ] )
         file_mpeg->total_astreams++;
   }
   if ( file_mpeg->total_vstreams )
      file_mpeg->has_video = 1;
   if ( file_mpeg->total_astreams )
      file_mpeg->has_audio = 1;
   //
   mpeg_io_close_file( file_mpeg->fs );
   //
   return file_mpeg;
}
/***************************************************************************/
static inline MPEG* mpeg_open( char *path )
{
   return mpeg_open_copy( path, 0 );
}
/***************************************************************************/
static inline int mpeg_close( MPEG *file_mpeg )
{
   mpeg_delete( file_mpeg );
   file_mpeg = 0;
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_read_toc( MPEG *file_mpeg )
{
   char string[ MPEG3_STRLEN ];
   int number1;
   //
   file_mpeg->is_program_stream = 1;
   mpeg_io_seek( file_mpeg->fs, 0 );
   fscanf( file_mpeg->fs->fd, "%s %d", string, &number1 );
   if ( number1 > 2 || number1 < 2 )
      return 1;
   mpeg_demux_read_titles( file_mpeg->demuxer );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_has_audio( MPEG *file_mpeg )
{
   return file_mpeg->has_audio;
}
/***************************************************************************/
static inline int mpeg_audio_channels( MPEG *file_mpeg, int stream )
{
   if ( file_mpeg->has_audio )
      return file_mpeg->atrack[ stream ]->channels;
   //
   return -1;
}
/***************************************************************************/
static inline int mpeg_sample_rate( MPEG *file_mpeg, int stream )
{
   if ( file_mpeg->has_audio )
      return file_mpeg->atrack[ stream ]->sample_rate;
   //
   return -1;
}
/***************************************************************************/
static inline long mpeg_get_sample( MPEG *file_mpeg, int stream )
{
   if ( file_mpeg->has_audio )
      return file_mpeg->atrack[ stream ]->current_position;
   //
   return -1;
}
/***************************************************************************/
static inline long mpeg_audio_samples( MPEG *file_mpeg, int stream )
{
   if ( file_mpeg->has_audio )
      return file_mpeg->atrack[ stream ]->total_samples;
   //
   return -1;
}
/***************************************************************************/
static inline int mpeg_has_video( MPEG *file_mpeg )
{
   return file_mpeg->has_video;
}
/***************************************************************************/
static inline int mpeg_video_width( MPEG *file_mpeg, int stream )
{
   if ( file_mpeg->has_video )
      return file_mpeg->vtrack[ stream ]->width;
   //
   return -1;
}
/***************************************************************************/
static inline int mpeg_video_height( MPEG *file_mpeg, int stream )
{
   if ( file_mpeg->has_video )
      return file_mpeg->vtrack[ stream ]->height;
   //
   return -1;
}
/***************************************************************************/
static inline float mpeg_frame_rate( MPEG *file_mpeg, int stream )
{
   if ( file_mpeg->has_video )
      return file_mpeg->vtrack[ stream ]->frame_rate;
   //
   return -1;
}
/***************************************************************************/
static inline long mpeg_video_frames( MPEG *file_mpeg, int stream )
{
   if ( file_mpeg->has_video )
      return file_mpeg->vtrack[ stream ]->total_frames;
   //
   return -1;
}
/***************************************************************************/
static inline long mpeg_get_frame( MPEG *file_mpeg, int stream )
{
   if ( file_mpeg->has_video )
      return file_mpeg->vtrack[ stream ]->current_position;
   //
   return -1;
}
/***************************************************************************/
static inline double mpeg_get_time( MPEG *file_mpeg )
{
   double atime = 0, vtime = 0;
   //
   if ( file_mpeg->is_transport_stream || file_mpeg->is_program_stream )
   {
      if ( file_mpeg->last_type_read == 1 )
      {
         atime = mpeg_demux_get_time( file_mpeg->atrack[ file_mpeg->last_stream_read ]->demuxer );
      }
      else if ( file_mpeg->last_type_read == 2 )
      {
         vtime = mpeg_demux_get_time( file_mpeg->vtrack[ file_mpeg->last_stream_read ]->demuxer );
      }
   }
   else
   {
      if ( file_mpeg->has_audio )
      {
         atime = mpeg_demux_tell_percentage( file_mpeg->atrack[ 0 ]->demuxer ) * mpeg_audio_samples( file_mpeg, 0 ) / mpeg_sample_rate( file_mpeg, 0 );
      }
      if ( file_mpeg->has_video )
      {
         vtime = mpeg_demux_tell_percentage( file_mpeg->vtrack[ 0 ]->demuxer ) * mpeg_video_frames( file_mpeg, 0 ) / mpeg_frame_rate( file_mpeg, 0 );
      }
   }
   //
   return MAX( atime, vtime );
}
/***************************************************************************/
static inline int mpeg_end_of_audio( MPEG *file_mpeg, int stream )
{
   return mpeg_demux_eof( file_mpeg->atrack[ stream ]->demuxer );
}
/***************************************************************************/
static inline int mpeg_read_frame( MPEG *file_mpeg, int stream, BITMAP *bmp )
{
   int result = -1;
   //
   if ( file_mpeg->has_video )
   {
      result = mpeg_video_read_frame( file_mpeg->vtrack[ stream ]->video, file_mpeg->vtrack[ stream ]->current_position, bmp );
      file_mpeg->last_type_read = 2;
      file_mpeg->last_stream_read = stream;
      file_mpeg->vtrack[ stream ]->current_position++;
   }
   //
   return result;
}
/***************************************************************************/
static inline int mpeg_drop_frames( MPEG *file_mpeg, long frames, int stream )
{
   int result = -1;
   //
   if ( file_mpeg->has_video )
   {
      result = mpeg_video_drop_frames( file_mpeg->vtrack[ stream ]->video, frames );
      if ( frames > 0 )
         file_mpeg->vtrack[ stream ]->current_position += frames;
      file_mpeg->last_type_read = 2;
      file_mpeg->last_stream_read = stream;
   }
   //
   return result;
}
/***************************************************************************/
static inline int mpeg_read_audio( MPEG *file_mpeg, float *output_f, short *output_i, int channel, long samples, int stream )
{
   int result = -1;
   //
   if ( file_mpeg->has_audio )
   {
      result = mpeg_audio_decode_audio( file_mpeg->atrack[ stream ]->audio, output_f, output_i, channel, file_mpeg->atrack[ stream ]->current_position, samples );
      file_mpeg->last_type_read = 1;
      file_mpeg->last_stream_read = stream;
      file_mpeg->atrack[ stream ]->current_position += samples;
   }
   //
   return result;
}
/***************************************************************************/
static inline mpeg_atrack_t* mpeg_new_atrack( MPEG *file_mpeg, int stream_id, int format, mpeg_demuxer_t *demuxer )
{
   mpeg_atrack_t *new_atrack;
   new_atrack = ( mpeg_atrack_t * ) calloc( 1, sizeof( mpeg_atrack_t ) );
   new_atrack->channels = 0;
   new_atrack->sample_rate = 0;
   new_atrack->total_samples = 0;
   new_atrack->current_position = 0;
   new_atrack->demuxer = mpeg_new_demuxer( file_mpeg, 1, 0, stream_id );
   if ( demuxer )
      mpeg_demux_copy_titles( new_atrack->demuxer, demuxer );
   new_atrack->audio = mpeg_audio_new( file_mpeg, new_atrack, format );
   if ( !new_atrack->audio )
   {
      mpeg_delete_atrack( file_mpeg, new_atrack );
      new_atrack = 0;
   }
   //
   return new_atrack;
}
/***************************************************************************/
static inline int mpeg_delete_atrack( MPEG *file_mpeg, mpeg_atrack_t *atrack )
{
   if ( atrack->audio )
      mpeg_audio_delete( atrack->audio );
   if ( atrack->demuxer )
      mpeg_delete_demuxer( atrack->demuxer );
   free( atrack );
   //
   return 0;
}
/***************************************************************************/
static inline unsigned char mpeg_packet_next_char( mpeg_demuxer_t *demuxer )
{
   return demuxer->raw_data[ demuxer->raw_offset ];
}
/***************************************************************************/
static inline unsigned char mpeg_packet_read_char( mpeg_demuxer_t *demuxer )
{
   return demuxer->raw_data[ demuxer->raw_offset++ ];
}
/***************************************************************************/
static inline unsigned int mpeg_packet_read_int16( mpeg_demuxer_t *demuxer )
{
   unsigned int a, b, result;
   //
   a = demuxer->raw_data[ demuxer->raw_offset++ ];
   b = demuxer->raw_data[ demuxer->raw_offset++ ];
   result = ( a << 8 ) | b;
   //
   return result;
}
/***************************************************************************/
static inline unsigned int mpeg_packet_next_int24( mpeg_demuxer_t *demuxer )
{
   unsigned int a, b, c, result;
   //
   a = demuxer->raw_data[ demuxer->raw_offset ];
   b = demuxer->raw_data[ demuxer->raw_offset + 1 ];
   c = demuxer->raw_data[ demuxer->raw_offset + 2 ];
   result = ( a << 16 ) | ( b << 8 ) | c;
   //
   return result;
}
/***************************************************************************/
static inline unsigned int mpeg_packet_read_int24( mpeg_demuxer_t *demuxer )
{
   unsigned int a, b, c, result;
   //
   a = demuxer->raw_data[ demuxer->raw_offset++ ];
   b = demuxer->raw_data[ demuxer->raw_offset++ ];
   c = demuxer->raw_data[ demuxer->raw_offset++ ];
   result = ( a << 16 ) | ( b << 8 ) | c;
   //
   return result;
}
/***************************************************************************/
static inline unsigned int mpeg_packet_read_int32( mpeg_demuxer_t *demuxer )
{
   unsigned int a, b, c, d, result;
   //
   a = demuxer->raw_data[ demuxer->raw_offset++ ];
   b = demuxer->raw_data[ demuxer->raw_offset++ ];
   c = demuxer->raw_data[ demuxer->raw_offset++ ];
   d = demuxer->raw_data[ demuxer->raw_offset++ ];
   result = ( a << 24 ) | ( b << 16 ) | ( c << 8 ) | d;
   //
   return result;
}
/***************************************************************************/
static inline unsigned int mpeg_packet_skip( mpeg_demuxer_t *demuxer, long length )
{
   demuxer->raw_offset += length;
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_get_adaptation_field( mpeg_demuxer_t *demuxer )
{
   long length;
   int pcr_flag;
   //
   demuxer->adaptation_fields++;
   length = mpeg_packet_read_char( demuxer );
   pcr_flag = ( mpeg_packet_read_char( demuxer ) >> 4 ) & 1; 
   if ( pcr_flag )
   {
      unsigned long clk_ref_base = mpeg_packet_read_int32( demuxer );
      unsigned int clk_ref_ext = mpeg_packet_read_int16( demuxer );
      if ( clk_ref_base > 0x7fffffff )
      {
         clk_ref_base = 0; 
         clk_ref_ext = 0; 
      }
      else 
      {
         clk_ref_base <<= 1; 
         clk_ref_base |= ( clk_ref_ext >> 15 ); 
         clk_ref_ext &= 0x01ff; 
      }
      demuxer->time = clk_ref_base + clk_ref_ext / 300;
      if ( length )
         mpeg_packet_skip( demuxer, length - 7 );
   }
   else
      mpeg_packet_skip( demuxer, length - 1 );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_get_program_association_table( mpeg_demuxer_t *demuxer )
{
   demuxer->program_association_tables++;
   demuxer->table_id = mpeg_packet_read_char( demuxer );
   demuxer->section_length = mpeg_packet_read_int16( demuxer ) & 0xfff;
   demuxer->transport_stream_id = mpeg_packet_read_int16( demuxer );
   mpeg_packet_skip( demuxer, demuxer->raw_size - demuxer->raw_offset );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_packet_get_data_buffer( mpeg_demuxer_t *demuxer )
{
   while( demuxer->raw_offset < demuxer->raw_size && demuxer->data_size < demuxer->data_allocated )
   {
      demuxer->data_buffer[ demuxer->data_size++ ] = demuxer->raw_data[ demuxer->raw_offset++ ];
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_get_pes_packet_header( mpeg_demuxer_t *demuxer, unsigned long *pts, unsigned long *dts )
{
   unsigned int pes_header_bytes = 0;
   unsigned int pts_dts_flags;
   int pes_header_data_length;
   //
   mpeg_packet_read_char( demuxer ); 
   pts_dts_flags = ( mpeg_packet_read_char( demuxer ) >> 6 ) & 0x3;
   pes_header_data_length = mpeg_packet_read_char( demuxer );
   if ( pts_dts_flags == 2 )
   {
      *pts = ( mpeg_packet_read_char( demuxer ) >> 1 ) & 7; 
      *pts <<= 15;
      *pts |= ( mpeg_packet_read_int16( demuxer ) >> 1 );
      *pts <<= 15;
      *pts |= ( mpeg_packet_read_int16( demuxer ) >> 1 );
      pes_header_bytes += 5;
   }
   else if ( pts_dts_flags == 3 )
   {
      *pts = ( mpeg_packet_read_char( demuxer ) >> 1 ) & 7; 
      *pts <<= 15;
      *pts |= ( mpeg_packet_read_int16( demuxer ) >> 1 );
      *pts <<= 15;
      *pts |= ( mpeg_packet_read_int16( demuxer ) >> 1 );
      *dts = ( mpeg_packet_read_char( demuxer ) >> 1 ) & 7; 
      *dts <<= 15;
      *dts |= ( mpeg_packet_read_int16( demuxer ) >> 1 );
      *dts <<= 15;
      *dts |= ( mpeg_packet_read_int16( demuxer ) >> 1 );
      pes_header_bytes += 10;
   }
   mpeg_packet_skip( demuxer, pes_header_data_length - pes_header_bytes );
   //
   return 0;
}
/***************************************************************************/
static inline int get_unknown_data( mpeg_demuxer_t *demuxer )
{
   mpeg_packet_skip( demuxer, demuxer->raw_size - demuxer->raw_offset );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_get_pes_packet_data( mpeg_demuxer_t *demuxer, unsigned int stream_id )
{
   unsigned long pts = 0, dts = 0;
   //
   if ( ( stream_id >> 4 ) == 12 || ( stream_id >> 4 ) == 13 )
   {
      if ( demuxer->astream == -1 )
         demuxer->astream = ( stream_id & 0x0f );
      if ( ( stream_id & 0x0f ) == demuxer->astream && demuxer->do_audio )
      {
         mpeg_get_pes_packet_header( demuxer, &pts, &dts );
         demuxer->pes_audio_time = pts;
         demuxer->audio_pid = demuxer->pid;
         return mpeg_packet_get_data_buffer( demuxer );
      }
   }
   else if ( ( stream_id >> 4 )==14 )
   {
      if ( demuxer->vstream == -1 )
         demuxer->vstream = ( stream_id & 0x0f );
      if ( ( stream_id & 0x0f ) == demuxer->vstream && demuxer->do_video )
      {
         mpeg_get_pes_packet_header( demuxer, &pts, &dts );
         demuxer->pes_video_time = pts;
         demuxer->video_pid = demuxer->pid;
         return mpeg_packet_get_data_buffer( demuxer );
      }
   }
   else 
   {
      return get_unknown_data( demuxer );
   }
   mpeg_packet_skip( demuxer, demuxer->raw_size - demuxer->raw_offset );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_get_pes_packet( mpeg_demuxer_t *demuxer )
{
   unsigned int stream_id;
   //
   demuxer->pes_packets++;
   stream_id = mpeg_packet_read_char( demuxer );
   mpeg_packet_read_int24( demuxer );
   mpeg_packet_read_int16( demuxer );
   if ( stream_id != MPEG3_PRIVATE_STREAM_2 && stream_id != MPEG3_PADDING_STREAM )
   {
      return mpeg_get_pes_packet_data( demuxer, stream_id );
   }
   else if ( stream_id == MPEG3_PRIVATE_STREAM_2 )
   {
	  DebugMessage( "MpegPlayer : stream_id == MPEG3_PRIVATE_STREAM_2\n" );
      mpeg_packet_skip( demuxer, demuxer->raw_size - demuxer->raw_offset );
      return 0;
   }
   else if ( stream_id == MPEG3_PADDING_STREAM )
   {
      mpeg_packet_skip( demuxer, demuxer->raw_size - demuxer->raw_offset );
      return 0;
   }
   else
   {
	  DebugMessage( "MpegPlayer : unknown stream_id in pes packet" );
      return 1;
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_get_payload( mpeg_demuxer_t *demuxer )
{
   if ( demuxer->payload_unit_start_indicator )
   {
      if ( demuxer->pid==0 )
         mpeg_get_program_association_table( demuxer );
      else if ( mpeg_packet_next_int24( demuxer ) == MPEG3_PACKET_START_CODE_PREFIX )
         mpeg_get_pes_packet( demuxer );
      else 
         mpeg_packet_skip( demuxer, demuxer->raw_size - demuxer->raw_offset );
   }
   else
   {
      if ( demuxer->pid == demuxer->audio_pid && demuxer->do_audio )
      {
         mpeg_packet_get_data_buffer( demuxer );
      }
      else if ( demuxer->pid == demuxer->video_pid && demuxer->do_video )
      {
         mpeg_packet_get_data_buffer( demuxer );
      }
      else 
         mpeg_packet_skip( demuxer, demuxer->raw_size - demuxer->raw_offset );
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_read_transport( mpeg_demuxer_t *demuxer )
{
   mpeg_title_t *title = demuxer->titles[ demuxer->current_title ];
   int result = mpeg_io_read_data( demuxer->raw_data, demuxer->packet_size, title->fs );
   unsigned int bits;
   int table_entry;
   //
   demuxer->raw_size = demuxer->packet_size;
   demuxer->raw_offset = 0;
   if ( result )
   {
	  DebugMessage( "MpegPlayer : mpeg_read_transport" );
      return 1;
   }
   if ( mpeg_packet_read_char( demuxer ) != MPEG3_SYNC_BYTE )
   {
	  DebugMessage( "MpegPlayer : mpeg_packet_read_char(demuxer) != MPEG3_SYNC_BYTE\n" );
      return 1;
   }
   bits = mpeg_packet_read_int24( demuxer ) & 0x00ffffff;
   demuxer->transport_error_indicator = ( bits >> 23 ) & 0x1;
   demuxer->payload_unit_start_indicator = ( bits >> 22 ) & 0x1;
   demuxer->pid = ( bits >> 8 ) & 0x00001fff;
   demuxer->transport_scrambling_control = ( bits >> 6 ) & 0x3;
   demuxer->adaptation_field_control = ( bits >> 4 ) & 0x3;
   demuxer->continuity_counter = bits & 0xf;
   if ( demuxer->transport_error_indicator )
   {
	  DebugMessage( "MpegPlayer : demuxer->transport_error_indicator\n" );
      return 1;
   }
   if ( demuxer->pid == 0x1fff )
   {
      demuxer->is_padding = 1; 
      return 0;
   }
   else
   {
      demuxer->is_padding = 0;
   }
   for( table_entry = 0, result = 0; table_entry < demuxer->total_pids; table_entry++ )
   {
      if ( demuxer->pid == demuxer->pid_table[ table_entry ] )
      {
         result = 1;
         break;
      }
   }
   if ( !result )
   {
      demuxer->pid_table[ table_entry ] = demuxer->pid;
      demuxer->continuity_counters[ table_entry ] = demuxer->continuity_counter; 
      demuxer->total_pids++;
   }
   result = 0;
   if ( demuxer->pid != MPEG3_PROGRAM_ASSOCIATION_TABLE && demuxer->pid != MPEG3_CONDITIONAL_ACCESS_TABLE && ( demuxer->adaptation_field_control == 1 || demuxer->adaptation_field_control == 3 ) )
   {
      if ( demuxer->continuity_counters[ table_entry ] != demuxer->continuity_counter )
      {
	 DebugMessage( "MpegPlayer : demuxer->continuity_counters[table_entry] != demuxer->continuity_counter\n" );
         demuxer->continuity_counters[ table_entry ] = demuxer->continuity_counter;
      }
      if ( ++( demuxer->continuity_counters[ table_entry ] ) > 15 )
         demuxer->continuity_counters[ table_entry ] = 0;
   }
   if ( demuxer->adaptation_field_control == 2 || demuxer->adaptation_field_control == 3 )
      result = mpeg_get_adaptation_field( demuxer );
   if ( demuxer->adaptation_field_control == 1 || demuxer->adaptation_field_control == 3 )
      result = mpeg_get_payload( demuxer );
   //
   return result;
}
/***************************************************************************/
static inline int mpeg_get_system_header( mpeg_demuxer_t *demuxer )
{
   int length = mpeg_packet_read_int16( demuxer );
   //
   mpeg_packet_skip( demuxer, length );
   //
   return 0;
}
/***************************************************************************/
static inline unsigned long mpeg_get_timestamp( mpeg_demuxer_t *demuxer )
{
   unsigned long timestamp;
   //
   timestamp = ( mpeg_packet_read_char( demuxer ) >> 1 ) & 7; 
   timestamp <<= 15;
   timestamp |= ( mpeg_packet_read_int16( demuxer ) >> 1 );
   timestamp <<= 15;
   timestamp |= ( mpeg_packet_read_int16( demuxer ) >> 1 );
   //
   return timestamp;
}
/***************************************************************************/
static inline int mpeg_get_pack_header( mpeg_demuxer_t *demuxer, unsigned int *header )
{
   unsigned long i, j;
   unsigned long clock_ref, clock_ref_ext;
   //
   if ( ( mpeg_packet_next_char( demuxer ) >> 4 ) == 2 )
   {
      demuxer->time = ( double )mpeg_get_timestamp( demuxer ) / 90000;
      mpeg_packet_read_int24( demuxer );
   }
   else if ( mpeg_packet_next_char( demuxer ) & 0x40 )
   {
      i = mpeg_packet_read_int32( demuxer );
      j = mpeg_packet_read_int16( demuxer );
      if ( i & 0x40000000 || ( i >> 28 ) == 2 )
      {
         clock_ref = ( ( i & 0x31000000 ) << 3 );
         clock_ref |= ( ( i & 0x03fff800 ) << 4 );
         clock_ref |= ( ( i & 0x000003ff ) << 5 );
         clock_ref |= ( ( j & 0xf800 ) >> 11 );
         clock_ref_ext = ( j >> 1 ) & 0x1ff;
         demuxer->time = ( double )( clock_ref + clock_ref_ext / 300 ) / 90000;
         mpeg_packet_read_int24( demuxer );
         i = mpeg_packet_read_char( demuxer ) & 0x7;
         mpeg_packet_skip( demuxer, i ); 
      }
   }
   else
   {
      mpeg_packet_skip( demuxer, 2 );
   }
   *header = mpeg_packet_read_int32( demuxer );
   if ( *header == MPEG3_SYSTEM_START_CODE )
   {
      mpeg_get_system_header( demuxer );
      *header = mpeg_packet_read_int32( demuxer );
   }
   return 0;
}
/***************************************************************************/
static inline int mpeg_get_ps_pes_packet( mpeg_demuxer_t *demuxer, unsigned int *header )
{
   unsigned long pts = 0, dts = 0;
   int stream_id;
   int pes_packet_length;
   int pes_packet_start;
   //
   stream_id = *header & 0xff;
   pes_packet_length = mpeg_packet_read_int16( demuxer );
   pes_packet_start = demuxer->raw_offset;
   if ( stream_id != MPEG3_PRIVATE_STREAM_2 && stream_id != MPEG3_PADDING_STREAM )
   {
      if ( ( mpeg_packet_next_char( demuxer ) >> 6 ) == 0x02 )
      {
         int pes_header_bytes = 0;
         int scrambling = ( mpeg_packet_read_char( demuxer ) >> 4 ) & 0x3;
         int pts_dts_flags = ( mpeg_packet_read_char( demuxer ) >> 6 ) & 0x3;
         int pes_header_data_length = mpeg_packet_read_char( demuxer );
         //
         if ( scrambling && ( demuxer->do_audio || demuxer->do_video ) )
         {
            if ( mpeg_decrypt_packet( demuxer->titles[ demuxer->current_title ]->fs->css, demuxer->raw_data ) )
            {
		   DebugMessage( "MpegPlayer : mpeg_get_ps_pes_packet: Decryption not available\n" );
               return 1;
            }
         }
         if ( pts_dts_flags == 2 )
         {
            pts = ( mpeg_packet_read_char( demuxer ) >> 1 ) & 7; 
            pts <<= 15;
            pts |= ( mpeg_packet_read_int16( demuxer ) >> 1 );
            pts <<= 15;
            pts |= ( mpeg_packet_read_int16( demuxer ) >> 1 );
            pes_header_bytes += 5;
         }
         else if ( pts_dts_flags == 3 )
         {
            pts = ( mpeg_packet_read_char( demuxer ) >> 1 ) & 7; 
            pts <<= 15;
            pts |= ( mpeg_packet_read_int16( demuxer ) >> 1 );
            pts <<= 15;
            pts |= ( mpeg_packet_read_int16( demuxer ) >> 1 );
            dts = ( mpeg_packet_read_char( demuxer ) >> 1 ) & 7; 
            dts <<= 15;
            dts |= ( mpeg_packet_read_int16( demuxer ) >> 1 );
            dts <<= 15;
            dts |= ( mpeg_packet_read_int16( demuxer ) >> 1 );
            pes_header_bytes += 10;
         }
         mpeg_packet_skip( demuxer, pes_header_data_length - pes_header_bytes );
      }
      else
      {
         int pts_dts_flags;
	 while( mpeg_packet_next_char( demuxer ) == 0xff )
         {
            mpeg_packet_read_char( demuxer );
         }
         if ( ( mpeg_packet_next_char( demuxer ) & 0x40 ) == 0x40 )
         {
            mpeg_packet_skip( demuxer, 2 );
         }
         pts_dts_flags = mpeg_packet_next_char( demuxer );
         if ( pts_dts_flags >= 0x30 )
         {
            pts = mpeg_get_timestamp( demuxer );
            dts = mpeg_get_timestamp( demuxer );
         }
         else if ( pts_dts_flags >= 0x20 )
         {
            pts = mpeg_get_timestamp( demuxer );
         }
         else if ( pts_dts_flags == 0x0f )
         {
            mpeg_packet_read_char( demuxer );
         }
         else
         {
            return 1; 
         }
      }
      if ( ( stream_id >> 4 ) == 0xc || ( stream_id >> 4 ) == 0xd )
      {
         pes_packet_length -= demuxer->raw_offset - pes_packet_start;
         if ( !demuxer->do_audio && !demuxer->do_video )
            demuxer->astream_table[ stream_id & 0x0f ] = AUDIO_MPEG;
         else if ( demuxer->astream == -1 ) 
            demuxer->astream = stream_id & 0x0f;
         if ( ( stream_id & 0x0f ) == demuxer->astream && demuxer->do_audio )
         {
            if ( pts )
               demuxer->pes_audio_time = pts;
            memcpy( &demuxer->data_buffer[ demuxer->data_size ], &demuxer->raw_data[ demuxer->raw_offset ], pes_packet_length );
            demuxer->data_size += pes_packet_length;
            demuxer->raw_offset += pes_packet_length;
         }
         else 
         {
            mpeg_packet_skip( demuxer, pes_packet_length );
         }
      }
      else if ( ( stream_id >> 4 ) == 0xe )
      {
         if ( !demuxer->do_audio && !demuxer->do_video ) 
            demuxer->vstream_table[ stream_id & 0x0f ] = 1;
         else if ( demuxer->vstream == -1 ) 
            demuxer->vstream = stream_id & 0x0f;
         pes_packet_length -= demuxer->raw_offset - pes_packet_start;
         if ( ( stream_id & 0x0f ) == demuxer->vstream && demuxer->do_video )
         {
            if ( pts )
               demuxer->pes_video_time = pts;
            memcpy( &demuxer->data_buffer[ demuxer->data_size ], &demuxer->raw_data[ demuxer->raw_offset ], pes_packet_length );
            demuxer->data_size += pes_packet_length;
            demuxer->raw_offset += pes_packet_length;
         }
         else 
         {
            mpeg_packet_skip( demuxer, pes_packet_length );
         }
      }
      else if ( stream_id == 0xbd && demuxer->raw_data[ demuxer->raw_offset ] != 0xff )
      {
         int format;
         if ( ( demuxer->raw_data[ demuxer->raw_offset ] & 0xf0 ) == 0xa0 )
            format = AUDIO_PCM;
         else
            format = AUDIO_AC3;
         stream_id = demuxer->raw_data[ demuxer->raw_offset ] - 0x80;
         if ( !demuxer->do_audio && !demuxer->do_video )
            demuxer->astream_table[ stream_id ] = format;
         else if ( demuxer->astream == -1 )
            demuxer->astream = stream_id;
         if ( stream_id == demuxer->astream && demuxer->do_audio )
         {
            demuxer->aformat = format;
            if ( pts )
               demuxer->pes_audio_time = pts;
            mpeg_packet_read_int32( demuxer );
            pes_packet_length -= demuxer->raw_offset - pes_packet_start;
            memcpy( &demuxer->data_buffer[ demuxer->data_size ], &demuxer->raw_data[ demuxer->raw_offset ], pes_packet_length );
            demuxer->data_size += pes_packet_length;
            demuxer->raw_offset += pes_packet_length;
         }
         else
         {
            pes_packet_length -= demuxer->raw_offset - pes_packet_start;
            mpeg_packet_skip( demuxer, pes_packet_length );
         }
      }
      else if ( stream_id == 0xbc || 1 )
      {
         pes_packet_length -= demuxer->raw_offset - pes_packet_start;
         mpeg_packet_skip( demuxer, pes_packet_length );
      }
   }
   else if ( stream_id == MPEG3_PRIVATE_STREAM_2 || stream_id == MPEG3_PADDING_STREAM )
   {
      pes_packet_length -= demuxer->raw_offset - pes_packet_start;
      mpeg_packet_skip( demuxer, pes_packet_length );
   }
   while( demuxer->raw_offset + 4 < demuxer->raw_size )
   {
      *header = mpeg_packet_read_int32( demuxer );
      if ( ( *header >> 8 ) != MPEG3_PACKET_START_CODE_PREFIX )
         demuxer->raw_offset -= 3;
      else
         break;
   }
   return 0;
}
/***************************************************************************/
static inline int mpeg_read_program( mpeg_demuxer_t *demuxer )
{
   int result = 0, count = 0;
   mpeg_title_t *title = demuxer->titles[ demuxer->current_title ];
   unsigned int header;
   //
   demuxer->raw_size = demuxer->packet_size;
   demuxer->raw_offset = 0;
   demuxer->data_size = 0;
   header = mpeg_io_read_int32( title->fs );
   result = mpeg_io_eof( title->fs );
   if ( !result )
      result = mpeg_io_seek_relative( title->fs, -4 );
   while( header != MPEG3_PACK_START_CODE && !result && count < demuxer->packet_size )
   {
      result = mpeg_io_seek_relative( title->fs, -1 );
      if ( !result )
      {
         header >>= 8;
         header |= mpeg_io_read_char( title->fs ) << 24;
         result = mpeg_io_seek_relative( title->fs, -1 );
      }
      count++;
   }
   if ( result )
   {
      return 1;
   }
   result = mpeg_io_read_data( demuxer->raw_data, demuxer->packet_size, title->fs );
   if ( result )
   {
	  DebugMessage( "MpegPlayer : mpeg_read_program" );
      return 1;
   }
   header = mpeg_packet_read_int32( demuxer );
   while( demuxer->raw_offset + 4 < demuxer->raw_size && !result )
   {
      if ( header == MPEG3_PACK_START_CODE )
      {
         result = mpeg_get_pack_header( demuxer, &header );
      }
      else if ( ( header >> 8 ) == MPEG3_PACKET_START_CODE_PREFIX )
      {
         result = mpeg_get_ps_pes_packet( demuxer, &header );
      }
   }
   //
   return result;
}
/***************************************************************************/
static inline double mpeg_lookup_time_offset( mpeg_demuxer_t *demuxer, long byte )
{
   int i;
   mpeg_title_t *title = demuxer->titles[ demuxer->current_title ];
   //
   if ( !title->timecode_table_size )
      return 0;
   for( i = title->timecode_table_size - 1; i >= 0 && title->timecode_table[ i ].start_byte > byte; i-- )
      ;
   if ( i < 0 )
      i = 0;
   return title->timecode_table[ i ].absolute_start_time - title->timecode_table[ i ].start_time;
}
/***************************************************************************/
static inline int mpeg_advance_timecode( mpeg_demuxer_t *demuxer, int reverse )
{
   mpeg_title_t *title = demuxer->titles[ demuxer->current_title ];
   int result = 0;
   int do_seek = 0;
   //
   if ( !title->timecode_table || !title->timecode_table_size || demuxer->generating_timecode )
      return 0;
   if ( !reverse )
   {
      if ( mpeg_io_tell( title->fs ) < title->timecode_table[ demuxer->current_timecode ].start_byte )
      {
         mpeg_io_seek( title->fs, title->timecode_table[ demuxer->current_timecode ].start_byte );
      }
	  while( !result && ( mpeg_io_tell( title->fs ) >= title->timecode_table[ demuxer->current_timecode ].end_byte || demuxer->current_program != title->timecode_table[ demuxer->current_timecode ].program ) )
      {
         demuxer->current_timecode++;
         if ( demuxer->current_timecode >= title->timecode_table_size )
         {
            demuxer->current_timecode = 0;
            if ( demuxer->current_title + 1 < demuxer->total_titles )
            {
               mpeg_demux_open_title( demuxer, demuxer->current_title + 1 );
               do_seek = 1;
            }
            else
            {
               mpeg_io_seek( title->fs, mpeg_io_total_bytes( title->fs ) );
               result = 1;
            }
         }
         title = demuxer->titles[ demuxer->current_title ];
      }
      if ( !result && do_seek )
      {
         mpeg_io_seek( title->fs, title->timecode_table[ demuxer->current_timecode ].start_byte );
      }
   }
   else
   {
	  while( !result && ( mpeg_io_tell( title->fs ) < title->timecode_table[ demuxer->current_timecode ].start_byte || demuxer->current_program != title->timecode_table[ demuxer->current_timecode ].program ) )
      {
         demuxer->current_timecode--;
         if ( demuxer->current_timecode < 0 )
         {
            if ( demuxer->current_title > 0 )
            {
               mpeg_demux_open_title( demuxer, demuxer->current_title - 1 );
               title = demuxer->titles[ demuxer->current_title ];
               demuxer->current_timecode = title->timecode_table_size - 1;
               do_seek = 1;
            }
            else
            {
               mpeg_io_seek( title->fs, 0 );
               demuxer->current_timecode = 0;
               result = 1;
            }
         }
      }
      if ( !result && do_seek ) 
         mpeg_io_seek( title->fs, title->timecode_table[ demuxer->current_timecode ].start_byte );
   }
   return result;
}
/***************************************************************************/
static inline int mpeg_read_next_packet( mpeg_demuxer_t *demuxer )
{
   int result = 0;
   MPEG *file_mpeg = ( MPEG * ) demuxer->file_mpeg;
   mpeg_title_t *title = demuxer->titles[ demuxer->current_title ];
   //
   demuxer->data_size = 0;
   demuxer->data_position = 0;
   if ( demuxer->reverse )
   {
      result = mpeg_io_seek_relative( title->fs, demuxer->packet_size );
      demuxer->reverse = 0;
   }
   if ( !result )
   {
      do
      {
         result = mpeg_advance_timecode( demuxer, 0 );
         if ( !result )
         {
            demuxer->time_offset = mpeg_lookup_time_offset( demuxer, mpeg_io_tell( title->fs ) );
            if ( file_mpeg->is_transport_stream )
            {
               result = mpeg_read_transport( demuxer );
            }
            else if ( file_mpeg->is_program_stream )
            {
               result = mpeg_read_program( demuxer );
            }
            else
            {
               result = mpeg_io_read_data( demuxer->data_buffer, demuxer->packet_size, title->fs );
               if ( !result )
                  demuxer->data_size = demuxer->packet_size;
            }
         }
	  } while( !result && demuxer->data_size == 0 && ( demuxer->do_audio || demuxer->do_video ) );
   }
   //
   return result;
}
/***************************************************************************/
static inline int mpeg_read_prev_packet( mpeg_demuxer_t *demuxer )
{
   int result = 0;
   MPEG *file_mpeg = ( MPEG * ) demuxer->file_mpeg;
   mpeg_title_t *title = demuxer->titles[ demuxer->current_title ];
   //
   demuxer->data_size = 0;
   demuxer->data_position = 0;
   do
   {
      result = mpeg_io_seek_relative( title->fs, -demuxer->packet_size );
      if ( !result )
         result = mpeg_advance_timecode( demuxer, 1 );
      if ( !result )
         demuxer->time_offset = mpeg_lookup_time_offset( demuxer, mpeg_io_tell( title->fs ) );
      if ( file_mpeg->is_transport_stream && !result )
      {
         result = mpeg_read_transport( demuxer );
         if ( !mpeg_io_bof( title->fs ) )
            result = mpeg_io_seek_relative( title->fs, -demuxer->packet_size );
      }
      else if ( file_mpeg->is_program_stream && !result )
      {
         result = mpeg_read_program( demuxer );
         if ( !mpeg_io_bof( title->fs ) )
            result = mpeg_io_seek_relative( title->fs, -demuxer->packet_size );
      }
      else if ( !result )
      {
         result = mpeg_io_read_data( demuxer->data_buffer, demuxer->packet_size, title->fs );
         if ( !result ) 
         {
            demuxer->data_size = demuxer->packet_size;
            result = mpeg_io_seek_relative( title->fs, -demuxer->packet_size );
         }
      }
   } while( !result && demuxer->data_size == 0 && ( demuxer->do_audio || demuxer->do_video ) );
   demuxer->reverse = 1;
   demuxer->error_flag = result;
   //
   return result;
}
/***************************************************************************/
static inline int mpeg_demux_read_data( mpeg_demuxer_t *demuxer, unsigned char *output, long size )
{
   long i;
   int result = 0;
   demuxer->error_flag = 0;
   if ( demuxer->data_position >= 0 )
   {
      for( i = 0; i < size && !result; )
      {
         int fragment_size = size - i;
         if ( fragment_size > demuxer->data_size - demuxer->data_position )
            fragment_size = demuxer->data_size - demuxer->data_position;
         memcpy( output + i, demuxer->data_buffer + demuxer->data_position, fragment_size );
         demuxer->data_position += fragment_size;
         i += fragment_size;
         if ( i < size )
         {
            result = mpeg_read_next_packet( demuxer );
         }
      }
   }
   else
   {
      long current_position = demuxer->data_position;
      result = mpeg_read_prev_packet( demuxer );
      if ( !result )
         demuxer->data_position = demuxer->data_size + current_position;
      memcpy( output, demuxer->data_buffer + demuxer->data_position, size );
      demuxer->data_position += size;
   }
   demuxer->error_flag = result;
   return result;
}
/***************************************************************************/
static inline unsigned int mpeg_demux_read_char_packet( mpeg_demuxer_t *demuxer )
{
   demuxer->error_flag = 0;
   if ( demuxer->data_position >= demuxer->data_size )
      demuxer->error_flag = mpeg_read_next_packet( demuxer );
   demuxer->next_char = demuxer->data_buffer[ demuxer->data_position++ ];
   //
   return demuxer->next_char;
}
/***************************************************************************/
static inline unsigned int mpeg_demux_read_prev_char_packet( mpeg_demuxer_t *demuxer )
{
   demuxer->error_flag = 0;
   demuxer->data_position--;
   if ( demuxer->data_position < 0 )
   {
      demuxer->error_flag = mpeg_read_prev_packet( demuxer );
      if ( !demuxer->error_flag )
         demuxer->data_position = demuxer->data_size - 1;
   }
   demuxer->next_char = demuxer->data_buffer[ demuxer->data_position ];
   //
   return demuxer->next_char;
}
/***************************************************************************/
static inline mpeg_demux_timecode_t* mpeg_append_timecode( mpeg_demuxer_t *demuxer, mpeg_title_t *title, long prev_byte, double prev_time, long next_byte, double next_time, int dont_store )
{
   mpeg_demux_timecode_t *new_table = 0;
   mpeg_demux_timecode_t *new_timecode = 0, *old_timecode = 0;
   long i;
   //
   if ( !title->timecode_table || title->timecode_table_allocation <= title->timecode_table_size )
   {
      if ( title->timecode_table_allocation == 0 ) 
         title->timecode_table_allocation = 1;
      else
         title->timecode_table_allocation *= 2;
      new_table = ( mpeg_demux_timecode_t * ) calloc( 1, sizeof( mpeg_demux_timecode_t ) * title->timecode_table_allocation );
      if ( title->timecode_table )
      {
         for( i = 0; i < title->timecode_table_size; i++ )
         {
            new_table[ i ] = title->timecode_table[ i ];
         }
         free( title->timecode_table );
      }
      title->timecode_table = new_table;
   }
   if ( !dont_store )
   {
      new_timecode = &title->timecode_table[ title->timecode_table_size ];
      new_timecode->start_byte = next_byte;
      new_timecode->start_time = next_time;
      new_timecode->absolute_start_time = 0;
      if ( title->timecode_table_size > 0 )
      {
         old_timecode = &title->timecode_table[ title->timecode_table_size - 1 ];
         old_timecode->end_byte = prev_byte;
         old_timecode->end_time = prev_time;
         new_timecode->absolute_start_time = prev_time - old_timecode->start_time + old_timecode->absolute_start_time;
         new_timecode->absolute_end_time = next_time;
      }
   }
   title->timecode_table_size++;
   return new_timecode;
}
/***************************************************************************/
static inline mpeg_demux_timecode_t* mpeg_demux_next_timecode( mpeg_demuxer_t *demuxer, int *current_title, int *current_timecode, int current_program )
{
   int done = 0;
   //
   while( !done )
   {
      if ( *current_timecode < demuxer->titles[ *current_title ]->timecode_table_size - 1 ) 
      {
         ( *current_timecode )++;
         if ( demuxer->titles[ *current_title ]->timecode_table[ *current_timecode ].program == current_program )
            return &( demuxer->titles[ *current_title ]->timecode_table[ *current_timecode ] );
      }
      else if ( *current_title < demuxer->total_titles - 1 )
      {
         ( *current_title )++;
         ( *current_timecode ) = 0;
         if ( demuxer->titles[ *current_title ]->timecode_table[ *current_timecode ].program == current_program )
            return &( demuxer->titles[ *current_title ]->timecode_table[ *current_timecode ] );
      }
      else
         done = 1;
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_demux_open_title( mpeg_demuxer_t *demuxer, int title_number )
{
   mpeg_title_t *title;
   //
   if ( title_number < demuxer->total_titles )
   {
      if ( demuxer->current_title >= 0 )
      {
         mpeg_io_close_file( demuxer->titles[ demuxer->current_title ]->fs );
         demuxer->current_title = -1;
      }
      title = demuxer->titles[ title_number ];
      if ( mpeg_io_open_file( title->fs ) )
      {
         demuxer->error_flag = 1;
	 DebugMessage( "MpegPlayer : mpeg_demux_open_title" );
      }
      else
      {
         demuxer->current_title = title_number;
      }
   }
   demuxer->current_timecode = 0;
   //
   return demuxer->error_flag;
}
/***************************************************************************/
static inline int mpeg_demux_assign_programs( mpeg_demuxer_t *demuxer )
{
   int current_program = 0;
   int current_title = 0;
   int current_timecode = 0;
   double current_time;
   mpeg_demux_timecode_t *timecode1;
   int total_programs = 1;
   int i;
   int total_timecodes;
   //
   for( i = 0, total_timecodes = 0; i < demuxer->total_titles; i++ )
      total_timecodes += demuxer->titles[ i ]->timecode_table_size;
   for( current_program = 0; current_program < total_programs; current_program++ )
   {
      current_time = 0;
      current_title = 0;
      current_timecode = -1;
	  while( ( timecode1 = mpeg_demux_next_timecode( demuxer, &current_title, &current_timecode, current_program ) ) )
      {
         timecode1->absolute_start_time = current_time;
         current_time += timecode1->end_time - timecode1->start_time;
         timecode1->absolute_end_time = current_time;
      }
   }
   demuxer->current_program = 0;
   //
   return 0;
}
/***************************************************************************/
static inline mpeg_demuxer_t* mpeg_new_demuxer( MPEG *file_mpeg, int do_audio, int do_video, int stream_id )
{
   mpeg_demuxer_t *demuxer = ( mpeg_demuxer_t * ) calloc( 1, sizeof( mpeg_demuxer_t ) );
   //
   demuxer->file_mpeg = file_mpeg;
   demuxer->packet_size = file_mpeg->packet_size;
   demuxer->do_audio = do_audio;
   demuxer->do_video = do_video;
   demuxer->raw_data = ( unsigned char* )calloc( 1, MPEG3_MAX_PACKSIZE );
   demuxer->data_buffer = ( unsigned char* )calloc( 1, MPEG3_MAX_PACKSIZE );
   demuxer->data_allocated = MPEG3_MAX_PACKSIZE;
   demuxer->audio_pid = stream_id;
   demuxer->video_pid = stream_id;
   demuxer->astream = stream_id;
   demuxer->vstream = stream_id;
   demuxer->current_title = -1;
   //
   return demuxer;
}
/***************************************************************************/
static inline int mpeg_delete_demuxer( mpeg_demuxer_t *demuxer )
{
   int i;
   //
   if ( demuxer->current_title >= 0 )
   {
      mpeg_io_close_file( demuxer->titles[ demuxer->current_title ]->fs );
   }
   for( i = 0; i < demuxer->total_titles; i++ )
   {
      mpeg_delete_title( demuxer->titles[ i ] );
   }
   if ( demuxer->data_buffer )
      free( demuxer->data_buffer );
   free( demuxer->raw_data );
   free( demuxer );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_demux_create_title( mpeg_demuxer_t *demuxer, int timecode_search, FILE *toc )
{
   int result = 0, done = 0, counter_start, counter;
   MPEG *file_mpeg = ( MPEG * ) demuxer->file_mpeg;
   long next_byte, prev_byte = 0;
   double next_time, prev_time = 0;
   long i;
   mpeg_title_t *title;
   unsigned long test_header = 0;
   mpeg_demux_timecode_t *timecode = 0;
   //
   demuxer->error_flag = 0;
   demuxer->generating_timecode = 1;
   if ( !demuxer->total_titles )
   {
      demuxer->titles[ 0 ] = mpeg_new_title( file_mpeg, file_mpeg->fs->path );
      demuxer->total_titles = 1;
      mpeg_demux_open_title( demuxer, 0 );
   }
   title = demuxer->titles[ 0 ];
   title->total_bytes = mpeg_io_total_bytes( title->fs );
   if ( file_mpeg->is_program_stream )
   {
      mpeg_io_seek( title->fs, 4 );
      for( i = 0; i < MPEG3_MAX_PACKSIZE && test_header != MPEG3_PACK_START_CODE; i++ )
      {
         test_header <<= 8;
         test_header |= mpeg_io_read_char( title->fs );
      }
      if ( i < MPEG3_MAX_PACKSIZE )
         demuxer->packet_size = i;
      mpeg_io_seek( title->fs, 0 );
   }
   else
      demuxer->packet_size = file_mpeg->packet_size;
   if ( file_mpeg->is_transport_stream || file_mpeg->is_program_stream )
   {
      mpeg_io_seek( title->fs, 0 );
	  while( !done && !result && !mpeg_io_eof( title->fs ) )
      {
         next_byte = mpeg_io_tell( title->fs );
         result = mpeg_read_next_packet( demuxer );
         if ( !result )
         {
            next_time = demuxer->time;
            if ( next_time < prev_time || next_time - prev_time > MPEG3_CONTIGUOUS_THRESHOLD || !title->timecode_table_size )
            {
               timecode = mpeg_append_timecode( demuxer, title, prev_byte, prev_time, next_byte, next_time, 0 );
               counter_start = next_time;
            }
            prev_time = next_time;
            prev_byte = next_byte;
            counter = next_time;
         }
         if ( next_byte > 0x100000 && ( !timecode_search || !toc ) )
            done = 1;
      }
      if ( !toc || !timecode_search )
      {
         result = mpeg_io_seek( title->fs, title->total_bytes );
         if ( !result )
            result = mpeg_read_prev_packet( demuxer );
      }
      if ( title->timecode_table && timecode )
      {
         timecode->end_byte = title->total_bytes;
         timecode->end_time = demuxer->time;
         timecode->absolute_end_time = timecode->end_time - timecode->start_time;
      }
   }
   mpeg_io_seek( title->fs, 0 );
   demuxer->generating_timecode = 0;
   return 0;
}
/***************************************************************************/
static inline int mpeg_demux_read_titles( mpeg_demuxer_t *demuxer )
{
   char string1[ MPEG3_STRLEN ], string2[ MPEG3_STRLEN ];
   long start_byte, end_byte;
   float start_time, end_time;
   mpeg_title_t *title = 0;
   MPEG *file_mpeg = ( MPEG * ) demuxer->file_mpeg;
   //
   while( !feof( file_mpeg->fs->fd ) )
   {
      fscanf( file_mpeg->fs->fd, "%s %s %ld %f %f", string1, string2, &end_byte, &start_time, &end_time );
      if ( !strncasecmp( string1, "PATH:", 5 ) )
      {
         title = demuxer->titles[ demuxer->total_titles++ ] = mpeg_new_title( file_mpeg, string2 );
         if ( demuxer->current_title < 0 )
            mpeg_demux_open_title( demuxer, 0 );
      }
      else if ( title )
      {
         start_byte = atol( string2 );
         if ( !strcasecmp( string1, "REGION:" ) )
         {
            mpeg_append_timecode( demuxer, title, 0, 0, 0, 0, 1 );
            title->timecode_table[ title->timecode_table_size - 1 ].start_byte = start_byte;
            title->timecode_table[ title->timecode_table_size - 1 ].end_byte = end_byte;
            title->timecode_table[ title->timecode_table_size - 1 ].start_time = start_time;
            title->timecode_table[ title->timecode_table_size - 1 ].end_time = end_time;
         }
         else if ( !strcasecmp( string1, "ASTREAM:" ) )
            demuxer->astream_table[ start_byte ] = end_byte;
         else if ( !strcasecmp( string1, "VSTREAM:" ) )
            demuxer->vstream_table[ start_byte ] = end_byte;
         else if ( !strcasecmp( string1, "SIZE:" ) )
            title->total_bytes = start_byte;
         else if ( !strcasecmp( string1, "PACKETSIZE:" ) )
            demuxer->packet_size = start_byte;
      }
   }
   mpeg_demux_assign_programs( demuxer );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_demux_copy_titles( mpeg_demuxer_t *dst, mpeg_demuxer_t *src )
{
   long i;
   MPEG *file_mpeg = ( MPEG * )dst->file_mpeg;
   mpeg_title_t *dst_title, *src_title;
   dst->packet_size = src->packet_size;
   dst->total_titles = src->total_titles;
   dst->total_programs = src->total_programs;
   for( i = 0; i < MPEG3_MAX_STREAMS; i++ )
   {
      dst->astream_table[ i ] = src->astream_table[ i ];
      dst->vstream_table[ i ] = src->vstream_table[ i ];
   }
   for( i = 0; i < src->total_titles; i++ )
   {
      src_title = src->titles[ i ];
      dst_title = dst->titles[ i ] = mpeg_new_title( file_mpeg, src->titles[ i ]->fs->path );
      mpeg_copy_title( dst_title, src_title );
   }
   mpeg_demux_open_title( dst, src->current_title );
   return 0;
}
/***************************************************************************/
static inline double mpeg_demux_length( mpeg_demuxer_t *demuxer )
{
   mpeg_title_t *title;
   int i, j;
   //
   for( i = demuxer->total_titles - 1; i >= 0; i-- )
   {
      title = demuxer->titles[ i ];
      for( j = title->timecode_table_size - 1; j >= 0; j-- )
      {
         if ( title->timecode_table[ j ].program == demuxer->current_program )
         {
            return title->timecode_table[ j ].end_time - title->timecode_table[ j ].start_time + title->timecode_table[ j ].absolute_start_time;
         }
      }
   }
   return 1;
}
/***************************************************************************/
static inline int mpeg_demux_eof( mpeg_demuxer_t *demuxer )
{
   if ( demuxer->current_title >= 0 )
   {
      if ( mpeg_io_eof( demuxer->titles[ demuxer->current_title ]->fs ) && demuxer->current_title >= demuxer->total_titles - 1 )
         return 1;
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_demux_bof( mpeg_demuxer_t *demuxer )
{
   if ( demuxer->current_title >= 0 )
   {
      if ( mpeg_io_bof( demuxer->titles[ demuxer->current_title ]->fs ) && demuxer->current_title <= 0 )
         return 1;
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_demux_seek_byte( mpeg_demuxer_t *demuxer, long byte )
{
   long current_position;
   MPEG *file_mpeg = ( MPEG * ) demuxer->file_mpeg;
   mpeg_title_t *title = demuxer->titles[ demuxer->current_title ];
   //
   demuxer->data_position = 0;
   demuxer->data_size = 0;
   demuxer->error_flag = mpeg_io_seek( title->fs, byte );
   if ( !demuxer->error_flag && ( file_mpeg->is_transport_stream || file_mpeg->is_program_stream ) )
   {
      current_position = mpeg_io_tell( title->fs );
      if ( byte % demuxer->packet_size )
      {
         demuxer->error_flag |= mpeg_io_seek( title->fs, current_position - ( current_position % demuxer->packet_size ) );
      }
   }
   return demuxer->error_flag;
}
/***************************************************************************/
static inline int mpeg_demux_seek_time( mpeg_demuxer_t *demuxer, double new_time )
{
   int i, j, done = 0, result = 0;
   double byte_offset, new_byte_offset;
   double guess = 0, minimum = 65535;
   mpeg_title_t *title;
   mpeg_demux_timecode_t *timecode;
   //
   demuxer->error_flag = 0;
   i = 0;
   j = 0;
   title = demuxer->titles[ i ];
   timecode = &title->timecode_table[ j ];
   while( !demuxer->error_flag && !( timecode->absolute_start_time <= new_time && timecode->absolute_end_time > new_time && timecode->program == demuxer->current_program ) )
   {
      j++;
      if ( j >= title->timecode_table_size )
      {
         i++;
         j = 0;
         if ( i >= demuxer->total_titles )
         {
            demuxer->error_flag = 1;
            return 1;
         }
         else
         {
            mpeg_demux_open_title( demuxer, i );
         }
      }
      title = demuxer->titles[ i ];
      timecode = &title->timecode_table[ j ];
   }
   demuxer->current_timecode = j;
   byte_offset = ( ( new_time - timecode->absolute_start_time ) / ( timecode->absolute_end_time - timecode->absolute_start_time ) * ( timecode->end_byte - timecode->start_byte ) + timecode->start_byte );
   while( !done && !result && byte_offset >= 0 )
   {
      result = mpeg_demux_seek_byte( demuxer, ( long )byte_offset );
      if ( !result )
      {
         result = mpeg_read_next_packet( demuxer );
         guess = demuxer->time + demuxer->time_offset;
         if ( fabs( new_time - guess ) >= fabs( minimum ) )
            done = 1;
         else
         {
            minimum = guess - new_time;
            new_byte_offset = byte_offset + ( ( new_time - guess ) / ( timecode->end_time - timecode->start_time ) * ( timecode->end_byte - timecode->start_byte ) );
            if ( labs( ( long )new_byte_offset - ( long )byte_offset ) < demuxer->packet_size )
               done = 1;
            byte_offset = new_byte_offset;
         }
      }
   }
   if ( !result && byte_offset > demuxer->packet_size && minimum > 0 )
   {
      mpeg_read_prev_packet( demuxer );
      mpeg_read_prev_packet( demuxer );
   }
   demuxer->error_flag = result;
   return result;
}
/***************************************************************************/
static inline int mpeg_demux_seek_percentage( mpeg_demuxer_t *demuxer, double percentage )
{
   double total_bytes = 0;
   double absolute_position;
   long relative_position;
   int i, new_title;
   mpeg_title_t *title;
   //
   demuxer->error_flag = 0;
   for( i = 0; i < demuxer->total_titles; i++ )
      total_bytes += demuxer->titles[ i ]->total_bytes;
   absolute_position = percentage * total_bytes;
   for( new_title = 0, total_bytes = 0; new_title < demuxer->total_titles; new_title++ )
   {
      total_bytes += demuxer->titles[ new_title ]->total_bytes;
      if ( absolute_position < total_bytes )
         break;
   }
   if ( new_title >= demuxer->total_titles )
   {
      new_title = demuxer->total_titles - 1;
   }
   title = demuxer->titles[ new_title ];
   total_bytes -= title->total_bytes;
   relative_position = ( long )( absolute_position - total_bytes );
   for( demuxer->current_timecode = 0; demuxer->current_timecode < title->timecode_table_size; demuxer->current_timecode++ )
   {
      if ( title->timecode_table[ demuxer->current_timecode ].start_byte <= relative_position && title->timecode_table[ demuxer->current_timecode ].end_byte > relative_position )
      {
         break;
      }
   }
   if ( demuxer->current_timecode >= title->timecode_table_size )
      demuxer->current_timecode = title->timecode_table_size - 1;
   while( demuxer->current_timecode < title->timecode_table_size - 1 && title->timecode_table[ demuxer->current_timecode ].program != demuxer->current_program )
   {
      demuxer->current_timecode++;
   }
   if ( new_title != demuxer->current_title )
   {
      demuxer->error_flag = mpeg_demux_open_title( demuxer, new_title );
   }
   if ( !demuxer->error_flag )
      demuxer->error_flag = mpeg_io_seek( title->fs, relative_position );
   return demuxer->error_flag;
}
/***************************************************************************/
static inline double mpeg_demux_tell_percentage( mpeg_demuxer_t *demuxer )
{
   double total_bytes = 0;
   double position = 0;
   int i;
   //
   demuxer->error_flag = 0;
   position = mpeg_io_tell( demuxer->titles[ demuxer->current_title ]->fs );
   for( i = 0; i < demuxer->total_titles; i++ )
   {
      if ( i == demuxer->current_title )
      {
         position += total_bytes;
      }
      total_bytes += demuxer->titles[ i ]->total_bytes;
   }
   //
   return position / total_bytes;
}
/***************************************************************************/
static inline double mpeg_demux_get_time( mpeg_demuxer_t *demuxer )
{
   return demuxer->time;
}
/***************************************************************************/
static inline mpeg_demuxer_t* mpeg_get_demuxer( MPEG *file_mpeg )
{
   if ( file_mpeg->is_program_stream || file_mpeg->is_transport_stream )
   {
      if ( file_mpeg->has_audio )
         return file_mpeg->atrack[ 0 ]->demuxer;
      else if ( file_mpeg->has_video )
         return file_mpeg->vtrack[ 0 ]->demuxer;
   }
   //
   return 0;
}
/***************************************************************************/
static inline mpeg_fs_t* mpeg_new_fs( char *path )
{
   mpeg_fs_t *fs = ( mpeg_fs_t * ) calloc( 1, sizeof( mpeg_fs_t ) );
   //
   fs->css = mpeg_new_css();
   strcpy( fs->path, path );
   //
   return fs;
}
/***************************************************************************/
static inline int mpeg_delete_fs( mpeg_fs_t *fs )
{
   mpeg_delete_css( fs->css );
   free( fs );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_copy_fs( mpeg_fs_t *dst, mpeg_fs_t *src )
{
   strcpy( dst->path, src->path );
   dst->current_byte = 0;
   //
   return 0;
}
/***************************************************************************/
static inline long mpeg_io_get_total_bytes( mpeg_fs_t *fs )
{
   fseek( fs->fd, 0, SEEK_END );
   fs->total_bytes = ftell( fs->fd );
   fseek( fs->fd, 0, SEEK_SET );
   //
   return fs->total_bytes;
}
/***************************************************************************/
static inline int mpeg_io_open_file( mpeg_fs_t *fs )
{
   mpeg_get_keys( fs->css, fs->path );
   if ( !( fs->fd = fopen( fs->path, "rb" ) ) )
   {
	  DebugMessage( "MpegPlayer : mpeg_io_open_file" );
      return 1;
   }
   fs->total_bytes = mpeg_io_get_total_bytes( fs );
   if ( !fs->total_bytes )
   {
      fclose( fs->fd );
      return 1;
   }
   fs->current_byte = 0;
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_io_close_file( mpeg_fs_t *fs )
{
   if ( fs->fd )
      fclose( fs->fd );
   fs->fd = 0;
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_io_read_data( unsigned char *buffer, long bytes, mpeg_fs_t *fs )
{
   int result = 0;
   //
   result = !fread( buffer, 1, bytes, fs->fd );
   fs->current_byte += bytes;
   //
   return ( result && bytes );
}
/***************************************************************************/
static inline int mpeg_io_seek( mpeg_fs_t *fs, long byte )
{
   fs->current_byte = byte;
   return fseek( fs->fd, byte, SEEK_SET );
}
/***************************************************************************/
static inline int mpeg_io_seek_relative( mpeg_fs_t *fs, long bytes )
{
   fs->current_byte += bytes;
   return fseek( fs->fd, fs->current_byte, SEEK_SET );
}
/***************************************************************************/
static inline mpeg_title_t* mpeg_new_title( MPEG *file_mpeg, char *path )
{
   mpeg_title_t *title = ( mpeg_title_t * ) calloc( 1, sizeof( mpeg_title_t ) );
   title->fs = mpeg_new_fs( path );
   title->file_mpeg = file_mpeg;
   return title;
}
/***************************************************************************/
static inline int mpeg_delete_title( mpeg_title_t *title )
{
   mpeg_delete_fs( title->fs );
   if ( title->timecode_table_size )
   {
      free( title->timecode_table );
   }
   free( title );
   return 0;
}
/***************************************************************************/
static inline int mpeg_copy_title( mpeg_title_t *dst, mpeg_title_t *src )
{
   int i;
   //
   mpeg_copy_fs( dst->fs, src->fs );
   dst->total_bytes = src->total_bytes;
   if ( src->timecode_table_size )
   {
      dst->timecode_table_allocation = src->timecode_table_allocation;
      dst->timecode_table_size = src->timecode_table_size;
      dst->timecode_table = ( mpeg_demux_timecode_t * ) calloc( 1, sizeof( mpeg_demux_timecode_t ) * dst->timecode_table_allocation );
      for( i = 0; i < dst->timecode_table_size; i++ )
      {
         dst->timecode_table[ i ] = src->timecode_table[ i ];
      }
   }
   //
   return 0;
}
/***************************************************************************/
static inline mpeg_vtrack_t* mpeg_new_vtrack( MPEG *file_mpeg, int stream_id, mpeg_demuxer_t *demuxer )
{
   mpeg_vtrack_t *new_vtrack;
   new_vtrack = ( mpeg_vtrack_t * ) calloc( 1, sizeof( mpeg_vtrack_t ) );
   new_vtrack->demuxer = mpeg_new_demuxer( file_mpeg, 0, 1, stream_id );
   if ( demuxer )
      mpeg_demux_copy_titles( new_vtrack->demuxer, demuxer );
   new_vtrack->current_position = 0;
   new_vtrack->video = mpeg_video_new( file_mpeg, new_vtrack );
   if ( !new_vtrack->video )
   {
      mpeg_delete_vtrack( file_mpeg, new_vtrack );
      new_vtrack = 0;
   }
   return new_vtrack;
}
/***************************************************************************/
static inline int mpeg_delete_vtrack( MPEG *file_mpeg, mpeg_vtrack_t *vtrack )
{
   if ( vtrack->video )
      mpeg_video_delete( vtrack->video );
   if ( vtrack->demuxer )
      mpeg_delete_demuxer( vtrack->demuxer );
   free( vtrack );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_audio_dct64_1( float *out0, float *out1, float *b1, float *b2, float *samples )
{
   float *costab = mpeg_pnts[ 0 ];
   //
   b1[ 0x00 ] = samples[ 0x00 ] + samples[ 0x1F ];
   b1[ 0x01 ] = samples[ 0x01 ] + samples[ 0x1E ];
   b1[ 0x1F ] = ( samples[ 0x00 ] - samples[ 0x1F ] ) * costab[ 0x0 ];
   b1[ 0x1E ] = ( samples[ 0x01 ] - samples[ 0x1E ] ) * costab[ 0x1 ];
   b1[ 0x02 ] = samples[ 0x02 ] + samples[ 0x1D ];
   b1[ 0x03 ] = samples[ 0x03 ] + samples[ 0x1C ];
   b1[ 0x1D ] = ( samples[ 0x02 ] - samples[ 0x1D ] ) * costab[ 0x2 ];
   b1[ 0x1C ] = ( samples[ 0x03 ] - samples[ 0x1C ] ) * costab[ 0x3 ];
   b1[ 0x04 ] = samples[ 0x04 ] + samples[ 0x1B ];
   b1[ 0x05 ] = samples[ 0x05 ] + samples[ 0x1A ];
   b1[ 0x1B ] = ( samples[ 0x04 ] - samples[ 0x1B ] ) * costab[ 0x4 ];
   b1[ 0x1A ] = ( samples[ 0x05 ] - samples[ 0x1A ] ) * costab[ 0x5 ];
   b1[ 0x06 ] = samples[ 0x06 ] + samples[ 0x19 ];
   b1[ 0x07 ] = samples[ 0x07 ] + samples[ 0x18 ];
   b1[ 0x19 ] = ( samples[ 0x06 ] - samples[ 0x19 ] ) * costab[ 0x6 ];
   b1[ 0x18 ] = ( samples[ 0x07 ] - samples[ 0x18 ] ) * costab[ 0x7 ];
   b1[ 0x08 ] = samples[ 0x08 ] + samples[ 0x17 ];
   b1[ 0x09 ] = samples[ 0x09 ] + samples[ 0x16 ];
   b1[ 0x17 ] = ( samples[ 0x08 ] - samples[ 0x17 ] ) * costab[ 0x8 ];
   b1[ 0x16 ] = ( samples[ 0x09 ] - samples[ 0x16 ] ) * costab[ 0x9 ];
   b1[ 0x0A ] = samples[ 0x0A ] + samples[ 0x15 ];
   b1[ 0x0B ] = samples[ 0x0B ] + samples[ 0x14 ];
   b1[ 0x15 ] = ( samples[ 0x0A ] - samples[ 0x15 ] ) * costab[ 0xA ];
   b1[ 0x14 ] = ( samples[ 0x0B ] - samples[ 0x14 ] ) * costab[ 0xB ];
   b1[ 0x0C ] = samples[ 0x0C ] + samples[ 0x13 ];
   b1[ 0x0D ] = samples[ 0x0D ] + samples[ 0x12 ];
   b1[ 0x13 ] = ( samples[ 0x0C ] - samples[ 0x13 ] ) * costab[ 0xC ];
   b1[ 0x12 ] = ( samples[ 0x0D ] - samples[ 0x12 ] ) * costab[ 0xD ];
   b1[ 0x0E ] = samples[ 0x0E ] + samples[ 0x11 ];
   b1[ 0x0F ] = samples[ 0x0F ] + samples[ 0x10 ];
   b1[ 0x11 ] = ( samples[ 0x0E ] - samples[ 0x11 ] ) * costab[ 0xE ];
   b1[ 0x10 ] = ( samples[ 0x0F ] - samples[ 0x10 ] ) * costab[ 0xF ];
   costab = mpeg_pnts[ 1 ];
   b2[ 0x00 ] = b1[ 0x00 ] + b1[ 0x0F ]; 
   b2[ 0x01 ] = b1[ 0x01 ] + b1[ 0x0E ]; 
   b2[ 0x0F ] = ( b1[ 0x00 ] - b1[ 0x0F ] ) * costab[ 0 ];
   b2[ 0x0E ] = ( b1[ 0x01 ] - b1[ 0x0E ] ) * costab[ 1 ];
   b2[ 0x02 ] = b1[ 0x02 ] + b1[ 0x0D ]; 
   b2[ 0x03 ] = b1[ 0x03 ] + b1[ 0x0C ]; 
   b2[ 0x0D ] = ( b1[ 0x02 ] - b1[ 0x0D ] ) * costab[ 2 ];
   b2[ 0x0C ] = ( b1[ 0x03 ] - b1[ 0x0C ] ) * costab[ 3 ];
   b2[ 0x04 ] = b1[ 0x04 ] + b1[ 0x0B ]; 
   b2[ 0x05 ] = b1[ 0x05 ] + b1[ 0x0A ]; 
   b2[ 0x0B ] = ( b1[ 0x04 ] - b1[ 0x0B ] ) * costab[ 4 ];
   b2[ 0x0A ] = ( b1[ 0x05 ] - b1[ 0x0A ] ) * costab[ 5 ];
   b2[ 0x06 ] = b1[ 0x06 ] + b1[ 0x09 ]; 
   b2[ 0x07 ] = b1[ 0x07 ] + b1[ 0x08 ]; 
   b2[ 0x09 ] = ( b1[ 0x06 ] - b1[ 0x09 ] ) * costab[ 6 ];
   b2[ 0x08 ] = ( b1[ 0x07 ] - b1[ 0x08 ] ) * costab[ 7 ];
   b2[ 0x10 ] = b1[ 0x10 ] + b1[ 0x1F ];
   b2[ 0x11 ] = b1[ 0x11 ] + b1[ 0x1E ];
   b2[ 0x1F ] = ( b1[ 0x1F ] - b1[ 0x10 ] ) * costab[ 0 ];
   b2[ 0x1E ] = ( b1[ 0x1E ] - b1[ 0x11 ] ) * costab[ 1 ];
   b2[ 0x12 ] = b1[ 0x12 ] + b1[ 0x1D ];
   b2[ 0x13 ] = b1[ 0x13 ] + b1[ 0x1C ];
   b2[ 0x1D ] = ( b1[ 0x1D ] - b1[ 0x12 ] ) * costab[ 2 ];
   b2[ 0x1C ] = ( b1[ 0x1C ] - b1[ 0x13 ] ) * costab[ 3 ];
   b2[ 0x14 ] = b1[ 0x14 ] + b1[ 0x1B ];
   b2[ 0x15 ] = b1[ 0x15 ] + b1[ 0x1A ];
   b2[ 0x1B ] = ( b1[ 0x1B ] - b1[ 0x14 ] ) * costab[ 4 ];
   b2[ 0x1A ] = ( b1[ 0x1A ] - b1[ 0x15 ] ) * costab[ 5 ];
   b2[ 0x16 ] = b1[ 0x16 ] + b1[ 0x19 ];
   b2[ 0x17 ] = b1[ 0x17 ] + b1[ 0x18 ];
   b2[ 0x19 ] = ( b1[ 0x19 ] - b1[ 0x16 ] ) * costab[ 6 ];
   b2[ 0x18 ] = ( b1[ 0x18 ] - b1[ 0x17 ] ) * costab[ 7 ];
   costab = mpeg_pnts[ 2 ];
   b1[ 0x00 ] = b2[ 0x00 ] + b2[ 0x07 ];
   b1[ 0x07 ] = ( b2[ 0x00 ] - b2[ 0x07 ] ) * costab[ 0 ];
   b1[ 0x01 ] = b2[ 0x01 ] + b2[ 0x06 ];
   b1[ 0x06 ] = ( b2[ 0x01 ] - b2[ 0x06 ] ) * costab[ 1 ];
   b1[ 0x02 ] = b2[ 0x02 ] + b2[ 0x05 ];
   b1[ 0x05 ] = ( b2[ 0x02 ] - b2[ 0x05 ] ) * costab[ 2 ];
   b1[ 0x03 ] = b2[ 0x03 ] + b2[ 0x04 ];
   b1[ 0x04 ] = ( b2[ 0x03 ] - b2[ 0x04 ] ) * costab[ 3 ];
   b1[ 0x08 ] = b2[ 0x08 ] + b2[ 0x0F ];
   b1[ 0x0F ] = ( b2[ 0x0F ] - b2[ 0x08 ] ) * costab[ 0 ];
   b1[ 0x09 ] = b2[ 0x09 ] + b2[ 0x0E ];
   b1[ 0x0E ] = ( b2[ 0x0E ] - b2[ 0x09 ] ) * costab[ 1 ];
   b1[ 0x0A ] = b2[ 0x0A ] + b2[ 0x0D ];
   b1[ 0x0D ] = ( b2[ 0x0D ] - b2[ 0x0A ] ) * costab[ 2 ];
   b1[ 0x0B ] = b2[ 0x0B ] + b2[ 0x0C ];
   b1[ 0x0C ] = ( b2[ 0x0C ] - b2[ 0x0B ] ) * costab[ 3 ];
   b1[ 0x10 ] = b2[ 0x10 ] + b2[ 0x17 ];
   b1[ 0x17 ] = ( b2[ 0x10 ] - b2[ 0x17 ] ) * costab[ 0 ];
   b1[ 0x11 ] = b2[ 0x11 ] + b2[ 0x16 ];
   b1[ 0x16 ] = ( b2[ 0x11 ] - b2[ 0x16 ] ) * costab[ 1 ];
   b1[ 0x12 ] = b2[ 0x12 ] + b2[ 0x15 ];
   b1[ 0x15 ] = ( b2[ 0x12 ] - b2[ 0x15 ] ) * costab[ 2 ];
   b1[ 0x13 ] = b2[ 0x13 ] + b2[ 0x14 ];
   b1[ 0x14 ] = ( b2[ 0x13 ] - b2[ 0x14 ] ) * costab[ 3 ];
   b1[ 0x18 ] = b2[ 0x18 ] + b2[ 0x1F ];
   b1[ 0x1F ] = ( b2[ 0x1F ] - b2[ 0x18 ] ) * costab[ 0 ];
   b1[ 0x19 ] = b2[ 0x19 ] + b2[ 0x1E ];
   b1[ 0x1E ] = ( b2[ 0x1E ] - b2[ 0x19 ] ) * costab[ 1 ];
   b1[ 0x1A ] = b2[ 0x1A ] + b2[ 0x1D ];
   b1[ 0x1D ] = ( b2[ 0x1D ] - b2[ 0x1A ] ) * costab[ 2 ];
   b1[ 0x1B ] = b2[ 0x1B ] + b2[ 0x1C ];
   b1[ 0x1C ] = ( b2[ 0x1C ] - b2[ 0x1B ] ) * costab[ 3 ];
   {
      float const cos0 = mpeg_pnts[ 3 ][ 0 ];
      float const cos1 = mpeg_pnts[ 3 ][ 1 ];
      b2[ 0x00 ] = b1[ 0x00 ] + b1[ 0x03 ];
      b2[ 0x03 ] = ( b1[ 0x00 ] - b1[ 0x03 ] ) * cos0;
      b2[ 0x01 ] = b1[ 0x01 ] + b1[ 0x02 ];
      b2[ 0x02 ] = ( b1[ 0x01 ] - b1[ 0x02 ] ) * cos1;
      b2[ 0x04 ] = b1[ 0x04 ] + b1[ 0x07 ];
      b2[ 0x07 ] = ( b1[ 0x07 ] - b1[ 0x04 ] ) * cos0;
      b2[ 0x05 ] = b1[ 0x05 ] + b1[ 0x06 ];
      b2[ 0x06 ] = ( b1[ 0x06 ] - b1[ 0x05 ] ) * cos1;
      b2[ 0x08 ] = b1[ 0x08 ] + b1[ 0x0B ];
      b2[ 0x0B ] = ( b1[ 0x08 ] - b1[ 0x0B ] ) * cos0;
      b2[ 0x09 ] = b1[ 0x09 ] + b1[ 0x0A ];
      b2[ 0x0A ] = ( b1[ 0x09 ] - b1[ 0x0A ] ) * cos1;
      b2[ 0x0C ] = b1[ 0x0C ] + b1[ 0x0F ];
      b2[ 0x0F ] = ( b1[ 0x0F ] - b1[ 0x0C ] ) * cos0;
      b2[ 0x0D ] = b1[ 0x0D ] + b1[ 0x0E ];
      b2[ 0x0E ] = ( b1[ 0x0E ] - b1[ 0x0D ] ) * cos1;
      b2[ 0x10 ] = b1[ 0x10 ] + b1[ 0x13 ];
      b2[ 0x13 ] = ( b1[ 0x10 ] - b1[ 0x13 ] ) * cos0;
      b2[ 0x11 ] = b1[ 0x11 ] + b1[ 0x12 ];
      b2[ 0x12 ] = ( b1[ 0x11 ] - b1[ 0x12 ] ) * cos1;
      b2[ 0x14 ] = b1[ 0x14 ] + b1[ 0x17 ];
      b2[ 0x17 ] = ( b1[ 0x17 ] - b1[ 0x14 ] ) * cos0;
      b2[ 0x15 ] = b1[ 0x15 ] + b1[ 0x16 ];
      b2[ 0x16 ] = ( b1[ 0x16 ] - b1[ 0x15 ] ) * cos1;
      b2[ 0x18 ] = b1[ 0x18 ] + b1[ 0x1B ];
      b2[ 0x1B ] = ( b1[ 0x18 ] - b1[ 0x1B ] ) * cos0;
      b2[ 0x19 ] = b1[ 0x19 ] + b1[ 0x1A ];
      b2[ 0x1A ] = ( b1[ 0x19 ] - b1[ 0x1A ] ) * cos1;
      b2[ 0x1C ] = b1[ 0x1C ] + b1[ 0x1F ];
      b2[ 0x1F ] = ( b1[ 0x1F ] - b1[ 0x1C ] ) * cos0;
      b2[ 0x1D ] = b1[ 0x1D ] + b1[ 0x1E ];
      b2[ 0x1E ] = ( b1[ 0x1E ] - b1[ 0x1D ] ) * cos1;
   }
   {
      float const cos0 = mpeg_pnts[ 4 ][ 0 ];
      b1[ 0x00 ] = b2[ 0x00 ] + b2[ 0x01 ];
      b1[ 0x01 ] = ( b2[ 0x00 ] - b2[ 0x01 ] ) * cos0;
      b1[ 0x02 ] = b2[ 0x02 ] + b2[ 0x03 ];
      b1[ 0x03 ] = ( b2[ 0x03 ] - b2[ 0x02 ] ) * cos0;
      b1[ 0x02 ] += b1[ 0x03 ];
      b1[ 0x04 ] = b2[ 0x04 ] + b2[ 0x05 ];
      b1[ 0x05 ] = ( b2[ 0x04 ] - b2[ 0x05 ] ) * cos0;
      b1[ 0x06 ] = b2[ 0x06 ] + b2[ 0x07 ];
      b1[ 0x07 ] = ( b2[ 0x07 ] - b2[ 0x06 ] ) * cos0;
      b1[ 0x06 ] += b1[ 0x07 ];
      b1[ 0x04 ] += b1[ 0x06 ];
      b1[ 0x06 ] += b1[ 0x05 ];
      b1[ 0x05 ] += b1[ 0x07 ];
      b1[ 0x08 ] = b2[ 0x08 ] + b2[ 0x09 ];
      b1[ 0x09 ] = ( b2[ 0x08 ] - b2[ 0x09 ] ) * cos0;
      b1[ 0x0A ] = b2[ 0x0A ] + b2[ 0x0B ];
      b1[ 0x0B ] = ( b2[ 0x0B ] - b2[ 0x0A ] ) * cos0;
      b1[ 0x0A ] += b1[ 0x0B ];
      b1[ 0x0C ] = b2[ 0x0C ] + b2[ 0x0D ];
      b1[ 0x0D ] = ( b2[ 0x0C ] - b2[ 0x0D ] ) * cos0;
      b1[ 0x0E ] = b2[ 0x0E ] + b2[ 0x0F ];
      b1[ 0x0F ] = ( b2[ 0x0F ] - b2[ 0x0E ] ) * cos0;
      b1[ 0x0E ] += b1[ 0x0F ];
      b1[ 0x0C ] += b1[ 0x0E ];
      b1[ 0x0E ] += b1[ 0x0D ];
      b1[ 0x0D ] += b1[ 0x0F ];
      b1[ 0x10 ] = b2[ 0x10 ] + b2[ 0x11 ];
      b1[ 0x11 ] = ( b2[ 0x10 ] - b2[ 0x11 ] ) * cos0;
      b1[ 0x12 ] = b2[ 0x12 ] + b2[ 0x13 ];
      b1[ 0x13 ] = ( b2[ 0x13 ] - b2[ 0x12 ] ) * cos0;
      b1[ 0x12 ] += b1[ 0x13 ];
      b1[ 0x14 ] = b2[ 0x14 ] + b2[ 0x15 ];
      b1[ 0x15 ] = ( b2[ 0x14 ] - b2[ 0x15 ] ) * cos0;
      b1[ 0x16 ] = b2[ 0x16 ] + b2[ 0x17 ];
      b1[ 0x17 ] = ( b2[ 0x17 ] - b2[ 0x16 ] ) * cos0;
      b1[ 0x16 ] += b1[ 0x17 ];
      b1[ 0x14 ] += b1[ 0x16 ];
      b1[ 0x16 ] += b1[ 0x15 ];
      b1[ 0x15 ] += b1[ 0x17 ];
      b1[ 0x18 ] = b2[ 0x18 ] + b2[ 0x19 ];
      b1[ 0x19 ] = ( b2[ 0x18 ] - b2[ 0x19 ] ) * cos0;
      b1[ 0x1A ] = b2[ 0x1A ] + b2[ 0x1B ];
      b1[ 0x1B ] = ( b2[ 0x1B ] - b2[ 0x1A ] ) * cos0;
      b1[ 0x1A ] += b1[ 0x1B ];
      b1[ 0x1C ] = b2[ 0x1C ] + b2[ 0x1D ];
      b1[ 0x1D ] = ( b2[ 0x1C ] - b2[ 0x1D ] ) * cos0;
      b1[ 0x1E ] = b2[ 0x1E ] + b2[ 0x1F ];
      b1[ 0x1F ] = ( b2[ 0x1F ] - b2[ 0x1E ] ) * cos0;
      b1[ 0x1E ] += b1[ 0x1F ];
      b1[ 0x1C ] += b1[ 0x1E ];
      b1[ 0x1E ] += b1[ 0x1D ];
      b1[ 0x1D ] += b1[ 0x1F ];
   }
   out0[ 0x10*16 ] = b1[ 0x00 ];
   out0[ 0x10*12 ] = b1[ 0x04 ];
   out0[ 0x10* 8 ] = b1[ 0x02 ];
   out0[ 0x10* 4 ] = b1[ 0x06 ];
   out0[ 0x10* 0 ] = b1[ 0x01 ];
   out1[ 0x10* 0 ] = b1[ 0x01 ];
   out1[ 0x10* 4 ] = b1[ 0x05 ];
   out1[ 0x10* 8 ] = b1[ 0x03 ];
   out1[ 0x10*12 ] = b1[ 0x07 ];
   out0[ 0x10*14 ] = b1[ 0x08 ] + b1[ 0x0C ];
   out0[ 0x10*10 ] = b1[ 0x0C ] + b1[ 0x0a ];
   out0[ 0x10* 6 ] = b1[ 0x0A ] + b1[ 0x0E ];
   out0[ 0x10* 2 ] = b1[ 0x0E ] + b1[ 0x09 ];
   out1[ 0x10* 2 ] = b1[ 0x09 ] + b1[ 0x0D ];
   out1[ 0x10* 6 ] = b1[ 0x0D ] + b1[ 0x0B ];
   out1[ 0x10*10 ] = b1[ 0x0B ] + b1[ 0x0F ];
   out1[ 0x10*14 ] = b1[ 0x0F ];
   {
      float tmp;
      tmp = b1[ 0x18 ] + b1[ 0x1C ];
      out0[ 0x10*15 ] = tmp + b1[ 0x10 ];
      out0[ 0x10*13 ] = tmp + b1[ 0x14 ];
      tmp = b1[ 0x1C ] + b1[ 0x1A ];
      out0[ 0x10*11 ] = tmp + b1[ 0x14 ];
      out0[ 0x10* 9 ] = tmp + b1[ 0x12 ];
      tmp = b1[ 0x1A ] + b1[ 0x1E ];
      out0[ 0x10* 7 ] = tmp + b1[ 0x12 ];
      out0[ 0x10* 5 ] = tmp + b1[ 0x16 ];
      tmp = b1[ 0x1E ] + b1[ 0x19 ];
      out0[ 0x10* 3 ] = tmp + b1[ 0x16 ];
      out0[ 0x10* 1 ] = tmp + b1[ 0x11 ];
      tmp = b1[ 0x19 ] + b1[ 0x1D ];
      out1[ 0x10* 1 ] = tmp + b1[ 0x11 ];
      out1[ 0x10* 3 ] = tmp + b1[ 0x15 ]; 
      tmp = b1[ 0x1D ] + b1[ 0x1B ];
      out1[ 0x10* 5 ] = tmp + b1[ 0x15 ];
      out1[ 0x10* 7 ] = tmp + b1[ 0x13 ];
      tmp = b1[ 0x1B ] + b1[ 0x1F ];
      out1[ 0x10* 9 ] = tmp + b1[ 0x13 ];
      out1[ 0x10*11 ] = tmp + b1[ 0x17 ];
      out1[ 0x10*13 ] = b1[ 0x17 ] + b1[ 0x1F ];
      out1[ 0x10*15 ] = b1[ 0x1F ];
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_audio_dct64( float *a, float *b, float *c )
{
   float bufs[ 0x40 ];
   //
   return mpeg_audio_dct64_1( a, b, bufs, bufs + 0x20, c );
}
/***************************************************************************/
static inline int mpeg_audio_dct36( float *inbuf, float *o1, float *o2, float *wintab, float *tsbuf )
{
   float tmp[ 18 ];
   //
   {
      float *in = inbuf;
      //
      in[ 17 ]+=in[ 16 ]; in[ 16 ]+=in[ 15 ]; in[ 15 ]+=in[ 14 ];
      in[ 14 ]+=in[ 13 ]; in[ 13 ]+=in[ 12 ]; in[ 12 ]+=in[ 11 ];
      in[ 11 ]+=in[ 10 ]; in[ 10 ]+=in[ 9 ]; in[ 9 ] +=in[ 8 ];
      in[ 8 ] +=in[ 7 ]; in[ 7 ] +=in[ 6 ]; in[ 6 ] +=in[ 5 ];
      in[ 5 ] +=in[ 4 ]; in[ 4 ] +=in[ 3 ]; in[ 3 ] +=in[ 2 ];
      in[ 2 ] +=in[ 1 ]; in[ 1 ] +=in[ 0 ];
      in[ 17 ]+=in[ 15 ]; in[ 15 ]+=in[ 13 ]; in[ 13 ]+=in[ 11 ]; in[ 11 ]+=in[ 9 ];
      in[ 9 ] +=in[ 7 ]; in[ 7 ] +=in[ 5 ]; in[ 5 ] +=in[ 3 ]; in[ 3 ] +=in[ 1 ];
      {
         float t3;
         {
            float t0, t1, t2;
            t0 = mpeg_COS6_2 * ( in[ 8 ] + in[ 16 ] - in[ 4 ] );
            t1 = mpeg_COS6_2 * in[ 12 ];
            t3 = in[ 0 ];
            t2 = t3 - t1 - t1;
            tmp[ 1 ] = tmp[ 7 ] = t2 - t0;
            tmp[ 4 ] = t2 + t0 + t0;
            t3 += t1;
            t2 = mpeg_COS6_1 * ( in[ 10 ] + in[ 14 ] - in[ 2 ] );
            tmp[ 1 ] -= t2;
            tmp[ 7 ] += t2;
         }
         {
            float t0, t1, t2;
            t0 = mpeg_cos9[ 0 ] * ( in[ 4 ] + in[ 8 ] );
            t1 = mpeg_cos9[ 1 ] * ( in[ 8 ] - in[ 16 ] );
            t2 = mpeg_cos9[ 2 ] * ( in[ 4 ] + in[ 16 ] );
            tmp[ 2 ] = tmp[ 6 ] = t3 - t0 - t2;
            tmp[ 0 ] = tmp[ 8 ] = t3 + t0 + t1;
            tmp[ 3 ] = tmp[ 5 ] = t3 - t1 + t2;
         }
      }
      {
         float t1, t2, t3;
         t1 = mpeg_cos18[ 0 ] * ( in[ 2 ] + in[ 10 ] );
         t2 = mpeg_cos18[ 1 ] * ( in[ 10 ] - in[ 14 ] );
         t3 = mpeg_COS6_1 * in[ 6 ];
         {
            float t0 = t1 + t2 + t3;
            tmp[ 0 ] += t0;
            tmp[ 8 ] -= t0;
         }
         t2 -= t3;
         t1 -= t3;
         t3 = mpeg_cos18[ 2 ] * ( in[ 2 ] + in[ 14 ] );
         t1 += t3;
         tmp[ 3 ] += t1;
         tmp[ 5 ] -= t1;
         t2 -= t3;
         tmp[ 2 ] += t2;
         tmp[ 6 ] -= t2;
      }
      {
         float t0, t1, t2, t3, t4, t5, t6, t7;
         t1 = mpeg_COS6_2 * in[ 13 ];
         t2 = mpeg_COS6_2 * ( in[ 9 ] + in[ 17 ] - in[ 5 ] );
         t3 = in[ 1 ] + t1;
         t4 = in[ 1 ] - t1 - t1;
         t5 = t4 - t2;
         t0 = mpeg_cos9[ 0 ] * ( in[ 5 ] + in[ 9 ] );
         t1 = mpeg_cos9[ 1 ] * ( in[ 9 ] - in[ 17 ] );
         tmp[ 13 ] = ( t4 + t2 + t2 ) * mpeg_tfcos36[ 17-13 ];
         t2 = mpeg_cos9[ 2 ] * ( in[ 5 ] + in[ 17 ] );
         t6 = t3 - t0 - t2;
         t0 += t3 + t1;
         t3 += t2 - t1;
         t2 = mpeg_cos18[ 0 ] * ( in[ 3 ] + in[ 11 ] );
         t4 = mpeg_cos18[ 1 ] * ( in[ 11 ] - in[ 15 ] );
         t7 = mpeg_COS6_1 * in[ 7 ];
         t1 = t2 + t4 + t7;
         tmp[ 17 ] = ( t0 + t1 ) * mpeg_tfcos36[ 17-17 ];
         tmp[ 9 ] = ( t0 - t1 ) * mpeg_tfcos36[ 17-9 ];
         t1 = mpeg_cos18[ 2 ] * ( in[ 3 ] + in[ 15 ] );
         t2 += t1 - t7;
         tmp[ 14 ] = ( t3 + t2 ) * mpeg_tfcos36[ 17-14 ];
         t0 = mpeg_COS6_1 * ( in[ 11 ] + in[ 15 ] - in[ 3 ] );
         tmp[ 12 ] = ( t3 - t2 ) * mpeg_tfcos36[ 17-12 ];
         t4 -= t1 + t7;
         tmp[ 16 ] = ( t5 - t0 ) * mpeg_tfcos36[ 17-16 ];
         tmp[ 10 ] = ( t5 + t0 ) * mpeg_tfcos36[ 17-10 ];
         tmp[ 15 ] = ( t6 + t4 ) * mpeg_tfcos36[ 17-15 ];
         tmp[ 11 ] = ( t6 - t4 ) * mpeg_tfcos36[ 17-11 ];
      }
      #define MACRO(                   v ) { float tmpval; tmpval = tmp[ ( v ) ] + tmp[ 17-( v ) ]; out2[ 9+( v ) ] = tmpval * w[ 27+( v ) ]; out2[ 8-( v ) ] = tmpval * w[ 26-( v ) ]; tmpval = tmp[ ( v ) ] - tmp[ 17-( v ) ]; ts[ SBLIMIT*( 8-( v ) ) ] = out1[ 8-( v ) ] + tmpval * w[ 8-( v ) ]; ts[ SBLIMIT*( 9+( v ) ) ] = out1[ 9+( v ) ] + tmpval * w[ 9+( v ) ]; }
      {
	 float *out2 = o2;
	 float *w = wintab;
	 float *out1 = o1;
	 float *ts = tsbuf;
         //
         MACRO( 0 );
         MACRO( 1 );
         MACRO( 2 );
         MACRO( 3 );
         MACRO( 4 );
         MACRO( 5 );
         MACRO( 6 );
         MACRO( 7 );
         MACRO( 8 );
      }
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_audio_dct12( float *in, float *rawout1, float *rawout2, float *wi, float *ts )
{
   #define DCT12_PART1              in5 = in[ 5*3 ]; in5 += ( in4 = in[ 4*3 ] ); in4 += ( in3 = in[ 3*3 ] ); in3 += ( in2 = in[ 2*3 ] ); in2 += ( in1 = in[ 1*3 ] ); in1 += ( in0 = in[ 0*3 ] ); in5 += in3; in3 += in1; in2 *= mpeg_COS6_1; in3 *= mpeg_COS6_1; 
   #define DCT12_PART2              in0 += in4 * mpeg_COS6_2; in4 = in0 + in2; in0 -= in2; in1 += in5 * mpeg_COS6_2; in5 = ( in1 + in3 ) * mpeg_tfcos12[ 0 ]; in1 = ( in1 - in3 ) * mpeg_tfcos12[ 2 ]; in3 = in4 + in5; in4 -= in5; in2 = in0 + in1; in0 -= in1;
   {
      float in0, in1, in2, in3, in4, in5;
      float *out1 = rawout1;
      ts[ SBLIMIT*0 ] = out1[ 0 ]; ts[ SBLIMIT*1 ] = out1[ 1 ]; ts[ SBLIMIT*2 ] = out1[ 2 ];
      ts[ SBLIMIT*3 ] = out1[ 3 ]; ts[ SBLIMIT*4 ] = out1[ 4 ]; ts[ SBLIMIT*5 ] = out1[ 5 ];
      DCT12_PART1
      {
         float tmp0, tmp1 = ( in0 - in4 );
         {
            float tmp2 = ( in1 - in5 ) * mpeg_tfcos12[ 1 ];
            tmp0 = tmp1 + tmp2;
            tmp1 -= tmp2;
         }
         ts[ ( 17-1 )*SBLIMIT ] = out1[ 17-1 ] + tmp0 * wi[ 11-1 ];
         ts[ ( 12+1 )*SBLIMIT ] = out1[ 12+1 ] + tmp0 * wi[ 6+1 ];
         ts[ ( 6 +1 )*SBLIMIT ] = out1[ 6 +1 ] + tmp1 * wi[ 1 ];
         ts[ ( 11-1 )*SBLIMIT ] = out1[ 11-1 ] + tmp1 * wi[ 5-1 ];
      }
      DCT12_PART2
      ts[ ( 17-0 )*SBLIMIT ] = out1[ 17-0 ] + in2 * wi[ 11-0 ];
      ts[ ( 12+0 )*SBLIMIT ] = out1[ 12+0 ] + in2 * wi[ 6+0 ];
      ts[ ( 12+2 )*SBLIMIT ] = out1[ 12+2 ] + in3 * wi[ 6+2 ];
      ts[ ( 17-2 )*SBLIMIT ] = out1[ 17-2 ] + in3 * wi[ 11-2 ];
      ts[ ( 6+0 )*SBLIMIT ] = out1[ 6+0 ] + in0 * wi[ 0 ];
      ts[ ( 11-0 )*SBLIMIT ] = out1[ 11-0 ] + in0 * wi[ 5-0 ];
      ts[ ( 6+2 )*SBLIMIT ] = out1[ 6+2 ] + in4 * wi[ 2 ];
      ts[ ( 11-2 )*SBLIMIT ] = out1[ 11-2 ] + in4 * wi[ 5-2 ];
   }
   in++;
   {
      float in0, in1, in2, in3, in4, in5;
      float *out2 = rawout2;
      DCT12_PART1
      {
         float tmp0, tmp1 = ( in0 - in4 );
         {
            float tmp2 = ( in1 - in5 ) * mpeg_tfcos12[ 1 ];
            tmp0 = tmp1 + tmp2;
            tmp1 -= tmp2;
         }
         out2[ 5-1 ] = tmp0 * wi[ 11-1 ];
         out2[ 0+1 ] = tmp0 * wi[ 6+1 ];
         ts[ ( 12+1 )*SBLIMIT ] += tmp1 * wi[ 1 ];
         ts[ ( 17-1 )*SBLIMIT ] += tmp1 * wi[ 5-1 ];
      }
      DCT12_PART2
      out2[ 5-0 ] = in2 * wi[ 11-0 ];
      out2[ 0+0 ] = in2 * wi[ 6+0 ];
      out2[ 0+2 ] = in3 * wi[ 6+2 ];
      out2[ 5-2 ] = in3 * wi[ 11-2 ];
      ts[ ( 12+0 )*SBLIMIT ] += in0 * wi[ 0 ];
      ts[ ( 17-0 )*SBLIMIT ] += in0 * wi[ 5-0 ];
      ts[ ( 12+2 )*SBLIMIT ] += in4 * wi[ 2 ];
      ts[ ( 17-2 )*SBLIMIT ] += in4 * wi[ 5-2 ];
   }
   in++; 
   {
      float in0, in1, in2, in3, in4, in5;
      float *out2 = rawout2;
      out2[ 12 ]=out2[ 13 ]=out2[ 14 ]=out2[ 15 ]=out2[ 16 ]=out2[ 17 ]=0.0;
      DCT12_PART1
      {
         float tmp0, tmp1 = ( in0 - in4 );
         {
            float tmp2 = ( in1 - in5 ) * mpeg_tfcos12[ 1 ];
            tmp0 = tmp1 + tmp2;
            tmp1 -= tmp2;
         }
         out2[ 11-1 ] = tmp0 * wi[ 11-1 ];
         out2[ 6 +1 ] = tmp0 * wi[ 6+1 ];
         out2[ 0+1 ] += tmp1 * wi[ 1 ];
         out2[ 5-1 ] += tmp1 * wi[ 5-1 ];
      }
      DCT12_PART2
      out2[ 11-0 ] = in2 * wi[ 11-0 ];
      out2[ 6 +0 ] = in2 * wi[ 6+0 ];
      out2[ 6 +2 ] = in3 * wi[ 6+2 ];
      out2[ 11-2 ] = in3 * wi[ 11-2 ];
      out2[ 0+0 ] += in0 * wi[ 0 ];
      out2[ 5-0 ] += in0 * wi[ 5-0 ];
      out2[ 0+2 ] += in4 * wi[ 2 ];
      out2[ 5-2 ] += in4 * wi[ 5-2 ];
   }
   return 0;
}
static inline mpeg_complex_t cmplx_mult( mpeg_complex_t a, mpeg_complex_t b )
{
   mpeg_complex_t ret;
   ret.real = a.real * b.real - a.imag * b.imag;
   ret.imag = a.real * b.imag + a.imag * b.real;
   return ret;
}
static inline int mpeg_audio_imdct_init( mpeg_audio_t *audio )
{
   int i, k;
   mpeg_complex_t angle_step;
   mpeg_complex_t current_angle;
   for( i = 0; i < AC3_N / 4; i++ )
   {
      mpeg_xcos1[ i ] = -cos( 2.0f * M_PI * ( 8 * i + 1 ) / ( 8 * AC3_N ) ); 
      mpeg_xsin1[ i ] = -sin( 2.0f * M_PI * ( 8 * i + 1 ) / ( 8 * AC3_N ) );
   }
   for( i = 0; i < AC3_N / 8; i++ )
   {
      mpeg_xcos2[ i ] = -cos( 2.0f * M_PI * ( 8 * i + 1 ) / ( 4 * AC3_N ) ); 
      mpeg_xsin2[ i ] = -sin( 2.0f * M_PI * ( 8 * i + 1 ) / ( 4 * AC3_N ) );
   }
   audio->ac3_w[ 0 ] = audio->ac3_w_1;
   audio->ac3_w[ 1 ] = audio->ac3_w_2;
   audio->ac3_w[ 2 ] = audio->ac3_w_4;
   audio->ac3_w[ 3 ] = audio->ac3_w_8;
   audio->ac3_w[ 4 ] = audio->ac3_w_16;
   audio->ac3_w[ 5 ] = audio->ac3_w_32;
   audio->ac3_w[ 6 ] = audio->ac3_w_64;
   for( i = 0; i < 7; i++ )
   {
      angle_step.real = cos( -2.0f * M_PI / ( 1 << ( i + 1 ) ) );
      angle_step.imag = sin( -2.0f * M_PI / ( 1 << ( i + 1 ) ) );
      current_angle.real = 1.0f;
      current_angle.imag = 0.0f;
      for ( k = 0; k < 1 << i; k++ )
      {
         audio->ac3_w[ i ][ k ] = current_angle;
         current_angle = cmplx_mult( current_angle, angle_step );
      }
   }
   //
   return 0;
}
/***************************************************************************/
static inline void swap_cmplx( mpeg_complex_t *a, mpeg_complex_t *b )
{
   mpeg_complex_t tmp;
   //
   tmp = *a;
   *a = *b;
   *b = tmp;
}
static inline void mpeg_audio_ac3_imdct_do_512( mpeg_audio_t *audio, float data[], float *y, int step, float *delay )
{
   int i, k;
   int p, q;
   int m;
   int two_m;
   int two_m_plus_one;
   float tmp_a_i;
   float tmp_a_r;
   float tmp_b_i;
   float tmp_b_r;
   float *y_ptr;
   float *delay_ptr;
   float *window_ptr;
   mpeg_complex_t *buf = audio->ac3_imdct_buf;
   //
   for( i = 0; i < AC3_N / 4; i++ )
   {
      buf[ i ].real = ( data[ AC3_N / 2 - 2 * i - 1 ] * mpeg_xcos1[ i ] ) - ( data[ 2 * i ] * mpeg_xsin1[ i ] );
      buf[ i ].imag = -( ( data[ 2 * i ] * mpeg_xcos1[ i ] ) + ( data[ AC3_N / 2 - 2 * i - 1 ] * mpeg_xsin1[ i ] ) );
   }
   for( i = 0; i < AC3_N / 4; i++ )
   {
      k = mpeg_bit_reverse_512[ i ];
      if ( k < i )
         swap_cmplx( &buf[ i ], &buf[ k ] );
   }
   for( m = 0; m < 7; m++ )
   {
      if ( m )
         two_m = ( 1 << m );
      else
         two_m = 1;
      two_m_plus_one = ( 1 << ( m + 1 ) );
      for( k = 0; k < two_m; k++ )
      {
         for( i = 0; i < AC3_N / 4; i += two_m_plus_one )
         {
            p = k + i;
            q = p + two_m;
            tmp_a_r = buf[ p ].real;
            tmp_a_i = buf[ p ].imag;
            tmp_b_r = buf[ q ].real * audio->ac3_w[ m ][ k ].real - buf[ q ].imag * audio->ac3_w[ m ][ k ].imag;
            tmp_b_i = buf[ q ].imag * audio->ac3_w[ m ][ k ].real + buf[ q ].real * audio->ac3_w[ m ][ k ].imag;
            buf[ p ].real = tmp_a_r + tmp_b_r;
            buf[ p ].imag = tmp_a_i + tmp_b_i;
            buf[ q ].real = tmp_a_r - tmp_b_r;
            buf[ q ].imag = tmp_a_i - tmp_b_i;
         }
      }
   }
   for( i = 0; i < AC3_N / 4; i++ )
   {
      tmp_a_r = buf[ i ].real;
      tmp_a_i = -buf[ i ].imag;
      buf[ i ].real = ( tmp_a_r * mpeg_xcos1[ i ] ) - ( tmp_a_i * mpeg_xsin1[ i ] );
      buf[ i ].imag = ( tmp_a_r * mpeg_xsin1[ i ] ) + ( tmp_a_i * mpeg_xcos1[ i ] );
   }
   y_ptr = y;
   delay_ptr = delay;
   window_ptr = mpeg_window;
   for( i = 0; i < AC3_N / 8; i++ ) 
   {
      *y_ptr = -buf[ AC3_N / 8 + i ].imag * *window_ptr++ + *delay_ptr++;
      y_ptr += step;
      *y_ptr = buf[ AC3_N / 8 - i - 1 ].real * *window_ptr++ + *delay_ptr++;
      y_ptr += step;
   }
   for( i = 0; i < AC3_N / 8; i++ ) 
   {
      *y_ptr = -buf[ i ].real * *window_ptr++ + *delay_ptr++;
      y_ptr += step;
      *y_ptr = buf[ AC3_N / 4 - i - 1 ].imag * *window_ptr++ + *delay_ptr++;
      y_ptr += step;
   }
   delay_ptr = delay;
   for( i = 0; i < AC3_N / 8; i++ )
   {
      *delay_ptr++ = -buf[ AC3_N / 8 + i ].real * *--window_ptr; 
      *delay_ptr++ = buf[ AC3_N / 8 - i - 1 ].imag * *--window_ptr; 
   }
   for( i = 0; i < AC3_N / 8; i++ ) 
   {
      *delay_ptr++ = buf[ i ].imag * *--window_ptr; 
      *delay_ptr++ = -buf[ AC3_N / 4 - i - 1 ].real * *--window_ptr; 
   }
}
static inline void mpeg_audio_ac3_imdct_do_256( mpeg_audio_t *audio, float data[], float *y, int step, float *delay )
{
   int i, k;
   int p, q;
   int m;
   int two_m;
   int two_m_plus_one;
   mpeg_complex_t *buf = audio->ac3_imdct_buf;
   float *y_ptr;
   float *delay_ptr;
   float *window_ptr;
   float tmp_a_i;
   float tmp_a_r;
   float tmp_b_i;
   float tmp_b_r;
   mpeg_complex_t *buf_1, *buf_2;
   //
   buf_1 = &buf[ 0 ];
   buf_2 = &buf[ 64 ];
   for( k = 0; k < AC3_N / 8; k++ )
   {
      p = 2 * ( AC3_N / 4 - 2 * k - 1 );
      q = 2 * ( 2 * k );
      buf_1[ k ].real = data[ p ] * mpeg_xcos2[ k ] - data[ q ] * mpeg_xsin2[ k ];
      buf_1[ k ].imag = - ( data[ q ] * mpeg_xcos2[ k ] + data[ p ] * mpeg_xsin2[ k ] ); 
      buf_2[ k ].real = data[ p + 1 ] * mpeg_xcos2[ k ] - data[ q + 1 ] * mpeg_xsin2[ k ];
      buf_2[ k ].imag = - ( data[ q + 1 ] * mpeg_xcos2[ k ] + data[ p + 1 ] * mpeg_xsin2[ k ] ); 
   }
   for( i = 0; i < AC3_N / 8; i++ )
   {
      k = mpeg_bit_reverse_256[ i ];
      if ( k < i )
      {
         swap_cmplx( &buf_1[ i ], &buf_1[ k ] );
         swap_cmplx( &buf_2[ i ], &buf_2[ k ] );
      }
   }
   for( m = 0; m < 6; m++ )
   {
      if ( m )
         two_m = ( 1 << m );
      else
         two_m = 1;
      two_m_plus_one = ( 1 << ( m + 1 ) );
      for( k = 0; k < two_m; k++ )
      {
         for( i = 0; i < AC3_N / 8; i += two_m_plus_one )
         {
            p = k + i;
            q = p + two_m;
            tmp_a_r = buf_1[ p ].real;
            tmp_a_i = buf_1[ p ].imag;
            tmp_b_r = buf_1[ q ].real * audio->ac3_w[ m ][ k ].real - buf_1[ q ].imag * audio->ac3_w[ m ][ k ].imag;
            tmp_b_i = buf_1[ q ].imag * audio->ac3_w[ m ][ k ].real + buf_1[ q ].real * audio->ac3_w[ m ][ k ].imag;
            buf_1[ p ].real = tmp_a_r + tmp_b_r;
            buf_1[ p ].imag = tmp_a_i + tmp_b_i;
            buf_1[ q ].real = tmp_a_r - tmp_b_r;
            buf_1[ q ].imag = tmp_a_i - tmp_b_i;
            tmp_a_r = buf_2[ p ].real;
            tmp_a_i = buf_2[ p ].imag;
            tmp_b_r = buf_2[ q ].real * audio->ac3_w[ m ][ k ].real - buf_2[ q ].imag * audio->ac3_w[ m ][ k ].imag;
            tmp_b_i = buf_2[ q ].imag * audio->ac3_w[ m ][ k ].real + buf_2[ q ].real * audio->ac3_w[ m ][ k ].imag;
            buf_2[ p ].real = tmp_a_r + tmp_b_r;
            buf_2[ p ].imag = tmp_a_i + tmp_b_i;
            buf_2[ q ].real = tmp_a_r - tmp_b_r;
            buf_2[ q ].imag = tmp_a_i - tmp_b_i;
         }
      }
   }
   for( i = 0; i < AC3_N / 8; i++ )
   {
      tmp_a_r = buf_1[ i ].real;
      tmp_a_i = -buf_1[ i ].imag;
      buf_1[ i ].real = ( tmp_a_r * mpeg_xcos2[ i ] ) - ( tmp_a_i * mpeg_xsin2[ i ] );
      buf_1[ i ].imag = ( tmp_a_r * mpeg_xsin2[ i ] ) + ( tmp_a_i * mpeg_xcos2[ i ] );
      tmp_a_r = buf_2[ i ].real;
      tmp_a_i = -buf_2[ i ].imag;
      buf_2[ i ].real = ( tmp_a_r * mpeg_xcos2[ i ] ) - ( tmp_a_i * mpeg_xsin2[ i ] );
      buf_2[ i ].imag = ( tmp_a_r * mpeg_xsin2[ i ] ) + ( tmp_a_i * mpeg_xcos2[ i ] );
   }
   y_ptr = y;
   delay_ptr = delay;
   window_ptr = mpeg_window;
   for( i = 0; i < AC3_N / 8; i++ ) 
   {
      *y_ptr = -buf[ AC3_N / 8 + i ].imag * *window_ptr++ + *delay_ptr++;
      y_ptr += step;
      *y_ptr = buf[ AC3_N / 8 - i - 1 ].real * *window_ptr++ + *delay_ptr++;
      y_ptr += step;
   }
   for( i = 0; i < AC3_N / 8; i++ ) 
   {
      *y_ptr = -buf[ i ].real * *window_ptr++ + *delay_ptr++;
      y_ptr += step;
      *y_ptr = buf[ AC3_N / 4 - i - 1 ].imag * *window_ptr++ + *delay_ptr++;
      y_ptr += step;
   }
   delay_ptr = delay;
   for( i = 0; i < AC3_N / 8; i++ )
   {
      *delay_ptr++ = -buf[ AC3_N / 8 + i ].real * *--window_ptr; 
      *delay_ptr++ = buf[ AC3_N / 8 - i - 1 ].imag * *--window_ptr; 
   }
   for( i = 0; i < AC3_N / 8; i++ ) 
   {
      *delay_ptr++ = buf[ i ].imag * *--window_ptr; 
      *delay_ptr++ = -buf[ AC3_N / 4 - i - 1 ].real * *--window_ptr; 
   }
}
/***************************************************************************/
static inline int mpeg_audio_head_check( unsigned long head )
{
   if ( ( head & 0xffe00000 ) != 0xffe00000 )
      return 1;
   if ( !( ( head >> 17 ) & 3 ) )
      return 1;
   if ( ( ( head >> 12 ) & 0xf ) == 0xf )
      return 1;
   if ( !( ( head >> 12 ) & 0xf ) )
      return 1;
   if ( ( ( head >> 10 ) & 0x3 ) == 0x3 )
      return 1;
   if ( ( ( head >> 19 ) & 1 ) == 1 && ( ( head >> 17 ) & 3 ) == 3 && ( ( head >> 16 ) & 1 ) == 1 )
      return 1;
   if ( ( head & 0xffff0000 ) == 0xfffe0000 )
      return 1;
   return 0;
}
static inline int mpeg_audio_decode_header( mpeg_audio_t *audio )
{
   if ( audio->newhead & ( 1 << 20 ) ) 
   {
      audio->lsf = ( audio->newhead & ( 1 << 19 ) ) ? 0x0 : 0x1;
      audio->mpeg_5 = 0;
   }
   else 
   {
      audio->lsf = 1;
      audio->mpeg_5 = 1;
   }
   audio->layer = 4 - ( ( audio->newhead >> 17 ) & 3 );
   if ( audio->mpeg_5 ) 
      audio->sampling_frequency_code = 6 + ( ( audio->newhead >> 10 ) & 0x3 );
   else
      audio->sampling_frequency_code = ( ( audio->newhead >> 10 ) & 0x3 ) + ( audio->lsf * 3 );
   audio->error_protection = ( ( audio->newhead >> 16 ) & 0x1 ) ^ 0x1;
   audio->bitrate_index = ( ( audio->newhead >> 12 ) & 0xf );
   audio->padding = ( ( audio->newhead >> 9 ) & 0x1 );
   audio->extension = ( ( audio->newhead >> 8 ) & 0x1 );
   audio->mode = ( ( audio->newhead >> 6 ) & 0x3 );
   audio->mode_ext = ( ( audio->newhead >> 4 ) & 0x3 );
   audio->copyright = ( ( audio->newhead >> 3 ) & 0x1 );
   audio->original = ( ( audio->newhead >> 2 ) & 0x1 );
   audio->emphasis = audio->newhead & 0x3;
   audio->channels = ( audio->mode == MPG_MD_MONO ) ? 1 : 2;
   if ( audio->channels > 1 ) 
      audio->single = -1;
   else
      audio->single = 3;
   audio->prev_framesize = audio->framesize;
   if ( !audio->bitrate_index )
      return 1;
   audio->bitrate = 1000 * mpeg_tabsel_123[ audio->lsf ][ audio->layer - 1 ][ audio->bitrate_index ];
   switch( audio->layer ) 
   {
      case 1:
         audio->framesize = ( long )mpeg_tabsel_123[ audio->lsf ][ 0 ][ audio->bitrate_index ] * 12000;
         audio->framesize /= mpeg_freqs[ audio->sampling_frequency_code ];
         audio->framesize = ( ( audio->framesize + audio->padding ) << 2 ) - 4;
         break;
      case 2:
         audio->framesize = ( long )mpeg_tabsel_123[ audio->lsf ][ 1 ][ audio->bitrate_index ] * 144000;
         audio->framesize /= mpeg_freqs[ audio->sampling_frequency_code ];
         audio->framesize += audio->padding - 4;
         break;
      case 3:
         if ( audio->lsf )
            audio->ssize = ( audio->channels == 1 ) ? 9 : 17;
         else
            audio->ssize = ( audio->channels == 1 ) ? 17 : 32;
         if ( audio->error_protection )
            audio->ssize += 2;
         audio->framesize = ( long )mpeg_tabsel_123[ audio->lsf ][ 2 ][ audio->bitrate_index ] * 144000;
         audio->framesize /= mpeg_freqs[ audio->sampling_frequency_code ] << ( audio->lsf );
         audio->framesize = audio->framesize + audio->padding - 4;
         break; 
         default:
      return 1;
}
if ( audio->framesize > MAXFRAMESIZE )
   return 1;
//
return 0;
}
/***************************************************************************/
static inline int mpeg_audio_prev_header( mpeg_audio_t *audio )
{
   int result = 0, i, len = audio->avg_framesize;
   //
   for( i = 0; i < len && !result; i++ )
   {
      mpeg_bits_getbits_reverse( audio->astream, 8 );
   }
   result |= mpeg_bits_refill( audio->astream );
   //
   return result;
}
/***************************************************************************/
static inline int mpeg_audio_read_header( mpeg_audio_t *audio )
{
   int try2 = 0;
   int result = 0;
   //
   switch( audio->format )
   {
      case AUDIO_AC3:
         result = 1;//mpeg_audio_read_ac3_header( audio );
         break;
      case AUDIO_MPEG:
         if ( audio->layer == 1 )
         {
		DebugMessage( "MpegPlayer : mpeg_audio_new: layer 1 not supported\n" );
            result = 1;
         }
         audio->newhead = mpeg_bits_showbits( audio->astream, 32 );
         if ( !mpeg_bits_eof( audio->astream ) && ( mpeg_audio_head_check( audio->newhead ) || mpeg_audio_decode_header( audio ) ) )
         {
            do
            {
               try2++;
               mpeg_bits_getbyte_noptr( audio->astream );
               audio->newhead = mpeg_bits_showbits( audio->astream, 32 );
		}while( !mpeg_bits_eof( audio->astream ) && try2 < 65536 && ( mpeg_audio_head_check( audio->newhead ) || mpeg_audio_decode_header( audio ) ) );
         }
         mpeg_bits_getbits( audio->astream, 32 );
         break;
      case AUDIO_PCM:
         mpeg_audio_read_pcm_header( audio );
         break;
         default:
      break;
   }
   //
   return mpeg_bits_eof( audio->astream );
}
/***************************************************************************/
static inline int mpeg_audio_dolayer1( mpeg_audio_t *audio )
{
   return 1; // No.
}
/***************************************************************************/
static inline int mpeg_audio_II_select_table( mpeg_audio_t *audio )
{
   static int translate[ 3 ][ 2 ][ 16 ] =
   {{{0, 2, 2, 2, 2, 2, 2, 0, 0, 0, 1, 1, 1, 1, 1, 0}, 
   {0, 2, 2, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0}}, 
   {{0, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0}, 
   {0, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}}, 
   {{0, 3, 3, 3, 3, 3, 3, 0, 0, 0, 1, 1, 1, 1, 1, 0}, 
   {0, 3, 3, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0}}};
   int table, sblim;
   static struct al_table *tables[ 5 ] =
   {alloc_0, alloc_1, alloc_2, alloc_3, alloc_4};
   static int sblims[ 5 ] = {27, 30, 8, 12, 30};
   //
   if ( audio->lsf ) 
      table = 4;
   else
      table = translate[ audio->sampling_frequency_code ][ 2 - audio->channels ][ audio->bitrate_index ];
   sblim = sblims[ table ];
   audio->alloc = tables[ table ];
   audio->II_sblimit = sblim;
   return 0;
}
/***************************************************************************/
static inline int mpeg_audio_II_step_one( mpeg_audio_t *audio, unsigned int *bit_alloc, int *scale )
{
   int stereo = audio->channels - 1;
   int sblimit = audio->II_sblimit;
   int jsbound = audio->jsbound;
   int sblimit2 = audio->II_sblimit << stereo;
   struct al_table *alloc1 = audio->alloc;
   int i, result = 0;
   unsigned int *scfsi_buf = audio->layer2_scfsi_buf;
   unsigned int *scfsi, *bita;
   int sc, step;
   //
   bita = bit_alloc;
   if ( stereo )
   {
      for( i = jsbound;i ; i--, alloc1 += ( 1 << step ) )
      {
         *bita++ = ( char )mpeg_bits_getbits( audio->astream, step = alloc1->bits );
         *bita++ = ( char )mpeg_bits_getbits( audio->astream, step );
      }
      for( i = sblimit-jsbound; i; i--, alloc1 += ( 1 << step ) )
      {
         bita[ 0 ] = ( char )mpeg_bits_getbits( audio->astream, step = alloc1->bits );
         bita[ 1 ] = bita[ 0 ];
         bita += 2;
      }
      bita = bit_alloc;
      scfsi = scfsi_buf;
      for( i = sblimit2; i; i-- )
      {
         if ( *bita++ )
            *scfsi++ = ( char )mpeg_bits_getbits( audio->astream, 2 );
      }
   }
   else 
   {
      for( i = sblimit; i; i--, alloc1 += ( 1 << step ) )
         *bita++ = ( char )mpeg_bits_getbits( audio->astream, step = alloc1->bits );
      bita = bit_alloc;
      scfsi = scfsi_buf;
      for( i = sblimit; i; i-- )
      {
         if ( *bita++ )
            *scfsi++ = ( char )mpeg_bits_getbits( audio->astream, 2 );
      }
   }
   bita = bit_alloc;
   scfsi = scfsi_buf;
   for( i = sblimit2; i; i-- ) 
   {
      if ( *bita++ )
      {
         switch( *scfsi++ ) 
         {
            case 0: 
               *scale++ = mpeg_bits_getbits( audio->astream, 6 );
               *scale++ = mpeg_bits_getbits( audio->astream, 6 );
               *scale++ = mpeg_bits_getbits( audio->astream, 6 );
               break;
            case 1 : 
               *scale++ = sc = mpeg_bits_getbits( audio->astream, 6 );
               *scale++ = sc;
               *scale++ = mpeg_bits_getbits( audio->astream, 6 );
               break;
            case 2: 
               *scale++ = sc = mpeg_bits_getbits( audio->astream, 6 );
               *scale++ = sc;
               *scale++ = sc;
               break;
               default: 
            *scale++ = mpeg_bits_getbits( audio->astream, 6 );
            *scale++ = sc = mpeg_bits_getbits( audio->astream, 6 );
            *scale++ = sc;
            break;
         }
      }
   }
   //
   return result | mpeg_bits_error( audio->astream );
}
/***************************************************************************/
static inline int mpeg_audio_II_step_two( mpeg_audio_t *audio, unsigned int *bit_alloc, float fraction[ 2 ][ 4 ][ SBLIMIT ], int *scale, int x1 )
{
   int i, j, k, ba, result = 0;
   int channels = audio->channels;
   int sblimit = audio->II_sblimit;
   int jsbound = audio->jsbound;
   struct al_table *alloc2, *alloc1 = audio->alloc;
   unsigned int *bita = bit_alloc;
   int d1, step;
   //
   for( i = 0; i < jsbound; i++, alloc1 += ( 1 << step ) )
   {
      step = alloc1->bits;
      for( j = 0; j < channels; j++ )
      {
         if ( ( ba = *bita++ ) )
         {
            k = ( alloc2 = alloc1 + ba )->bits;
            if ( ( d1 = alloc2->d ) < 0 ) 
            {
               float cm = mpeg_muls[ k ][ scale[ x1 ] ];
               fraction[ j ][ 0 ][ i ] = ( ( float )( ( int )mpeg_bits_getbits( audio->astream, k ) + d1 ) ) * cm;
               fraction[ j ][ 1 ][ i ] = ( ( float )( ( int )mpeg_bits_getbits( audio->astream, k ) + d1 ) ) * cm;
               fraction[ j ][ 2 ][ i ] = ( ( float )( ( int )mpeg_bits_getbits( audio->astream, k ) + d1 ) ) * cm;
            }
            else 
            {
               static int *table[] = 
               {0, 0, 0, mpeg_grp_3tab, 0, mpeg_grp_5tab, 0, 0, 0, mpeg_grp_9tab};
               unsigned int idx, *tab, m = scale[ x1 ];
               idx = ( unsigned int )mpeg_bits_getbits( audio->astream, k );
               tab = ( unsigned int* )( table[ d1 ] + idx + idx + idx );
               fraction[ j ][ 0 ][ i ] = mpeg_muls[ *tab++ ][ m ];
               fraction[ j ][ 1 ][ i ] = mpeg_muls[ *tab++ ][ m ];
               fraction[ j ][ 2 ][ i ] = mpeg_muls[ *tab ][ m ]; 
            }
            scale += 3;
         }
         else
            fraction[ j ][ 0 ][ i ] = fraction[ j ][ 1 ][ i ] = fraction[ j ][ 2 ][ i ] = 0.0;
      }
   }
   for( i = jsbound; i < sblimit; i++, alloc1 += ( 1 << step ) )
   {
      step = alloc1->bits;
      bita++;	
      if ( ( ba = *bita++ ) )
      {
         k=( alloc2 = alloc1+ba )->bits;
         if ( ( d1 = alloc2->d ) < 0 )
         {
            float cm;
            cm = mpeg_muls[ k ][ scale[ x1 + 3 ] ];
            fraction[ 1 ][ 0 ][ i ] = ( fraction[ 0 ][ 0 ][ i ] = ( float )( ( int )mpeg_bits_getbits( audio->astream, k ) + d1 ) ) * cm;
            fraction[ 1 ][ 1 ][ i ] = ( fraction[ 0 ][ 1 ][ i ] = ( float )( ( int )mpeg_bits_getbits( audio->astream, k ) + d1 ) ) * cm;
            fraction[ 1 ][ 2 ][ i ] = ( fraction[ 0 ][ 2 ][ i ] = ( float )( ( int )mpeg_bits_getbits( audio->astream, k ) + d1 ) ) * cm;
            cm = mpeg_muls[ k ][ scale[ x1 ] ];
            fraction[ 0 ][ 0 ][ i ] *= cm; 
            fraction[ 0 ][ 1 ][ i ] *= cm; 
            fraction[ 0 ][ 2 ][ i ] *= cm;
         }
         else
         {
            static int *table[] = {0, 0, 0, mpeg_grp_3tab, 0, mpeg_grp_5tab, 0, 0, 0, mpeg_grp_9tab};
            unsigned int idx, *tab, m1, m2;
            m1 = scale[ x1 ]; 
            m2 = scale[ x1+3 ];
            idx = ( unsigned int )mpeg_bits_getbits( audio->astream, k );
            tab = ( unsigned int* )( table[ d1 ] + idx + idx + idx );
            fraction[ 0 ][ 0 ][ i ] = mpeg_muls[ *tab ][ m1 ]; 
            fraction[ 1 ][ 0 ][ i ] = mpeg_muls[ *tab++ ][ m2 ];
            fraction[ 0 ][ 1 ][ i ] = mpeg_muls[ *tab ][ m1 ]; 
            fraction[ 1 ][ 1 ][ i ] = mpeg_muls[ *tab++ ][ m2 ];
            fraction[ 0 ][ 2 ][ i ] = mpeg_muls[ *tab ][ m1 ]; 
            fraction[ 1 ][ 2 ][ i ] = mpeg_muls[ *tab ][ m2 ];
         }
         scale += 6;
      }
      else 
      {
         fraction[ 0 ][ 0 ][ i ] = fraction[ 0 ][ 1 ][ i ] = fraction[ 0 ][ 2 ][ i ] =
         fraction[ 1 ][ 0 ][ i ] = fraction[ 1 ][ 1 ][ i ] = fraction[ 1 ][ 2 ][ i ] = 0.0;
      }
   }
   if ( sblimit > SBLIMIT )
      sblimit = SBLIMIT;
   for( i = sblimit; i < SBLIMIT; i++ )
   {
      for( j = 0; j < channels; j++ )
         fraction[ j ][ 0 ][ i ] = fraction[ j ][ 1 ][ i ] = fraction[ j ][ 2 ][ i ] = 0.0;
   }
   //
   return result | mpeg_bits_error( audio->astream );
}
/***************************************************************************/
static inline int mpeg_audio_dolayer2( mpeg_audio_t *audio )
{
   int i, j, result = 0;
   int channels = audio->channels;
   float fraction[ 2 ][ 4 ][ SBLIMIT ]; 
   unsigned int bit_alloc[ 64 ];
   int scale[ 192 ];
   int single = audio->single;
   //
   if ( audio->error_protection )
      mpeg_bits_getbits( audio->astream, 16 );
   mpeg_audio_II_select_table( audio );
   audio->jsbound = ( audio->mode == MPG_MD_JOINT_STEREO ) ? ( audio->mode_ext << 2 ) + 4 : audio->II_sblimit;
   if ( channels == 1 || single == 3 )
      single = 0;
   result |= mpeg_audio_II_step_one( audio, bit_alloc, scale );
   for( i = 0; i < SCALE_BLOCK && !result; i++ )
   {
      result |= mpeg_audio_II_step_two( audio, bit_alloc, fraction, scale, i >> 2 );
      for( j = 0; j < 3; j++ ) 
      {
         if ( single >= 0 )
         {
            mpeg_audio_synth_mono( audio, fraction[ single ][ j ], audio->pcm_sample, &( audio->pcm_point ) );
         }
         else 
         {
            int p1 = audio->pcm_point;
            mpeg_audio_synth_stereo( audio, fraction[ 0 ][ j ], 0, audio->pcm_sample, &p1 );
            mpeg_audio_synth_stereo( audio, fraction[ 1 ][ j ], 1, audio->pcm_sample, &( audio->pcm_point ) );
         }
         if ( audio->pcm_point / audio->channels >= audio->pcm_allocated - MPEG3AUDIO_PADDING * audio->channels )
         {
            mpeg_audio_replace_buffer( audio, audio->pcm_allocated + MPEG3AUDIO_PADDING * audio->channels );
         }
      }
   }
   //
   return result;
}
/***************************************************************************/
static inline int mpeg_audio_III_get_scale_factors_1( mpeg_audio_t *audio, int *scf, struct gr_info_s *gr_info, int ch, int gr )
{
   static unsigned char slen[ 2 ][ 16 ] = 
   {{0, 0, 0, 0, 3, 1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4}, 
   {0, 1, 2, 3, 0, 1, 2, 3, 1, 2, 3, 1, 2, 3, 2, 3}};
   int numbits;
   int num0 = slen[ 0 ][ gr_info->scalefac_compress ];
   int num1 = slen[ 1 ][ gr_info->scalefac_compress ];
   //
   if ( gr_info->block_type == 2 ) 
   {
      int i = 18;
      numbits = ( num0 + num1 ) * 18;
      if ( gr_info->mixed_block_flag ) 
      {
         for( i = 8; i; i-- )
            *scf++ = mpeg_bits_getbits( audio->astream, num0 );
         i = 9;
         numbits -= num0; 
      }
      for( ; i; i-- )
         *scf++ = mpeg_bits_getbits( audio->astream, num0 );
      for( i = 18; i; i-- )
         *scf++ = mpeg_bits_getbits( audio->astream, num1 );
      *scf++ = 0; 
      *scf++ = 0; 
      *scf++ = 0; 
   }
   else 
   {
      int i;
      int scfsi = gr_info->scfsi;
      if ( scfsi < 0 )
      {
         for( i = 11; i; i-- )
         {
            *scf++ = mpeg_bits_getbits( audio->astream, num0 );
         }
         for( i = 10; i; i-- )
            *scf++ = mpeg_bits_getbits( audio->astream, num1 );
         numbits = ( num0 + num1 ) * 10 + num0;
         *scf++ = 0;
      }
      else 
      {
         numbits = 0;
         if ( !( scfsi & 0x8 ) ) 
         {
            for( i = 0; i < 6; i++ )
            {
               *scf++ = mpeg_bits_getbits( audio->astream, num0 );
            }
            numbits += num0 * 6;
         }
         else 
         {
            scf += 6; 
         }
         if ( !( scfsi & 0x4 ) ) 
         {
            for( i = 0; i < 5; i++ )
               *scf++ = mpeg_bits_getbits( audio->astream, num0 );
            numbits += num0 * 5;
         }
         else 
         {
            scf += 5;
         }
         if ( !( scfsi & 0x2 ) ) 
         {
            for( i = 0; i < 5; i++ )
               *scf++ = mpeg_bits_getbits( audio->astream, num1 );
            numbits += num1 * 5;
         }
         else 
         {
            scf += 5; 
         }
         if ( !( scfsi & 0x1 ) ) 
         {
            for( i = 0; i < 5; i++ )
               *scf++ = mpeg_bits_getbits( audio->astream, num1 );
            numbits += num1 * 5;
         }
         else 
         {
            scf += 5;
         }
         *scf++ = 0; 
      }
   }
   return numbits;
}
/***************************************************************************/
static inline int mpeg_audio_III_get_scale_factors_2( mpeg_audio_t *audio, int *scf, struct gr_info_s *gr_info, int i_stereo )
{
   unsigned char *pnt;
   int i, j, n = 0, numbits = 0;
   unsigned int slen;
   static unsigned char stab[ 3 ][ 6 ][ 4 ] = 
   {{{6, 5, 5, 5 }, {6, 5, 7, 3 }, {11, 10, 0, 0}, 
   {7, 7, 7, 0 }, {6, 6, 6, 3 }, {8, 8, 5, 0}}, 
   {{9, 9, 9, 9 }, {9, 9, 12, 6 }, {18, 18, 0, 0}, 
   {12, 12, 12, 0 }, {12, 9, 9, 6 }, {15, 12, 9, 0}}, 
   {{6, 9, 9, 9 }, {6, 9, 12, 6 }, {15, 18, 0, 0}, 
   {6, 15, 12, 0 }, {6, 12, 9, 6 }, {6, 18, 9, 0}}};
   //
   if ( i_stereo ) 
      slen = mpeg_i_slen2[ gr_info->scalefac_compress >> 1 ];
   else
      slen = mpeg_n_slen2[ gr_info->scalefac_compress ];
   gr_info->preflag = ( slen >> 15 ) & 0x1;
   n = 0; 
   if ( gr_info->block_type == 2 ) 
   {
      n++;
      if ( gr_info->mixed_block_flag )
         n++;
   }
   pnt = stab[ n ][ ( slen >> 12 ) & 0x7 ];
   for( i = 0; i < 4; i++ )
   {
      int num = slen & 0x7;
      slen >>= 3;
      if ( num ) 
      {
         for( j = 0; j < ( int )( pnt[ i ] ); j++ )
            *scf++ = mpeg_bits_getbits( audio->astream, num );
         numbits += pnt[ i ] * num;
      }
      else 
      {
         for( j = 0; j < ( int )( pnt[ i ] ); j++ )
            *scf++ = 0;
      }
   }
   n = ( n << 1 ) + 1;
   for( i = 0; i < n; i++ )
      *scf++ = 0;
   return numbits;
}
/***************************************************************************/
static inline int mpeg_audio_III_dequantize_sample( mpeg_audio_t *audio, float xr[ SBLIMIT ][ SSLIMIT ], int *scf, struct gr_info_s *gr_info, int sfreq, int part2bits )
{
   int shift = 1 + gr_info->scalefac_scale;
   float *xrpnt = ( float* )xr;
   int l[ 3 ], l3;
   int part2remain = gr_info->part2_3_length - part2bits;
   int *me;
   int num = mpeg_bits_getbitoffset( audio->astream );
   long mask = mpeg_bits_getbits( audio->astream, num );
   //
   mask = mask << ( BITSHIFT + 8 - num );
   part2remain -= num;
   {
      int bv = gr_info->big_values;
      int region1 = gr_info->region1start;
      int region2 = gr_info->region2start;
      l3 = ( ( 576 >> 1 ) - bv ) >> 1; 
      if ( bv <= region1 ) 
      {
         l[ 0 ] = bv; 
         l[ 1 ] = 0; 
         l[ 2 ] = 0;
      }
      else 
      {
         l[ 0 ] = region1;
         if ( bv <= region2 ) 
         {
            l[ 1 ] = bv - l[ 0 ]; l[ 2 ] = 0;
         }
         else 
         {
            l[ 1 ] = region2 - l[ 0 ]; 
            l[ 2 ] = bv - region2;
         }
      }
   }
   if ( gr_info->block_type == 2 ) 
   {
      int i, max[ 4 ];
      int step = 0, lwin = 3, cb = 0;
      float v = 0.0;
      int *m, mc;
      if ( gr_info->mixed_block_flag ) 
      {
         max[ 3 ] = -1;
         max[ 0 ] = max[ 1 ] = max[ 2 ] = 2;
         m = mpeg_map[ sfreq ][ 0 ];
         me = mpeg_mapend[ sfreq ][ 0 ];
      }
      else 
      {
         max[ 0 ] = max[ 1 ] = max[ 2 ] = max[ 3 ] = -1;
         m = mpeg_map[ sfreq ][ 1 ];
         me = mpeg_mapend[ sfreq ][ 1 ];
      }
      mc = 0;
      for( i = 0; i < 2; i++ ) 
      {
         int lp = l[ i ];
         struct newhuff *h = mpeg_ht + gr_info->table_select[ i ];
         for( ; lp; lp--, mc-- ) 
         {
	    int x, y = 0;
	    //
            if ( !mc ) 
            {
               mc = *m++;
               xrpnt = ( ( float* )xr ) + ( *m++ );
               lwin = *m++;
               cb = *m++;
               if ( lwin == 3 ) 
               {
                  v = gr_info->pow2gain[ ( *scf++ ) << shift ];
                  step = 1;
               }
               else 
               {
                  v = gr_info->full_gain[ lwin ][ ( *scf++ ) << shift ];
                  step = 3;
               }
            }
            {
	       short *val = h->table;
               REFRESH_MASK;
		   while( ( y = *val++ ) < 0 )
               {
                  if ( mask < 0 )
                     val -= y;
                  num--;
                  mask <<= 1;
               }
               x = y >> 4;
               y &= 0xf;
            }
            if ( x == 15 && h->linbits ) 
            {
               max[ lwin ] = cb;
               REFRESH_MASK;
               x += ( ( unsigned long )mask ) >> ( BITSHIFT + 8 - h->linbits );
               num -= h->linbits + 1;
               mask <<= h->linbits;
               if ( mask < 0 )
                  *xrpnt = -mpeg_ispow[ x ] * v;
               else
                  *xrpnt = mpeg_ispow[ x ] * v;
               mask <<= 1;
            }
            else if ( x ) 
            {
               max[ lwin ] = cb;
               if ( mask < 0 )
                  *xrpnt = -mpeg_ispow[ x ] * v;
               else
                  *xrpnt = mpeg_ispow[ x ] * v;
               num--;
               mask <<= 1;
            }
            else
               *xrpnt = 0.0;
            xrpnt += step;
            if ( y == 15 && h->linbits ) 
            {
               max[ lwin ] = cb;
               REFRESH_MASK;
               y += ( ( unsigned long ) mask ) >> ( BITSHIFT + 8 - h->linbits );
               num -= h->linbits + 1;
               mask <<= h->linbits;
               if ( mask < 0 )
                  *xrpnt = -mpeg_ispow[ y ] * v;
               else
                  *xrpnt = mpeg_ispow[ y ] * v;
               mask <<= 1;
            }
            else if ( y ) 
            {
               max[ lwin ] = cb;
               if ( mask < 0 )
                  *xrpnt = -mpeg_ispow[ y ] * v;
               else
                  *xrpnt = mpeg_ispow[ y ] * v;
               num--;
               mask <<= 1;
            }
            else
               *xrpnt = 0.0;
            xrpnt += step;
         }
      }
      for( ;l3 && ( part2remain + num > 0 ); l3-- ) 
      {
         struct newhuff *h = mpeg_htc + gr_info->count1table_select;
	 short *val = h->table, a =0;
	 //
	 REFRESH_MASK;
	 while( ( a = *val++ ) < 0 )
         {
            if ( mask < 0 )
               val -= a;
            num--;
            mask <<= 1;
         }
         if ( part2remain + num <= 0 ) 
         {
            num -= part2remain + num;
            break;
         }
         for( i = 0; i < 4; i++ ) 
         {
            if ( !( i & 1 ) ) 
            {
               if ( !mc ) 
               {
                  mc = *m++;
                  xrpnt = ( ( float* )xr ) + ( *m++ );
                  lwin = *m++;
                  cb = *m++;
                  if ( lwin == 3 ) 
                  {
                     v = gr_info->pow2gain[ ( *scf++ ) << shift ];
                     step = 1;
                  }
                  else 
                  {
                     v = gr_info->full_gain[ lwin ][ ( *scf++ ) << shift ];
                     step = 3;
                  }
               }
               mc--;
            }
            if ( ( a & ( 0x8 >> i ) ) ) 
            {
               max[ lwin ] = cb;
               if ( part2remain + num <= 0 ) 
               {
                  break;
               }
               if ( mask < 0 ) 
                  *xrpnt = -v;
               else
                  *xrpnt = v;
               num--;
               mask <<= 1;
            }
            else
               *xrpnt = 0.0;
            xrpnt += step;
         }
      }
      if ( lwin < 3 ) 
      {
	 while( 1 )
         {
            for( ;mc > 0; mc-- ) 
            {
               *xrpnt = 0.0; 
               xrpnt += 3; 
               *xrpnt = 0.0; 
               xrpnt += 3;
            }
            if ( m >= me )
               break;
            mc = *m++;
            xrpnt = ( ( float* )xr ) + *m++;
            if ( *m++ == 0 )
               break; 
            m++; 
         }
      }
      gr_info->maxband[ 0 ] = max[ 0 ] + 1;
      gr_info->maxband[ 1 ] = max[ 1 ] + 1;
      gr_info->maxband[ 2 ] = max[ 2 ] + 1;
      gr_info->maxbandl = max[ 3 ] + 1;
      {
         int rmax = max[ 0 ] > max[ 1 ] ? max[ 0 ] : max[ 1 ];
         rmax = ( rmax > max[ 2 ] ? rmax : max[ 2 ] ) + 1;
         gr_info->maxb = rmax ? mpeg_shortLimit[ sfreq ][ rmax ] : mpeg_longLimit[ sfreq ][ max[ 3 ] + 1 ];
      }
   }
   else 
   {
      int *pretab = gr_info->preflag ? pretab1 : pretab2;
      int i, max = -1;
      int cb = 0;
      int *m = mpeg_map[ sfreq ][ 2 ];
      float v = 0.0;
      int mc = 0;
      for( i = 0; i < 3; i++ ) 
      {
	 int lp = l[ i ];
	 struct newhuff *h = mpeg_ht + gr_info->table_select[ i ];
	 for( ; lp; lp--, mc-- )
	 {
	    int x, y = 0;
	    //
	    if ( !mc )
	    {
	       mc = *m++;
	       cb = *m++;
	       if ( cb == 21 )
		  v = 0.0;
	       else
		  v = gr_info->pow2gain[ ( ( *scf++ ) + ( *pretab++ ) ) << shift ];
	    }
	    {
	       short *val = h->table;
	       REFRESH_MASK;
		   while( ( y = *val++ ) < 0 )
	       {
		  if ( mask < 0 )
		     val -= y;
		  num--;
		  mask <<= 1;
	       }
	       x = y >> 4;
               y &= 0xf;
            }
            if ( x == 15 && h->linbits ) 
            {
               max = cb;
               REFRESH_MASK;
               x += ( ( unsigned long ) mask ) >> ( BITSHIFT + 8 - h->linbits );
               num -= h->linbits + 1;
               mask <<= h->linbits;
               if ( mask < 0 )
                  *xrpnt++ = -mpeg_ispow[ x ] * v;
               else
                  *xrpnt++ = mpeg_ispow[ x ] * v;
               mask <<= 1;
            }
            else if ( x ) 
            {
               max = cb;
               if ( mask < 0 )
                  *xrpnt++ = -mpeg_ispow[ x ] * v;
               else
                  *xrpnt++ = mpeg_ispow[ x ] * v;
               num--;
               mask <<= 1;
            }
            else
               *xrpnt++ = 0.0;
            if ( y == 15 && h->linbits ) 
            {
               max = cb;
               REFRESH_MASK;
               y += ( ( unsigned long ) mask ) >> ( BITSHIFT + 8 - h->linbits );
               num -= h->linbits + 1;
               mask <<= h->linbits;
               if ( mask < 0 )
                  *xrpnt++ = -mpeg_ispow[ y ] * v;
               else
                  *xrpnt++ = mpeg_ispow[ y ] * v;
               mask <<= 1;
            }
            else if ( y ) 
            {
               max = cb;
               if ( mask < 0 )
                  *xrpnt++ = -mpeg_ispow[ y ] * v;
               else
                  *xrpnt++ = mpeg_ispow[ y ] * v;
               num--;
               mask <<= 1;
            }
            else
               *xrpnt++ = 0.0;
         }
      }
      for( ; l3 && ( part2remain + num > 0 ); l3-- ) 
      {
         struct newhuff *h = mpeg_htc + gr_info->count1table_select;
	 short *val = h->table, a =0;
	 //
         REFRESH_MASK;
	 while( ( a = *val++ ) < 0 )
         {
            if ( mask < 0 )
               val -= a;
            num--;
            mask <<= 1;
         }
         if ( part2remain + num <= 0 ) 
         {
            num -= part2remain + num;
            break;
         }
         for( i = 0; i < 4; i++ ) 
         {
            if ( !( i & 1 ) ) 
            {
               if ( !mc ) 
               {
                  mc = *m++;
                  cb = *m++;
                  if ( cb == 21 )
                     v = 0.0;
                  else
                     v = gr_info->pow2gain[ ( ( *scf++ ) + ( *pretab++ ) ) << shift ];
               }
               mc--;
            }
            if ( ( a & ( 0x8 >> i ) ) )
            {
               max = cb;
               if ( part2remain + num <= 0 ) 
               {
                  break;
               }
               if ( mask < 0 )
                  *xrpnt++ = -v;
               else
                  *xrpnt++ = v;
               num--;
               mask <<= 1;
            }
            else
               *xrpnt++ = 0.0;
         }
      }
      gr_info->maxbandl = max + 1;
      gr_info->maxb = mpeg_longLimit[ sfreq ][ gr_info->maxbandl ];
   }
   part2remain += num;
   mpeg_bits_start_reverse( audio->astream );
   mpeg_bits_getbits_reverse( audio->astream, num );
   mpeg_bits_start_forward( audio->astream );
   num = 0;
   while( xrpnt < &xr[ SBLIMIT ][ 0 ] )
      *xrpnt++ = 0.0;
   while( part2remain > 16 )
   {
      mpeg_bits_getbits( audio->astream, 16 ); 
      part2remain -= 16;
   }
   if ( part2remain > 0 )
   {
      mpeg_bits_getbits( audio->astream, part2remain );
   }
   else if ( part2remain < 0 ) 
   {
	  DebugMessage( "MpegPlayer : mpeg_audio_III_dequantize_sample: Can't rewind stream %d bits!\n", -part2remain);
      return 1; 
   }
   return 0;
}
/***************************************************************************/
static inline int mpeg_audio_III_get_side_info( mpeg_audio_t *audio, struct mpeg_III_sideinfo *si, int channels, int ms_stereo, long sfreq, int single, int lsf )
{
   int ch, gr;
   int powdiff = ( single == 3 ) ? 4 : 0;
   static const int tabs[ 2 ][ 5 ] = {{2, 9, 5, 3, 4 } , {1, 8, 1, 2, 9 } };
   const int *tab = tabs[ lsf ];
   //
   si->main_data_begin = mpeg_bits_getbits( audio->astream, tab[ 1 ] );
   if ( channels == 1 )
      si->private_bits = mpeg_bits_getbits( audio->astream, tab[ 2 ] );
   else 
      si->private_bits = mpeg_bits_getbits( audio->astream, tab[ 3 ] );
   if ( !lsf )
   {
      for( ch = 0; ch < channels; ch++ )
      {
         si->ch[ ch ].gr[ 0 ].scfsi = -1;
         si->ch[ ch ].gr[ 1 ].scfsi = mpeg_bits_getbits( audio->astream, 4 );
      }
   }
   for( gr = 0; gr < tab[ 0 ]; gr++ ) 
   {
      for( ch = 0; ch < channels; ch++ )
      {
	 struct gr_info_s *gr_info = &( si->ch[ ch ].gr[ gr ] );
         gr_info->part2_3_length = mpeg_bits_getbits( audio->astream, 12 );
         gr_info->big_values = mpeg_bits_getbits( audio->astream, 9 );
         if ( gr_info->big_values > 288 ) 
         {
		DebugMessage( "MpegPlayer : mpeg_III_get_side_info: big_values too large!\n" );
            gr_info->big_values = 288;
         }
         gr_info->pow2gain = mpeg_gainpow2 + 256 - mpeg_bits_getbits( audio->astream, 8 ) + powdiff;
         if ( ms_stereo )
            gr_info->pow2gain += 2;
         gr_info->scalefac_compress = mpeg_bits_getbits( audio->astream, tab[ 4 ] );
         if ( mpeg_bits_getbits( audio->astream, 1 ) ) 
         {
            int i;
            gr_info->block_type = mpeg_bits_getbits( audio->astream, 2 );
            gr_info->mixed_block_flag = mpeg_bits_getbits( audio->astream, 1 );
            gr_info->table_select[ 0 ] = mpeg_bits_getbits( audio->astream, 5 );
            gr_info->table_select[ 1 ] = mpeg_bits_getbits( audio->astream, 5 );
            gr_info->table_select[ 2 ] = 0;
            for( i = 0; i < 3; i++ )
               gr_info->full_gain[ i ] = gr_info->pow2gain + ( mpeg_bits_getbits( audio->astream, 3 ) << 3 );
            if ( gr_info->block_type == 0 ) 
            {
		   DebugMessage( "MpegPlayer : Blocktype == 0 and window-switching == 1 not allowed.\n" );
               return 1;
            }
            if ( !lsf || gr_info->block_type == 2 )
               gr_info->region1start = 36 >> 1;
            else 
            {
               if ( sfreq == 8 )
                  gr_info->region1start = 108 >> 1;
               else
                  gr_info->region1start = 54 >> 1;
            }
            gr_info->region2start = 576 >> 1;
         }
         else 
         {
            int i, r0c, r1c;
            for( i = 0; i < 3; i++ )
               gr_info->table_select[ i ] = mpeg_bits_getbits( audio->astream, 5 );
            r0c = mpeg_bits_getbits( audio->astream, 4 );
            r1c = mpeg_bits_getbits( audio->astream, 3 );
            gr_info->region1start = mpeg_bandInfo[ sfreq ].longIdx[ r0c + 1 ] >> 1 ;
            gr_info->region2start = mpeg_bandInfo[ sfreq ].longIdx[ r0c + 1 + r1c + 1 ] >> 1;
            gr_info->block_type = 0;
            gr_info->mixed_block_flag = 0;
         }
         if ( !lsf )
            gr_info->preflag = mpeg_bits_getbits( audio->astream, 1 );
         gr_info->scalefac_scale = mpeg_bits_getbits( audio->astream, 1 );
         gr_info->count1table_select = mpeg_bits_getbits( audio->astream, 1 );
      }
   }
   return 0;
}
/***************************************************************************/
static inline int mpeg_audio_III_hybrid( mpeg_audio_t *audio, float fsIn[ SBLIMIT ][ SSLIMIT ], float tsOut[ SSLIMIT ][ SBLIMIT ], int ch, struct gr_info_s *gr_info )
{
   float *tspnt = ( float * ) tsOut;
   float *rawout1, *rawout2;
   int bt, sb = 0;
   //
   {
      int b = audio->mp3_blc[ ch ];
      //
      rawout1 = audio->mp3_block[ b ][ ch ];
      b = -b + 1;
      rawout2 = audio->mp3_block[ b ][ ch ];
      audio->mp3_blc[ ch ] = b;
   }
   if ( gr_info->mixed_block_flag ) 
   {
      sb = 2;
      mpeg_audio_dct36( fsIn[ 0 ], rawout1, rawout2, mpeg_win[ 0 ], tspnt );
      mpeg_audio_dct36( fsIn[ 1 ], rawout1 + 18, rawout2 + 18, mpeg_win1[ 0 ], tspnt + 1 );
      rawout1 += 36; 
      rawout2 += 36; 
      tspnt += 2;
   }
   bt = gr_info->block_type;
   if ( bt == 2 )
   {
      for( ; sb < gr_info->maxb; sb += 2, tspnt += 2, rawout1 += 36, rawout2 += 36 ) 
      {
         mpeg_audio_dct12( fsIn[ sb ] , rawout1 , rawout2 , mpeg_win[ 2 ] , tspnt );
         mpeg_audio_dct12( fsIn[ sb + 1 ], rawout1 + 18, rawout2 + 18, mpeg_win1[ 2 ], tspnt + 1 );
      }
   }
   else 
   {
      for( ; sb < gr_info->maxb; sb += 2, tspnt += 2, rawout1 += 36, rawout2 += 36 ) 
      {
         mpeg_audio_dct36( fsIn[ sb ], rawout1, rawout2, mpeg_win[ bt ], tspnt );
         mpeg_audio_dct36( fsIn[ sb + 1 ], rawout1 + 18, rawout2 + 18, mpeg_win1[ bt ], tspnt + 1 );
      }
   }
   for( ; sb < SBLIMIT; sb++, tspnt++ ) 
   {
      int i;
      for( i = 0; i < SSLIMIT; i++ ) 
      {
         tspnt[ i * SBLIMIT ] = *rawout1++;
         *rawout2++ = 0.0;
      }
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_audio_III_antialias( mpeg_audio_t *audio, float xr[ SBLIMIT ][ SSLIMIT ], struct gr_info_s *gr_info )
{
   int sblim;
   //
   if ( gr_info->block_type == 2 ) 
   {
      if ( !gr_info->mixed_block_flag ) 
         return 1;
      sblim = 1; 
   }
   else 
   {
      sblim = gr_info->maxb-1;
   }
   {
      int sb;
      float *xr1 = ( float* )xr[ 1 ];
      //
      for( sb = sblim; sb; sb--, xr1 += 10 ) 
      {
         int ss;
         float *cs, *ca;
         float *xr2;
         //
         cs = mpeg_aa_cs;
         ca = mpeg_aa_ca;
         xr2 = xr1;
         for( ss = 7; ss >= 0; ss-- )
         {
	    float bu, bd;
            //
            bu = *--xr2;
            bd = *xr1;
            *xr2 = ( bu * ( *cs ) ) - ( bd * ( *ca ) );
            *xr1++ = ( bd * ( *cs++ ) ) + ( bu * ( *ca++ ) );
         }
      }
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_audio_III_i_stereo( mpeg_audio_t *audio, float xr_buf[ 2 ][ SBLIMIT ][ SSLIMIT ], int *scalefac, struct gr_info_s *gr_info, int sfreq, int ms_stereo, int lsf )
{
   float ( *xr )[ SBLIMIT*SSLIMIT ] = ( float ( * )[ SBLIMIT*SSLIMIT ] ) xr_buf;
   struct mpeg_bandInfoStruct *bi = &mpeg_bandInfo[ sfreq ];
   const float *tab1, *tab2;
   int tab;
   static const float *tabs[ 3 ][ 2 ][ 2 ] = 
   {
      {{mpeg_tan1_1, mpeg_tan2_1 } , {mpeg_tan1_2, mpeg_tan2_2 } }, 
      {{mpeg_pow1_1[ 0 ], mpeg_pow2_1[ 0 ] } , {mpeg_pow1_2[ 0 ], mpeg_pow2_2[ 0 ] } } , 
      {{mpeg_pow1_1[ 1 ], mpeg_pow2_1[ 1 ] } , {mpeg_pow1_2[ 1 ], mpeg_pow2_2[ 1 ] } } 
   };
   //
   tab = lsf + ( gr_info->scalefac_compress & lsf );
   tab1 = tabs[ tab ][ ms_stereo ][ 0 ];
   tab2 = tabs[ tab ][ ms_stereo ][ 1 ];
   if ( gr_info->block_type == 2 ) 
   {
      int lwin, do_l = 0;
      //
      if ( gr_info->mixed_block_flag )
         do_l = 1;
      for( lwin = 0; lwin < 3; lwin++ ) 
      {
         int is_p, sb, idx, sfb = gr_info->maxband[ lwin ]; 
         if ( sfb > 3 )
            do_l = 0;
         for( ; sfb < 12 ; sfb++ ) 
         {
            is_p = scalefac[ sfb * 3 + lwin - gr_info->mixed_block_flag ]; 
            if ( is_p != 7 ) 
            {
               float t1, t2;
               sb = bi->shortDiff[ sfb ];
               idx = bi->shortIdx[ sfb ] + lwin;
               t1 = tab1[ is_p ]; 
               t2 = tab2[ is_p ];
               for( ; sb > 0; sb--, idx += 3 ) 
               {
                  float v = xr[ 0 ][ idx ];
                  xr[ 0 ][ idx ] = v * t1;
                  xr[ 1 ][ idx ] = v * t2;
               }
            }
         }
         is_p = scalefac[ 11 * 3 + lwin - gr_info->mixed_block_flag ]; 
         sb = bi->shortDiff[ 12 ];
         idx = bi->shortIdx[ 12 ] + lwin;
         if ( is_p != 7 ) 
         {
            float t1, t2;
            t1 = tab1[ is_p ]; 
            t2 = tab2[ is_p ];
            for( ; sb > 0; sb--, idx += 3 ) 
            {
               float v = xr[ 0 ][ idx ];
               xr[ 0 ][ idx ] = v * t1;
               xr[ 1 ][ idx ] = v * t2;
            }
         }
      } 
      if ( do_l ) 
      {
         int sfb = gr_info->maxbandl;
         int idx = bi->longIdx[ sfb ];
         for ( ; sfb < 8; sfb++ ) 
         {
            int sb = bi->longDiff[ sfb ];
            int is_p = scalefac[ sfb ]; 
            if ( is_p != 7 ) 
            {
               float t1, t2;
               t1 = tab1[ is_p ]; 
               t2 = tab2[ is_p ];
               for( ; sb > 0; sb--, idx++ ) 
               {
                  float v = xr[ 0 ][ idx ];
                  xr[ 0 ][ idx ] = v * t1;
                  xr[ 1 ][ idx ] = v * t2;
               }
            }
            else 
               idx += sb;
         }
      } 
   } 
   else 
   {
      int sfb = gr_info->maxbandl;
      int is_p, idx = bi->longIdx[ sfb ];
      for( ; sfb < 21; sfb++ ) 
      {
         int sb = bi->longDiff[ sfb ];
         is_p = scalefac[ sfb ]; 
         if ( is_p != 7 ) 
         {
            float t1, t2;
            t1 = tab1[ is_p ]; 
            t2 = tab2[ is_p ];
            for( ; sb > 0; sb--, idx++ ) 
            {
               float v = xr[ 0 ][ idx ];
               xr[ 0 ][ idx ] = v * t1;
               xr[ 1 ][ idx ] = v * t2;
            }
         }
         else
            idx += sb;
      }
      is_p = scalefac[ 20 ];
      if ( is_p != 7 ) 
      {
         int sb;
         float t1 = tab1[ is_p ], t2 = tab2[ is_p ]; 
         for( sb = bi->longDiff[ 21 ]; sb > 0; sb--, idx++ )
         {
            float v = xr[ 0 ][ idx ];
            xr[ 0 ][ idx ] = v * t1;
            xr[ 1 ][ idx ] = v * t2;
         }
      }
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_audio_read_layer3_frame( mpeg_audio_t *audio )
{
   int result = mpeg_audio_read_header( audio );
   //
   if ( !result )
   {
      audio->bsbufold = audio->bsbuf;
      audio->bsbuf = audio->bsspace[ audio->bsnum ] + 512;
      audio->bsnum ^= 1;
      result = mpeg_bits_read_buffer( audio->astream, audio->bsbuf, audio->framesize );
   }
   //
   return result;
}
/***************************************************************************/
static inline int mpeg_audio_dolayer3( mpeg_audio_t *audio )
{
   int gr, ch, ss;
   int scalefacs[ 2 ][ 39 ]; 
   struct mpeg_III_sideinfo sideinfo;
   int channels = audio->channels;
   int single = audio->single;
   int ms_stereo, i_stereo;
   int sfreq = audio->sampling_frequency_code;
   int stereo1, granules;
   //
   audio->bsbufold = audio->bsbuf;
   audio->bsbuf = audio->bsspace[ audio->bsnum ] + 512;
   audio->bsnum ^= 1;
   if ( mpeg_bits_read_buffer( audio->astream, audio->bsbuf, audio->framesize ) )
      return 1;
   mpeg_bits_use_ptr( audio->astream, audio->bsbuf );
   if ( audio->error_protection )
      mpeg_bits_getbits( audio->astream, 16 );
   if ( channels == 1 )
   {
      stereo1 = 1;
      single = 0;
   }
   else
   {
      stereo1 = 2;
   }
   if ( audio->mode == MPG_MD_JOINT_STEREO )
   {
      ms_stereo = ( audio->mode_ext & 0x2 ) >> 1;
      i_stereo = audio->mode_ext & 0x1;
   }
   else
      ms_stereo = i_stereo = 0;
   if ( audio->lsf )
   {
      granules = 1;
   }
   else 
   {
      granules = 2;
   }
   if ( mpeg_audio_III_get_side_info( audio, &sideinfo, channels, ms_stereo, sfreq, single, audio->lsf ) )
      return 1;
   if ( sideinfo.main_data_begin >= 512 )
      return 1;
   if ( sideinfo.main_data_begin )
   {
      memcpy( audio->bsbuf + audio->ssize - sideinfo.main_data_begin, 
      audio->bsbufold + audio->prev_framesize - sideinfo.main_data_begin, 
      sideinfo.main_data_begin );
      mpeg_bits_use_ptr( audio->astream, audio->bsbuf + audio->ssize - sideinfo.main_data_begin );
   }
   for( gr = 0; gr < granules; gr++ )
   {
      float hybridIn [ 2 ][ SBLIMIT ][ SSLIMIT ];
      float hybridOut[ 2 ][ SSLIMIT ][ SBLIMIT ];
      {
         struct gr_info_s *gr_info = &( sideinfo.ch[ 0 ].gr[ gr ] );
         long part2bits;
         if ( audio->lsf )
            part2bits = mpeg_audio_III_get_scale_factors_2( audio, scalefacs[ 0 ], gr_info, 0 );
         else
            part2bits = mpeg_audio_III_get_scale_factors_1( audio, scalefacs[ 0 ], gr_info, 0, gr );
         if ( mpeg_audio_III_dequantize_sample( audio, hybridIn[ 0 ], scalefacs[ 0 ], gr_info, sfreq, part2bits ) )
         {
            mpeg_bits_use_demuxer( audio->astream );
            return 1;
         }
      }
      if ( channels == 2 ) 
      {
         struct gr_info_s *gr_info = &( sideinfo.ch[ 1 ].gr[ gr ] );
         long part2bits;
         if ( audio->lsf ) 
            part2bits = mpeg_audio_III_get_scale_factors_2( audio, scalefacs[ 1 ], gr_info, i_stereo );
         else
            part2bits = mpeg_audio_III_get_scale_factors_1( audio, scalefacs[ 1 ], gr_info, 1, gr );
         if ( mpeg_audio_III_dequantize_sample( audio, hybridIn[ 1 ], scalefacs[ 1 ], gr_info, sfreq, part2bits ) )
         {
            mpeg_bits_use_demuxer( audio->astream );
            return 1;
         }
         if ( ms_stereo )
         {
            int i;
            int maxb = sideinfo.ch[ 0 ].gr[ gr ].maxb;
            if ( sideinfo.ch[ 1 ].gr[ gr ].maxb > maxb )
               maxb = sideinfo.ch[ 1 ].gr[ gr ].maxb;
            for( i = 0; i < SSLIMIT * maxb; i++ )
            {
               float tmp0 = ( ( float* )hybridIn[ 0 ] )[ i ];
               float tmp1 = ( ( float* )hybridIn[ 1 ] )[ i ];
               ( ( float* )hybridIn[ 0 ] )[ i ] = tmp0 + tmp1;
               ( ( float* )hybridIn[ 1 ] )[ i ] = tmp0 - tmp1;
            }
         }
         if ( i_stereo )
            mpeg_audio_III_i_stereo( audio, hybridIn, scalefacs[ 1 ], gr_info, sfreq, ms_stereo, audio->lsf );
         if ( ms_stereo || i_stereo || ( single == 3 ) ) 
         {
            if ( gr_info->maxb > sideinfo.ch[ 0 ].gr[ gr ].maxb ) 
               sideinfo.ch[ 0 ].gr[ gr ].maxb = gr_info->maxb;
            else
               gr_info->maxb = sideinfo.ch[ 0 ].gr[ gr ].maxb;
         }
         switch( single ) 
         {
            case 3:
               {
		  int i;
		  float *in0 = ( float* )hybridIn[ 0 ], *in1 = ( float* )hybridIn[ 1 ];
                  for( i = 0; i < SSLIMIT * gr_info->maxb; i++, in0++ )
                     *in0 = ( *in0 + *in1++ ); 
               }
               break;
            case 1:
               {
		  int i;
		  float *in0 = ( float* )hybridIn[ 0 ], *in1 = ( float* )hybridIn[ 1 ];
                  //
                  for( i = 0; i < SSLIMIT * gr_info->maxb; i++ )
                     *in0++ = *in1++;
               }
               break;
            }
         }
         for( ch = 0; ch < stereo1; ch++ )
         {
            struct gr_info_s *gr_info = &( sideinfo.ch[ ch ].gr[ gr ] );
            //
            mpeg_audio_III_antialias( audio, hybridIn[ ch ], gr_info );
            mpeg_audio_III_hybrid( audio, hybridIn[ ch ], hybridOut[ ch ], ch, gr_info );
         }
         for( ss = 0; ss < SSLIMIT; ss++ )
         {
            if ( single >= 0 )
            {
               mpeg_audio_synth_mono( audio, hybridOut[ 0 ][ ss ], audio->pcm_sample, &( audio->pcm_point ) );
            }
            else 
            {
               int p1 = audio->pcm_point;
               mpeg_audio_synth_stereo( audio, hybridOut[ 0 ][ ss ], 0, audio->pcm_sample, &p1 );
               mpeg_audio_synth_stereo( audio, hybridOut[ 1 ][ ss ], 1, audio->pcm_sample, &( audio->pcm_point ) );
            }
            if ( audio->pcm_point / audio->channels >= audio->pcm_allocated - MPEG3AUDIO_PADDING * audio->channels )
            {
               mpeg_audio_replace_buffer( audio, audio->pcm_allocated + MPEG3AUDIO_PADDING * audio->channels );
            }
         }
      }
      mpeg_bits_use_demuxer( audio->astream );
      //
      return 0;
   }
   /***************************************************************************/
   static inline mpeg_audio_t* mpeg_audio_allocate_struct( MPEG *file_mpeg, mpeg_atrack_t *track )
   {
      mpeg_audio_t *audio = ( mpeg_audio_t * ) calloc( 1, sizeof( mpeg_audio_t ) );
      audio->file_mpeg = file_mpeg;
      audio->track = track;
      audio->astream = mpeg_bits_new_stream( file_mpeg, track->demuxer );
      audio->outscale = 1;
      audio->bsbuf = audio->bsspace[ 1 ];
      audio->init = 1;
      audio->bo = 1;
      audio->channels = 1;
      //
      return audio;
}
/***************************************************************************/
static inline int mpeg_audio_delete_struct( mpeg_audio_t *audio )
{
   mpeg_bits_delete_stream( audio->astream );
   if ( audio->pcm_sample )
      free( audio->pcm_sample );
   free( audio );
   return 0;
}
/***************************************************************************/
static inline int mpeg_audio_replace_buffer( mpeg_audio_t *audio, long new_allocation )
{
   long i;
   //
   if ( !audio->pcm_sample )
   {
      audio->pcm_sample = ( float * ) calloc( 1, sizeof( float ) * new_allocation * audio->channels );
      audio->pcm_allocated = new_allocation;
   }
   else
   {
      float *new_samples = ( float * ) calloc( 1, sizeof( float ) * new_allocation * audio->channels );
      for( i = 0; i < audio->pcm_allocated * audio->channels; i++ )
      {
         new_samples[ i ] = audio->pcm_sample[ i ];
      }
      free( audio->pcm_sample );
      audio->pcm_sample = new_samples;
      audio->pcm_allocated = new_allocation;
   }
   return 0;
}
/***************************************************************************/
static inline int mpeg_audio_read_frame( mpeg_audio_t *audio )
{
   int result = mpeg_audio_read_header( audio );
   //
   if ( !result )
   {
      switch( audio->format )
      {
         case AUDIO_AC3:
            result = 1; //mpeg_audio_do_ac3( audio );
            break;	
         case AUDIO_MPEG:
            switch( audio->layer )
            {
            case 1:
               result = mpeg_audio_dolayer1( audio );
               break;
            case 2:
               result = mpeg_audio_dolayer2( audio );
               break;
            case 3:
               result = mpeg_audio_dolayer3( audio );
               break;
            default:
               result = 1;
               break;
         }
         break;
         case AUDIO_PCM:
            result = mpeg_audio_do_pcm( audio );
            break;
      }
   }
   if ( !result )
   {
      mpeg_bits_byte_align( audio->astream );
   }
   //
   return result;
}
/***************************************************************************/
static inline int mpeg_audio_get_length( mpeg_audio_t *audio, mpeg_atrack_t *track )
{
   long result = 0;
   long framesize1 = 0, total1 = 0;
   long framesize2 = 0, total2 = 0;
   long total_framesize = 0, total_frames = 0;
   long byte_limit = 131072; 
   long total_bytes = 0;
   long major_framesize; 
   long minor_framesize; 
   long major_total;
   long minor_total;
   MPEG *file_mpeg = ( MPEG * ) audio->file_mpeg;
   //
   mpeg_bits_seek_start( audio->astream );
   audio->pcm_point = 0;
   result = mpeg_audio_read_frame( audio ); 
   audio->samples_per_frame = audio->pcm_point / audio->channels;
   switch( audio->format )
   {
      case AUDIO_AC3:
         audio->avg_framesize = audio->framesize;
         break;
      case AUDIO_MPEG:
         framesize1 = audio->framesize;
         total_bytes += audio->framesize;
         total1 = 1;
	 while( !result && total_bytes < byte_limit )
         {
            audio->pcm_point = 0;
            result = mpeg_audio_read_frame( audio );
            total_bytes += audio->framesize;
            if ( audio->framesize != framesize1 )
            {
               framesize2 = audio->framesize;
               total2 = 1;
               break;
         }
         else
         {
            total1++;
         }
      }
	  while( !result && total_bytes < byte_limit )
      {
         audio->pcm_point = 0;
         result = mpeg_audio_read_frame( audio );
         total_bytes += audio->framesize;
         if ( audio->framesize != framesize2 )
         {
            break;
         }
         else
         {
            total2++;
         }
      }
      audio->pcm_point = 0;
      result = mpeg_audio_read_frame( audio );
      if ( audio->framesize != framesize1 && audio->framesize != framesize2 )
      {
	 while( !result && total_bytes < byte_limit )
         {
            audio->pcm_point = 0;
            result = mpeg_audio_read_frame( audio );
            total_bytes += audio->framesize;
            if ( !result )
            {
               total_framesize += audio->framesize;
               total_frames++;
            }
         }
         audio->avg_framesize = 4 + ( float )( total_framesize + framesize1 + framesize2 ) / ( total_frames + total1 + total2 );
      }
      else
      {
         major_framesize = framesize2 > framesize1 ? framesize2 : framesize1;
         major_total = framesize2 > framesize1 ? total2 : total1;
         minor_framesize = framesize2 > framesize1 ? framesize1 : framesize2;
         minor_total = framesize2 > framesize1 ? total1 : total2;
         audio->avg_framesize = 4 + ( float )( major_framesize * major_total + minor_framesize * minor_total ) / ( major_total + minor_total );
      }
      break;
      case AUDIO_PCM:
         break;
   }
   if ( file_mpeg->is_audio_stream )
   {
      result = ( long )( ( float )mpeg_demuxer_total_bytes( audio->astream->demuxer ) / audio->avg_framesize * audio->samples_per_frame );
   }
   else
   {
      result = ( long )( mpeg_demux_length( audio->astream->demuxer ) * track->sample_rate );
   }
   audio->pcm_point = 0;
   mpeg_bits_seek_start( audio->astream );
   mpeg_audio_reset_synths( audio );
   //
   return result;
}
static inline int mpeg_audio_seek( mpeg_audio_t *audio, long position )
{
   int result = 0;
   MPEG *file_mpeg = ( MPEG * ) audio->file_mpeg;
   mpeg_atrack_t *track = ( mpeg_atrack_t * ) audio->track;
   long frame_number;
   long byte_position = 0;
   double time_position;
   //
   if ( audio->sample_seek < 0 )
   {
      audio->pcm_position = position;
      audio->pcm_size = 0;
      return 0;
   }
   if ( !file_mpeg->is_audio_stream )
   {
      time_position = ( double )position / track->sample_rate;
      result |= mpeg_bits_seek_time( audio->astream, time_position );
      audio->pcm_position = mpeg_bits_packet_time( audio->astream ) * track->sample_rate;
   }
   else
   {
      frame_number = position / audio->samples_per_frame;
      byte_position = ( long )( audio->avg_framesize * frame_number );
      audio->pcm_position = frame_number * audio->samples_per_frame;
      if ( byte_position < audio->avg_framesize * 2 )
      {
         result |= mpeg_bits_seek_start( audio->astream );
         audio->pcm_position = 0;
      }
      else
      {
         result |= mpeg_bits_seek_byte( audio->astream, byte_position );
      }
   }
   if ( byte_position >= audio->avg_framesize * 2 && audio->layer == 3 && !result )
   {
      result |= mpeg_audio_prev_header( audio );
      result |= mpeg_audio_read_layer3_frame( audio );
   }
   mpeg_audio_reset_synths( audio );
   audio->pcm_size = 0;
   audio->pcm_point = 0;
   //
   return result;
}
/***************************************************************************/
static inline mpeg_audio_t* mpeg_audio_new( MPEG *file_mpeg, mpeg_atrack_t *track, int format )
{
   mpeg_audio_t *audio = mpeg_audio_allocate_struct( file_mpeg, track );
   int result = 0;
   //
   mpeg_audio_new_decode_tables( audio );
   audio->percentage_seek = -1;
   audio->sample_seek = -1;
   audio->format = format;
   if ( format == AUDIO_UNKNOWN )
   {
      if ( ( ( mpeg_bits_showbits( audio->astream, 32 ) & 0xffff0000 ) >> 16 ) == MPEG3_AC3_START_CODE )
         audio->format = AUDIO_AC3;
      else
         audio->format = AUDIO_MPEG;
   }
   result = mpeg_audio_read_header( audio );
   mpeg_audio_replace_buffer( audio, 262144 );
   if ( !result )
   {
      track->channels = audio->channels;
      switch( audio->format )
      {
         case AUDIO_AC3:
            track->sample_rate = mpeg_ac3_samplerates[ audio->sampling_frequency_code ];
            break;
         case AUDIO_MPEG:
            track->sample_rate = mpeg_freqs[ audio->sampling_frequency_code ];
            break;
         case AUDIO_PCM:
            track->sample_rate = 48000;
            break;
      }
      track->total_samples = mpeg_audio_get_length( audio, track );
      result |= mpeg_bits_seek_start( audio->astream );
   }
   else
   {
      mpeg_audio_delete_struct( audio );
      audio = 0;
   }
   //
   return audio;
}
/***************************************************************************/
static inline int mpeg_audio_delete( mpeg_audio_t *audio )
{
   mpeg_audio_delete_struct( audio );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_audio_read_raw( mpeg_audio_t *audio, unsigned char *output, long *size, long max_size )
{
   int result = 0;
   //
   *size = 0;
   switch( audio->format )
   {
      case AUDIO_AC3:
         if ( mpeg_bits_read_buffer( audio->astream, output, audio->framesize ) )
            return 1;
         *size = audio->framesize;
         break;
      case AUDIO_MPEG:
         result = mpeg_audio_read_header( audio );
         if ( !result )
         {
            if ( max_size < 4 )
               return 1;
            *output++ = ( audio->newhead & 0xff000000 ) >> 24;
            *output++ = ( audio->newhead & 0xff0000 ) >> 16;
            *output++ = ( audio->newhead & 0xff00 ) >> 8;
            *output++ = ( audio->newhead & 0xff );
            *size += 4;
            if ( max_size < 4 + audio->framesize )
               return 1;
            if ( mpeg_bits_read_buffer( audio->astream, output, audio->framesize ) )
               return 1;
            *size += audio->framesize;
         }
         break;
      case AUDIO_PCM:
         if ( mpeg_bits_read_buffer( audio->astream, output, audio->framesize ) )
            return 1;
         *size = audio->framesize;
         break;
      }
      //
      return result;
}
/***************************************************************************/
static inline int mpeg_audio_decode_audio( mpeg_audio_t *audio, float *output_f, short *output_i, int channel, long start_position, long len )
{
   long allocation_needed = len + MPEG3AUDIO_PADDING;
   long i, j, result = 0;
   long attempts;
   //
   if ( audio->pcm_allocated < allocation_needed )
   {
      mpeg_audio_replace_buffer( audio, allocation_needed );
   }
   if ( audio->percentage_seek >= 0 )
   {
      mpeg_bits_seek_percentage( audio->astream, audio->percentage_seek );
      audio->pcm_position = start_position;
      audio->pcm_size = 0;
      audio->percentage_seek = -1;
   }
   else
   {
      if ( start_position >= audio->pcm_position && start_position < audio->pcm_position + audio->pcm_size && start_position + len <= audio->pcm_size )
      {
         ;
      }
      else if ( start_position <= audio->pcm_position + audio->pcm_size && start_position >= audio->pcm_position )
      {
         for( i = 0, j = ( start_position - audio->pcm_position ) * audio->channels;
            j < audio->pcm_size * audio->channels;
         i++, j++ )
         {
            audio->pcm_sample[ i ] = audio->pcm_sample[ j ];
         }
         audio->pcm_point = i;
         audio->pcm_size -= start_position - audio->pcm_position;
         audio->pcm_position = start_position;
      }
      else
      {
         result = mpeg_audio_seek( audio, start_position );
         audio->sample_seek = -1;
         if ( start_position < audio->pcm_position )
            audio->pcm_position = start_position;
      }
      audio->sample_seek = -1;
   }
   if ( !result )
   {
      attempts = 0;
	  while( attempts < 6 && !mpeg_bits_eof( audio->astream ) && audio->pcm_size + audio->pcm_position < start_position + len )
      {
         result = mpeg_audio_read_frame( audio );
         if ( result )
            attempts++;
         audio->pcm_size = audio->pcm_point / audio->channels;
      }
   }
   if ( output_f )
   {
      for( i = 0, j = ( start_position - audio->pcm_position ) * audio->channels + channel; 
         i < len && j < audio->pcm_size * audio->channels; 
      i++, j += audio->channels )
      {
         output_f[ i ] = audio->pcm_sample[ j ];
      }
      for( ; i < len; i++ )
      {
         output_f[ i ] = 0;
      }
   }
   else if ( output_i )
   {
      int sample;
      for( i = 0, j = ( start_position - audio->pcm_position ) * audio->channels + channel; 
         i < len && j < audio->pcm_size * audio->channels; 
      i++, j += audio->channels )
      {
         sample = ( int )( audio->pcm_sample[ j ] * 32767 );
         if ( sample > 32767 )
            sample = 32767;
         else if ( sample < -32768 )
            sample = -32768;
         ( ( signed short* )output_i )[ i ] = ( signed short )sample^0x8000;
      }
      for( ; i < len; i++ )
      {
         output_i[ i ] = 0;
      }
   }
   if ( audio->pcm_point > 0 )
      return 0;
   else
      return result;
}
/***************************************************************************/
static inline int mpeg_audio_read_pcm_header( mpeg_audio_t *audio )
{
   unsigned int code = mpeg_bits_getbits( audio->astream, 16 );
   //
   while( !mpeg_bits_eof( audio->astream ) && code != MPEG3_PCM_START_CODE )
   {
      code <<= 8;
      code &= 0xffff;
      code |= mpeg_bits_getbits( audio->astream, 8 );
   }
   audio->avg_framesize = audio->framesize = 0x7db;
   audio->channels = 2;
   return mpeg_bits_eof( audio->astream );
}
/***************************************************************************/
static inline int mpeg_audio_do_pcm( mpeg_audio_t *audio )
{
   int i, j, k;
   MPEG3_INT16 sample;
   int frame_samples = ( audio->framesize - 3 ) / audio->channels / 2;
   //
   if ( mpeg_bits_read_buffer( audio->astream, audio->ac3_buffer, frame_samples * audio->channels * 2 ) )
      return 1;
   if ( audio->pcm_point / audio->channels >= audio->pcm_allocated - MPEG3AUDIO_PADDING * audio->channels )
   {
      mpeg_audio_replace_buffer( audio, audio->pcm_allocated + MPEG3AUDIO_PADDING * audio->channels );
   }
   k = 0;
   for( i = 0; i < frame_samples; i++ )
   {
      for( j = 0; j < audio->channels; j++ )
      {
         sample = ( ( MPEG3_INT16 )( audio->ac3_buffer[ k++ ] ) ) << 8;
         sample |= audio->ac3_buffer[ k++ ];
         audio->pcm_sample[ audio->pcm_point + i * audio->channels + j ] = ( float )sample / 32767;
      }
   }
   audio->pcm_point += frame_samples * audio->channels;
   //
   return 0;
}
/***************************************************************************/
#define WRITE_SAMPLE(            samples, sum ) {( *samples ) = ( sum ); }
static inline int mpeg_audio_synth_stereo( mpeg_audio_t *audio, float *bandPtr, int channel, float *out, int *pnt )
{
   const int step = 2;
   float *samples = out + *pnt;
   float sum;
   float *b0, ( *buf )[ 0x110 ];
   int bo1;
   //
   if ( !channel )
   {
      audio->bo--;
      audio->bo &= 0xf;
      buf = audio->synth_stereo_buffs[ 0 ];
   }
   else 
   {
      samples++;
      buf = audio->synth_stereo_buffs[ 1 ];
   }
   if ( audio->bo & 0x1 )
   {
      b0 = buf[ 0 ];
      bo1 = audio->bo;
      mpeg_audio_dct64( buf[ 1 ] + ( ( audio->bo + 1 ) & 0xf ), buf[ 0 ] + audio->bo, bandPtr );
   }
   else 
   {
      b0 = buf[ 1 ];
      bo1 = audio->bo + 1;
      mpeg_audio_dct64( buf[ 0 ] + audio->bo, buf[ 1 ] + audio->bo + 1, bandPtr );
   }
   {
      int j;
      float *window = mpeg_decwin + 16 - bo1;
      for( j = 16; j; j--, b0 += 0x10, window += 0x20, samples += step )
      {
         sum = window[ 0x0 ] * b0[ 0x0 ];
         sum -= window[ 0x1 ] * b0[ 0x1 ];
         sum += window[ 0x2 ] * b0[ 0x2 ];
         sum -= window[ 0x3 ] * b0[ 0x3 ];
         sum += window[ 0x4 ] * b0[ 0x4 ];
         sum -= window[ 0x5 ] * b0[ 0x5 ];
         sum += window[ 0x6 ] * b0[ 0x6 ];
         sum -= window[ 0x7 ] * b0[ 0x7 ];
         sum += window[ 0x8 ] * b0[ 0x8 ];
         sum -= window[ 0x9 ] * b0[ 0x9 ];
         sum += window[ 0xA ] * b0[ 0xA ];
         sum -= window[ 0xB ] * b0[ 0xB ];
         sum += window[ 0xC ] * b0[ 0xC ];
         sum -= window[ 0xD ] * b0[ 0xD ];
         sum += window[ 0xE ] * b0[ 0xE ];
         sum -= window[ 0xF ] * b0[ 0xF ];
         WRITE_SAMPLE( samples, sum );
      }
      sum = window[ 0x0 ] * b0[ 0x0 ];
      sum += window[ 0x2 ] * b0[ 0x2 ];
      sum += window[ 0x4 ] * b0[ 0x4 ];
      sum += window[ 0x6 ] * b0[ 0x6 ];
      sum += window[ 0x8 ] * b0[ 0x8 ];
      sum += window[ 0xA ] * b0[ 0xA ];
      sum += window[ 0xC ] * b0[ 0xC ];
      sum += window[ 0xE ] * b0[ 0xE ];
      WRITE_SAMPLE( samples, sum );
      b0 -= 0x10;
      window -= 0x20;
      samples += step;
      window += bo1 << 1;
      for( j = 15; j; j--, b0 -= 0x10, window -= 0x20, samples += step )
      {
         sum = -window[ -0x1 ] * b0[ 0x0 ];
         sum -= window[ -0x2 ] * b0[ 0x1 ];
         sum -= window[ -0x3 ] * b0[ 0x2 ];
         sum -= window[ -0x4 ] * b0[ 0x3 ];
         sum -= window[ -0x5 ] * b0[ 0x4 ];
         sum -= window[ -0x6 ] * b0[ 0x5 ];
         sum -= window[ -0x7 ] * b0[ 0x6 ];
         sum -= window[ -0x8 ] * b0[ 0x7 ];
         sum -= window[ -0x9 ] * b0[ 0x8 ];
         sum -= window[ -0xA ] * b0[ 0x9 ];
         sum -= window[ -0xB ] * b0[ 0xA ];
         sum -= window[ -0xC ] * b0[ 0xB ];
         sum -= window[ -0xD ] * b0[ 0xC ];
         sum -= window[ -0xE ] * b0[ 0xD ];
         sum -= window[ -0xF ] * b0[ 0xE ];
         sum -= window[ -0x0 ] * b0[ 0xF ];
         WRITE_SAMPLE( samples, sum );
      }
   }
   *pnt += 64;
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_audio_synth_mono( mpeg_audio_t *audio, float *bandPtr, float *samples, int *pnt )
{
   float *samples_tmp = audio->synth_mono_buff;
   float *tmp1 = samples_tmp;
   int i, ret;
   int pnt1 = 0;
   //
   ret = mpeg_audio_synth_stereo( audio, bandPtr, 0, samples_tmp, &pnt1 );
   samples += *pnt;
   for( i = 0; i < 32; i++ )
   {
      *samples = *tmp1;
      samples++;
      tmp1 += 2;
   }
   *pnt += 32;
   return ret;
}
/***************************************************************************/
static inline int mpeg_audio_reset_synths( mpeg_audio_t *audio )
{
   int i, j, k;
   for( i = 0; i < 2; i++ )
   {
      for( j = 0; j < 2; j++ )
      {
         for( k = 0; k < 0x110; k++ )
         {
            audio->synth_stereo_buffs[ i ][ j ][ k ] = 0;
         }
      }
   }
   for( i = 0; i < 64; i++ )
   {
      audio->synth_mono_buff[ i ] = 0;
      audio->layer2_scfsi_buf[ i ] = 0;
   }
   for( i = 0; i < 2; i++ )
   {
      for( j = 0; j < 2; j++ )
      {
         for( k = 0; k < SBLIMIT * SSLIMIT; k++ )
         {
            audio->mp3_block[ i ][ j ][ k ] = 0;
         }
      }
   }
   audio->mp3_blc[ 0 ] = 0;
   audio->mp3_blc[ 1 ] = 0;
   for( i = 0; i < audio->channels; i++ )
   {
      for( j = 0; j < AC3_N / 2; j++ )
      {
         audio->ac3_delay[ i ][ j ] = 0;
      }
   }
   return 0;
}
/***************************************************************************/
static int mpeg_tabsel_123[ 2 ][ 3 ][ 16 ] =
{
   {{0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448, }, 
   {0, 32, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 384, }, 
   {0, 32, 40, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, } }, 
   {{0, 32, 48, 56, 64, 80, 96, 112, 128, 144, 160, 176, 192, 224, 256, }, 
   {0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, }, 
   {0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, } }
};
static inline int mpeg_audio_init_layer2( mpeg_audio_t *audio )
{
   static double mulmul[ 27 ] = 
   {
      0.0 , -2.0/3.0 , 2.0/3.0 , 
      2.0/7.0 , 2.0/15.0 , 2.0/31.0, 2.0/63.0 , 2.0/127.0 , 2.0/255.0 , 
      2.0/511.0 , 2.0/1023.0 , 2.0/2047.0 , 2.0/4095.0 , 2.0/8191.0 , 
      2.0/16383.0 , 2.0/32767.0 , 2.0/65535.0 , 
      -4.0/5.0 , -2.0/5.0 , 2.0/5.0, 4.0/5.0 , 
      -8.0/9.0 , -4.0/9.0 , -2.0/9.0 , 2.0/9.0 , 4.0/9.0 , 8.0/9.0 
   };
   static int base[ 3 ][ 9 ] = 
   {
      {1 , 0, 2 , } , 
      {17, 18, 0 , 19, 20 , } , 
      {21, 1, 22, 23, 0, 24, 25, 2, 26 } 
   };
   static int tablen[ 3 ] = {3, 5, 9 };
   static int *itable, *tables[ 3 ] = {mpeg_grp_3tab, mpeg_grp_5tab, mpeg_grp_9tab};
   int i, j, k, l, len;
   float *table;
   for( i = 0; i < 3; i++ )
   {
      itable = tables[ i ];
      len = tablen[ i ];
      for( j = 0; j < len; j++ )
         for( k = 0; k < len; k++ )
      for( l = 0; l < len; l++ )
      {
         *itable++ = base[ i ][ l ];
         *itable++ = base[ i ][ k ];
         *itable++ = base[ i ][ j ];
      }
   }
   for( k = 0; k < 27; k++ )
   {
      double m = mulmul[ k ];
      table = mpeg_muls[ k ];
      for( j = 3, i = 0; i < 63; i++, j-- )
         *table++ = m * pow( 2.0, ( double )j / 3.0 );
      *table++ = 0.0;
   }
   return 0;
}
/***************************************************************************/
static inline int mpeg_audio_init_layer3( mpeg_audio_t *audio )
{
   int i, j, k, l;
   int down_sample_sblimit = 32;
   //
   audio->mp3_block[ 0 ][ 0 ][ 0 ] = 0;
   audio->mp3_blc[ 0 ] = 0;
   audio->mp3_blc[ 1 ] = 0;
   for( i = -256; i < 118 + 4; i++ )
      mpeg_gainpow2[ i + 256 ] = pow( ( double )2.0, -0.25 * ( double )( i + 210 ) );
   for( i = 0; i < 8207; i++ )
      mpeg_ispow[ i ] = pow( ( double )i, ( double )4.0 / 3.0 );
   for( i = 0; i < 8; i++ ) 
   {
      static double Ci[ 8 ] = {-0.6, -0.535, -0.33, -0.185, -0.095, -0.041, -0.0142, -0.0037};
      double sq = sqrt( 1.0+Ci[ i ]*Ci[ i ] );
      mpeg_aa_cs[ i ] = 1.0/sq;
      mpeg_aa_ca[ i ] = Ci[ i ]/sq;
   }
   for( i = 0; i < 18; i++ ) 
   {
      mpeg_win[ 0 ][ i ] = mpeg_win[ 1 ][ i ] = 0.5 * sin( M_PI / 72.0 * ( double )( 2 * ( i + 0 ) + 1 ) ) / cos ( M_PI * ( double )( 2 * ( i + 0 ) + 19 ) / 72.0 );
      mpeg_win[ 0 ][ i+18 ] = mpeg_win[ 3 ][ i+18 ] = 0.5 * sin( M_PI / 72.0 * ( double )( 2 * ( i + 18 ) + 1 ) ) / cos ( M_PI * ( double )( 2 * ( i + 18 ) + 19 ) / 72.0 );
   }
   for( i = 0; i < 6; i++ ) 
   {
      mpeg_win[ 1 ][ i + 18 ] = 0.5 / cos ( M_PI * ( double ) ( 2*( i+18 )+19 ) / 72.0 );
      mpeg_win[ 3 ][ i + 12 ] = 0.5 / cos ( M_PI * ( double ) ( 2*( i+12 )+19 ) / 72.0 );
      mpeg_win[ 1 ][ i + 24 ] = 0.5 * sin( M_PI / 24.0 * ( double )( 2 * i + 13 ) ) / cos ( M_PI * ( double )( 2 * ( i + 24 )+ 19 ) / 72.0 );
      mpeg_win[ 1 ][ i + 30 ] = mpeg_win[ 3 ][ i ] = 0.0;
      mpeg_win[ 3 ][ i + 6 ] = 0.5 * sin( M_PI / 24.0 * ( double )( 2 * i + 1 ) ) / cos ( M_PI * ( double )( 2 * ( i + 6 )+ 19 ) / 72.0 );
   }
   for( i = 0; i < 9; i++ )
      mpeg_COS9[ i ] = cos( M_PI / 18.0 * ( double )i );
   for( i = 0; i < 9; i++ )
      mpeg_tfcos36[ i ] = 0.5 / cos ( M_PI * ( double ) ( i*2+1 ) / 36.0 );
   for( i = 0; i < 3; i++ )
      mpeg_tfcos12[ i ] = 0.5 / cos ( M_PI * ( double ) ( i*2+1 ) / 12.0 );
   mpeg_COS6_1 = cos( M_PI / 6.0 * ( double ) 1 );
   mpeg_COS6_2 = cos( M_PI / 6.0 * ( double ) 2 );
   mpeg_cos9[ 0 ] = cos( 1.0 * M_PI / 9.0 );
   mpeg_cos9[ 1 ] = cos( 5.0 * M_PI / 9.0 );
   mpeg_cos9[ 2 ] = cos( 7.0 * M_PI / 9.0 );
   mpeg_cos18[ 0 ] = cos( 1.0 * M_PI / 18.0 );
   mpeg_cos18[ 1 ] = cos( 11.0 * M_PI / 18.0 );
   mpeg_cos18[ 2 ] = cos( 13.0 * M_PI / 18.0 );
   for( i = 0; i < 12; i++ ) 
   {
      mpeg_win[ 2 ][ i ] = 0.5 * sin( M_PI / 24.0 * ( double ) ( 2 * i + 1 ) ) / cos( M_PI * ( double )( 2 * i + 7 ) / 24.0 );
      for( j = 0; j < 6; j++ )
         mpeg_COS1[ i ][ j ] = cos( M_PI / 24.0 * ( double ) ( ( 2 * i + 7 ) * ( 2 * j + 1 ) ) );
   }
   for( j = 0; j < 4; j++ ) 
   {
      static int len[ 4 ] = {36, 36, 12, 36};
      for( i = 0; i < len[ j ]; i += 2 )
         mpeg_win1[ j ][ i ] = + mpeg_win[ j ][ i ];
      for( i = 1; i < len[ j ]; i += 2 )
         mpeg_win1[ j ][ i ] = - mpeg_win[ j ][ i ];
   }
   for( i = 0; i < 16; i++ ) 
   {
      double t = tan( ( double ) i * M_PI / 12.0 );
      mpeg_tan1_1[ i ] = t / ( 1.0 + t );
      mpeg_tan2_1[ i ] = 1.0 / ( 1.0 + t );
      mpeg_tan1_2[ i ] = M_SQRT2 * t / ( 1.0 + t );
      mpeg_tan2_2[ i ] = M_SQRT2 / ( 1.0 + t );
      for( j = 0; j < 2; j++ ) 
      {
         double base = pow( 2.0, -0.25 * ( j + 1.0 ) );
         double p1 = 1.0, p2 = 1.0;
         if ( i > 0 ) 
         {
            if ( i & 1 )
               p1 = pow( base, ( i + 1.0 ) * 0.5 );
            else
               p2 = pow( base, i * 0.5 );
         }
         mpeg_pow1_1[ j ][ i ] = p1;
         mpeg_pow2_1[ j ][ i ] = p2;
         mpeg_pow1_2[ j ][ i ] = M_SQRT2 * p1;
         mpeg_pow2_2[ j ][ i ] = M_SQRT2 * p2;
      }
   }
   for( j = 0; j < 9; j++ ) 
   {
      struct mpeg_bandInfoStruct *bi = &mpeg_bandInfo[ j ];
      int *mp;
      int cb, lwin;
      int *bdf;
      mp = mpeg_map[ j ][ 0 ] = mpeg_mapbuf0[ j ];
      bdf = bi->longDiff;
      for( i = 0, cb = 0; cb < 8; cb++, i += *bdf++ ) 
      {
         *mp++ = ( *bdf ) >> 1;
         *mp++ = i;
         *mp++ = 3;
         *mp++ = cb;
      }
      bdf = bi->shortDiff + 3;
      for( cb = 3; cb < 13; cb++ ) 
      {
         int l = ( *bdf++ ) >> 1;
         for( lwin = 0; lwin < 3; lwin++ ) 
         {
            *mp++ = l;
            *mp++ = i + lwin;
            *mp++ = lwin;
            *mp++ = cb;
         }
         i += 6 * l;
      }
      mpeg_mapend[ j ][ 0 ] = mp;
      mp = mpeg_map[ j ][ 1 ] = mpeg_mapbuf1[ j ];
      bdf = bi->shortDiff+0;
      for( i = 0, cb = 0; cb < 13; cb++ ) 
      {
         int l = ( *bdf++ ) >> 1;
         for( lwin = 0; lwin < 3; lwin++ ) 
         {
            *mp++ = l;
            *mp++ = i + lwin;
            *mp++ = lwin;
            *mp++ = cb;
         }
         i += 6 * l;
      }
      mpeg_mapend[ j ][ 1 ] = mp;
      mp = mpeg_map[ j ][ 2 ] = mpeg_mapbuf2[ j ];
      bdf = bi->longDiff;
      for( cb = 0; cb < 22 ; cb++ ) 
      {
         *mp++ = ( *bdf++ ) >> 1;
         *mp++ = cb;
      }
      mpeg_mapend[ j ][ 2 ] = mp;
   }
   for( j = 0; j < 9; j++ ) 
   {
      for( i = 0; i < 23; i++ ) 
      {
         mpeg_longLimit[ j ][ i ] = ( mpeg_bandInfo[ j ].longIdx[ i ] - 1 + 8 ) / 18 + 1;
         if ( mpeg_longLimit[ j ][ i ] > ( down_sample_sblimit ) )
            mpeg_longLimit[ j ][ i ] = down_sample_sblimit;
      }
      for( i = 0; i < 14; i++ ) 
      {
         mpeg_shortLimit[ j ][ i ] = ( mpeg_bandInfo[ j ].shortIdx[ i ] - 1 ) / 18 + 1;
         if ( mpeg_shortLimit[ j ][ i ] > ( down_sample_sblimit ) )
            mpeg_shortLimit[ j ][ i ] = down_sample_sblimit;
      }
   }
   for( i = 0; i < 5; i++ ) 
   {
      for( j = 0; j < 6; j++ ) 
      {
         for( k = 0; k < 6; k++ ) 
         {
            int n = k + j * 6 + i * 36;
            mpeg_i_slen2[ n ] = i | ( j << 3 ) | ( k << 6 ) | ( 3 << 12 );
         }
      }
   }
   for( i = 0; i < 4; i++ ) 
   {
      for( j = 0; j < 4; j++ ) 
      {
         for( k = 0; k < 4; k++ ) 
         {
            int n = k + j * 4 + i * 16;
            mpeg_i_slen2[ n+180 ] = i | ( j << 3 ) | ( k << 6 ) | ( 4 << 12 );
         }
      }
   }
   for( i = 0; i < 4; i++ ) 
   {
      for( j = 0; j < 3; j++ ) 
      {
         int n = j + i * 3;
         mpeg_i_slen2[ n + 244 ] = i | ( j << 3 ) | ( 5 << 12 );
         mpeg_n_slen2[ n + 500 ] = i | ( j << 3 ) | ( 2 << 12 ) | ( 1 << 15 );
      }
   }
   for( i = 0; i < 5; i++ ) 
   {
      for( j = 0; j < 5; j++ ) 
      {
         for( k = 0; k < 4; k++ ) 
         {
            for( l = 0; l < 4; l++ ) 
            {
               int n = l + k * 4 + j * 16 + i * 80;
               mpeg_n_slen2[ n ] = i | ( j << 3 ) | ( k << 6 ) | ( l << 9 ) | ( 0 << 12 );
            }
         }
      }
   }
   for( i = 0; i < 5; i++ ) 
   {
      for( j = 0; j < 5; j++ ) 
      {
         for( k = 0; k < 4; k++ ) 
         {
            int n = k + j * 4 + i * 20;
            mpeg_n_slen2[ n + 400 ] = i | ( j << 3 ) | ( k << 6 ) | ( 1 << 12 );
         }
      }
   }
   return 0;
}
/***************************************************************************/
static inline int mpeg_audio_new_decode_tables( mpeg_audio_t *audio )
{
   int i, j, k, kr, divv;
   float *costab;
   int idx;
   long scaleval = audio->outscale;
   for( i = 0; i < 5; i++ )
   {
      kr = 0x10 >> i; 
      divv = 0x40 >> i;
      costab = mpeg_pnts[ i ];
      for( k = 0; k < kr; k++ )
         costab[ k ] = 1.0 / ( 2.0 * cos( M_PI * ( ( double )k * 2.0 + 1.0 ) / ( double )divv ) );
   }
   idx = 0;
   scaleval = -scaleval;
   for( i = 0, j = 0; i < 256; i++, j++, idx += 32 )
   {
      if ( idx < 512 + 16 )
         mpeg_decwin[ idx+16 ] = mpeg_decwin[ idx ] = ( double )mpeg_intwinbase[ j ] / 65536.0 * ( double )scaleval;
      if ( i % 32 == 31 )
         idx -= 1023;
      if ( i % 64 == 63 )
         scaleval = -scaleval;
   }
   for( ; i < 512; i++, j--, idx += 32 )
   {
      if ( idx < 512 + 16 )
         mpeg_decwin[ idx + 16 ] = mpeg_decwin[ idx ] = ( double )mpeg_intwinbase[ j ] / 65536.0 * ( double )scaleval;
      if ( i % 32 == 31 )
         idx -= 1023;
      if ( i % 64 == 63 )
         scaleval = -scaleval;
   }
   audio->ac3_lfsr_state = 1;
   mpeg_audio_imdct_init( audio );
   mpeg_audio_init_layer2( audio ); 
   mpeg_audio_init_layer3( audio );
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_get_cbp( mpeg_slice_t *slice )
{
   int code;
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   //
   if ( ( code = mpeg_slice_showbits9( slice_buffer ) ) >= 128 )
   {
      code >>= 4;
      mpeg_slice_flushbits( slice_buffer, mpeg_CBPtab0[ code ].len );
      return mpeg_CBPtab0[ code ].val;
   }
   if ( code >= 8 )
   {
      code >>= 1;
      mpeg_slice_flushbits( slice_buffer, mpeg_CBPtab1[ code ].len );
      return mpeg_CBPtab1[ code ].val;
   }
   if ( code < 1 )
   {
      slice->fault = 1;
      return 0;
   }
   mpeg_slice_flushbits( slice_buffer, mpeg_CBPtab2[ code ].len );
   //
   return mpeg_CBPtab2[ code ].val;
}
/***************************************************************************/
static inline int mpeg_video_clearblock( mpeg_slice_t *slice, int comp, int size )
{
   slice->sparse[ comp ] = 1;
   if ( size == 6 )
   {
      #ifdef DOS
         bzero( slice->block[ comp ], sizeof( short ) * 64 * 6 );
      #else
         memset( slice->block[ comp ], 0, sizeof( short ) * 64 * 6 );
      #endif
   }
   else
   {
      memset( slice->block[ comp ], 0, sizeof( short ) * 64 * size );
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_getdclum( mpeg_slice_buffer_t *slice_buffer )
{
   int code, size, val;
   //
   code = mpeg_slice_showbits5( slice_buffer );
   if ( code < 31 )
   {
      size = mpeg_DClumtab0[ code ].val;
      mpeg_slice_flushbits( slice_buffer, mpeg_DClumtab0[ code ].len );
   }
   else 
   {
      code = mpeg_slice_showbits9( slice_buffer ) - 0x1f0;
      size = mpeg_DClumtab1[ code ].val;
      mpeg_slice_flushbits( slice_buffer, mpeg_DClumtab1[ code ].len );
   }
   if ( size == 0 )
      val = 0;
   else 
   {
      val = mpeg_slice_getbits( slice_buffer, size );
      if ( ( val & ( 1 << ( size - 1 ) ) ) == 0 )
         val -= ( 1 << size ) - 1;
   }
   //
   return val;
}
/***************************************************************************/
static inline int mpeg_video_getdcchrom( mpeg_slice_buffer_t *slice_buffer )
{
   int code, size, val;
   //
   code = mpeg_slice_showbits5( slice_buffer );
   if ( code < 31 )
   {
      size = mpeg_DCchromtab0[ code ].val;
      mpeg_slice_flushbits( slice_buffer, mpeg_DCchromtab0[ code ].len );
   }
   else 
   {
      code = mpeg_slice_showbits( slice_buffer, 10 ) - 0x3e0;
      size = mpeg_DCchromtab1[ code ].val;
      mpeg_slice_flushbits( slice_buffer, mpeg_DCchromtab1[ code ].len );
   }
   if ( size == 0 )
      val = 0;
   else 
   {
      val = mpeg_slice_getbits( slice_buffer, size );
      if ( ( val & ( 1 << ( size - 1 ) ) ) == 0 )
         val -= ( 1 << size ) - 1;
   }
   //
   return val;
}
/***************************************************************************/
static inline int mpeg_video_getintrablock( mpeg_slice_t *slice, mpeg_video_t *video, int comp, int dc_dct_pred[] )
{
   int val, i, j = 0, sign;
   unsigned int code;
   mpeg_DCTtab_t *tab = 0;
   short *bp = slice->block[ comp ];
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   //
   if ( comp < 4 ) 
      bp[ 0 ] = ( dc_dct_pred[ 0 ] += mpeg_video_getdclum( slice_buffer ) ) << 3;
   else if ( comp == 4 ) 
      bp[ 0 ] = ( dc_dct_pred[ 1 ] += mpeg_video_getdcchrom( slice_buffer ) ) << 3;
   else 
      bp[ 0 ] = ( dc_dct_pred[ 2 ] += mpeg_video_getdcchrom( slice_buffer ) ) << 3;
   if ( slice->fault )
      return 1;
   for( i = 1; ; i++ )
   {
      code = mpeg_slice_showbits16( slice_buffer );
      if ( code >= 16384 )
         tab = &mpeg_DCTtabnext[ ( code >> 12 ) - 4 ];
      else if ( code >= 1024 )
         tab = &mpeg_DCTtab0[ ( code >> 8 ) - 4 ];
      else if ( code >= 512 )
         tab = &mpeg_DCTtab1[ ( code >> 6 ) - 8 ];
      else if ( code >= 256 )
         tab = &mpeg_DCTtab2[ ( code >> 4 ) - 16 ];
      else if ( code >= 128 )
         tab = &mpeg_DCTtab3[ ( code >> 3 ) - 16 ];
      else if ( code >= 64 )
         tab = &mpeg_DCTtab4[ ( code >> 2 ) - 16 ];
      else if ( code >= 32 )
         tab = &mpeg_DCTtab5[ ( code >> 1 ) - 16 ];
      else if ( code >= 16 )
         tab = &mpeg_DCTtab6[ code - 16 ];
      else 
      {
         slice->fault = 1;
         return 1;
      }
      mpeg_slice_flushbits( slice_buffer, tab->len );
      if ( tab->run == 64 )
         break; 
      if ( tab->run == 65 )
      {
         i += mpeg_slice_getbits( slice_buffer, 6 );
         if ( ( val = mpeg_slice_getbits( slice_buffer, 8 ) ) == 0 ) 
            val = mpeg_slice_getbits( slice_buffer, 8 );
         else if ( val == 128 ) 
            val = mpeg_slice_getbits( slice_buffer, 8 ) - 256;
         else if ( val > 128 ) 
            val -= 256;
         if ( ( sign = ( val < 0 ) ) != 0 )
            val= -val;
      }
      else 
      {
         i += tab->run;
         val = tab->level;
         sign = mpeg_slice_getbit( slice_buffer );
      }
      if ( i < 64 )
         j = video->mpeg_zigzag_scan_table[ i ];
      else
      {
         slice->fault = 1;
         return 1;
      }
      {
         val = ( val * slice->quant_scale * video->intra_quantizer_matrix[ j ] ) >> 3;
         val = ( val - 1 ) | 1;
      }
      bp[ j ] = sign ? -val : val;
   }
   if ( j != 0 ) 
   {
      slice->sparse[ comp ] = 0;
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_getinterblock( mpeg_slice_t *slice, mpeg_video_t *video, int comp )
{
   int val, i, j = 0, sign;
   unsigned int code;
   mpeg_DCTtab_t *tab; 
   short *bp = slice->block[ comp ];
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   //
   for( i = 0; ; i++ )
   {
      code = mpeg_slice_showbits16( slice_buffer );
      if ( code >= 16384 )
      {
         if ( i == 0 ) 
            tab = &mpeg_DCTtabfirst[ ( code >> 12 ) - 4 ];
         else 
            tab = &mpeg_DCTtabnext[ ( code >> 12 ) - 4 ];
      }
      else if ( code >= 1024 )
         tab = &mpeg_DCTtab0[ ( code >> 8 ) - 4 ];
      else if ( code >= 512 )
         tab = &mpeg_DCTtab1[ ( code >> 6 ) - 8 ];
      else if ( code >= 256 )
         tab = &mpeg_DCTtab2[ ( code >> 4 ) - 16 ];
      else if ( code >= 128 )
         tab = &mpeg_DCTtab3[ ( code >> 3 ) - 16 ];
      else if ( code >= 64 )
         tab = &mpeg_DCTtab4[ ( code >> 2 ) - 16 ];
      else if ( code >= 32 )
         tab = &mpeg_DCTtab5[ ( code >> 1 ) - 16 ];
      else if ( code >= 16 )
         tab = &mpeg_DCTtab6[ code - 16 ];
      else 
      {
         slice->fault = 1;
         return 1;
      }
      mpeg_slice_flushbits( slice_buffer, tab->len );
      if ( tab->run == 64 )
         break; 
      if ( tab->run == 65 )
      {
         i += mpeg_slice_getbits( slice_buffer, 6 );
         if ( ( val = mpeg_slice_getbits( slice_buffer, 8 ) ) == 0 ) 
            val = mpeg_slice_getbits( slice_buffer, 8 );
         else if ( val == 128 ) 
            val = mpeg_slice_getbits( slice_buffer, 8 ) - 256;
         else if ( val > 128 ) 
            val -= 256;
         if ( ( sign = ( val < 0 ) ) != 0 )
            val = -val;
      }
      else 
      {
         i += tab->run;
         val = tab->level;
         sign = mpeg_slice_getbit( slice_buffer );
      }
      j = video->mpeg_zigzag_scan_table[ i ];
      {
         val = ( ( ( val << 1 )+1 ) * slice->quant_scale * video->non_intra_quantizer_matrix[ j ] ) >> 4;
         val = ( val - 1 ) | 1;
      }
      bp[ j ] = sign ? -val : val;
   }
   if ( j != 0 ) 
   {
      slice->sparse[ comp ] = 0;
   }
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_getmpeg_2intrablock( mpeg_slice_t *slice, mpeg_video_t *video, int comp, int dc_dct_pred[] )
{
   int val, i, j = 0, sign, nc;
   unsigned int code;
   mpeg_DCTtab_t *tab;
   short *bp;
   int *qmat;
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   //
   bp = slice->block[ comp ];
   qmat = ( comp < 4 || video->chroma_format == CHROMA420 ) ? video->intra_quantizer_matrix : video->chroma_intra_quantizer_matrix;
   if ( comp < 4 ) 
      val = ( dc_dct_pred[ 0 ] += mpeg_video_getdclum( slice_buffer ) );
   else if ( ( comp & 1 ) == 0 ) 
      val = ( dc_dct_pred[ 1 ] += mpeg_video_getdcchrom( slice_buffer ) );
   else 
      val = ( dc_dct_pred[ 2 ] += mpeg_video_getdcchrom( slice_buffer ) );
   if ( slice->fault )
      return 1;
   bp[ 0 ] = val << ( 3 - video->dc_prec );
   nc = 0;
   for( i = 1; ; i++ )
   {
      code = mpeg_slice_showbits16( slice_buffer );
      if ( code >= 16384 && !video->intravlc )
         tab = &mpeg_DCTtabnext[ ( code >> 12 ) - 4 ];
      else if ( code >= 1024 )
      {
         if ( video->intravlc ) 
            tab = &mpeg_DCTtab0a[ ( code >> 8 ) - 4 ];
         else 
            tab = &mpeg_DCTtab0[ ( code >> 8 ) - 4 ];
      }
      else if ( code >= 512 )
      {
         if ( video->intravlc ) 
            tab = &mpeg_DCTtab1a[ ( code >> 6 ) - 8 ];
         else 
            tab = &mpeg_DCTtab1[ ( code >> 6 ) - 8 ];
      }
      else if ( code >= 256 )
         tab = &mpeg_DCTtab2[ ( code >> 4 ) - 16 ];
      else if ( code >= 128 )
         tab = &mpeg_DCTtab3[ ( code >> 3 ) - 16 ];
      else if ( code >= 64 )
         tab = &mpeg_DCTtab4[ ( code >> 2 ) - 16 ];
      else if ( code >= 32 )
         tab = &mpeg_DCTtab5[ ( code >> 1 ) - 16 ];
      else if ( code >= 16 )
         tab = &mpeg_DCTtab6[ code - 16 ];
      else 
      {
         slice->fault = 1;
         return 1;
      }
      mpeg_slice_flushbits( slice_buffer, tab->len );
      if ( tab->run == 64 )
         break; 
      if ( tab->run == 65 )
      {
         i += mpeg_slice_getbits( slice_buffer, 6 );
         val = mpeg_slice_getbits( slice_buffer, 12 );
         if ( ( val & 2047 ) == 0 )
         {
            slice->fault = 1;
            return 1;
         }
         if ( ( sign = ( val >= 2048 ) ) != 0 )
            val = 4096 - val;
      }
      else 
      {
         i += tab->run;
         val = tab->level;
         sign = mpeg_slice_getbit( slice_buffer );
      }
      j = ( video->altscan ? video->mpeg_alternate_scan_table : video->mpeg_zigzag_scan_table )[ i ];
      val = ( val * slice->quant_scale * qmat[ j ] ) >> 4;
      bp[ j ] = sign ? -val : val;
      nc++;
   }
   if ( j != 0 )
   {
      slice->sparse[ comp ] = 0;
   }
   //
   return 1;
}
/***************************************************************************/
static inline int mpeg_video_getmpeg_2interblock( mpeg_slice_t *slice, mpeg_video_t *video, int comp )
{
   int val, i, j = 0, sign, nc;
   unsigned int code;
   mpeg_DCTtab_t *tab;
   short *bp;
   int *qmat;
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   //
   bp = slice->block[ comp ];
   qmat = ( comp < 4 || video->chroma_format == CHROMA420 ) ? video->non_intra_quantizer_matrix : video->chroma_non_intra_quantizer_matrix;
   nc = 0;
   for( i = 0; ; i++ )
   {
      code = mpeg_slice_showbits16( slice_buffer );
      if ( code >= 16384 )
      {
         if ( i == 0 )
            tab = &mpeg_DCTtabfirst[ ( code >> 12 ) - 4 ];
         else
            tab = &mpeg_DCTtabnext[ ( code >> 12 ) - 4 ];
      }
      else if ( code >= 1024 )
         tab = &mpeg_DCTtab0[ ( code >> 8 ) - 4 ];
      else if ( code >= 512 )
         tab = &mpeg_DCTtab1[ ( code >> 6 ) - 8 ];
      else if ( code >= 256 )
         tab = &mpeg_DCTtab2[ ( code >> 4 ) - 16 ];
      else if ( code >= 128 )
         tab = &mpeg_DCTtab3[ ( code >> 3 ) - 16 ];
      else if ( code >= 64 )
         tab = &mpeg_DCTtab4[ ( code >> 2 ) - 16 ];
      else if ( code >= 32 )
         tab = &mpeg_DCTtab5[ ( code >> 1 ) - 16 ];
      else if ( code >= 16 )
         tab = &mpeg_DCTtab6[ code - 16 ];
      else 
      {
         slice->fault = 1;
         return 1;
      }
      mpeg_slice_flushbits( slice_buffer, tab->len );
      if ( tab->run == 64 )
         break; 
      if ( tab->run == 65 )
      {
         i += mpeg_slice_getbits( slice_buffer, 6 );
         val = mpeg_slice_getbits( slice_buffer, 12 );
         if ( ( val & 2047 ) == 0 )
         {
            slice->fault = 1;
            return 1;
         }
         if ( ( sign = ( val >= 2048 ) ) != 0 )
            val = 4096 - val;
      }
      else 
      {
         i += tab->run;
         val = tab->level;
         sign = mpeg_slice_getbit( slice_buffer );
      }
      j = ( video->altscan ? video->mpeg_alternate_scan_table : video->mpeg_zigzag_scan_table )[ i ];
      val = ( ( ( val << 1 )+1 ) * slice->quant_scale * qmat[ j ] ) >> 5;
      bp[ j ] = sign ? ( -val ) : val ;
      nc++;
   }
   if ( j != 0 ) 
   {
      slice->sparse[ comp ] = 0;
   }
   return 0;
}
/***************************************************************************/
static inline void mpeg_slice_loop( mpeg_slice_t *slice )
{
   mpeg_video_t *video = ( mpeg_video_t * ) slice->video;
   int result = 1;
   //
   while( !slice->done )
   {
      if ( !slice->done )
      {
         result = 1;
         if ( slice->buffer_step > 0 )
         {
		while( slice->current_buffer <= slice->last_buffer )
            {
               if ( !video->slice_buffers[ slice->current_buffer ].done && slice->current_buffer <= slice->last_buffer )
               {
                  result = 0;
                  break;
               }
               slice->current_buffer += slice->buffer_step;
            }
         }
         else
         {
		while( slice->current_buffer >= slice->last_buffer )
            {
               if ( !video->slice_buffers[ slice->current_buffer ].done && slice->current_buffer >= slice->last_buffer )
               {
                  result = 0;
                  break;
               }
               slice->current_buffer += slice->buffer_step;
            }
         }
         if ( !result && slice->current_buffer >= 0 && slice->current_buffer < video->total_slice_buffers )
         {
            slice->slice_buffer = &( video->slice_buffers[ slice->current_buffer ] );
            slice->slice_buffer->done = 1;
            mpeg_decode_slice( slice );
         }
         else 
         {
         }
      }
      if ( ( slice->current_buffer > slice->last_buffer ) || ( slice->last_buffer == 0 ) )
         break;
   }
}
/***************************************************************************/
static inline int mpeg_video_get_macroblocks( mpeg_video_t *video, int framenum )
{
   mpeg_slice_buffer_t *slice_buffer; 
   int i;
   int current_buffer;
   mpeg_bits_t *vstream = video->vstream;
   //
   video->total_slice_buffers = 0;
   current_buffer = 0;
   while( !mpeg_bits_eof( vstream ) && mpeg_bits_showbits32_noptr( vstream ) >= MPEG3_SLICE_MIN_START && mpeg_bits_showbits32_noptr( vstream ) <= MPEG3_SLICE_MAX_START )
   {
      if ( current_buffer >= video->slice_buffers_initialized )
         mpeg_new_slice_buffer( &( video->slice_buffers[ video->slice_buffers_initialized++ ] ) );
      slice_buffer = &( video->slice_buffers[ current_buffer ] );
      slice_buffer->buffer_size = 0;
      slice_buffer->current_position = 0;
      slice_buffer->bits_size = 0;
      slice_buffer->done = 0;
      do
      {
         if ( slice_buffer->buffer_allocation <= slice_buffer->buffer_size )
            mpeg_expand_slice_buffer( slice_buffer );
         slice_buffer->data[ slice_buffer->buffer_size++ ] = mpeg_bits_getbyte_noptr( vstream );
	  } while( !mpeg_bits_eof( vstream ) && mpeg_bits_showbits24_noptr( vstream ) != MPEG3_PACKET_START_CODE_PREFIX );
      if ( slice_buffer->buffer_allocation <= slice_buffer->buffer_size + 4 )
         mpeg_expand_slice_buffer( slice_buffer );
      slice_buffer->data[ slice_buffer->buffer_size++ ] = 0;
      slice_buffer->data[ slice_buffer->buffer_size++ ] = 0;
      slice_buffer->data[ slice_buffer->buffer_size++ ] = 1;
      slice_buffer->data[ slice_buffer->buffer_size++ ] = 0;
      slice_buffer->bits_size = 0;
      current_buffer++;
      video->total_slice_buffers++;
   }
   if ( video->total_slice_buffers > 0 )
   {
      for( i = 0; i < video->total_slice_decoders; i++ )
      {
         if ( i == 0 && video->total_slice_decoders > 1 )
         {
            video->slice_decoders[ i ].current_buffer = 0;
            video->slice_decoders[ i ].buffer_step = 1;
            video->slice_decoders[ i ].last_buffer = ( video->total_slice_buffers - 1 );
         }
         else if ( i == 1 )
         {
            video->slice_decoders[ i ].current_buffer = video->total_slice_buffers - 1;
            video->slice_decoders[ i ].buffer_step = -1;
            video->slice_decoders[ i ].last_buffer = 0;
         }
         else
         {
            video->slice_decoders[ i ].current_buffer = i;
            video->slice_decoders[ i ].buffer_step = 1;
            video->slice_decoders[ i ].last_buffer = video->total_slice_buffers - 1;
         }
      }
   }
   if ( video->total_slice_buffers > 0 )
   {
      for( i = 0; i < video->total_slice_decoders; i++ )
      {
         mpeg_slice_loop( &video->slice_decoders[ i ] );
      }
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_allocate_decoders( mpeg_video_t *video, int decoder_count )
{
   int i;
   MPEG *file_mpeg = ( MPEG * ) video->file_mpeg;
   //
   if ( video->total_slice_decoders != file_mpeg->cpus )
   {
      for( i = 0; i < video->total_slice_decoders; i++ )
      {
         mpeg_delete_slice_decoder( &( video->slice_decoders[ i ] ) );
      }
      for( i = 0; i < file_mpeg->cpus && i < MPEG3_MAX_CPUS; i++ )
      {
         mpeg_new_slice_decoder( video, &( video->slice_decoders[ i ] ) );
         video->slice_decoders[ i ].thread_number = i;
      }
      video->total_slice_decoders = file_mpeg->cpus;
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_getpicture( mpeg_video_t *video, int framenum )
{
   int i, result = 0;
   MPEG *file_mpeg = ( MPEG * ) video->file_mpeg;
   //
   if ( video->pict_struct == FRAME_PICTURE && video->secondfield )
   {
      video->secondfield = 0;
   }
   if ( !video->mpeg_2 )
   {
      video->current_repeat = video->repeat_count = 0;
   }
   mpeg_video_allocate_decoders( video, file_mpeg->cpus );
   for( i = 0; i < 3; i++ )
   {
      if ( video->pict_type == B_TYPE )
      {
         video->newframe[ i ] = video->auxframe[ i ];
      }
      else 
      {
         if ( !video->secondfield && !video->current_repeat )
         {
            unsigned char* tmp = video->oldrefframe[ i ];
            //
            video->oldrefframe[ i ] = video->refframe[ i ];
            video->refframe[ i ] = tmp;
         }
         video->newframe[ i ] = video->refframe[ i ];
      }
      if ( video->pict_struct == BOTTOM_FIELD )
      {
         video->newframe[ i ] += ( i == 0 ) ? video->coded_picture_width : video->chrom_width;
      }
   }
   if ( !video->current_repeat )
   {
      if ( !( video->skip_bframes && video->pict_type == B_TYPE ) || ( video->repeat_count >= ( 100 + 100 * video->skip_bframes ) ) )
         result = mpeg_video_get_macroblocks( video, framenum );
   }
   video->output_src = 0;
   if ( framenum > -1 && !result )
   {
      if ( video->pict_struct == FRAME_PICTURE || video->secondfield )
      {
         if ( video->pict_type == B_TYPE )
         {
            video->output_src = video->auxframe;
         }
         else
         {
            video->output_src = video->oldrefframe;
         }
      }
      else 
      {
         mpeg_video_display_second_field( video );
      }
   }
   if ( video->mpeg_2 )
   {
      video->current_repeat += 100;
   }
   if ( video->pict_struct != FRAME_PICTURE )
      video->secondfield = !video->secondfield;
   //
   return result;
}
/***************************************************************************/
static inline int mpeg_video_getseqhdr( mpeg_video_t *video )
{
   int i;
   int aspect_ratio, vbv_buffer_size;
   int constrained_parameters_flag;
   int load_intra_quantizer_matrix, load_non_intra_quantizer_matrix;
   //
   video->horizontal_size = mpeg_bits_getbits( video->vstream, 12 );
   video->vertical_size = mpeg_bits_getbits( video->vstream, 12 );
   aspect_ratio = mpeg_bits_getbits( video->vstream, 4 );
   video->framerate_code = mpeg_bits_getbits( video->vstream, 4 );
   video->bitrate = mpeg_bits_getbits( video->vstream, 18 );
   mpeg_bits_getbit_noptr( video->vstream ); 
   vbv_buffer_size = mpeg_bits_getbits( video->vstream, 10 );
   constrained_parameters_flag = mpeg_bits_getbit_noptr( video->vstream );
   video->frame_rate = mpeg_frame_rate_table[ video->framerate_code ];
   load_intra_quantizer_matrix = mpeg_bits_getbit_noptr( video->vstream );
   if ( load_intra_quantizer_matrix )
   {
      for( i = 0; i < 64; i++ )
         video->intra_quantizer_matrix[ video->mpeg_zigzag_scan_table[ i ] ] = mpeg_bits_getbyte_noptr( video->vstream );
   }
   else 
   {
      for( i = 0; i < 64; i++ )
         video->intra_quantizer_matrix[ i ] = mpeg_default_intra_quantizer_matrix[ i ];
   }
   load_non_intra_quantizer_matrix = mpeg_bits_getbit_noptr( video->vstream );
   if ( load_non_intra_quantizer_matrix )
   {
      for( i = 0; i < 64; i++ )
         video->non_intra_quantizer_matrix[ video->mpeg_zigzag_scan_table[ i ] ] = mpeg_bits_getbyte_noptr( video->vstream );
   }
   else 
   {
      for( i = 0; i < 64; i++ )
         video->non_intra_quantizer_matrix[ i ] = 16;
   }
   for( i = 0; i < 64; i++ )
   {
      video->chroma_intra_quantizer_matrix[ i ] = video->intra_quantizer_matrix[ i ];
      video->chroma_non_intra_quantizer_matrix[ i ] = video->non_intra_quantizer_matrix[ i ];
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_sequence_extension( mpeg_video_t *video )
{
   int prof_lev;
   int horizontal_size_extension, vertical_size_extension;
   int bit_rate_extension, vbv_buffer_size_extension, low_delay;
   int frame_rate_extension_n, frame_rate_extension_d;
   //
   video->mpeg_2 = 1;
   video->scalable_mode = SC_NONE; 
   prof_lev = mpeg_bits_getbyte_noptr( video->vstream );
   video->prog_seq = mpeg_bits_getbit_noptr( video->vstream );
   video->chroma_format = mpeg_bits_getbits( video->vstream, 2 );
   horizontal_size_extension = mpeg_bits_getbits( video->vstream, 2 );
   vertical_size_extension = mpeg_bits_getbits( video->vstream, 2 );
   bit_rate_extension = mpeg_bits_getbits( video->vstream, 12 );
   mpeg_bits_getbit_noptr( video->vstream );
   vbv_buffer_size_extension = mpeg_bits_getbyte_noptr( video->vstream );
   low_delay = mpeg_bits_getbit_noptr( video->vstream );
   frame_rate_extension_n = mpeg_bits_getbits( video->vstream, 2 );
   frame_rate_extension_d = mpeg_bits_getbits( video->vstream, 5 );
   video->horizontal_size = ( horizontal_size_extension << 12 ) | ( video->horizontal_size & 0x0fff );
   video->vertical_size = ( vertical_size_extension << 12 ) | ( video->vertical_size & 0x0fff );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_sequence_display_extension( mpeg_video_t *video )
{
   int colour_primaries = 0, transfer_characteristics = 0;
   int display_horizontal_size, display_vertical_size;
   int colour_description = mpeg_bits_getbit_noptr( video->vstream );
   //
   if ( colour_description )
   {
      colour_primaries = mpeg_bits_getbyte_noptr( video->vstream );
      transfer_characteristics = mpeg_bits_getbyte_noptr( video->vstream );
      video->matrix_coefficients = mpeg_bits_getbyte_noptr( video->vstream );
   }
   display_horizontal_size = mpeg_bits_getbits( video->vstream, 14 );
   mpeg_bits_getbit_noptr( video->vstream );
   display_vertical_size = mpeg_bits_getbits( video->vstream, 14 );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_quant_matrix_extension( mpeg_video_t *video )
{
   int i;
   int load_intra_quantiser_matrix, load_non_intra_quantiser_matrix;
   int load_chroma_intra_quantiser_matrix;
   int load_chroma_non_intra_quantiser_matrix;
   //
   if ( ( load_intra_quantiser_matrix = mpeg_bits_getbit_noptr( video->vstream ) ) != 0 )
   {
      for( i = 0; i < 64; i++ )
      {
         video->chroma_intra_quantizer_matrix[ video->mpeg_zigzag_scan_table[ i ] ] = video->intra_quantizer_matrix[ video->mpeg_zigzag_scan_table[ i ] ] = mpeg_bits_getbyte_noptr( video->vstream );
      }
   }
   if ( ( load_non_intra_quantiser_matrix = mpeg_bits_getbit_noptr( video->vstream ) ) != 0 )
   {
      for ( i = 0; i < 64; i++ )
      {
         video->chroma_non_intra_quantizer_matrix[ video->mpeg_zigzag_scan_table[ i ] ] = video->non_intra_quantizer_matrix[ video->mpeg_zigzag_scan_table[ i ] ] = mpeg_bits_getbyte_noptr( video->vstream );
      }
   }
   if ( ( load_chroma_intra_quantiser_matrix = mpeg_bits_getbit_noptr( video->vstream ) ) != 0 )
   {
      for( i = 0; i < 64; i++ )
         video->chroma_intra_quantizer_matrix[ video->mpeg_zigzag_scan_table[ i ] ] = mpeg_bits_getbyte_noptr( video->vstream );
   }
   if ( ( load_chroma_non_intra_quantiser_matrix = mpeg_bits_getbit_noptr( video->vstream ) ) != 0 )
   {
      for( i = 0; i < 64; i++ )
         video->chroma_non_intra_quantizer_matrix[ video->mpeg_zigzag_scan_table[ i ] ] = mpeg_bits_getbyte_noptr( video->vstream );
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_sequence_scalable_extension( mpeg_video_t *video )
{
   int layer_id;
   //
   video->scalable_mode = mpeg_bits_getbits( video->vstream, 2 ) + 1; 
   layer_id = mpeg_bits_getbits( video->vstream, 4 );
   if ( video->scalable_mode == SC_SPAT )
   {
      video->llw = mpeg_bits_getbits( video->vstream, 14 ); 
      mpeg_bits_getbit_noptr( video->vstream );
      video->llh = mpeg_bits_getbits( video->vstream, 14 ); 
      video->hm = mpeg_bits_getbits( video->vstream, 5 );
      video->hn = mpeg_bits_getbits( video->vstream, 5 );
      video->vm = mpeg_bits_getbits( video->vstream, 5 );
      video->vn = mpeg_bits_getbits( video->vstream, 5 );
   }
   if ( video->scalable_mode == SC_TEMP )
	  DebugMessage( "MpegPlayer : mpeg_video_sequence_scalable_extension: temporal scalability not implemented\n" );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_picture_display_extension( mpeg_video_t *video )
{
   int n, i;
   short frame_centre_horizontal_offset[ 3 ];
   short frame_centre_vertical_offset[ 3 ];
   //
   if ( video->prog_seq || video->pict_struct != FRAME_PICTURE )
      n = 1;
   else 
      n = video->repeatfirst ? 3 : 2;
   for( i = 0; i < n; i++ )
   {
      frame_centre_horizontal_offset[ i ] = ( short )mpeg_bits_getbits( video->vstream, 16 );
      mpeg_bits_getbit_noptr( video->vstream );
      frame_centre_vertical_offset[ i ] = ( short )mpeg_bits_getbits( video->vstream, 16 );
      mpeg_bits_getbit_noptr( video->vstream );
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_picture_coding_extension( mpeg_video_t *video )
{
   int chroma_420_type, composite_display_flag;
   int v_axis = 0, field_sequence = 0, sub_carrier = 0, burst_amplitude = 0, sub_carrier_phase = 0;
   //
   video->h_forw_r_size = mpeg_bits_getbits( video->vstream, 4 ) - 1;
   video->v_forw_r_size = mpeg_bits_getbits( video->vstream, 4 ) - 1;
   video->h_back_r_size = mpeg_bits_getbits( video->vstream, 4 ) - 1;
   video->v_back_r_size = mpeg_bits_getbits( video->vstream, 4 ) - 1;
   video->dc_prec = mpeg_bits_getbits( video->vstream, 2 );
   video->pict_struct = mpeg_bits_getbits( video->vstream, 2 );
   video->topfirst = mpeg_bits_getbit_noptr( video->vstream );
   video->frame_pred_dct = mpeg_bits_getbit_noptr( video->vstream );
   video->conceal_mv = mpeg_bits_getbit_noptr( video->vstream );
   video->qscale_type = mpeg_bits_getbit_noptr( video->vstream );
   video->intravlc = mpeg_bits_getbit_noptr( video->vstream );
   video->altscan = mpeg_bits_getbit_noptr( video->vstream );
   video->repeatfirst = mpeg_bits_getbit_noptr( video->vstream );
   chroma_420_type = mpeg_bits_getbit_noptr( video->vstream );
   video->prog_frame = mpeg_bits_getbit_noptr( video->vstream );
   if ( video->repeat_count > 100 )
      video->repeat_count = 0;
   video->repeat_count += 100;
   video->current_repeat = 0;
   if ( video->prog_seq )
   {
      if ( video->repeatfirst )
      {
         if ( video->topfirst )
            video->repeat_count += 200;
         else
            video->repeat_count += 100;
      }
   }
   else if ( video->prog_frame )
   {
      if ( video->repeatfirst )
      {
         video->repeat_count += 50;
      }
   }
   composite_display_flag = mpeg_bits_getbit_noptr( video->vstream );
   if ( composite_display_flag )
   {
      v_axis = mpeg_bits_getbit_noptr( video->vstream );
      field_sequence = mpeg_bits_getbits( video->vstream, 3 );
      sub_carrier = mpeg_bits_getbit_noptr( video->vstream );
      burst_amplitude = mpeg_bits_getbits( video->vstream, 7 );
      sub_carrier_phase = mpeg_bits_getbyte_noptr( video->vstream );
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_picture_spatial_scalable_extension( mpeg_video_t *video )
{
   video->pict_scal = 1; 
   video->lltempref = mpeg_bits_getbits( video->vstream, 10 );
   mpeg_bits_getbit_noptr( video->vstream );
   video->llx0 = mpeg_bits_getbits( video->vstream, 15 );
   if ( video->llx0 >= 16384 )
      video->llx0 -= 32768;
   mpeg_bits_getbit_noptr( video->vstream );
   video->lly0 = mpeg_bits_getbits( video->vstream, 15 );
   if ( video->lly0 >= 16384 )
      video->lly0 -= 32768;
   video->stwc_table_index = mpeg_bits_getbits( video->vstream, 2 );
   video->llprog_frame = mpeg_bits_getbit_noptr( video->vstream );
   video->llfieldsel = mpeg_bits_getbit_noptr( video->vstream );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_picture_temporal_scalable_extension( mpeg_video_t *video )
{
   DebugMessage( "MpegPlayer : mpeg_video_picture_temporal_scalable_extension: temporal scalability not supported\n" );
   //
   return 0;
}
/***************************************************************************/
static inline unsigned int mpeg_bits_next_startcode( mpeg_bits_t* stream )
{
   mpeg_bits_byte_align( stream );
   while( ( mpeg_bits_showbits32_noptr( stream ) >> 8 ) != MPEG3_PACKET_START_CODE_PREFIX && !mpeg_bits_eof( stream ) )
   {
      mpeg_bits_getbyte_noptr( stream );
   }
   //
   return mpeg_bits_showbits32_noptr( stream );
}
/***************************************************************************/
static inline int mpeg_video_ext_user_data( mpeg_video_t *video )
{
   int code = mpeg_bits_next_startcode( video->vstream );
   //
   while( ( code == MPEG3_EXT_START_CODE ) || ( code == MPEG3_USER_START_CODE && !mpeg_bits_eof( video->vstream ) ) )
   {
      mpeg_bits_refill( video->vstream );
      if ( code == MPEG3_EXT_START_CODE )
      {
         int ext_id = mpeg_bits_getbits( video->vstream, 4 );
         switch( ext_id )
         {
            case SEQ_ID:
               mpeg_video_sequence_extension( video );
               break;
            case DISP_ID:
               mpeg_video_sequence_display_extension( video );
               break;
            case QUANT_ID:
               mpeg_video_quant_matrix_extension( video );
               break;
            case SEQSCAL_ID:
               mpeg_video_sequence_scalable_extension( video );
               break;
            case PANSCAN_ID:
               mpeg_video_picture_display_extension( video );
               break;
            case CODING_ID:
               mpeg_video_picture_coding_extension( video );
               break;
            case SPATSCAL_ID:
               mpeg_video_picture_spatial_scalable_extension( video );
               break;
            case TEMPSCAL_ID:
               mpeg_video_picture_temporal_scalable_extension( video );
               break;
            default:
		   DebugMessage( "MpegPlayer : mpeg_video_ext_user_data: reserved extension start code ID %d\n", ext_id );
               break;
         }
      }
      code = mpeg_bits_next_startcode( video->vstream );
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_getgophdr( mpeg_video_t *video )
{
   int drop_flag, closed_gop, broken_link;
   //
   drop_flag = mpeg_bits_getbit_noptr( video->vstream );
   video->gop_timecode.hour = mpeg_bits_getbits( video->vstream, 5 );
   video->gop_timecode.minute = mpeg_bits_getbits( video->vstream, 6 );
   mpeg_bits_getbit_noptr( video->vstream );
   video->gop_timecode.second = mpeg_bits_getbits( video->vstream, 6 );
   video->gop_timecode.frame = mpeg_bits_getbits( video->vstream, 6 );
   closed_gop = mpeg_bits_getbit_noptr( video->vstream );
   broken_link = mpeg_bits_getbit_noptr( video->vstream );
   //
   return mpeg_bits_error( video->vstream );
}
/***************************************************************************/
static inline int mpeg_video_getpicturehdr( mpeg_video_t *video )
{
   int temp_ref, vbv_delay;
   //
   video->pict_scal = 0; 
   temp_ref = mpeg_bits_getbits( video->vstream, 10 );
   video->pict_type = mpeg_bits_getbits( video->vstream, 3 );
   vbv_delay = mpeg_bits_getbits( video->vstream, 16 );
   if ( video->pict_type == P_TYPE || video->pict_type == B_TYPE )
   {
      video->full_forw = mpeg_bits_getbit_noptr( video->vstream );
      video->forw_r_size = mpeg_bits_getbits( video->vstream, 3 ) - 1;
   }
   if ( video->pict_type == B_TYPE )
   {
      video->full_back = mpeg_bits_getbit_noptr( video->vstream );
      video->back_r_size = mpeg_bits_getbits( video->vstream, 3 ) - 1;
   }
   while( mpeg_bits_getbit_noptr( video->vstream ) && !mpeg_bits_eof( video->vstream ) )
      mpeg_bits_getbyte_noptr( video->vstream );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_get_header( mpeg_video_t *video, int dont_repeat )
{
   unsigned int code;
   //
   if ( video->repeat_count - video->current_repeat >= 100 && !dont_repeat )
   {
      return 0;
   }
   if ( dont_repeat )
   {
      video->repeat_count = 0;
      video->current_repeat = 0;
   }
   else
      video->repeat_count -= video->current_repeat;
   while( 1 )
   {
      code = mpeg_bits_next_startcode( video->vstream );
      if ( mpeg_bits_eof( video->vstream ) )
         return 1;
      if ( code != MPEG3_SEQUENCE_END_CODE ) mpeg_bits_refill( video->vstream );
      {
         switch( code )
         {
            case MPEG3_SEQUENCE_START_CODE:
               video->found_seqhdr = 1;
               mpeg_video_getseqhdr( video ); 
               mpeg_video_ext_user_data( video );
               break;
            case MPEG3_GOP_START_CODE:
               mpeg_video_getgophdr( video );
               mpeg_video_ext_user_data( video );
               break;
            case MPEG3_PICTURE_START_CODE:
               mpeg_video_getpicturehdr( video );
               mpeg_video_ext_user_data( video );
               if ( video->found_seqhdr )
                  return 0; 
               break;
            case MPEG3_SEQUENCE_END_CODE:
               mpeg_bits_refill( video->vstream );
               break;
            default:
               break;
         }
      }
   }
   //
   return 1; 
}
/***************************************************************************/
static inline int mpeg_video_ext_bit_info( mpeg_slice_buffer_t *slice_buffer )
{
   while( mpeg_slice_getbit( slice_buffer ) )
      mpeg_slice_getbyte( slice_buffer );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_getslicehdr( mpeg_slice_t *slice, mpeg_video_t *video )
{
   int slice_vertical_position_extension, intra_slice;
   int qs;
   //
   slice_vertical_position_extension = ( video->mpeg_2 && video->vertical_size > 2800 ) ? 
   mpeg_slice_getbits( slice->slice_buffer, 3 ) : 0;
   if ( video->scalable_mode == SC_DP )
      slice->pri_brk = mpeg_slice_getbits( slice->slice_buffer, 7 );
   qs = mpeg_slice_getbits( slice->slice_buffer, 5 );
   slice->quant_scale = video->mpeg_2 ? ( video->qscale_type ? mpeg_non_linear_mquant_table[ qs ] : ( qs << 1 ) ) : qs;
   if ( mpeg_slice_getbit( slice->slice_buffer ) )
   {
      intra_slice = mpeg_slice_getbit( slice->slice_buffer );
      mpeg_slice_getbits( slice->slice_buffer, 7 );
      mpeg_video_ext_bit_info( slice->slice_buffer );
   }
   else 
      intra_slice = 0;
   //
   return slice_vertical_position_extension;
}
/***************************************************************************/
#define W1                       2841 
#define W2                       2676 
#define W3                       2408 
#define W5                       1609 
#define W6                       1108 
#define W7                       565 
/***************************************************************************/
static inline int mpeg_video_idctrow( short *blk )
{
   int x0, x1, x2, x3, x4, x5, x6, x7, x8;
   //
   if ( !( ( x1 = blk[ 4 ]<<11 ) | ( x2 = blk[ 6 ] ) | ( x3 = blk[ 2 ] ) | ( x4 = blk[ 1 ] ) | ( x5 = blk[ 7 ] ) | ( x6 = blk[ 5 ] ) | ( x7 = blk[ 3 ] ) ) )
   {
      blk[ 0 ]=blk[ 1 ]=blk[ 2 ]=blk[ 3 ]=blk[ 4 ]=blk[ 5 ]=blk[ 6 ]=blk[ 7 ]=blk[ 0 ]<<3;
      return 1;
   }
   x0 = ( blk[ 0 ]<<11 ) + 128; 
   x8 = W7*( x4+x5 );
   x4 = x8 + ( W1-W7 )*x4;
   x5 = x8 - ( W1+W7 )*x5;
   x8 = W3*( x6+x7 );
   x6 = x8 - ( W3-W5 )*x6;
   x7 = x8 - ( W3+W5 )*x7;
   x8 = x0 + x1;
   x0 -= x1;
   x1 = W6*( x3+x2 );
   x2 = x1 - ( W2+W6 )*x2;
   x3 = x1 + ( W2-W6 )*x3;
   x1 = x4 + x6;
   x4 -= x6;
   x6 = x5 + x7;
   x5 -= x7;
   x7 = x8 + x3;
   x8 -= x3;
   x3 = x0 + x2;
   x0 -= x2;
   x2 = ( 181*( x4+x5 )+128 )>>8;
   x4 = ( 181*( x4-x5 )+128 )>>8;
   blk[ 0 ] = ( x7+x1 )>>8;
   blk[ 1 ] = ( x3+x2 )>>8;
   blk[ 2 ] = ( x0+x4 )>>8;
   blk[ 3 ] = ( x8+x6 )>>8;
   blk[ 4 ] = ( x8-x6 )>>8;
   blk[ 5 ] = ( x0-x4 )>>8;
   blk[ 6 ] = ( x3-x2 )>>8;
   blk[ 7 ] = ( x7-x1 )>>8;
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_idctcol( short *blk )
{
   int x0, x1, x2, x3, x4, x5, x6, x7, x8;
   //
   if ( !( ( x1 = ( blk[ 8 * 4 ]<<8 ) ) | ( x2 = blk[ 8 * 6 ] ) | ( x3 = blk[ 8 * 2 ] ) | ( x4 = blk[ 8*1 ] ) | ( x5 = blk[ 8 * 7 ] ) | ( x6 = blk[ 8 * 5 ] ) | ( x7 = blk[ 8 * 3 ] ) ) )
   {
      blk[ 8*0 ]=blk[ 8*1 ]=blk[ 8 * 2 ]=blk[ 8 * 3 ]=blk[ 8 * 4 ]=blk[ 8 * 5 ]=blk[ 8 * 6 ]=blk[ 8 * 7 ]=
      ( blk[ 8*0 ]+32 )>>6;
      return 1;
   }
   x0 = ( blk[ 8*0 ]<<8 ) + 8192;
   x8 = W7*( x4+x5 ) + 4;
   x4 = ( x8+( W1-W7 )*x4 )>>3;
   x5 = ( x8-( W1+W7 )*x5 )>>3;
   x8 = W3*( x6+x7 ) + 4;
   x6 = ( x8-( W3-W5 )*x6 )>>3;
   x7 = ( x8-( W3+W5 )*x7 )>>3;
   x8 = x0 + x1;
   x0 -= x1;
   x1 = W6*( x3+x2 ) + 4;
   x2 = ( x1-( W2+W6 )*x2 )>>3;
   x3 = ( x1+( W2-W6 )*x3 )>>3;
   x1 = x4 + x6;
   x4 -= x6;
   x6 = x5 + x7;
   x5 -= x7;
   x7 = x8 + x3;
   x8 -= x3;
   x3 = x0 + x2;
   x0 -= x2;
   x2 = ( 181 * ( x4 + x5 ) + 128 ) >> 8;
   x4 = ( 181 * ( x4 - x5 ) + 128 ) >> 8;
   blk[ 8 * 0 ] = ( x7 + x1 ) >> 14;
   blk[ 8 * 1 ] = ( x3 + x2 ) >> 14;
   blk[ 8 * 2 ] = ( x0 + x4 ) >> 14;
   blk[ 8 * 3 ] = ( x8 + x6 ) >> 14;
   blk[ 8 * 4 ] = ( x8 - x6 ) >> 14;
   blk[ 8 * 5 ] = ( x0 - x4 ) >> 14;
   blk[ 8 * 6 ] = ( x3 - x2 ) >> 14;
   blk[ 8 * 7 ] = ( x7 - x1 ) >> 14;
   //
   return 0;
}
/***************************************************************************/
static inline void mpeg_video_idct_conversion( short* block )
{
   int i;
   //
   for( i = 0; i < 8; i++ )
      mpeg_video_idctrow( block + 8 * i );
   for( i = 0; i < 8; i++ )
      mpeg_video_idctcol( block + i );
}
/***************************************************************************/
static inline int mpeg_video_get_macroblock_address( mpeg_slice_t *slice )
{
   int code = 0, val = 0;
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   //
   while( ( code = mpeg_slice_showbits( slice_buffer, 11 ) ) < 24 )
   {
      if ( code != 15 )
      {
         if ( code == 8 )
         {
            val += 33;
         }
         else 
         {
            slice->fault = 1;
            return 1;
         }
      }
      mpeg_slice_flushbits( slice_buffer, 11 );
   }
   if ( code >= 1024 )
   {
      mpeg_slice_flushbit( slice_buffer );
      return val + 1;
   }
   if ( code >= 128 )
   {
      code >>= 6;
      mpeg_slice_flushbits( slice_buffer, mpeg_MBAtab1[ code ].len );
      return val + mpeg_MBAtab1[ code ].val;
   }
   code -= 24;
   mpeg_slice_flushbits( slice_buffer, mpeg_MBAtab2[ code ].len );
   //
   return val + mpeg_MBAtab2[ code ].val;
}
/***************************************************************************/
static inline int mpeg_video_getsp_imb_type( mpeg_slice_t *slice )
{
   mpeg_slice_buffer_t *slice_buffer = slice_buffer;
   unsigned int code = mpeg_slice_showbits( slice_buffer, 4 );
   //
   if ( !code )
   {
      slice->fault = 1;
      return 0;
   }
   mpeg_slice_flushbits( slice_buffer, mpeg_spIMBtab[ code ].len );
   //
   return mpeg_spIMBtab[ code ].val;
}
/***************************************************************************/
static inline int mpeg_video_getsp_pmb_type( mpeg_slice_t *slice )
{
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   int code = mpeg_slice_showbits( slice_buffer, 7 );
   //
   if ( code < 2 )
   {
      slice->fault = 1;
      return 0;
   }
   if ( code >= 16 )
   {
      code >>= 3;
      mpeg_slice_flushbits( slice_buffer, mpeg_spPMBtab0[ code ].len );
      //
      return mpeg_spPMBtab0[ code ].val;
   }
   mpeg_slice_flushbits( slice_buffer, mpeg_spPMBtab1[ code ].len );
   //
   return mpeg_spPMBtab1[ code ].val;
}
/***************************************************************************/
static inline int mpeg_video_getsp_bmb_type( mpeg_slice_t *slice )
{
   mpeg_VLCtab_t *p;
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   int code = mpeg_slice_showbits9( slice_buffer );
   //
   if ( code >= 64 ) 
      p = &mpeg_spBMBtab0[ ( code >> 5 ) - 2 ];
   else if ( code >= 16 ) 
      p = &mpeg_spBMBtab1[ ( code >> 2 ) - 4 ];
   else if ( code >= 8 ) 
      p = &mpeg_spBMBtab2[ code - 8 ];
   else 
   {
      slice->fault = 1;
      return 0;
   }
   mpeg_slice_flushbits( slice_buffer, p->len );
   //
   return p->val;
}
/***************************************************************************/
static inline int mpeg_video_get_imb_type( mpeg_slice_t *slice )
{
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   //
   if ( mpeg_slice_getbit( slice_buffer ) )
   {
      return 1;
   }
   if ( !mpeg_slice_getbit( slice_buffer ) )
   {
      slice->fault = 1;
   }
   return 17;
}
/***************************************************************************/
static inline int mpeg_video_get_pmb_type( mpeg_slice_t *slice )
{
   int code;
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   //
   if ( ( code = mpeg_slice_showbits( slice_buffer, 6 ) ) >= 8 )
   {
      code >>= 3;
      mpeg_slice_flushbits( slice_buffer, mpeg_PMBtab0[ code ].len );
      //
      return mpeg_PMBtab0[ code ].val;
   }
   if ( code == 0 )
   {
      slice->fault = 1;
      return 0;
   }
   mpeg_slice_flushbits( slice_buffer, mpeg_PMBtab1[ code ].len );
   //
   return mpeg_PMBtab1[ code ].val;
}
/***************************************************************************/
static inline int mpeg_video_get_bmb_type( mpeg_slice_t *slice )
{
   int code;
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   //
   if ( ( code = mpeg_slice_showbits( slice_buffer, 6 ) ) >= 8 )
   {
      code >>= 2;
      mpeg_slice_flushbits( slice_buffer, mpeg_BMBtab0[ code ].len );
      //
      return mpeg_BMBtab0[ code ].val;
   }
   if ( code == 0 )
   {
      slice->fault = 1;
      return 0;
   }
   mpeg_slice_flushbits( slice_buffer, mpeg_BMBtab1[ code ].len );
   //
   return mpeg_BMBtab1[ code ].val;
}
/***************************************************************************/
static inline int mpeg_video_get_dmb_type( mpeg_slice_t *slice )
{
   if ( !mpeg_slice_getbit( slice->slice_buffer ) )
   {
      slice->fault=1;
   }
   //
   return 1;
}
/***************************************************************************/
static inline int mpeg_video_get_snrmb_type( mpeg_slice_t *slice )
{
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   int code = mpeg_slice_showbits( slice_buffer, 3 );
   //
   if ( code == 0 )
   {
      slice->fault = 1;
      return 0;
   }
   mpeg_slice_flushbits( slice_buffer, mpeg_SNRMBtab[ code ].len );
   //
   return mpeg_SNRMBtab[ code ].val;
}
/***************************************************************************/
static inline int mpeg_video_get_mb_type( mpeg_slice_t *slice, mpeg_video_t *video )
{
   if ( video->scalable_mode == SC_SNR )
   {
      return mpeg_video_get_snrmb_type( slice );
   }
   else
   {
      switch( video->pict_type )
      {
         case I_TYPE:
            return video->pict_scal ? mpeg_video_getsp_imb_type( slice ) : mpeg_video_get_imb_type( slice );
         case P_TYPE:
            return video->pict_scal ? mpeg_video_getsp_pmb_type( slice ) : mpeg_video_get_pmb_type( slice );
         case B_TYPE:
            return video->pict_scal ? mpeg_video_getsp_bmb_type( slice ) : mpeg_video_get_bmb_type( slice );
         case D_TYPE:
            return mpeg_video_get_dmb_type( slice );
         default: 
            break;
      }
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_macroblock_modes( mpeg_slice_t *slice, mpeg_video_t *video, int *pmb_type, int *pstwtype, int *pstwclass, int *pmotion_type, int *pmv_count, int *pmv_format, int *pdmv, int *pmvscale, int *pdct_type )
{
   int mb_type;
   int stwtype, stwcode, stwclass;
   int motion_type = 0, mv_count, mv_format, dmv, mvscale;
   int dct_type;
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   static unsigned char stwc_table[ 3 ][ 4 ] = {{6, 3, 7, 4}, {2, 1, 5, 4}, {2, 5, 7, 4} };
   static unsigned char stwclass_table[ 9 ] = {0, 1, 2, 1, 1, 2, 3, 3, 4};
   //
   mb_type = mpeg_video_get_mb_type( slice, video );
   if ( slice->fault )
      return 1;
   if ( mb_type & MB_WEIGHT )
   {
      if ( video->stwc_table_index == 0 )
         stwtype = 4;
      else
      {
         stwcode = mpeg_slice_getbits2( slice_buffer );
         stwtype = stwc_table[ video->stwc_table_index - 1 ][ stwcode ];
      }
   }
   else
      stwtype = ( mb_type & MB_CLASS4 ) ? 8 : 0;
   stwclass = stwclass_table[ stwtype ];
   if ( mb_type & ( MB_FORWARD | MB_BACKWARD ) )
   {
      if ( video->pict_struct == FRAME_PICTURE )
      {
         motion_type = video->frame_pred_dct ? MC_FRAME : mpeg_slice_getbits2( slice_buffer );
      }
      else 
      {
         motion_type = mpeg_slice_getbits2( slice_buffer );
      }
   }
   else if ( ( mb_type & MB_INTRA ) && video->conceal_mv )
   {
      motion_type = ( video->pict_struct == FRAME_PICTURE ) ? MC_FRAME : MC_FIELD;
   }
   if ( video->pict_struct == FRAME_PICTURE )
   {
      mv_count = ( motion_type == MC_FIELD && stwclass < 2 ) ? 2 : 1;
      mv_format = ( motion_type == MC_FRAME ) ? MV_FRAME : MV_FIELD;
   }
   else
   {
      mv_count = ( motion_type == MC_16X8 ) ? 2 : 1;
      mv_format = MV_FIELD;
   }
   dmv = ( motion_type == MC_DMV ); 
   mvscale = ( ( mv_format == MV_FIELD ) && ( video->pict_struct == FRAME_PICTURE ) );
   dct_type = ( video->pict_struct == FRAME_PICTURE ) && ( !video->frame_pred_dct ) && ( mb_type & ( MB_PATTERN | MB_INTRA ) ) ? mpeg_slice_getbit( slice_buffer ) : 0;
   *pmb_type = mb_type;
   *pstwtype = stwtype;
   *pstwclass = stwclass;
   *pmotion_type = motion_type;
   *pmv_count = mv_count;
   *pmv_format = mv_format;
   *pdmv = dmv;
   *pmvscale = mvscale;
   *pdct_type = dct_type;
   //
   return 0;
}
/***************************************************************************/
static inline void mpeg_video_calc_mv( int *pred, int r_size, int motion_code, int motion_r, int full_pel_vector )
{
   int lim = 16 << r_size;
   int vec = full_pel_vector ? ( *pred >> 1 ) : ( *pred );
   //
   if ( motion_code > 0 )
   {
      vec += ( ( motion_code - 1 ) << r_size ) + motion_r + 1;
      if ( vec >= lim )
         vec -= lim + lim;
   }
   else if ( motion_code < 0 )
   {
      vec -= ( ( -motion_code - 1 ) << r_size ) + motion_r + 1;
      if ( vec < -lim )
         vec += lim + lim;
   }
   *pred = full_pel_vector ? ( vec << 1 ) : vec;
}
/***************************************************************************/
static inline void mpeg_video_calc_dmv( mpeg_video_t *video, int DMV[][ 2 ], int *dmvector, int mvx, int mvy )
{
   if ( video->pict_struct == FRAME_PICTURE )
   {
      if ( video->topfirst )
      {
         DMV[ 0 ][ 0 ] = ( ( mvx + ( mvx>0 ) ) >> 1 ) + dmvector[ 0 ];
         DMV[ 0 ][ 1 ] = ( ( mvy + ( mvy>0 ) ) >> 1 ) + dmvector[ 1 ] - 1;
         DMV[ 1 ][ 0 ] = ( ( 3 * mvx + ( mvx > 0 ) ) >> 1 ) + dmvector[ 0 ];
         DMV[ 1 ][ 1 ] = ( ( 3 * mvy + ( mvy > 0 ) ) >> 1 ) + dmvector[ 1 ] + 1;
      }
      else 
      {
         DMV[ 0 ][ 0 ] = ( ( 3 * mvx + ( mvx>0 ) ) >> 1 ) + dmvector[ 0 ];
         DMV[ 0 ][ 1 ] = ( ( 3 * mvy + ( mvy>0 ) ) >> 1 ) + dmvector[ 1 ] - 1;
         DMV[ 1 ][ 0 ] = ( ( mvx + ( mvx>0 ) ) >> 1 ) + dmvector[ 0 ];
         DMV[ 1 ][ 1 ] = ( ( mvy + ( mvy>0 ) ) >> 1 ) + dmvector[ 1 ] + 1;
      }
   }
   else 
   {
      DMV[ 0 ][ 0 ] = ( ( mvx + ( mvx > 0 ) ) >> 1 ) + dmvector[ 0 ];
      DMV[ 0 ][ 1 ] = ( ( mvy + ( mvy > 0 ) ) >> 1 ) + dmvector[ 1 ];
      if ( video->pict_struct == TOP_FIELD )
         DMV[ 0 ][ 1 ]--;
      else 
         DMV[ 0 ][ 1 ]++;
   }
}
/***************************************************************************/
static inline int mpeg_video_get_mv( mpeg_slice_t *slice )
{
   int code;
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   //
   if ( mpeg_slice_getbit( slice_buffer ) )
   {
      return 0;
   }
   if ( ( code = mpeg_slice_showbits9( slice_buffer ) ) >= 64 )
   {
      code >>= 6;
      mpeg_slice_flushbits( slice_buffer, mpeg_MVtab0[ code ].len );
      //
      return mpeg_slice_getbit( slice_buffer ) ? -mpeg_MVtab0[ code ].val : mpeg_MVtab0[ code ].val;
   }
   if ( code >= 24 )
   {
      code >>= 3;
      mpeg_slice_flushbits( slice_buffer, mpeg_MVtab1[ code ].len );
      //
      return mpeg_slice_getbit( slice_buffer ) ? -mpeg_MVtab1[ code ].val : mpeg_MVtab1[ code ].val;
   }
   if ( ( code -= 12 ) < 0 )
   {
      slice->fault = 1;
      return 1;
   }
   mpeg_slice_flushbits( slice_buffer, mpeg_MVtab2[ code ].len );
   //
   return mpeg_slice_getbit( slice_buffer ) ? -mpeg_MVtab2[ code ].val : mpeg_MVtab2[ code ].val;
}
/***************************************************************************/
static inline int mpeg_video_get_dmv( mpeg_slice_t *slice )
{
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   if ( mpeg_slice_getbit( slice_buffer ) )
   {
      return mpeg_slice_getbit( slice_buffer ) ? -1 : 1;
   }
   else 
   {
      return 0;
   }
}
/***************************************************************************/
static inline void mpeg_video_motion_vector( mpeg_slice_t *slice, mpeg_video_t *video, int *PMV, int *dmvector, int h_r_size, int v_r_size, int dmv, int mvscale, int full_pel_vector )
{
   int motion_r;
   int motion_code = mpeg_video_get_mv( slice );
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   //
   if ( slice->fault )
      return;
   motion_r = ( h_r_size != 0 && motion_code != 0 ) ? mpeg_slice_getbits( slice_buffer, h_r_size ) : 0;
   mpeg_video_calc_mv( &PMV[ 0 ], h_r_size, motion_code, motion_r, full_pel_vector );
   if ( dmv )
      dmvector[ 0 ] = mpeg_video_get_dmv( slice );
   motion_code = mpeg_video_get_mv( slice );
   if ( slice->fault )
      return;
   motion_r = ( v_r_size != 0 && motion_code != 0 ) ? mpeg_slice_getbits( slice_buffer, v_r_size ) : 0;
   if ( mvscale )
      PMV[ 1 ] >>= 1; 
   mpeg_video_calc_mv( &PMV[ 1 ], v_r_size, motion_code, motion_r, full_pel_vector );
   if ( mvscale )
      PMV[ 1 ] <<= 1;
   if ( dmv )
      dmvector[ 1 ] = mpeg_video_get_dmv( slice );
}
/***************************************************************************/
static inline int mpeg_video_motion_vectors( mpeg_slice_t *slice, mpeg_video_t *video, int PMV[ 2 ][ 2 ][ 2 ], int dmvector[ 2 ], int mv_field_sel[ 2 ][ 2 ], int s, int mv_count, int mv_format, int h_r_size, int v_r_size, int dmv, int mvscale )
{
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   //
   if ( mv_count == 1 )
   {
      if ( mv_format == MV_FIELD && !dmv )
      {
         mv_field_sel[ 1 ][ s ] = mv_field_sel[ 0 ][ s ] = mpeg_slice_getbit( slice_buffer );
      }
      mpeg_video_motion_vector( slice, video, PMV[ 0 ][ s ], dmvector, h_r_size, v_r_size, dmv, mvscale, 0 );
      if ( slice->fault )
         return 1;
      PMV[ 1 ][ s ][ 0 ] = PMV[ 0 ][ s ][ 0 ];
      PMV[ 1 ][ s ][ 1 ] = PMV[ 0 ][ s ][ 1 ];
   }
   else 
   {
      mv_field_sel[ 0 ][ s ] = mpeg_slice_getbit( slice_buffer );
      mpeg_video_motion_vector( slice, video, PMV[ 0 ][ s ], dmvector, h_r_size, v_r_size, dmv, mvscale, 0 );
      if ( slice->fault )
         return 1;
      mv_field_sel[ 1 ][ s ] = mpeg_slice_getbit( slice_buffer );
      mpeg_video_motion_vector( slice, video, PMV[ 1 ][ s ], dmvector, h_r_size, v_r_size, dmv, mvscale, 0 );
      if ( slice->fault )
         return 1;
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_initdecoder( mpeg_video_t *video )
{
   int blk_cnt_tab[ 3 ] = {6, 8, 12};
   int cc;
   int i;
   long size[ 4 ], padding[ 2 ];
   //
   if ( !video->mpeg_2 )
   {
      video->prog_seq = 1;
      video->prog_frame = 1;
      video->pict_struct = FRAME_PICTURE;
      video->frame_pred_dct = 1;
      video->chroma_format = CHROMA420;
      video->matrix_coefficients = 5;
   }
   video->mb_width = ( video->horizontal_size + 15 ) / 16;
   video->mb_height = ( video->mpeg_2 && !video->prog_seq ) ? ( 2 * ( ( video->vertical_size + 31 ) / 32 ) ) : ( ( video->vertical_size + 15 ) / 16 );
   video->coded_picture_width = 16 * video->mb_width;
   video->coded_picture_height = 16 * video->mb_height;
   video->chrom_width = ( video->chroma_format == CHROMA444 ) ? video->coded_picture_width : ( video->coded_picture_width >> 1 );
   video->chrom_height = ( video->chroma_format != CHROMA420 ) ? video->coded_picture_height : ( video->coded_picture_height >> 1 );
   video->blk_cnt = blk_cnt_tab[ video->chroma_format - 1 ];
   padding[ 0 ] = 16 * video->coded_picture_width;
   size[ 0 ] = video->coded_picture_width * video->coded_picture_height + padding[ 0 ] * 2;
   padding[ 1 ] = 16 * video->chrom_width;
   size[ 1 ] = video->chrom_width * video->chrom_height + 2 * padding[ 1 ];
   size[ 2 ] = ( video->llw * video->llh );
   size[ 3 ] = ( video->llw * video->llh ) / 4;
   video->yuv_buffer[ 0 ] = ( unsigned char* )calloc( 1, ( size[ 0 ] + padding[ 0 ] ) + 2 * ( size[ 1 ] + padding[ 1 ] ) );
   video->yuv_buffer[ 1 ] = ( unsigned char* )calloc( 1, ( size[ 0 ] + padding[ 0 ] ) + 2 * ( size[ 1 ] + padding[ 1 ] ) );
   video->yuv_buffer[ 2 ] = ( unsigned char* )calloc( 1, ( size[ 0 ] + padding[ 0 ] ) + 2 * ( size[ 1 ] + padding[ 1 ] ) );
   if ( video->scalable_mode == SC_SPAT )
   {
      video->yuv_buffer[ 3 ] = ( unsigned char* )calloc( 1, size[ 2 ] + 2 * size[ 3 ] );
      video->yuv_buffer[ 4 ] = ( unsigned char* )calloc( 1, size[ 2 ] + 2 * size[ 3 ] );
   }
   for( cc = 0; cc < 3; cc++ )
   {
      video->llframe0[ cc ] = 0;
      video->llframe1[ cc ] = 0;
      video->newframe[ cc ] = 0;
   }
   video->refframe[ 0 ] = video->yuv_buffer[ 0 ];
   video->oldrefframe[ 0 ] = video->yuv_buffer[ 1 ];
   video->auxframe[ 0 ] = video->yuv_buffer[ 2 ];
   video->refframe[ 2 ] = video->yuv_buffer[ 0 ] + size[ 0 ] + padding[ 0 ];
   video->oldrefframe[ 2 ] = video->yuv_buffer[ 1 ] + size[ 0 ] + padding[ 0 ];
   video->auxframe[ 2 ] = video->yuv_buffer[ 2 ] + size[ 0 ] + padding[ 0 ];
   video->refframe[ 1 ] = video->yuv_buffer[ 0 ] + size[ 0 ] + padding[ 0 ] + size[ 1 ] + padding[ 1 ];
   video->oldrefframe[ 1 ] = video->yuv_buffer[ 1 ] + size[ 0 ] + padding[ 0 ] + size[ 1 ] + padding[ 1 ];
   video->auxframe[ 1 ] = video->yuv_buffer[ 2 ] + size[ 0 ] + padding[ 0 ] + size[ 1 ] + padding[ 1 ];
   if ( video->scalable_mode == SC_SPAT )
   {
      video->llframe0[ 0 ] = video->yuv_buffer[ 3 ] + padding[ 0 ] 	 ;
      video->llframe1[ 0 ] = video->yuv_buffer[ 4 ] + padding[ 0 ] 	 ;
      video->llframe0[ 2 ] = video->yuv_buffer[ 3 ] + padding[ 1 ] + size[ 2 ]	 ;
      video->llframe1[ 2 ] = video->yuv_buffer[ 4 ] + padding[ 1 ] + size[ 2 ]	 ;
      video->llframe0[ 1 ] = video->yuv_buffer[ 3 ] + padding[ 1 ] + size[ 2 ] + size[ 3 ];
      video->llframe1[ 1 ] = video->yuv_buffer[ 4 ] + padding[ 1 ] + size[ 2 ] + size[ 3 ];
   }
   video->cr_to_r = ( long * ) calloc( 1, sizeof( long ) * 256 );
   video->cr_to_g = ( long * ) calloc( 1, sizeof( long ) * 256 );
   video->cb_to_g = ( long * ) calloc( 1, sizeof( long ) * 256 );
   video->cb_to_b = ( long * ) calloc( 1, sizeof( long ) * 256 );
   video->cr_to_r_ptr = video->cr_to_r + 128;
   video->cr_to_g_ptr = video->cr_to_g + 128;
   video->cb_to_g_ptr = video->cb_to_g + 128;
   video->cb_to_b_ptr = video->cb_to_b + 128;
   for( i = -128; i < 128; i++ )
   {
      video->cr_to_r_ptr[ i ] = ( long )( 1.371 * 65536 * i );
      video->cr_to_g_ptr[ i ] = ( long )( -0.698 * 65536 * i );
      video->cb_to_g_ptr[ i ] = ( long )( -0.336 * 65536 * i );
      video->cb_to_b_ptr[ i ] = ( long )( 1.732 * 65536 * i );
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_deletedecoder( mpeg_video_t *video )
{
   free( video->yuv_buffer[ 0 ] );
   free( video->yuv_buffer[ 1 ] );
   free( video->yuv_buffer[ 2 ] );
   if ( video->llframe0[ 0 ] )
   {
      free( video->yuv_buffer[ 3 ] );
      free( video->yuv_buffer[ 4 ] );
   }
   free( video->cr_to_r );
   free( video->cr_to_g );
   free( video->cb_to_g );
   free( video->cb_to_b );
   //
   return 0;
}
/***************************************************************************/
static inline void mpeg_video_init_scantables( mpeg_video_t *video )
{
   video->mpeg_zigzag_scan_table = mpeg_zig_zag_scan_nommx;
   video->mpeg_alternate_scan_table = mpeg_alternate_scan_nommx;
}
/***************************************************************************/
static inline mpeg_video_t* mpeg_video_allocate_struct( MPEG *file_mpeg, mpeg_vtrack_t *track )
{
   mpeg_video_t *video = ( mpeg_video_t * ) calloc( 1, sizeof( mpeg_video_t ) );
   video->file_mpeg = file_mpeg;
   video->track = track;
   video->vstream = mpeg_bits_new_stream( file_mpeg, track->demuxer );
   video->last_number = -1;
   video->framenum = -1;
   video->percentage_seek = -1;
   video->frame_seek = -1;
   mpeg_video_init_scantables( video );
   mpeg_video_init_output();
   //
   return video;
}
/***************************************************************************/
static inline int mpeg_video_delete_struct( mpeg_video_t *video )
{
   int i;
   //
   mpeg_bits_delete_stream( video->vstream );
   if ( video->x_table )
   {
      free( video->x_table );
      free( video->y_table );
   }
   if ( video->total_slice_decoders )
   {
      for( i = 0; i < video->total_slice_decoders; i++ )
         mpeg_delete_slice_decoder( &( video->slice_decoders[ i ] ) );
   }
   for( i = 0; i < video->slice_buffers_initialized; i++ )
      mpeg_delete_slice_buffer( &( video->slice_buffers[ i ] ) );
   free( video );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_read_frame_backend( mpeg_video_t *video, int skip_bframes )
{
   int result = 0;
   //
   if ( mpeg_bits_eof( video->vstream ) )
      return 1;
   else
      result = mpeg_video_get_header( video, 0 );
   //
   video->skip_bframes = skip_bframes;
   if ( !result )
   {
      result = mpeg_video_getpicture( video, video->framenum );
      if ( !result )
      {
         video->last_number = video->framenum;
         video->framenum++;
      }
   }
   //
   return result;
}
/***************************************************************************/
static inline int* mpeg_video_get_scaletable( int input_w, int output_w )
{
   int *result = ( int * ) calloc( 1, sizeof( int ) * output_w );
   float i;
   float scale = ( float )input_w / output_w;
   //
   for( i = 0; i < output_w; i++ )
   {
      result[ ( int )i ] = ( int )( scale * i );
   }
   //
   return result;
}
/***************************************************************************/
static inline int mpeg_video_get_firstframe( mpeg_video_t *video )
{
   int result = 0;
   //
   if ( video->framenum < 0 )
   {
      video->repeat_count = video->current_repeat = 0;
      result = mpeg_video_read_frame_backend( video, 0 );
      mpeg_bits_seek_byte( video->vstream, 0 );
      mpeg_video_match_refframes( video );
   }
   //
   return result;
}
/***************************************************************************/
static inline mpeg_video_t* mpeg_video_new( MPEG *file_mpeg, mpeg_vtrack_t *track )
{
   mpeg_video_t *video;
   int result = 0;
   //
   video = mpeg_video_allocate_struct( file_mpeg, track );
   result = mpeg_video_get_header( video, 1 );
   if ( !result )
   {
      int hour, minute, second, frame;
      mpeg_video_initdecoder( video );
      video->decoder_initted = 1;
      track->width = video->horizontal_size;
      track->height = video->vertical_size;
      track->frame_rate = video->frame_rate;
      if ( file_mpeg->is_video_stream )
      {
         mpeg_bits_seek_start( video->vstream );
         result = mpeg_video_next_code( video->vstream, MPEG3_GOP_START_CODE );
         if ( !result )
         {
            mpeg_bits_getbits( video->vstream, 32 );
            result = mpeg_video_getgophdr( video );
         }
         hour = video->gop_timecode.hour;
         minute = video->gop_timecode.minute;
         second = video->gop_timecode.second;
         frame = video->gop_timecode.frame;
         video->first_frame = ( long )( hour * 3600 * video->frame_rate + minute * 60 * video->frame_rate +second * video->frame_rate +frame );
         video->frames_per_gop = 16;
         mpeg_bits_seek_end( video->vstream );
         mpeg_bits_start_reverse( video->vstream );
         result = mpeg_video_prev_code( video->vstream, MPEG3_GOP_START_CODE );
         mpeg_bits_start_forward( video->vstream );
         mpeg_bits_getbits( video->vstream, 8 );
         if ( !result )
            result = mpeg_video_getgophdr( video );
         hour = video->gop_timecode.hour;
         minute = video->gop_timecode.minute;
         second = video->gop_timecode.second;
         frame = video->gop_timecode.frame;
         video->last_frame = ( long )( hour * 3600 * video->frame_rate + minute * 60 * video->frame_rate +second * video->frame_rate +frame );
	 while( !result )
         {
            result = mpeg_video_next_code( video->vstream, MPEG3_PICTURE_START_CODE );
            if ( !result )
            {
               mpeg_bits_getbyte_noptr( video->vstream );
               video->last_frame++;
            }
         }
         track->total_frames = video->last_frame - video->first_frame + 1;
         mpeg_bits_seek_start( video->vstream );
      }
      else
      {
         video->first_frame = 0;
         track->total_frames = video->last_frame = ( long )( mpeg_demux_length( video->vstream->demuxer ) * video->frame_rate );
         video->first_frame = 0;
      }
      video->maxframe = track->total_frames;
      mpeg_bits_seek_start( video->vstream );
   }
   else
   {
      mpeg_video_delete( video );
      video = 0;
   }
   //
   return video;
}
/***************************************************************************/
static inline int mpeg_video_delete( mpeg_video_t *video )
{
   if ( video->decoder_initted )
   {
      mpeg_video_deletedecoder( video );
   }
   mpeg_video_delete_struct( video );
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_read_frame( mpeg_video_t *video, long frame_number, BITMAP *bmp )
{
   int result = 0;
   //
   video->want_yvu = 0;
   video->out_w = bmp->w;
   video->out_h = bmp->h;
   video->in_w = bmp->w;
   video->in_h = bmp->h;
   video->in_x = 0;
   video->in_y = 0;
   if ( !video->x_table )
   {
      video->x_table = mpeg_video_get_scaletable( video->in_w, video->out_w );
      video->y_table = mpeg_video_get_scaletable( video->in_h, video->out_h );
   }
   mpeg_video_get_firstframe( video );
   if ( !result )
      result = mpeg_video_seek( video );
   if ( !result )
      result = mpeg_video_read_frame_backend( video, 0 );
   if ( video->output_src )
      mpeg_video_present_frame( video, bmp );
   video->percentage_seek = -1;
   //
   return result;
}
/***************************************************************************/
static inline int mpeg_video_ditherframe( mpeg_video_t *video, unsigned char **src, BITMAP *bmp )
{
   int h = 0;
   unsigned char *y_in, *cb_in, *cr_in;
   long y_l, r_l, b_l, g_l;
   unsigned char *clipArray_ptr = gClipArray_ptr;
   int w = -1;
   //
   for( h = 0; h < video->out_h; h++ )
   {
      y_in = &src[ 0 ][ ( video->y_table[ h ] + video->in_y ) * video->coded_picture_width ] + video->in_x;
      cb_in = &src[ 1 ][ ( ( video->y_table[ h ] + video->in_y ) >> 1 ) * video->chrom_width ] + ( video->in_x >> 2 );
      cr_in = &src[ 2 ][ ( ( video->y_table[ h ] + video->in_y ) >> 1 ) * video->chrom_width ] + ( video->in_x >> 1 );
      for( w = 0; w < video->horizontal_size; w++ )
      {
	 y_l = *y_in++;
	 y_l <<= 16; r_l = ( y_l + video->cr_to_r[ *cr_in ] ) >> 16;
	 g_l = ( y_l + video->cr_to_g[ *cr_in ] + video->cb_to_g[ *cb_in ] ) >> 16;
         b_l = ( y_l + video->cb_to_b[ *cb_in ] ) >> 16;
         _putpixel16( bmp, w, h, makecol16( clipArray_ptr[ r_l ], clipArray_ptr[ g_l ], clipArray_ptr[ b_l ] ) );
         if ( w & 1 )
         {
            cr_in++;
            cb_in++;
         }
      }
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_ditherframe444( mpeg_video_t *video, unsigned char *src[] )
{
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_dithertop( mpeg_video_t *video, unsigned char *src[], BITMAP *bmp )
{
   return mpeg_video_ditherframe( video, src, bmp );
}
/***************************************************************************/
static inline int mpeg_video_dithertop444( mpeg_video_t *video, unsigned char *src[] )
{
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_ditherbot( mpeg_video_t *video, unsigned char *src[] )
{
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_ditherbot444( mpeg_video_t *video, unsigned char *src[] )
{
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_init_output( void )
{
   int i, value;
   //
   for( i = 0; i < 256; i++ )
   {
      value = ( int )( 1.1644 * i - 255 * 0.0627 + 0.5 );
      if ( value < 0 )
         value = 0;
      else if ( value > 255 )
         value = 255;
      mpeg_601_to_rgb[ i ] = value;
   }
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_present_frame( mpeg_video_t *video, BITMAP *bmp )
{
   int i, j, h;
   unsigned char **src = video->output_src;
   //
   if ( doClippingArrays )
   {
      for( h=-512;h<=512;h++ )
      {
         gClipArray_ptr[ h ]=CLIP( h );
         gClipArray_ptr16r[ h ]=( CLIP( h ) & 0xf8 ) << 7;
         gClipArray_ptr16g[ h ]=( CLIP( h ) & 0xf8 ) << 2;
         gClipArray_ptr16b[ h ]=( CLIP( h ) & 0xf8 ) >> 3;
         if ( gClipArray_ptr[ h ] == 0x00 ) 
            gClipArray_ptr[ h ] = 0x01;
         if ( gClipArray_ptr16b[ h ] == 0x00 ) 
            gClipArray_ptr16b[ h ] = 0x01;
      }
      doClippingArrays = 0;
   }
   if ( video->want_yvu )
   {
      long size[ 2 ];
      long offset[ 2 ];
      if ( !video->y_output )
         return 0;
      if ( video->in_x == 0 && video->in_w >= video->coded_picture_width )
      {
         size[ 0 ] = video->coded_picture_width * video->in_h;
         size[ 1 ] = video->chrom_width * ( int )( ( float )video->in_h / 2 + 0.5 );
         offset[ 0 ] = video->coded_picture_width * video->in_y;
         offset[ 1 ] = video->chrom_width * ( int )( ( float )video->in_y / 2 + 0.5 );
         memcpy( video->y_output, src[ 0 ] + offset[ 0 ], size[ 0 ] );
         memcpy( video->u_output, src[ 1 ] + offset[ 1 ], size[ 1 ] );
         memcpy( video->v_output, src[ 2 ] + offset[ 1 ], size[ 1 ] );
      }
      else
      {
         for( i = 0, j = video->in_y; i < video->in_h; i++, j++ )
         {
			memcpy( video->y_output + i * video->in_w, src[ 0 ] + j * video->coded_picture_width + video->in_x, video->in_w );
			memcpy( video->u_output + i * video->in_w / 4, src[ 1 ] + j * video->chrom_width / 2 + video->in_x / 4, video->in_w / 4 );
			memcpy( video->v_output + i * video->in_w / 4, src[ 2 ] + j * video->chrom_width / 2 + video->in_x / 4, video->in_w / 4 );
		 }
	  }
	  return 0;
   }
   if ( video->prog_seq )
   {
	  if ( video->chroma_format != CHROMA444 )
	  {
		 mpeg_video_ditherframe( video, src, bmp );
	  }
	  else
      {
         mpeg_video_ditherframe444( video, src );
      }
   }
   else
   {
      if ( ( video->pict_struct == FRAME_PICTURE && video->topfirst ) || video->pict_struct == BOTTOM_FIELD )
      {
         if ( video->chroma_format != CHROMA444 )
         {
            mpeg_video_dithertop( video, src, bmp );
            mpeg_video_ditherbot( video, src );
         }
         else 
         {
            mpeg_video_dithertop444( video, src );
            mpeg_video_ditherbot444( video, src );
         }
      }
      else 
      {
         if ( video->chroma_format != CHROMA444 )
         {
            mpeg_video_ditherbot( video, src );
            mpeg_video_dithertop( video, src, bmp );
         }
         else 
         {
            mpeg_video_ditherbot444( video, src );
            mpeg_video_dithertop444( video, src );
         }
      }
   }
   //
   return 0;
}
static inline int mpeg_video_display_second_field( mpeg_video_t *video )
{
   return 0;
}
/***************************************************************************/
static inline int mpeg_audio_read_ac3_header( mpeg_ac3_t *audio, unsigned char *header )
{
   int result = 0;
   //
   audio->flags = 0;
   result = a52_syncinfo( header, &audio->flags, &audio->samplerate, &audio->bitrate );
   if ( result )
   {
      audio->framesize = result;
      audio->channels = 0;
      if ( audio->flags & A52_LFE )
         audio->channels++;
      switch( audio->flags & A52_CHANNEL_MASK )
      {
         case A52_CHANNEL:
            audio->channels++;
            break;
         case A52_MONO:
            audio->channels++;
            break;
         case A52_STEREO:
            audio->channels += 2;
            break;
         case A52_3F:
            audio->channels += 3;
            break;
         case A52_2F1R:
            audio->channels += 3;
            break;
         case A52_3F1R:
            audio->channels += 4;
            break;
         case A52_2F2R:
            audio->channels += 4;
            break;
         case A52_3F2R:
            audio->channels += 5;
            break;
         case A52_DOLBY:
            audio->channels += 2;
            break;
         default:
		DebugMessage( "MpegPlayer : mpeg_ac3_header: unknown channel code: %p\n", audio->flags & A52_CHANNEL_MASK );
	    break;
      }
   }
   //
   return result;
}
/***************************************************************************/
static inline int mpeg_audio_do_ac3( mpeg_ac3_t *audio, char *frame, int frame_size, float **output, int render )
{
   int output_position = 0;
   sample_t level = 1;
   int i, j, k, l;
   //
   a52_frame( audio->state, frame, &audio->flags, &level, 0 );
   a52_dynrng( audio->state, NULL, NULL );
   for( i = 0; i < 6; i++ )
   {
      if ( !a52_block( audio->state ) )
      {
         l = 0;
         if ( render )
         {
            for( j = 0; j < audio->channels; j++ )
            {
               for( k = 0; k < 256; k++ )
               {
                  output[ j ][ output_position + k ] = ( ( sample_t* )audio->output )[ l ];
                  l++;
               }
            }
         }
         output_position += 256;
      }
   }
   //
   return output_position;
}
/***************************************************************************/
static inline int a52_syncinfo ( uint8_t * buf, int * flags, int * sample_rate, int * bit_rate )
{
   static int rate[] = {32, 40, 48, 56, 64, 80, 96, 112, 
   128, 160, 192, 224, 256, 320, 384, 448, 
   512, 576, 640};
   static uint8_t lfeon[ 8 ] = {0x10, 0x10, 0x04, 0x04, 0x04, 0x01, 0x04, 0x01};
   int frmsizecod;
   int bitrate;
   int half;
   int acmod;
   //
   if ( ( buf[ 0 ] != 0x0b ) || ( buf[ 1 ] != 0x77 ) )	
      return 0;
   if ( buf[ 5 ] >= 0x60 )	
      return 0;
   half = halfrate[ buf[ 5 ] >> 3 ];
   acmod = buf[ 6 ] >> 5;
   *flags = ( ( ( ( buf[ 6 ] & 0xf8 ) == 0x50 ) ? A52_DOLBY : acmod ) | ( ( buf[ 6 ] & lfeon[ acmod ] ) ? A52_LFE : 0 ) );
   frmsizecod = buf[ 4 ] & 63;
   if ( frmsizecod >= 38 )
      return 0;
   bitrate = rate [ frmsizecod >> 1 ];
   *bit_rate = ( bitrate * 1000 ) >> half;
   switch ( buf[ 4 ] & 0xc0 ) 
   {
      case 0:
         *sample_rate = 48000 >> half;
         return 4 * bitrate;
      case 0x40:
         *sample_rate = 44100 >> half;
         return 2 * ( 320 * bitrate / 147 + ( frmsizecod & 1 ) );
      case 0x80:
         *sample_rate = 32000 >> half;
         return 6 * bitrate;
         default:
      return 0;
}
}
/***************************************************************************/
static inline int a52_frame ( a52_state_t * state, uint8_t * buf, int * flags, sample_t * level, sample_t bias )
{
   static sample_t clev[ 4 ] = {LEVEL_3DB, LEVEL_45DB, LEVEL_6DB, LEVEL_45DB};
   static sample_t slev[ 4 ] = {LEVEL_3DB, LEVEL_6DB, 0, LEVEL_6DB};
   int chaninfo;
   int acmod;
   //
   state->fscod = buf[ 4 ] >> 6;
   state->halfrate = halfrate[ buf[ 5 ] >> 3 ];
   state->acmod = acmod = buf[ 6 ] >> 5;
   a52_bitstream_set_ptr ( buf + 6 );
   bitstream_get ( 3 );	
   if ( ( acmod == 2 ) && ( bitstream_get ( 2 ) == 2 ) )	
      acmod = A52_DOLBY;
   if ( ( acmod & 1 ) && ( acmod != 1 ) )
      state->clev = clev[ bitstream_get ( 2 ) ];	
   if ( acmod & 4 )
      state->slev = slev[ bitstream_get ( 2 ) ];	
   state->lfeon = bitstream_get ( 1 );
   state->output = a52_downmix_init ( acmod, *flags, level, 
   state->clev, state->slev );
   if ( state->output < 0 )
      return 1;
   if ( state->lfeon && ( *flags & A52_LFE ) )
      state->output |= A52_LFE;
   *flags = state->output;
   state->dynrng = state->level = 2 * *level;
   state->bias = bias;
   state->dynrnge = 1;
   state->dynrngcall = NULL;
   state->cplba.deltbae = DELTA_BIT_NONE;
   state->ba[ 0 ].deltbae = state->ba[ 1 ].deltbae = state->ba[ 2 ].deltbae =
   state->ba[ 3 ].deltbae = state->ba[ 4 ].deltbae = DELTA_BIT_NONE;
   chaninfo = !acmod;
   do 
   {
      bitstream_get ( 5 );	
      if ( bitstream_get ( 1 ) )	
         bitstream_get ( 8 );	
      if ( bitstream_get ( 1 ) )	
         bitstream_get ( 8 );	
      if ( bitstream_get ( 1 ) )	
         bitstream_get ( 7 );	
   } while ( chaninfo-- );
   bitstream_get ( 2 );	
   if ( bitstream_get ( 1 ) )	
      bitstream_get ( 14 );	
   if ( bitstream_get ( 1 ) )	
      bitstream_get ( 14 );	
   if ( bitstream_get ( 1 ) )
   {
      int addbsil;
      addbsil = bitstream_get ( 6 );
      do 
      {
         bitstream_get ( 8 );	
	  } while ( addbsil-- );
   }
   //
   return 0;
}
/***************************************************************************/
static inline void a52_dynrng ( a52_state_t * state, sample_t ( * call ) ( sample_t, void * ), void * data )
{
   state->dynrnge = 0;
   if ( call )
   {
      state->dynrnge = 1;
      state->dynrngcall = call;
      state->dynrngdata = data;
   }
}
/***************************************************************************/
static int parse_exponents ( int expstr, int ngrps, uint8_t exponent, uint8_t * dest )
{
   int exps;
   //
   while ( ngrps-- )
   {
      exps = bitstream_get ( 7 );
      exponent += exp_1[ exps ];
      if ( exponent > 24 )
         return 1;
      switch ( expstr ) 
      {
         case EXP_D45:
            *( dest++ ) = exponent;
            *( dest++ ) = exponent;
         case EXP_D25:
            *( dest++ ) = exponent;
         case EXP_D15:
            *( dest++ ) = exponent;
         }
         exponent += exp_2[ exps ];
         if ( exponent > 24 )
            return 1;
         switch ( expstr ) 
         {
         case EXP_D45:
            *( dest++ ) = exponent;
            *( dest++ ) = exponent;
         case EXP_D25:
            *( dest++ ) = exponent;
         case EXP_D15:
            *( dest++ ) = exponent;
         }
         exponent += exp_3[ exps ];
         if ( exponent > 24 )
            return 1;
         switch ( expstr ) 
         {
         case EXP_D45:
            *( dest++ ) = exponent;
            *( dest++ ) = exponent;
         case EXP_D25:
            *( dest++ ) = exponent;
         case EXP_D15:
            *( dest++ ) = exponent;
         }
      }	
      return 0;
   }
   /***************************************************************************/
   static int parse_deltba ( int8_t * deltba )
   {
      int deltnseg, deltlen, delta, j;
      memset ( deltba, 0, 50 );
      deltnseg = bitstream_get ( 3 );
      j = 0;
      do 
      {
         j += bitstream_get ( 5 );
         deltlen = bitstream_get ( 4 );
         delta = bitstream_get ( 3 );
         delta -= ( delta >= 4 ) ? 3 : 4;
         if ( !deltlen )
            continue;
         if ( j + deltlen >= 50 )
            return 1;
	 while ( deltlen-- )
            deltba[ j++ ] = delta;
	  } while ( deltnseg-- );
      return 0;
   }
   static inline int zero_snr_offsets ( int nfchans, a52_state_t * state )
   {
      int i;
      if ( ( state->csnroffst ) || ( state->chincpl && state->cplba.bai >> 3 ) || ( state->lfeon && state->lfeba.bai >> 3 ) )
         return 0;
      for ( i = 0; i < nfchans; i++ )
      {
         if ( state->ba[ i ].bai >> 3 )	
            return 0;
      }
      return 1;
   }
   static inline int16_t dither_gen ( void )
   {
      static uint16_t lfsr_state = 1;
      int16_t state;
      state = dither_lut[ lfsr_state >> 8 ] ^ ( lfsr_state << 8 );
      lfsr_state = ( uint16_t ) state;
      return state;
   }
   static void coeff_get ( sample_t * coeff, expbap_t * expbap, quantizer_t * quantizer, sample_t level, int dither, int end )
   {
      int i;
      uint8_t * exp;
      int8_t * bap;
      sample_t factor[ 25 ];
      for ( i = 0; i <= 24; i++ )
         factor[ i ] = scale_factor[ i ] * level;
      exp = expbap->exp;
      bap = expbap->bap;
      for ( i = 0; i < end; i++ ) 
      {
         int bapi;
         bapi = bap[ i ];
         switch ( bapi ) 
         {
         case 0:
            if ( dither )
            {
               coeff[ i ] = dither_gen() * LEVEL_3DB * factor[ exp[ i ] ];
               continue;
            }
            else
            {
               coeff[ i ] = 0;
               continue;
            }
         case -1:
            if ( quantizer->q1_ptr >= 0 )
            {
               coeff[ i ] = quantizer->q1[ quantizer->q1_ptr-- ] * factor[ exp[ i ] ];
               continue;
            }
            else
            {
               int code;
               code = bitstream_get ( 5 );
               quantizer->q1_ptr = 1;
               quantizer->q1[ 0 ] = q_1_2[ code ];
               quantizer->q1[ 1 ] = q_1_1[ code ];
               coeff[ i ] = q_1_0[ code ] * factor[ exp[ i ] ];
               continue;
            }
         case -2:
            if ( quantizer->q2_ptr >= 0 )
            {
               coeff[ i ] = quantizer->q2[ quantizer->q2_ptr-- ] * factor[ exp[ i ] ];
               continue;
            }
            else
            {
               int code;
               code = bitstream_get ( 7 );
               quantizer->q2_ptr = 1;
               quantizer->q2[ 0 ] = q_2_2[ code ];
               quantizer->q2[ 1 ] = q_2_1[ code ];
               coeff[ i ] = q_2_0[ code ] * factor[ exp[ i ] ];
               continue;
            }
         case 3:
            coeff[ i ] = q_3[ bitstream_get ( 3 ) ] * factor[ exp[ i ] ];
            continue;
         case -3:
            if ( quantizer->q4_ptr == 0 )
            {
               quantizer->q4_ptr = -1;
               coeff[ i ] = quantizer->q4 * factor[ exp[ i ] ];
               continue;
            }
            else
            {
               int code;
               code = bitstream_get ( 7 );
               quantizer->q4_ptr = 0;
               quantizer->q4 = q_4_1[ code ];
               coeff[ i ] = q_4_0[ code ] * factor[ exp[ i ] ];
               continue;
            }
         case 4:
            coeff[ i ] = q_5[ bitstream_get ( 4 ) ] * factor[ exp[ i ] ];
            continue;
            default:
         coeff[ i ] = ( ( bitstream_get_2 ( bapi ) << ( 16 - bapi ) ) *
         factor[ exp[ i ] ] );
      }
   }
}
/***************************************************************************/
static void coeff_get_coupling ( a52_state_t * state, int nfchans, sample_t * coeff, sample_t ( * samples )[ 256 ], quantizer_t * quantizer, uint8_t dithflag[ 5 ] )
{
   int cplbndstrc, bnd, i, i_end, ch;
   uint8_t * exp;
   int8_t * bap;
   sample_t cplco[ 5 ];
   //
   exp = state->cpl_expbap.exp;
   bap = state->cpl_expbap.bap;
   bnd = 0;
   cplbndstrc = state->cplbndstrc;
   i = state->cplstrtmant;
   while ( i < state->cplendmant )
   {
      i_end = i + 12;
	  while ( cplbndstrc & 1 )
      {
         cplbndstrc >>= 1;
         i_end += 12;
      }
      cplbndstrc >>= 1;
      for ( ch = 0; ch < nfchans; ch++ )
         cplco[ ch ] = state->cplco[ ch ][ bnd ] * coeff[ ch ];
      bnd++;
	  while ( i < i_end )
      {
         sample_t cplcoeff;
         int bapi;
         bapi = bap[ i ];
         switch ( bapi ) 
         {
            case 0:
               cplcoeff = LEVEL_3DB * scale_factor[ exp[ i ] ];
               for ( ch = 0; ch < nfchans; ch++ )
                  if ( ( state->chincpl >> ch ) & 1 )
               {
                  if ( dithflag[ ch ] )
		     samples[ ch ][ i ] = ( cplcoeff * cplco[ ch ] * dither_gen() );
                  else
                     samples[ ch ][ i ] = 0;
               }
               i++;
               continue;
            case -1:
               if ( quantizer->q1_ptr >= 0 )
               {
                  cplcoeff = quantizer->q1[ quantizer->q1_ptr-- ];
                  break;
               }
               else
               {
                  int code;
                  code = bitstream_get ( 5 );
                  quantizer->q1_ptr = 1;
                  quantizer->q1[ 0 ] = q_1_2[ code ];
                  quantizer->q1[ 1 ] = q_1_1[ code ];
                  cplcoeff = q_1_0[ code ];
                  break;
               }
            case -2:
               if ( quantizer->q2_ptr >= 0 )
               {
                  cplcoeff = quantizer->q2[ quantizer->q2_ptr-- ];
                  break;
               }
               else
               {
                  int code;
                  code = bitstream_get ( 7 );
                  quantizer->q2_ptr = 1;
                  quantizer->q2[ 0 ] = q_2_2[ code ];
                  quantizer->q2[ 1 ] = q_2_1[ code ];
                  cplcoeff = q_2_0[ code ];
                  break;
               }
            case 3:
               cplcoeff = q_3[ bitstream_get ( 3 ) ];
               break;
            case -3:
               if ( quantizer->q4_ptr == 0 )
               {
                  quantizer->q4_ptr = -1;
                  cplcoeff = quantizer->q4;
                  break;
               }
               else
               {
                  int code;
                  code = bitstream_get ( 7 );
                  quantizer->q4_ptr = 0;
                  quantizer->q4 = q_4_1[ code ];
                  cplcoeff = q_4_0[ code ];
                  break;
            }
            case 4:
               cplcoeff = q_5[ bitstream_get ( 4 ) ];
               break;
            default:
               cplcoeff = bitstream_get_2 ( bapi ) << ( 16 - bapi );
         }
         cplcoeff *= scale_factor[ exp[ i ] ];
         for ( ch = 0; ch < nfchans; ch++ )
         {
            if ( ( state->chincpl >> ch ) & 1 )
               samples[ ch ][ i ] = cplcoeff * cplco[ ch ];
         }
         i++;
      }
   }
}
/***************************************************************************/
static inline int a52_block ( a52_state_t * state )
{
   static const uint8_t nfchans_tbl[] = {2, 1, 2, 3, 3, 4, 4, 5, 1, 1, 2};
   static int rematrix_band[ 4 ] = {25, 37, 61, 253};
   int i, nfchans, chaninfo;
   uint8_t cplexpstr, chexpstr[ 5 ], lfeexpstr, do_bit_alloc, done_cpl;
   uint8_t blksw[ 5 ], dithflag[ 5 ];
   sample_t coeff[ 5 ];
   int chanbias;
   quantizer_t quantizer;
   sample_t * samples;
   //
   nfchans = nfchans_tbl[ state->acmod ];
   for ( i = 0; i < nfchans; i++ )
      blksw[ i ] = bitstream_get ( 1 );
   for ( i = 0; i < nfchans; i++ )
      dithflag[ i ] = bitstream_get ( 1 );
   chaninfo = !state->acmod;
   do 
   {
      if ( bitstream_get ( 1 ) )
      {
         int dynrng;
         dynrng = bitstream_get_2 ( 8 );
         if ( state->dynrnge )
         {
            sample_t range;
            range = ( ( ( ( dynrng & 0x1f ) | 0x20 ) << 13 ) *
            scale_factor[ 3 - ( dynrng >> 5 ) ] );
            if ( state->dynrngcall )
               range = state->dynrngcall ( range, state->dynrngdata );
            state->dynrng = state->level * range;
         }
      }
   } while ( chaninfo-- );
   if ( bitstream_get ( 1 ) )
   {
      state->chincpl = 0;
      if ( bitstream_get ( 1 ) )
      {
         static uint8_t bndtab[ 16 ] = {31, 35, 37, 39, 41, 42, 43, 44, 
         45, 45, 46, 46, 47, 47, 48, 48};
         int cplbegf;
         int cplendf;
         int ncplsubnd;
         for ( i = 0; i < nfchans; i++ )
            state->chincpl |= bitstream_get ( 1 ) << i;
         switch ( state->acmod ) 
         {
            case 0: case 1:
               return 1;
            case 2:
               state->phsflginu = bitstream_get ( 1 );
            }
            cplbegf = bitstream_get ( 4 );
            cplendf = bitstream_get ( 4 );
            if ( cplendf + 3 - cplbegf < 0 )
               return 1;
         state->ncplbnd = ncplsubnd = cplendf + 3 - cplbegf;
         state->cplstrtbnd = bndtab[ cplbegf ];
         state->cplstrtmant = cplbegf * 12 + 37;
         state->cplendmant = cplendf * 12 + 73;
         state->cplbndstrc = 0;
         for ( i = 0; i < ncplsubnd - 1; i++ )
            if ( bitstream_get ( 1 ) )
         {
            state->cplbndstrc |= 1 << i;
            state->ncplbnd--;
         }
      }
   }
   if ( state->chincpl )
   {
      int j, cplcoe;
      cplcoe = 0;
      for ( i = 0; i < nfchans; i++ )
         if ( ( state->chincpl ) >> i & 1 )
      if ( bitstream_get ( 1 ) )
      {
         int mstrcplco, cplcoexp, cplcomant;
         cplcoe = 1;
         mstrcplco = 3 * bitstream_get ( 2 );
         for ( j = 0; j < state->ncplbnd; j++ )
         {
            cplcoexp = bitstream_get ( 4 );
            cplcomant = bitstream_get ( 4 );
            if ( cplcoexp == 15 )
               cplcomant <<= 14;
            else
               cplcomant = ( cplcomant | 0x10 ) << 13;
            state->cplco[ i ][ j ] =
            cplcomant * scale_factor[ cplcoexp + mstrcplco ];
         }
      }
      if ( ( state->acmod == 2 ) && state->phsflginu && cplcoe )
         for ( j = 0; j < state->ncplbnd; j++ )
      if ( bitstream_get ( 1 ) )	
         state->cplco[ 1 ][ j ] = -state->cplco[ 1 ][ j ];
   }
   if ( ( state->acmod == 2 ) && ( bitstream_get ( 1 ) ) )
   {
      int end;
      state->rematflg = 0;
      end = ( state->chincpl ) ? state->cplstrtmant : 253;	
      i = 0;
      do
         state->rematflg |= bitstream_get ( 1 ) << i;
	  while ( rematrix_band[ i++ ] < end );
      }
   cplexpstr = EXP_REUSE;
   lfeexpstr = EXP_REUSE;
   if ( state->chincpl )	
      cplexpstr = bitstream_get ( 2 );
   for ( i = 0; i < nfchans; i++ )
      chexpstr[ i ] = bitstream_get ( 2 );
   if ( state->lfeon ) 
      lfeexpstr = bitstream_get ( 1 );
   for ( i = 0; i < nfchans; i++ )
      if ( chexpstr[ i ] != EXP_REUSE )
   {
      if ( ( state->chincpl >> i ) & 1 )
         state->endmant[ i ] = state->cplstrtmant;
      else
      {
         int chbwcod;
         chbwcod = bitstream_get ( 6 );
         if ( chbwcod > 60 )
            return 1;
         state->endmant[ i ] = chbwcod * 3 + 73;
      }
   }
   do_bit_alloc = 0;
   if ( cplexpstr != EXP_REUSE )
   {
      int cplabsexp, ncplgrps;
      do_bit_alloc = 64;
      ncplgrps = ( ( state->cplendmant - state->cplstrtmant ) / ( 3 << ( cplexpstr - 1 ) ) );
      cplabsexp = bitstream_get ( 4 ) << 1;
      if ( parse_exponents ( cplexpstr, ncplgrps, cplabsexp, state->cpl_expbap.exp + state->cplstrtmant ) )
         return 1;
   }
   for ( i = 0; i < nfchans; i++ )
      if ( chexpstr[ i ] != EXP_REUSE )
   {
      int grp_size, nchgrps;
      do_bit_alloc |= 1 << i;
      grp_size = 3 << ( chexpstr[ i ] - 1 );
      nchgrps = ( state->endmant[ i ] + grp_size - 4 ) / grp_size;
      state->fbw_expbap[ i ].exp[ 0 ] = bitstream_get ( 4 );
      if ( parse_exponents ( chexpstr[ i ], nchgrps, state->fbw_expbap[ i ].exp[ 0 ], state->fbw_expbap[ i ].exp + 1 ) )
         return 1;
      bitstream_get ( 2 );	
   }
   if ( lfeexpstr != EXP_REUSE )
   {
      do_bit_alloc |= 32;
      state->lfe_expbap.exp[ 0 ] = bitstream_get ( 4 );
      if ( parse_exponents ( lfeexpstr, 2, state->lfe_expbap.exp[ 0 ], state->lfe_expbap.exp + 1 ) )
         return 1;
   }
   if ( bitstream_get ( 1 ) )
   {
      do_bit_alloc = -1;
      state->bai = bitstream_get ( 11 );
   }
   if ( bitstream_get ( 1 ) )
   {
      do_bit_alloc = -1;
      state->csnroffst = bitstream_get ( 6 );
      if ( state->chincpl )	
         state->cplba.bai = bitstream_get ( 7 );
      for ( i = 0; i < nfchans; i++ )
         state->ba[ i ].bai = bitstream_get ( 7 );
      if ( state->lfeon )
         state->lfeba.bai = bitstream_get ( 7 );
   }
   if ( ( state->chincpl ) && ( bitstream_get ( 1 ) ) )
   {
      do_bit_alloc |= 64;
      state->cplfleak = 9 - bitstream_get ( 3 );
      state->cplsleak = 9 - bitstream_get ( 3 );
   }
   if ( bitstream_get ( 1 ) )
   {
      do_bit_alloc = -1;
      if ( state->chincpl )	
         state->cplba.deltbae = bitstream_get ( 2 );
      for ( i = 0; i < nfchans; i++ )
         state->ba[ i ].deltbae = bitstream_get ( 2 );
      if ( state->chincpl &&	( state->cplba.deltbae == DELTA_BIT_NEW ) && parse_deltba ( state->cplba.deltba ) )
         return 1;
      for ( i = 0; i < nfchans; i++ )
         if ( ( state->ba[ i ].deltbae == DELTA_BIT_NEW ) && parse_deltba ( state->ba[ i ].deltba ) )
      return 1;
   }
   if ( do_bit_alloc )
   {
      if ( zero_snr_offsets ( nfchans, state ) )
      {
         memset ( state->cpl_expbap.bap, 0, sizeof ( state->cpl_expbap.bap ) );
         for ( i = 0; i < nfchans; i++ )
            memset ( state->fbw_expbap[ i ].bap, 0, 
         sizeof ( state->fbw_expbap[ i ].bap ) );
         memset ( state->lfe_expbap.bap, 0, sizeof ( state->lfe_expbap.bap ) );
      }
      else
      {
         if ( state->chincpl && ( do_bit_alloc & 64 ) )
            a52_bit_allocate ( state, &state->cplba, state->cplstrtbnd, state->cplstrtmant, state->cplendmant, state->cplfleak << 8, state->cplsleak << 8, &state->cpl_expbap );
         for ( i = 0; i < nfchans; i++ )
         {
            if ( do_bit_alloc & ( 1 << i ) )
               a52_bit_allocate ( state, state->ba + i, 0, 0, state->endmant[ i ], 0, 0, state->fbw_expbap +i );
         }
         if ( state->lfeon && ( do_bit_alloc & 32 ) ) 
         {
            state->lfeba.deltbae = DELTA_BIT_NONE;
            a52_bit_allocate ( state, &state->lfeba, 0, 0, 7, 0, 0, &state->lfe_expbap );
         }
      }
   }
   if ( bitstream_get ( 1 ) )
   {
      i = bitstream_get ( 9 );	
	  while ( i-- )
         bitstream_get ( 8 );
   }
   samples = state->samples;
   if ( state->output & A52_LFE )
      samples += 256;	
   chanbias = a52_downmix_coeff ( coeff, state->acmod, state->output, state->dynrng, state->clev, state->slev );
   quantizer.q1_ptr = quantizer.q2_ptr = quantizer.q4_ptr = -1;
   done_cpl = 0;
   for ( i = 0; i < nfchans; i++ ) 
   {
      int j;
      coeff_get ( samples + 256 * i, state->fbw_expbap +i, &quantizer, coeff[ i ], dithflag[ i ], state->endmant[ i ] );
      if ( ( state->chincpl >> i ) & 1 ) 
      {
         if ( !done_cpl ) 
         {
            done_cpl = 1;
            coeff_get_coupling ( state, nfchans, coeff, ( sample_t ( * )[ 256 ] )samples, &quantizer, dithflag );
         }
         j = state->cplendmant;
      }
      else
         j = state->endmant[ i ];
      do
         ( samples + 256 * i )[ j ] = 0;
	  while ( ++j < 256 );
      }
   if ( state->acmod == 2 ) 
   {
      int j, end, band, rematflg;
      end = ( ( state->endmant[ 0 ] < state->endmant[ 1 ] ) ?
      state->endmant[ 0 ] : state->endmant[ 1 ] );
      i = 0;
      j = 13;
      rematflg = state->rematflg;
      do 
      {
         if ( ! ( rematflg & 1 ) ) 
         {
            rematflg >>= 1;
            j = rematrix_band[ i++ ];
            continue;
         }
         rematflg >>= 1;
         band = rematrix_band[ i++ ];
         if ( band > end )
            band = end;
         do 
         {
            sample_t tmp0, tmp1;
            tmp0 = samples[ j ];
            tmp1 = ( samples+256 )[ j ];
            samples[ j ] = tmp0 + tmp1;
            ( samples+256 )[ j ] = tmp0 - tmp1;
	 } while ( ++j < band );
	  } while ( j < end );
   }
   if ( state->lfeon ) 
   {
      if ( state->output & A52_LFE ) 
      {
         coeff_get ( samples - 256, &state->lfe_expbap, &quantizer, state->dynrng, 0, 7 );
         for ( i = 7; i < 256; i++ )
            ( samples-256 )[ i ] = 0;
         a52_imdct_512 ( samples - 256, samples + 1536 - 256, state->bias );
      }
      else
      {
         coeff_get ( samples + 1280, &state->lfe_expbap, &quantizer, 
         0, 0, 7 );
      }
   }
   i = 0;
   if ( nfchans_tbl[ state->output & A52_CHANNEL_MASK ] < nfchans )
      for ( i = 1; i < nfchans; i++ )
   if ( blksw[ i ] != blksw[ 0 ] )
      break;
   if ( i < nfchans ) 
   {
      if ( state->downmixed ) 
      {
         state->downmixed = 0;
         a52_upmix ( samples + 1536, state->acmod, state->output );
      }
      for ( i = 0; i < nfchans; i++ ) 
      {
         sample_t bias;
         bias = 0;
         if ( !( chanbias & ( 1 << i ) ) )
            bias = state->bias;
         if ( coeff[ i ] ) 
         {
            if ( blksw[ i ] )
               a52_imdct_256 ( samples + 256 * i, samples + 1536 + 256 * i, bias );
            else 
               a52_imdct_512 ( samples + 256 * i, samples + 1536 + 256 * i, bias );
         }
         else
         {
            int j;
            for ( j = 0; j < 256; j++ )
               ( samples + 256 * i )[ j ] = bias;
         }
      }
      a52_downmix ( samples, state->acmod, state->output, state->bias, state->clev, state->slev );
   }
   else
   {
      nfchans = nfchans_tbl[ state->output & A52_CHANNEL_MASK ];
      a52_downmix ( samples, state->acmod, state->output, 0, state->clev, state->slev );
      if ( !state->downmixed ) 
      {
         state->downmixed = 1;
         a52_downmix ( samples + 1536, state->acmod, state->output, 0, state->clev, state->slev );
      }
      if ( blksw[ 0 ] )
         for ( i = 0; i < nfchans; i++ )
      a52_imdct_256 ( samples + 256 * i, samples + 1536 + 256 * i, state->bias );
      else 
         for ( i = 0; i < nfchans; i++ )
      a52_imdct_512 ( samples + 256 * i, samples + 1536 + 256 * i, state->bias );
   }
   //
   return 0;
}
/***************************************************************************/
static inline void rec( unsigned char *s, unsigned char *d, int lx2, int h )
{
   int j;
   //
   for( j = 0; j < h; j++, s += lx2, d += lx2 )
   {
      d[ 0 ] = s[ 0 ]; d[ 1 ] = s[ 1 ]; d[ 2 ] = s[ 2 ]; d[ 3 ] = s[ 3 ];
      d[ 4 ] = s[ 4 ]; d[ 5 ] = s[ 5 ]; d[ 6 ] = s[ 6 ]; d[ 7 ] = s[ 7 ];
      d[ 8 ] = s[ 8 ]; d[ 9 ] = s[ 9 ]; d[ 10 ] = s[ 10 ]; d[ 11 ] = s[ 11 ];
      d[ 12 ] = s[ 12 ]; d[ 13 ] = s[ 13 ]; d[ 14 ] = s[ 14 ]; d[ 15 ] = s[ 15 ];
   }
}
/***************************************************************************/
static inline void recc( unsigned char *s, unsigned char *d, int lx2, int h )
{
   int j;
   //
   for( j = 0; j < h; j++, s += lx2, d += lx2 )
   {
      d[ 0 ] = s[ 0 ]; d[ 1 ] = s[ 1 ]; d[ 2 ] = s[ 2 ]; d[ 3 ] = s[ 3 ];
      d[ 4 ] = s[ 4 ]; d[ 5 ] = s[ 5 ]; d[ 6 ] = s[ 6 ]; d[ 7 ] = s[ 7 ];
   }
}
/***************************************************************************/
static inline void reca( unsigned char *s, unsigned char *d, int lx2, int h )
{
   int j;
   //
   for( j = 0; j < h; j++, s +=lx2, d +=lx2 )
   {
      d[ 0 ] = ( unsigned int )( d[ 0 ] + s[ 0 ] + 1 ) >> 1;
      d[ 1 ] = ( unsigned int )( d[ 1 ] + s[ 1 ] + 1 ) >> 1;
      d[ 2 ] = ( unsigned int )( d[ 2 ] + s[ 2 ] + 1 ) >> 1;
      d[ 3 ] = ( unsigned int )( d[ 3 ] + s[ 3 ] + 1 ) >> 1;
      d[ 4 ] = ( unsigned int )( d[ 4 ] + s[ 4 ] + 1 ) >> 1;
      d[ 5 ] = ( unsigned int )( d[ 5 ] + s[ 5 ] + 1 ) >> 1;
      d[ 6 ] = ( unsigned int )( d[ 6 ] + s[ 6 ] + 1 ) >> 1;
      d[ 7 ] = ( unsigned int )( d[ 7 ] + s[ 7 ] + 1 ) >> 1;
      d[ 8 ] = ( unsigned int )( d[ 8 ] + s[ 8 ] + 1 ) >> 1;
      d[ 9 ] = ( unsigned int )( d[ 9 ] + s[ 9 ] + 1 ) >> 1;
      d[ 10 ] = ( unsigned int )( d[ 10 ] + s[ 10 ] + 1 ) >> 1;
      d[ 11 ] = ( unsigned int )( d[ 11 ] + s[ 11 ] + 1 ) >> 1;
      d[ 12 ] = ( unsigned int )( d[ 12 ] + s[ 12 ] + 1 ) >> 1;
      d[ 13 ] = ( unsigned int )( d[ 13 ] + s[ 13 ] + 1 ) >> 1;
      d[ 14 ] = ( unsigned int )( d[ 14 ] + s[ 14 ] + 1 ) >> 1;
      d[ 15 ] = ( unsigned int )( d[ 15 ] + s[ 15 ] + 1 ) >> 1;
   }
}
/***************************************************************************/
static inline void recac( unsigned char *s, unsigned char *d, int lx2, int h )
{
   int j;
   //
   for( j = 0; j < h; j++, s += lx2, d += lx2 )
   {
      d[ 0 ] = ( unsigned int )( d[ 0 ] + s[ 0 ] + 1 )>>1;
      d[ 1 ] = ( unsigned int )( d[ 1 ] + s[ 1 ] + 1 )>>1;
      d[ 2 ] = ( unsigned int )( d[ 2 ] + s[ 2 ] + 1 )>>1;
      d[ 3 ] = ( unsigned int )( d[ 3 ] + s[ 3 ] + 1 )>>1;
      d[ 4 ] = ( unsigned int )( d[ 4 ] + s[ 4 ] + 1 )>>1;
      d[ 5 ] = ( unsigned int )( d[ 5 ] + s[ 5 ] + 1 )>>1;
      d[ 6 ] = ( unsigned int )( d[ 6 ] + s[ 6 ] + 1 )>>1;
      d[ 7 ] = ( unsigned int )( d[ 7 ] + s[ 7 ] + 1 )>>1;
   }
}
/***************************************************************************/
static inline void recvmpeg( unsigned char *s, unsigned char *d, int lx, int lx2, int h )
{
   unsigned char *dp, *sp, *sp2;
   int j;
   //
   sp = s;
   sp2 = s + lx;
   dp = d;
   for( j = 0; j < h; j++ )
   {
      dp[ 0 ] = ( unsigned int )( sp[ 0 ] + sp2[ 0 ] + 1 ) >> 1;
      dp[ 1 ] = ( unsigned int )( sp[ 1 ] + sp2[ 1 ] + 1 ) >> 1;
      dp[ 2 ] = ( unsigned int )( sp[ 2 ] + sp2[ 2 ] + 1 ) >> 1;
      dp[ 3 ] = ( unsigned int )( sp[ 3 ] + sp2[ 3 ] + 1 ) >> 1;
      dp[ 4 ] = ( unsigned int )( sp[ 4 ] + sp2[ 4 ] + 1 ) >> 1;
      dp[ 5 ] = ( unsigned int )( sp[ 5 ] + sp2[ 5 ] + 1 ) >> 1;
      dp[ 6 ] = ( unsigned int )( sp[ 6 ] + sp2[ 6 ] + 1 ) >> 1;
      dp[ 7 ] = ( unsigned int )( sp[ 7 ] + sp2[ 7 ] + 1 ) >> 1;
      dp[ 8 ] = ( unsigned int )( sp[ 8 ] + sp2[ 8 ] + 1 ) >> 1;
      dp[ 9 ] = ( unsigned int )( sp[ 9 ] + sp2[ 9 ] + 1 ) >> 1;
      dp[ 10 ] = ( unsigned int )( sp[ 10 ] + sp2[ 10 ] + 1 ) >> 1;
      dp[ 11 ] = ( unsigned int )( sp[ 11 ] + sp2[ 11 ] + 1 ) >> 1;
      dp[ 12 ] = ( unsigned int )( sp[ 12 ] + sp2[ 12 ] + 1 ) >> 1;
      dp[ 13 ] = ( unsigned int )( sp[ 13 ] + sp2[ 13 ] + 1 ) >> 1;
      dp[ 14 ] = ( unsigned int )( sp[ 14 ] + sp2[ 14 ] + 1 ) >> 1;
      dp[ 15 ] = ( unsigned int )( sp[ 15 ] + sp2[ 15 ] + 1 ) >> 1;
      sp+= lx2;
      sp2+= lx2;
      dp+= lx2;
   }
}
/***************************************************************************/
static inline void recvc( unsigned char *s, unsigned char *d, int lx, int lx2, int h )
{
   unsigned char *dp, *sp, *sp2;
   int j;
   //
   sp = s;
   sp2 = s+lx;
   dp = d;
   for( j = 0; j < h; j++ )
   {
      dp[ 0 ] = ( unsigned int )( sp[ 0 ]+sp2[ 0 ]+1 )>>1;
      dp[ 1 ] = ( unsigned int )( sp[ 1 ]+sp2[ 1 ]+1 )>>1;
      dp[ 2 ] = ( unsigned int )( sp[ 2 ]+sp2[ 2 ]+1 )>>1;
      dp[ 3 ] = ( unsigned int )( sp[ 3 ]+sp2[ 3 ]+1 )>>1;
      dp[ 4 ] = ( unsigned int )( sp[ 4 ]+sp2[ 4 ]+1 )>>1;
      dp[ 5 ] = ( unsigned int )( sp[ 5 ]+sp2[ 5 ]+1 )>>1;
      dp[ 6 ] = ( unsigned int )( sp[ 6 ]+sp2[ 6 ]+1 )>>1;
      dp[ 7 ] = ( unsigned int )( sp[ 7 ]+sp2[ 7 ]+1 )>>1;
      sp+= lx2;
      sp2+= lx2;
      dp+= lx2;
   }
}
/***************************************************************************/
static inline void recva( unsigned char *s, unsigned char *d, int lx, int lx2, int h )
{
   unsigned char *dp, *sp, *sp2;
   int j;
   //
   sp = s;
   sp2 = s+lx;
   dp = d;
   for ( j=0; j<h; j++ )
   {
      dp[ 0 ] = ( dp[ 0 ] + ( ( unsigned int )( sp[ 0 ]+sp2[ 0 ]+1 )>>1 ) + 1 )>>1;
      dp[ 1 ] = ( dp[ 1 ] + ( ( unsigned int )( sp[ 1 ]+sp2[ 1 ]+1 )>>1 ) + 1 )>>1;
      dp[ 2 ] = ( dp[ 2 ] + ( ( unsigned int )( sp[ 2 ]+sp2[ 2 ]+1 )>>1 ) + 1 )>>1;
      dp[ 3 ] = ( dp[ 3 ] + ( ( unsigned int )( sp[ 3 ]+sp2[ 3 ]+1 )>>1 ) + 1 )>>1;
      dp[ 4 ] = ( dp[ 4 ] + ( ( unsigned int )( sp[ 4 ]+sp2[ 4 ]+1 )>>1 ) + 1 )>>1;
      dp[ 5 ] = ( dp[ 5 ] + ( ( unsigned int )( sp[ 5 ]+sp2[ 5 ]+1 )>>1 ) + 1 )>>1;
      dp[ 6 ] = ( dp[ 6 ] + ( ( unsigned int )( sp[ 6 ]+sp2[ 6 ]+1 )>>1 ) + 1 )>>1;
      dp[ 7 ] = ( dp[ 7 ] + ( ( unsigned int )( sp[ 7 ]+sp2[ 7 ]+1 )>>1 ) + 1 )>>1;
      dp[ 8 ] = ( dp[ 8 ] + ( ( unsigned int )( sp[ 8 ]+sp2[ 8 ]+1 )>>1 ) + 1 )>>1;
      dp[ 9 ] = ( dp[ 9 ] + ( ( unsigned int )( sp[ 9 ]+sp2[ 9 ]+1 )>>1 ) + 1 )>>1;
      dp[ 10 ] = ( dp[ 10 ] + ( ( unsigned int )( sp[ 10 ]+sp2[ 10 ]+1 )>>1 ) + 1 )>>1;
      dp[ 11 ] = ( dp[ 11 ] + ( ( unsigned int )( sp[ 11 ]+sp2[ 11 ]+1 )>>1 ) + 1 )>>1;
      dp[ 12 ] = ( dp[ 12 ] + ( ( unsigned int )( sp[ 12 ]+sp2[ 12 ]+1 )>>1 ) + 1 )>>1;
      dp[ 13 ] = ( dp[ 13 ] + ( ( unsigned int )( sp[ 13 ]+sp2[ 13 ]+1 )>>1 ) + 1 )>>1;
      dp[ 14 ] = ( dp[ 14 ] + ( ( unsigned int )( sp[ 14 ]+sp2[ 14 ]+1 )>>1 ) + 1 )>>1;
      dp[ 15 ] = ( dp[ 15 ] + ( ( unsigned int )( sp[ 15 ]+sp2[ 15 ]+1 )>>1 ) + 1 )>>1;
      sp+= lx2;
      sp2+= lx2;
      dp+= lx2;
   }
}
/***************************************************************************/
static inline void recvac( unsigned char *s, unsigned char *d, int lx, int lx2, int h )
{
   unsigned char *dp, *sp, *sp2;
   int j;
   //
   sp = s;
   sp2 = s+lx;
   dp = d;
   for ( j=0; j<h; j++ )
   {
      dp[ 0 ] = ( dp[ 0 ] + ( ( unsigned int )( sp[ 0 ]+sp2[ 0 ]+1 )>>1 ) + 1 )>>1;
      dp[ 1 ] = ( dp[ 1 ] + ( ( unsigned int )( sp[ 1 ]+sp2[ 1 ]+1 )>>1 ) + 1 )>>1;
      dp[ 2 ] = ( dp[ 2 ] + ( ( unsigned int )( sp[ 2 ]+sp2[ 2 ]+1 )>>1 ) + 1 )>>1;
      dp[ 3 ] = ( dp[ 3 ] + ( ( unsigned int )( sp[ 3 ]+sp2[ 3 ]+1 )>>1 ) + 1 )>>1;
      dp[ 4 ] = ( dp[ 4 ] + ( ( unsigned int )( sp[ 4 ]+sp2[ 4 ]+1 )>>1 ) + 1 )>>1;
      dp[ 5 ] = ( dp[ 5 ] + ( ( unsigned int )( sp[ 5 ]+sp2[ 5 ]+1 )>>1 ) + 1 )>>1;
      dp[ 6 ] = ( dp[ 6 ] + ( ( unsigned int )( sp[ 6 ]+sp2[ 6 ]+1 )>>1 ) + 1 )>>1;
      dp[ 7 ] = ( dp[ 7 ] + ( ( unsigned int )( sp[ 7 ]+sp2[ 7 ]+1 )>>1 ) + 1 )>>1;
      sp+= lx2;
      sp2+= lx2;
      dp+= lx2;
   }
}
/***************************************************************************/
static inline void rech( unsigned char *s, unsigned char *d, int lx2, int h )
{
   unsigned char *dp, *sp;
   unsigned int s1, s2;
   int j;
   //
   sp = s;
   dp = d;
   for ( j=0; j<h; j++ )
   {
      s1=sp[ 0 ];
      dp[ 0 ] = ( unsigned int )( s1+( s2=sp[ 1 ] )+1 )>>1;
      dp[ 1 ] = ( unsigned int )( s2+( s1=sp[ 2 ] )+1 )>>1;
      dp[ 2 ] = ( unsigned int )( s1+( s2=sp[ 3 ] )+1 )>>1;
      dp[ 3 ] = ( unsigned int )( s2+( s1=sp[ 4 ] )+1 )>>1;
      dp[ 4 ] = ( unsigned int )( s1+( s2=sp[ 5 ] )+1 )>>1;
      dp[ 5 ] = ( unsigned int )( s2+( s1=sp[ 6 ] )+1 )>>1;
      dp[ 6 ] = ( unsigned int )( s1+( s2=sp[ 7 ] )+1 )>>1;
      dp[ 7 ] = ( unsigned int )( s2+( s1=sp[ 8 ] )+1 )>>1;
      dp[ 8 ] = ( unsigned int )( s1+( s2=sp[ 9 ] )+1 )>>1;
      dp[ 9 ] = ( unsigned int )( s2+( s1=sp[ 10 ] )+1 )>>1;
      dp[ 10 ] = ( unsigned int )( s1+( s2=sp[ 11 ] )+1 )>>1;
      dp[ 11 ] = ( unsigned int )( s2+( s1=sp[ 12 ] )+1 )>>1;
      dp[ 12 ] = ( unsigned int )( s1+( s2=sp[ 13 ] )+1 )>>1;
      dp[ 13 ] = ( unsigned int )( s2+( s1=sp[ 14 ] )+1 )>>1;
      dp[ 14 ] = ( unsigned int )( s1+( s2=sp[ 15 ] )+1 )>>1;
      dp[ 15 ] = ( unsigned int )( s2+sp[ 16 ]+1 )>>1;
      sp+= lx2;
      dp+= lx2;
   }
}
/***************************************************************************/
static inline void rechc( unsigned char *s, unsigned char *d, int lx2, int h )
{
   unsigned char *dp, *sp;
   unsigned int s1, s2;
   int j;
   //
   sp = s;
   dp = d;
   for ( j=0; j<h; j++ )
   {
      s1=sp[ 0 ];
      dp[ 0 ] = ( unsigned int )( s1+( s2=sp[ 1 ] )+1 )>>1;
      dp[ 1 ] = ( unsigned int )( s2+( s1=sp[ 2 ] )+1 )>>1;
      dp[ 2 ] = ( unsigned int )( s1+( s2=sp[ 3 ] )+1 )>>1;
      dp[ 3 ] = ( unsigned int )( s2+( s1=sp[ 4 ] )+1 )>>1;
      dp[ 4 ] = ( unsigned int )( s1+( s2=sp[ 5 ] )+1 )>>1;
      dp[ 5 ] = ( unsigned int )( s2+( s1=sp[ 6 ] )+1 )>>1;
      dp[ 6 ] = ( unsigned int )( s1+( s2=sp[ 7 ] )+1 )>>1;
      dp[ 7 ] = ( unsigned int )( s2+sp[ 8 ]+1 )>>1;
      sp+= lx2;
      dp+= lx2;
   }
}
/***************************************************************************/
static inline void recha( unsigned char *s, unsigned char *d, int lx2, int h )
{
   unsigned char *dp, *sp;
   unsigned int s1, s2;
   int j;
   //
   sp = s;
   dp = d;
   for ( j = 0; j < h; j++ )
   {
      s1 = sp[ 0 ];
      dp[ 0 ] = ( dp[ 0 ] + ( ( unsigned int )( s1 + ( s2 = sp[ 1 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 1 ] = ( dp[ 1 ] + ( ( unsigned int )( s2 + ( s1 = sp[ 2 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 2 ] = ( dp[ 2 ] + ( ( unsigned int )( s1 + ( s2 = sp[ 3 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 3 ] = ( dp[ 3 ] + ( ( unsigned int )( s2 + ( s1 = sp[ 4 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 4 ] = ( dp[ 4 ] + ( ( unsigned int )( s1 + ( s2 = sp[ 5 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 5 ] = ( dp[ 5 ] + ( ( unsigned int )( s2 + ( s1 = sp[ 6 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 6 ] = ( dp[ 6 ] + ( ( unsigned int )( s1 + ( s2 = sp[ 7 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 7 ] = ( dp[ 7 ] + ( ( unsigned int )( s2 + ( s1 = sp[ 8 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 8 ] = ( dp[ 8 ] + ( ( unsigned int )( s1 + ( s2 = sp[ 9 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 9 ] = ( dp[ 9 ] + ( ( unsigned int )( s2 + ( s1 = sp[ 10 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 10 ] = ( dp[ 10 ] + ( ( unsigned int )( s1 + ( s2 = sp[ 11 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 11 ] = ( dp[ 11 ] + ( ( unsigned int )( s2 + ( s1 = sp[ 12 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 12 ] = ( dp[ 12 ] + ( ( unsigned int )( s1 + ( s2 = sp[ 13 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 13 ] = ( dp[ 13 ] + ( ( unsigned int )( s2 + ( s1 = sp[ 14 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 14 ] = ( dp[ 14 ] + ( ( unsigned int )( s1 + ( s2 = sp[ 15 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 15 ] = ( dp[ 15 ] + ( ( unsigned int )( s2 + sp[ 16 ] + 1 ) >> 1 ) + 1 ) >> 1;
      sp += lx2;
      dp += lx2;
   }
}
/***************************************************************************/
static inline void rechac( unsigned char *s, unsigned char *d, int lx2, int h )
{
   unsigned char *dp, *sp;
   unsigned int s1, s2;
   int j;
   //
   sp = s;
   dp = d;
   for( j = 0; j < h; j++ )
   {
      s1 = sp[ 0 ];
      dp[ 0 ] = ( dp[ 0 ] + ( ( unsigned int )( s1 + ( s2 = sp[ 1 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 1 ] = ( dp[ 1 ] + ( ( unsigned int )( s2 + ( s1 = sp[ 2 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 2 ] = ( dp[ 2 ] + ( ( unsigned int )( s1 + ( s2 = sp[ 3 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 3 ] = ( dp[ 3 ] + ( ( unsigned int )( s2 + ( s1 = sp[ 4 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 4 ] = ( dp[ 4 ] + ( ( unsigned int )( s1 + ( s2 = sp[ 5 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 5 ] = ( dp[ 5 ] + ( ( unsigned int )( s2 + ( s1 = sp[ 6 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 6 ] = ( dp[ 6 ] + ( ( unsigned int )( s1 + ( s2 = sp[ 7 ] ) + 1 ) >> 1 ) + 1 ) >> 1;
      dp[ 7 ] = ( dp[ 7 ] + ( ( unsigned int )( s2 + sp[ 8 ] + 1 ) >> 1 ) + 1 ) >> 1;
      sp += lx2;
      dp += lx2;
   }
}
/***************************************************************************/
static inline void rec4( unsigned char *s, unsigned char *d, int lx, int lx2, int h )
{
   unsigned char *dp, *sp, *sp2;
   unsigned int s1, s2, s3, s4;
   int j;
   //
   sp = s;
   sp2 = s+lx;
   dp = d;
   for ( j=0; j<h; j++ )
   {
      s1=sp[ 0 ]; s3=sp2[ 0 ];
      dp[ 0 ] = ( unsigned int )( s1+( s2=sp[ 1 ] )+s3+( s4=sp2[ 1 ] )+2 )>>2;
      dp[ 1 ] = ( unsigned int )( s2+( s1=sp[ 2 ] )+s4+( s3=sp2[ 2 ] )+2 )>>2;
      dp[ 2 ] = ( unsigned int )( s1+( s2=sp[ 3 ] )+s3+( s4=sp2[ 3 ] )+2 )>>2;
      dp[ 3 ] = ( unsigned int )( s2+( s1=sp[ 4 ] )+s4+( s3=sp2[ 4 ] )+2 )>>2;
      dp[ 4 ] = ( unsigned int )( s1+( s2=sp[ 5 ] )+s3+( s4=sp2[ 5 ] )+2 )>>2;
      dp[ 5 ] = ( unsigned int )( s2+( s1=sp[ 6 ] )+s4+( s3=sp2[ 6 ] )+2 )>>2;
      dp[ 6 ] = ( unsigned int )( s1+( s2=sp[ 7 ] )+s3+( s4=sp2[ 7 ] )+2 )>>2;
      dp[ 7 ] = ( unsigned int )( s2+( s1=sp[ 8 ] )+s4+( s3=sp2[ 8 ] )+2 )>>2;
      dp[ 8 ] = ( unsigned int )( s1+( s2=sp[ 9 ] )+s3+( s4=sp2[ 9 ] )+2 )>>2;
      dp[ 9 ] = ( unsigned int )( s2+( s1=sp[ 10 ] )+s4+( s3=sp2[ 10 ] )+2 )>>2;
      dp[ 10 ] = ( unsigned int )( s1+( s2=sp[ 11 ] )+s3+( s4=sp2[ 11 ] )+2 )>>2;
      dp[ 11 ] = ( unsigned int )( s2+( s1=sp[ 12 ] )+s4+( s3=sp2[ 12 ] )+2 )>>2;
      dp[ 12 ] = ( unsigned int )( s1+( s2=sp[ 13 ] )+s3+( s4=sp2[ 13 ] )+2 )>>2;
      dp[ 13 ] = ( unsigned int )( s2+( s1=sp[ 14 ] )+s4+( s3=sp2[ 14 ] )+2 )>>2;
      dp[ 14 ] = ( unsigned int )( s1+( s2=sp[ 15 ] )+s3+( s4=sp2[ 15 ] )+2 )>>2;
      dp[ 15 ] = ( unsigned int )( s2+sp[ 16 ]+s4+sp2[ 16 ]+2 )>>2;
      sp+= lx2;
      sp2+= lx2;
      dp+= lx2;
   }
}
/***************************************************************************/
static inline void rec4c( unsigned char *s, unsigned char *d, int lx, int lx2, int h )
{
   unsigned char *dp, *sp, *sp2;
   unsigned int s1, s2, s3, s4;
   int j;
   //
   sp = s;
   sp2 = s+lx;
   dp = d;
   for ( j=0; j<h; j++ )
   {
      s1=sp[ 0 ]; s3=sp2[ 0 ];
      dp[ 0 ] = ( unsigned int )( s1+( s2=sp[ 1 ] )+s3+( s4=sp2[ 1 ] )+2 )>>2;
      dp[ 1 ] = ( unsigned int )( s2+( s1=sp[ 2 ] )+s4+( s3=sp2[ 2 ] )+2 )>>2;
      dp[ 2 ] = ( unsigned int )( s1+( s2=sp[ 3 ] )+s3+( s4=sp2[ 3 ] )+2 )>>2;
      dp[ 3 ] = ( unsigned int )( s2+( s1=sp[ 4 ] )+s4+( s3=sp2[ 4 ] )+2 )>>2;
      dp[ 4 ] = ( unsigned int )( s1+( s2=sp[ 5 ] )+s3+( s4=sp2[ 5 ] )+2 )>>2;
      dp[ 5 ] = ( unsigned int )( s2+( s1=sp[ 6 ] )+s4+( s3=sp2[ 6 ] )+2 )>>2;
      dp[ 6 ] = ( unsigned int )( s1+( s2=sp[ 7 ] )+s3+( s4=sp2[ 7 ] )+2 )>>2;
      dp[ 7 ] = ( unsigned int )( s2+sp[ 8 ]+s4+sp2[ 8 ]+2 )>>2;
      sp+= lx2;
      sp2+= lx2;
      dp+= lx2;
   }
}
/***************************************************************************/
static inline void rec4a( unsigned char *s, unsigned char *d, int lx, int lx2, int h )
{
   unsigned char *dp=d, *sp=s, *sp2=s+lx;
   unsigned int s1, s2, s3, s4;
   int j;
   //
   for ( j=0; j<h; j++ )
   {
      s1=sp[ 0 ]; s3=sp2[ 0 ];
      dp[ 0 ] = ( dp[ 0 ] + ( ( unsigned int )( s1+( s2=sp[ 1 ] )+s3+( s4=sp2[ 1 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 1 ] = ( dp[ 1 ] + ( ( unsigned int )( s2+( s1=sp[ 2 ] )+s4+( s3=sp2[ 2 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 2 ] = ( dp[ 2 ] + ( ( unsigned int )( s1+( s2=sp[ 3 ] )+s3+( s4=sp2[ 3 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 3 ] = ( dp[ 3 ] + ( ( unsigned int )( s2+( s1=sp[ 4 ] )+s4+( s3=sp2[ 4 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 4 ] = ( dp[ 4 ] + ( ( unsigned int )( s1+( s2=sp[ 5 ] )+s3+( s4=sp2[ 5 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 5 ] = ( dp[ 5 ] + ( ( unsigned int )( s2+( s1=sp[ 6 ] )+s4+( s3=sp2[ 6 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 6 ] = ( dp[ 6 ] + ( ( unsigned int )( s1+( s2=sp[ 7 ] )+s3+( s4=sp2[ 7 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 7 ] = ( dp[ 7 ] + ( ( unsigned int )( s2+( s1=sp[ 8 ] )+s4+( s3=sp2[ 8 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 8 ] = ( dp[ 8 ] + ( ( unsigned int )( s1+( s2=sp[ 9 ] )+s3+( s4=sp2[ 9 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 9 ] = ( dp[ 9 ] + ( ( unsigned int )( s2+( s1=sp[ 10 ] )+s4+( s3=sp2[ 10 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 10 ] = ( dp[ 10 ] + ( ( unsigned int )( s1+( s2=sp[ 11 ] )+s3+( s4=sp2[ 11 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 11 ] = ( dp[ 11 ] + ( ( unsigned int )( s2+( s1=sp[ 12 ] )+s4+( s3=sp2[ 12 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 12 ] = ( dp[ 12 ] + ( ( unsigned int )( s1+( s2=sp[ 13 ] )+s3+( s4=sp2[ 13 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 13 ] = ( dp[ 13 ] + ( ( unsigned int )( s2+( s1=sp[ 14 ] )+s4+( s3=sp2[ 14 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 14 ] = ( dp[ 14 ] + ( ( unsigned int )( s1+( s2=sp[ 15 ] )+s3+( s4=sp2[ 15 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 15 ] = ( dp[ 15 ] + ( ( unsigned int )( s2+sp[ 16 ]+s4+sp2[ 16 ]+2 )>>2 ) + 1 )>>1;
      sp+= lx2;
      sp2+= lx2;
      dp+= lx2;
   }
}
/***************************************************************************/
static inline void rec4ac( unsigned char *s, unsigned char *d, int lx, int lx2, int h )
{
   unsigned char *dp=d, *sp=s, *sp2=s+lx;
   unsigned int s1, s2, s3, s4;
   int j;
   //
   for ( j=0; j<h; j++ )
   {
      s1=sp[ 0 ]; s3=sp2[ 0 ];
      dp[ 0 ] = ( dp[ 0 ] + ( ( unsigned int )( s1+( s2=sp[ 1 ] )+s3+( s4=sp2[ 1 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 1 ] = ( dp[ 1 ] + ( ( unsigned int )( s2+( s1=sp[ 2 ] )+s4+( s3=sp2[ 2 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 2 ] = ( dp[ 2 ] + ( ( unsigned int )( s1+( s2=sp[ 3 ] )+s3+( s4=sp2[ 3 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 3 ] = ( dp[ 3 ] + ( ( unsigned int )( s2+( s1=sp[ 4 ] )+s4+( s3=sp2[ 4 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 4 ] = ( dp[ 4 ] + ( ( unsigned int )( s1+( s2=sp[ 5 ] )+s3+( s4=sp2[ 5 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 5 ] = ( dp[ 5 ] + ( ( unsigned int )( s2+( s1=sp[ 6 ] )+s4+( s3=sp2[ 6 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 6 ] = ( dp[ 6 ] + ( ( unsigned int )( s1+( s2=sp[ 7 ] )+s3+( s4=sp2[ 7 ] )+2 )>>2 ) + 1 )>>1;
      dp[ 7 ] = ( dp[ 7 ] + ( ( unsigned int )( s2+sp[ 8 ]+s4+sp2[ 8 ]+2 )>>2 ) + 1 )>>1;
      sp+= lx2;
      sp2+= lx2;
      dp+= lx2;
   }
}
/***************************************************************************/
static inline void recon_comp( mpeg_video_t *video, unsigned char *src, unsigned char *dst, int lx, int lx2, int w, int h, int x, int y, int dx, int dy, int addflag )
{
   int switcher;
   unsigned char *s, *d;
   //
   switcher = ( dx & 1 ) << 3 | ( dy & 1 ) << 2 | w;
   if ( addflag )
      switcher |= 2; 
   s = src + lx * ( y + ( dy >> 1 ) ) + x + ( dx >> 1 );
   d = dst + lx * y + x;
   {
      switch( switcher )
      {
         case 0x3:
            reca( s, d, lx2, h );
            break;
         case 0x2:
            recac( s, d, lx2, h );
            break;
         case 0x1:
            rec( s, d, lx2, h );
            break;
         case 0x0:
            recc( s, d, lx2, h );
            break;
         case 0x7:
            recva( s, d, lx, lx2, h );
            break;
         case 0x6:
            recvac( s, d, lx, lx2, h );
            break;
         case 0x5:
            recvmpeg( s, d, lx, lx2, h );
            break;
         case 0x4:
            recvc( s, d, lx, lx2, h );
            break;
         case 0x9:
            rech( s, d, lx2, h );
            break;
         case 0x8:
            rechc( s, d, lx2, h );
            break;
            default:
         break;
      }
   }
   switch( switcher ) 
   {
      case 0xb:
         recha( s, d, lx2, h );
         break;
      case 0xa:
         rechac( s, d, lx2, h );
         break;
      case 0xf:
         rec4a( s, d, lx, lx2, h );
         break;
      case 0xe:
         rec4ac( s, d, lx, lx2, h );
         break;
      case 0xd:
         rec4( s, d, lx, lx2, h );
         break;
      case 0xc:
         rec4c( s, d, lx, lx2, h );
         break;
         default:
      break;
   }
}
/***************************************************************************/
static void recon( mpeg_video_t *video, unsigned char *src[], int sfield, unsigned char *dst[], int dfield, int lx, int lx2, int w, int h, int x, int y, int dx, int dy, int addflag )
{
   recon_comp( video, ( src[ 0 ] + ( sfield ? ( lx2 >> 1 ) : 0 ) ), 
   dst[ 0 ] + ( dfield ? ( lx2 >> 1 ) : 0 ), 
   lx, lx2, w, h, x, y, dx, dy, addflag );
   if ( video->chroma_format != CHROMA444 )
   {
      lx >>= 1; 
      dx /= 2; 
      lx2 >>= 1; 
      w = 0; 
      x >>= 1; 
   }
   if ( video->chroma_format == CHROMA420 )
   {
      h >>= 1; 
      dy /= 2; 
      y >>= 1; 
   }
   recon_comp( video, ( src[ 1 ] + ( sfield ? ( lx2 >> 1 ) : 0 ) ), dst[ 1 ] + ( dfield ? ( lx2 >> 1 ) : 0 ), lx, lx2, w, h, x, y, dx, dy, addflag );
   recon_comp( video, ( src[ 2 ] + ( sfield ? ( lx2 >> 1 ) : 0 ) ), dst[ 2 ] + ( dfield ? ( lx2 >> 1 ) : 0 ), lx, lx2, w, h, x, y, dx, dy, addflag );
}
/***************************************************************************/
#define WIDTH                    1
static inline int mpeg_video_reconstruct( mpeg_video_t *video, int bx, int by, int mb_type, int motion_type, int PMV[ 2 ][ 2 ][ 2 ], int mv_field_sel[ 2 ][ 2 ], int dmvector[ 2 ], int stwtype )
{
   int currentfield;
   unsigned char **predframe;
   int DMV[ 2 ][ 2 ];
   int stwtop, stwbot;
   //
   stwtop = stwtype % 3; 
   stwbot = stwtype / 3;
   if ( ( mb_type & MB_FORWARD ) || ( video->pict_type == P_TYPE ) )
   {
      if ( video->pict_struct == FRAME_PICTURE )
      {
         if ( ( motion_type == MC_FRAME ) || !( mb_type & MB_FORWARD ) )
         {
            {
               if ( stwtop < 2 )
                  recon( video, video->oldrefframe, 0, video->newframe, 0, video->coded_picture_width, video->coded_picture_width << 1, WIDTH, 8, bx, by, PMV[ 0 ][ 0 ][ 0 ], PMV[ 0 ][ 0 ][ 1 ], stwtop );
               if ( stwbot < 2 )
                  recon( video, video->oldrefframe, 1, video->newframe, 1, video->coded_picture_width, video->coded_picture_width << 1, WIDTH, 8, bx, by, PMV[ 0 ][ 0 ][ 0 ], PMV[ 0 ][ 0 ][ 1 ], stwbot );
            }
         }
         else if ( motion_type == MC_FIELD ) 
         {
            if ( stwtop < 2 )
               recon( video, video->oldrefframe, mv_field_sel[ 0 ][ 0 ], video->newframe, 0, video->coded_picture_width << 1, video->coded_picture_width << 1, WIDTH, 8, bx, by >> 1, PMV[ 0 ][ 0 ][ 0 ], PMV[ 0 ][ 0 ][ 1 ] >> 1, stwtop );
            if ( stwbot < 2 )
               recon( video, video->oldrefframe, mv_field_sel[ 1 ][ 0 ], video->newframe, 1, video->coded_picture_width << 1, video->coded_picture_width << 1, WIDTH, 8, bx, by >> 1, PMV[ 1 ][ 0 ][ 0 ], PMV[ 1 ][ 0 ][ 1 ] >> 1, stwbot );
         }
         else if ( motion_type == MC_DMV )
         {
            mpeg_video_calc_dmv( video, DMV, dmvector, PMV[ 0 ][ 0 ][ 0 ], PMV[ 0 ][ 0 ][ 1 ] >> 1 );
            if ( stwtop < 2 )
            {
               recon( video, video->oldrefframe, 0, video->newframe, 0, video->coded_picture_width << 1, video->coded_picture_width << 1, WIDTH, 8, bx, by>>1, PMV[ 0 ][ 0 ][ 0 ], PMV[ 0 ][ 0 ][ 1 ] >> 1, 0 );
               recon( video, video->oldrefframe, 1, video->newframe, 0, video->coded_picture_width << 1, video->coded_picture_width << 1, WIDTH, 8, bx, by>>1, DMV[ 0 ][ 0 ], DMV[ 0 ][ 1 ], 1 );
            }
            if ( stwbot < 2 )
            {
               recon( video, video->oldrefframe, 1, video->newframe, 1, video->coded_picture_width << 1, video->coded_picture_width << 1, WIDTH, 8, bx, by>>1, PMV[ 0 ][ 0 ][ 0 ], PMV[ 0 ][ 0 ][ 1 ]>>1, 0 );
               recon( video, video->oldrefframe, 0, video->newframe, 1, video->coded_picture_width << 1, video->coded_picture_width<<1, WIDTH, 8, bx, by>>1, DMV[ 1 ][ 0 ], DMV[ 1 ][ 1 ], 1 );
            }
         }
      }
      else 
      {
         currentfield = ( video->pict_struct == BOTTOM_FIELD );
         if ( ( video->pict_type == P_TYPE ) && video->secondfield && ( currentfield != mv_field_sel[ 0 ][ 0 ] ) )
            predframe = video->refframe;
         else
            predframe = video->oldrefframe; 
         if ( ( motion_type == MC_FIELD ) || !( mb_type & MB_FORWARD ) )
         {
            if ( stwtop < 2 )
               recon( video, predframe, mv_field_sel[ 0 ][ 0 ], video->newframe, 0, video->coded_picture_width << 1, video->coded_picture_width << 1, WIDTH, 16, bx, by, PMV[ 0 ][ 0 ][ 0 ], PMV[ 0 ][ 0 ][ 1 ], stwtop );
         }
         else 
            if ( motion_type == MC_16X8 )
         {
            if ( stwtop < 2 )
            {
               recon( video, predframe, mv_field_sel[ 0 ][ 0 ], video->newframe, 0, video->coded_picture_width << 1, video->coded_picture_width << 1, WIDTH, 8, bx, by, PMV[ 0 ][ 0 ][ 0 ], PMV[ 0 ][ 0 ][ 1 ], stwtop );
               if ( ( video->pict_type==P_TYPE ) && video->secondfield && ( currentfield!=mv_field_sel[ 1 ][ 0 ] ) )
                  predframe = video->refframe;
               else
                  predframe = video->oldrefframe; 
               recon( video, predframe, mv_field_sel[ 1 ][ 0 ], video->newframe, 0, video->coded_picture_width << 1, video->coded_picture_width << 1, WIDTH, 8, bx, by+8, PMV[ 1 ][ 0 ][ 0 ], PMV[ 1 ][ 0 ][ 1 ], stwtop );
            }
         }
         else if ( motion_type == MC_DMV ) 
         {
            if ( video->secondfield )
               predframe = video->refframe; 
            else
               predframe = video->oldrefframe; 
            mpeg_video_calc_dmv( video, DMV, dmvector, PMV[ 0 ][ 0 ][ 0 ], PMV[ 0 ][ 0 ][ 1 ] );
            recon( video, video->oldrefframe, currentfield, video->newframe, 0, video->coded_picture_width << 1, video->coded_picture_width << 1, WIDTH, 16, bx, by, PMV[ 0 ][ 0 ][ 0 ], PMV[ 0 ][ 0 ][ 1 ], 0 );
            recon( video, predframe, !currentfield, video->newframe, 0, video->coded_picture_width << 1, video->coded_picture_width << 1, WIDTH, 16, bx, by, DMV[ 0 ][ 0 ], DMV[ 0 ][ 1 ], 1 );
         }
      }
      stwtop = stwbot = 1;
   }
   if ( mb_type & MB_BACKWARD )
   {
      if ( video->pict_struct == FRAME_PICTURE )
      {
         if ( motion_type == MC_FRAME )
         {
            if ( stwtop < 2 )
               recon( video, video->refframe, 0, video->newframe, 0, video->coded_picture_width, video->coded_picture_width << 1, WIDTH, 8, bx, by, PMV[ 0 ][ 1 ][ 0 ], PMV[ 0 ][ 1 ][ 1 ], stwtop );
            if ( stwbot < 2 )
               recon( video, video->refframe, 1, video->newframe, 1, video->coded_picture_width, video->coded_picture_width << 1, WIDTH, 8, bx, by, PMV[ 0 ][ 1 ][ 0 ], PMV[ 0 ][ 1 ][ 1 ], stwbot );
         }
         else 
         {
            if ( stwtop < 2 )
            {
               recon( video, video->refframe, mv_field_sel[ 0 ][ 1 ], video->newframe, 0, ( video->coded_picture_width << 1 ), ( video->coded_picture_width<<1 ), WIDTH, 8, bx, ( by >> 1 ), PMV[ 0 ][ 1 ][ 0 ], ( PMV[ 0 ][ 1 ][ 1 ] >> 1 ), stwtop );
            }
            if ( stwbot < 2 )
            {
               recon( video, video->refframe, mv_field_sel[ 1 ][ 1 ], video->newframe, 1, ( video->coded_picture_width << 1 ), ( video->coded_picture_width << 1 ), WIDTH, 8, bx, ( by>>1 ), PMV[ 1 ][ 1 ][ 0 ], ( PMV[ 1 ][ 1 ][ 1 ]>>1 ), stwbot );
            }
         }
      }
      else 
      {
         if ( motion_type == MC_FIELD )
         {
            recon( video, video->refframe, mv_field_sel[ 0 ][ 1 ], video->newframe, 0, video->coded_picture_width << 1, video->coded_picture_width << 1, WIDTH, 16, bx, by, PMV[ 0 ][ 1 ][ 0 ], PMV[ 0 ][ 1 ][ 1 ], stwtop );
         }
         else if ( motion_type==MC_16X8 )
         {
            recon( video, video->refframe, mv_field_sel[ 0 ][ 1 ], video->newframe, 0, video->coded_picture_width << 1, video->coded_picture_width << 1, WIDTH, 8, bx, by, PMV[ 0 ][ 1 ][ 0 ], PMV[ 0 ][ 1 ][ 1 ], stwtop );
            recon( video, video->refframe, mv_field_sel[ 1 ][ 1 ], video->newframe, 0, video->coded_picture_width << 1, video->coded_picture_width << 1, WIDTH, 8, bx, by+8, PMV[ 1 ][ 1 ][ 0 ], PMV[ 1 ][ 1 ][ 1 ], stwtop );
         }
      }
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_next_code( mpeg_bits_t* stream, unsigned int code )
{
   while( !mpeg_bits_eof( stream ) && mpeg_bits_showbits32_noptr( stream ) != code )
   {
      mpeg_bits_getbyte_noptr( stream );
   }
   //
   return mpeg_bits_eof( stream );
}
/***************************************************************************/
static inline int mpeg_video_prev_code( mpeg_bits_t* stream, unsigned int code )
{
   while( !mpeg_bits_bof( stream ) && mpeg_bits_showbits_reverse( stream, 32 ) != code )
   {
      mpeg_bits_getbits_reverse( stream, 8 );
   }
   //
   return mpeg_bits_bof( stream );
}
/***************************************************************************/
static inline long mpeg_video_goptimecode_to_frame( mpeg_video_t *video )
{
   return ( long )( video->gop_timecode.hour * 3600 * video->frame_rate + video->gop_timecode.minute * 60 * video->frame_rate +video->gop_timecode.second * video->frame_rate +video->gop_timecode.frame ) - 1 - video->first_frame;
}
/***************************************************************************/
static inline int mpeg_video_match_refframes( mpeg_video_t *video )
{
   unsigned char *dst, *src;
   int i, size;
   //
   for( i = 0; i < 3; i++ )
   {
      if ( video->newframe[ i ] )
      {
         if ( video->newframe[ i ] == video->refframe[ i ] )
         {
            src = video->refframe[ i ];
            dst = video->oldrefframe[ i ];
         }
         else
         {
            src = video->oldrefframe[ i ];
            dst = video->refframe[ i ];
         }
         if ( i == 0 )
            size = video->coded_picture_width * video->coded_picture_height + 32 * video->coded_picture_width;
         else 
            size = video->chrom_width * video->chrom_height + 32 * video->chrom_width;
         memcpy( dst, src, size );
      }
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_seek( mpeg_video_t *video )
{
   long this_gop_start;
   int result = 0;
   MPEG *file_mpeg = ( MPEG * ) video->file_mpeg;
   mpeg_bits_t *vstream = video->vstream;
   double percentage;
   long frame_number;
   int match_refframes = 1;
   //
   if ( video->percentage_seek >= 0 )
   {
      percentage = video->percentage_seek;
      video->percentage_seek = -1;
      mpeg_bits_seek_percentage( vstream, percentage );
      mpeg_bits_start_reverse( vstream );
      result = mpeg_video_prev_code( vstream, MPEG3_GOP_START_CODE );
      if ( !result ) mpeg_bits_getbits_reverse( vstream, 32 );
         mpeg_bits_start_forward( vstream );
      if ( mpeg_bits_tell_percentage( vstream ) < 0 ) mpeg_bits_seek_percentage( vstream, 0 );
         result = 0;
	  while( !result && mpeg_bits_tell_percentage( vstream ) < percentage )
      {
         result = mpeg_video_read_frame_backend( video, 0 );
         if ( match_refframes )
            mpeg_video_match_refframes( video );
         match_refframes = 0;
      }
   }
   else if ( video->frame_seek >= 0 )
   {
      frame_number = video->frame_seek;
      video->frame_seek = -1;
      if ( frame_number < 0 )
         frame_number = 0;
      if ( frame_number > video->maxframe )
         frame_number = video->maxframe;
      if ( frame_number < 16 )
      {
         video->repeat_count = video->current_repeat = 0;
         mpeg_bits_seek_start( vstream );
         video->framenum = 0;
         result = mpeg_video_drop_frames( video, frame_number - video->framenum );
      }
      else
      {
         if ( ( frame_number < video->framenum || frame_number - video->framenum > MPEG3_SEEK_THRESHOLD ) )
         {
            if ( file_mpeg->is_video_stream )
            {
               mpeg_vtrack_t *track = ( mpeg_vtrack_t * ) video->track;
               long byte = ( long )( ( float )( mpeg_demuxer_total_bytes( vstream->demuxer ) / track->total_frames ) * frame_number );
               long minimum = 65535;
               int done = 0;
               //
               do
               {
                  result = mpeg_bits_seek_byte( vstream, byte );
                  mpeg_bits_start_reverse( vstream );
                  if ( !result )
                     result = mpeg_video_prev_code( vstream, MPEG3_GOP_START_CODE );
                  mpeg_bits_start_forward( vstream );
                  mpeg_bits_getbits( vstream, 8 );
                  if ( !result )
                     result = mpeg_video_getgophdr( video );
                  this_gop_start = mpeg_video_goptimecode_to_frame( video );
                  if ( labs( this_gop_start - frame_number ) >= labs( minimum ) ) 
                     done = 1;
                  else
                  {
                     minimum = this_gop_start - frame_number;
                     byte += ( long )( ( float )( frame_number - this_gop_start ) * ( float )( mpeg_demuxer_total_bytes( vstream->demuxer ) / track->total_frames ) );
                     if ( byte < 0 )
                        byte = 0;
                  }
		   } while( !result && !done );
               if ( !result )
               {
                  video->framenum = this_gop_start;
                  result = mpeg_video_drop_frames( video, frame_number - video->framenum );
               }
            }
            else
            {
               mpeg_bits_seek_time( vstream, ( double )frame_number / video->frame_rate );
               percentage = mpeg_bits_tell_percentage( vstream );
               mpeg_bits_start_reverse( vstream );
               mpeg_video_prev_code( vstream, MPEG3_GOP_START_CODE );
               mpeg_bits_getbits_reverse( vstream, 32 );
               mpeg_bits_start_forward( vstream );
		   while( !result && mpeg_bits_tell_percentage( vstream ) < percentage )
               {
                  result = mpeg_video_read_frame_backend( video, 0 );
                  if ( match_refframes )
                     mpeg_video_match_refframes( video );
                  match_refframes = 0;
               }
            }
            video->framenum = frame_number;
         }
         else
         {
            mpeg_video_drop_frames( video, frame_number - video->framenum );
         }
      }
   }
   //
   return result;
}
/***************************************************************************/
static inline int mpeg_video_drop_frames( mpeg_video_t *video, long frames )
{
   int result = 0;
   long frame_number = video->framenum + frames;
   //
   while( !result && frame_number > video->framenum )
   {
      result = mpeg_video_read_frame_backend( video, frame_number - video->framenum );
   }
   //
   return result;
}
/***************************************************************************/
static inline int mpeg_new_slice_buffer( mpeg_slice_buffer_t *slice_buffer )
{
   slice_buffer->data = ( unsigned char * ) calloc( 1, 1024 );
   slice_buffer->buffer_size = 0;
   slice_buffer->buffer_allocation = 1024;
   slice_buffer->current_position = 0;
   slice_buffer->bits_size = 0;
   slice_buffer->bits = 0;
   slice_buffer->done = 0;
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_delete_slice_buffer( mpeg_slice_buffer_t *slice_buffer )
{
   free( slice_buffer->data );
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_expand_slice_buffer( mpeg_slice_buffer_t *slice_buffer )
{
   int i;
   unsigned char *new_buffer = ( unsigned char * ) calloc( 1, slice_buffer->buffer_allocation * 2 );
   //
   for( i = 0; i < slice_buffer->buffer_size; i++ )
      new_buffer[ i ] = slice_buffer->data[ i ];
   free( slice_buffer->data );
   slice_buffer->data = new_buffer;
   slice_buffer->buffer_allocation *= 2;
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_video_addblock( mpeg_slice_t *slice, mpeg_video_t *video, int comp, int bx, int by, int dct_type, int addflag )
{
   int cc, i, iincr;
   unsigned char *rfp;
   short *bp;
   //
   cc = ( comp < 4 ) ? 0 : ( comp & 1 ) + 1;
   if ( cc == 0 )
   {
      if ( video->pict_struct == FRAME_PICTURE )
      {
         if ( dct_type )
         {
            rfp = video->newframe[ 0 ] +
            video->coded_picture_width * ( by + ( ( comp & 2 ) >> 1 ) ) + bx + ( ( comp & 1 ) << 3 );
            iincr = ( video->coded_picture_width << 1 );
         }
         else
         {
            rfp = video->newframe[ 0 ] + 
            video->coded_picture_width * ( by + ( ( comp & 2 ) << 2 ) ) + bx + ( ( comp & 1 ) << 3 );
            iincr = video->coded_picture_width;
         }
      }
      else 
      {
         rfp = video->newframe[ 0 ] + ( video->coded_picture_width << 1 ) * ( by + ( ( comp & 2 ) << 2 ) ) + bx + ( ( comp & 1 ) << 3 );
         iincr = ( video->coded_picture_width << 1 );
      }
   }
   else 
   {
      if ( video->chroma_format != CHROMA444 )
         bx >>= 1;
      if ( video->chroma_format == CHROMA420 )
         by >>= 1;
      if ( video->pict_struct == FRAME_PICTURE )
      {
         if ( dct_type && ( video->chroma_format != CHROMA420 ) )
         {
            rfp = video->newframe[ cc ] + video->chrom_width * ( by + ( ( comp & 2 ) >> 1 ) ) + bx + ( comp & 8 );
            iincr = ( video->chrom_width << 1 );
         }
         else 
         {
            rfp = video->newframe[ cc ] + video->chrom_width * ( by + ( ( comp & 2 ) << 2 ) ) + bx + ( comp & 8 );
            iincr = video->chrom_width;
         }
      }
      else 
      {
         rfp = video->newframe[ cc ] + ( video->chrom_width << 1 ) * ( by + ( ( comp & 2 ) << 2 ) ) + bx + ( comp & 8 );
         iincr = ( video->chrom_width << 1 );
      }
   }
   bp = slice->block[ comp ];
   if ( addflag )
   {
      for( i = 0; i < 8; i++ )
      {
         rfp[ 0 ] = CLIP( bp[ 0 ] + rfp[ 0 ] );
         rfp[ 1 ] = CLIP( bp[ 1 ] + rfp[ 1 ] );
         rfp[ 2 ] = CLIP( bp[ 2 ] + rfp[ 2 ] );
         rfp[ 3 ] = CLIP( bp[ 3 ] + rfp[ 3 ] );
         rfp[ 4 ] = CLIP( bp[ 4 ] + rfp[ 4 ] );
         rfp[ 5 ] = CLIP( bp[ 5 ] + rfp[ 5 ] );
         rfp[ 6 ] = CLIP( bp[ 6 ] + rfp[ 6 ] );
         rfp[ 7 ] = CLIP( bp[ 7 ] + rfp[ 7 ] );
         rfp += iincr;
         bp += 8;
      }
   }
   else 
   {
      for( i = 0; i < 8; i++ )
      {
         rfp[ 0 ] = CLIP( bp[ 0 ] + 128 );
         rfp[ 1 ] = CLIP( bp[ 1 ] + 128 );
         rfp[ 2 ] = CLIP( bp[ 2 ] + 128 );
         rfp[ 3 ] = CLIP( bp[ 3 ] + 128 );
         rfp[ 4 ] = CLIP( bp[ 4 ] + 128 );
         rfp[ 5 ] = CLIP( bp[ 5 ] + 128 );
         rfp[ 6 ] = CLIP( bp[ 6 ] + 128 );
         rfp[ 7 ] = CLIP( bp[ 7 ] + 128 );
         rfp+= iincr;
         bp += 8;
      }
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_decode_slice( mpeg_slice_t *slice )
{
   mpeg_video_t *video = ( mpeg_video_t * ) slice->video;
   int comp;
   int mb_type, cbp, motion_type = 0, dct_type;
   int macroblock_address, mba_inc, mba_max;
   int slice_vert_pos_ext;
   unsigned int code;
   int bx, by;
   int dc_dct_pred[ 3 ];
   int mv_count, mv_format, mvscale;
   int pmv[ 2 ][ 2 ][ 2 ], mv_field_sel[ 2 ][ 2 ];
   int dmv, dmvector[ 2 ];
   int qs;
   int stwtype, stwclass; 
   int snr_cbp;
   int i;
   mpeg_slice_buffer_t *slice_buffer = slice->slice_buffer;
   //
   mba_max = video->mb_width * video->mb_height;
   if ( video->pict_struct != FRAME_PICTURE )
      mba_max >>= 1; 
   macroblock_address = 0; 
   mba_inc = 0;
   slice->fault = 0;
   code = mpeg_slice_getbits( slice_buffer, 32 );
   slice_vert_pos_ext = mpeg_video_getslicehdr( slice, video );
   dc_dct_pred[ 0 ] = dc_dct_pred[ 1 ] = dc_dct_pred[ 2 ] = 0;
   pmv[ 0 ][ 0 ][ 0 ] = pmv[ 0 ][ 0 ][ 1 ] = pmv[ 1 ][ 0 ][ 0 ] = pmv[ 1 ][ 0 ][ 1 ] = 0;
   pmv[ 0 ][ 1 ][ 0 ] = pmv[ 0 ][ 1 ][ 1 ] = pmv[ 1 ][ 1 ][ 0 ] = pmv[ 1 ][ 1 ][ 1 ] = 0;
   for( i = 0; slice_buffer->current_position < slice_buffer->buffer_size; i++ )
   {
      if ( mba_inc == 0 )
      {
         if ( !mpeg_slice_showbits( slice_buffer, 23 ) )
            return 0;
         mba_inc = mpeg_video_get_macroblock_address( slice );
         if ( slice->fault )
            return 1;
         if ( i == 0 )
         {
            macroblock_address = ( ( slice_vert_pos_ext << 7 ) + ( code & 255 ) - 1 ) * video->mb_width + mba_inc - 1;
            mba_inc = 1;
         }
      }
      if ( slice->fault )
         return 1;
      if ( macroblock_address >= mba_max )
      {
         return 1;
      }
      if ( mba_inc == 1 )
      {
         mpeg_video_macroblock_modes( slice, video, &mb_type, &stwtype, &stwclass, &motion_type, &mv_count, &mv_format, &dmv, &mvscale, &dct_type );
         if ( slice->fault )
            return 1;
         if ( mb_type & MB_QUANT )
         {
            qs = mpeg_slice_getbits( slice_buffer, 5 );
            if ( video->mpeg_2 )
               slice->quant_scale = video->qscale_type ? mpeg_non_linear_mquant_table[ qs ] : ( qs << 1 );
            else 
               slice->quant_scale = qs;
            if ( video->scalable_mode == SC_DP )
               slice->quant_scale = slice->quant_scale;
         }
         if ( ( mb_type & MB_FORWARD ) || ( ( mb_type & MB_INTRA ) && video->conceal_mv ) )
         {
            if ( video->mpeg_2 )
               mpeg_video_motion_vectors( slice, video, pmv, dmvector, mv_field_sel, 0, mv_count, mv_format, video->h_forw_r_size, video->v_forw_r_size, dmv, mvscale );
            else
               mpeg_video_motion_vector( slice, video, pmv[ 0 ][ 0 ], dmvector, video->forw_r_size, video->forw_r_size, 0, 0, video->full_forw );
         }
         if ( slice->fault )
            return 1;
         if ( mb_type & MB_BACKWARD )
         {
            if ( video->mpeg_2 )
               mpeg_video_motion_vectors( slice, video, pmv, dmvector, mv_field_sel, 1, mv_count, mv_format, video->h_back_r_size, video->v_back_r_size, 0, mvscale );
            else
               mpeg_video_motion_vector( slice, video, pmv[ 0 ][ 1 ], dmvector, video->back_r_size, video->back_r_size, 0, 0, video->full_back );
         }
         if ( slice->fault )
            return 1;
         if ( ( mb_type & MB_INTRA ) && video->conceal_mv )
            mpeg_slice_flushbit( slice_buffer );
         if ( mb_type & MB_PATTERN )
         {
            cbp = mpeg_video_get_cbp( slice );
            if ( video->chroma_format == CHROMA422 )
            {
               cbp = ( cbp << 2 ) | mpeg_slice_getbits2( slice_buffer ); 
            }
            else if ( video->chroma_format == CHROMA444 )
            {
               cbp = ( cbp << 6 ) | mpeg_slice_getbits( slice_buffer, 6 ); 
            }
         }
         else
            cbp = ( mb_type & MB_INTRA ) ? ( ( 1 << video->blk_cnt ) - 1 ) : 0;
         if ( slice->fault )
            return 1;
         mpeg_video_clearblock( slice, 0, video->blk_cnt );
         for( comp = 0; comp < video->blk_cnt; comp++ )
         {
            if ( cbp & ( 1 << ( video->blk_cnt - comp - 1 ) ) )
            {
               if ( mb_type & MB_INTRA )
               {
                  if ( video->mpeg_2 )
                     mpeg_video_getmpeg_2intrablock( slice, video, comp, dc_dct_pred );
                  else
                     mpeg_video_getintrablock( slice, video, comp, dc_dct_pred );
               }
               else 
               {
                  if ( video->mpeg_2 ) 
                     mpeg_video_getmpeg_2interblock( slice, video, comp );
                  else 
                     mpeg_video_getinterblock( slice, video, comp );
               }
               if ( slice->fault )
                  return 1;
            }
         }
         if ( !( mb_type & MB_INTRA ) )
            dc_dct_pred[ 0 ] = dc_dct_pred[ 1 ] = dc_dct_pred[ 2 ] = 0;
         if ( ( mb_type & MB_INTRA ) && !video->conceal_mv )
         {
            pmv[ 0 ][ 0 ][ 0 ] = pmv[ 0 ][ 0 ][ 1 ] = pmv[ 1 ][ 0 ][ 0 ] = pmv[ 1 ][ 0 ][ 1 ] = 0;
            pmv[ 0 ][ 1 ][ 0 ] = pmv[ 0 ][ 1 ][ 1 ] = pmv[ 1 ][ 1 ][ 0 ] = pmv[ 1 ][ 1 ][ 1 ] = 0;
         }
         if ( ( video->pict_type == P_TYPE ) && !( mb_type & ( MB_FORWARD | MB_INTRA ) ) )
         {
            pmv[ 0 ][ 0 ][ 0 ] = pmv[ 0 ][ 0 ][ 1 ] = pmv[ 1 ][ 0 ][ 0 ] = pmv[ 1 ][ 0 ][ 1 ] = 0;
            if ( video->pict_struct == FRAME_PICTURE ) 
               motion_type = MC_FRAME;
            else
            {
               motion_type = MC_FIELD;
               mv_field_sel[ 0 ][ 0 ] = ( video->pict_struct == BOTTOM_FIELD );
            }
         }
         if ( stwclass == 4 )
         {
            pmv[ 0 ][ 0 ][ 0 ] = pmv[ 0 ][ 0 ][ 1 ] = pmv[ 1 ][ 0 ][ 0 ] = pmv[ 1 ][ 0 ][ 1 ] = 0;
            pmv[ 0 ][ 1 ][ 0 ] = pmv[ 0 ][ 1 ][ 1 ] = pmv[ 1 ][ 1 ][ 0 ] = pmv[ 1 ][ 1 ][ 1 ] = 0;
         }
      }
      else 
      {
         mpeg_video_clearblock( slice, 0, video->blk_cnt );
         dc_dct_pred[ 0 ] = dc_dct_pred[ 1 ] = dc_dct_pred[ 2 ] = 0;
         if ( video->pict_type == P_TYPE )
            pmv[ 0 ][ 0 ][ 0 ] = pmv[ 0 ][ 0 ][ 1 ] = pmv[ 1 ][ 0 ][ 0 ] = pmv[ 1 ][ 0 ][ 1 ] = 0;
         if ( video->pict_struct == FRAME_PICTURE )
            motion_type = MC_FRAME;
         else
         {
            motion_type = MC_FIELD;
            mv_field_sel[ 0 ][ 0 ] = mv_field_sel[ 0 ][ 1 ] = ( video->pict_struct == BOTTOM_FIELD );
         }
         stwtype = ( video->pict_type == I_TYPE ) ? 8 : 0;
         mb_type &= ~MB_INTRA;
         cbp = 0; 
      }
      snr_cbp = 0;
      bx = 16 * ( macroblock_address % video->mb_width );
      by = 16 * ( macroblock_address / video->mb_width );
      if ( !( mb_type & MB_INTRA ) )
         mpeg_video_reconstruct( video, bx, by, mb_type, motion_type, pmv, mv_field_sel, dmvector, stwtype );
      for( comp = 0; comp < video->blk_cnt; comp++ )
      {
         if ( ( cbp | snr_cbp ) & ( 1 << ( video->blk_cnt - 1 - comp ) ) )
         {
            mpeg_video_idct_conversion( slice->block[ comp ] );
            mpeg_video_addblock( slice, video, comp, bx, by, dct_type, ( mb_type & MB_INTRA ) == 0 );
         }
      }
      macroblock_address++;
      mba_inc--;
   }
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_new_slice_decoder( void *video, mpeg_slice_t *slice )
{
   slice->video = video;
   slice->done = 0;
   //
   return 0;
}
/***************************************************************************/
static inline int mpeg_delete_slice_decoder( mpeg_slice_t *slice )
{
   slice->done = 1;
   //
   return 0;
}
/***************************************************************************/
static inline void ifft2 ( complex_t * buf )
{
   double r, i;
   //
   r = buf[ 0 ].real;
   i = buf[ 0 ].imag;
   buf[ 0 ].real += buf[ 1 ].real;
   buf[ 0 ].imag += buf[ 1 ].imag;
   buf[ 1 ].real = r - buf[ 1 ].real;
   buf[ 1 ].imag = i - buf[ 1 ].imag;
}
/***************************************************************************/
static inline void ifft4 ( complex_t * buf )
{
   double tmp1, tmp2, tmp3, tmp4, tmp5, tmp6, tmp7, tmp8;
   //
   tmp1 = buf[ 0 ].real + buf[ 1 ].real;
   tmp2 = buf[ 3 ].real + buf[ 2 ].real;
   tmp3 = buf[ 0 ].imag + buf[ 1 ].imag;
   tmp4 = buf[ 2 ].imag + buf[ 3 ].imag;
   tmp5 = buf[ 0 ].real - buf[ 1 ].real;
   tmp6 = buf[ 0 ].imag - buf[ 1 ].imag;
   tmp7 = buf[ 2 ].imag - buf[ 3 ].imag;
   tmp8 = buf[ 3 ].real - buf[ 2 ].real;
   buf[ 0 ].real = tmp1 + tmp2;
   buf[ 0 ].imag = tmp3 + tmp4;
   buf[ 2 ].real = tmp1 - tmp2;
   buf[ 2 ].imag = tmp3 - tmp4;
   buf[ 1 ].real = tmp5 + tmp7;
   buf[ 1 ].imag = tmp6 + tmp8;
   buf[ 3 ].real = tmp5 - tmp7;
   buf[ 3 ].imag = tmp6 - tmp8;
}
/***************************************************************************/
#define BUTTERFLY(               a0, a1, a2, a3, wr, wi ) do {tmp5 = a2.real * wr + a2.imag * wi; tmp6 = a2.imag * wr - a2.real * wi; tmp7 = a3.real * wr - a3.imag * wi; tmp8 = a3.imag * wr + a3.real * wi; tmp1 = tmp5 + tmp7; tmp2 = tmp6 + tmp8; tmp3 = tmp6 - tmp8; tmp4 = tmp7 - tmp5; a2.real = a0.real - tmp1; a2.imag = a0.imag - tmp2; a3.real = a1.real - tmp3; a3.imag = a1.imag - tmp4; a0.real += tmp1; a0.imag += tmp2; a1.real += tmp3; a1.imag += tmp4; } while ( 0 )
#define BUTTERFLY_ZERO(          a0, a1, a2, a3 ) do {tmp1 = a2.real + a3.real; tmp2 = a2.imag + a3.imag; tmp3 = a2.imag - a3.imag;	tmp4 = a3.real - a2.real; a2.real = a0.real - tmp1; a2.imag = a0.imag - tmp2; a3.real = a1.real - tmp3; a3.imag = a1.imag - tmp4; a0.real += tmp1; a0.imag += tmp2; a1.real += tmp3; a1.imag += tmp4; } while ( 0 )
#define BUTTERFLY_HALF(          a0, a1, a2, a3, w ) do {tmp5 = ( a2.real + a2.imag ) * w; tmp6 = ( a2.imag - a2.real ) * w; tmp7 = ( a3.real - a3.imag ) * w; tmp8 = ( a3.imag + a3.real ) * w; tmp1 = tmp5 + tmp7; tmp2 = tmp6 + tmp8; tmp3 = tmp6 - tmp8; tmp4 = tmp7 - tmp5; a2.real = a0.real - tmp1; a2.imag = a0.imag - tmp2; a3.real = a1.real - tmp3; a3.imag = a1.imag - tmp4; a0.real += tmp1; a0.imag += tmp2; a1.real += tmp3; a1.imag += tmp4; } while ( 0 )
static inline void ifft8 ( complex_t * buf )
{
   double tmp1, tmp2, tmp3, tmp4, tmp5, tmp6, tmp7, tmp8;
   //
   ifft4 ( buf );
   ifft2 ( buf + 4 );
   ifft2 ( buf + 6 );
   BUTTERFLY_ZERO ( buf[ 0 ], buf[ 2 ], buf[ 4 ], buf[ 6 ] );
   BUTTERFLY_HALF ( buf[ 1 ], buf[ 3 ], buf[ 5 ], buf[ 7 ], roots16[ 1 ] );
}
/***************************************************************************/
static void ifft_pass ( complex_t * buf, sample_t * weight, int n )
{
   complex_t * buf1;
   complex_t * buf2;
   complex_t * buf3;
   double tmp1, tmp2, tmp3, tmp4, tmp5, tmp6, tmp7, tmp8;
   int i;
   //
   buf++;
   buf1 = buf + n;
   buf2 = buf + 2 * n;
   buf3 = buf + 3 * n;
   BUTTERFLY_ZERO ( buf[ -1 ], buf1[ -1 ], buf2[ -1 ], buf3[ -1 ] );
   i = n - 1;
   do 
   {
      BUTTERFLY ( buf[ 0 ], buf1[ 0 ], buf2[ 0 ], buf3[ 0 ], weight[ n ], weight[ 2*i ] );
      buf++;
      buf1++;
      buf2++;
      buf3++;
      weight++;
   } while ( --i );
}
/***************************************************************************/
static void ifft16 ( complex_t * buf )
{
   ifft8 ( buf );
   ifft4 ( buf + 8 );
   ifft4 ( buf + 12 );
   ifft_pass ( buf, roots16 - 4, 4 );
}
/***************************************************************************/
static void ifft32 ( complex_t * buf )
{
   ifft16 ( buf );
   ifft8 ( buf + 16 );
   ifft8 ( buf + 24 );
   ifft_pass ( buf, roots32 - 8, 8 );
}
/***************************************************************************/
static void ifft64_c ( complex_t * buf )
{
   ifft32 ( buf );
   ifft16 ( buf + 32 );
   ifft16 ( buf + 48 );
   ifft_pass ( buf, roots64 - 16, 16 );
}
/***************************************************************************/
static void ifft128_c ( complex_t * buf )
{
   ifft32 ( buf );
   ifft16 ( buf + 32 );
   ifft16 ( buf + 48 );
   ifft_pass ( buf, roots64 - 16, 16 );
   ifft32 ( buf + 64 );
   ifft32 ( buf + 96 );
   ifft_pass ( buf, roots128 - 32, 32 );
}
/***************************************************************************/
static inline void a52_imdct_512 ( sample_t * data, sample_t * delay, sample_t bias )
{
   int i, k;
   sample_t t_r, t_i, a_r, a_i, b_r, b_i, w_1, w_2;
   const sample_t * window = a52_imdct_window;
   //
   for ( i = 0; i < 128; i++ ) 
   {
      k = fftorder[ i ];
      t_r = pre1[ i ].real;
      t_i = pre1[ i ].imag;
      buf[ i ].real = t_i * data[ 255-k ] + t_r * data[ k ];
      buf[ i ].imag = t_r * data[ 255-k ] - t_i * data[ k ];
   }
   ifft128 ( buf );
   for ( i = 0; i < 64; i++ ) 
   {
      t_r = post1[ i ].real;
      t_i = post1[ i ].imag;
      a_r = t_r * buf[ i ].real + t_i * buf[ i ].imag;
      a_i = t_i * buf[ i ].real - t_r * buf[ i ].imag;
      b_r = t_i * buf[ 127-i ].real + t_r * buf[ 127-i ].imag;
      b_i = t_r * buf[ 127-i ].real - t_i * buf[ 127-i ].imag;
      w_1 = window[ 2*i ];
      w_2 = window[ 255-2*i ];
      data[ 2*i ] = delay[ 2*i ] * w_2 - a_r * w_1 + bias;
      data[ 255-2*i ] = delay[ 2*i ] * w_1 + a_r * w_2 + bias;
      delay[ 2*i ] = a_i;
      w_1 = window[ 2*i+1 ];
      w_2 = window[ 254-2*i ];
      data[ 2*i+1 ] = delay[ 2*i+1 ] * w_2 + b_r * w_1 + bias;
      data[ 254-2*i ] = delay[ 2*i+1 ] * w_1 - b_r * w_2 + bias;
      delay[ 2*i+1 ] = b_i;
   }
}
/***************************************************************************/
static inline void a52_imdct_256( sample_t data[], sample_t delay[], sample_t bias )
{
   int i, k;
   sample_t t_r, t_i, a_r, a_i, b_r, b_i, c_r, c_i, d_r, d_i, w_1, w_2;
   complex_t * buf1, * buf2;
   const sample_t * window = a52_imdct_window;
   //
   buf1 = &buf[ 0 ];
   buf2 = &buf[ 64 ];
   for ( i = 0; i < 64; i++ ) 
   {
      k = fftorder[ i ];
      t_r = pre2[ i ].real;
      t_i = pre2[ i ].imag;
      buf1[ i ].real = t_i * data[ 254-k ] + t_r * data[ k ];
      buf1[ i ].imag = t_r * data[ 254-k ] - t_i * data[ k ];
      buf2[ i ].real = t_i * data[ 255-k ] + t_r * data[ k+1 ];
      buf2[ i ].imag = t_r * data[ 255-k ] - t_i * data[ k+1 ];
   }
   ifft64 ( buf1 );
   ifft64 ( buf2 );
   for ( i = 0; i < 32; i++ ) 
   {
      t_r = post2[ i ].real;
      t_i = post2[ i ].imag;
      a_r = t_r * buf1[ i ].real + t_i * buf1[ i ].imag;
      a_i = t_i * buf1[ i ].real - t_r * buf1[ i ].imag;
      b_r = t_i * buf1[ 63-i ].real + t_r * buf1[ 63-i ].imag;
      b_i = t_r * buf1[ 63-i ].real - t_i * buf1[ 63-i ].imag;
      c_r = t_r * buf2[ i ].real + t_i * buf2[ i ].imag;
      c_i = t_i * buf2[ i ].real - t_r * buf2[ i ].imag;
      d_r = t_i * buf2[ 63-i ].real + t_r * buf2[ 63-i ].imag;
      d_i = t_r * buf2[ 63-i ].real - t_i * buf2[ 63-i ].imag;
      w_1 = window[ 2*i ];
      w_2 = window[ 255-2*i ];
      data[ 2*i ] = delay[ 2*i ] * w_2 - a_r * w_1 + bias;
      data[ 255-2*i ] = delay[ 2*i ] * w_1 + a_r * w_2 + bias;
      delay[ 2*i ] = c_i;
      w_1 = window[ 128+2*i ];
      w_2 = window[ 127-2*i ];
      data[ 128+2*i ] = delay[ 127-2*i ] * w_2 + a_i * w_1 + bias;
      data[ 127-2*i ] = delay[ 127-2*i ] * w_1 - a_i * w_2 + bias;
      delay[ 127-2*i ] = c_r;
      w_1 = window[ 2*i+1 ];
      w_2 = window[ 254-2*i ];
      data[ 2*i+1 ] = delay[ 2*i+1 ] * w_2 - b_i * w_1 + bias;
      data[ 254-2*i ] = delay[ 2*i+1 ] * w_1 + b_i * w_2 + bias;
      delay[ 2*i+1 ] = d_r;
      w_1 = window[ 129+2*i ];
      w_2 = window[ 126-2*i ];
      data[ 129+2*i ] = delay[ 126-2*i ] * w_2 + b_r * w_1 + bias;
      data[ 126-2*i ] = delay[ 126-2*i ] * w_1 - b_r * w_2 + bias;
      delay[ 126-2*i ] = d_i;
   }
}
/***************************************************************************/
static inline void a52_imdct_init ( uint32_t mm_accel )
{
   int i, k;
   //
   for ( i = 0; i < 3; i++ )
      roots16[ i ] = cos ( ( M_PI / 8 ) * ( i + 1 ) );
   for ( i = 0; i < 7; i++ )
      roots32[ i ] = cos ( ( M_PI / 16 ) * ( i + 1 ) );
   for ( i = 0; i < 15; i++ )
      roots64[ i ] = cos ( ( M_PI / 32 ) * ( i + 1 ) );
   for ( i = 0; i < 31; i++ )
      roots128[ i ] = cos ( ( M_PI / 64 ) * ( i + 1 ) );
   for ( i = 0; i < 64; i++ ) 
   {
      k = fftorder[ i ] / 2 + 64;
      pre1[ i ].real = cos ( ( M_PI / 256 ) * ( k - 0.25 ) );
      pre1[ i ].imag = sin ( ( M_PI / 256 ) * ( k - 0.25 ) );
   }
   for ( i = 64; i < 128; i++ ) 
   {
      k = fftorder[ i ] / 2 + 64;
      pre1[ i ].real = -cos ( ( M_PI / 256 ) * ( k - 0.25 ) );
      pre1[ i ].imag = -sin ( ( M_PI / 256 ) * ( k - 0.25 ) );
   }
   for ( i = 0; i < 64; i++ ) 
   {
      post1[ i ].real = cos ( ( M_PI / 256 ) * ( i + 0.5 ) );
      post1[ i ].imag = sin ( ( M_PI / 256 ) * ( i + 0.5 ) );
   }
   for ( i = 0; i < 64; i++ ) 
   {
      k = fftorder[ i ] / 4;
      pre2[ i ].real = cos ( ( M_PI / 128 ) * ( k - 0.25 ) );
      pre2[ i ].imag = sin ( ( M_PI / 128 ) * ( k - 0.25 ) );
   }
   for ( i = 0; i < 32; i++ ) 
   {
      post2[ i ].real = cos ( ( M_PI / 128 ) * ( i + 0.5 ) );
      post2[ i ].imag = sin ( ( M_PI / 128 ) * ( i + 0.5 ) );
   }
   ifft128 = ifft128_c;
   ifft64 = ifft64_c;
}
/***************************************************************************/
#define CONVERT( acmod, output ) ( ( ( output ) << 3 ) + ( acmod ) )
static inline int a52_downmix_init ( int input, int flags, sample_t * level, sample_t clev, sample_t slev )
{
   static uint8_t table[ 11 ][ 8 ] = 
   {
      {A52_CHANNEL, 	A52_DOLBY, 	A52_STEREO, 	A52_STEREO, 
      A52_STEREO, 	A52_STEREO, 	A52_STEREO, 	A52_STEREO}, 
      {A52_MONO, 	A52_MONO, 	A52_MONO, 	A52_MONO, 
      A52_MONO, 	A52_MONO, 	A52_MONO, 	A52_MONO}, 
      {A52_CHANNEL, 	A52_DOLBY, 	A52_STEREO, 	A52_STEREO, 
      A52_STEREO, 	A52_STEREO, 	A52_STEREO, 	A52_STEREO}, 
      {A52_CHANNEL, 	A52_DOLBY, 	A52_STEREO, 	A52_3F, 
      A52_STEREO, 	A52_3F, 	A52_STEREO, 	A52_3F}, 
      {A52_CHANNEL, 	A52_DOLBY, 	A52_STEREO, 	A52_STEREO, 
      A52_2F1R, 	A52_2F1R, 	A52_2F1R, 	A52_2F1R}, 
      {A52_CHANNEL, 	A52_DOLBY, 	A52_STEREO, 	A52_STEREO, 
      A52_2F1R, 	A52_3F1R, 	A52_2F1R, 	A52_3F1R}, 
      {A52_CHANNEL, 	A52_DOLBY, 	A52_STEREO, 	A52_3F, 
      A52_2F2R, 	A52_2F2R, 	A52_2F2R, 	A52_2F2R}, 
      {A52_CHANNEL, 	A52_DOLBY, 	A52_STEREO, 	A52_3F, 
      A52_2F2R, 	A52_3F2R, 	A52_2F2R, 	A52_3F2R}, 
      {A52_CHANNEL1, 	A52_MONO, 	A52_MONO, 	A52_MONO, 
      A52_MONO, 	A52_MONO, 	A52_MONO, 	A52_MONO}, 
      {A52_CHANNEL2, 	A52_MONO, 	A52_MONO, 	A52_MONO, 
      A52_MONO, 	A52_MONO, 	A52_MONO, 	A52_MONO}, 
      {A52_CHANNEL, 	A52_DOLBY, 	A52_STEREO, 	A52_DOLBY, 
      A52_DOLBY, 	A52_DOLBY, 	A52_DOLBY, 	A52_DOLBY}
   };
   int output;
   //
   output = flags & A52_CHANNEL_MASK;
   if ( output > A52_DOLBY )
      return -1;
   output = table[ output ][ input & 7 ];
   if ( ( output == A52_STEREO ) && ( ( input == A52_DOLBY ) || ( ( input == A52_3F ) && ( clev == LEVEL_3DB ) ) ) )
      output = A52_DOLBY;
   if ( flags & A52_ADJUST_LEVEL )
      switch ( CONVERT ( input & 7, output ) )
   {
      case CONVERT ( A52_3F, A52_MONO ):
         *level *= LEVEL_3DB / ( 1 + clev );
         break;
      case CONVERT ( A52_STEREO, A52_MONO ):
      case CONVERT ( A52_2F2R, A52_2F1R ):
      case CONVERT ( A52_3F2R, A52_3F1R ):
         level_3db:
         *level *= LEVEL_3DB;
         break;
      case CONVERT ( A52_3F2R, A52_2F1R ):
         if ( clev < LEVEL_PLUS3DB - 1 )
            goto level_3db;
      case CONVERT ( A52_3F, A52_STEREO ):
      case CONVERT ( A52_3F1R, A52_2F1R ):
      case CONVERT ( A52_3F1R, A52_2F2R ):
      case CONVERT ( A52_3F2R, A52_2F2R ):
         *level /= 1 + clev;
         break;
      case CONVERT ( A52_2F1R, A52_MONO ):
         *level *= LEVEL_PLUS3DB / ( 2 + slev );
         break;
      case CONVERT ( A52_2F1R, A52_STEREO ):
      case CONVERT ( A52_3F1R, A52_3F ):
         *level /= 1 + slev * LEVEL_3DB;
         break;
      case CONVERT ( A52_3F1R, A52_MONO ):
         *level *= LEVEL_3DB / ( 1 + clev + 0.5 * slev );
         break;
      case CONVERT ( A52_3F1R, A52_STEREO ):
         *level /= 1 + clev + slev * LEVEL_3DB;
         break;
      case CONVERT ( A52_2F2R, A52_MONO ):
         *level *= LEVEL_3DB / ( 1 + slev );
         break;
      case CONVERT ( A52_2F2R, A52_STEREO ):
      case CONVERT ( A52_3F2R, A52_3F ):
         *level /= 1 + slev;
         break;
      case CONVERT ( A52_3F2R, A52_MONO ):
         *level *= LEVEL_3DB / ( 1 + clev + slev );
         break;
      case CONVERT ( A52_3F2R, A52_STEREO ):
         *level /= 1 + clev + slev;
         break;
      case CONVERT ( A52_MONO, A52_DOLBY ):
         *level *= LEVEL_PLUS3DB;
         break;
      case CONVERT ( A52_3F, A52_DOLBY ):
      case CONVERT ( A52_2F1R, A52_DOLBY ):
         *level *= 1 / ( 1 + LEVEL_3DB );
         break;
      case CONVERT ( A52_3F1R, A52_DOLBY ):
      case CONVERT ( A52_2F2R, A52_DOLBY ):
         *level *= 1 / ( 1 + 2 * LEVEL_3DB );
         break;
      case CONVERT ( A52_3F2R, A52_DOLBY ):
         *level *= 1 / ( 1 + 3 * LEVEL_3DB );
         break;
   }
   //
   return output;
}
/***************************************************************************/
static inline int a52_downmix_coeff ( sample_t * coeff, int acmod, int output, sample_t level, sample_t clev, sample_t slev )
{
   switch ( CONVERT ( acmod, output & A52_CHANNEL_MASK ) )
   {
      case CONVERT ( A52_CHANNEL, A52_CHANNEL ):
      case CONVERT ( A52_MONO, A52_MONO ):
      case CONVERT ( A52_STEREO, A52_STEREO ):
      case CONVERT ( A52_3F, A52_3F ):
      case CONVERT ( A52_2F1R, A52_2F1R ):
      case CONVERT ( A52_3F1R, A52_3F1R ):
      case CONVERT ( A52_2F2R, A52_2F2R ):
      case CONVERT ( A52_3F2R, A52_3F2R ):
      case CONVERT ( A52_STEREO, A52_DOLBY ):
         coeff[ 0 ] = coeff[ 1 ] = coeff[ 2 ] = coeff[ 3 ] = coeff[ 4 ] = level;
         return 0;
      case CONVERT ( A52_CHANNEL, A52_MONO ):
         coeff[ 0 ] = coeff[ 1 ] = level * LEVEL_6DB;
         return 3;
      case CONVERT ( A52_STEREO, A52_MONO ):
         coeff[ 0 ] = coeff[ 1 ] = level * LEVEL_3DB;
         return 3;
      case CONVERT ( A52_3F, A52_MONO ):
         coeff[ 0 ] = coeff[ 2 ] = level * LEVEL_3DB;
         coeff[ 1 ] = level * clev * LEVEL_PLUS3DB;
         return 7;
      case CONVERT ( A52_2F1R, A52_MONO ):
         coeff[ 0 ] = coeff[ 1 ] = level * LEVEL_3DB;
         coeff[ 2 ] = level * slev * LEVEL_3DB;
         return 7;
      case CONVERT ( A52_2F2R, A52_MONO ):
         coeff[ 0 ] = coeff[ 1 ] = level * LEVEL_3DB;
         coeff[ 2 ] = coeff[ 3 ] = level * slev * LEVEL_3DB;
         return 15;
      case CONVERT ( A52_3F1R, A52_MONO ):
         coeff[ 0 ] = coeff[ 2 ] = level * LEVEL_3DB;
         coeff[ 1 ] = level * clev * LEVEL_PLUS3DB;
         coeff[ 3 ] = level * slev * LEVEL_3DB;
         return 15;
      case CONVERT ( A52_3F2R, A52_MONO ):
         coeff[ 0 ] = coeff[ 2 ] = level * LEVEL_3DB;
         coeff[ 1 ] = level * clev * LEVEL_PLUS3DB;
         coeff[ 3 ] = coeff[ 4 ] = level * slev * LEVEL_3DB;
         return 31;
      case CONVERT ( A52_MONO, A52_DOLBY ):
         coeff[ 0 ] = level * LEVEL_3DB;
         return 0;
      case CONVERT ( A52_3F, A52_DOLBY ):
         clev = LEVEL_3DB;
      case CONVERT ( A52_3F, A52_STEREO ):
      case CONVERT ( A52_3F1R, A52_2F1R ):
      case CONVERT ( A52_3F2R, A52_2F2R ):
         coeff[ 0 ] = coeff[ 2 ] = coeff[ 3 ] = coeff[ 4 ] = level;
         coeff[ 1 ] = level * clev;
         return 7;
      case CONVERT ( A52_2F1R, A52_DOLBY ):
         slev = 1;
      case CONVERT ( A52_2F1R, A52_STEREO ):
         coeff[ 0 ] = coeff[ 1 ] = level;
         coeff[ 2 ] = level * slev * LEVEL_3DB;
         return 7;
      case CONVERT ( A52_3F1R, A52_DOLBY ):
         clev = LEVEL_3DB;
         slev = 1;
      case CONVERT ( A52_3F1R, A52_STEREO ):
         coeff[ 0 ] = coeff[ 2 ] = level;
         coeff[ 1 ] = level * clev;
         coeff[ 3 ] = level * slev * LEVEL_3DB;
         return 15;
      case CONVERT ( A52_2F2R, A52_DOLBY ):
         slev = LEVEL_3DB;
      case CONVERT ( A52_2F2R, A52_STEREO ):
         coeff[ 0 ] = coeff[ 1 ] = level;
         coeff[ 2 ] = coeff[ 3 ] = level * slev;
         return 15;
      case CONVERT ( A52_3F2R, A52_DOLBY ):
         clev = LEVEL_3DB;
      case CONVERT ( A52_3F2R, A52_2F1R ):
         slev = LEVEL_3DB;
      case CONVERT ( A52_3F2R, A52_STEREO ):
         coeff[ 0 ] = coeff[ 2 ] = level;
         coeff[ 1 ] = level * clev;
         coeff[ 3 ] = coeff[ 4 ] = level * slev;
         return 31;
      case CONVERT ( A52_3F1R, A52_3F ):
         coeff[ 0 ] = coeff[ 1 ] = coeff[ 2 ] = level;
         coeff[ 3 ] = level * slev * LEVEL_3DB;
         return 13;
      case CONVERT ( A52_3F2R, A52_3F ):
         coeff[ 0 ] = coeff[ 1 ] = coeff[ 2 ] = level;
         coeff[ 3 ] = coeff[ 4 ] = level * slev;
         return 29;
      case CONVERT ( A52_2F2R, A52_2F1R ):
         coeff[ 0 ] = coeff[ 1 ] = level;
         coeff[ 2 ] = coeff[ 3 ] = level * LEVEL_3DB;
         return 12;
      case CONVERT ( A52_3F2R, A52_3F1R ):
         coeff[ 0 ] = coeff[ 1 ] = coeff[ 2 ] = level;
         coeff[ 3 ] = coeff[ 4 ] = level * LEVEL_3DB;
         return 24;
      case CONVERT ( A52_2F1R, A52_2F2R ):
         coeff[ 0 ] = coeff[ 1 ] = level;
         coeff[ 2 ] = level * LEVEL_3DB;
         return 0;
      case CONVERT ( A52_3F1R, A52_2F2R ):
         coeff[ 0 ] = coeff[ 2 ] = level;
         coeff[ 1 ] = level * clev;
         coeff[ 3 ] = level * LEVEL_3DB;
         return 7;
      case CONVERT ( A52_3F1R, A52_3F2R ):
         coeff[ 0 ] = coeff[ 1 ] = coeff[ 2 ] = level;
         coeff[ 3 ] = level * LEVEL_3DB;
         return 0;
      case CONVERT ( A52_CHANNEL, A52_CHANNEL1 ):
         coeff[ 0 ] = level;
         coeff[ 1 ] = 0;
         return 0;
      case CONVERT ( A52_CHANNEL, A52_CHANNEL2 ):
         coeff[ 0 ] = 0;
         coeff[ 1 ] = level;
         return 0;
   }
   return -1;	
}
/***************************************************************************/
static void mix2to1 ( sample_t * dest, sample_t * src, sample_t bias )
{
   int i;
   //
   for ( i = 0; i < 256; i++ )
      dest[ i ] += src[ i ] + bias;
}
/***************************************************************************/
static void mix3to1 ( sample_t * samples, sample_t bias )
{
   int i;
   //
   for ( i = 0; i < 256; i++ )
      samples[ i ] += samples[ i + 256 ] + samples[ i + 512 ] + bias;
}
/***************************************************************************/
static void mix4to1 ( sample_t * samples, sample_t bias )
{
   int i;
   //
   for ( i = 0; i < 256; i++ )
      samples[ i ] += ( samples[ i + 256 ] + samples[ i + 512 ] + samples[ i + 768 ] + bias );
}
/***************************************************************************/
static void mix5to1 ( sample_t * samples, sample_t bias )
{
   int i;
   //
   for ( i = 0; i < 256; i++ )
      samples[ i ] += ( samples[ i + 256 ] + samples[ i + 512 ] + samples[ i + 768 ] + samples[ i + 1024 ] + bias );
}
/***************************************************************************/
static void mix3to2 ( sample_t * samples, sample_t bias )
{
   int i;
   sample_t common;
   //
   for ( i = 0; i < 256; i++ ) 
   {
      common = samples[ i + 256 ] + bias;
      samples[ i ] += common;
      samples[ i + 256 ] = samples[ i + 512 ] + common;
   }
}
/***************************************************************************/
static void mix21to2 ( sample_t * left, sample_t * right, sample_t bias )
{
   int i;
   sample_t common;
   //
   for ( i = 0; i < 256; i++ ) 
   {
      common = right[ i + 256 ] + bias;
      left[ i ] += common;
      right[ i ] += common;
   }
}
/***************************************************************************/
static void mix21toS ( sample_t * samples, sample_t bias )
{
   int i;
   sample_t surround;
   //
   for ( i = 0; i < 256; i++ ) 
   {
      surround = samples[ i + 512 ];
      samples[ i ] += bias - surround;
      samples[ i + 256 ] += bias + surround;
   }
}
/***************************************************************************/
static void mix31to2 ( sample_t * samples, sample_t bias )
{
   int i;
   sample_t common;
   //
   for ( i = 0; i < 256; i++ ) 
   {
      common = samples[ i + 256 ] + samples[ i + 768 ] + bias;
      samples[ i ] += common;
      samples[ i + 256 ] = samples[ i + 512 ] + common;
   }
}
/***************************************************************************/
static void mix31toS ( sample_t * samples, sample_t bias )
{
   int i;
   sample_t common, surround;
   //
   for ( i = 0; i < 256; i++ ) 
   {
      common = samples[ i + 256 ] + bias;
      surround = samples[ i + 768 ];
      samples[ i ] += common - surround;
      samples[ i + 256 ] = samples[ i + 512 ] + common + surround;
   }
}
/***************************************************************************/
static void mix22toS ( sample_t * samples, sample_t bias )
{
   int i;
   sample_t surround;
   //
   for ( i = 0; i < 256; i++ ) 
   {
      surround = samples[ i + 512 ] + samples[ i + 768 ];
      samples[ i ] += bias - surround;
      samples[ i + 256 ] += bias + surround;
   }
}
/***************************************************************************/
static void mix32to2 ( sample_t * samples, sample_t bias )
{
   int i;
   sample_t common;
   //
   for ( i = 0; i < 256; i++ ) 
   {
      common = samples[ i + 256 ] + bias;
      samples[ i ] += common + samples[ i + 768 ];
      samples[ i + 256 ] = common + samples[ i + 512 ] + samples[ i + 1024 ];
   }
}
/***************************************************************************/
static void mix32toS ( sample_t * samples, sample_t bias )
{
   int i;
   sample_t common, surround;
   //
   for ( i = 0; i < 256; i++ ) 
   {
      common = samples[ i + 256 ] + bias;
      surround = samples[ i + 768 ] + samples[ i + 1024 ];
      samples[ i ] += common - surround;
      samples[ i + 256 ] = samples[ i + 512 ] + common + surround;
   }
}
/***************************************************************************/
static void move2to1 ( sample_t * src, sample_t * dest, sample_t bias )
{
   int i;
   //
   for ( i = 0; i < 256; i++ )
      dest[ i ] = src[ i ] + src[ i + 256 ] + bias;
}
/***************************************************************************/
static void zero ( sample_t * samples )
{
   int i;
   //
   for ( i = 0; i < 256; i++ )
      samples[ i ] = 0;
}
/***************************************************************************/
static inline void a52_downmix ( sample_t * samples, int acmod, int output, sample_t bias, sample_t clev, sample_t slev )
{
   switch ( CONVERT ( acmod, output & A52_CHANNEL_MASK ) )
   {
      case CONVERT ( A52_CHANNEL, A52_CHANNEL2 ):
         memcpy ( samples, samples + 256, 256 * sizeof ( sample_t ) );
         break;
      case CONVERT ( A52_CHANNEL, A52_MONO ):
      case CONVERT ( A52_STEREO, A52_MONO ):
         mix_2to1:
         mix2to1 ( samples, samples + 256, bias );
         break;
      case CONVERT ( A52_2F1R, A52_MONO ):
         if ( slev == 0 )
            goto mix_2to1;
      case CONVERT ( A52_3F, A52_MONO ):
         mix_3to1:
         mix3to1 ( samples, bias );
         break;
      case CONVERT ( A52_3F1R, A52_MONO ):
	 if ( slev == 0 )
            goto mix_3to1;
      case CONVERT ( A52_2F2R, A52_MONO ):
         if ( slev == 0 )
            goto mix_2to1;
         mix4to1 ( samples, bias );
         break;
      case CONVERT ( A52_3F2R, A52_MONO ):
         if ( slev == 0 )
            goto mix_3to1;
         mix5to1 ( samples, bias );
         break;
      case CONVERT ( A52_MONO, A52_DOLBY ):
         memcpy ( samples + 256, samples, 256 * sizeof ( sample_t ) );
         break;
      case CONVERT ( A52_3F, A52_STEREO ):
      case CONVERT ( A52_3F, A52_DOLBY ):
         mix_3to2:
         mix3to2 ( samples, bias );
         break;
      case CONVERT ( A52_2F1R, A52_STEREO ):
         if ( slev == 0 )
            break;
      mix21to2 ( samples, samples + 256, bias );
      break;
      case CONVERT ( A52_2F1R, A52_DOLBY ):
         mix21toS ( samples, bias );
         break;
      case CONVERT ( A52_3F1R, A52_STEREO ):
         if ( slev == 0 )
            goto mix_3to2;
         mix31to2 ( samples, bias );
         break;
      case CONVERT ( A52_3F1R, A52_DOLBY ):
         mix31toS ( samples, bias );
         break;
      case CONVERT ( A52_2F2R, A52_STEREO ):
         if ( slev == 0 )
            break;
      mix2to1 ( samples, samples + 512, bias );
      mix2to1 ( samples + 256, samples + 768, bias );
      break;
      case CONVERT ( A52_2F2R, A52_DOLBY ):
         mix22toS ( samples, bias );
         break;
      case CONVERT ( A52_3F2R, A52_STEREO ):
         if ( slev == 0 )
            goto mix_3to2;
         mix32to2 ( samples, bias );
         break;
      case CONVERT ( A52_3F2R, A52_DOLBY ):
         mix32toS ( samples, bias );
         break;
      case CONVERT ( A52_3F1R, A52_3F ):
         if ( slev == 0 )
            break;
      mix21to2 ( samples, samples + 512, bias );
      break;
      case CONVERT ( A52_3F2R, A52_3F ):
         if ( slev == 0 )
	    break;
      mix2to1 ( samples, samples + 768, bias );
      mix2to1 ( samples + 512, samples + 1024, bias );
      break;
      case CONVERT ( A52_3F1R, A52_2F1R ):
         mix3to2 ( samples, bias );
         memcpy ( samples + 512, samples + 768, 256 * sizeof ( sample_t ) );
         break;
      case CONVERT ( A52_2F2R, A52_2F1R ):
         mix2to1 ( samples + 512, samples + 768, bias );
         break;
      case CONVERT ( A52_3F2R, A52_2F1R ):
         mix3to2 ( samples, bias );
         move2to1 ( samples + 768, samples + 512, bias );
         break;
      case CONVERT ( A52_3F2R, A52_3F1R ):
         mix2to1 ( samples + 768, samples + 1024, bias );
         break;
      case CONVERT ( A52_2F1R, A52_2F2R ):
         memcpy ( samples + 768, samples + 512, 256 * sizeof ( sample_t ) );
	 break;
      case CONVERT ( A52_3F1R, A52_2F2R ):
         mix3to2 ( samples, bias );
         memcpy ( samples + 512, samples + 768, 256 * sizeof ( sample_t ) );
         break;
      case CONVERT ( A52_3F2R, A52_2F2R ):
         mix3to2 ( samples, bias );
         memcpy ( samples + 512, samples + 768, 256 * sizeof ( sample_t ) );
         memcpy ( samples + 768, samples + 1024, 256 * sizeof ( sample_t ) );
         break;
      case CONVERT ( A52_3F1R, A52_3F2R ):
         memcpy ( samples + 1027, samples + 768, 256 * sizeof ( sample_t ) );
         break;
   }
}
/***************************************************************************/
static inline void a52_upmix ( sample_t * samples, int acmod, int output )
{
   switch ( CONVERT ( acmod, output & A52_CHANNEL_MASK ) )
   {
      case CONVERT ( A52_CHANNEL, A52_CHANNEL2 ):
         memcpy ( samples + 256, samples, 256 * sizeof ( sample_t ) );
         break;
      case CONVERT ( A52_3F2R, A52_MONO ):
         zero ( samples + 1024 );
      case CONVERT ( A52_3F1R, A52_MONO ):
      case CONVERT ( A52_2F2R, A52_MONO ):
         zero ( samples + 768 );
      case CONVERT ( A52_3F, A52_MONO ):
      case CONVERT ( A52_2F1R, A52_MONO ):
         zero ( samples + 512 );
      case CONVERT ( A52_CHANNEL, A52_MONO ):
      case CONVERT ( A52_STEREO, A52_MONO ):
         zero ( samples + 256 );
         break;
      case CONVERT ( A52_3F2R, A52_STEREO ):
      case CONVERT ( A52_3F2R, A52_DOLBY ):
         zero ( samples + 1024 );
      case CONVERT ( A52_3F1R, A52_STEREO ):
      case CONVERT ( A52_3F1R, A52_DOLBY ):
	 zero ( samples + 768 );
      case CONVERT ( A52_3F, A52_STEREO ):
      case CONVERT ( A52_3F, A52_DOLBY ):
         mix_3to2:
         memcpy ( samples + 512, samples + 256, 256 * sizeof ( sample_t ) );
         zero ( samples + 256 );
         break;
      case CONVERT ( A52_2F2R, A52_STEREO ):
      case CONVERT ( A52_2F2R, A52_DOLBY ):
         zero ( samples + 768 );
      case CONVERT ( A52_2F1R, A52_STEREO ):
      case CONVERT ( A52_2F1R, A52_DOLBY ):
         zero ( samples + 512 );
         break;
      case CONVERT ( A52_3F2R, A52_3F ):
         zero ( samples + 1024 );
      case CONVERT ( A52_3F1R, A52_3F ):
      case CONVERT ( A52_2F2R, A52_2F1R ):
         zero ( samples + 768 );
         break;
      case CONVERT ( A52_3F2R, A52_3F1R ):
         zero ( samples + 1024 );
         break;
      case CONVERT ( A52_3F2R, A52_2F1R ):
         zero ( samples + 1024 );
      case CONVERT ( A52_3F1R, A52_2F1R ):
         mix_31to21:
         memcpy ( samples + 768, samples + 512, 256 * sizeof ( sample_t ) );
         goto mix_3to2;
      case CONVERT ( A52_3F2R, A52_2F2R ):
         memcpy ( samples + 1024, samples + 768, 256 * sizeof ( sample_t ) );
         goto mix_31to21;
   }
}
/***************************************************************************/
#define UPDATE_LEAK()            do {fastleak += fdecay; if ( fastleak > psd + fgain ) fastleak = psd + fgain; slowleak += sdecay; if ( slowleak > psd + sgain ) slowleak = psd + sgain; } while ( 0 )
#define COMPUTE_MASK()           do {if ( psd > dbknee ) mask -= ( psd - dbknee ) >> 2; if ( mask > hth [ i >> halfrate ] ) mask = hth [ i >> halfrate ]; mask -= snroffset + 128 * deltba[ i ]; mask = ( mask > 0 ) ? 0 : ( ( -mask ) >> 5 ); mask -= floor; } while ( 0 )
static inline void a52_bit_allocate ( a52_state_t * state, ba_t * ba, int bndstart, 	 int start, int end, int fastleak, int slowleak, 	 expbap_t * expbap )
{
   static int slowgain[ 4 ] = {0x540, 0x4d8, 0x478, 0x410};
   static int dbpbtab[ 4 ] = {0xc00, 0x500, 0x300, 0x100};
   static int floortab[ 8 ] = {0x910, 0x950, 0x990, 0x9d0,0xa10, 0xa90, 0xb10, 0x1400};
   int i, j;
   uint8_t * exp;
   int8_t * bap;
   int fdecay, fgain, sdecay, sgain, dbknee, floor, snroffset;
   int psd, mask;
   int8_t * deltba;
   int * hth;
   int halfrate;
   //
   halfrate = state->halfrate;
   fdecay = ( 63 + 20 * ( ( state->bai >> 7 ) & 3 ) ) >> halfrate;	
   fgain = 128 + 128 * ( ba->bai & 7 );	
   sdecay = ( 15 + 2 * ( state->bai >> 9 ) ) >> halfrate;	
   sgain = slowgain[ ( state->bai >> 5 ) & 3 ];	
   dbknee = dbpbtab[ ( state->bai >> 3 ) & 3 ];	
   hth = hthtab[ state->fscod ];
   deltba = ( ba->deltbae == DELTA_BIT_NONE ) ? baptab + 156 : ba->deltba;
   floor = floortab[ state->bai & 7 ];	
   snroffset = 960 - 64 * state->csnroffst - 4 * ( ba->bai >> 3 ) + floor;
   floor >>= 5;
   exp = expbap->exp;
   bap = expbap->bap;
   i = bndstart;
   j = start;
   if ( start == 0 )
   {
      int lowcomp;
      lowcomp = 0;
      j = end - 1;
      do 
      {
         if ( i < j ) 
         {
            if ( exp[ i+1 ] == exp[ i ] - 2 )
               lowcomp = 384;
            else if ( lowcomp && ( exp[ i+1 ] > exp[ i ] ) )
               lowcomp -= 64;
         }
	 psd = 128 * exp[ i ];
         mask = psd + fgain + lowcomp;
	 COMPUTE_MASK();
         bap[ i ] = ( baptab+156 )[ mask + 4 * exp[ i ] ];
         i++;
	  } while ( ( i < 3 ) || ( ( i < 7 ) && ( exp[ i ] > exp[ i-1 ] ) ) );
      fastleak = psd + fgain;
      slowleak = psd + sgain;
	  while ( i < 7 )
      {
         if ( i < j ) 
         {
            if ( exp[ i+1 ] == exp[ i ] - 2 )
               lowcomp = 384;
            else if ( lowcomp && ( exp[ i+1 ] > exp[ i ] ) )
               lowcomp -= 64;
         }
         psd = 128 * exp[ i ];
	 UPDATE_LEAK();
         mask = ( ( fastleak + lowcomp < slowleak ) ?
         fastleak + lowcomp : slowleak );
	 COMPUTE_MASK();
         bap[ i ] = ( baptab+156 )[ mask + 4 * exp[ i ] ];
         i++;
      }
      if ( end == 7 )	
         return;
      do 
      {
         if ( exp[ i+1 ] == exp[ i ] - 2 )
            lowcomp = 320;
         else if ( lowcomp && ( exp[ i+1 ] > exp[ i ] ) )
            lowcomp -= 64;
         psd = 128 * exp[ i ];
	 UPDATE_LEAK();
         mask = ( ( fastleak + lowcomp < slowleak ) ?
         fastleak + lowcomp : slowleak );
	 COMPUTE_MASK();
         bap[ i ] = ( baptab+156 )[ mask + 4 * exp[ i ] ];
         i++;
	  } while ( i < 20 );
	  while ( lowcomp > 128 )
      {
         lowcomp -= 128;
         psd = 128 * exp[ i ];
	 UPDATE_LEAK();
         mask = ( ( fastleak + lowcomp < slowleak ) ?
         fastleak + lowcomp : slowleak );
	 COMPUTE_MASK();
         bap[ i ] = ( baptab+156 )[ mask + 4 * exp[ i ] ];
         i++;
      }
      j = i;
   }
   do 
   {
      int startband, endband;
      //
      startband = j;
      endband = ( ( bndtab-20 )[ i ] < end ) ? ( bndtab-20 )[ i ] : end;
      psd = 128 * exp[ j++ ];
	  while ( j < endband )
      {
         int next, delta;
         next = 128 * exp[ j++ ];
         delta = next - psd;
         switch ( delta >> 9 ) 
         {
            case -6: case -5: case -4: case -3: case -2:
               psd = next;
               break;
            case -1:
               psd = next + latab[ ( -delta ) >> 1 ];
               break;
            case 0:
               psd += latab[ delta >> 1 ];
               break;
            }
         }
	 UPDATE_LEAK();
         mask = ( fastleak < slowleak ) ? fastleak : slowleak;
	 COMPUTE_MASK();
         i++;
         j = startband;
         do 
         {
            bap[ j ] = ( baptab+156 )[ mask + 4 * exp[ j ] ];
	 } while ( ++j < endband );
   } while ( j < end );
}
/***************************************************************************/
static inline void a52_bitstream_set_ptr ( uint8_t * buf )
{
   int align;
   //
   align = ( long )buf & 3;
   buffer_start = ( uint32_t * ) ( buf - align );
   a52_bits_left = 0;
   bitstream_get ( align * 8 );
}
/***************************************************************************/
static inline void bitstream_fill_current( void )
{
   uint32_t tmp;
   //
   tmp = *( buffer_start++ );
   a52_current_word = swab32( tmp );
}
/***************************************************************************/
static inline uint32_t a52_bitstream_get_bh( uint32_t num_bits )
{
   uint32_t result;
   //
   num_bits -= a52_bits_left;
   result = ( ( a52_current_word << ( 32 - a52_bits_left ) ) >> ( 32 - a52_bits_left ) );
   bitstream_fill_current();
   if ( num_bits != 0 )
      result = ( result << num_bits ) | ( a52_current_word >> ( 32 - num_bits ) );
   a52_bits_left = 32 - num_bits;
   //
   return result;
}
/***************************************************************************/
static inline int32_t a52_bitstream_get_bh_2( uint32_t num_bits )
{
   int32_t result;
   //
   num_bits -= a52_bits_left;
   result = ( ( ( ( int32_t )a52_current_word ) << ( 32 - a52_bits_left ) ) >> ( 32 - a52_bits_left ) );
   bitstream_fill_current();
   if ( num_bits != 0 )
      result = ( result << num_bits ) | ( a52_current_word >> ( 32 - num_bits ) );
   a52_bits_left = 32 - num_bits;
   //
   return result;
}
/***************************************************************************/
// Programa Principal.
/***************************************************************************/
//static HWND *mHWND;
static MPEG *file = NULL;
static short *audio_output;
static long total_samples, total_frames;
static float actual_sample, actual_frame, porcentaje_sample, porcentaje_frame;
static int video_width, video_height;
static int audio_canales, audio_rate;
static BITMAP *video = NULL;
static AUDIOSTREAM *audio = NULL;
//
static int quit( void );
static int load( void );
static void hchangelbl( int i );
static void maximize_video( void );
//
#define AUDIO_BUFFER                  ( 1024 * 3 )
/***************************************************************************/
GUIMENU my_menu[] =
{
   { "&Cargar Fichero\t(ctrl+L)",   load,          NULL },
   { "",                            NULL,          NULL },
   { "&Salir\t(ctrl+Q)",            quit,          NULL },
   { NULL,                          NULL,          NULL }
};
/***************************************************************************/
GUIMENU menu[] =
{
   { "&File",                       NULL,             my_menu },
   { NULL,                          NULL,             NULL }
};
/***************************************************************************/
#define V_W		450
#define V_H		340
//
DIALOG the_dialog[] =
{
   /* (dialog proc)             (x)     (y)       (w)      (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { WindowProcedure,             0,      0,      V_W,     V_H,    0,     0,    0,    0,       0,    0,    "Mpeg Player" },
   { dMenuProcedure,              5,      5,        0,       0,    0,     0,    0,    0,       0,    0,    menu },
   { BitmapProcedure,             5,     25,   V_W-10,  V_H-50,    0,     0,    0,    0,       0,    0,    NULL },
   { ProgressProcedure,           5, V_H-20,   V_W-10,      16,    0,     0,    0,    0,     100,    0,    NULL},
   { NULL,                        0,      0,        0,       0,    0,     0,    0,    0,       0,    0,    NULL }
};
/***************************************************************************/
static l_int maximized = FALSE;
//
static void maximize_video( void )
{
   destroy_bitmap( ( BITMAP *)the_dialog [ 2 ].dp );
   the_dialog[ 2 ].dp = create_bitmap( the_dialog[ 2 ].w, the_dialog[ 2 ].h );
}
/***************************************************************************/
static int quit( void )
{
   BreakApp( &Me );
   //
   return D_CLOSE;
}
/***************************************************************************/
static void close_mpeg( void )
{
   if ( audio )
   {
	  while( get_audio_stream_buffer( audio ) == NULL )
		 ;
   }
   //
   free_audio_stream_buffer( audio );
   if ( audio )
	  stop_audio_stream( audio );
   mpeg_close( file );
   file = NULL;
}
/***************************************************************************/
static void show_my_img( void )
{
   BITMAP *my_bmp = NULL;
   BITMAP *my_img = NULL;
   //
   maximized = TRUE;
   maximize_video();
   //
   my_bmp = LoadImage( "./bmp/bitmaps/mmedia.bmp" );
   my_img = the_dialog[ 2 ].dp;
   stretch_blit( my_bmp, my_img, 0, 0, my_bmp->w, my_bmp->h, 0, 0, my_img->w, my_img->h );
}
/***************************************************************************/
static int load( void )
{
   l_text buf = IOBox("Select File", IOBOX_OPEN, NULL, Filter, true);
   //
   if ( buf )
   {
	  buf += 2;
	  fix_filename_case( buf );
	  //
	  DebugMessage( "MpegPlayer : Open '%s'", buf );
	  //
	  if ( file )
	 close_mpeg();
	  //
	  maximized = TRUE;
	  maximize_video();
	  //
	  file = mpeg_open( buf );
	  if ( file )
	  {
	 video_width   = mpeg_video_width( file, 0 );
	 video_height  = mpeg_video_height( file, 0 );
	 audio_canales = mpeg_audio_channels( file, 0 );
	 audio_rate    = mpeg_sample_rate( file, 0 );
	 total_frames  = mpeg_video_frames( file, 0 );
	 total_samples = mpeg_audio_samples( file, 0 );
	 //
	 if ( video_width && video_height )
	 {
		if ( video )
		   destroy_bitmap( video );
		video = create_bitmap( video_width, video_height );
	 }
	 //
	 if ( audio )
		stop_audio_stream( audio );
	 if ( mpeg_has_audio( file ) )
	 {
		audio = play_audio_stream( AUDIO_BUFFER / audio_canales, 16, ( audio_canales == 2 ? TRUE : FALSE ), audio_rate / audio_canales, 255, 128 );
	 }
	  }
   }
   //
   return 0;
}
/***************************************************************************/
void show_video( MPEG *file )
{
   int result = mpeg_read_frame( file, 0, video );
   BITMAP *b = ( BITMAP *)the_dialog[ 2 ].dp;
   //
   if ( result == -1 )
	  close_mpeg();
   else if ( b )
	  stretch_blit( video, b, 0, 0, video->w, video->h, 0, 0, b->w, b->h );
}
//
void RefreshMpegPlayer( void *A )
{
	 if ( file )
	 {
		if ( mpeg_has_audio( file ) )
		{
		   actual_sample     = mpeg_get_sample( file, 0 );
		   porcentaje_sample = ( float )( ( actual_sample / total_samples ) * 100 );
		   //
		   audio_output = get_audio_stream_buffer( audio );
		   if ( audio_output )
		   {
		  l_int result = mpeg_read_audio( file, 0, audio_output, 1, AUDIO_BUFFER, 0 );
		  //
		  free_audio_stream_buffer( audio );
		  if ( result == -1 )
			 close_mpeg();
		   }
		}
		//
		if ( mpeg_has_video( file ) && (l_int)porcentaje_sample > (l_int)porcentaje_frame )
		{
		   while( (l_int)porcentaje_sample > (l_int)porcentaje_frame )
		   {
		  actual_frame     = mpeg_get_frame( file, 0 );
		  porcentaje_frame = ( float )( ( actual_frame / total_frames ) * 100 );
		  mpeg_drop_frames( file, 1+(porcentaje_sample-porcentaje_frame), 0 );
		   }
		}
		if ( mpeg_has_video( file ) )
		   show_video( file );
		//
		if ( mpeg_has_audio( file ) && mpeg_end_of_audio( file, 0 ) )
		   close_mpeg();
	 }
	 //
	 the_dialog[ 4 ].d2 = ( l_int )porcentaje_sample;
}
//
PDialog test_dialog;
//
PTimer T = 0;
//
l_int Main( l_int argc, l_text *argv )
{
	Filter = NewFileTypes("Multimedia Files","mpeg,mp3,mp4,mpg", NewFileTypes("All files",NULL, NULL ));
	show_my_img();
	CentreDialog( the_dialog );
	T = NewTimer( &Me, 31, &RefreshMpegPlayer, NULL );
	test_dialog = ActivateDialog( &Me, the_dialog, 0, 0 );
	//
	return true;
}
//
void Close( void )
{
   close_mpeg();
   KillTimer( T );
   CloseDialog( test_dialog );
}
