#include "kernel.h"
#include "console.h"
/****************************************************************************/
l_ulong AppVersion	= ULONG_ID( 0, 0, 0, 1 );
char AppName[]	= "Console";
l_uid NeededLibs[] = { "conlib", "" };
/****************************************************************************/
static l_int VideoRows = 24;
static l_int VideoCols = 80;
static l_int CursorX = 0;
static l_int CursorY = 0;
static l_int CursorIniRow = 6;
static l_int CursorEndRow = 7;
static l_int Attr = 0x07;
static l_byte VideoChar[ 25 ][ 80 ];
static l_byte VideoAttr[ 25 ][ 80 ];
//
static _go32_dpmi_registers regs_0x10;
static _go32_dpmi_seginfo   info_0x10;
static _go32_dpmi_seginfo   old_vector_0x10;
static _go32_dpmi_seginfo old_handler_0x08, new_handler_0x08;
//
l_int My_Int_0x10( _go32_dpmi_registers *r );
//
#define dossel 					 ( os_type == OSTYPE_UNKNOWN ? _fargetsel() : _go32_info_block.selector_for_linear_memory  )
#define CO80                     ( os_type == OSTYPE_UNKNOWN ? 0xB8000 : _go32_info_block.linear_address_of_primary_screen )
#define VIDADDR( r, c )          ( CO80 + 2 * ( ( ( r ) * VideoCols ) + ( c ) ) )
//
static l_int GetChar( l_int x, l_int y )
{
   _farsetsel( dossel );
   return _farnspeekw( VIDADDR( y, x ) ) & 0xFF;
}
//
static l_int GetAttr( l_int x, l_int y )
{
   _farsetsel( dossel );
   return _farnspeekw( VIDADDR( y, x ) ) >> 8;
}
//
static void PutCh( l_int y, l_int x, l_int c, l_int a )
{
   _farsetsel( dossel );
   _farnspokew( VIDADDR( y, x ), ( a * 0x100 ) + c );
}
//
static void ClearScreen( l_int x, l_int y, l_int w, l_int h, l_int a )
{
   l_int i, j;
   //
   for( j = y; j <= h; j++ )
   {
	  for( i = x; i <= w; i++ )
	  {
		 PutCh( j, i, ' ', a );
	  }
   }
}
//
static void MyScroll( l_int x, l_int y, l_int w, l_int h, l_byte d, l_byte a )
{
   l_int i, px, py;
   //
   if ( !d )
   {
	  ClearScreen( x, y, w, h, a );
	  //
	  return;
   }
   //
   for( py = y; py <= h; py++ )
   {
	  l_int yy = py - 1;
	  //
	  for( px = x; px <= w; px++ )
	  {
		 PutCh( yy, px, GetChar( px, py ), GetAttr( px, py ) );
	  }
   }
   //
   ClearScreen( x, h - 1, w, h - 1, a );
}
//
static void DisplayChar( l_int ch )
{
   switch( ch )
   {
      case 0x0a: // Line feed //
         CursorY++;
         break;
      case 0x0d: // Return //
         CursorX = 0;
         break;
	  case 0x8: // Backspace //
		 CursorX--;
         break;
	  case 0x7: // Bell //
         break;
      default:
         PutCh( CursorY, CursorX++, ch, Attr );
         break;
   }
   //
   if ( CursorX >= VideoCols )
   {
	  CursorX = 0;
	  CursorY++;
   }
   if ( CursorX < 0 )
   {
	  CursorX = VideoCols - 1;
	  CursorY--;
   }
   if ( CursorY >= VideoRows )
   {
	  MyScroll( 0, 1, VideoCols - 1, VideoRows, 1, Attr );
	  CursorY = VideoRows - 1;
   }
   if ( CursorY < 0 )
   {
	  MyScroll( 0, 0, VideoCols - 1, VideoRows, -1, Attr );
	  CursorY = 0;
   }
}
//
l_int My_Int_0x10( _go32_dpmi_registers *r )
{
   register l_int al = r->h.al;
   register l_int ah = r->h.ah;
   register l_int bl = r->h.bl;
   register l_int bh = r->h.bh;
   register l_int cl = r->h.cl;
   register l_int ch = r->h.ch;
   register l_int dl = r->h.dl;
   register l_int dh = r->h.dh;
   register l_int cx = r->x.cx;
   register l_int i, j;
   //
   switch( ah )
   {
	  case 0: // Set mode //
		 Attr = 0x07;
		 ClearScreen( 0, 0, VideoCols - 1, VideoRows - 1, Attr );
		 break;
	  case 1: // Set cursor type //
		 break;
	  case 2: // Move cursor position //
		 CursorX = dl;
		 CursorY = dh;
		 break;
	  case 3: // Get cursor position //
		 r->h.dl = CursorX;
		 r->h.dh = CursorY;
		 break;
	  case 6: // Scroll down/clear window //
		 Attr = bh;
		 MyScroll( cl, ch, dl, dh, al, Attr );
		 break;
	  case 7: // Scroll up/clear window //
		 Attr = bh;
		 MyScroll( cl, ch, dl, dh, -al, Attr );
		 break;
	  case 8: // Read character and attribute at cursor //
		 r->h.al = GetChar( CursorX, CursorY );
		 r->h.ah = GetAttr( CursorX, CursorY );
		 break;
	  case 9: // Write character( s ) and attribute( s ) at cursor //
		 i = cx;
		 Attr = bl;
		 //
		 for( i = 0; i < cx; i++ )
			DisplayChar( al );
		 break;
	  case 0x0a: // Write character at cursor //
		 i = cx;
		 Attr = 0x07;
		 //
		 for( i = 0; i < cx; i++ )
			DisplayChar( al );
		 break;
	  case 0x0e: // Write character in teletype mode //
		 Attr = 0x07;
		 DisplayChar( al );
		 break;
	  case 0x0f: // Return information //
		 r->h.al = 0x03;
		 r->h.ah = VideoCols;
		 break;
	  case 0x11: // Font stuff... //
		 switch( al )
		 {
		 case 0x02: // Load rom 8x8 font ( set 50/42 lines )
		 case 0x12:
			if ( VideoRows == 24 )
			   VideoRows = 50;
			else
			   VideoRows = 24;
			//
			ClearScreen( 0, 0, VideoCols, VideoRows, 0x07 );
			break;
		 case 0x30: // Get font information
			r->h.cl = 0x10;
			r->h.ch = 0x10;
			r->h.dl = 0x18;
			r->x.es = 0xC000;
			r->x.bp = 0x422C;
			r->x.cx = 0x10;
			break;
		 default:
//			_go32_dpmi_simulate_int( 0x10, r );
			break;
	  }
	  break;
	  case 0x12: // Display information //
		 switch( bl )
		 {
		 case 0x10: // Get config info
			r->h.bh = 0;
			r->h.bl = 0;
			r->x.cx = 0;
			break;
		 default:
//			_go32_dpmi_simulate_int( 0x10, r );
			//	printf( "Unimplemented int 10 function 0x12 sub-function 0x%02X\n", *bregs[ BL ] );
			break;
	  }
	  break;
	  case 0x1a: // Get/Set display combination //
		 r->h.bl = 8;
		 break;
	  default:
		 //	dbg( "Unimplemented int 10 function: %02X. AL = %02X  BL = %02X\n", ah, al, bl );
//		 _go32_dpmi_simulate_int( 0x10, r );
		 break;
   }
   //
   return 1;
} END_OF_STATIC_FUNCTION( My_Int_0x10 );
//
static void My_Int_0x08( void )
{
   l_int x, y;
   //
   for( y = 0; y < VideoRows; y++ )
   {
	  for( x = 0; x < VideoCols; x++ )
	  {
		 VideoChar[ y ][ x ] = GetChar( x, y );
		 VideoAttr[ y ][ x ] = GetAttr( x, y );
	  }
   }
   SetConsoleVideoMem( &Me, VideoChar, VideoAttr, CursorX, CursorY );
   //
}  END_OF_STATIC_FUNCTION( My_Int_0x08 );
//
static void InstallInt0x08( void )
{
   disable();
   LOCK_FUNCTION( My_Int_0x10 );
   //
   _go32_dpmi_get_protected_mode_interrupt_vector( 0x08, &old_handler_0x08 );
   //
   new_handler_0x08.pm_offset = ( l_int )My_Int_0x08;
   new_handler_0x08.pm_selector = _go32_my_cs();
   _go32_dpmi_chain_protected_mode_interrupt_vector( 0x08, &new_handler_0x08 );
   //
   enable();
}
//
static void RemoveInt0x08( void )
{
   _go32_dpmi_set_protected_mode_interrupt_vector( 0x08, &old_handler_0x08 );
}
//
static void InstallInt0x10( void )
{
   disable();
   LOCK_FUNCTION( My_Int_0x10 );
   __KeyboardStop();
   //
   memset( VideoChar, 0x20, 80 * 25 );
   memset( VideoAttr, Attr, 80 * 25 );
   Cls( &Me );
   ClearScreen( 0, 0, VideoCols - 1, VideoRows - 1, Attr );
   //
   _go32_dpmi_get_real_mode_interrupt_vector( 0x10, &old_vector_0x10 );
   info_0x10.pm_offset = ( l_int )My_Int_0x10;
   _go32_dpmi_allocate_real_mode_callback_iret( &info_0x10, &regs_0x10 );
   _go32_dpmi_set_real_mode_interrupt_vector( 0x10, &info_0x10 );
   //
   enable();
}
//
static void RemoveInt0x10( void )
{
   disable();
   _go32_dpmi_set_real_mode_interrupt_vector( 0x10, &old_vector_0x10 );
   _go32_dpmi_free_real_mode_callback( &info_0x10 );
   __KeyboardRestart();
   enable();
}
/****************************************************************************/
l_int Main( int arc, l_text *arv )
{
   InstallInt0x10();
   InstallInt0x08();
   //
   system( "command.com" );
   //
   RemoveInt0x08();
   RemoveInt0x10();
   //
   return false;
}
////////////////////////////////////////////////////////////////////////////////
void Close( void )
{
}


