#include "kernel.h"
#include "datedit/datedit.h"
#include "iodlg.h"
//
l_ulong AppVersion	= ULONG_ID(0,0,0,1);
char AppName[]		= "DatEdit Dialog Gui Library";
l_uid nUID			= "DatEdit";
l_uid NeededLibs[] = { "allegui", "iodlg", "" };
//
PFileTypes Filter = 0;
//
#define my_SCREEN_W		640
#define my_SCREEN_H		480
/* 80 characters * maximum character width (6 bytes for UTF8) */
#define FILENAME_LENGTH (80*6)


/* creates a monochrome image containing the alpha channel of this bitmap */
static BITMAP *get_alpha_bitmap(BITMAP *bmp)
{
   BITMAP *alpha;
   int x, y, c, a;
   int got = FALSE;

   if (bitmap_color_depth(bmp) != 32)
      return NULL;

   alpha = create_bitmap_ex(24, bmp->w, bmp->h);

   for (y=0; y<bmp->h; y++) {
      for (x=0; x<bmp->w; x++) {
	 c = getpixel(bmp, x, y);
	 a = geta32(c);
	 putpixel(alpha, x, y, makecol24(a, a, a));
	 if (a)
	    got = TRUE;
      }
   }

   if (!got) {
      destroy_bitmap(alpha);
      return NULL;
   }

   return alpha;
}



/* checks whether the selection is capable of having an alpha channel */
static int alpha_query(int popup)
{
   if (popup) {
      DATAFILE *dat = grabber_single_selection();
      return ((dat) && ((dat->type == DAT_BITMAP) || (dat->type == DAT_RLE_SPRITE)));
   }

   return TRUE;
}



/* views the current alpha channel */
static int view_alpha(void)
{
   DATAFILE *dat = grabber_single_selection();
   DATAFILE tmpdat;
   RLE_SPRITE *rle;
   BITMAP *bmp;
   int ret = 0;
   int i;

   if ((!dat) || ((dat->type != DAT_BITMAP) && (dat->type != DAT_RLE_SPRITE))) {
/*      Alert("You must select a single bitmap or RLE sprite",
		"object before you can view the alpha channel",
		NULL, "Sorry", NULL, 13, 0); */
      return 0;
   }

   if (dat->type == DAT_RLE_SPRITE) {
      rle = dat->dat;
      bmp = create_bitmap_ex(rle->color_depth, rle->w, rle->h);
      clear_to_color(bmp, bmp->vtable->mask_color);
      draw_rle_sprite(bmp, rle, 0, 0);
   }
   else
      bmp = dat->dat;

   tmpdat.dat = get_alpha_bitmap(bmp);

   if (tmpdat.dat) {
      tmpdat.type = DAT_BITMAP;
      tmpdat.size = 0;
      tmpdat.prop = NULL;

      for (i=0; datedit_object_info[i]->type != DAT_END; i++) {
	 if ((datedit_object_info[i]->type == DAT_BITMAP) && (datedit_object_info[i]->dclick)) {
	    ret = datedit_object_info[i]->dclick(&tmpdat);
	    break;
	 }
      }

      destroy_bitmap(tmpdat.dat);
   }
//   else
//	  Alert("There is no alpha channel in this image", NULL, NULL, "Sorry", NULL, 13, 0);

   if (dat->type == DAT_RLE_SPRITE)
      destroy_bitmap(bmp);

   return ret;
}



/* exports the current alpha channel */
static int export_alpha(void)
{
   DATAFILE *dat = grabber_single_selection();
   DATAFILE tmpdat;
   RLE_SPRITE *rle;
   BITMAP *bmp;
   char buf[256];
   l_text name;
   char *ext;
   int ret = 0;

   if ((!dat) || ((dat->type != DAT_BITMAP) && (dat->type != DAT_RLE_SPRITE))) {
/*      Alert("You must select a single bitmap or RLE sprite",
		"object before you can export the alpha channel",
		NULL, "Sorry", NULL, 13, 0); */
      return 0;
   }

   if (dat->type == DAT_RLE_SPRITE) {
      rle = dat->dat;
      bmp = create_bitmap_ex(rle->color_depth, rle->w, rle->h);
      clear_to_color(bmp, bmp->vtable->mask_color);
      draw_rle_sprite(bmp, rle, 0, 0);
   }
   else
      bmp = dat->dat;

   tmpdat.dat = get_alpha_bitmap(bmp);

   if (tmpdat.dat) {
      tmpdat.type = DAT_BITMAP;
      tmpdat.size = 0;
      tmpdat.prop = NULL;

      ext = (char*)datedit_export_ext(DAT_BITMAP);
      sprintf(buf, "Export alpha (%s)", ext);

      strcpy(name, grabber_import_file);
      *get_filename(name) = 0;

	  name = IOBox("Select File", IOBOX_OPEN, NULL, Filter, true);
	  if (name)//FileSelect(buf, name, ext))
	  {
	  name+=2;
	 fix_filename_case(name);
	 strcpy(grabber_import_file, name);
//	 grabber_busy_mouse(TRUE);
//    SaveMouse( CUR_BUSY );
	 datedit_export(&tmpdat, name);
//    RestoreMouse();
//	 grabber_busy_mouse(FALSE);
      }

      destroy_bitmap(tmpdat.dat);
      ret = D_REDRAW;
   }
//   else
//	  Alert("There is no alpha channel in this image", NULL, NULL, "Sorry", NULL, 13, 0);

   if (dat->type == DAT_RLE_SPRITE)
      destroy_bitmap(bmp);

   return ret;
}



/* deletes the current alpha channel */
static int delete_alpha(void)
{
   DATAFILE *dat = grabber_single_selection();
   RLE_SPRITE *rle;
   BITMAP *bmp;
   int got = FALSE;
   int ret = 0;
   int x, y, c, r, g, b, a;

   if ((!dat) || ((dat->type != DAT_BITMAP) && (dat->type != DAT_RLE_SPRITE))) {
/*      Alert("You must select a single bitmap or RLE sprite",
		"object before you can delete the alpha channel",
		NULL, "Sorry", NULL, 13, 0); */
      return 0;
   }

   if (dat->type == DAT_RLE_SPRITE) {
      rle = dat->dat;
      bmp = create_bitmap_ex(rle->color_depth, rle->w, rle->h);
      clear_to_color(bmp, bmp->vtable->mask_color);
      draw_rle_sprite(bmp, rle, 0, 0);
   }
   else
      bmp = dat->dat;

   if (bitmap_color_depth(bmp) == 32) {
      for (y=0; y<bmp->h; y++) {
	 for (x=0; x<bmp->w; x++) {
	    c = getpixel(bmp, x, y);
	    r = getr32(c);
	    g = getg32(c);
	    b = getb32(c);
	    a = geta32(c);
	    putpixel(bmp, x, y, makecol32(r, g, b));
	    if (a)
	       got = TRUE;
	 }
      }
   }

   if (got) {
      if (dat->type == DAT_RLE_SPRITE) {
	 destroy_rle_sprite(dat->dat);
	 dat->dat = get_rle_sprite(bmp);
      }

//      Alert("Success: alpha channel moved to /dev/null", NULL, NULL, "Cool", NULL, 13, 0);
      ret = D_REDRAW;
   }
//   else
//	  Alert("There is no alpha channel in this image", NULL, NULL, "Sorry", NULL, 13, 0);

   if (dat->type == DAT_RLE_SPRITE)
      destroy_bitmap(bmp);

   return ret;
}



/* worker function for importing alpha channels */
static BITMAP *do_alpha_import(BITMAP *bmp, int *changed, RGB *pal)
{
   BITMAP *newbmp;
   DATAFILE *alpha;
   char buf[256];
   l_text name;
   char *ext;
   int x, y, c, r, g, b, a;

   *changed = FALSE;

   ext = (char*)datedit_grab_ext(DAT_BITMAP);
   sprintf(buf, "Import alpha (%s)", ext);

   strcpy(name, grabber_import_file);
   *get_filename(name) = 0;

	  name = IOBox("Select File", IOBOX_OPEN, NULL, Filter, true);
   if (name)//FileSelect(buf, name, ext))
   {
	 name+=2;
      fix_filename_case(name);
      strcpy(grabber_import_file, name);
//    SaveMouse( CUR_BUSY );
//      grabber_busy_mouse(TRUE);
	  alpha = datedit_grab(name, name, DAT_BITMAP, -1, -1, -1, -1, -1);

	  if ((alpha) && (alpha->dat)) {
	 if (pal)
		select_palette(pal);

	 newbmp = create_bitmap_ex(32, bmp->w, bmp->h);
	 blit(bmp, newbmp, 0, 0, 0, 0, bmp->w, bmp->h);
	 destroy_bitmap(bmp);
	 bmp = newbmp;

	 if (pal)
		unselect_palette();

	 select_palette(datedit_last_read_pal);

	 for (y=0; y<bmp->h; y++) {
	    for (x=0; x<bmp->w; x++) {
	       if (getpixel(bmp, x, y) != bitmap_mask_color(bmp)) {
		  c = getpixel(alpha->dat, x, y);
		  r = getr_depth(bitmap_color_depth(alpha->dat), c);
		  g = getg_depth(bitmap_color_depth(alpha->dat), c);
		  b = getb_depth(bitmap_color_depth(alpha->dat), c);
		  a = (r+g+b)/3;

		  bmp->line[y][x*4+_rgb_a_shift_32/8] = a;
	       }
	    }
	 }

	 unselect_palette();
	 _unload_datafile_object(alpha);

	 *changed = TRUE;
      }

//      grabber_busy_mouse(FALSE);
//        RestoreMouse();
   }

   return bmp;
}



/* imports an alpha channel over the top of the current selection */
static int import_alpha(void)
{
   DATAFILE *dat = grabber_single_selection();
   RLE_SPRITE *rle;
   BITMAP *bmp;
   int changed;

   if ((!dat) || ((dat->type != DAT_BITMAP) && (dat->type != DAT_RLE_SPRITE))) {
/*      Alert("You must select a single bitmap or RLE sprite",
		"object before you can import an alpha channel",
		NULL, "Sorry", NULL, 13, 0); */
      return 0;
   }

   if (dat->type == DAT_RLE_SPRITE) {
      rle = dat->dat;
      bmp = create_bitmap_ex(rle->color_depth, rle->w, rle->h);
      clear_to_color(bmp, bmp->vtable->mask_color);
      draw_rle_sprite(bmp, rle, 0, 0);
      bmp = do_alpha_import(bmp, &changed, NULL);
      destroy_rle_sprite(rle);
      dat->dat = get_rle_sprite(bmp);
      destroy_bitmap(bmp);
   }
   else
      dat->dat = do_alpha_import(dat->dat, &changed, NULL);

   if (changed)
      view_alpha();

   return D_REDRAW;
}



/* reads an alpha channel over the top of the grab source bitmap */
static int read_alpha(void)
{
   int changed;

   if (!grabber_graphic) {
//      Alert("You must read in a bitmap file before", "you can add an alpha channel to it", NULL, "OK", NULL, 13, 0);
      return 0;
   }

   grabber_graphic = do_alpha_import(grabber_graphic, &changed, grabber_palette);

//   if (changed)
//      Alert("Alpha channel imported successfully: this will be", "used when you next grab a bitmap or RLE sprite", NULL, "OK", NULL, 13, 0);

   return D_REDRAW;
}



/* menu commands for doing stuff to the alpha channel */
static GUIMENU alpha_sub_menu[] =
{
   { "&View Alpha",     view_alpha,    NULL,    0,    NULL, NULL},
   { "&Import Alpha",   import_alpha,  NULL,    0,    NULL, NULL},
   { "&Export Alpha",   export_alpha,  NULL,    0,    NULL, NULL},
   { "&Delete Alpha",   delete_alpha,  NULL,    0,    NULL, NULL},
   { NULL,              NULL,          NULL,    0,    NULL, NULL}
};



/* per-object alpha channel menu */
static GUIMENU alpha_menu =
{
   "Alpha Channel",
   NULL,
   alpha_sub_menu,
   0,
   NULL,
   NULL
};



/* alpha channel command for the file menu */
static GUIMENU read_alpha_menu =
{
   "Read Alpha Channel",
   read_alpha,
   NULL,
   0,
   NULL,
   NULL
};



/* plugin interface header */
DATEDIT_GUIMENU_INFO datalpha_menu1 =
{
   &alpha_menu,
   alpha_query,
   DATEDIT_GUIMENU_POPUP | DATEDIT_GUIMENU_OBJECT,
   0
};



DATEDIT_GUIMENU_INFO datalpha_menu2 =
{
   &read_alpha_menu,
   NULL,
   DATEDIT_GUIMENU_FILE,
   0
};



/* creates a new FLIC object */
static void *makenew_fli(long *size)
{
   char *v = malloc(1);

   *v = 0;
   *size = 1;

   return v;
}



/* displays a FLIC object in the grabber object view window */
static void plot_fli(AL_CONST DATAFILE *dat, int x, int y)
{
   textout_ex(DialogGuiScreen, font, "Double-click in the item list to play it", x, y+32, gui_fg_color, gui_bg_color);
}



/* callback to quit out of the FLI player */
static int fli_stopper(void)
{
   poll_mouse();

   if ((keypressed()) || (mouse_b))
      return 1;
   else
      return 0;
}



/* handles double-clicking on a FLIC object in the grabber */
static int view_fli(DATAFILE *dat)
{
//   show_mouse(NULL);
//   clear_to_color(DialogGuiScreen, gui_mg_color);
   play_memory_fli(dat->dat, DialogGuiScreen, TRUE, fli_stopper);
   do {
      poll_mouse();
   } while( mouse_b );
   clear_keybuf();
   set_palette(datedit_current_palette);
//   show_mouse(DialogGuiScreen);
   return D_REDRAW;
}



/* plugin interface header */
DATEDIT_OBJECT_INFO datfli_info =
{ 
   DAT_FLI, 
   "FLI/FLC animation", 
   NULL,
   makenew_fli,
   NULL,
   plot_fli,
   view_fli,
   NULL
};



DATEDIT_GRABBER_INFO datfli_grabber =
{ 
   DAT_FLI, 
   "fli;flc",
   "fli;flc",
   NULL,
   NULL
};



/* creates a new font object */
static void* makenew_font(long* size)
{
    FONT *f;
    FONT_MONO_DATA* mf = 0, * mfread = font->data;

	f = malloc(sizeof(FONT));

    f->height = font->height;
    f->vtable = font->vtable;

    while( mfread) {
        int i;

        if(mf) {
		mf->next = malloc(sizeof(FONT_MONO_DATA));
            mf = mf->next;
	} else f->data = mf = malloc(sizeof(FONT_MONO_DATA));

        mf->begin = mfread->begin;
        mf->end = mfread->end;
        mf->next = 0;
	mf->glyphs = malloc(sizeof(FONT_GLYPH*) * (mf->end - mf->begin));

        for(i = mf->begin; i < mf->end; i++) {
            FONT_GLYPH *gsrc = mfread->glyphs[i - mf->begin], *gdest;
            int sz = ((gsrc->w + 7) / 8) * gsrc->h;

		gdest = malloc(sizeof(FONT_GLYPH) + sz);
            gdest->w = gsrc->w;
            gdest->h = gsrc->h;
            memcpy(gdest->dat, gsrc->dat, sz);

            mf->glyphs[i - mf->begin] = gdest;
        }

        mfread = mfread->next;
    }

    return f;
}



/* displays a font in the grabber object view window */
static void plot_font(AL_CONST DATAFILE *dat, int x, int y)
{
    FONT* f = dat->dat;
    int cx = x, bufpos, ch;
    char buf[8];

    y += 32;

    if(f->vtable == font_vtable_mono) {
        FONT_MONO_DATA* mf = f->data;

	while( mf) {
            for(ch = mf->begin; ch < mf->end; ch++) {
                bufpos = usetc(buf, ch);
                usetc(buf + bufpos, 0);

		if(text_length(f, buf) + cx + 4 > my_SCREEN_W) {
                    cx = x;
                    y += text_height(f) + 4;
                }

		if(y > my_SCREEN_H) break;

		textout_ex(DialogGuiScreen, f, buf, cx, y, gui_fg_color, -1);

                cx += text_length(f, buf) + 4;
            }

            mf = mf->next;
        }
    } else {
        FONT_COLOR_DATA* cf = f->data;

	while( cf) {
            for(ch = cf->begin; ch < cf->end; ch++) {
                bufpos = usetc(buf, ch);
                usetc(buf + bufpos, 0);

		if(text_length(f, buf) + cx + 4 > my_SCREEN_W) {
                    cx = x;
                    y += text_height(f) + 4;
                }

		if(y > my_SCREEN_H) break;

		textout_ex(DialogGuiScreen, f, buf, cx, y, -1, -1);

                cx += text_length(f, buf) + 4;
            }

            cf = cf->next;
        }
    }
}


/* returns a description string for a font object */
static void get_font_desc(AL_CONST DATAFILE *dat, char *s)
{
   FONT *fnt = (FONT *)dat->dat;
   char *mono = (fnt->vtable == font_vtable_mono) ? "mono" : "color";
   int ranges = 0;
   int glyphs = 0;

    if(fnt->vtable == font_vtable_mono) {
        FONT_MONO_DATA* mf = fnt->data;

	while( mf) {
            ranges++;
            glyphs += mf->end - mf->begin;
            mf = mf->next;
        }
    } else {
        FONT_COLOR_DATA* cf = fnt->data;

	while( cf) {
            ranges++;
            glyphs += cf->end - cf->begin;
            cf = cf->next;
        }
    }

   sprintf(s, "%s font, %d range%s, %d glyphs", mono, ranges, (ranges==1) ? "" : "s", glyphs);
}



/* exports a font into an external file */
static int export_font(AL_CONST DATAFILE* dat, AL_CONST char* filename)
{
    char buf[1024], tmp[1024];
    PACKFILE *pack;
    FONT* f = dat->dat;

    BITMAP *bmp;
    PALETTE pal;

    int w = 0, h = 0, max = 0, i;

    if(stricmp(get_extension(filename), "txt") == 0) {
        replace_extension(buf, filename, "pcx", sizeof(buf));

        if(exists(buf)) {
            i = datedit_ask("%s already exists, overwrite", buf);
            if(i == 27 || i == 'n' || i == 'N') return TRUE;
        }

	pack = pack_fopen(filename, F_WRITE);
	if(!pack) return FALSE;

        if(f->vtable == font_vtable_mono) {
            FONT_MONO_DATA* mf = f->data;
	    while( mf) {
                sprintf(tmp, "%s 0x%04X 0x%04X\n", (mf == f->data) ? get_filename(buf) : "-", mf->begin, mf->end - 1);
		pack_fputs(tmp, pack);
                mf = mf->next;
            }
        } else {
            FONT_COLOR_DATA* cf = f->data;
	    while( cf) {
                sprintf(tmp, "%s 0x%04X 0x%04X\n", (cf == f->data) ? get_filename(buf) : "-", cf->begin, cf->end - 1);
		pack_fputs(tmp, pack);
                cf = cf->next;
            }
        }

	pack_fclose(pack);

        filename = buf;
    } else {
        int multi = 0;

        if(f->vtable == font_vtable_mono) {
            if( ((FONT_MONO_DATA*)f->data) ->next) multi = 1;
        } else {
            if( ((FONT_COLOR_DATA*)f->data)->next) multi = 1;
        }

        if(multi) {
            i = datedit_ask("Really export multi-range font as bitmap rather than .txt");
            if(i == 27 || i == 'n' || i == 'N') return TRUE;
        }
    }

    if(f->vtable == font_vtable_mono) {
        FONT_MONO_DATA* mf = f->data;

	while( mf) {
            for(i = mf->begin; i < mf->end; i++) {
                FONT_GLYPH* g = mf->glyphs[i - mf->begin];

                max++;
                if(g->w > w) w = g->w;
                if(g->h > h) h = g->h;
            }

            mf = mf->next;
        }
    } else {
        FONT_COLOR_DATA* cf = f->data;
        int i;

	while( cf) {
            for(i = cf->begin; i < cf->end; i++) {
                BITMAP* g = cf->bitmaps[i - cf->begin];

                max++;
                if(g->w > w) w = g->w;
                if(g->h > h) h = g->h;
            }

            cf = cf->next;
        }
    }

    w = (w + 16) & 0xFFF0;
    h = (h + 16) & 0xFFF0;

    bmp = create_bitmap_ex(8, 1 + w * 16, 1 + h * ((max + 15) / 16) );
    clear_to_color(bmp, 255);

    max = 0;

    if(f->vtable == font_vtable_mono) {
        FONT_MONO_DATA* mf = f->data;

	while( mf) {
            for(i = mf->begin; i < mf->end; i++) {
		textprintf_ex(bmp, f, 1 + w * (max & 15), 1 + h * (max / 16), 1, 0, "%c", i);
                max++;
            }

            mf = mf->next;
        }
    } else {
        FONT_COLOR_DATA* cf = f->data;

	while( cf) {
            for(i = cf->begin; i < cf->end; i++) {
		textprintf_ex(bmp, f, 1 + w * (max & 15), 1 + h * (max / 16), -1, 0, "%c", i);
                max++;
            }

            cf = cf->next;
        }
    }

    get_palette(pal);

    save_bitmap(filename, bmp, pal);
    destroy_bitmap(bmp);

    return (errno == 0);
}



/* magic number for GRX format font files */
#define FONTMAGIC    0x19590214L



/* import routine for the GRX font format */
static FONT* import_grx_font(AL_CONST char* filename)
{
    PACKFILE *pack;
    FONT *f;
    FONT_MONO_DATA *mf;
    FONT_GLYPH **gl;
    int w, h, num, i;
    int* wtab = 0;

    pack = pack_fopen(filename, F_READ);
    if(!pack) return 0;

    if(pack_igetl(pack) != FONTMAGIC) {
	pack_fclose(pack);
        return 0;
    }
    pack_igetl(pack);

	f = malloc(sizeof(FONT));
	mf = malloc(sizeof(FONT_MONO_DATA));

    f->data = mf;
    f->vtable = font_vtable_mono;
    mf->next = 0;

    w = pack_igetw(pack);
    h = pack_igetw(pack);

    f->height = h;

    mf->begin = pack_igetw(pack);
    mf->end = pack_igetw(pack) + 1;
    num = mf->end - mf->begin;

	gl = mf->glyphs = malloc(sizeof(FONT_GLYPH*) * num);

    if(pack_igetw(pack) == 0) {
        for(i = 0; i < 38; i++) pack_getc(pack);
	wtab = malloc(sizeof(int) * num);
        for(i = 0; i < num; i++) wtab[i] = pack_igetw(pack);
    } else {
        for(i = 0; i < 38; i++) pack_getc(pack);
    }

    for(i = 0; i < num; i++) {
        int sz;

        if(wtab) w = wtab[i];

        sz = ((w + 7) / 8) * h;
	gl[i] = malloc(sizeof(FONT_GLYPH) + sz);
        gl[i]->w = w;
        gl[i]->h = h;

        pack_fread(gl[i]->dat, sz, pack);
    }

    if(!pack_feof(pack)) {
        char cw_char[1024];

        strcpy(cw_char, filename);
        strcpy(get_extension(cw_char), "txt");
        i = datedit_ask("Save font copyright message into '%s'", cw_char);
        if(i != 27 && i != 'n' && i != 'N') {
            PACKFILE* pout = pack_fopen(cw_char, F_WRITE);
            if(pout) {
		while( !pack_feof(pack)) {
                    i = pack_fread(cw_char, 1024, pack);
                    pack_fwrite(cw_char, i, pout);
                }
            }
            pack_fclose(pout);
        }
    }

    pack_fclose(pack);
    if(wtab) free(wtab);

    return f;
}



/* import routine for the 8x8 and 8x16 BIOS font formats */
static FONT* import_bios_font(AL_CONST char* filename)
{
    PACKFILE *pack;
    FONT *f;
    FONT_MONO_DATA *mf;
    FONT_GLYPH **gl;
    int i, h;

	f = malloc(sizeof(FONT));
	mf = malloc(sizeof(FONT_MONO_DATA));
	gl = malloc(sizeof(FONT_GLYPH*) * 256);

    pack = pack_fopen(filename, F_READ);
    if(!pack) return 0;

	h = (pack->todo == 2048) ? 8 : 16;

    for(i = 0; i < 256; i++) {
	gl[i] = malloc(sizeof(FONT_GLYPH) + 8);
        gl[i]->w = gl[i]->h = 8;
        pack_fread(gl[i]->dat, 8, pack);
    }

    f->vtable = font_vtable_mono;
    f->data = mf;
    f->height = h;

    mf->begin = 0;
    mf->end = 256;
    mf->glyphs = gl;
    mf->next = 0;

    pack_fclose(pack);

    return f;
}



/* state information for the bitmap font importer */
static BITMAP *import_bmp = NULL;

static int import_x = 0;
static int import_y = 0;



/* import_bitmap_font_mono:
 *  Helper for import_bitmap_font, below.
 */
static int import_bitmap_font_mono(FONT_GLYPH** gl, int num)
{
    int w = 1, h = 1, i;

    for(i = 0; i < num; i++) {
        if(w > 0 && h > 0) datedit_find_character(import_bmp, &import_x, &import_y, &w, &h);
        if(w <= 0 || h <= 0) {
            int j;

		gl[i] = malloc(sizeof(FONT_GLYPH) + 8);
            gl[i]->w = 8;
            gl[i]->h = 8;

            for(j = 0; j < 8; j++) gl[i]->dat[j] = 0;
        } else {
            int sx = ((w + 7) / 8), j, k;

		gl[i] = malloc(sizeof(FONT_GLYPH) + sx * h);
            gl[i]->w = w;
            gl[i]->h = h;

            for(j = 0; j < sx * h; j++) gl[i]->dat[j] = 0;
            for(j = 0; j < h; j++) {
                for(k = 0; k < w; k++) {
                    if(getpixel(import_bmp, import_x + k + 1, import_y + j + 1))
                        gl[i]->dat[(j * sx) + (k / 8)] |= 0x80 >> (k & 7);
                }
            }

            import_x += w;
        }
    }

    return 0;
}



/* import_bitmap_font_color:
 *  Helper for import_bitmap_font, below.
 */
static int import_bitmap_font_color(BITMAP** bits, int num)
{
    int w = 1, h = 1, i;

    for(i = 0; i < num; i++) {
        if(w > 0 && h > 0) datedit_find_character(import_bmp, &import_x, &import_y, &w, &h);
        if(w <= 0 || h <= 0) {
            bits[i] = create_bitmap_ex(8, 8, 8);
            if(!bits[i]) return -1;
	    clear_to_color(bits[i], 255);
        } else {
            bits[i] = create_bitmap_ex(8, w, h);
            if(!bits[i]) return -1;
            blit(import_bmp, bits[i], import_x + 1, import_y + 1, 0, 0, w, h);
            import_x += w;
        }
    }

    return 0;
}



/* bitmap_font_ismono:
 *  Helper for import_bitmap_font, below.
 */
static int bitmap_font_ismono(BITMAP *bmp)
{
    int x, y, col = -1, pixel;

    for(y = 0; y < bmp->h; y++) {
        for(x = 0; x < bmp->w; x++) {
            pixel = getpixel(bmp, x, y);
            if(pixel == 0 || pixel == 255) continue;
            if(col > 0 && pixel != col) return 0;
            col = pixel;
        }
    }

    return 1;
}



/* upgrade_to_color, upgrade_to_color_data:
 *  Helper functions. Upgrades a monochrome font to a color font.
 */
static FONT_COLOR_DATA* upgrade_to_color_data(FONT_MONO_DATA* mf)
{
	FONT_COLOR_DATA* cf = malloc(sizeof(FONT_COLOR_DATA));
	BITMAP** bits = malloc(sizeof(BITMAP*) * (mf->end - mf->begin));
    int i;

    cf->begin = mf->begin;
    cf->end = mf->end;
    cf->bitmaps = bits;
    cf->next = 0;

    for(i = mf->begin; i < mf->end; i++) {
        FONT_GLYPH* g = mf->glyphs[i - mf->begin];
        BITMAP* b = create_bitmap_ex(8, g->w, g->h);
	clear_to_color(b, 0);
        b->vtable->draw_glyph(b, g, 0, 0, 1, 0);

        bits[i - mf->begin] = b;
        free(g);
    }

    free(mf->glyphs);
    free(mf);

    return cf;
}



static void upgrade_to_color(FONT* f)
{
    FONT_MONO_DATA* mf = f->data;
    FONT_COLOR_DATA * cf, *cf_write = 0;

    if(f->vtable == font_vtable_color) return;
    f->vtable = font_vtable_color;

    while( mf) {
        FONT_MONO_DATA* mf_next = mf->next;

        cf = upgrade_to_color_data(mf);
        if(!cf_write) f->data = cf;
        else cf_write->next = cf;

        cf_write = cf;
        mf = mf_next;
    }
}



/* bitmap_font_count:
 *  Helper for `import_bitmap_font', below.
 */
static int bitmap_font_count(BITMAP* bmp)
{
    int x = 0, y = 0, w = 0, h = 0;
    int num = 0;

    while( 1) {
        datedit_find_character(bmp, &x, &y, &w, &h);
        if (w <= 0 || h <= 0) break;
        num++;
        x += w;
    }

    return num;
}



/* import routine for the Allegro .pcx font format */
static FONT* import_bitmap_font(AL_CONST char* fname, int begin, int end, int cleanup)
{
    /* NB: `end' is -1 if we want every glyph */
    FONT *f;

    if(fname) {
        PALETTE junk;

        if(import_bmp) destroy_bitmap(import_bmp);
        import_bmp = load_bitmap(fname, junk);

        import_x = 0;
        import_y = 0;
    }

    if(!import_bmp) return 0;

    if(bitmap_color_depth(import_bmp) != 8) {
        destroy_bitmap(import_bmp);
        import_bmp = 0;
        return 0;
    }

	f = malloc(sizeof(FONT));
    if(end == -1) end = bitmap_font_count(import_bmp) + begin;

    if (bitmap_font_ismono(import_bmp)) {

	FONT_MONO_DATA* mf = malloc(sizeof(FONT_MONO_DATA));

	mf->glyphs = malloc(sizeof(FONT_GLYPH*) * (end - begin));

        if( import_bitmap_font_mono(mf->glyphs, end - begin) ) {

            free(mf->glyphs);
            free(mf);
            free(f);
            f = 0;

        } else {

            f->data = mf;
            f->vtable = font_vtable_mono;
            f->height = mf->glyphs[0]->h;

            mf->begin = begin;
            mf->end = end;
            mf->next = 0;
        }

    } else {

	FONT_COLOR_DATA* cf = malloc(sizeof(FONT_COLOR_DATA));
	cf->bitmaps = malloc(sizeof(BITMAP*) * (end - begin));

        if( import_bitmap_font_color(cf->bitmaps, end - begin) ) {

            free(cf->bitmaps);
            free(cf);
            free(f);
            f = 0;

        } else {

            f->data = cf;
            f->vtable = font_vtable_color;
            f->height = cf->bitmaps[0]->h;

            cf->begin = begin;
            cf->end = end;
            cf->next = 0;

        }

    }

    if(cleanup) {
        destroy_bitmap(import_bmp);
        import_bmp = 0;
    }

    return f;
}



/* import routine for the multiple range .txt font format */
static FONT* import_scripted_font(AL_CONST char* filename)
{
    char buf[1024], *bmp_str, *start_str = 0, *end_str = 0;
    FONT *f, *f2;
    PACKFILE *pack;
    int begin, end;

    pack = pack_fopen(filename, F_READ);
    if(!pack) return 0;

	f = malloc(sizeof(FONT));
    f->data = NULL;
    f->height = 0;
    f->vtable = NULL;

    while( pack_fgets(buf, sizeof(buf)-1, pack)) {
        bmp_str = strtok(buf, " \t");
        if(bmp_str) start_str = strtok(0, " \t");
        if(start_str) end_str = strtok(0, " \t");

        if(!bmp_str || !start_str || !end_str) {
            datedit_error("Bad font description (expecting 'file.pcx start end')");

		free(f);
            pack_fclose(pack);

            return 0;
        }

        if(bmp_str[0] == '-') bmp_str = 0;
        begin = strtol(start_str, 0, 0);
        if(end_str) end = strtol(end_str, 0, 0) + 1;
        else end = -1;

        if(begin <= 0 || (end > 0 && end < begin)) {
            datedit_error("Bad font description (expecting 'file.pcx start end'); start > 0, end > start");

		free(f);
            pack_fclose(pack);

            return 0;
        }

        f2 = import_bitmap_font(bmp_str, begin, end, FALSE);
        if(!f2) {
            if(bmp_str) datedit_error("Unable to read font images from %s", bmp_str);
            else datedit_error("Unable to read continuation font images");

		free(f);
            pack_fclose(pack);

            return 0;
        }

        if(!f->vtable) f->vtable = f2->vtable;
        if(!f->height) f->height = f2->height;

        if(f2->vtable != f->vtable) {
            upgrade_to_color(f);
            upgrade_to_color(f2);
        }

        /* add to end of linked list */

        if(f->vtable == font_vtable_mono) {
            FONT_MONO_DATA* mf = f->data;
            if(!mf) f->data = f2->data;
            else {
		while( mf->next) mf = mf->next;
                mf->next = f2->data;
            }
            free(f2);
        } else {
            FONT_COLOR_DATA* cf = f->data;
            if(!cf) f->data = f2->data;
            else {
		while( cf->next) cf = cf->next;
                cf->next = f2->data;
            }
            free(f2);
        }
    }

    destroy_bitmap(import_bmp);
    import_bmp = 0;

    pack_fclose(pack);
    return f;
}



/* imports a font from an external file (handles various formats) */
static void *grab_font(AL_CONST char *filename, long *size, int x, int y, int w, int h, int depth)
{
   PACKFILE *f;
   int id;

   if (stricmp(get_extension(filename), "fnt") == 0) {
      f = pack_fopen(filename, F_READ);
      if (!f)
	 return NULL;

      id = pack_igetl(f);
      pack_fclose(f);

      if (id == FONTMAGIC)
	 return import_grx_font(filename);
      else
	 return import_bios_font(filename);
   }
   else if (stricmp(get_extension(filename), "txt") == 0) {
      return import_scripted_font(filename);
   }
   else {
      return import_bitmap_font(filename, ' ', -1, TRUE);
   }
}



/* helper for save_font, below */
static int save_mono_font(FONT* f, PACKFILE* pack)
{
    FONT_MONO_DATA* mf = f->data;
    int i = 0;

    /* count number of ranges */
    while( mf) {
        i++;
        mf = mf->next;
    }
    pack_mputw(i, pack);

    mf = f->data;

    while( mf) {

        /* mono, begin, end-1 */
        pack_putc(1, pack);
        pack_mputl(mf->begin, pack);
        pack_mputl(mf->end - 1, pack);

        for(i = mf->begin; i < mf->end; i++) {
            FONT_GLYPH* g = mf->glyphs[i - mf->begin];

            pack_mputw(g->w, pack);
            pack_mputw(g->h, pack);

            pack_fwrite(g->dat, ((g->w + 7) / 8) * g->h, pack);
        }

        mf = mf->next;

    }

    /* TODO: return FALSE on failure */
    return TRUE;
}



/* helper for save_font, below */
static int save_color_font(FONT* f, PACKFILE* pack)
{
    FONT_COLOR_DATA* cf = f->data;
    int i = 0;

    /* count number of ranges */
    while( cf) {
        i++;
        cf = cf->next;
    }
    pack_mputw(i, pack);

    cf = f->data;

    while( cf) {

        /* mono, begin, end-1 */
        pack_putc(0, pack);
        pack_mputl(cf->begin, pack);
        pack_mputl(cf->end - 1, pack);

        for(i = cf->begin; i < cf->end; i++) {
            BITMAP* g = cf->bitmaps[i - cf->begin];
            int y;

            pack_mputw(g->w, pack);
            pack_mputw(g->h, pack);

            for(y = 0; y < g->h; y++) {
                pack_fwrite(g->line[y], g->w, pack);
            }

        }

        cf = cf->next;

    }

    /* TODO: return FALSE on failure */
    return TRUE;
}



/* saves a font into a datafile */
static int save_font(DATAFILE *dat, AL_CONST int *fixed_prop, int pack, int pack_kids, int strip, int sort, int verbose, int extra, PACKFILE *f)
{
    FONT* font = dat->dat;

    pack_mputw(0, f);

    if (font->vtable == font_vtable_mono)
       return save_mono_font(font, f);
    else
       return save_color_font(font, f);
}



/* for the font viewer/editor */
static FONT *the_font;

static char char_string[8] = "0x0020";

static char *range_getter(int index, int *list_size);
static int import_proc(int msg, DIALOG *d, int c);
static int delete_proc(int msg, DIALOG *d, int c);
static int font_view_proc(int msg, DIALOG *d, int c);



static DIALOG char_dlg[] =
{
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key)    (flags)     (d1)           (d2)     (dp)                 (dp2) (dp3) */
   { WindowProcedure, 0,    0,    225,  73,   0,    0,    0,       0,          0,             0,       NULL,                NULL, NULL  },
   { TextProcedure,       16,   16,   0,    0,    0,    0,    0,       0,          0,             0,       "Base character:",   NULL, NULL  },
   { EditProcedure,       144,  16,   56,   8,    0,    0,    0,       0,          6,             0,       char_string,         NULL, NULL  },
   { ButtonProcedure,     72,   44,   81,   17,   0,    0,    13,      D_EXIT,     0,             0,       "OK",                NULL, NULL  },
   { NULL,              0,    0,    0,    0,    0,    0,    0,       0,          0,             0,       NULL,                NULL, NULL  }
};



static DIALOG view_font_dlg[] =
{
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key)    (flags)     (d1)           (d2)     (dp)              (dp2) (dp3) */
   { ClearProcedure,      0,    0,    0,    0,    0,    0,    0,       0,          0,             0,       NULL,             NULL, NULL  },
   { font_view_proc,    0,    100,  0,    0,    0,    0,    0,       D_EXIT,     0,             0,       NULL,             NULL, NULL  }, 
   { ListProcedure,       0,    0,    161,  100,  0,    0,    0,       D_EXIT,     0,             0,       range_getter,     NULL, NULL  },
   { import_proc,       180,  8,    113,  17,   0,    0,    'i',     D_EXIT,     0,             0,       "&Import Range",  NULL, NULL  }, 
   { delete_proc,       180,  40,   113,  17,   0,    0,    'd',     D_EXIT,     0,             0,       "&Delete Range",  NULL, NULL  }, 
   { ButtonProcedure,     180,  72,   113,  17,   0,    0,    27,      D_EXIT,     0,             0,       "Exit",           NULL, NULL  },
   { NULL,              0,    0,    0,    0,    0,    0,    0,       0,          0,             0,       NULL,             NULL, NULL  }
};


#define VIEWER          1
#define RANGE_LIST      2



/* dialog callback for retrieving information about the font range list */
static char *range_getter(int index, int *list_size)
{
    static char buf[64];
    FONT* f = the_font;

    if(index < 0) {
        if(!list_size) return 0;

        *list_size = 0;
        if(f->vtable == font_vtable_mono) {
            FONT_MONO_DATA* mf = f->data;
	    while( mf) {
                (*list_size)++;
                mf = mf->next;
            }
        } else {
            FONT_COLOR_DATA* cf = f->data;
	    while( cf) {
                (*list_size)++;
                cf = cf->next;
            }
        }

        return 0;
    }

    if(f->vtable == font_vtable_mono) {
        FONT_MONO_DATA* mf = f->data;
	while( index) {
            index--;
            mf = mf->next;
        }

        sprintf(buf, "%04X-%04X, mono", mf->begin, mf->end - 1);
    } else {
        FONT_COLOR_DATA* cf = f->data;
	while( index) {
            index--;
            cf = cf->next;
        }

        sprintf(buf, "%04X-%04X, color", cf->begin, cf->end - 1);
    }

   return buf;
}



/* imports a new font range */
static int import_proc(int msg, DIALOG *d, int c)
{
//   char name[80*6]; /* 80 chars * max UTF8 char width */
   l_text name;
   int ret = ButtonProcedure(msg, d, c);
   FONT *fnt, *f;
   long size;
   int base;
   int i;

   if (ret & D_CLOSE) {
      #define EXT_LIST  "bmp;fnt;lbm;pcx;tga"

      strcpy(name, grabber_import_file);
      *get_filename(name) = 0;

	  name = IOBox("Select File", IOBOX_OPEN, NULL, Filter, true);
//	  if (FileSelect("Import range (" EXT_LIST ")", name, EXT_LIST))
	  if (name)
	  {
	  name+=2;
	 fix_filename_case(name);
	 strcpy(grabber_import_file, name);

//	 grabber_busy_mouse(TRUE);
//    SaveMouse( CUR_BUSY );
	 fnt = grab_font(name, &size, -1, -1, -1, -1, -1);
//	 grabber_busy_mouse(FALSE);
//    RestoreMouse();

	 if (!fnt) {
	    datedit_error("Error importing %s", name);
	 }
	 else {
	    int import_begin, import_end;

	    CentreDialog(char_dlg);
	    SetDialogColor(char_dlg, gui_fg_color, gui_bg_color);
		DoWindow(char_dlg, 2);

	    base = strtol(char_string, NULL, 0);

        if(fnt->vtable == font_vtable_mono) {
            FONT_MONO_DATA* mf = fnt->data;
            import_end = (mf->end += (base - mf->begin));
            import_begin = mf->begin = base;
        } else {
            FONT_COLOR_DATA* cf = fnt->data;
            import_end = (cf->end += (base - cf->begin));
            import_begin = cf->begin = base;
        }

	    f = the_font;

	    if(f->vtable == font_vtable_mono) {
	        FONT_MONO_DATA* mf = f->data;
		while( mf) {
	            if(mf->end > import_begin && mf->begin < import_end) {
//		    Alert("Warning, data overlaps with an", "existing range. This almost", "certainly isn't what you want", "Silly me", NULL, 13, 0);
                    break;
	            }

	            mf = mf->next;
	        }
	    }

	    if(f->vtable != fnt->vtable) {
	        upgrade_to_color(f);
	        upgrade_to_color(fnt);
	    }

	    f = the_font;
	    i = 0;

        if(f->vtable == font_vtable_mono) {
            FONT_MONO_DATA* mf = f->data, * mf2 = fnt->data;

            if(mf->begin > import_begin) {
				mf2->next = mf;
                f->data = mf2;
            } else {
                i++;
		while( mf->next && mf->next->begin < import_begin) {
                    mf = mf->next;
                    i++;
                }
                mf2->next = mf->next;
                mf->next = mf2;
            }
        } else {
            FONT_COLOR_DATA* cf = f->data, * cf2 = fnt->data;

            if(cf->begin > import_begin) {
                cf2->next = cf;
                f->data = cf2;
            } else {
                i++;
		while( cf->next && cf->next->begin < import_begin) {
					cf = cf->next;
                    i++;
                }
                cf2->next = cf->next;
                cf->next = cf2;
            }
        }
            
	    view_font_dlg[RANGE_LIST].d1 = i;

	    ObjectMessage(view_font_dlg+VIEWER, MSG_START, 0);
	    ObjectMessage(view_font_dlg+RANGE_LIST, MSG_START, 0);
	    
	    grabber_modified(TRUE);
	 }
      }

      return D_REDRAW;
   }

   return ret;
}



/* deletes a font range */
static int delete_proc(int msg, DIALOG *d, int c)
{
   int ret = ButtonProcedure(msg, d, c);
   FONT *fnt;
   int i;

   if (ret & D_CLOSE) {
      fnt = the_font;

        if(fnt->vtable == font_vtable_mono) {

            FONT_MONO_DATA* mf = fnt->data, * mf_prev = 0;

            if(!mf->next) {
//		Alert("Deletion not possible:", "fonts must always have at", "least one character range", "Sorry", NULL, 13, 0);
                return D_O_K;
            }

            i = view_font_dlg[RANGE_LIST].d1;
	    while( i--) {
                mf_prev = mf;
                mf = mf->next;
            }

            if(mf_prev) mf_prev->next = mf->next;
            else fnt->data = mf->next;

            for(i = mf->begin; i < mf->end; i++) free(mf->glyphs[i - mf->begin]);
            free(mf->glyphs);
            free(mf);

        } else {

            FONT_COLOR_DATA* cf = fnt->data, * cf_prev = 0;

            if(!cf->next) {
//		Alert("Deletion not possible:", "fonts must always have at", "least one character range", "Sorry", NULL, 13, 0);
                return D_O_K;
            }

            i = view_font_dlg[RANGE_LIST].d1;
	    while( i--) {
                cf_prev = cf;
                cf = cf->next;
            }

            if(cf_prev) cf_prev->next = cf->next;
            else fnt->data = cf->next;

            for(i = cf->begin; i < cf->end; i++) destroy_bitmap(cf->bitmaps[i - cf->begin]);
            free(cf->bitmaps);
            free(cf);

        }

      ObjectMessage(view_font_dlg+VIEWER, MSG_START, 0);
      ObjectMessage(view_font_dlg+RANGE_LIST, MSG_START, 0);

      grabber_modified(TRUE);

      return D_REDRAW;
   }

   return ret;
}



/* displays the font contents */
static int font_view_proc(int msg, DIALOG *d, int c)
{
   BITMAP** bits = 0;
   FONT_GLYPH** gl = 0;
   FONT *fnt;
   int x, y, w, h, i;
   int font_h;
   int begin, end;

   switch (msg) {

      case MSG_START:
	 if (!d->dp)
	    d->dp = create_bitmap(d->w, d->h);
	 d->dp2 = NULL;
	 d->d1 = 0;
	 d->d2 = 0;
	 break;

      case MSG_END:
	 destroy_bitmap(d->dp);
	 d->dp = NULL;
	 break;

      case MSG_DRAW:
	 clear_to_color(d->dp, gui_mg_color);

	 fnt = the_font;
	 i = view_font_dlg[RANGE_LIST].d1;

     if(fnt->vtable == font_vtable_mono) {

        FONT_MONO_DATA* mf = fnt->data;
	while( i--) mf = mf->next;

        gl = mf->glyphs;
        begin = mf->begin;
        end = mf->end;

     } else {

        FONT_COLOR_DATA* cf = fnt->data;
	while( i--) cf = cf->next;

        bits = cf->bitmaps;
        begin = cf->begin;
        end = cf->end;

     }

	 font_h = text_height(fnt);
	 y = d->d1 + 8;

	 for (i = begin; i < end; i++) {
	    if (gl) {
	       w = gl[i - begin]->w;
	       h = gl[i - begin]->h;
	    }
	    else {
	        w = bits[i - begin]->w;
	        h = bits[i - begin]->h;
	    }

	    textprintf_ex(d->dp, font, 32, y+font_h/2-4, gui_fg_color, gui_mg_color, "U+%04X %3dx%-3d %c", i, w, h, i);
	    textprintf_ex(d->dp, fnt, 200, y, (gl ? gui_fg_color : -1), makecol(0xC0, 0xC0, 0xC0), "%c", i);

	    y += font_h * 3/2;
	 }

	 if (d->flags & D_GOTFOCUS) {
	    for (i=0; i<d->w; i+=2) {
	       putpixel(d->dp, i,   0,      gui_fg_color);
	       putpixel(d->dp, i+1, 0,      gui_bg_color);
	       putpixel(d->dp, i,   d->h-1, gui_bg_color);
	       putpixel(d->dp, i+1, d->h-1, gui_fg_color);
	    }
	    for (i=1; i<d->h-1; i+=2) {
	       putpixel(d->dp, 0,      i,   gui_bg_color);
	       putpixel(d->dp, 0,      i+1, gui_fg_color);
	       putpixel(d->dp, d->w-1, i,   gui_fg_color);
	       putpixel(d->dp, d->w-1, i+1, gui_bg_color);
	    }
	 }

	 blit(d->dp, DialogGuiScreen, 0, 0, d->x, d->y, d->w, d->h);

	 d->d2 = y - d->d1;
	 d->dp2 = fnt;
	 break; 

      case MSG_WANTFOCUS:
	 return D_WANTFOCUS;

      case MSG_CHAR:
	 switch (c >> 8) {

	    case KEY_UP:
	       d->d1 += 8;
	       break;

	    case KEY_DOWN:
	       d->d1 -= 8;
	       break;

	    case KEY_PGUP:
	       d->d1 += d->h*2/3;
	       break;

	    case KEY_PGDN:
	       d->d1 -= d->h*2/3;
	       break;

	    case KEY_HOME:
	       d->d1 = 0;
	       break;

	    case KEY_END:
	       d->d1 = -d->d2 + d->h;
	       break;

	    default:
	       return D_O_K;
	 }

	 if (d->d1 < -d->d2 + d->h)
	    d->d1 = -d->d2 + d->h;

	 if (d->d1 > 0)
	    d->d1 = 0;

	 d->flags |= D_DIRTY;
	 return D_USED_CHAR;

      case MSG_CLICK:
	 if (mouse_b & 2)
	    return D_CLOSE;

	 x = mouse_x;
	 y = mouse_y;

//	 show_mouse(NULL);

	 while( mouse_b) {
	    poll_mouse();

	    if (mouse_y != y) {
	       d->d1 += (y - mouse_y);
	       position_mouse(x, y);

	       if (d->d1 < -d->d2 + d->h)
		  d->d1 = -d->d2 + d->h;

	       if (d->d1 > 0)
		  d->d1 = 0;

	       ObjectMessage(d, MSG_DRAW, 0);
	    }
	 }

//	 show_mouse(DialogGuiScreen);
	 return D_O_K;

      case MSG_KEY:
	 return D_CLOSE;

	  case MSG_IDLE:
	 if(d->fg != view_font_dlg[RANGE_LIST].d1) {
	    d->fg = view_font_dlg[RANGE_LIST].d1;
	    d->d1 = 0;
	    ObjectMessage(d, MSG_DRAW, 0);
	 }
   }

   return D_O_K;
}



/* handles double-clicking on a font in the grabber */
static int view_font(DATAFILE *dat)
{
//   show_mouse(NULL);
//   clear_to_color(DialogGuiScreen, gui_mg_color);

   the_font = dat->dat;

   view_font_dlg[RANGE_LIST].d1 = 0;
   view_font_dlg[RANGE_LIST].d2 = 0;

   view_font_dlg[VIEWER].w = my_SCREEN_W;
   view_font_dlg[VIEWER].h = my_SCREEN_H - view_font_dlg[VIEWER].y;

   SetDialogColor(view_font_dlg, gui_fg_color, gui_bg_color);

   view_font_dlg[0].bg = gui_mg_color;

   DoWindow(view_font_dlg, VIEWER);

   dat->dat = the_font;

//   show_mouse(DialogGuiScreen);
   return D_REDRAW;
}



/* plugin interface header */
DATEDIT_OBJECT_INFO datfont_info =
{
   DAT_FONT, 
   "Font", 
   get_font_desc,
   makenew_font,
   save_font,
   plot_font,
   view_font,
   NULL
};



DATEDIT_GRABBER_INFO datfont_grabber =
{
   DAT_FONT, 
   "txt;fnt;bmp;lbm;pcx;tga",
   "txt;bmp;pcx;tga",
   grab_font,
   export_font
};



/* handles grabbing data from a bitmap */
static void grabbit(DATAFILE *dat, int box)
{
   int xgrid, ygrid;
   int using_mouse;
   int x, y, w, h;
   int ox = -1, oy = -1, ow = -1, oh = -1;
   char buf[80];
   BITMAP *graphiccopy;
   BITMAP *pattern;
   BITMAP *bmp;
   RLE_SPRITE *spr;

   grabber_get_grid_size(&xgrid, &ygrid);

//   show_mouse(NULL);
   grabber_sel_palette(grabber_palette);

//   if (bitmap_color_depth(DialogGuiScreen) == 8)
//      clear_to_color(DialogGuiScreen, gui_mg_color);

   graphiccopy = create_bitmap(grabber_graphic->w, grabber_graphic->h);
   blit(grabber_graphic, graphiccopy, 0, 0, 0, 0, grabber_graphic->w, grabber_graphic->h);

//   if (bitmap_color_depth(DialogGuiScreen) != 8)
//      clear_to_color(DialogGuiScreen, gui_mg_color);

   blit(graphiccopy, DialogGuiScreen, 0, 0, 0, 0, my_SCREEN_W, my_SCREEN_H);

   pattern = create_bitmap(2, 2);

   putpixel(pattern, 0, 0, gui_bg_color);
   putpixel(pattern, 1, 1, gui_bg_color);
   putpixel(pattern, 0, 1, gui_fg_color);
   putpixel(pattern, 1, 0, gui_fg_color);

   do {
      poll_mouse();
   } while( mouse_b);

   set_mouse_range(0, 0, grabber_graphic->w-1, grabber_graphic->h-1);

   do {
      poll_mouse();

      x = mouse_x;
      y = mouse_y;

      if ((x >= grabber_graphic->w) || (y >= grabber_graphic->h)) {
	 x = MID(0, x, grabber_graphic->w-1);
	 y = MID(0, y, grabber_graphic->h-1);
	 position_mouse(x, y);
      }

      if (!box) {
	 x = (x / xgrid) * xgrid;
	 y = (y / ygrid) * ygrid;
      }

      if ((x != ox) || (y != oy)) {

	 blit(graphiccopy, DialogGuiScreen, 0, 0, 0, 0, grabber_graphic->w, grabber_graphic->h);
	 hline(DialogGuiScreen, 0, grabber_graphic->h, grabber_graphic->w, 0);
	 vline(DialogGuiScreen, grabber_graphic->w, 0, grabber_graphic->h, 0);

	 drawing_mode(DRAW_MODE_COPY_PATTERN, pattern, 0, 0);

	 if (box) {
		hline(DialogGuiScreen, 0, y-1, grabber_graphic->w, 0);
		vline(DialogGuiScreen, x-1, 0, grabber_graphic->h, 0);
	 }
	 else {
		hline(DialogGuiScreen, x-1, y-1, grabber_graphic->w, 0);
		vline(DialogGuiScreen, x-1, y-1, grabber_graphic->h, 0);
	 }

	 drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);

	 sprintf(buf, "%d, %d", x, y);
	 rectfill(DialogGuiScreen, 0, my_SCREEN_H-8, my_SCREEN_W, my_SCREEN_H, gui_fg_color);
	 textout_ex(DialogGuiScreen, font, buf, 0, my_SCREEN_H-8, gui_bg_color, -1);

	 ox = x;
	 oy = y;
      }

      if (keypressed()) {
	 switch (readkey() >> 8) {

	    case KEY_UP:
	       position_mouse(mouse_x, MAX(mouse_y-ygrid, 0));
	       break;

	    case KEY_DOWN:
	       position_mouse(mouse_x, mouse_y+ygrid);
	       break;

	    case KEY_RIGHT:
	       position_mouse(mouse_x+xgrid, mouse_y);
	       break;

	    case KEY_LEFT:
	       position_mouse(MAX(mouse_x-xgrid, 0), mouse_y);
	       break;

	    case KEY_ENTER:
	    case KEY_SPACE:
	       goto gottheposition;

	    case KEY_ESC:
	       goto getmeoutofhere;
	 }
      }
   } while( !mouse_b);

   if (mouse_b & 2)
      goto getmeoutofhere;

   gottheposition:

   if (box) {
      int xx, yy, ww, hh;

      xx = 0;
      yy = 0;

      datedit_find_character(grabber_graphic, &xx, &yy, &ww, &hh);

      while( (ww > 0) && (hh > 0)) {
	 if ((x > xx) && (x <= xx+ww) &&
	     (y > yy) && (y <= yy+hh)) {
	    x = xx+1;
	    y = yy+1;
	    w = ww;
	    h = hh;
	    goto gotthesize;
	 }

	 xx += ww;
	 datedit_find_character(grabber_graphic, &xx, &yy, &ww, &hh);
      }

      x = 0;
      y = 0;
      w = grabber_graphic->w;
      h = grabber_graphic->h;
   }
   else {
      using_mouse = (mouse_b & 1);

      do {
	 poll_mouse();

	 w = mouse_x;
	 h = mouse_y;

	 if ((w < x) || (w > grabber_graphic->w) || ( h < y) || (h > grabber_graphic->h)) {
	    w = MID(x, w, grabber_graphic->w);
	    h = MID(y, h, grabber_graphic->h);
	    position_mouse(w, h);
	 }

	 w = ((w - x) / xgrid + 1) * xgrid;
	 if (x+w > grabber_graphic->w)
	    w = grabber_graphic->w - x;

	 h = ((h - y) / ygrid + 1) * ygrid;
	 if (y+h > grabber_graphic->h)
	    h = grabber_graphic->h - y;

	 if ((w != ow) || (h != oh)) {

		blit(graphiccopy, DialogGuiScreen, 0, 0, 0, 0, grabber_graphic->w, grabber_graphic->h);
		hline(DialogGuiScreen, 0, grabber_graphic->h, grabber_graphic->w, 0);
		vline(DialogGuiScreen, grabber_graphic->w, 0, grabber_graphic->h, 0);

	    drawing_mode(DRAW_MODE_COPY_PATTERN, pattern, 0, 0);
		hline(DialogGuiScreen, x-1, y-1, x+w, 0);
		hline(DialogGuiScreen, x-1, y+h, x+w, 0);
		vline(DialogGuiScreen, x-1, y-1, y+h, 0);
		vline(DialogGuiScreen, x+w, y-1, y+h, 0);
	    drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);

	    sprintf(buf, "%d, %d (%dx%d)", x, y, w, h);
		rectfill(DialogGuiScreen, 0, my_SCREEN_H-8, my_SCREEN_W, my_SCREEN_H, gui_fg_color);
		textout_ex(DialogGuiScreen, font, buf, 0, my_SCREEN_H-8, gui_bg_color, -1);

	    ow = w;
	    oh = h;
	 }

	 if (keypressed()) {
	    switch (readkey() >> 8) {

	       case KEY_UP:
		  position_mouse(mouse_x, MAX(mouse_y-ygrid, 0));
		  break;

	       case KEY_DOWN:
		  position_mouse(mouse_x, mouse_y+ygrid);
		  break;

	       case KEY_RIGHT:
		  position_mouse(mouse_x+xgrid, mouse_y);
		  break;

	       case KEY_LEFT:
		  position_mouse(MAX(mouse_x-xgrid, 0), mouse_y);
		  break;

	       case KEY_ENTER:
	       case KEY_SPACE:
		  goto gotthesize;

	       case KEY_ESC:
		  goto getmeoutofhere;
	    }
	 }

	 if (mouse_b & 2)
	    goto getmeoutofhere;

      } while(  (((mouse_b) && (using_mouse)) || ((!mouse_b) && (!using_mouse))));
   }

   gotthesize:

   if ((w > 0) && (h > 0)) {
      bmp = create_bitmap_ex(bitmap_color_depth(grabber_graphic), w, h);
      clear_to_color(bmp, bmp->vtable->mask_color);
      blit(grabber_graphic, bmp, x, y, 0, 0, w, h);

      if (dat->type == DAT_RLE_SPRITE) {
	 spr = get_rle_sprite(bmp);
	 destroy_bitmap(bmp);
	 destroy_rle_sprite(dat->dat);
	 dat->dat = spr;
      }
      else {
	 destroy_bitmap(dat->dat);
	 dat->dat = bmp;
      }

      sprintf(buf, "%d", x);
      datedit_set_property(dat, DAT_XPOS, buf);

      sprintf(buf, "%d", y);
      datedit_set_property(dat, DAT_YPOS, buf);

      sprintf(buf, "%d", w);
      datedit_set_property(dat, DAT_XSIZ, buf);

      sprintf(buf, "%d", h);
      datedit_set_property(dat, DAT_YSIZ, buf);

      datedit_set_property(dat, DAT_ORIG, grabber_graphic_origin);
      datedit_set_property(dat, DAT_DATE, grabber_graphic_date);
   }

   getmeoutofhere:

//   if (mouse_b)
//      clear_to_color(DialogGuiScreen, gui_mg_color);

   set_mouse_range(0, 0, my_SCREEN_W-1, my_SCREEN_H-1);

   destroy_bitmap(graphiccopy);
   destroy_bitmap(pattern);
//   show_mouse(DialogGuiScreen);

   do {
      poll_mouse();
   } while( mouse_b);
}



/* checks whether our grab command is allowed at the moment */
static int grabber_query(int popup)
{
   DATAFILE *dat = grabber_single_selection();

   if ((!dat) || (!grabber_graphic))
      return FALSE;

   return ((dat->type == DAT_BITMAP) || (dat->type == DAT_RLE_SPRITE) || (dat->type == DAT_C_SPRITE) || (dat->type == DAT_XC_SPRITE));
}



/* menu hook for our special grab command */
static int grabber(void)
{
   DATAFILE *dat = grabber_single_selection();

   if ((!dat) || (!grabber_graphic))
      return D_O_K;

   if ((dat->type != DAT_BITMAP) && (dat->type != DAT_RLE_SPRITE) &&
       (dat->type != DAT_C_SPRITE) && (dat->type != DAT_XC_SPRITE))
      return D_O_K;

   grabbit(dat, FALSE);

   datedit_sort_properties(dat->prop);
   grabber_select_property(DAT_NAME);

   grabber_modified(TRUE);

   return D_REDRAW;
}



/* checks whether our box grab command is allowed at the moment */
static int boxgrab_query(int popup)
{
   DATAFILE *dat = grabber_single_selection();

   if (!dat) {
      if (popup)
	 return FALSE;
//      else
//	 Alert("You must create and select a single object to", "contain the data before you can box-grab it", NULL, "OK", NULL, 13, 0);
   }
   else if ((dat->type != DAT_BITMAP) && (dat->type != DAT_RLE_SPRITE) &&
	    (dat->type != DAT_C_SPRITE) && (dat->type != DAT_XC_SPRITE)) {
      if (popup)
	 return FALSE;
//      else
//	 Alert("You can only box-grab to a bitmap object!", NULL, NULL, "OK", NULL, 13, 0);
   }
   else if (!grabber_graphic) {
      if (popup)
	 return FALSE;
//      else
//	 Alert("You must read in a bitmap file", "before you can box-grab data from it", NULL, "OK", NULL, 13, 0);
   }

   return TRUE;
}



/* menu hook for the box grab command */
static int boxgrab(void)
{
   DATAFILE *dat = grabber_single_selection();

   if ((!dat) || (!grabber_graphic))
      return D_O_K;

   if ((dat->type != DAT_BITMAP) && (dat->type != DAT_RLE_SPRITE) &&
       (dat->type != DAT_C_SPRITE) && (dat->type != DAT_XC_SPRITE))
      return D_O_K;

   grabbit(dat, TRUE);

   datedit_sort_properties(dat->prop);
   grabber_select_property(DAT_NAME);

   grabber_modified(TRUE);

   return D_REDRAW;
}



/* checks whether our ungrab command is allowed at the moment */
static int ungrab_query(int popup)
{
   DATAFILE *dat = grabber_single_selection();

   if ((!dat) ||
       ((dat->type != DAT_BITMAP) && (dat->type != DAT_RLE_SPRITE) &&
	(dat->type != DAT_C_SPRITE) && (dat->type != DAT_XC_SPRITE))) {
      if (popup)
	 return FALSE;
//      else
//	 Alert("Only bitmap objects can be ungrabbed!", NULL, NULL, "OK", NULL, 13, 0);
   }

   return TRUE;
}



/* menu hook for the ungrab command */
static int ungrab(void)
{
   DATAFILE *dat = grabber_single_selection();

   if (!dat)
      return D_O_K;

   if ((dat->type != DAT_BITMAP) && (dat->type != DAT_RLE_SPRITE) &&
       (dat->type != DAT_C_SPRITE) && (dat->type != DAT_XC_SPRITE))
      return D_O_K;

   if (grabber_graphic)
      destroy_bitmap(grabber_graphic);

   if (dat->type == DAT_RLE_SPRITE) {
      RLE_SPRITE *spr = (RLE_SPRITE *)dat->dat;
      grabber_graphic = create_bitmap_ex(spr->color_depth, spr->w, spr->h);
      clear_bitmap(grabber_graphic);
      draw_rle_sprite(grabber_graphic, spr, 0, 0);
   }
   else {
      BITMAP *bmp = (BITMAP *)dat->dat;
      grabber_graphic = create_bitmap_ex(bitmap_color_depth(bmp), bmp->w, bmp->h);
      blit(bmp, grabber_graphic, 0, 0, 0, 0, bmp->w, bmp->h);
   }

   if (bitmap_color_depth(grabber_graphic) == 8)
      memcpy(grabber_palette, datedit_current_palette, sizeof(PALETTE));
   else
      generate_optimized_palette(grabber_graphic, grabber_palette, NULL);

//   Alert("Bitmap data copied to the input buffer", NULL, NULL, "OK", NULL, 13, 0);

   return D_O_K;
}



/* hook ourselves into the grabber menu system */
static GUIMENU grabber_menu =
{
   "Grab",
   grabber,
   NULL,
   0,
   NULL,
   NULL
};



DATEDIT_GUIMENU_INFO datgrab_grabber_menu =
{
   &grabber_menu,
   grabber_query,
   DATEDIT_GUIMENU_OBJECT,
   0
};



static GUIMENU boxgrab_menu =
{
   "Box Grab\t(ctrl+B)",
   boxgrab,
   NULL,
   0,
   NULL,
   NULL
};



DATEDIT_GUIMENU_INFO datgrab_boxgrab_menu =
{
   &boxgrab_menu,
   boxgrab_query,
   DATEDIT_GUIMENU_OBJECT | DATEDIT_GUIMENU_POPUP,
   2  /* ctrl+B */
};



static GUIMENU ungrab_menu =
{
   "Ungrab",
   ungrab,
   NULL,
   0,
   NULL,
   NULL
};



DATEDIT_GUIMENU_INFO datgrab_ungrab_menu =
{
   &ungrab_menu,
   ungrab_query,
   DATEDIT_GUIMENU_OBJECT | DATEDIT_GUIMENU_POPUP,
   0
};



static int gg_text_proc(int msg, DIALOG *d, int c);
static int gg_edit_proc(int msg, DIALOG *d, int c);



static void *added_item[1024];
static int added_count;



/* helper for cropping bitmaps */
static BITMAP *crop_bitmap(BITMAP *bmp, int *tx, int *ty)
{
   int tw, th, i, j, c;
   int changed = FALSE;

   *tx = 0;
   *ty = 0;
   tw = bmp->w;
   th = bmp->h;

   if ((tw > 0) && (th > 0)) {
      c = getpixel(bmp, 0, 0);

      for (j=*ty; j<(*ty)+th; j++) {       /* top of image */
	 for (i=*tx; i<(*tx)+tw; i++) {
	    if (getpixel(bmp, i, j) != c)
	       goto finishedtop;
	 }
	 (*ty)++;
	 th--;
	 changed = TRUE;
      }

      finishedtop:

      for (j=(*ty)+th-1; j>*ty; j--) {     /* bottom of image */
	 for (i=*tx; i<(*tx)+tw; i++) {
	    if (getpixel(bmp, i, j) != c)
	       goto finishedbottom;
	 }
	 th--;
	 changed = TRUE;
      }

      finishedbottom:

      for (j=*tx; j<*(tx)+tw; j++) {       /* left of image */
	 for (i=*ty; i<(*ty)+th; i++) {
	    if (getpixel(bmp, j, i) != c)
	       goto finishedleft;
	 }
	 (*tx)++;
	 tw--;
	 changed = TRUE;
      }

      finishedleft:

      for (j=*(tx)+tw-1; j>*tx; j--) {     /* right of image */
	 for (i=*ty; i<(*ty)+th; i++) {
	    if (getpixel(bmp, j, i) != c)
	       goto finishedright;
	 }
	 tw--;
	 changed = TRUE;
      }

      finishedright:
	 ;
   }

   if ((tw != 0) && (th != 0) && (changed)) {
      BITMAP *b2 = create_bitmap_ex(bitmap_color_depth(bmp), tw, th);
      clear_to_color(b2, b2->vtable->mask_color);
      blit(bmp, b2, *tx, *ty, 0, 0, tw, th);
      destroy_bitmap(bmp);
      return b2;
   }
   else
      return bmp;
}



/* worker function for counting bitmap objects */
static int do_bitmap_check(DATAFILE *dat, int *param, int param2)
{
   if ((dat->type == DAT_BITMAP) || (dat->type == DAT_RLE_SPRITE) ||
       (dat->type == DAT_C_SPRITE) || (dat->type == DAT_XC_SPRITE)) 
      (*param)++;

   return D_O_K;
}



/* checks whether our grab-from-grid command is allowed at the moment */
static int griddler_query(int popup)
{
   DATAFILE *dat;
   AL_CONST char *s;

   if (popup) {
      dat = grabber_single_selection();

      if (!dat)
	 return FALSE;

      if ((dat->type != DAT_BITMAP) && (dat->type != DAT_RLE_SPRITE) &&
	  (dat->type != DAT_C_SPRITE) && (dat->type != DAT_XC_SPRITE)) 
	 return FALSE;

      s = get_datafile_property(dat, DAT_NAME);

      if ((s[0]) && (uisdigit(s[strlen(s)-1])))
	 return TRUE;
      else
	 return FALSE;
   }

   return TRUE;
}



/* checks whether a bitmap contains any data */
static int bitmap_is_empty(BITMAP *bmp)
{
   int x, y;
   int c = getpixel(bmp, 0, 0);

   for (y=0; y<bmp->h; y++)
      for (x=0; x<bmp->w; x++)
	 if (getpixel(bmp, x, y) != c)
	    return FALSE;

   return TRUE;
}



/* helper for grabbing images from a grid */
static void *griddlit(DATAFILE **parent, AL_CONST char *name, int c, int type, int skipempty, int autocrop, int depth, int x, int y, int w, int h)
{
   DATAFILE *dat;
   BITMAP *bmp;
   void *v;
   char buf[256];
   RGB tmprgb = datedit_current_palette[0];
   int tx = 0, ty = 0;

   if ((type == DAT_RLE_SPRITE) || (type == DAT_C_SPRITE) || (type == DAT_XC_SPRITE)) {
      datedit_current_palette[0].r = 63;
      datedit_current_palette[0].g = 0;
      datedit_current_palette[0].b = 63;
   }
   select_palette(datedit_current_palette);

   bmp = create_bitmap_ex(depth, w, h);
   clear_to_color(bmp, bmp->vtable->mask_color);
   blit(grabber_graphic, bmp, x, y, 0, 0, w, h);

   unselect_palette();
   datedit_current_palette[0] = tmprgb;

   if ((skipempty) && (bitmap_is_empty(bmp))) {
      destroy_bitmap(bmp);
      return NULL;
   }

   if (autocrop)
      bmp = crop_bitmap(bmp, &tx, &ty);

   if (type == DAT_RLE_SPRITE) {
      v = get_rle_sprite(bmp);
      destroy_bitmap(bmp);
   }
   else
      v = bmp;

   sprintf(buf, "%s%03d", name, c);

   *parent = datedit_insert(*parent, &dat, buf, type, v, 0);

   sprintf(buf, "%d", x);
   datedit_set_property(dat, DAT_XPOS, buf);

   sprintf(buf, "%d", y);
   datedit_set_property(dat, DAT_YPOS, buf);

   sprintf(buf, "%d", w);
   datedit_set_property(dat, DAT_XSIZ, buf);

   sprintf(buf, "%d", h);
   datedit_set_property(dat, DAT_YSIZ, buf);

   if (tx || ty) {
      sprintf(buf, "%d", tx);
      datedit_set_property(dat, DAT_XCRP, buf);

      sprintf(buf, "%d", ty);
      datedit_set_property(dat, DAT_YCRP, buf);
   }

   datedit_set_property(dat, DAT_ORIG, grabber_graphic_origin);
   datedit_set_property(dat, DAT_DATE, grabber_graphic_date);

   datedit_sort_properties(dat->prop);

   if (added_count < (int)(sizeof(added_item)/sizeof(added_item[0])))
      added_item[added_count++] = v;

   return v;
}



/* grabs images from a grid, using boxes of color #255 */
static void *box_griddle(DATAFILE **parent, AL_CONST char *name, int type, int skipempty, int autocrop, int depth)
{
   void *ret = NULL;
   void *item;
   int x, y, w, h;
   int c = 0;

   x = 0;
   y = 0;

   datedit_find_character(grabber_graphic, &x, &y, &w, &h);

   while( (w > 0) && (h > 0)) {
      item = griddlit(parent, name, c, type, skipempty, autocrop, depth, x+1, y+1, w, h);
      if (item) {
	 c++;
	 if (!ret)
	    ret = item;
      }

      x += w;
      datedit_find_character(grabber_graphic, &x, &y, &w, &h);
   }

   return ret;
}



/* grabs images from a regular grid */
static void *grid_griddle(DATAFILE **parent, AL_CONST char *name, int type, int skipempty, int autocrop, int depth, int xgrid, int ygrid)
{
   void *ret = NULL;
   void *item;
   int x, y;
   int c = 0;

   for (y=0; y+ygrid<=grabber_graphic->h; y+=ygrid) {
      for (x=0; x+xgrid<=grabber_graphic->w; x+=xgrid) {
	 item = griddlit(parent, name, c, type, skipempty, autocrop, depth, x, y, xgrid, ygrid);
	 if (item) {
	    c++;
	    if (!ret)
	       ret = item;
	 }
      } 
   }

   return ret;
}



static char griddle_xgrid[8] = "32";
static char griddle_ygrid[8] = "32";
static char griddle_name[256] = "";



/* dialog callback for retrieving the contents of the object type list */
static AL_CONST char *typelist_getter(int index, int *list_size)
{
   static char *str[] =
   {
      "Bitmap",
      "RLE Sprite",
      "Compiled Sprite",
      "Mode-X Compiled Sprite"
   };

   if (index < 0) {
      if (list_size)
	 *list_size = 4;
      return NULL;
   }

   return str[index];
}



/* dialog callback for retrieving the contents of the color depth list */
static AL_CONST char *depthlist_getter(int index, int *list_size)
{
   static char *str[] =
   {
      "256 color palette",
      "15 bit hicolor",
      "16 bit hicolor",
      "24 bit truecolor",
      "32 bit truecolor"
   };

   if (index < 0) {
      if (list_size)
	 *list_size = 5;
      return NULL;
   }

   return str[index];
}



static DIALOG griddle_dlg[] =
{
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key)    (flags)     (d1)           (d2)     (dp)                     (dp2) (dp3) */
   { WindowProcedure, 0,    0,    277,  305,  0,    0,    0,       0,          0,             0,       NULL,                    NULL, NULL  },
   { CentreTextProcedure,      138,  8,    0,    0,    0,    0,    0,       0,          0,             0,       "Grab from Grid",        NULL, NULL  },
   { RadioProcedure,      16,   32,   121,  13,   0,    0,    0,       D_SELECTED, 0,             0,       "Use col #255",          NULL, NULL  },
   { RadioProcedure,      16,   56,   121,  13,   0,    0,    0,       0,          0,             0,       "Regular grid",          NULL, NULL  },
   { gg_text_proc,      160,  58,   0,    0,    0,    0,    0,       0,          0,             0,       "X-grid:",               NULL, NULL  },
   { gg_edit_proc,      224,  58,   40,   14,    0,    0,    0,       0,          4,             0,       griddle_xgrid,           NULL, NULL  },
   { gg_text_proc,      160,  80,   0,    0,    0,    0,    0,       0,          0,             0,       "Y-grid:",               NULL, NULL  },
   { gg_edit_proc,      224,  82,   40,   14,    0,    0,    0,       0,          4,             0,       griddle_ygrid,           NULL, NULL  },
   { CheckProcedure,      16,   82,   123,  13,   0,    0,    0,       0,          0,             0,       "Skip empties:",         NULL, NULL  },
   { CheckProcedure,      16,   106,  91,   13,   0,    0,    0,       0,          0,             0,       "Autocrop:",             NULL, NULL  },
   { TextProcedure,       16,   138,  0,    0,    0,    0,    0,       0,          0,             0,       "Name:",                 NULL, NULL  },
   { EditProcedure,       64,   138,  204,  14,    0,    0,    0,       0,          255,           0,       griddle_name,            NULL, NULL  },
   { TextProcedure,       16,   160,  0,    0,    0,    0,    0,       0,          0,             0,       "Type:",                 NULL, NULL  },
   { ListProcedure,       64,   160,  197,  36,   0,    0,    0,       0,          0,             0,       (void*)typelist_getter,  NULL, NULL  },
   { TextProcedure,       16,   208,  0,    0,    0,    0,    0,       0,          0,             0,       "Cols:",                 NULL, NULL  },
   { ListProcedure,       64,   208,  197,  44,   0,    0,    0,       0,          0,             0,       (void*)depthlist_getter, NULL, NULL  },
   { ButtonProcedure,     50,   272,  81,   17,   0,    0,    13,      D_EXIT,     0,             0,       "OK",                    NULL, NULL  },
   { ButtonProcedure,     146,  272,  81,   17,   0,    0,    27,      D_EXIT,     0,             0,       "Cancel",                NULL, NULL  },
   { NULL,              0,    0,    0,    0,    0,    0,    0,       0,          0,             0,       NULL,                    NULL, NULL  }
};


#define GRIDDLE_DLG_BOXES        2
#define GRIDDLE_DLG_GRID         3
#define GRIDDLE_DLG_XGRID        5
#define GRIDDLE_DLG_YGRID        7
#define GRIDDLE_DLG_EMPTIES      8
#define GRIDDLE_DLG_AUTOCROP     9
#define GRIDDLE_DLG_NAME         11
#define GRIDDLE_DLG_TYPE         13
#define GRIDDLE_DLG_DEPTH        15
#define GRIDDLE_DLG_OK           16
#define GRIDDLE_DLG_CANCEL       17



/* wrapper for TextProcedure, that greys out when invalid */
static int gg_text_proc(int msg, DIALOG *d, int c)
{
   if (msg == MSG_IDLE) {
      int valid = (griddle_dlg[GRIDDLE_DLG_BOXES].flags & D_SELECTED) ? 0 : 1;
      int enabled = (d->flags & D_DISABLED) ? 0 : 1;

      if (valid != enabled) {
	 if (valid)
	    d->flags &= ~D_DISABLED;
	 else
	    d->flags |= D_DISABLED;

	 ObjectMessage(d, MSG_DRAW, 0);
      }

      return D_O_K;
   }
   else
      return TextProcedure(msg, d, c);
}



/* wrapper for EditProcedure, that greys out when invalid */
static int gg_edit_proc(int msg, DIALOG *d, int c)
{
   if (msg == MSG_IDLE) {
      int valid = (griddle_dlg[GRIDDLE_DLG_BOXES].flags & D_SELECTED) ? 0 : 1;
      int enabled = (d->flags & D_DISABLED) ? 0 : 1;

      if (valid != enabled) {
	 if (valid)
	    d->flags &= ~D_DISABLED;
	 else
	    d->flags |= D_DISABLED;

	 ObjectMessage(d, MSG_DRAW, 0);
      }

      return D_O_K;
   }
   else
      return EditProcedure(msg, d, c);
}



/* checks whether an object name matches the grid grab base name */
static int grid_name_matches(AL_CONST char *gn, AL_CONST char *n)
{
   AL_CONST char *s;

   if (strncmp(gn, n, strlen(gn)) != 0)
      return FALSE;

   s = n + strlen(gn);
   if (*s == 0)
      return FALSE;

   while( *s) {
      if (!uisdigit(*s))
	 return FALSE;
      s++;
   }

   return TRUE;
}



/* handle the griddle command */
static int griddler(void)
{
   DATAFILE *dat;
   DATAFILE **parent;
   void *selitem;
   char *s;
   int c;
   int xgrid, ygrid;
   int done, type, skipempty, autocrop;
   int depth = 8;

   if (!grabber_graphic) {
/*      Alert("You must read in a bitmap file",
		"before you can grab data from it",
		NULL, "OK", NULL, 13, 0); */
      return D_O_K;
   }

   grabber_get_selection_info(&dat, &parent);

   if (!dat) {
      griddle_name[0] = 0;
      depth = bitmap_color_depth(grabber_graphic);
   }
   else {
      strcpy(griddle_name, get_datafile_property(dat, DAT_NAME));
      s = griddle_name + strlen(griddle_name) - 1;
      while( (s >= griddle_name) && (uisdigit(*s))) {
	 *s = 0;
	 s--;
      }

      if (dat->type == DAT_BITMAP) {
	 griddle_dlg[GRIDDLE_DLG_TYPE].d1 = 0;
	 depth = bitmap_color_depth(dat->dat);
      }
      else if (dat->type == DAT_RLE_SPRITE) {
	 griddle_dlg[GRIDDLE_DLG_TYPE].d1 = 1;
	 depth = ((RLE_SPRITE *)dat->dat)->color_depth;
      }
      else if (dat->type == DAT_C_SPRITE) {
	 griddle_dlg[GRIDDLE_DLG_TYPE].d1 = 2;
	 depth = bitmap_color_depth(dat->dat);
      }
      else if (dat->type == DAT_XC_SPRITE) {
	 griddle_dlg[GRIDDLE_DLG_TYPE].d1 = 3;
	 depth = 8;
      }
   }

   if (depth == 8)
      griddle_dlg[GRIDDLE_DLG_DEPTH].d1 = 0;
   else if (depth == 15)
      griddle_dlg[GRIDDLE_DLG_DEPTH].d1 = 1;
   else if (depth == 16)
      griddle_dlg[GRIDDLE_DLG_DEPTH].d1 = 2;
   else if (depth == 24)
      griddle_dlg[GRIDDLE_DLG_DEPTH].d1 = 3;
   else if (depth == 32)
      griddle_dlg[GRIDDLE_DLG_DEPTH].d1 = 4;

   CentreDialog(griddle_dlg);
   SetDialogColor(griddle_dlg, gui_fg_color, gui_bg_color);

   if (DoWindow(griddle_dlg, GRIDDLE_DLG_NAME) == GRIDDLE_DLG_CANCEL)
      return D_REDRAW;

   grabber_sel_palette(grabber_palette);

   do {
      done = TRUE;

      for (c=0; (*parent)[c].type != DAT_END; c++) {
	 if (grid_name_matches(griddle_name, get_datafile_property((*parent)+c, DAT_NAME))) {
	    *parent = datedit_delete(*parent, c);
	    done = FALSE;
	    break;
	 }
      }
   } while( !done);

   switch (griddle_dlg[GRIDDLE_DLG_TYPE].d1) {

      case 0:
      default:
	 type = DAT_BITMAP;
	 break;

      case 1:
	 type = DAT_RLE_SPRITE;
	 break;

      case 2:
	 type = DAT_C_SPRITE;
	 break;

      case 3:
	 type = DAT_XC_SPRITE;
	 break;
   }

   switch (griddle_dlg[GRIDDLE_DLG_DEPTH].d1) {

      case 0:
      default:
	 depth = 8;
	 break;

      case 1:
	 depth = 15;
	 break;

      case 2:
	 depth = 16;
	 break;

      case 3:
	 depth = 24;
	 break;

      case 4:
	 depth = 32;
	 break;
   }

   skipempty = griddle_dlg[GRIDDLE_DLG_EMPTIES].flags & D_SELECTED;
   autocrop = griddle_dlg[GRIDDLE_DLG_AUTOCROP].flags & D_SELECTED;

   added_count = 0;

   if (griddle_dlg[GRIDDLE_DLG_BOXES].flags & D_SELECTED)
      selitem = box_griddle(parent, griddle_name, type, skipempty, autocrop, depth);
   else {
      xgrid = MID(1, atoi(griddle_xgrid), 0xFFFF);
      ygrid = MID(1, atoi(griddle_ygrid), 0xFFFF);
      selitem = grid_griddle(parent, griddle_name, type, skipempty, autocrop, depth, xgrid, ygrid);
   }

   datedit_sort_datafile(*parent);
   grabber_rebuild_list(selitem, TRUE);
   grabber_select_property(DAT_NAME);

   for (c=0; c<added_count; c++)
      grabber_set_selection(added_item[c]);

   if (selitem) {
      grabber_sel_palette(grabber_palette);
      grabber_modified(TRUE);
   }
//   else
//	  Alert("No grid found - nothing grabbed!", NULL, NULL, "Hmm...", NULL, 13, 0);

   set_config_string("grabber", "griddle_xgrid", griddle_xgrid);
   set_config_string("grabber", "griddle_ygrid", griddle_ygrid);

   if (griddle_dlg[GRIDDLE_DLG_BOXES].flags & D_SELECTED)
      set_config_string("grabber", "griddle_mode", "boxes");
   else
      set_config_string("grabber", "griddle_mode", "grid");

   if (griddle_dlg[GRIDDLE_DLG_EMPTIES].flags & D_SELECTED)
      set_config_string("grabber", "griddle_empties", "y");
   else
      set_config_string("grabber", "griddle_empties", "n");

   if (griddle_dlg[GRIDDLE_DLG_AUTOCROP].flags & D_SELECTED)
      set_config_string("grabber", "griddle_autocrop", "y");
   else
      set_config_string("grabber", "griddle_autocrop", "n");

   set_config_int("grabber", "griddle_type", griddle_dlg[GRIDDLE_DLG_TYPE].d1);

   return D_REDRAW;
}



/* worker function for cropping an image */
static int do_autocropper(DATAFILE *dat, int *param, int param2)
{
   BITMAP *bmp;
   RLE_SPRITE *spr;
   char buf[256];
   int tx, ty;

   if ((dat->type != DAT_BITMAP) && (dat->type != DAT_RLE_SPRITE) &&
       (dat->type != DAT_C_SPRITE) && (dat->type != DAT_XC_SPRITE)) {
      (*param)++;
      return D_O_K;
   }

   if (dat->type == DAT_RLE_SPRITE) {
      spr = (RLE_SPRITE *)dat->dat;
      bmp = create_bitmap_ex(spr->color_depth, spr->w, spr->h);
      clear_to_color(bmp, bmp->vtable->mask_color);
      draw_rle_sprite(bmp, spr, 0, 0);
      destroy_rle_sprite(spr);
      bmp = crop_bitmap(bmp, &tx, &ty);
      dat->dat = get_rle_sprite(bmp);
      destroy_bitmap(bmp);
   }
   else
      dat->dat = crop_bitmap((BITMAP *)dat->dat, &tx, &ty);

   if (tx || ty) {
      sprintf(buf, "%d", tx);
      datedit_set_property(dat, DAT_XCRP, buf);

      sprintf(buf, "%d", ty);
      datedit_set_property(dat, DAT_YCRP, buf);
   }
   
   grabber_modified(TRUE);

   return D_REDRAW;
}



/* checks whether our autocrop command is allowed at the moment */
static int autocrop_query(int popup)
{
   int n, p;

   if (popup) {
      p = 0;
      grabber_foreach_selection(do_bitmap_check, &n, &p, 0);
      return (p > 0);
   }

   return TRUE;
}



/* menu hook for the autocrop command */
static int autocrop(void)
{
   char buf[80];
   int ret, n;
   int p = 0;

   ret = grabber_foreach_selection(do_autocropper, &n, &p, 0);

   if (n <= 0) {
//      Alert ("Nothing to crop!", NULL, NULL, "OK", NULL, 13, 0);
   }
   else if (p > 0) {
      sprintf(buf, "%d non-bitmap object%s ignored", p, (p==1) ? " was" : "s were");
//      Alert(buf, NULL, NULL, "OK", NULL, 13, 0);
   }

   return ret;
}



/* initialisation code for the grab-from-grid plugin */
void datgrid_init(void)
{
   int i;

   if (DialogGuiScreen) {
      if (my_SCREEN_H < 400) {
	 griddle_dlg[0].h = griddle_dlg[0].h * 2/3;
	 for (i=1; griddle_dlg[i].proc; i++) {
	    griddle_dlg[i].y = griddle_dlg[i].y * 2/3;
	    if (griddle_dlg[i].h > 32)
	       griddle_dlg[i].h -= 8;
	 }
      }
   }

   sprintf(griddle_xgrid, "%d", get_config_int("grabber", "griddle_xgrid", 32));
   sprintf(griddle_ygrid, "%d", get_config_int("grabber", "griddle_ygrid", 32));

   if (stricmp(get_config_string("grabber", "griddle_mode", ""), "grid") == 0) {
      griddle_dlg[GRIDDLE_DLG_BOXES].flags &= ~D_SELECTED;
      griddle_dlg[GRIDDLE_DLG_GRID].flags |= D_SELECTED;
   }
   else {
      griddle_dlg[GRIDDLE_DLG_BOXES].flags |= D_SELECTED;
      griddle_dlg[GRIDDLE_DLG_GRID].flags &= ~D_SELECTED;
   }

   if (strpbrk(get_config_string("grabber", "griddle_empties", ""), "yY1"))
      griddle_dlg[GRIDDLE_DLG_EMPTIES].flags |= D_SELECTED;
   else
      griddle_dlg[GRIDDLE_DLG_EMPTIES].flags &= ~D_SELECTED;

   if (strpbrk(get_config_string("grabber", "griddle_autocrop", ""), "yY1"))
      griddle_dlg[GRIDDLE_DLG_AUTOCROP].flags |= D_SELECTED;
   else
      griddle_dlg[GRIDDLE_DLG_AUTOCROP].flags &= ~D_SELECTED;

   griddle_dlg[GRIDDLE_DLG_TYPE].d1 = get_config_int("grabber", "griddle_type", 0);
}



/* hook ourselves into the grabber menu system */
static GUIMENU griddler_menu =
{
   "Grab from Grid",
   griddler,
   NULL,
   0,
   NULL,
   NULL
};



DATEDIT_GUIMENU_INFO datgrid_griddler_menu =
{
   &griddler_menu,
   griddler_query,
   DATEDIT_GUIMENU_FILE | DATEDIT_GUIMENU_POPUP,
   0
};



static GUIMENU autocrop_menu =
{
   "Autocrop",
   autocrop,
   NULL,
   0,
   NULL,
   NULL
};



DATEDIT_GUIMENU_INFO datgrid_autocrop_menu =
{
   &autocrop_menu,
   autocrop_query,
   DATEDIT_GUIMENU_OBJECT | DATEDIT_GUIMENU_POPUP,
   0
};



/* checks whether this bitmap has an alpha channel */
static int bitmap_has_alpha(BITMAP *bmp)
{
   int x, y, c;

   if (bitmap_color_depth(bmp) != 32)
      return FALSE;

   for (y=0; y<bmp->h; y++) {
      for (x=0; x<bmp->w; x++) {
	 c = getpixel(bmp, x, y);
	 if (geta32(c))
	    return TRUE;
      }
   }

   return FALSE;
}



/* checks whether this RLE sprite has an alpha channel */
static int rle_has_alpha(AL_CONST RLE_SPRITE *spr)
{
   signed long *p32;
   int x, y, c;

   if (spr->color_depth != 32)
      return FALSE;

   p32 = (signed long *)spr->dat;

   for (y=0; y<spr->h; y++) {
      while( (unsigned long)*p32 != MASK_COLOR_32) {
	 if (*p32 < 0) {
	    p32++;
	 }
	 else {
	    x = *p32;
	    p32++;

	    while( x-- > 0) {
	       c = (unsigned long)*p32;
	       if (geta32(c))
		  return TRUE;
	       p32++;
	    }
	 }
      }

      p32++;
   }

   return FALSE;
}



/* creates a new bitmap object */
static void *makenew_bitmap(long *size)
{
   BITMAP *bmp = create_bitmap_ex(8, 32, 32);

   clear_bitmap(bmp);
   textout_centre_ex(bmp, font, "Hi!", 16, 12, 1, -1);

   return bmp;
}



/* returns a bitmap description string */
static void get_bitmap_desc(AL_CONST DATAFILE *dat, char *s)
{
   BITMAP *bmp = (BITMAP *)dat->dat;

   sprintf(s, "bitmap (%dx%d, %d bit)", bmp->w, bmp->h, bitmap_color_depth(bmp));

   if (bitmap_has_alpha(bmp))
      strcat(s, " +alpha");
}



/* exports a bitmap into an external file */
static int export_bitmap(AL_CONST DATAFILE *dat, AL_CONST char *filename)
{
   return (save_bitmap(filename, (BITMAP *)dat->dat, datedit_current_palette) == 0);
}



/* draws a bitmap onto the grabber object view window */
static void plot_bitmap(AL_CONST DATAFILE *dat, int x, int y)
{
   BITMAP *b = dat->dat;
   int w = b->w;
   int h = b->h;
   fixed scale;

   if (w > my_SCREEN_W-x-8) {
      scale = itofix(my_SCREEN_W-x-8) / w;
      w = (w * scale) >> 16;
      h = (h * scale) >> 16;
   }

   if (h > my_SCREEN_H-y-40) {
      scale = itofix(my_SCREEN_H-y-40) / h;
      w = (w * scale) >> 16;
      h = (h * scale) >> 16;
   }

   rect(DialogGuiScreen, x, y+32, x+w+1, y+32+h+1, gui_fg_color);

   if ((bitmap_color_depth(DialogGuiScreen) == 8) && (bitmap_color_depth(b) > 8)) {
      if ((w != b->w) || (h != b->h))
	 textout_ex(DialogGuiScreen, font, "<reduced color scaled preview>", x, y+18, gui_fg_color, gui_bg_color);
      else
	 textout_ex(DialogGuiScreen, font, "<reduced color preview>", x, y+18, gui_fg_color, gui_bg_color);
   }
   else if ((w != b->w) || (h != b->h))
	  textout_ex(DialogGuiScreen, font, "<scaled preview>", x, y+18, gui_fg_color, gui_bg_color);

   if (bitmap_color_depth(b) == bitmap_color_depth(DialogGuiScreen)) {
      if (dat->type != DAT_BITMAP)
	 stretch_sprite(DialogGuiScreen, b, x+1, y+33, w, h);
      else
	 stretch_blit(b, DialogGuiScreen, 0, 0, b->w, b->h, x+1, y+33, w, h);
   }
   else {
      BITMAP *b2 = create_bitmap_ex(bitmap_color_depth(b), w, h);
      clear_to_color(b2, makecol_depth(bitmap_color_depth(b2), 0x80, 0x80, 0x80));

      if (dat->type != DAT_BITMAP) {
	 RGB tmprgb = datedit_current_palette[0];

	 datedit_current_palette[0].r = 63;
	 datedit_current_palette[0].g = 0;
	 datedit_current_palette[0].b = 63;
	 select_palette(datedit_current_palette);

	 stretch_sprite(b2, b, 0, 0, w, h);
	 blit(b2, DialogGuiScreen, 0, 0, x+1, y+33, w, h);

	 unselect_palette();
	 datedit_current_palette[0] = tmprgb;
      }
      else {
	 stretch_blit(b, b2, 0, 0, b->w, b->h, 0, 0, w, h);
	 blit(b2, DialogGuiScreen, 0, 0, x+1, y+33, w, h);
      }

      destroy_bitmap(b2);
   }
}



/* handles double-clicking on a bitmap in the grabber */
static int view_bitmap(DATAFILE *dat)
{
   BITMAP *b = dat->dat;
   fixed scale = itofix(1);
   fixed prevscale = itofix(1);
   int x = 0;
   int y = 0;
   int prevx = 0;
   int prevy = 0;
   BITMAP *bc = NULL;
   int done = FALSE;
   int c;

//   show_mouse(NULL);
//   clear_to_color(DialogGuiScreen, gui_mg_color);
   blit(b, DialogGuiScreen, 0, 0, 0, 0, b->w, b->h);

   clear_keybuf();
   do {
      poll_mouse();
   } while( mouse_b );

   do {
      if ((x != prevx) || (y != prevy) || (scale != prevscale)) {
	 prevx = itofix(my_SCREEN_W) / scale;
	 prevy = itofix(my_SCREEN_H) / scale;

	 if ((b->w >= prevx) && (x+prevx > b->w))
	    x = b->w-prevx;
	 else if ((b->w < prevx) && (x > 0))
	    x = 0;

	 if ((b->h >= prevy) && (y+prevy > b->h))
	    y = b->h-prevy;
	 else if ((b->h < prevy) && (y > 0))
	    y = 0;

	 if (x < 0)
	    x = 0;

	 if (y < 0)
	    y = 0;

//	 if (scale != prevscale)
//	    clear_to_color(DialogGuiScreen, gui_mg_color);

	 if (!bc) {
	    bc = create_bitmap(b->w, b->h);
	    blit(b, bc, 0, 0, 0, 0, b->w, b->h);
	 }

	 stretch_blit(bc, DialogGuiScreen, x, y, b->w-x, b->h-y, 0, 0, ((b->w-x)*scale)>>16, ((b->h-y)*scale)>>16);

	 prevx = x;
	 prevy = y;
	 prevscale = scale;
      }

      while( keypressed()) {
	 c = readkey();

	 switch (c >> 8) {

	    case KEY_DOWN:
	       y += 4;
	       break;

	    case KEY_RIGHT:
	       x += 4;
	       break;

	    case KEY_UP:
	       y -= 4;
	       break;

	    case KEY_LEFT:
	       x -= 4;
	       break;

	    case KEY_HOME:
	       x = 0;
	       y = 0;
	       break;

	    case KEY_END:
	       x = 65536;
	       y = 65536;
	       break;

	    case KEY_PGUP:
	       if (scale > itofix(1)/16)
		  scale /= 2;
	       break;

	    case KEY_PGDN:
	       if (scale < itofix(16))
		  scale *= 2;
	       break;

	    default:
	       switch (c & 0xFF) {

		  case '+':
		     if (scale < itofix(16))
			scale *= 2;
		     break;

		  case '-':
		     if (scale > itofix(1)/16)
			scale /= 2;
		     break;

		  default:
		     done = TRUE;
		     break;
	       }
	       break;
	 }
      }

      poll_mouse();

      if (mouse_b)
	 done = TRUE;

   } while( !done );

   if (bc)
      destroy_bitmap(bc);

   clear_keybuf();

   do {
      poll_mouse();
   } while( mouse_b);

//   clear_bitmap(DialogGuiScreen);
//   show_mouse(DialogGuiScreen);

   return D_REDRAW;
}



/* reads a bitmap from an external file */
static void *grab_bitmap(AL_CONST char *filename, long *size, int x, int y, int w, int h, int depth)
{
   BITMAP *bmp;

   if (depth > 0) {
      int oldcolordepth = _color_depth;

      _color_depth = depth;
      set_color_conversion(COLORCONV_TOTAL);

      bmp = load_bitmap(filename, datedit_last_read_pal);

      _color_depth = oldcolordepth;
      set_color_conversion(COLORCONV_NONE);
   }
   else
      bmp = load_bitmap(filename, datedit_last_read_pal);

   if (!bmp)
      return NULL;

   if ((x >= 0) && (y >= 0) && (w >= 0) && (h >= 0)) {
      BITMAP *b2 = create_bitmap_ex(bitmap_color_depth(bmp), w, h);
      clear_to_color(b2, bitmap_mask_color(b2));
      blit(bmp, b2, x, y, 0, 0, w, h);
      destroy_bitmap(bmp);
      return b2;
   }
   else 
      return bmp;
}



/* saves a bitmap into the datafile format */
static int save_datafile_bitmap(DATAFILE *dat, AL_CONST int *fixed_prop, int pack, int pack_kids, int strip, int sort, int verbose, int extra, PACKFILE *f)
{
   BITMAP *bmp = (BITMAP *)dat->dat;
   int x, y, c, r, g, b, a;
   unsigned short *p16;
   unsigned long *p32;
   int depth;

   if (bitmap_has_alpha(bmp))
      depth = -32;
   else
      depth = bitmap_color_depth(bmp);

   pack_mputw(depth, f);
   pack_mputw(bmp->w, f);
   pack_mputw(bmp->h, f);

   switch (depth) {

      case 8:
	 /* 256 colors */
	 for (y=0; y<bmp->h; y++)
	    for (x=0; x<bmp->w; x++)
	       pack_putc(bmp->line[y][x], f);
	 break;

      case 15:
      case 16:
	 /* hicolor */
	 for (y=0; y<bmp->h; y++) {
	    p16 = (unsigned short *)bmp->line[y];

	    for (x=0; x<bmp->w; x++) {
	       c = p16[x];
	       r = getr_depth(depth, c);
	       g = getg_depth(depth, c);
	       b = getb_depth(depth, c);
	       c = ((r<<8)&0xF800) | ((g<<3)&0x7E0) | ((b>>3)&0x1F);
	       pack_iputw(c, f);
	    }
	 }
	 break;

      case 24:
	 /* 24 bit truecolor */
	 for (y=0; y<bmp->h; y++) {
	    for (x=0; x<bmp->w; x++) {
	       r = bmp->line[y][x*3+_rgb_r_shift_24/8];
	       g = bmp->line[y][x*3+_rgb_g_shift_24/8];
	       b = bmp->line[y][x*3+_rgb_b_shift_24/8];
	       pack_putc(r, f);
	       pack_putc(g, f);
	       pack_putc(b, f);
	    }
	 }
	 break;

      case 32:
	 /* 32 bit truecolor */
	 for (y=0; y<bmp->h; y++) {
	    p32 = (unsigned long *)bmp->line[y];

	    for (x=0; x<bmp->w; x++) {
	       c = p32[x];
	       r = getr32(c);
	       g = getg32(c);
	       b = getb32(c);
	       pack_putc(r, f);
	       pack_putc(g, f);
	       pack_putc(b, f);
	    }
	 }
	 break;

      case -32:
	 /* 32 bit truecolor with alpha channel */
	 for (y=0; y<bmp->h; y++) {
	    p32 = (unsigned long *)bmp->line[y];

	    for (x=0; x<bmp->w; x++) {
	       c = p32[x];
	       r = getr32(c);
	       g = getg32(c);
	       b = getb32(c);
	       a = geta32(c);
	       pack_putc(r, f);
	       pack_putc(g, f);
	       pack_putc(b, f);
	       pack_putc(a, f);
	    }
	 }
	 break;
   }

   /* TODO: return FALSE on failure */
   return TRUE;
}



/* creates a new RLE sprite object */
static void *makenew_rle_sprite(long *size)
{
   BITMAP *bmp = makenew_bitmap(size);
   RLE_SPRITE *spr = get_rle_sprite(bmp);

   destroy_bitmap(bmp);

   return spr;
}



/* draws an RLE sprite onto the grabber object view window */
static void plot_rle_sprite(AL_CONST DATAFILE *dat, int x, int y)
{
   RLE_SPRITE *s = dat->dat;
   BITMAP *b = create_bitmap_ex(s->color_depth, s->w, s->h);
   DATAFILE tmpdat;

   tmpdat.dat = b;
   tmpdat.type = DAT_RLE_SPRITE;
   tmpdat.size = 0;
   tmpdat.prop = NULL;

   clear_to_color(b, b->vtable->mask_color);
   draw_rle_sprite(b, s, 0, 0);
   plot_bitmap(&tmpdat, x, y);
   destroy_bitmap(b);
}



/* handles double-clicking on an RLE sprite in the grabber */
static int view_rle_sprite(DATAFILE *dat)
{
   RLE_SPRITE *spr = dat->dat;
   BITMAP *bmp = create_bitmap_ex(spr->color_depth, spr->w, spr->h);
   DATAFILE tmpdat;

   tmpdat.dat = bmp;
   tmpdat.type = DAT_RLE_SPRITE;
   tmpdat.size = 0;
   tmpdat.prop = NULL;

   clear_to_color(bmp, bmp->vtable->mask_color);
   draw_rle_sprite(bmp, spr, 0, 0);
   view_bitmap(&tmpdat);
   destroy_bitmap(bmp);
   return D_REDRAW;
}



/* returns a description string for an RLE sprite */
static void get_rle_sprite_desc(AL_CONST DATAFILE *dat, char *s)
{
   RLE_SPRITE *spr = (RLE_SPRITE *)dat->dat;

   sprintf(s, "RLE sprite (%dx%d, %d bit)", spr->w, spr->h, spr->color_depth);

   if (rle_has_alpha(spr))
      strcat(s, " +alpha");
}



/* exports an RLE sprite into an external file */
static int export_rle_sprite(AL_CONST DATAFILE *dat, AL_CONST char *filename)
{
   BITMAP *bmp;
   RLE_SPRITE *rle = (RLE_SPRITE *)dat->dat;
   int ret;

   bmp = create_bitmap_ex(rle->color_depth, rle->w, rle->h);
   clear_to_color(bmp, bmp->vtable->mask_color);
   draw_rle_sprite(bmp, rle, 0, 0);

   ret = (save_bitmap(filename, bmp, datedit_current_palette) == 0);

   destroy_bitmap(bmp);
   return ret;
}



/* reads an RLE sprite from an external file */
static void *grab_rle_sprite(AL_CONST char *filename, long *size, int x, int y, int w, int h, int depth)
{
   BITMAP *bmp;
   RLE_SPRITE *spr;

   bmp = (BITMAP *)grab_bitmap(filename, size, x, y, w, h, depth);
   if (!bmp)
      return NULL;

   spr = get_rle_sprite(bmp);
   destroy_bitmap(bmp);

   return spr;
}



/* saves an RLE sprite into the datafile format */
static int save_rle_sprite(DATAFILE *dat, AL_CONST int *fixed_prop, int pack, int pack_kids, int strip, int sort, int verbose, int extra, PACKFILE *f)
{
   RLE_SPRITE *spr = (RLE_SPRITE *)dat->dat;
   int x, y, c, r, g, b, a;
   signed short *p16;
   signed long *p32;
   unsigned long eol_marker;
   int depth;

   if (rle_has_alpha(spr))
      depth = -32;
   else
      depth = spr->color_depth;

   pack_mputw(depth, f);
   pack_mputw(spr->w, f);
   pack_mputw(spr->h, f);
   pack_mputl(spr->size, f);

   switch (depth) {

      case 8:
	 /* 256 colors, easy! */
	 pack_fwrite(spr->dat, spr->size, f);
	 break;

      case 15:
      case 16:
	 /* hicolor */
	 p16 = (signed short *)spr->dat;
	 eol_marker = (depth == 15) ? MASK_COLOR_15 : MASK_COLOR_16;

	 for (y=0; y<spr->h; y++) {
	    while( (unsigned short)*p16 != (unsigned short)eol_marker) {
	       if (*p16 < 0) {
		  /* skip count */
		  pack_iputw(*p16, f);
		  p16++;
	       }
	       else {
		  /* solid run */
		  x = *p16;
		  p16++;

		  pack_iputw(x, f);

		  while( x-- > 0) {
		     c = (unsigned short)*p16;
		     r = getr_depth(depth, c);
		     g = getg_depth(depth, c);
		     b = getb_depth(depth, c);
		     c = ((r<<8)&0xF800) | ((g<<3)&0x7E0) | ((b>>3)&0x1F);
		     pack_iputw(c, f);
		     p16++;
		  }
	       }
	    }

	    /* end of line */
	    pack_iputw(MASK_COLOR_16, f);
	    p16++;
	 }
	 break;

      case 24:
      case 32:
	 /* truecolor */
	 p32 = (signed long *)spr->dat;
	 eol_marker = (depth == 24) ? MASK_COLOR_24 : MASK_COLOR_32;

	 for (y=0; y<spr->h; y++) {
	    while( (unsigned long)*p32 != eol_marker) {
	       if (*p32 < 0) {
		  /* skip count */
		  pack_iputl(*p32, f);
		  p32++;
	       }
	       else {
		  /* solid run */
		  x = *p32;
		  p32++;

		  pack_iputl(x, f);

		  while( x-- > 0) {
		     c = (unsigned long)*p32;
		     r = getr_depth(depth, c);
		     g = getg_depth(depth, c);
		     b = getb_depth(depth, c);
		     pack_putc(r, f);
		     pack_putc(g, f);
		     pack_putc(b, f);
		     p32++;
		  }
	       }
	    }

	    /* end of line */
	    pack_iputl(MASK_COLOR_32, f);
	    p32++;
	 }
	 break;

      case -32:
	 /* truecolor with alpha channel */
	 p32 = (signed long *)spr->dat;

	 for (y=0; y<spr->h; y++) {
	    while( (unsigned long)*p32 != MASK_COLOR_32) {
	       if (*p32 < 0) {
		  /* skip count */
		  pack_iputl(*p32, f);
		  p32++;
	       }
	       else {
		  /* solid run */
		  x = *p32;
		  p32++;

		  pack_iputl(x, f);

		  while( x-- > 0) {
		     c = (unsigned long)*p32;
		     r = getr32(c);
		     g = getg32(c);
		     b = getb32(c);
		     a = geta32(c);
		     pack_putc(r, f);
		     pack_putc(g, f);
		     pack_putc(b, f);
		     pack_putc(a, f);
		     p32++;
		  }
	       }
	    }

	    /* end of line */
	    pack_iputl(MASK_COLOR_32, f);
	    p32++;
	 }
	 break;
   }

   /* TODO: return FALSE on failure */
   return TRUE;
}



/* returns a description string for a compiled sprite object */
static void get_c_sprite_desc(AL_CONST DATAFILE *dat, char *s)
{
   BITMAP *bmp = (BITMAP *)dat->dat;

   sprintf(s, "compiled sprite (%dx%d, %d bit)", bmp->w, bmp->h, bitmap_color_depth(bmp));
}



/* returns a description string for a mode-X compiled sprite object */
static void get_xc_sprite_desc(AL_CONST DATAFILE *dat, char *s)
{
   BITMAP *bmp = (BITMAP *)dat->dat;

   if (bitmap_color_depth(bmp) == 8)
      sprintf(s, "mode-X compiled sprite (%dx%d)", bmp->w, bmp->h);
   else
      sprintf(s, "!!! %d bit XC sprite not possible !!!", bitmap_color_depth(bmp));
}



/* plugin interface header */
DATEDIT_OBJECT_INFO datbitmap_info =
{
   DAT_BITMAP, 
   "Bitmap", 
   get_bitmap_desc,
   makenew_bitmap,
   save_datafile_bitmap,
   plot_bitmap,
   view_bitmap,
   NULL
};



DATEDIT_GRABBER_INFO datbitmap_grabber =
{
   DAT_BITMAP, 
   "bmp;lbm;pcx;tga",
   "bmp;pcx;tga",
   grab_bitmap,
   export_bitmap
};



/* plugin interface header */
DATEDIT_OBJECT_INFO datrlesprite_info =
{
   DAT_RLE_SPRITE, 
   "RLE sprite", 
   get_rle_sprite_desc,
   makenew_rle_sprite,
   save_rle_sprite,
   plot_rle_sprite,
   view_rle_sprite,
   NULL
};



DATEDIT_GRABBER_INFO datrlesprite_grabber =
{
   DAT_RLE_SPRITE, 
   "bmp;lbm;pcx;tga",
   "bmp;pcx;tga",
   grab_rle_sprite,
   export_rle_sprite
};



/* plugin interface header */
DATEDIT_OBJECT_INFO datcsprite_info =
{
   DAT_C_SPRITE, 
   "Compiled sprite", 
   get_c_sprite_desc,
   makenew_bitmap,
   save_datafile_bitmap,
   plot_bitmap,
   view_bitmap,
   NULL
};



DATEDIT_GRABBER_INFO datcsprite_grabber =
{
   DAT_C_SPRITE, 
   "bmp;lbm;pcx;tga",
   "bmp;pcx;tga",
   grab_bitmap,
   export_bitmap
};



/* plugin interface header */
DATEDIT_OBJECT_INFO datxcsprite_info =
{ 
   DAT_XC_SPRITE, 
   "Compiled X-sprite", 
   get_xc_sprite_desc,
   makenew_bitmap,
   save_datafile_bitmap,
   plot_bitmap,
   view_bitmap,
   NULL
};



DATEDIT_GRABBER_INFO datxcsprite_grabber =
{ 
   DAT_XC_SPRITE, 
   "bmp;lbm;pcx;tga",
   "bmp;pcx;tga",
   grab_bitmap,
   export_bitmap
};



/* checks whether our image changing commands are allowed at the moment */
static int change_query(int popup)
{
   int n, p;

   if (popup) {
      p = 0;
      grabber_foreach_selection(do_bitmap_check, &n, &p, 0);
      return (p > 0);
   }

   return TRUE;
}



/* worker function for changing the type of an image */
static int do_changetype(DATAFILE *dat, int *param, int type)
{
   BITMAP *bmp;
   RLE_SPRITE *spr;
   int x, y, c, r, g, b;

   if ((dat->type != DAT_BITMAP) && (dat->type != DAT_RLE_SPRITE) &&
       (dat->type != DAT_C_SPRITE) && (dat->type != DAT_XC_SPRITE)) {
      (*param)++;
      return D_O_K;
   }

   if (dat->type == type)
      return D_O_K;

   if (dat->type == DAT_RLE_SPRITE) {
      spr = (RLE_SPRITE *)dat->dat;
      bmp = create_bitmap_ex(spr->color_depth, spr->w, spr->h);
      clear_to_color(bmp, bmp->vtable->mask_color);
      draw_rle_sprite(bmp, spr, 0, 0);
      dat->dat = bmp;
      destroy_rle_sprite(spr);
   }
   else if (type == DAT_RLE_SPRITE) {
      bmp = (BITMAP *)dat->dat;
      spr = get_rle_sprite(bmp);
      dat->dat = spr;
      destroy_bitmap(bmp);
   }
   else if ((type == DAT_C_SPRITE) || (type == DAT_XC_SPRITE)) {
      bmp = (BITMAP *)dat->dat;
      if (bitmap_color_depth(bmp) == 32) {
	 for (y=0; y<bmp->h; y++) {
	    for (x=0; x<bmp->w; x++) {
	       c = getpixel(bmp, x, y);
	       r = getr32(c);
	       g = getg32(c);
	       b = getb32(c);
	       putpixel(bmp, x, y, makecol32(r, g, b));
	    }
	 }
      }
   }

   dat->type = type;

   return D_REDRAW;
}



/* changes the type of bitmap data */
static int changetype(void)
{
   int type = (int)ActiveMenu->dp;
   char buf[80];
   int ret, n;
   int p = 0;

   ret = grabber_foreach_selection(do_changetype, &n, &p, type);

   if (n <= 0) {
//      Alert ("Nothing to re-type!", NULL, NULL, "OK", NULL, 13, 0);
   }
   else if (p > 0) {
      sprintf(buf, "%d non-bitmap object%s ignored", p, (p==1) ? " was" : "s were");
//      Alert(buf, NULL, NULL, "OK", NULL, 13, 0);
   }

   if (n > p)
      grabber_rebuild_list(NULL, FALSE);

   return ret;
}



/* worker function for changing the color depth of bitmap data */
static int do_changedepth(DATAFILE *dat, int *param, int depth)
{
   BITMAP *bmp, *bmp2;
   RLE_SPRITE *spr;
   RGB tmprgb = datedit_current_palette[0];

   if ((dat->type != DAT_BITMAP) && (dat->type != DAT_RLE_SPRITE) &&
       (dat->type != DAT_C_SPRITE) && (dat->type != DAT_XC_SPRITE)) {
      (*param)++;
      return D_O_K;
   }

   if (dat->type == DAT_RLE_SPRITE) {
      spr = (RLE_SPRITE *)dat->dat;
      if (spr->color_depth == depth)
	 return D_O_K;

      bmp = create_bitmap_ex(spr->color_depth, spr->w, spr->h);
      clear_to_color(bmp, bmp->vtable->mask_color);
      draw_rle_sprite(bmp, spr, 0, 0);
      bmp2 = create_bitmap_ex(depth, bmp->w, bmp->h);

      datedit_current_palette[0].r = 63;
      datedit_current_palette[0].g = 0;
      datedit_current_palette[0].b = 63;
      select_palette(datedit_current_palette);

      blit(bmp, bmp2, 0, 0, 0, 0, bmp->w, bmp->h);

      unselect_palette();
      datedit_current_palette[0] = tmprgb;

      dat->dat = get_rle_sprite(bmp2);
      destroy_bitmap(bmp);
      destroy_bitmap(bmp2);
      destroy_rle_sprite(spr);
   }
   else {
      bmp = (BITMAP *)dat->dat;
      if (bitmap_color_depth(bmp) == depth)
	 return D_O_K;
      bmp2 = create_bitmap_ex(depth, bmp->w, bmp->h);

      if ((dat->type == DAT_C_SPRITE) || (dat->type == DAT_XC_SPRITE)) {
	 datedit_current_palette[0].r = 63;
	 datedit_current_palette[0].g = 0;
	 datedit_current_palette[0].b = 63;
      }
      select_palette(datedit_current_palette);

      blit(bmp, bmp2, 0, 0, 0, 0, bmp->w, bmp->h);

      unselect_palette();
      datedit_current_palette[0] = tmprgb;

      dat->dat = bmp2;
      destroy_bitmap(bmp);
   }

   return D_REDRAW;
}



/* changes the color depth of bitmap data */
static int changedepth(void)
{
   int depth = (int)ActiveMenu->dp;
   char buf[80];
   int ret, n;
   int p = 0;

//    SaveMouse( CUR_BUSY );
//   grabber_busy_mouse(TRUE);

   ret = grabber_foreach_selection(do_changedepth, &n, &p, depth);

//   grabber_busy_mouse(FALSE);
//     RestoreMouse();

   if (n <= 0) {
//      Alert ("Nothing to re-format!", NULL, NULL, "OK", NULL, 13, 0);
   }
   else if (p > 0) {
      sprintf(buf, "%d non-bitmap object%s ignored", p, (p==1) ? " was" : "s were");
//      Alert(buf, NULL, NULL, "OK", NULL, 13, 0);
   }

   return ret;
}



static GUIMENU type_menu[] =
{
   { "To &Bitmap",               changetype,    NULL,    0,    (void *)DAT_BITMAP, NULL      },
   { "To &RLE Sprite",           changetype,    NULL,    0,    (void *)DAT_RLE_SPRITE, NULL  },
   { "To &Compiled Sprite",      changetype,    NULL,    0,    (void *)DAT_C_SPRITE, NULL    },
   { "To &X-Compiled Sprite",    changetype,    NULL,    0,    (void *)DAT_XC_SPRITE, NULL   },
   { NULL,                       NULL,          NULL,    0,    NULL, NULL	    }
};



static GUIMENU depth_menu[] =
{
   { "&256 color palette",       changedepth,   NULL,    0,    (void *)8, NULL},
   { "1&5 bit hicolor",          changedepth,   NULL,    0,    (void *)15, NULL},
   { "1&6 bit hicolor",          changedepth,   NULL,    0,    (void *)16, NULL},
   { "2&4 bit truecolor",        changedepth,   NULL,    0,    (void *)24, NULL},
   { "&32 bit truecolor",        changedepth,   NULL,    0,    (void *)32, NULL},
   { NULL,                       NULL,          NULL,    0,    NULL, NULL}
};



/* hook ourselves into the grabber menu system */
static GUIMENU change_type_menu =
{
   "Change Type",
   NULL,
   type_menu,
   0,
   NULL,
   NULL
};



DATEDIT_GUIMENU_INFO datitype_type_menu =
{
   &change_type_menu,
   change_query,
   DATEDIT_GUIMENU_OBJECT | DATEDIT_GUIMENU_POPUP,
   0
};



static GUIMENU change_depth_menu =
{
   "Color Depth",
   NULL,
   depth_menu,
   0,
   NULL,
   NULL
};



DATEDIT_GUIMENU_INFO datitype_depth_menu =
{
   &change_depth_menu,
   change_query,
   DATEDIT_GUIMENU_OBJECT | DATEDIT_GUIMENU_POPUP,
   0
};




/* creates a new MIDI object */
static void *makenew_midi(long *size)
{
   MIDI *mid;
   int c;

   mid = malloc(sizeof(MIDI));
   mid->divisions = 120;

   for (c=0; c<MIDI_TRACKS; c++) {
      mid->track[c].data = NULL;
      mid->track[c].len = 0;
   }

   return mid;
}



/* displays a MIDI file in the grabber object view window */
static void plot_midi(AL_CONST DATAFILE *dat, int x, int y)
{
   textout_ex(DialogGuiScreen, font, "Double-click in the item list to play it", x, y+32, gui_fg_color, gui_bg_color);
}



/* handles double-clicking on a MIDI file in the grabber */
static int dclick_midi(DATAFILE *dat)
{
   play_midi(dat->dat, FALSE);
   return D_O_K;
}



/* exports a MIDI object into an external file */
static int export_midi(AL_CONST DATAFILE *dat, AL_CONST char *filename)
{
   MIDI *midi = (MIDI *)dat->dat;
   PACKFILE *f;
   int num_tracks;
   int c;

   num_tracks = 0;
   for (c=0; c<MIDI_TRACKS; c++)
      if (midi->track[c].len > 0)
	 num_tracks++;

   errno = 0; 

   f = pack_fopen(filename, F_WRITE);

   if (f) {
      pack_fputs("MThd", f);                 /* MIDI header */
      pack_mputl(6, f);                      /* size of header chunk */
      pack_mputw(1, f);                      /* type 1 */
      pack_mputw(num_tracks, f);             /* number of tracks */
      pack_mputw(midi->divisions, f);        /* beat divisions */

      for (c=0; c<MIDI_TRACKS; c++) {        /* for each track */
	 if (midi->track[c].len > 0) {
	    pack_fputs("MTrk", f);           /* write track data */
	    pack_mputl(midi->track[c].len, f); 
	    pack_fwrite(midi->track[c].data, midi->track[c].len, f);
	 }
      }

      pack_fclose(f);
   }

   return (errno == 0);
}



/* imports a MIDI object from an external file */
static void *grab_midi(AL_CONST char *filename, long *size, int x, int y, int w, int h, int depth)
{
   return load_midi(filename);
}



/* saves a MIDI object in the datafile format */
static int save_midi(DATAFILE *dat, AL_CONST int *fixed_prop, int pack, int pack_kids, int strip, int sort, int verbose, int extra, PACKFILE *f)
{
   MIDI *midi = (MIDI *)dat->dat;
   int c;

   pack_mputw(midi->divisions, f);

   for (c=0; c<MIDI_TRACKS; c++) {
      pack_mputl(midi->track[c].len, f);
      if (midi->track[c].len > 0)
	 pack_fwrite(midi->track[c].data, midi->track[c].len, f);
   }

   /* TODO: return FALSE on failure */
   return TRUE;
}



/* plugin interface header */
DATEDIT_OBJECT_INFO datmidi_info =
{
   DAT_MIDI, 
   "MIDI file", 
   NULL,
   makenew_midi,
   save_midi,
   plot_midi,
   dclick_midi,
   NULL
};



DATEDIT_GRABBER_INFO datmidi_grabber =
{
   DAT_MIDI, 
   "mid",
   "mid",
   grab_midi,
   export_midi
};




/* creates a new palette object */
static void *makenew_palette(long *size)
{
   RGB *pal = malloc(sizeof(PALETTE));

   memcpy(pal, default_palette, sizeof(PALETTE));
   *size = sizeof(PALETTE);

   return pal;
}



/* tests whether two palettes are identical */
static int compare_palettes(RGB *p1, RGB *p2)
{
   int c;

   for (c=0; c<PAL_SIZE; c++) {
      if ((p1[c].r != p2[c].r) || 
	  (p1[c].g != p2[c].g) || 
	  (p1[c].b != p2[c].b))
	 return TRUE;
   }

   return FALSE;
}



/* draws a palette onto the grabber object view window */
static void plot_palette(AL_CONST DATAFILE *dat, int x, int y)
{
   int c;

   if (compare_palettes(dat->dat, datedit_current_palette)) {
	  if (bitmap_color_depth(DialogGuiScreen) > 8) {
	 select_palette(dat->dat);
	 for (c=0; c<PAL_SIZE; c++)
		rectfill(DialogGuiScreen, x+(c&15)*8, y+(c/16)*8+32,
		     x+(c&15)*8+7, y+(c/16)*8+39, palette_color[c]);
	 unselect_palette();
	 textout_ex(DialogGuiScreen, font, "A different palette", x+160, y+32, gui_fg_color, gui_bg_color);
	 textout_ex(DialogGuiScreen, font, "is currently in use.", x+160, y+40, gui_fg_color, gui_bg_color);
	 textout_ex(DialogGuiScreen, font, "To select this one,", x+160, y+48, gui_fg_color, gui_bg_color);
	 textout_ex(DialogGuiScreen, font, "double-click on it", x+160, y+56, gui_fg_color, gui_bg_color);
	 textout_ex(DialogGuiScreen, font, "in the item list.", x+160, y+64, gui_fg_color, gui_bg_color);
      }
      else {
	 textout_ex(DialogGuiScreen, font, "A different palette is currently in use.", x, y+32, gui_fg_color, gui_bg_color);
	 textout_ex(DialogGuiScreen, font, "To select this one, double-click on it", x, y+40, gui_fg_color, gui_bg_color);
	 textout_ex(DialogGuiScreen, font, "in the item list.", x, y+48, gui_fg_color, gui_bg_color);
      }
   }
   else {
      for (c=0; c<PAL_SIZE; c++)
	 rectfill(DialogGuiScreen, x+(c&15)*8, y+(c/16)*8+32,
		  x+(c&15)*8+7, y+(c/16)*8+39, palette_color[c]);
   }
}



/* handles double-clicking on a palette object in the grabber */
static int use_palette(DATAFILE *dat)
{
   grabber_sel_palette(dat->dat);
   return D_REDRAW;
}



/* exports a palette into an external file */
static int export_palette(AL_CONST DATAFILE *dat, AL_CONST char *filename)
{
   BITMAP *bmp;
   int ret;

   bmp = create_bitmap_ex(8, 32, 8);
   clear_bitmap(bmp);
   textout_ex(bmp, font, "PAL.", 0, 0, 255, 0);

   ret = (save_bitmap(filename, bmp, (RGB *)dat->dat) == 0);

   destroy_bitmap(bmp);
   return ret;
}



/* grabs a palette from an external file */
static void *grab_palette(AL_CONST char *filename, long *size, int x, int y, int w, int h, int depth)
{
   int oldcolordepth = _color_depth;
   RGB *pal;
   BITMAP *bmp;

   _color_depth = 8;
   set_color_conversion(COLORCONV_TOTAL);

   pal = malloc(sizeof(PALETTE));
   bmp = load_bitmap(filename, pal);

   _color_depth = oldcolordepth;
   set_color_conversion(COLORCONV_NONE);

   if (!bmp) {
	  free(pal);
      return NULL;
   }

   destroy_bitmap(bmp);

   *size = sizeof(PALETTE);
   return pal;
}



/* checks whether our grab command is allowed at the moment */
static int grabber_query2(int popup)
{
   DATAFILE *dat = grabber_single_selection();

   return ((dat) && (dat->type == DAT_PALETTE) && (grabber_graphic));
}



DATEDIT_GUIMENU_INFO datpal_grabber_menu =
{
   &grabber_menu,
   grabber_query2,
   DATEDIT_GUIMENU_OBJECT,
   0
};



/* plugin interface header */
DATEDIT_OBJECT_INFO datpal_info =
{
   DAT_PALETTE, 
   "Palette", 
   NULL,
   makenew_palette,
   NULL,
   plot_palette,
   use_palette,
   NULL
};



DATEDIT_GRABBER_INFO datpal_grabber =
{
   DAT_PALETTE, 
   "bmp;lbm;pcx;tga",
   "bmp;pcx;tga",
   grab_palette,
   export_palette
};



/* creates a new sample object */
static void *makenew_sample(long *size)
{
   return create_sample(8, FALSE, 11025, 1024);
}



/* displays a sample in the grabber object view window */
static void plot_sample(AL_CONST DATAFILE *dat, int x, int y)
{
   textout_ex(DialogGuiScreen, font, "Double-click in the item list to play it", x, y+32, gui_fg_color, gui_bg_color);
}



/* handles double-clicking on a sample in the grabber */
static int dclick_sample(DATAFILE *dat)
{
   play_sample(dat->dat, 255, 127, 1000, FALSE);
   return D_O_K;
}



/* returns an information string describing a sample object */
static void get_sample_desc(AL_CONST DATAFILE *dat, char *s)
{
   SAMPLE *sample = (SAMPLE *)dat->dat;
   long sec = (sample->len + sample->freq/2) * 10 / MAX(sample->freq, 1);
   char *type = (sample->stereo) ? "stereo" : "mono";

   sprintf(s, "sample (%d bit %s, %d, %ld.%ld sec)", sample->bits, type, sample->freq, sec/10, sec%10);
}



/* exports a sample into an external file */
static int export_sample(AL_CONST DATAFILE *dat, AL_CONST char *filename)
{
   SAMPLE *spl = (SAMPLE *)dat->dat;
   int bps = spl->bits/8 * ((spl->stereo) ? 2 : 1);
   int len = spl->len * bps;
   int i;
   signed short s;
   PACKFILE *f;

   errno = 0;
   
   f = pack_fopen(filename, F_WRITE);

   if (f) {
      pack_fputs("RIFF", f);                 /* RIFF header */
      pack_iputl(36+len, f);                 /* size of RIFF chunk */
      pack_fputs("WAVE", f);                 /* WAV definition */
      pack_fputs("fmt ", f);                 /* format chunk */
      pack_iputl(16, f);                     /* size of format chunk */
      pack_iputw(1, f);                      /* PCM data */
      pack_iputw((spl->stereo) ? 2 : 1, f);  /* mono/stereo data */
      pack_iputl(spl->freq, f);              /* sample frequency */
      pack_iputl(spl->freq*bps, f);          /* avg. bytes per sec */
      pack_iputw(bps, f);                    /* block alignment */
      pack_iputw(spl->bits, f);              /* bits per sample */
      pack_fputs("data", f);                 /* data chunk */
      pack_iputl(len, f);                    /* actual data length */

      if (spl->bits == 8) {
	 pack_fwrite(spl->data, len, f);     /* write the data */
      }
      else {
	 for (i=0; i < (int)spl->len * ((spl->stereo) ? 2 : 1); i++) {
	    s = ((signed short *)spl->data)[i];
	    pack_iputw(s^0x8000, f);
	 }
      }

      pack_fclose(f);
   }

   return (errno == 0);
}



/* imports a sample from an external file */
static void *grab_sample(AL_CONST char *filename, long *size, int x, int y, int w, int h, int depth)
{
   return load_sample(filename);
}



/* saves a sample into the datafile format */
static int save_sample_in_datafile(DATAFILE *dat, AL_CONST int *fixed_prop, int pack, int pack_kids, int strip, int sort, int verbose, int extra, PACKFILE *f)
{
   SAMPLE *spl = (SAMPLE *)dat->dat;

   pack_mputw((spl->stereo) ? -spl->bits : spl->bits, f);
   pack_mputw(spl->freq, f);
   pack_mputl(spl->len, f);
   if (spl->bits == 8) {
      pack_fwrite(spl->data, spl->len * ((spl->stereo) ? 2 : 1), f);
   }
   else {
      int i;

      for (i=0; i < (int)spl->len * ((spl->stereo) ? 2 : 1); i++) {
	 pack_iputw(((unsigned short *)spl->data)[i], f);
      }
   }

   /* TODO: return FALSE on failure */
   return TRUE;
}



/* returns a description string for a GUS patch object */
static void get_patch_desc(AL_CONST DATAFILE *dat, char *s)
{
   sprintf(s, "MIDI instrument (%ld bytes)", dat->size);
}



/* plugin interface header */
DATEDIT_OBJECT_INFO datpatch_info =
{
   DAT_PATCH,
   "GUS patch",
   get_patch_desc,
   NULL,
   NULL,
   NULL,
   NULL,
   NULL
};



DATEDIT_GRABBER_INFO datpatch_grabber =
{
   DAT_PATCH,
   "pat",
   "pat",
   NULL,
   NULL
};



/* plugin interface header */
DATEDIT_OBJECT_INFO datsample_info =
{
   DAT_SAMPLE, 
   "Sample", 
   get_sample_desc,
   makenew_sample,
   save_sample_in_datafile,
   plot_sample,
   dclick_sample,
   NULL
};



DATEDIT_GRABBER_INFO datsample_grabber =
{
   DAT_SAMPLE, 
   "voc;wav",
   "wav",
   grab_sample,
   export_sample
};



/* what a pointless way to spend my Sunday afternoon :-) */
static int worms(void)
{
   #define MAX_SIZE     65536

   int *xpos = malloc(sizeof(int)*MAX_SIZE);
   int *ypos = malloc(sizeof(int)*MAX_SIZE);
   int head, tail, dir, dead, quit, score, counter;
   int tx, ty, x, y, i, c1, c2, c3;

   again:

   head = 0;
   tail = 0;
   dir = 0;
   dead = FALSE;
   quit = FALSE;
   score = 0;
   counter = 0;
   tx = -100;
   ty = -100;

//   show_mouse(NULL);

   if (bitmap_color_depth(DialogGuiScreen) <= 8) {
      PALETTE pal;

      get_palette(pal);

      pal[0].r = 0;
      pal[0].g = 0;
      pal[0].b = 0;

      pal[1].r = 31;
      pal[1].g = 31;
      pal[1].b = 31;

      pal[254].r = 31;
      pal[254].g = 31;
      pal[254].b = 31;

      pal[255].r = 63;
      pal[255].g = 63;
      pal[255].b = 63;

      set_palette(pal);

      c1 = 0;
      c2 = 1;
      c3 = 255;
   }
   else {
      c1 = gui_fg_color;
      c2 = gui_mg_color;
      c3 = gui_bg_color;
   }

//   clear_to_color(DialogGuiScreen, c1);

   xor_mode(TRUE);

   xpos[0] = my_SCREEN_W/2;
   ypos[0] = my_SCREEN_H/2;

   putpixel(DialogGuiScreen, xpos[0], ypos[0], c3);

   while( (!dead) && (!quit) ) {
      x = xpos[head];
      y = ypos[head];

      switch (dir) {
	 case 0: x++; break;
	 case 1: y++; break;
	 case 2: x--; break;
	 case 3: y--; break;
      }

      if (x >= my_SCREEN_W)
	 x -= my_SCREEN_W;
      else if (x < 0)
	 x += my_SCREEN_W;

      if (y >= my_SCREEN_H)
	 y -= my_SCREEN_H;
      else if (y < 0)
	 y += my_SCREEN_H;

      head++;
      if (head >= MAX_SIZE)
	 head = 0;

      xpos[head] = x;
      ypos[head] = y;

      counter++;

      if (counter & 15) {
	 putpixel(DialogGuiScreen, xpos[tail], ypos[tail], c3);

	 tail++;
	 if (tail >= MAX_SIZE)
	    tail = 0;
      }

      i = tail;

      while( i != head) {
	 if ((x == xpos[i]) && (y == ypos[i])) {
	    dead = TRUE;
	    break;
	 }

	i++;
	 if (i >= MAX_SIZE)
	    i = 0;
      }

      if (!(counter % (1+(score+2)/3)))
	 vsync();

	  putpixel(DialogGuiScreen, x, y, c3);

      if ((tx < 0) || (ty < 0)) {
	 do {
	    tx = 16+rand()%(my_SCREEN_W-32);
	    ty = 16+rand()%(my_SCREEN_H-32);
	 } while( (ABS(x-tx)+ABS(y-ty)) < 64);

	 circle(DialogGuiScreen, tx, ty, 8, c2);
	 circle(DialogGuiScreen, tx, ty, 5, c2);
	 circle(DialogGuiScreen, tx, ty, 2, c2);
      }

      if ((ABS(x-tx)+ABS(y-ty)) < 9) {
	 circle(DialogGuiScreen, tx, ty, 8, c2);
	 circle(DialogGuiScreen, tx, ty, 5, c2);
	 circle(DialogGuiScreen, tx, ty, 2, c2);

	 tx = -100;
	 ty = -100;

	 score++;
      }

	  textprintf_ex(DialogGuiScreen, font, 0, 0, c2, c1, "Score: %d", score);

      if (keypressed()) {
	 switch (readkey()>>8) {

	    case KEY_RIGHT:
	       dir = 0;
	       break;

	    case KEY_DOWN:
	       dir = 1;
	       break;

	    case KEY_LEFT:
	       dir = 2;
	       break;

	    case KEY_UP:
	       dir = 3;
	       break;

	    case KEY_ESC:
	       quit = TRUE;
	       break;
	 }
      }
   }

   xor_mode(FALSE);
//   clear_to_color(DialogGuiScreen, gui_fg_color);
   set_palette(datedit_current_palette);

   do {
      poll_keyboard();
   } while(  ((key[KEY_ESC]) || (key[KEY_RIGHT]) || (key[KEY_LEFT]) || (key[KEY_UP]) || (key[KEY_DOWN])));

   clear_keybuf();
//   show_mouse(DialogGuiScreen);

   if (dead) {
      char buf[64];
      sprintf(buf, "Score: %d", score);
//      if (Alert(buf, NULL, NULL, "Play", "Quit", 'p', 'q') == 1)
	 goto again;
   }

   free(xpos);
   free(ypos);

   return D_REDRAW;
}



/* hook ourselves into the grabber menu system */
static GUIMENU worms_menu =
{
   "Worms",
   worms,
   NULL,
   0,
   NULL,
   NULL
};



DATEDIT_GUIMENU_INFO datworms_menu =
{
   &worms_menu,
   NULL,
   DATEDIT_GUIMENU_HELP,
   0
};

PALETTE datedit_current_palette;
PALETTE datedit_last_read_pal;

static char info_msg[] = "For internal use by the grabber";

DATAFILE datedit_info = { info_msg, DAT_INFO, sizeof(info_msg), NULL };

static int file_datasize;

void (*grabber_sel_palette)() = NULL;
void (*grabber_select_property)() = NULL;
void (*grabber_get_grid_size)() = NULL;
void (*grabber_rebuild_list)() = NULL;
void (*grabber_get_selection_info)() = NULL;
int (*grabber_foreach_selection)() = NULL;
DATAFILE *(*grabber_single_selection)() = NULL;
void (*grabber_set_selection)() = NULL;
void (*grabber_busy_mouse)() = NULL;
void (*grabber_modified)() = NULL;

BITMAP *grabber_graphic = NULL;
PALETTE grabber_palette;

char grabber_import_file[256] = "";
char grabber_graphic_origin[256] = "";
char grabber_graphic_date[256] = "";



/* comparison function for sorting the menu list */
static int menu_cmp(GUIMENU *m1, GUIMENU *m2)
{
   if ((m1->child) && (!m2->child))
      return 1;
   else if ((m2->child) && (!m1->child))
      return -1;
   else
      return stricmp(m1->text, m2->text);
}



/* main initialisation routine */
void datedit_init(void)
{
   int done, i;

      #include "datedit/plugins.h"

   do {
      done = TRUE;

      for (i=0; datedit_object_info[i+1]->type != DAT_END; i++) {
	 if (stricmp(datedit_object_info[i]->desc, datedit_object_info[i+1]->desc) > 0) {
	    DATEDIT_OBJECT_INFO *tmp = datedit_object_info[i];
	    datedit_object_info[i] = datedit_object_info[i+1];
	    datedit_object_info[i+1] = tmp;
	    done = FALSE;
	 }
      }
   } while( !done);

   do {
      done = TRUE;

      for (i=0; (datedit_menu_info[i]) && (datedit_menu_info[i+1]); i++) {
	 if (menu_cmp(datedit_menu_info[i]->menu, datedit_menu_info[i+1]->menu) > 0) {
	    DATEDIT_GUIMENU_INFO *tmp = datedit_menu_info[i];
	    datedit_menu_info[i] = datedit_menu_info[i+1];
	    datedit_menu_info[i+1] = tmp;
	    done = FALSE;
	 }
      }
   } while( !done);
}



/* export raw binary data */
static int export_binary(AL_CONST DATAFILE *dat, AL_CONST char *filename)
{
   PACKFILE *f = pack_fopen(filename, F_WRITE);
   int ret = TRUE;

   if (f) {
      if (pack_fwrite(dat->dat, dat->size, f) < dat->size)
	 ret = FALSE;

      pack_fclose(f);
   }
   else
      ret = FALSE;

   return ret;
}



/* grab raw binary data */
static void *grab_binary(AL_CONST char *filename, long *size, int x, int y, int w, int h, int depth)
{
   void *mem;
   long sz = file_size(filename);
   PACKFILE *f;

   if (sz <= 0)
      return NULL;

   mem = malloc(sz);

   f = pack_fopen(filename, F_READ);
   if (!f) {
      free(mem);
      return NULL; 
   }

   if (pack_fread(mem, sz, f) < sz) {
      pack_fclose(f);
      free(mem);
      return NULL;
   }

   pack_fclose(f);

   *size = sz;
   return mem;
}



/* save raw binary data */
static int save_binary(DATAFILE *dat, AL_CONST int *fixed_prop, int pack, int pack_kids, int strip, int sort, int verbose, int extra, PACKFILE *f)
{
   if (pack_fwrite(dat->dat, dat->size, f) < dat->size)
      return FALSE;

   return TRUE;
}



/* export a child datafile */
static int export_datafile(AL_CONST DATAFILE *dat, AL_CONST char *filename)
{
   DATEDIT_SAVE_DATAFILE_OPTIONS options = {-1,    /* pack      */
					    -1,    /* strip     */
					    -1,    /* sort      */
					    TRUE, /* verbose   */
					    TRUE, /* write_msg */
					    FALSE  /* backup    */ };

   return datedit_save_datafile((DATAFILE *)dat->dat, filename, NULL, &options, NULL);
}



/* loads object names from a header file, if they are missing */
static void load_header(DATAFILE *dat, AL_CONST char *filename)
{
   char buf[160], buf2[160];
   int datsize, i, c, c2;
   PACKFILE *f;

   datsize = 0;
   while( dat[datsize].type != DAT_END)
      datsize++;

   f = pack_fopen(filename, F_READ);

   if (f) {
      while( pack_fgets(buf, 160, f) != 0)
      {
	 if (strncmp(buf, "#define ", 8) == 0) {
	    c2 = 0;
	    c = 8;

	    while( (buf[c]) && (buf[c] != ' '))
	       buf2[c2++] = buf[c++];

	    buf2[c2] = 0;
	    while( buf[c]==' ')
	       c++;

	    i = 0;
	    while( (buf[c] >= '0') && (buf[c] <= '9')) {
	       i *= 10;
	       i += buf[c] - '0';
	       c++;
	    }

	    if ((i < datsize) && (!*get_datafile_property(dat+i, DAT_NAME)))
	       datedit_set_property(dat+i, DAT_NAME, buf2);
	 }
      }

      pack_fclose(f);
   }
   else {
      /* don't let the error propagate */
      errno = 0;
   }
}



/* generates object names, if they are missing */
static int generate_names(DATAFILE *dat, int n)
{
   int i;

   while( dat->type != DAT_END) {
      if (!*get_datafile_property(dat, DAT_NAME)) {
	 char tmp[32];

	 sprintf(tmp, "%03d_%c%c%c%c", n++,
			(dat->type>>24)&0xFF, (dat->type>>16)&0xFF,
			(dat->type>>8)&0xFF, dat->type&0xFF);

	 for (i=4; tmp[i]; i++) {
	    if (((tmp[i] < '0') || (tmp[i] > '9')) &&
		((tmp[i] < 'A') || (tmp[i] > 'Z')) &&
		((tmp[i] < 'a') || (tmp[i] > 'z')))
	       tmp[i] = 0;
	 }

	 datedit_set_property(dat, DAT_NAME, tmp);
      }

      if (dat->type == DAT_FILE)
	 n = generate_names((DATAFILE *)dat->dat, n);

      dat++;
   }

   return n;
}



/* retrieves whatever grabber information is stored in a datafile */
static DATAFILE *extract_info(DATAFILE *dat, int save)
{
   DATAFILE_PROPERTY *prop;
   int i;

   if (save) {
      if (datedit_info.prop) {
	 prop = datedit_info.prop;
	 while( prop->type != DAT_END) {
	    if (prop->dat)
		   free(prop->dat);
	    prop++;
	 }
	 free(datedit_info.prop);
	 datedit_info.prop = NULL;
      }
   }

   for (i=0; dat[i].type != DAT_END; i++) {
      if (dat[i].type == DAT_INFO) {
	 if (save) {
	    prop = dat[i].prop;
	    while( (prop) && (prop->type != DAT_END)) {
	       datedit_set_property(&datedit_info, prop->type, prop->dat);
	       prop++;
	    }
	 }

	 dat = datedit_delete(dat, i);
	 i--;
      }
   }

   return dat;
}



/* grabs a child datafile */
static void *grab_datafile(AL_CONST char *filename, long *size, int x, int y, int w, int h, int depth)
{
   DATAFILE *dat = load_datafile(filename);

   if (dat) {
      load_header(dat, datedit_pretty_name(filename, "h", TRUE));
      generate_names(dat, 0);
      dat = extract_info(dat, FALSE);
   }

   return dat;
}



/* queries whether this property belongs in the current strip mode */
static int should_save_prop(int type, AL_CONST int *fixed_prop, int strip)
{
   if (strip == 0)
      return TRUE;

   if (fixed_prop) {
      while( *fixed_prop) {
	 if (type == *fixed_prop++)
	    return TRUE;
      }
   }

   if (strip >= 2) {
      return FALSE;
   }
   else {
      return ((type != DAT_ORIG) &&
	      (type != DAT_DATE) &&
	      (type != DAT_XPOS) && 
	      (type != DAT_YPOS) &&
	      (type != DAT_XSIZ) && 
	      (type != DAT_YSIZ) &&
              (type != DAT_XCRP) &&
              (type != DAT_YCRP));
   }
}



/* wrapper to avoid division by zero when files are empty */
/*static int percent(int a, int b)
{
   if (a)
      return (b * 100) / a;
   else
      return 0;
} */



/* saves an object */
static int save_object(DATAFILE *dat, AL_CONST int *fixed_prop, int pack, int pack_kids, int strip, int sort, int verbose, PACKFILE *f)
{
   int i, ret;
   DATAFILE_PROPERTY *prop;
   int (*save)(DATAFILE *, AL_CONST int *, int, int, int, int, int, int, PACKFILE *);

   prop = dat->prop;
   datedit_sort_properties(prop);

   while( (prop) && (prop->type != DAT_END) )
   {
      if (should_save_prop(prop->type, fixed_prop, strip))
      {
	 pack_mputl(DAT_PROPERTY, f);
	 pack_mputl(prop->type, f);
	 pack_mputl(strlen(prop->dat), f);
	 if (pack_fwrite(prop->dat, strlen(prop->dat), f) < (signed)strlen(prop->dat))
	    return FALSE;
	 file_datasize += 12 + strlen(prop->dat);
      }

      prop++;
   }

//   if (verbose)
//      my_datedit_startmsg("%-28s", get_datafile_property(dat, DAT_NAME));

   pack_mputl(dat->type, f);
   f = pack_fopen_chunk(f, ((!pack) && (pack_kids) && (dat->type != DAT_FILE)));
   file_datasize += 12;

   save = NULL;

   for (i=0; datedit_object_info[i]->type != DAT_END; i++) {
      if (datedit_object_info[i]->type == dat->type) {
	 save = datedit_object_info[i]->save;
	 break;
      }
   }

   if (!save)
      save = save_binary;

   if (dat->type == DAT_FILE) {
//      if (verbose)
//	 my_datedit_endmsg("");

      ret = save((DATAFILE *)dat->dat, fixed_prop, pack, pack_kids, strip, sort, verbose, FALSE, f);

//      if (verbose)
//	 my_datedit_startmsg("End of %-21s", get_datafile_property(dat, DAT_NAME));
   }
   else
      ret = save(dat, fixed_prop, (pack || pack_kids), FALSE, strip, sort, verbose, FALSE, f);

   pack_fclose_chunk(f);

   if (verbose) {
      if ((!pack) && (pack_kids) && (dat->type != DAT_FILE)) {
//	 my_datedit_endmsg("%7d bytes into %-7d (%d%%)",
//			_packfile_datasize, _packfile_filesize,
//			percent(_packfile_datasize, _packfile_filesize));
      }
//      else
//	 my_datedit_endmsg("");
   }

   if (dat->type == DAT_FILE)
      file_datasize += 4;
   else
      file_datasize += _packfile_datasize;

   return ret;
}



/* saves a datafile */
static int save_datafile(DATAFILE *dat, AL_CONST int *fixed_prop, int pack, int pack_kids, int strip, int sort, int verbose, int extra, PACKFILE *f)
{
   int c, size;

   if (sort)
      datedit_sort_datafile(dat);

   size = 0;
   while( dat[size].type != DAT_END)
      size++;

   pack_mputl(extra ? size+1 : size, f);

   for (c=0; c<size; c++) {
      if (!save_object(dat+c, fixed_prop, pack, pack_kids, strip, sort, verbose, f))
	 return FALSE;
   }

   return TRUE;
}



/* creates a new datafile */
static void *makenew_file(long *size)
{
   DATAFILE *dat = malloc(sizeof(DATAFILE));

   dat->dat = NULL;
   dat->type = DAT_END;
   dat->size = 0;
   dat->prop = NULL;

   return dat;
}



/* header block for datafile objects */
DATEDIT_OBJECT_INFO datfile_info =
{
   DAT_FILE, 
   "Datafile", 
   NULL, 
   makenew_file,
   save_datafile,
   NULL,
   NULL,
   NULL
};



DATEDIT_GRABBER_INFO datfile_grabber =
{
   DAT_FILE, 
   "dat",
   "dat",
   grab_datafile,
   export_datafile
};



/* dummy header block to use as a terminator */
DATEDIT_OBJECT_INFO datend_info =
{
   DAT_END,
   NULL,
   NULL,
   NULL,
   NULL,
   NULL,
   NULL,
   NULL
};



DATEDIT_GRABBER_INFO datend_grabber =
{
   DAT_END,
   NULL,
   NULL,
   NULL,
   NULL
};



/* list of available object types */
DATEDIT_OBJECT_INFO *datedit_object_info[32] =
{
   &datfile_info, &datend_info,  &datend_info,  &datend_info,
   &datend_info,  &datend_info,  &datend_info,  &datend_info,
   &datend_info,  &datend_info,  &datend_info,  &datend_info,
   &datend_info,  &datend_info,  &datend_info,  &datend_info,
   &datend_info,  &datend_info,  &datend_info,  &datend_info,
   &datend_info,  &datend_info,  &datend_info,  &datend_info,
   &datend_info,  &datend_info,  &datend_info,  &datend_info,
   &datend_info,  &datend_info,  &datend_info,  &datend_info
};



/* list of available object grabber routines */
DATEDIT_GRABBER_INFO *datedit_grabber_info[32] =
{
   &datfile_grabber, &datend_grabber,  &datend_grabber,  &datend_grabber,
   &datend_grabber,  &datend_grabber,  &datend_grabber,  &datend_grabber,
   &datend_grabber,  &datend_grabber,  &datend_grabber,  &datend_grabber,
   &datend_grabber,  &datend_grabber,  &datend_grabber,  &datend_grabber,
   &datend_grabber,  &datend_grabber,  &datend_grabber,  &datend_grabber,
   &datend_grabber,  &datend_grabber,  &datend_grabber,  &datend_grabber,
   &datend_grabber,  &datend_grabber,  &datend_grabber,  &datend_grabber,
   &datend_grabber,  &datend_grabber,  &datend_grabber,  &datend_grabber
};



/* list of active menu hooks */
DATEDIT_GUIMENU_INFO *datedit_menu_info[32] =
{
   NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
   NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
   NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
   NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL
};



/* adds a new object to the list of available methods */
void datedit_register_object(DATEDIT_OBJECT_INFO *info)
{
   int i = 0;

   while( datedit_object_info[i]->type != DAT_END)
      i++;

   datedit_object_info[i] = info;
}



/* adds a new grabber to the list of available methods */
void datedit_register_grabber(DATEDIT_GRABBER_INFO *info)
{
   int i = 0;

   while( datedit_grabber_info[i]->type != DAT_END)
      i++;

   datedit_grabber_info[i] = info;
}



/* adds a new entry to the list of active menus */
void datedit_register_menu(DATEDIT_GUIMENU_INFO *info)
{
   int i = 0;

   while( datedit_menu_info[i])
      i++;

   datedit_menu_info[i] = info;
}



/* adds extensions to filenames if they are missing, or changes them */
char *datedit_pretty_name(AL_CONST char *name, AL_CONST char *ext, int force_ext)
{
   static char buf[256];
   char *s;

   strcpy(buf, name);

   s = get_extension(buf);
   if ((s > buf) && (*(s-1)=='.')) {
      if (force_ext)
	 strcpy(s, ext);
   }
   else {
      *s = '.';
      strcpy(s+1, ext);
   }

   fix_filename_case(buf);
   return buf;
}



/* returns a description string for an object */
AL_CONST char *datedit_desc(AL_CONST DATAFILE *dat)
{
   static char buf[256];
   int i;

   for (i=0; datedit_object_info[i]->type != DAT_END; i++) {
      if (datedit_object_info[i]->type == dat->type) {
	 if (datedit_object_info[i]->get_desc)
	    datedit_object_info[i]->get_desc(dat, buf);
	 else
	    strcpy(buf, datedit_object_info[i]->desc);

	 return buf;
      }
   }

   sprintf(buf, "binary data (%ld bytes)", dat->size);
   return buf;
}



/* qsort callback for comparing datafile objects */
static int dat_cmp(AL_CONST void *e1, AL_CONST void *e2)
{
   DATAFILE *d1 = (DATAFILE *)e1;
   DATAFILE *d2 = (DATAFILE *)e2;

   return stricmp(get_datafile_property((AL_CONST DATAFILE*)d1, DAT_NAME), get_datafile_property((AL_CONST DATAFILE*)d2, DAT_NAME));
}



/* sorts a datafile */
void datedit_sort_datafile(DATAFILE *dat)
{
   int len;

   if (dat) {
      len = 0;
      while( dat[len].type != DAT_END)
	 len++;

      if (len > 1)
	 qsort(dat, len, sizeof(DATAFILE), dat_cmp);
   }
}



/* qsort callback for comparing datafile properties */
static int prop_cmp(AL_CONST void *e1, AL_CONST void *e2)
{
   DATAFILE_PROPERTY *p1 = (DATAFILE_PROPERTY *)e1;
   DATAFILE_PROPERTY *p2 = (DATAFILE_PROPERTY *)e2;

   return p1->type - p2->type;
}



/* sorts a list of object properties */
void datedit_sort_properties(DATAFILE_PROPERTY *prop)
{
   int len;

   if (prop) {
      len = 0;
      while( prop[len].type != DAT_END)
	 len++;

      if (len > 1)
	 qsort(prop, len, sizeof(DATAFILE_PROPERTY), prop_cmp);
   }
}



/* splits bitmaps into sub-sprites, using regions bounded by col #255 */
void datedit_find_character(BITMAP *bmp, int *x, int *y, int *w, int *h)
{
   int c1;
   int c2;

   if (bitmap_color_depth(bmp) == 8) {
      c1 = 255;
      c2 = 255;
   }
   else {
      c1 = makecol_depth(bitmap_color_depth(bmp), 255, 255, 0);
      c2 = makecol_depth(bitmap_color_depth(bmp), 0, 255, 255);
   }

   /* look for top left corner of character */
   while(  ((getpixel(bmp, *x, *y) != c1) || (getpixel(bmp, *x+1, *y) != c2) || (getpixel(bmp, *x, *y+1) != c2) || (getpixel(bmp, *x+1, *y+1) == c1) || (getpixel(bmp, *x+1, *y+1) == c2)))
   {
      (*x)++;
      if (*x >= bmp->w) {
	 *x = 0;
	 (*y)++;
	 if (*y >= bmp->h) {
	    *w = 0;
	    *h = 0;
	    return;
	 }
      }
   }

   /* look for right edge of character */
   *w = 0;
   while( (getpixel(bmp, *x+*w+1, *y) == c2) &&
	  (getpixel(bmp, *x+*w+1, *y+1) != c2) &&
	  (*x+*w+1 <= bmp->w))
      (*w)++;

   /* look for bottom edge of character */
   *h = 0;
   while( (getpixel(bmp, *x, *y+*h+1) == c2) &&
	  (getpixel(bmp, *x+1, *y+*h+1) != c2) &&
	  (*y+*h+1 <= bmp->h))
      (*h)++;
}



/* cleans up an object type string, and packs it */
int datedit_clean_typename(AL_CONST char *type)
{
   int c1, c2, c3, c4;

   if (!type)
      return 0;

   c1 = (*type) ? utoupper(*(type++)) : ' ';
   c2 = (*type) ? utoupper(*(type++)) : ' ';
   c3 = (*type) ? utoupper(*(type++)) : ' ';
   c4 = (*type) ? utoupper(*(type++)) : ' ';

   return DAT_ID(c1, c2, c3, c4);
}



/* sets an object property string */
void datedit_set_property(DATAFILE *dat, int type, AL_CONST char *value)
{
   int i, size, pos;

   if (dat->prop) {
      pos = -1;
      for (size=0; dat->prop[size].type != DAT_END; size++)
	 if (dat->prop[size].type == type)
	    pos = size;

      if ((value) && (strlen(value) > 0)) {
	 if (pos >= 0) {
		dat->prop[pos].dat = realloc(dat->prop[pos].dat, strlen(value)+1);
	    strcpy(dat->prop[pos].dat, value);
	 }
	 else {
		dat->prop = realloc(dat->prop, sizeof(DATAFILE_PROPERTY)*(size+2));
	    dat->prop[size+1] = dat->prop[size];
	    dat->prop[size].type = type;
		dat->prop[size].dat = malloc(strlen(value)+1);
	    strcpy(dat->prop[size].dat, value);
	 }
      }
      else {
	 if (pos >= 0) {
		free(dat->prop[pos].dat);
	    for (i=pos; i<size; i++)
	       dat->prop[i] = dat->prop[i+1];
		dat->prop = realloc(dat->prop, sizeof(DATAFILE_PROPERTY)*size);
	 }
      }
   }
   else {
      if ((value) && (strlen(value) > 0)) {
	 dat->prop = malloc(sizeof(DATAFILE_PROPERTY) * 2);
	 dat->prop[0].type = type;
	 dat->prop[0].dat = malloc(strlen(value)+1);
	 strcpy(dat->prop[0].dat, value);
	 dat->prop[1].type = DAT_END;
	 dat->prop[1].dat = NULL;
      }
   }
}



/* loads a datafile */
DATAFILE *datedit_load_datafile(AL_CONST char *name, int compile_sprites, AL_CONST char *password)
{
   char *pretty_name;
   DATAFILE *datafile;

   _compile_sprites = compile_sprites;

   if (!compile_sprites) {
      typedef void (*DF)(void *);
      register_datafile_object(DAT_C_SPRITE, NULL, (void (*)(void *))destroy_bitmap);
      register_datafile_object(DAT_XC_SPRITE, NULL, (void (*)(void *))destroy_bitmap);
   }

   if (name)
      pretty_name = datedit_pretty_name(name, "dat", FALSE);
   else
      pretty_name = NULL;

   if ((pretty_name) && (exists(pretty_name))) {
//      my_datedit_msg("Reading %s", pretty_name);

      packfile_password(password);

      datafile = load_datafile(pretty_name);

      packfile_password(NULL);

      if (!datafile) {
//	 my_datedit_error("Error reading %s", pretty_name);
	 return NULL;
      }
      else {
	 load_header(datafile, datedit_pretty_name(name, "h", TRUE));
	 generate_names(datafile, 0);
      }
   }
   else {
//      if (pretty_name)
//	 my_datedit_msg("%s not found: creating new datafile", pretty_name);

	  datafile = malloc(sizeof(DATAFILE));
      datafile->dat = NULL;
      datafile->type = DAT_END;
      datafile->size = 0;
      datafile->prop = NULL;
   }

   datafile = extract_info(datafile, TRUE);

   return datafile;
}



/* works out what name to give an exported object */
void datedit_export_name(AL_CONST DATAFILE *dat, AL_CONST char *name, AL_CONST char *ext, char *buf)
{
   AL_CONST char *obname = get_datafile_property(dat, DAT_NAME);
   AL_CONST char *oborig = get_datafile_property(dat, DAT_ORIG);
   char tmp[32];
   int i;

   if (name)
      strcpy(buf, name);
   else
      strcpy(buf, oborig);

   if (*get_filename(buf) == 0) {
      if (*oborig) {
	 strcat(buf, get_filename(oborig));
      }
      else {
	 strcat(buf, obname);
	 if (!ALLEGRO_LFN)
	    get_filename(buf)[8] = 0;
      }
   }

   if (ext) {
      strcpy(tmp, ext);
      for (i=0; tmp[i]; i++) {
	 if (tmp[i] == ';') {
	    tmp[i] = 0;
	    break;
	 }
      }
      strcpy(buf, datedit_pretty_name(buf, tmp, ((name == NULL) || (!*get_extension(name)))));
   }
   else
      fix_filename_case(buf);
}



/* exports a datafile object */
int datedit_export(AL_CONST DATAFILE *dat, AL_CONST char *name)
{
//   AL_CONST char *obname = get_datafile_property(dat, DAT_NAME);
   int (*export)(AL_CONST DATAFILE *dat, AL_CONST char *filename) = NULL;
   char buf[256], tmp[256];
   char *ext = NULL;
   char *tok;
   int i;

   for (i=0; datedit_grabber_info[i]->type != DAT_END; i++) {
      if ((datedit_grabber_info[i]->type == dat->type) && (datedit_grabber_info[i]->export_ext)) {
	 ext = datedit_grabber_info[i]->export_ext;
	 break;
      }
   }

   datedit_export_name(dat, name, ext, buf);

/*   if (exists(buf)) {
      i = my_datedit_ask("%s already exists, overwrite", buf);
      if (i == 27)
	 return FALSE;
      else if ((i == 'n') || (i == 'N'))
	 return TRUE;
   }*/

//   my_datedit_msg("Exporting %s -> %s", obname, buf);

   ext = get_extension(buf);

   for (i=0; datedit_grabber_info[i]->type != DAT_END; i++) {
      if ((datedit_grabber_info[i]->type == dat->type) && (datedit_grabber_info[i]->export_ext) && (datedit_grabber_info[i]->export)) {
	 strcpy(tmp, datedit_grabber_info[i]->export_ext);
	 tok = strtok(tmp, ";");
	 while( tok) {
	    if (stricmp(tok, ext) == 0) {
	       export = datedit_grabber_info[i]->export;
	       break;
	    }
	    tok = strtok(NULL, ";");
	 }
      }
   }

   if (!export)
      export = export_binary;

   if (!export(dat, buf)) {
      delete_file(buf);
//      my_datedit_error("Error writing %s", buf);
      return FALSE;
   }

   return TRUE;
}



/* deletes a datafile object */
DATAFILE *datedit_delete(DATAFILE *dat, int i)
{
   _unload_datafile_object(dat+i);

   do {
      dat[i] = dat[i+1];
      i++;
   } while( dat[i].type != DAT_END);

   return realloc(dat, sizeof(DATAFILE)*i);
}



/* fixup function for the strip options */
int datedit_striptype(int strip)
{
   if (strip >= 0)
      return strip;
   else
      return 0;
}



/* fixup function for the pack options */
int datedit_packtype(int pack)
{
   if (pack >= 0) {
      char buf[80];

      sprintf(buf, "%d", pack);
      datedit_set_property(&datedit_info, DAT_PACK, buf);

      return pack;
   }
   else {
      AL_CONST char *p = get_datafile_property(&datedit_info, DAT_PACK);

      if (*p)
	 return atoi(p);
      else
	 return 0;
   }
}



/* fixup function for the sort options */
int datedit_sorttype(int sort)
{
   if (sort >= 0) {
      sort = (sort == 1 ? TRUE : FALSE);
      datedit_set_property(&datedit_info, DAT_SORT, sort ? "y" : "n");

      return sort;
   }
   else {
      AL_CONST char *p = get_datafile_property(&datedit_info, DAT_SORT);

      if (*p)
	 return (utolower(*p)=='y');
      else
	 return TRUE;  /* sort if SORT property not present */
   }
}



/* saves a datafile */
int datedit_save_datafile(DATAFILE *dat, AL_CONST char *name, AL_CONST int *fixed_prop, AL_CONST DATEDIT_SAVE_DATAFILE_OPTIONS *options, AL_CONST char *password)
{
   char *pretty_name;
   char backup_name[256];
   int pack, strip, sort;
   PACKFILE *f;
   int ret;

   packfile_password(password);

   pack = datedit_packtype(options->pack);
   strip = datedit_striptype(options->strip);
   sort = datedit_sorttype(options->sort);

   strcpy(backup_name, datedit_pretty_name(name, "bak", TRUE));
   pretty_name = datedit_pretty_name(name, "dat", FALSE);

//   if (options->write_msg)
//      my_datedit_msg("Writing %s", pretty_name);

   delete_file(backup_name);
   rename(pretty_name, backup_name);

   f = pack_fopen(pretty_name, (pack >= 2) ? F_WRITE_PACKED : F_WRITE_NOPACK);

   if (f) {
      pack_mputl(DAT_MAGIC, f);
      file_datasize = 12;

      ret = save_datafile(dat, fixed_prop, (pack >= 2), (pack >= 1), strip, sort, options->verbose, (strip <= 0), f);

      if ((ret == TRUE) && (strip <= 0)) {
	 datedit_set_property(&datedit_info, DAT_NAME, "GrabberInfo");
	 ret = save_object(&datedit_info, NULL, FALSE, FALSE, FALSE, FALSE, FALSE, f);
      }

      pack_fclose(f); 
   }
   else
      ret = FALSE;

   if (ret == FALSE) {
      delete_file(pretty_name);
//      my_datedit_error("Error writing %s", pretty_name);
      packfile_password(NULL);
      return FALSE;
   }
   
   if (!options->backup)
      delete_file(backup_name);

   if (options->verbose) {
//      int file_filesize = file_size(pretty_name);
//      my_datedit_msg("%-28s%7d bytes into %-7d (%d%%)", "- GLOBAL COMPRESSION -",
//		  file_datasize, file_filesize, percent(file_datasize, file_filesize));
   }

   packfile_password(NULL);
   return TRUE;
}



/* writes object definitions into a header file */
static void save_header(AL_CONST DATAFILE *dat, FILE *f, AL_CONST char *prefix)
{
   int c;

   fprintf(f, "\n");

   for (c=0; dat[c].type != DAT_END; c++) {
      fprintf(f, "#define %s%-*s %-8d /* %c%c%c%c */\n", 
	      prefix, 32-(int)strlen(prefix),
	      get_datafile_property(dat+c, DAT_NAME), c,
	      (dat[c].type>>24), (dat[c].type>>16)&0xFF, 
	      (dat[c].type>>8)&0xFF, (dat[c].type&0xFF));

      if (dat[c].type == DAT_FILE) {
	 char p[256];

	 strcpy(p, prefix);
	 strcat(p, get_datafile_property(dat+c, DAT_NAME));
	 strcat(p, "_");

	 save_header((DATAFILE *)dat[c].dat, f, p);
      }
   }

   if (*prefix)
      fprintf(f, "#define %s%-*s %d\n", prefix, 32-(int)strlen(prefix), "COUNT", c);

   fprintf(f, "\n");
}



/* helper for renaming files (works across drives) */
static int rename_file(AL_CONST char *oldname, AL_CONST char *newname)
{
   PACKFILE *oldfile, *newfile;
   int c;

   oldfile = pack_fopen(oldname, F_READ);
   if (!oldfile)
      return -1;

   newfile = pack_fopen(newname, F_WRITE);
   if (!newfile) {
      pack_fclose(oldfile);
      return -1;
   }

   c = pack_getc(oldfile);

   while( c != EOF) {
      pack_putc(c, newfile);
      c = pack_getc(oldfile);
   } 

   pack_fclose(oldfile);
   pack_fclose(newfile);

   delete_file(oldname); 

   return 0;
}



/* checks whether the header needs updating */
static int cond_update_header(AL_CONST char *tn, AL_CONST char *n, int verbose)
{
   PACKFILE *f1, *f2;
   char b1[256], b2[256];
   int i;
   int differ = FALSE;

   if (!exists(n)) {
      if (rename_file(tn, n) != 0)
	 return FALSE;
   }
   else {
      f1 = pack_fopen(tn, F_READ);
      if (!f1)
	 return FALSE;

      f2 = pack_fopen(n, F_READ);
      if (!f2) {
	 pack_fclose(f1);
	 return FALSE;
      }

      for (i=0; i<4; i++) {
	 /* skip date, which may differ */
	 pack_fgets(b1, 255, f1);
	 pack_fgets(b2, 255, f2);
      }

      while( (!pack_feof(f1)) && (!pack_feof(f2)) && (!differ)) {
	 pack_fgets(b1, 255, f1);
	 pack_fgets(b2, 255, f2);
	 if (strcmp(b1, b2) != 0)
	    differ = TRUE;
      }

      if ((!pack_feof(f1)) || (!pack_feof(f2)))
	 differ = TRUE;

      pack_fclose(f1);
      pack_fclose(f2);

      if (differ) {
//	 if (verbose)
//	    my_datedit_msg("%s has changed: updating", n);

	 delete_file(n);
	 rename_file(tn, n);
      }
      else {
//	 if (verbose)
//	    my_datedit_msg("%s has not changed: no update", n);

	 delete_file(tn);
      }
   }

   return TRUE;
}



/* exports a datafile header */
int datedit_save_header(AL_CONST DATAFILE *dat, AL_CONST char *name, AL_CONST char *headername, AL_CONST char *progname, AL_CONST char *prefix, int verbose)
{
   char *pretty_name, *tmp_name;
   char tm[80];
   char p[80];
   time_t now;
   FILE *f;
   int c;

   #ifdef HAVE_MKSTEMP

      char tmp_buf[] = "XXXXXX";
      char tmp[512];
      int tmp_fd;

      tmp_fd = mkstemp(tmp_buf);
      if (tmp_fd < 0) {
//	 my_datedit_error("Error creating temporary file");
	 return FALSE;
      }
      close(tmp_fd);
      tmp_name = uconvert_ascii(tmp_buf, tmp);

   #else

      tmp_name = tmpnam(NULL);

   #endif

   if (prefix)
      strcpy(p, prefix);
   else
      strcpy(p, get_datafile_property(&datedit_info, DAT_HPRE));

   if ((p[0]) && (p[(strlen(p)-1)] != '_'))
      strcat(p, "_");

   pretty_name = datedit_pretty_name(headername, "h", FALSE);
//   my_datedit_msg("Writing ID's into %s", pretty_name);

   f = fopen(tmp_name, "w");
   if (f) {
      time(&now);
      strcpy(tm, asctime(localtime(&now)));
      for (c=0; tm[c]; c++)
	 if ((tm[c] == '\r') || (tm[c] == '\n'))
	    tm[c] = 0;

      fprintf(f, "/* Allegro datafile object indexes, produced by %s v" ALLEGRO_VERSION_STR ", " ALLEGRO_PLATFORM_STR " */\n", progname);
      fprintf(f, "/* Datafile: %s */\n", name);
      fprintf(f, "/* Date: %s */\n", tm);
      fprintf(f, "/* Do not hand edit! */\n");

      save_header(dat, f, p);

      fclose(f);

      if (!cond_update_header(tmp_name, pretty_name, verbose)) {
//	 my_datedit_error("Error writing %s", pretty_name);
	 return FALSE;
      }
   }
   else {
//      my_datedit_error("Error writing %s", pretty_name);
      return FALSE;
   }

   return TRUE;
}



/* converts a file timestamp from ASCII to integer representation */
long datedit_asc2ftime(AL_CONST char *time)
{
   static char *sep = "-,: ";
   char tmp[256], *tok;
   struct tm t;

   memset(&t, 0, sizeof(struct tm));

   strcpy(tmp, time);
   tok = strtok(tmp, sep);

   if (tok) {
      t.tm_mon = atoi(tok)-1;
      tok = strtok(NULL, sep);
      if (tok) {
	 t.tm_mday = atoi(tok);
	 tok = strtok(NULL, sep);
	 if (tok) {
	    t.tm_year = atoi(tok)-1900;
	    tok = strtok(NULL, sep);
	    if (tok) {
	       t.tm_hour = atoi(tok);
	       tok = strtok(NULL, sep);
	       if (tok) {
		  t.tm_min = atoi(tok);
	       }
	    }
	 }
      }
   }

   {
      /* make timezone adjustments by converting to time_t with adjustment 
       * from local time, then back again as GMT (=UTC) */
      time_t tm = mktime (&t);
      if (tm != (time_t)-1) {               /* cast needed in djgpp */
	 struct tm *temp = gmtime (&tm);
	 if (temp) memcpy (&t, temp, sizeof t);
      }
   }

   return mktime(&t);
}



/* converts a file timestamp from integer to ASCII representation */
AL_CONST char *datedit_ftime2asc(long time)
{
   static char buf[80];

   time_t tim = time;
   struct tm *t = gmtime(&tim);

   sprintf(buf, "%d-%02d-%d, %d:%02d",
		t->tm_mon+1, t->tm_mday, t->tm_year+1900,
		t->tm_hour, t->tm_min);

   return buf;
}



/* converts a file timestamp to international ASCII representation */
AL_CONST char *datedit_ftime2asc_int(long time)
{
   static char month[12][4] =
   {
       "JAN", "FEB", "MAR", "APR", "MAY", "JUN",
       "JUL", "AUG", "SEP", "OCT", "NOV", "DEC"
   };

   static char buf[80];

   time_t tim = time;
   struct tm *t = gmtime(&tim);

   sprintf(buf, "%.3s-%02d-%d, %d:%02d",
		month[t->tm_mon%12], t->tm_mday, t->tm_year+1900,
		t->tm_hour, t->tm_min);

   return buf;
}



/* grabs an object from a disk file */
DATAFILE *datedit_grab(AL_CONST char *filename, AL_CONST char *name, int type, int x, int y, int w, int h, int colordepth)
{
   static DATAFILE dat;
   void *(*grab)(AL_CONST char *filename, long *size, int x, int y, int w, int h, int depth) = NULL;
   char *ext = get_extension(filename);
   char *tok;
   char tmp[256];
   int c;

   dat.dat = NULL;
   dat.size = 0;
   dat.prop = NULL;

   if (type) {
      dat.type = type;
   }
   else {
      dat.type = DAT_DATA;

      if ((ext) && (*ext)) {
	 for (c=0; datedit_grabber_info[c]->type != DAT_END; c++) {
	    if (datedit_grabber_info[c]->grab_ext) {
	       strcpy(tmp, datedit_grabber_info[c]->grab_ext);
	       tok = strtok(tmp, ";");
	       while( tok) {
		  if (stricmp(tok, ext) == 0) {
		     dat.type = datedit_grabber_info[c]->type;
		     goto found_type;
		  }
		  tok = strtok(NULL, ";");
	       }
	    }
	 }
      }
   }

   found_type:

   for (c=0; datedit_grabber_info[c]->type != DAT_END; c++) {
      if ((datedit_grabber_info[c]->type == dat.type) && (datedit_grabber_info[c]->grab_ext) && (datedit_grabber_info[c]->grab)) {
	 if ((ext) && (*ext)) {
	    strcpy(tmp, datedit_grabber_info[c]->grab_ext);
	    tok = strtok(tmp, ";");
	    while( tok) {
	       if (stricmp(tok, ext) == 0) {
		  grab = datedit_grabber_info[c]->grab;
		  goto found_grabber;
	       }
	       tok = strtok(NULL, ";");
	    }
	 }
	 if (!grab)
	    grab = datedit_grabber_info[c]->grab;
      }
   }

   if (!grab)
      grab = grab_binary;

   found_grabber:

   dat.dat = grab(filename, &dat.size, x, y, w, h, colordepth);

   if (dat.dat) {
      datedit_set_property(&dat, DAT_NAME, name);
      datedit_set_property(&dat, DAT_ORIG, filename);
      datedit_set_property(&dat, DAT_DATE, datedit_ftime2asc(file_time(filename)));
   }
   else {
//      my_datedit_error("Error reading %s as type %c%c%c%c", filename,
//		    dat.type>>24, (dat.type>>16)&0xFF,
//		    (dat.type>>8)&0xFF, dat.type&0xFF);
   }

   return &dat;
}



/* grabs an object over the top of an existing one */
int datedit_grabreplace(DATAFILE *dat, AL_CONST char *filename, AL_CONST char *name, AL_CONST char *type, int colordepth, int x, int y, int w, int h)
{
   DATAFILE *tmp = datedit_grab(filename, name, 
				datedit_clean_typename(type), 
				x, y, w, h, colordepth);

   if (tmp->dat) {
      _unload_datafile_object(dat);
      *dat = *tmp;
      return TRUE;
   }
   else
      return FALSE;
}



/* updates an object in-place */
int datedit_grabupdate(DATAFILE *dat, AL_CONST char *filename, int x, int y, int w, int h)
{
   DATAFILE *tmp = datedit_grab(filename, "dummyname", dat->type, x, y, w, h, -1);
   DATAFILE_PROPERTY *tmp_prop;

   if (tmp->dat) {
      /* exchange properties */
      tmp_prop = tmp->prop;
      tmp->prop = dat->prop;
      dat->prop = tmp_prop;

      datedit_set_property(tmp, DAT_DATE, get_datafile_property(dat, DAT_DATE));

      /* adjust color depth? */
      if ((dat->type == DAT_BITMAP) || (dat->type == DAT_RLE_SPRITE) ||
	  (dat->type == DAT_C_SPRITE) || (dat->type == DAT_XC_SPRITE)) {

	 int src_depth, dest_depth;

	 if (dat->type == DAT_RLE_SPRITE) {
	    dest_depth = ((RLE_SPRITE *)dat->dat)->color_depth;
	    src_depth = ((RLE_SPRITE *)tmp->dat)->color_depth;
	 }
	 else {
	    dest_depth = bitmap_color_depth(dat->dat);
	    src_depth = bitmap_color_depth(tmp->dat);
	 }

	 if (src_depth != dest_depth) {
	    BITMAP *b1, *b2;

	    if (dat->type == DAT_RLE_SPRITE) {
	       RLE_SPRITE *spr = (RLE_SPRITE *)tmp->dat;
	       b1 = create_bitmap_ex(src_depth, spr->w, spr->h);
	       clear_to_color(b1, b1->vtable->mask_color);
	       draw_rle_sprite(b1, spr, 0, 0);
	       destroy_rle_sprite(spr);
	    }
	    else
	       b1 = (BITMAP *)tmp->dat;

//	    if (dest_depth == 8)
//	       my_datedit_msg("Warning: lossy conversion from truecolor to 256 colors!");

	    if ((dat->type == DAT_RLE_SPRITE) ||
		(dat->type == DAT_C_SPRITE) || (dat->type == DAT_XC_SPRITE)) {
	       datedit_last_read_pal[0].r = 63;
	       datedit_last_read_pal[0].g = 0;
	       datedit_last_read_pal[0].b = 63;
	    }

	    select_palette(datedit_last_read_pal);
	    b2 = create_bitmap_ex(dest_depth, b1->w, b1->h);
	    blit(b1, b2, 0, 0, 0, 0, b1->w, b1->h);
	    unselect_palette();

	    if (dat->type == DAT_RLE_SPRITE) {
	       tmp->dat = get_rle_sprite(b2);
	       destroy_bitmap(b1);
	       destroy_bitmap(b2);
	    }
	    else {
	       tmp->dat = b2;
	       destroy_bitmap(b1);
	    }
	 }
      }

      _unload_datafile_object(dat);
      *dat = *tmp;
      return TRUE;
   }
   else
      return FALSE;
}



/* grabs a new object, inserting it into the datafile */
DATAFILE *datedit_grabnew(DATAFILE *dat, AL_CONST char *filename, AL_CONST char *name, AL_CONST char *type, int colordepth, int x, int y, int w, int h)
{
   DATAFILE *tmp = datedit_grab(filename, name, 
				datedit_clean_typename(type), 
				x, y, w, h, colordepth);
   int len;

   if (tmp->dat) {
      len = 0;
      while( dat[len].type != DAT_END)
	 len++;

	  dat = realloc(dat, sizeof(DATAFILE)*(len+2));
      dat[len+1] = dat[len];
      dat[len] = *tmp;
      return dat;
   }
   else
      return NULL;
}



/* inserts a new object into the datafile */
DATAFILE *datedit_insert(DATAFILE *dat, DATAFILE **ret, AL_CONST char *name, int type, void *v, long size)
{
   int len;

   len = 0;
   while( dat[len].type != DAT_END)
      len++;

   dat = realloc(dat, sizeof(DATAFILE)*(len+2));
   dat[len+1] = dat[len];

   dat[len].dat = v;
   dat[len].type = type;
   dat[len].size = size;
   dat[len].prop = NULL;
   datedit_set_property(dat+len, DAT_NAME, name);

   if (ret)
      *ret = dat+len;

   return dat;
}



/* wrapper for examining numeric property values */
int datedit_numprop(AL_CONST DATAFILE *dat, int type)
{
   AL_CONST char *p = get_datafile_property(dat, type);

   if (*p)
      return atoi(p);
   else 
      return -1;
}



/* scans plugins to find available import/export file extensions */
static AL_CONST char *make_ext_list(int type, int grab)
{
   static char buf[256];
   char extlist[256][16];
   int num_ext = 0;
   char *s;
   int i, j;
   int done;

   for (i=0; datedit_grabber_info[i]->type != DAT_END; i++) {
      if (datedit_grabber_info[i]->type == type) {
	 if (grab)
	    s = datedit_grabber_info[i]->grab_ext;
	 else if (!grab)
	    s = datedit_grabber_info[i]->export_ext;
	 else
	    s = NULL;

	 if (s) {
	    strcpy(buf, s);
	    s = strtok(buf, ";");
	    while( s) {
	       for (j=0; j<num_ext; j++) {
		  if (stricmp(s, extlist[j]) == 0)
		     break;
	       }
	       if (j >= num_ext) {
		  strcpy(extlist[num_ext], s);
		  num_ext++;
	       }
	       s = strtok(NULL, ";");
	    }
	 }
      }
   }

   if (num_ext <= 0)
      return NULL;

   do {
      done = TRUE;

      for (i=0; i<num_ext-1; i++) {
	 if (stricmp(extlist[i], extlist[i+1]) > 0) {
	    strcpy(buf, extlist[i]);
	    strcpy(extlist[i], extlist[i+1]);
	    strcpy(extlist[i+1], buf);
	    done = FALSE;
	 }
      }
   } while( !done);

   buf[0] = 0;

   for (i=0; i<num_ext; i++) {
      strcat(buf, extlist[i]);
      if (i<num_ext-1)
	 strcat(buf, ";");
   }

   return buf;
}



/* returns a list of suitable file extensions for this object type */
AL_CONST char *datedit_grab_ext(int type)
{
   return make_ext_list(type, TRUE);
}



/* returns a list of suitable file extensions for this object type */
AL_CONST char *datedit_export_ext(int type)
{
   return make_ext_list(type, FALSE);
}



/* conditionally update an out-of-date object */
int datedit_update(DATAFILE *dat, int verbose, int *changed)
{
//   AL_CONST char *name = get_datafile_property(dat, DAT_NAME);
   AL_CONST char *origin = get_datafile_property(dat, DAT_ORIG);
   AL_CONST char *date = get_datafile_property(dat, DAT_DATE);
   time_t origt, datat;

   if (!*origin) {
//      my_datedit_msg("%s has no origin data - skipping", name);
      return TRUE;
   }

   if (!exists(origin)) {
//      my_datedit_msg("%s: %s not found - skipping", name, origin);
      return TRUE;
   }

   if (*date) {
      origt = file_time(origin);
      datat = datedit_asc2ftime(date);

      if ((origt/60) <= (datat/60)) {
//	 if (verbose)
//	    my_datedit_msg("%s: %s has not changed - skipping", name, origin);
	 return TRUE;
      }
   }

//   my_datedit_msg("Updating %s -> %s", origin, name);

   if (changed)
      *changed = TRUE;

   return datedit_grabupdate(dat, origin, 
			     datedit_numprop(dat, DAT_XPOS), 
			     datedit_numprop(dat, DAT_YPOS), 
			     datedit_numprop(dat, DAT_XSIZ), 
			     datedit_numprop(dat, DAT_YSIZ));
}



/* unconditionally update an object */
int datedit_force_update(DATAFILE *dat, int verbose, int *changed)
{
//   AL_CONST char *name = get_datafile_property(dat, DAT_NAME);
   AL_CONST char *origin = get_datafile_property(dat, DAT_ORIG);

   if (!*origin) {
//      my_datedit_msg("%s has no origin data - skipping", name);
      return TRUE;
   }

   if (!exists(origin)) {
//      my_datedit_msg("%s: %s not found - skipping", name, origin);
      return TRUE;
   }

//   my_datedit_msg("Updating %s -> %s", origin, name);

   if (changed)
      *changed = TRUE;

   return datedit_grabupdate(dat, origin, 
			     datedit_numprop(dat, DAT_XPOS), 
			     datedit_numprop(dat, DAT_YPOS), 
			     datedit_numprop(dat, DAT_XSIZ), 
			     datedit_numprop(dat, DAT_YSIZ));
}
//
l_int LibMain( int argc, l_text *argv )
{
	Filter = NewFileTypes("Allegro Datafile","dat", NewFileTypes("All files",NULL, NULL ));
SYSEXPORT( datedit_menu_info );
SYSEXPORT( ActiveMenu );
SYSEXPORT( datedit_current_palette );
SYSEXPORT( SetDialogColor );
SYSEXPORT( datedit_object_info );
SYSEXPORT( datedit_sort_datafile );
SYSEXPORT( datedit_desc );
SYSEXPORT( datedit_asc2ftime );
SYSEXPORT( datedit_ftime2asc_int );
SYSEXPORT( datedit_set_property );
SYSEXPORT( datedit_sort_properties );
SYSEXPORT( datedit_clean_typename );
SYSEXPORT( datedit_info );
SYSEXPORT( datedit_pretty_name );
SYSEXPORT( datedit_load_datafile );
SYSEXPORT( datedit_save_datafile );
SYSEXPORT( datedit_save_header );
SYSEXPORT( datedit_update );
SYSEXPORT( datedit_force_update );
SYSEXPORT( grabber_import_file );
SYSEXPORT( datedit_grab_ext );
SYSEXPORT( grabber_graphic );
SYSEXPORT( datedit_grab );
SYSEXPORT( datedit_last_read_pal );
SYSEXPORT( grabber_palette );
SYSEXPORT( _unload_datafile_object );
SYSEXPORT( grabber_graphic_origin );
SYSEXPORT( datedit_ftime2asc );
SYSEXPORT( grabber_graphic_date );
SYSEXPORT( datedit_grabreplace );
SYSEXPORT( datedit_export_ext );
SYSEXPORT( datedit_export_name );
SYSEXPORT( datedit_export );
SYSEXPORT( datedit_delete );
SYSEXPORT( datedit_insert );
SYSEXPORT( datedit_numprop );
SYSEXPORT( datedit_grabupdate );
SYSEXPORT( grabber_sel_palette );
SYSEXPORT( grabber_select_property );
SYSEXPORT( grabber_get_grid_size );
SYSEXPORT( grabber_rebuild_list );
SYSEXPORT( grabber_get_selection_info );
SYSEXPORT( grabber_foreach_selection );
SYSEXPORT( grabber_single_selection );
SYSEXPORT( grabber_set_selection );
SYSEXPORT( grabber_busy_mouse );
SYSEXPORT( grabber_modified );
SYSEXPORT( datedit_init );
//
/*SYSEXPORT( datedit_menu_info );
SYSEXPORT( datedit_current_palette );
SYSEXPORT( datedit_object_info );
SYSEXPORT( datedit_sort_datafile );
SYSEXPORT( datedit_asc2ftime );
SYSEXPORT( datedit_ftime2asc_int );
SYSEXPORT( datedit_set_property );
SYSEXPORT( datedit_sort_properties );
SYSEXPORT( datedit_clean_typename );
SYSEXPORT( datedit_info );
SYSEXPORT( datedit_pretty_name );
SYSEXPORT( datedit_load_datafile );
SYSEXPORT( datedit_save_datafile );
SYSEXPORT( datedit_save_header );
SYSEXPORT( datedit_update );
SYSEXPORT( datedit_force_update );
SYSEXPORT( grabber_import_file );
SYSEXPORT( datedit_grab_ext );
SYSEXPORT( grabber_graphic );
SYSEXPORT( datedit_grab );
SYSEXPORT( datedit_last_read_pal );
SYSEXPORT( grabber_palette );
SYSEXPORT( _unload_datafile_object );
SYSEXPORT( grabber_graphic_origin );
SYSEXPORT( datedit_ftime2asc );
SYSEXPORT( grabber_graphic_date );
SYSEXPORT( datedit_grabreplace );
SYSEXPORT( datedit_export_ext );
SYSEXPORT( datedit_export_name );
SYSEXPORT( datedit_export );
SYSEXPORT( datedit_delete );
SYSEXPORT( datedit_insert );
SYSEXPORT( grabber_sel_palette );
SYSEXPORT( grabber_select_property );
SYSEXPORT( grabber_get_grid_size );
SYSEXPORT( grabber_rebuild_list );
SYSEXPORT( grabber_get_selection_info );
SYSEXPORT( grabber_foreach_selection );
SYSEXPORT( grabber_single_selection );
SYSEXPORT( grabber_set_selection );
SYSEXPORT( grabber_busy_mouse );
SYSEXPORT( grabber_modified );
SYSEXPORT( datedit_init ); */
}
//
void Close(void)
{
}
