// Associated include file : ManView.H

#include "Sword.H"
#include "ToolBox/Image/Image.H"

#include "ManView.H"

// Global Vars

char *IdentTMandelScroller = "TMandelScroller";
short RegTMandelScroller;

char *IdentTMandelImage = "TMandelImage";
short RegTMandelImage;

char *IdentTMandelViewer = "TMandelViewer";
short RegTMandelViewer;

// --- TMandelScroller

// This object is a image scroller window in which we insert
// the mandelbrot image.

DEFINE(TMandelScroller);

DEFINE_EVENTS_TABLE(TMandelScroller,TImageScroller)
  MOUSELDOWN()
  MOUSERDOWN()
END_EVENTS_TABLE

TMandelScroller::TMandelScroller(void)
{ Defaults();
}

TMandelScroller::TMandelScroller(int X, int Y, int L, int H, TScreenBitmap *_Image,
                                 TLift *HLift, TLift *VLift)
{ Defaults();
  Init(X,Y,L,H,_Image,HLift,VLift);
}

TMandelScroller::TMandelScroller(TRect R, TScreenBitmap *_Image, TLift *HLift, TLift *VLift)
{ Defaults();
  Init(R.X1(),R.Y1(),R.Width(),R.Height(),_Image,HLift,VLift);
}

void TMandelScroller::Defaults(void)
{ // Object identification
  Ident=IdentTMandelScroller;
  Register=RegTMandelScroller;
  // Default values
  // ...
}

void TMandelScroller::Init(int X, int Y, int L, int H, TScreenBitmap *_Image,
                           TLift *HLift, TLift *VLift)
{ // Derived constructions
  TImageScroller::Init(X,Y,L,H,_Image,HLift,VLift);
  // New constructions
  // ...
}

boolean TMandelScroller::MouseLDown(TPoint Where, int )
{ // Mouse click in the image must caused a Zoom In action
  // Test if the click is within this object
  if (GetStatus(sfMouseIn))
  { TMandelImage *MImage = (TMandelImage*)Image;
    TPoint        P;
    // Compute local coordinate of the click
    MakeLocal(Where,P);
    // Perform the zoom coordinates changement
    MImage->ZoomIn(P.X()+DecX,P.Y()+DecY);
    // Compute the new mandelbrot image
    MImage->Render();
    // Tell SWORD to redraw the image
    Invalidate();
    return TRUE;
  }
  return FALSE;
}

boolean TMandelScroller::MouseRDown(TPoint , int )
{ // Mouse right-click in the image must caused a Zoom Out action
  // Test if the click is within this object
  if (GetStatus(sfMouseIn))
  { TMandelImage *MImage = (TMandelImage*)Image;
    // Perform the zoom-out coordinates changement
    MImage->ZoomOut();
    // Compute the new mandelbrot image
    MImage->Render();
    // Tell SWORD to redraw the image
    Invalidate();
    return TRUE;
  }
  return FALSE;
}

// --- TMandelImage

// This object is a bitmap in memory in which we will draw
// mandelbrot fractal before showing it on the screen. That
// allow quick scrolling in the image, as it is in memory
// (hidden part will not be computed many times).

DEFINE(TMandelImage);

TMandelImage::TMandelImage(void)
{ Defaults();
}

TMandelImage::TMandelImage(int Width, int Height)
{ Defaults();
  Init(Width,Height);
}

void TMandelImage::Defaults(void)
{ // Object identification
  Ident=IdentTMandelImage;
  Register=RegTMandelImage;
  // Other default values
  Iter=128;
  NZooms=0;
  x1=rX1=-1.0;
  x2=rX2=1.0;
  y1=rY1=-1.0;
  y2=rY2=1.0;
}

void TMandelImage::Init(int Width, int Height)
{ // Derived constructions
  TScreenBitmap::Init(Width,Height);
  // Other constructions
  SetViewport(-2.2,-1.2,1.0,1.2);
  Render();
}

void TMandelImage::Render(void)
{ int   i, j, k;
  float R, I, CR, CI, Inter;
  float Mod;
  //
  BeginPhase("Rendering Mandelbrot fractal...");
  for(i=0;i<W;i++)
  { CR=rX1+(rX2-rX1)*i/(W-1);
    BeginDraw();
    for(j=0;j<H;j++)
    { CI=rY1+(rY2-rY1)*j/(H-1);
      R=0.0;I=0.0;
      for(Mod=0.0, k=0; (k<Iter)&&(Mod<4.0); k++)
      { Inter=R*R - I*I + CR;
        I=2*R*I + CI;
        R=Inter;
        Mod=R*R+I*I;
      }
      GrPlot(i, j, Dither->GetRGBColor(i, j, (128+k)%256, 0, 0));
    }
    EndDraw();
    AvancePhase(100*i/(W-1));
  }
  EndPhase();
}

void TMandelImage::SetViewport(float _x1, float _y1, float _x2, float _y2)
{ x1=_x1; y1=_y1;
  x2=_x2; y2=_y2;
  // Compute real coordinate for drawing depending
  // on window aspect ratio
  // :: Compute correct width and height
  float rW=x2-x1;
  float rH=y2-y1;
  if (rW/rH>(float)W/H) rH=(float)H*rW/W;
                   else rW=(float)W*rH/H;
  // :: Center this correct width and height onto window
  float cX=(x2+x1)/2;
  float cY=(y2+y1)/2;
  rX1=cX-rW/2;       rY1=cY-rH/2;
  rX2=cX+rW/2;       rY2=cY+rH/2;
}

void TMandelImage::ZoomIn(int pX1, int pY1, int pX2, int pY2)
{ if (PushZoomStatus()) return;
  // Zoom coordinates are given in pixels in the image.
  // Convert that into float coordinates
  SetViewport( rX1+(rX2-rX1)*pX1/(W-1),
               rY1+(rY2-rY1)*pY1/(H-1),
               rX1+(rX2-rX1)*pX2/(W-1),
               rY1+(rY2-rY1)*pY2/(H-1) );
}

void TMandelImage::ZoomIn(int X, int Y)
{ if (PushZoomStatus()) return;
  // Zoom with only two coordinates = zoom x2 around
  // this center (given in pixel in the image coordinates).
  float cX = rX1+(rX2-rX1)*X/(W-1);
  float cY = rY1+(rY2-rY1)*Y/(H-1);
  SetViewport(cX-(x2-x1)/4, cY-(y2-y1)/4,
              cX+(x2-x1)/4, cY+(y2-y1)/4);
}

void TMandelImage::ZoomOut(void)
{ if (NZooms)
  { // There are zoom status on the zoom-stack
    // just restore the last pushed zoom status
    NZooms--;
    SetViewport(Zooms[NZooms].X1,Zooms[NZooms].Y1,
                Zooms[NZooms].X2,Zooms[NZooms].Y2);
  }
  else
  { // No zoom on the zoom-stack. Zoom out by a factor
    // of 2
    float cX=(x1+x2)/2;
    float cY=(y1+y2)/2;
    SetViewport(cX-(x2-x1), cY-(y2-y1),
                cX+(x2-x1), cY+(y2-y1));
  }
}

boolean TMandelImage::PushZoomStatus(void)
{ // Check for the maximum number of zoom positions in memory
  if (NZooms==50) return TRUE;
  // push zoom status
  Zooms[NZooms].X1=x1;
  Zooms[NZooms].Y1=y1;
  Zooms[NZooms].X2=x2;
  Zooms[NZooms].Y2=y2;
  NZooms++;
  return FALSE;
}

// --- TMandelViewer

// This object is the mandelbrot viewer window. It just
// contain a mandelbrot scroller in order to show the
// fractal with scrolling capabilities

DEFINE(TMandelViewer);

TMandelViewer::TMandelViewer(void)
{ Defaults();
}

TMandelViewer::TMandelViewer(int X, int Y, int L, int H, TScreenBitmap *_Image)
{ Defaults();
  Init(X,Y,L,H,_Image);
}

TMandelViewer::TMandelViewer(TRect R, TScreenBitmap *_Image)
{ Defaults();
  Init(R.X1(),R.Y1(),R.Width(),R.Height(),_Image);
}

void TMandelViewer::Defaults(void)
{ // Object identification
  Ident=IdentTMandelViewer;
  Register=RegTMandelViewer;
  // Other default values
  // ...
}

void TMandelViewer::Init(int X, int Y, int L, int H, TScreenBitmap *_Image)
{ // Derived constructions
  TImageViewer::Init(X,Y,L,H,"Mandelbrot Fractal viewer",_Image);
  // Other constructions
  // ...
}

TImageScroller *TMandelViewer::CreateImageScroller(TRect R, TScreenBitmap *_Image, TLift *HLift, TLift *VLift)
{ return new TMandelScroller(R,_Image,HLift,VLift);
}
