;
; UDMAJR.ASM    Written 18-Dec-2004 by Jack R. Ellis
;
; UDMAJR is free software.  You can redistribute and/or modify it under
; the terms of the GNU General Public License (hereafter called GPL) as
; published by the Free Software Foundation, either version 2 of GPL or
; any later versions at your option.  UDMAJR is distributed in the hope
; that it will be useful, but WITHOUT ANY WARRANTY and without even the
; implied warranties of MERCHANTABILITY nor of FITNESS FOR A PARTICULAR
; PURPOSE!  See the GPL for details.  You ought to have received a copy
; of the GPL with these UDMA files.  If not, write to the Free Software
; Foundation Inc., 59 Temple Place Ste. 330, Boston, MA 02111-1307 USA.
; http://www.gnu.org/licenses/
;
; Special thanks to Luchezar I. Georgiev for his INESTIMABLE advice and
; help in research, revising, enhancing, and test of the original UDMA!
;
; This is a DOS driver designed to handle 1 to 4 UltraDMA hard-disks on
; PC motherboards having a VIA 8235 or equivalent chipset.   The driver
; determines which of the IDE units are actually UltraDMA hard-disks at
; initialization and will run all such disks.	All UltraDMA disks from
; mode 0 ATA-16 thru mode 7 ATA-166 may be used.    An UltraDMA disk is
; assumed to handle full LBA mode (63 sectors, 255 heads and a designed
; cylinder count).   "LBA mode" I-O requests are supported for FreeDOS,
; MS-DOS V7.xx, and other systems that allow them.   LBA values over 28
; bits shall cause the driver to use "Read/Write Extended" DMA commands
; and need an ATA-6 or newer hard-disk.	  LBA values of 28 bits or less
; shall use regular DMA commands.   24-bit "CHS mode" is also supported
; for MS-DOS V6.xx and earlier.	  Data accessed using CHS calls must be
; located in the initial 8-GB of the disk.
;
; The driver intercepts BIOS INT13 read or write requests only.	  Other
; INT13 requests (including seeks) and read/write requests with invalid
; parameters will be "passed" back to the BIOS or some other driver for
; handling.   If a user I-O buffer is not DWORD aligned, crosses a 64K-
; boundary or fails a VDS "lock", the I-O request will use a 64K buffer
; in XMS memory with UltraDMA to or from the buffer, to avoid "passing"
; these requests to the BIOS for execution in slow PIO mode!   Although
; UltraDMA specifies word-aligned buffers, ERRATA in some chipsets does
; require DWORD alignment and avoiding a 64K DMA address boundary!
;
; NOTE:	  UDMAJR is a "short" variant of the full UDMA driver.	  It is
; intended for "RAM disk" and other space-limited systems.   UDMAJR has
; all UDMA run-time functions, i.e. it supports up to 4 disks, supports
; LBA-48 and any earlier address mode, provides all return codes (shown
; below), and permits "DMA only" use when an XMS driver is not present.
; To hold UDMAJR.SYS at 2048 bytes, the following items are omitted:
;   A) The check for an 80386 or better CPU.
;   B) Hard-disk names and controller "bus" data displays.
;   C) All initialization read tests and the read-rate display.
; Users who REQUIRE these items should employ the full UDMA driver.
;
; Beginning with version 1.6 of this driver, the following return codes
; have been added to help in diagnosing "problem" systems and chipsets.
; On exit from successful I-O requests, the AH-register is zero and the
; carry flag is reset.	 If an error occurs, the carry flag is SET, and
; the AH-register contains one of the following codes:
;
;  Code 08h - DMA timed out
;	0Fh - DMA error
;	20h - Controller busy before I-O
;	21h - Controller busy after I-O
;	80h - First DRQ timed out
;	AAh - Disk not ready before I-O
;	ABh - Disk not ready after I-O
;	CCh - Write FAULT before I-O
;	CDh - Write FAULT after I-O
;	E0h - Hard error at I-O end
;	FFh - XMS memory error
;
;
; Revision History:
; ----------------
;  V7.4  18-Dec-04   JE   Fixed DRQ timeout code
;  V7.3  11-Dec-04   JE   Fixed FOOLISH disk-select bug (My apologies!)
;  V7.2  10-Dec-04   JE   No EDD BIOS error abort, more code reductions
;  V7.1	  2-Dec-04   JE	  Total revision, derived from V1.6 UDMA2
;  V7.0	 14-Feb-04   JE	  Merged new init, V6.8 run-time, V6.9 CHS code
;  V6.9	  8-Feb-04   JE	  Any CHS "geometry" O.K., buffered I.D. input
;  V6.8	 28-Jan-04   JE	  If no EDD/DPTE, 4 disks at 80h-83h units only
;  V6.7	 16-Jan-04   JE	  Renumbered to replace UDMA, init code reduced
;  V2.2	 25-Dec-03   JE	  Corrected "read test" diagnostic messages
;  V2.1	 24-Dec-03   JE	  Use XMS for read tests, to reduce UDMA size
;  V2.0	 21-Dec-03   JE	  Controller-name displays, multi-sector tests
;  V1.9	  6-Dec-03   JE	  Fixed VDS init bug, buffer/diagnostic "swap"
;  V1.8	  3-Dec-03   JE	  Fixed "STI" bug, "DMA only" now 528 bytes
;  V1.7	 25-Nov-03   JE	  If no XMS driver, allow "DMA only" usage
;  V1.6	 22-Nov-03   JE	  Fixed init reads, added full error codes
;  V1.5	 15-Nov-03   JE	  Added all UDMA init functions but ctlr. name
;  V1.4	 14-Nov-03   JE	  Corrected DMA-status reset
;  V1.3	 13-Nov-03   JE	  "DoIO" does ALL I-O, "XMS error" now 0FFh
;  V1.2	 12-Nov-03   JE	  No "timeout error", other size reductions
;  V1.1	  7-Nov-03   JE	  Used 80386 test from V5.9 UDMA
;  V1.0	  6-Nov-03   JE	  Initial release (had been named UDMA-E)
;
;
; General Program Equations.
;
%define VER 'V7.4, 18-Dec-2004.'
RDYTO	equ	008h		;384-msec minimum I-O timeout.
BIOSTMR equ	0046Ch		;BIOS "tick" timer address.
HDISKS	equ	00475h		;BIOS hard-disk count address.
VDSFLAG equ	0047Bh		;BIOS "Virtual DMA" flag address.
CR	equ	00Dh		;ASCII carriage-return.
LF	equ	00Ah		;ASCII line-feed.
;
; Driver Return Codes.
;
DMATIMO equ	0E8h		;DMA timeout code, 008h at exit.
DMAERR	equ	0EFh		;DMA error   code, 00Fh at exit.
CTLRERR equ	000h		;Ctlr. busy  code, 020h/021h at exit.
DRQTIMO equ	080h		;DRQ timeout code.
DISKERR equ	08Ah		;Disk-busy   code, 0AAh/0ABh at exit.
WFLTERR equ	0ACh		;Write-fault code, 0CCh/0CDh at exit.
HARDERR equ	0BFh		;Hard-error  code, 0E0H at exit.
XMSERR	equ	0FFh		;XMS memory-error code.
;
; IDE Controller Register Definitions.
;
CDATA	equ	001F0h		;Data port.
CSUBCM	equ	CDATA+1		;Subcommand register.
CSECCT	equ	CDATA+2		;I-O sector count.
CDSEL	equ	CDATA+6		;Disk-select and upper LBA.
CCMD	equ	CDATA+7		;Command register.
CSTAT	equ	CDATA+7		;Primary status register.
CSTAT2	equ	CDATA+206h	;Alternate status register.
;
; Controller Status and Command Definitions.
;
BSY	equ	080h		;IDE controller is busy.
RDY	equ	040h		;IDE disk is "ready".
FLT	equ	020h		;IDE disk has a "fault".
DRQ	equ	008h		;IDE data request.
ERR	equ	001h		;IDE general error flag.
DMI	equ	004h		;DMA interrupt occured.
DME	equ	002h		;DMA error occurred.
DRCMD	equ	0C8h		;DMA read command (write is 0CAh,
				;    LBA48 commands are 025h/035h).
SETM	equ	003h		;Set Mode subcommand.
SETF	equ	0EFh		;Set Features command.
LBABITS equ	0E0h		;Fixed LBA command bits.
;
; LBA "Device Address Packet" Layout.
;
struc	DAP
DapPL	resb	1		;Packet length.
	resb	1		;(Reserved).
DapSC	resb	1		;I-O sector count.
	resb	1		;(Reserved).
DapBuf	resd	1		;I-O buffer address (segment:offset).
DapLBA	resw	1		;48-bit logical block address (LBA).
DapLBA1	resw	1
DapLBA2	resw	1
endstruc
;
; DOS "Request Packet" Layout.
;
struc	RP
	resb	2		;(Unused by us).
RPOp	resb	1		;Opcode.
RPStat	resw	1		;Status word.
	resb	9		;(Unused by us).
RPSize	resd	1		;Resident driver size.
endstruc
RPERR	equ	08003h		;Packet "error" flags.
RPDON	equ	00100h		;Packet "done" flag.
;
; DOS Driver Device Header.
;
@	dd	0FFFFFFFFh	;Link to next header block.
	dw	08000h		;Driver "device attributes".
	dw	Strat		;"Strategy" routine offset.
IDEAd	equ	$-1		;(Lower IDE status address, after Init).
	dw	DevInt		;"Device-Interrupt" routine offset.
DMAAd	equ	$-2		;(DMA command-reg. address, after Init).
	db	'UDMAJR$',0,0	;Driver name (NO "arrowhead" for V7.1+).
;
; Resident Driver "Data Page" Variables.
;
VLF	db	0		;VDS lock flag (1 = user buffer locked).
Units	dd	0000000FFh	;IDE "active units" table, set by Init:
	dd	0000000FFh	;  Byte 0:    BIOS unit (0FFh inactive).
	dd	0000000FFh	;  Byte 1:    CHS sectors per head.
	dd	0000000FFh	;  Byte 2-3:  CHS sectors/cylinder.
PRDAd	dd	IOAdr-@		;PRD 32-bit command addr. (Init set).
	db	0		;IDE "upper" sector count (always 0).
LBA2	db	0FAh,0F0h,08Ah	;IDE "upper" LBA bits 24-47.
SecCt	db	080h		;IDE "lower" sector count.
LBA	db	0,0,0		;IDE "lower" LBA bits 0-23.
DSCmd	db	0		;IDE disk-select and LBA commands.
IOCmd	db	0		;IDE command byte.
XMHdl	dw	0		;XMS buffer handle number (Init set).
XMOfs	dd	0		;XMS 32-bit buffer offset (Init set).
VDSLn	dd	ResEnd-@	;VDS and XMS buffer length.
VDSOf	dd	0		;VDS 32-bit offset.
VDSSg	dd	0		;VDS 16-bit segment (hi-order zero).
IOAdr	dd	0		;VDS and DMA 32-bit address.
IOLen	dd	080000000h	;DMA byte count and "end" flag.
XMSSH	equ	VDSOf		;(XMS parameters share the VDS block).
XMSDH	equ	VDSSg+2
;
; Driver Entry Routine.  For CHS requests, the registers contain:
;
;   AH      Request code.  We handle 002h read and 003h write.
;   AL      I-O sector count.
;   CH      Lower 8 bits of starting cylinder.
;   CL      Starting sector and upper 2 bits of cylinder.
;   DH      Starting head.
;   DL      BIOS unit number.  We handle 080h and up (hard-disks).
;   ES:BX   I-O buffer address.
;
; For LBA requests, the registers contain:
;
;   AH      Request code.  We handle 042h read and 043h write.
;   DL      BIOS unit number.  We handle 080h and up (hard-disks).
;   DS:SI   Pointer to Device Address Packet ("DAP"), described above.
;
Entry	pusha			;Entry -- save all CPU registers.
	mov	bp,16		;Reset active-units table index.
@LastU	equ	$-2		;(Last-unit index, set by Init.  MUST
				;  be 16 while Init TESTS all disks!). 
NextU	sub	bp,byte 4	;Any more active units to check?
	js	QuickX		;No, request NOT for us -- exit quick!
	cmp	dl,[cs:bp+Units-@] ;Does request unit match our table?
	jne	NextU		;No, see if more table entries remain.
	sti			;Ensure CPU interrupts are enabled.
	cld			;Ensure FORWARD "string" commands!
	push	ds		;Save CPU segment registers.
	push	es
	mov	dl,0BEh		;Mask out LBA and write request bits.
	and	dl,ah
	cmp	dl,002h		;Is this a CHS or LBA read or write?
	jne	Pass		;No, let BIOS handle this request.
	shl	ah,1		;Is this an LBA read or write request?
	jns	ValSC		;No, go validate CHS sector count.
	mov	al,[si+DapSC]	;Get "DAP" I-O sector count.
	cmp	dword [si+DapBuf],byte -1 ;64-bit "DAP" buffer address?
	jne	ValSC		;No, go validate "DAP" sector count.
Pass	pop	es		;"Pass" request -- reload segment regs.
	pop	ds
QuickX	mov	bp,sp		;Reload CPU flags saved by Int 13h.
	push	word [bp+20]
	popf
	popa			;Reload all CPU registers.
	jmp	0000:0000	;Go to next routine in Int 13h chain.
@PrvI13 equ	$-4		;(Previous INT13 vector, set by Init).
ValSC	dec	al		;Is sector count zero or over 128?
	js	Pass		;Yes?  Let BIOS handle this "No-No!".
	inc	ax		;Restore sector count -- LBA request?
	js	GetDAP		;Yes, get remaining "DAP" parameters.
	xchg	ax,cx		;CHS -- save request code and sectors.
	mov	si,0003Fh	;Set SI-reg. to starting sector.
	and	si,ax
	dec	si
	mov	di,dx		;Set DI-reg. to starting head.
	shr	al,6		;Set AX-reg. to cylinder number.
	xchg	al,ah
	mul	word [cs:bp+Units+2-@]  ;Convert cylinder to sectors.
	xchg	ax,di		     ;Swap low-order and head number.
	mov	al,[cs:bp+Units+1-@] ;Convert head to sectors.
	mul	ah
	add	si,ax		;Add to starting sector.
	add	si,di		;Add in cylinder sectors.
	adc	dl,dh
	xchg	ax,bx		;Get buffer offset in AX-register.
	xchg	ax,cx		;Swap offset with request/sectors.
	xor	di,di		;Reset upper LBA address bits.
	jmp	short SetDS	;Go set our DS-register.
GetDAP	les	cx,[si+DapBuf]	;Get "DAP" I-O buffer address.
	mov	di,[si+DapLBA2]	;Get "DAP" logical block address.
	mov	dx,[si+DapLBA1]
	mov	si,[si+DapLBA]
SetDS	push	cs		;Set our DS-register.
	pop	ds
	xor	bx,bx		;Zero BX-reg. for relative commands.
	mov	[bx+LBA-@],si	;Save 48-bit logical block address.
	mov	[bx+LBA+2-@],dl
	mov	[bx+LBA2-@],dh
	mov	[bx+LBA2+2-@],di
	shr	dx,12		;Shift out LBA bits 16-27.
	or	di,dx		;Anything in LBA bits 28-47?
	jz	LBA28		;No, use LBA28 read/write command.
	shl	ah,3		;LBA48 -- get command as 020h/030h.
	jmp	short GetAdr	;Go get IDE and LBA address bytes.
LBA28	xchg	dh,[bx+LBA2-@]	;LBA28 -- reload & reset bits 24-27.
	or	ah,(DRCMD+1)	;Get LBA28 read/write command + 5.
GetAdr	mov	di,CDSEL-00100h	;Get primary device-address bytes.
@DMALo1	equ	$-1		;(Lower DMA command address, Init set).
	shr	bp,3		;Is this a primary-channel request?
	jz	DevAdr		;Yes, set IDE & PCI address bytes.
	mov	di,CDSEL+00680h	;Get secondary device-address bytes.
@DMALo2	equ	$-1		;(Lower DMA command address, Init set).
DevAdr	mov	[bx+IDEAd-@],di ;Set current IDE & PCI address bytes.
	mov	dl,(LBABITS/32)	;Get disk-select & LBA command bits.
	rcl	dl,5
	or	dl,dh		;"Or" in LBA28 bits 24-27 (if any).
	mov	dh,005h		;Get final IDE command byte.
	xor	dh,ah		;(LBA28 = C8h/CAh, LBA48 = 25h/35h).
	mov	[bx+DSCmd-@],dx	;Set LBA and IDE command bytes.
	mov	[bx+SecCt-@],al	;Set I-O sector count.
	mov	ah,0		;Set VDS/XMS and DMA buffer lengths.
	shl	ax,1
	mov	[bx+VDSLn+1-@],ax
	mov	[bx+IOLen+1-@],ax
	mov	[bx+VDSOf-@],cx	;Set VDS offset and segment.
	mov	[bx+VDSOf+2-@],bx
	mov	[bx+VDSSg-@],es
	or	dword [bx+IOAdr-@],byte -1 ;Invalidate VDS address.
	mov	ax,08103h	           ;VDS "lock" user buffer.
	mov	dx,0000Ch
	call	VDLock
	jc	GoToBuf		           ;Error -- do buffered I-O.
	cmp	dword [bx+IOAdr-@],byte -1 ;Got a valid VDS address?
	je	NoVDS		;No, set 20-bit user buffer address.
	inc	byte [bx+VLF-@]	;Set VDS "lock" flag.
	mov	ax,[bx+IOAdr-@]	;Get low-order VDS buffer address.
ChkOdd	test	al,003h		;Is user I-O buffer 32-bit aligned?
	jnz	NoLock		;No, use buffered I-O routines below.
	mov	cx,[bx+IOLen-@]	;Get lower ending DMA address.
	dec	cx		;(IOLen - 1 + IOAdr).
	add	ax,cx		;Would this I-O cross a 64K boundary?
	jc	NoLock		;Yes, use buffered I-O routines below.
	call	DoDMA		;Do direct DMA I-O with user's buffer.
Done	mov	bp,sp		;Done -- point to saved registers.
	mov	[bp+19],al	;Set error code in exiting AH-reg.
	rcr	byte [bp+24],1	;Set error flag in exiting carry
	rol	byte [bp+24],1	;  bit, in flags saved by Int 13h.
	call	VDUnlk		;If needed, "unlock" user I-O buffer.
	pop	es		;Reload all CPU registers and exit.
	pop	ds
	popa
	iret
NoVDS	mov	ax,16		;No VDS -- get 20-bit buffer segment.
	mul	word [bx+VDSSg-@]
	add	ax,[bx+VDSOf-@]	;Add in buffer offset value.
	adc	dx,bx
	mov	[bx+IOAdr-@],ax	;Set 20-bit user buffer address.
	mov	[bx+IOAdr+2-@],dx
	jmp	short ChkOdd	;Go check for "odd" buffer alignment.
NoLock	call	VDUnlk		;Buffered I-O -- "unlock" user buffer.
GoToBuf	jmp	BufIO		;Go to buffered I-O routines below.
	db	0		;(Unused alignment "filler").
;
; Subroutine to execute read and write commands.
;
BufDMA	mov	dword [bx+IOAdr-@],0  ;Buffered -- set XMS buffer addr.
@XBufAd	equ	$-4		;(XMS 32-bit buffer address, Init set).
DoDMA	mov	dx,[bx+DMAAd-@]	;Ensure any previous DMA is stopped!
	in	al,dx		;(On some older chipsets, if DMA is
	and	al,0FEh		;  running, reading an IDE register
	out	dx,al		;  causes the chipset to "HANG"!!).
	mov	al,[bx+DSCmd-@]	;Select our desired disk.
	and	al,0F0h
	mov	dl,[bx+IDEAd-@]
	mov	dh,001h
	out	dx,al
	mov	di,dx		;Save IDE drive-select address.
	mov	es,bx		;Point to low-memory BIOS timer.
	mov	si,BIOSTMR
	mov	cx,(RDYTO*256)+FLT   ;Get timeout count & "fault" mask.
	add	ch,[es:si]	     ;Set CH-reg. with timeout limit.
	call	ChkRdy		     ;Await controller- and disk-ready.
	jc	IOExit		     ;If error, exit immediately!
	test	byte [bx+IOCmd-@],012h  ;Is this a write request?
	jnz	SetDMA		;Yes, reset DMA command register.
	mov	al,008h		;Get "DMA read" command bit.
SetDMA	mov	dx,[bx+DMAAd-@]	;Reset DMA commands and set DMA mode.
	out	dx,al
	inc	dx		;Point to DMA status register.
	inc	dx
	in	al,dx		;Reset DMA status register.
	or	al,006h		;(Done this way so we do NOT alter
	out	dx,al		;  the "DMA capable" status flags!).
	push	si		;Save BIOS timer pointer.
	inc	dx		;Set PRD pointer to our DMA address.
	inc	dx
	mov	si,(PRDAd-@)
	outsd
	mov	bx,001F7h	;Set IDE parameter-output flags.
NxtPar	lea	dx,[di+CSECCT-CDSEL-1] ;Point to IDE sector count -1.
IDEPar	inc	dx		;Output all ten LBA48 parameter bytes.
	outsb			;(1st 4 overlayed by 2nd 4 if LBA28!).
	shr	bx,1		;More parameters to go in this group?
	jc	IDEPar		;Yes, loop back and output next one.
	jnz	NxtPar		;If first 4 done, go output last 6.
	pop	si		;Reload BIOS timer pointer.
	mov	dh,003h		;Get IDE alternate-status address.
	dec	dx		;(Primary-status address | 300h - 1).
ChkDRQ	cmp	ch,[es:si]	;Too long without 1st data-request?
	je	ErrDRQ		;Yes?  Return carry and DRQ error!
	in	al,dx		;Read IDE alternate status.
	and	al,DRQ		;Has 1st data-request arrived?
	jz	ChkDRQ		;No, loop back and check again.
	mov	dx,[bx+DMAAd-@]	;Set DMA Start/Stop bit (starts DMA).
	in	al,dx
	inc	ax
	out	dx,al
ChkDMA	inc	dx		;Point to DMA status register.
	inc	dx
	in	al,dx		;Read DMA controller status.
	dec	dx		;Point back to DMA command register.
	dec	dx
	and	al,DMI+DME	;DMA interrupt or DMA error?
	jnz	HltDMA		;Yes, halt DMA and check results.
	cmp	ch,[es:si]	;Has our DMA transfer timed out?
	jne	ChkDMA		;No, loop back and check again.
HltDMA	push	ax		;Save ending DMA status.
	in	al,dx		;Reset DMA Start/Stop bit.
	and	al,0FEh
	out	dx,al
	pop	ax		;Reload ending DMA status.
	cmp	al,DMI		;Did DMA end with only an interrupt?
	jne	ErrDMA		;No?  Go see what went wrong.
	inc	dx		;Reread DMA controller status.
	inc	dx
	in	al,dx
	test	al,DME		;Any "late" DMA error after DMA end?
	jnz	DMAEnd		;Yes?  Return carry and DMA error!
	inc	cx		;Check "fault" and hard-error at end.
ChkRdy	lea	dx,[di+CSTAT-CDSEL] ;Read IDE primary status.
	in	al,dx
	test	al,BSY+RDY	;Controller or disk still busy?
	jg	ChkErr		;No, check for "fault" or hard-error.
	cmp	ch,[es:si]	;Too long without becoming ready?
	jne	ChkRdy		;No, loop back and check again.
	test	al,BSY		;BAAAD News!  Did controller go ready?
	mov	ax,(256*CTLRERR)+DISKERR ;(Get not-ready error codes).
	jmp	short WhichE	;Go see which error code to return.
ErrDRQ	mov	al,DRQTIMO	;BAAAD News!  Get DRQ error code.
	stc			;Set carry flag (error) and exit!
IOExit	ret
ChkErr	and	al,cl		;Disk "fault" or hard-error?
	jz	IOExit		;No, all is well -- go exit below.
	test	al,FLT		;BAAAD News!  Is the disk "faulted"?
	mov	ax,(256*WFLTERR)+HARDERR ;(Get hardware error codes).
WhichE	jz	EndErr		;If "zero", use AL-reg. return code.
	mov	al,ah		;Use AH-reg. return code of this pair.
EndErr	add	al,cl		;Add 1 if error occurred at I-O end.
	stc			;Set carry flag to denote "error"!
	ret			;Exit.
ErrDMA	test	al,DME		;BAAAD News!  Did DMA end with error?
DMAEnd	mov	ax,(256*DMAERR)+DMATIMO	 ;(Get DMA error codes).
	jmp	short WhichE	;Go see which error code to return.
;
; Subroutine to do a VDS "lock" or "unlock".
;
VDUnlk	shr	byte [bx+VLF-@],1 ;Was input buffer "locked" by VDS?
	jnc	VDExit		;No, just exit below.
VDUnlX	mov	ax,08104h	;Get VDS "unlock" parameters.
	xor	dx,dx
VDLock	mov	di,(VDSLn-@)	;Point to VDS parameter block.
	push	cs
	pop	es
	push	bx		;Save BX-reg. and execute VDS request.
	int	04Bh
VDDone	pop	bx		;Reload our BX-register.
	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs
	pop	ds
VDExit	ret			;Exit.
NoXMEnd	equ	$		;End of resident "No XMS" driver.
;
; Buffered I-O routines, put here so they and the XMSMov subroutine
;   can be "dismissed" during driver-init if no XMS driver is found!
;
BufIO	shl	dword [bx+VDSOf-@],16  ;Convert to XMS handle & offset.
	test	byte [bx+IOCmd-@],012h ;Is this a write request?
	jnz	BufOut		;Yes, use output routine below.
	call	BufDMA		;Input all data to driver XMS buffer.
	jc	Done		;If error, post return code & exit!
	call	XMSMov		;Move XMS data to user input buffer.
	jmp	short DoneJ	;Done -- post any return code & exit.
BufOut	call	XMSMov		;Output -- move data to XMS buffer.
	jc	Done		;If error, post return code & exit!
	call	BufDMA		;Output all data from XMS buffer.
DoneJ	jmp	Done		;Done -- go post return code & exit.
;
; Subroutine to move data to or from the driver's XMS buffer.
;
XMSMov	push	cs		;Point ES-reg. to our data.
	pop	es
	mov	di,(XMSDH-@)	;Point to XMS destination field.
	jnz	XMSet		;If output, just set XMS destination!
	mov	si,(XMSSH-@)	;Point to XMS user-buffer address.
	movsw			;Move user-buffer address from
	movsw			;  XMS source to XMS destination.
	movsw
	mov	di,(XMSSH-@)	;Point to XMS source field.
XMSet	mov	si,(XMHdl-@)	;Set XMS handle and buffer offset as
	movsw			;  input source or output destination.
	movsw
	movsw
XMGo	mov	ah,00Bh		;Get XMS "move data" request code.
XMCall	push	bx		;Save BX-reg. and execute XMS request.
	call	0000:0000	;(SI-reg. points to IOLen after move).
@XEntry equ	$-4		;(XMS "entry" address, set by init).
	dec	ax		;Return 0 if success, -1 if error.
	sar	ax,1		;If error, set carry flag on.
	jmp	short VDDone	;Go reload BX-reg. and exit above.
	db	0		;(Unused alignment "filler").
ResEnd	equ	$		;End of resident driver.
;
; Initialization Tables And Variables.
;
	align	4
HDNames dw	PMMsg		;Table of hard-disk "name" pointers.
	dw	PSMsg
	dw	SMMsg
	dw	SSMsg
Modes	db	'16. '		;Mode 0 = ATA-16  UltraDMA mode table.
	db	'25. '		;Mode 1 = ATA-25.
	db	'33. '		;Mode 2 = ATA-33.
	db	'44. '		;Mode 3 = ATA-44  (Rare but possible).
	db	'66. '		;Mode 4 = ATA-66.
	db	'100.'		;Mode 5 = ATA-100.
	db	'133.'		;Mode 6 = ATA-133.
	db	'166.'		;Mode 7 = ATA-166.
Packet	dd	0		;"Init" request packet address.
HDCount db	0		;Remaining hard-disk count.
EDDFlag db	0		;"EDD BIOS present" flag.
HDUnit	db	0		;Current BIOS unit number.
HDIndex db	0		;IDE "index" number.
EDDBuff dd	30		;Start of 30-byte EDD input buffer.
;
; "Strategy" routine -- At entry, ES:BX points to the DOS init request
;   packet, whose address is saved for processing below.
;
Strat	push	es		;Save DOS request-packet address.
	push	bx
	pop	dword [cs:Packet]
	retf			;Exit & await DOS "Device Interrupt".
;
; "Device-Interrupt" routine -- This routine initializes the driver.
;
DevInt	pushf			;Entry -- save CPU flags.
	pushad			;Save all CPU registers.
	push	ds
	push	es
	push	cs		;Set our DS-register.
	pop	ds
	les	bx,[Packet]	;Point to DOS request packet.
	cmp	byte [es:bx+RPOp],0  ;Is this an "Init" packet?
	jne	near I_BadP	;No?  Go post errors and exit quick!
	mov	dx,UDMsg	;Display driver "title" message.
	call	I_Dspl
	xor	edi,edi		;Get PCI BIOS "I.D." code.
	mov	ax,0B101h
	int	01Ah
	push	cs		;Reload our DS-register.
	pop	ds
	cmp	edx,"PCI "	;Is PCI BIOS V2.0C or newer?
	mov	dx,PEMsg	;(Get error message pointer if not).
	jne	I_PCEr		;No?  Go display message and exit!
	mov	si,LBA2		;Point to interface byte table.
	cld			;Ensure FORWARD "string" commands!
I_GetD	mov	ecx,000010100h	;We want class 1 storage, subclass 1 IDE.
	lodsb			;Get next "valid" PCI interface byte.
	mov	cl,al
	push	si		;Search for our PCI class code.
	mov	ax,0B103h	;(Returns bus/device/function in BX).
	xor	si,si
	int	01Ah
	push	cs		;Reload our DS-register.
	pop	ds
	pop	si
	jnc	I_GotD		;Found our boy!  Go process it.
	cmp	si,LBA		;More interface bytes to try?
	jb	I_GetD		;Yes, go try next one.
	mov	dx,NEMsg	;BAAAD News!  Point to error message.
	jmp	short I_PCEr	;Go display error message and exit.
I_GotD	push	bx		;Save PCI bus/device/function.
	mov	ax,0B108h	;Get low-order PCI command byte.
	mov	di,4
	int	01Ah
	push	cs		;Reload our DS-register.
	pop	ds
	pop	bx		;Reload PCI bus/device/function.
	and	cl,005h		;Mask Bus-Master and I-O Space bits.
	cmp	cl,005h		;Is this how our controller is set up?
	je	I_Base		;Yes, get our PCI base address.
	mov	dx,MEMsg	;Cannot USE it -- point to error msg.
I_PCEr	jmp	I_EOut		;Go display "INVALID" message & exit!
I_Base	mov	ax,0B109h	;Get PCI base address (register 4).
	mov	di,32
	int	01Ah
	push	cs		;Reload our DS-register.
	pop	ds
	xchg	ax,cx		;Post our DMA controller address.
	and	al,0FCh
	mov	[DMAAd],ax
	mov	[@DMALo1],al	;Set low-order DMA address bytes.
	add	[@DMALo2],al
	mov	cx,4		;Set hex address in display message.
	mov	si,DspAd
I_Hex	rol	ax,4		;Get next hex digit in low-order.
	push	ax		;Save remaining digits.
	and	al,00Fh		;Mask off next hex digit.
	cmp	al,009h		;Is digit 0-9?
	jbe	I_HexA		;Yes, convert to ASCII.
	add	al,007h		;Add A-F offset.
I_HexA	add	al,030h		;Convert digit to ASCII.
	mov	[si],al		;Set next ASCII digit in message.
	inc	si		;Bump message address.
	pop	ax		;Reload remaining digits.
	loop	I_Hex		;If more digits to go, loop back.
	mov	dx,PCMsg	;Display controller I-O address.
	call	I_Dspl
	mov	ax,04300h	;Inquire about an XMS manager.
	int	02Fh
	push	cs		;Reload our DS-register.
	pop	ds
	mov	dx,NXMsg	;Point to "No XMS manager" message.
	cmp	al,080h		;Is an XMS manager installed?
	jne	I_XErr		;No, display message & disable XMS.
	mov	ax,04310h	;Get XMS manager "entry" address.
	int	02Fh
	push	cs		;Reload our DS-register.
	pop	ds
	push	es		;Save XMS manager "entry" address.
	push	bx
	pop	dword [@XEntry]
	mov	ah,009h		;Ask XMS manager for 128K of memory.
	mov	dx,128
	call	XMCall
	jc	I_XMErr		;If error, display msg. & disable XMS.
	mov	[XMHdl],dx	;Save XMS buffer handle number.
	mov	ah,00Ch		;"Lock" our XMS memory (direct call,
	call	far [@XEntry]	;  as "XMCall" saves & restores BX!).
	push	cs		;Reload our DS-register.
	pop	ds
	dec	ax		;Did XMS memory get locked?
	jnz	I_RidX		;No?  Unuseable -- get RID of it!
	shl	edx,16		;Get unaligned XMS buffer address.
	mov	dx,bx
	mov	eax,edx		;Find 1st 64K boundary after start.
	add	eax,65536
	xor	ax,ax
	mov	[@XBufAd],eax	;Set final XMS 32-bit buffer address.
	sub	eax,edx		;Initialize "offset" into XMS memory.
	mov	[XMOfs],eax
	mov	ah,005h		;Local-enable "A20 line" FOREVER!
	call	XMCall		;("A20" CANNOT turn off during DMA!).
	jnc	I_Stop		;If no error, ensure DMA is stopped.
	mov	ah,00Dh		;Unuseable!  Unlock our XMS memory.
	mov	dx,[XMHdl]
	call	XMCall
I_RidX	xor	dx,dx		;Load & reset our XMS buffer handle.
	xchg	dx,[XMHdl]
	mov	ah,00Ah		;Free our XMS memory.
	call	XMCall
I_XMErr mov	dx,XEMsg	;Point to "XMS setup error" message.
I_XErr	call	I_Dspl		;Display desired XMS error message.
	mov	dx,NBMsg	;Display "no buffered I-O" message.
	call	I_Dspl
	mov	ax,(Pass-GoToBuf-3) ;Reject buffered I-O using a
	mov	[GoToBuf+1],ax	    ;  "dirty-nasty" code change!
	mov	ax,(NoXMEnd-@)	;Reduce resident driver size.
	mov	[VDSLn],ax
I_Stop	mov	dx,[DMAAd]	;Ensure any previous DMA is stopped.
	in	al,dx		;(See "DoDMA" routine notes below).
	and	al,0FEh
	out	dx,al
	add	dx,byte 8	;Stop secondary-channel DMA, also!
	in	al,dx
	and	al,0FEh
	out	dx,al
	xor	eax,eax		;Zero EAX-reg. for 20-bit addressing.
	mov	es,ax		;Point ES-reg. to low memory.
	or	al,[es:HDISKS]	;Did BIOS find any hard-disks?
	jz	near I_RScn	;No?  Display "No disk" and exit!
	mov	[@BIOSHD],al	;Set BIOS hard-disk count below.
	mov	ax,cs		;Set our code segment in VDS block.
	mov	[VDSSg],ax
	shl	eax,4		;Get 20-bit driver virtual address.
	cli			;Avoid interrupts during VDS tests.
	test	byte [es:VDSFLAG],020h ;Are "VDS services" active?
	jz	I_SetA		;No, set 20-bit virtual addresses.
	mov	ax,08103h	;"Lock" this driver into memory.
	mov	dx,0000Ch
	call	VDLock
	jc	near I_VErr	;Error?  Display error msg. & exit!
	inc	byte [VDSOf]	;Set initialization VDS "lock" flag.
	mov	eax,[IOAdr]	;Get 32-bit starting driver address.
I_SetA	sti			;Re-enable CPU interrupts.
	add	[PRDAd],eax	;Set relocated 32-bit PRD address.
	mov	ah,041h		;See if we have an EDD BIOS.
	mov	bx,055AAh
	mov	dl,080h
	int	013h
	push	cs
	pop	ds
	jc	I_Scan		;No, scan for disks without EDD.
	cmp	bx,0AA55h	;Did BIOS "reverse" our entry code?
	jne	I_Scan		;No, scan for disks without EDD.
	xchg	ax,cx		;Set "EDD BIOS present" flag.
	and	al,004h
	mov	[EDDFlag],al
	jmp	I_Scan		;Go scan for UltraDMA disks to use.
I_RScn	mov	dx,NDMsg	;Point to "No disk to use" message.
	xor	ax,ax		;Load & reset EDD BIOS flag.
	xchg	al,[EDDFlag]
	or	ax,ax		;Were we scanning v.s. DPTE data?
	jz	near I_Err	;No?  Display "No disk" and exit!
I_Scan	mov	ax,00080h	;Reset hard-disk unit number & index.
	mov	[HDUnit],ax
	mov	byte [HDCount],0  ;Reset remaining hard-disk count.
@BIOSHD equ	$-1		;(BIOS hard-disk count, set above).
I_Next	movzx	bx,[HDIndex]	;Get disk unit-number index.
	cmp	bh,[EDDFlag]	;Are we using DPTE data from BIOS?
	je	I_ChMS		;No, check disk at "fixed" addresses.
	mov	ah,048h		;Get next BIOS disk's EDD parameters.
	mov	dl,[HDUnit]
	mov	si,EDDBuff
	int	013h
	push	cs
	pop	ds
	jc	I_ErEDD		;Error?  Ignore this BIOS unit!
	cmp	dword [si+26],byte -1  ;Valid DPTE pointer?
	je	near I_More	;No, ignore unit & check for more.
	les	si,[si+26]	;Get this disk's DPTE pointer.
	mov	bx,15		;Calculate DPTE checksum.
	xor	cx,cx
I_CkSm	add	cl,[es:bx+si]
	dec	bx
	jns	I_CkSm
	jcxz	I_EDOK		;If checksum O.K., use parameters.
I_ErEDD	mov	dx,EBMsg	;Display "EDD error" and ignore unit!
	jmp	short I_ErrD
I_EDOK	mov	bx,00010h	;Initialize IDE unit number index.
	and	bl,[es:si+4]
	shr	bl,4
	mov	ax,[es:si]	;Get disk's IDE base address.
	cmp	ax,CDATA	;Is this a primary-channel disk?
	je	I_Indx		;Yes, set disk unit-number index.
	cmp	ax,(CDATA-080h)	;Is this a secondary-channel disk?
	jne	I_More		;No, ignore unit & check for more.
	inc	bx		;Adjust for secondary channel.
	inc	bx
I_Indx	mov	[HDIndex],bl	;Set disk's unit number index.
I_ChMS	mov	ax,bx		;Separate channel and master/slave.
	shr	al,1
	mov	ah,(LBABITS/32)	;Get drive-select "nibble".
	rcl	ah,5
	ror	al,1		;Set channel offset (2nd = 080h).
	mov	[@HDOffs],al
	mov	dx,CDSEL	;Get IDE disk-select address.
	xor	dl,al
	mov	al,ah		;Select master or slave disk.
	out	dx,al
	shl	bx,2		;Save disk's "Units" table index.
	push	bx
	shr	bx,1		;Get "channel name" message index.
	mov	dx,[bx+HDNames] ;Display disk's IDE channel name.
	call	I_Dspl		;("Primary master", etc.).
	mov	ah,008h		;Get BIOS CHS values for this disk.
	mov	dl,[HDUnit]
	int	013h
	push	cs
	pop	ds
	xchg	ax,dx		;Set AX-reg. with head-number value.
	pop	bx		;Reload disk's unit number index.
	mov	dx,NPMsg	;Point to "parameter ERROR" message.
	jc	I_ErrD		;Error -- display msg. & ignore disk.
	mov	al,cl		;Save sectors per head value.
	and	al,03Fh		;(Clear 2 high-order cylinder bits).
	mov	[bx+Units+1-@],al
	inc	ah		;Get heads/cylinder (BIOS value + 1).
	mul	ah		;Save sectors per cylinder value.
	mov	[bx+Units+2-@],ax
	mov	al,[HDUnit]	;Activate this disk in main driver.
	mov	[bx+Units-@],al
	push	bx		;Validate this UltraDMA disk.
	call	I_ValD
	pop	bx
	jnc	I_More		;If no errors, check for more disks.
	mov	byte [bx+Units-@],0FFh  ;DELETE disk in main driver!
I_ErrD	call	I_Dspl		;Display error for this disk.
	mov	dx,CRMsg	;Display error-message suffix.
	call	I_Dspl
I_More	add	word [HDUnit],00101h  ;Bump BIOS unit & disk index.
	cmp	word [EDDFlag],08400h ;No EDD and all 4 units tested?
	je	I_AnyD		;Yes, see if we found any disks.
	dec	byte [HDCount]	;More BIOS disks to check?
	jnz	I_Next		;Yes, loop back and do next one.
I_AnyD	mov	bx,16		;Set up to scan for last unit.
I_ChkU	sub	bx,byte 4	;Any more active units to check?
	js	near I_RScn	;No, see if we should do a re-scan.
	cmp	byte [bx+Units-@],0FFh  ;Is this unit active?
	je	I_ChkU		;No, loop back and check next unit.
	add	bx,byte 4	;Post last-unit index in main driver.
	mov	[@LastU],bx
	mov	ax,03513h	;Get current Int 13h vector.
	call	I_Int21
	push	es		;Save previous Int 13h vector.
	push	bx
	pop	dword [@PrvI13]
	mov	ax,02513h	;"Hook" this driver into Int 13h.
	mov	dx,(Entry-@)
	call	I_Int21
	xor	ax,ax		;Load & reset driver length.
	xchg	ax,[VDSLn]
	mov	dx,RPDON	;Go post "success" and exit.	
	jmp	short I_Exit
I_VErr	mov	dx,VEMsg	;VDS "lock" error!  Point to message.
I_Err	push	dx		;Save error message pointer.
	shr	byte [VDSOf],1	;Was driver "locked" by VDS?
	jnc	I_XUnl		;No, get rid of our XMS memory.
	call	VDUnlX		;"Unlock" this driver from memory.
I_XUnl	mov	cx,[XMHdl]	;Did we reserve any XMS memory?
	jcxz	I_LdMP		;No, display desired error message.
	mov	ah,00Dh		;Unlock our XMS memory buffer.
	mov	dx,cx
	push	dx
	call	XMCall
	mov	ah,00Ah		;Free our XMS memory buffer.
	pop	dx
	call	XMCall
	mov	ah,006h		;Do local-disable of "A20 line".
	call	XMCall
I_LdMP	pop	dx		;Reload error message pointer.
I_EOut	call	I_Dspl		;Display error message and suffix.
	mov	dx,Suffix
	call	I_Dspl
I_BadP	xor	ax,ax		;Get "null" length & error flags.
	mov	dx,RPDON+RPERR
I_Exit	les	bx,[Packet]	;Post results in "init" packet.
	mov	[es:bx+RPSize],ax
	mov	[es:bx+RPSize+2],cs
	mov	[es:bx+RPStat],dx
	pop	es		;Reload all CPU registers and exit.
	pop	ds
	popad
	popf
	retf
;
; Subroutine to "validate" an UltraDMA hard-disk.
;
I_ValD	mov	al,0ECh		;Issue "Identify Device" command.
	call	I_Cmd
	jc	I_DErr		;If error, exit & display message!
	add	dx,byte (CDATA-CCMD)  ;Point to PIO data register.
	in	ax,dx		;Read I.D. bytes 0 and 1.
	xchg	ax,si		;Save "ATA/ATAPI" flag word.
	mov	cx,52		;Skip I.D. bytes 2-105.
I_Skp1	in	ax,dx
	loop	I_Skp1
	in	ax,dx		;Read I.D. bytes 106 and 107.
	mov	bh,al		;Save "DMA valid" flag byte.
	mov	cl,34		;Skip I.D. bytes 108-175.
I_Skp2	in	ax,dx
	loop	I_Skp2
	in	ax,dx		;Read I.D. bytes 176 and 177.
	mov	bl,ah		;Save "UltraDMA selected" flag byte.
	mov	cl,167		;Skip remaining I.D. data.
I_Skp3	in	ax,dx
	loop	I_Skp3
	shl	si,1		;Is this an "ATA" hard-disk?
	jc	I_DErr		;No?  Exit & display message!
	test	bh,004h		;Are UltraDMA flag bits valid?
	jz	I_DErr		;No?  Exit & display message!
	mov	di,Modes	;Point to UltraDMA mode table.
	mov	al,'0'		;Initialize "current mode" value.
	mov	cl,002h		;Set rotating mode-check bit.
	cmp	bl,001h		;Will disk do UltraDMA mode 0?
	jae	I_NxtM		;Yes, find its best UltraDMA mode.
I_DErr	mov	dx,DEMsg	;Not an UltraDMA disk!   Point to msg.
I_SErr	stc			;Set carry flag (error!) and exit.
	ret
I_NxtM	mov	[CurMode],al	;Update "current mode" value in message.
	cmp	bl,cl		;Will disk do next UltraDMA mode?
	jb	I_GotM		;No, use current mode.
	inc	ax		;Set up for next UltraDMA mode.
	add	di,byte 4
	shl	cl,1		;More UltraDMA modes to check?
	jnz	I_NxtM		;Yes, loop back.
I_GotM	push	ax		;Save "current mode" value.
	mov	eax,[di]	;Set UltraDMA mode in set-mode message.
	mov	[DMode],eax
	inc	dx		;Set mode-select subcode.
	mov	al,SETM
	out	dx,al
	pop	ax		;Set desired UltraDMA mode.
	add	al,010h
	inc	dx
	out	dx,al
	mov	al,SETF		;Issue set-features command to disk.
	call	I_Cmd
	mov	dx,SEMsg	;Point to "Set-mode" error message.
	jc	I_SErr		;If error, set carry flag and exit.
	mov	dx,MSMsg	;Display "Set to mode" message.
I_Dspl	mov	ah,009h
I_Int21	int	021h
	push	cs		;Reload our DS-register.
	pop	ds
	clc			;Clear carry (no errors!) and exit.
	ret
;
; Subroutine to issue disk initialization commands.
;
I_Cmd	mov	dx,CCMD		;Issue desired init command.
	xor	dl,0
@HDOffs	equ	$-1
	out	dx,al
	xor	si,si		;Point to low-memory BIOS timer.
	mov	es,si
	mov	si,BIOSTMR
	mov	cl,RDYTO	;Set I-O timeout limit in CL-reg.
	add	cl,[es:si]
I_CmdW	cmp	cl,[es:si]	;Has our command timed out?
	je	I_CmdE		;Yes, set CPU carry flag & exit.
	in	al,dx		;Get IDE controller status.
	test	al,BSY+RDY	;Controller or disk still busy?
	jle	I_CmdW		;Yes, loop back and check again.
	test	al,ERR		;Did command cause any errors?
	jz	I_CmdX		;No, leave carry flag off & exit.
I_CmdE	stc			;Error!  Set CPU carry flag.
I_CmdX	ret			;Exit.
;
; Initialization Messages.
;
UDMsg	db	CR,LF,'UDMAJR Disk Driver ',VER,CR,LF,'$'
PEMsg	db	'PCI BIOS below V2.0C$'
NEMsg	db	'No controller found$'
MEMsg	db	'Bus-Master setup BAD$'
NXMsg	db	'No XMS manager$'
XEMsg	db	'XMS init error$'
NBMsg	db	'; using only DMA I-O!',CR,LF,'$'
VEMsg	db	'VDS lock error$'
PCMsg	db	'UltraDMA controller at I-O address '
DspAd	db	'0000h'
CRMsg	db	'.',CR,LF,'$'
NDMsg	db	'No disks to use$'
EBMsg	db	'EDD error!  BIOS unit ignored$'
PMMsg	db	'Primary-master $'
PSMsg	db	'Primary-slave $'
SMMsg	db	'Secondary-master $'
SSMsg	db	'Secondary-slave $'
MSMsg	db	'disk set to UltraDMA mode '
CurMode db	'0, ATA-'
DMode	db	'16. ',CR,LF,'$'
DEMsg	db	'is not UltraDMA$'
NPMsg	db	'CHS-values ERROR$'
SEMsg	db	'Set-Mode error$'
Suffix	db	'; driver NOT loaded!',CR,LF,'$'
