.include "xms.inc"

		.struct sh_handle_cache
handle	.dw			; search handle number
offset	.dw			; offset of search handle in cache (relative to segment)
		.ends

sh_cache_size = 8						; 8 search handles cached in low memory
search_handle_cache:
		.db search_handle * sh_cache_size .dup 0
cached_search_handles:
		.db sh_handle_cache * sh_cache_size .dup 0
mru_search_handle .struct sh_handle_cache (. - sh_handle_cache)
cache_xms_handle:
		.dw		0
num_search_handles:
		.dd		1024

init_lfn_find:
		push	es
		; initialize the handle offsets in the list of cached search handles
		push	cs
		pop		es
		mov		cx,sh_cache_size
		mov		di,cached_search_handles
		mov		eax,(search_handle_cache<<16)+1	; eax(0-15) will hold handle number
		cld
	?set_cache_list_loop:
		stosd
		add		eax,(search_handle<<16)+1
		loop	?set_cache_list_loop

		call	xms_init
		jc		?no_xms
		mov		eax,search_handle
		imul	eax,[num_search_handles]
		add		eax,1024-1
		shr		eax,10
		call	xms_allocate_block
		jc		?no_xms
		mov		[cache_xms_handle],ax
		mov		cx,[num_search_handles]
		xor		bx,bx
		mov		si,search_handle_cache
		mov		dx,ax
		xor		edi,edi
	?clear_xms_loop:
		push	ecx
		mov		ecx,search_handle
		call	xms_move_block
		add		edi,ecx
		pop		ecx
		loop	?clear_xms_loop
		jmp		?exit
	?no_xms:
		movw	[num_search_handles],sh_cache_size
	?exit:
		pop		es
		ret
; end of init_lfn_find

uninit_lfn_find:
		mov		ax,[cache_xms_handle]
		or		ax,ax
		jz		?exit
		call	xms_free_block
	?exit:
		ret
; end of uninit_lfn_find

shift_cached_handles:
		; shifts the cached search handles down one so the top one can be
		; the most recently used one. Copies the overwritten handle to the
		; top.
		; in
		;	al=index of element to be moved to the top
		; out
		;	ax=offset of search handle
		;	dx=search handle number
		;	cx=0
		push	si
		push	di
		movzx	eax,al
		lea		di,[eax*sh_handle_cache+cached_search_handles]
		lea		si,[di+sh_handle_cache]
		mov		cx,sh_cache_size-1
		sub		cl,al
		mov		dx,[di+sh_handle_cache.handle]
		mov		ax,[di+sh_handle_cache.offset]
		rep
		movsd
		mov		[di+sh_handle_cache.handle],dx
		mov		[di+sh_handle_cache.offset],ax
		pop		di
		pop		si
		ret
; end of shift_cached_handles

read_search_handle:
		; Retrieves the search handle from extended memory
		; in
		;	ax=offset of search handle
		;	dx=search handle number
		; out
		;	dx=0
		;	ecx=size of search handle
		;	ax=0 (if everything went ok with the move)
		push	esi
		push	di
		push	bx
		push	es
		push	cs
		pop		es
		; setup the source offset (bx:esi (bx=xms handle))
		movzx	esi,dx
		dec		esi
		mov		ecx,search_handle
		imul	esi,ecx
		; setup the destination offset (es:edi)
		movzx	edi,ax
		; setup the xms source and destination handles handles
		mov		bx,[cache_xms_handle]	; source handle
		xor		dx,dx					; destination handle (use es)
		call	xms_move_block
		pop		es
		pop		bx
		pop		di
		pop		esi
		ret
; end of read_search_handle

write_search_handle:
		; Retrieves the search handle from extended memory
		; in
		;	ax=offset of search handle
		;	dx=search handle number
		; out
		;	dx=xms handle
		;	ecx=size of search handle
		;	ax=0 (if everything went ok with the move)
		push	esi
		push	di
		push	bx
		; setup the destination offset (dx:edi (dx=xms handle))
		movzx	edi,dx
		dec		edi
		mov		ecx,search_handle
		imul	edi,ecx
		; setup the source offset (ds:esi)
		movzx	esi,ax
		; setup the xms source and destination handles handles
		mov		dx,[cs:cache_xms_handle]	; source handle
		xor		bx,bx					; destination handle (use ds)
		call	xms_move_block
		pop		bx
		pop		di
		pop		esi
		ret
; end of write_search_handle

find_handle_by_number:
		; Searches for the specified search handle number in the list of
		; cached search handles
		; in
		;	bx=search handle number
		; out
		;	al=index into list of cached search handles. Will be 0
		;		if the handle isn't found.
		;	cf clear=handle found
		;	cf set=handle not found in list of cached search handles
		push	di
		mov		di,cached_search_handles+sh_handle_cache*(sh_cache_size-1)
		mov		al,sh_cache_size-1
	?try_next_handle:
		cmp		[di+sh_handle_cache.handle],bx
		je		?exit					; cmp above clears carry
		sub		di,sh_handle_cache
		sub		al,1					; will set carry if al was 0
		jnc		?try_next_handle
		inc		al						; does not affect carry
	?exit:
		pop		di
		ret
; end of find_handle_by_number

find_handle_by_offset:
		; Searches for the specified search handle offset in the list of
		; cached search handles
		; in
		;	bx=search handle offset
		; out
		;	al=index into list of cached search handles. Will be 0
		;		if the handle isn't found.
		;	cf clear=handle found
		;	cf set=handle not found in list of cached search handles
		push	di
		mov		di,cached_search_handles+sh_handle_cache*(sh_cache_size-1)
		mov		al,sh_cache_size-1
	?try_next_handle:
		cmp		[di+sh_handle_cache.offset],bx
		je		?exit					; cmp above clears carry
		sub		di,sh_handle_cache
		sub		al,1					; will set carry if al was 0
		jnc		?try_next_handle
		inc		al						; does not affect carry
	?exit:
		pop		di
		ret
; end of find_handle_by_offset

access_search_handle:
		; Makes sure the specified search handle is in low memory. As a side
		; effect, converts the handle number to an offset usable by the rest
		; of the system. Uses a LRU style caching algorithm.
		; in
		;	bx=handle number
		; out
		;	bx=handle offset
		cmp		bx,1
		jb		?return
		cmp		[num_search_handles],bx
		jb		?return
		; see if the desired handle is in low memory
		call	find_handle_by_number
		pushf
		; make it the most recently used handle (even if it's not in low
		; memory)
		; al holds the index
		call	shift_cached_handles
		popf
		jnc		?exit					; search handle number found
		; Bring the handle into low memory. If the cached handle number is 0,
		; the slot is free and does not need to be written back to extended
		; memory.
		cmpw	[mru_search_handle.handle],0
		je		?never_used
		; Save the `unwanted' search handle data.
		mov		dx,[mru_search_handle.handle]
		mov		ax,[mru_search_handle.offset]
		call	write_search_handle
	?never_used:
		; Retrieve the `wanted' search handle data.
		mov		dx,[mru_search_handle.handle]
		mov		ax,[mru_search_handle.offset]
		mov		[mru_search_handle.handle],bx
		call	read_search_handle
	?exit:
		mov		bx,[mru_search_handle.offset]
	?return:
		ret
; end of access_search_handle

get_current_psp:
		; Retrieve the current psp (PID) from dos
		; out
		;	ax=psp segment
		push	bx
		mov		ah,0x51
		int		0x21
		mov		ax,bx
		pop		bx
		ret
; end of get_current_psp		

allocate_search_handle:
		; get a new search handle
		; out
		;	bx=search handle offset
		;	ax=search handle number
		push	si
		push	dx
		; first, see if there is a free handle in low memory
		mov		bx,search_handle_cache
		mov		cx,sh_cache_size
	?find_low_handle_loop:
		cmpw	[bx+search_handle.app_psp],0
		je		?free_low_handle_found
		add		bx,search_handle
		loop	?find_low_handle_loop
		mov		si,1
		mov		cx,[num_search_handles]
	?find_free_handle_loop:
		mov		bx,si
		call	access_search_handle
		cmpw	[bx+search_handle.app_psp],0
		je		?free_handle_found
		inc		si
		loop	?find_free_handle_loop
		; No free handles
		xor		bx,bx					; zero handle offset
		xor		ax,ax					; zero handle number
		stc
		jmp		?exit
	?free_low_handle_found:
		call	find_handle_by_offset
		call	shift_cached_handles
		; ax=handle offset, dx=handle number
		mov		si,dx
		; bx already has handle offset
	?free_handle_found:
		; A free handle has been found
		call	get_current_psp
		mov		[bx+search_handle.app_psp],ax
		mov		ax,si					; return handle number in ax
		clc
	?exit:
		pop		dx
		pop		si
		ret
; end of allocate_search_handle

free_search_handle:
		; in
		;	bx=handle number
		call	access_search_handle
		jc		?exit
		; free the handle
		movw	[bx+search_handle.app_psp],0
		mov		ax,[mru_search_handle.offset]
		mov		dx,[mru_search_handle.handle]
		; write the freed search handle data out to extended memory
		call	write_search_handle
		clc
	?exit:
		ret
; end of free_search_handle

		.struct FindData
attr	.dd				; file attributes
cr_time	.dd				; creation time (QWORD)
		.dd
la_time	.dd				; last access time (QWORD)
		.dd
mo_time	.dd				; last modification time (QWORD)
		.dd
fsize_h	.dd				; High 32 bits of file size
fsize	.dd				; low 32 bits of file size
		.db 8 .dup		; reserved
lfn		.db 260 .dup	; full file name
sfn		.db 14 .dup		; short file name
		.ends	; FindData

convert_dos_time:
		; in
		;	eax=dos date/time
		;	dl=10ms units past eax
		;	cx=conversion flag
		; out
		;	si!=0
		;		edx=0
		;		eax=dos date/time
		;	si==0
		;		edx:eax=number of 100ns intervals since 1601/1/1
		cmp		si,0					; time convertion flag
		jnel	?use_dos_time
		; 100ns intervals since 1601/1/1! (ugh, how long ago was that!?)
		;Bit(s)	Description	(Table 1351)
		; 15-11	hours (0-23)
		; 10-5	minutes
		; 4-0	seconds/2
		;
		;Bitfields for file date:
		;Bit(s)	Description	(Table 1352)
		; 15-9	year - 1980
		; 8-5	month
		; 4-0	day
		push	ecx
		movzx	ecx,dl
		push	eax						; save date and time
		and		eax,0x1f				; extract seconds/2
		; need to multiply eax by 200 (5*5*8) to get 10ms units
		shl		eax,3					; eax*=8
		lea		eax,[eax+eax*4]			; eax*=5 (40)
		lea		eax,[eax+eax*4]			; eax*=5 (200)
		add		ecx,eax
		; add in the minutes
		movzxw	eax,[esp]
		and		ax,0x07e0				; extract minutes * 32
		; need to multiply minutes by 6000 (5*5*5*3*16), but first need to
		; divide eax by 32, so shift right 1 (divide by 2) and then multiply
		; by 375 (5*5*5*3), for a total of 187.5
		shr		eax,1					; eax/=2 (0.5)
		lea		eax,[eax+eax*4]			; eax*=5 (2.5)
		lea		eax,[eax+eax*4]			; eax*=5 (12.5)
		lea		eax,[eax+eax*4]			; eax*=5 (62.5)
		lea		eax,[eax+eax*2]			; eax*=3 (187.5)
		add		ecx,eax
		; add in the hours
		movzxb	eax,[esp+1]
		and		al,0xf8
		; need to multiply hours by 360000 (5*5*5*5*9*64) but first need to
		; divide eax by 8, so shift left 3 (multiply by 8) and then multiply
		; by 5625 (5*5*5*5*9), for a total of 45000
		shl		eax,3					; eax*=8 (8)
		lea		eax,[eax+eax*4]			; eax*=5 (40)
		lea		eax,[eax+eax*4]			; eax*=5 (200)
		lea		eax,[eax+eax*4]			; eax*=5 (1000)
		lea		eax,[eax+eax*4]			; eax*=5 (5000)
		lea		eax,[eax+eax*8]			; eax*=9 (45000)
		add		ecx,eax
		; ecx now holds the time of day in 10ms units

		; get number of days since 1601/1/1
		movzxb	eax,[esp+3]
		shr		ax,1
		add		ax,1980-1601
		imul	edx,eax,97
		imul	eax,eax,365*400
		add		eax,edx
		xor		edx,edx
		push	ecx
		mov		ecx,400
		div		ecx
		pop		ecx
		mov		edx,eax
		mov		ax,[esp+2]
		shr		ax,5
		and		eax,0x0f
		dec		eax						; eax holds month (jan=0)
		add		edx,[?month_start+eax*2]; add in start of month
		pop		eax
		shr		eax,16
		and		al,0x1f
		dec		al
		add		edx,edx
		; edx now holds the number of days since 1601/1/1
		mov		eax,24*3600*100			; 10ms intervals in a day
		mul		edx
		add		eax,ecx
		adc		edx,0
		; edx:eax now holds number of 10ms intervals since 1601/1/1 0:00:00
		push	edx
		mov		ecx,10*1000*10			; convert from 10ms to 100ns
		mul		ecx
		pop		ecx
		imul	ecx,ecx,10*1000*10
		add		edx,ecx
		; edx:eax now holds number of 100ns intervals since 1601/1/1
		pop		ecx
		jmp		?exit
	?month_start:
		;		jan feb mar apr may jun jul aug sep oct nov dec
		.dd		0  ,31 ,59 ,90 ,120,151,181,212,243,273,304,334
	?use_dos_time:
		xor		edx,edx
	?exit:
		ret
; end of convert_dos_time

do_file_search:
		; Search for the file spec in the search handle. Then copies the file
		; data to the application's FindData record;
		; in
		;	bx=search handle
		; out
		;	ax=error code if cf set
		;	cx=unicode conversion flags if cf set (always 0 for now)
		;	cf set=not found
		;	cf clear=found
		push	si
		push	es
		push	di
		mov		dl,[lfn_only_flag]		; matching with lfn only is optional
		call	scan_directory
		mov		si,ax					; save dirent address
		mov		ax,DE_file_not_found
		jcl		?exit
		mov		es,[bp+sf.ES]
		mov		di,[bp+sf.DI]
		; copy file attributes
		movzxb	eax,[si+dirent.attr]
		mov		[es:di+FindData.attr],eax
		; copy file times
		mov		cx,[bp+sf.SI]			; date/time conversion flag
		cmpw	[dos_version],0x0700	; W95 is ms-dos 7.x
		jae		?copy_extra_time_fields
		; use modification time for both creation time and last access time
		mov		eax,[si+dirent.time]	; grabs date as well
		mov		dl,0
		call	convert_dos_time
		mov		[es:di+FindData.cr_time],eax
		mov		[es:di+FindData.cr_time+4],edx
		mov		[es:di+FindData.la_time],eax
		mov		[es:di+FindData.la_time+4],edx
		mov		[es:di+FindData.mo_time],eax
		mov		[es:di+FindData.mo_time+4],edx
		jmp		?time_fields_copied
	?copy_extra_time_fields:
		; copy creation time
		mov		eax,[si+dirent.cr_time]	; grabs date as well
		mov		dl,[si+dirent.crt_cs]	; 10ms units
		call	convert_dos_time
		mov		[es:di+FindData.cr_time],eax
		mov		[es:di+FindData.cr_time+4],edx
		; copy last access time (date, actually)
		mov		ax,[si+dirent.la_date]
		shl		eax,16
		mov		dl,0
		call	convert_dos_time
		mov		[es:di+FindData.la_time],eax
		mov		[es:di+FindData.la_time+4],edx
		; copy last modification time
		mov		eax,[si+dirent.time]	; grabs date as well
		mov		dl,0
		call	convert_dos_time
		mov		[es:di+FindData.mo_time],eax
		mov		[es:di+FindData.mo_time+4],edx
	?time_fields_copied:
		; copy file size
		mov		eax,[si+dirent.size]
		mov		[es:di+FindData.fsize],eax
		movd	[es:di+FindData.fsize_h],0	; is this correct for fat32???
		; copy file names
		mov		si,mde_long_name
		cmpb	[si],0
		jne		?has_long_name
		mov		si,mde_short_name			; some extra stuff will be copied
	?has_long_name:
		mov		cx,260/4
		lea		di,[di+FindData.lfn]
		rep
		movsd
		mov		si,mde_short_name
		; di already points to the sfn field
		mov		cx,14/2
		rep
		movsw
		; cx is 0, lie about unicode conversion problems
	?exit:
		pop		di
		pop		es
		pop		si
		ret
; end of do_file_search

lfn_find_first:
		mov		ax,DE_path_not_found
		mov		si,dx					; search spec at ds:dx
		cmpb	[si],0
		jel		?error
		push	cs
		pop		es
		mov		di,filename_buffer
		mov		cx,260
		call	strncpy
		push	cs
		pop		ds
		movb	[di+260],0
		; sort out the path to the directory to search
		call	allocate_search_handle
		mov		ax,DE_no_more_handles
		jc		?error
		mov		cx,[bp+sf.CX]
		call	resolve_path
		jc		?invalid_path
		lea		di,[bx+search_handle.spec]
		call	strcpy
		; now try to find the file(s)
		call	do_file_search
		jc		?file_not_found
		mov		ax,[mru_search_handle.handle]
		mov		[bp+sf.AX],ax			; Search handle number
		mov		[bp+sf.CX],cx			; Unicode conversion flags
		jmp		?exit
	?invalid_path:
		mov		ax,DE_path_not_found
		jmp		?error
	?file_not_found:
		mov		bx,[mru_search_handle.handle]
		call	free_search_handle
		mov		ax,DE_file_not_found
	?error:
		mov		[bp+sf.AX],ax
		orb		[bp+sf.FLAGS],1
	?exit:
		ret
; end of lfn_find_first

lfn_find_next:
		; in
		;	bx=search handle number
		;	si=date flags
		;	es:di=FindData buffer
		push	cs
		pop		ds
		call	access_search_handle	; bx already holds search handle number
		jb		?invalid_handle
		call	get_current_psp
		cmp		[bx+search_handle.app_psp],ax
		jne		?invalid_handle
		call	do_file_search
		jc		?no_more_files
		mov		[bp+sf.CX],cx			; Unicode conversion flags
		jmp		?exit
	?no_more_files:
		cmp		ax,DE_file_not_found
		jne		?error
		mov		ax,DE_no_more_files
		jmp		?error
	?invalid_handle:
		mov		ax,DE_invalid_handle
	?error:
		mov		[bp+sf.AX],ax
		orb		[bp+sf.FLAGS],1
	?exit:
		ret
; end of lfn_find_next

lfn_find_close:
		; in
		;	bx=search handle number
		push	cs
		pop		ds
		push	bx
		call	access_search_handle
		mov		si,bx
		pop		bx
		jc		?invalid_handle
		call	get_current_psp
		cmp		[si+search_handle.app_psp],ax
		jne		?invalid_handle
		call	free_search_handle		; bx still holds search handle number
		jnc		?exit
	?invalid_handle:
		movw	[bp+sf.AX],DE_invalid_handle
		orb		[bp+sf.FLAGS],1
	?exit:
		ret
; end of lfn_find_close
