; Cute Mouse Driver - a tiny mouse driver
; Copyright (c) 1997-2002 Nagy Daniel <nagyd@users.sourceforge.net>
;
; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;

CTMVER		equ	"1.9"
CTMRELEASE	equ	"1.9 beta 1"
driverversion	equ	705h	; Microsoft driver version

WARN
%NOINCL
.model tiny
;.286
;.386

FASTER_CODE	 = 0		; Optimize by speed instead size
OVERFLOW_PROTECT = 0		; Prevent internal variables overflow
FOOLPROOF	 = 1		; Check passed to driver arguments correctness

USE_286 equ (@CPU and 4)
USE_386 equ (@CPU and 8)

_ARG_DI_	equ	word ptr [bp]
_ARG_SI_	equ	word ptr [bp+2]
_ARG_BP_	equ	word ptr [bp+4]

IF USE_286

_ARG_BX_	equ	word ptr [bp+8]
_ARG_DX_	equ	word ptr [bp+10]
_ARG_CX_	equ	word ptr [bp+12]
_ARG_AX_	equ	word ptr [bp+14]
_ARG_ES_	equ	word ptr [bp+16]
_ARG_DS_	equ	word ptr [bp+18]
_ARG_OFFS_	equ	24

PUSHALL		equ	pusha
POPALL		equ	popa

ELSE ;---------- USE_286

_ARG_BX_	equ	word ptr [bp+6]
_ARG_DX_	equ	word ptr [bp+8]
_ARG_CX_	equ	word ptr [bp+10]
_ARG_AX_	equ	word ptr [bp+12]
_ARG_ES_	equ	word ptr [bp+14]
_ARG_DS_	equ	word ptr [bp+16]
_ARG_OFFS_	equ	22

PUSHALL		equ	push	ax cx dx bx bp si di
POPALL		equ	pop	di si bp bx dx cx ax

ENDIF ;---------- USE_286

j		equ	jmp short

movSeg		macro	dest,src
		push	src
		pop	dest
	endm

saveFAR		macro	addr,segm,offs
		mov	word ptr addr[0],offs
		mov	word ptr addr[2],segm
	endm

PS2serv		macro	serv,errlabel
		mov	ax,serv
		int	15h
	IFNB <errlabel>
		jc	errlabel
		or	ah,ah
		jnz	errlabel
	ENDIF
	endm

PrintS		macro	addr
	IFNB <addr>
	 IFDIFI <addr>,<dx>
		mov	dx,offset addr
	 ENDIF
	ENDIF
		mov	ah,9
		int	21h
	endm

POINT		struc
  X		dw	0
  Y		dw	0
	ends

.code
		org	0
TSRstart	label
		org	100h
start:		jmp	real_start

; UNINITIALIZED DATA 
;!!! WARNING: don't init variables in uninitialized data section because
;		it not presented in executable image

		org	4Eh		; reuse part of PSP
TSRdata?	label

;----- application state begins here -----
;!!! WARNING: variables order between StartSaveArea and LenDefArea must be
;		syncronized with variables order after StartDefArea

		evendata
StartSaveArea = $

mickey8		POINT	?		; mickeys per 8 pixel ratios
;;+*doublespeed	dw	?		; double-speed threshold (mickeys/sec)
startscan	dw	?		; screen mask/cursor start scanline
endscan		dw	?		; cursor mask/cursor end scanline
;---------- hotspot, screenmask and cursormask must follow as is
hotspot		POINT	?		; cursor bitmap hot spot
screenmask	db	2*16 dup (?)	; user defined screen mask
cursormask	db	2*16 dup (?)	; user defined cursor mask
cursorhidden	db	?		; 0 - cursor visible, else hidden
;;+*nolightpen?	db	?		; 0 - emulate light pen
		evendata
LenDefArea = $ - StartSaveArea		; initialized by softreset_21

rangemax	POINT	?		; horizontal/vertical range max
upleft		POINT	?		; upper left of update region
lowright	POINT	?		; lower right of update region
pos		POINT	?		; virtual cursor position
granpos		POINT	?		; granulated virtual cursor position
UIR@		dd	?		; user interrupt routine address

		evendata
StartClearArea = $

rounderr	POINT	?		; rounding error after mickeys->pixels
		evendata
LenClearArea1 = $ - StartClearArea	; cleared by setpos_04

rangemin	POINT	?		; horizontal/vertical range min
		evendata
LenClearArea2 = $ - StartClearArea	; cleared by setupvideo

cursortype	db	?		; 0 - software, else hardware
callmask	db	?		; user interrupt routine call mask
mickeys		POINT	?		; mouse move since last access
BUTTLASTSTATE	struc
  counter	dw	?
  lastrow	dw	?
  lastcol	dw	?
	ends
buttpress	BUTTLASTSTATE	?,?,?
buttrelease	BUTTLASTSTATE	?,?,?
		evendata
LenClearArea3 = $ - StartClearArea	; cleared by softreset_21
LenSaveArea = $ - StartSaveArea

;----- registers values for RIL -----
;!!! WARNING: registers order and RGROUPDEF contents must be fixed

		evendata
StartVRegsArea = $

regs_SEQC	db	5 dup (?)
reg_MISC	db	?
regs_CRTC	db	25 dup (?)
regs_ATC	db	21 dup (?)
regs_GRC	db	9 dup (?)
reg_FC		db	?
reg_GPOS1	db	?
reg_GPOS2	db	?
		evendata
LenVRegsArea = $ - StartVRegsArea

		evendata
StartDefVRegsArea = $

def_SEQC	db	5 dup (?)
def_MISC	db	?
def_CRTC	db	25 dup (?)
def_ATC		db	21 dup (?)
def_GRC		db	9 dup (?)
def_FC		db	?
def_GPOS1	db	?
def_GPOS2	db	?
		evendata
ERRIF ($-StartDefVRegsArea ne LenVRegsArea) "VRegs area contents corrupted!"

;----- old interrupt vectors -----

		evendata
oldint33	dd	?		; old INT 33 handler address
oldIRQaddr	dd	?		; old IRQ handler address


; INITIALIZED DATA 

		evendata
TSRdata		label

StartDefArea = $
		POINT	<8,16>			; mickey8
;;+*		dw	64			; doublespeed
		dw	77FFh,7700h		; startscan, endscan
		POINT	<0,0>			; hotspot
		dw	0011111111111111b	; screenmask
		dw	0001111111111111b
		dw	0000111111111111b
		dw	0000011111111111b
		dw	0000001111111111b
		dw	0000000111111111b
		dw	0000000011111111b
		dw	0000000001111111b
		dw	0000000000111111b
		dw	0000000000011111b
		dw	0000000111111111b
		dw	0000000011111111b
		dw	0011000011111111b
		dw	1111100001111111b
		dw	1111100001111111b
		dw	1111110011111111b
		dw	0000000000000000b	; cursormask
		dw	0100000000000000b
		dw	0110000000000000b
		dw	0111000000000000b
		dw	0111100000000000b
		dw	0111110000000000b
		dw	0111111000000000b
		dw	0111111100000000b
		dw	0111111110000000b
		dw	0111110000000000b
		dw	0110110000000000b
		dw	0100011000000000b
		dw	0000011000000000b
		dw	0000001100000000b
		dw	0000001100000000b
		dw	0000000000000000b
		db	1			; cursorhidden
;;+*		db	0			; nolightpen?
		evendata
ERRIF ($-StartDefArea ne LenDefArea) "Defaults area contents corrupted!"

;----- driver and video state begins here -----

		evendata
UIRunlock	db	1		; 0 - user intr routine is in progress
videounlock	db	1		; 0 - screen manipulation is in progress

cursorptr	dd	0		; video memory pointer
cursordrawn	db	0		; 1 - restore screen data

videomode	db	?		; video mode number
granumask	POINT	<-1,-1>
textbuf		label	word
buffer@		dd	?		; pointer to screen sprite copy

vdata1		dw	10h,1, 10h,3, 10h,4, 10h,5, 10h,8, 08h,2
VDATA1cnt	equ	($-vdata1)/4
vdata2		dw	10h,1, 10h,4, 10h,0105h, 10h,0FF08h, 08h,0F02h
VDATA2cnt	equ	($-vdata2)/4

;----- table of pointers to registers values for RIL -----
RGROUPDEF	struc
  port@		dw	?
  regs@		dw	?
  def@		dw	?
  regscnt	db	1
  rmodify?	db	0
RGROUPDEF	ends

		evendata
videoregs@	label
	RGROUPDEF <3D4h,regs_CRTC,def_CRTC,25>	; CRTC
	RGROUPDEF <3C4h,regs_SEQC,def_SEQC,5>	; Sequencer
	RGROUPDEF <3CEh,regs_GRC, def_GRC, 9>	; Graphics controller
	RGROUPDEF <3C0h,regs_ATC, def_ATC, 20>	; VGA attrib controller
	RGROUPDEF <3C2h,reg_MISC, def_MISC>	; VGA misc output and input
	RGROUPDEF <3DAh,reg_FC,   def_FC>	; Feature Control
	RGROUPDEF <3CCh,reg_GPOS1,def_GPOS1>	; Graphics 1 Position
	RGROUPDEF <3CAh,reg_GPOS2,def_GPOS2>	; Graphics 2 Position


; IRQ HANDLERS 

;

IRQhandler	proc
		;cld
		push	ds es
		PUSHALL
		movSeg	ds,cs
IRQproc		db	0EBh			; if PS/2 "j PS2proc"
		db	PS2proc-$-1		; else "mov al,20h"
		out	20h,al			; {20h} end of interrupt

		db	0BAh			; MOV DX,word
IO_address	dw	?			; COM port IO address
		push	dx
		inc	dx
		inc	dx
		in	al,dx			; {3FAh} IIR (intr id reg)
		add	dx,3			; {3FDh} LSR: clear error bits
		in	ax,dx			; {3FEh} MSR: clear state bits
		pop	dx
		xchg	bx,ax			; OPTIMIZE: instead MOV BL,AL
		in	al,dx			; {3F8h} flush receive buffer
		db	0B9h			; MOV CX,word
IOdone		db	?,0			; processed bytes counter

		test	bl,00001010b		; Framing/Overrun Error?
		jz	@@checkdata
		xor	cx,cx			; restart sequence if errors
		mov	[IOdone],cl		; clear counter

@@checkdata:	shr	bl,1
		jnc	@@exitIRQ		; jump if data not ready
		db	0E8h			; CALL NEAR to mouseproc
mouseproc	dw	MSMproc-$-2

@@exitIRQ:	jmp	@rethandler
IRQhandler	endp

;
;!!! WARNING: buffer for copy of screen sprite when serial protocols

bufferSERIAL	label				; requires 3*16 bytes

;
;				Enable PS/2
;
;
; In:	none
; Out:	none
; Use:	none
; Modf:	BX, ES
; Call:	disablePS2
;
enablePS2	proc
		call	disablePS2
		movSeg	es,cs
		mov	bx,offset DGROUP:IRQhandler
		PS2serv	0C207h			; es:bx=ptr to handler
		mov	bh,1			; set mouse on
		PS2serv	0C200h
		ret
enablePS2	endp

;
;				Disable PS/2
;
;
; In:	none
; Out:	none
; Use:	none
; Modf:	BX, ES
; Call:	none
;
disablePS2	proc
		xor	bx,bx			; set mouse off
		PS2serv	0C200h
		mov	es,bx			; clear PS/2 mouse handler
		PS2serv	0C207h			; es:bx=ptr to handler
		ret
disablePS2	endp

;

PS2proc		proc
		mov	bp,sp
		mov	ax,[bp+_ARG_OFFS_+6]	; Status
		mov	bx,[bp+_ARG_OFFS_+4]	; X
		mov	cx,[bp+_ARG_OFFS_+2]	; Y
		test	al,00100000b		; check Y sign bit
		jz	@@checkxsign
		mov	ch,-1
@@checkxsign:	test	al,00010000b		; check X sign bit
		jz	@@yreverse
		mov	bh,-1
@@yreverse:	neg	cx			; reverse Y movement
		call	mouseupdate
		POPALL
		pop	es ds
		retf
PS2proc		endp
ERRIF ($-bufferSERIAL lt 3*16) "PS/2 handler too small for buffer!"

;
;!!! WARNING: buffer for copy of screen sprite when PS2 protocol

bufferPS2	label				; requires 3*16 bytes

;
;			Enable serial interrupt in PIC
;
;
; In:	none
; Out:	none
; Use:	IO_address
; Modf:	AX, DX, SI, IOdone, MSLTbuttons
; Call:	INT 21/25
;
enableCOM	proc
;---------- set new IRQ handler
		db	0B8h			; MOV AX,word
IRQintnum	dw	2500h			; INT number of selected IRQ
		mov	dx,offset DGROUP:IRQhandler
		int	21h			; set INT in DS:DX

;---------- set communication parameters (speed, parity, etc.)
		mov	si,[IO_address]
		lea	dx,[si+3]
		mov	al,10000000b
		out	dx,al			; {3FBh} LCR: DLAB on
		xchg	dx,si
		mov	ax,96			; 1200 baud rate
		out	dx,ax			; {3F8h},{3F9h} divisor latch

		xchg	dx,si
		db	0B8h			; MOV AX,word
COMLCR		db	00000010b,00001011b	; {3FBh} LCR: DLAB off, no
		out	dx,ax			;  parity, stop=1, length=7/8
						; {3FCh} MCR: DTR/RTS/OUT2 on
		inc	dx
		inc	dx			; {3FDh} LSR: clear error bits
		in	ax,dx			; {3FEh} MSR: clear state bits
		mov	dx,si
		in	al,dx			; {3F8h} flush receive buffer

		inc	dx
		mov	ax,1			; {3F9h} IER: enable DR intr
		out	dx,ax			; {3FAh} FCR: disable FIFO

		dec	ax			; OPTIMIZE: instead MOV AL,0
		mov	[IOdone],al
		mov	[MSLTbuttons],al
;----------
		in	al,21h			; {21h} get PIC mask
		db	24h			; AND AL,byte
notPICstate	db	?			; clear bit to enable interrupt
		out	21h,al			; enable serial interrupts
		ret
enableCOM	endp

;
;			Disable serial interrupt of PIC
;
;
; In:	none
; Out:	none
; Use:	IO_address, oldIRQaddr
; Modf:	AX, DX
; Call:	INT 21/25
;
disableCOM	proc
		in	al,21h			; {21h} get PIC mask
		db	0Ch			; OR AL,byte
PICstate	db	?			; set bit to disable interrupt
		out	21h,al			; disable serial interrupts
;----------
		mov	dx,[IO_address]
		add	dx,3
		xor	ax,ax			; {3FBh} LCR: DLAB off
		out	dx,ax			; {3FCh} MCR: DTR/RTS/OUT2 off
		dec	dx
		dec	dx
		;mov	al,0
		out	dx,al			; {3F9h} IER: interrupts off

;---------- restore old IRQ handler
		mov	ax,[IRQintnum]
		push	ds
		lds	dx,[oldIRQaddr]
		int	21h			; set INT in DS:DX
		pop	ds
		ret
disableCOM	endp

;
;		Process mouse bytes the Microsoft/Logitech way
;

MSLTproc	proc
		test	al,01000000b		; synchro check
		jz	@@MSLT_2		; jump if non first byte

		mov	[IOdone],1		; request next 2/3 bytes
		mov	[MSLT_1],al
MSLTCODE1	db	091h			; if LT/WM "xchg ax,cx" else "ret"
		sub	al,3			; first byte after 3 bytes?
		jnz	@@MSLTret
		mov	[MSLTbuttons],al	; release middle button if yes
@@MSLTret:	ret

@@MSLT_2:	jcxz	@@MSLTret		; skip nonfirst byte at start
		inc	[IOdone]		; request next byte
		loop	@@MSLT_3
		mov	[MSLT_X],al		; keep X movement LO
		ret

@@MSLT_3:	loop	@@LTWM_4
		xchg	cx,ax			; OPTIMIZE: instead MOV CL,AL
		db	0B4h			; MOV AH,byte
MSLT_1		db	?			; first byte of MS/LT protocol

		;mov	al,0			; CX was 0
		shr	ax,2			; bits 1-0 - X movement HI
		db	0Ch			; OR AL,byte
MSLT_X		db	?
		mov	bh,ah
		cbw
		xchg	bx,ax			; BX=X movement

		mov	al,0
		shr	ax,2			; bits 3-2 - Y movement HI
		or	al,cl
		mov	cl,ah			; bits 5-4 - L/R buttons
		cbw
		xchg	cx,ax			; CX=Y movement

		xor	al,[MSLTbuttons]
		and	al,00000011b		; L/R buttons change mask
		mov	dl,al
		or	dl,bl			; nonzero if L/R buttons state
		or	dl,cl			;  changed or mouse moved
MSLTCODE2	db	0EBh			; if MS3 "jnz" else "j"
		db	@@MSLTupdate-$-1
		or	al,00000100b		; empty event toggles button
		j	@@MSLTupdate

@@LTWM_4:	mov	[IOdone],ch		; CH=0, request next 3/4 bytes
MSLTCODE3	db	0B1h,3			; if LT "mov cl,3" else
						; if WM "mov cl,2" else "ret"
		shr	al,cl			; (MS defines only 3 bytes)
		xor	al,[MSLTbuttons]
		and	al,00000100b		; nonzero if state changed
		jz	@@MSLTret
		xor	cx,cx
		xor	bx,bx
@@MSLTupdate:	db	034h			; XOR AL,byte
MSLTbuttons	db	?			; buttons state for MS3/LT
		mov	[MSLTbuttons],al
		j	mouseupdate
MSLTproc	endp

;
;		Process mouse bytes the Mouse Systems way
;

MSMproc		proc
		jcxz	@@MSM_1
		cbw
		dec	cx
		jz	@@MSM_24
		dec	cx
		jz	@@MSM_3
		loop	@@MSM_5

@@MSM_24:	add	[MSM_X],ax
@@MSMnext:	inc	[IOdone]		; request next byte
@@MSMret:	ret

@@MSM_1:	mov	[MSM_buttons],al	; save buttons state
		mov	[MSM_X],cx		; CX=0
		and	al,11111000b		; sync check: AL should
		cmp	al,10000000b		;  be equal to 10000lmrb
		;je	@@MSMnext
		;ret
		jne	@@MSMret

@@MSM_3:	mov	[MSM_Y],ax
		j	@@MSMnext

@@MSM_5:	mov	[IOdone],ch		; CH=0, request next 5 bytes
		db	05h			; ADD AX,word
MSM_Y		dw	?
		xchg	cx,ax			; OPTIMIZE: instead MOV CX,AX
		neg	cx
		db	0BBh			; MOV BX,word
MSM_X		dw	?
		db	0B0h			; MOV AL,byte
MSM_buttons	db	?
		not	al			; bits 2-0 - L/M/R buttons
		test	al,00000110b		; check the L and M buttons
		jpe	mouseupdate		; skip if the same
		xor	al,00000110b		; swap them
		;j	mouseupdate
MSMproc		endp
ERRIF ($-bufferPS2 lt 3*16) "Serial handler too small for buffer!"

;
;			Update mouse status
;
;
; In:	AL			(new buttons state)
;	BX			(X mouse movement)
;	CX			(Y mouse movement)
; Out:	none
; Use:	callmask, granpos, mickeys, UIR@
; Modf:	AX, CX, DX, BX, SI, DI, buttstatus, UIRunlock
; Call:	updateposition, updatebutton, showcursor
;
mouseupdate	proc
		test	al,00000011b		; check the L and R buttons
LEFTHANDCODE	db	?			; JMP SHORT for PS2^LHAND/JPE
		db	@@updatemask-$-1
		xor	al,00000011b		; swap
@@updatemask:	db	024h			; AND AL,byte
buttonsmask	db	00000111b
		xchg	di,ax			; OPTIMIZE: instead MOV DI,AX

;---------- recalculate mickey counters and screen position
		db	0B2h			; MOV DL,byte
mresolutionX	db	0
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
IF (offset X) eq 0
		xor	bx,bx
ELSE
		mov	bx,offset X
ENDIF
		call	updateposition

		db	0B2h			; MOV DL,byte
mresolutionY	db	0
		xchg	ax,cx
		mov	bl,offset Y		; OPTIMIZE: BL instead BX
		call	updateposition
		or	cl,al			; bit 0=movement flag

;---------- recalculate buttons state
		mov	ax,[mickeys.Y]
		xchg	ax,di
		xchg	dx,ax			; OPTIMIZE: instead MOV DX,AX
		mov	dh,dl			; DH=buttons new state
		xchg	dl,[buttstatus]
		xor	dl,dh			; DL=buttons change state
IF FASTER_CODE
		jz	@@callUIR		; CL=unrolled change state
ENDIF
		xor	bx,bx			; buttpress array index
		mov	al,00000010b		; indicate that 1 is pressed
		call	updatebutton
		mov	al,00001000b		; indicate that 2 is pressed
		call	updatebutton
		mov	al,00100000b		; indicate that 3 is pressed
		call	updatebutton

;---------- call User Interrupt Routine
@@callUIR:	dec	[UIRunlock]
		jnz	@@updatedone		; exit if user proc running

		and	cl,[callmask]		; is there a user call mask?
		jz	@@updateshow		; skip if not
		db	0BBh			; MOV BX,word
buttstatus	db	0,0			; buttons status
		mov	ax,[granpos.X]
		mov	dx,[granpos.Y]
		xchg	ax,cx
		mov	si,[mickeys.X]
		;mov	di,[mickeys.Y]
		push	ds
		sti
		call	[UIR@]
		pop	ds
@@updateshow:	call	showcursor
;----------
@@updatedone:	inc	[UIRunlock]
		ret
mouseupdate	endp

;
; In:	AX			(mouse movement)
;	BX			(offset X/offset Y)
;	DL			(resolution)
; Out:	AX			(1 - mickey counter changed)
; Use:	mickey8, rangemax, rangemin, granumask
; Modf:	DX, SI, mickeys, rounderr, pos, granpos
;
updateposition	proc
		or	ax,ax
		jz	@@uposret
		mov	si,ax
		jg	@@resolute
		neg	si

;---------- apply resolution (AX=movement, SI=abs(AX), DL=reslevel)
@@resolute:	mov	dh,0
		dec	dx
		jz	@@newmickeys		; skip resolution=1
		cmp	si,7
		ja	@@rescheck		; medium [-7..+7] movement...
		cmp	si,2
		jbe	@@newmickeys		; ...except [-2..+2] movement
@@resmul2:	shl	ax,1			; ...multiply on 2
		j	@@newmickeys

@@rescheck:	inc	dx
		jnz	@@resmul

		mov	dl,10			; auto resolution=
		shr	si,2			;  min(10,abs(AX)/4)
		cmp	si,dx
		jae	@@resmul
		mov	dx,si
@@resmul:
IF FASTER_CODE
		cmp	dl,2
		je	@@resmul2
ENDIF
		imul	dx			; multiply on resolution

;---------- apply mickeys per 8 pixels ratio
@@newmickeys:	add	word ptr mickeys[bx],ax
IF FASTER_CODE
		mov	si,word ptr mickey8[bx]
		cmp	si,8
		je	@@newpos
		shl	ax,3
		dec	si
		jle	@@newpos
		add	ax,word ptr rounderr[bx]
		inc	si
		cwd
		idiv	si
		mov	word ptr rounderr[bx],dx
		or	ax,ax
		jz	@@uposdone
ELSE
		shl	ax,3
		add	ax,word ptr rounderr[bx]
		cwd
		idiv	word ptr mickey8[bx]
		mov	word ptr rounderr[bx],dx
ENDIF
@@newpos:	add	ax,word ptr pos[bx]

;---------- cut new position by ranges and save
@savecutpos:	mov	dx,word ptr rangemax[bx]
		cmp	ax,dx
		jge	@@cutpos
		mov	dx,word ptr rangemin[bx]
		cmp	ax,dx
		jg	@@savepos
@@cutpos:	xchg	ax,dx			; OPTIMIZE: instead MOV AX,DX

@@savepos:	mov	word ptr pos[bx],ax	; new position
		and	al,byte ptr granumask[bx]
		mov	word ptr granpos[bx],ax	; new granulated position
@@uposdone:	mov	ax,1
@@uposret:	ret
updateposition	endp

;
; In:	AL			(unrolled press bit mask)
;	CL			(unrolled buttons change state)
;	DL			(buttons change state)
;	DH			(buttons new state)
;	BX			(buttpress array index)
; Out:	CL
;	DX			(shifted state)
;	BX			(next index)
; Use:	granpos
; Modf:	AX, SI, buttpress, buttrelease
;
updatebutton	proc
		shr	dx,1
		jnc	@@nextbutton		; jump if button unchanged
		mov	si,offset DGROUP:buttpress
		or	dl,dl
		js	@@updbutton		; jump if button pressed
		shl	al,1			; indicate that it released
		mov	si,offset DGROUP:buttrelease
@@updbutton:	or	cl,al
		inc	[si+bx].counter
		mov	ax,[granpos.Y]
		mov	[si+bx].lastrow,ax
		mov	ax,[granpos.X]
		mov	[si+bx].lastcol,ax
@@nextbutton:	add	bx,SIZE BUTTLASTSTATE
		ret
updatebutton	endp

; END OF IRQ HANDLERS 


; INT 10 HANDLER 

		evendata
RILtable	dw	offset DGROUP:RIL_F0	; RIL functions
		dw	offset DGROUP:RIL_F1
		dw	offset DGROUP:RIL_F2
		dw	offset DGROUP:RIL_F3
		dw	offset DGROUP:RIL_F4
		dw	offset DGROUP:RIL_F5
		dw	offset DGROUP:RIL_F6
		dw	offset DGROUP:RIL_F7

int10handler	proc
		cld
		or	ah,ah			; set video mode?
		jz	@@setmodreq
		cmp	ah,11h			; font manipulation function
		je	@@setnewfont
		cmp	ax,4F02h		; VESA set video mode?
		je	@@setmodreq
		cmp	ah,0F0h			; RIL func requested?
		jb	@@jmpold10
		cmp	ah,0F7h
		jbe	@@RIL
		cmp	ah,0FAh
		je	@@RIL_FA
@@jmpold10:	db	0EAh			; JMP FAR to old INT 10 handler
oldint10	dd	?

@@setnewfont:	cmp	al,30h
		je	@@jmpold10
		;j	@@setmodreq

;========== set video mode
@@setmodreq:	push	ax
		mov	al,2			; OPTIMIZE: AL instead AX
		pushf				;!!! Logitech MouseWare
		push	cs			;  Windows driver workaround
		call	handler33		; hide mouse cursor
		pop	ax
		pushf
		call	cs:[oldint10]
		push	ds es
		PUSHALL
		movSeg	ds,cs
		mov	[cursorhidden],1	; normalize flag
		call	setupvideo
@@exitINT10:	jmp	@rethandler

;========== RIL
@@RIL:		push	ds es
		PUSHALL
		movSeg	ds,cs
		mov	bp,sp
		mov	al,ah
		and	ax,0Fh			;!!! AH must be 0 for RIL_*
		mov	si,ax
		shl	si,1
		call	RILtable[si]
		j	@@exitINT10
;----------
@@RIL_FA:	movSeg	es,cs			; RIL FA - Interrogate driver
		mov	bx,offset DGROUP:RILversion
		iret
int10handler	endp

;
; RIL F0 - Read one register
;
;
; In:	DX			(group index)
;	BX			(register #)
; Out:	BL			(value)
; Use:	videoregs@
; Modf:	AL, SI
; Call:	none
;
RIL_F0		proc
		mov	si,dx
		mov	si,videoregs@[si].regs@
		cmp	dx,20h
		jae	@@retBL			; jump if single register
		add	si,bx
@@retBL:	lodsb
		mov	byte ptr [_ARG_BX_],al
		ret
RIL_F0		endp

;
; RIL F1 - Write one register
;
;
; In:	DX			(group index)
;	BL			(value for single reg)
;	BL			(register # otherwise)
;	BH			(value otherwise)
; Out:	BL			(value)
; Use:	none
; Modf:	AX
; Call:	RILwrite
;
RIL_F1		proc
		mov	ah,bl
		cmp	dx,20h
		jae	RILwrite		; jump if single registers
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		mov	byte ptr [_ARG_BX_],ah
		;j	RILwrite
RIL_F1		endp

;
; In:	DX			(group index)
;	AL			(register # for regs group)
;	AH			(value to write)
; Out:	none
; Use:	videoregs@
; Modf:	AL, DX, BX, DI
; Call:	RILoutAH, RILgroupwrite
;
RILwrite	proc
		xor	bx,bx
		mov	di,dx
		cmp	dx,20h
		mov	dx,videoregs@[di].port@
		mov	videoregs@[di].rmodify?,dl ; OPTIMIZE: DL instead 1
		mov	di,videoregs@[di].regs@
		jae	@@RWsingle		; jump if single register
		mov	bl,al
@@RWsingle:	mov	[di+bx],ah
		jae	RILoutAH
		;j	RILgroupwrite
RILwrite	endp

;
; In:	DX			(IO port)
;	AL			(register #)
;	AH			(value to write)
; Out:	none
; Use:	videoregs@
; Modf:	none
; Call:	none
;
RILgroupwrite	proc
		cmp	dl,0C0h
		je	@@writeATC		; jump if ATTR controller
		out	dx,ax
		ret
@@writeATC:	push	ax dx
		mov	dx,videoregs@[(SIZE RGROUPDEF)*5].port@
		in	al,dx			; {3DAh} force address mode
		pop	dx ax
		out	dx,al			; {3C0h} select ATC register
RILoutAH:	xchg	al,ah
		out	dx,al			; {3C0h} modify ATC register
		xchg	al,ah
		ret
RILgroupwrite	endp

;
; RIL F2 - Read register range
;
;
; In:	CH			(starting register #)
;	CL			(# of registers)
;	DX			(group index: 0,8,10h,18h)
;	ES:BX			(buffer, CL bytes size)
; Out:	none
; Use:	videoregs@
; Modf:	AX, CX, SI, DI
; Call:	none
;
RIL_F2		proc
		mov	di,bx
		mov	si,dx
		mov	si,videoregs@[si].regs@
		mov	al,ch
		;mov	ah,0
		add	si,ax
RILmemcopy:	sti
		mov	ch,0
		shr	cx,1
		rep	movsw
		adc	cx,cx
		rep	movsb
		ret
RIL_F2		endp

;
; RIL F3 - Write register range
;
;
; In:	CH			(starting register #)
;	CL			(# of registers, >0)
;	DX			(group index: 0,8,10h,18h)
;	ES:BX			(buffer, CL bytes size)
; Out:	none
; Use:	videoregs@
; Modf:	AX, CX, DX, BX, DI
; Call:	RILgroupwrite
;
RIL_F3		proc
		mov	di,dx
		mov	dx,videoregs@[di].port@
		mov	videoregs@[di].rmodify?,dl ; OPTIMIZE: DL instead 1
		mov	di,videoregs@[di].regs@
RILgrouploop:	xor	ax,ax
		xchg	al,ch
		add	di,ax
@@R3loop:	mov	ah,es:[bx]
		mov	[di],ah
		inc	bx
		inc	di
		call	RILgroupwrite
		inc	ax			; OPTIMIZE: AX instead AL
		loop	@@R3loop
		ret
RIL_F3		endp

;
; RIL F4 - Read register set
;
;
; In:	CX			(# of registers, >0)
;	ES:BX			(table of registers records)
; Out:	none
; Use:	videoregs@
; Modf:	AL, CX, BX, DI
; Call:	none
;
RIL_F4		proc
		sti
		mov	di,bx
@@R4loop:	mov	bx,es:[di]
		inc	di
		inc	di
		mov	bx,videoregs@[bx].regs@
		mov	al,es:[di]
		inc	di
		xlat
		stosb
		loop	@@R4loop
		ret
RIL_F4		endp

;
; RIL F5 - Write register set
;
;
; In:	CX			(# of registers, >0)
;	ES:BX			(table of registers records)
; Out:	none
; Use:	none
; Modf:	AX, CX, DX, SI
; Call:	RILwrite
;
RIL_F5		proc
		mov	si,bx
@@R5loop:	lods	word ptr es:[si]
		xchg	dx,ax			; OPTIMIZE: instead MOV DX,AX
		lods	word ptr es:[si]
		call	RILwrite
		loop	@@R5loop
		ret
RIL_F5		endp

;
; RIL F7 - Define registers default
;
;
; In:	DX			(group index)
;	ES:BX			(table of one-byte entries)
; Out:	none
; Use:	videoregs@
; Modf:	CL, SI, DI, ES, DS
; Call:	RILmemcopy
;
RIL_F7		proc
		mov	si,bx
		mov	di,dx
		mov	cl,videoregs@[di].regscnt
		mov	videoregs@[di].rmodify?,cl ; OPTIMIZE: CL instead 1
		mov	di,videoregs@[di].def@
		push	es ds
		pop	es ds
		j	RILmemcopy
RIL_F7		endp

;
; RIL F6 - Revert registers to default
;
;
; In:	none
; Out:	none
; Use:	videoregs@
; Modf:	AX, CX, DX, BX, SI, DI, ES
; Call:	RILgrouploop
;
RIL_F6		proc
		movSeg	es,ds
		mov	si,offset DGROUP:videoregs@+(SIZE RGROUPDEF)*8

@@R6loop:	sub	si,SIZE RGROUPDEF
		xor	cx,cx
		xchg	cl,[si].rmodify?
		jcxz	@@R6next

		mov	bx,[si].def@
		mov	di,[si].regs@
		mov	dx,[si].port@
		mov	cl,[si].regscnt
		dec	cx			; OPTIMIZE: CX instead CL
		jnz	@@R6group
		mov	al,[bx]
		stosb
		out	dx,al
		;j	@@R6next		; OPTIMIZE: single regs
		j	@@R6loop		;  handled first

@@R6group:	inc	cx			; OPTIMIZE: CX instead CL
		;mov	ch,0
		call	RILgrouploop

@@R6next:	cmp	si,offset DGROUP:videoregs@
		ja	@@R6loop
		ret
RIL_F6		endp

; END OF INT 10 HANDLER 


;
;			Draw mouse cursor
;

drawcursor	proc
		jz	@graphcursor		; jump if graphics mode
		call	gettxtoffset
		cmp	[cursortype],ch		; OPTIMIZE: CH instead 0
		jz	@@softcursor		; jump if software cursor

;========== draw hardware text mode cursor
		shr	di,1
		mov	dx,videoregs@[0].port@	; CRTC port

		mov	ax,di
		mov	ah,al
		mov	al,0Fh
		out	dx,ax			; cursor position lo

		xchg	ax,di			; OPTIMIZE: instead MOV AX,DI
		mov	al,0Eh
		out	dx,ax			; cursor position hi
		ret

;========== draw software text mode cursor
@@softcursor:	jcxz	@@drawtext		; jump if cursor not drawn
		cmp	di,word ptr cursorptr[0]
		je	@@drawtextdone		; exit if position not changed
		push	di
		call	restoretext
		pop	di

@@drawtext:	mov	ax,[granpos.Y]		; get cursor position
		mov	bx,[granpos.X]
		mov	si,8			; OPTIMIZE: instead -[granumask.Y]
		call	checkifseen
		jc	@@drawret		; exit if not seen

		mov	word ptr cursorptr[0],di
		mov	es,word ptr cursorptr[2]
		mov	ax,es:[di]		; save char under cursor
		mov	textbuf[0],ax
		and	ax,[startscan]
		xor	ax,[endscan]
		stosw				; draw to new position
		mov	textbuf[2],ax

@@drawtextdone:	inc	[cursordrawn]		; we have to restore later
@@drawret:	ret

;========== draw graphics mode cursor
@graphcursor:	; if [videomode] not in [4-6, 0Dh-13h] "ret" else "push cx"
		db	?
		call	updatevregs
		mov	bx,[granpos.X]
		sub	bx,[hotspot.X]
		and	bx,[granumask.X]	; virtual granulated X
		mov	ax,[granpos.Y]
		sub	ax,[hotspot.Y]		; virtual Y

		push	ax
		call	getgroffset
		pop	ax cx
; cx=old cursordrawn, bx=X, ax=Y, di=video row offset, si=nextrow

		jcxz	@@drawgraph		; jump if cursor not drawn
		cmp	bx,[cursorX]
		jne	@@restsprite
		db	081h,0FFh		; CMP DI,word
cursorrow	dw	?
		je	@@drawgrdone		; exit if position not changed

; bx=X, ax=Y, di=video row offset, si=nextrow
@@restsprite:	push	ax bx
		call	restoresprite
		pop	bx ax

;---------- check if cursor seen and not in update region
; bx=X, ax=Y, di=video row offset, si=nextrow
@@drawgraph:	mov	[cursorX],bx
		mov	[cursorY],ax
		mov	[cursorrow],di
		mov	[nextrow],si

		db	0B1h			; MOV CL,byte
bitmapshift	db	?			; mode 13h=1, 0Dh=4, other=3
		shr	bx,cl
		shl	bx,cl			; virtual screen aligned X
		mov	si,16			; cursor height
		call	checkifseen
		jc	@@drawgrret		; exit if not seen

;---------- precompute some sprite parameters
		push	ax			; push [cursorY]
		shr	dx,cl			; right X byte offset
		mov	ax,[scanline]
		cmp	dx,ax
		jbe	@@spritesize
		xchg	dx,ax			; DX=min(DX,scanline)
@@spritesize:	sar	bx,cl			; left X byte offset (signed)
		db	0B8h			; MOV AX,word
cursorX		dw	?
		sub	cl,3			; mode 0Dh=1
		sar	ax,cl			; sprite shift for non 13h modes
		neg	bx
		jge	@@savespritesz
		add	dx,bx
		sub	di,bx
		xor	bx,bx
		and	al,7			; shift in byte (X%8)
@@savespritesz:	mov	word ptr cursorptr[0],di
		mov	[spritewidth],dx
		neg	al
		add	al,8			; 8+|X| or 8-X%8
		cbw				; OPTIMIZE: instead MOV AH,0
		push	ax bx

;---------- save sprite and draw cursor at new cursor position
		mov	al,0D6h			; screen source
		call	copysprite		; save new sprite

		pop	si cx ax		; spriteshift,,[cursorY]
		les	di,[cursorptr]
		;mov	dx,[nextrow]
		;xor	bx,bx			; mask offset (2*16 bytes)

@@makeloop:	cmp	ax,[maxcoordY]
		jae	@@makenexty		; jump if Y > max || Y < 0

		push	ax dx bx di
		call	makerow
		pop	di bx dx ax

@@makenexty:	inc	ax
		add	di,dx
		xor	dx,[nextxor]
		inc	bx
		inc	bx
		cmp	bx,2*16
		jb	@@makeloop
;----------
@@drawgrdone:	inc	[cursordrawn]
@@drawgrret:	;j	restorevregs
drawcursor	endp

;
;		Restore graphics card video registers
;

restorevregs	proc
		; if [planar videomode 0Dh-12h] "push ds" else "ret"
		db	?
		pop	es
		mov	bx,offset DGROUP:vdata1
		mov	cx,VDATA1cnt
		j	@writevideo
restorevregs	endp

;
;		Save & update graphics card video registers
;

updatevregs	proc
		; if [planar videomode 0Dh-12h] "push ds" else "ret"
		db	?
		pop	es
		mov	bx,offset DGROUP:vdata1
		mov	cx,VDATA1cnt
		mov	ah,0F4h			; read register set
		int	10h

		mov	bx,offset DGROUP:vdata2
		mov	cx,VDATA2cnt
@writevideo:	mov	ah,0F5h			; write register set
		int	10h
		ret
updatevregs	endp

;
;			Restore old screen content
;

restorescreen	proc
		jcxz	@restret		; exit if cursor was not drawn
		jnz	restoretext		; jump if text mode
		call	updatevregs
		call	restoresprite
		j	restorevregs
restorescreen	endp

;

restoretext	proc
		les	di,[cursorptr]
		mov	ax,es:[di]
		cmp	ax,textbuf[2]
		jne	@restret
		mov	ax,textbuf[0]
		stosw				; restore old text char attrib
@restret:	ret
restoretext	endp

;

restoresprite	proc
		mov	al,0D7h			; screen destination
		;j	copysprite		; restore old sprite
restoresprite	endp

;
;		Copy screen sprite back and forth
;
;
; In:	AL			(0D6h-scr src./0D7h-scr dst.)
; Out:	DX = [nextrow]
;	BX = 0
; Use:	buffer@, cursorptr
; Modf:	AX, CX, ES
; Call:	none
;
copysprite	proc	C uses si di ds
		cmp	al,0D6h
		mov	NEXTOFFSCODE[1],al
		db	0B8h			; MOV AX,word
cursorY		dw	?			; cursor screen Y pos
		db	0BAh			; MOV DX,word
nextrow		dw	?			; next row offset
		mov	bx,16			; sprite height in rows
		les	di,[buffer@]
		lds	si,[cursorptr]
		je	@@copyloop
		push	ds es
		pop	ds es
		xchg	si,di

@@copyloop:	push	dx
		db	03Dh			; CMP AX,word
maxcoordY	dw	?			; screen height
		jae	@@copynexty		; jump if Y > max || Y < 0

		db	0B9h			; MOV CX,word
spritewidth	dw	?			; seen part of sprite in bytes
		sub	dx,cx
		rep	movsb

@@copynexty:	inc	ax
NEXTOFFSCODE	db	01h,?			; ADD SI,DX/ADD DI,DX
		pop	dx
		db	081h,0F2h		; XOR DX,word
nextxor		dw	?			; even times remain unchanged
		dec	bx
		jnz	@@copyloop
		ret
copysprite	endp

;
;		Transform the cursor mask row to screen
;
;
; In:	BX			(mask offset)
;	CX			(sprite shift when non 13h modes)
;	SI			(sprite shift when mode 13h)
;	ES:DI			(video memory pointer)
; Out:	none
; Use:	screenmask, cursormask
; Modf:	AX, CX, DX, BX, SI, DI
; Call:	none
;
makerow		proc
		mov	dx,word ptr screenmask[bx]
		mov	bx,word ptr cursormask[bx]
		mov	ax,[spritewidth]
		cmp	[videomode],13h
		jne	makerowno13
;----------
		mov	cx,si
		shl	dx,cl
		shl	bx,cl
		xchg	cx,ax			; OPTIMIZE: instead MOV CX,AX

@@loopx13:	mov	al,0
		shl	dx,1
		jnc	@@chkcmask		; jump if most sign bit zero
		mov	al,es:[di]
@@chkcmask:	shl	bx,1
		jnc	@@putbyte13		; jump if most sign bit zero
		xor	al,0Fh
@@putbyte13:	stosb
		loop	@@loopx13
		ret

;---------- display cursor row in modes other than 13h
makerowno13:	push	cx
		xchg	si,ax			; OPTIMIZE: instead MOV SI,AX
		mov	ax,0FFh
@@maskshift:	stc
		rcl	dx,1
		rcl	al,1			; al:dh:dl shifted screenmask
		shl	bx,1
		rcl	ah,1			; ah:bh:bl shifted cursormask
		loop	@@maskshift
		xchg	dh,bl			; al:bl:dl - ah:bh:dh

@@loopxno13:	push	dx
		cmp	[videomode],0Dh		; is videomode 0Dh-12h?
		jae	@@planar		; jump if multiplanar mode
		and	al,es:[di]
		xor	al,ah
		stosb
		j	@@nextxno13

@@planar:	xchg	cx,ax			; OPTIMIZE: instead MOV CX,AX
		mov	dx,3CEh
		mov	ax,5			; set write mode
		out	dx,ax
		mov	ax,0803h		; data ANDed with latched data
		out	dx,ax
		xchg	es:[di],cl
		mov	ah,18h			; data XORed with latched data
		out	dx,ax
		xchg	es:[di],ch
		inc	di

@@nextxno13:	xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		pop	bx
		dec	si
		jnz	@@loopxno13
		pop	cx
		ret
makerow		endp

;
;		Return graphic mode video memory offset
;
;
; In:	AX			(Y coordinate in pixels)
; Out:	DI			(video memory offset)
;	SI			(offset to next row)
; Use:	videomode
; Modf:	AX, DX, ES
; Call:	@getoffsret
;
getgroffset	proc
;
;4/5 (320x200x4)   byte offset = (y/2)*80 + (y%2)*2000h + (x*2)/8
;  6 (640x200x2)   byte offset = (y/2)*80 + (y%2)*2000h + x/8
;0Dh (320x200x16)  byte offset = y*40 + x/8, bit offset = 7 - (x % 8)
;0Eh (640x200x16)  byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;0Fh (640x350x4)   byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;10h (640x350x16)  byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;11h (640x480x2)   byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;12h (640x480x16)  byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;13h (320x200x256) byte offset = y*320 + x
;HGC (720x348x2)   byte offset = (y%4)*2000h + (y/4)*90 + x/8
;			bit offset = 7 - (x % 8)
;
		xor	di,di
		mov	es,di
		db	0BAh			; MOV DX,word
scanline	dw	?
		mov	si,dx			; [nextrow]
		cmp	byte ptr cursorptr[3],0A0h
		je	@getoffsret		; jump if not videomode 4-6
		mov	si,2000h
		sar	ax,1			; AX=Y/2
		jnc	@getoffsret
		mov	di,si			; DI=(Y%2)*2000h
		mov	si,-(2000h-80)
		j	@getoffsret
getgroffset	endp

;
;		Return text mode video memory offset
;
;
; In:	none
; Out:	DI			(video memory offset=row*[0:44Ah]*2+column*2)
; Use:	0:44Ah, 0:44Eh, granpos
; Modf:	AX, DX, ES
; Call:	getpageoffset
;
gettxtoffset	proc
		xor	dx,dx
		mov	es,dx
		mov	di,[granpos.X]
		mov	al,[bitmapshift]
		dec	ax			; OPTIMIZE: AX instead AL
		xchg	cx,ax
		sar	di,cl			; DI=column*2
		xchg	cx,ax			; OPTIMIZE: instead MOV CX,AX
		mov	ax,[granpos.Y]
		sar	ax,2			; AX=row*2=Y/4
		mov	dx,es:[44Ah]		; screen width

@getoffsret:	imul	dx			; AX=row*screen width
		add	ax,es:[44Eh]		; add video page offset
		add	di,ax
		ret
gettxtoffset	endp

;
;		Check if cursor in update region
;

checkifseen	proc
		db	0BAh			; MOV DX,word
cursorwidth	db	?,0

;---------- check if cursor shape seen on the screen
		add	si,ax
		jle	@@retunseen
		add	dx,bx
		jle	@@retunseen
		cmp	ax,[maxcoordY]
		jge	@@retunseen
		cmp	bx,[maxcoordX]
		jge	@@retunseen

;---------- check if cursor shape intersects with update region
		cmp	ax,[lowright.Y]
		jg	@@retseen
		cmp	bx,[lowright.X]
		jg	@@retseen
		cmp	si,[upleft.Y]
		jle	@@retseen
		cmp	dx,[upleft.X]
		jle	@@retseen

@@retunseen:	stc
		ret
@@retseen:	clc
		ret
checkifseen	endp


; INT 33 HANDLER SERVICES 

;

setupvideo	proc
		mov	si,LenClearArea2/2	; clear area 2
ERRIF (LenClearArea2 mod 2 ne 0) "LenClearArea2 must be even!"
		j	@setvideo
setupvideo	endp

;
; 21 - Software reset
;
;
; In:	none
; Out:	[AX] = 21h/FFFFh	(not installed/installed)
;	[BX] = 2/3/FFFFh	(number of buttons)
; Use:	0:449h, 0:44Ah, 0:463h, 0:484h, 0:487h, 0:488h, 0:4A8h, shape
; Modf:	StartSaveArea, videomode, nextxor, buffer@, restorevregs, updatevregs,
;	@graphcursor, scanline, maxcoordX, maxcoordY, granumask, cursorptr,
;	bitmapshift, cursorwidth, rangemax, StartClearArea
; Call:	hidecursor, @savecutpos
;
softreset_21	proc
buttonscnt	equ	byte ptr [$+3]		; buttons count (2/3)
		mov	[_ARG_BX_],3
		mov	[_ARG_AX_],0FFFFh

		movSeg	es,ds
		mov	si,offset DGROUP:StartDefArea
		mov	di,offset DGROUP:StartSaveArea
		mov	cx,LenDefArea/2
ERRIF (LenDefArea mod 2 ne 0) "LenDefArea must be even!"
		rep	movsw			; make default values
		call	hidecursor
		mov	si,LenClearArea3/2	; clear area 3
ERRIF (LenClearArea3 mod 2 ne 0) "LenClearArea3 must be even!"

;---------- setup video regs values for current video mode
@setvideo:	push	si
		movSeg	es,ds
		xor	ax,ax
		mov	ds,ax
		mov	ax,ds:[463h]		; CRTC base I/O port address
		mov	es:videoregs@[0].port@,ax ; 3D4h/3B4h
		add	ax,6
		mov	es:videoregs@[(SIZE RGROUPDEF)*5].port@,ax
		mov	al,ds:[449h]		; current video mode
		push	ax
;----------
		mov	ah,9
		cmp	al,11h			; VGA videomodes?
		jae	@@getshift

		cbw				; OPTIMIZE: instead MOV AH,0
		cmp	al,0Fh			; 0F-10 videomodes?
		jb	@@checkcga
		test	byte ptr ds:[487h],01100000b ; =60h
		jz	@@getshift		; jump if only 64K of VRAM
		mov	ah,2
		j	@@getshift

@@checkcga:	cmp	al,4			; not color text modes?
		jae	@@getshift
		mov	cl,ds:[488h]		; get display combination
		and	cl,00011111b		; =1Fh
		cmp	cl,9			; EGA/ECD+MDA?
		je	@@lines350
		cmp	cl,3			; MDA+EGA/ECD?
		jne	@@getshift
@@lines350:	mov	ah,13h
;----------
@@getshift:	lds	si,dword ptr ds:[4A8h]
		lds	si,dword ptr ds:[si]
		add	ah,al
		mov	al,5 SHL 2
		shr	ax,2
		add	si,ax			; SI += (AL+AH)*64+5

		mov	di,offset DGROUP:StartDefVRegsArea
		push	di
		mov	al,3
		stosb				; def_SEQ[0]=3
		mov	cx,50/2
		rep	movsw			; copy default registers value
		mov	al,0
		stosb				; def_ATC[20]=0; VGA only
		movsw				; def_GRC (9 registers)
		movsw
		movsw
		movsw
		movsb
		stosw				; def_FC=0, def_GPOS1=0
		inc	ax			; OPTIMIZE: instead MOV AL,1
		stosb				; def_GPOS2=1

		movSeg	ds,es
		pop	si
		mov	di,offset DGROUP:StartVRegsArea
		mov	cx,LenVRegsArea/2
ERRIF (LenVRegsArea mod 2 ne 0) "LenVRegsArea must be even!"
		rep	movsw			; copy current registers value

		dec	ax			; OPTIMIZE: instead MOV AL,0
		mov	cx,8
		mov	di,offset DGROUP:videoregs@[0].rmodify?
@@setmodify:	stosb
		add	di,(SIZE RGROUPDEF)-1
		loop	@@setmodify

;---------- set parameters for current video mode
; mode	 seg   screen  cell scan planar  VX/
;			    line	byte
;  0	B800h  640x200 16x8   -    -	  -
;  1	B800h  640x200 16x8   -    -	  -
;  2	B800h  640x200	8x8   -    -	  -
;  3	B800h  640x200	8x8   -    -	  -
;  4	B800h  640x200	2x1  80   no	  8
;  5	B800h  640x200	2x1  80   no	  8
;  6	B800h  640x200	1x1  80   no	  8
;  7	B000h  640x200	8x8   -    -	  -
; 0Dh	A000h  320x200	2x1  40    +	 16
; 0Eh	A000h  640x200	1x1  80    +	  8
; 0Fh	A000h  640x350	1x1  80    +	  8
; 10h	A000h  640x350	1x1  80    +	  8
; 11h	A000h  640x480	1x1  80    +	  8
; 12h	A000h  640x480	1x1  80    +	  8
; 13h	A000h  320x200	2x1 320   no	  2
;
		pop	ax			; current video mode
; mode 0-3
		mov	bx,0B8C3h		; B800h: 0-3/RET opcode
		mov	cx,0304h		; 16x8: 0-1
		mov	dx,0FFFFh
		mov	di,200			; x200: 4-6, 0Dh-0Eh, 13h
		cmp	al,2
		jb	@@settext
		dec	cx			; 8x8: 2-3, 7
		cmp	al,4
		jb	@@settext
; mode 7
		cmp	al,7
		jne	@@checkgraph
		mov	bh,0B0h			; B000h: 7

@@settext:	mov	ch,1
		mov	dh,-8
		shl	dl,cl

		xor	ax,ax
		mov	es,ax
		add	al,es:[484h]		; screen height-1
		jz	@@screenw
		inc	ax			; OPTIMIZE: AX instead AL
IF USE_286
		shl	ax,3
ELSE
		mov	ah,8
		mul	ah
ENDIF
		xchg	di,ax			; OPTIMIZE: instead MOV DI,AX

@@screenw:	mov	ax,es:[44Ah]		; screen width
		j	@@setcommon

; mode 4-6
@@checkgraph:	;mov	cx,0303h		; sprite: 3 bytes/row
		;mov	dx,0FFFFh		; 1x1: 6, 0Eh-12h
		;mov	bx,0B8C3h		; B800h: 4-6/RET opcode
		push	bx
		;mov	di,200			; x200: 4-6, 0Dh-0Eh, 13h
		mov	si,2000h xor -(2000h-80) ; [nextxor] for modes 4-6
		;movSeg	es,ds
		db	0BBh			; MOV BX,word
BUFFADDR	dw	offset DGROUP:bufferPS2
		cmp	al,6
		je	@@setgraphics
		jb	@@set2x1
; mode 8-
		pop	bx
		mov	bh,0A0h			; A000h: 0Dh-
; for modes 0Dh-13h we'll be storing the cursor shape and the hided
; screen contents at free space in video memory to save driver memory
		mov	si,0A000h
		mov	es,si
		xor	si,si			; [nextxor] for modes >6
		cmp	al,13h
		ja	@@nonstandard
		jb	@@mode0812
; mode 13h
		push	bx
		mov	bx,0FA00h		; =320*200
		mov	cx,1000h		; sprite: 16 bytes/row
@@set320:	inc	cx			; OPTIMIZE: instead INC CL
@@set2x1:	dec	dx			; OPTIMIZE: instead MOV DL,-2
		j	@@setgraphics
; mode 8-0Dh
@@mode0812:	cmp	al,0Dh
		jb	@@nonstandard
		mov	bl,1Eh			; PUSH DS opcode
		push	bx
		mov	bx,3E82h		; 16002: 0Dh-0Eh
		je	@@set320
; mode 0Eh-12h
		cmp	al,0Fh
		jb	@@setgraphics
		mov	di,350			; x350: 0Fh-10h
		mov	bh,7Eh			; 32386: 0Fh-10h
		cmp	al,11h
		jb	@@setgraphics
		mov	di,480			; x480: 11h-12h
		mov	bh,9Eh			; 40578: 11h-12h

@@setgraphics:	mov	[videomode],al
		mov	[nextxor],si
		saveFAR	[buffer@],es,bx
		pop	ax			; multiplanar modes opcodes
		mov	byte ptr [restorevregs],al
		mov	byte ptr [updatevregs],al
		xchg	bx,ax			; OPTIMIZE: instead MOV BH,AH
		mov	al,51h			; PUSH CX opcode
		j	@@setgcommon

@@nonstandard:
;;+++++ for text modes: bh := 0B8h, dl := 0FFh, cl := 3, j @@settext
		;mov	dx,0FFFFh
		;mov	di,200
		mov	al,0C3h			; RET opcode

@@setgcommon:	mov	byte ptr [@graphcursor],al
		mov	ax,640			; [maxcoordX]
		shr	ax,cl
		mov	[scanline],ax		; screen line width in bytes

@@setcommon:	shl	ax,cl
		mov	[maxcoordX],ax
		mov	[maxcoordY],di
		mov	byte ptr [granumask.X],dl
		mov	byte ptr [granumask.Y],dh
		mov	byte ptr cursorptr[3],bh
		mov	[bitmapshift],cl	; log2(screen/memory ratio)
						;  (mode 13h=1, 0-1/0Dh=4, other=3)
		shl	ch,cl
		mov	[cursorwidth],ch
		pop	si

;---------- set ranges and center cursor (AX=maxcoordX, DI=maxcoordY)
		mov	cx,ax
		dec	ax
		mov	[rangemax.X],ax		; set right X range
		shr	cx,1			; X middle

		mov	dx,di
		dec	di
		mov	[rangemax.Y],di		; set lower Y range
		shr	dx,1			; Y middle

;---------- set cursor position (CX=X, DX=Y, SI=area size to clear)
@setpos:	cli
		movSeg	es,ds
		mov	di,offset DGROUP:StartClearArea
		xchg	cx,si
		xor	ax,ax
		rep	stosw

		xchg	ax,dx			; OPTIMIZE: instead MOV AX,DX
IF (offset Y) eq 0
		xor	bx,bx
ELSE
		mov	bx,offset Y
ENDIF
		call	@savecutpos
		xchg	ax,si			; OPTIMIZE: instead MOV AX,SI
		mov	bl,offset X		; OPTIMIZE: BL instead BX
		jmp	@savecutpos
softreset_21	endp

;
; 1F - Disable mouse driver
;
;
; In:	none
; Out:	[AX] = 1Fh/FFFFh	(success/unsuccess)
;	[ES:BX]			(old int33 handler)
; Use:	oldint33, oldint10
; Modf:	AX, CX, DX, BX, ES, disabled?, cursorhidden
; Call:	INT 21/35, INT 21/25, disablePS2/disableCOM, hidecursor
;
disabledrv_1F	proc
		les	ax,[oldint33]
		mov	[_ARG_ES_],es
		mov	[_ARG_BX_],ax

		db	0E8h			; CALL NEAR to disableproc
disableproc	dw	disablePS2-$-2

		mov	al,0
		mov	[buttstatus],al
		inc	ax			; OPTIMIZE: instead MOV AL,1
		cmp	[disabled?],al
		je	@@disabled
		mov	[cursorhidden],al	; normalize flag
		call	hidecursor

;---------- check if INT 33 or INT 10 were intercepted
;	    (i.e. handlers segments not equal to CS)
		mov	cx,cs
		mov	ax,3533h
		int	21h
		mov	dx,es
		cmp	dx,cx
		jne	althandler_18

		mov	al,10h			; OPTIMIZE: instead MOV AX,3510h
		int	21h
		mov	ax,es
		sub	ax,cx
		jne	althandler_18

		inc	ax			; OPTIMIZE: instead MOV AL,1
		mov	[disabled?],al

;---------- restore old INT 10 handler
		mov	ax,2510h
		lds	dx,[oldint10]
		int	21h
@@disabled:	ret
disabledrv_1F	endp

;
; 18 - Set alternate user interrupt handler
;
;
; In:	CX			(call mask)
;	ES:DX			(FAR routine)
; Out:	[AX] = 18h/FFFFh	(success/unsuccess)
;
althandler_18	proc
		mov	[_ARG_AX_],0FFFFh
		ret
althandler_18	endp

;
; 19 - Get alternate user interrupt handler
;
;
; In:	CX			(call mask)
; Out:	[CX]			(0=not found)
;	[BX:DX]			(FAR routine)
;
althandler_19	proc
		mov	[_ARG_CX_],0
		ret
althandler_19	endp

;
; 00 - Reset driver and read status
;
;
; In:	none
; Out:	[AX] = 0/FFFFh		(not installed/installed)
;	[BX] = 2/3/FFFFh	(number of buttons)
; Use:	none
; Modf:	none
; Call:	softreset_21, enabledriver_20
;
resetdriver_00	proc
		call	softreset_21
		;j	enabledriver_20
resetdriver_00	endp

;
; 20 - Enable mouse driver
;
;
; In:	none
; Out:	[AX] = 20h/FFFFh	(success/unsuccess)
; Use:	none
; Modf:	AX, CX, DX, BX, ES, disabled?, oldint10
; Call:	INT 21/35, INT 21/25, setupvideo, enablePS2/enableCOM
;
enabledriver_20	proc
		xor	cx,cx
		xchg	cl,[disabled?]
		jcxz	@@enabled

;---------- set new INT 10 handler
		mov	ax,3510h
		int	21h			; get INT in ES:BX
		saveFAR	[oldint10],es,bx
		mov	ah,25h			; OPTIMIZE: instead MOV AX,2510H
		mov	dx,offset DGROUP:int10handler
		int	21h			; set INT in DS:DX
;----------
@@enabled:	call	setupvideo
		db	0E9h			; JMP NEAR to enableproc
enableproc	dw	enablePS2-$-2
enabledriver_20	endp

;
; 03 - Get cursor position and buttons status
;
;
; In:	none
; Out:	[BX]			(buttons status)
;	[CX]			(X - column)
;	[DX]			(Y - row)
; Use:	buttstatus, granpos
; Modf:	AX, CX, DX
; Call:	none
;
status_03	proc
		;mov	ah,0
		mov	al,[buttstatus]
		mov	cx,[granpos.X]
		mov	dx,[granpos.Y]
		j	@retBCDX
status_03	endp

;
; 05 - Get button press data
;
;
; In:	BX			(button number)
; Out:	[AX]			(buttons status)
;	[BX]			(press times)
;	[CX]			(last press X)
;	[DX]			(last press Y)
; Use:	none
; Modf:	SI, buttpress
; Call:	@retbuttstat
;
pressdata_05	proc
		mov	si,offset DGROUP:buttpress
		j	@retbuttstat
pressdata_05	endp

;
; 06 - Get button release data
;
;
; In:	BX			(button number)
; Out:	[AX]			(buttons status)
;	[BX]			(release times)
;	[CX]			(last release X)
;	[DX]			(last release Y)
; Use:	buttstatus
; Modf:	AX, CX, DX, BX, SI, butrelease
; Call:	none
;
releasedata_06	proc
		mov	si,offset DGROUP:buttrelease
@retbuttstat:	;mov	ah,0
		mov	al,[buttstatus]
		mov	[_ARG_AX_],ax
IF FOOLPROOF
		cmp	bx,2
		ja	@@ret56
ENDIF
ERRIF (6 ne SIZE BUTTLASTSTATE) "BUTTLASTSTATE structure size changed!"
		shl	bx,1
		add	si,bx			; SI+BX=buttrelease
		shl	bx,1			;  +button*SIZE BUTTLASTSTATE
		xor	ax,ax
		xchg	[bx+si.counter],ax
		mov	cx,[bx+si.lastcol]
		mov	dx,[bx+si.lastrow]
@retBCDX:	mov	[_ARG_CX_],cx
		mov	[_ARG_DX_],dx
@retBX:		mov	[_ARG_BX_],ax
@@ret56:	ret
releasedata_06	endp

;
; 0B - Get motion counters
;
;
; In:	none
; Out:	[CX]			(number of mickeys mouse moved
;	[DX]			 horizontally/vertically since last call)
; Use:	none
; Modf:	mickeys
; Call:	none
;
mickeys_0B	proc
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		xor	cx,cx
		xor	dx,dx
		xchg	[mickeys.X],cx
		xchg	[mickeys.Y],dx
		j	@retBCDX
mickeys_0B	endp

;
; 15 - Get driver storage requirements
;
;
; In:	none
; Out:	[BX]			(buffer size)
; Use:	LenSaveArea
; Modf:	AX
; Call:	none
;
storagereq_15	proc
		mov	ax,LenSaveArea
		j	@retBX
storagereq_15	endp

;
; 1B - Get mouse sensitivity
;
;
; In:	none
; Out:	[BX]			(number of mickeys per 8 pix
;	[CX]			 horizontally/vertically)
;	[DX]			(speed threshold in mickeys/second)
; Use:	mickey8, /doublespeed/
; Modf:	AX, CX, /DX/
; Call:	none
;
sensetivity_1B	proc
		mov	ax,[mickey8.X]
		mov	cx,[mickey8.Y]
;;+*		mov	dx,[doublespeed]
		j	@retBCDX
sensetivity_1B	endp

;
; 1E - Get display page
;
;
; In:	none
; Out:	[BX]			(display page number)
; Use:	0:462h
; Modf:	AX, DS
; Call:	none
;
videopage_1E	proc
		xor	ax,ax
		mov	ds,ax
		mov	al,ds:[462h]
		j	@retBX
videopage_1E	endp

;
; 01 - Show mouse cursor
;
;
; In:	none
; Out:	none
; Use:	none
; Modf:	AX, lowright.Y, cursorhidden
; Call:	@showcursor, @showcheck
;
showcursor_01	proc
		mov	byte ptr lowright.Y[1],-1 ; make lowright.Y negative
		cmp	[cursorhidden],ah	; OPTIMIZE: AH instead 0
		jz	@showcursor		; jump if already shown
		dec	[cursorhidden]
		j	@showcheck
showcursor_01	endp

;
; 02 - Hide mouse cursor
;
;
; In:	none
; Out:	none
; Use:	none
; Modf:	cursorhidden
; Call:	none
;
hidecursor_02	proc
		inc	[cursorhidden]
IF FOOLPROOF
		jnz	hidecursor		; jump if counter not overflow
		dec	[cursorhidden]
ENDIF
hidecursor:	mov	dx,offset DGROUP:restorescreen
		j	@cursorproc
hidecursor_02	endp

;
; 07 - Set horizontal cursor range
;
;
; In:	CX			(min X)
;	DX			(max X)
; Out:	none
; Use:	none
; Modf:	BX
; Call:	@setnewrange
;
hrange_07	proc
IF (offset X) eq 0
		xor	bx,bx
ELSE
		mov	bx,offset X
ENDIF
		j	@setnewrange
hrange_07	endp

;
; 08 - Set vertical cursor range
;
;
; In:	CX			(min Y)
;	DX			(max Y)
; Out:	none
; Use:	pos
; Modf:	CX, DX, BX, rangemin, rangemax
; Call:	setpos_04
;
vrange_08	proc
IF (offset Y) eq 0
		xor	bx,bx
ELSE
		mov	bx,offset Y
ENDIF
IF FOOLPROOF
@setnewrange:	xchg	ax,cx			; OPTIMIZE: instead MOV AX,CX
		cmp	ax,dx
		jle	@@saverange
		xchg	ax,dx
@@saverange:	mov	word ptr rangemin[bx],ax
ELSE
@setnewrange:	mov	word ptr rangemin[bx],cx
ENDIF
		mov	word ptr rangemax[bx],dx
		mov	cx,[pos.X]
		mov	dx,[pos.Y]
		;j	setpos_04
vrange_08	endp

;
; 04 - Position mouse cursor
;
;
; In:	CX			(X - column)
;	DX			(Y - row)
; Out:	none
; Use:	cursorhidden, granumask.Y
; Modf:	AX, CX, SI, videounlock, cursordrawn
; Call:	@setpos
;
setpos_04	proc
		mov	si,LenClearArea1/2	; clear area 1
ERRIF (LenClearArea1 mod 2 ne 0) "LenClearArea1 must be even!"
		call	@setpos

showcursor:	cmp	[cursorhidden],0
@showcheck:	jnz	@showret		; exit if cursor hidden

@showcursor:	mov	dx,offset DGROUP:drawcursor
@cursorproc:	dec	[videounlock]
		jnz	@@showdone		; exit if drawing in progress
		sti
		xor	cx,cx
		xchg	cl,[cursordrawn]	; mark cursor as not drawn
		mov	ax,[granumask.Y]
		inc	ax			; Zero flag if graphics
		call	dx			; CX=old cursordrawn

@@showdone:	inc	[videounlock]		; drawing stopped
@showret:	ret
setpos_04	endp

;
; 09 - Define graphics cursor
;
;
; In:	BX			(hot spot X)
;	CX			(hot spot Y)
;	ES:DX			(pointer to bitmaps)
; Out:	none
; Use:	none
; Modf:	AX, CX, BX, SI, DI, ES, hotspot, screenmask, cursormask
; Call:	@showret, hidecursor, showcursor
;
graphcursor_09	proc
;---------- compare user shape with internal area
		mov	si,offset DGROUP:hotspot
		lodsw
		cmp	ax,bx
		jne	@@redrawgraph
		lodsw
		xor	ax,cx
		jne	@@redrawgraph
		mov	di,dx
		mov	al,16+16		; OPTIMIZE: AL instead AX
		xchg	ax,cx
		repe	cmpsw
		xchg	cx,ax			; OPTIMIZE: instead MOV CX,AX
		je	@showret

;---------- copy user shape to internal area
@@redrawgraph:	push	ds ds es
		pop	ds es
		mov	di,offset DGROUP:hotspot
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		stosw
		xchg	ax,cx			; OPTIMIZE: instead MOV AX,CX
		stosw
		mov	si,dx
		mov	cx,16+16
		rep	movsw
		pop	ds

redrawscreen:	call	hidecursor
		j	showcursor
graphcursor_09	endp

;
; 0A - Define text cursor
;
;
; In:	BX			(0 - SW, else HW text cursor)
;	CX			(screen mask/start scanline)
;	DX			(cursor mask/end scanline)
; Out:	none
; Use:	none
; Modf:	AX, CX, BX, cursortype, startscan, endscan
; Call:	INT 10/01, @showret, redrawscreen
;
textcursor_0A	proc
		xchg	cx,bx
		jcxz	@@difftext		; jump if software cursor

		mov	ch,bl
		mov	cl,dl
		mov	ah,1
		int	10h			; set cursor shape & size
		mov	cl,1

@@difftext:	cmp	cl,[cursortype]
		jne	@@redrawtext
		cmp	bx,[startscan]
		jne	@@redrawtext
		cmp	dx,[endscan]
		je	@showret		; exit if cursor not changed

@@redrawtext:	mov	[cursortype],cl
		mov	[startscan],bx
		mov	[endscan],dx
		j	redrawscreen
textcursor_0A	endp

;
; 10 - Define screen region for updating
;
;
; In:	CX, DX			(X/Y of upper left corner)
;	SI, DI			(X/Y of lower right corner)
; Out:	none
; Use:	none
; Modf:	AX, CX, DX, DI, upleft, lowright
; Call:	redrawscreen
;
updateregion_10	proc
		mov	ax,[_ARG_SI_]
IF FOOLPROOF
		cmp	cx,ax
		jle	@@regionX
		xchg	cx,ax
@@regionX:	mov	[upleft.X],cx
		mov	[lowright.X],ax
		xchg	ax,di			; OPTIMIZE: instead MOV AX,DI
		cmp	dx,ax
		jle	@@regionY
		xchg	dx,ax
@@regionY:	mov	[upleft.Y],dx
		mov	[lowright.Y],ax
ELSE
		mov	[upleft.X],cx
		mov	[upleft.Y],dx
		mov	[lowright.X],ax
		mov	[lowright.Y],di
ENDIF
		j	redrawscreen
updateregion_10	endp

;
; 16 - Save driver state
;
;
; In:	BX			(buffer size)
;	ES:DX			(buffer)
; Out:	none
; Use:	StartSaveArea
; Modf:	CX, SI, DI
; Call:	none
;
savestate_16	proc
IF FOOLPROOF
;-		cmp	bx,LenSaveArea		;!!! TurboPascal IDE workaround
;-		jb	@stateret
ENDIF
		mov	di,dx
		mov	si,offset DGROUP:StartSaveArea
		mov	cx,LenSaveArea/2
ERRIF (LenSaveArea mod 2 ne 0) "LenSaveArea must be even!"
		rep	movsw
@stateret:	ret
savestate_16	endp

;
; 17 - Restore driver state
;
;
; In:	BX			(buffer size)
;	ES:DX			(saved state buffer)
; Out:	none
; Use:	none
; Modf:	SI, DI, DS, ES, StartSaveArea
; Call:	@stateret, hidecursor, showcursor
;
restorestate_17	proc
IF FOOLPROOF
;-		cmp	bx,LenSaveArea		;!!! TurboPascal IDE workaround
;-		jb	@stateret
ENDIF
;---------- do nothing if SaveArea is not changed
;;+*		mov	si,offset DGROUP:StartSaveArea
;;+*		mov	di,dx
;;+*		mov	cx,LenSaveArea/2
ERRIF (LenSaveArea mod 2 ne 0) "LenSaveArea must be even!"
;;+*		repe	cmpsw
;;+*		je	@stateret

;---------- change SaveArea
		push	es dx
		call	hidecursor
		movSeg	es,ds
		pop	si ds
		mov	di,offset DGROUP:StartSaveArea
		mov	cx,LenSaveArea/2
ERRIF (LenSaveArea mod 2 ne 0) "LenSaveArea must be even!"
		rep	movsw
		jmp	showcursor
restorestate_17	endp

;
; 0D - Light pen emulation On
;
;
; In:	none
; Out:	none
; Use:	none
; Modf:	none
; Call:	lightpenoff_0E
;
;;+*lightpenon_0D	proc
;;+*		mov	al,0
;;+*		;j	lightpenoff_0E
;;+*lightpenon_0D	endp

;
; 0E - Light pen emulation Off
;
;
; In:	none
; Out:	none
; Use:	none
; Modf:	nolightpen?
; Call:	none
;
;;+*lightpenoff_0E	proc
;;+*		mov	[nolightpen?],al	; OPTIMIZE: AL instead nonzero
;;+*		ret
;;+*lightpenoff_0E	endp

;
; 14 - Exchange user interrupt routines
;
;
; In:	CX			(new call mask)
;	ES:DX			(new FAR routine)
; Out:	[CX]			(old call mask)
;	[ES:DX]			(old FAR routine)
; Use:	callmask, UIR@
; Modf:	AX
; Call:	UIR_0C
;
exchangeUIR_14	proc
		;mov	ah,0
		mov	al,[callmask]
		mov	[_ARG_CX_],ax
		mov	ax,word ptr UIR@[0]
		mov	[_ARG_DX_],ax
		mov	ax,word ptr UIR@[2]
		mov	[_ARG_ES_],ax
		;j	UIR_0C
exchangeUIR_14	endp

;
; 0C - Define user interrupt routine
;
;
; In:	CX			(call mask)
;	ES:DX			(FAR routine)
; Out:	none
; Use:	none
; Modf:	UIR@, callmask
; Call:	none
;
UIR_0C		proc
		saveFAR [UIR@],es,dx
		mov	[callmask],cl
		ret
UIR_0C		endp

;
; 0F - Set mickeys/pixels ratios
;
;
; In:	CX			(number of mickeys per 8 pix
;	DX			 horizontally/vertically)
; Out:	none
; Use:	/doublespeed/
; Modf:	none
; Call:	sensetivity_1A
;
sensetivity_0F	proc
		mov	bx,cx
		mov	cx,dx
;;+*		mov	dx,[doublespeed]
		;j	sensetivity_1A
sensetivity_0F	endp

;
; 1A - Set mouse sensitivity
;
;
; In:	BX			(number of mickeys per 8 pix
;	CX			 horizontally/vertically)
;	DX			(speed threshold in mickeys/second)
; Out:	none
; Use:	none
; Modf:	mickey8
; Call:	doublespeed_13
;
sensetivity_1A	proc
IF FOOLPROOF
		or	bx,bx
		jz	doublespeed_13		; ignore wrong ratio
		jcxz	doublespeed_13		; ignore wrong ratio
ENDIF
		mov	[mickey8.X],bx
		mov	[mickey8.Y],cx
		;j	doublespeed_13
sensetivity_1A	endp

;
; 13 - Define double-speed threshold
;
;
; In:	DX			(speed threshold in mickeys/second)
; Out:	none
; Use:	none
; Modf:	/DX/, /doublespeed/
; Call:	none
;
doublespeed_13	proc
;;+*		or	dx,dx
;;+*		jnz	@@savespeed
;;+*		mov	dl,64
;;+*@@setspeed:	mov	[doublespeed],dx
		;ret
doublespeed_13	endp

;
; 0D 0E 11 12 18 19 1C 1D - Null function for not implemented calls
;

nullfunc	proc
		ret
nullfunc	endp

;
;				INT 33 handler
;

		evendata
handler33table	dw	offset DGROUP:resetdriver_00
		dw	offset DGROUP:showcursor_01
		dw	offset DGROUP:hidecursor_02
		dw	offset DGROUP:status_03
		dw	offset DGROUP:setpos_04
		dw	offset DGROUP:pressdata_05
		dw	offset DGROUP:releasedata_06
		dw	offset DGROUP:hrange_07
		dw	offset DGROUP:vrange_08
		dw	offset DGROUP:graphcursor_09
		dw	offset DGROUP:textcursor_0A
		dw	offset DGROUP:mickeys_0B
		dw	offset DGROUP:UIR_0C
		dw	offset DGROUP:nullfunc	;lightpenon_0D
		dw	offset DGROUP:nullfunc	;lightpenoff_0E
		dw	offset DGROUP:sensetivity_0F
		dw	offset DGROUP:updateregion_10
		dw	offset DGROUP:nullfunc	;11 - genius driver only
		dw	offset DGROUP:nullfunc	;12 - large graphics cursor
		dw	offset DGROUP:doublespeed_13
		dw	offset DGROUP:exchangeUIR_14
		dw	offset DGROUP:storagereq_15
		dw	offset DGROUP:savestate_16
		dw	offset DGROUP:restorestate_17
		dw	offset DGROUP:althandler_18
		dw	offset DGROUP:althandler_19
		dw	offset DGROUP:sensetivity_1A
		dw	offset DGROUP:sensetivity_1B
		dw	offset DGROUP:nullfunc	;1C - InPort mouse only
		dw	offset DGROUP:nullfunc	;1D - define display page #
		dw	offset DGROUP:videopage_1E
		dw	offset DGROUP:disabledrv_1F
		dw	offset DGROUP:enabledriver_20
		dw	offset DGROUP:softreset_21

handler33	proc
		cld
		or	ah,ah
		jnz	@@iret
		push	ds
		movSeg	ds,cs
		cmp	al,21h
		ja	language_23
		push	es
		PUSHALL
		mov	si,ax			;!!! AX must be unchanged
		mov	bp,sp
		shl	si,1
		call	handler33table[si]	; call by calculated offset
@rethandler:	POPALL
		pop	es ds
@@iret:		iret

;
; 23 - Get language for messages
; Out:	[BX]					(language code: 0 - English)
;
language_23:	cmp	al,23h
		je	@@iretBX0

;
; 24 - Get software version, mouse type and IRQ
; Out:	[AX] = 24h/FFFFh			(installed/error)
;	[BX]					(version)
;	[CH] = 1=bus/2=serial/3=InPort/4=PS2/5=HP (mouse type)
;	[CL]					(interrupt #/0=PS/2)
; Use:	driverversion
;
version_24:	cmp	al,24h
		jne	no_version_24
		mov	bx,driverversion
		db	0B9h			; MOV CX,word
mouseinfo	db	?,4
no_version_24:

;
; 26 - Get maximum virtual screen coordinates
; Out:	[BX]					(mouse disabled flag)
;	[CX]					(max virtual screen X)
;	[DX]					(max virtual screen Y)
; Use:	maxcoordY
;
maxscreen_26:	cmp	al,26h
		jne	no_maxscreen_26
		db	0BBh			; MOV BX,word
disabled?	db	1,0			; 1 - driver disabled
		db	0B9h			; MOV CX,word
maxcoordX	dw	?			; screen width
		mov	dx,[maxcoordY]
		dec	cx
		dec	dx
no_maxscreen_26:

;
; 27 - Get screen/cursor masks and mickey counters
; Out:	[AX]			(screen mask/start scanline)
;	[BX]			(cursor mask/end scanline)
;	[CX]			(number of mickeys mouse moved
;	[DX]			 horizontally/vertically since last call)
; Modf:	mickeys
;
;;+*cursor_27:	cmp	al,27h
;;+*		jne	no_cursor_27
;;+*		mov	ax,[startscan]
;;+*		mov	bx,[endscan]
;;+*		xor	cx,cx
;;+*		xor	dx,dx
;;+*		xchg	[mickeys.X],cx
;;+*		xchg	[mickeys.Y],dx
;;+*		pop	ds
;;+*		iret
;;+*no_cursor_27:

;
; 31 - Get current virtual cursor coordinates
; Out:	[AX]					(min virtual cursor X)
;	[BX]					(min virtual cursor Y)
;	[CX]					(max virtual cursor X)
;	[DX]					(max virtual cursor Y)
; Use:	rangemin, rangemax
;
cursrange_31:	cmp	al,31h
		jne	no_cursrange_31
		mov	ax,[rangemin.X]
		mov	bx,[rangemin.Y]
		lds	cx,[rangemax]
		mov	dx,ds
		pop	ds
		iret
no_cursrange_31:

;
; 32 - Get active advanced functions
; Out:	[AX]					(active functions flag)
;	[CX] = 0
;	[DX] = 0
;	[BX] = 0
;
active_32:	cmp	al,32h
		jne	no_active_32
		mov	ax,0100010000001100b	; active: 26 2A 31 32
		xor	cx,cx
		xor	dx,dx
@@iretBX0:	xor	bx,bx
		pop	ds
		iret
no_active_32:

;
; 4D - Get pointer to copyright string
; Out:	[ES:DI]					(copyright string)
; Use:	IDstring
;
copyright_4D:	cmp	al,4Dh
		jne	no_copyright_4D
		movSeg	es,ds
		mov	di,offset DGROUP:IDstring
no_copyright_4D:

;
; 6D - Get pointer to version
; Out:	[ES:DI]					(version string)
; Use:	msversion
;
version_6D:	cmp	al,6Dh
		jne	no_version_6D
		movSeg	es,ds
		mov	di,offset DGROUP:msversion
no_version_6D:

;
; 2A - Get cursor hot spot
; Out:	[AX]					(cursor visibility counter)
;	[BX]					(hot spot X)
;	[CX]					(hot spot Y)
;	[DX] = 1=bus/2=serial/3=InPort/4=PS2/5=HP (mouse type)
; Use:	cursorhidden, hotspot
;
hotspot_2A:	cmp	al,2Ah
		jne	@@iretDS
		;mov	ah,0
		mov	al,[cursorhidden]
		mov	bx,[hotspot.X]
		mov	cx,[hotspot.Y]
		db	0BAh			; MOV DX,word
mouseinfo1	db	4,0
@@iretDS:	pop	ds
		iret
handler33	endp

; END OF INT 33 SERVICES 


RILversion	label
msversion	db	driverversion / 100h,driverversion MOD 100h
IDstring	db	'CuteMouse ',CTMVER,0
IDstringlen = $ - IDstring

TSRend		label


; INITIALIZATION PART DATA 

.data

SaveMemStrat	dw	0
SaveUMBLink	dw	0
XMSentry	dd	0

mousetype	db	?			; 0=PS/2,1=MSys,2=LT,3=MS,4=WM
options		db	0
OPT_PS2		equ	00000001b
OPT_SERIAL	equ	00000010b
OPT_COMforced	equ	00000100b
OPT_PS2after	equ	00001000b
OPT_3button	equ	00010000b
OPT_NOMSYS	equ	00100000b
OPT_LEFTHAND	equ	01000000b
OPT_NOUMB	equ	10000000b

OPTION		struc
  C		db	?
  M		db	0
  PROC@		dw	?
	ends

		evendata
OPTABLE		OPTION	<'P',OPT_PS2,			@ret>
		OPTION	<'S',OPT_SERIAL,		_serialopt>
;;+		OPTION	<'A',OPT_SERIAL+OPT_COMforced,	_addropt>
		OPTION	<'Y',OPT_NOMSYS,		@ret>
		OPTION	<'V',OPT_PS2after,		@ret>
		OPTION	<'3' and not 20h,OPT_3button,	@ret>
		OPTION	<'R',,				_resolution>
		OPTION	<'L',OPT_LEFTHAND,		@ret>
		OPTION	<'B',,				_checkdriver>
		OPTION	<'W',OPT_NOUMB,			@ret>
		OPTION	<'U',,				unloadTSR>
		OPTION	<'?' and not 20h,,		EXITMSG>
OPTABLEend	label

INCLUDE ctmouse.msg

.code

; Real Start 

real_start:	cld
;---------- save old INT 33h handler
		mov	ax,3533h
		int	21h
		saveFAR [oldint33],es,bx

;---------- parse command line and find mouse
@@start:	PrintS	Copyright		; 'Cute Mouse Driver'
		mov	si,80h
		lodsb
		cbw				; OPTIMIZE: instead MOV AH,0
		mov	bx,ax
		mov	[si+bx],ah		; OPTIMIZE: AH instead 0
		call	commandline		; examine command line

		mov	cx,word ptr oldint33[2]
		jcxz	@@startfind
		mov	ax,1Fh			; disable old driver
		pushf				;!!! Logitech MouseWare
		call	[oldint33]		;  Windows driver workaround
;----------
@@startfind:	mov	al,[options]
		test	al,OPT_PS2+OPT_SERIAL
		jnz	@@find1COM
		or	al,OPT_PS2+OPT_SERIAL

@@find1COM:	test	al,OPT_PS2after
		jz	@@findPS2
		call	searchCOM
		jnc	@@serialfound

@@findPS2:	test	al,OPT_PS2+OPT_PS2after
		jz	@@find2COM
		push	ax
		mov	al,0
		mov	mouseinfo[0],al
		mov	[mousetype],al
		call	checkPS2
		pop	ax
		jnc	@@mousefound

@@find2COM:	test	al,OPT_PS2after
		jnz	@@devnotfound
		test	al,OPT_SERIAL+OPT_NOMSYS
		jz	@@devnotfound
		call	searchCOM		; call if no /V and serial
		jnc	@@serialfound

@@devnotfound:	mov	dx,offset DGROUP:E_notfound ; 'Error: device not found'
		jmp	EXITENABLE

@@serialfound:	mov	al,2
		mov	mouseinfo[1],al
		mov	[mouseinfo1],al

;---------- check if CuteMouse driver already installed
@@mousefound:	call	getCuteMouse
		mov	dx,offset DGROUP:S_reset ; 'Resident part reset to'
		mov	bx,4C02h		; terminate, al=return code
		je	@@setupdriver

;---------- allocate UMB memory, if possible, and set INT 33 handler
		mov	bx,(TSRend-TSRstart+15) SHR 4
		push	bx
		call	prepareTSR		; new memory segment in ES
		mov	si,offset DGROUP:oldint33
		mov	di,si
		movsw				; copy oldint33 contents
		movsw
		push	ds
		movSeg	ds,es
		mov	[disabled?],1		; copied back in setupdriver
		mov	ax,2533h		; setup new INT 33 handler
		mov	dx,offset DGROUP:handler33
		int	21h			; set INT in DS:DX
		pop	ds ax
		mov	dx,offset DGROUP:S_installed ; 'Installed at'
		mov	bl,0			; errorlevel
;----------
@@setupdriver:	push	ax			; size of TSR for INT 21/31
		PrintS
		mov	cl,[mousetype]
		mov	dx,offset DGROUP:S_forPS2 ; 'PS/2 port'
		or	cl,cl
		jz	@@printmode

		PrintS	S_atCOM
		mov	dx,offset DGROUP:S_forWM ; 'Microsoft wheel mouse'
		cmp	cl,4
		je	@@printmode
		mov	dx,offset DGROUP:S_forMS ; 'Microsoft mouse'
		cmp	cl,2
		ja	@@printmode
		mov	dx,offset DGROUP:S_forLT ; 'Logitech MouseMan'
		je	@@printmode
		mov	dx,offset DGROUP:S_forMSYS ; 'Mouse Systems mouse'
		inc	bx			; OPTIMIZE: BX instead BL

@@printmode:	push	bx			; exit function and errorlevel
		PrintS
		call	setupdriver

;---------- close all handles (20 pieces) to prevent decreasing system
;	    pool of handles if INT 21/31 used
		mov	bx,19
@@closeall:	mov	ah,3Eh
		int	21h			; close file in BX
		dec	bx
		jns	@@closeall
;----------
		pop	ax dx
		int	21h

;
;		Setup resident driver code and parameters
;

setupdriver	proc
		;mov	cl,[mousetype]

;---------- detect VGA card (VGA ATC have one more register)
		mov	ax,1A00h
		int	10h			; get display type in bx
		cmp	al,1Ah
		jne	@@lhand
		xchg	ax,bx			; OPTIMIZE: instead MOV AL,BL
		cmp	al,7			; monochrome VGA?
		je	@@setATCVGA
		cmp	al,8			; color VGA?
		jne	@@lhand
@@setATCVGA:	inc	videoregs@[(SIZE RGROUPDEF)*3].regscnt

;---------- setup left hand mode handling
@@lhand:	mov	al,0EBh			; JMP SHORT
		mov	bl,07Ah			; JPE
		or	cl,cl
		jz	@@checklhand		; jump if PS/2 mouse
		xchg	ax,bx
@@checklhand:	test	[options],OPT_LEFTHAND
		jz	@@setlhand
		xchg	ax,bx
@@setlhand:	mov	[LEFTHANDCODE],al

;---------- setup buttons count and mask
		mov	al,3
		test	[options],OPT_3button
		jnz	@@sethandler
		or	cl,cl
		je	@@setbuttons		; jump if PS/2 mouse
		cmp	cl,al			; OPTIMIZE: AL instead 3
		jne	@@sethandler		; jump if no MS mode
@@setbuttons:	mov	[buttonsmask],al	; OPTIMIZE: AL instead 0011b
		dec	ax			; OPTIMIZE: instead MOV AL,2
		mov	[buttonscnt],al

;---------- setup mouse handlers code
@@sethandler:	cmp	cl,1
		jb	@@setother		; jump if PS/2 mouse

		mov	word ptr [IRQproc],20B0h ; MOV AL,20h
		mov	[BUFFADDR],offset DGROUP:bufferSERIAL
		mov	[enableproc],enableCOM-enableproc-2
		mov	[disableproc],disableCOM-disableproc-2
		je	@@setother		; jump if Mouse Systems mouse
		mov	[mouseproc],MSLTproc-mouseproc-2
		cmp	cl,3
		jb	@@setother		; jump if Logitech mouse
		mov	MSLTCODE3[1],2
		ja	@@setother		; jump if wheel mouse

		cmp	al,2
		je	@@setMSproc		; jump if MS2
		mov	[MSLTCODE2],075h	; JNZ
@@setMSproc:	mov	al,0C3h			; RET
		mov	[MSLTCODE1],al
		mov	[MSLTCODE3],al

;---------- setup, if required, other parameters
@@setother:	push	es ds es ds
		pop	es ds
		mov	si,offset DGROUP:oldint10
		mov	di,si
		movsw				; get back [oldint10]...
		movsw
		mov	al,[disabled?]
		pop	ds
		mov	[disabled?],al		; ...and [disabled?]

		mov	al,byte ptr [IRQintnum]	; save old IRQ handler
		mov	ah,35h
		int	21h			; get INT in ES:BX
		mov	ax,es
		pop	es
		mov	di,offset DGROUP:oldIRQaddr
		xchg	ax,bx
		stosw
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		stosw

;---------- copy TSR image (even if ES=DS - this is admissible)
		mov	si,offset DGROUP:TSRdata
		mov	di,si
		mov	cx,(TSRend-TSRdata+1)/2
		rep	movsw

;---------- call INT 33/0000 (Reset driver)
		pop	ax
		pushf				;!!! Logitech MouseWare
		push	cs ax			;  Windows driver workaround
		mov	ax,offset DGROUP:handler33
		push	es ax
		xor	ax,ax			; reset driver
		retf
setupdriver	endp

;
;		Check given or all COM-ports for mouse connection
;

searchCOM	proc
		;mov	[COMLCR],2
		mov	dx,offset DGROUP:detectmouse
		call	COMloop
		jnc	@searchret

		test	[options],OPT_NOMSYS
		stc
		jnz	@searchret

		mov	[COMLCR],3
		mov	[mousetype],1		; set Mouse Systems mode
		mov	dx,offset DGROUP:checkUART
		;j	COMloop
searchCOM	endp

;

COMloop		proc
		push	ax
		mov	cx,1			; scan only current COM port
		test	[options],OPT_COMforced
		jnz	@@checkCOM
		mov	cl,4			; scan all COM ports

@@COMloop:	push	cx
		mov	al,'5'
		sub	al,cl
		call	setCOMport
		pop	cx
@@checkCOM:	push	cx dx
		mov	si,[IO_address]
		call	dx
		pop	dx cx
		jnc	@@searchbreak
		loop	@@COMloop
@@searchbreak:	pop	ax
@searchret:	ret
COMloop		endp

;
;			Check if UART available
;
;
; In:	SI = [IO_address]
; Out:	Carry flag		(no UART detected)
; Use:	none
; Modf:	AX, DX
; Call:	none
;
checkUART	proc
		or	si,si
		jz	@@noUART

;---------- check UART registers for dummy bits
		lea	dx,[si+4]		; {3FCh} MCR (modem ctrl reg)
		 in	ax,dx			; {3FDh} LSR (line status reg)
		test	al,11100000b		; =0E0h; reserved bits
		 jnz	@@noUART
		inc	dx
		 in	al,dx			; {3FDh} LSR (line status reg)
		inc	ax
		 jz	@@noUART		; jump if AX was 0FFFFh

;---------- check LCR function
		cli
		dec	dx
		 dec	dx
		 mov	al,10111111b		; =0BFh
		 out	dx,al			; {3FBh} LCR: DLAB on, space
		in	al,dx			;  parity, stop=2, length=8
		 mov	ah,al
		mov	al,00000010b		; =2
		 out	dx,al			; {3FBh} LCR: DLAB off, no
		 in	al,dx			;  parity, stop=1, length=7
		sti
		cmp	ax,1011111100000010b	; =0BF02h
		 jne	@@noUART
;----------
		dec	dx
		 dec	dx
		 in	al,dx			; {3F9h} IER (int enable reg)
		test	al,11110000b		; =0F0h; reserved bits
		;clc
		 jz	@@UARTret

@@noUART:	stc
@@UARTret:	ret
checkUART	endp

;
;			Detect mouse type if present
;
;
; In:	SI = [IO_address]
; Out:	Carry flag		(no UART or mouse found)
; Use:	0:46Ch, COMLCR
; Modf:	AX, CX, DX, BX, ES, mousetype
; Call:	checkUART, readUART
;
detectmouse	proc
		call	checkUART
		jc	@searchret

		lea	dx,[si+3]
		 mov	ax,0000000010000000b	; =0080h
		 out	dx,ax			; {3FBh} LCR: DLAB on
		xchg	dx,si			; {3FCh} MCR: DTR/RTS/OUT2 off
		 ;mov	ah,0
		 mov	al,96			; 1200 baud rate
		 out	dx,ax			; {3F8h},{3F9h} divisor latch
		xchg	dx,si
		 mov	al,[COMLCR]		; {3FBh} LCR: DLAB off, no
		 out	dx,al			;  parity, stop=1, length=7/8

		inc	dx
		push	dx
		 inc	dx			; {3FDh} LSR: clear error bits
		 in	ax,dx			; {3FEh} MSR: clear state bits
		mov	dx,si
		 in	al,dx			; {3F8h} flush receive buffer
		inc	dx
		 ;in	ax,dx			; {3FAh} IIR: clear THRE event
		 xor	ax,ax			; {3F9h} IER: interrupts off
		 out	dx,ax			; {3FAh} FCR: disable FIFO

;---------- wait current+next timer tick then raise RTS line
		xor	ax,ax
		 mov	es,ax
@@tickget:	mov	dx,es:[46Ch]
@@tickwait:	cmp	dx,es:[46Ch]
		 je	@@tickwait		; wait timer tick change
		xor	al,1
		 jnz	@@tickget

		pop	dx
		 mov	al,3
		 out	dx,al			; {3FCh} MCR: DTR/RTS on, OUT2 off

;---------- detect if Microsoft or Logitech mouse present
		mov	bx,0103h		; bl=mousetype, bh=no `M'
		mov	cx,4			; scan 4 first bytes
		inc	dx			; SI=3F8h, DX=3FDh
@@detmloop:	call	readUART
		jc	@@detmdone
		cmp	al,8
		 je	@@skipPNP		; jump if PNP data starts
		cmp	al,'M'
		 jne	@@checkWM
		 mov	bh,0			; MS compatible mouse found...
@@checkWM:	cmp	al,'Z'
		 jne	@@checkLT
		 mov	bl,4			; ...wheel mouse found
@@checkLT:	cmp	al,'3'
		 jne	@@detmnext
		 mov	bl,2			; ...Logitech mouse found
@@detmnext:	loop	@@detmloop

@@skipPNP:	mov	ch,91			; =5*18.2 ticks, ~5 seconds
@@PNPtime:	mov	cl,es:[46Ch]		; current timer value
@@PNPloop:	call	readUART
		 jc	@@detmdone		; jump if no more data
		cmp	cl,es:[46Ch]
		 je	@@PNPloop		; loop if same tick
		dec	ch
		 jnz	@@PNPtime		; loop if not timeout end 

@@detmdone:	mov	[mousetype],bl		; save 2=LT, 3=MS or 4=WM
		neg	bh			; nonzero makes carry flag
		ret
detectmouse	endp

;
; In:	DX = [IO_address]+5
;	SI = [IO_address]
;	ES = 0
; Out:	Carry flag		(set if timeout)
;	AL			(received data when no timeout)
; Use:	0:46Ch
; Modf:	AH, DI
;
readUART	proc
		mov	ah,2+1			; length of silence in ticks
						; (include rest of curr tick)
@@readloop:	mov	di,es:[46Ch]
@@readwait:	in	al,dx			; {3FDh} LSR (line status reg)
		test	al,1
		 jnz	@@readret		; jump if data ready
		cmp	di,es:[46Ch]
		 je	@@readwait		; jump if same tick
		dec	ah
		 jnz	@@readloop		; wait next tick of silence
@retC:		stc				; indicate timeout
		ret

@@readret:	xchg	dx,si
		 in	al,dx			; {3F8h} receive byte
		xchg	dx,si
		;clc
		ret
readUART	endp

;
;				Check for PS/2
;
;
; In:	none
; Out:	Carry flag
; Use:	none
; Modf:	AX, CX, BX
; Call:	INT 11, INT 15/C2xx
;
checkPS2	proc
		int	11h			; get equipment list
		test	al,00000100b		; =4
		jz	@retC			; jump if PS/2 not indicated
		mov	bh,3
		PS2serv 0C205h,@retC		; initialize mouse, bh=datasize
		mov	bh,3
		PS2serv 0C203h,@retC		; set mouse resolution bh
		movSeg	es,cs
		mov	bx,offset DGROUP:PS2dummy
		PS2serv	0C207h,@retC		; mouse, es:bx=ptr to handler
		xor	bx,bx
		mov	es,bx
		PS2serv	0C207h			; mouse, es:bx=ptr to handler
		;clc
		ret
PS2dummy:	retf
checkPS2	endp

;
;				Set COM port
;
;
; In:	AL			(COM port letter)
; Out:	none
; Use:	0:400
; Modf:	AX, BX, DI, ES, com_port
; Call:	setIRQ, setIOaddr
;
setCOMport	proc
		mov	[com_port],al
		sub	al,'1'

		xor	bx,bx
		mov	es,bx
		mov	bl,al

		shr	al,1			; 0=COM1, 1=COM2, ...
		mov	al,'4'			; IRQ4 for COM1/3
		sbb	al,0			; IRQ3 for COM2/4
		call	setIRQ

		shl	bx,1
		mov	ax,es:400h[bx]
		movSeg	es,ds
		mov	di,offset DGROUP:S_atIO
		;j	setIOaddr
setCOMport	endp

;
;				Set I/O address
;
;
; In:	AX			(I/O address)
;	ES:DI			(string for 4 digits)
; Out:	none
; Use:	none
; Modf:	CL, DI, IO_address, S_atIO
; Call:	none
;
setIOaddr	proc
		mov	[IO_address],ax
		mov	cl,12
@@loophex:	push	ax
		shr	ax,cl
		and	al,0Fh
		cmp	al,10
		sbb	al,69h
		das
		stosb
		pop	ax
		sub	cl,4
		jae	@@loophex
		ret
setIOaddr	endp

;
;				Set IRQ number
;
;
; In:	AL			(IRQ number letter)
; Out:	none
; Use:	none
; Modf:	AL, CL, IRQno, mouseinfo, IRQintnum, PICstate, notPICstate
; Call:	none
;
setIRQ		proc
		mov	[IRQno],al
		sub	al,'0'
		mov	mouseinfo[0],al
		mov	cl,al
		add	al,8
		mov	byte ptr [IRQintnum],al
		mov	al,1
		shl	al,cl
		mov	[PICstate],al		; PIC interrupt disabler
		not	al
		mov	[notPICstate],al	; PIC interrupt enabler
		ret
setIRQ		endp

;
;		Check if CuteMouse driver is installed
;
;
; In:	none
; Out:	Zero flag		(zero if installed)
;	ES			(driver segment)
; Use:	oldint33, IDstring
; Modf:	AX, CX, SI, DI
; Call:	INT 33/004D
;
getCuteMouse	proc
		mov	ax,word ptr oldint33[2]
		neg	ax
		sbb	ax,ax
		inc	ax
		jne	@ret			; jump if handler seg is zero
		mov	al,4Dh			; OPTIMIZE: AL instead AX
		pushf				;!!! Logitech MouseWare
		call	[oldint33]		;  Windows driver workaround
		cmp	di,offset DGROUP:IDstring
		jne	@ret
		mov	si,di
		mov	cx,IDstringlen
		rep	cmpsb
@ret:		ret
getCuteMouse	endp


; COMMAND LINE PARSING 

;
;			Parse Serial option
;

_serialopt	proc
		lodsb
		cmp	al,'1'
		jb	@@fixret
		cmp	al,'4'
		ja	@@fixret
		or	[options],OPT_COMforced
		call	setCOMport		; '/Sc' -> set COM port
		lodsb
		cmp	al,'2'
		jb	@@fixret
		cmp	al,'7'
		jbe	setIRQ			; '/Sci' -> set IRQ line
@@fixret:	dec	si			; fixup for command line ptr
		ret
_serialopt	endp

;
;			Parse Resolution option
;

_resolution	proc
		;mov	ah,0
		lodsb				; first argument
		sub	al,'0'
		cmp	al,9
		ja	@@fixupres		; jump if AL < 0 || AL > 9

		mov	ah,al
		lodsb				; second argument
		sub	al,'0'
		cmp	al,9
		jbe	@@setres		; jump if AL >= 0 && AL <= 9

@@fixupres:	dec	si			; fixup for command line ptr
		mov	al,ah			; drop invalid argument
@@setres:	mov	[mresolutionX],ah
		mov	[mresolutionY],al
		ret
_resolution	endp

;
;		Check if mouse services already present
;

_checkdriver	proc
		mov	cx,word ptr oldint33[2]
		jcxz	@ret
		;mov	ah,0
		mov	al,21h			; OPTIMIZE: AL instead AX
		int	33h
		inc	ax
		jnz	@ret
		mov	dx,offset DGROUP:E_mousepresent ; 'Mouse service already...'
		j	EXITMSG
_checkdriver	endp

;
; In:	DS:SI			(null terminated command line)
;
commandline	proc
		lodsb
		or	al,al
		jz	@ret			; exit if end of command line
		cmp	al,' '
		jbe	commandline		; skips spaces and controls
		cmp	al,'/'			; option character?
		jne	@@badoption
		lodsb
		and	al,not 20h		; upper case

		mov	dx,offset DGROUP:Syntax	; 'Options:'
		mov	bx,offset DGROUP:OPTABLE
@@optloop:	cmp	al,[bx].C
		jne	@@nextopt
		mov	ah,[bx].M
		or	[options],ah
		call	[bx].PROC@
		j	commandline
@@nextopt:	add	bx,SIZE OPTION
		cmp	bx,offset DGROUP:OPTABLEend
		jb	@@optloop

@@badoption:	mov	dx,offset DGROUP:E_option ; 'Error: Invalid option'

EXITMSG:	mov	bx,dx
		inc	dx
		mov	bl,[bx]
		mov	bh,4Ch			; terminate, al=return code
		PrintS
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		int	21h
commandline	endp


; TSR MANAGEMENT 

;
;			Unload driver and quit
;

unloadTSR	proc
		call	getCuteMouse		; check if CTMOUSE installed
		mov	dx,offset DGROUP:E_nocute ; 'CuteMouse driver is not installed!'
		jne	EXITMSG

		push	es
		mov	ax,1Fh			; disable CuteMouse driver
		pushf				;!!! Logitech MouseWare
		call	[oldint33]		;  Windows driver workaround
		mov	cx,es
		pop	es
		cmp	al,1Fh
		mov	dx,offset DGROUP:E_notunload ; 'Driver unload failed...'
		jne	@@exitenable

		saveFAR	[oldint33],cx,bx
		push	ds
		mov	ds,cx
		mov	dx,bx
		mov	ax,2533h		; restore old int33 handler
		int	21h			; set INT in DS:DX
		pop	ds
		call	FreeMem
		mov	dx,offset DGROUP:S_unloaded ; 'Driver successfully unloaded...'
EXITENABLE:	mov	cx,word ptr oldint33[2]
		jcxz	EXITMSG			; exit if no old driver
@@exitenable:	mov	ax,20h			; enable old driver
		pushf				;!!! Logitech MouseWare
		call	[oldint33]		;  Windows driver workaround
		j	EXITMSG
unloadTSR	endp

;
; Prepare memory for TSR
;
; In:	BX			(TSR size)
;	DS			(PSP segment)
; Out:	ES			(memory segment to be TSR)
;	BH			(exit code for INT 21)
; Use:	PSP:2Ch, MCB:8
; Modf:	AX, CX, DX, SI, DI
; Call:	INT 21/49, AllocUMB
;
prepareTSR	proc
		mov	cx,ds:[2Ch]
		jcxz	@@allocmem		; suggested by Matthias Paul
		mov	es,cx			; release environment
		mov	ah,49h
		int	21h
		mov	word ptr ds:[2Ch],0	; suggested by Matthias Paul

@@allocmem:	call	AllocUMB
		mov	ax,ds
		mov	es,dx
		mov	bh,31h			; TSR exit, al=return code
		cmp	dx,ax
		je	@@prepareret		; exit if TSR "in place"

		push	ds
		dec	ax			; current MCB
		dec	dx			; target MCB...
		mov	es,dx
		inc	dx
		mov	es:[1],dx		; ...set owner to itself
		mov	ds,ax
		mov	si,8
		mov	di,si
		movsw				; ...and copy process name
		movsw
		movsw
		movsw
		pop	ds

		mov	ah,26h
		int	21h			; create PSP
		mov	ax,es:[3]
		mov	es,dx			; target segment...
		add	ax,dx			; =targetMCB.size+targetPSP
		mov	es:[2],ax		; ...fix "seg beyond memblock"

		mov	bh,4Ch			; terminate, al=return code
@@prepareret:	ret
prepareTSR	endp


; MEMORY HANDLING 

;
; Get XMS handler address
;
; In:	none
; Out:	Carry flag		(set if no XMS support)
; Use:	none
; Modf:	AX, BX, XMSentry
; Call:	INT 2F/4300, INT 2F/4310
;
getXMSaddr	proc
		mov	ax,4300h
		int	2fh			; XMS: installation check
		cmp	al,80h
		stc
		jne	@@getXMSret		; return if XMS not present
		push	es
		mov	ax,4310h		; XMS: Get Driver Address
		int	2Fh
		saveFAR [XMSentry],es,bx
		pop	es
		clc
@@getXMSret:	ret
getXMSaddr	endp

;
; Save allocation strategy
;
; In:	none
; Out:	Carry flag		(no UMB link supported)
; Use:	none
; Modf:	AX, SaveMemStrat, SaveUMBLink
; Call:	INT 21/5800, INT 21/5802
;
SaveStrategy	proc
		mov	ax,5800h
		int	21h			; get DOS memory alloc strategy
		mov	[SaveMemStrat],ax
		mov	ax,5802h
		int	21h			; get UMB link state
		mov	byte ptr [SaveUMBLink],al
		ret
SaveStrategy	endp

;
; Restore allocation strategy
;
; In:	none
; Out:	none
; Use:	SaveMemStrat, SaveUMBLink
; Modf:	AX, BX
; Call:	INT 21/5801, INT 21/5803
;
RestoreStrategy	proc
		mov	bx,[SaveMemStrat]
		mov	ax,5801h
		int	21h			; set DOS memory alloc strategy
		mov	bx,[SaveUMBLink]
		mov	ax,5803h
		int	21h			; set UMB link state
		ret
RestoreStrategy	endp

;
; Allocate high memory
;
; In:	BX			(required memory size in para)
;	DS			(current memory segment)
; Out:	DX			(seg of new memory or DS)
; Use:	XMSentry
; Modf:	AX, ES
; Call:	INT 21/48, INT 21/49, INT 21/58,
;	SaveStrategy, RestoreStrategy, getXMSaddr
;
AllocUMB	proc
		push	bx
		test	[options],OPT_NOUMB
		jnz	@@allocasis		; jump if UMB prohibited
		mov	ax,ds
		cmp	ah,0A0h
		jae	@@allocasis		; jump if already loaded hi

;---------- check if UMB is DOS type
		call	SaveStrategy
		mov	bx,1			; add UMB to MCB chain
		mov	ax,5803h
		int	21h			; set UMB link state

; try to set a best strategy to allocate DOS supported UMBs
		mov	bl,41h			; OPTIMIZE: BL instead BX
		mov	ax,5801h		; hi mem, best fit
		int	21h			; set alloc strategy
		jnc	@@allocDOSUMB

; try a worse one then
		mov	bl,81h			; OPTIMIZE: BL instead BX
		mov	ax,5801h		; hi mem then low mem, best fit
		int	21h			; set alloc strategy

@@allocDOSUMB:	pop	bx
		push	bx
		mov	ah,48h
		int	21h			; allocate UMB memory (size=BX)
		pushf
		xchg	dx,ax			; OPTIMIZE: instead MOV DX,AX
		call	RestoreStrategy		; restore allocation strategy
		popf
		jc	@@allocXMSUMB
		cmp	dh,0A0h			; exit if allocated mem is
		jae	@@allocret		;  is above 640k,
		mov	es,dx
		mov	ah,49h			;  else free it
		int	21h

;---------- try XMS driver to allocate UMB
@@allocXMSUMB:	call	getXMSaddr
		jc	@@allocasis
		pop	dx
		push	dx
		mov	ah,10h			; XMS: Request UMB (size=DX)
		call	[XMSentry]		; ...AX=1 -> BX=seg, DX=size
		dec	ax
		jnz	@@allocasis
		pop	ax
		push	ax
		cmp	bx,ax
		mov	dx,bx
		jae	@@allocret
		mov	ah,11h			; XMS: Release UMB (seg=DX)
		call	[XMSentry]

;---------- use current memory segment
@@allocasis:	mov	dx,ds

@@allocret:	pop	bx
		ret
AllocUMB	endp

;
; In:	ES			(segment to free)
; Out:	none
; Use:	XMSentry
; Modf:	AH, DX
; Call:	INT 21/49, getXMSaddr
;
FreeMem		proc
		call	getXMSaddr
		jc	@@DOSfree

		mov	dx,es
		mov	ah,11h			; XMS: Release UMB
		call	[XMSentry]

@@DOSfree:	mov	ah,49h
		int	21h			; free allocated memory
		ret
FreeMem		endp

;

		end	start
