/*  Mem

    Copyright (c) Express Software 1997.
    All Rights Reserved.

    Created by: Joseph Cosentino.
    With help from Michal Meller for the convmemfree routine.

    Clean up and bug fixes March 2001, Bart Oldeman

*/

// I N C L U D E S //////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <mem.h>
#include <string.h>

// D E F I N E S ////////////////////////////////////////////////////////////

#define BYTE   unsigned char
#define uchar  unsigned char
#define ushort unsigned int
#define ulong  unsigned long

#define FALSE 0
#define TRUE  1

#define MAX_ITEM   150
#define MAX_VEC    100
#define MAX_DRIVE  20
#define MAX_HANDLE 255
#define MT_NONE    0
#define MT_SYSCODE 1
#define MT_SYSDATA 2
#define MT_PROGRAM 3
#define MT_DEVICE  4
#define MT_ENV     5
#define MT_DATA    6
#define MT_FREE    7
#define MT_MAP     8
#define PAGE_SIZE  23

// S T R U C T U R E S //////////////////////////////////////////////////////

typedef struct device_header
{
    struct device_header *next;
    ushort attr;
    ushort strategy_rutin;
    ushort interrupt_rutin;
    uchar name[8];
} DEVICE_HEADER;

typedef struct
{
    struct device_header *addr;
    uchar devname[9];
    uchar progname[9];
    ushort attr;
    uchar drive_num;
    uchar drives[MAX_DRIVE];
} DEVINFO;

typedef struct dpb
{
    uchar drive_num;
    uchar unit_number;
    ushort bytes_in_sector;
    uchar lsec_num;
    uchar log_base;
    ushort reserved_num;
    uchar FAT_num;
    ushort rootentry_num;
    ushort first_sector;
    ushort largest_cluster;
    ushort sectors_in_FAT;
    ushort root_firstsector;
    DEVICE_HEADER *device_addr;
    uchar media_desc;
    uchar block_flag;
    struct dpb *next_dpb;
} DPB;

typedef struct
{
    uchar type;
    ushort start;
    ushort size;
    uchar unused[3];
    uchar name[8];
} SD_HEADER;

typedef struct
{
    uchar type;
    ushort owner;
    ushort size;
    uchar unused[3];
    uchar name[8];
} MCB;

typedef struct
{
    uchar type;
    ushort seg;
    ushort owner;
    ushort environment;
    uchar name[10];
    ulong size;
    uchar vecnum;
    uchar vectors[MAX_VEC];
} MINFO;

typedef struct
{
    ushort handle;
    ushort pages;
} EMS_HANDLE;

typedef struct
{
    ushort handle;
    ulong size;
    ushort locks;
} XMS_HANDLE;

struct MCBheader {
    char type;
    unsigned int pid;
    unsigned int length;
    char reserved[3];
    char progname[8];
};

// G L O B A L S ////////////////////////////////////////////////////////////

char EnvString[50];
static MCB *first_mcb=NULL;
static MINFO mlist[MAX_ITEM];
static DEVICE_HEADER *first_dev=NULL;
static DPB *first_dpb=NULL;
static DEVINFO dlist[MAX_ITEM];
static ushort mlistnum=0,dlistnum=0,ems_totalhandle=0, ems_freehandle=0, ems_usehandle=0,ems_frame=0;
static uchar *typenames[]= { "             ", "system code  ", "system data  ", "program      ", "device driver", "environment  ", "data area    ", "free         ", };
static uchar ems_installed=FALSE, ems_name[]="EMMXXXX0", ems_vermajor=0, ems_verminor=0;
static ulong ems_size=0L, ems_free=0L;
static EMS_HANDLE ems_handles[MAX_HANDLE];
static uchar xms_installed=FALSE;
static void far (*xms_drv)(void);
static ulong xms_total=0L, xms_free=0L, xms_largest=0L, xms_used=0L;
static uchar xms_vermajor=0, xms_verminor=0, xms_hma=0, xms_a20=0;
static XMS_HANDLE xms_handles[MAX_HANDLE];
static ushort xms_usehandle=0, xms_freehandle=255;
static uchar upper_installed=FALSE;
static ulong upper_total=0L, upper_free=0L, upper_large=0L;
static ushort upper_index=65535;

// P R O T O T Y P E S //////////////////////////////////////////////////////

static void check_ems(void);
static void check_xms(void);
static void check_upper(void);
static void check_memory(void);
static uchar get_upperlink(void);
static int set_upperlink(uchar);
static void search_vectors(MINFO *);
static void search_sd(MCB *);
static void register_mcb(MCB *);
static int is_psp(MCB *);
static ushort env_seg(MCB *);
static void make_mcb_list(void);
static void make_dev_list(void);
static void normal_list(char *argv0);
static void full_list(void);
static void device_list(void);
static void ems_list(void);
static void xms_list(void);
static ushort our_bios_memsize();

// F U N C T I O N S ////////////////////////////////////////////////////////

ushort our_bios_memsize()
{
    asm int 12h;
    return _AX;
}

int convert(long num, char *des)
{
    char temp[32];
    int c=0, n=0;

    if (num == 0)
        {
        des[0]=0x30;
        des[1]=0;
        n=1;
        } // end if.
    else
        {
        temp[31]=0;
        while (num > 0)
            {
            if (((c+1)%4)==0)
                temp[30-c++]=',';

            temp[30-c++]=(char)(num%10)+0x30;
            num/=10;
            } // end while.

        for (n=0;n<=c;n++)
            des[n]=temp[31-c+n];

        } // end else.

    return n;

} // end convert.

/////////////////////////////////////////////////////////////////////////////

char *get_os(void)
{
    BYTE OEMNumber, DOSMajor30;

    asm { mov AH, 0x30                  // Get DOS version.
        mov AL, 0x00                    // Get OEM number.
        int 0x21
        mov OEMNumber, BH
        mov DOSMajor30, AL
        } // end asm.

    if (DOSMajor30 == 0x00)
        DOSMajor30 = 0x01;              // DOS 1.x.

    switch (OEMNumber)
        {
        case 0xFD:
            strcpy(EnvString, "FreeDOS");
            break;
        case 0xFF:
            if (DOSMajor30 <= 6)
                strcpy(EnvString, "MS-DOS");
            else
                strcpy(EnvString, "Windows");
            break;
        case 0x00:
            strcpy(EnvString, "PC-DOS");
            break;
        case 0xEE:
            strcpy(EnvString, "DR-DOS");
            break;
        case 0xEF:
            strcpy(EnvString, "Novell");
            break;
        case 0x66:
            strcpy(EnvString, "PTS-DOS");
            break;
        case 0x5E:
            strcpy(EnvString, "RxDOS");
            break;
        default:
            strcpy(EnvString, "An unknown operating system");

        } // end switch.

    return EnvString;

} // end Check_DOS_Version.

/////////////////////////////////////////////////////////////////////////////

static void check_ems(void)
{
    void far *int67;

    int67=(void *)getvect(0x67);
    if (int67 == NULL)
        return;

    asm { push ds
        push si
        push di
        les di,int67
        mov di,10
        lea si,ems_name
        mov cx,8
        cld
        repe cmpsb
        pop di
        pop si
        pop ds
        jz _found
        } // end asm.
    return;

_found:
    ems_installed=TRUE;

    asm { mov ah,41h
        int 67h
        or ah,ah
	jz _skip1
	jmp _error
    }
_skip1:
    asm {
        mov ems_frame,bx

        mov ah,46h
        int 67h
        or ah,ah
	jz _skip2
	jmp _error
    }
_skip2:
    asm {
        mov bl,al
        and al,0fh
        and bl,0f0h
        mov cl,4
        shr bl,cl
        mov ems_vermajor,bl
        mov ems_verminor,al

        mov ah,42h
        int 67h
        or ah,ah
        jnz _error
        mov word ptr ems_size,dx
        mov word ptr ems_size[2],0
        mov word ptr ems_free,bx
        mov word ptr ems_free[2],0
        } // end asm.

    ems_size*=16384L;
    ems_free*=16384L;

    asm push  di
    _ES=FP_SEG(&ems_handles);
    _DI=FP_OFF(&ems_handles);
    asm { mov ah,4Dh
        int 67h
        pop di
        or ah,ah
        jnz _error
        mov ems_usehandle,bx
        } // end asm.

    if (ems_vermajor >= 4)
        {
        asm { mov ax,5402h
            int 67h
            or ah,ah
            jnz _error
            mov ems_totalhandle,bx
            } // end asm.
        } // end if.
    else
        {
        ems_totalhandle=ems_usehandle;
        } // end else.

    ems_freehandle=ems_totalhandle - ems_usehandle;
    return;

_error:
    puts("EMS INTERNAL ERROR.");
    exit(1);

} // end check_ems.

/////////////////////////////////////////////////////////////////////////////

static void check_xms(void)
{
    ushort i, freehandles = 0;

    asm mov ax,4300h
    asm int 2fh
    asm cmp al,80h
    asm je _found
    return;

_found:
    xms_installed=TRUE;

    asm mov ax,4310h
    asm int 2fh
    asm mov word ptr xms_drv,bx
    asm mov word ptr xms_drv[2],es

    asm mov ah,0
    (*xms_drv)();
    asm mov xms_vermajor,ah
    asm mov xms_verminor,al
    asm mov xms_hma,dl

    asm mov ah,8
    (*xms_drv)();
    asm mov word ptr xms_free,dx
    asm mov word ptr xms_free[2],0
    asm mov word ptr xms_largest,ax
    asm mov word ptr xms_largest[2],0
    xms_free*=1024L;
    xms_total=xms_free;
    xms_largest*=1024L;

    asm mov ah,7
    (*xms_drv)();
    asm or bl,bl
    asm jz _noerror
    asm jmp _error    
_noerror:        
    asm mov xms_a20,al

    memset(xms_handles, 0, sizeof(xms_handles));
    xms_usehandle=0;
        
    for (i=0;i<65535 && freehandles <= xms_freehandle;i++)
        {
        asm mov ah,0Eh
        _DX=i;
        (*xms_drv)();
        asm or ax,ax
        asm jnz _found_block
        freehandles++;
        continue;
_found_block:
        asm mov byte ptr xms_freehandle,bl
        if (xms_usehandle < MAX_HANDLE)
            {
            asm push  di
            _ES=FP_SEG(&xms_handles);
            _DI=FP_OFF(&xms_handles);
            asm {
                mov ax,xms_usehandle
                mov cl,3
                shl ax,cl
                add di,ax
                mov ax,i
                mov es:[di],ax
                mov es:[di+2],dx
                mov es:[di+6],bh
                pop di
                } // end asm.

            xms_handles[xms_usehandle].size*=1024L;
            xms_total += xms_handles[xms_usehandle].size;
            xms_usehandle++;
            } // end if.

        } // end for.

    return;

_error:
    puts("XMS INTERNAL ERROR.");
    exit(1);

} // end check_ems.

/////////////////////////////////////////////////////////////////////////////

static uchar get_upperlink(void)
{
    asm { mov ax,5802h
        int 21h
        } // end asm.

    return(_AL);

} // end get_upperlink.

/////////////////////////////////////////////////////////////////////////////

static int set_upperlink(uchar link)
{
    asm { mov ax,5803h
        xor bh,bh
        mov bl,link
        int 21h
        jc _noumb
        } // end asm.
    return(1);

_noumb:
    asm cmp ax,1
    asm jne _trash
    return(0);

_trash:
    return(-1);

} // end set_upperlink.

/////////////////////////////////////////////////////////////////////////////

static void check_upper(void)
{
    uchar origlink;

    origlink=get_upperlink();
    switch (set_upperlink(1))
        {
        case 1:
            upper_installed=TRUE;
            break;

        case 0:
            upper_installed=FALSE;
            break;

        case -1:
            puts("SYSTEM MEMORY TRASHED!");
            exit(1);
            break;

        } // end switch.

    set_upperlink(origlink);

} // end check_upper.

/////////////////////////////////////////////////////////////////////////////

static void check_memory(void)
{
    check_ems();
    check_xms();
    check_upper();

} // end check_memory.

/////////////////////////////////////////////////////////////////////////////


long convmemfree(char progname[8])
{
    struct MCBheader far *firstmcbpointer, far *currentmcbblock;
    unsigned int far *temp;
    unsigned long freemem;

    freemem = 0;

    // Get pointer to LL struct.
    _AH=0x52;
    geninterrupt(0x21);

    // In LL in offset -02 there's pointer to first mem block (segment only).
    temp = MK_FP(_ES, _BX - 0x2);
    firstmcbpointer = (struct MCBheader far *)MK_FP(*temp, 0);
    currentmcbblock = firstmcbpointer;
    do
	{
        if (FP_SEG(currentmcbblock) < 0xA000 &&
            ((currentmcbblock->pid==0) || strncmp(currentmcbblock->progname, progname, 8)==0)) 
            freemem += ((long)currentmcbblock->length+1) * 16;
        
        if (currentmcbblock->type == 'Z')  // Z means last memory block.
            break;

        currentmcbblock = MK_FP(FP_SEG(currentmcbblock) + 1 + currentmcbblock->length, 0);
        } // end do.
    while (1);

    return ((long)freemem);

} // end convmemfree.

/////////////////////////////////////////////////////////////////////////////

static int is_psp(MCB *mcb)
{
    asm { les bx,mcb
        mov ax,es
        inc ax
        mov es,ax
        mov ax,TRUE
        cmp word ptr es:[bx],20CDh
        je __exit
        mov ax,FALSE
        } // end asm.

__exit:
    return(_AX);

} // end is_psp.

/////////////////////////////////////////////////////////////////////////////

static ushort env_seg(MCB *mcb)
{
    MCB *env;

    asm { les bx,mcb
        mov ax,es
        inc ax
        mov es,ax
        mov bx,ax
        mov ax,es:[2Ch]
        dec ax
        mov es,ax
        inc ax
        cmp es:[1],bx
        je __exit
        mov ax,0
        } // end asm.

__exit:
    return(_AX);

} // end env_seg.

/////////////////////////////////////////////////////////////////////////////

static void search_vectors(MINFO *m)
{
    ushort i;
    ulong begin, end, iv;
    uchar far *ivp;

    begin=(ulong)(m->seg + 1) << 4;
    end=begin + m->size;
    for (i=0;i<256;i++)
        {
        memcpy(&ivp, MK_FP(0, i*4), 4);
        iv=((ulong)(FP_SEG(ivp) + 1) << 4) + (ulong)FP_OFF(ivp);
        if ((iv > begin) && (iv < end) && (m->vecnum < MAX_VEC))
            m->vectors[m->vecnum++]=(uchar)i;

        } // end for.

} // end search_vectors.

/////////////////////////////////////////////////////////////////////////////

static void search_sd(MCB *mcb)
{
    ushort begin, end;
    SD_HEADER *sd;

    sd=MK_FP(FP_SEG(mcb) + 1, 0);
    begin=FP_SEG(mcb);
    end=FP_SEG(mcb) + mcb->size;
    while ((FP_SEG(sd) > begin) && (FP_SEG(sd) < end) && (mlistnum < MAX_ITEM))
        {
        mlistnum++;
        mlist[mlistnum].seg=sd->start;
        mlist[mlistnum].size=(ulong)sd->size << 4;
        switch (sd->type)
            {
            case 'D':
            case 'I':
                mlist[mlistnum].name[0]=' ';
                strncpy(&mlist[mlistnum].name[1], sd->name, 8);
                strupr(mlist[mlistnum].name);
                mlist[mlistnum].type=MT_DEVICE;
                break;

            case 'F':
                strcpy(mlist[mlistnum].name, " FILES");
                mlist[mlistnum].type=MT_DATA;
                break;

            case 'X':
                strcpy(mlist[mlistnum].name, " FCBS");
                mlist[mlistnum].type=MT_DATA;
                break;

            case 'C':
            case 'B':
                strcpy(mlist[mlistnum].name, " BUFFERS");
                mlist[mlistnum].type=MT_DATA;
                break;

            case 'L':
                strcpy(mlist[mlistnum].name, " LASTDRV");
                mlist[mlistnum].type=MT_DATA;
                break;

            case 'S':
                strcpy(mlist[mlistnum].name, " STACKS");
                mlist[mlistnum].type=MT_DATA;
                break;

            default:
                strcpy(mlist[mlistnum].name, " ??????");
                mlist[mlistnum].type=MT_DATA;
                break;

            } // end switch.

        sd=MK_FP(sd->start + sd->size, 0);
        } // end while.

} // end search_sd.

/////////////////////////////////////////////////////////////////////////////

void check_name(uchar *name)
{
    ushort i;

    for (i=0;name[i];i++)
        if (name[i] < ' ')
            {
            name[i] = '\0';
            break;
            } // end if.

} // end check_name.

/////////////////////////////////////////////////////////////////////////////

static void register_mcb(MCB *mcb)
{
    mlist[mlistnum].seg=FP_SEG(mcb);
    mlist[mlistnum].owner=mcb->owner;
    mlist[mlistnum].size=(ulong)mcb->size << 4;
    if (mlist[mlistnum].seg <= 0x9FFF)
        {
        if (is_psp(mcb))
            {
            strncpy(mlist[mlistnum].name, mcb->name, 8);
            check_name(mlist[mlistnum].name);
            strupr(mlist[mlistnum].name);
            mlist[mlistnum].environment=env_seg(mcb);
            mlist[mlistnum].type=MT_PROGRAM;
            } // end if.

        if (!mcb->owner)
            {
            mlist[mlistnum].type=MT_FREE;
            } // end if.
        else if (mcb->owner <= 0x0008)
            {
            strcpy(mlist[mlistnum].name, "DOS");
            if (!strncmp(mcb->name, "SD", 2))
                {
                mlist[mlistnum].type=MT_SYSDATA;
                search_sd(mcb);
                } // end if.
            else if (!strncmp(mcb->name, "SC", 2))
                {
                mlist[mlistnum].type=MT_SYSCODE;
                } // end else.
            else
                mlist[mlistnum].type=MT_SYSCODE;

            } // end else.

        } // end if.
    else
        {
        if (!mcb->owner)
            {
            mlist[mlistnum].type=MT_FREE;
            } // end if.
        else if (mcb->owner <= 0x0008)
            {
            strcpy(mlist[mlistnum].name, "DOS");
            if (!strncmp(mcb->name, "SD", 2))
                {
                mlist[mlistnum].type=MT_SYSDATA;
                search_sd(mcb);
                } // end if.
            else if (!strncmp(mcb->name, "SC", 2))
                {
                mlist[mlistnum].type=MT_SYSCODE;
                } // end else.

            } // end else.
        else if (mcb->owner > 0x0008)
            {
            mlist[mlistnum].environment=env_seg(mcb);
            mlist[mlistnum].type=MT_PROGRAM;
            strncpy(mlist[mlistnum].name, mcb->name, 8);
            strupr(mlist[mlistnum].name);
            } // end else.

        } // end else.

} // end register_mcb.

/////////////////////////////////////////////////////////////////////////////
static int mcb_made = 0;

static void make_mcb_list(void)
{
    ushort i, j;
    MCB *cur_mcb;
    uchar origlink;

    if (mcb_made) return;
    mcb_made = 1;

    memset(mlist, 0, sizeof(mlist));
    check_memory();
    asm { mov ah,52h
        int 21h
        mov ax,es:[bx-2]
        mov word ptr first_mcb[2],ax
        mov word ptr first_mcb,0
        } // end asm.

    if (upper_installed)
        {
        origlink=get_upperlink();
        set_upperlink(1);
        } // end if.

    cur_mcb=(MCB *)MK_FP(first_mcb, 0);
    while((mlistnum < MAX_ITEM) && (cur_mcb->type != 'Z'))
        {
        register_mcb(cur_mcb);
        cur_mcb=(MCB *)MK_FP(FP_SEG(cur_mcb) + cur_mcb->size + 1, 0);
        ++mlistnum;
        } // end while.

    register_mcb(cur_mcb);
    cur_mcb=(MCB *)MK_FP(FP_SEG(cur_mcb) + cur_mcb->size + 1, 0);
    ++mlistnum;
    if (upper_installed)
        set_upperlink(origlink);

    for (i=0;i<mlistnum;i++)
        if ((mlist[i].seg >= 0x9000) && (mlist[i].seg <= 0xB000) && (mlist[i].type == MT_SYSCODE))
            {
            upper_index=i;
            break;
            } // end if.
    
    for (i=upper_index;i<mlistnum;i++) {
        if (mlist[i].type == MT_FREE)
            {
            upper_free+=mlist[i].size;
            if (mlist[i].size > upper_large)
                upper_large=mlist[i].size;

            } // end if.
        if (mlist[i].seg >= 0xA000)
            upper_total += mlist[i].size + 16;
    }
    
    for (i=0;i<mlistnum;i++)
        {
        if (mlist[i].type == MT_PROGRAM)
            for(j=0;j<mlistnum;j++)
               if ((mlist[i].seg != mlist[j].seg) && (mlist[j].owner == mlist[i].seg+1))
                   {
                   strcpy(mlist[j].name, mlist[i].name);
                   mlist[j].type=(mlist[i].environment == mlist[j].seg+1) ? MT_ENV : MT_DATA;
                   } // end if.

        if (mlist[i].type != MT_SYSDATA)
            search_vectors(&mlist[i]);

        } // end for.

    for (i=0;i<mlistnum;i++)
        if (mlist[i].seg+1 == _psp)
            {
            mlist[i+1].size+=mlist[i].size + 16;
            mlist[i].type=MT_MAP;
            for (j=0;j<mlistnum;j++)
                if (mlist[j].seg+1 == mlist[i].environment)
                    {
                    if (j == i-1)
                        {
                        mlist[j].type=MT_MAP;
                        } // end if.
                    else
                        {
                        mlist[j].type=MT_FREE;
                        mlist[j].name[0]='\0';
                        } // end else.

                    break;
                    } // end if.

            break;
            } // end if.

} // end make_mcb_list.

/////////////////////////////////////////////////////////////////////////////

static void make_dev_list(void)
{
    ushort i, j;
    DEVICE_HEADER *cur_dev;
    DPB *cur_dpb;

    memset(dlist, 0, sizeof(dlist));
    make_mcb_list();
    asm { mov ah,52h
        int 21h
        add bx,22h
        mov word ptr first_dev[2],es
        mov word ptr first_dev,bx
        } // end asm.

    cur_dev=first_dev;
    while ((FP_OFF(cur_dev) != 0xFFFF) && (dlistnum < MAX_ITEM))
        {
        dlist[dlistnum].addr=cur_dev;
        dlist[dlistnum].attr=cur_dev->attr;
        strncpy(dlist[dlistnum].devname, cur_dev->name, 8);
        check_name(dlist[dlistnum].devname);
        strupr(dlist[dlistnum].devname);
        cur_dev=cur_dev->next;
        ++dlistnum;
        } // end while.

    for (i=0;i<dlistnum;i++)
        for (j=0;j<mlistnum;j++)
            if (mlist[j].seg == FP_SEG(dlist[i].addr))
                strcpy(dlist[i].progname, (mlist[j].name[0] == ' ') ? &mlist[j].name[1] : mlist[j].name);

    asm { mov ah,52h
        int 21h
        les bx,es:[bx]
        mov word ptr first_dpb[2],es
        mov word ptr first_dpb,bx
        } // end asm.

    cur_dpb=first_dpb;
    while (FP_OFF(cur_dpb) != 0xFFFF)
        {
        for (i=0;i<dlistnum;i++)
            if (dlist[i].addr == cur_dpb->device_addr)
                {
                dlist[i].drives[dlist[i].drive_num++]=cur_dpb->drive_num+'A';
                break;
                } // end if.

        cur_dpb=cur_dpb->next_dpb;
        } // end while.

    for (i=0;i<dlistnum;i++)
        {
        if ((dlist[i].attr & 0x8000) == 0)
            dlist[i].devname[0]='\0';

        if (dlist[i].drive_num)
            {
            if (dlist[i].drive_num == 1)
                sprintf(dlist[i].devname, "%c:", dlist[i].drives[0]);
            else
                sprintf(dlist[i].devname, "%c: - %c:", dlist[i].drives[0], dlist[i].drives[dlist[i].drive_num-1]);

            } // end if.

        } // end for.

} // end make_dev_list.

/////////////////////////////////////////////////////////////////////////////

static void normal_list(char *argv0)
{
    char buffer[32], xms_buffer[32], ems_buffer[32], xms_lbuffer[32], xms_ubuffer[32], bytes_free[32], upper_buffer[32];
    ushort i, num;
    unsigned memory, memused;
    long temp;

    make_mcb_list();
    get_os();
    memory=our_bios_memsize();
    xms_free=xms_free/1024;
    xms_total=xms_total/1024;
    xms_largest=xms_largest/1024;
    xms_used=xms_total-xms_free;
    convert(xms_free, xms_buffer);
    convert(xms_total, xms_lbuffer);
    convert(xms_used, xms_ubuffer);
    convert(upper_large, upper_buffer);
    i = strlen(argv0);
    while (argv0[i] != '\\' && argv0[i] != '.' && argv0[i] != ':' && i>0) i--;
    if (argv0[i] == '.') {
        argv0[i] = '\0';
        while (argv0[i] != '\\' && argv0[i] != '.' && argv0[i] != ':' && i>0) i--;
    }
    if (i>0) i++;
    temp=convmemfree(&argv0[i]);
    upper_free=upper_free/1024;
    upper_large=upper_large/1024;
    upper_total=upper_total/1024;
    memused=memory - temp/1024;
    printf("\nMemory Type        Total        Used        Free\n"
	   "----------------  --------   ---------    --------\n"
           "Conventional %12uK %10uK %10luK\n", memory, memused, temp/1024);
    printf("Upper        %12luK %10luK %10luK\n"
           "Extended (XMS) %10sK %10sK %10sK\n"
	   "----------------  --------   ---------    --------\n\n", upper_total, upper_total-upper_free, upper_free, xms_lbuffer, xms_ubuffer, xms_buffer);

    convert(ems_size, buffer);
    printf("Total Expanded (EMS) %18luM (%s bytes)\n", ems_size/2048/512, buffer);
    convert(ems_free, ems_buffer);
    printf("Free Expanded (EMS)  %18luM (%s bytes)\n\n", ems_free/2048/512, ems_buffer);
    convert(temp, bytes_free);
    printf("Largest executable program size %8luK (%s bytes)\n", temp/1024, bytes_free);
    printf("Largest free upper memory block %8luK (%s bytes)\n", upper_large, upper_buffer);
    if (xms_hma)
        printf("%s: the high memory area exists.\n", EnvString);
    else
        printf("%s: the high memory area does not exist.\n", EnvString);
        
} // end normal_list.

/////////////////////////////////////////////////////////////////////////////

static void upper_list(void)
{
    ushort i;

    make_mcb_list();
    get_os();
    for (i=0;i<mlistnum;i++)
	if ((mlist[i].type == MT_SYSCODE) || (mlist[i].type == MT_SYSDATA) || (mlist[i].type == MT_FREE) || (mlist[i].type == MT_PROGRAM) || (mlist[i].type == MT_DEVICE))
            printf("  %04x   %6lu   %-9s   %-13s\n", mlist[i].seg, mlist[i].size, mlist[i].name, typenames[mlist[i].type]);
} // end upper_list.

/////////////////////////////////////////////////////////////////////////////

static void full_list(void)
{
    ushort i, j, pos;
    uchar line[81];

    make_mcb_list();
    puts("segment   size       name         type");
    puts("------- --------  ----------  -------------");
    for (i=0;i<mlistnum;i++)
        if (mlist[i].type != MT_MAP)
            {
            sprintf(line, "  %04x   %6lu   %-9s   %-4s", mlist[i].seg, mlist[i].size, mlist[i].name, typenames[mlist[i].type]);
	    puts(line);
            } // end if.

} // end full_list.

/////////////////////////////////////////////////////////////////////////////

static void device_list(void)
{
    ushort i, num;

    make_dev_list();
    puts("   address     attr    name       program   ");
    puts(" -----------  ------ ----------  ---------- ");
        //  XXXX:XXXX    XXXX   XXXXXXXX    XXXXXXXX
    for (i=0;i<dlistnum;i++)
         printf("  %Fp   %04x   %-8s   %-8s  \n", dlist[i].addr, dlist[i].attr, dlist[i].devname, dlist[i].progname);

} // end device_list.

/////////////////////////////////////////////////////////////////////////////

static void ems_list(void)
{
    ushort i;
    uchar *line, numstr[20], handlename[9];

    check_ems();
    if (!ems_installed)
        {
        puts("  EMS driver not installed in system.  ");
        } // end if.
    else
	{
        line="  EMS driver version                   ";
        sprintf(numstr, "%1i.%1i", (int)ems_vermajor, (int)ems_verminor);
        strncpy(&line[22], numstr, strlen(numstr));
        puts(line);
        line="  EMS page frame                       ";
        sprintf(numstr, "%04X", ems_frame);
        strncpy(&line[22], numstr, strlen(numstr));
        puts(line);
        line="  Total EMS memory                     ";
        sprintf(numstr, "%lu bytes", ems_size);
        strncpy(&line[22], numstr, strlen(numstr));
        puts(line);
        line="  Free EMS memory                      ";
        sprintf(numstr, "%lu bytes", ems_free);
        strncpy(&line[22], numstr, strlen(numstr));
        puts(line);
        line="  Total handles                        ";
        sprintf(numstr, "%u", ems_totalhandle);
        strncpy(&line[22], numstr, strlen(numstr));
        puts(line);
        line="  Free handles                         ";
        sprintf(numstr, "%u", ems_freehandle);
        strncpy(&line[22], numstr, strlen(numstr));
        puts(line);

        puts("                                       ");
        puts("  Handle   Pages   Size      Name     ");
        puts(" -------- ------  ------   ---------- ");
        for (i=0;i<ems_usehandle;i++)
            {
            memset(handlename, 0, sizeof(handlename));
            if (ems_vermajor >= 4)
                {
                if (ems_handles[i].handle == 0)
                    {
                    strcpy(handlename, "SYSTEM");
                    } // end if.
                else
		    {
                    asm push di
                    _DX=ems_handles[i].handle;
                    _ES=FP_SEG(&handlename);
                    _DI=FP_OFF(&handlename);
                    asm mov ax,5300h
                    asm int 67h
                    asm pop di
                    } // end else.

                strupr(handlename);
                } // end if.

            printf("     %-7u %-6u %-9lu %-9s  \n", ems_handles[i].handle, ems_handles[i].pages, (ulong)ems_handles[i].pages * 16384L, handlename);
            } // end for.

        } // end else.

} // end ems_list.

/////////////////////////////////////////////////////////////////////////////

static void xms_list(void)
{
    uchar *line, numstr[20];
    ushort i;

    make_mcb_list();
    if (!xms_installed)
        {
        puts("XMS driver not installed in system.");
        } // end if.
    else
        {
        printf("\r");
        line="XMS driver version                     ";
        sprintf(numstr, "%i.%i", (ushort)xms_vermajor, (ushort)xms_verminor);
        strncpy(&line[26], numstr, strlen(numstr));
        puts(line);

        line="HMA state                              ";
        sprintf(numstr, "%s", (xms_hma) ? "exists" : "not exists");
        strncpy(&line[26], numstr, strlen(numstr));
        puts(line);

        line="A20 line state                         ";
        sprintf(numstr, "%s", (xms_a20) ? "enabled" : "disabled");
        strncpy(&line[26], numstr, strlen(numstr));
        puts(line);

        line="Free XMS memory                        ";
        sprintf(numstr, "%lu bytes", xms_free);
        strncpy(&line[26], numstr, strlen(numstr));
        puts(line);

        line="Largest free XMS block                 ";
        sprintf(numstr, "%lu bytes", xms_largest);
        strncpy(&line[26], numstr, strlen(numstr));
        puts(line);

        line="Free handles                           ";
        sprintf(numstr, "%u", xms_freehandle);
        strncpy(&line[26], numstr, strlen(numstr));
        puts(line);
        printf("\n");
        if (xms_usehandle)
            {
            puts(" Block   Handle   Size       Locks");
            puts("------- -------- ------    ---------");
            for (i=0;i<xms_usehandle;i++)
                printf("   %-6u %-7u %-9lu %-12u\n", i, xms_handles[i].handle, xms_handles[i].size, xms_handles[i].locks);

            } // end if.

        puts("                                          ");
        if (upper_installed)
            {
            line="Free upper memory                       ";
            sprintf(numstr, "%lu bytes", upper_free);
            strncpy(&line[26], numstr, strlen(numstr));
            puts(line);
            line="Largest upper block                     ";
            sprintf(numstr, "%lu bytes", upper_large);
            strncpy(&line[26], numstr, strlen(numstr));
            puts(line);
            } // end if.
        else
            {
            puts("Upper memory            not available   ");
            } // end else.

        } // end else.

} // end xms_list.

/////////////////////////////////////////////////////////////////////////////

int classify_args(int narg, char *rawargs[], char *fileargs[], char *optargs[])
{
    int index, jndex, kndex;
    char *argptr;

    for (index=0,jndex=0,kndex=0;index<narg;index++)
        {
        argptr = rawargs[index];
        if (*argptr == '/')
            {
            argptr++;
            optargs[kndex++] = argptr;
            } // end if.
        else
            {
            fileargs[jndex++] = argptr;
            } // end else.
            
        } // end for.

   return kndex;

} // end classify_args.

/////////////////////////////////////////////////////////////////////////////

int main(int argc, char *argv[])
{
    char *fileargs[64], *optargs[64];
    int n_options, n_files, index, help_flag=0;

    n_options = classify_args(argc, argv, fileargs, optargs);
    for (index=0;index<n_options;index++)
        {
        if (optargs[index][0] == '?') help_flag=1;
        else if (optargs[index][0] == 'U' || optargs[index][0] == 'u') upper_list();
        else if (optargs[index][0] == 'F' || optargs[index][0] == 'f') full_list();
        else if (optargs[index][0] == 'D' || optargs[index][0] == 'd') device_list();
        else if (optargs[index][0] == 'E' || optargs[index][0] == 'e') ems_list();
        else if (optargs[index][0] == 'X' || optargs[index][0] == 'x') xms_list();
        else
            {
            printf("Invalid parameter - /%s\n", strupr(optargs[index]));
            exit(1);
            } // end else.
            
        } // end for.
        
    if (help_flag)
        {
	printf("Displays the amount of used and free memory in your system.\n\n"
               "Syntax: MEM [/E] [/F] [/D] [/U] [/X] [/?]\n"
               "  /E  Reports all information about Expanded Memory\n"
               "  /F  Full list of memory blocks\n"
               "  /D  List of device drivers currently in memory\n"
               "  /U  List of programs in conventional and upper memory\n"
               "  /X  Reports all information about Extended Memory\n"
               "  /?  Displays this help message\n");
        exit(1);
        } // end if.

    normal_list(argv[0]);
    return 0;

} // end main.
