/* MEM - Displays the memory amounts in system.
   Copyright (c) 2001-2003 Arkady V.Belousov
   Based on code from Joseph Cosentino,
   see HISTORY.TXT and Credits section in MEM.TXT.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <dos.h>
#include <limits.h>
#include <stddef.h>	/* size_t, offsetof() */
#include <stdio.h>
#include <stdlib.h>	/* abort(), exit() */
#include <string.h>

#include "defines.h"
#include "systypes.h"
#include "memtypes.h"
#include "mem.h"


#define VERSION "1.5 beta"

#define MSG_not_option	"not option: "
#define MSG_bad_option	"bad option: "
#define MSG_use_help	"Use /? for help\n"

#define MSG_no_mem	": out of memory"
#define MSG_chain_bad	" chain is corrupted"
#define MSG_XMS_call	"XMS call "
#define MSG_EMM_call	"EMM call "

#define EXIT_OK		0
#define EXIT_OPTION	1
#define EXIT_ESC	1

/*----------------------------------------------------------------------*/

LOCAL SYSINFO	SYS;
LOCAL MEMINFO	MEM;
LOCAL EXTINFO	EXT;
LOCAL XMSINFO	XMS;
LOCAL EMSINFO	EMS;

#ifdef ASM86
#  define callint(no) (geninterrupt (no), AX)
#  pragma warn -eff /* disable "code has no effect" warning */
#  define AX _AX
#  define AH _AH
#  define AL _AL
#  define CX _CX
#  define DX _DX
#  define DH _DH
#  define DL _DL
#  define BX _BX
#  define BH _BH
#  define BL _BL
#  define DI _DI
#  define ES _ES
#  define CFLAG (_FLAGS & 1)
#else
   LOCAL union REGS r;
   LOCAL struct SREGS segr;
   LOCAL word FAST_ callint (word no) { return int86x (no, &r, &r, &segr); }
#  define AX r.x.ax
#  define AH r.h.ah
#  define AL r.h.al
#  define CX r.x.cx
#  define DX r.x.dx
#  define DH r.h.dh
#  define DL r.h.dl
#  define BX r.x.bx
#  define BH r.h.bh
#  define BL r.h.bl
#  define DI r.x.di
#  define ES segr.es
#  define CFLAG (r.x.cflag)
#endif

/*======================================================================*/

#include "mem-sys.inc"
#include "mem-mem.inc"
#include "mem-dev.inc"
#include "mem-ext.inc"
#include "mem-xms.inc"
#include "mem-ems.inc"
#include "mem-tot.inc"
#include "mem-say.inc"

/*----------------------------------------------------------------------*/

LOCAL byte	MEM_BUFFER [32*1024u];
LOCAL size_t	mem_left = sizeof MEM_BUFFER;

/* allocate temporary area for array (may be used until next call) */
LOCAL void *FAST_ mem_tmp (size_t sz, count n) {
/*!*/	sz = sz;
	return n > mem_left/sz ? NULL : MEM_BUFFER + sizeof MEM_BUFFER - mem_left;
}

LOCAL void *FAST_ mem_add (CStr errmsg, size_t sz) {
	if (sz > mem_left) {
		say_error (errmsg, MSG_no_mem);
		abort (); return NULL;
	}
	mem_left -= sz;
	return MEM_BUFFER + sizeof MEM_BUFFER - mem_left - sz;
}

/*----------------------------------------------------------------------*/

LOCAL void FAST_ memfill (PStr p, size_t sz) {
	/* assume (sz > 0); */
/*!*/	do *p++ = ' '; while (--(int)sz);
}

LOCAL void FAST_ canonize_name (PStr dst, CBFP src, size_t sz) {
	/* assume (sz > 0); */
/*!*/	sz = sz;
	do {	byte ch = *src++;
		if (ch < (byte)' ') {
			if (ch == 0) break;
			ch = ' ';
		}
		*dst++ = ch;
/*!*/	} while (--(int)sz);
	*dst = 0;
}

LOCAL char FAST_ fstrdiff (CBFP s1, PCStr s2) {
	char ch;
	do { ch = *s2, s2++; } while (ch && *s1++ == ch);
	return ch;
}

/* Return string for ID from map m. If ID not found, return last string */
LOCAL PCStr FAST_ MAP_get (const MAP *m, byte ID, count sz) {
/*!*/	while (--(int)sz && m->ID != ID) m++;
	return m->str;
}

/*======================================================================*/

#define HELP() \
   say ("MEM v" VERSION " - Displays the memory amounts in system.\n" \
	"Copyright (c) 2001-2003, licensed under GPL2.\n\n" \
	"Syntax: MEM {options}\n\n" \
	"Options:\n\n" \
/*	"  /C  List programs.\n"*/ \
	"  /D  List device drivers.\n" \
/*	"  /I  List interrupt vectors.\n"*/ \
	"  /A  List all memory blocks.\n" \
	"  /X  Report Extended memory details.\n" \
	"  /E  Report Expanded memory details.\n" \
	"  /T  Report size of all memory types (by default if no options given).\n" \
	"  /P  Pause after each screen with information.\n")

int STDCALL_ main (int argc, PCStr *argv) {
	enum {  A_PROGS = 0x01, A_DEV   = 0x02, A_ALL   = 0x04,
		A_IVT	= 0x08, A_XMS   = 0x10, A_EMS   = 0x20,
		A_TOTAL = 0x40, A_PAUSE = 0x80
	};
	static const struct OPTION { char ch; count opt; } options [] = {
		{ 'C', A_PROGS }, { 'D', A_DEV   }, { 'A', A_ALL   },
/*		{ 'I', A_IVT  },*/{ 'X', A_XMS   }, { 'E', A_EMS   },
		{ 'T', A_TOTAL }, { 'P', A_PAUSE }, { 0, 0 }
	};

	/*--- parse command line ---*/

	count act = 0;
	char ch = '\0';
	PCStr arg;
	for (;; arg++, ch = *arg) {
		if (ch == '\0') {
			if (--argc == 0) break;
			arg = *++argv, ch = *arg;
			if (ch == '-') ch = '/';
			if (ch != '/') {
				say_error (MSG_not_option, arg);
				say (MSG_use_help);
				return EXIT_OPTION;
		}	}

		if (ch == '/') ch = *++arg;
		if (ch == '?') { HELP (); return EXIT_OK; }

		const struct OPTION *p = options;
		ch &= ~0x20; /* fast ASCII conversion to upper case */
		while (ch != p->ch) {
			p++; if (p->ch == 0) {
				say_error (MSG_bad_option, *argv);
				say (MSG_use_help);
				return EXIT_OPTION;
		}	}
		act |= p->opt;
	}

	/*--- initialize ---*/

	SYS.screen.pause = act & A_PAUSE,
	SYS.screen.filled = 0;
	SYS_get (); MEM_get (); EXT_get ();
	XMS.state = XMS_get ();
	EMS.state = EMS_get ();

	/*--- process ---*/

	if (act & (A_ALL | A_PROGS))	memory_list ();
	if (act & A_DEV)		device_list ();
/*	if (act & A_IVT)		IVT_list ();*/
	if (act & A_XMS)		XMS_list ();
	if (act & A_EMS)		EMS_list ();
	if (act & A_TOTAL ||
	    (act & ~A_PAUSE) == 0)	total_list ();

	if (act & A_PAUSE)		pause (0, UINT_MAX);
	return EXIT_OK;
}
