	page	59,132
	title	UIDE -- DOS "Universal IDE" Caching Driver.
;
; This is a general-purpose caching driver for DOS drives handled thru
; Int 13h BIOS I-O requests.   UIDE runs SATA and UltraDMA disks on up
; to 10 "Legacy" or "Native PCI" controllers via an "internal" driver.
; It caches up to 34 BIOS units, including A: and B: diskettes.   UIDE
; also runs up to 8 CD/DVD drives and caches their data-file requests.
; CD/DVD "raw" input (audio & trackwriters) is handled but not cached.
; I-O for up to 200 other "external" devices (USB, Firewire, etc.) can
; also be cached.   "Write Through" caching is done for output.   UIDE
; uses XMS memory for its search tables and can cache 5 Megabytes to 4
; GIGABYTES of data!
;
; When its /P switch is given, UIDE places its search tables in memory
; after the driver, for up to 10% more speed on protected-mode systems
; due to fewer XMS calls!   /P limits the driver to 170-MB caches with
; V7.10 MS-DOS when loaded in the HMA, and it limits UIDE to a maximum
; 1900-MB cache (1.85-GB) if loaded in upper- or DOS memory.   /P adds
; 32 bytes per megabyte of cache to the driver's run-time size.
;
;
; General Program Equations.
;
	.386p			;Allow use of 80386 instructions.
s	equ	<short>		;Make a conditional jump "short".
MAXBIOS	equ	34		;Maximum BIOS units handled.
CDTYP	equ	07Ch		;CD and DVD device type code.
EXTYP	equ	07Eh		;"External" device type code.
CDXUNIT	equ	48		;CD/DVD/"external" starting unit no.
STACK	equ	520		;Driver local-stack size.
RMAXLBA	equ	00006DD39h	;Redbook (audio) maximum LBA value.
COOKSL	equ	2048		;CD/DVD "cooked" sector length.
RAWSL	equ	2352		;CD/DVD "raw" sector length.
RDYTO	equ	008h		;Disk 389-msec minimum I-O timeout.
CMDTO	equ	00Ah		;CD/DVD 500-msec min. command timeout.
SEEKTO	equ	037h		;CD/DVD 3-second min. "seek"  timeout.
STARTTO	equ	07Fh		;CD/DVD 7-second min. startup timeout.
HDWRFL	equ	00410h		;BIOS hardware-installed flag.
DKTSTAT	equ	00441h		;BIOS diskette status byte.
BIOSTMR equ	0046Ch		;BIOS "tick" timer address.
HDISKS	equ	00475h		;BIOS hard-disk count address.
VDSFLAG equ	0047Bh		;BIOS "Virtual DMA" flag address.
HDI_OFS	equ	0048Eh-BIOSTMR	;BIOS hard-disk int. flag "offset".
MCHDWFL	equ	0048Fh		;Diskette hardware media-change flags.
IXM	equ	4096		;IOCTL transfer-length multiplier.
CR	equ	00Dh		;ASCII carriage-return.
LF	equ	00Ah		;ASCII line-feed.
TAB	equ	009h		;ASCII "tab".
;
; Driver Error Return Codes.
;
MCHANGE	equ	006h		;Diskette "media change".
GENERR	equ	00Ch		;CD/DVD "general error".
DMAERR	equ	00Fh		;Hard disk DMA error.
CTLRERR equ	020h		;Hard disk controller not-ready.
DISKERR equ	0AAh		;Hard disk drive not-ready.
FAULTED	equ	0CCh		;Hard disk drive FAULTED.
HARDERR equ	0E0h		;Hard disk I-O error.
XMSERR	equ	0FFh		;XMS memory error.
;
; "Legacy IDE" Controller I-O Base Addresses.
;
NPDATA	equ	001F0h		;Normal primary      base address.
NSDATA	equ	00170h		;Normal secondary    base address.
APDATA	equ	001E8h		;Alternate primary   base address.
ASDATA	equ	00168h		;Alternate secondary base address.
;
; IDE Controller Register Offsets.
;
CDATA	equ	0		;Data port offset.
CDSEL	equ	6		;Disk-select and upper LBA offset.
CCMD	equ	7		;Command-register offset.
CSTAT	equ	7		;Primary-status register offset.
;
; Controller Status and Command Definitions.
;
BSY	equ	080h		;IDE controller is busy.
RDY	equ	040h		;IDE disk is "ready".
FLT	equ	020h		;IDE disk has a "fault".
DRQ	equ	008h		;IDE data request.
ERR	equ	001h		;IDE general error flag.
DMI	equ	004h		;DMA interrupt occured.
DME	equ	002h		;DMA error occurred.
MSEL	equ	0A0h		;"Master" device-select bits.
SSEL	equ	0B0h		;"Slave"  device-select bits.
DRCMD	equ	0C8h		;DMA read command (write is 0CAh,
				;    LBA48 commands are 025h/035h).
LBABITS equ	0E0h		;Fixed LBA command bits.
;
; Byte, Word, and Double-Word Definitions.
;
BDF	struc
lb	db	?
hb	db	?
BDF	ends

WDF	struc
lw	dw	?
hw	dw	?
WDF	ends

DDF	struc
dwd	dd	?
DDF	ends
;
; LBA "Device Address Packet" Layout.
;
DAP	struc
DapPL	db	?		;Packet length.
	db	?		;(Reserved).
DapSC	db	?		;I-O sector count.
	db	?		;(Reserved).
DapBuf	dd	?		;I-O buffer address (segment:offset).
DapLBA	dw	?		;48-bit logical block address (LBA).
DapLBA1	dd	?
DAP	ends
;
; DOS "Request Packet" Layout.
;
RP	struc
RPHLen	db	?		;Header byte count.
RPSubU	db	?		;Subunit number.
RPOp	db	?		;Opcode.
RPStat	dw	?		;Status word.
	db	8 dup (?)	;(Unused by us).
RPUnit	db	?		;Number of units found.
RPLen	dw	?		;Resident driver 32-bit length.
RPSeg	dw	?		;(Segment must ALWAYS be set!).
RPCL	dd	?		;Command-line data pointer.
RP	ends
RPERR	equ	08003h		;Packet "error" flags.
RPDON	equ	00100h		;Packet "done" flag.
RPBUSY	equ	00200h		;Packet "busy" flag.
;
; IOCTL "Request Packet" Layout.
;
IOC	struc
	db	13 dup (?)	;Request "header" (unused by us).
	db	?		;Media descriptor byte (Unused by us).
IOCAdr	dd	? 		;Data-transfer address.
IOCLen	dw	?		;Data-transfer length.
	dw	?		;Starting sector (unused by us).
	dd	?		;Volume I.D. pointer (unused by us).
IOC	ends
;
; Read Long "Request Packet" Layout.
;
RL	struc
	db	13 dup (?)	;Request "header" (unused by us).
RLAM	db	?		;Addressing mode.
RLAddr	dd	?		;Data-transfer address.
RLSC	dw	?		;Data-transfer sector count.
RLSec	dd	?		;Starting sector number.
RLDM	db	?		;Data-transfer mode.
RLIntlv	db	?		;Interleave size.
RLISkip	db	?		;Interleave skip factor.
RL	ends
;
; Segment Declarations.
;
CODE	segment	public use16 'CODE'
	assume	cs:CODE,ds:CODE
;
; DOS Driver Device Header.
;
@	dd	0FFFFFFFFh	;Link to next header block.
	dw	0C800h		;Driver "device attributes".
	dw	(Strat-@)	;"Strategy" routine pointer.
DevIntP	dw	(I_Init-@)	;Device-Interrupt routine pointer.
DvrNam	db	'UDVD1   '	;Driver name for use by SHCDX33E, etc.
	dw	0		;(Reserved).
	db	0		;First assigned drive letter.
CDUnits	db	0		;Number of CD/DVD drives (0 to 8).
ExEntrP	dw	(ExEntr-@)	;External-entry pointer.   MUST be at
				;  address 0016h!   Other drivers can
				;  call "ExEntr" via our code-segment
				;  and this pointer.   A zero pointer
				;  is set for "stand alone" use or if
				;  the user /E switch is omitted.
;
; VDS Parameter Block.
;
VDSLn	dd	(NormEnd-@)	;VDS block length.
VDSOf	dd	0		;VDS 32-bit buffer offset.
VDSSg	dd	0		;VDS 16-bit segment (hi-order zero).
VDSAd	dd	0		;VDS 32-bit physical buffer address.
;
; UltraDMA Command-List.
;
IOAdr	dd	0		;DMA 32-bit buffer address.
IOLen	dd	080000000h	;DMA byte count and "end" flag.
;
; ATAPI "Packet" Area (always 12 bytes for a CD/DVD).
;
Packet	db	0		;Opcode.
	db	0		;Unused (LUN and reserved).
PktLBA	dd	0		;CD/DVD logical block address.
PktLH	db	0		;"Transfer length" (sector count).
PktLn	dw	0		;Middle- and low-order sector count.
PktRM	db	0		;Read mode ("Raw" Read Long only).
	dw	0		;Unused ATAPI "pad" bytes (required).
;
; "PRD" Address and IDE Parameter Area.
;
PRDAd	dd	(IOAdr-@)	;PRD 32-bit command addr. (Init set).
	db	0		;IDE "upper" sector count (always 0).
LBA2	db	0,0,0		;IDE "upper" LBA bits 24-47.
SecCt	db	0		;IDE "lower" sector count.
LBA	db	0,0,0		;IDE "lower" LBA bits 0-23.
DSCmd	db	0		;IDE disk-select and LBA commands.
IOCmd	db	0		;IDE command byte.
Try	equ	LBA2.lb		;CD/DVD I-O retry counter.
UserB	equ	SecCt.dwd	;CD/DVD working I-O buffer address.
UserL	equ	DSCmd.lw	;CD/DVD working I-O buffer length.
;
; General Driver "Data Page" Variables.
;
IOF	db	001h		;I-O control flags --
				;  Bit 7:  Driver is busy.
				;  Bit 0:  Cache must be flushed.
VLF	db	0		;VDS lock flag (01h = buffer locked).
DMAAd	dw	0		;DMA status/command register address.
IdeDA	dw	0		;IDE data register address.
RqPkt	dd	0		;Request I-O packet address.
RqBuf	dd	0		;Request I-O buffer address.
RqXMS	dd	128		;Request XMS buffer offset.
RqSec	db	0		;Request I-O sector count.
RqCmd	db	0		;Request I-O command bits.
RqCSC	db	0		;Request current-block sector count.
RqTyp	db	0		;Request device-type flag.
RqLBA	dw	0,0,0		;Request initial LBA address.
RqUNo	db	0		;Request "cache unit" number.
	db	0		;(Unused at present).
CBLBA	dw	0,0,0		;Current-buffer initial LBA.
CBUNo	db	0		;Current-buffer "cache unit" number.
CBSec	db	0		;Current-buffer sector count.
CBLst	dw	0		;Current-buffer "last" LRU index.
CBNxt	dw	0		;Current-buffer "next" LRU index.
AudAP	dw	0		;CD/DVD audio-start address pointer.
XMHdl	equ	AudAP.lw	;(Used as XMS "handle" during Init).
XMF	db	0		;CD/DVD XMS move  flag (01h if so).
DMF	db	0		;CD/DVD DMA input flag (01h if so).
EFlag	equ	DMF.lb		;(External-entry flag during Init).
STLmt	dw	0		;Cache binary-search limit index.
LUTop	dw	0		;Least-recently-used "top" index.
LUEnd	dw	0		;Least-recently-used "end" index.
;
; BIOS "Active Unit" Table.   The beginning of this table MUST be at
;   driver address 07Fh or below, for BX-reg. addressing at "Entry".
;
Units	db	MAXBIOS dup (0)	;Reserve space for maximum devices.
;
; DOS "Strategy" Routine, used only by our CD/DVD requests.    This
;   routine "fits in here", which helps support up to 34 BIOS units.
;
Strat:	mov	cs:RqPkt.lw,bx	;Save DOS request-packet address.
	mov	cs:RqPkt.hw,es
	retf			;Exit & await DOS "Device Interrupt".
;
; Miscellaneous Driver Variables.
;
PFlag	db	0		;Protected-mode caching (Init only).
CStack	dd	0		;Caller's saved stack pointer.
LBABuf	dw	00082h,0	;"Calculated LBA" buffer  (8 bytes).
PCISubC	dw	00100h		;PCI subclass/function  (Init only).
HMASize	dw	HMALEN2		;HMA space required     (Init only).
;
; "Callback" Driver Pointer for CD/DVD/"external" drives, and Cache
; "Working Buffer" (both "share" the CD/DVD Audio Function Buffer).
;
DvrPtr	equ	[$].dwd		;CD/DVD/"external" driver pointer.
WBLBA	equ	[$+4].lw	;Working-buffer initial LBA.
WBUNo	equ	[$+10].lb	;Working-buffer "cache unit" number.
WBSec	equ	[$+11].lb	;Working-buffer sector count.
WBLst	equ	[$+12].lw	;Working-buffer "last" LRU index.
WBNxt	equ	[$+14].lw	;Working-buffer "next" LRU index.
;
; CD/DVD Audio Function Buffer (16 bytes) for most CD/DVD "audio"
;   requests.    Its variables below are only for initialization.
;
InBuf	equ	$
UTblP	dw	UTable		;Initialization unit table pointer.
NoDMA	db	0		;"No UltraDMA" flag.
UMode	dw	0		;UltraDMA "mode select" bits.
UFlag	db	0		;UltraDMA "mode valid" flags.
USNdx	db	5		;Unit-select index.
USelB	db	082h		;PCI "modes" and unit-select byte.
DVDTblA	dw	(DVD1Tbl-@)	;PCI "scan" address table pointer.
DVDTblB	dw	(DVD1Tbl-@)	;CD/DVD setup address table pointer.
SFlag	db	5		;User cache-size flag, default = 5.
HFlag	db	0		;"Use HMA space" flag.
BiosHD	db	0FFh		;Number of BIOS hard disks.
HDUnit	db	0		;Current BIOS unit number.
;
; Hard-Disk and Diskette Int 13h Entry Routine.   For a CHS request,
;   the registers are:
;
;   AH      Request code.  We handle 002h read and 003h write.
;   AL      I-O sector count.
;   CH      Lower 8 bits of starting cylinder.
;   CL      Starting sector and upper 2 bits of cylinder.
;   DH      Starting head.
;   DL      BIOS unit number:  000h/001h diskettes, 080h+ hard-disks.
;   ES:BX   I-O buffer address.
;
; For an LBA request, the registers are:
;
;   AH      Request code.  We handle 042h read and 043h write.
;   DL      BIOS unit number:  000h/001h diskettes, 080h+ hard-disks.
;   DS:SI   Pointer to Device Address Packet ("DAP") as shown above.
;
Entry:	pushf			;Save CPU flags and BP-reg.
	push	bp
	mov	bp,0		;Reset active-units table index.
@LastU	equ	[$-2].lw	;(Last-unit index, set by Init).
NextU:	dec	bp		;Any more active units to check?
	js s	QuickX		    ;No, not for us -- exit quick!
	cmp	dl,cs:[bp+Units-@]  ;Does request unit match table?
	jne s	NextU		    ;No, see if more units remain.
	cli			;Disable CPU interrupts.
	bts	cs:IOF,7	;Is this driver currently "busy"?
	jc s	InvalF		;Yes?  Set "invalid function" & exit!
Switch:	mov	cs:CStack.lw,sp	;Save caller's stack pointer.
	mov	cs:CStack.hw,ss
	push	cs		;Switch to our driver stack, to avoid
	pop	ss		;  CONFIG.SYS "STACKS=0,0" problems!!
	mov	sp,0
@Stack	equ	[$-2].lw	;(Starting stack pointer, Init set).
	sti			;Re-enable CPU interrupts.
	pushad			;Save all CPU registers.
	push	ds
	push	es
	pusha			;Issue "A20 local-enable" request.
	mov	ah,005h
	call	A20Req
	popa
	jc s	A20Er		;If "A20" error, use routine below.
	db	0EAh		;Go to main Int 13h request handler.
@HDMain	dd	(HDReq-@)	;(Main entry address, set by Init).
PassRq:	call	A20Dis		;Pass request -- Disable "A20" line.
	cli			;Disable CPU interrupts.
	and	[bx+IOF-@].lw,1	;Reset all but "flush cache" flag.
	pop	es		;Reload all CPU registers.
	pop	ds
	popad
	lss	sp,cs:CStack	;Switch back to caller's stack.
QuickX:	pop	bp		;Reload BP-register and CPU flags.
	popf
	db	0EAh		;Go to next routine in Int 13h chain.
@Prv13A	dd	0		;(Prior Int 13h vector, set by Init).
	db	0		;(Unused alignment "filler").
HDExit:	pushf			;Exit -- save error flag and code.
	push	ax
	call	VDUnlk		;Do VDS "unlock" and disable "A20".
	pop	ax		;Reload error code and error flag.
	popf
Abandn:	mov	bp,sp		;Point to saved registers on stack.
	mov	[bp+33],al	;Set error code in exiting AH-reg.
	cli			;Disable CPU interrupts.
	lahf			;Save error flag (carry) for below.
	and	IOF.lw,00001h	;Reset all but "flush cache" flag.
	sahf			;Reload error flag (carry bit).
	pop	es		;Reload all CPU registers.
	pop	ds
	popad
	lss	sp,cs:CStack	;Switch back to caller's stack.
GetOut:	mov	bp,sp		;Set error flag in exiting carry bit.
	rcr	[bp+8].lb,1	;(One of the flags saved by Int 13h).
	rol	[bp+8].lb,1
	pop	bp		;Reload BP-register and CPU flags.
	popf
	iret			;Exit.
A20Er:	mov	al,XMSERR	;"A20" error!  Post "XMS error" code.
	mov	IOF,al		;Flush cache on next caching request.
	jmp s	Abandn		;Abandon this request and exit QUICK!
InvalF:	mov	ah,001h		;We are BUSY, you fool!  Set "invalid
	jmp s	GetOut		;  function" code and get out, QUICK!
;
; DOS Device-Interrupt Routine.   This logic handles CD/DVD requests.
;   Our driver stack is not used, as SHCDX33E/MSCDEX provide a stack.
;
DevInt:	pushf			;Save CPU flags & disable interrupts.
	cli
	pushad			;Save all CPU registers but ES-reg.
	push	ds
	lds	si,cs:RqPkt	    ;Get DOS request-packet address.
	mov	[si].RPStat,0810Ch  ;Post default "general error".
	bts	cs:IOF.lb,7	    ;Is this driver currently "busy"?
	jc s	DvIntX		    ;Yes?  Reload registers and exit!
	sti			;Re-enable CPU interrupts.
	push	es		;Save ES-register now.
	mov	ah,005h		;Issue "A20 local-enable" request.
	call	A20Req		;(Makes critical settings on exit).
	jc s	DvIntE		;If "A20" failure, go exit below.
	db	0EAh		;Go to main CD/DVD request handler.
@CDMain	dd	(CDReq-@)	;(CD/DVD request address, Init set).
DvIRtn:	call	VDUnlk		;Do "VDS unlock" and disable "A20".
DvIntE:	pop	es		;Reload ES-register now.
	cli			;Disable CPU interrupts.
	and	[bx+IOF-@].lw,1	;Reset all but "flush cache" flag.
DvIntX:	pop	ds		;Reload all remaining CPU registers.
	popad
	popf			;Reload CPU flags and exit.
	retf
;
; Subroutine to issue a VDS "unlock" request.   Note that this logic
;   is only used on exit from I-O requests, so it "falls through" to
;   issue an "A20 local-disable" before the driver actually exits.
;
VDUnlk:	shr	[bx+VLF-@].lb,1	;"Unlock" -- Was I-O buffer "locked"?
	jnc s	A20Dis		;No, go issue "A20 local disable".
	mov	ax,08104h	;Get VDS "unlock" parameters.
	mov	dx,0
	mov	di,(VDSLn-@)	;Point to VDS parameter block.
	push	ds
	pop	es
	int	04Bh		;Execute "VDS unlock" request.
;
; Subroutine to issue "A20" local-enable and local-disable requests.
;
A20Dis:	mov	ah,006h		;"A20 local disable" -- get XMS code.
A20Req:	db	09Ah		;Call XMS manager for "A20" request.
@XEntry	equ	dword ptr $	;  (XMS "entry" address, set by Init).
	nop			;  If no XMS memory, set AX-reg. to 1.
	mov	ax,00001h
	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs
	pop	ds
	xor	bx,bx		;Rezero BX-reg. for relative commands.
	sub	al,1		;Zero AL-reg. if success, -1 if error.
	ret			;Exit.
RqIndex	dw	(Ctl1Tbl-@)	;Request cache-table index, here as
				;  it does not "fit" anywhere else!
;
; "External" and CD/DVD Drive Entry Routine.    This logic is called
;   with a "pushf" and then a "call far" command, so the main driver
;   can handle an external/CD/DVD request exactly like an "Int 13h".
;   It sets up read/write requests for processing at "Switch" below.
;   At entry, the registers contain:
;
;     EAX     32-bit user buffer address.    MUST be valid, and must
;		already have been "VDS locked" if needed.
;     DS:BX   "Call back" I-O routine address, used to do I-O for an
;		LBA address that is NOT found to be "cached".
;     CL      I-O sector count (1 to 128) of 512-byte sectors.
;     CH      000h for a read request, 002h for a write request.
;     SI      LBA address bits 0-15.
;     DX      LBA address bits 16-31.
;     DI      LBA address bits 32-47.
;     BP      Bits 0-7 are the "cache unit" number, which must begin
;		with CDXUNIT+8 (CDXUNIT to CDXUNIT+7 are for our own
;		CD/DVD drives 0 to 7).   Bits 8-15 are ignored.
;
;   At exit, the carry bit is zero for no I-O errors or is SET if an
;   I-O error occurred.   If so, the AH-reg. has the I-O error code.
;
;   Note that if the user has not requested "external entry" support
;   with the /E switch, all of this logic is dismissed during driver
;   init by adjusting its stack offset, if the driver loads into HMA
;   space.   [For upper-memory loads, this logic remains in memory].
;
ExEntr:	pushf			;Save CPU flags and BP-reg.
	push	bp
	cli			;Disable CPU interrupts.
	bts	cs:IOF,7	;Is this driver currently "busy"?
	jc s	InvalF		;Yes?  Set "invalid function" & exit!
	and	bp,000FFh	;Ensure an 8-bit "cache unit" number!
	mov	cs:RqTyp,EXTYP	;Set "external" device type code.
	push	ds		;Set our "callback" I-O routine ptr.
	push	bx
	pop	cs:DvrPtr
	jmp s	ExEnt2		;Go to "ExEntr part 2" below.
;
; "External Reset" Routine, ALWAYS 32 bytes after "ExEntr" above!
;   External drivers can call here to request a cache flush after
;   a removable-media change or an I-O error on their drives.
;
ExRSet:	or	cs:IOF,001h	;Flush cache on next caching request.
	retf			;Exit.
	db	0		;(Unused alignment "filler").
;
; "External" Entry Routine, Part 2.   These commands are put here so
;   our "External Reset" routine can remain 32 bytes after "ExEntr".
;
ExEnt2:	mov	cs:RqSec.lw,cx  ;Set sectors and I-O command.
	jmp	Switch		;Go switch to our stack and do I-O.
NXSTACK	equ	ExEntr+STACK	;Starting stack address if /E omitted.
NormEnd	equ	$+STACK		;End of driver's data and entry logic.
				;  All that follows can go in the HMA.
;
; Main CD/DVD request handler.
;
CDReq:	call	ClrPkt		       ;Clear our ATAPI packet area.
	mov	[bx+XMF-@],bx	       ;Reset XMS & DMA-input flags.
	les	si,[bx+RqPkt-@]	       ;Reload DOS request packet ptr.
	mov	es:[si.RPStat],RPDON   ;Set packet status to "done".
	mov	al,es:[si+RPSubU].lb   ;Get our CD/DVD drive number.
	mov	ah,20		       ;Get drive's "audio-start" ptr.
	mul	ah
	add	ax,offset (UTable+8-@)
@UTable	equ	[$-2].lw
	xchg	ax,di
	mov	[bx+AudAP-@],di
	mov	eax,cs:[di-8]	       ;Set drive DMA/IDE addresses.
	mov	[bx+DMAAd-@],eax
	mov	al,es:[si+RPOp]	       ;Get packet request code.
	mov	di,offset (DspTbl1-@)  ;Point to 1st dispatch table.
@DspTB1	equ	[$-2].lw	       ;(Dispatch address, Init set).
	call	Dispat		;Dispatch to desired request handler.
	db	0EAh		;Return to CD/DVD exit logic.
	dw	(DvIRtn-@)
@CDXSeg	dw	0		;(Driver segment address, Init set).
	db	0,0		;(Unused alignment "filler").
;
; Main hard-disk request handler.
;
HDReq:	cmp	bp,CDXUNIT	;CD/DVD or "external unit" request?
	jae	SetBuf		;Yes, go set I-O parameters below.
	mov	dl,0BEh		;Mask out LBA & write request bits.
	and	dl,ah
	cmp	dl,002h		    ;CHS or LBA read/write request?
	mov	dl,cs:[bp+TypeF-@]  ;(Set unit "type" flag anyway).
@TypeF	equ	[$-2].lw
	mov	RqTyp,dl
	je s	ChkLBA		;Yes, check for actual LBA request.
	cmp	dl,0C0h		;Not for us -- Diskette I-O request?
	jne s	Pass		;No, "flush" cache and pass request.
	cmp	ah,005h		;Diskette track-format request?
	jne s	Pass1		;No, "pass" other diskette requests.
	nop			;(Unused alignment "filler").
Pass:	or	IOF,001h	;Flush cache on next caching request.
Pass1:	db	0EAh		;Return to request "pass" routine.
	dw	(PassRq-@)
@PasSeg	dw	0		;(Driver segment address, Init set).
ChkLBA:	mov	di,sp		;Point DI-reg. to current stack.
	shl	ah,1		;Is this an LBA read or write request?
	jns s	ValSC		;No, go validate CHS sector count.
	push	ds		;Save this driver's DS-reg.
	mov	ds,[di+2]	;Reload "DAP" segment into DS-reg.
	cmp	[si].DapBuf,-1	;Check for 64-bit "DAP" I-O buffer.
	mov	al,[si].DapSC	;(Get "DAP" I-O sector count).
	les	dx,[si].DapLBA1	;(Get "DAP" LBA bits 16-47).
	mov	di,es
	les	cx,[si].DapBuf	;(Get "DAP" I-O buffer address).
	mov	si,[si].DapLBA	;(Get "DAP" LBA bits 0-15).
	pop	ds		;(Reload this driver's DS-reg.).
	je s	Pass		;If 64-bit buffer, pass this request!
ValSC:	dec	al		;Is sector count zero or over 128?
	js s	Pass		;Yes?  Let BIOS handle this "No-No!".
	inc	ax		;Restore sector count -- LBA request?
	js s	ZeroBX		;Yes, go set driver's DS-reg.
	xchg	ax,cx		;CHS -- save request code and sectors.
	mov	es,[di]		;Reload buffer segment into ES-reg.
	xor	di,di		;Reset upper LBA address bits.
	mov	si,0003Fh	;Set SI-reg. to starting sector.
	and	si,ax
	dec	si
	shr	al,6		;Set AX-reg. to starting cylinder.
	xchg	al,ah
	xchg	ax,dx		    ;Swap cylinder & head values.
	mov	al,cs:[bp+CHSec-@]  ;Get disk CHS sectors/head value.
@CHSec	equ	[$-2].lw
	or	al,al		    ;Were disk CHS values legitimate?
	jz s	Pass		    ;No?  Let BIOS have this request!
	push	ax		    ;Save CHS sectors/head value.
	mul	ah		    ;Convert head to sectors.
	add	si,ax		    ;Add result to starting sector.
	pop	ax		    ;Reload CHS sectors/head value.
	mul	cs:[bp+CHSHd-@].lb  ;Convert cylinder to sectors.
@CHSHd	equ	[$-2].lw
	mul	dx
	add	si,ax		;Add to head/sector value.
	adc	dx,di
	xchg	ax,bx		;Get buffer offset in AX-register.
	xchg	ax,cx		;Swap offset with request/sectors.
ZeroBX:	xor	bx,bx		;Zero BX-reg. for relative logic.
	and	ah,006h		;Save sector count & command bits.
	mov	[bx+RqSec-@],ax
	mov	[bx+VDSOf-@].lw,cx  ;Set VDS I-O buffer address.
	mov	[bx+VDSOf-@].hw,bx
	mov	[bx+VDSSg-@],es
	movzx	eax,al		    ;Multiply sectors by 512 bytes.
	shl	eax,9
	mov	[bx+VDSLn-@],eax    ;Set VDS buffer length.
	call	VDLock		    ;Do VDS "lock" on user buffer.
	jc s	Pass		    ;VDS error:  Pass this request!
SetBuf:	mov	[bx+RqBuf-@],eax    ;Set user I-O buffer address.
	mov	[bx+RqLBA-@],di	    ;Set initial request LBA.
	mov	[bx+RqLBA+2-@],dx
	mov	[bx+RqLBA+4-@],si
	mov	[bx+RqUNo-@],bp	    ;Set "cache unit" number.
GoMain:	jmp	Cache		;Go to main cache or driver routine.
	db	0EAh		;Return to hard-disk exit routine,
	dw	(HDExit-@)	;  used by stand-alone driver only.
@HXSeg1	dw	0		;(Driver segment address, Init set).
	db	0,0		;(Unused alignment "filler").
;
; Hard Disk Parameter Tables.
;
TypeF	db	MAXBIOS dup (0)	;Device type and "sub-unit":
				;  00h-27h:  SATA or UltraDMA disk.
				;  7Ch:      CD or DVD drive.
				;  7Eh:      "External" drive.
				;  80h:      BIOS-handled disk.
				;  C0h:      BIOS-handled diskette.
CHSec	db	MAXBIOS dup (0)	;Sectors-per-head table.
CHSHd	db	MAXBIOS dup (0)	;Heads-per-cylinder table.
;
; Hard Disk Controller I-O Address Table, set during Initialization.
;
Ctl1Tbl	dw	0FFFFh		;Controller 1 DMA base address.
Ctl1Pri	dw	NPDATA		;   "Legacy" primary   addresses.
Ctl1Sec	dw	NSDATA		;   "Legacy" secondary addresses.
Ctl2Tbl	dw	0FFFFh		;Controller 2 DMA base address.
Ctl2Pri	dw	APDATA		;   "Legacy" primary   addresses.
Ctl2Sec	dw	ASDATA		;   "Legacy" secondary addresses.
	dw	24 dup (0FFFFh)	;Controller 3 to 10 I-O addresses.
CtlTEnd	equ	$		   ;End of controller I-O tables.
CTLTSIZ	equ	(Ctl2Tbl-Ctl1Tbl)  ;Size of a controller I-O table.
;
; CD/DVD Drive Parameter Tables.
;
UTable	dw	0FFFFh		;Unit 0 DMA address (set by Init).
	dw	0FFFFh		;	IDE address (set by Init).
	dw	0FFFFh		;	(Unused alignment "filler").
	db	0FFh		;	Device-select (set by Init).
	db	0FFh		;	Media-change flag.
	dd	0FFFFFFFFh	;	Current audio-start address.
	dd	0FFFFFFFFh	;	Current audio-end   address.
	dd	0FFFFFFFFh	;	Last-session starting LBA.
	db	140 dup (0FFh)	;Unit 1 to 7 Drive Parameter Tables.
UTblEnd	equ	$		;(End of all CD/DVD drive tables).
;
; Int 15h Descriptor Table, used for BIOS "protected-mode" moves.
;
MoveDT	dd	0,0		       ;"Null" descriptor.
	dd	0,0		       ;GDT descriptor.
SrcDsc	dd	00000FFFFh,000009300h  ;Source segment descriptor.
DstDsc	dd	00000FFFFh,000009300h  ;Dest.  segment descriptor.
	dd	0,0		       ;(Used by BIOS).
	dd	0,0		       ;(Used by BIOS).
;
; Global Descriptor Table, used during our own "real-mode" moves.
;
OurGDT	dd	0,0		;"Null" segment to start.
	dw	0FFFFh		;Conforming code segment.
GDT_CSL	dw	0
GDT_CSM	db	0
	dw	0009Fh
GDT_CSH	db	0
GDT_DS	dd	00000FFFFh,000CF9300h  ;4-GB data segment in pages.
GDTLen	equ	($-OurGDT)
;
; Global Descriptor Table Pointer, used by our "lgdt" command.
;
GDTP	dw	GDTLen		;GDT length (always 24 bytes).
GDTPAdr	dd	(OurGDT-@)	;GDT 32-bit address (Set By Init).
;
; CD/DVD Dispatch Table for DOS request codes 0 through 14.   Each
;   entry in the following tables has 4 bits of "IOCTL byte count"
;   and 12 bits of offset for its "handler" routine, so "handlers"
;   MUST be located within 4K bytes from the start of this driver!
;
DspTbl1	dw	DspLmt1		;Number of valid request codes.
	dw	Try2D-USR	;Invalid-request handler address.
DspTblA	dw	0		;00 -- Init  (handled by I_Init).
	dw	0		;01 -- Media Check	(unused).
	dw	0		;02 -- Build BPB	(unused).
	dw	Try3D-USR	;03 -- IOCTL Input.
	dw	0		;04 -- Input		(unused).
	dw	0		;05 -- Input no-wait	(unused).
	dw	0		;06 -- Input Status	(unused).
	dw	0		;07 -- Input flush	(unused).
	dw	0		;08 -- Output		(unused).
	dw	0		;09 -- Output & verify	(unused).
	dw	0		;10 -- Output status	(unused).
	dw	0		;11 -- Output flush	(unused).
	dw	Try4D-USR	;12 -- IOCTL Output.
	dw	RqExit-USR	;13 -- Device Open     (ignored).
	dw	RqExit-USR	;14 -- Device Close    (ignored).
DspLmt1	equ	($-DspTblA)/2	;Table request-code limit.
;
; CD/DVD Dispatch Table for DOS request codes 128 through 136.
;
DspTbl2	dw	DspLmt2		;Number of valid request codes.
	dw	0		;Invalid-request handler address.
DspTblB	dw	RqRead-USR	;128 -- Read Long.
	dw	0		;129 -- Reserved	(unused).
	dw	RqSeek-USR	;130 -- Read Long Prefetch.
	dw	RqSeek-USR	;131 -- Seek.
	dw	RqPlay-USR	;132 -- Play Audio.
	dw	RqStop-USR	;133 -- Stop Audio.
	dw	0		;134 -- Write Long	(unused).
	dw	0		;135 -- Wr. Long Verify	(unused).
	dw	RqRsum-USR	;136 -- Resume Audio.
DspLmt2	equ	($-DspTblB)/2	;Table request-code limit.
;
; CD/DVD Dispatch table for IOCTL Input requests.
;
DspTbl3	dw	DspLmt3		    ;Number of valid request codes.
	dw	0		    ;Invalid-request handler address.
DspTblC	dw	ReqDHA+(5*IXM)-USR  ;00 -- Device-header address.
	dw	RqHLoc+(6*IXM)-USR  ;01 -- Current head location.
	dw	0		    ;02 -- Reserved	    (unused).
	dw	0		    ;03 -- Error Statistics (unused).
	dw	0		    ;04 -- Audio chan. info (unused).
	dw	0		    ;05 -- Read drive bytes (unused).
	dw	ReqDS +(5*IXM)-USR  ;06 -- Device status.
	dw	ReqSS +(4*IXM)-USR  ;07 -- Sector size.
	dw	ReqVS +(5*IXM)-USR  ;08 -- Volume size.
	dw	ReqMCS+(2*IXM)-USR  ;09 -- Media-change status.
	dw	ReqADI+(7*IXM)-USR  ;10 -- Audio disk info.
	dw	ReqATI+(7*IXM)-USR  ;11 -- Audio track info.
	dw	ReqAQI+(11*IXM)-USR ;12 -- Audio Q-channel info.
	dw	0		    ;13 -- Subchannel info  (unused).
	dw	0		    ;14 -- Read UPC code    (unused).
	dw	ReqASI+(11*IXM)-USR ;15 -- Audio status info.
DspLmt3	equ	($-DspTblC)/2	    ;Table request-code limit.
;
; CD/DVD Dispatch table for IOCTL Output requests.
;
DspTbl4	dw	DspLmt4		    ;Number of valid request codes.
	dw	0		    ;Invalid-request handler address.
DspTblD	dw	ReqEJ +(1*IXM)-USR  ;00 -- Eject Disk.
	dw	ReqLU +(2*IXM)-USR  ;01 -- Lock/Unlock Door.
	dw	ReqRS +(1*IXM)-USR  ;02 -- Reset drive.
	dw	0		    ;03 -- Audio control    (unused).
	dw	0		    ;04 -- Write ctl. bytes (unused).
	dw	ReqCl +(1*IXM)-USR  ;05 -- Close tray.
DspLmt4	equ	($-DspTblD)/2	    ;Table request-code limit.
;
; CD/DVD Function-Code "Dispatch" Routines.
;
USR:	mov	al,3		;Unsupported request!  Get error code.
ReqErr:	les	si,[bx+RqPkt-@]	;Reload DOS request-packet address.
	mov	ah,081h		;Post error flags & code in packet.
	mov	es:[si+RPStat],ax
RqExit:	ret			;Exit ("ignored" request handler).
Try2D:	sub	al,080h		;Not request code 0-15:  subtract 128.
	mov	di,(DspTbl2-@)	;Point to 2nd DOS dispatch table.
@DspTB2	equ	[$-2].lw	;(Dispatch table address, Init set).
	jmp s	Dispat		;Go try request-dispatch again.
Try3D:	mov	di,(DspTbl3-@)	;Point to IOCTL Input dispatch table.
@DspTB3	equ	[$-2].lw	;(Dispatch table address, Init set).
	jmp s	TryIOC		;Go get actual IOCTL request code.
Try4D:	mov	di,(DspTbl4-@)	;Point to IOCTL Output table.
@DspTB4	equ	[$-2].lw	;(Dispatch table address, Init set).
TryIOC:	les	si,es:[si+IOCAdr]  ;Get actual IOCTL request code.
	mov	al,es:[si]
	les	si,[bx+RqPkt-@]	;Reload DOS request-packet address.
	nop			;(Unused alignment "filler").
Dispat:	cmp	al,cs:[di]	;Is request code out-of-bounds?
	inc	di		;(Skip past table-limit value).
	inc	di
	jae s	Dspat1		;Yes?  Dispatch to error handler!
	inc	di		;Skip past error-handler address.
	inc	di
	xor	ah,ah		;Point to request-handler address.
	shl	ax,1
	add	di,ax
Dspat1:	mov	dx,cs:[di]	;Get handler address from table.
	mov	di,00FFFh
	and	di,dx
	xor	dx,di		;IOCTL request (xfr length > 0)?
	jz s	DspGo		;No, dispatch to desired handler.
	shr	dx,12		   ;Ensure correct IOCTL transfer
	mov	es:[si+IOCLen],dx  ;  length is set in DOS packet.
	les	si,es:[si+IOCAdr]  ;Get IOCTL data-transfer address.
DspGo:	add	di,(USR-@)	   ;Dispatch to desired handler.
@DspOfs	equ	[$-2].lw	   ;(HMA dispatch offset, Init set).
	jmp	di
;
; CD/DVD subroutine to validate starting RedBook disk sector numbers.
;
ValSN:	mov	eax,es:[si+RLSec]  ;Get starting sector number.
ValSN1:	mov	dl,es:[si+RLAM]	;Get desired addressing mode.
	cmp	dl,001h		;HSG or RedBook addressing?
	ja s	ValSNE		;No?  Return "sector not found".
	jne s	RqExit		;HSG -- exit (accept any DVD value).
	call	CvtLBA		;RedBook -- get starting sector.
	cmp	eax,RMAXLBA	;Is starting sector too big?
	jbe s	RqExit		;No, all is well -- go exit above.
ValSNE:	pop	ax		;Error!  Discard our exit address.
SectNF:	mov	al,8		;Sector not found!  Get error code.
	jmp s	ReqErr		;Go post "sector not found" and exit.
;
; CD/DVD "Read Long" request handler.   All CD/DVD data is read here.
;
RqRead:	call	ValSN		;Validate starting sector number.
	call	MultiS		;Handle Multi-Session disk if needed.
	jc s	ReqErr		;If error, post return code & exit.
	mov	cx,es:[si+RLSC]	;Get request sector count.
	jcxz	RqExit		;If zero, simply exit.
	xchg	cl,ch		;Save swapped sector count.
	mov	[bx+PktLn-@],cx
	cmp	es:[si.RLDM],1	;"Cooked" or "raw" read mode?
	ja s	SectNF		;Neither?  Return "sector not found"!
	mov	dl,028h		;Get "cooked" input values.
	mov	ax,COOKSL
	jb s	RqRL1		;If "cooked" input, set values.
	mov	dl,0BEh		;Get "raw" input values.
	mov	ax,RAWSL
	mov	[bx+PktRM-@].lb,0F8h  ;Set "raw" input flags.
RqRL1:	mov	[bx+Packet-@].lb,dl   ;Set "packet" opcode.
	mul	es:[si+RLSC]	      ;Get desired input byte count.
	test	dx,dx		      ;More than 65535 bytes desired?
	jnz s	SectNF		      ;Yes?  Return sector not found!
	mov	[bx+VDSLn-@],ax	      ;Set 32-bit VDS buffer length.
	mov	[bx+VDSLn+2-@],bx
	mov	[bx+IOLen-@],ax	      ;Set 24-bit DMA buffer length.
	mov	[bx+IOLen+2-@],bl     ;(Save DMA "end of list" byte).
	les	ax,es:[si+RLAddr]     ;Set user input-buffer address.
	mov	[bx+VDSOf-@],ax
	mov	[bx+VDSSg-@],es
	mov	[bx+UserB+2-@],es
	call	VDLock		      ;Do VDS "lock" on user buffer.
	les	si,[bx+RqPkt-@]	      ;Reload request packet address.
	jc s	RqRL2		      ;"Lock" error:  Use "PIO mode".
	test	[bx+DMAAd-@].lb,001h  ;Is this drive using UltraDMA?
	jnz s	RqRL2		      ;No, check for "raw mode" read.
	call	SetAdr		      ;Set 32-bit DMA buffer address.
@NoXM2:	adc	[bx+XMF-@].lw,00100h  ;Set XMS-move & DMA-input flags.
RqRL2:	cmp	[bx+PktRM-@],bl	      ;Is this a "raw mode" read?
@NoCA:	jne s	RqRL4		      ;Yes, use "non-cached" logic.
				      ;("jmp s RqRL4" if no caching).
	movzx	cx,es:[si+RLSC].lb    ;Get sector count & read code.
	shl	cl,2		      ;Multiply by 4 for 2K sectors.
	movzx	bp,es:[si+RPSubU].lb  ;Get our CD/DVD drive number.
	add	bp,CDXUNIT	      ;Convert to our "cache unit".
	mov	eax,[bx+PktLBA-@]   ;Convert packet LBA to "little
	call	Swp32		    ;  endian" format for caching.
	shl	eax,2		    ;Multiply by 4 for 2K sectors.
	push	eax		    ;Put 32-bit LBA in DX/SI-regs.
	pop	si		    ;  and zero DI-reg. (hi-order)
	pop	dx		    ;  as CD/DVD drives do not use
	xor	di,di		    ;  48-bit LBAs!
	mov	eax,[bx+VDSAd-@]    ;Get 32-bit input buffer addr.
	cli			    ;Disable interrupts, so we exit
				    ;  from I-O with interrupts OFF!
	pushf			    ;Save CPU flags and call "RqEntr"
	push	cs		    ;  for input -- with flags saved,
	call	RqEntr		    ;  the call acts like an Int 13h!
	mov	al,ah		    ;Move any error code into AL-reg.
	lahf			    ;Save carry flag (CD/DVD error).
	or	[bx+IOF-@].lb,080h  ;Post driver "busy" flag again.
	sahf			    ;Reload carry flag (CD error).
	sti			;NOW safe to re-enable interrupts!
	jc s	RqRL5		;Any CD/DVD errors during input?
RqRL3:	ret			;No, all is well -- exit.
RqRL4:	call	RLRead		;Noncached -- Input desired data.
	jnc s	RqRL3		;If no errors, exit above.
RqRL5:	cmp	al,0FFh		;Did we get an "XMS move" error?
	jne s	RqRL6		;No, set returned error code as-is.
	mov	al,GENERR	;Get CD/DVD "general error" code.
	nop			;(Unused alignment "filler").
RqRL6:	jmp	ReqErr		;Set error code in packet & exit.
	db	0		;(Unused alignment "filler").
;
; Subroutine to enter our hard-disk logic for cached CD/DVD input.
;
RqEntr:	pushf			;Save CPU flags and BP-reg.
	push	bp
	mov	[bx+RqSec-@],cx	      ;Set sectors and I-O command.
	mov	[bx+RqTyp-@].lb,CDTYP ;Set CD/DVD device type code.
	db	0EAh		      ;Go switch stacks and do input.
	dw	(Switch-@)
@RqESeg	dw	0		;(Driver segment address, Init set).
;
; Subroutine to initialize the next UltraDMA I-O transfer.
;   This subroutine is place here for alignment.
;
IniDMA:	mov	dx,[bx+DMAAd-@]	;Get DMA command-register address.
	mov	si,(PRDAd-@)	;Get Physical-Region Descriptor addr.
	out	dx,al		;Reset DMA commands and set DMA mode.
	inc	dx		;Point to DMA status register.
	inc	dx
	in	al,dx		;Reset DMA status register.
	or	al,006h		;(Done this way so we do NOT alter
	out	dx,al		;  the "DMA capable" status flags!).
	inc	dx		;Point to PRD address register.
	inc	dx
	outsd			;Set PRD pointer to our DMA address.
	ret			;Exit.
;
; CD/DVD "Seek" request handler.
;
RqSeek:	call	RdAST1		;Read current "audio" status.
	call	ClrPkt		;Reset our ATAPI packet area.
	jc s	RqSK2		;If status error, do DOS seek.
	mov	al,[di+1]	;Get "audio" status flag.
	cmp	al,011h		;Is drive in "play audio" mode?
	je s	RqSK1		;Yes, validate seek address.
	cmp	al,012h		;Is drive in "pause" mode?
	jne s	RqSK2		;No, do DOS seek below.
RqSK1:	call	ValSN		;Validate desired seek address.
	mov	di,[bx+AudAP-@]	;Point to audio-start address.
	cmp	eax,cs:[di+4]	;Is address past "play" area?
	ja s	RqSK2		;Yes, do DOS seek below.
	mov	cs:[di],eax	;Update audio-start address.
	call	PlayA		;Issue "Play Audio" command.
	jc s	RqPLE		;If error, post return code & exit.
	cmp	[di+1].lb,011h	;Were we playing audio before?
	je s	RqPLX		;Yes, post "busy" status and exit.
	call	ClrPkt		;Reset our ATAPI packet area.
	jmp s	RqStop		;Go put drive back in "pause" mode.
RqSK2:	call	ValSN		;Validate desired seek address.
	call	MultiS		;Handle Multi-Session disk if needed.
	jc s	RqPLE		;If error, post return code & exit.
	mov	[bx+Packet-@].lb,02Bh  ;Set "seek" command code.
RqSK3:	call	DoCmd		;Issue desired command to drive.
	jc s	RqPLE		;If error, post return code & exit.
RqSKX:	ret			;Exit.
;
; CD/DVD "Play Audio" request handler.
;
RqPlay:	cmp	es:[si.RLSC].dwd,0  ;Is sector count zero?
	je s	RqSKX		    ;Yes, just exit above.
	mov	eax,es:[si+RLAddr]  ;Validate audio-start address.
	call	ValSN1
	mov	di,[bx+AudAP-@]	;Save drive's audio-start address.
	mov	cs:[di],eax
	add	eax,es:[si+18]	;Calculate audio-end address.
	mov	edx,RMAXLBA	;Get maximum audio address.
	jc s	RqPL1		;If "end" WAY too big, use max.
	cmp	eax,edx		;Is "end" address past maximum?
	jbe s	RqPL2		;No, use "end" address as-is.
RqPL1:	xchg	eax,edx		;Set "end" address to maximum.
RqPL2:	mov	cs:[di+4],eax	;Save drive's audio-end address.
	call	PlayA		;Issue "Play Audio" command.
RqPLE:	jc s	RqDS0		;If error, post return code & exit.
RqPLX:	jmp	RdAST4		;Go post "busy" status and exit.
;
; CD/DVD "Stop Audio" request handler.
;
RqStop:	mov	[bx+Packet-@].lb,04Bh ;Set "Pause/Resume" command.
	jmp	DoCmd		      ;Go pause "audio" and exit.
;
; CD/DVD "Resume Audio" request handler.
;
RqRsum:	inc	[bx+PktLn+1-@].lb  ;Set "Resume" flag for above.
	call	RqStop		;Issue "Pause/Resume" command.
	jmp s	RqPLE		;Go exit through "RqPlay" above.
;
; CD/DVD IOCTL Input "Device Header Address" handler.
;
ReqDHA:	push	ds		;Return our base driver address.
	push	bx
	pop	es:[si+1].dwd
	ret			;Exit.
;
; CD/DVD IOCTL Input "Current Head Location" handler.
;
RqHLoc:	mov	[bx+Packet-@].dwd,001400042h   ;Set command bytes.
	mov	al,16		;Set input byte count of 16.
	call	RdAST3		;Issue "Read Subchannel" request.
	jc s	RqDS0		;If error, post return code & exit.
	mov	es:[si+1],bl	;Return "HSG" addressing mode.
	call	SwpLBA		;Return "swapped" head location.
	mov	es:[si+2],eax
	jmp s	RqVSX		;Go post "busy" status and exit.
;
; CD/DVD IOCTL Input "Device Status" handler.
;
ReqDS:	mov	[bx+Packet-@].dwd,0002A005Ah  ;Set up mode-sense.
	mov	al,16		;Use input byte count of 16.
	call	DoBuf1		;Issue mode-sense for hardware data.
RqDS0:	jc s	RqVSE		;If error, post return code & exit.
	mov	eax,00214h	;Get our basic driver status flags.
	mov	cl,070h		;Get media status byte.
	xor	cl,[di+2]
	shr	cl,1		;Door open, or closed with no disk?
	jnz s	RqDS1		;No, check "drive locked" status.
	adc	ax,00800h	;Post "door open" & "no disk" flags.
RqDS1:	test	[di+14].lb,002h	;Drive pushbutton "locked out"?
	jnz s	RqVSS		;No, set flags in IOCTL and exit.
	or	al,002h		;Set "door locked" status flag.
	jmp s	RqVSS		;Go set flags in IOCTL and exit.
;
; CD/DVD IOCTL Input "Sector Size" handler.
;
ReqSS:	cmp	es:[si+1].lb,1	;Is read mode "cooked" or "raw"?
	ja	RqGenE		;Neither -- Post "general error".
	mov	ax,RAWSL	;Get "raw" sector length.
	je s	RqSS1		;If "raw" mode, set sector length.
	mov	ax,COOKSL	;Get "cooked" sector length.
RqSS1:	mov	es:[si+2],ax	;Post sector length in IOCTL packet.
	ret			;Exit.
;
; CD/DVD IOCTL Input "Volume Size" handler.
;
ReqVS:	mov	[bx+Packet-@].lb,025h  ;Set "Read Capacity" code.
	mov	al,008h		;Get 8 byte data-transfer length.
	call	DoBuf2		;Issue "Read Capacity" command.
RqVSE:	jc s	RqASIE		;If error, post return code & exit.
	mov	eax,[di]	;Get and "swap" volume size.
	call	Swp32
RqVSS:	mov	es:[si+1],eax	;Set desired IOCTL data value.
RqVSX:	jmp s	RqATIX		;Go post "busy" status and exit.
;
; CD/DVD IOCTL Input "Media-Change Status" handler.
;
ReqMCS:	call	DoCmd		;Issue "Test Unit Ready" command.
	mov	di,[bx+AudAP-@]	;Get media-change flag from table.
	mov	al,cs:[di-1]
	mov	es:[si+1],al	;Return media-change flag to user.
	ret			;Exit.
;
; CD/DVD IOCTL Input "Audio Disk Info" handler.
;
ReqADI:	mov	al,0AAh		;Specify "lead-out" session number.
	call	RdTOC		;Read disk table-of-contents (TOC).
	mov	es:[si+3],eax	;Set "lead out" LBA addr. in IOCTL.
	mov	ax,[di+2]	;Set first & last tracks in IOCTL.
	mov	es:[si+1],ax
	jmp s	RqATIX		;Go post "busy" status and exit.
;
; CD/DVD IOCTL Input "Audio Track Info" handler.
;
ReqATI:	mov	al,es:[si+1]	;Specify desired session (track) no.
	call	RdTOC		;Read disk table-of-contents (TOC).
	mov	es:[si+2],eax	;Set track LBA address in IOCTL.
	mov	al,[di+5]
	shl	al,4
	mov	es:[si+6],al
RqATIX:	jmp	RdAST		;Go post "busy" status and exit.
;
; CD/DVD IOCTL Input "Audio Q-Channel Info" handler.
;
ReqAQI:	mov	ax,04010h	;Set "data in", use 16-byte count.
	call	RdAST2		;Read current "audio" status.
	jc s	RqASIE		;If error, post return code & exit.
	mov	eax,[di+5]	;Set ctrl/track/index in IOCTL.
	mov	es:[si+1],eax
	mov	eax,[di+13]	;Set time-on-track in IOCTL.
	mov	es:[si+4],eax
	mov	edx,[di+9]	;Get time-on-disk & clear high
	shl	edx,8		;  order time-on-track in IOCTL.
	jmp s	RqASI4		;Go set value in IOCTL and exit.
;
; CD/DVD IOCTL Input "Audio Status Info" handler.
;
ReqASI:	mov	ax,04010h	;Set "data in", use 16-byte count.
	call	RdAST2		;Read current "audio" status.
RqASIE:	jc s	RqErrJ		;If error, post return code & exit.
	mov	es:[si+1],bx	;Reset audio "paused" flag.
	xor	eax,eax		;Reset starting audio address.
	xor	edx,edx		;Reset ending audio address.
	cmp	[di+1].lb,011h  ;Is drive now "playing" audio?
	jne s	RqASI1		;No, check for audio "pause".
	mov	di,[bx+AudAP-@]	;Point to drive's audio data.
	mov	eax,cs:[di]	;Get current audio "start" addr.
	jmp s	RqASI2		;Go get current audio "end" addr.
RqASI1:	cmp	[di+1].lb,012h	;Is drive now in audio "pause"?
	jne s	RqASI3		;No, return "null" addresses.
	inc	es:[si+1].lb	;Set audio "paused" flag.
	call	SwpLBA		;Convert time-on-disk to LBA addr.
	call	CvtLBA
	mov	di,[bx+AudAP-@]	;Point to drive's audio data.
RqASI2:	mov	edx,cs:[di+4]	;Get current audio "end" address.
RqASI3:	mov	es:[si+3],eax	;Set audio "start" addr. in IOCTL.
RqASI4:	mov	es:[si+7],edx	;Set audio "end" address in IOCTL.
	ret			;Exit.
;
; CD/DVD IOCTL Output "Eject Disk" handler.
;
ReqEJ:	mov	[bx+Packet-@].lw,0011Bh  ;Set "eject" commands.
	mov	[bx+PktLBA+2-@].lb,002h  ;Set "eject" function.
	jmp	RqSK3			 ;Go do "eject" & exit.
;
; CD/DVD IOCTL Output "Lock/Unlock Door" handler.
;
ReqLU:	mov	al,es:[si+1]	;Get "lock" or "unlock" function.
	cmp	al,001h		;Is function byte too big?
	ja s	RqGenE		;Yes, post "general error" & exit.
	mov	cx,0001Eh	    ;Get "lock" & "unlock" commands.
RqLU1:	mov	[bx+Packet-@],cx    ;Set "packet" command bytes.
	mov	[bx+PktLBA+2-@],al  ;Set "packet" function byte.
	call	DoCmd		    ;Issue desired command to drive.
	jc s	RqErrJ		;If error, post return code & exit.
	jmp s	RdAST		;Go post "busy" status and exit.
;
; CD/DVD IOCTL Output "Reset Drive" handler.
;
ReqRS:	call	HltDMA		;Stop previous DMA & select drive.
	inc	dx		;Point to IDE command register.
	mov	al,008h		;Do an ATAPI "soft reset" command.
	out	dx,al
	call	ChkTO		;Await controller-ready.
	jnc s	SwpLBX		;If no timeout, go exit below.
RqGenE:	mov	al,GENERR	;General error!  Get return code.
RqErrJ:	jmp	ReqErr		;Go post error return code and exit.
	db	0		;(Unused alignment "filler").
;
; CD/DVD IOCTL Output "Close Tray" handler.
;
ReqCL:	mov	al,003h		;Get "close tray" function byte.
	mov	cx,0011Bh	;Get "eject" & "close" commands.
	jmp s	RqLU1		;Go do "close tray" command above.
;
; CD/DVD subroutine to read disk "Table of Contents" (TOC) values.
;
RdTOCE:	pop	cx		;Error -- Discard exit address.
	jmp s	RqErrJ		;Go post return code and exit.
RdTOC:	mov	[bx+Packet-@].lw,00243h  ;Set TOC and MSF bytes.
	call	DoTOC1		;Issue "Read Table of Contents" cmd.
	jc s	RdTOCE		;If error, exit immediately.
;
; CD/DVD subroutine to "swap" the 4 bytes of a a 32-bit value.
;
SwpLBA:	mov	eax,[di+8]	;Get audio-end or buffer LBA value.
Swp32:	xchg	al,ah		;"Swap" original low-order bytes.
	rol	eax,16		;"Exchange" low- and high-order.
	xchg	al,ah		;"Swap" ending low-order bytes.
SwpLBX:	ret			;Exit.
;
; CD/DVD subroutine to read current "audio" status & disk addresses.
;
RdAST:	call	ClrPkt		  ;Status only -- reset ATAPI packet.
RdAST1:	mov	ax,00004h	  ;Clear "data in", use 4-byte count.
RdAST2:	mov	[bx+Packet-@].dwd,001000242h  ;Set command bytes.
	mov	[bx+PktLBA-@],ah  ;Set "data in" flag (RdAST2 only).
RdAST3:	call	DoBuf1		  ;Issue "Read Subchannel" command.
	jc s	RdASTX		  ;If error, exit immediately.
	cmp	[di+1].lb,011h	  ;Is a "play audio" in progress?
	clc			  ;(Zero carry flag in any case).
	jne s	RdASTX		  ;No, just exit below.
RdAST4:	push	si		  ;Save SI- and ES-regs.
	push	es
	les	si,[bx+RqPkt-@]	  ;Reload DOS request-packet addr.
	or	es:[si.RPStat],RPBUSY  ;Set "busy" status bit.
	pop	es		  ;Reload ES- and SI-regs.
	pop	si
RdASTX:	ret			  ;Exit.
;
; Subroutine to handle CD/DVD Multi-Session disks for reads & seeks.
;   Multi-Session disks require (A) saving the last-session starting
;   LBA for a new disk after any media-change and (B) "offsetting" a
;   read of the VTOC or initial directory block, sector 16 or 17, to
;   access the VTOC/directory of the disk's last session.
;
MultiS:	mov	di,[bx+AudAP-@]		;Point to drive variables.
	cmp	cs:[di+11].lb,0FFh	;Is last-session LBA valid?
	jne s	MultS1			;Yes, proceed with request.
	mov	[bx+Packet-@].lb,043h	;Set "Read TOC" command.
	inc	[bx+PktLBA-@].lb	;Set "format 1" request.
	call	DoTOC2			;Read first & last session.
	jc s	MultSX			;If any error, exit below.
	mov	[bx+PktLBA-@],bl	;Reset "format 1" request.
	mov	al,[di+3]		;Get last-session number.
	call	DoTOC1		;Read disk info for last session.
	jc s	MultSX		;If error, exit with carry set.
	call	SwpLBA		;"Swap" & save last-session LBA addr.
	mov	di,[bx+AudAP-@]
	mov	cs:[di+8],eax
	call	ClrPkt		   ;Reset our ATAPI packet area.
MultS1:	mov	eax,es:[si+RLSec]  ;Get starting sector number.
	mov	dx,ax		   ;Save low-order sector number.
	and	al,0FEh		;"Mask" sector to an even number.
	cmp	eax,16		;Sector 16 (VTOC) or 17 (directory)?
	xchg	ax,dx		;(Restore low-order sector number).
	jne s	MultS2		;No, set sector in packet.
	add	eax,cs:[di+8]	;Offset sector to last-session start.
MultS2:	call	Swp32		;"Swap" sector into packet as LBA.
	mov	[bx+PktLBA-@],eax
	clc			;Clear carry flag (no errors).
MultSX:	ret			;Exit.
;
; Subroutine to ensure CD/DVD UltraDMA is stopped and then select our
;   drive.   For some older chipsets, if UltraDMA is running, reading
;   an IDE register causes the chipset to "HANG"!!
;
HltDMA:	mov	dx,[bx+DMAAd-@]	;Get drive UltraDMA command address.
	test	dl,001h		;Is this drive using UltraDMA?
	jnz s	HltDM1		;No, just select "master" or "slave".
	in	al,dx		;Ensure any previous DMA is stopped!
	and	al,0FEh
	out	dx,al
HltDM1:	mov	dx,[bx+IdeDA-@]	;Point to IDE device-select register.
	add	dx,6
	mov	di,[bx+AudAP-@]	;Get drive's IDE device-select byte.
	mov	al,cs:[di-2]
	out	dx,al		;Select IDE "master" or "slave" unit.
	ret			;Exit.
;
; Ye Olde UltraDMA "Internal Driver" Subroutine.   This routine sets
;   up I-O requests, then calls "DoDMA" below to perform actual I-O.
;
UdmaIO:	mov	ax,[bx+RqLBA+4-@]  ;Set LBA bits 0 thru 15, unneeded
	mov	[bx+LBA-@],ax	   ;  in setting LBA28/LBA48 command.
	mov	bp,[bx+RqTyp-@]	   ;Get this disk's "type" flags.
	mov	ax,bp		   ;Get disk's IDE controller number.
	shr	al,2
	mov	ah,CTLTSIZ	   ;Point to disk's I-O addresses.
	mul	ah
	add	ax,(Ctl1Tbl-@)
@CtlTbl	equ	[$-2].lw
	xchg	ax,si
	lods	cs:[si].dwd	   ;Set controller base address and
	mov	[bx+DMAAd-@],eax   ;  primary-channel data address.
	rcr	bp,2		   ;Primary channel I-O request?
	jnc s	GetSec		   ;Yes, get sector count & commands.
	add	[bx+DMAAd-@].lw,8  ;Use secondary DMA controller.
	lods	cs:[si].lw	   ;Set secondary channel data addr.
	mov	[bx+IdeDA-@],ax
GetSec:	mov	ax,[bx+RqSec-@]	   ;Get sector count and command bits.
	les	di,[bx+RqLBA-@]	   ;Set LBA bits 16 thru 47.
	mov	dx,es
	mov	[bx+LBA2+1-@],di
	mov	[bx+LBA2-@],dh
	mov	[bx+LBA+2-@],dl
	shr	dx,12		   ;Shift out LBA bits 16-27.
	or	di,dx		   ;Anything in LBA bits 28-47?
	jz s	LBA28		   ;No, use LBA28 read/write command.
	shl	ah,3		   ;LBA48 -- get command as 020h/030h.
	jmp s	SetCmd		   ;Go set LBA and IDE command bytes.
LBA28:	xchg	dh,[bx+LBA2-@]	   ;LBA28:  Reload & reset bits 24-27.
	or	ah,(DRCMD+1)	   ;Get LBA28 read/write command + 5.
SetCmd:	shl	bp,1		   ;Shift master/slave into LBA cmds.
	mov	dl,(LBABITS/32)
	rcl	dl,5
	or	dl,dh		   ;"Or" in LBA28 bits 24-27 (if any).
	mov	dh,005h		   ;Get final IDE command byte.
	xor	dh,ah		   ;(LBA28 C8h/CAh, LBA48 25h/35h).
	mov	[bx+DSCmd-@],dx	   ;Set LBA and IDE command bytes.
	mov	[bx+SecCt-@],al	   ;Set disk I-O sector count.
	mov	[bx+IOLen-@],bl	   ;Set 24-bit DMA buffer length.
	mov	ah,0		   ;  We include bit 16 for 65536
	shl	ax,1		   ;  bytes, as a few "odd" chips
	mov	[bx+IOLen+1-@],ax  ;  may NOT run O.K. without it!
	mov	eax,[bx+RqBuf-@]   ;Get user I-O buffer address.
	call	UseCB		   ;Can current cache buffer be used?
	jnc s	UBufRq		   ;Yes, go do buffered I-O below.
	call	SetAdr		   ;Set user or Main XMS buffer addr.
	jnc s	DoDMA		   ;If user buffer, do direct DMA.
@NoXM1:	nop			     ;(Unused alignment "filler").
UBufRq:	test	[bx+IOCmd-@].lb,012h ;Buffered I-O:  Read request?
	jz s	UBufIn		     ;Yes, use input routine below.
	call	UBufMv		     ;Move user data to XMS buffer.
	jnc s	DoDMA		     ;If no error, do output & exit.
UBufEr:	ret			     ;XMS or I-O error:  Exit now!
UBufIn:	call	DoDMA		     ;Do input to our XMS buffer.
	jc s	UBufEr		     ;If any I-O errors, exit above!
	nop			     ;(Unused alignment "filler").
UBufMv:	movzx	ecx,[bx+SecCt-@].lb  ;Get number of sectors to move.
	mov	esi,[bx+IOAdr-@]     ;Set move addresses for a read.
	mov	edi,esi
	xchg	edi,[bx+RqBuf-@]
UBufM1:	jz s	UBufM2		;Is this a read from XMS memory?
	xchg	esi,edi		;No, "swap" source & destination.
UBufM2:	shl	ecx,9		;Convert sectors to byte count.
	jmp	MvData		;Go move all XMS data, then exit.
;
; Subroutine to handle hard-disk UltraDMA I-O requests for "UdmaIO".
;
DoDMA:	call	MovCLX		;Move DMA command-list up to XMS.
	jc s	DoDMAX		;Command-list move ERROR?  Exit NOW!
	mov	dx,[bx+DMAAd-@]	;Ensure any previous DMA is stopped!
	in	al,dx		;(On some older chipsets, if DMA is
	and	ax,000FEh	;  running, reading an IDE register
	out	dx,al		;  causes the chipset to "HANG"!!).
	mov	di,[bx+IdeDA-@]	;Get our disk's IDE base address.
	mov	al,[bx+DSCmd-@]	;Select our desired disk.
	and	ax,000F0h
	lea	dx,[di+CDSEL]
	out	dx,al
	mov	es,bx		;Point to low-memory BIOS timer.
	mov	si,BIOSTMR
	mov	cx,((RDYTO*256)+FLT)  ;Get timeout & "fault" mask.
	add	ch,es:[si]	      ;Set timeout limit in CH-reg.
	call	ChkRdy		      ;Await controller/disk ready.
	jc s	DoDMAX		      ;If any errors, exit below!
	test	[bx+IOCmd-@].lb,012h  ;Is this a write request?
	jnz s	SetDMA		      ;Yes, init DMA transfer.
	mov	al,008h		;Get "DMA read" command bit.
SetDMA:	push	si		;Save BIOS timer pointer.
	call	IniDMA		;Initialize this DMA I-O transfer.
	mov	ax,001F7h	;Set IDE parameter-output flags.
NxtPar:	lea	dx,[di+CDATA+1]	;Point to IDE sector count reg. - 1.
IDEPar:	inc	dx		;Output all ten LBA48 parameter bytes.
	outsb			;(1st 4 overlayed by 2nd 4 if LBA28!).
	shr	ax,1		;More parameters to go in this group?
	jc s	IDEPar		;Yes, loop back and output next one.
	jnz s	NxtPar		;If first 4 done, go output last 6.
	pop	si		;Reload BIOS timer pointer.
	dec	dh		;"Legacy IDE" controller channel?
@DRQ:	jmp s	DMAGo		;No, forget about awaiting 1st DRQ.
	mov	dh,003h		;Get IDE alternate-status address.
	dec	dx		;(Primary-status address | 300h - 1).
ChkDRQ:	cmp	ch,es:[si]	;Too long without 1st data-request?
	je s	ErrDMA		;Yes?  Return carry and DMA error!
	in	al,dx		;Read IDE alternate status.
	and	al,DRQ		;Has 1st data-request arrived?
	jz s	ChkDRQ		;No, loop back and check again.
DMAGo:	call	RunDMA		;Start and monitor our DMA transfer.
	jnz s	ErrDMA		;Problems?  Return carry & DMA error!
	inc	cx		;Check "fault" and hard-error at end.
ChkRdy:	lea	dx,[di+CSTAT]	;Read IDE primary status.
	in	al,dx
	cmp	ch,es:[si]	;Too long without becoming ready?
	je s	RdyErr		;Yes?  Go see what went wrong.
	test	al,BSY+RDY	;Controller or disk still busy?
	jle s	ChkRdy		;Yes, loop back and check again.
	and	al,cl		;Disk "fault" or hard-error?
	jnz s	HdwErr		;Yes?  Go see what went wrong.
DoDMAX:	ret			;End of request -- Return to "UdmaIO".
ErrDMA:	mov	al,DMAERR	;BAAAD News!  Post DMA error code.
	stc			;Set carry flag (error!) and exit.
	ret
RdyErr:	test	al,BSY		;BAAAD News!  Did controller go ready?
	mov	ax,(256*CTLRERR)+DISKERR ;(Get not-ready error codes).
	jmp s	WhichE		;Go see which error code to return.
HdwErr:	test	al,FLT		;BAAAD News!  Is the disk "faulted"?
	mov	ax,(256*FAULTED)+HARDERR ;(Get hardware error codes).
WhichE:	jz s	Kaput		;If "zero", use AL-reg. return code.
	mov	al,ah		;Use AH-reg. return code of this pair.
Kaput:	stc			;Set carry flag (error!) and exit.
	ret
;
; Subroutine to move our DMA command-list up to XMS memory for I-O.
;   This avoids "DMA Hangs" if we are loaded in UMBPCI "Shadow RAM"!
;
MovCLX:	mov	ecx,8		 ;Set command-list move parameters
	mov	esi,(IOAdr-@)	 ;  and then fall thru to "MvData".
@CLAddr	equ	[$-4].dwd	 ;(Command-list address, Init set).
	mov	edi,[bx+PRDAd-@]
;
; Subroutine to do XMS moves.   At entry, the move parameters are:
;
;   ECX:   Number of bytes to move.   MUST be an even value as only
;	     whole 16-bit words are moved!   Up to a 4-GB value can
;	     be used as data is moved in 2K sections, which permits
;	     interrupt handling between sections.
;   ESI:   "True" 32-bit source data address (NOT segment/offset!).
;   EDI:   "True" 32-bit destination address (NOT segment/offset!).
;
; At exit, the carry bit is zero for no errors or is SET for a move
; error.   If so, the AL-reg. has a 0FFh "XMS error" code.   The DS
; and ES-regs. are saved/restored, and the BX-reg. is zero at exit.
; All other registers are NOT saved, for speed.   Only a "protected
; mode" move (JEMM386, etc.) can post an error.   "Real mode" moves
; (UMBPCI, etc.) have NO error codes to post!
;
MvData:	push	ds		;Save DS- and ES-regs.
	push	es
	sti			;Ensure CPU interrupts are enabled.
	cld			;Ensure FORWARD "string" commands.
	mov	ebp,ecx		;Save byte count in EBP-reg.
	mov	edx,2048	;Set 2K section count in EDX-reg.
	smsw	ax		;Get CPU "machine status word".
	shr	ax,1		;Are we now in "protected mode"?
	jc s	MvProt		;Yes, go use "protected mode" logic.
MvRNxt:	mov	ecx,edx		;Set move section count in CX-reg.
	cli			;Disable CPU interrupts.
	cmp	edx,ebp		;At least 2048 bytes left?
	jbe s	MvRGo		;Yes, use full section count.
	mov	ecx,ebp		;Use remaining byte count.
MvRGo:	db	02Eh,00Fh	;"lgdt cs:GDTP", coded the hard-way
	db	001h,016h	;   to avoid any annoying V5.1 MASM
	dw	(GDTP-@)	;   warning messages about it!
@GDTP	equ	[$-2].lw
	mov	eax,cr0		;Set "protected-mode" control bit.
	or	al,001h
	mov	cr0,eax
	mov	bx,(GDT_DS-OurGDT)  ;Set DS- & ES-reg. "selectors".
	mov	ds,bx
	mov	es,bx
	shr	cx,2		;Convert byte count to dword count.
	db	0F3h,067h	;Move all 32-bit words using "rep"
	movsd			;  and "address-override" prefixes.
	db	067h,090h	;("Override" & "nop" for 386 BUG!).
	jnc s	MvRClP		;"Odd" 16-bit word to be moved?
	db	067h		;Yes, move it also, using an
	movsw			;  "address-override" prefix.
	db	067h,090h	;("Override"/"nop" for 386 BUG!).
MvRClP:	dec	ax		;Clear "protected-mode" control bit.
	mov	cr0,eax
	sti			;Allow interrupts after next command.
	sub	ebp,edx		;Any more data sections to move?
	ja s	MvRNxt		;Yes, go move next data section.
MvDone:	xor	ax,ax		;Success!  Reset carry and error code.
MvExit:	pop	es		;Reload caller's segment registers.
	pop	ds
	mov	bx,0		;Rezero BX-reg. but save carry bit.
	ret			;Exit.
MvProt:	push	edx		;"Protected mode" -- save section ct.
	push	esi		;Save move source & destination addrs.
	push	edi
	push	ebp		;Save remaining move byte count.
	mov	ecx,edx		;Set move section count in CX-reg.
	cmp	edx,ebp		;At least 2048 bytes left?
	jbe s	MvPGo		;Yes, use full section count.
	mov	cx,bp		;Use remaining byte count instead.
MvPGo:	shr	cx,1		;Convert byte count to word count.
	push	cs		;Point ES-register to this driver.
	pop	es
	push	edi		;Set up destination descriptor.
	mov	di,(DstDsc+2-@)
@MvDsc1	equ	[$-2].lw
	pop	ax
	stosw
	pop	ax
	stosb
	mov	es:[di+2],ah
	push	esi		      ;Set up source descriptor ("sub"
	sub	di,(DstDsc-SrcDsc+3)  ;  will zero carry for Int 15h).
	pop	ax
	stosw
	pop	ax
	stosb
	mov	es:[di+2],ah
	mov	si,(MoveDT-@)	;Point to move descriptor table.
@MvDsc2	equ	[$-2].lw
	mov	ah,087h		;Have JEMM386/BIOS move next section.
	int	015h
	pop	ebp		;Reload all 32-bit move parameters.
	pop	edi
	pop	esi
	pop	edx
	sti			;Ensure CPU interrupts are enabled.
	cld			;Ensure FORWARD "string" commands.
	mov	al,XMSERR	;Get our XMS error code.
	jc s	MvExit		;If any BIOS error, exit immediately.
	add	esi,edx		;Update source and dest. addresses.
	add	edi,edx
	sub	ebp,edx		;Any more data sections to move?
	ja s	MvProt		;Yes, go move next data section.
	jmp s	MvDone		;Done:  Set "success" code and exit.
;
; Subroutine to read all data for a CD/DVD "Read Long" request.
;
RLRead:	cmp	[bx+DMF-@],bl	  ;Will this input be using DMA?
	je s	RlRdA		  ;No, go do input using "PIO mode".
	call	UseCB		  ;Can current cache buffer be used?
	cmc			  ;If so, set XMS-move flag for below.
	adc	[bx+XMF-@],bl
	call	MovCLX		  ;Move DMA command-list up to XMS.
	jc s	RLRdE		  ;Failed?  Return "general error"!
RlRdA:	call	DoIO		  ;Do desired UltraDMA input request.
	jc s	RLRdX		  ;If any errors, exit below.
	or	[bx+XMF-@],bl	  ;Do we need an XMS buffer "move"?
	jz s	RLRdX		  ;No, just exit below.
	mov	ecx,[bx+VDSLn-@]  ;Get parameters for an XMS move.
	mov	esi,[bx+IOAdr-@]
	mov	edi,[bx+VDSAd-@]
	call	MvData		  ;Move data from XMS to user buffer.
RLRdE:	mov	al,GENERR	  ;Use "general error" code if error.
RLRdX:	ret			  ;Exit.
;
; Subroutine to issue a CD/DVD "Play Audio" command.   At entry, the
;   DI-reg. points to the audio-start address for this drive.
;
PlayA:	mov	eax,cs:[di]	;Set "packet" audio-start address.
	call	CvtMSF
	mov	[bx+PktLBA+1-@],eax
	mov	eax,cs:[di+4]	;Set "packet" audio-end address.
	call	CvtMSF
	mov	[bx+PktLH-@],eax
	mov	[bx+Packet-@].lb,047h  ;Set "Play Audio" command.
	jmp s	DoCmd		       ;Start playing audio & exit.
;
; Ye Olde CD/DVD I-O Subroutine.   All CD/DVD I-O gets executed here!
;
DoTOC1:	mov	[bx+PktLH-@],al	   ;"TOC" -- set packet session no.
DoTOC2:	mov	al,12		   ;Use 12-byte "TOC" alloc. count.
	nop			   ;(Unused alignment "filler").
DoBuf1:	mov	[bx+PktLn+1-@],al  ;Buffered -- set packet count.
DoBuf2:	mov	[bx+VDSLn-@],al	   ;Set 8-bit data-transfer length.
	mov	[bx+VDSOf-@].lw,(InBuf-@)  ;Use our buffer for I-O.
	mov	[bx+UserB+2-@],ds
	jmp s	DoCmd1		   ;Go zero hi-order and start I-O.
DoCmd:	mov	[bx+VDSLn-@],bl    ;Command only -- zero xfr length.
DoCmd1:	mov	[bx+VDSLn+1-@],bx  ;Zero hi-order VDS buffer length.
DoIO:	push	si		   ;Save SI- and ES-registers.
	push	es
	mov	[bx+Try-@].lb,4	;Set request retry count of 4.
DoIO1:	call	HltDMA		;Stop previous DMA & select drive.
	call	ChkTO		;Await controller-ready.
	jc s	DoIO4		;Timeout!  Handle as a "hard error".
	cmp	[bx+DMF-@],bl	;UltraDMA input request?
	je s	DoIO2		;No, output ATAPI "packet" command.
	mov	al,008h		;Get DMA-input command bit.
	call	IniDMA		;Initialize our DMA input transfer.
DoIO2:	mov	ax,[bx+VDSOf-@]	;Reset data-transfer buffer address.
	mov	[bx+UserB-@],ax
	mov	ax,[bx+VDSLn-@]	;Reset data-transfer byte count.
	mov	[bx+UserL-@],ax
	mov	dx,[bx+IdeDA-@]	;Point to IDE "features" register.
	inc	dx
	mov	al,[bx+DMF-@]	;If UltraDMA input, set "DMA" flag.
	out	dx,al
	add	dx,3		;Point to byte count registers.
	mov	si,(UserL-@)	;Output data-transfer length.
	outsb
	inc	dx
	outsb
	inc	dx		;Point to command register.
	inc	dx
	mov	al,0A0h		;Issue "Packet" command.
	out	dx,al
	mov	cl,DRQ		;Await controller- and data-ready.
	call	ChkTO1
	jc s	DoIO4		;Timeout!  Handle as a "hard error".
	xchg	ax,si		;Save BIOS timer address.
	mov	dx,[bx+IdeDA-@]	;Point to IDE data register.
	mov	cl,6		;Output all 12 "Packet" bytes.
	mov	si,(Packet-@)
	rep	outsw
	xchg	ax,si		;Reload BIOS timer address.
	mov	ah,STARTTO	;Allow 7 seconds for drive startup.
	cmp	[bx+DMF-@],bl	;UltraDMA input request?
	je s	DoIO12		;No, start "PIO mode" transfer below.
	mov	[bx+UserL-@],bx	;Reset transfer length (DMA does it).
	mov	ch,ah		;Set 7-second timeout in AH-reg.
	add	ch,es:[si]
	call	RunDMA		;Start and monitor next DMA transfer.
	jnz s	DoIO4		;Problems?  Handle as a "hard error"!
	call	ChkTO		;Await final controller-ready.
	jc s	DoIO4		;Timeout!  Handle as a "hard error"!
DoIO3:	mov	si,[bx+AudAP-@]	;Get drive media-change flag pointer.
	dec	si
	and	ax,00001h	;Did controller detect any errors?
	jz s	DoIO6		;No, see if all data was transferred.
	sub	dx,6		;Get controller's sense key value.
	in	al,dx
	shr	al,4
	cmp	al,006h		;Is sense key "Unit Attention"?
	je s	DoIO7		;Yes, check for prior media-change.
	mov	ah,0FFh		;Get 0FFh flag for "Not Ready".
	cmp	al,002h		;Is sense key "Drive Not Ready"?
	je s	DoIO8		;Yes, go set our media-change flag.
DoIO4:	mov	dx,[bx+IdeDA-@]	;Hard error!  Point to command reg.
	add	dx,7
	mov	al,008h		;Issue ATAPI "soft reset" to drive.
	out	dx,al
	mov	al,11		;Get "hard error" return code.
	dec	[bx+Try-@].lb	;Do we have more I-O retries left?
	jz s	DoIO9		;No, set carry & return error code.
	jmp	DoIO1		;Try re-executing this I-O request.
DoIO6:	cmp	[bx+UserL-@],bx	;Was all desired data input?
	jne s	DoIO4		;No?  Handle as a "hard error".
	mov	cs:[si].lb,001h	;Set "no media change" flag.
	jmp s	DoIO10		;Go reload regs. and exit below.
DoIO7:	mov	al,002h		;"Attention":  Get "Not Ready" code.
DoIO8:	xchg	ah,cs:[si]	;Load & set our media-change flag.
	or	[bx+IOF-@].lb,1	    ;Flush cache on next caching req.
	mov	cs:[si+12].lb,0FFh  ;Make last-session LBA invalid.
	dec	ah		    ;Media-change flag already set?
	jnz s	DoIO9		;Yes, set carry flag and exit.
	mov	al,15		;Return "Invalid Media Change".
DoIO9:	stc			;Set carry flag (error!).
DoIO10:	pop	es		;Reload ES- and SI-registers.
	pop	si
	mov	di,(InBuf-@)	;For audio, point to our buffer.
	ret			;Exit.
	db	0		;(Unused alignment "filler").
DoIO11:	mov	ah,SEEKTO	;"PIO mode" -- Get "seek" timeout.
DoIO12:	call	ChkTO2		;Await controller-ready.
	jc s	DoIO4		;Timeout!  Handle as a "hard error".
	test	al,DRQ		;Did we also get a data-request?
	jz s	DoIO3		;No, go check for any input errors.
	dec	dx		;Get controller-buffer byte count.
	dec	dx
	in	al,dx
	mov	ah,al
	dec	dx
	in	al,dx
	mov	dx,[bx+IdeDA-@]	;Point to IDE data register.
	mov	si,[bx+UserL-@]	;Get our data-transfer length.
	or	si,si		;Any remaining bytes to input?
	jz s	DoIO14		;No, "eat" all residual data.
	cmp	si,ax		;Remaining bytes > buffer count?
	jbe s	DoIO13		;No, input all remaining bytes.
	mov	si,ax		;Use buffer count as input count.
DoIO13:	les	di,[bx+UserB-@]	;Get input data-transfer address.
	mov	cx,si		;Input all 16-bit data words.
	shr	cx,1
	rep	insw
	add	[bx+UserB-@],si	;Increment data-transfer address.
	sub	[bx+UserL-@],si	;Decrement data-transfer length.
	sub	ax,si		;Any data left in controller buffer?
	jz s	DoIO11		;No, await next controller-ready.
DoIO14:	xchg	ax,cx		;"Eat" all residual input data.
	shr	cx,1		;(Should be NO residual data as we
DoIO15:	in	ax,dx		;  always set an exact byte count.
	loop	DoIO15		;  This logic is only to be SAFE!).
	jmp s	DoIO11		;Go await next controller-ready.
;
; Subroutine to convert "RedBook" MSF values to an LBA sector number.
;
CvtLBA:	mov	cx,ax		;Save "seconds" & "frames" in CX-reg.
	shr	eax,16		;Get "minute" value.
	cmp	ax,99		;Is "minute" value too large?
	ja s	CvtLBE		;Yes, return -1 error value.
	cmp	ch,60		;Is "second" value too large?
	ja s	CvtLBE		;Yes, return -1 error value.
	cmp	cl,75		;Is "frame" value too large?
	ja s	CvtLBE		;Yes, return -1 error value.
	xor	edx,edx		;Zero EDX-reg. for 32-bit math below.
	mov	dl,60		;Convert "minute" value to "seconds".
	mul	dl		;(Multiply by 60, obviously!).
	mov	dl,ch		;Add in "second" value.
	add	ax,dx
	mov	dl,75		;Convert "second" value to "frames".
	mul	edx		;(Multiply by 75 "frames"/second).
	mov	dl,150		;Subtract offset - "frame".
	sub	dl,cl		;("Adds" frame, "subtracts" offset).
	sub	eax,edx
	ret			;Exit.
CvtLBE:	or	eax,-1		;Too large!  Set -1 error value.
	ret			;Exit.
	db	0		;(Unused alignment "filler").
;
; Subroutine to test for CD/DVD I-O timeouts.   At entry the CL-reg.
;   is 008h to test for a data-request, also.   At exit, the DX-reg.
;   points to the drive's IDE primary-status reg.   The AH-, SI- and
;   ES-regs. will be lost.
;
ChkTO:	xor	cx,cx		;Check for only controller-ready.
ChkTO1:	mov	ah,CMDTO	;Use 500-msec command timeout.
ChkTO2:	mov	es,bx		;Point to low-memory BIOS timer.
	mov	si,BIOSTMR
	add	ah,es:[si]	;Set timeout limit in AH-reg.
ChkTO3:	cmp	ah,es:[si]	;Has our I-O timed out?
	stc			;(If so, set carry flag).
	je s	ChkTOX		;Yes?  Exit with carry flag on.
	mov	dx,[bx+IdeDA-@]	;Read IDE primary status.
	add	dx,7
	in	al,dx
	test	al,BSY		;Is our controller still busy?
	jnz s	ChkTO3		;Yes, loop back and test again.
	or	cl,cl		;Are we also awaiting I-O data?
	jz s	ChkTOX		;No, just exit.
	test	al,cl		;Is data-request (DRQ) also set?
	jz s	ChkTO3		;No, loop back and test again.
ChkTOX:	ret			;Exit:  Carry "on" denotes a timeout!
;
; Subroutine to convert an LBA sector number to "RedBook" minutes,
;   seconds, and "frames" format (MSF).
;
CvtMSF:	add	eax,150		;Add in offset.
	push	eax		;Get address in DX:AX-regs.
	pop	ax
	pop	dx
	mov	cx,75		;Divide by 75 "frames"/second.
	div	cx
	shl	eax,16		;Set "frames" remainder in upper EAX.
	mov	al,dl
	ror	eax,16
	mov	cl,60		;Divide quotient by 60 seconds/min.
	div	cl
	ret			;Exit -- EAX-reg. contains MSF value.
;
; Subroutine to issue a VDS "lock" request.
;
VDLock:	pusha			     ;Save all 16-bit CPU registers.
	push	ds
	or	[bx+VDSAd-@].dwd,-1  ;Invalidate VDS buffer address.
	mov	ax,08103h	     ;Get VDS "lock" parameters.
	mov	dx,0000Ch
	mov	di,(VDSLn-@)	     ;Point to VDS parameter block.
	push	ds
	pop	es
	int	04Bh		     ;Execute "VDS lock" VDS request.
	sti			     ;RESTORE all critical settings!
	cld			     ;(NEVER "trust" external code!).
	pop	ds		     ;Reload all 16-bit registers.
	popa
	jc s	VDLokX		     ;VDS error:  Exit immediately!
	mov	eax,[bx+VDSAd-@]     ;Get 32-bit VDS buffer address.
	cmp	eax,-1		     ;Is VDS address not "all-ones"?
	jb s	VDLokF		     ;Yes, set VDS "lock" flag.
	movzx	eax,[bx+VDSSg-@].lw  ;VDS logic is NOT present --
	shl	eax,4		     ;  set 20-bit buffer address.
	add	eax,[bx+VDSOf-@]
	mov	[bx+VDSAd-@],eax
VDLokF:	adc	[bx+VLF-@],bl	     ;Set VDS "lock" flag from carry.
VDLokX:	ret			     ;Exit.
;
; Subroutine to test if the current cache buffer can be used for DMA.
;   If so, the cache-buffer address is set for UltraDMA and the carry
;   flag is "off" at exit.   If not, or if the stand-alone driver (no
;   cache) is in use, the carry flag is "on" at exit.
;
UseCB:	mov	cl,[bx+RqCSC-@]	      ;Get current I-O sector count.
				      ;("stc" & "ret" if no caching).
	cmp	cl,[bx+RqSec-@]	      ;Is all I-O in one cache block?
	jb s	VDLokX		      ;No, exit and use regular I-O.
	or	[bx+RqCmd-@].lb,080h  ;Do NOT cache data, after I-O!
	mov	eax,[bx+RqXMS-@]      ;Do DMA thru the cache buffer!
	mov	[bx+IOAdr-@],eax
	ret			      ;Exit.
;
; Subroutine to start and monitor the next UltraDMA I-O transfer.
;
RunDMA:	mov	es:[si+HDI_OFS],bl  ;Reset BIOS disk-interrupt flag.
	mov	dx,[bx+DMAAd-@]	    ;Point to DMA command register.
	in	al,dx		;Set DMA Start/Stop bit (starts DMA).
	inc	ax
	out	dx,al
RunDM1:	inc	dx		;Point to DMA status register.
	inc	dx
	in	ax,dx		;Read DMA controller status.
	dec	dx		;Point back to DMA command register.
	dec	dx
	and	al,DMI+DME	;DMA interrupt or DMA error?
	jnz s	RunDM2		;Yes, halt DMA and check results.
	cmp	ch,es:[si]	;Has our DMA transfer timed out?
	je s	RunDM2		    ;Yes?  Halt DMA & post timeout!
	cmp	es:[si+HDI_OFS],bl  ;Did BIOS get a disk interrupt?
	je s	RunDM1		    ;No, loop back & retest status.
	mov	al,DMI		;Set "simulated" interrupt flag.
RunDM2:	push	ax		;Save ending DMA status.
	in	al,dx		;Reset DMA Start/Stop bit.
	and	al,0FEh
	out	dx,al
	pop	ax		;Reload ending DMA status.
	cmp	al,DMI		;Did DMA end with only an interrupt?
	jne s	RunDMX		;No?  Exit with "non-zero" (error!).
	inc	dx		;Reread DMA controller status.
	inc	dx
	in	al,dx
	test	al,DME		;Any "late" DMA error after DMA end?
RunDMX:	ret			;Exit -- "Non-zero" indicates error.
;
; Subroutine to set the 32-bit DMA address for an UltraDMA transfer.
;   At entry the EAX-reg. has the address from a VDS "lock" request.
;   At exit, if this address was unsuitable for UltraDMA, the driver
;   "Main XMS buffer" address has been set and the carry flag is on.
;
SetAdr:	test	al,003h		      ;Is user buffer 32-bit aligned?
	jnz s	SetAd1		      ;No, use our XMS memory buffer.
	cmp	[bx+VDSAd-@].hw,009h  ;Is DMA beyond our 640K limit?
	ja s	SetAd1		      ;Yes, use our XMS memory buffer.
	mov	cx,[bx+IOLen-@]	  ;Get (byte count - 1) + I-O address.
	dec	cx
	add	cx,ax		  ;Would I-O cross a 64K boundary?
	jnc s	SetAd2		  ;No, set final buffer address.
SetAd1:	stc			  ;Set carry to denote "our buffer".
	mov	eax,0		  ;Get our 32-bit XMS buffer addr.
@XBAddr	equ	[$-4].dwd	  ;(XMS buffer address, Init set).
SetAd2:	mov	[bx+IOAdr-@],eax  ;Set final 32-bit buffer addr.
	ret			  ;Exit.
;
; Subroutine to clear our ATAPI "packet" area for the next request.
;
ClrPkt:	mov	[bx+Packet-@],bx    ;Zero 1st 10 ATAPI packet bytes.
	mov	[bx+Packet+2-@],bx  ;(Last 2 are unused "pad" bytes).
	mov	[bx+Packet+4-@],bx
	mov	[bx+Packet+6-@],bx
	mov	[bx+Packet+8-@],bx
	ret			    ;Exit.
HMALEN1	equ	($-CDReq)	;(Length of all HMA driver routines).
;
; Main Caching Routine.
;
Cache:	cmp	[bx+RqTyp-@].lb,0C0h ;Is this a diskette request?
	jne s	Flush		     ;No, see if cache needs a flush.
	mov	es,bx		     ;Point ES-reg. to low memory.
	mov	cl,01Fh		     ;Get BIOS diskette status code.
	and	cl,es:[DKTSTAT]
	cmp	cl,MCHANGE	;Diskette media-change detected?
	je s	FlushA		;Yes, go flush our cache now.
Flush:	test	[bx+IOF-@].lb,1	;Does cache need to be flushed?
	jz s	NextIO		;No, proceed with this request.
FlushA:	mov	ecx,0		;Move reset block to search table,
@TC1	equ	[$-4].dwd	;  MUCH faster than storing words!
	mov	esi,0
@STblR	equ	[$-4].dwd
	mov	edi,0
@STblA	equ	[$-4].dwd
	call	MvData
	nop			    ;(Unused alignment "filler").
FlushE:	jc	EndIO		    ;XMS error?  Get out immediately!
	mov	[bx+STLmt-@],bx	    ;Reset binary-search limit index.
	or	[bx+LUTop-@].dwd,-1 ;Reset LRU start & end indexes.
	and	[bx+IOF-@].lb,0FEh  ;O.K. to reset flush flag now.
NextIO:	push	ds		    ;Set ES-reg. same as DS-reg.
	pop	es
	mov	al,0		;Get "granularity" (sectors/block).
@GRAN1	equ	[$-1].lb	;(Block "granularity", set by Init).
	cmp	al,[bx+RqSec-@]	;Will we need multiple cache blocks?
	jbe s	SetSC		;Yes, use "full block" sector count.
	mov	al,[bx+RqSec-@]	;Use remaining request sector count.
SetSC:	mov	[bx+RqCSC-@],al	;Set maximum I-O sector count.
	mov	[bx+RqXMS-@],bx ;Reset lower XMS buffer offset.
	mov	dx,0		;Set initial binary-search offset.
@MP1	equ	[$-2].lw	;(Search-table midpoint, Init set).
	mov	bp,dx		;Set initial binary-search index.
	dec	bp
Search:	shr	dx,1		;Divide binary-search offset by 2.
	cmp	bp,[bx+STLmt-@]	;Is our search index too high?
	jae s	SrchLo		;Yes, cut search index by offset.
	call	STGet		;Get next binary-search table index.
	jc s	FlushE		;If any XMS error, exit immediately!
	mov	ax,WBLBA	;Get next cache entry into buffer.
	call	CBGet
	jc s	FlushE		;If any XMS error, exit immediately!
	mov	si,(RqLBA-@)	;Compare initial request & table LBA.
	call	CComp
	jae s	ChkEnd		;Request >= table:  Check for found.
SrchLo:	sub	bp,dx		;Subtract offset from search ptr.
	or	dx,dx		;Offset zero, i.e. last compare?
	jnz s	Search		;No, go compare next table entry.
	jmp s	NoFind		;Handle this request as "not found".
ChkEnd:	je s	Found		;Request = table:  Treat as "found".
	mov	cl,[bx+CBSec-@]	;Calculate and set ending entry LBA.
	mov	si,(CBLBA-@)
	call	CalcEA
	mov	si,(RqLBA-@)	;Compare request start & entry end.
	call	CComp1
	jb s	Found		   ;Request < Entry:  Handle as found.
	ja s	SrchHi		   ;Request > Entry:  Bump search ptr.
	cmp	[bx+CBSec-@].lb,0  ;Is this cache entry "full"?
@GRAN2	equ	[$-1].lb	   ;(Block "granularity", Init set).
	jb s	Found		;No, handle this request as "found".
	align	4		;(Unused alignment "filler").
SrchHi:	add	bp,dx		;Add offset to search pointer.
	or	dx,dx		;Offset zero, i.e. last compare?
	jnz s	Search		;No, go compare next table entry.
	inc	bp		;Bump index to next table entry.
	nop			;(Unused alignment "filler").
NoFind:	mov	WBLBA+2,bp	;Unfound:  Save search-table offset.
	mov	bp,[bx+STLmt-@]	;Get next "free" cache-table index
	call	STGet		;  and leave index in work buffer.
	jc s	NFind1		;If any XMS error, exit immediately!
	movzx	ecx,bp		  ;Get move-up word count.
	movzx	esi,[WBLBA+2].lw
	sub	cx,si		  ;Any search indexes to move up?
	jz s	NFind2		  ;No, go set up new cache entry.
	shl	ecx,1		  ;Set move-up byte count.
	shl	esi,1		  ;Set 32-bit move-up addresses.
	add	esi,dword ptr 0
@STblB	equ	[$-4].dwd
	lea	edi,[esi+2]
	call	ZipZip		;"Zip" data to search buffer & back.
	jc s	NFind1		;If any XMS error, exit immediately!
	mov	bp,WBLBA+2	;Set up table-index "put" to XMS.
	add	bx,1
	call	STPut		;Insert "free" index in search table.
NFind1:	jc s	UpdErr		;If XMS error, exit immediately!
NFind2:	inc	[bx+STLmt-@].lw	;Advance binary-search limit index.
	mov	si,(RqLBA-@)	;Set 48-bit LBA in new entry.
	mov	di,(CBLBA-@)
	movsd
	movsw
	movsb			;Set "cache unit" in new entry.
	mov	[di].lb,0	;Reset new entry's sector count.
	mov	ax,WBLBA	;Reload "free" cache-table index.
Found:	mov	RqIndex,ax	   ;Post current cache-entry index.
	mov	cx,[bx+RqLBA+4-@]  ;Get starting I-O offset in block.
	sub	cx,[bx+CBLBA+4-@]
	mov	[bx+RqXMS-@],cl	   ;Set starting XMS sector offset.
	mov	[bx+LBA2+1-@],cx   ;Save starting I-O sector offset.
	cmp	[bx+CBSec-@],bl	   ;Is this a new cache-table entry?
	je s	ReLink		   ;Yes, relink entry as top-of-list.
	push	ax		   ;Unlink this entry from LRU list.
	call	UnLink
	pop	ax
ReLink:	mov	[bx+RqXMS+2-@],bx  ;Reset upper XMS buffer offset.
	movzx	edx,ax		   ;Get 32-bit cache block number.
	shl	edx,2		   ;Shift number to starting sector.
@GRSSC1	equ	[$-1].lb	   ;("Granularity" shift, Init set).
	add	[bx+RqXMS-@],edx   ;Add to "preset" sector offset.
	shl	[bx+RqXMS-@].dwd,9	  ;Convert sectors to bytes.
	add	[bx+RqXMS-@].dwd,020000h  ;Add in XMS "base" address.
@XBase1	equ	[$-4].dwd		  ;(XMS "base", set by Init).
	mov	cx,0FFFFh	;Make this entry "top of list".
	or	[bx+CBLst-@],cx	;Set this entry's "last" index.
	mov	dx,ax		;Swap top-of-list & entry index.
	xchg	dx,[bx+LUTop-@]
	mov	[bx+CBNxt-@],dx ;Set this entry's "next" index.
	cmp	dx,cx		;Is this the only LRU index?
	mov	cx,ax		;(Get this entry's index in CX-reg.).
	je s	ReLnk1		;Yes, make entry last on LRU list.
	push	ax		;Link entry to prior "top of list".
	call	UnLnk3
	pop	ax
	jmp s	ReLnk2		;Go deal with I-O sector count.
Update:	mov	[bx+CBSec-@],cl	;Update this entry's total sectors.
	call	CBPut		;Update this cache-table entry.
UpdErr:	jc s	EndIOJ		;If any XMS error, exit immediately!
	movzx	cx,[bx+RqCSC-@]	;Calculate ending LBA for this I-O.
	mov	si,(RqLBA-@)
	call	CalcEA
	inc	bp		;Skip to next search index.
	nop			;(Unused alignment "filler").
Ovrlap:	cmp	bp,[bx+STLmt-@]	;More cache-table entries to check?
	jae s	CachIO		;No, O.K. to handle "found" entry.
	call	STGet		;Get next search table index.
	jc s	EndIOJ		;If any XMS error, bail out now!
	mov	ax,WBLBA	;Get next cache entry into buffer.
	call	CBGet
	jc s	EndIOJ		;If any XMS error, exit immediately!
	mov	si,(LBABuf-@)	;Compare request end & entry start.
	call	CComp
	jbe s	CachIO		;Request <= entry:  O.K. to proceed.
	push	bp		;Delete this overlapping table entry.
	call	Delete
	pop	bp
	jmp s	Ovrlap		  ;Go check for more entry overlap.
ReLnk1:	mov	[bx+LUEnd-@],ax	  ;Make entry last on LRU list, too!
ReLnk2:	mov	cx,[bx+LBA2+1-@]  ;Reload initial I-O sector offset.
	mov	ch,0		  ;Get entry's available sectors.
@GRAN3	equ	[$-1].lb	  ;(Block "granularity", Init set).
	sub	ch,cl
	cmp	ch,[bx+RqCSC-@]	;More I-O sectors than available?
	jae s	Larger		;No, retain maximum sector count.
	mov	[bx+RqCSC-@],ch	;Reduce current I-O sector count.
	align	4		;(Unused alignment "filler").
Larger:	add	cl,[bx+RqCSC-@]	;Get ending I-O sector number.
	cmp	cl,[bx+CBSec-@]	;More sectors than entry has now?
	ja s	Update		;Yes, update entry sectors.
	inc	bx		;Reset Z-flag for "put" into XMS.
	call	CBPut		;Update this cache-table entry.
EndIOJ:	jc s	EndIO		      ;If any XMS error, exit fast!
	test	[bx+RqCmd-@].lb,002h  ;Is this a read request?
	jz s	BufMov		      ;Yes, move cache data to user.
CachIO:	bts	[bx+RqCmd-@].lb,6     ;I-O done during a prior block?
	jc s	BfMore		      ;Yes, buffer more cache data.
	cmp	[bx+RqTyp-@].lb,CDTYP ;Is this one of our disks/CDs?
	ja s	OtherU		      ;No, go check for other units.
	je s	CDVDIO		;CD/DVD -- go call "RLRead" below.
	call	UdmaIO		;Call "UdmaIO" for SATA/IDE disks.
	jmp s	ErrChk		;Go check for any I-O errors.
CDVDIO:	call	RLRead		;Call "RLRead" for CD/DVD input.
	jmp s	ErrChk		;Go check for any I-O errors.
OtherU:	or	[bx+RqTyp-@],bl	;Is this unit handled by the BIOS?
	js s	BiosIO		;Yes, go "Call the BIOS" below.
	call	DvrPtr		;"External" unit:  Call its driver.
	jmp s	ErrChk		;Go check for any I-O errors.
BiosIO:	lds	si,CStack	;BIOS I-O:  Reload caller CPU flags.
	push	[si+2].lw
	popf
	pop	es		;Reload all CPU registers.
	pop	ds
	popad
	pushf			;"Call the BIOS" to do this request.
	db	09Ah
@Prv13B	dd	0		;(Prior Int 13h vector, set by Init).
	pushad			;Save all CPU registers again.
	push	ds
	push	es
	mov	al,ah		;Move any BIOS error code to AL-reg.
ErrChk:	sti			;Restore critical driver settings.
	cld
	push	ss		;Reload this driver's DS- & ES-regs.
	pop	ds
	push	ds
	pop	es
	mov	bx,0		;Rezero BX-reg. but save carry.
	jnc s	InCach		;If no error, go see about caching.
IOErr:	push	ax		;Error!  Save returned error code.
	mov	ax,RqIndex	;Delete cache entry for this I-O.
	call	ScanD
	pop	ax		;Reload returned error code.
	stc			;Set carry again to denote error.
EndIO:	db	0EAh		;Return to hard-disk exit routine.
	dw	(HDExit-@)
@HXSeg2	dw	0		;(Driver segment address, Init set).
InCach:	or	[bx+RqCmd-@],bl	;Did user data get cached during I-O?
	js s	ChkFul		;Yes, check if cache tables are full.
BfMore:	or	al,1		     ;Ensure we LOAD data into cache!
BufMov:	movzx	ecx,[bx+RqCSC-@].lb  ;Set XMS move sector count.
	mov	esi,[bx+RqXMS-@]     ;Set desired XMS buffer address.
	mov	edi,[bx+RqBuf-@]     ;Set user buffer as destination.
	call	UBufM1		     ;Move data between user and XMS.
	jc s	IOErr			;If error, use routine above.
ChkFul:	cmp	[bx+STLmt-@].lw,08000h  ;Is binary-search table full?
@TE1	equ	[$-2].lw		;(Table "end", Init set).
	jb s	MoreIO		   ;No, check for more sectors to go.
	push	ax		   ;Delete least-recently-used entry.
	mov	ax,[bx+LUEnd-@]
	call	ScanD
	pop	ax
MoreIO:	xor	ax,ax		   ;Reset error code for exit above.
	movzx	cx,[bx+RqCSC-@]	   ;Get current I-O sector count.
	sub	[bx+RqSec-@],cl	   ;More data sectors left to handle?
	jz s	EndIO		   ;No, return to our "exit" routine.
	add	[bx+RqLBA+4-@],cx  ;Update current LBA for next I-O.
	adc	[bx+RqLBA+2-@],bx
	adc	[bx+RqLBA-@],bx
	shl	cx,1		   ;Convert sector count to bytes.
	add	[bx+RqBuf+1-@],cl  ;Update user I-O buffer address.
	adc	[bx+RqBuf+2-@],bx
	jmp	NextIO		   ;Go handle next cache data block.
;
; Subroutine to calculate ending request or cache-table LBA values.
;
CalcEA:	mov	di,(LBABuf-@)	;Point to address-comparison buffer.
	push	[si].dwd	;Move high-order LBA into buffer.
	pop	[di].dwd
	add	cx,[si+4]	;Calculate ending LBA.
	mov	[di+4],cx
	adc	[di+2],bx
	adc	[di],bx
CalcEX:	ret			;Exit.
;
; Subroutine to do 7-byte "cache unit" and LBA comparisons.
;
CComp:	mov	di,(CBLBA-@)	;Point to current-buffer LBA value.
CComp1:	mov	cl,[bx+RqUNo-@]	;Compare our "unit" & table "unit".
	cmp	cl,[bx+CBUNo-@]
	jne s	CalcEX		;If units are different, exit now.
	mov	cx,3		;Compare our LBA & cache-table LBA.
	rep	cmpsw
	ret			;Exit -- Main routine checks results.
;
; Subroutines to do search-table index and cache-entry XMS moves.
;
WBGet:	xor	di,di		;Work buffer:  Set Z-flag for "get".
WBPut:	mov	edi,(WBLBA-@)	;Set 32-bit "working" buffer address.
@WBAdr1	equ	[$-4].dwd	;(Working-buffer address, Init set).
	jmp s	CEMov		;Go move data to/from working buffer.
CBGet:	xor	di,di		;Entry buffer:  Set Z-flag for "get".
CBPut:	mov	edi,(CBLBA-@)	;Set 32-bit "current" buffer address.
@CBAdr1	equ	[$-4].dwd	;(Current-buffer address, Init set).
CEMov:	push	ax		;Save all needed 16-bit regs.
	push	cx
	push	dx
	pushf			;Save CPU Z-flag from above.
	xor	ecx,ecx		;Set cache-table entry size.
	mov	cl,12
	mul	cx		;Multiply cache index by entry size.
	push	dx		;Get cache-entry offset in ESI-reg.
	push	ax
	pop	esi
	add	esi,010000h	;Add cache-table "base" address.
@CTBas1	equ	[$-4].dwd	;(Cache-table "base", Init set).
	jmp s	CEMov1		;Go get or put cache entry below.
STGet:	xor	di,di		;Search index:  Set Z-flag for get.
STPut:	mov	edi,(WBLBA-@)	;Set 32-bit "working" buffer addr.
@WBAdr2	equ	[$-4].dwd	;(Working-buffer addr., Init set).
	push	ax		;Save all needed 16-bit regs.
	push	cx
	push	dx
	pushf			;Save CPU Z-flag from above.
	mov	ecx,2		;Set binary-search index size.
	movzx	esi,bp		;Get 32-bit offset of requested
	shl	esi,1		;  index in binary-search table.
	add	esi,dword ptr 0	;Add starting search-table addr.
@STblC	equ	[$-4].dwd
CEMov1:	popf			;Reload CPU flags.
	jz s	CEMov2		;Are we "getting" from XMS memory?
	xchg	esi,edi		;No, "swap" source and destination.
CEMov2:	push	bp		;Save BP-reg. now (helps alignment).
	call	MvData		;"Get" or "put" desired cache entry.
	pop	bp		;Reload all 16-bit regs. we used.
	pop	dx
	pop	cx
	pop	ax
	jc s	CEMErr		;If XMS error, post "flush" & exit.
	ret			;All is well -- exit.
CEMErr:	or	[bx+IOF-@].lb,1	;BAD News!  Post "flush cache" flag.
	stc			;Set carry again after "or" command.
	mov	al,XMSERR	;Post "XMS error" return code.
CEMovX:	ret			;Exit.
;
; Subroutine to "zip" all required search-table data into our search
;   buffer, then "zip" it BACK to the search table using a move-down
;   or move-up offset.   This is required for protected-mode as some
;   BIOS and EMM logic can reject an XMS move whose addresses appear
;   to "overlap".   At exit the carry flag is set for any XMS error.
;   If so, the AL-reg. has a 0FFh "XMS error" return code.
;
ZipZip:	push	ecx		;Save move length & destination addr.
	push	edi
	mov	edi,0		;Set search-buffer destination addr.
@SBufA	equ	[$-4].dwd
	call	MvData		;Send needed data to search buffer.
	pop	edi		;Reload move destination & length.
	pop	ecx
	jc s	CEMErr		;If XMS error, post "flush" & exit.
	mov	esi,0		;Set search-buffer source address.
@SBufB	equ	[$-4].dwd
	call	MvData		;Bring data BACK from search buffer.
	jc s	CEMErr		;If XMS error, post "flush" & exit.
	ret			;All is well -- exit.
;
; Subroutine to search for a cache-table index, and then delete it.
;   Our binary-search table has unit/LBA order, not LRU order!   To
;   delete an LRU index, we do a binary-search for it and then pass
;   the ending BP-reg. value to our "Delete" routine below.
;
ScanD:	call	CBGet		;Get target cache-table entry.
	jc s	ScanDE		;If XMS error, go exit below.
	mov	dx,0		;Set initial binary-search offset.
@MP2	equ	[$-2].lw	;(Search-table midpoint, Init set).
	mov	bp,dx		;Set initial binary-search index.
	dec	bp
	nop			;(Unused alignment "filler").
ScanD1:	shr	dx,1		;Divide binary-search offset by 2.
	cmp	bp,[bx+STLmt-@]	;Is our search pointer too high?
	jae s	ScanD3		;Yes, cut search pointer by offset.
	call	STGet		;Get next binary-search table index.
	jc s	ScanDE		;If any XMS error, go exit below.
	mov	ax,WBLBA	;Get next cache entry in work buffer.
	call	WBGet
	jc s	ScanDE		;If any XMS error, go exit below.
	mov	si,(CBLBA-@)	;Set up target v.s. work comparison.
	mov	di,(WBLBA-@)
	mov	cl,[si+6]	;Compare target unit v.s. work unit.
	cmp	cl,[di+6]
	jne s	ScanD2		;If units differ, check results now.
	mov	cx,3		;Compare target LBA v.s. work LBA.
	rep	cmpsw
	je s	Delete		;Target = entry:  BP has our offset.
ScanD2:	ja s	ScanD4		;Target > entry:  Adjust offset up.
	nop			;(Unused alignment "filler").
ScanD3:	sub	bp,dx		;Subtract offset from search ptr.
	jmp s	ScanD5		;Go see if we did our last compare.
ScanD4:	add	bp,dx		;Add offset to search pointer.
ScanD5:	or	dx,dx		;Offset zero, i.e. last compare?
	jnz s	ScanD1		;No, go compare next table entry.
	call	CEMErr		;Not found!  Handle as an XMS error!
ScanDE:	jmp s	UnLnkE		;Go discard stack data and exit.
	db	0		;(Unused alignment "filler").
;
; Subroutine to delete a cache index from our search table.
;
Delete:	mov	WBLBA,ax	     ;Save table index being deleted.
	movzx	ecx,[bx+STLmt-@].lw  ;Get move-down word count.
	sub	cx,bp
	dec	cx		;Any other indexes to move down?
	jz s	Delet2		;No, put our index in "free" list.
	shl	ecx,1		;Set move-down byte count.
	movzx	edi,bp		;Set 32-bit move-down addresses.
	shl	edi,1
	add	edi,dword ptr 0
@STblD	equ	[$-4].dwd
	lea	esi,[edi+2]
	smsw	dx		;Get CPU "machine status" word.
	shr	dx,1		;Are we now using "protected mode"?
	jc s	Delet1		;Yes, use "ZipZip" move logic below.
	call	MvData		;Move down needed search-table data.
	jmp s	Delet2		;Go put deleted index in "free" list.
	db	0		;(Unused alignment "filler").
Delet1:	call	ZipZip		;"Zip" data to search buffer & back.
	jc s	UnLnkE		;If any XMS error, go exit below.
Delet2:	dec	[bx+STLmt-@].lw	;Decrement search-table limit index.
	mov	bp,[bx+STLmt-@]	;Set up table-index "put" to XMS.
	inc	bx
	call	STPut		;Put deleted index in "free" list.
	jc s	UnLnkE		;If any XMS error, go exit below.
;
; Subroutine to unlink a cache-table entry from the LRU list.
;
UnLink:	mov	cx,[bx+CBLst-@]	;Get current entry's "last" index.
	mov	dx,[bx+CBNxt-@]	;Get current entry's "next" index.
	cmp	cx,-1		;Is this entry top-of-list?
	je s	UnLnk1		;Yes, "promote" next entry.
	mov	ax,cx		;Get "prior" entry into work buffer.
	call	WBGet
	jc s	UnLnkE		;If any XMS error, go exit below!
	mov	WBNxt,dx	;Update working entry's "next" index.
	inc	bx		;Put work-buffer entry back into XMS.
	call	WBPut
	jnc s	UnLnk2		;If no XMS error, check end-of-list.
UnLnkE:	pop	cx		;XMS error!  Discard exit address &
	pop	cx		;  AX/BP parameter saved upon entry.
	jmp	EndIO		;Go post XMS error code and get out!
UnLnk1:	mov	[bx+LUTop-@],dx	;Make next entry top-of-list.
UnLnk2:	cmp	dx,-1		;Is this entry end-of-list?
	jne s	UnLnk3		;No, link next to prior entry.
	mov	[bx+LUEnd-@],cx	;Make prior entry end-of-list.
	ret			;Exit.
UnLnk3:	mov	ax,dx		;Get "next" entry into work buffer.
	call	WBGet
	jc s	UnLnkE		;If any XMS error, go exit above!
	mov	WBLst,cx	;Update working entry's "last" index.
	inc	bx		;Put work-buffer entry back into XMS.
	call	WBPut
	jc s	UnLnkE		;If any XMS error, go exit above!
	ret			;All is well -- exit.
	dd	0,0		;(Unused alignment "filler").
HMALEN2	equ	($-CDReq)	;(Length of normal HMA cache logic).
;
; Initialization Cache-Sizes Table.    For the protected-mode cache,
;   each midpoint is multiplied by 2 and each cache-tables XMS value
;   is multiplied by 2/3, before being used by our init logic.
;
CachSiz	dw	(5*128)		; 5-MB "small" cache for all drivers.
	db	16		;   8K sectors per cache block
	db	4		;      and sector-shift count.
	dw	512		;   512 binary-search midpoint.
	dw	12		;   12K cache-tables XMS memory.
	db	"   5"		;   Title message cache-size text.
	dw	(15*64)		;15-MB "medium 1" cache.
	db	32,5		;   16K granularity values.
	dw	512		;   512 binary-search midpoint.
	dw	18		;   18K cache-tables XMS memory.
	db	"  15"		;   Title message cache-size text.
	dw	(25*64)		;25-MB "medium 2" cache.
	db	32,5		;   16K granularity values.
	dw	1024		;   1024 binary-search midpoint.
	dw	30		;   30K cache-tables XMS memory.
	db	"  25"		;   Title message cache-size text.
	dw	(40*32)		;40-MB "medium 3" cache.
	db	64,6		;   32K granularity values.
	dw	1024		;   1024 binary-search midpoint.
	dw	24		;   24K cache-tables XMS memory.
	db	"  40"		;   Title message cache-size text.
	dw	(50*32)		;50-MB "medium 4" cache.
	db	64,6		;   32K granularity values.
	dw	1024		;   1024 binary-search midpoint.
	dw	30		;   30K cache-tables XMS memory.
	db	"  50"		;   Title message cache-size text.
LCBlks	dw	(80*16)		;80-MB to 4093-MB "large" cache.
	db	128,7		;   64K granularity values.
LCVars	dw	1024		;   1024 to 32768 search midpoint.
	dw	36		;   36K to 1152K cache-tables XMS.
LCSize	db	"  80"		;   Title message cache-size text.
	dw	(80*16)		;80-MB default cache, used if loaded
	db	128,7		;   in upper/DOS memory and the user
	dw	1024		;   cache size does NOT fit into the
	dw	24		;   memory that follows this driver!
	db	"  80"
;
; CD/DVD I-O Address Table.   The 1st word for each set is its DMA
;   controller base address, which is set by Init.   A value of -1
;   denotes an unused controller.   A value of 0 in the DVD1 table
;   indicates an old system having no DMA controller, whose CD/DVD
;   drives must run in "PIO mode" only.    The second word is each
;   controller's default "legacy" base I-O address.    For "native
;   PCI mode", all base I-O addresses are set during Init.   There
;   is an EXTRA address set, so that on old systems, the first set
;   of "legacy" addresses can be used even with no DMA controller.
;   Note that this table is used only during init.    At run-time,
;   a CD/DVD drive's addresses are taken from its parameter table.
;
DVD1Tbl	dw	0		;IDE1 primary-master parameters.
DVD1Pri	dw	NPDATA
	dw	0		;IDE1 secondary-master parameters.
DVD1Sec	dw	NSDATA
DVD2Tbl	dw	0FFFFh		;IDE2 primary-master parameters.
DVD2Pri	dw	APDATA
	dw	0FFFFh		;IDE2 secondary-master parameters.
DVD2Sec	dw	ASDATA
	dd	18 dup (0FFFFh)	;IDE3 to IDE10 & "extra" parameters.
DVDTEnd	equ	$		;End of CD/DVD addresses table.
DVDTSIZ	equ	DVD2Tbl-DVD1Tbl	;Size of a CD/DVD address table.
;
; Initialization UltraDMA "Mode" Table (digit count in low 4 bits).
;
Modes	dw	01602h		;Mode 0, ATA-16.
	dw	02502h		;Mode 1, ATA-25.
	dw	03302h		;Mode 2, ATA-33.
	dw	04402h		;Mode 3, ATA-44.
	dw	06602h		;Mode 4, ATA-66.
	dw	01003h		;Mode 5, ATA-100.
	dw	01333h		;Mode 6, ATA-133.
	dw	01663h		;Mode 7, ATA-166.
;
; Initialization Messages.
;
EDDBuff equ	$
TTLMsg	db	CR,LF,'UIDE, 11-16-2009.   '
TTL2	db	'    -MB Cache, $'
TTL3	db	'CD/DVD name is '
TTLName	db	'         ',CR,LF,'$'
NPMsg	db	'No V2.0C+ PCI, BIOS I-O only!',CR,LF,'$'
NIMsg	db	'/N3 illegal$'
NXMsg	db	'No XMS manager$'
DNMsg	db	' d'
CDName	db	'isk is '
DName	equ	$
DNEnd	equ	DName+40
BCMsg	db	'BAD$'
PCMsg	db	'IDE0'
PCMsg1	db	' Controller at I-O address '
PCMsg2	db	'    h, Chip I.D. '
PCMsg3	db	'        h.',CR,LF,'$'
NonUMsg	db	'Disks run by the BIOS:  '
NonUCt	db	'0.',CR,LF,'$'
EDDMsg	db	'EDD BIOS$'
CHSMsg	db	'CHS'
UnitMsg	db	' data ERROR, D'
UnitNam	db	'rive '
UnitNo	db	'A: ',CR,LF,'$'
CDMsg	db	'CD'
CDMsgNo	db	'0:  '
CtlMsg	db	'IDE'
CtlrNo	db	'0 $'
PriMsg	db	'Primary-$'
SecMsg	db	'Secondary-$'
MstMsg	db	'master$'
SlvMsg	db	'slave$'
IEMsg	db	' Identify ERROR!',CR,LF,'$'
UEMsg	db	' is not UltraDMA!',CR,LF,'$'
NDMsg	db	'Nothing to use$'
VEMsg	db	'VDS init error$'
CPUMsg	db	'No 80386+ CPU'
Suffix	db	'; UIDE not loaded!',CR,LF,'$'
;
; Driver "Final Initialization" Routine.    The driver stack must be
;   cleared here, since the stack address for protected-mode drivers
;   "varies" when those drivers are loaded in upper or DOS memory!
;
I_ClrS:	dec	bx		;Clear driver stack (helps debug).
	dec	bx
	and	[bx].lw,0
	dec	ax
	jnz s	I_ClrS
	cmp	ax,@LastU	;Any BIOS disks/diskettes to use?
	je s	I_End		;No, reload all registers and exit.
	mov	ax,02513h	;"Hook" this driver into Int 13h.
	mov	dx,(Entry-@)
	int	021h
I_End:	pop	dx		;Reload 16-bit registers we used.
	pop	bx
	pop	ax
	pop	es		;Reload CPU segment registers.
	pop	ds
	popf			;Reload CPU flags and exit.
	retf
;
; Protected-Mode Caching Routines.   When the /P switch is given, our
; driver initialization shall "overlay" the XMS Caching Routines with
; these routines, starting here and extending down to "I_Init" below.
;
; Main Protected-Mode Caching Routine.
;
	align	16		     ;(Helps find "bugs" easier!).
PCache:	cmp	[bx+RqTyp-@].lb,0C0h ;Is this a diskette request?
	jne s	PFlush		     ;No, see if cache needs a flush.
	mov	es,bx		     ;Point ES-reg. to low memory.
	mov	cl,01Fh		     ;Get BIOS diskette status code.
	and	cl,es:[DKTSTAT]
	cmp	cl,MCHANGE	;Diskette media-change detected?
	je s	PFlshA		;Yes, go flush our cache now.
PFlush:	test	[bx+IOF-@].lb,1	;Does cache need to be flushed?
	jz s	PNxtIO		;No, proceed with this request.
PFlshA:	mov	ax,0FFFFh	    ;Get binary-search table data.
@TC2	equ	[$-2].lw	    ;(Table count - 1, Init set).
	mov	si,offset (SrchT-@)
@TE2	equ	[$-2].lw	    ;(Table upper-limit, Init set).
PFlshT:	dec	si		    ;Flush our binary-search table.
	dec	si
	mov	cs:[si],ax
	dec	ax
	jns s	PFlshT
	mov	[bx+STLmt-@],si	    ;Reset "free" cache-index ptr.
	or	[bx+LUTop-@].dwd,-1 ;Reset LRU start & end indexes.
	and	[bx+IOF-@].lb,0FEh  ;O.K. to reset flush flag now.
PNxtIO:	push	ds		    ;Set ES-reg. same as DS-reg.
	pop	es
	mov	al,0		;Get "granularity" (sectors/block).
@GRAN4	equ	[$-1].lb	;(Block "granularity", set by Init).
	cmp	al,[bx+RqSec-@]	;Will we need multiple cache blocks?
	jbe s	PSetSC		;Yes, use "full block" sector count.
	mov	al,[bx+RqSec-@]	;Use remaining request sector count.
PSetSC:	mov	[bx+RqCSC-@],al	;Set maximum I-O sector count.
	mov	[bx+RqXMS-@],bx ;Reset lower XMS buffer offset.
	mov	dx,0		;Set initial binary-search offset.
@MP3	equ	[$-2].lw	;(Search-table midpoint, Init set).
	mov	bp,offset (SrchT-2-@) ;Set table midpoint address.
@MP4	equ	[$-2].lw	      ;(Midpoint address, Init set).
PSrch:	shr	dx,1		;Divide binary-search offset by 2.
	and	dl,0FEh		;Ensure offset remains "even".
	cmp	bp,[bx+STLmt-@]	;Is our search pointer too high?
	jae s	PSrchL		;Yes, cut search pointer by offset.
	mov	ax,cs:[bp]	;Get next binary-search table index.
	call	PCBGet		;Get next cache entry into buffer.
	jc	PEndIO		;If any XMS error, exit immediately!
	mov	si,(RqLBA-@)	;Compare initial request & table LBA.
	call	PComp
	jae s	PChkNd		;Request >= table:  Check for found.
PSrchL:	sub	bp,dx		;Subtract offset from search ptr.
	or	dx,dx		;Offset zero, i.e. last compare?
	jnz s	PSrch		;No, go compare next table entry.
	jmp s	PNFind		;Handle this request as "not found".
PChkNd:	je s	PFound		;Request = table:  Treat as "found".
	mov	cl,[bx+CBSec-@]	;Calculate and set ending entry LBA.
	mov	si,(CBLBA-@)
	call	PCalcA
	mov	si,(RqLBA-@)	;Compare request start & entry end.
	call	PComp1
	jb s	PFound		   ;Request < Entry:  Handle as found.
	ja s	PSrchH		   ;Request > Entry:  Bump search ptr.
	cmp	[bx+CBSec-@].lb,0  ;Is this cache entry "full"?
@GRAN5	equ	[$-1].lb	   ;(Block "granularity", Init set).
	jb s	PFound		;No, handle this request as "found".
PSrchH:	add	bp,dx		;Add offset to search pointer.
	or	dx,dx		;Offset zero, i.e. last compare?
	jnz s	PSrch		;No, go compare next table entry.
	inc	bp		;Bump pointer to next table entry.
	inc	bp
PNFind:	push	cs		;Unfound:  Set search table segment.
	pop	es
	mov	di,[bx+STLmt-@]	;Get next "free" cache-entry index.
	mov	ax,es:[di]
	std			;Enable a "backward" move, and set
	lea	si,[di-2]	;  move-up source address & count.
	mov	cx,di		;  We do the "std" early, for more
	sub	cx,bp		;  time margin v.s. AMD CPU "bugs"!
	shr	cx,1
	rep	movs es:[di].lw,cs:[si].lw  ;Move up higher indexes.
	stosw				    ;Store new search index.
	cld			   ;Re-enable FORWARD "string" cmds.
	push	ds		   ;Reload this driver's ES-reg.
	pop	es
	add	[bx+STLmt-@].lw,2  ;Bump binary-search table limit.
	mov	si,(RqLBA-@)	   ;Set 48-bit LBA in new entry.
	mov	di,(CBLBA-@)
	movsd
	movsw
	movsb			   ;Set "cache unit" in new entry.
	mov	[di],bl		   ;Reset new entry's sector count.
PFound:	mov	RqIndex,ax	   ;Post current cache-entry index.
	mov	cx,[bx+RqLBA+4-@]  ;Get starting I-O offset in block.
	sub	cx,[bx+CBLBA+4-@]
	mov	[bx+RqXMS-@],cl	   ;Set starting XMS sector offset.
	mov	[bx+LBA2+1-@],cx   ;Save starting I-O sector offset.
	cmp	[bx+CBSec-@],bl	   ;Is this a new cache-table entry?
	je s	PRLink		   ;Yes, relink entry as top-of-list.
	push	ax		   ;Unlink this entry from LRU list.
	call	PULink
	pop	ax
PRLink:	mov	[bx+RqXMS+2-@],bx  ;Reset upper XMS buffer offset.
	movzx	edx,ax		   ;Get 32-bit cache block number.
	shl	edx,2		   ;Shift number to starting sector.
@GRSSC2	equ	[$-1].lb	   ;("Granularity" shift, Init set).
	add	[bx+RqXMS-@],edx   ;Add to "preset" sector offset.
	shl	[bx+RqXMS-@].dwd,9	  ;Convert sectors to bytes.
	add	[bx+RqXMS-@].dwd,020000h  ;Add in XMS "base" address.
@XBase2	equ	[$-4].dwd		  ;(XMS "base", set by Init).
	mov	cx,0FFFFh	;Make this entry "top of list".
	or	[bx+CBLst-@],cx	;Set this entry's "last" index.
	mov	dx,ax		;Swap top-of-list & entry index.
	xchg	dx,[bx+LUTop-@]
	mov	[bx+CBNxt-@],dx ;Set this entry's "next" index.
	cmp	dx,cx		;Is this the only LRU index?
	mov	cx,ax		;(Get this entry's index in CX-reg.).
	je s	PRLnk1		;Yes, make entry last on LRU list.
	push	ax		;Link entry to prior "top of list".
	call	PULnk3
	pop	ax
	jmp s	PRLnk2		;Go deal with I-O sector count.
PUpdat:	mov	[bx+CBSec-@],cl	;Update this entry's total sectors.
	call	PCBPut		;Update this cache-table entry.
	jc s	PNdIOJ		;If any XMS error, exit immediately!
	movzx	cx,[bx+RqCSC-@]	;Calculate ending LBA for this I-O.
	mov	si,(RqLBA-@)
	call	PCalcA
	inc	bp		;Skip to next search index.
	inc	bp
POvrlp:	cmp	bp,[bx+STLmt-@]	;More cache-table entries to check?
	jae s	PCchIO		;No, O.K. to handle "found" entry.
        mov     ax,cs:[bp]      ;Set next search table index.
        call    PCBGet          ;Get next cache entry into buffer.
	jc s	PNdIOJ		;If any XMS error, exit immediately!
	mov	si,(LBABuf-@)	;Compare request end & entry start.
	call	PComp
	jbe s	PCchIO		;Request <= entry:  O.K. to proceed.
	push	bp		;Delete this overlapping table entry.
	call	PDelet
	pop	bp
	jmp s	POvrlp		  ;Go check for more entry overlap.
PRLnk1:	mov	[bx+LUEnd-@],ax	  ;Make entry last on LRU list, too!
PRLnk2:	mov	cx,[bx+LBA2+1-@]  ;Reload initial I-O sector offset.
	mov	ch,0		  ;Get entry's available sectors.
@GRAN6	equ	[$-1].lb	  ;(Block "granularity", Init set).
	sub	ch,cl
	cmp	ch,[bx+RqCSC-@]	;More I-O sectors than available?
	jae s	PLrger		;No, retain maximum sector count.
	mov	[bx+RqCSC-@],ch	;Reduce current I-O sector count.
PLrger:	add	cl,[bx+RqCSC-@]	;Get ending I-O sector number.
	cmp	cl,[bx+CBSec-@]	;More sectors than entry has now?
	ja s	PUpdat		;Yes, update entry sectors.
	inc	bx		;Reset Z-flag for "put" into XMS.
	call	PCBPut		;Update this cache-table entry.
PNdIOJ:	jc s	PEndIO		      ;If any XMS error, exit fast!
	test	[bx+RqCmd-@].lb,002h  ;Is this a read request?
	jz s	PBfMov		      ;Yes, move cache data to user.
PCchIO:	bts	[bx+RqCmd-@].lb,6     ;I-O done during a prior block?
	jc s	PBfMor		      ;Yes, buffer more cache data.
	cmp	[bx+RqTyp-@].lb,CDTYP ;Is this one of our disks/CDs?
	ja s	POther		      ;No, go check for other units.
	je s	PCDVD		;CD/DVD -- go call "RLRead" below.
	call	UdmaIO+PCOFFS	;Call "UdmaIO" for SATA/IDE disks.
	jmp s	PErChk		;Go check for any I-O errors.
PCDVD:	call	RLRead+PCOFFS	;Call "RLRead" for CD/DVD input.
	jmp s	PErChk		;Go check for any I-O errors.
POther:	or	[bx+RqTyp-@],bl	;Is this unit handled by the BIOS?
	js s	PBios		;Yes, go "Call the BIOS" below.
	call	DvrPtr		;"External" unit:  Call its driver.
	jmp s	PErChk		;Go check for any I-O errors.
PBios:	lds	si,CStack	;BIOS I-O:  Reload caller CPU flags.
	push	[si+2].lw
	popf
	pop	es		;Reload all CPU registers.
	pop	ds
	popad
	pushf			;"Call the BIOS" to do this request.
	db	09Ah
@Prv13C	dd	0		;(Prior Int 13h vector, set by Init).
	pushad			;Save all CPU registers again.
	push	ds
	push	es
	mov	al,ah		;Move any BIOS error code to AL-reg.
PErChk:	sti			;Restore critical driver settings.
	cld
	push	ss		;Reload this driver's DS- & ES-regs.
	pop	ds
	push	ds
	pop	es
	mov	bx,0		;Rezero BX-reg. but save carry.
	jnc s	PInCch		;If no error, go see about caching.
PIOErr:	push	ax		;Error!  Save returned error code.
	mov	ax,RqIndex	;Delete cache entry for this I-O.
	call	PScanD
	pop	ax		;Reload returned error code.
	stc			;Set carry again to denote error.
PEndIO:	db	0EAh		;Return to hard-disk exit routine.
	dw	(HDExit-@)
@HXSeg3	dw	0		;(Driver segment address, Init set).
PInCch:	or	[bx+RqCmd-@],bl	;Did user data get cached during I-O?
	js s	PCkFul		;Yes, check if cache tables are full.
PBfMor:	or	al,1		     ;Ensure we LOAD data into cache!
PBfMov:	movzx	ecx,[bx+RqCSC-@].lb  ;Set XMS move sector count.
	mov	esi,[bx+RqXMS-@]     ;Set desired XMS buffer address.
	mov	edi,[bx+RqBuf-@]     ;Set user buffer as destination.
	call	UBufM1+PCOFFS	     ;Move data between user and XMS.
	jc s	PIOErr			   ;If error, use rtn. above.
PCkFul:	cmp	[bx+STLmt-@].lw,(SrchT-@)  ;Binary-search table full?
@TE3	equ	[$-2].lw		   ;(Table "end", Init set).
	jb s	PMorIO		   ;No, check for more sectors to go.
	push	ax		   ;Delete least-recently-used entry.
	mov	ax,[bx+LUEnd-@]
	call	PScanD
	pop	ax
PMorIO:	xor	ax,ax		   ;Reset error code for exit above.
	movzx	cx,[bx+RqCSC-@]	   ;Get current I-O sector count.
	sub	[bx+RqSec-@],cl	   ;More data sectors left to handle?
	jz s	PEndIO		   ;No, return to our "exit" routine.
	add	[bx+RqLBA+4-@],cx  ;Update current LBA for next I-O.
	adc	[bx+RqLBA+2-@],bx
	adc	[bx+RqLBA-@],bx
	shl	cx,1		   ;Convert sector count to bytes.
	add	[bx+RqBuf+1-@],cl  ;Update user I-O buffer address.
	adc	[bx+RqBuf+2-@],bx
	jmp	PNxtIO		   ;Go handle next cache data block.
;
; Subroutine to calculate ending request or cache-table LBA values.
;
PCalcA:	mov	di,(LBABuf-@)	;Point to address-comparison buffer.
	push	[si].dwd	;Move high-order LBA into buffer.
	pop	[di].dwd
	add	cx,[si+4]	;Calculate ending LBA.
	mov	[di+4],cx
	adc	[di+2],bx
	adc	[di],bx
PCalcX:	ret			;Exit.
;
; Subroutine to do 7-byte "cache unit" and LBA comparisons.
;
PComp:	mov	di,(CBLBA-@)	;Point to current-buffer LBA value.
PComp1:	mov	cl,[bx+RqUNo-@]	;Compare our "unit" & table "unit".
	cmp	cl,[bx+CBUNo-@]
	jne s	PCalcX		;If units are different, exit now.
	mov	cx,3		;Compare our LBA & cache-table LBA.
	rep	cmpsw
	ret			;Exit -- Main routine checks results.
;
; Subroutines to do search-table index and cache-entry XMS moves.
;
PWBGet:	xor	di,di		;Work buffer:  Set Z-flag for "get".
PWBPut:	mov	edi,(WBLBA-@)	;Set 32-bit "working" buffer address.
@WBAdr3	equ	[$-4].dwd	;(Working-buffer address, Init set).
	jmp s	PCEMov		;Go move data to/from working buffer.
PCBGet:	xor	di,di		;Entry buffer:  Set Z-flag for "get".
PCBPut:	mov	edi,(CBLBA-@)	;Set 32-bit "current" buffer address.
@CBAdr2	equ	[$-4].dwd	;(Current-buffer address, Init set).
PCEMov:	push	ax		;Save all needed 16-bit regs.
	push	cx
	push	dx
	pushf			;Save CPU Z-flag from above.
	xor	ecx,ecx		;Set cache-table entry size.
	mov	cl,12
	mul	cx		;Multiply cache index by entry size.
	push	dx		;Get cache-entry offset in ESI-reg.
	push	ax
	pop	esi
	add	esi,010000h	;Add cache-table "base" address.
@CTBas2	equ	[$-4].dwd	;(Cache-table "base", Init set).
	popf			;Reload CPU flags.
	jz s	PCEMv1		;Are we "getting" from XMS memory?
	xchg	esi,edi		;No, "swap" source and destination.
PCEMv1:	push	bp		;Save BP-reg. now (helps alignment).
	call	MvData+PCOFFS	;"Get" or "put" desired cache entry.
	pop	bp		;Reload all 16-bit regs. we used.
	pop	dx
	pop	cx
	pop	ax
	jnc s	PCEMvX		;If no BIOS/JEMM386 error, go exit.
PCEMvE:	or	[bx+IOF-@].lb,1	;BAD News!  Post "flush cache" flag.
	stc			;Set carry again after "or" command.
	mov	al,XMSERR	;Post "XMS error" return code.
PCEMvX:	ret			;Exit.
;
; Subroutine to scan for and delete a search-table index.
;
PScanD:	mov	cx,0		;Set up a 16-bit "scan" command.
@TC3	equ	[$-2].lw	;(Search-table count, set by Init).
	mov	di,offset (SrchT-@)
@TA1	equ	[$-2].lw	;(Search-table start, set by Init).
	push	cs
	pop	es
	repne	scasw		;"Scan" for index in search table.
	lea	bp,[di-2]	;Index is at [di-2], due to "scasw".
	call	PCBGet		;Get cache entry from XMS to buffer.
	jc s	PULnkE		;If any XMS error, go exit below.
;
; Subroutine to delete a cache index from our search table.
;
PDelet:	mov	di,bp		;Point DI-reg. to index in table.
	lea	si,[di+2]	;Set move-down source and count.
	mov	cx,[bx+STLmt-@]
	sub	cx,si
	shr	cx,1
	push	cs		;Set binary-search table segment and
	pop	es		;  move down remaining table indexes.
	rep	movs es:[di].lw,cs:[si].lw
	stosw			;Put deleted index in free-list.
	push	ds		;Reload this driver's ES-reg.
	pop	es
	sub	[bx+STLmt-@].lw,2  ;Decrement table limit.
;
; Subroutine to unlink a cache-table entry from the LRU list.
;
PULink:	mov	cx,[bx+CBLst-@]	;Get current entry's "last" index.
	mov	dx,[bx+CBNxt-@]	;Get current entry's "next" index.
	cmp	cx,-1		;Is this entry top-of-list?
	je s	PULnk1		;Yes, "promote" next entry.
	mov	ax,cx		;Get "prior" entry into work buffer.
	call	PWBGet
	jc s	PULnkE		;If any XMS error, go exit below!
	mov	WBNxt,dx	;Update working entry's "next" index.
	inc	bx		;Put work-buffer entry back into XMS.
	call	PWBPut
	jnc s	PULnk2		;If no XMS error, check end-of-list.
PULnkE:	pop	cx		;XMS error!  Discard exit address &
	pop	cx		;  AX/BP parameter saved upon entry.
	jmp	PEndIO		;Go post XMS error code and get out!
PULnk1:	mov	[bx+LUTop-@],dx	;Make next entry top-of-list.
PULnk2:	cmp	dx,-1		;Is this entry end-of-list?
	jne s	PULnk3		;No, link next to prior entry.
	mov	[bx+LUEnd-@],cx	;Make prior entry end-of-list.
	ret			;Exit.
PULnk3:	mov	ax,dx		;Get "next" entry into work buffer.
	call	PWBGet
	jc s	PULnkE		;If any XMS error, go exit above!
	mov	WBLst,cx	;Update working entry's "last" index.
	inc	bx		;Put work-buffer entry back into XMS.
	call	PWBPut
	jc s	PULnkE		;If any XMS error, go exit above!
	ret			;All is well -- exit.
	db	0		     ;(Unused alignment "filler").
PCLSIZE	equ	($-PCache)	     ;(Protected cache-logic size).
HMALEN3	equ	(HMALEN1+PCLSIZE)    ;(Length of HMA caching logic).
PCOFFS	equ	(PCache-Cache)	     ;(Protected-cache "offset").
LMT710	equ	(HMALEN3+(170*16*2)) ;(170-MB limit if V7.10 MS-DOS).
SrchT	equ	(Cache+PCLSIZE)	     ;(Resident search-table start).
ResEnd	equ	(SrchT+STACK)	     ;(Memory-resident driver size).
;
; Driver Initialization Routine.
;
I_Init:	pushf			;Entry -- save CPU flags.
	push	ds		;Save CPU segment registers.
	push	es
	push	cs		;Point DS-reg. to this driver.
	pop	ds
	push	ax		;Save needed 16-bit CPU registers.
	push	bx
	push	dx
	les	bx,RqPkt	 ;Point to DOS "Init" packet.
	cmp	es:[bx].RPOp,0	 ;Is this really an "Init" packet?
	jne s	I_Exit		 ;No??  Reload registers and exit!
	mov	ax,es:[bx].RPLen ;Save our "free memory" limit.
	mov	PktLBA.lw,ax
	mov	ax,es:[bx].RPSeg
	mov	PktLBA.hw,ax
	mov	es:[bx].RPStat,RPDON+RPERR ;Set init packet defaults.
	and	es:[bx].RPLen,0
	mov	es:[bx].RPSeg,cs
	push	sp		;See if CPU is an 80286 or newer.
	pop	ax		;(80286+ push SP, then decrement it).
	cmp	ax,sp		;Did SP-reg. get saved "decremented"?
	jne s	I_Junk		;Yes?  CPU is an 8086/80186, TOO OLD!
	pushf			;80386 test -- save CPU flags.
	push	07000h		;Try to set NT|IOPL status flags.
	popf
	pushf			;Get resulting CPU status flags.
	pop	ax
	popf			;Reload starting CPU flags.
	test	ah,070h		;Did any NT|IOPL bits get set?
	jnz s	I_Sv32		;Yes, go save 32-bit CPU registers.
I_Junk:	mov	dx,(CPUMsg-@)	;Point to "No 80386+ CPU" message.
I_Quit:	call	I_Msg		;Display "No 80386" or msg. suffix.
I_Exit:	pop	dx		;Reload 16-bit registers we used.
	pop	bx
	pop	ax
	pop	es		;Reload CPU segment registers.
	pop	ds
	popf			;Reload CPU flags and exit.
	retf
I_VErr:	mov	VEMsg.dwd,eax	;Set prefix in "VDS init error" msg.
I_VEr1:	mov	dx,(VEMsg-@)	;Point to "VDS init error" message.
I_Err:	push	dx		;Init ERROR!  Save message pointer.
	shr	VDSOf.lb,1	;Was driver "locked" by VDS?
	jnc s	I_XDis		;No, see if we reserved XMS memory.
	mov	ax,08104h	;"Unlock" this driver from memory.
	xor	dx,dx
	call	I_VDS
I_XDis:	mov	dx,XMHdl	;Load our XMS "handle" number.
	or	dx,dx		;Did we reserve any XMS memory?
	jz s	I_LDMP		;No, reload pointer & display msg.
	mov	ah,00Dh		;Unlock and "free" our XMS memory.
	push	dx
	call	I_XMS
	mov	ah,00Ah
	pop	dx
	call	I_XMS
I_LDMP:	pop	dx		;Reload error message pointer.
	call	I_Msg		;Display desired error message.
	popad			;Reload all 32-bit CPU registers.
	mov	dx,(Suffix-@)	;Display message suffix and exit!
	jmp s	I_Quit
I_Sv32:	pushad			;Save all 32-bit CPU registers.
	les	si,es:[bx].RPCL	;Get command-line data pointer.
	xor	bx,bx		;Zero BX-reg. for relative logic.
I_NxtC:	mov	al,es:[si]	;Get next command-line byte.
	inc	si		;Bump pointer past first byte.
	cmp	al,'/'		;Is byte a slash?
	je s	I_NxtS		;Yes, see what next "switch" byte is.
	cmp	al,0		;Is byte the command-line terminator?
	je s	I_TrmJ		;Yes, go check for valid device name.
	cmp	al,LF		;Is byte an ASCII line-feed?
	je s	I_TrmJ		;Yes, go check for valid device name.
	cmp	al,CR		;Is byte an ASCII carriage-return?
I_TrmJ:	je	I_Term		;Yes, go check for valid device name.
	cmp	al,'-'		;Is byte a dash?
	jne s	I_NxtC		;No, check next command-line byte.
I_NxtS:	mov	ax,es:[si]	;Get next 2 command-line bytes.
	and	al,0DFh		;Mask out 1st byte's lower-case bit.
	cmp	ax,"3N"		;Are these 2 switch bytes "N3"?
	jne s	I_ChkA		;No, see if 1st byte is "A" or "a".
	mov	XMF,al		;Set "No XMS memory" flag.
	mov	HFlag,bl	;Ensure NO use of HMA space!
	inc	si		;Skip 1st command-line byte.
	jmp s	I_CkBA		;Go set up our stand-alone driver.
I_ChkA:	cmp	al,'A'		;Is switch byte an "A" or "a"?
	jne s	I_ChkB		   ;No, see if byte is "B" or "b".
	mov	al,(ASDATA-0100h)  ;Reverse all "Legacy IDE" addrs.
	mov	Ctl1Sec.lb,al
	mov	DVD1Sec.lb,al
	mov	al,(APDATA-0100h)
	mov	Ctl1Pri.lb,al
	mov	DVD1Pri.lb,al
	mov	al,(NSDATA-0100h)
	mov	Ctl2Sec.lb,al
	mov	DVD2Sec.lb,al
	mov	al,(NPDATA-0100h)
	mov	Ctl2Pri.lb,al
	mov	DVD2Pri.lb,al
	inc	si		;Point to next command-line byte.
I_ChkB:	cmp	al,'B'		;Is switch byte a "B" or "b"?
	jne s	I_ChkE		;No, see if byte is "E" or "e".
I_CkBA:	mov	HMASize,HMALEN1	;Set stand-alone driver HMA size.
	db	066h,068h	;Enable stand-alone disk driver.
	db	0E8h
	dw	(UdmaIO-GoMain-3)
	db	0EAh
	pop	[GoMain].dwd
	mov	@NoCA.lb,0EBh   ;Disable all CD/DVD caching.
	db	068h		;Disable I-O through cache buffers.
	stc
	ret
	pop	UseCB.lw
	mov	EFlag,bl	;Reset "External Entry" flag.
	mov	PFlag,bl	;Reset protected-caching flag.
	mov	TTL2.lb,'$'	;Disable cache size in "title".
	inc	si		;Point to next command-line byte.
I_ChkE:	cmp	al,'E'		;Is switch byte an "E" or "e"?
	jne s	I_ChkH		;No, see if byte is "H" or "h".
	cmp	GoMain.lb,0E8h	;"Stand alone" driver requested?
	je s	I_ChkH		;Yes, must ignore the /E switch!
	mov	EFlag,al	;Retain "External Entry" logic.
	inc	si		;Point to next command-line byte.
I_ChkH:	cmp	al,'H'		;Is switch byte an "H" or "h"?
	jne s	I_CkN1		;No, see if next 2 bytes are "N1".
	cmp	XMF,bl		;"No XMS memory" also requested?
	jne s	I_CkN1		;Yes, must ignore the /H switch!
	mov	HFlag,0FFh	;Set "use HMA space" flag.
	inc	si		;Point to next command-line byte.
I_CkN1:	cmp	ax,"1N"		;Are these 2 switch bytes "N1"?
	jne s	I_CkN2		;No, see if switch bytes are "N2".
	mov	BiosHD,bl	;Disable all hard-disk handling.
	inc	si		;Skip these 2 command-line bytes.
	inc	si
I_CkN2:	cmp	ax,"2N"		;Are these 2 switch bytes "N2"?
	jne s	I_ChkP		;No, see if byte is "P" or "p".
	mov	DVDTblB,ax	;Disable all CD/DVD drive handling.
	inc	si		;Skip these 2 command-line bytes.
	inc	si
I_ChkP:	cmp	al,'P'		;Is switch byte a "P" or "p"?
	jne s	I_ChkQ		;No, see if byte is "Q" or "q".
	cmp	GoMain.lb,0E8h	;"Stand alone" driver requested?
	je s	I_ChkQ		;Yes, must ignore the /P switch!
	mov	PFlag,0FFh	;Set protected-mode caching flag.
	mov	HMASize,HMALEN3	;Set protected-mode driver HMA size.
	inc	si		;Point to next command-line byte.
I_ChkQ:	cmp	al,'Q'		;Is switch byte a "Q" or "q"?
	jne s	I_ChkS		;No, see if byte is "S" or "s".
	mov	@DRQ.lb,075h	;Enable "data request" tests above.
	inc	si		;Point to next command-line byte.
I_ChkS:	cmp	al,'S'		;Is switch byte an "S" or "s"?
	jne s	I_ChkU		;No, see if byte is "U" or "u".
	call	GetSiz		;Get desired user cache size.
	jmp s	I_NxtJ		;Continue scanning for a terminator.
I_ChkU:	cmp	al,'U'		;Is switch byte a "U" or "u"?
	jne s	I_ChkD		;No, see if 2 bytes are "D:" or "d:".
	inc	si		;Bump pointer past "U" or "u".
	mov	al,ah		;Get 2nd command-line byte.
	and	al,0DFh		;Mask out 1st lower-case bit (020h).
	cmp	al,'X'		;Is 2nd byte an "X" or "x"?
	jne s	I_NxtJ		;No, continue scan for a terminator.
	mov	NoDMA,al	;Set CD/DVD "No UltraDMA" flag.
	inc	si		;Bump pointer past "X" or "x".
I_NxtJ:	jmp	I_NxtC		;Continue scanning for a terminator.
I_ChkD:	cmp	ax,":D"		;Are next 2 bytes "D:" or "d:"?
	jne s	I_NxtJ		;No, continue scan for a terminator.
	inc	si		;Bump pointer past "device" switch.
	inc	si
	mov	cx,8		;Set driver-name byte count of 8.
	mov	di,(DvrNam-@)	;Blank out previous driver name.
	mov	bx,cx
I_Nam0:	dec	bx
	mov	[bx+DvrNam-@].lb,' '
	jnz s	I_Nam0
I_Nam1:	mov	al,es:[si]	;Get next device-name byte.
	cmp	al,TAB		;Is byte a "tab"?
	je s	I_NxtJ		;Yes, handle above, "name" has ended!
	cmp	al,' '		;Is byte a space?
	je s	I_NxtJ		;Yes, handle above, "name" has ended!
	cmp	al,'/'		;Is byte a slash?
	je s	I_NxtJ		;Yes, handle above, "name" has ended!
	cmp	al,0		;Is byte the command-line terminator?
	je s	I_Term		;Yes, go check for valid device name.
	cmp	al,LF		;Is byte an ASCII line-feed?
	je s	I_Term		;Yes, go check for valid device name.
	cmp	al,CR		;Is byte an ASCII carriage-return?
	je s	I_Term		;Yes, go check for valid device name.
	cmp	al,'a'		;Ensure letters are upper-case.
	jb s	I_Nam2
	cmp	al,'z'
	ja s	I_Nam2
	and	al,0DFh
I_Nam2:	cmp	al,'!'		;Is this byte an exclamation point?
	je s	I_Nam3		;Yes, store it in device name.
	cmp	al,'#'		;Is byte below a pound-sign?
	jb s	I_Nam4		;Yes, Invalid!  Blank first byte.
	cmp	al,')'		;Is byte a right-parenthesis or less?
	jbe s	I_Nam3		;Yes, store it in device name.
	cmp	al,'-'		;Is byte a dash?
	je s	I_Nam3		;Yes, store it in device name.
	cmp	al,'0'		;Is byte below a zero?
	jb s	I_Nam4		;Yes, invalid!  Blank first byte.
	cmp	al,'9'		;Is byte a nine or less?
	jbe s	I_Nam3		;Yes, store it in device name.
	cmp	al,'@'		;Is byte below an "at sign"?
	jb s	I_Nam4		;Yes, invalid!  Blank first byte.
	cmp	al,'Z'		;Is byte a "Z" or less?
	jbe s	I_Nam3		;Yes, store it in device name.
	cmp	al,'^'		;Is byte below a carat?
	jb s	I_Nam4		;Yes, invalid!  Blank first byte.
	cmp	al,'~'		;Is byte above a tilde?
	ja s	I_Nam4		;Yes, invalid!  Blank first byte.
	cmp	al,'|'		;Is byte an "or" symbol?
	je s	I_Nam4		;Yes, invalid!  Blank first byte.
I_Nam3:	mov	[di],al		;Store next byte in device name.
	inc	si		;Bump command-line pointer.
	inc	di		;Bump device-name pointer.
	loop	I_Nam1		;If more name bytes to go, loop back.
	jmp s	I_NxtJ		      ;Go get next command byte.
I_Nam4:	mov	[bx+DvrNam-@].lb,' '  ;Invalid!  Blank first byte.
	jmp s	I_NxtJ		      ;Go get next command byte.
I_Term:	mov	ah,'.'		   ;Set a period for "stores" below.
	mov	cx,8		   ;Set driver-name byte count of 8.
	mov	di,(DvrNam-@)	   ;Point to our driver name.
	cmp	[di].lb,' '	   ;Is driver "name" valid?
	jne s	I_SetN		   ;Yes, set name in "title" message.
	mov	[di+4].dwd,"   1"  ;Set our default driver "name".
	mov	[di].dwd,"DVDU"
I_SetN:	mov	al,[di]		;Get next driver "name" byte.
	cmp	al,' '		;End of driver "name"?
	je s	I_PMC		;Yes, check on protected-mode caching.
	mov	[di+(TTLName-DvrNam)],ax  ;Store title byte & period.
	inc	di		;Bump driver "name" pointer.
	loop	I_SetN		;If more name bytes to go, loop back.
I_PMC:	sar	PFlag,1		;Is protected-mode caching desired?
	jnz s	I_PMCA		;Yes, use routines below.
	sar	HFlag,1		;Restricted to upper/DOS memory?
	jz s	I_NoHA		;Yes, go set upper/DOS memory size.
	mov	ax,04A01h	;Get total "free HMA" space.
	call	I_In2F
	cmp	bx,HMASize	;Enough HMA for our driver logic?
	jae s	I_Siz0		;Yes, check for stand-alone driver.
I_NoHA:	mov	HFlag,0		;Ensure NO use of HMA space!
	mov	ax,HMASize	;Set driver upper/DOS memory size.
	add	VDSLn.lw,ax
I_Siz0:	cmp	GoMain.lb,0E8h	;Is this the "stand alone" driver?
	je	I_ExtE		;Yes, check on external-entry logic.
	mov	al,12		;Point to desired cache-size table.
	mul	SFlag
	add	ax,(CachSiz-@)
	xchg	ax,si
	movzx	eax,[si].lw	;Set binary-search limit index.
	mov	@TE1,ax
	shl	eax,1		;Set binary-search table length.
	mov	@TC1,eax
	mov	ax,[si+4]	;Set binary-search starting offsets.
	mov	@MP1,ax
	mov	@MP2,ax
	movzx	ecx,[si+6].lw	  ;Preset cache-tables XMS size.
	jmp	I_SetG		  ;Go set cache "granularity" below.
I_PMCA:	cmp	LCBlks,(1900*16)  ;Protected:  User's cache too big?
	jbe s	I_PMCB		  ;Yes, see if we can use HMA space.
	mov	LCBlks,(1900*16)  ;Cut "large" cache size to 1900-MB.
	mov	LCVars.dwd,(576*65536)+16384
	mov	LCSize.dwd,"0091"
I_PMCB:	sar	HFlag,1		  ;Restricted to upper/DOS memory?
	jz s	I_NoHM		  ;Yes, go set upper/DOS memory size.
	mov	ax,04A01h	  ;Get total "free HMA" space.
	call	I_In2F
	cmp	bx,LMT710	;Less HMA than V7.10 MS-DOS limit?
	jb s	I_HSiz		;Yes, use only whatever HMA we have.
	cmp	bx,(14*1024)	;At least 14K of "free HMA"?
	jae s	I_HSiz		;Yes, usually O.K. to use all of it.
	mov	bx,LMT710	;May be V7.10 MS-DOS!  Set HMA limit.
I_HSiz:	sub	bx,HMASize	;Enough HMA for all our driver logic?
	jb s	I_NoHM		;No, we must load in normal memory!
	cmp	GoMain.lb,0E8h	;Is this the "stand alone" driver?
	je	I_ExtE		;Yes, see about external-entry logic.
	mov	dx,bx		;Set free HMA in DX-reg. for below.
	cmp	dx,(5*128*2)	;Can our 5-MB cache fit in the HMA?
	jb s	I_NoHM		;No, we must load in normal memory!
	movzx	cx,SFlag	;Does user want only a 5-MB cache?
	jcxz	I_HSz2		;Yes, O.K. to set it up now.
	cmp	dx,(15*64*2)	;Can our 15-MB cache fit in the HMA?
	jb s	I_NoHM		;No, nicer to load in normal memory!
	cmp	cl,1		;Does user want only a 15-MB cache?
	jbe s	I_HSz2		;Yes, O.K. to set it up now.
	cmp	dx,(40*32*2)	;Can 40- or 80-MB cache fit the HMA?
	jae s	I_HSz1		;Yes, see if more can be used.
	mov	cl,1		;Must default to only a 15-MB cache.
	jmp s	I_HSz2		;Go set up our 15-MB cache now.
I_HSz1:	cmp	cl,3		;Does user want only a 40-MB cache?
	je s	I_HSz2		;Yes, O.K. to set it up now.
	cmp	cl,4		;Does user want a 25/50-MB cache?
	ja s	I_Siz1		;No, see if "large" cache is O.K.
	cmp	dx,(50*32*2)	;Can 25- or 50-MB cache fit the HMA?
	jae s	I_HSz2		;Yes, O.K. to set it up now.
	dec	cx		;Must default to 15- or 40-MB cache.
I_HSz2:	jmp s	I_SetC		;Go set up desired HMA cache now.
I_NoHM:	mov	HFlag,0		;Ensure NO use of HMA space!
	mov	ax,HMASize	;Set driver upper/DOS memory size.
	add	VDSLn.lw,ax
	xor	edx,edx		;Get 32-bit "free" memory limit.
	xchg	dx,PktLBA.hw
	shl	edx,4
	add	edx,PktLBA
	xor	eax,eax		;Subtract 32-bit driver "base" size
	mov	ax,cs		;  for available search-table space.
	add	ax,((ResEnd-@)/16)
	shl	eax,4
	sub	edx,eax
	jb s	I_Min1		;Limit < driver?  Assume minimum!
	cmp	edx,65536	;64K or more of free memory?
	jb s	I_MinM		;No, see if we have minimum needed.
	mov	dx,0FFFFh	;Use 64K - 1 as free memory size.
I_MinM:	cmp	dx,(50*32*2)	;Enough memory for a 100-MB cache?
	jae s	I_Size		;Yes, go see what cache is desired.
I_Min1:	mov	dx,(50*32*2)	;Kernel CRETINS, UIDE loads in > 8K!
I_Size:	movzx	cx,SFlag	;Get user cache-size flag.
	cmp	cl,5		;Has user requested a "large" cache?
	jb s	I_SetC		;No, O.K. to set up lower sizes now.
I_Siz1:	mov	ax,LCBlks	;Get required search-table memory.
	shl	ax,1
	cmp	dx,ax		;Enough memory for requested cache?
	jae s	I_SetC		;Yes, go set it up now.
	mov	cl,6		;Must use our default 80-MB cache.
I_SetC:	mov	al,12		;Point to desired cache-size table.
	mul	cl
	add	ax,(CachSiz-@)
	xchg	ax,si
	mov	ax,[si]		;Set cache-entry counts.
	add	@TC2,ax
	mov	@TC3,ax
	shl	ax,1		;Set binary-search limit addresses.
	add	@TE2,ax
	add	@TE3,ax
	sar	HFlag,1		;Will we be loading in the HMA?
	jnz s	I_SetM		;Yes, set search midpoint values.
	add	VDSLn.lw,ax	;Set final resident-driver size.
I_SetM:	mov	ax,[si+4]	;Get binary-search midpoint * 2 (our
	shl	ax,1		;  CachSiz table has XMS-cache data).
	mov	@MP3,ax		;Set binary-search midpoint values.
	add	@MP4,ax
	xor	dx,dx		;Get cache-table XMS memory size.
	mov	ax,[si+6]
	shl	ax,1		;Drop to 2/3 for protected cache as
	mov	cx,3		;  binary-search data is not in XMS!
	div	cx
	movzx	ecx,ax		;Preset cache-tables XMS memory size.
I_SetG:	mov	ax,[si+2]	;Set cache "granularity" values.
	mov	@GRAN1,al
	mov	@GRAN2,al
	mov	@GRAN3,al
	mov	@GRAN4,al
	mov	@GRAN5,al
	mov	@GRAN6,al
	mov	@GRSSC1,ah
	mov	@GRSSC2,ah
	mov	ah,0		;Multiply number of cache blocks
	shr	ax,1		;  times (sectors-per-block / 2).
	mul	[si].lw
	push	dx		;Get 32-bit cache data XMS size.
	push	ax
	pop	eax
	mov	RqBuf,eax	;Save cache XMS size for below.
	add	eax,ecx		;Add in cache-tables XMS memory.
	add	RqXMS,eax	;Save total required XMS memory.
	mov	eax,[si+8]	;Set cache size in "title" message.
	mov	TTL2.dwd,eax
I_ExtE:	shl	EFlag,1		;Retain external-entry logic?
	jnz s	I_TTL		;Yes, go display driver "title".
	and	ExEntrP,0	;Reset external-entry pointer.
	sar	HFlag,1		;Will we be loading in the HMA?
	jz s	I_TTL		;No, go display driver "title".
	mov	ax,(NXSTACK-@)	;Dismiss external-entry logic.
	mov	VDSLn.lw,ax
I_TTL:	mov	dx,(TTLMsg-@)	;Display driver "title" message.
	call	I_Msg
	mov	dx,(TTL3-@)
	call	I_Msg
	mov	ax,cs		;Set "fixed" driver segment values.
	mov	@CDMain.hw,ax
	mov	@HDMain.hw,ax
	mov	@CDXSeg,ax
	mov	@HXSeg1,ax
	mov	@PasSeg,ax
	mov	@RqESeg,ax
	mov	@HXSeg2,ax
	mov	@HXSeg3,ax
	mov	ax,(DevInt-@)	;Set run-time "Device Interrupt" ptr.
	mov	DevIntP,ax
	mov	ax,03513h	;Get and save Int 13h vector.
	call	I_In21
	push	es
	push	bx
	pop	eax
	mov	@Prv13A,eax
	mov	@Prv13B,eax
	mov	@Prv13C,eax
	xor	eax,eax		;Zero EAX-reg. for 20-bit addressing.
	mov	es,ax		;Point ES-reg. to low memory.
	mov	al,es:[HDISKS]	;Save our BIOS hard-disk count.
	and	BiosHD,al
	mov	ax,cs		;Set 20-bit VDS driver address.
	mov	VDSSg.lw,ax
	shl	eax,4
	mov	VDSAd,eax
	cli			      ;Avoid interrupts in VDS tests.
	test	es:[VDSFLAG].lb,020h  ;Are "VDS services" active?
	jz s	I_REnI		      ;No, re-enable CPU interrupts.
	mov	ax,08103h	;"Lock" this driver into memory.
	mov	dx,0000Ch
	call	I_VDS
	jc	I_VEr1		;If "lock" error, display msg. & exit!
	inc	VDSOf.lb	;Set initialization VDS "lock" flag.
I_REnI:	sti			;Re-enable CPU interrupts.
	mov	eax,VDSAd	;Get final driver 32-bit address.
	add	PRDAd,eax	;Set "No XMS memory" PRD address.
	add	GDTPAdr,eax	;Relocate "real mode" GDT base addr.
	add	@WBAdr1,eax	;Relocate "working" buffer addresses.
	add	@WBAdr2,eax
	add	@WBAdr3,eax
	add	@CBAdr1,eax	;Relocate cache-entry buffer addrs.
	add	@CBAdr2,eax	;Relocate cache-entry buffer addrs.
	add	@CLAddr,eax	;Relocate command-list source address.
	mov	GDT_CSL,ax	;Set lower CS-descriptor address.
	shr	eax,16		;Set upper CS-descriptor address.
	mov	GDT_CSM,al
	mov	GDT_CSH,ah
	xor	edi,edi		;Get PCI BIOS "I.D." code.
	mov	al,001h
	call	I_In1A
	cmp	edx," ICP"	;Is PCI BIOS V2.0C or newer?
	je s	I_ScnC		;Yes, scan for all IDE controllers.
	mov	dx,(NPMsg-@)	;Display "No V2.0C+ PCI" message.
	call	I_Msg
	jmp s	I_NoXM		;Go see if we need an XMS manager.
I_ScnC:	mov	al,LBABuf.lb	;Get next "interface bit" value.
	and	ax,00003h
	or	ax,PCISubC	;"Or" in subclass & current function.
	call	I_PCIC		;Test for specific PCI class/subclass.
	rol	LBABuf.lb,4	;Swap both "interface bit" values.
	mov	al,LBABuf.lb	;Load next "interface bit" value.
	or	al,al		;Both "interface" values tested?
	jns s	I_ScnC		;No, loop back and test 2nd one.
	add	PCISubC.lb,004h	;More PCI function codes to try?
	jnc s	I_ScnC		;Yes, loop back & try next function.
	cmp	DVDTblA,(DVD1Tbl-@)  ;Any CD/DVD controllers found?
	ja s	I_Scn1		;Yes, see about "Native PCI" checks.
	add	DVDTblA,DVDTSIZ	;Save "legacy" addresses for oldies!
I_Scn1:	test	al,001h		;Have we tested "Native PCI" ctlrs.?
	mov	LBABuf.lb,093h	;(Set "Native PCI" interface bits).
	jz s	I_ScnC		;No, loop back and test them, also.
;
; These additional controller tests could be added, if necessary:
;
;	mov	ax,00400h	;Scan for "RAID" controller.
;	call	I_PCIC
;	mov	ax,00520h	;Scan for ADMA "Single-stepping" ctlr.
;	call	I_PCIC
;	mov	ax,00530h	;Scan for ADMA "Continuous DMA" ctlr.
;	call	I_PCIC
;	mov	ax,00600h	;Scan for standard SATA controller.
;	call	I_PCIC
;	mov	ax,00601h	;Scan for "AHCI" SATA controller.
;	call	I_PCIC
;
I_NoXM:	shl	XMF,1		;Has user requested no XMS memory?
	jz s	I_XMgr		;No, go see about an XMS manager.
	mov	dx,(NIMsg-@)	;Point to "/N3 illegal" message.
	cmp	PRDAd.hw,00Ah	;Are we loaded in upper-memory?
	jae	I_Err		;Yes?  Display error message & exit.
	db	0B8h		;Disable hard-disk buffered I-O.
	pop	eax
	mov	@NoXM1.lw,ax
	db	066h,0B8h
	db	0E9h
	dw	(Pass1-(@NoXM1+5))
	db	0
	mov	[@NoXM1+2].dwd,eax
	db	0B8h		;Get "clc" and "ret" commands.
	clc
	ret
	mov	MovCLX.lw,ax	;Disable command-list moves to XMS.
	db	0B8h		;Get "stc" and "ret" commands.
	stc
	ret
	mov	SetAd1.lw,ax	;Disable any use of an XMS buffer.
	mov	al,090h		;Disable all "A20" request calls.
	mov	A20Req.lb,al
	mov	@NoXM2.lb,al	;Disable CD/DVD buffered input via
	db	066h,0B8h	;  "nop"/"cmc"/"adc [bx+DMF-@],bl"
	cmc			;  at "@NoXM2" above -- direct DMA
	adc	[bx+DMF-@],bl	;  if buffer O.K., PIO mode if not.
	mov	[@NoXM2+1].dwd,eax
	jmp	I_HDCh		;Go check on our hard-disk count.
I_XMgr:	mov	ax,04300h	;Inquire if we have an XMS manager.
	call	I_In2F
	mov	dx,(NXMsg-@)	;Point to "No XMS manager" message.
	cmp	al,080h		;Is an XMS manager installed?
	jne	I_Err		;No?  Display error message and exit!
	mov	ax,04310h	;Get XMS manager "entry" address.
	call	I_In2F
	push	es		;Save XMS manager "entry" addresses.
	push	bx
	pop	@XEntry
	mov	al,@Gran1	;Request all necessary XMS memory.
	and	al,080h		;(009h V2.0 XMS command for small &
	mov	ah,009h		;  medium caches, 089h V3.0 command
	or	ah,al		;  for our large caches).
	mov	edx,RqXMS
	call	I_XMS
	jz s	I_XLok		;If no errors, "lock" XMS memory.
I_XErr:	mov	eax," SMX"	;BAAAD News!  Get "XMS" msg. prefix.
	jmp	I_VErr		;Go display "XMS init error" & exit!
I_XLok:	mov	XMHdl,dx	;Save our XMS "handle" number.
	mov	ah,00Ch		;"Lock" our XMS memory.
	call	I_XMS
	jnz s	I_XErr		;If error, display message and exit!
	shl	edx,16		;Get unaligned 32-bit buffer address.
	or	dx,bx
	mov	eax,edx		;Copy 32-bit address to EAX-reg.
	jz s	I_XBAd		;Any low-order XMS buffer "offset"?
	mov	ax,0FFFFh	;Yes, align address to an even 64K.
	inc	eax
I_XBAd:	mov	@XBAddr,eax	;Save aligned "main buffer" address.
	add	@XBase1,eax	;Initialize cache-data base addrs.
	add	@XBase2,eax
	mov	cx,ax		;Get buffer "offset" in XMS memory.
	sub	cx,dx
	mov	edx,000010000h	;Put command-list after XMS buffer.
	jcxz	I_PRDA		;Is buffer already on a 64K boundary?
	mov	edx,-32		;No, put command-list before buffer.
	dec	@XBase1.hw	;Decrement cache-data bases by 64K.
	dec	@XBase2.hw
I_PRDA:	add	eax,edx		;Set our 32-bit PRD address.
	mov	PRDAd,eax
	mov	ax,[RqBuf+1].lw	;Get needed cache XMS in 256K blocks.
	shl	eax,18		;Convert from 256K blocks to bytes.
	add	eax,@XBase1	;Add cache-data XMS base address.
	mov	@CTBas1,eax	;Set cache-table XMS base addresses.
	mov	@CTBas2,eax
	mov	edx,@TC1	;Set binary-search reset block addr.
	mov	ecx,edx
	shl	ecx,1
	add	ecx,edx
	shl	ecx,1
	add	ecx,eax
	mov	@STblR,ecx
	add	ecx,edx		;Set our "ZipZip" buffer addresses.
	mov	@SBufA,ecx
	mov	@SBufB,ecx
	add	ecx,edx		;Set binary-search table addresses.
	mov	@STblA,ecx
	mov	@STblB,ecx
	mov	@STblC,ecx
	mov	@STblD,ecx
	cmp	GoMain.lb,0E8h	;Is this the "stand alone" driver?
	je s	I_HDCh		;Yes, forget about running diskettes!
	xor	ax,ax		;Point ES-reg. to low memory.
	mov	es,ax
	mov	al,es:[HDWRFL]	;Get BIOS "hardware installed" flag.
	test	al,001h		;Any diskettes on this system?
	jz s	I_HDCh		;No, scan for available hard-disks.
	mov	al,es:[MCHDWFL]	;Use diskette media-change bits
	and	al,011h		;  for our number of diskettes.
I_FScn:	test	al,001h		;Can next unit signal media-changes?
	jz s	I_FMor		;No?  CANNOT use this old "clunker"!
	push	ax		;Save our diskette counter.
	mov	al,0C0h		;Get "diskette" device-type flags.
	call	I_CHSD		;Get and save diskette's CHS values.
	pop	ax		;Reload our diskette counter.
	inc	@LastU		;Bump unit-table index.
I_FMor:	inc	HDUnit		;Bump BIOS unit number.
	inc	UnitNo.lb	;Bump error-message unit number.
	shr	al,4		;Another diskette to check?
	jnz s	I_FScn		;Yes, loop back and do next one.
I_HDCh:	cmp	BiosHD,0	;Any BIOS hard-disks to use?
	je	I_SCD		;No, scan for CD/DVD units to use.
	mov	HDUnit,080h	    ;Set 1st BIOS hard-disk unit.
	mov	UnitNam.dwd," ksi"  ;Set messages for hard-disks.
	mov	[UnitNo+1].lw,".h"
I_Next:	mov	ah,HDUnit	    ;Set unit no. in error message.
	mov	cx,2
	mov	si,(UnitNo-1-@)
	call	I_HexA
	mov	al,080h		;Get "BIOS disk" device-type.
	call	I_CHSD		;Get and save next disk's CHS values.
	mov	ah,041h		;Get EDD "extensions" for this disk.
	mov	bx,055AAh
	call	I_In13
	jc s	I_NoUJ		;If none, bump non-UltraDMA disk count.
	cmp	bx,0AA55h	;Did BIOS "reverse" our entry code?
	jne s	I_NoUJ		;No, bump non-UltraDMA disk count.
	test	cl,004h		;Does this disk have "EDD" extensions?
	jz s	I_NoUJ		;No, bump non-UltraDMA disk count.
	mov	si,(EDDBuff-@)	;Point to "EDD" input buffer.
	mov	[si].lw,30	;Set 30-byte buffer size.
	mov	ah,048h		;Get this disk's "EDD" parameters.
	call	I_In13
	jc s	I_ErED		;Error?  Display msg. & ignore!
	cmp	[si].lw,30	;Did we get at least 30 bytes?
	jb s	I_NoUJ		;No, bump non-UltraDMA disk count.
	cmp	[si+26].dwd,-1	;Valid "DPTE" pointer?
	je s	I_NoUJ		;No, bump non-UltraDMA disk count.
	les	si,[si+26]	;Get this disk's "DPTE" pointer.
	mov	bx,15		;Calculate "DPTE" checksum.
	xor	cx,cx
I_CkSm:	add	cl,es:[bx+si]
	dec	bx
	jns s	I_CkSm
	jcxz	I_EDOK		;If checksum O.K., use parameters.
I_ErED:	mov	dx,(EDDMsg-@)	;Display "EDD BIOS ERROR" message.
	call	I_Msg
	mov	dx,(UnitMsg-@)
	call	I_Msg
I_NoUJ:	jmp s	I_NoUD		;Go bump number of non-UltraDMA disks.
I_EDOK:	mov	bx,00010h	;Initialize IDE unit number index.
	and	bl,es:[si+4]
	shr	bl,4
	mov	ax,es:[si]	;Get disk's IDE base address.
	mov	CtlrNo.lb,'0'	;Reset display to "Ctlr. 0".
	mov	si,(Ctl1Tbl-@)	;Point to IDE address table.
I_ITbl:	cmp	[si].lw,-1	;Is this IDE table active?
	je s	I_NoUJ		;No, go bump non-UltraDMA disk count.
	cmp	ax,[si+2]	;Is disk on this primary channel?
	je s	I_ChMS		;Yes, set disk channel and unit.
	inc	bx		;Adjust index for secondary channel.
	inc	bx
	cmp	ax,[si+4]	;Is disk on this secondary channel?
	je s	I_ChMS		;Yes, set disk channel and unit.
	inc	bx		;Adjust values for next controller.
	inc	bx
	add	si,CTLTSIZ
	inc	CtlrNo.lb
	cmp	si,(CtlTEnd-@)	;More IDE controllers to check?
	jb s	I_ITbl		;Yes, loop back and check next one.
	jmp s	I_NoUJ		;Go bump non-UltraDMA disk count.
I_ChMS:	push	bx		;Save disk's caching unit number.
	mov	IdeDA,ax	;Save disk's base IDE address.
	add	ax,CDSEL	;Point to IDE device-select register.
	xchg	ax,dx
	mov	al,bl		;Get drive-select command byte.
	shl	al,4
	or	al,LBABITS
	out	dx,al		;Select master or slave disk.
	push	ax		;Save drive-select and caching unit.
	push	bx
	mov	dx,(CtlMsg-@)	;Display IDE controller number.
	call	I_Msg
	pop	ax		;Reload caching unit number.
	mov	dx,(PriMsg-@)	;Point to "Primary" channel name.
	test	al,002h		;Is this a primary-channel disk?
	jz s	I_PRNm		;Yes, display "Primary" channel.
	mov	dx,(SecMsg-@)	;Point to "Secondary" channel name.
I_PRNm:	call	I_Msg		;Display disk's IDE channel name.
	pop	ax		;Reload drive-select byte.
	mov	dx,(MstMsg-@)	;Point to "master" disk name.
	test	al,010h		;Is this disk the master?
	jz s	I_MSNm		;Yes, display "master" name.
	mov	dx,(SlvMsg-@)	;Point to "slave" disk name.
I_MSNm:	call	I_Msg		;Display disk's master/slave name.
	call	I_ValD		;Validate disk as an UltraDMA unit.
	pop	ax		;Reload caching unit number.
	jc s	I_UDEr		;If any errors, display message.
	mov	bx,@LastU	;Change this disk's device-type to
	mov	[bx+TypeF-@],al	;  "IDE" and include channel/unit.
	jmp s	I_More		;Go check for any more BIOS disks.
I_UDEr:	call	I_Msg		;NOT UltraDMA -- Display error msg.
I_NoUD:	cmp	GoMain.lb,0E8h	;Is this the "stand alone" driver?
	je s	I_IgnD		;Yes, ignore disk and scan for more.
	inc	NonUCt		;Bump number of non-UltraDMA disks.
	cmp	NonUCt,'9'	;Over 9 non-UltraDMA hard disks?
	jbe s	I_More		;No, check for more disks.
	mov	NonUCt.lw,"+9"	;Set 9+ non-UltraDMA count.
I_More:	inc	@LastU		;Bump our unit-table index.
I_IgnD:	inc	HDUnit		;Bump BIOS unit number.
	cmp	@LastU,MAXBIOS	;More entries in our units table?
	jae s	I_SCD		;No, check for any CD/DVD drives.
	dec	BiosHD		;More BIOS disks to check?
	jnz	I_Next		;Yes, loop back and do next one.
I_SCD:	xor	bx,bx		;Zero BX-reg. for CD/DVD logic.
	mov	DNMsg.lw," ,"	;Change start of disk "name" msg.
	mov	CtlrNo.lb,'0'-1	;Reset display controller number.
I_SCD1:	test	USNdx,001h	;Is this a new I-O channel?
	jnz s	I_SCD2		;No, test for new controller.
	add	DVDTblB,4	;Bump to next I-O addresses.
I_SCD2:	cmp	USNdx,004h	;Is this a new controller?
	jb s	I_SCD3		;No, get unit's I-O addresses.
	inc	CtlrNo.lb	;Bump display controller number.
	mov	USNdx,bl	;Reset unit-select index.
I_SCD3:	mov	si,DVDTblB	;Load I-O parameter pointer.
	cmp	si,(DVDTEnd-@)	;Any more IDE units to check?
	jae s	I_AnyN		;No, check for any drives to use.
	mov	al,USNdx	;Set indicated unit-select byte.
	shl	al,4
	or	al,MSEL
	mov	USelB,al
	inc	USNdx		   ;Bump unit-select index.
	mov	eax,[si]	   ;Get this unit's I-O addresses.
	cmp	ax,-1		   ;Any more controllers to check?
	je s	I_AnyN		   ;No, check for any drives to use.
	mov	[bx+DMAAd-@],eax   ;Set this unit's I-O addresses.
	cmp	NoDMA,bl	   ;Is all UltraDMA disabled?
	jne s	I_SCD4		   ;Yes, default to no DMA address.
	or	ax,ax		   ;"Legacy IDE" with no DMA ctlr.?
	jnz s	I_SCD5		   ;No, go validate this unit.
I_SCD4:	or	[bx+DMAAd-@].lw,-1 ;Invalidate UltraDMA address.
I_SCD5:	call	I_VCD		   ;Validate unit as ATAPI CD/DVD.
	jc s	I_SCD1		   ;If invalid, merely ignore it.
	mov	si,UTblP	   ;Update unit-table parameters.
	mov	eax,[bx+DMAAd-@]
	mov	[si],eax
	mov	al,USelB
	mov	[si+6],al
	add	si,20		   ;Update unit-table pointer.
	mov	UTblP,si
	inc	[bx+CDUnits-@].lb  ;Bump number of active units.
	inc	CDMsgNo		   ;Bump display unit number.
	cmp	si,(UTblEnd-@)	   ;Can we install another drive?
	jb	I_SCD1		   ;Yes, loop back & check for more.
I_AnyN:	cmp	NonUCt.lb,'0'	;Do we have any non-UltraDMA disks?
	je s	I_AnyD		;No, check for any drives to use.
	mov	dx,(NonUMsg-@)	;Display "Non-UltraDMA disks" msg.
	call	I_Msg
I_AnyD:	mov	dx,(NDMsg-@)	;Point to "Nothing to use" message.
	movzx	ax,CDUnits.lb	;Get total CD/DVD and BIOS drives.
	add	ax,@LastU	;Did we find any drives to use?
	jz	I_Err		;No?  Display error message and exit!
	cmp	ah,CDUnits	;Set FreeDOS "UIDE$" CD/DVD name,
	jne s	I_PMCM		;  if no CD/DVD drivers were found.
	mov	DvrNam.dwd,"EDIU"
	mov	[DvrNam+4].dwd,"   $"
I_PMCM:	sar	PFlag,1		;Using protected-mode caching?
	jz s	I_20LE		;No, see if we can use HMA space.
	cld			;Overlay all protected-mode caching
	mov	cx,(PCLSIZE/2)	;  routines onto main caching logic.
	mov	si,(PCache-@)
	mov	di,(Cache-@)
	push	ds
	pop	es
	rep	movsw
I_20LE:	sar	HFlag,1		;Can we use any HMA space?
	jz	I_RBlk		;No, see about reset-block setup.
	mov	ah,005h		;Issue "A20 local-enable" request.
	call	I_XMS
I_A20E:	mov	eax," 02A"	;Get "A20" error-message prefix.
	jnz s	I_HMAX		;If any "A20" error, bail out QUICK!!
	mov	bx,HMASize	;Get required driver HMA space.
	sar	PFlag,1		;Using protected-mode caching?
	jz s	I_HMA		;No, go request our needed HMA.
	mov	ax,@TC3		;Add in binary-search table size.
	shl	ax,1
	add	bx,ax
I_HMA:	mov	ax,04A02h	;Request needed memory in the HMA.
	call	I_In2F
	mov	ax,es		;Get returned HMA segment address.
	cmp	ax,-1		;Is segment = 0FFFFh?
	jne s	I_HMA1		;No, VERY unusual but not an error.
	cmp	ax,di		;Is offset also = 0FFFFh?
	jne s	I_HMA1		;No, set binary-search table addr.
	call	I_A20D		;BAAAD News!  Do "A20 local-disable".
	mov	eax," AMH"	;Get "HMA" error-message prefix.
I_HMAX:	jmp	I_VErr		;Go display error message & exit!
I_HMA1:	push	ax		;Save starting HMA address.
	push	di
	movzx	eax,ax		;Get 32-bit HMA segment address.
	shl	eax,4
	movzx	ecx,di		  ;Adjust "real mode" GDT base addr.
	add	cx,(OurGDT-CDReq)
	add	ecx,eax
	mov	GDTPAdr,ecx
	mov	GDT_CSL,ax	;Adjust lower CS-descriptor address.
	shr	eax,16		;Adjust upper CS-descriptor address.
	mov	GDT_CSM,al
	mov	GDT_CSH,ah
	pop	eax		;Set CD/DVD entry routine pointer.
	mov	@CDMain,eax
	add	ax,(HDReq-CDReq)   ;Set disk entry routine pointer.
	mov	@HDMain,eax
	lea	ax,[di-(CDReq-@)]  ;Get starting HMA logic offset.
	add	@UTable,ax	   ;Adjust CD/DVD table pointers.
	add	@DspTB1,ax
	add	@DspTB2,ax
	add	@DspTB3,ax
	add	@DspTB4,ax
	add	@DspOfs,ax	;Adjust CD/DVD dispatch offset.
	add	@TypeF,ax	;Adjust disk "CHS tables" offsets.
	add	@CHSec,ax
	add	@CHSHd,ax
	add	@CtlTbl,ax	;Adjust disk controller-table ptr.
	add	@GDTP,ax	;Adjust "MvData" routine offsets.
	add	@MvDsc1,ax
	add	@MvDsc2,ax
	sar	PFlag,1		;Are we using protected-mode caching?
	jz s	I_HMA2		;No, move needed logic up to the HMA.
	add	di,HMASize	;Set protected search-table address.
	mov	@TA1-PCOFFS,di
	mov	ax,@TC3		;Set protected search-table limits.
	shl	ax,1
	add	ax,di
	mov	@TE2-PCOFFS,ax
	mov	@TE3-PCOFFS,ax
	mov	ax,@MP3		;Set protected search-midpoint addr.
	dec	ax
	dec	ax
	add	ax,di
	mov	@MP4-PCOFFS,ax
I_HMA2:	mov     cx,HMASize      ;Move all needed logic up to the HMA.
	shr     cx,1
	mov	si,(CDReq-@)
	les	di,@CDMain
	rep	movsw
	call	I_A20D		;Issue "A20 local-disable" request.
	mov	ax,(GDTP-@)	;Restore "MvData" routine offsets,
	mov	@GDTP,ax	;  so we need NOT call "MvData" in
	mov	ax,(DstDsc+2-@)	;  the HMA for reset-block moves!!
	mov	@MvDsc1,ax
	mov	ax,(MoveDT-@)
	mov	@MvDsc2,ax
I_RBlk:	cmp	GoMain.lb,0E8h	;Is this the "stand alone" driver?
	je s	I_Done		;Yes, go post "init" packet results.
	sar	PFlag,1		;Using protected-mode caching?
	jnz s	I_Done		;Yes, go post "init" packet results.
	mov	ah,005h		;Issue "A20 local-enable" request.
	call	I_XMS
	jnz	I_A20E		;If any "A20" error, bail out NOW!
	mov	edx,@TC1	;Initialize "reset block" count.
	shr	edx,1
	xor	bp,bp		;Initialize cache-table index.
I_RBlA:	mov	ax,bp		;Set next 200 reset-block indexes
	mov	ecx,200		;  in init tables & messages area.
	mov	si,(CachSiz-@)
I_RBlB:	mov	[si],ax
	inc	ax
	inc	si
	inc	si
	loop	I_RBlB
	xor	esi,esi		;Set 32-bit move source address.
	mov	si,(CachSiz-@)	;(Offset of our "reset" buffer +
	add	esi,VDSAd	;   32-bit driver "base" address).
	movzx	edi,bp		;Set 32-bit move destination addr.
	shl	edi,1		;(2 * current "reset" index +
	add	edi,@STblR	;   XMS "reset block" address).
	mov	bp,ax		;Update next cache-table index.
	mov	cl,200		;Get 200-word move length.
	cmp	ecx,edx		;At least 200 words left to go?
	jbe s	I_RBlC		;Yes, use full 200-word count.
	mov	cx,dx		;Use remaining word count.
I_RBlC:	shl	ecx,1		;Convert word count to byte count.
	push	edx		;Save move count and cache index.
	push	bp
	call	MvData		;Move next indexes to "reset block".
	pop	bp		;Reload cache index and move count.
	pop	edx
	jnc s	I_RBlD		;If no XMS errors, check move count.
	call	I_A20D		;BAAAD News!  Do "A20 local-disable".
	jmp	I_XErr		;Go display "XMS init error" & exit!
I_RBlD:	sub	dx,200		;More "reset block" indexes to set?
	ja s	I_RBlA		;Yes, loop back and do next 200.
	call	I_A20D		;Issue "A20 local-disable" request.
I_Done:	popad			;Done!  Reload 32-bit CPU registers.
	mov	ax,VDSLn.lw	;Set initial driver stack pointer.
	mov	@Stack,ax
	les	bx,RqPkt	;Set results in DOS "init" packet.
	mov	es:[bx].RPLen,ax
	mov	es:[bx].RPStat,RPDON
	xchg	ax,bx		;Set up to clear driver's stack.
	mov	ax,(STACK/2)
	jmp	I_ClrS		;Go clear driver's stack and exit.
;
; Init subroutine to get the user's desired /Snnnn cache size.
;
GetSiz:	inc	si		   ;Bump scan ptr. past "S" or "s".
	mov	di,(LCBlks-@)	   ;Point to variable block count.
	mov	[di+4].dwd,(36*65536)+1024 ;Reset large-cache values.
	mov	[di+8].dwd,"    "  ;Reset variable "title".
GtSiz0:	mov	[di].lw,08000h	   ;Invalidate cache-block count.
GtSiz1:	movzx	ax,es:[si].lb	;Get next command-line byte.
	cmp	al,'9'		;Is byte greater than a '9'?
	ja s	GtSiz2		;Yes, go see what is in bucket.
	cmp	al,'0'		;Is byte less than a '0'?
	jb s	GtSiz2		;Yes, go see what is in bucket.
	inc	si		;Bump scan pointer past digit.
	cmp	[di+8].lb,' '	;Have we already found 4 digits?
	jne s	GtSiz0		;Yes, set INVALID & keep scanning.
	push	[di+9].dwd	;Shift "title" bytes 1 place left.
	pop	[di+8].dwd
	mov	[di+11],al	;Insert next "title" message byte.
	mov	cx,[di]		;Multiply current block size by 10.
	shl	[di].lw,2
	add	[di],cx
	shl	[di].lw,1
	and	al,00Fh		;"Add in" next cache size digit.
	add	[di],ax
	jmp s	GtSiz1		;Go scan for more cache-size digits.
GtSiz2:	xor	cx,cx		;Set 5-MB cache size flag.
	cmp	[di].lw,15	;Does user want less than 15-MB?
	jb s	GtSizX		;Yes, give user 5-MB and exit.
	mov	cl,2		;Set 25-MB cache size flag.
	cmp	[di].lw,25	;Does user want exactly 25-MB?
	je s	GTSizX		;Yes, set cache-size flag and exit.
	dec	cx		;Set 15-MB cache size flag.
	cmp	[di].lw,40	;Does user want less than 40-MB?
	jb s	GtSizX		;Yes, give user 15-MB and exit.
	mov	cl,4		;Set 50-MB cache size flag.
	cmp	[di].lw,50	;Does user want exactly 50-MB?
	je s	GTSizX		;Yes, set cache-size flag and exit.
	dec	cx		;Set 40-MB cache size flag.
	cmp	[di].lw,80	;Does user want less than 80-MB?
	jb s	GTSizX		;Yes, give user 40-MB and exit.
	mov	cl,5		;Set "large" cache size flag.
	cmp	[di].lw,4093	;Is cache size invalid or too big?
	ja s	GtSizE		;Yes?  Ignore & set default cache.
	mov	al,128		;Set initial 128-MB cache limit.
GtSiz3:	cmp	[di],ax		;User cache size < current limit?
	jb s	GTSiz4		;Yes, set user cache-block count.
	shl	ax,1		;Double current cache-size limit.
	shl	[di+4].dwd,1	;Double variable cache parameters.
	jmp s	GtSiz3		;Go check user's cache size again.
GtSiz4:	shl	[di].lw,4	;Cache blocks = (16 * Megabytes).
	jmp s	GtSizX		;Go set cache-size flag and exit.
GtSizE:	mov	[di].lw,(80*16)   ;Error!  Set our default cache.
	mov	[di+8].dwd,"08  "
GtSizX:	mov	SFlag,cl	  ;Set desired cache-size flag.
GtSRet:	ret			  ;Exit.
;
; Init subroutine to test for and set up a PCI disk controller.
;
I_PCIC:	mov	IdeDA,ax	     ;Save subclass & function codes.
	and	LBABuf.hw,0	     ;Reset PCI device index.
I_PCI1:	cmp	RqIndex,(CtlTEnd-@)  ;More space in address tables?
 	jae s	GtSRet		     ;No, go exit above.
	mov	ax,IdeDA	;Test PCI class 1, subclass/function.
	mov	ecx,000010003h	;(Returns bus/device/function in BX).
	xchg	ax,cx
	mov	si,LBABuf.hw
	call	I_In1A
	jc s	GtSRet		;Controller not found -- exit above.
	inc	LBABuf.hw	;Bump device index for another test.
	xor	di,di		;Get controller Vendor & Device I.D.
	call	I_PCID
	push	ecx		;Save Vendor and Device I.D.
	mov	di,32		;Save DMA controller base address.
	call	I_PCID
	xchg	ax,cx
	and	al,0FCh
	mov	DMAAd,ax
	mov	si,(PCMsg2-@)	;Set controller address in msg.
	call	I_Hex
	mov	si,(PCMsg3-@)	;Set vendor I.D. in message.
	pop	ax
	call	I_Hex
	pop	ax		;Set Device I.D. in message.
	call	I_Hex
	mov	di,4		;Get low-order PCI command byte.
	call	I_PCID
	not	cl		;Get "Bus Master" & "I-O Space" bits.
	and	cl,005h		;Is controller using both BM and IOS?
	jz s	I_PCI2		;Yes, save this controller's data.
	mov	dx,(BCMsg-@)	;Display "BAD controller!", and hope
	call	I_Msg		;  our user can find a better BIOS!
	mov	dx,(PCMsg1-@)
	call	I_Msg
I_PCIJ:	jmp s	I_PCI1		;Go test for more same-class ctlrs.
I_PCI2:	mov	si,RqIndex	;Get current I-O address table ptr.
	mov	ax,DMAAd	;Set controller DMA base address.
	mov	[si],ax
	test	LBABuf.lb,001h	;Is this a "Native PCI" controller?
	jz s	I_PCI3		;No, set CD/DVD table addresses.
	mov	di,16		;Set primary-channel base address.
	call	I_PCID
	and	cl,0FCh
	mov	[si+2],cx
	mov	di,24		;Set secondary-channel base address.
	call	I_PCID
	and	cl,0FCh
	mov	[si+4],cx
I_PCI3:	cld			;Ensure FORWARD "string" commands.
	mov	di,DVDTblA	;Point to CD/DVD addresses table.
	push	cs
	pop	es
	lodsd			;Set CD/DVD primary addresses.
	stosd
	add	ax,8		;Set CD/DVD secondary addresses.
	stosw
	movsw
	mov	DVDTblA,di	;Bump CD/DVD table pointer.
	mov	dx,(PCMsg-@)	;Display all controller data.
	call	I_Msg
	inc	[PCMsg+3].lb	;Bump controller number in message.
	add	RqIndex,CTLTSIZ	;Bump controller address-table ptr.
	jmp s	I_PCIJ		;Go test for more same-class ctlrs.
;
; Init Subroutine to "validate" an UltraDMA hard-disk.
;
I_ValD:	xor	bx,bx		;Point ES-reg. to low memory.
	mov	es,bx
	mov	dx,[bx+IdeDA-@]	;Issue "Identify Device" command.
	add	dx,CCMD
	mov	al,0ECh
	out	dx,al
	mov	bx,BIOSTMR	;Point to low-memory BIOS timer.
	mov	cl,RDYTO	;Set I-O timeout limit in CL-reg.
	add	cl,es:[bx]
I_IDly:	cmp	cl,es:[bx]	;Has our command timed out?
	je s	I_DErr		;Yes, set "Identify" message & exit.
	in	al,dx		;Get IDE controller status.
	test	al,BSY+RDY	;Controller or disk still busy?
	jle s 	I_IDly		;Yes, loop back and check again.
	test	al,ERR		;Did command cause any errors?
	jz s	I_PIO		;No, read I.D. data using PIO.
I_DErr:	mov	dx,(IEMsg-@)	;Point to "Identify error" msg.
I_SErr:	stc			;Set carry flag (error) & exit.
	ret
I_PIO:	add	dx,(CDATA-CCMD)	;Point to PIO data register.
	in	ax,dx		;Read I.D. bytes 0 and 1.
	shl	ax,1		;Save "ATA/ATAPI" flag in carry bit.
	mov	cx,27		;Skip I.D. bytes 2-53 and
I_Skp1:	in	ax,dx		;  read I.D. bytes 54-55.
	loop	I_Skp1
	push	cs		;Point to disk-name message.
	pop	es
	mov	di,(DName-@)
	mov	cl,26		;Read & swap disk name into message.
I_RdNm:	xchg	ah,al		;(I.D. bytes 54-93.  Bytes 94-105 are
	stosw			;  also read but are ignored.   Bytes
	in	ax,dx		;  106-107 are left in the AX-reg.).
	loop	I_RdNm
	xchg	ax,bx		;Save "UltraDMA valid" flag in BL-reg.
	mov	cl,35		;Skip I.D. bytes 108-175 &
I_Skp2:	in	ax,dx		;  read I.D. bytes 176-177.
	loop	I_Skp2
	mov	bh,ah		;Save "UltraDMA mode" flags in BH-reg.
	mov	cl,167		;Skip remaining I.D. data.
I_Skp3:	in	ax,dx
	loop	I_Skp3
	mov	dx,(UEMsg-@)	;Point to "is not UltraDMA" message.
	rcr	bl,1		;Shift "ATA/ATAPI" flag into BL-reg.
	and	bl,082h		;ATAPI disk, or UltraDMA bits invalid?
	jle s	I_SErr		;Yes?  Exit & display error message.
	mov	di,(Modes-@)	;Point to UltraDMA mode table.
	or	bh,bh		;Will disk do UltraDMA mode 0?
	jz s	I_SErr		;No?  Exit & display message!
I_NxtM:	cmp	bh,bl		;Will disk do next UltraDMA mode?
	jb s	I_GotM		;No, use current mode.
	inc	di		;Point to next mode table value.
	inc	di
	shl	bl,1		;More UltraDMA modes to check?
	jnz s	I_NxtM		;Yes, loop back.
I_GotM:	mov	si,(DNEnd-@)	;Point to end of disk name.
I_NxtN:	cmp	si,(DName-@)	;Are we at the disk-name start?
	je s	I_Name		;Yes, disk name is all spaces!
	dec	si		;Decrement disk name pointer.
	cmp	[si].lb,' '	;Is this name byte a space?
	je s	I_NxtN		;No, continue scan for non-space.
	inc	si		;Skip non-space character.
	mov	[si].lw," ,"	;End disk name with comma & space.
	inc	si
	inc	si
I_Name:	jmp	I_VC17		;Go display disk name/"mode" & exit.
;
; Init subroutine to set BIOS CHS data for a hard-disk or diskette.
;
I_CHSD:	push	ax		;Save unit's device-type flag.
	mov	ah,008h		;Get BIOS CHS data for this unit.
	call	I_In13
	jc s	I_CHSE		;If BIOS error, zero both values.
	and	cl,03Fh		;Get sectors/head value (low 6 bits).
	jz s	I_CHSE		;If zero, ERROR!  Zero both values.
	inc	dh		;Get heads/cylinder (BIOS value + 1).
	jnz s	I_CHST		;If non-zero, save data in our table.
I_CHSE:	xor	cl,cl		;Error!  Zero CHS data for this unit.
	xor	dh,dh
I_CHST:	mov	di,@LastU	;Point to "active units" table.
	mov	al,HDUnit	;Set BIOS unit number in our table.
	mov	[di+Units-@],al
	mov	[di+CHSec-@],cl ;Set unit's CHS data in our table.
	mov	[di+CHSHd-@],dh
	pop	ax		;Reload and set device-type flag.
	mov	[di+TypeF-@],al
	or	cl,cl		;Valid CHS values for this unit?
	jnz s	I_VC7		;Yes, go exit below.
	mov	dx,(CHSMsg-@)	;Display "CHS data error" & exit.
	jmp	I_Msg
;
; Init subroutine to "validate" an IDE unit as an ATAPI CD/DVD drive.
;
I_VCD:	mov	dx,[bx+DMAAd-@]	;Get unit UltraDMA command address.
	test	dl,001h		;Will this unit be using UltraDMA?
	jnz s	I_VC0		;No, just select "master" or "slave".
	in	al,dx		;Ensure any previous DMA is stopped!
	and	al,0FEh
	out	dx,al
I_VC0:	mov	dx,[bx+IdeDA-@]	;Point to IDE device-select register.
	add	dx,6
	mov	al,USelB	;Select IDE "master" or "slave" unit.
	out	dx,al
	call	ChkTO		;Await controller-ready.
	jc s	I_VC7		;If timeout, go exit below.
	mov	al,0A1h		;Issue "Identify Packet Device" cmd.
	out	dx,al
	call	ChkTO		;Await controller-ready.
	jc s	I_VC7		;If timeout, go exit below.
	test	al,DRQ		;Did we also get a data-request?
	jz s	I_VC6		;No, go set carry & exit below.
	sub	dx,7		;Point back to IDE data register.
	in	ax,dx		;Read I.D. word 0, main device flags.
	and	ax,0DF03h	;Mask off flags for an ATAPI CD/DVD.
	xchg	ax,si		;Save main device flags in SI-reg.
	mov	cx,26		;Skip I.D. words 1-26 (unimportant).
I_VC1:	in	ax,dx
	loop	I_VC1
	mov	di,(CDName-@)	;Point to drive "name" buffer.
	push	cs
	pop	es
	mov	cl,20		;Read & swap words 27-46 into buffer.
I_VC2:	in	ax,dx		;(Manufacturer "name" of this drive).
	xchg	ah,al
	stosw
	loop	I_VC2
	mov	cl,7		;Skip I.D. words 47-52 (unimportant)
I_VC3:	in	ax,dx		;  and read I.D. word 53 into AX-reg.
	loop	I_VC3
	mov	UFlag,al	;Save UltraDMA "valid" flags.
	mov	cl,35		;Skip I.D. words 54-87 (unimportant)
I_VC4:	in	ax,dx		;  and read I.D. word 88 into AX-reg.
	loop	I_VC4
	mov	UMode.lb,ah	;Save posted UltraDMA "mode" value.
	mov	cl,167		;Skip all remaining I.D. data.
I_VC5:	in	ax,dx
	loop	I_VC5
	cmp	si,08500h	;Do device flags say "ATAPI CD/DVD"?
	je s	I_VC8		;Yes, see about UltraDMA use.
I_VC6:	stc			;Set carry flag on (error!).
I_VC7:	ret			;Exit.
I_VC8:	test	UFlag,004h	;Valid UltraDMA "mode" bits?
	jz s	I_VC9		;No, disable drive UltraDMA.
	cmp	UMode.lb,bl	      ;Can drive do mode 0 minimum?
	jne s	I_VC10		      ;Yes, display "Unit n:" msg.
I_VC9:	or	[bx+DMAAd-@].lb,1     ;Disable this drive's UltraDMA.
I_VC10:	mov	dx,(CDMsg-@)	      ;Display "CDn:  " message.
	call	I_Msg
	mov	dx,(PriMsg-@)	      ;Point to "Primary" message.
	test	[bx+IdeDA-@].lb,080h  ;Primary-channel drive?
	jnz s	I_VC11		      ;Yes, display "Primary" msg.
	mov	dx,(SecMsg-@)	      ;Point to "Secondary" message.
I_VC11:	call	I_Msg		   ;Display CD/DVD's IDE channel.
	mov	dx,(MstMsg-@)	   ;Point to "Master" message.
	cmp	USelB,SSEL	   ;Is this drive a "slave"?
	jnz s	I_VC12		   ;No, display "Master".
	mov	dx,(SlvMsg-@)	   ;Point to "Slave" message.
I_VC12:	call	I_Msg		   ;Display "Master" or "Slave".
	mov	si,(CDName+40-@)   ;Point to end of CD/DVD name.
I_VC13:	cmp	si,(CDName-@)	   ;Are we at the vendor-name start?
	je s	I_VC14		   ;Yes, vendor name is all spaces!
	dec	si		   ;Decrement vendor-name pointer.
	cmp	[si].lb,' '	   ;Is this name byte a space?
	je s	I_VC13		   ;No, continue scan for non-space.
	inc	si		   ;Skip non-space character.
	mov	[si].lw," ,"	   ;End disk name with comma/space.
	inc	si		   ;Skip comma and space.
	inc	si
I_VC14:	test	[bx+DMAAd-@].lb,1  ;Will this drive use "PIO mode"?
	jz s	I_VC15		   ;No, get drive's UltraDMA "mode".
	mov	[si].dwd,"OIP"	   ;Set "PIO" after drive name.
	add	si,3
	jmp s	I_VC18		;Go set message terminators.
I_VC15:	mov	cx,UMode	;Initialize UltraDMA "mode" scan.
	mov	di,(Modes-2-@)
I_VC16:	inc	di		;Advance to next UltraDMA "mode".
	inc	di
	shr	cx,1		;Will drive do next "mode"?
	jnz s	I_VC16		;Yes, keep scanning for maximum.
I_VC17:	mov	[si].dwd,"-ATA"	;Set "ATA-" after drive name.
	add	si,4
	mov	ax,[di]		;Set UltraDMA "mode" in message.
	mov	cl,00Fh
	and	cl,al
	call	I_HexA
I_VC18:	mov	[si].dwd,0240A0D2Eh  ;Set message terminators.
	mov	dx,(DNMsg-@)	     ;Display vendor name & "mode".
	call	I_Msg
	clc			;Reset carry (no errors) and exit.
	ret
;
; Subroutines to issue initialization "external" calls.
;
I_A20D:	mov	ah,006h		;"A20 local-disable" -- get XMS code.
I_XMS:	call	@XEntry		;XMS -- issue desired request.
	dec	ax		;Zero AX-reg. if success, -1 if error.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_VDS:	push	bx		;VDS -- save our BX-reg.
	mov	di,(VDSLn-@)	;Point to VDS parameter block.
	push	cs
	pop	es
	int	04Bh		;Execute VDS "lock" or "unlock".
	jmp s	I_MsgX		;Go reload our BX-reg. and exit.
I_In13:	mov	dl,HDUnit	;BIOS data -- set BIOS unit in DL-reg.
	int	013h		;Issue BIOS data interrupt.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_PCID:	push	bx		;Save PCI bus/device/function codes.
	push	si		;Save IDE address-table pointer.
	mov	al,00Ah		;Set "PCI doubleword" request code.
	call	I_In1A		;Get desired 32-bit word from PCI.
	pop	si		;Reload IDE address-table pointer.
	pop	bx		;Reload PCI bus/device/function.
	ret			;Exit.
I_In1A:	mov	ah,0B1h		;Issue PCI BIOS interrupt.
	int	01Ah
	jmp s	I_IntX		;Restore driver settings, then exit.
I_Msg:	push	bx		;Message -- save our BX-register.
	mov	ah,009h		;Issue DOS "display string" request.
	int	021h
I_MsgX:	pop	bx		;Reload our BX-register.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_In21:	int	021h		;General DOS request -- issue Int 21h.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_In2F:	int	02Fh		;"Multiplex" -- issue XMS/HMA request.
I_IntX:	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs
	pop	ds
	ret			;Exit.
;
; Subroutine to convert a 4-digit hex number to ASCII for messages.
;   At entry, the number is in the AX-reg., and the message pointer
;   is in the SI-reg.   At exit, the SI-reg. is updated and the CX-
;   reg. is zero.
;
I_Hex:	mov	cx,4		;Set 4-digit count.
I_HexA:	rol	ax,4		;Get next hex digit in low-order.
	push	ax		;Save remaining digits.
	and	al,00Fh		;Mask off next hex digit.
	cmp	al,009h		;Is digit 0-9?
	jbe s	I_HexB		;Yes, convert to ASCII.
	add	al,007h		;Add A-F offset.
I_HexB:	add	al,030h		;Convert digit to ASCII.
	mov	[si],al		;Store next digit in message.
	inc	si		;Bump message pointer.
	pop	ax		;Reload remaining digits.
	loop	I_HexA		;If more digits to go, loop back.
	ret			;Exit.
CODE	ends
	end
