/*
   DISKCOPY.EXE, floppy diskette duplicator similar to MSDOS Diskcopy.
   Copyright (C) 1998, Matthew Stanford.
   Copyright (C) 1999, 2000, Imre Leber.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have recieved a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


   If you have any questions, comments, suggestions, or fixes please
   email me at:  ilebr@vub.ac.be

*/

#include <conio.h> 
#include <ctype.h>
#include <dos.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <io.h>
#include <errno.h>
#include <string.h>

#include "drive.h"
#include "diskcopy.h"
#include "memtypes.h"
#include "misc.h"
#include "waitfinp.h"
#include "tdrvcpy.h"
#include "critical.h"
#include "parser.h"
#include "exepath.h"

int SavedVerify, SavedCBreak;

int main(int argc, char *argv[])
{
   int           memkind;
   char          loop1, loop2;
   struct        dfree free;
   char          buf[BUFFERSIZE];
   unsigned long sector, total, disksize, allocated, beginsector, endsector;
   int           Arg1IsFile, Arg2IsFile;
   int           UseImageFile, ImageModus, Overwrite;
   int           i, enable;
   int           audible, verify, fallthrough, askdisk;
   char          sdrive, tdrive;
   int           HardDiskOk;
   int           informative;
   int           handle;
   char          switchchar;
   int           tocopysectors;
   unsigned      tobuffer;
#ifndef ALL_RECOVERY_MODE
   int           recoverymode;
#endif
   int           bytespersector;
   
   struct IniParserStruct*  ParsedData;
   
   memkind      = UNALLOCATED;
   UseImageFile = FALSE;

   SetExePath(argv[0]);             /* Remember executable's path.     */
   CriticalHandlerOn();             /* Install critical error handler. */

   switchchar   = SwitchChar();
   
   /* Check arguments */
   if ((argc == 2) && ((argv[1][0] == '/') || (argv[1][0] == switchchar)) &&
       (argv[1][1] == '?'))
   {
      ShowHelp (switchchar);
      return COPYSUCCESS;
   }

   if (argc > 2)
   {
      Arg1IsFile = IsFile(argv[1]);
      Arg2IsFile = IsFile(argv[2]);
   }

#ifdef DEF_HELP      
   if (argc < 3)
   {
      /* The way we do it.                */
      ShowHelp(switchchar);
      return INITERROR;
   }
   else if ((!HasFloppyForm(argv[1]) && !Arg1IsFile) ||
       (!HasFloppyForm(argv[2]) && !Arg2IsFile))
#else
   if ((argc < 3) || (!HasFloppyForm(argv[1]) && !Arg1IsFile)
        || (!HasFloppyForm(argv[2]) && !Arg2IsFile))
#endif
   {
      /* The way (DR and MS) DOS does it. */
      printf("Invalid drive specification or non removable media.\n");
      return INITERROR;
   }
   
   if (ParseIniFile(NULL) < 0)              /* Parse .ini file. */
   {
      printf("Error reading configuration file.\n");
      return INITERROR;
   }

   ParsedData = GetParsedData();
   
   audible      = ParsedData->audible;
   verify       = ParsedData->verify;
   HardDiskOk   = ParsedData->UseSWAP;
   informative  = ParsedData->informative;
   Overwrite    = ParsedData->overwrite;
   fallthrough  = ParsedData->autoexit;
   askdisk      = ParsedData->askdisk;
#ifndef ALL_RECOVERY_MODE   
   recoverymode = ParsedData->mode;
#endif
   
   /* Check arguments. */
   for (i = 3; i < argc; i++)
   {
       if (strlen(argv[i]) > 3)
       {   
          printf ("Invalid switch: %s\n", argv[i]);
          return INITERROR;
       }
     
       enable = TRUE;
       if (strlen(argv[i]) == 3)
       {
          if (argv[i][2] == '-')
          {
             enable = FALSE;
          }
          else
          {
             printf ("Invalid switch: %s\n", argv[i]);
             return INITERROR;
          }
       }
       
       if ((argv[i][0] == switchchar) || (argv[i][0] == '/'))
          switch (argv[i][1])
          {    
            case 'a':                      /* DOS is case insensitive. */
            case 'A':
                     audible = enable;
                     break;
            case 'v':
            case 'V':
                     verify = enable;
                     break;
            case 'm':
            case 'M':
                     HardDiskOk = !enable;
                     break;
            case 'i':
            case 'I':
                     informative = enable;
                     break;
            case 'o':
            case 'O':
                     Overwrite = enable;
                     break;
            case 'x':
            case 'X':
                     fallthrough = enable;
                     break;
             case 'd':
             case 'D':
                      askdisk = !enable;
                      break;

#ifndef ALL_RECOVERY_MODE
             case 'r':
             case 'R':
                       recoverymode = enable;
                       break;
#else
             case 'r':
             case 'R':
#endif
             case '1':
                      printf("Warning: option %s doesn't do anything!\n", 
                             argv[i]);
                       break;

             default:
                      printf ("Invalid switch: %s\n", argv[i]);
                      return INITERROR;
          }
       else
       {
          printf("Too many parameters: %s\n", argv[i]);
          return INITERROR;
       }
   }

   if (Arg1IsFile && Arg2IsFile)
   {
      if ((access(argv[2], EXISTS) == 0) &&
          Overwrite                      &&
          (remove(argv[2]) == -1))
      {
         puts("File is write protected!");
         return INITERROR;
      }

      if (CopyFile(argv[1], argv[2]))
      {
         printf("%s copied to %s.\n", argv[1], argv[2]);
         if (audible) Beep();
         return COPYSUCCESS;
      }
      else
      {
         printf("Problem copying %s to %s.\n", argv[1], argv[2]);
         return CRITICAL;
      }
   }
   else if (Arg1IsFile)
   {
      if (access(argv[1], READPERMISSION) != 0)
      {
         printf("File not found: %s\n", argv[1]);
         return INITERROR;
      }
      UseImageFile = TRUE;
      ImageModus   = READIMAGE;
   }
   else if (Arg2IsFile)
   {
      if (access(argv[2], EXISTS) == 0)
      {
         if (Overwrite)
         {
            if (remove(argv[2]) != 0)
            {
               printf("File is write protected!");
               return INITERROR;
            }
         }
         else
         {
            printf("File already exists!");
            return INITERROR;
         }
      }
      UseImageFile = TRUE;
      ImageModus   = WRITEIMAGE;
   }

   sdrive = (char) toupper(argv[1][0]) - 65;
   tdrive = (char) toupper(argv[2][0]) - 65;

   loop1 = 'Y';    /* initialize loop1         */
   loop2 = 'Y';    /* initialize loop2         */

   ctrlbrk(OnCBreak);
   SavedVerify = getverify();
   setverify (verify);
   SavedCBreak = getcbrk();
   setcbrk(1);     /* set control-break to ON  */
   atexit(OnExit); /* make sure that all allocated memory is released when
                       program stops. */
   
   if ((sdrive != tdrive) && (!UseImageFile) && BiosReportsTwoDrives()) 
   {
     TwoDriveCopy(sdrive, tdrive, audible, fallthrough, askdisk 
#ifndef ALL_RECOVERY_MODE
        ,recoverymode
#endif
        );

     return COPYSUCCESS;
   }

   endsector  = 0;
   while (loop1 != 'N') /* loop1 */
   {
         if (!UseImageFile || (ImageModus == WRITEIMAGE))
         {
            if (askdisk)
            {
               printf("\nInsert SOURCE diskette into drive %s\n", argv[1]);
               printf("\nPress any key to continue . . .");
               WaitForInput();
            }
            else 
               askdisk = TRUE; /* Always ask after the first run. */

            if (endsector == 0)
            {
               getdfree (sdrive+1, &free);

               if (CriticalErrorOccured())
               {
                  puts ("\nDisk not ready!");
                  return CRITICAL;
               }
               
               free.df_total = GetFullClusterCount(sdrive+1);
               if (free.df_total == 0)
               {
                  printf("\nInvalid drive specification or non removable media.\n");
                  return INITERROR;
               }
               total          = free.df_total * free.df_sclus;
               bytespersector = free.df_bsec;
               disksize       = total * bytespersector;
            }
      }
      else
      {
         handle = open(argv[1], O_RDONLY|O_BINARY);
         if (handle < 0)
         {
            printf("\nUnable to open image file.\n");
            return INITERROR;
         }
         disksize = filelength(handle);
         close(handle);
      }
      puts("");

      if (UseImageFile)
         switch(SetImageFile(argv[ImageModus], ImageModus, disksize))
         {
            case EZERO:
                 allocated = disksize;
                 break; 

            case DISKTOSMALL:
                 printf("\nNot enough disk space on target drive!\n");
                 return INITERROR;

            default:
                 printf ("Error accessing image file: %s\n", argv[ImageModus]);
                 return INITERROR;
         }
      else if ((memkind = InitializeFittingMemory (disksize, HardDiskOk, &allocated))
                                   == 0)
      {
        puts("Insufficient memory for disk copy.\n");
        return INITERROR;
      }

      if (!UseImageFile || (ImageModus == WRITEIMAGE))
      {
         beginsector  =  endsector;
         endsector   +=  allocated / BYTESPERSECTOR;
         if (endsector > total) endsector = total;
      }

      if (!UseImageFile || (ImageModus == WRITEIMAGE))
         printf("\nCopying %d clusters, %d sectors per cluster, %d bytes per sector.\n"
                "Drive size is %ld bytes.",
                free.df_total, free.df_sclus, free.df_bsec, disksize);

      if (informative && (!UseImageFile))
      {
         printf(" Using ");

         switch(memkind)
         {
           case EMS:      puts("EMS.\n");                                break;
           case XMS:      puts("XMS.\n");                                break;
           case HARDDISK: puts("swap file.\n");                          break;
           case BUFFERS:  printf("buffer of %ld bytes.\n\n", allocated); break;
         }
      }
      else if (!UseImageFile || (ImageModus == WRITEIMAGE))
         puts("\n");

      if (!UseImageFile || (ImageModus == WRITEIMAGE))
      {
         if (!UseImageFile)
            printf("Reading SOURCE diskette . . .");
         else
            printf("Creating image file . . .");

         PrepareForWriting();
         for (sector = beginsector; sector < endsector;
                                             sector = sector + TOCOPYSECTORS)
         {
             if (sector < endsector - TOCOPYSECTORS)
                tocopysectors = TOCOPYSECTORS;
             else
                tocopysectors = (int) (endsector - sector);
              tobuffer = (unsigned) (tocopysectors * BYTESPERSECTOR);

#ifndef ALL_RECOVERY_MODE
              if (recoverymode)
#endif              
                 ReadSectors(sdrive, tocopysectors, (int) sector, buf, 
                             bytespersector);
#ifndef ALL_RECOVERY_MODE           
               else
                  if (absread(sdrive, tocopysectors, (int) sector, buf) != 0)
                     printf("\nMedia error reading from sector %ld.\n", sector);
#endif
               if (!WriteMemoryBlock(buf, tobuffer))
               {
                  puts("\nUnsuspected memory error.\n");
                  SetErrorStopped();
                  return CRITICAL;
               }
         }
         puts("");
      }

      if (audible && (!UseImageFile)) Beep();

      if (!UseImageFile || (ImageModus == READIMAGE))
      {
         while (loop2 != 'N') /* loop2 */
         {
               if (askdisk)
               {
                  if (!UseImageFile) puts("");
                  printf("Insert TARGET diskette into drive %s \n", argv[2]);
                  printf("\nPress any key to continue . . .");
                  WaitForInput();
               }
               else
                  askdisk = TRUE;

               /* Check disk capacity is the same as that of the original
                  diskette. */
               for (;;)
               {
                   getdfree (tdrive+1, &free);
                   free.df_total = GetFullClusterCount(tdrive+1);
                   total = free.df_total * free.df_sclus;
                   if (disksize == total * free.df_bsec) break;

                   if (CriticalErrorOccured())
                      puts("\nDisk not ready!");
                   else
                      puts("\nDiskette does not have the same capacity as the original.");
         
                   if (fallthrough) return NONFATAL;

                   printf("\nPut a diskette with the right capacity in drive %s, \n",
                            argv[2]);
                   puts("or press CTRL-C to cancel.");
                   WaitForInput();
               }

               if (UseImageFile)
               {
                  printf("\n\nCopying %d clusters, %d sectors per cluster, %d bytes per sector\n"
                         "Drive size is %ld bytes.\n",
                         free.df_total, free.df_sclus, free.df_bsec, disksize);

                  beginsector  =  endsector;
                  endsector   +=  allocated / BYTESPERSECTOR;
                  if (endsector > total) endsector = total;
               }

               if (UseImageFile) 
                  printf("\nWriting image file . . .");
               else
                  printf("\n\nWriting to TARGET diskette in drive . . .");

               PrepareForReading();
               for (sector = beginsector; sector < endsector;
                                             sector = sector + TOCOPYSECTORS)
               {
                   if (sector < endsector - TOCOPYSECTORS)
                      tocopysectors = TOCOPYSECTORS;
                   else
                      tocopysectors = (int) (endsector - sector);
                   tobuffer = (unsigned) (tocopysectors * BYTESPERSECTOR);

                   if (!ReadMemoryBlock(buf, tobuffer))
                   {
                      puts("\nUnsuspected memory error.");
                      SetErrorStopped();
                      return CRITICAL;
                   }

                   if (abswrite(tdrive, tocopysectors, (int) sector, buf) != 0)
                      printf("\nMedia error writing to sector %ld.\n", sector);
               }
               puts("");

               if (memkind != BUFFERS)
               {
                  loop2 = (fallthrough) ? 'N' : 'X'; 
                  while ((loop2 != 'N') && (loop2 != 'Y'))
                  {
                        ClrKbd();
                        if (audible) Beep();
                        printf("\nDo you want another copy of this ");
                        if (UseImageFile)
                        {
                           endsector = 0;
                           printf("image file (Y/N)?");
                        }
                        else
                           printf("disk (Y/N)?");

                        loop2 = toupper(getch());
                        puts("");
                  }
                  if (UseImageFile && (loop2 == 'Y')) puts("");
               }
               else
               {
                  puts("");
                  loop2 = 'N';
               }

          } /*  loop2 */

          loop2 = 'X';
          ReleaseMemory();
      }

      loop1 = 'X'; 
      if (!UseImageFile && !fallthrough && (endsector == total))
      {
         while ((loop1 != 'N') && (loop1 != 'Y'))
         {
               ClrKbd();
               printf("\nCopy another disk (Y/N)?");
               loop1 = toupper(getch());
               puts("");
               endsector = 0;
         }
      }
      else if (UseImageFile || (fallthrough && (endsector == total))) 
      {
           loop1 = 'N';
           if (fallthrough) 
              puts("");
           else
              if (audible && (ImageModus == WRITEIMAGE)) Beep();
      }

   } /* end loop1 */

   return COPYSUCCESS;
}

void ShowHelp (char switchchar)
{
   printf ("Diskcopy %s\nCopy one diskette or image file to another diskette or image file.\n\n",
      VERSION);

   printf ("(C) 1998 by Matthew Stanford.\n(C) 1999, 2000 by Imre Leber.\n\n");

#ifdef ALL_RECOVERY_MODE
   printf("Diskcopy <source> <destination> [%ca] [%cv] [%cm] [%ci] [%co] [%cx] [%cd] [%c1]\n"
#else          
   printf("Diskcopy <source> <destination> [%ca] [%cv] [%cm] [%ci] [%co] [%cx] [%cd] [%cr] [%c1]\n"
#endif
          "source:      drive or image file to copy from.\n"
          "destination: drive or image file to copy to.\n"
          "\n" 
          "%ca : give an audible warning for user action.\n"
          "%cv : verify writes.\n"
          "%cm : only use memory for disk copy.\n"
          "%ci : show memory usage (informative).\n"
          "%co : overwrite destination, if it already exists (in case of an image file).\n"
          "%cx : always automaticaly exit.\n"
          "%cd : assume disk already in drive.\n"
#ifndef ALL_RECOVERY_MODE          
          "%cr : go into disk error recovery mode.\n"
#endif
          "%c1 : no-op, only included for MS-DOS compatibility.\n"
          "\n"
          "Remarks: a minus sign after an option disables the option.\n"
          "         Source and destination must be the same size.\n"
          "         You may specify the same drive for source and destination.\n",
          switchchar,
          switchchar,
          switchchar,
          switchchar,
          switchchar,
          switchchar,
          switchchar,
          switchchar,
          switchchar,
          switchchar,
          switchchar,
          switchchar,
          switchchar,
          switchchar,
          switchchar,
          switchchar,
          switchchar,
          switchchar);
}

void OnExit (void)
{
   ReleaseMemory();
   setcbrk   (SavedCBreak);
   setverify (SavedVerify);
}

int OnCBreak (void)
{
  OnExit();
  return ABORT;
}

