
/************************************************************************/
/*                                                                      */
/* FILE: intcmds.c                                                      */
/* Miscellaneous internal commands                                      */
/*                                                                      */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* This file is part of G-COM.                                          */
/* G-COM is derived from DOS-C source (GPL).                            */
/*                                                                      */
/* (C) Copyright 1999-2000  Roberto Gordo Saez   (GCOM)                 */
/* (C) Copyright 1995-1998  Pasquale J. Villani  (DOSC)                 */
/*                                                                      */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2, or (at your option)  */
/* any later version.                                                   */
/*                                                                      */
/* This program is distributed in the hope that it will be useful, but  */
/* WITHOUT ANY WARRANTY; without even the implied warranty of           */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU     */
/* General Public License for more details.                             */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to the Free Software          */
/* Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.                  */
/*                                                                      */
/************************************************************************/


#include "globals.h"


BOOL cmd_beep (char*, int, char*);
BOOL cmd_break (char*, int, char*);
BOOL cmd_cd (char*, int, char*);
BOOL cmd_echo (char*, int, char*);
BOOL cmd_echo_dot (char*, int, char*);
BOOL cmd_path (char*, int, char*);
BOOL cmd_pause (char*, int, char*);
BOOL cmd_set (char*, int, char*);
BOOL cmd_unset (char*, int, char*);
BOOL cmd_ver (char*, int, char*);
BOOL cmd_verify (char*, int, char*);


/* -------------------------------------------------------------------- */
/* cmd_beep - BEEP command                                              */
/* -------------------------------------------------------------------- */

BOOL cmd_beep(char *cmd_name, int argc, char *arguments)
{
    DosWrite(STDERR, (char FAR *)"\a", 1);
    return TRUE;
}


/* -------------------------------------------------------------------- */
/* cmd_break - BREAK command                                            */
/* -------------------------------------------------------------------- */

BOOL cmd_break(char *cmd_name, int argc, char *arguments)
{
    if(argc > 1)
    {
        error_message(ON_OFF);
        return FALSE;
    }

    if(*arguments != '\0')
    {
        if(stricmp(arguments, "ON"))
            DosSetBreak(TRUE);
        else if(stricmp(arguments, "OFF"))
            DosSetBreak(FALSE);
        else
        {
            error_message(ON_OFF);
            return FALSE;
        }
        return TRUE;
    }

    DosWrite(STDOUT, (char FAR *)"BREAK is ", 9);
    DosWrite(STDOUT, DosGetBreak() ? (char FAR *)"ON " : (char FAR *)"OFF", 3);
    DosWrite(STDOUT, (char FAR *)newline, 2);

    return TRUE;
}


/* -------------------------------------------------------------------- */
/* cmd_cd - CD command                                                  */
/* -------------------------------------------------------------------- */

BOOL cmd_cd(char *cmd_name, int argc, char *arguments)
{
    int OldDrive, NewDrive = -1;
    char directory[MAX_CMDLINE];

    if(argc > 1)
    {
        error_message(INV_NUM_PARAMS);
        return FALSE;
    }

    OldDrive = DosGetDrive();

    *directory = '\0';
    parsing("DP", (char FAR *)arguments, &NewDrive, directory);

    if(NewDrive < 0)
        NewDrive = default_drive;
    else
        DosSetDrive(NewDrive);

    /* Do pwd function for no path case */
    if(*directory == '\0')
    {
        DosPwd(NewDrive + 1, (char FAR*)directory);
        printf("%c:\\%s\n\r", 'A' + NewDrive, directory);
        DosSetDrive(OldDrive);
        return TRUE;
    }

    if(DosCd((char FAR *)directory) != DE_SUCCESS)
    {
        error_message(INV_DIR);
        DosSetDrive(OldDrive);
        return FALSE;
    }

    DosSetDrive(OldDrive);
    return TRUE;
}


/* -------------------------------------------------------------------- */
/* cmd_echo - ECHO command                                              */
/* -------------------------------------------------------------------- */

BOOL cmd_echo(char *cmd_name, int argc, char *arguments)
{
    if(argc == 0)
    {
        DosWrite(STDOUT, (char FAR *)"ECHO is ", 8);
        DosWrite(STDOUT, (char FAR *)echo_flag ? "ON " : "OFF", 3);
        DosWrite(STDOUT, (char FAR *)newline, 2);
        return TRUE;
    }
    else if(argc == 1)
    {
        if(stricmp(arguments, "ON"))
        {
            echo_flag = TRUE;
            return TRUE;
        }
        else if(stricmp(arguments, "OFF"))
        {
            echo_flag = FALSE;
            return TRUE;
        }
    }

    for(--arguments; issep(*arguments); --arguments);
    arguments+=2;
    DosWrite(STDOUT, (char FAR *)arguments, strlen(arguments));
    DosWrite(STDOUT, (char FAR *)newline, 2);

    return TRUE;
}


BOOL cmd_echo_dot(char *cmd_name, int argc, char *arguments)
{
    char *pointer = arguments;

    for(--pointer; *pointer != '.'; --pointer);

    pointer++;
    if(*pointer != '\0')
        DosWrite(STDOUT, (char FAR *)pointer, strlen(pointer));
    DosWrite(STDOUT, (char FAR *)newline, 2);

    return TRUE;
}


/* -------------------------------------------------------------------- */
/* cmd_path - PATH command                                              */
/* -------------------------------------------------------------------- */

BOOL cmd_path(char *cmd_name, int argc, char *arguments)
{
    char FAR *p_env;
    char *PATH = "PATH";

    if(argc == 0)
    {
        p_env = fskipwh(EnvLookup(PATH));
        if(*p_env == '\0')
            DosWrite(STDOUT, (char FAR *)"No PATH\n\r", 9);
        else
        {
            DosWrite(STDOUT, (char FAR *)PATH, 4);
            DosWrite(STDOUT, (char FAR *)"=", 1);
            DosWrite(STDOUT, p_env, fstrlen(p_env));
            DosWrite(STDOUT, (char FAR *)newline, 2);
        }
    }
    else if(argc == 1)
    {
        if(*arguments == ';')
        {
            if(*(arguments+1) == '\0')
                EnvClearVar(PATH);
            else
            {
                error_message(PARAM_FORMAT);
                return FALSE;
            }
        }
        else
                EnvSetVar(PATH,arguments);
    }
    else
    {
        error_message(INV_NUM_PARAMS);
        return FALSE;
    }

    return TRUE;
}


/* -------------------------------------------------------------------- */
/* cmd_pause - PAUSE command                                            */
/* -------------------------------------------------------------------- */

BOOL cmd_pause(char *cmd_name, int argc, char *arguments)
{
    if(argc == 0)
        DosWrite(STDOUT, (char FAR *)"Press any key to continue . . .\n\r", 33);
    else
    {
        DosWrite(STDOUT, (char FAR *)arguments, strlen(arguments));
        DosWrite(STDOUT, (char FAR *)newline, 2);
    }

    DosGetChar();
    return TRUE;
}


/* -------------------------------------------------------------------- */
/* cmd_set - SET command                                                */
/* -------------------------------------------------------------------- */

BOOL cmd_set(char *cmd_name, int argc, char *arguments)
{
    char *p, *end;
    char FAR *str;
    char *PROMPT = "PROMPT";

    if(argc == 0)
    {
        EnvDump();
        return TRUE;
    }

    for(p = arguments; !issep(*p) && *p != '='; p++);
    end = p;
    p = skipwh(p);

    if(*p != '=')
    {
        if(*p == '\0')
        {
            *end = '\0';
            str = EnvLookup(arguments);
            DosWrite(STDOUT, str, fstrlen(str));
            DosWrite(STDOUT, (char FAR *)newline, 2);
            return TRUE;
        }
        error_message(INV_SYNTAX);
        return FALSE;
    }

    *end = '\0';
    if(*++p == '\0')
    {
        /* Clear variable in environment */
        EnvClearVar(arguments);

        /* Update system PROMPT immediately */
        if(stricmp(arguments, PROMPT))
            strcpy(prompt_string, default_prompt);
    }
    else
    {
        EnvSetVar(arguments, p);

        /* Update system PROMPT immediately */
        if(stricmp(arguments, PROMPT))
        {
            if(!strncpy(prompt_string, p, MAX_CMDLINE))
                error_message(CMDLINE_LONG);
        }
    }

    return TRUE;
}


/* -------------------------------------------------------------------- */
/* cmd_unset - UNSET command                                            */
/* -------------------------------------------------------------------- */

BOOL cmd_unset(char *cmd_name, int argc, char *arguments)
{
    char *PROMPT = "PROMPT";

    if(argc != 1)
    {
        error_message(INV_NUM_PARAMS);
        return FALSE;
    }

    /* Clear variable in environment */
    EnvClearVar(arguments);

    /* Update system PROMPT immediately */
    if(stricmp(arguments, PROMPT))
        strcpy(prompt_string, default_prompt);

    return TRUE;
}


/* -------------------------------------------------------------------- */
/* cmd_ver - VER command                                                */
/* -------------------------------------------------------------------- */

BOOL cmd_ver(char *cmd_name, int argc, char *arguments)
{
    BOOL xflag = FALSE;

    parsing("[X]", (char FAR *)arguments, &xflag);

    DosWrite(STDOUT, (char FAR *)newline, 2);

    if(xflag)
    {
        DosWrite(STDOUT, (char FAR *)str_line, 68);
        printf(announce, (unsigned int)MAJOR_RELEASE, (unsigned int)MINOR_RELEASE);
        DosWrite(STDOUT, (char FAR *)str_line, 68);
        DosWrite(STDOUT, (char FAR *)copyright, strlen(copyright));
        DosWrite(STDOUT, (char FAR *)newline, 2);
        DosWrite(STDOUT, (char FAR *)str_line, 68);
    }
    else
        printf(version, (unsigned int)MAJOR_RELEASE, (unsigned int)MINOR_RELEASE);

    return TRUE;
}


/* -------------------------------------------------------------------- */
/* cmd_verify - VERIFY command                                          */
/* -------------------------------------------------------------------- */

BOOL cmd_verify(char *cmd_name, int argc, char *arguments)
{
    if(argc > 1)
    {
        error_message(ON_OFF);
        return FALSE;
    }

    if(*arguments != '\0')
    {
        if(stricmp(arguments, "on"))
            DosSetVerify(TRUE);
        else if(stricmp(arguments, "off"))
            DosSetVerify(FALSE);
        else
        {
            error_message(ON_OFF);
            return FALSE;
        }
        return TRUE;
    }

    DosWrite(STDOUT, (char FAR *)"VERIFY is ", 10);
    DosWrite(STDOUT, DosGetVerify() ? (char FAR *)"ON " : (char FAR *)"OFF", 3);
    DosWrite(STDOUT, (char FAR *)newline, 2);

    return TRUE;
}
