/*
 *  BATCH.C - batch file processor for COMMAND.COM.
 *
 *  Comments:
 *
 * ??/??/?? (Evan Jeffrey)
 *   started.
 *
 * 15 Jul 1995 (Tim Norman)
 *   modes and bugfixes.
 *
 * 08 Aug 1995 (Matt Rains)
 *   i have cleaned up the source code. changes now bring this source
 *   into guidelines for recommended programming practice.
 *
 *   i have added some constants to help making changes easier.
 *
 * 29 Jan 1996 (Steffan Kaiser)
 *   made a few cosmetic changes
 *
 * 05 Feb 1996 (Tim Norman)
 *   changed to comply with new first/rest calling scheme
 *
 * 14 Jun 1997 (Steffen Kaiser)
 *   bug fixes.  added error level expansion %?.  ctrl-break handling
 *
 * 16 Jul 1998 (Hans B Pufal)
 *   Totally reorganised in conjunction with COMMAND.C (cf) to implement
 *   proper BATCH file nesting and other improvements.
 *
 * 16 Jul 1998 (John P Price <linux-guru@gcfl.net>)
 *   Seperated commands into individual files.
 *
 * 19 Jul 1998 (Hans B Pufal) [HBP_001]
 *   Preserve state of echo flag across batch calls.
 *
 * 19 Jul 1998 (Hans B Pufal) [HBP_002]
 *   Implementation of FOR command
 *
 * 20-Jul-1998 (John P Price <linux-guru@gcfl.net>)
 *  added error checking after malloc calls
 *
 * 27-Jul-1998 (John P Price <linux-guru@gcfl.net>)
 * - added config.h include
 *
 * 02-Aug-1998 (Hans B Pufal) [HBP_003]
 *  Fixed bug in ECHO flag restoration at exit from batch file
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <io.h>
#include <dos.h>
#include <dir.h>
#include <conio.h>

#include "command.h"
#include "batch.h"

struct bcontext
 *bc = NULL;                    /*  The stack of current batch contexts.
                                 * NULL when no batch is active
                                 */

unsigned int echo = 1;          /* The echo flag */

#define BUFFSIZE  512
char
  textline[BUFFSIZE];           /* Buffer for reading Batch file lines */

char *find_arg(int n)
{
/* Returns a pointer to the n'th parameter of the current batch file.
 * If no such parameter exists returns pointer to empty string.
 * If no batch file is current, returns NULL
 *
 */

  char *pp;

  dprintf(("[find_arg (%d)]\n", n));

  if (bc == NULL)
    return NULL;

  n += bc->shiftlevel;
  pp = bc->params;

  while (*pp && n--)            /* Step up the strings till we reach the end
                                 * or the one we want */
    pp += strlen(pp) + 1;

  return pp;
}

/* HBP_002 { FOR command support */

char *batch_params(char *s1, char *s2)
{
  /*
     * Batch_params builds a parameter list in newlay allocated memory.
     * The parameters consist of null terminated strings with a final
     * NULL character signalling the end of the parameters.
     *
   */

  char
   *dp = (char *)malloc(strlen(s1) + strlen(s2) + 3);

  /* JPP 20-Jul-1998 added error checking */
  if (dp == NULL)
  {
    error_out_of_memory();
    return NULL;
  }

  if (s1 && *s1)
  {
    s1 = stpcpy(dp, s1);
    *s1++ = '\0';
  }
  else
    s1 = dp;

  while (*s2)
  {
    if (isspace(*s2) || strchr(",;", *s2))
    {
      *s1++ = '\0';
      s2++;
      while (*s2 && strchr(" ,;", *s2))
        s2++;
      continue;
    }

    if ((*s2 == '"') || (*s2 == '\''))
    {
      char
        st = *s2;

      do
        *s1++ = *s2++;
      while (*s2 && (*s2 != st));
    }

    *s1++ = *s2++;
  }

  *s1++ = '\0';
  *s1 = '\0';

  return dp;
}

/* HBP_002 } */

void exit_batch(char *msg)
{
/*
 * If a batch file is current, exits it, freeing the context block and
 * chaining back to the previous one.
 *
 * If no new batch context is found, sets ECHO back ON.
 *
 * If the parameter is non-null or not empty, it is printed as an exit
 * message
 */

  dprintf(("exit_batch (..)\n"));

  if (bc)
  {
    struct bcontext
     *t = bc;

    if (bc->bfile)
      fclose(bc->bfile);

    if (bc->params)
      free(bc->params);

/* HBP_002 { FOR command support */

    if (bc->forproto)
      free(bc->forproto);

    if (bc->ffind)
      free(bc->ffind);

/* HBP_002 } */

/* HBP_003 { fix echo restore */

    echo = bc->echo;              /* Preserve echo state across batch calls */

/* HBP_003 fix echo restore } */

    bc = bc->prev;
    free(t);
  }

/* HBP_001 } */

  if (msg && *msg)
    printf("%s\n", msg);
}

int batch(char *fullname, char *firstword, char *param)
{
  /*
   * Start batch file execution
   *
   * The firstword parameter is the full filename of the batch file.
   *
   */

  FILE *bf = fopen(fullname, "rt");

  dprintf(("batch ('%s', '%s', '%s')  bf = %x\n", fullname, firstword,
           param, bf));

  if (bf == NULL)
  {
    perror("opening batch file");
    return 1;
  }

/* HBP_002 { FOR command support */

  while (bc && bc->forvar)      /* Kill any and all FOR contexts */
    exit_batch(NULL);

/* HBP_002 } */

  if (bc == NULL)               /* No curent batch file, create a new context */
  {
    struct bcontext
     *n = (struct bcontext *)malloc(sizeof(struct bcontext));

    /* JPP 20-Jul-1998 added error checking */
    if (n == NULL)
    {
      error_out_of_memory();
      return 1;
    }
    n->prev = bc;
    bc = n;
  }
  else if (bc->bfile)           /* Then we are transferring to another batch */
  {
    fclose(bc->bfile);
    free(bc->params);
  }

  bc->bfile = bf;
  bc->echo = echo;              /* Preserve echo across batch calls [HBP_001] */
  bc->shiftlevel = 0;

/* HBP_002 { FOR command support */
  bc->ffind = NULL;
  bc->forvar = '\0';
  bc->forproto = NULL;
  bc->params = batch_params(firstword, param);

/* HBP_002 } */

  return 0;
}

char *readbatchline(int *eflag)
{
  /*
   * Read and return the next executable line form the current batch file
   *
   * If no batch file is current or no further executable lines are found
   * return NULL.
   *
   * Here we also look out for FOR bcontext structures which trigger the
   * FOR expansion code.
   *
   * Set eflag to 0 if line is not to be echoed else 1
   */

  char
   *first,
   *ip;

  if (bc == NULL)               /* No batch */
    return NULL;

  dprintf(("readbatchline ()\n"));

  while (1)
  {
    if (chkCBreak(BREAK_BATCHFILE)) /* User halt */
    {
      while (bc)
        exit_batch(NULL);
      return NULL;
    }

    /* HBP_002 { FOR command support */

    if (bc == NULL)             /* No batch */
      return NULL;

    if (bc->forvar)             /* If its a FOR context... */
    {
      char
       *sp = bc->forproto,      /* pointer to prototype command */
       *dp = textline,          /* Place to expand protoype */
       *fv = find_arg(0);       /* Next list element */

      if ((fv == NULL) || (*fv == '\0'))  /* End of list so... */
      {
        exit_batch(NULL);       /* just exit this context */
        continue;
      }

      if (strcspn(fv, "?*") == strlen(fv))  /* element is wild file */
        bc->shiftlevel++;       /* No use it and shift list */
      else
        /* Wild file spec, find first (or next) file name */
      {
        if (bc->ffind)          /* First already done fo do next */
          fv = !findnext(bc->ffind) ? bc->ffind->ff_name : NULL;
        else
          /*  For first find, allocate a find first block */
        {
          if ((bc->ffind = (struct ffblk *)malloc(sizeof(struct ffblk))) == NULL)
          {
            error_out_of_memory();  /* JPP 20-Jul-1998 added error checking */
            return NULL;
          }

          fv = !findfirst(fv, bc->ffind, FA_NORMAL) ?
              bc->ffind->ff_name : NULL;
        }

        if (fv == NULL)         /* Null indicates no more files.. */
        {
          free(bc->ffind);      /* free the buffer */
          bc->ffind = NULL;
          bc->shiftlevel++;     /* On to next list element */
          continue;
        }
      }

      /* At this point, fv points to parameter string */

      while (*sp)
      {
        if ((*sp == '%') && (*(sp + 1) == bc->forvar))  /* replace % var */
          dp = stpcpy(dp, fv), sp += 2;
        else
          *dp++ = *sp++;        /* Else just copy */
      }

      *dp = '\0';

      *eflag = echo;

      return textline;
    }

    /* HBP_002 } */

    if (fgets(textline, BUFFSIZE, bc->bfile) == NULL) /* End of file.... */
    {
      exit_batch(NULL);

      if (bc == NULL)
        return NULL;

      continue;
    }

    /* Strip leading spaces and trailing space/control chars */

    for (first = textline; isspace(*first); first++)
      ;

    for (ip = first + strlen(first) - 1; isspace(*ip) || iscntrl(*ip);
         ip--)
      ;

    *++ip = '\0';

    /* ignore labels and empty lines */

    if (*first == ':' || *first == 0)
      continue;

    if (*first == '@')          /* don't echo this line */
    {
      do
        first++;
      while (isspace(*first));

      *eflag = 0;
    }
    else
      *eflag = echo;

    break;
  }

  return first;
}
