/*
 *  DIR.C - dir internal command
 *
 *  Comments:
 *
 *  01/29/97 (Tim Norman) ---------------------------------------------------
 *    started.
 *
 *  06/13/97 (Tim Norman) ---------------------------------------------------
 *    Fixed code.
 *
 *  07/12/97 (Tim Norman) ---------------------------------------------------
 *    Fixed bug that caused the root directory to be unlistable
 *
 *  07/12/97 (Marc Desrochers) ----------------------------------------------
 *    Changed to use maxx, maxy instead of findxy()
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <dir.h>
#include <dos.h>
#include <io.h>
#include <conio.h>
#include <string.h>
#include "command.h"

/* error messages */
#define TOO_MANY_PARAMETERS "Too many parameters!\n"
#define INVALID_SWITCH "Invalid switch!\n"
#define PATH_NOT_FOUND "Path not found!\n"

/* useful macros */
#define MEM_ERR fprintf (stderr, "Memory error.\n"); return 1;

/* flag definitions */
enum
{
  DIR_RECURSE = 0x1,
  DIR_PAGE = 0x2
};

/*
 * dir_read_param
 *
 * read the parameters from the command line
 */
int
dir_read_param(char *line, char **param, int *flags)
{
  int slash = 0;

  *param = NULL;

  /* scan the command line, processing switches */
  while (*line)
  {
    /* process switch */
    if (*line == '/' || slash)
    {
      line++;
      slash = 0;
      if (toupper(*line) == 'R')
        *flags |= DIR_RECURSE;
      else if (toupper(*line) == 'P')
        *flags |= DIR_PAGE;
      else
      {
        fprintf(stderr, INVALID_SWITCH);
        return 1;
      }
      line++;
      continue;
    }

    /* process parameter */
    if (!isspace(*line))
    {
      if (*param)
      {
        fprintf(stderr, TOO_MANY_PARAMETERS);
        return 1;
      }

      /* skip to end of line or next whitespace or next / */
      *param = line;
      while (*line && !isspace(*line) && *line != '/')
        line++;

      /* if end of line, return */
      if (!*line)
        return 0;

      /* if parameter, remember to process it later */
      if (*line == '/')
        slash = 1;

      *line++ = 0;
      continue;
    }

    line++;
  }

  if (slash)
  {
    fprintf(stderr, INVALID_SWITCH);
    return 1;
  }

  return 0;
}

/*
 * extend_file
 *
 * extend the filespec, possibly adding wildcards
 */
void
extend_file(char **file)
{
  char *tmp;

  if (!*file)
    return;

  /* if no file spec, change to *.* */
  if (!**file)
  {
    free(*file);
    *file = strdup("*.*");
    return;
  }

  /* if starts with . add * in front */
  if (**file == '.')
  {
    tmp = malloc(strlen(*file) + 2);
    if (tmp)
    {
      *tmp = '*';
      strcpy(&tmp[1], *file);
    }
    free(*file);
    *file = tmp;
    return;
  }

  /* if no . add .* */
  if (strchr(*file, '.') == NULL)
  {
    tmp = malloc(strlen(*file) + 3);
    if (tmp)
    {
      strcpy(tmp, *file);
      strcat(tmp, ".*");
    }
    free(*file);
    *file = tmp;
    return;
  }
}

/*
 * dir_parse_pathspec
 *
 * split the pathspec into drive, directory, and filespec
 */
int
dir_parse_pathspec(char *pathspec, int *drive, char **dir, char **file)
{
  char *start,
   *tmp,
    orig_dir[128];
  int i,
    wildcards = 0;

  /* get the drive and change to it */
  if (pathspec[1] == ':')
  {
    *drive = toupper(pathspec[0]) - 'A';
    start = pathspec + 2;
    setdisk(*drive);
  }
  else
  {
    *drive = getdisk();
    start = pathspec;
  }

  getcwd(orig_dir, 128);

  /* check for wildcards */
  for (i = 0; pathspec[i]; i++)
    if (pathspec[i] == '*' || pathspec[i] == '?')
      wildcards = 1;

  /* check if this spec is a directory */
  if (!wildcards)
  {
    if (chdir(pathspec) == 0)
    {
      *file = strdup("*.*");
      if (!*file)
      {
        MEM_ERR
      }
      tmp = getcwd(NULL, 128);
      if (!tmp)
      {
        free(*file);
        chdir(orig_dir);
        MEM_ERR
      }
      *dir = strdup(&tmp[2]);
      free(tmp);
      if (!*dir)
      {
        free(*file);
        chdir(orig_dir);
        MEM_ERR
      }

      chdir(orig_dir);
      return 0;
    }
  }

  /* find the file spec */
  tmp = strrchr(start, '\\');

  /* if no path is specified */
  if (!tmp)
  {
    *file = strdup(start);
    extend_file(file);
    if (!*file)
    {
      MEM_ERR
    }

    tmp = getcwd(NULL, 128);
    if (!tmp)
    {
      free(*file);
      chdir(orig_dir);
      MEM_ERR
    }
    *dir = strdup(&tmp[2]);
    free(tmp);
    if (!*dir)
    {
      free(*file);
      chdir(orig_dir);
      MEM_ERR
    }

    return 0;
  }

  /* get the filename */
  *file = strdup(tmp + 1);
  extend_file(file);
  if (!*file)
  {
    MEM_ERR
  }

  *tmp = 0;

  /* change to this directory and get its full name */
  if (chdir(start) < 0)
  {
    fprintf(stderr, PATH_NOT_FOUND);
    *tmp = '\\';
    free(*file);
    chdir(orig_dir);
    return 1;
  }

  tmp = getcwd(NULL, 128);
  if (!tmp)
  {
    free(*file);
    MEM_ERR
  }
  *dir = strdup(&tmp[2]);
  free(tmp);
  if (!*dir)
  {
    free(*file);
    MEM_ERR
  }

  *tmp = '\\';

  chdir(orig_dir);
  return 0;
}

/*
 * pause
 *
 * pause until a key is pressed
 */
int
pause(void)
{
  int c;

  if ((isatty(0)) && (isatty(1)))
  {
    printf("Press any key to continue . . .");
    c = getch();
    printf("\n");
    if (c == 27 || c == 3)
    {
      return 1;
    }
    if (c == 0)
    {
      getch();
    }
  }
  return 0;
}

/*
 * incline
 *
 * increment our line if paginating, display message at end of screen
 */
int
incline(int *line, int flags)
{
  if (!(flags & DIR_PAGE))
    return 0;

  (*line)++;

  if (*line >= *maxy)
  {
    *line = 0;
    return pause();
  }

  return 0;
}

/*
 * dir_print_header
 *
 * print the header for the dir command
 */
int
dir_print_header(int drive, int *line, int flags)
{
  struct media_id
  {
    int info_level,
      serial1,
      serial2;
    char vol_id[11],
      file_sys[8];
  }
  media;
  struct ffblk f;
  struct SREGS s;
  union REGS r;

  /* change to the drive */
  setdisk(drive);
  if (getdisk() != drive)
  {
    fprintf(stderr, "Invalid drive specification\n");
    return 1;
  }

  /* get the media ID of the drive */
  media.info_level = 0;
  r.x.bx = drive + 'A' - '@';
  r.x.cx = 0x866;
  s.ds = FP_SEG(&media);
  r.x.dx = FP_OFF(&media);
  r.x.ax = 0x440d;
  int86x(0x21, &r, &r, &s);
  media.vol_id[10] = NULL;

  /* print drive info */
  printf(" Volume in drive %c", drive + 'A');

  if (findfirst("\\*.*", &f, FA_LABEL) == 0)
  {
    printf(" is %s\n", f.ff_name);
  }
  else
  {
    printf(" has no label\n");
  }
  if (incline(line, flags) != 0)
    return 1;

  /* print the volume serial number if the return was successful */
  if (!r.x.cflag)
  {
    printf(" Volume Serial Number is %04X-%04X\n", media.serial2, media.serial1);
    if (incline(line, flags) != 0)
      return 1;
  }

  return 0;
}

/*
 * convert
 *
 * insert commas into a number
 */
int
convert(long num, char *des)
{
  char temp[32];
  int c = 0,
    n = 0;

  if (num == 0)
  {
    des[0] = 0x30;
    des[1] = 0;
    n = 1;
  }
  else
  {
    temp[31] = 0;
    while (num > 0)
    {
      if (((c + 1) % 4) == 0)
        temp[30 - c++] = ',';
      temp[30 - c++] = (char)(num % 10) + 0x30;
      num /= 10;
    }
    for (n = 0; n <= c; n++)
      des[n] = temp[31 - c + n];
  }
  return n;
}

/*
 * dir_list
 *
 * list the files in the directory
 */
int
dir_list(int drive, char *directory, char *filespec, int *line,
         int flags)
{
  char pathspec[128],
   *ext,
    buffer[32];
  struct ffblk file;
  union REGS r;
  long bytecount = 0,
    filecount = 0,
    dircount = 0;
  int time;

  printf(" Directory of %c:%s\n", drive + 'A', directory);
  if (incline(line, flags) != 0)
    return 1;
  printf("\n");
  if (incline(line, flags) != 0)
    return 1;

  if (directory[strlen(directory) - 1] == '\\')
    sprintf(pathspec, "%c:%s%s", drive + 'A', directory, filespec);
  else
    sprintf(pathspec, "%c:%s\\%s", drive + 'A', directory, filespec);

  if (findfirst(pathspec, &file, FA_RDONLY | FA_ARCH | FA_DIREC) != 0)
  {
    printf("File not found.\n");
    incline(line, flags);
    return 1;
  }

  do
  {
    if (file.ff_name[0] == '.')
      printf("%-13s", file.ff_name);
    else
    {
      ext = strchr(file.ff_name, '.');
      if (!ext)
        ext = "";
      else
        *ext++ = 0;

      printf("%-8s %-3s ", file.ff_name, ext);
    }

    if (file.ff_attrib & FA_DIREC)
    {
      printf("%-14s", "<DIR>");
      dircount++;
    }
    else
    {
      convert(file.ff_fsize, buffer);
      printf("   %10s ", buffer);
      bytecount += file.ff_fsize;
      filecount++;
    }

    printf("%.2d-%.2d-%d", ((file.ff_fdate >> 5) & 0x000f),
           (file.ff_fdate & 0x001f), ((file.ff_fdate >> 9) + 80));
    time = file.ff_ftime >> 5 >> 6;
    printf("  %2d:%.2u%c\n",
           (time == 0 ? 12 : (time <= 12 ? time : time - 12)),
           ((file.ff_ftime >> 5) & 0x003f),
           (time <= 11 ? 'a' : 'p'));

    if (incline(line, flags) != 0)
      return 1;
  }
  while (findnext(&file) == 0);

  if (filecount || dircount)
  {
    /* print number of files and bytes */
    convert(filecount, buffer);
    printf("   %6s file%c", buffer, filecount == 1 ? ' ' : 's');
    convert(bytecount, buffer);
    printf("   %12s byte%c\n", buffer, bytecount == 1 ? ' ' : 's');
    if (incline(line, flags) != 0)
      return 1;

    /* print number of dirs and bytes free */
    printf("%9d dirs", dircount);
    r.h.ah = 0x36;
    r.h.dl = drive + 'A' - '@';
    int86(0x21, &r, &r);
    convert((long)r.x.ax * r.x.bx * r.x.cx, buffer);
    printf(" %15s bytes free\n", buffer);
    if (incline(line, flags) != 0)
      return 1;
  }
  else
  {
    printf("File not found\n");
    return 1;
  }

  printf("\n");
  if (incline(line, flags) != 0)
    return 1;

  return 0;
}

/*
 * dir
 *
 * internal dir command
 */
#pragma argsused
int
dir(char *first, char *rest)
{
  int flags = 0;
  char *param;
  int line = 0;
  int drive,
    orig_drive;
  char *directory,
   *filespec,
    orig_dir[128];

  /* read the parameters */
  if (dir_read_param(rest, &param, &flags) != 0)
    return 1;

  /* default to current directory */
  if (!param)
    param = ".";

  /* save the current directory info */
  orig_drive = getdisk();
  getcwd(orig_dir, 128);

  /* parse the directory info */
  if (dir_parse_pathspec(param, &drive, &directory, &filespec) != 0)
  {
    setdisk(orig_drive);
    chdir(orig_dir);
    return 1;
  }

  /* print the header */
  if (dir_print_header(drive, &line, flags) != 0)
    return 1;

  setdisk(orig_drive);
  chdir(orig_dir);

  if (dir_list(drive, directory, filespec, &line, flags) != 0)
    return 1;

  return 0;
}
