/*
 * MISC.C -- Misc. Functions
 *
 * 07/12/98 (Rob Lake)
 *  started
 *
 * 07/13/98 (Rob Lake)
 *  moved functions in here
 *
 * 27-Jul-1998 (John P Price <linux-guru@gcfl.net>)
 * - added config.h include
 *
 * 09-Aug-1998 (Rob Lake <rlake@cs.mun.ca>)
 * - changed split function
 * - added freep function
 *
 * 10-Aug-1998 ska
 * - fix someone removed the ^Break catcher --> readded, this time using
 *   ANSI-like (DOS emulation) of signal handling functions
 *
 * 09-Sep-1998 ska
 * - fix/chg: ^Break handler now handles external program properly
 *
 * 1998/12/04 ska
 * - chg: vcgetchar() moved here
 * - add: vcgetcstr() gets visual user input and validates it
 *
 * 1999/01/24 ska
 * bugfix: split(): the loop skipping over leading whitespaces crashed
 *	at end of string ('\0') (reported by Eric Kohl <ekohl@abo.rhein-zeitung.de>)
 */

#include "config.h"

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <dos.h>
#include <io.h>
#include <fcntl.h>
#include <direct.h>
#include <ctype.h>
#include <string.h>
#include <dir.h>
#include <sys/stat.h>
#include <signal.h>
#include "command.h"
#include "batch.h"

/*
 * exist -- Checks if a file exists
 *
 */
int exist(const char *fn)
{
  return (access(fn, 0) == 0);
}

/*
 * get a character out-of-band and honor Ctrl-Break characters
 */
int cgetchar(void)
{
  int c;

  if ((c = getch()) == 0)
    c = getch() << 8;

  if (c == 3)
    ctrlBreak = 1;

  return c;
}

/* Visible cgetchar() that does not advance the cursor */
int vcgetchar(void)
{
  int ch;

  ch = cgetchar();
  putchar(isprint(ch) ? ch : ' ');
  putchar('\b');
  fflush(stdout);

  return ch;
}

/*
 *  get a visual character that must match one of the supplied
 *  ones
 */
int vcgetcstr(const char *const legalCh)
{
  int ch;

  fflush(stdout);               /* Make sure the pending output arrives the
                                   screen as we bypass the stdio interface */
  while ((ch = toupper(vcgetchar())) == 0
         || !strchr(legalCh, ch))
  {
    if (cbreak)
    {
      ch = '\3';                /* This is ^C for ^Break */
      break;
    }
    beep();                     /* hit erroreous character */
  }
  putchar('\n');                /* advance to next line */
  return ch;
}

/*
 * Check if Ctrl-Break was pressed during the last calls
 */
int chkCBreak(int mode)
{
  static int leaveAll = 0;      /* leave all batch files */
  int c;

  switch (mode)
  {
    case BREAK_ENDOFBATCHFILES:
      leaveAll = 0;
      return 0;

    case 0:
      if (!bc)
        goto justCheck;

    case BREAK_BATCHFILE:
      if (leaveAll)
        return 1;
      if (!ctrlBreak)
        return 0;

      /* we need to be sure the string arrives on the screen! */
      do
        cprintf("\r\nCtrl-Break pressed.\r\nCancel batch file '%s'? (Yes/No/All) ", bc && bc->bfnam ? bc->bfnam : "");
      while (!strchr("YNA\3", c = toupper(cgetchar())) || !c);

      cputs("\r\n");

      if (c == 'N')
        return ctrlBreak = 0;   /* ignore */

      leaveAll = c == 'A' || c == '\3'; /* leave all batch files */

      break;

    justCheck:
    case BREAK_FORCMD:         /* FOR commands are part of batch processing */
      if (leaveAll)
        return 1;
      /* fall through */

    case BREAK_INPUT:
      if (!ctrlBreak)
        return 0;
      break;
  }

  ctrlBreak = 0;                /* state processed */
  return 1;
}

/*
 * function to destructively split a string into an array of strings
 *
 * Changed return value to int to determine number of args
 *   -- Rob Lake (06/29/98)
 *
 */

#if 0                           /* nested comment - ska */
int split(char *s, char **p)
{
  int sc = 0,
    pc = 0;                     /* string and parameter counters */

  while (s[sc])
  {
    if (sc && isspace(s[sc]) && !isspace(s[sc - 1]))
      s[sc] = 0;

    else if (!isspace(s[sc]) && (sc == 0 || isspace(s[sc - 1]) ||
                                 s[sc - 1] == 0))
      p[pc++] = &s[sc];

    sc++;
  }

  p[pc] = NULL;
  return pc;
}
#endif

/*
 * split - splits a line up into separate arguments, deliminators
 *      are spaces and /'s
 * Rewritten by: Rob Lake (rlake@cs.mun.ca)
 *      started 03/08/98
 * 1998/08/12 ska
 *                                                                              changed to use a dynamic array and perfectly fit */

char **split(char *s, int *args)
{
  char **arg,
  **p,
   *start,
   *q;
  unsigned ac,
    ch;
  int len;

  arg = malloc(sizeof(char *));
  if (!arg)
    return NULL;
  *arg = NULL;

  ac = 0;
  while ((ch = *s) != '\0')
  {
    while (isspace(ch = *s) || iscntrl(ch))     /* skip leading spaces */
      ++s;
    if(isspace(ch) || iscntrl(ch))   /* skip leading spaces */
    {
      ++s;
      continue;
    }
    start = s;
    if (ch == '/')              /* the first character can be '/' */
      ++s;
    /* skip to next word delimiter or start of next option */
    while (isprint(ch = *s) && !isspace(ch) && ch != '/')
      ++s;
    if (s != start)             /* a word was found */
    {
      /* add new entry for new argument */
      arg = realloc(p = arg, (ac + 2) * sizeof(char *));
      if (!arg)
      {
        freep(p);
        return NULL;
      }
      /* create new entry */
      q = arg[ac] = malloc((len = s - start) + 1);
      arg[++ac] = NULL;
      if (!q)
      {
        freep(arg);
        return NULL;
      }
      memcpy(q, start, len);
      q[len] = '\0';
    }
  }

  *args = ac;
  return arg;
}

/*
 * freep -- frees memory used for a call to split
 *
 */
void freep(char **p)
{
  char **q;

  q = p;
  while (*q)
    free(*q++);
  free(p);
}

/*
 * internal function to get the first argument from the parameter list
 * and return a pointer to the beginning of the next argument, or NULL if
 * there are no more arguments
 *
 */
char *parse_firstarg(char *s)
{
  char *place;

  /* skip over first argument */
  place = s;
  while (*place && !isspace(*place))
    place++;

  if (*place)
  {
    /* mark the end of the first parameter */
    *place++ = 0;

    /* skip over whitespace before next argument */
    while (isspace(*place))
      place++;

    /* if there is something here, return a pointer to it, else NULL */
    if (*place)
      return place;
    else
      return NULL;
  }
  else
    return NULL;
}

/*
 * Internally BEEP (getting the user's attention)
 * Note: One should consider implementing a _visual_ beep
 *   rather an audible beep and an option to switch between them.
 */
void beep(void)
{
  sound(900);
  delay(400);
  nosound();
  delay(100);
}

void beep_low(void)
{
  sound(900);
  delay(400);
  nosound();
  delay(100);
}


