/*
 * COPY.C -- Internal Copy Command
 *
 * 01-Aug-98 (Rob Lake z63rrl@morgan.ucs.mun.ca)
 *      - started
 *
 * 13-Aug-1998 (John P. Price)
 * - fixed memory leak problem in copy function.
 * - fixed copy function so it would work with wildcards in the source
 *
 */

#include <stdio.h>
#include <string.h>
#include <dir.h>
#include <dos.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <errno.h>
#include <io.h>
#include <fcntl.h>
#include <ctype.h>
#include <alloc.h>              //for coreleft()

#include "config.h"

#ifdef INCLUDE_CMD_COPY
/* #define DEBUG */
#include "command.h"
#include "strings.h"

#include "copy.h"

int isDir(char *fn)
{
  if (exist(fn) == 0)
    return 0;
  return _chmod(fn, 0) & FA_DIREC;
}

int do_switches(char *arg, unsigned *flags)
{
  char c;
  if (stricmp(arg, "/-Y") == 0)
  {
    *flags |= PROMPT;
    *flags &= ~NPROMPT;
    return 1;
  }
  else if (strlen(arg) > 2)
  {
    display_string(TEXT_ERROR_TOO_MANY_PARAMETERS);
    return 0;
  }
  c = toupper(arg[1]);
  switch (c)
  {
    case 'V':
      *flags |= VERIFY;
      break;
    case 'A':
      *flags |= ASCII;
      *flags &= ~BINARY;
      break;
    case 'B':
      *flags |= BINARY;
      *flags &= ~ASCII;
      break;
    case 'Y':
      *flags &= ~PROMPT;
      *flags |= NPROMPT;
      break;
    default:
      error_invalid_switch(arg[1]);
      return 0;
  }
  return 1;
}

int add_file(struct files *f, char *arg, int *source, int *dest,
             unsigned *flags)
{
  if (*dest)
  {
    display_string(TEXT_ERROR_TOO_MANY_PARAMETERS);
    return 0;
  }
  if (*source)
  {
    *dest = 1;
    f->flag = 0;
  }
  else
  {
    *source = 1;
    f->flag = SOURCE;
  }
  strcpy(f->file, arg);
  f->flag |= *flags & ASCII ? ASCII : BINARY;
  if ((f->next = (struct files *)malloc(sizeof(struct files))) == NULL)
  {
    error_out_of_memory();
    return 0;
  }
  f = f->next;
  f->flag = 0;
  f->next = NULL;
  return 1;
}

int add_files(struct files *f, char *arg, int *source, int *dest,
              int *count, unsigned *flags)
{
  char t[128];
  int j,
    k;

  if (*dest)
  {
    display_string(TEXT_ERROR_TOO_MANY_PARAMETERS);
    return 0;
  }

  j = 0;
  k = 0;

  while (arg[j] == '+')
    j++;

  while (arg[j] != '\0')
  {
    t[k] = arg[j++];
    if (t[k] == '+' || arg[j] == '\0')
    {
      if (!k)
        continue;
      if (arg[j] == '\0' && t[k] != '+')
        k++;
      t[k] = '\0';
      *count += 1;
      strcpy(f->file, t);
      *source = 1;
      if (*flags & ASCII)
        f->flag |= *flags | SOURCE | ASCII;
      else
        f->flag |= *flags | BINARY | SOURCE;
      if ((f->next = (struct files *)malloc(sizeof(struct files))) == NULL)
      {
        error_out_of_memory();
        return 0;
      }
      f = f->next;
      f->next = NULL;
      k = 0;
      f->flag = 0;
      continue;
    }
    k++;
  }
  if (arg[--j] == '+')
    *source = 0;

  return 1;
}

int get_dest(struct files *f, struct files *dest)
{
  struct files *p,
   *start = f;

  while (f->next != NULL)
  {
    p = f;
    f = f->next;
  }

  f = p;

  if ((f->flag & SOURCE) == 0)
  {
    free(p->next);
    p->next = NULL;
    strcpy(dest->file, f->file);
    dest->flag = f->flag;
    dest->next = NULL;
    f = start;
    return 1;
  }

  return 0;
}

int parse_cmd(struct files *f, int argc, char **arg, unsigned *flags)
{
  int i,
    dest,
    source,
    count;

  dest = 0;
  source = 0;
  count = 0;

  for (i = 0; i < argc; i++)
  {
    if (arg[i][0] == '/')
    {
      if (!do_switches(arg[i], flags))
        return -1;
    }
    else
    {
      if (strcmp(arg[i], "+") == 0)
        source = 0;
      else if (strchr(arg[i], '+') == NULL && source)
      {
        if (!add_file(f, arg[i], &source, &dest, flags))
          return -1;
        f = f->next;
        count++;
      }
      else
      {
        if (!add_files(f, arg[i], &source, &dest, &count, flags))
          return -1;
        while (f->next != NULL)
          f = f->next;
      }
    }
  }
  dprintf(("parse_cmd: flags has %s\n", *flags & ASCII ? "ASCII" : "BINARY"));
  return count;
}

void delete_list(struct files *f)
{
  struct files *temp;

  while (f != NULL)
  {
    temp = f;
    f = f->next;
    free(temp);
  }
}

int overwrite(char *fn)
{
  char inp[10];

  printf("Overwrite %s (Yes/No/All)?", fn);
  scanf("%s", &inp);
  strupr(inp);
  if (inp[0] != 'Y' && inp[0] != 'A')
    return 0;
  if (inp[0] == 'A')
    return 2;
  return 1;
}

#define BUFF_SIZE 16384         /* 16k = max buffer size */

//int copy(struct files source, char *dest, int append, unsigned *flags)
int copy(char *source, char *dest, int append, unsigned *flags)
{
  char *buffer;
  unsigned bsiz,
    n,
    i,
    eof,
    wrote,
    attrib;
  int fd,
    old_verify;
  struct ftime srctime;
  FILE *fsrc,
   *fdest;

  eof = 0;
  dprintf(("checking mode\n"));
  attrib = _chmod(source, 0);
  fd = open(source, O_RDONLY);
  if (fd == -1)
  {
    display_string(TEXT_ERROR_CANNOT_OPEN_SOURCE, source);
    return 0;
  }
  dprintf(("getting time\n"));
  getftime(fd, &srctime);
  if (close(fd) == -1)
    perror("copy");

  fsrc = fopen(source, "rb");

  if (fsrc == NULL)
  {
    display_string(TEXT_ERROR_CANNOT_OPEN_SOURCE, source);
    return 0;
  }

  dprintf(("copy: flags has %s\n", *flags & ASCII ? "ASCII" : "BINARY"));
  dprintf(("copy: source flags has %s\n", source.flag & ASCII ? "ASCII" : "BINARY"));

  if (access(dest, 0) != 0)
  {
    dprintf(("opening/creating\n"));
    fdest = fopen(dest, "wb");
  }
  else if (!append)
  {
    if (strcmp(dest, source) == 0)
    {
      display_string(TEXT_ERROR_COPIED_ONTO_SELF);
      fclose(fsrc);
      return 0;
    }
    dprintf(("_chmod(%s, 1, 0);\n", dest));
    _chmod(dest, 1, 0);
    dprintf(("unlink(%s);\n", dest));
    unlink(dest);
    fdest = fopen(dest, "wb");
  }
  else
  {
    if (strcmp(dest, source) == 0)
    {
      fclose(fsrc);
      return 0;
    }
    dprintf(("opening/appending\n"));
    fdest = fopen(dest, "ab");
  }

  if (fdest == NULL)
  {
    fclose(fsrc);
    error_path_not_found();
    return 0;
  }

  if ((bsiz = coreleft()) > BUFF_SIZE)
    bsiz = BUFF_SIZE;
  else if (bsiz < 128)
  {
    error_out_of_memory();
    return 0;
  }

  buffer = (char *)malloc(bsiz);
  if (buffer == NULL)
  {
    fclose(fdest);
    fclose(fsrc);
    error_out_of_memory();
    return 0;
  }

  strupr(source);
  puts(source);
  if (*flags & VERIFY)
  {
    old_verify = getverify();
    setverify(1);
  }
  do
  {
    n = fread(buffer, sizeof(char), bsiz, fsrc);
    if (*flags & ASCII)
    {
      for (i = 0; i < n; i++)
      {
        if (buffer[i] == 0x1A)
        {
          eof = 1;
          break;
        }
      }
      n = i;
    }
    if (n == 0)
      break;
    wrote = fwrite(buffer, sizeof(char), n, fdest);
    if (wrote != n)
    {
      display_string(TEXT_ERROR_WRITING_DEST);
      free(buffer);
      fclose(fdest);
      fclose(fsrc);
      return 0;
    }
  }
  while (n && !eof);
  dprintf(("setting time\n"));
  setftime(fileno(fdest), &srctime);
  if (*flags & ASCII)
  {
    buffer[0] = 0x1A;
    buffer[1] = '\0';
    dprintf(("appending ^Z\n"));
    fwrite(buffer, sizeof(char), 1, fdest);
  }
  free(buffer);
  fclose(fdest);
  fclose(fsrc);
  dprintf(("setting mode\n"));
  _chmod(dest, 1, attrib);
  if (*flags & VERIFY)
    setverify(old_verify);
  return 1;
}

#pragma argsused
//int setup_copy(struct files *sources, char **p, int multiple,
//        char drive_d[MAXDRIVE], char dir_d[MAXDIR], char file_d[MAXFILE],
//               char ext_d[MAXEXT], int *append, unsigned *flags)

int setup_copy(struct files *sources, char **p, int multiple,
               char *drive_d, char *dir_d, char *file_d,
               char *ext_d, int *append, unsigned *flags)
{
  char drive_s[MAXDRIVE],
    dir_s[MAXDIR],
    file_s[MAXFILE],
    ext_s[MAXEXT];
  char from_merge[MAXPATH];
  char *real_dest,
   *real_source;
  int done,
    all = 0,
    copied = 0;
  struct ffblk ffblk;

  real_source = (char *)malloc(MAXPATH);
  real_dest = (char *)malloc(MAXPATH);

  if (!real_source || !real_dest)
  {
    error_out_of_memory();
    delete_list(sources);
    free(real_source);
    free(real_dest);
    freep(p);
    return 0;
  }

  while (sources->next != NULL)
  {
    fnsplit(sources->file, drive_s, dir_s, file_s, ext_s);
    done = FINDFIRST(sources->file, &ffblk, FA_ARCH);
    if (done)
    {
      error_file_not_found();
      delete_list(sources);
      freep(p);
      free(real_source);
      free(real_dest);
      return 0;
    }
    while (!done)
    {
      fnmerge(from_merge, drive_d, dir_d, file_d, ext_d);
      if (from_merge[strlen(from_merge) - 1] == '\\')
        from_merge[strlen(from_merge) - 1] = 0;
      if (isDir(from_merge))
      {
        multiple = 0;
        strcat(from_merge, "\\");
        strcat(from_merge, ffblk.ff_name);
      }
      else
        multiple = 1;
      strcpy(real_dest, from_merge);
      strupr(real_dest);
      fnmerge(real_source, drive_s, dir_s, ffblk.ff_name, NULL);
      dprintf(("copying %s -> %s (%sappending%s)\n",
               real_source, real_dest,
               *append ? "" : "not ",
               sources->flag & ASCII ? ", ASCII" : ", BINARY"));
      if (access(real_dest, 0) == 0 && !all)
      {
        int over = overwrite(real_dest);
        if (over == 2)
          all = 1;
        else if (over == 0)
          goto next;
        else if (multiple)
          all = 1;
      }
//      if (copy(*sources, real_dest, *append, flags))
      if (copy(real_source, real_dest, *append, flags))
        copied++;
    next:
      done = FINDNEXT(&ffblk);
      if (multiple)
        *append = 1;
    }
    sources = sources->next;
  }
  free(real_source);
  free(real_dest);
  return copied;
}

#pragma argsused
int cmd_copy(char *rest)
{
  char **p;
  char drive_d[MAXDRIVE],
    dir_d[MAXDIR],
    file_d[MAXFILE],
    ext_d[MAXEXT];

  int argc,
    dest_has_wildcards,
    dest_found,
    multiple,
    append,
    files,
    copied;
  struct files *sources,
    dest,
   *start;
  unsigned flags = 0;

  p = split(rest, &argc);

  if (argc == 0)
  {
    error_req_param_missing();
    return 0;
  }

  sources = (struct files *)malloc(sizeof(struct files));
  if (!sources)
  {
    error_out_of_memory();
    return 0;
  }
  sources->next = NULL;
  sources->flag = 0;

  if ((files = parse_cmd(sources, argc, p, &flags)) == -1)
  {
    delete_list(sources);
    freep(p);
    return 0;
  }
  else if (files == 0)
  {
    error_req_param_missing();
    delete_list(sources);
    freep(p);
    return 0;
  }
  start = sources;

  dest_found = get_dest(sources, &dest);
  if (dest_found)
  {
    fnsplit(dest.file, drive_d, dir_d, file_d, ext_d);
    if (isDir(dest.file))
    {
      strcat(dir_d, file_d);
      strcat(dir_d, ext_d);
      file_d[0] = 0;
      ext_d[0] = 0;
    }
  }

  if (strchr(dest.file, '*') || strchr(dest.file, '?'))
    dest_has_wildcards = 1;
  else
    dest_has_wildcards = 0;

  if (strchr(rest, '+'))
    multiple = 1;
  else
    multiple = 0;

  append = 0;
  copied = 0;

  if (dest_found && !dest_has_wildcards)
  {
    copied = setup_copy(sources, p, multiple, drive_d, dir_d, file_d, ext_d, &append, &flags);
  }
  else if (dest_found && dest_has_wildcards)
  {
    display_string(TEXT_NOT_IMPLEMENTED_YET);
    delete_list(sources);
    freep(p);
    return 0;
  }
  else if (!dest_found && !multiple)
  {
    fnsplit(sources->file, drive_d, dir_d, file_d, ext_d);
    if (isDir(sources->file))
    {
      strcat(dir_d, file_d);
      strcat(dir_d, ext_d);
      file_d[0] = 0;
      ext_d[0] = 0;
    }
    copied = setup_copy(sources, p, 0, "", "", file_d, ext_d, &append, &flags);
  }
  else
  {
    fnsplit(sources->file, drive_d, dir_d, file_d, ext_d);
    if (isDir(sources->file))
    {
      strcat(dir_d, file_d);
      strcat(dir_d, ext_d);
      file_d[0] = 0;
      ext_d[0] = 0;
    }
    strupr(sources->file);
    puts(sources->file);
    append = 1;
    copied = setup_copy(sources->next, p, multiple, drive_d, dir_d, file_d, ext_d, &append, &flags) + 1;
  }

  delete_list(start);
  freep(p);
  printf("        %d file(s) copied\n", copied);
  return 1;
}

#ifdef DEBUG_STANDALONE
#pragma argsused
int main(int argc, char *argv[])
{
  dprintf(("coreleft: %u\n", coreleft()));
  if (argc == 1)
    argv[1][0] = 0;
  cmd_copy("copy", argv[1]);
  dprintf(("coreleft: %u\n", coreleft()));
  return 1;
}
#endif                          /* DEBUG_STANDALONE */
/* #undef DEBUG */
#endif                          /* INCLUDE_CMD_COPY */
