/*
Copyright (C) 1998, 1999, 2000 Wabasoft

Modifications for DOS by Mohan Embar
http://www.thisiscool.com/

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later version. 

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE. See the GNU General Public License for more details. 

You should have received a copy of the GNU General Public License along with
this program; if not, write to the Free Software Foundation, Inc., 675 Mass Ave,
Cambridge, MA 02139, USA. 
*/

/*

If you're looking here, you've probably already looked at (or worked on) the
nm<platform>_a.c file.

This file is the second step in porting the VM. In it, you need to place the
main application loop and class loader. If you get through this, you'll have
a basic VM running and from there you can implement the native functions
so you can actually see something on the screen.

The WabaVM, like other applications, needs some type of main application loop
where the OS sends it key clicks, mouse presses, etc. The VM also needs to
load classes from memory or disk or wherever.

Different platforms have different main application loops. For example, under
Win32, a program has a WinMain() function. Under PalmOS, applications have
a PilotMain() function.

You'll want to implement a main application loop for the platform you are
porting to. Then, when things start up (initialization), you need to:

- parse the launch command and then call VmInit()
- call VmStartApp passing the class name of the program's main window

You'll also need to figure out how programs will launch the VM. The VM
is normally passed parameters telling it what class heap, class name, etc.
to use. It's usually another program that does this but you could do it
by building a waba launcher that draws a tray of icons and when you click
one, it runs the WabaVM with various parameters as an example of something
different. Whatever you do for this is platform-specific.

Just before the program exits, you'll should call VmStopApp(mainWinObj);

That's it for initialization and exit, then you need to do some work to
hook up key presses, mouse presses, etc. to the VM. When a key is pressed,
you'll want to invoke the main window's onEvent() method. Here's an
example of how to do that:

    mainWinObj = globalMainWin;
    if (!mainWinObj)
        return;
    vclass = WOBJ_class(mainWinObj); // get runtime class
    method = getMethod(vclass, createUtfString("_postEvent"),
        createUtfString("(IIIIII)V"), &vclass);
    if (method != NULL)
        {
        params[0].obj = mainWinObj;
        params[1].intValue = type; // type
        params[2].intValue = key; // key
        params[3].intValue = 0; // x
        params[4].intValue = 0; // y
        params[5].intValue = 0; // modifiers
        params[6].intValue = 0; // timeStamp
        executeMethod(vclass, method, params, 7);
        }

This gets the runtime class of the main window object, gets a
reference to the _postEvent method, fills in an array of parameters
to pass and then calls executeMethod() to execute the method.

There is also a quickbind version of this which is quicker than the
above. You can look at nmpalm_b.c or nmwin32_b.c to get an idea how
that works (look for the #ifdef QUICKBIND) but you should start
without using the quickbind method.

You'll want to hook up mouse or pen click, key presses, calls from
the OS to repaint the window (which should call the main window's
_doPaint routine).

Overall, you need to make sure the following main window functions
get called:

_onTimerTick()
_postEvent() for key presses and mouse/pen clicks and moves

You'll need to map special keys (escape, etc.) if the device has
them to suitable numbers for waba. To do this, you can see how the
other native implementations do key mapping (a big switch statement).

The _onTimerTick() one can be tricky for devices that don't have
a built in timer (like PalmOS). When using one of those devices,
copy the code from the nmpalm_b.c to deal with timers. Basically,
this consists of getting the current time and looking for an event
up until the time the next timer period expires.

The VM only needs a single timer, the waba core classes have the
code to support multiple timers even though the underlying system
only supports a single one.

The last code you need to implement is:

nativeLoadClass()

This is the function that load a class from memory, disk or whatever.
It looks like this:

static uchar *nativeLoadClass(UtfString className, uint32 *size)

and it is passed a class name (not zero terminated, the UtfString
contains the len in:

className.len

and it returns a pointer to the class in memory after it is loaded
and fills in the *size variable with the size of the class in bytes.

The classes loaded are classes that are compiled by a java compiler
or other compiler that can generate code in the subset of java
bytecode that waba supports.

*/

#include <grx20.h>
#include <grxkeys.h>
#include <time.h>

#ifdef QUICKBIND
static int32 postPaintMethodMapNum = -1;
static int32 postEventMethodMapNum = -1;
static int32 onTimerTickMethodMapNum = -1;
#endif

static WObject globalMainWin = 0;
static int32 globalTimerInterval = 0;
static int32 globalTimerStart = 0;

static int g_mainWinWidth = 0;
static int g_mainWinHeight = 0;
static int nRunning = 0;
static int nGetVMIsColor = 1;

#define MAX_CLASSPATHS 20
static int numClassPaths = -1;
static char *classPaths[MAX_CLASSPATHS];
static int nExitCode = 0;

static void WinEraseWindow();

typedef struct
{
    int isPDB;
    uint32 size;
    uchar *ptr;
} MemFile;

#define MAX_MEM_FILES 16
static MemFile memFiles[MAX_MEM_FILES];
static uint32 numMemFiles = 0;
static int memFileNotSupported = 0;

#define WARP_CORE_FILENAME "waba.wrp"

static void usage()
{
    printf("Waba (TM) Virtual Machine Version 1.0 Beta for 80x86\n");
    printf("Copyright (C) Wabasoft 1998. All rights reserved.\n");
    printf("\n");
    printf("Usage: waba [options] appclass [warpfile]\n");
    printf("\n");
    printf("Options:\n");
    printf("  /l   Assign size of class heap (e.g. /l 10000)\n");
    printf("  /m   Assign size of object heap (e.g. /m 20000)\n");
    printf("  /s   Assign size of stack (e.g. /s 2000)\n");
    printf("  /t   Assign size of native stack (e.g. /t 50)\n");
    printf("  /w   Assign window width (e.g. /w 160)\n");
    printf("  /h   Assign window height (e.g. /h 160)\n");
    printf("  /b   Black and white\n");
    printf("\n");
    printf("Example:\n");
    printf("  waba /m 20000 MyApp myapp.wrp\n");
    printf("\n");
    exit(1);
}

static char* getDir(char* szBuf, char* szPath)
{
    int i, nLen;

    strcpy(szBuf,szPath);
    nLen = strlen(szBuf);
    for (i=nLen-1; i>=0; --i)
    {
        char ch = szBuf[i];
        if (ch=='/' || ch=='\\')
        {
            szBuf[i+1] = '\0';
            return szBuf;
        }
    }

    return NULL;
}

#ifdef _WIN32

static int32 getTimeStamp()
	{
	static uint32 maxStamp = 1 << 30;
	static uint32 maxLow = 0xFFFFFFFF / 10000;
	uint32 millis;
	SYSTEMTIME tm;
	FILETIME ftm;

	GetSystemTime(&tm);
	SystemTimeToFileTime(&tm, &ftm);
	millis = ((ftm.dwHighDateTime % 10000) * maxLow) + (ftm.dwLowDateTime / 10000);
	millis = millis % (uint32)maxStamp;
	return (int32)millis;
	}

#else

static int32 getTimeStamp()
{
	static uint32 maxStamp = 1 << 30;
    static uint32 lSecondsBase = 980000000L;
    int32 nResult;
    struct timeval tv;
    gettimeofday(&tv,NULL);
    nResult =                 
        (tv.tv_sec-lSecondsBase)*1000 + (tv.tv_usec / 1000L);
	nResult %= (uint32)(1L << 30);
    return nResult;
}

#endif

static uchar *readFileIntoMemory(char *path, int nullTerminate, uint32 *size)
{
    FILE* pFile;
    uchar *p;
    uint32 lenRead, len, lenWant;
    char* pszTemp;

    for (pszTemp = path; *pszTemp; ++pszTemp)
    {
        if (*pszTemp == '/')
            *pszTemp = '\\';
    }
    if (nullTerminate != 0)
        nullTerminate = 1;

    pFile = fopen(path, "rb");
    if (!pFile)
        return NULL;

    if (fseek(pFile,0,SEEK_END))
    {
        fclose(pFile);
        return NULL;
    }

    len = ftell(pFile);
    if (len == -1)
    {
        fclose(pFile);
        return NULL;
    }

    if (fseek(pFile,0,SEEK_SET))
    {
        fclose(pFile);
        return NULL;
    }

    lenWant = len + nullTerminate;
    
    if (size)
        *size = lenWant;

    p = (uchar *)xmalloc(lenWant);
    if (p != NULL)
    {
        lenRead = fread(p,1,len,pFile);
        if (len != lenRead)
        {
            xfree(p);
            p = NULL;
        }
    }
    else
        VmQuickError(ERR_CantAllocateMemory);

    if (p && nullTerminate)
        p[len] = 0;

    fclose(pFile);

//  printf("  read %s\n",path);
    return p;
}

static uchar *loadFromMem(char *path, uint32 pathLen, uint32 *size)
{
    uchar *baseP, *offP, *p;
    uint32 i, off, nextOff, top, bot, mid;
    uint32 nameLen, minLen, numRecs;
    int cmp;
    int numRecsOffset;
    int isPDB;

#ifdef DEBUGVERBOSE
    fprintf(pfilLog, "Looking for %s...\n", path);
    fflush(pfilLog);
#endif
    
    // look in memory mapped files
    for (i = 0; i < numMemFiles; i++)
    {
        baseP = memFiles[i].ptr;
        isPDB = memFiles[i].isPDB;

        if (isPDB)
            numRecs = getUInt16(baseP + 76);
        else
            numRecs = getUInt32(baseP + 4);

        if (numRecs == 0)
            continue;

        // NOTE: We do a binary search to find the class. So, a search
        // for N classes occurs in O(nlogn) time.
        top = 0;
        bot = numRecs;
        while (1)
            {
            mid = (bot + top) / 2;

            if (isPDB)
                offP = baseP + 78 + (mid * 8);
            else
                offP = baseP + 8 + (mid * 4);

            off = getUInt32(offP);
            p = baseP + off;
            nameLen = getUInt16(p);
            p += 2;
            if (pathLen > nameLen)
                minLen = nameLen;
            else
                minLen = pathLen;
            cmp = xstrncmp(path, p, minLen);
            if (!cmp)
                {
                if (pathLen == nameLen)
                    {
                    if (size != NULL)
                        {

                        if (isPDB)
                        {
                            if (mid < numRecs-1)
                                nextOff = getUInt32(offP + 8);
                            else
                                nextOff = memFiles[i].size;
                        }
                        else
                        {
                            nextOff = getUInt32(offP + 4);
                        }
                        *size = nextOff - off - nameLen - 2;
                        }
#ifdef DEBUGVERBOSE
                    fprintf(pfilLog, "  found %s!\n", path);
                    fflush(pfilLog);
#endif
                    return p + nameLen;
                    }
                if (pathLen > nameLen)
                    cmp = 1;
                else
                    cmp = -1;
                }
            if (mid == top)
                break; // not found
            if (cmp < 0)
                bot = mid;
            else
                top = mid;
            }
    }
    return NULL;
}

static void freeMemMapFile(MemFile *memFile)
{
    if (memFile->ptr)
        xfree(memFile->ptr);
}

static uchar *memMapFile(char *path)
{
    MemFile memFile;

    if (numMemFiles == MAX_MEM_FILES)
    {
#ifdef DEBUGVERBOSE
        fprintf(stderr, "Too many warp files\n");
        fflush(pfilLog);
#endif
        return NULL;
    }

    memFile.ptr = readFileIntoMemory(path, 0, &memFile.size);

    if
    (
        !memFile.ptr
        ||
        (
            strncmp(memFile.ptr, "Wrp1", 4)
            &&
            strncmp(memFile.ptr+60, "Wrp1", 4)
        )
    )
    {
#ifdef DEBUGVERBOSE
        fprintf(stderr,"Error loading file: %s\n",path);
        fflush(pfilLog);
#endif
        freeMemMapFile(&memFile);
        return NULL;
    }

    memFile.isPDB = !strncmp(memFile.ptr+60, "Wrp1", 4);
    memFiles[numMemFiles++] = memFile;
    return memFile.ptr;
}

static uchar *nativeLoadClass(UtfString className, uint32 *size)
{
    uchar *p;
    uint16 len, i;
    char path[128];

    
    // try loading from memory mapped files first
    // make full path by appending .class
    len = (uint16) className.len + 6;
    if (len > 128)
        return NULL;
    xstrncpy(path, className.str, className.len);
    xstrncpy(&path[className.len], ".class", 6);
    path[className.len+6] = '\0';
//  printf("Trying to load class %s...\n", path);
    p = loadFromMem(path, len, size);
    if (p != NULL)
        return p;

    // not found in memory mapped files, try loading it from the CLASSPATH
    if (numClassPaths == -1)
        {
        char *s, *sp;
        classPaths[0] = ".";
        numClassPaths = 1;
        s = getenv("CLASSPATH");
        if (s != NULL)
            {
            // NOTE: we duplicate the CLASSPATH here since strtok() modifies
            // it but we never explicitly free it, we let the OS free it when
            // the program exits. Also note we don't need to deal with UNICODE
            // since this section does not applicable to WinCE
            i = xstrlen(s);
            sp = xmalloc(i + 1);
            xstrncpy(sp, s, i);
            sp[i] = 0;
            s = sp;

            // parse through the elements of CLASSPATH      
            sp = strtok(s, ";");
            while (sp != NULL)
                {
                classPaths[numClassPaths++] = sp;
                if (numClassPaths == MAX_CLASSPATHS)
                    break;
                sp = strtok(NULL, ";");
                }
            }
        }
    // NOTE: we never free the memory pointers we allocate here. We let the
    // OS clean up memory when the process exits. This works well but if we
    // ever do free these pointers, we need to make sure we differentiate
    // them from the memory mapped file pointers (nativeLoadClassFromMem())
    for (i = 0; i < numClassPaths; i++)
        {
        len = xstrlen(classPaths[i]);
        xstrncpy(path, classPaths[i], len);
        if (len>0 && path[len-1] != '\\')
            xstrncpy(&path[len++], "\\", 1);
        xstrncpy(&path[len], className.str, className.len);
        len += (uint16) className.len;
        xstrncpy(&path[len], ".class", 6);
        len += 6;
        path[len] = 0;
        p = readFileIntoMemory(path, 0, size);
        if (p != NULL)
            break;
        }
    if (p == NULL)
        return NULL;
    // validate
    if (getUInt32(p) != (uint32)0xCAFEBABE)
        {
        VmError(ERR_BadClass, NULL, &className, NULL);
        xfree(p);
        return NULL;
        }
    return p;
}

static void dumpStackTrace()
	{
	WClass *wclass;
	WClassMethod *method;
	UtfString className, methodName, methodDesc;
	uint32 i, n, stackPtr;

	if (vmStackPtr == 0)
		return;
	AllocConsole();
#ifdef DUMPERRORTRACE
	{
	char *msg;

	msg = errorMessages[vmStatus.errNum - 1];
#ifdef DEBUGVERBOSE
	if (msg)
		fprintf(pfilLog, "ERROR - %s\n", msg);
	fprintf(pfilLog, "ERROR arg1: %s\n", vmStatus.arg1);
	fprintf(pfilLog, "ERROR arg2: %s\n", vmStatus.arg2);
#endif
	}
#endif
	stackPtr = vmStackPtr;
	while (stackPtr != 0)
		{
		wclass = (WClass *)vmStack[--stackPtr].refValue;
		method = (WClassMethod *)vmStack[--stackPtr].refValue;
		className = getUtfString(wclass, wclass->classNameIndex);
		methodName = getUtfString(wclass, METH_nameIndex(method));
		methodDesc = getUtfString(wclass, METH_descIndex(method));

#ifdef DEBUGVERBOSE
		for (i = 0; i < className.len; i++)
			fprintf(pfilLog, "%c", className.str[i]);
		fprintf(pfilLog, ".");
		for (i = 0; i < methodName.len; i++)
			fprintf(pfilLog, "%c", methodName.str[i]);
		for (i = 0; i < methodDesc.len; i++)
			fprintf(pfilLog, "%c", methodDesc.str[i]);
		fprintf(pfilLog, "\n");
#endif

		if ((METH_accessFlags(method) & ACCESS_NATIVE) > 0)
			stackPtr -= vmStack[--stackPtr].intValue;
		else
			{
			n = METH_maxLocals(method) + METH_maxStack(method);
#ifdef DEBUGVERBOSE
			for (i = 0; i < n; i++)
				fprintf(pfilLog, "%d:%d ", i, vmStack[--stackPtr].intValue);
			fprintf(pfilLog, "\n");
#endif
			}
		stackPtr -= 3;
		}
	}

static WObject startApp(char* szEXEPath, char *cmdLine)
{
    int vmStackSize, nmStackSize, classHeapSize, objectHeapSize;
    char *className;
    int i;
    char szBuf[255];

    vmStackSize = 8192;
    nmStackSize = 8192;
    classHeapSize = 80000;
    objectHeapSize = 80000;
    g_mainWinWidth = g_mainWinHeight = 160;
    className = 0;

    // parse options
    i = 0;
    while (cmdLine[i] == '/')
    {
        char c;
        int32 value;

        c = cmdLine[++i];
        if (!c)
            break;
        if (cmdLine[++i] != ' ')
            break;
        i++;
        // parse integer - CE doesn't always have atoi()
        value = 0;
        while (cmdLine[i] >= '0' && cmdLine[i] <= '9')
        {
            value = (value * 10) + (cmdLine[i] - '0');
            i++;
        }
        if (c == 'l')
            classHeapSize = value;
        else if (c == 'm')
            objectHeapSize = value;
        else if (c == 's')
            vmStackSize = value;
        else if (c == 't')
            nmStackSize = value;
        else if (c == 'w')
            g_mainWinWidth = value;
        else if (c == 'b')
            nGetVMIsColor = 0;
        else if (c == 'h')
            g_mainWinHeight = value;
        while (cmdLine[i] == ' ')
            i++;
    }

    // parse class name
    className = &cmdLine[i];
    while (cmdLine[i] && cmdLine[i] != ' ')
        i++;
    if (cmdLine[i] == ' ')
        cmdLine[i++] = 0;

    // run "welcome" app (no app class specified)
    if (className[0] == 0)
        className = "waba/ui/Welcome";

    // memory map warp files
    memMapFile(strcat(getDir(szBuf,szEXEPath),WARP_CORE_FILENAME));
    while (cmdLine[i])
    {
        char *path, delim;

        while (cmdLine[i] == ' ')
            i++;
        delim = ' ';
        if (cmdLine[i] == '"')
            {
            delim = '"';
            i++;
            }
        path = &cmdLine[i];
        while (cmdLine[i] && cmdLine[i] != delim)
            i++;
        if (cmdLine[i] == delim)
            cmdLine[i++] = 0;
        memMapFile(path);
    }

#ifdef DEBUGCMDLINE
    printf("mainWinWidth %d\n", g_mainWinWidth);
    printf("mainWinHeight %d\n", g_mainWinHeight);
    printf("vmStackSize %d\n", vmStackSize);
    printf("nmStackSize %d\n", nmStackSize);
    printf("classHeapSize %d\n", classHeapSize);
    printf("objectHeapSize %d\n", objectHeapSize);
    printf("className #%s#\n", className);
#endif
 
    VmInit(vmStackSize, nmStackSize, classHeapSize, objectHeapSize);
    return VmStartApp(className);
}

static void stopApp(WObject mainWinObj)
{
    uint32 i;

    VmStopApp(mainWinObj);
    VmFree();

    // free memory mapped files
    for (i = 0; i < numMemFiles; i++)
        freeMemMapFile(&memFiles[i]);
}

typedef enum {NONE, UP, DOWN, MOVE} EMouseOp;

static int nXltOffX, nXltOffY;

static void mouse(WObject mainWinObj, int nX, int nY, EMouseOp eMouseOp)
{
	WClass *vclass;
	WClassMethod *method;
	Var params[7];
	int32 type, x, y;

	if (eMouseOp == DOWN)
		{
		type = 200; // PenEvent.PEN_DOWN
		}
	else if (eMouseOp == UP)
		{
		type = 202; // PenEvent.PEN_UP
		}
	else
		type = 201; // PenEvent.PEN_MOVE
	x = nX;
	y = nY;
	vclass = WOBJ_class(mainWinObj); // get runtime class
#ifdef QUICKBIND
	method = getMethodByMapNum(vclass, &vclass, (uint16)postEventMethodMapNum);
#else
	method = getMethod(vclass, createUtfString("_postEvent"),
		createUtfString("(IIIIII)V"), &vclass);
#endif
	if (method != NULL)
		{
		params[0].obj = mainWinObj;
		params[1].intValue = type; // type
		params[2].intValue = 0; // key
		params[3].intValue = x-nXltOffX; // x
		params[4].intValue = y-nXltOffY; // y
		params[5].intValue = 0; // modifiers
		params[6].intValue = 0; // timeStamp
		executeMethod(vclass, method, params, 7);
    }
}

static void keypress(WObject mainWinObj, GrMouseEvent* pEvt)
{
	WClass *vclass;
	WClassMethod *method;
	Var params[7];
	int32 type, key, mod;

	type = 100; // KeyEvent.KEY_PRESS
	key = pEvt->key;


#ifdef DEBUGVERBOSE
    fprintf(pfilLog, "Key: %d\n", key);
    fflush(pfilLog);
#endif

    if (key == GrKey_Alt_F4)
    {
        nRunning = 0;
        return;
    }

	switch (key)
		{
		case GrKey_PageUp:      key = 75000; break; // IKeys.PAGE_UP
		case GrKey_PageDown:    key = 75001; break; // IKeys.PAGE_DOWN
		case GrKey_Home:        key = 75002; break; // IKeys.HOME
		case GrKey_End:         key = 75003; break; // IKeys.END
		case GrKey_Up:          key = 75004; break; // IKeys.UP
		case GrKey_Down:        key = 75005; break; // IKeys.DOWN
		case GrKey_Left:        key = 75006; break; // IKeys.LEFT
		case GrKey_Right:       key = 75007; break; // IKeys.RIGHT
		case GrKey_Insert:      key = 75008; break; // IKeys.INSERT
		case GrKey_Return:      key = 75009; break; // IKeys.ENTER
		case GrKey_Tab:         key = 75010; break; // IKeys.TAB
		case GrKey_BackSpace:   key = 75011; break; // IKeys.BACKSPACE
		case GrKey_Escape:      key = 75012; break; // IKeys.ESCAPE
		case GrKey_Delete:      key = 75013; break; // IKeys.DELETE
		}

	mod = 0;
	if (pEvt->kbstat & GR_KB_CTRL)
		mod |= (1 << 1); // IKeys.CONTROL
	if (pEvt->kbstat & GR_KB_SHIFT)
		mod |= (1 << 2); // IKeys.SHIFT

	vclass = WOBJ_class(mainWinObj); // get runtime class
#ifdef QUICKBIND
	method = getMethodByMapNum(vclass, &vclass, (uint16)postEventMethodMapNum);
#else
	method = getMethod(vclass, createUtfString("_postEvent"),
		createUtfString("(IIIIII)V"), &vclass);
#endif
	if (method != NULL)
	{
		params[0].obj = mainWinObj;
		params[1].intValue = type; // type
		params[2].intValue = key; // key
		params[3].intValue = 0; // x
		params[4].intValue = 0; // y
		params[5].intValue = mod; // modifiers
		params[6].intValue = 0; // timeStamp

        executeMethod(vclass, method, params, 7);
	}
}

static void timerCheck()
	{
	WObject mainWinObj;
	int32 now, diff;
	WClass *vclass;
	WClassMethod *method;
	Var params[1];

	mainWinObj = globalMainWin;
	if (!mainWinObj)
		return;
	now = getTimeStamp();
	diff = now - globalTimerStart;
	if (diff < 0)
		diff += (1L << 30); // max stamp is (1 << 30)
	if (diff < globalTimerInterval)
		return;
	vclass = WOBJ_class(mainWinObj); // get runtime class
#ifdef QUICKBIND
	method = getMethodByMapNum(vclass, &vclass,
		(uint16)onTimerTickMethodMapNum);
#else
	method = getMethod(vclass, createUtfString("_onTimerTick"),
		createUtfString("()V"), &vclass);
#endif
	if (method != NULL)
		{
		params[0].obj = mainWinObj;
		executeMethod(vclass, method, params, 1);
		}
	globalTimerStart = now;
	}

static int calcEventTimeout()
		{
		int timeout;

		if (globalTimerInterval <= 0)
			timeout = -1;
		else
			{
			int32 now, diff;

			// NOTE: calculate the ticks until the next interval
			now = getTimeStamp();
			diff = now - globalTimerStart;
			if (diff < 0)
				diff += (1L << 30); // max stamp is (1 << 30)
			timeout = globalTimerInterval - diff;
			if (timeout <= 0)
				timeout = 0;
			if (timeout <= 0)
				timeout = 1;
			}
		return timeout;
		}

static void drawMainWin()
{
	WObject mainWinObj;
	WClass *vclass;
	WClassMethod *method;
	Var params[7];

	mainWinObj = globalMainWin;
	if (!mainWinObj)
		return;
	WinEraseWindow();
	vclass = WOBJ_class(mainWinObj); // get runtime class
#ifdef QUICKBIND
	method = getMethodByMapNum(vclass, &vclass, (uint16)postPaintMethodMapNum);
#else
	method = getMethod(vclass, createUtfString("_doPaint"),
		createUtfString("(IIII)V"), &vclass);
#endif
	if (method == NULL)
		return;
	params[0].obj = mainWinObj;
	params[1].intValue = 0; // x
	params[2].intValue = 0; // y
	params[3].intValue = g_mainWinWidth; // width
	params[4].intValue = g_mainWinHeight; // height
	executeMethod(vclass, method, params, 5);
}

int main(int nArgC, char* arpszArgV[])
{
    WObject mainWinObj;
    char szCmdLine[500];
    int i;
    int nMouse = 0;
	GrMouseEvent evt;

#ifdef DEBUGVERBOSE
    pfilLog = fopen("c:\\temp\\waba.log", "w");
#endif

    if (nArgC == 1)
        usage();

    szCmdLine[0] = '\0';
    for (i=1; i<nArgC; ++i)
    {
        if (i>1)
            strcat(szCmdLine," ");
        strcat(szCmdLine,arpszArgV[i]);
    }
    
    mainWinObj = startApp(arpszArgV[0],szCmdLine);

#ifdef QUICKBIND
    if (mainWinObj != 0)
        {
        WClass *vclass;

        // cache method map numbers for commonly called methods
        vclass = WOBJ_class(mainWinObj);
        postPaintMethodMapNum = getMethodMapNum(vclass, createUtfString("_doPaint"),
            createUtfString("(IIII)V"), SEARCH_ALL);
        postEventMethodMapNum = getMethodMapNum(vclass, createUtfString("_postEvent"),
            createUtfString("(IIIIII)V"), SEARCH_ALL);
        onTimerTickMethodMapNum = getMethodMapNum(vclass, createUtfString("_onTimerTick"),
            createUtfString("()V"), SEARCH_ALL);
        if (postPaintMethodMapNum == -1 || postEventMethodMapNum == -1 ||
            onTimerTickMethodMapNum == -1)
            mainWinObj = 0;
        }
#endif

    if (mainWinObj == 0 || vmStatus.errNum > 0)
    {
#ifdef DEBUGVERBOSE
        fprintf(stderr, "Error initializing VM or creating main window\n");
        fflush(pfilLog);
#endif
        return 1;
    }

    if(GrMouseDetect())
    {
        nMouse = 1;
	    GrMouseEventMode(1);
	    GrMouseInit();
	    GrMouseSetColors(GrBlack(),GrWhite());
	    GrMouseDisplayCursor();
    }

    drawMainWin();

    nRunning = 1;
    while (nRunning)
    {
        EMouseOp eMouseOp = NONE;
		int timeout;
		timeout = calcEventTimeout();

#ifdef DEBUGVERBOSE
        print("Before get event");
        fprintf(pfilLog, "Timeout: %d\n", timeout);
        fflush(pfilLog);
#endif
        GrMouseGetEventT(GR_M_EVENT,&evt,timeout);
#ifdef DEBUGVERBOSE
        print("After get event");
#endif

		if(evt.flags & GR_M_MOTION)      eMouseOp = MOVE;
		if(evt.flags & GR_M_LEFT_DOWN)   eMouseOp = DOWN;
		if(evt.flags & GR_M_LEFT_UP)     eMouseOp = UP;

		if (timeout != -1)
			timerCheck();
        
        if (eMouseOp != NONE)
        {
#ifdef DEBUGVERBOSE
            print("Before mouse");
#endif
            mouse(mainWinObj,evt.x,evt.y,eMouseOp);
#ifdef DEBUGVERBOSE
            print("After mouse");
#endif
        }
        else if (evt.flags & GR_M_KEYPRESS)
        {
#ifdef DEBUGVERBOSE
            print("Before key");
#endif
            keypress(mainWinObj, &evt);
#ifdef DEBUGVERBOSE
            print("After key");
            fprintf(pfilLog,"Key code: %d\n", evt.key);
            fflush(pfilLog);
#endif
        }
    }

    if (nMouse)
    {
	    GrMouseUnInit();
    }
    
    stopApp(mainWinObj);
#ifdef DEBUGVERBOSE
    fclose(pfilLog);
#endif

    return nExitCode;
}

static int getVMIsColor()
{
    return nGetVMIsColor;
}
