/*
    MSGLIB - a message handling library
    Copyright (C) 1995,1997-98  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/* $Id: MSGCONF2.H 1.7 1998/10/07 04:42:53 ska Rel ska $
   $Locker: ska $	$Name:  $	$State: Rel $

	This holds configuration options for the method #2 to access messages.

	This file can be sourced-in in non-method #2 environments to
	get access to the identifiers, however, no change to the msg*()
	macros is made.

*/

#ifndef __MSGCONF2_H
#define __MSGCONF2_H

#ifndef MSG_METHOD
		/* default to method #2 */
#define MSG_METHOD 2
#endif /*#	!defined(MSG_METHOD) */


#include <portable.h>

/*
 *	A numerical message ID looks like this (binary/dual visualization):
 *	GGGG.IIII IIII.IIII
 *	==> 4 bit group ID, 12 bit index
 */
#ifdef MSGGROUPMAX
#undef MSGGROUPMAX
#endif /*#	defined(MSGGROUPMAX) */
#define MSGGROUPMAX 16
#define msgGroup(msg) (((msg) >> 12) & 0xF)
#define msgIndex(msg) ((msg) & 0xFFF)
#define mkMsgID(group,index) (((group) << 12) | ((index) & 0xfff))

/*
 *	This string will identify a message catalogue file (in conjunction
 *	with magic numbers).
 */
#define MSG_IDSTRING "MSGLIB"


/*
 *	How an opened catalogue is identified by within MSGLIB.
 */
#define MSG2FILE int
#define MSG2NOFILE (-1)			/* error/empty value */


/* Special character sets, substitutions for varios non-codepage
	settings */
#define CHARSET_NONE		0x7fff	/* none specific, match always */
#define CHARSET_7BITASCII	0x7ffe	/* 7bit ASCII 0x20..0x7e */
#define CHARSET_UNICODE16	0x7ffd	/* 16bit (2 byte) UNI CODE */


/* Magic numbers */
/* Currently those numbers serve two purposes:
	1) to identify the structure within the binary message file,&
	2) to verify the correctness of the structure chain within the
		file. For it a simple check mechanism is used:
			+ the magic number must be unequal to zero,
			+ the sum of low and high byte must be zero,
			+ the high byte must be less than or equal to MSG_MN_MAXHI
*/
#define MSG_MN_STRU0	0x01ff		/* structure #0 */
#define MSG_MN_STRU1	0x02fe		/* structure #1 */
#define MSG_MN_IDS		0x03fd		/* message IDs <-> names */
#define MSG_MN_CSMAP	0x04fc		/* character set mapping */

#define MSG_MN_MAXHI	0x04		/* highest HI-byte */

#include <algnbyte.h>	/* disable alignment -> align at byte boundary! */

/* Various types
	msgTLngcode	language code; signed; zero means invalid/empty code;
				negative values are reserved for special error codes;
				values >= 20000 are reserved for special codes, not
				directly dedicated to particular languages.
	msgTCharset	character set (DOS code page) in which the messages are
				presentable; signed; zero means invalid/empty set;
				negative values are reserved for special error codes;
				values >= 20000 are reserved for special sets, not
				directly dedicated to particular DOS code pages
	msgTGroup	message group; unsigned
	msgTMagic	magic number; unsigned
	msgTExit	exit code; unsigned

	type of count/size values is always (int).
*/
#ifdef _MICROC_
#define msgTLngcode int
#define msgTCharset int
#define msgTGroup byte
#define msgTClass byte
#define msgTExit byte
#define msgTMagic word
#define MSGID2 unsigned
#else /*#	!(defined(_MICROC_)) */
typedef int msgTLngcode;
typedef int msgTCharset;
typedef byte msgTGroup;
typedef byte msgTClass;
typedef byte msgTExit;
typedef word msgTMagic;
typedef unsigned MSGID2;
#endif /*#	defined(_MICROC_) */

struct MsgStructMN {				/* standard fields */
	msgTMagic mmn_magicnumber;
	dword mmn_next;				/* next item in chain */
};

struct MsgStruct0 {					/* structure #0 */
	msgTMagic ms0_magicnumber;
	dword ms0_address;				/* first struct1 of this file */
	byte ms0_id[6];					/* message library id string */
	byte ms0_version;				/* version id */
};

struct MsgStruct1 {					/* structure #1 */
	msgTMagic ms1_magicnumber;
	dword ms1_next;					/* next lng */
	msgTLngcode ms1_lng;			/* code of language */
	int ms1_charsets;				/* number of supported character sets */
	int ms1_groups;					/* number of group ids */
	int ms1_aliases;				/* number of aliases */
};

struct MsgStruct2 {					/* structure #2 */
	dword ms2_ms3;					/* ref to associated 1st struc #3 */
	int ms2_num;					/* number of messages */
	msgTGroup ms2_gid;				/* group ID */
};

struct MsgStruct3 {					/* structure #3 */
	dword ms3_string;				/* reference to string */
	int ms3_length;					/* length of message string */
	msgTExit ms3_exitcode;			/* Exit code */
	msgTClass ms3_class;			/* class of message */
};

struct MsgStructIDS {				/* entry of assoc msgID<->msgName */
	msgTMagic msIDS_magicnumber;
	dword msIDS_next;				/* next entry */
};

struct MsgStructID {				/* one assoc msgID <-> msgName */
	int msID_length;				/* length of name of this entry */
	MSGID2 msID_id;					/* msgID of this item */
									/* == 0 ==> end of list */
};

#include <algndflt.h>	/* enable default alignment */

#ifdef _MICROC_
#define MsgStruMN struct MsgStructMN
#define MsgStru0 struct MsgStruct0
#define MsgStru1 struct MsgStruct1
#define MsgStru2 struct MsgStruct2
#define MsgStru3 struct MsgStruct3
#define MsgStruIDS struct MsgStructIDS
#define MsgStruID struct MsgStructID
#else /*#	!(defined(_MICROC_)) */
typedef struct MsgStructMN MsgStruMN;
typedef struct MsgStruct0 MsgStru0;
typedef struct MsgStruct1 MsgStru1;
typedef struct MsgStruct2 MsgStru2;
typedef struct MsgStruct3 MsgStru3;
typedef struct MsgStructIDS MsgStruIDS;
typedef struct MsgStructID MsgStruID;
#endif /*#	defined(_MICROC_) */


void msg2Init(void);
/*	Initialize the access method #2 subsystem
	+ Aquiring the currently active character set & language code
	+ Reading the default catalogues
 */

#define msg2Exit()
/* Shutdown the access method #2 subsystem
	--> nothing
*/


#if MSG_METHOD == 2
	/* Definitions for access method #2 only */


extern msgTCharset msg_CurCharset;			/* Character set to use */
extern msgTLngcode msg_CurLngcode;			/* language code to use */

/*
 *	Macros mapping the global names into access method #2 conform
 *	names.
 *	Because all access method exist in parallel to each other within the
 *	archive MSGLIB.LIB, the identifiers must differ from each other.
 */
#define MSGID MSGID2

#define msgLock msg2Lock
#define msgRetrieve msg2Retrieve
#define msgUnlock msg2Unlock
#define msgUnlockString msg2UnlockString
#define msgExitCode msg2ExitCode
#define MSGFILE MSG2FILE
#define MSGNOFILE MSG2NOFILE
#define msgLanguage msg2Language
#define msgCharset msg2Charset
#define msgRereadCats msg2RereadCats
#define msgCatalogue msg2Catalogue
#define msgDatabase msg2Database
#define msgIdentCat msg2IdentCat
#define msgMatchLngcode msg2MatchLngcode
#define msgValidMagicNumber msg2ValidMagicNumber
#define msgMatchCharset msg2MatchCharset
	/* From msgemit.c */
#define msgErrFct msg2ErrFct
#define interactive msg2interactive
#define message msg2message
#define error msg2error
#define fatal msg2fatal
#define warning msg2warning
#define informative msg2informative
#define smessage msg2smessage
#define setNoiseLevel msg2setNoiseLevel
#define getNoiseLevel msg2getNoiseLevel
#define msgInitMV msg2InitMV
	/* From msgprntf.c */
#define msgGetAnsiFlag msg2GetAnsiFlag
#define msgSetAnsiFlag msg2SetAnsiFlag
#define checkAnsi msg2checkAnsi
#define skprintf msg2skprintf
#define fd_sprintf msg2fd_sprintf
#define fd_fprintf msg2fd_fprintf
	/* From msghlpsc.c */
#define hlpScreen msg2hlpScreen
	/* From getopt */
#define getopt msg2getopt
#define getopt1 msg2getopt1
#define getopt2 msg2getopt2
#define getoptG msg2getoptG

int getopt(int argc, char **argv, char *opt, char *argopt);
int getopt1(int argc, char **argv, char *opt, char *argopt);
int getopt2(int argc, char **argv, char *opt, char *argopt, char *specopt);
int getoptG(int argc, char **argv);

int msg2ValidMagicNumber(msgTMagic mnum);
/*
 *	Check if 'mnum' is a valid magic number
 */

int msg_mtchCSet(msgTCharset cs1, msgTCharset cs2);
/*
 *	Check if those two character sets are identical
 */

#define msg2MatchCharset(cset)		\
	msg_mtchCSet(msg_CurCharset, cset)
/*
 *	Check if the character set 'cset' matches the currently active
 *	character set.
 */

#define msg2MatchLngcode(lng)		\
	(msg_CurLngcode == (lng))
/*
 *	Check if the language code 'lng' is equal to the currently active
 *	language (country) code.
 */

void msg2Language(msgTLngcode lng);
/*
 *	Set which language to use, by the default the language is the
 *	currently active country code.
 */

void msg2Charset(msgTCharset cset);
/*
 *	Set which character set to use, by the default is the currently active
 *	codepage.
 */

int msg2RereadCats(void);
/*
 *	Re-read all catalogues, thus, applying parameters changed lately.
 *
 *	If MSG_SINGLE_CATALOGUE_FILE is defined, no such chain exists,
 *	but all the standard catalogues are read.
 */


/*********************
		Catalogue file functions
				****************/

int msg2Catalogue(const char * const fnam);
/*
 *	Append the catalogue 'fnam' at the end of the catalogue chain.
 *	(Last items superceed previous items). And insert any found
 *	language definitions that matches better or equal than the currently
 *	found languages into the global catalogue structure (msg_glbCat).
 *
 *	This function should be called, if the user wants to use a catalogue
 *	that is shipped as an external and free-standing file itself.
 *
 *	The file must be a binary catalogue in the compiler's native format.
 *
 *	The catalogue is searched the following order to gain the knowledge
 *	about if the file is a catalogue at all:
 *	1) probe the end of the file for MsgStru0,
 *	2) probe the beginning of the file for any valid magic number.
 *	
 *	If step 1) succeeds, anything between the beginning of MsgStru0
 *	and the position MsgStru0 contains is considered to be part of
 *	of the catalogue. If step 2) succeeds and the first item is a
 *	MsgStru0, everything between the position contained in MsgStru0 up to
 *	the end of the file is considered to form the catalogue. If step 2)
 *	succeeds, but the first entry is not a MsgStru0, the whole file is
 *	treated as a catalogue.
 *
 *	With this information, msgDatabase() is called (see there for more
 *	details).
 *
 *	Return:	same as msgDatabase()
 */

int msg2Database(MSGFILE f, dword iM(*)start, dword iM(*)limit);
/*
 *	Same as msgCatalogue(), but the start and end of the catalogue is
 *	known and the file has been opened.
 *
 *	This function should be called, if the user wants to use a catalogue
 *	that is physically embedded into another file. However, that file
 *	itself must not be compressed or something like that.
 *
 *	The function steps the chain of lamguage definitions within the
 *	catalogue until it reaches the position 'limit', an invalid
 *	position of the next item (to be more specific: a zero 'next'
 *	pointer), a MsgStru0 magic number, an invalid magic number,
 *	or if a read error occures during reading the control area of
 *	the language definitions.
 *
 *	No message text is retrieved so far, nor is the text area scanned
 *	for read errors. So a success return value does not indicate that
 *	the message strings are physically available, only that the
 *	logical access paths are available.
 *
 *	Return value:
 * 		 > 0: successful scan of the whole catalogue, number of scanned
 *				language definitions.
 *		 < 0: an error occured during read/invalif magic number or such;
 *				abs(rv) == number of successfully scanned language definitions
 *		== 0: no successful scan of one language definition; that may
 *				indicate an empty or completely broken catalogue.
 *
 *	The file can be closed upon return of this function, because msgDatabase()
 *	duplicates its own copy of the file descriptor and takes care to close
 *	it if applicable.
 */

int msg2IdentMSGLIB(const MsgStru0 * const msg0);
/* Verify if the structure is filled with valid values.

	Return:
 		== 0: OK
 		 > 0: specific error
 		 < 0: unspecific error
*/

MSGFILE msg2IdentCat(const char * const fnam, dword * const Xstart
	, dword * const Xlimit);
/*
 *	Identify if the catalogue file really contains a valid catalogue
 *
 *	On success, the start and limit varibles receive the absolute address
 *	to the first item within the catalogue and to the first byte, that's no
 *	part of the item chain
 *
 *	Return:
 *		== MSGNOFILE: on failure
 *		else: opened catalogue file
 */



#define msgInit() msgInitMI(); msgInitMV(); msg2Init();
/* Initialize the message system for method #2.
	Call the appropriate initialization functions of the various
	subsystems:
	+ messager interpreter
	+ messager visulator
	+ access method #2
*/


/*********************
		Message accessing functions
				****************/

char *msg2Lock(MSGID2 msgid);
/* Lock the message with the numerical message ID msgid and return a
	pointer to the message text.

	The locked message must be unlocked by either the msgUnlock() or
	msgUnlockString() function.

	If a message is locked multiple times, it shares the same resources.

	The returned string must not be modified.

	Return:
		always != NULL, the pointer to the message string. If no such
		message was found within the catalogue file, a default message
		is generated.
*/

char *msg2Retrieve(MSGID2 msgid);
/* Unlock the previous "msgRetrieve"ed message, then lock the specified
	one.

	This function is a combination of msgUnlock() and msgLock()

	Return:
		sam as msgLock(msgid);
*/

void msg2Unlock(MSGID2 msgid);
/* Unlock the message with the numerical message ID.

	'msgid' might be an invalid msgID or not locked at all.
*/

void msg2UnlockString(char *str);
/* Unlock the message which assoicated msgLock() call returned 'str'.

	'str' might be an invalid pointer or any other valid pointer into the
	data area, but not returned by msgLock().
*/

int msg2ExitCode(MSGID2 msgid);
/* Return the exit code associated with the message.

	During this call the message will be locked & unlocked automatically.

	Return:
		0..255: An exit code; if the message is not a member of the "E" class,
			the return value is invalid, maybe a random number.
*/


#endif /*#	MSG_METHOD == 2 */

#endif /*#	!defined(__MSGCONF2_H) */
