/*
    MSGLIB - a message handling library
    Copyright (C) 1995,1997-98  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/* $RCSfile: MSG2IDNT.C $
   $Locker: ska $	$Name:  $	$State: Rel $

	MSGFILE msgIdentCat(char *fnam, dword *pos, dword *limit)

	Identify if a file is a binary message library and return an
	opened file descriptor and the start and end file positions.

	The file must be a binary catalogue in the compiler's native format.

	The catalogue is searched the following order to gain the knowledge
	about if the file is a catalogue at all:
	1) probe the end of the file for MsgStru0,
	2) probe the beginning of the file for any valid magic number.
	
	If step 1) succeeds, anything between the beginning of MsgStru0
	and the position MsgStru0 contains is considered to be part of
	of the catalogue. If step 2) succeeds and the first item is a
	MsgStru0, everything between the position contained in MsgStru0 up to
	the end of the file is considered to form the catalogue. If step 2)
	succeeds, but the first entry is not a MsgStru0, the whole file is
	treated as a catalogue.

	Return:	MSGNOFILE:	failure
			else: opened catalogue file descriptor
 
*/

#include <portable.h>
#include <string.h>
#include "msg2.h"
#include "fileIO.h"

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: MSG2IDNT.C 1.3 1998/10/07 04:42:53 ska Rel ska $";
#endif /*#	defined(RCS_Version) */

MSGFILE msgIdentCat(const char * const fnam, dword * const Xstart
	, dword * const Xlimit)
{	MSGFILE fd;
	MsgStru0 msg0;

#ifdef _MICROC_
#define start Xstart
#define limit Xlimit
#else /*#	!(defined(_MICROC_)) */
#define start (*Xstart)
#define limit (*Xlimit)
#endif /*#	defined(_MICROC_) */

	if((fd = msg2OpenCat(fnam)) == MSGNOFILE)
		return MSGNOFILE;

	if(!Mseeke(fd) || !Mgetpos(fd, limit)
#ifdef _MICROC_
	 || limit->hi == 0 && limit->lo <= sizeof(MsgStru0)
#else /*#	!(defined(_MICROC_)) */
	 || limit <= sizeof(MsgStru0)
#endif /*#	defined(_MICROC_) */
	 || !Mseek(fd, -sizeof(MsgStru0))
	 || !MSread(fd, msg0)
	)
		goto errRet;

	if(msg2IdentMSGLIB(aS(msg0)) == 0) {		/* MSGLIB ends in valid Stru0 */
		/* The start of the entry chain starts at ms0_address and
			ends at the start of the Stru0, because *limit is currently
			the end of file position, it must be decremented by the size
			of it.

			The ms0_address member is relative to the end of file rather
			than the beginning of the structure as any other item
			of the MSGLIB chain.
		*/
		longcpy(start, msg0.ms0_address);
		longadd(start, limit);		/* transform relative -> absolute addr */

		longsubu(limit, sizeof(msg0));
	}
	else {		/* try the start of the file for a valid magic number */
		if(!Mseekb(fd)
		 || !MSread(fd, msg0)
		 || !msgValidMagicNumber(msg0.ms0_magicnumber))
		 	goto errRet;

		if(msg0.ms0_magicnumber != MSG_MN_STRU0)
			longset(start, 0);
		else
			/* Stru0 must be skip, otherwise it would be interpreted as
				the end of the item chain */
			longcpy(start, msg0.ms0_address);
				/* no transformation necessary, because start of Stru0
					is zero "0" */

		/* limit is already set to the end of the file */
	}

	return fd;

errRet:
	msg2CloseCat(fd);
	return MSGNOFILE;
}
