/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggTheora SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE Theora SOURCE CODE IS COPYRIGHT (C) 2002-2003                *
 * by the Xiph.Org Foundation http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

  function: Ogg encoder application; makes an Ogg Theora/Vorbis
            file from compatible input, using Allegro and APEG

 ********************************************************************/

#define _GNU_SOURCE
#define _LARGEFILE_SOURCE
#define _LARGEFILE64_SOURCE
#define _FILE_OFFSET_BITS 64

#ifndef _REENTRANT
#define _REENTRANT
#endif

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <time.h>
#include <math.h>
#include <ctype.h>

#include <allegro.h>

#include <theora/theora.h>
#include <vorbis/codec.h>
#include <vorbis/vorbisenc.h>

#include "apeg.h"

#define SW	640
#define SH	480

#ifdef _MSC_VER
/* supply missing headers and functions to Win32 */

#include <fcntl.h>

#define strcasecmp stricmp
static double rint(double x)
{
	if (x < 0.0)
		return ceil(x - 0.5);
	else
		return floor(x + 0.5);
}
#endif

const char *optstring = "o:a:A:v:V:s:S:f:F:e:";
struct option options [] = {
	{"output",required_argument,NULL,'o'},
	{"audio-rate-target",required_argument,NULL,'A'},
	{"video-rate-target",required_argument,NULL,'V'},
	{"audio-quality",required_argument,NULL,'a'},
	{"video-quality",required_argument,NULL,'v'},
	{"aspect-numerator",optional_argument,NULL,'s'},
	{"aspect-denominator",optional_argument,NULL,'S'},
	{"framerate-numerator",optional_argument,NULL,'f'},
	{"framerate-denominator",optional_argument,NULL,'F'},
	{"auto-encode",optional_argument,NULL,'e'},
	{NULL,0,NULL,0}
};


static int openfile_pclose(void *_f)
{
	FILE *f = _f;
	return pclose(f);
}

static int openfile_getc(void *_f)
{
	FILE *f = _f;
	return fgetc(f);
}

static int openfile_ungetc(int c, void *_f)
{
	FILE *f = _f;
	return ungetc(c, f);
}

static long openfile_fread(void *p, long n, void *_f)
{
	FILE *f = _f;
	unsigned char *cp = (unsigned char *)p;
	return fread(cp, 1, n, f);
}

static int openfile_putc(int c, void *_f)
{
	FILE *f = _f;
	return fputc(c, f);
}

static long openfile_fwrite(AL_CONST void *p, long n, void *_f)
{
	FILE *f = _f;
	AL_CONST unsigned char *cp = (AL_CONST unsigned char *)p;
	return fwrite(cp, 1, n, f);
}

static int openfile_fseek(void *_f, int offset)
{
	FILE *f = _f;
	return fseek(f, offset, SEEK_CUR);
}

static int openfile_feof(void *_f)
{
	FILE *f = _f;
	return feof(f);
}

static int openfile_ferror(void *_f)
{
	FILE *f = _f;
	return ferror(f);
}

static PACKFILE_VTABLE openfile_vtable =
{
	openfile_pclose,
	openfile_getc,
	openfile_ungetc,
	openfile_fread,
	openfile_putc,
	openfile_fwrite,
	openfile_fseek,
	openfile_feof,
	openfile_ferror
};


/* You'll go to Hell for using globals. */
static PACKFILE *outfile=NULL;
static PACKFILE *audio=NULL;
static struct {
	enum { type_none, type_yuv4mpeg2, type_ff } type;
	PACKFILE *pf;

	char *path;
	struct al_ffblk ff_info;
	int ff_done;
} video;

/* Audio, video, and output filenames (TODO: or pipe-commands; should rename
   them) */
static char audio_fname[512];
static char video_fname[512];
static char ofile_fname[512];

static int audio_ch;
static int audio_hz;
static int audio_bt;

static float audio_q;
static int audio_r;

static float audio_skip;

static char askip_n[6]="0";


static int video_x;
static int video_y;
static int frame_x;
static int frame_y;
static int frame_x_offset;
static int frame_y_offset;
static int video_hzn=-1;
static int video_hzd=-1;
static int video_an=-1;
static int video_ad=-1;

static int video_r;
static int video_q;

//static float video_skip;

static int override_fps;
static int override_aspect;

static char fps_n[6]="0", fps_d[5]="0";
static char aspect_n[6]="0", aspect_d[5]="0";


static int serial1;
static int serial2;

static char video_kbps[9]="0";
static char audio_kbps[8]="0";

static int one_shot;

static int vstate;
static char *yuvframe[2];

static char buffer[4096];

static int quit_now;


static void usage(void)
{
	fprintf(stderr,
"Usage: encoder_example [options] [input files...]\n"
"\n"
"Options:\n"
"\n"
"  -o --output <filename.ogg>     file name for encoded output.\n"
"\n"
"  -a --audio-quality <n>         Vorbis quality selector from -1 to 10\n"
"                                 (-1 yields smallest files but lowest\n"
"                                 fidelity; 10 yields highest fidelity\n"
"                                 but large files. '2' is a reasonable\n"
"                                 default).\n"
"  -A --audio-rate-target <n>     bitrate target for Vorbis audio;\n"
"                                 use -a and not -A if at all possible,\n"
"                                 as -a gives higher quality for a given\n"
"                                 bitrate.\n"
"\n"
"  -v --video-quality <n>         Theora quality selector fro 0 to 10\n"
"                                 (0 yields smallest files but lowest\n"
"                                 video quality. 10 yields highest\n"
"                                 fidelity but large files).\n"
"  -V --video-rate-target <n>     bitrate target for Theora video\n"
"\n"
"  -s --aspect-numerator <n>      Aspect ratio numerator, default is 0\n"
"                                 or extracted from YUV input file\n"
"  -S --aspect-denominator <n>    Aspect ratio denominator, default is 0\n"
"                                 or extracted from YUV input file\n"
"  -f --framerate-numerator <n>   Frame rate numerator, can be extracted\n"
"                                 from YUV input file. ex: 30000000\n"
"  -F --framerate-denominator <n> Frame rate denominator, can be extracted\n"
"                                 from YUV input file. ex: 1000000\n"
"                                 The frame rate nominator divided by this\n"
"                                 determines the frame rate in units per tick\n"
"\n"
"  -e --auto-encode               Start encoding automatically, and\n"
"                                 automatically exit\n"
"\n"
"ogg_encoder accepts uncompressed RIFF WAV data for audio and uncompressed\n"
"YUV4MPEG2 data or a series of compatible image formats (BMP, PCX, or TGA) for\n"
"video. Input files *cannot* be FIFO buffers. If you wish to use FIFO buffers,\n"
"select Audio/Video pipe in the GUI and specify a command to write the active\n"
"FIFO to stdout. (example: 'cat stream.yuv')\n\n");
	exit(1);
}


static int quit()
{
	if(alert("", "Are you sure you want to quit?", "", "Yes", "No", 0, 0) == 1)
		exit(0);
	return D_O_K;
}

static int d_tail_proc(int msg, DIALOG *d, int c)
{
	if(quit_now || (one_shot && msg == MSG_IDLE) ||
	   (msg == MSG_CLICK && (d->flags&D_EXIT)))
		return D_CLOSE;

	return D_O_K;
	(void)d;
	(void)c;
}

static int message(const char *buf)
{
	static DIALOG about_dlg[6] = {
		{ .proc=d_shadow_box_proc, .x=0, .y=0, .w=320, .h=160 },
		{ .proc=d_textbox_proc, .x=4, .y=4, .w=320-4*2, .h=160-32,
		  .dp=NULL },
		{ .proc=d_button_proc, .x=160-32, .y=160-24, .w=64, .h=16,
		  .flags=D_EXIT, .dp="Okay" },
		{ .proc=d_tail_proc },
		{ .proc=d_yield_proc },
		{ .proc=NULL }
	};
	about_dlg[1].dp = (void*)buf;
	centre_dialog(about_dlg);
	set_dialog_color(about_dlg, gui_fg_color, gui_bg_color);

	popup_dialog(about_dlg, 2);

	return D_O_K;
}


static void cleanup()
{
	free(yuvframe[0]);
	yuvframe[0] = NULL;
	free(yuvframe[1]);
	yuvframe[1] = NULL;
	vstate = 0;

	if(outfile)
		pack_fclose(outfile);
	outfile = NULL;
	if(audio)
		pack_fclose(audio);
	audio = NULL;
	if(video.pf)
		pack_fclose(video.pf);
	video.pf = NULL;
	if(video.type == type_ff)
		al_findclose(&video.ff_info);

	free(video.path);
	video.path = NULL;

	video.type = type_none;
}

static unsigned int sample_delay;
static int fetch_audio(PACKFILE *audio, ogg_page *audiopage,
                       ogg_stream_state *vo, vorbis_dsp_state *vd,
                       vorbis_block *vb)
{
	const int toread = sizeof(buffer)/(audio_bt/8)/audio_ch;
	ogg_packet op;
	int bytesread;
	int sampread;
	int i, j;

	while(TRUE)
	{
		/* process any audio already buffered */
		if(ogg_stream_pageout(vo, audiopage))
			return 1;
		if(ogg_stream_eos(vo))
			return 0;

		/* read and process more audio */
		if(sample_delay)
		{
			if(audio_bt == 8)
			{
				for(bytesread = 0;bytesread < toread*audio_ch &&
				                  sample_delay > 0;++bytesread)
				{
					((char*)buffer)[bytesread] = 0x80;
					--sample_delay;
				}
			}
			else
			{
				for(bytesread = 0;bytesread < toread*audio_ch &&
                                  sample_delay > 0;bytesread+=2)
                {
                    ((short*)buffer)[bytesread/2] = 0;
                    --sample_delay;
                }
			}
		}
		else
			bytesread = pack_fread(buffer, toread*audio_bt*audio_ch/8, audio);
		sampread = bytesread/(audio_bt/8)/audio_ch;

		if(sampread > 0)
		{
			uint16_t *buf16 = (uint16_t*)buffer;
			uint8_t *buf8 = (uint8_t*)buffer;
			float **vorbis_buffer;

			vorbis_buffer = vorbis_analysis_buffer(vd, sampread);

			/* uninterleave samples */
			for(j = 0;j < audio_ch;j++)
			{
				float *vb = vorbis_buffer[j];
				for(i = 0;i < sampread;i++)
				{
					if(audio_bt == 16)
						vb[i] = ((buf16[(i*audio_ch)+j]^0x8000)/32767.5) - 1.0;
					else
						vb[i] = (buf8[(i*audio_ch)+j]/127.5) - 1.0;
				}
			}
		}
		vorbis_analysis_wrote(vd, ((sampread>=0) ? sampread : 0));

		while(vorbis_analysis_blockout(vd, vb) == 1)
		{
			/* analysis, assume we want to use bitrate management */
			vorbis_analysis(vb, NULL);
			vorbis_bitrate_addblock(vb);

			/* weld packets into the bitstream */
			while(vorbis_bitrate_flushpacket(vd, &op))
				ogg_stream_packetin(vo, &op);
		}
	}
}


static void convert_rgb(BITMAP *bmp, unsigned char *yuv_buf,
                        int yuv_w, int yuv_h)
{
	int y_stride = yuv_w;
	int uv_stride = yuv_w / 2;

	unsigned char *y_dst = yuv_buf + yuv_w*frame_y_offset + frame_x_offset;
	unsigned char *u_dst = yuv_buf + (yuv_w*yuv_h) + ((yuv_w/2)*
	                                  (frame_y_offset/2)) + frame_x_offset/2;
	unsigned char *v_dst = yuv_buf + (yuv_w*yuv_h*5/4) + ((yuv_w/2)*
	                                  (frame_y_offset/2)) + frame_x_offset/2;

	int w = MIN(bmp->w, yuv_w)&~1;
	int h = MIN(bmp->h, yuv_h)&~1;
	int x, y;

	for(y = 0;y < h;y+=2)
	{
		for(x = 0;x < w;++x)
		{
			int r, g, b, val;

			val = _getpixel24(bmp, x, y);
			r = getb24(val);
			g = getg24(val);
			b = getr24(val);

			val = (0.299*r + 0.587*g + 0.114*b + 0.5) + 16;
			y_dst[x] = MID(16, val, 235);

			if(!(x&1))
			{
				val = _getpixel24(bmp, x, y+1);
				r += getb24(val);
				g += getg24(val);
				b += getr24(val);
				val = _getpixel24(bmp, x+1, y);
				r += getb24(val);
				g += getg24(val);
				b += getr24(val);
				val = _getpixel24(bmp, x+1, y+1);
				r += getb24(val);
				g += getg24(val);
				b += getr24(val);

				r /= 4;
				g /= 4;
				b /= 4;

				val = ( 0.500*r - 0.419*g - 0.081*b + 0.5) + 128;
				val = MID(16, val, 239);
				u_dst[x>>1] = val;

				val = (-0.169*r - 0.331*g + 0.500*b + 0.5) + 128;
				val = MID(16, val, 239);
				v_dst[x>>1] = val;
			}
		}
		y_dst += y_stride;
		++y;

		for(x = 0;x < w;++x)
		{
			int r, g, b, val;

			val = _getpixel24(bmp, x, y);
			r = getb24(val);
			g = getg24(val);
			b = getr24(val);

			val = (0.299*r + 0.587*g + 0.114*b + 0.5) + 16;
			y_dst[x] = MID(16, val, 235);
		}
		y_dst += y_stride;
		u_dst += uv_stride;
		v_dst += uv_stride;
	}
}

static int fetch_video(ogg_page *videopage, ogg_stream_state *to,
                       theora_state *td)
{
	yuv_buffer yuv;
	ogg_packet op;
	char *line;
	int i, e;

	/* get a video page */
	while(TRUE)
	{
		if(ogg_stream_pageout(to, videopage))
			return 1;
		if(ogg_stream_eos(to))
 			return 0;

		/* read and process more video */
		/* video strategy reads one frame ahead so we know when we're
		   at end of stream and can mark last video frame as such
		   (vorbis audio has to flush one frame past last video frame
		   due to overlap and thus doesn't need this extra work */

		/* have two frame buffers full (if possible) before
		   proceeding.  after first pass and until eos, one will
		   always be full when we get here */

		for(i = vstate;i < 2;i++)
		{
			if(video.type == type_yuv4mpeg2)
			{
				char frame[6];
				int ret, c;

				/* match and skip the frame header */
				ret = pack_fread(frame, 6, video.pf);
				if(ret < 6)
					break;
				if(memcmp(frame, "FRAME", 5) != 0)
				{
					if(alert("", "Loss of framing in YUV input data", "",
					         "Stop", (audio ? "Continue" : NULL), KEY_ESC,
					          0) == 1)
						return -1;
					return 0;
				}
				if(frame[5]!='\n')
				{
					int j;
					for(j = 0;j < 79;j++)
					{
						c = pack_getc(video.pf);
						if(c == EOF && pack_feof(video.pf))
							j = 79;
						else if(c == '\n')
							break;
					}
					if(j==79)
					{
						if(alert("", "Error parsing YUV frame header", "",
						         "Stop", (audio ? "Continue" : NULL), KEY_ESC,
						          0) == 1)
							return -1;
						return 0;
					}
				}

				/* read the Y plane into our frame buffer with centering */
				line = yuvframe[i] + video_x*frame_y_offset + frame_x_offset;
				for(e = 0;e < frame_y;e++)
				{
					ret = pack_fread(line, frame_x, video.pf);
					if(ret != frame_x)
						break;
					line += video_x;
				}
				/* now get U plane*/
				line = yuvframe[i] + (video_x*video_y) + ((video_x/2)*
				       (frame_y_offset/2)) + frame_x_offset/2;
				for(e = 0;e < frame_y/2;e++)
				{
					ret = pack_fread(line, frame_x/2, video.pf);
					if(ret != frame_x/2)
						break;
					line += video_x/2;
				}
				/* and the V plane*/
				line = yuvframe[i] + (video_x*video_y*5/4) + ((video_x/2)*
				       (frame_y_offset/2)) + frame_x_offset/2;
				for(e = 0;e < frame_y/2;e++)
				{
					ret = pack_fread(line, frame_x/2, video.pf);
					if(ret != frame_x/2)
						break;
					line += video_x/2;
				}
			}
			else if(video.type == type_ff)
			{
				BITMAP *bmp;
				int cd;

				if(video.ff_done)
					break;

				snprintf(buffer, sizeof(buffer), "%s%s", video.path,
				         video.ff_info.name);

				cd = get_color_depth();

				/* Force image to load as 24-bit */
				set_color_depth(24);
				bmp = load_bitmap(buffer, NULL);
				set_color_depth(cd);

				if(!bmp)
				{
					if(alert("Couldn't load image", video.ff_info.name, "",
					         "Stop", (audio ? "Continue" : NULL), 0, 0) == 1)
						return -1;
					video.ff_done = 1;
					break;
				}
				convert_rgb(bmp, yuvframe[i], video_x, video_y);
				destroy_bitmap(bmp);
				video.ff_done = al_findnext(&video.ff_info);
			}
			vstate++;
		}

		if(vstate < 1)
		{
			/* can't get here unless stream has no video */
			if(alert("", "Video input contains no frames.", "", "Stop",
			         (audio ? "Continue" : NULL), KEY_ESC, 0) == 1)
				return -1;
			return 0;
		}

		/* Theora is a one-frame-in,one-frame-out system; submit a frame
		   for compression and pull out the packet */
		yuv.y_width = video_x;
		yuv.y_height = video_y;
		yuv.y_stride = video_x;

		yuv.uv_width = video_x/2;
		yuv.uv_height = video_y/2;
		yuv.uv_stride = video_x/2;

		yuv.y = yuvframe[0];
		yuv.u = yuvframe[0] + video_x*video_y;
		yuv.v = yuvframe[0] + video_x*video_y*5/4;

		theora_encode_YUVin(td, &yuv);

		/* if there's only one frame, it's the last in the stream */
		if(vstate < 2)
			theora_encode_packetout(td, 1, &op);
		else
			theora_encode_packetout(td, 0, &op);

		ogg_stream_packetin(to, &op);

		line = yuvframe[0];
		yuvframe[0] = yuvframe[1];
		yuvframe[1] = line;
		vstate--;
	}
}


static int d_encoder_proc(int msg, DIALOG *d, int c)
{
	static ogg_int64_t audio_bytesout;
	static ogg_int64_t video_bytesout;
	static double timebase;
	static int audioflag;
	static int videoflag;
	static int do_video;

	static ogg_stream_state to; /* take physical pages, weld into a logical
	                               stream of packets */
	static ogg_stream_state vo; /* take physical pages, weld into a logical
	                               stream of packets */
	static ogg_page         og; /* one Ogg bitstream page. Vorbis packets are
	                               inside */
	static ogg_packet       op; /* one raw packet of data for decode */

	static ogg_page audiopage;
	static ogg_page videopage;

	static vorbis_info      vi; /* struct that stores all the static vorbis
	                               bitstream settings */
	static vorbis_comment   vc; /* struct that stores all the user comments */
	static vorbis_dsp_state vd; /* central state for the local packet->PCM
	                               decoder */
	static vorbis_block     vb; /* working space for packet->PCM decode */

	static theora_state   td;
	static theora_comment tc;

	static enum {
		init, working, done
	} status;

	int ret;
	(void)c;

	switch(msg)
	{
		case MSG_START:
			d[1].dp = "Cancel";
			status = init;
			audio_bytesout = 0;
			video_bytesout = 0;
			timebase = 0.0;
			audioflag = 0;
			videoflag = 0;

			memset(&to, 0, sizeof(to));
			memset(&vo, 0, sizeof(vo));
			memset(&og, 0, sizeof(og));
			memset(&op, 0, sizeof(op));
			memset(&audiopage, 0, sizeof(audiopage));
			memset(&videopage, 0, sizeof(videopage));
			memset(&td, 0, sizeof(td));
			memset(&tc, 0, sizeof(tc));
			memset(&vi, 0, sizeof(vi));
			memset(&vc, 0, sizeof(vc));
			memset(&vd, 0, sizeof(vd));
			memset(&vb, 0, sizeof(vb));

			if(video.type == type_none && !audio)
			{
				alert("", "No files specified!", "", "Okay", NULL, 0, 0);
				return D_CLOSE;
			}

			ogg_stream_init(&to, serial1);
			ogg_stream_init(&vo, serial2);

			/* Set up Theora encoder */
			if(video.type != type_none)
			{
				theora_info    ti;

				/* Theora has a divisible-by-sixteen restriction for the
				   encoded video size */
				/* scale the frame size up to the next multiple of 16 and
				   calculate offsets */
				video_x = (frame_x+15) & ~15;
				video_y = (frame_y+15) & ~15;

				/* We force the offset to be even. This ensures that the
				   chroma samples align properly with the luma samples. */
	 			frame_x_offset = ((video_x-frame_x)/2) & ~1;
				frame_y_offset = ((video_y-frame_y)/2) & ~1;

				theora_info_init(&ti);

				ti.width = video_x;
				ti.height = video_y;
				ti.frame_width = frame_x;
				ti.frame_height = frame_y;
				ti.offset_x = frame_x_offset;
				ti.offset_y = frame_y_offset;
				ti.fps_numerator = video_hzn;
				ti.fps_denominator = video_hzd;
				ti.aspect_numerator = video_an;
				ti.aspect_denominator = video_ad;
				ti.colorspace = OC_CS_UNSPECIFIED;
				ti.pixelformat = OC_PF_420;
				ti.target_bitrate = video_r;
				ti.quality = video_q;

				ti.dropframes_p = 0;
				ti.quick_p = 1;
				ti.keyframe_auto_p = 1;
				ti.keyframe_frequency = 64;
				ti.keyframe_frequency_force = 64;
				ti.keyframe_data_target_bitrate = video_r*1.5;
				ti.keyframe_auto_threshold = 80;
				ti.keyframe_mindistance = 8;
				ti.noise_sensitivity = 1;

				theora_encode_init(&td, &ti);
				theora_info_clear(&ti);
			}

			/* initialize Vorbis too, assuming we have audio to compress. */
			if(audio)
			{
				vorbis_info_init(&vi);
				if(audio_q > -99.0)
					ret = vorbis_encode_init_vbr(&vi, audio_ch, audio_hz,
					                             audio_q);
				else
					ret = vorbis_encode_init(&vi, audio_ch, audio_hz, -1,
					                         audio_r, -1);
				if(ret)
				{
					alert("The Vorbis encoder could not set up a mode",
                          "according to the requested quality or bitrate",
					      "", "Okay", NULL, 0, 0);
					ogg_stream_clear(&vo);
					vorbis_info_clear(&vi);
					pack_fclose(audio);
					audio = NULL;
					return D_CLOSE;
				}

				vorbis_comment_init(&vc);
				vorbis_analysis_init(&vd, &vi);
				vorbis_block_init(&vd, &vb);
			}

			/* write the bitstream header packets with proper page
			   interleave. first packet will get its own page automatically */
			if(video.type != type_none)
			{
				theora_encode_header(&td, &op);
				ogg_stream_packetin(&to, &op);
				if(ogg_stream_pageout(&to, &og) == 0)
				{
					alert("", "Internal Ogg library error", "", "Okay", NULL,
					      0, 0);
					ogg_stream_clear(&to);
					cleanup();
					return D_CLOSE;
				}
				pack_fwrite(og.header, og.header_len, outfile);
				pack_fwrite(og.body, og.body_len, outfile);

				/* create the remaining theora headers */
				theora_comment_init(&tc);
				theora_encode_comment(&tc, &op);
				ogg_stream_packetin(&to, &op);
				theora_encode_tables(&td, &op);
				ogg_stream_packetin(&to, &op);

				/* initialize the double frame buffer */
				free(yuvframe[0]);
				free(yuvframe[1]);
				yuvframe[0] = malloc(video_x*video_y*3/2);
				yuvframe[1] = malloc(video_x*video_y*3/2);
				if(!yuvframe[0] || !yuvframe[1])
				{
					alert("Internal error", "",
					      "(yuvframe[] allocation failed)","Okay", NULL, 0, 0);
					cleanup();
					return D_CLOSE;
				}

				/* clear the frame as it may be larger than the actual video
				   data: fill Y with 0x10 and UV with 0x80, for black data */
				memset(yuvframe[0], 0x10, video_x*video_y);
				memset(yuvframe[0]+video_x*video_y, 0x80, video_x*video_y/2);
				memset(yuvframe[1], 0x10, video_x*video_y);
				memset(yuvframe[1]+video_x*video_y, 0x80, video_x*video_y/2);
			}

			if(audio)
			{
				ogg_packet header;
				ogg_packet header_comm;
				ogg_packet header_code;

				vorbis_analysis_headerout(&vd, &vc, &header, &header_comm,
				                          &header_code);
				/* automatically placed in its own page */
				ogg_stream_packetin(&vo, &header);
				if(ogg_stream_pageout(&vo, &og) == 0)
				{
					alert("", "Internal Ogg library error", "", "Okay", NULL,
					      0, 0);
					return D_CLOSE;
				}
				pack_fwrite(og.header, og.header_len, outfile);
				pack_fwrite(og.body, og.body_len, outfile);

				/* remaining vorbis header packets */
				ogg_stream_packetin(&vo, &header_comm);
				ogg_stream_packetin(&vo, &header_code);
			}

			/* Flush the rest of our headers. This ensures
			   the actual data in each stream will start
			   on a new page, as per spec. */
			if(video.type != type_none)
			{
				while(1)
				{
					int result = ogg_stream_flush(&to, &og);
					if(result < 0)
					{
						/* can't get here */
						alert("", "Internal Ogg library error", "", "Okay",
						      NULL, 0, 0);
						return D_CLOSE;
					}
					if(result == 0)
						break;
					pack_fwrite(og.header, og.header_len, outfile);
					pack_fwrite(og.body, og.body_len, outfile);
				}
			}
			if(audio)
			{
				while(1)
				{
					int result = ogg_stream_flush(&vo, &og);
					if(result < 0)
					{
						/* can't get here */
						alert("", "Internal Ogg library error", "", "Okay",
						      NULL, 0, 0);
						return D_CLOSE;
					}
					if(result == 0)
						break;
					pack_fwrite(og.header, og.header_len, outfile);
					pack_fwrite(og.body, og.body_len, outfile);
				}
			}
			return D_O_K;

		case MSG_END:
			if(audio)
			{
				vorbis_block_clear(&vb);
				vorbis_dsp_clear(&vd);
				vorbis_info_clear(&vi);
				vorbis_comment_clear(&vc);
			}
			if(video.type != type_none)
			{
				theora_clear(&td);
				theora_comment_clear(&tc);
			}
			ogg_stream_clear(&vo);
			ogg_stream_clear(&to);
			return D_O_K;

//		redraw_now:
		case MSG_DRAW:
		{
			static int akbps;
			static int vkbps;
			int min, hrs;
			float sec;

			if(audio && video.type != type_none)
			{
				textprintf_centre_ex(screen, font, d->x, d->y, d->fg, d->bg,
				                     "Audio: %d channels %dhz %d bits",
				                      audio_ch, audio_hz, audio_bt);
			}
			else if(audio)
				textprintf_centre_ex(screen, font, d->x, d->y+9, d->fg, d->bg,
				                     "Audio: %d channels %dhz %d bits",
				                      audio_ch, audio_hz, audio_bt);
			if(video.type != type_none)
				textprintf_centre_ex(screen, font, d->x, d->y+9, d->fg, d->bg,
				                     "Video: %dx%d %.02f fps",
				                     frame_x, frame_y, (float)video_hzn/
				                                       (float)video_hzd);

			if(status == init)
			{
				textprintf_centre_ex(screen, font, d->x, d->y+27,
				                     d->fg, d->bg, "Starting encoder...");
				akbps = 0;
				vkbps = 0;
				return D_O_K;
			}

			hrs = (int)timebase / 3600;
			min = ((int)timebase/60) % 60;
			sec = timebase - (float)((int)timebase/60) * 60.0;

			if(do_video)
			{
				if(video_bytesout > 0 && timebase > 0.0)
					vkbps = rint(video_bytesout*8.0/timebase*0.001);
			}
			else
			{
				if(audio_bytesout > 0 && timebase > 0.0)
					akbps = rint(audio_bytesout*8.0/timebase*0.001);
			}

			if(status == done)
			{
				textprintf_centre_ex(screen, font, d->x, d->y+27, d->fg, d->bg,
				                     "* %d:%02d:%05.2f *", hrs, min, sec);
				return D_O_K;
			}

			textprintf_centre_ex(screen, font, d->x, d->y+27, d->fg, d->bg,
			                     "%d:%02d:%05.2f", hrs, min, sec);
			if(audio && video.type != type_none)
				textprintf_centre_ex(screen, font, d->x, d->y+36, d->fg, d->bg,
				                     "audio: %4dkbps <> video: %4dkbps",
				                     akbps, vkbps);
			else if(audio)
				textprintf_centre_ex(screen, font, d->x, d->y+36, d->fg, d->bg,
				                     "audio: %4dkbps", akbps);
			else if(video.type != type_none)
				textprintf_centre_ex(screen, font, d->x, d->y+36, d->fg, d->bg,
				                     "video: %4dkbps", vkbps);

			return D_O_K;
		}

		default:
		{
			ogg_int64_t gp;
			double audiotime = 0.0;
			double videotime = 0.0;

			if(quit_now)
				return D_CLOSE;

			if(status == done)
			{
				rest(1);
				return D_O_K;
			}
			rest(0);

			if(audio && audioflag == 0)
			{
				/* is there an audio page flushed?  If not, fetch one */
				audioflag = fetch_audio(audio, &audiopage, &vo, &vd, &vb);
				if(audioflag < 0)
					return D_CLOSE;
			}

			if(video.type != type_none && videoflag == 0)
			{
				/* is there a video page flushed?  If not, fetch one */
				videoflag = fetch_video(&videopage, &to, &td);
				if(videoflag < 0)
					return D_CLOSE;
			}

			/* no pages of either?  Must be end of stream. */
			if(!audioflag && !videoflag)
			{
				status = done;
				if(one_shot)
					return D_CLOSE;
				d[1].dp = "Close";
				return D_REDRAW;
			}

			/* which is earlier; the end of the audio page or the end of
			   the video page? Flush the earlier to stream */
			if(audioflag && videoflag)
			{
				gp = ogg_page_granulepos(&videopage);
				videotime = theora_granule_time(&td, gp);

				gp = ogg_page_granulepos(&audiopage);
				audiotime = vorbis_granule_time(&vd, gp);

				do_video = (videotime<audiotime);
			}
			else if(videoflag)
			{
				gp = ogg_page_granulepos(&videopage);
				videotime = theora_granule_time(&td, gp);
				do_video = 1;
			}
			else if(audioflag)
			{
				gp = ogg_page_granulepos(&audiopage);
				audiotime = vorbis_granule_time(&vd, gp);
				do_video = 0;
			}

			if(do_video)
			{
				/* flush a video page */
				video_bytesout += pack_fwrite(videopage.header,
				                              videopage.header_len, outfile);
				video_bytesout += pack_fwrite(videopage.body,
				                              videopage.body_len, outfile);
				videoflag = 0;
				timebase = videotime;
			}
			else
			{
				/* flush an audio page */
				audio_bytesout += pack_fwrite(audiopage.header,
				                              audiopage.header_len, outfile);
				audio_bytesout += pack_fwrite(audiopage.body,
				                              audiopage.body_len, outfile);
				audioflag = 0;
				timebase = audiotime;
			}

			if(status == init)
			{
				status = working;
				return D_REDRAW;
			}

			return D_REDRAWME;
		}
	}

	return D_O_K;
}

static DIALOG main_dialog[];
static int audio_slider_callback(void *dp3, int d2)
{
	static char buf[5];
	DIALOG *d = &main_dialog[(int)dp3];

	snprintf(buf, sizeof(buf), "%2d.%d", d2/10, d2%10);
	d->dp = buf;
	if(gfx_driver)
		object_message(d, MSG_DRAW, 0);

	return D_O_K;
}

static int video_slider_callback(void *dp3, int d2)
{
	static char buf[5];
	DIALOG *d = &main_dialog[(int)dp3];

	snprintf(buf, sizeof(buf), "%2d.%d", d2/10, d2%10);
	d->dp = buf;
	if(gfx_driver)
		object_message(d, MSG_DRAW, 0);

	return D_O_K;
}


static int what_is();

static int show_about()
{
	return message("Ogg Vorbis/Ogg Theora encoder\n"
	               "Using APEG v"APEG_VERSION_STR"\n"
	               "http://kcat.strangesoft.net/\n"
	               "\n"
	               "Original code is Copyright the Xiph.Org Foundation\n"
	               "http://www.xiph.org/");
}

static MENU file_menu[] =
{
	{ .text="&Load settings...", .flags=D_DISABLED },
	{ .text="&Save settings...", .flags=D_DISABLED },
	{ .text="" },
	{ .text="&Quit", .proc=quit },
	{ .text=NULL },
};

static MENU help_menu[] =
{
	{ .text="&What is...", .proc=what_is },
	{ .text="" },
	{ .text="&About...", .proc=show_about },
	{ .text=NULL },
};

static MENU main_menu[] =
{
	{ .text="&File", .child=file_menu },
	{ .text="&Help", .child=help_menu },
	{ .text=NULL }
};


static DIALOG main_dialog[] =
{
	{ .proc=d_clear_proc, .w=SW, .h=SH },
	{ .proc=d_menu_proc, .x=-1, .y=-1, .dp=main_menu },
#define ENCODE_BTN	2
	{ .proc=d_button_proc, .x=SW/4-32, .y=448, .w=64, .h=16, .flags=D_EXIT,
	  .dp="Encode" },
#define TEST_BTN	(ENCODE_BTN+1)
	{ .proc=d_button_proc, .x=SW*2/4-32, .y=448, .w=64, .h=16, .flags=D_EXIT,
	  .dp="Test" },
#define CANCEL_BTN	(TEST_BTN+1)
	{ .proc=d_button_proc, .x=SW*3/4-32, .y=448, .w=64, .h=16, .flags=D_EXIT,
	  .dp="Quit" },
#define MAIN_AREA_END (CANCEL_BTN)

#define VIDEO_AREA_START (MAIN_AREA_END+1)
	{ .proc=d_box_proc, .x=6, .y=16, .w=628, .h=164 },
	{ .proc=d_ctext_proc, .x=320, .y=18, .dp="Video Options" },
#define VIDEO_RADIO_BTN(x)	(VIDEO_AREA_START+2+(x))
	{ .proc=d_radio_proc, .x=14, .y=30, .w=96, .h=10, .d1=1, .flags=D_EXIT,
	  .dp="Video file" },
	{ .proc=d_radio_proc, .x=120, .y=30, .w=96, .h=10, .d1=1,
	  .flags=D_EXIT, .dp="Video pipe" },
	{ .proc=d_radio_proc, .x=224, .y=30, .w=120, .h=10, .d1=1,
	  .flags=D_SELECTED|D_EXIT, .dp="Disable Video" },
	{ .proc=d_box_proc, .x=28, .y=42, .w=522, .h=12, .flags=D_DISABLED },
#define VIDEO_BROWSE_BTN	(VIDEO_RADIO_BTN(0)+4)
	{ .proc=d_button_proc, .x=28+522, .y=42, .w=64, .h=12,
	  .flags=D_DISABLED|D_EXIT, .dp="Browse" },
	{ .proc=d_edit_proc, .x=32, .y=44, .w=522-8, .h=8, .flags=D_DISABLED,
	  .d1=sizeof(video_fname)-1, .dp=video_fname },

#define VIDEO_QUALITY_RADIO(x) ((x) ? VIDEO_QUALITY_RADIO_1 : \
                                      VIDEO_QUALITY_RADIO_0)
#define VIDEO_QUALITY_RADIO_0	(VIDEO_BROWSE_BTN+2)
	{ .proc=d_radio_proc, .x=14, .y=60, .w=80, .h=10, .d1=3,
	  .flags=D_SELECTED|D_EXIT|D_DISABLED, .dp="Quality:" },
	{ .proc=d_text_proc, .x=32, .y=74, .flags=D_DISABLED, .dp=" 3.0" },
	{ .proc=d_slider_proc, .x=72, .y=72, .w=SW-72-32, .h=12,
	  .flags=D_DISABLED, .d1=100, .d2=30, .dp2=video_slider_callback,
	  .dp3=(void*)(VIDEO_QUALITY_RADIO_0+1) },

#define VIDEO_QUALITY_RADIO_1	(VIDEO_QUALITY_RADIO_0+3)
	{ .proc=d_radio_proc, .x=14, .y=88, .w=80, .h=10,
	  .flags=D_DISABLED|D_EXIT, .d1=3, .dp="Bitrate:" },
	{ .proc=d_box_proc, .x=30, .y=100, .w=76, .h=12, .flags=D_DISABLED },
	{ .proc=d_edit_proc, .x=32, .y=102, .w=72, .h=8, .flags=D_DISABLED,
	  .d1=sizeof(video_kbps)-1, .dp=video_kbps },
	{ .proc=d_text_proc, .x=108, .y=102, .flags=D_DISABLED,
	  .dp="kbit/s (45 - 2000)" },

#define VIDEO_ASPECT_CHECK	(VIDEO_QUALITY_RADIO_1+4)
	{ .proc=d_check_proc, .x=14, .y=115, .w=176+14, .h=10,
	  .flags=D_DISABLED|D_EXIT, .d1=1, .dp="Aspect Ratio modifier:" },
	{ .proc=d_box_proc, .x=204+4, .y=114, .w=52, .h=12, .flags=D_DISABLED },
	{ .proc=d_text_proc, .x=204+52+8+2, .y=116, .flags=D_DISABLED, .dp="/" },
	{ .proc=d_box_proc, .x=204+52+16+4, .y=114, .w=44, .h=12, .flags=D_DISABLED },
#define VIDEO_ASPECT_VAL	(VIDEO_ASPECT_CHECK+4)
	{ .proc=d_edit_proc, .x=204+6, .y=116, .w=48, .h=8, .flags=D_DISABLED,
	  .d1=sizeof(aspect_n)-1, .dp=aspect_n },
	{ .proc=d_edit_proc, .x=204+52+16+6, .y=116, .w=40, .h=8,
	  .flags=D_DISABLED, .d1=sizeof(aspect_d)-1, .dp=aspect_d },
#define VIDEO_FPS_CHECK	(VIDEO_ASPECT_VAL+2)
	{ .proc=d_check_proc, .x=14, .y=129, .w=104+14, .h=10,
	  .flags=D_DISABLED|D_EXIT, .d1=1, .dp="Override FPS:" },
	{ .proc=d_box_proc, .x=204+4, .y=128, .w=52, .h=12, .flags=D_DISABLED },
	{ .proc=d_text_proc, .x=204+52+8+2, .y=130, .flags=D_DISABLED, .dp="/" },
	{ .proc=d_box_proc, .x=204+52+16+4, .y=128, .w=44, .h=12, .flags=D_DISABLED },
#define VIDEO_FPS_VAL	(VIDEO_FPS_CHECK+4)
	{ .proc=d_edit_proc, .x=204+6, .y=130, .w=48, .h=8, .flags=D_DISABLED,
	  .d1=sizeof(fps_n)-1, .dp=fps_n },
	{ .proc=d_edit_proc, .x=204+52+16+6, .y=130, .w=40, .h=8,
	  .flags=D_DISABLED, .d1=sizeof(fps_d)-1, .dp=fps_d },
#define VIDEO_AREA_END	(VIDEO_FPS_VAL+1)


#define AUDIO_AREA_START (VIDEO_AREA_END+1)
	{ .proc=d_box_proc, .x=6, .y=176+16, .w=628, .h=164 },
	{ .proc=d_ctext_proc, .x=320, .y=176+18, .dp="Audio Options" },
#define AUDIO_RADIO_BTN(x)	(AUDIO_AREA_START+2+(x))
	{ .proc=d_radio_proc, .x=14, .y=176+30, .w=96, .h=10, .d1=2,
	  .flags=D_EXIT, .dp="Audio file" },
	{ .proc=d_radio_proc, .x=120, .y=176+30, .w=96, .h=10, .d1=2,
	  .flags=D_EXIT, .dp="Audio pipe" },
	{ .proc=d_radio_proc, .x=224, .y=176+30, .w=120, .h=10, .d1=2,
	  .flags=D_SELECTED|D_EXIT, .dp="Disable Audio" },
	{ .proc=d_box_proc, .x=28, .y=176+42, .w=522, .h=12, .flags=D_DISABLED },
#define AUDIO_BROWSE_BTN	(AUDIO_RADIO_BTN(0)+4)
	{ .proc=d_button_proc, .x=28+522, .y=176+42, .w=64, .h=12,
	  .flags=D_DISABLED|D_EXIT, .dp="Browse" },
	{ .proc=d_edit_proc, .x=32, .y=176+44, .w=522-8, .h=8, .flags=D_DISABLED,
	  .d1=sizeof(audio_fname)-1, .dp=audio_fname },

#define AUDIO_QUALITY_RADIO(x) ((x) ? AUDIO_QUALITY_RADIO_1 : \
                                      AUDIO_QUALITY_RADIO_0)
#define AUDIO_QUALITY_RADIO_0	(AUDIO_BROWSE_BTN+2)
	{ .proc=d_radio_proc, .x=14, .y=176+60, .w=80, .h=10, .d1=4,
	  .flags=D_SELECTED|D_EXIT|D_DISABLED, .dp="Quality:" },
	{ .proc=d_text_proc, .x=32, .y=176+74, .flags=D_DISABLED,
	  .dp=" 3.0" },
	{ .proc=d_slider_proc, .x=72, .y=176+72, .w=SW-72-32, .h=12,
	  .flags=D_DISABLED, .d1=100, .d2=30, .dp2=audio_slider_callback,
	  .dp3=(void*)(AUDIO_QUALITY_RADIO_0+1) },

#define AUDIO_QUALITY_RADIO_1	(AUDIO_QUALITY_RADIO_0+3)
	{ .proc=d_radio_proc, .x=14, .y=176+88, .w=80, .h=10,
	  .flags=D_DISABLED|D_EXIT, .d1=4, .dp="Bitrate:" },
	{ .proc=d_box_proc, .x=30, .y=176+100, .w=68, .h=12, .flags=D_DISABLED },
	{ .proc=d_edit_proc, .x=32, .y=176+102, .w=64, .h=8, .flags=D_DISABLED,
	  .d1=sizeof(audio_kbps)-1, .dp=audio_kbps },
	{ .proc=d_text_proc, .x=100, .y=176+102, .flags=D_DISABLED,
	  .dp="kbit/s (1 - 512)" },

#define AUDIO_SKIP_CHECK (AUDIO_QUALITY_RADIO_1+4)
    { .proc=d_check_proc, .x=14, .y=176+115, .w=104+14, .h=10,
      .flags=D_DISABLED|D_EXIT, .d1=1, .dp="Start Offset:" },
    { .proc=d_box_proc, .x=132+4, .y=176+114, .w=52, .h=12, .flags=D_DISABLED },
    { .proc=d_text_proc, .x=132+52+8+2, .y=176+116, .flags=D_DISABLED,
      .dp="sec." },
#define AUDIO_SKIP_VAL (AUDIO_SKIP_CHECK+3)
    { .proc=d_edit_proc, .x=132+6, .y=176+116, .w=48, .h=8, .flags=D_DISABLED,
      .d1=sizeof(askip_n)-1, .dp=askip_n },
#define AUDIO_AREA_END	(AUDIO_SKIP_VAL)


#define OFILE_AREA_START	(AUDIO_AREA_END+1)
#define OFILE_RADIO_BTN(x)	(OFILE_AREA_START+(x))
	{ .proc=d_radio_proc, .x=16, .y=192*2+24, .w=100, .h=10,
	  .flags=D_SELECTED|D_EXIT, .d1=5, .dp="Output file" },
	{ .proc=d_radio_proc, .x=120, .y=192*2+24, .w=100, .h=10, .flags=D_EXIT,
	  .d1=5, .dp="Output pipe" },
	{ .proc=d_box_proc, .x=16, .y=192*2+36, .w=522, .h=12 },
#define OFILE_BROWSE_BTN	(OFILE_RADIO_BTN(0)+3)
	{ .proc=d_button_proc, .x=16+522, .y=192*2+36, .w=64, .h=12,
	  .flags=D_EXIT, .dp="Browse" },
	{ .proc=d_edit_proc, .x=20, .y=192*2+38, .w=522-8, .h=8,
	  .d1=sizeof(ofile_fname)-1, .dp=ofile_fname },
#define OFILE_AREA_END	(OFILE_AREA_START+1)

	{ .proc=d_tail_proc },
	{ .proc=d_yield_proc },
	{ .proc=NULL }
};


static DIALOG encoder_dialog[] =
{
	{ .proc=d_shadow_box_proc, .x=0, .y=0, .w=SH, .h=160 },
	{ .proc=d_encoder_proc, .x=240, .y=60 },
	{ .proc=d_button_proc, .x=240-32, .y=160-24, .w=64, .h=16, .flags=D_EXIT,
	  .dp="Cancel" },

	{ .proc=d_tail_proc },
	{ .proc=NULL }
};


static const char *descriptions[] = {
	[ENCODE_BTN]            = "Encode the selected video and/or audio sources",
	[TEST_BTN]              = "Test play the specified output file",
	[CANCEL_BTN]            = "Quit the program",

	[VIDEO_RADIO_BTN(0)]    = "Read video data from the specified file",
	[VIDEO_RADIO_BTN(1)]    = "Read video data from the specified external\n"
	                          "program's output",
	[VIDEO_RADIO_BTN(2)]    = "Disable video input",

	[VIDEO_BROWSE_BTN-1]    = "The video source to read data from.\n"
	                          "ogg_encoder's directory will be used as a\n"
	                          "base if an absolute path isn't specified",
	[VIDEO_BROWSE_BTN+1]    = "The video source to read data from.\n"
	                          "ogg_encoder's directory will be used as a\n"
	                          "base if an absolute path isn't specified",

	[VIDEO_QUALITY_RADIO_0] = "Use a quality-based compression method",
	[VIDEO_QUALITY_RADIO_1] = "Use a bitrate-based compression method",

	[VIDEO_ASPECT_CHECK]    = "Specify an aspect ratio multiplier (numbers\n"
	                          "greater than 1 stretch the width, less than 1\n"
	                          "stretch the height)",
	[VIDEO_ASPECT_VAL  ]    = "Aspect multiplier numerator (this will be\n"
	                          "divided by the demoninator)",
	[VIDEO_ASPECT_VAL+1]    = "Aspect multiplier denominator (the numerator\n"
	                          "will be divided by this)",

	[VIDEO_FPS_CHECK]       = "Override detected frame rate",
	[VIDEO_FPS_VAL  ]       = "Frame rate numerator (this will be divided by\n"
	                          "the demoninator)",
	[VIDEO_FPS_VAL+1]       = "Aspect multiplier denominator (the numerator\n"
	                          "will be divided by this)",


	[AUDIO_RADIO_BTN(0)]    = "Read audio data from the specified file",
	[AUDIO_RADIO_BTN(1)]    = "Read audio data from the specified external\n"
	                          "program's output",
	[AUDIO_RADIO_BTN(2)]    = "Disable audio input",

	[AUDIO_BROWSE_BTN-1]    = "The audio source to read data from.\n"
	                          "ogg_encoder's directory will be used as a\n"
	                          "base if an absolute path isn't specified",
	[AUDIO_BROWSE_BTN+1]    = "The audio source to read data from.\n"
	                          "ogg_encoder's directory will be used as a\n"
	                          "base if an absolute path isn't specified",

	[AUDIO_QUALITY_RADIO_0] = "Use a quality-based compression method",
	[AUDIO_QUALITY_RADIO_1] = "Use a bitrate-based compression method",

	[AUDIO_SKIP_CHECK]      = "Encoded audio offset. Positive values will\n"
	                          "skip samples, negative will insert silence",

	[OFILE_RADIO_BTN(0)]    = "Write to the specified file",
	[OFILE_RADIO_BTN(1)]    = "Write to the specified external program's\n"
	                          "standard input"
};

static int what_is()
{
	DIALOG *iter = main_dialog;
	unsigned int idx;
	int x, y;

	while(iter->proc)
		++iter;

	while(mouse_b)
		rest(1);

	select_mouse_cursor(MOUSE_CURSOR_QUESTION);
	show_mouse(gui_get_screen());

	while(!mouse_b)
		rest(1);
	x = mouse_x;
	y = mouse_y;

	select_mouse_cursor(MOUSE_CURSOR_ARROW);
	show_mouse(gui_get_screen());

	while(--iter > main_dialog)
	{
		if(x >= iter->x && x < iter->x+iter->w &&
		   y >= iter->y && y < iter->y+iter->h)
			break;
	}

	idx = iter-main_dialog;
	if(idx < (sizeof(descriptions)/sizeof(char*)))
	{
		static DIALOG popup[] = {
			{ .proc=d_box_proc },
			{ .proc=d_text_proc },
			{ .proc=d_text_proc },
			{ .proc=d_text_proc },

			{ .proc=d_tail_proc, .w=SW, .h=SH, .flags=D_EXIT },
			{ .proc=d_yield_proc },
			{ .proc=NULL }
		};
		const char *msg = descriptions[idx];
		if(msg)
		{
			int th = text_height(font);
			int i;

			popup[1].dp = strdup(msg);
			popup[2].dp = "";
			popup[3].dp = "";
			popup[0].w = 8;
			for(i = 0;i < 2;++i)
			{
				int w;
				char *ptr = strchr(popup[i+1].dp, '\n');
				if(ptr)
				{
					*(ptr++) = 0;
					popup[i+2].dp = ptr;
				}
				w = text_length(font, popup[i+1].dp)+8;
				popup[0].w = MAX(w, popup[0].w);
				if(!ptr)
					break;
			}

			popup[0].h = th*(i+1) + i + 8;

			popup[0].x = x+mouse_sprite->w;
			if(popup[0].x+popup[0].w > SW)
				popup[0].x = SW-popup[0].w;
			popup[0].y = y+mouse_sprite->h;
			if(popup[0].y+popup[0].h > SH)
				popup[0].y = SH-popup[0].h;

			popup[1].x = popup[2].x = popup[3].x = popup[0].x+4;
			popup[1].y = popup[0].y+4;
			popup[2].y = popup[1].y+text_height(font)+1;
			popup[3].y = popup[2].y+text_height(font)+1;
			set_dialog_color(popup, gui_fg_color, gui_mg_color);

			while(mouse_b)
				rest(1);

			popup_dialog(popup, -1);
			free(popup[1].dp);
		}
	}

	while(mouse_b)
		rest(1);

	return D_O_K;
}


static void fixup_video_area(void)
{
	int i;
	if(!(main_dialog[VIDEO_RADIO_BTN(2)].flags&D_SELECTED))
	{
		for(i = VIDEO_RADIO_BTN(2)+1;i <= VIDEO_AREA_END;++i)
				main_dialog[i].flags &= ~D_DISABLED;
		if((main_dialog[VIDEO_QUALITY_RADIO_0].flags&D_SELECTED))
		{
			for(i = VIDEO_QUALITY_RADIO_1+1;i < VIDEO_ASPECT_CHECK;++i)
					main_dialog[i].flags |= D_DISABLED;
		}
		else
		{
			for(i = VIDEO_QUALITY_RADIO_0+1;i < VIDEO_QUALITY_RADIO_1;++i)
					main_dialog[i].flags |= D_DISABLED;
		}
		if(!(main_dialog[VIDEO_ASPECT_CHECK].flags&D_SELECTED))
		{
			for(i = VIDEO_ASPECT_CHECK+1;i < VIDEO_FPS_CHECK;++i)
					main_dialog[i].flags |= D_DISABLED;
		}
		if(!(main_dialog[VIDEO_FPS_CHECK].flags&D_SELECTED))
		{
			for(i = VIDEO_FPS_CHECK+1;i <= VIDEO_AREA_END;++i)
					main_dialog[i].flags |= D_DISABLED;
		}
	}
	else
	{
		for(i = VIDEO_RADIO_BTN(2)+1;i < VIDEO_AREA_END;++i)
			main_dialog[i].flags |= D_DISABLED;
	}
}

static void fixup_audio_area(void)
{
	int i;
	if(!(main_dialog[AUDIO_RADIO_BTN(2)].flags&D_SELECTED))
	{
		for(i = AUDIO_RADIO_BTN(2)+1;i <= AUDIO_AREA_END;++i)
				main_dialog[i].flags &= ~D_DISABLED;
		if((main_dialog[AUDIO_QUALITY_RADIO_0].flags&D_SELECTED))
		{
			for(i = AUDIO_QUALITY_RADIO_1+1;i < AUDIO_SKIP_CHECK;++i)
					main_dialog[i].flags |= D_DISABLED;
		}
		else
		{
			for(i = AUDIO_QUALITY_RADIO_0+1;i < AUDIO_QUALITY_RADIO_1;++i)
					main_dialog[i].flags |= D_DISABLED;
		}
		if(!(main_dialog[AUDIO_SKIP_CHECK].flags&D_SELECTED))
		{
			for(i = AUDIO_SKIP_CHECK+1;i <= AUDIO_AREA_END;++i)
				main_dialog[i].flags |= D_DISABLED;
		}
	}
	else
	{
		for(i = AUDIO_RADIO_BTN(2)+1;i <= AUDIO_AREA_END;++i)
			main_dialog[i].flags |= D_DISABLED;
	}
}


static void id_file(char *f)
{
	PACKFILE *test;
	unsigned char buffer[4];
	int ret;

	/* open it, look for magic */
	test=pack_fopen(f,"rb");
	if(!test)
		return;

	ret=pack_fread(buffer,4,test);
	if(ret==4)
	{
		if(memcmp(buffer,"RIFF",4) == 0)
		{
			/* possible WAV file */
			strncpy(audio_fname, f, sizeof(audio_fname));
			main_dialog[AUDIO_RADIO_BTN(0)].flags |= D_SELECTED;
			main_dialog[AUDIO_RADIO_BTN(1)].flags &= ~D_SELECTED;
			main_dialog[AUDIO_RADIO_BTN(2)].flags &= ~D_SELECTED;
		}
		else if(memcmp(buffer,"YUV4",4) == 0 || memcmp(buffer,"OETF",4) == 0)
		{
			/* possible YUV2MPEG2 format file */
			strncpy(video_fname, f, sizeof(video_fname));
			main_dialog[VIDEO_RADIO_BTN(0)].flags |= D_SELECTED;
			main_dialog[VIDEO_RADIO_BTN(1)].flags &= ~D_SELECTED;
			main_dialog[VIDEO_RADIO_BTN(2)].flags &= ~D_SELECTED;
		}
	}

	pack_fclose(test);
}


static int open_audio(void)
{
	const char *msg = "";
	PACKFILE *f = NULL;
	int length;
	int i;

	audio_ch=0;
	audio_hz=0;
	audio_bt=0;

	if(is_relative_filename(audio_fname))
	{
		char *buf = strdup(audio_fname);
		if(!buf)
			return 1;
		get_executable_name(buffer, sizeof(buffer));
		make_absolute_filename(audio_fname, buffer, buf, sizeof(audio_fname));
		free(buf);
	}

	if((main_dialog[AUDIO_RADIO_BTN(1)].flags&D_SELECTED))
	{
		FILE *fp = popen(audio_fname, "r");
		if(fp)
		{
			f = pack_fopen_vtable(&openfile_vtable, fp);
			if(!f)
				pclose(fp);
		}
	}
	else if((main_dialog[AUDIO_RADIO_BTN(0)].flags&D_SELECTED))
		f = pack_fopen(audio_fname, F_READ);
	if(f == NULL)
	{
		alert("Could not open audio", audio_fname, "", "Okay", NULL, 0, 0);
		return 1;
	}

	memset(buffer, 0, sizeof(buffer));

	pack_fread(buffer, 12, f);          /* check RIFF header */
	if(memcmp(buffer, "RIFF", 4) != 0 || memcmp(buffer+8, "WAVE", 4) != 0)
	{
		msg = "Not a RIFF WAV file";
		goto getout;
	}

	while(TRUE)
	{
		if (pack_fread(buffer, 4, f) != 4)
		{
			msg = "Premature EOF";
			break;
		}

		length = pack_igetl(f);          /* read chunk length */

		if(memcmp(buffer, "fmt ", 4) == 0)
		{
			i = pack_igetw(f);            /* should be 1 for PCM data */
			length -= 2;
			if (i != 1)
			{
				msg = "Not PCM format";
				goto getout;
			}

			audio_ch = pack_igetw(f);     /* mono or stereo data */
			length -= 2;

			audio_hz = pack_igetl(f);         /* sample frequency */
			length -= 4;

			pack_igetl(f);                /* skip six bytes */
			pack_igetw(f);
			length -= 6;

			audio_bt = pack_igetw(f);         /* 8 or 16 bit data? */
			length -= 2;
			if(audio_bt != 8 && audio_bt != 16)
			{
				msg = "Invalid bit depth (must be 8 or 16)";
				goto getout;
			}
		}
		else if(memcmp(buffer, "data", 4) == 0)
		{
			if (audio_ch == 0)
			{
				msg = "Not a RIFF WAV file";
				goto getout;
			}

			if(audio_skip >= 0.0f)
			{
				sample_delay  = audio_skip*(float)audio_hz;
				sample_delay *= audio_ch;
			}
			else
			{
				sample_delay = 0;
				pack_fseek(f, -(int)(audio_skip*(float)audio_hz)*audio_ch*audio_bt/8);
			}

			audio = f;
			return 0;
		}

		/* skip the remainder of the chunk */
		pack_fseek(f, length);
	}

getout:
	alert("Error reading from", audio_fname, msg, "Okay", NULL, 0, 0);
	pack_fclose(f);
	return 1;
}

static int open_video(void)
{
	PACKFILE *test = NULL;
//	unsigned char buffer[80];
	int ret;
	int tmp_video_hzn, tmp_video_hzd, tmp_video_an, tmp_video_ad;

	if(is_relative_filename(video_fname))
	{
		char *buf = strdup(video_fname);
		if(!buf)
			return 1;
		get_executable_name(buffer, sizeof(buffer));
		make_absolute_filename(video_fname, buffer, buf, sizeof(video_fname));
		free(buf);
	}

	/* open it, look for magic */
	if((main_dialog[VIDEO_RADIO_BTN(1)].flags&D_SELECTED))
	{
		FILE *fp = popen(video_fname, "r");
		if(fp)
		{
			test = pack_fopen_vtable(&openfile_vtable, fp);
			if(!test)
				pclose(fp);
		}
	}
	else if((main_dialog[VIDEO_RADIO_BTN(0)].flags&D_SELECTED))
		test = pack_fopen(video_fname, F_READ);
	if(!test)
	{
		alert("Could not open video", video_fname, "", "Okay", NULL, 0, 0);
		return 1;
	}

	ret = pack_fread(buffer, 4, test);
	if(ret < 4)
		goto yuv_err;

	if(memcmp(buffer, "YUV4", 4) == 0)
	{
		/* possible YUV2MPEG2 format file */
		/* read until newline, or 80 cols, whichever happens first */
		int i;
		for(i = 0;i < 79;i++)
		{
			ret = pack_getc(test);
			if(ret == EOF && pack_feof(test))
				goto yuv_err;
			if(ret == '\n')
				break;
			buffer[i] = ret&0xFF;
		}
		if(i == 79)
		{
			alert("Error parsing video header", "not a YUV2MPEG2 file?", "",
			      "Okay", NULL, 0, 0);
		}

		buffer[i] = '\0';

		if(memcmp(buffer,"MPEG",4) == 0)
		{
			char interlace;

			if(buffer[4] != '2')
			{
				alert("Incorrect YUV input file version",
				      "YUV4MPEG2 required", "", "Okay", NULL, 0, 0);
				pack_fclose(test);
				return 1;
			}

			ret = sscanf(buffer, "MPEG2 W%d H%d F%d:%d I%c A%d:%d",
			             &frame_x, &frame_y, &tmp_video_hzn, &tmp_video_hzd,
			             &interlace, &tmp_video_an, &tmp_video_ad);
			if(ret < 7)
				goto yuv_err;

			/* update fps and aspect ratio globals if not specified in the
			   command line */
			if(video_hzn == -1) video_hzn = tmp_video_hzn;
			if(video_hzd == -1) video_hzd = tmp_video_hzd;
			if(video_an == -1) video_an = tmp_video_an;
			if(video_ad == -1) video_ad = tmp_video_ad;

			if(interlace != 'p')
			{
				alert("Input video is interlaced",
				      "Theora handles only progressive scan", "",
				      "Okay", NULL, 0, 0);
				pack_fclose(test);
				return 1;
			}

			video.pf = test;
			video.type = type_yuv4mpeg2;
			return 0;
		}
	}
	else if (memcmp(buffer, "OETF", 4) == 0)
	{
		frame_x = -1;
		frame_y = -1;
		video.ff_done = -1;
		while(TRUE)
		{
			char *opt, *val;

			opt = pack_fgets(buffer, sizeof(buffer), test);
			if(!opt)
				break;

			/* Cut out comments */
			val = strpbrk(opt, "#;");
			if(val) *val = 0;

			/* Find start of option name */
			while(isspace(*opt) && *opt)
				++opt;

			if(!(*opt))
				continue;

			/* Find end of option name, and null-terminate it */
			val = opt;
			while(!isspace(*val) && *val != '=' && *val)
				++val;
			if(!(*val)) continue;
			*(val++) = 0;

			/* Find start of the option value */
			while((isspace(*val) || *val == '=') && *val)
				++val;

			if(!(*val))
				continue;

			if(strcasecmp(opt, "filespec") == 0)
			{
				int c;
				char *ptr = get_filename(val);
				if(!ptr)
					break;
				c = *ptr;
				*ptr = 0;
				video.path = strdup(val);
				*ptr = c;

				video.ff_done = al_findfirst(val, &video.ff_info, ~FA_DIREC);
				continue;
			}
			else if(strcasecmp(opt, "aspect") == 0)
			{
				char *ptr;

				ptr = strpbrk(val, ":/");
				if(ptr)
					*(ptr++) = 0;
				else
					ptr = "1";

				if(video_an == -1) video_an = atoi(val);
				if(video_ad == -1) video_ad = atoi(ptr);
				continue;
			}
			else if(strcasecmp(opt, "framerate") == 0)
			{
				char *ptr;

				ptr = strpbrk(val, ":/");
				if(ptr)
					*(ptr++) = 0;
				else
					ptr = "1";

				if(video_hzn == -1) video_hzn = atoi(val);
				if(video_hzd == -1) video_hzd = atoi(ptr);
				continue;
			}
			else if(strcasecmp(opt, "framesize") == 0)
			{
				char *ptr;

				ptr = strpbrk(val, "x");
				if(ptr)
				{
					*(ptr++) = 0;
					frame_y = atoi(ptr);
				}

				frame_x = atoi(val);
				continue;
			}
			else
			{
				alert("Unknown setting:", opt, "", "Okay", NULL, 0, 0);
				pack_fgets(buffer, sizeof(buffer), test);
			}
		}
		pack_fclose(test);

		if(video.ff_done)
		{
			alert("No image files found from", video_fname, "", "Okay", NULL,
			      0, 0);
			return 1;
		}
		if(video_hzn <= 0 || video_hzd <= 0)
		{
			alert("No, or an illegal, frame rate defined", video_fname, "",
			      "Okay", NULL, 0, 0);
			al_findclose(&video.ff_info);
			return 1;
		}
		if(video_an <= 0 || video_ad <= 0)
		{
			alert("No, or an illegal, aspect ratio defined", video_fname, "",
			      "Okay", NULL, 0, 0);
			al_findclose(&video.ff_info);
			return 1;
		}
		if(frame_x <= 0 || frame_y <= 0)
		{
			snprintf(buffer, sizeof(buffer), "%d x %d", frame_x, frame_y);
			alert("Bad frame size", buffer, "", "Okay", NULL, 0, 0);
			al_findclose(&video.ff_info);
			return 1;
		}

		video.type = type_ff;
		return 0;
	}

yuv_err:
	alert("Error reading from", video_fname, "", "Okay", NULL, 0, 0);
	pack_fclose(test);
	return 1;
}



typedef struct {
	const char *fname;
	APEG_STREAM *stream;
} MEDIA_INFO;


static int d_video_proc(int msg, DIALOG *d, int c)
{
	MEDIA_INFO *mi = d->dp;

	switch(msg)
	{
		case MSG_START:
			mi->stream = apeg_open_stream(mi->fname);
			if(!mi->stream)
			{
				alert("Could not open file", mi->fname, "for preview", "Okay", NULL, 0, 0);
				return D_CLOSE;
			}

			if((mi->stream->flags&APEG_HAS_AUDIO) && !digi_driver)
			{
				if(!(mi->stream->flags&APEG_HAS_VIDEO))
				{
					alert("Internal error", "", "(could not init sound)", "Okay",
					      NULL, 0, 0);
					return D_CLOSE;
				}
			}

			// Get w/h
			if(d->w <= 0 || d->h <= 0)
			{
				int w = 320, h = 240;

				if((mi->stream->flags&APEG_HAS_VIDEO))
					apeg_get_video_size(mi->stream, &w, &h);

				if(d->w <= 0 && d->h <= 0)
				{
					d->w = w;
					d->h = h;
				}
				else if(d->w <= 0)
					d->w = d->h*w/h;
				else
					d->h = d->w*h/w;
			}

			d->d1 = 255;
			break;

		case MSG_END:
			apeg_close_stream(mi->stream);
			mi->stream = NULL;
			break;

		case MSG_DRAW: {
			BITMAP *target = gui_get_screen();

			if((mi->stream->flags&APEG_HAS_VIDEO))
				stretch_blit(mi->stream->bitmap, target, 0, 0,
				             mi->stream->w, mi->stream->h, d->x, d->y,
				             d->w, d->h);
			else
			{
				rectfill(target, d->x, d->y, d->x+d->w-1, d->y+d->h-1, gui_bg_color);
				textout_ex(target, font, "Audio Only", d->x+2, d->y+2, gui_fg_color, -1);
			}

			} break;

		case MSG_IDLE:
			if(d->d1)
			{
				switch(apeg_advance_stream(mi->stream, FALSE))
				{
					case APEG_ERROR:
						apeg_set_stream_rate(mi->stream, 0.0f);
						alert("APEG Decode Error", "", apeg_error, "Okay", NULL, 0, 0);
						return D_CLOSE;

					case APEG_EOF:
						object_message(d, MSG_USER, 0);
						return D_REDRAW;

					case APEG_OK:
						if(mi->stream->frame_updated > 0)
							return D_REDRAWME;
						if(mi->stream->frame_updated == 0 || mi->stream->audio.flushed)
							return D_O_K;
						break;
				}
			}

			rest(1);
			break;

		case MSG_USER:
			d->d1 = c;
			apeg_set_stream_rate(mi->stream, (float)c / 255.0f);
			break;
	}
	return D_O_K;
	(void)c;
}

static int d_vidbox_proc(int msg, DIALOG *d, int c)
{
	int ret;

	if(msg == MSG_DRAW && !d->d1)
	{
		// Fix the width/height
		d->w = d[1].w + 4;
		d->h = d[1].h + 64;

		// Set the play/pause button position
		d[2].x = d->x + 8;
		d[2].y = d->y + d->h-d[2].h-d[5].h-6;

		// Set the stop button position
		d[3].x = d[2].x + d[2].w + 2;
		d[3].y = d[2].y;

		// Set the time slider's position and width
		d[4].x = d[3].x + d[3].w + 2;
		d[4].y = d[2].y + 2;
		d[4].w = d->x+d->w - d[4].x - 4;

		// Framedrop check box
		d[5].x = d->x + 12;
		d[5].y = d->y + d->h-d[5].h-4;

		// Time counter
		d[6].x = d->x + d->w - d[6].w - 4;
		d[6].y = d->y + d->h-d[6].h-6;

		centre_dialog(d);
		d->d1 = 1;
	}
	ret = d_box_proc(msg, d, c);
	if(msg == MSG_DRAW)
	{
		BITMAP *bmp = gui_get_screen();
		textprintf_ex(bmp, font, d->x+8, d->y+4, d->fg, -1, "APEG Output Test (%dx%d)", d[1].w, d[1].h);
		rect(bmp, d[1].x-1, d[1].y-1, d[1].x+d[1].w, d[1].y+d[1].h, d->fg);
	}
	return ret;
}

static int d_buttonctrl_proc(int msg, DIALOG *d, int c)
{
	int ret = d_button_proc(msg, d, c);

	d->d1 = d[-1].d1;
	if((d->flags&D_SELECTED) && !gui_mouse_b())
	{
		if(d->d1 == 0)
			d->d1 = 255;
		else
			d->d1 = 0;

		object_message(d-1, MSG_USER, d->d1);

		d->flags &= ~D_SELECTED;
		return D_REDRAWME;
	}

	if(msg == MSG_DRAW)
	{
		BITMAP *bmp = gui_get_screen();
		int col = ((d->flags&D_SELECTED) ? d->bg : d->fg);
		int off = ((d->flags&D_SELECTED) ? 1 : 0);
		if(d->d1)
		{
			vline(bmp, d->x+((d->w-1)*2/5)+off, d->y+((d->h-1)*2/5)+off,
			           d->y+((d->h-1)*3/5)+off, col);
			vline(bmp, d->x+((d->w-1)*3/5)+off, d->y+((d->h-1)*2/5)+off,
			           d->y+((d->h-1)*3/5)+off, col);
		}
		else
			triangle(bmp, d->x+((d->w-1)*2/5)+off, d->y+((d->h-1)*2/5)+off,
			              d->x+((d->w-1)*3/5)+off, d->y+((d->h-1)*1/2)+off,
			              d->x+((d->w-1)*2/5)+off, d->y+((d->h-1)*3/5)+off, col);
	}

	return ret;
}

static int d_stopbutton_proc(int msg, DIALOG *d, int c)
{
	int ret = d_button_proc(msg, d, c);
	if(msg == MSG_DRAW)
	{
		int off = ((d->flags&D_SELECTED) ? 1 : 0);
		BITMAP *bmp = gui_get_screen();
		rectfill(bmp, d->x+((d->w-1)*2/5)+off, d->y+((d->h-1)*2/5)+off,
		              d->x+((d->w-1)*3/5)+off, d->y+((d->h-1)*3/5)+off,
		         ((d->flags&D_SELECTED)?d->bg:d->fg));
	}
	return ret;
}

static int d_framedrop_check_proc(int msg, DIALOG *d, int c)
{
	int ret = d_check_proc(msg, d, c);
	if((d->flags&D_SELECTED) != d->d2)
	{
		apeg_enable_framedrop((d->flags&D_SELECTED));
		d->d2 = (d->flags&D_SELECTED);
	}
	return ret;
}

static int d_timeslider_proc(int msg, DIALOG *d, int c)
{
	if(msg == MSG_START)
	{
		MEDIA_INFO *mi = d->dp3;
		if(mi->stream)
			d->d1 = mi->stream->length*10.0;
		if(!d->d1)
			d->d1 = 1;
	}
	else if(msg == MSG_IDLE)
	{
		MEDIA_INFO *mi = d->dp3;
		int curr_pos = mi->stream->pos*10.0;
		if(d->d2 != curr_pos)
		{
			d->d2 = curr_pos;
			return D_REDRAWME;
		}
	}

	return d_slider_proc(msg, d, c);
}

static int d_timetext_proc(int msg, DIALOG *d, int c)
{
	int ret = d_rtext_proc(msg, d, c);

	if(msg == MSG_IDLE)
	{
		static unsigned int last_time;
		MEDIA_INFO *mi = d->dp3;
		unsigned int curr_time = mi->stream->pos;
		if(curr_time != last_time)
		{
			last_time = curr_time;
			snprintf(d->dp, d->d1, "%u:%02u:%02u", curr_time/60/60,
			                       (curr_time/60)%60, curr_time%60);
			return D_REDRAWME;
		}
	}

	return ret;
}

char tt_buf[16] = "0:00:00";

static DIALOG vidplay_dlg[] = {
#define VIDBOX_PROC 0
	{ d_vidbox_proc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, NULL, NULL, NULL },

#define VIDEO_PROC 1
	{ d_video_proc, 2, 16, 0, 0, 0, 0, 0, 0, 0, 0, NULL, NULL, NULL },

	{ d_buttonctrl_proc, 0, 0, 20, 20, 0, 0, ' ', 0, 0, 0, "", NULL, NULL },
	{ d_stopbutton_proc, 0, 0, 20, 20, 0, 0, 0, D_EXIT, 0, 0, "", NULL, NULL },
#define SLIDER_PROC 4
	{ d_timeslider_proc, 0, 0, 32, 16, 0, 0, 0, 0, 2, 1, NULL, NULL, NULL },
	{ d_framedrop_check_proc, 0, 0, 84, 12, 0, 0, 'f', 0, 0, 0, "&Framedrop", NULL, NULL },

#define TIMEINF_PROC 6
	{ d_timetext_proc, 0, 0, 56, 8, 0, 0, 0, 0, sizeof(tt_buf), 0, tt_buf, NULL, NULL },

	{ NULL, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, NULL, NULL, NULL }
};

static void simple_player(const char *fname)
{
	MEDIA_INFO lmi = { NULL, NULL };
	PALETTE pal;
	get_palette(pal);

	install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL);

	lmi.fname = fname;
	vidplay_dlg[VIDBOX_PROC].d1 = 0;
	vidplay_dlg[VIDEO_PROC].w = 0;
	vidplay_dlg[VIDEO_PROC].h = 0;
	vidplay_dlg[VIDEO_PROC].dp = &lmi;
	vidplay_dlg[SLIDER_PROC].d2 = 0;
	vidplay_dlg[SLIDER_PROC].dp3 = &lmi;
	vidplay_dlg[TIMEINF_PROC].dp3 = &lmi;

	do_dialog(vidplay_dlg, -1);

/*	stream = apeg_open_stream(fname);
	if(!stream)
	{
		alert("Could not open output file", fname,
		      "for preview", "Okay", NULL, 0, 0);
		return;
	}

	if((stream->flags&APEG_HAS_AUDIO) && !digi_driver)
	{
		if(!(stream->flags&APEG_HAS_VIDEO))
		{
			alert("Internal error", "", "(could not init sound)", "Okay",
			      NULL, 0, 0);
			apeg_close_stream(stream);
			set_palette(pal);
			return;
		}
	}


	// Get w/h
	if((stream->flags&APEG_HAS_VIDEO))
	{
		int tw, th;
		apeg_get_video_size(stream, &w, &h);

		// Get scale
		tw = SW;
		th = SW * h / w;
		if(th > SH)
		{
			th = SH;
			tw = SH * w / h;
		}
		w = tw;
		h = th;
	}
	else
	{
		w = 160;
		h = 32;
	}

	trgt = create_sub_bitmap(screen, (SW-w)/2, (SH-h)/2, w, h);
	if(!trgt)
	{
		alert("Internal error", "", "(could not create screen sub-bitmap)",
		      "Okay", NULL, 0, 0);
		apeg_close_stream(stream);
		set_palette(pal);
		remove_sound();
		return;
	}

	if(!(stream->flags&APEG_HAS_VIDEO))
	{
		int x, y;
		x = (SW-w)/2;
		y = (SH-h)/2;

		rectfill(screen, x, y, x+w-1, y+h-1, gui_bg_color);
		rect(screen, x, y, x+w-1, y+h-1, gui_fg_color);

		textout_centre_ex(screen, font, "Audio only",
		                  x+w/2, y+h/2-4, gui_fg_color,
		                  gui_bg_color);
	}
	else
		clear_to_color(screen, makecol(0, 0, 0));

	while(!key[KEY_ESC] && apeg_advance_stream(stream, FALSE) == APEG_OK)
	{
		if(stream->frame_updated > 0)
		{
			stretch_blit(stream->bitmap, trgt, 0, 0,
			             stream->w, stream->h, 0, 0,
			             trgt->w, trgt->h);
		}
		else if(stream->frame_updated < 0)
		{
			if(!stream->audio.flushed)
				rest(1);
		}
	}

	destroy_bitmap(trgt);
	apeg_close_stream(stream);*/

	set_palette(pal);
	remove_sound();
	clear_keybuf();
}


void close_callback()
{
	quit_now = 1;
}


int main(int argc, char *argv[])
{
	int c, long_option_index;

	set_uformat(U_UTF8);
	if(allegro_init() != 0)
	{
		fprintf(stderr, "Allegro error: %s\n", allegro_error);
		exit(1);
	}
	atexit(cleanup);

	install_timer();
	install_keyboard();
	install_mouse();

#ifdef _WIN32
	/* We need to set stdin/stdout to binary mode. Damn windows. */
	/* if we were reading/writing a file, it would also need to in
	   binary mode, eg, fopen("file.wav","wb"); */
	/* Beware the evil ifdef. We avoid these where we can, but this one we
	   cannot. Don't add any more, you'll probably go to hell if you do. */
	_setmode( _fileno( stdin ), _O_BINARY );
	_setmode( _fileno( stdout ), _O_BINARY );
#endif

	while((c=getopt_long(argc, argv, optstring, options,
	                     &long_option_index)) != EOF)
	{
		switch(c)
		{
			case 'o':
				strncpy(ofile_fname, optarg, sizeof(ofile_fname));
				break;

			case 'a':
			{
				int val = floor(atof(optarg)*10.0);
				main_dialog[AUDIO_QUALITY_RADIO(0)].flags |= D_SELECTED;
				main_dialog[AUDIO_QUALITY_RADIO(1)].flags &= ~D_SELECTED;
				main_dialog[AUDIO_QUALITY_RADIO(0)+2].d2 = MID(0, val, 100);
				audio_slider_callback((void*)AUDIO_QUALITY_RADIO(0)+1,
				                     main_dialog[AUDIO_QUALITY_RADIO(0)+2].d2);
				break;
			}

			case 'v':
			{
				int val = floor(atof(optarg)*10.0);
				main_dialog[VIDEO_QUALITY_RADIO(0)].flags |= D_SELECTED;
				main_dialog[VIDEO_QUALITY_RADIO(1)].flags &= ~D_SELECTED;
				main_dialog[VIDEO_QUALITY_RADIO(0)+2].d2 = MID(0, val, 100);
				video_slider_callback((void*)VIDEO_QUALITY_RADIO(0)+1,
				                     main_dialog[VIDEO_QUALITY_RADIO(0)+2].d2);
				break;
			}

			case 'A':
			{
				float val = atof(optarg);
				val = MID(1.0, val, 512.0);
				main_dialog[AUDIO_QUALITY_RADIO(1)].flags |= D_SELECTED;
				main_dialog[AUDIO_QUALITY_RADIO(0)].flags &= ~D_SELECTED;
				if(floor(val) == val)
					snprintf(audio_kbps, sizeof(audio_kbps), "%i",
					         (int)val);
				else
					snprintf(audio_kbps, sizeof(audio_kbps), "%f", val);
				break;
			}

			case 'V':
			{
				float val = atof(optarg);
				val = MID(45.0, val, 2000.0);
				main_dialog[VIDEO_QUALITY_RADIO(1)].flags |= D_SELECTED;
				main_dialog[VIDEO_QUALITY_RADIO(0)].flags &= ~D_SELECTED;
				if(floor(val) == val)
					snprintf(video_kbps, sizeof(video_kbps), "%i",
					         (int)floor(val));
				else
					snprintf(video_kbps, sizeof(video_kbps), "%f", val);
				break;
			}

			case 's':
				snprintf(aspect_n, sizeof(aspect_n), "%i", atoi(optarg));
				main_dialog[VIDEO_ASPECT_CHECK].flags |= D_SELECTED;
				override_aspect = !0;
				break;

			case 'S':
				snprintf(aspect_d, sizeof(aspect_d), "%i", atoi(optarg));
				main_dialog[VIDEO_ASPECT_CHECK].flags |= D_SELECTED;
				override_aspect = !0;
				break;

			case 'f':
				snprintf(fps_n, sizeof(fps_n), "%i", atoi(optarg));
				main_dialog[VIDEO_FPS_CHECK].flags |= D_SELECTED;
				override_fps = !0;
				break;

			case 'F':
				snprintf(fps_d, sizeof(fps_d), "%i", atoi(optarg));
				main_dialog[VIDEO_FPS_CHECK].flags |= D_SELECTED;
				override_fps = !0;
				break;

			case 'e':
				one_shot = TRUE;
				break;

			default:
				usage();
		}
	}

	while(optind<argc)
	{
		/* assume that anything following the options must be a filename */
		id_file(argv[optind]);
		optind++;
	}


	c = desktop_color_depth();
	if(c > 0)
	{
		set_color_depth(c);
		if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, SW, SH, 0, 0) != 0)
		{
			allegro_message("Could not create 640x480 window\n");
			exit(1);
		}
	}
	if(!gfx_driver)
	{
		int modes[] = { 32, 16, 15, 8, 0 };
		int i;
		for(i = 0;modes[i];++i)
		{
			set_color_depth(modes[i]);
			if(set_gfx_mode(GFX_AUTODETECT, SW, SH, 0, 0) == 0)
				break;
		}
		if(!modes[i])
		{
			allegro_message("Could not init 640x480 graphics\n");
			exit(1);
		}
	}

	// Try to set a reasonable background mode
	if(set_display_switch_mode(SWITCH_BACKGROUND) == -1)
		set_display_switch_mode(SWITCH_BACKAMNESIA);
	set_close_button_callback(close_callback);

#if 0
	gui_fg_color = makecol(  0,   0,   0);
	gui_mg_color = makecol(128, 128, 128);
	gui_bg_color = makecol(255, 255, 255);
#else
	gui_fg_color = makecol(213, 213, 237);
	gui_mg_color = makecol( 74,  81, 128);
	gui_bg_color = makecol( 57,  64,  98);
#endif
	set_dialog_color(main_dialog, gui_fg_color, gui_bg_color);
	set_dialog_color(encoder_dialog, gui_fg_color, gui_bg_color);
	set_dialog_color(vidplay_dlg, gui_fg_color, gui_bg_color);
	centre_dialog(encoder_dialog);

	gui_mouse_focus = TRUE;

	enable_hardware_cursor();

	while(TRUE)
	{
		int ret;

		cleanup();

		if(strlen(ofile_fname) && is_relative_filename(ofile_fname))
		{
			char *buf = strdup(ofile_fname);
			if(!buf)
				return 1;
			get_executable_name(buffer, sizeof(buffer));
			make_absolute_filename(ofile_fname, buffer, buf,
			                       sizeof(ofile_fname));
			free(buf);
		}

		if((main_dialog[OFILE_RADIO_BTN(0)].flags&D_SELECTED) &&
		   file_exists(ofile_fname, ~FA_DIREC, NULL))
			main_dialog[TEST_BTN].flags &= ~D_DISABLED;
		else
			main_dialog[TEST_BTN].flags |= D_DISABLED;

		if((main_dialog[AUDIO_RADIO_BTN(2)].flags&D_SELECTED) &&
		   (main_dialog[VIDEO_RADIO_BTN(2)].flags&D_SELECTED))
			main_dialog[ENCODE_BTN].flags |= D_DISABLED;
		else
			main_dialog[ENCODE_BTN].flags &= ~D_DISABLED;

		fixup_video_area();
		fixup_audio_area();

		select_mouse_cursor(MOUSE_CURSOR_ARROW);
		ret = do_dialog(main_dialog, -1);
		switch(ret)
		{
			case -1:
				continue;

			case OFILE_RADIO_BTN(0):
			case OFILE_RADIO_BTN(1):
			{
				int b = ret - OFILE_RADIO_BTN(0);
				main_dialog[OFILE_RADIO_BTN(  b)].flags |= D_SELECTED;
				main_dialog[OFILE_RADIO_BTN(1-b)].flags &= ~D_SELECTED;
				continue;
			}

			case TEST_BTN:
				simple_player(ofile_fname);
				continue;

			case CANCEL_BTN:
				quit();

			case VIDEO_RADIO_BTN(0):
				main_dialog[VIDEO_RADIO_BTN(0)].flags |= D_SELECTED;
				main_dialog[VIDEO_RADIO_BTN(1)].flags &= ~D_SELECTED;
				main_dialog[VIDEO_RADIO_BTN(2)].flags &= ~D_SELECTED;
				continue;
			case VIDEO_RADIO_BTN(1):
				main_dialog[VIDEO_RADIO_BTN(0)].flags &= ~D_SELECTED;
				main_dialog[VIDEO_RADIO_BTN(1)].flags |= D_SELECTED;
				main_dialog[VIDEO_RADIO_BTN(2)].flags &= ~D_SELECTED;
				continue;
			case VIDEO_RADIO_BTN(2):
				main_dialog[VIDEO_RADIO_BTN(0)].flags &= ~D_SELECTED;
				main_dialog[VIDEO_RADIO_BTN(1)].flags &= ~D_SELECTED;
				main_dialog[VIDEO_RADIO_BTN(2)].flags |= D_SELECTED;
				continue;

			case VIDEO_QUALITY_RADIO_0:
			case VIDEO_QUALITY_RADIO_1:
			{
				int b = (ret-VIDEO_QUALITY_RADIO_0) /
				        (VIDEO_QUALITY_RADIO_1-VIDEO_QUALITY_RADIO_0);
				main_dialog[VIDEO_QUALITY_RADIO(  b)].flags |= D_SELECTED;
				main_dialog[VIDEO_QUALITY_RADIO(1-b)].flags &= ~D_SELECTED;
				continue;
			}

			case VIDEO_ASPECT_CHECK:
				if((override_aspect=!override_aspect))
					main_dialog[VIDEO_ASPECT_CHECK].flags |= D_SELECTED;
				else
					main_dialog[VIDEO_ASPECT_CHECK].flags &= ~D_SELECTED;
				continue;

			case VIDEO_FPS_CHECK:
				if((override_fps=!override_fps))
					main_dialog[VIDEO_FPS_CHECK].flags |= D_SELECTED;
				else
					main_dialog[VIDEO_FPS_CHECK].flags &= ~D_SELECTED;
				continue;

			case AUDIO_RADIO_BTN(0):
				main_dialog[AUDIO_RADIO_BTN(0)].flags |= D_SELECTED;
				main_dialog[AUDIO_RADIO_BTN(1)].flags &= ~D_SELECTED;
				main_dialog[AUDIO_RADIO_BTN(2)].flags &= ~D_SELECTED;
				continue;
			case AUDIO_RADIO_BTN(1):
				main_dialog[AUDIO_RADIO_BTN(0)].flags &= ~D_SELECTED;
				main_dialog[AUDIO_RADIO_BTN(1)].flags |= D_SELECTED;
				main_dialog[AUDIO_RADIO_BTN(2)].flags &= ~D_SELECTED;
				continue;
			case AUDIO_RADIO_BTN(2):
				main_dialog[AUDIO_RADIO_BTN(0)].flags &= ~D_SELECTED;
				main_dialog[AUDIO_RADIO_BTN(1)].flags &= ~D_SELECTED;
				main_dialog[AUDIO_RADIO_BTN(2)].flags |= D_SELECTED;
				continue;

			case AUDIO_QUALITY_RADIO_0:
			case AUDIO_QUALITY_RADIO_1:
			{
				int b = (ret-AUDIO_QUALITY_RADIO_0) /
				        (AUDIO_QUALITY_RADIO_1-AUDIO_QUALITY_RADIO_0);
				main_dialog[AUDIO_QUALITY_RADIO(  b)].flags |= D_SELECTED;
				main_dialog[AUDIO_QUALITY_RADIO(1-b)].flags &= ~D_SELECTED;
				continue;
			}

			case AUDIO_SKIP_CHECK:
			{
				static int override = 0;
				if((override=!override))
					main_dialog[AUDIO_SKIP_CHECK].flags |= D_SELECTED;
				else
					main_dialog[AUDIO_SKIP_CHECK].flags &= ~D_SELECTED;
				continue;
			}


			case OFILE_BROWSE_BTN:
			case AUDIO_BROWSE_BTN:
			case VIDEO_BROWSE_BTN:
			{
				const char *msg;
				const char *ext;
				char *bkp = strdup(main_dialog[ret+1].dp);

				if(ret == AUDIO_BROWSE_BTN)
				{
					if((main_dialog[AUDIO_RADIO_BTN(1)].flags&D_SELECTED))
					{
						msg = "Select audio pipe command";
						ext = "";
					}
					else
					{
						msg = "Select audio source";
						ext = "WAV";
					}
				}
				else if(ret == VIDEO_BROWSE_BTN)
				{
					if((main_dialog[VIDEO_RADIO_BTN(1)].flags&D_SELECTED))
					{
						msg = "Select video pipe command";
						ext = "";
					}
					else
					{
						msg = "Select video source";
						ext = "YUV;TES";
					}
				}
				else
				{
					if((main_dialog[OFILE_RADIO_BTN(1)].flags&D_SELECTED))
					{
						msg = "Select output pipe command";
						ext = "";
					}
					else
					{
						msg = "Select output file";
						ext = "OGG;OGM";
					}
				}

				if(file_select_ex(msg, main_dialog[ret+1].dp, ext,
				                  main_dialog[ret+1].d1, 320, 240) == 0)
					strcpy(main_dialog[ret+1].dp, bkp);

				free(bkp);
				continue;
			}
		}

		if(quit_now)
			break;

		if((main_dialog[AUDIO_RADIO_BTN(2)].flags&D_SELECTED) &&
		   (main_dialog[VIDEO_RADIO_BTN(2)].flags&D_SELECTED))
		{
			alert("", "You must enable video and/or audio input!", "",
			      "Okay", NULL, 0, 0);
			continue;
		}

		if(!(main_dialog[VIDEO_RADIO_BTN(2)].flags&D_SELECTED))
		{
			if((main_dialog[VIDEO_QUALITY_RADIO_0].flags&D_SELECTED))
			{
				video_q = rint(main_dialog[VIDEO_QUALITY_RADIO_0+2].d2*0.63);
				if(video_q<0 || video_q>63)
				{
					alert("Illegal video quality", "(choose 0 through 10)", "",
					      "Okay", NULL, 0, 0);
					continue;
				}
				video_r=0;
			}
			else// if(((main_dialog[VIDEO_QUALITY_RADIO_1].flags&D_SELECTED))
			{
				video_r = atof(main_dialog[VIDEO_QUALITY_RADIO_1+2].dp) *
				          1000.0;
				if(video_r<45000 || video_r>2000000)
				{
					alert("Illegal video bitrate",
					      "(choose 45kbps through 2000kbps)", "", "Okay",
					      NULL, 0, 0);
					continue;
				}
				video_q=0;
			}

			if((main_dialog[VIDEO_ASPECT_CHECK].flags&D_SELECTED))
			{
				video_an = atoi(main_dialog[VIDEO_ASPECT_VAL  ].dp);
				video_ad = atoi(main_dialog[VIDEO_ASPECT_VAL+1].dp);
				if(video_an <= 0 || video_ad <=0)
				{
					alert("Illegal aspect mode",
					      "(values should be greater than 0)", "", "Okay",
					      NULL, 0, 0);
					continue;
				}
			}
			else
			{
				video_an = -1;
				video_ad = -1;
			}

			if((main_dialog[VIDEO_FPS_CHECK].flags&D_SELECTED))
			{
				video_hzn = atoi(main_dialog[VIDEO_FPS_VAL  ].dp);
				video_hzd = atoi(main_dialog[VIDEO_FPS_VAL+1].dp);
				if(video_hzn <= 0 || video_hzd <=0)
				{
					alert("Illegal frame rate",
					      "(values should be greater than 0)", "", "Okay",
					      NULL, 0, 0);
					continue;
				}
			}
			else
			{
				video_hzn = -1;
				video_hzd = -1;
			}

			if(open_video() != 0)
				continue;
		}

		if(!(main_dialog[AUDIO_RADIO_BTN(2)].flags&D_SELECTED))
		{
			if((main_dialog[AUDIO_QUALITY_RADIO_0].flags&D_SELECTED))
			{
				audio_q = main_dialog[AUDIO_QUALITY_RADIO_0+2].d2*0.0099;
				if(audio_q<0.0 || audio_q>1.0)
				{
					alert("Illegal audio quality", "(choose 0 through 10)", "",
					      "Okay", NULL, 0, 0);
					continue;
				}
				audio_r=-1;
			}
			else// if(((main_dialog[AUDIO_QUALITY_RADIO_1].flags&D_SELECTED))
			{
				audio_r = atof(main_dialog[AUDIO_QUALITY_RADIO_1+2].dp) *
				          1000.0;
				if(audio_r<1000 || audio_r>512000)
				{
					alert("Illegal audio bitrate",
					      "(choose 1kbps through 512kbps)", "", "Okay",
					      NULL, 0, 0);
					continue;
				}
				audio_q=-99;
			}

			if((main_dialog[AUDIO_SKIP_CHECK].flags&D_SELECTED))
				audio_skip = atof(main_dialog[AUDIO_SKIP_VAL].dp);
			else
				audio_skip = 0.0f;

			if(open_audio() != 0)
				continue;
		}


		if(strlen(ofile_fname) <= 0)
		{
			alert("", "No output specified", "","Yes", NULL, 0, 0);
			continue;
		}

		if(is_relative_filename(ofile_fname))
		{
			char *buf = strdup(ofile_fname);
			if(!buf)
				return 1;
			get_executable_name(buffer, sizeof(buffer));
			make_absolute_filename(ofile_fname, buffer, buf,
			                       sizeof(ofile_fname));
			free(buf);
		}

		if((main_dialog[OFILE_RADIO_BTN(1)].flags&D_SELECTED))
		{
			FILE *fp = popen(ofile_fname, "w");
			if(fp)
			{
				outfile = pack_fopen_vtable(&openfile_vtable, fp);
				if(!outfile)
					pclose(fp);
			}
		}
		else if((main_dialog[OFILE_RADIO_BTN(0)].flags&D_SELECTED))
		{
			if(file_exists(ofile_fname, ~0, NULL))
			{
				if(alert(ofile_fname, "File exists! Overwrite?",
				         "(The file will be irreversably lost!)",
				         "Yes", "No", 0, 0) == 2)
					continue;
			}
			outfile = pack_fopen(ofile_fname, F_WRITE);
		}

		if(!outfile)
		{
			alert("Couldn't create output", ofile_fname, "", "Okay",
				  NULL, 0, 0);
			continue;
		}

		/* yayness.  Set up Ogg output stream */
		srand(time(NULL));
		{
			/* need two inequal serial numbers */
			serial1 = rand();
			serial2 = rand();
			if(serial1 == serial2)
				serial2 ^= ~0;
		}

		do_dialog(encoder_dialog, -1);
		if(one_shot || quit_now)
			break;
	}

	return 0;
}
END_OF_MAIN()
