/*
 * A(n insanely) featured example program for the APEG add-on library
 */

#include <string.h>
#include <stdio.h>

#include <allegro.h>
#include "apeg.h"


#ifdef _MSC_VER
#define strcasecmp stricmp
#endif

/* Globals to help read command line arguments */
static char **global_argv;
static int global_argc;

/* Searches the command line for the named parameter */
static int check_param(const char *param, int *idx)
{
	int i;
	for(i = global_argc-1;i >= 1;--i)
	{
		if(strcasecmp(global_argv[i], param) == 0)
		{
			if(idx) *idx = i;
			return i;
		}
	}

	return 0;
}


/* Some crap to use a YUV overlay in X using Xv */
#ifdef USE_XV
#include <xalleg.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>
#include <X11/extensions/Xvlib.h>

static XShmSegmentInfo shminfo1;

/* We want I420, not YV12. YV12 has the chrome values switched */
#define GUID_I420_PLANAR  0x30323449
#define GUID_YV12_PLANAR  0x32315659
static int portNum = -1;
static int was_autopainting;

/* Opens the overlay port and returns an XvImage, storing the color key value */
XvImage *init_display(int w, int h, int format, int *colorkey)
{
	static XvAdaptorInfo *info = NULL;
	static unsigned int numAdaptors = 0;

	XvImage *xvimage = NULL;
	unsigned int i, j, k, numImages;

	int adaptor = -1;
	Atom atom;

	if (!colorkey) {
		fprintf(stderr, "Can't store overlay color key\n");
		return NULL;
	}

	if (_xwin.display == NULL) {
		fprintf(stderr, "Can't open display\n");
		return NULL;
	}

	/* Lock the display */
	acquire_screen();

	if(info == NULL && XvQueryAdaptors(_xwin.display, _xwin.window,
	                                   &numAdaptors, &info) != Success)
		goto end;

	for(i = 0; i < numAdaptors; i++)
	{
		if(info[i].type & XvImageMask)
		{
			/* Adaptor has XvImage support */
			XvImageFormatValues *formats = XvListImageFormats(_xwin.display,
			                                      info[i].base_id, &numImages);

			for(j = 0; j < numImages; j++)
			{
				if(formats[j].id == format)
				{
					/* It supports our format */
					adaptor = i;

					for(k = 0; k < info[i].num_ports; k++)
					{
						/* try to grab a port */
						if(XvGrabPort(_xwin.display, info[i].base_id + k,
						              CurrentTime) == Success)
						{
							portNum = info[i].base_id + k;
							break;
						}
					}
				}
				if(portNum != -1) break;
			}
			XFree(formats);
		}
	}

	if (adaptor == -1) {
		fprintf(stderr, "No suitable Xv adaptor found\n");
		goto end;
	}

	if (portNum == -1) {
		fprintf(stderr, "No free Xv adaptor ports found\n");
		goto end;
	}

	/* Get the colorkey for the overlay */
	atom = XInternAtom(_xwin.display, "XV_COLORKEY", True);
	if(XvGetPortAttribute(_xwin.display, portNum, atom, colorkey) != Success) {
		XvUngrabPort(_xwin.display, portNum, CurrentTime);
		portNum = -1;
		fprintf(stderr, "Could not get overlay color key\n");
		goto end;
	}

	xvimage = (XvImage*)XvShmCreateImage(_xwin.display, portNum, format, NULL,
	                                     w, h, (void*)&shminfo1);
	if(!xvimage) {
		XvUngrabPort(_xwin.display, portNum, CurrentTime);
		portNum = -1;
		fprintf(stderr, "Could not create %ix%i Xv bitmap\n", w, h);
		goto end;
	}

	shminfo1.shmid = shmget(IPC_PRIVATE, xvimage->data_size, IPC_CREAT|0777);
	shminfo1.shmaddr = xvimage->data = shmat(shminfo1.shmid, 0, 0);
	shmctl(shminfo1.shmid, IPC_RMID, 0);
	shminfo1.readOnly = False;

	if(!XShmAttach(_xwin.display, &shminfo1))
		fprintf(stderr, "XShmAttach failed! (fatal?)\n");

	/* Turn off auto-painting the colorkey, temporarilly */
	atom = XInternAtom(_xwin.display, "XV_AUTOPAINT_COLORKEY", True);
	if(XvGetPortAttribute(_xwin.display, portNum, atom, &was_autopainting) !=
	   Success) {
		fprintf(stderr, "Could not determine auto-painting status\n");
		was_autopainting = 1;
	}
	if(was_autopainting && XvSetPortAttribute(_xwin.display, portNum, atom,
	                                          0) != Success)
		fprintf(stderr, "Could not disable auto-painting\n");

	XSync(_xwin.display, False);

	if(format == GUID_I420_PLANAR)
		fprintf(stderr, "I420 overlay enabled\n");
	else if(format == GUID_YV12_PLANAR)
		fprintf(stderr, "YV12 overlay enabled\n");

end:
	/* Release the display before exiting */
	release_screen();
	return xvimage;
}

/* Copies an XvImage to the overlay */
void blit_xvimage(XvImage *xvimage, int src_x, int src_y, int src_w,
                  int src_h, int dst_x, int dst_y, int dst_w, int dst_h)
{
	acquire_screen();
	XvShmPutImage(_xwin.display, portNum, _xwin.window, _xwin.gc, xvimage,
	              src_x, src_y, src_w, src_h, dst_x, dst_y, dst_w, dst_h,
	              True);
	release_screen();
}

/* Closes the overlay */
void exit_display(XvImage *xvimage)
{
	acquire_screen();
	{
		/* Restore auto-painting status */
		Atom atom = XInternAtom(_xwin.display, "XV_AUTOPAINT_COLORKEY", True);
		if(was_autopainting && XvSetPortAttribute(_xwin.display, portNum,
		                                          atom, was_autopainting) !=
		                       Success)
			fprintf(stderr, "Could not restore auto-painting mode\n");

		if(shminfo1.shmaddr)
		{
			XShmDetach(_xwin.display, &shminfo1);
			/*XDestroyImage(xvimage);*/
			shmdt(shminfo1.shmaddr);
		}

		free(xvimage);

		if(portNum != -1)
			XvUngrabPort(_xwin.display, portNum, CurrentTime);
		portNum = -1;
	}
	release_screen();
}


/* Initialize the window, and get an I420 xvimage */
typedef struct AList {
	int switch_chrome;
	int str_w, str_h;
	int colorkey;
	XvImage *xvimage;
	int do_fs;
} AList;
static int InitYUVDisplay(APEG_STREAM *stream, int coded_w, int coded_h,
                          void *arg)
{
	int ret = -1;
	AList *args = (AList*)arg;

	if(gfx_driver)
		exit_display(args->xvimage);

	/* Gets the frame extents, taking into account the video's aspect ratio */
	apeg_get_video_size(stream, &args->str_w, &args->str_h);

	/* Overlay must use the desktop's depth (for proper color-keying) */
	set_color_depth(desktop_color_depth());
	if(args->do_fs)
	{
		int w, h;

		/* For fullscreen, try to use the same size as the current desktop */
		if(get_desktop_resolution(&w, &h) != 0)
		{
			w = 640;
			h = 480;
		}

		ret = set_gfx_mode(GFX_AUTODETECT_FULLSCREEN, w, h, 0, 0);
		if(ret == 0)
		{
			/* Expand the frame extents to fill the screen (yay for hw
			   scaling) */
			if(stream->aspect_ratio > (float)w/(float)h)
			{
				args->str_w = w;
				args->str_h = (int)((float)w / stream->aspect_ratio);
			}
			else
			{
				args->str_w = (int)((float)h * stream->aspect_ratio);
				args->str_h = h;
			}
		}
	}

	if(ret != 0)
	{
		int w = (args->str_w+3)&(~3);
		int h = (args->str_h+3)&(~3);
		if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, w, h, 0, 0) != 0)
			return 1;
	}

	args->switch_chrome = FALSE;
	args->xvimage = init_display(coded_w, coded_h, GUID_I420_PLANAR,
	                             &args->colorkey);
	if(!args->xvimage)
	{
		args->xvimage = init_display(coded_w, coded_h, GUID_YV12_PLANAR,
		                             &args->colorkey);
		if(!args->xvimage)
		{
			set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
			return 1;
		}
		args->switch_chrome = TRUE;
	}

	return 0;
}

/* Copy the uncompressed YUV data to the XvImage and blit it to the display */
static void DisplayYUVFrame(APEG_STREAM *stream, unsigned char **yuv_src, void *arg)
{
	AList *args = (AList*)arg;
	XvImage *xvimage = args->xvimage;
	int plane_size = xvimage->width * xvimage->height;
	unsigned char *y_plane = xvimage->data;
	unsigned char *u_plane = xvimage->data + plane_size;
	unsigned char *v_plane = u_plane + plane_size/4;
	int x, y;

	if(args->switch_chrome)
	{
		unsigned char *u_tmp = u_plane;
		u_plane = v_plane;
		v_plane = u_tmp;
	}

	if(stream->pixel_format == APEG_444)
	{
		/* Downsample 4:4:4 -> 4:2:0 */
		unsigned char *u_src = yuv_src[1];
		unsigned char *v_src = yuv_src[2];
		int x, y;

		memcpy(y_plane, yuv_src[0], plane_size);
		for(y = 0;y < (stream->h+1)/2;++y)
		{
			for(x = 0;x < (stream->w+1)/2;++x)
			{
				u_plane[x] = u_src[x*2];
				v_plane[x] = v_src[x*2];
			}
			u_plane += xvimage->width/2;
			v_plane += xvimage->width/2;
			u_src += xvimage->width*2;
			v_src += xvimage->width*2;
		}
	}
	else if(stream->pixel_format == APEG_422)
	{
		/* Downsample 4:2:2 -> 4:2:0 */
		int w16 = xvimage->width/2;
		int y;

		memcpy(y_plane, yuv_src[0], plane_size);
		for(y = 0;y < (stream->h+1)/2;++y)
		{
			memcpy(u_plane + y*w16, yuv_src[1] + y*2*w16, xvimage->width/2);
			memcpy(v_plane + y*w16, yuv_src[2] + y*2*w16, xvimage->width/2);
		}
	}
	else if(stream->pixel_format == APEG_420)
	{
		memcpy(y_plane, yuv_src[0], plane_size);
		memcpy(u_plane, yuv_src[1], plane_size/4);
		memcpy(v_plane, yuv_src[2], plane_size/4);
	}

	x = (SCREEN_W-args->str_w)/2;
	y = (SCREEN_H-args->str_h)/2;
	acquire_screen();
	rectfill(screen, x, y, x+args->str_w-1, y+args->str_h-1, args->colorkey);
	blit_xvimage(xvimage, 0, 0, stream->w, stream->h,
	                      x, y, args->str_w, args->str_h);
	release_screen();
}
#endif


int main(int argc, char *argv[])
{
	APEG_STREAM *stream;
	int showed, skipped;
	float speed_mult;
	int x_off, y_off;
	int sw, sh;
	int ret, depth;
	void *mpeg_dat;
#ifdef USE_XV
	AList args;
#endif

	if(allegro_init() != 0)
	{
		fprintf(stderr, "Error initializing Allegro: %s\n", allegro_error);
		return -1;
	}

	global_argv = argv;
	global_argc = argc;

	// Make sure we have at least a file to play, or see if the user is asking
	// for help
	if(argc < 2 || check_param("-help", NULL))
	{
		allegro_message(
"Usage: %s <filename> [options]\n"
"  <filename> can be either an MPEG/Ogg file or an embedded datafile object.\n"
"  [-memplay] load the whole file into memory before playing.\n"
"  [-speed x] playback at x normal speed.\n"
"  [-bpp x] use x-bit color, if available (default: display depth).\n"
"  [-downsample x] downsample audio to the power of x (0,1,2;default: 0).\n"
"  [-downchannel] force mono audio output.\n"
"  [-largeaudiobuffer] use a large audio buffer (for CPU intense streams).\n"
"  [-nosound] disable audio playback.\n"
"  [-novideo] disable video playback.\n"
"  [-quality x] use quality level x (0,1,2;default: 2).\n"
"  [-framedrop] enable frame-dropping.\n"
"  [-fs] fullscreen playback.\n"
#ifdef USE_XV
"  [-overlay] attempt to use a YUV overlay (-bpp is ignored).\n"
#endif
"\n"
"Esc will terminate playback, and F will switch between fullscreen and\n"
"windowed playback.\n", argv[0]);
		return 1;
	}

	install_timer();
	install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL);
	set_volume_per_voice(0);

#ifdef USE_XV
	memset(&args, 0, sizeof(args));
	// Set overlay callbacks
	if(check_param("-overlay", NULL))
		apeg_set_display_callbacks(InitYUVDisplay, DisplayYUVFrame, &args);
#endif

	// Set gfx depth
	depth = desktop_color_depth();
	if(!depth)
		depth = 32;

	// Override decoding depth
	if(check_param("-bpp", &ret) > 0)
	{
		++ret;
		if(argc > ret)
			depth = atoi(argv[ret]);
	}
	apeg_set_display_depth(depth);

	// Set video decode quality
	if(check_param("-quality", &ret) > 0)
	{
		++ret;
		if(argc > ret)
		{
			int q = atoi(argv[ret]);
			apeg_set_quality(q);
		}
	}

	// Set downsampling rate
	if(check_param("-downsample", &ret) > 0)
	{
		++ret;
		if(argc > ret)
		{
			int d = atoi(argv[ret]);
			apeg_downsample_audio(d);
		}
	}

	// Set downchanneling
	if(check_param("-downchannel", NULL))
		apeg_downchannel_audio(TRUE);

	// Enable frame dropping
	if(check_param("-framedrop", NULL))
		apeg_enable_framedrop(TRUE);

	if(check_param("-largeaudiobuffer", NULL))
		apeg_set_audio_bufsize(apeg_set_audio_bufsize(-1)*3);

	// Disable video playback?
	if(check_param("-novideo", NULL))
		apeg_ignore_video(TRUE);

	// Disable audio stream?
	if(check_param("-nosound", NULL))
		apeg_ignore_audio(TRUE);

#ifdef USE_XV
	// Select fullscreen?
	if(check_param("-fs", NULL))
		args.do_fs = TRUE;
#endif

	// Optionally load the stream into memory, then open it
	mpeg_dat = NULL;
	if(check_param("-memplay", NULL))
	{
		int size = file_size(argv[1]);

		if(size > 0)
		{
			mpeg_dat = malloc(size);
			if(mpeg_dat)
			{
				PACKFILE *pf = pack_fopen(argv[1], F_READ);
				if(pf)
				{
					size = pack_fread(mpeg_dat, size, pf);
					pack_fclose(pf);
					if(size < 0)
					{
						free(mpeg_dat);
						mpeg_dat = NULL;
					}
				}
				else
				{
					free(mpeg_dat);
					mpeg_dat = NULL;
				}
			}
		}

		if(!mpeg_dat)
		{
			allegro_message("Couldn't load %s!\n", argv[1]);
			return -1;
		}

		stream = apeg_open_memory_stream(mpeg_dat, size);
	}
	else
		stream = apeg_open_stream(argv[1]);

	if(!stream)
	{
		allegro_message("APEG Error: %s\n", apeg_error);
		ret = -1;
		goto end;
	}

	if((stream->flags & APEG_HAS_VIDEO))
		apeg_get_video_size(stream, &sw, &sh);

	if((stream->flags & APEG_MPG_VIDEO))
		fprintf(stderr, "Video: MPEG-1\n"
		                " %dx%d (%.2f) => %dx%d @ %.02ffps %dkbps\n",
		        stream->w, stream->h, stream->aspect_ratio, sw, sh,
		        stream->frame_rate, stream->bit_rate/1000);
	else if((stream->flags & APEG_THEORA_VIDEO))
		fprintf(stderr, "Video: Theora\n"
		                " %dx%d (%.2f) => %dx%d @ %.02ffps\n",
		        stream->w, stream->h, stream->aspect_ratio, sw, sh,
		        stream->frame_rate);

	if((stream->flags & APEG_MPG_AUDIO))
		fprintf(stderr, "Audio: MPEG layer-%d\n"
		                " %dhz, %d channels %dkbps\n",
		        stream->audio.layer, stream->audio.freq,
		        stream->audio.channels, stream->audio.kbps);
	else if((stream->flags & APEG_VORBIS_AUDIO))
		fprintf(stderr, "Audio: Vorbis\n"
		                " %dhz, %d channels\n",
		        stream->audio.freq, stream->audio.channels);
	fprintf(stderr, "Total playtime: ");
	if(stream->length > 60.0*60.0)
		fprintf(stderr, "%uh ", (unsigned int)stream->length/(60*60));
	if(stream->length > 60.0)
		fprintf(stderr, "%02um ", (unsigned int)(stream->length/60)%60);
	fprintf(stderr, "%02us\n", (unsigned int)stream->length%60);

	// Make sure we have something to play...
	if(!(stream->flags&(APEG_HAS_VIDEO|APEG_HAS_AUDIO)))
	{
		allegro_message("No audio or video to play!\n");
		ret = -1;
		goto end;
	}

	// Set the timer speed
	speed_mult = 1.0f;
	if(check_param("-speed", &ret) > 0)
	{
		++ret;
		if(ret < argc)
		{
			float m = atof(argv[ret]);
			if(m > 0.0f)
			{
				speed_mult = m;
				apeg_set_stream_rate(stream, m);
			}
		}
	}

	// Create a custom-fit window for windowed playback.
	if(!gfx_driver && (!check_param("-fs", NULL) ||
	                   !(stream->flags&APEG_HAS_VIDEO)))
	{
		int w = 640;
		int h = 480;

		// The window needs to be a multiple of 4(I think?), so make sure
		// it's big enough.
		if((stream->flags&APEG_HAS_VIDEO))
		{
			w = (sw+3)&(~3);
			h = (sh+3)&(~3);
		}

		set_color_depth(depth);
		if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, w, h, 0, 0) != 0)
		{
			allegro_message("Couldn't create %ix%ix%i window:\n%s\n",
			                w, h, depth, allegro_error);
			ret = -1;
			goto end;
		}
	}

	// Set fullscreen mode now
	if(!gfx_driver)
	{
		set_color_depth(depth);
		if(set_gfx_mode(GFX_AUTODETECT_FULLSCREEN, 640, 480, 0, 0) != 0)
		{
			allegro_message("Couldn't set mode 640x480x%i:\n%s\n", depth,
			                allegro_error);
			ret = -1;
			goto end;
		}
	}

	// Reset APEG's color tables since we just entered a valid video mode
	apeg_reset_colors(stream);

	gui_fg_color = makecol(0, 0, 0);
	gui_mg_color = makecol(128, 128, 128);
	gui_bg_color = makecol(255, 255, 255);

	// Try to set a reasonable background mode
	if(set_display_switch_mode(SWITCH_BACKGROUND) == -1)
		set_display_switch_mode(SWITCH_BACKAMNESIA);

	install_keyboard();

	// Get the offsets to center the image
	x_off = (SCREEN_W-sw) / 2;
	y_off = (SCREEN_H-sh) / 2;

replay:
	// Clear the screen
	clear_to_color(screen, makecol(0, 0, 0));

	if(!(stream->flags&APEG_HAS_VIDEO))
	{
		textprintf_ex(screen, font, 0, 0, makecol(255, 255, 255), -1,
		              "Playing: %s", get_filename(argv[1]));
		if((stream->flags & APEG_MPG_AUDIO))
		{
			textprintf_ex(screen, font, 0, text_height(font),
			              makecol(255, 255, 255), -1,
			              "Audio: MPEG layer-%d", stream->audio.layer);
			textprintf_ex(screen, font, 0, text_height(font)*2,
			              makecol(255, 255, 255), -1,
			              " %dhz, %d channels", stream->audio.freq,
			              stream->audio.channels);
			textprintf_ex(screen, font, 0, text_height(font)*3,
			              makecol(255, 255, 255), -1,
			              " %dkbps", stream->audio.kbps);
		}
		else if((stream->flags & APEG_VORBIS_AUDIO))
		{
			textprintf_ex(screen, font, 0, text_height(font),
			              makecol(255, 255, 255), -1, "Audio: Vorbis");
			textprintf_ex(screen, font, 0, text_height(font)*2,
			              makecol(255, 255, 255), -1, " %dhz, %d channels",
			              stream->audio.freq, stream->audio.channels);
		}
	}

	showed = skipped = 0;

	// Reset the video timer (it has been running since we opened the video).
	if((stream->flags & APEG_HAS_VIDEO))
	{
		stream->timer = -1;
		while(stream->timer == -1)
			;
	}

	// Get the next video frame
	while((ret = apeg_advance_stream(stream, FALSE)) == APEG_OK)
	{
		// If the bitmap was updated, blit it to the screen, else mark it as
		// skipped, and just continue on
		if(stream->frame_updated > 0)
		{
			++showed;
			if(stream->bitmap)
				stretch_blit(stream->bitmap, screen, 0, 0, stream->w,
				             stream->h, x_off, y_off, sw, sh);
		}
		else if(stream->frame_updated == 0)
			++skipped;

		// Check if a key was pressed
		if(keypressed())
		{
			int sc = readkey() >> 8;

			if(sc == KEY_ESC)
				break;

			if(sc == KEY_F)
			{
				if((stream->flags & APEG_HAS_VIDEO) && stream->bitmap)
				{
					int to_fs;

					// Pause the stream while we attempt mode switching
					apeg_set_stream_rate(stream, 0.0f);

					to_fs = gfx_driver->windowed;
					if(to_fs)
					{
						if(set_gfx_mode(GFX_AUTODETECT_FULLSCREEN, 640, 480,
						                0, 0) != 0)
						{
							allegro_message("Couldn't set 640x480x%i:\n%s\n",
							                depth, allegro_error);
							ret = -1;
							goto end;
						}
					}
					if(!to_fs || !gfx_driver)
					{
						int w = (sw+3)&(~3);
						int h = (sh+3)&(~3);

						if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, w, h, 0,
						                0) != 0)
						{
							allegro_message(
							          "Couldn't create %ix%ix%i window:\n%s\n",
							          w, h, depth, allegro_error);
							ret = -1;
							goto end;
						}
					}

					apeg_reset_colors(stream);

					x_off = (SCREEN_W-sw) / 2;
					y_off = (SCREEN_H-sh) / 2;

					clear_keybuf();

					/* Resume the stream */
					apeg_set_stream_rate(stream, speed_mult);
				}
#ifdef USE_XV
				else if((stream->flags & APEG_HAS_VIDEO))
				{
					int w16, h16;

					apeg_set_stream_rate(stream, 0.0f);

					/* Save the image size */
					w16 = args.xvimage->width;
					h16 = args.xvimage->height;

					args.do_fs = gfx_driver->windowed;
					if(InitYUVDisplay(stream, w16, h16, &args) != 0)
					{
						ret = -1;
						goto end;
					}

					clear_keybuf();

					apeg_set_stream_rate(stream, speed_mult);
				}
#endif
				else
					fprintf(stderr, "Can't go fullscreen!\n");
			}
		}

		if(!stream->audio.flushed && stream->timer <= 0)
			rest(1);
	}
	apeg_set_stream_rate(stream, 0.0);

	// APEG_EOF == no error
	if(ret == APEG_EOF)
		ret = APEG_OK;

	// If error, print a message, else print some stats.
reprint:
	install_mouse();

	enable_hardware_cursor();
	select_mouse_cursor(MOUSE_CURSOR_ARROW);

	if(ret == APEG_ERROR)
	{
		alert("APEG Error", apeg_error, "", "Okay", NULL, 0, 0);
	}
	else if((stream->flags & APEG_HAS_VIDEO) && stream->frame > 0)
	{
		static char l1[128], l2[128], l3[128];
		snprintf(l1, sizeof(l1), "Frames Shown:   %-6i", showed);
		snprintf(l2, sizeof(l2), "Frames Skipped: %-6i", skipped);
		snprintf(l3, sizeof(l3), "Average Speed: %2.2ffps (%3.1f%%)",
		         showed * stream->frame_rate / stream->frame,
		         showed * 100.0f / stream->frame);

		// Give the choice to replay
		if(alert(l1, l2, l3, "Okay", "Replay", KEY_ESC, 0) == 2)
		{
			remove_mouse();

			if((ret = apeg_reset_stream(stream)) != APEG_OK)
				goto reprint;

			apeg_set_stream_rate(stream, speed_mult);
			goto replay;
		}
	}

#ifdef USE_XV
	exit_display(args.xvimage);
#endif
	set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);

end:
	// Close the mpeg
	apeg_close_stream(stream);
	free(mpeg_dat);

	return ret;
}
END_OF_MAIN()
